;**************************************************************************
;**************************************************************************
;*
;*                         SMS Controller Firmware
;* 
;*                          Version 1.0  10/10/04
;*                           
;*                          Created by P.B.Smith
;*                  (C)2004 Silicon Chip Publications P/L
;*                          All Rights Reserved.
;*
;*                         www.siliconchip.com.au
;*
;*            +-----------------------------------------------+
;*            | This source code is NOT in the public domain! |
;*            +-----------------------------------------------+
;*
;************************************^*************************************
;*
;* Change record
;*
;* Version 1.1  12/01/05
;* - Increased receive buffer size from 128 bytes to 208 bytes.
;* - Added receive buffer overflow flag, so any message size is
;*   handled without error.
;*
;* Version 1.2  08/11/06
;* - Fixed bug in compr_usr routine that was causing incorrect
;*   {string} matching in certain cases.
;* - Removed leading zeros from COUNT command's version output.
;*   This release will display as "v=1.2" instead of "v=01.02".
;*  
;**************************************************************************

.nolist
.include "8515ndef.inc"		;AVR definitions
.list

.equ	VERSION_MAJOR	='1'
.equ	VERSION_MINOR	='2'

;**************************************************************************
;*
;* Registers
;*
;**************************************************************************

.def	astate		=R1	;SREG save (used by int0)
.def	bstate		=R2	;SREG save (used by uart interrupts)
.def	sys_tmr_lo	=R3	;ticks every msec
.def	sys_tmr_hi	=R4	;ticks every msec * 256
.def	min_tmr_lo	=R5	;one minute timer 
.def	min_tmr_hi	=R6	;one minute timer
.def	rx_frame_tmr	=R7	;frame receive timer
.def	rx_ack_tmr	=R8	;acknowledge receive timer
.def	rx_hdr_cnt	=R9	;receive header byte counter
.def	rx_byte_cnt	=R10	;receive frame byte count 
.def	rx_msg_typ	=R11	;last frame message type
.def	tx_byte_cnt	=R12	;transmit frame byte count
.def	tx_retry_cnt	=R13	;transmit frame retry count
.def	tx_seq_no	=R14	;last sent frame sequence number
.def	sms_mem_loc	=R15	;last SMS memory location

.def	A		=R16	;scratch
.def	B		=R17	;scratch
.def	C		=R18	;scratch
.def	D		=R19	;scratch
.def	E		=R20	;scratch
.def	lcnt		=R21	;scratch (loop counter)
.def	stat_cnt	=R22	;cmd_handler return status
.def	aflags		=R23	;various flags (see below)
.def	cflags		=R24
.def	iflags		=R25
.def	XL		=R26	;scratch pointers...
.def	XH		=R27	;
.def	YL		=R28	;
.def	YH		=R29	;
.def	ZL		=R30	;
.def	ZH		=R31	;

; Bit variables

; aflags

.equ	rx_buff_lok	=0	;receive buffer lock
.equ	tx_buff_lok	=1	;transmit buffer lock
.equ	tx_msg_req	=2	;transmit frame request
.equ	rx_too_big	=3	;receive frame length > 208 bytes 
.equ	tx_msg_retry	=4	;retransmit frame request
.equ	tx_fatal_err	=5	;fatal transmit error occurred
.equ	in_sample	=6	;in_port second sample
.equ	tick_flag	=7	;watchdog keep-alive tick

; cflags

.equ	time_out	=0	;charge timer expired
.equ	time_sync	=1	;one minute timer reset
.equ	got_pstat	=2	;got phone status
.equ	serv_err	=3	;operator services error
.equ	log_me_out	=4	;user logout pending

; iflags (used by interrupt handlers)

.equ	tx_frame_ip	=0	;transmit frame in progress
.equ	tx_ack_pend	=1	;transmit acknowledge pending
.equ	rx_frame_ip	=2	;receive frame in progress
.equ	rx_header_ip	=3	;receive header (part) of frame in progress
.equ	rx_ack_pend	=4	;receive acknowledge pending
.equ	rx_frame_rdy	=5	;complete frame received
.equ	rx_to_null	=6	;receive to bit bucket
.equ	rx_delay	=7	;receive delay (skip data)

; eeflags

.equ	user_ack	=0	;set to acknowledge user messages
.equ	log_stat	=1	;set when user logged in
.equ	pwd_stat	=2	;set when password programmed


;**************************************************************************
;*
;* Hardware specific
;*
;**************************************************************************

; Port B bits

.equ	comms_err_led	=0	;phone (fbus) comms error
.equ	serv_err_led	=1	;no service error
.equ	send_err_led	=2	;message send failed
.equ	del_err_led	=3	;message delete failed
.equ	in_use_led	=4	;in use (logged in)

; Port D bits

.equ	chg_ctrl	=4	;charge circuit on/off control
.equ	erase_jmp	=5	;erase eeprom jumper
.equ	pass_jmp	=6	;erase password jumper
.equ	prog_jmp	=7	;program jumper

;NOTE: INT0 (PD2) is used for generating 'software' interrupts.
;Do not connect this pin externally!


;**************************************************************************
;*
;* Constants
;*
;**************************************************************************

.equ	BAUD_115200	=3	;baud rate divisor (7.3728MHz crystal)
.equ	RX_FRAME_TIMEOUT=100	;receive frame timeout (millisecs, 255 max)
.equ	RX_ACK_TIMEOUT	=255	;receive ack timeout (millisecs, 255 max)
.equ	TX_RETRY_COUNT	=3	;transmit frame retry count

.equ	PHONE_POWER_ON	=6	;phone power-on time (secs, 65 max)
.equ	SMS_SEND_TIMEOUT=20	;send sms timeout (secs, 65 max)
.equ	SMS_RETRY_COUNT =5	;max. sms send attempts
.equ	DISCHARGE_LEVEL	=1	;default battery discharge level (0-4)
.equ	TOP_UP_TIME	=10	;battery top-up time (minutes, 65535 max)

; As least two models from the 61xx-based series do not provide battery
; level information over the fbus (via get_pstatus). The 3210 & 3310 have
; been identified so far. Without this information, it is still possible
; to use these models by performing a fixed charge/discharge cycling
; scheme. The default times used are:
 
.equ	CHARGE_TIME	=40	;battery charge time (10-240 minutes)
.equ	DISCHARGE_TIME	=8*60	;battery discharge time (minutes, 65535 max)

; Offsets to elements in pstat_array (see charge_control)

.equ	state		=0	;state machine state
.equ	pmode		=1	;mode
.equ	sigstr		=2	;signal strength
.equ	uflags		=3	;unknown
.equ	psource		=4	;power source
.equ	blevel		=5	;battery level
.equ	utemp		=6	;discharge level or charge time


;**************************************************************************
;*
;* Interrupt vectors
;*
;**************************************************************************

	.cseg
	.org	$0

	rjmp	reset		;reset
	rjmp	int0_irq	;INT0 (used internally)
	rjmp	bad_irq		;INT1
	rjmp	bad_irq		;timer 1 capture
	rjmp    bad_irq		;timer 1 compare A
	rjmp	bad_irq		;timer 1 compare B
	rjmp	bad_irq 	;timer 1 overflow
	rjmp	tmr0_irq	;timer 0 overflow
	rjmp	bad_irq		;SPI transfer complete
	rjmp	rx_irq		;UART receive complete
	rjmp	tx_irq		;UART data register empty
	rjmp	bad_irq		;UART transmit complete
	rjmp	bad_irq		;analog comparator
bad_irq:
	reti


;**************************************************************************
;*
;* Reset entry point
;*
;* Initialise 8515 I/O registers & flags
;*
;**************************************************************************
	
reset:	cli
	ldi	A,low(RAMEND)		;set stack top
	out	SPL,A
	ldi	A,high(RAMEND)
	out	SPH,A

	clr	A
	out	MCUCR,A			;init control register
	sbi	ACSR,ACD		;power down comparator

; Init i/o ports
	
	ldi	A,0
	out	DDRA,A			;port A: 7 - 0 in (in port)
	ldi	A,0b11110000		;unused inputs pulled up
	out	PORTA,A

	ldi	A,0b00011111		;port B: 7-5 in (ISP), 4-0 out (LEDs) 
	out	DDRB,A
	ldi	A,0b11111111		;inputs pulled up, LEDs off
	out	PORTB,A

	ldi	A,0b11111111		;port C: 7-0 out (out port) 
	out	DDRC,A
	ldi	A,0b00000000		;all bits low (drivers off)
	out	PORTC,A

	ldi	A,0b00010110		;port D: 7-5 in (jumpers)
	out	DDRD,A			;4,2,1 out (chg_ctrl, int0, txd), 0 in (rxd) 
	ldi	A,0b11101100		;inputs pulled up, chg_ctrl low, int0 high
	out	PORTD,A

; Init timer 0 for 1ms overflow interrupt

	ldi	A,1<<CS01|1<<CS00	;CT0 clocked at CK/64
	out	TCCR0,A
	ldi	A,141			;1ms overflow count
	out	TCNT0,A			;
	ldi	A,1<<TOIE0		;enable CT0 overflow interrupt
	out	TIMSK,A

	rcall	init_fifo		;init in_port fifo & state save
	rcall	init_comms		;init send/receive subsystem
	sei				;enable interrupts

;**************************************************************************
;*
;* Initialisation
;*
;**************************************************************************

; Erase eeprom if first power up or jumper installed

init:
	sbis	PIND,erase_jmp		;skip if 'erase' jumper not in
	rjmp	ini1			;else go do it

	ldi	ZL,low(eemagic)
	ldi	ZH,high(eemagic)
	rcall	ee_read
	cpi	A,$5A			;magic number?
	breq	ini3			;continue if found

ini1:	rcall	erase_eeprom		;wipe eeprom

	ldi	A,$F0
	out	PORTB,A			;light leds 1-4

ini2:	sbis	PIND,erase_jmp
	rjmp	ini2			;wait until jumper removed	

	ldi	B,low(1000)
	ldi	C,high(1000)
	rcall	sleep_msecs		;pause a moment...
	rjmp	reset			;then restart

; Disable password function if jumper installed

ini3:	sbic	PIND,pass_jmp		;skip if 'erase password' jumper in
	rjmp	ini4			
	
	rcall	get_eeflags
	sbrs	A,pwd_stat
	rjmp	ini4			;ignore if not enabled

	cbr	A,1<<pwd_stat		;disable password function
	rcall	put_eeflags			

; Enable watchdog (atmega sequence)

ini4:	wdr
	ldi	A,1<<WDTOE|1<<WDE
	out	WDTCR,A
	ldi	A,1<<WDE|1<<WDP2|1<<WDP1|1<<WDP0 ;max (1.9s) timeout
	out	WDTCR,A

; Restore output port state

	ldi	ZL,low(out_port)
	ldi	ZH,high(out_port)
	rcall	ee_read			;get previous output port state
	out	PORTC,A			;restore it

; Init charge control
	
ini5:	clr	A
	sts	pstat_array+state,A	;state 0
	clr	cflags			;related flags
	cbi	PORTD,chg_ctrl		;charger on

	sbi	PORTB,in_use_led	;turn off the 'in use' led
	cbi	PORTB,comms_err_led	;turn on the 'comms error'led

; Wait for phone power-on time & then initialise... 

	ldi	B,low(PHONE_POWER_ON*1000)
	ldi	C,high(PHONE_POWER_ON*1000)
	rjmp	ini7
	
ini6:	ldi	B,low(2000)		;two sec delay on retries
	ldi	C,high(2000)

ini7:	rcall	sleep_msecs		;sleep for phone power-on time
	rcall	init_fifo		;reinit in_port (caps charged)
	rcall	init_comms		;init send/receive subsystem

; Send 'get SMSC' frame

	rcall	build_get_smsc
	rcall	send_frame

	ldi	YL,low(rx_buff)
	ldi	YH,high(rx_buff)
	
; Wait for phone response (or timeout)...

	ldi	C,8			;for 2.048 sec timeout (+0/-0.25)
	add	C,sys_tmr_hi		;timeout tick

ini8:	rcall	wait_frame
	brts	ini9

; Frame received, check type...	

	ldd	A,Y+3
	cpi	A,$2			;message type = SMS?
	brne	ini8			;keep looking if not

	ldd	A,Y+9
	cpi	A,$34			;'SMSC' enclosed? ($35 if failed, +10=reason)	
	breq	ini10			;continue if it is

; SMSC not returned, check if phone powered off...

ini9:	rcall	get_pstatus		;get phone status
	sbrs	cflags,got_pstat
	rjmp	ini6

	cpi	A,4			;powered off?
	brne	ini6			;restart if not		

; Phone is powered off, press the power button!

	ldi	A,$0D			;'power' key code
	ldi	B,1			;press code
	rcall	build_press_key		;build the frame
	rcall	send_frame		;send it

; Phone enters power-up init state now, so we don't need
; to send a key release frame...

	rjmp	ini5			;go restart

; Save SMSC address for use later

ini10:	rcall	save_smsc		;save SMSC address

; Check if this is a restart due to delete failure...

	sbic	PORTB,del_err_led	;skip if delete sms failure
	rjmp	ini11			;else go complete init
	
	rcall	delete_sms		;try the delete again
	brts	ini6			;restart if failed
	
	sbi	PORTB,del_err_led	;turn off the 'delete error' led

ini11:	rcall	get_eeflags
	sbrc	A,log_stat		;skip if user not logged in
	cbi	PORTB,in_use_led	;'in use' led on
	sbi	PORTB,serv_err_led	;'no service' led off
	sbi	PORTB,comms_err_led	;'comms error' led off
	rjmp	mn4


;**************************************************************************
;*
;* Foreground processing loop
;*
;*  Detects & displays network status
;*  Controls charger
;*  Polls SIM memory for received messages
;*  Deletes messages after processing
;*  Sends user acknowledge messages
;*  Reports input port state changes
;*
;**************************************************************************
	
main:	rcall	get_pstatus		;get phone status
	rcall	charge_control		;battery charge control

; Wait about one second before looking for the next message...

	ldi	C,4			;for 1.024 sec timeout (+0/-0.25)
	add	C,sys_tmr_hi		;timeout tick
mn0:	rcall	wait_frame
	brtc	mn0			;ignore unsolicited frames

; Format & send 'get sms' frame

	rcall	build_get_sms		;build the 'get sms' request message
	rcall	send_frame		;send it

	ldi	YH,high(rx_buff)	;pointer to receive buffer
	ldi	YL,low(rx_buff)

; Wait for phone response (or timeout)...

	ldi	C,8			;for 2.048 sec timeout (+0/-0.25)
	add	C,sys_tmr_hi		;timeout tick
mn1:	rcall	wait_frame
	brts	mn7			;abort if error/timeout

; Frame received, look for expected response 
 
	ldd	A,Y+3
	cpi	A,$14			;msg_type = SMS function?
	brne	mn1			;go try again if not

	ldd	A,Y+9
	cpi	A,8			;read SMS successful?
	breq	mn2			;go if it was

	cpi	A,9			;read failed?
	brne	mn1			;loop to keep looking if not

; Empty memory location, do in_port stuff instead...

	rcall	in_port_handler		;test & respond on in_port change
	rjmp	mn4			;go reset to first SIM location

; SMS message successfully retrieved...

mn2:	clr	stat_cnt		;reset message status/error
	rcall	get_msg			;decode message string & save
	brts	mn3			;just delete if duplicate or other error

	rcall	cmd_handler		;interpret & execute command
mn3:	rcall	delete_sms		;delete message from phone
	brts	mn6			;abort if delete failed

	rcall	resp_handler		;send user response

	mov	A,sms_mem_loc
	inc	A			;bump to next location
	cpi	A,5
	brlo	mn5			;scan locations 1 to 4
	
mn4:	ldi	A,1			;reset to first location
mn5:	mov	sms_mem_loc,A	
	rjmp	main			;loop

mn6:	cbi	PORTB,del_err_led	;turn on 'delete error' led
	rjmp	mn8

mn7:	sbrc	aflags,rx_too_big	;skip if no buffer overflow
	rjmp	mn3			;...else attempt to delete the message
mn8:	rjmp	ini5			;enter init state


;**************************************************************************
;*
;* Get & save phone status
;*
;**************************************************************************

get_pstatus:
	rcall	build_get_stat		;build 'get status' frame
	rcall	send_frame		;send it

	ldi	YH,high(rx_buff)	;Y points to receive buffer
	ldi	YL,low(rx_buff)

	ldi	C,4			;one second (+0/-0.25) timeout
	add	C,sys_tmr_hi
gst1:	rcall	wait_frame
	brts	gst3			;abort if timeout

	ldd	A,Y+3
	cpi	A,4			;msg_type = phone status?
	brne	gst1

; Save phone status

	ldi	XH,high(rx_buff+10)
	ldi	XL,low(rx_buff+10)
	ldi	ZH,high(pstat_array+1)
	ldi	ZL,low(pstat_array+1)
	ldi	lcnt,5
gst2:	ld	A,X+
	st	Z+,A
	dec	lcnt
	brne	gst2

	sbr	cflags,1<<got_pstat
	ldd	A,Y+10			;return 'mode' byte
	clt				;no error
gst3:	ret


;**************************************************************************
;*
;* Charge control state machine
;*
;**************************************************************************

charge_control:
	ldi	YH,high(pstat_array)	;pointer to status array
	ldi	YL,low(pstat_array)
	
	ldi	ZH,high(cc_jmp_tbl)
	ldi	ZL,low(cc_jmp_tbl)

	ldd	A,Y+state
	cpi	A,10
	brlo	cct1
	rjmp	reset			;crash trap	

cct1:	add	ZL,A
	clr	A
	adc	ZH,A
	ijmp

cc_jmp_tbl:
	rjmp	s00
	rjmp	s10
	rjmp	s20
	rjmp	s30
	rjmp	s40
	rjmp	s50
	rjmp	s60
	rjmp	s70
	rjmp	s80
	rjmp	s90

;**************************************************************************
;*
;* State 0: initialise
;*
;**************************************************************************

s00:	sbrc	cflags,got_pstat
	rjmp	s01

; Load 3210/3310 fixed charge time value

	ldi	ZL,low(eecharge)
	ldi	ZH,high(eecharge)		
	rcall	ee_read			;battery charge time
	sts	pstat_array+utemp,A	;...to status array
	ldi	A,6
	rjmp	s02

; Load 5110/6110 discharge level value

s01:	ldi	ZL,low(eelevel)
	ldi	ZH,high(eelevel)		
	rcall	ee_read			;battery discharge level
	sts	pstat_array+utemp,A	;...to status array
	cpi	A,4
	breq	s02			;level 4 (charger always on)
	
	ldi	A,1
s02:	std	Y+state,A
	cbi	PORTD,chg_ctrl		;charger on
	ret

;**************************************************************************
;*
;* States 1->3: charge to level 4 with 'top-up' period,
;* then discharge to the programmed level
;*
;**************************************************************************

; Charge until battery level = 4

s10:	ldd	A,Y+blevel
	cpi	A,4
	brlo	s11

	cbr	cflags,1<<time_sync|1<<time_out
	ldi	A,low(TOP_UP_TIME)
	sts	chg_tmr_lo,A
	ldi	A,high(TOP_UP_TIME)
	sts	chg_tmr_hi,A

	ldi	A,2
	std	Y+state,A
s11:	ret	

; Top-up charge (unless over-volt/temp cut-out)

s20:	sbrc	cflags,time_out
	rjmp	s21			;time is up...

	ldd	A,Y+psource		;power source indicator
	cpi	A,1			;DC input? (2 = battery)
	breq	s22
	
s21:	sbi	PORTD,chg_ctrl		;charger off
	ldi	A,3
	std	Y+state,A
s22:	ret

; Discharge until battery level <= programmed discharge level

s30:	ldd	A,Y+blevel
	ldd	B,Y+utemp
	cp	B,A
	brlo	s31

	ldi	A,0
	std	Y+state,A
s31:	ret	

;**************************************************************************
;*
;* States 4->5: permanent charge (discharge level = 4)
;*
;**************************************************************************

; Monitor for over-volt/temp

s40:	ldd	A,Y+psource		;power source indicator
	cpi	A,1			;DC input? (2 = battery)
	breq	s41

	sbi	PORTD,chg_ctrl		;charger off
	ldi	A,5
	std	Y+state,A
s41:	ret

; Over volt/temp occurred, discharge until battery level < 4

s50:	ldd	A,Y+blevel
	cpi	A,4
	breq	s51

	ldi	A,0
	std	Y+state,A
s51:	ret	

;**************************************************************************
;*
;* States 6->9: fixed period charge/discharge
;*
;**************************************************************************
 
; Load charge timer

s60:	cbr	cflags,1<<time_sync|1<<time_out
	ldd	A,Y+utemp
	sts	chg_tmr_lo,A
	clr	A
	sts	chg_tmr_hi,A
	
	ldi	A,7
	std	Y+state,A
	ret

; Wait for charge period

s70:	sbrs	cflags,time_out
	rjmp	s71

	ldi	A,8
	std	Y+state,A
s71:	ret

; Load discharge timer

s80:	cbr	cflags,1<<time_sync|1<<time_out
	ldi	A,low(DISCHARGE_TIME)
	sts	chg_tmr_lo,A
	ldi	A,high(DISCHARGE_TIME)
	sts	chg_tmr_hi,A
	sbi	PORTD,chg_ctrl		;charger off
	
	ldi	A,9
	std	Y+state,A
	ret

; Wait for discharge period

s90:	sbrs	cflags,time_out
	rjmp	s91

	ldi	A,0
	std	Y+state,A
s91:	ret


;**************************************************************************
;*
;* Delete a message from SIM memory
;*
;* Entry: {sms_mem_loc} = location
;*
;**************************************************************************

delete_sms:
	sbrc	aflags,tx_buff_lok
	rjmp	delete_sms		;wait if transmit buffer not empty

	rcall	build_del_sms		;create a 'delete SMS' message frame
	rcall	send_frame		;send it

	ldi	YL,low(rx_buff)		;receive buffer pointer
	ldi	YH,high(rx_buff)

	ldi	C,20			;for 5.12 sec timeout (+0/-0.25)
	add	C,sys_tmr_hi		;timeout tick
del1:	rcall	wait_frame
	brts	del2			;abort if timeout

	ldd	A,Y+3
	cpi	A,$14			;'SMS function' message?
	brne	del1
	
	ldd	A,Y+9
	cpi	A,$0B			;'SMS delete OK' message?
	brne	del1
	
	clt				;success
	rjmp	del3

del2:	set				;failure
del3:	ret


;**************************************************************************
;*
;* Check & notify on input port bit change
;*
;* Entry: {s_buff}   = fifo buffer of changed bits (see tmr0_irq)
;*        {in_mask}  = pattern defining notify bits
;*        {bit_temp} = last bit pattern processed
;*
;**************************************************************************

in_port_handler:
	rcall	get_eeflags
	sbrs	A,log_stat		;continue if user logged in
	rjmp	iph7			;else skip notification

	lds	B,bit_temp
	tst	B			;more bit(s) to do from last time?
	brne	iph3
	
iph1:	cli				;tmr0_irq modifies {s_buff_len} too...
	lds	A,s_buff_len
	dec	A
	brmi	iph7			;fifo empty, exit

	sts	s_buff_len,A
	lds	YL,s_buff_tptr		;get fifo tail pointer
	lds	YH,s_buff_tptr+1
	ld	B,Y+			;pop a byte
	sei
	
	ldi	A,low(s_buff+16)	;physical buffer end?
	cp	YL,A
	ldi	A,high(s_buff+16)
	cpc	YH,A
	brcs	iph2

	ldi	YL,low(s_buff)		;wrap pointer
	ldi	YH,high(s_buff)

iph2:	sts	s_buff_tptr,YL		;save fifo tail pointer
	sts	s_buff_tptr+1,YH

iph3:	ldi	ZL,low(in_mask)
	ldi	ZH,high(in_mask)
	rcall	ee_read			;get notify mask

	and	B,A
	brne	iph4			;continue if notify on these bit(s)
		
	sts	bit_temp,B		;else clear all
	rjmp	iph1			;and get a fresh one

iph4:	sbis	PIND,prog_jmp		;continue if not in program mode
	rjmp	iph7			;else skip remaining (no notification) 

	clr	A			;bit mask
	clr	D			;index number
	sec
iph5:	rol	A
	mov	C,B
	and	C,A
	brne	iph6			;exit when bit match
	inc	D
	rjmp	iph5	

iph6:	ser	C
	eor	A,C			;invert mask

	and	B,A			;turn off this bit
	sts	bit_temp,B		;save for next time

	lsl	D			;index (bit #) * 16
	lsl	D
	lsl	D
	lsl	D
	ldi	ZH,high(in_tbl)		;user 'IN' strings start here
	ldi	ZL,low(in_tbl)	
	add	ZL,D			;add index
	clr	D
	adc	ZH,D

	ldi	YL,low(str_buff)
	ldi	YH,high(str_buff)
	rcall	ee_read			;check for undefined string
	tst	A
	breq	iph7			;abort if zero length

	ldi	lcnt,16			;assume max. length
	rcall	mov_ee_ram		;copy notification string to buffer
	
	clr	A
	st	Y,A			;need this if 16 bytes long

	rcall	send_msg		;send the message
iph7:	sei				;(see iph1)
	ret


;**************************************************************************
;*
;* Send user acknowledge or error response
;*
;* Entry: {stat_cnt} = cmd_handler status
;*
;**************************************************************************

resp_handler:
	cpi	stat_cnt,4
	breq	rsh6			;no response (local or auth error)

	cpi	stat_cnt,3		;special case for 'STAT' & 'COUNT'...
	breq	rsh4			;go send string in {str_buff}

	rcall	get_eeflags
	sbrs	A,user_ack		;continue if user acknowledge enabled
	rjmp	rsh5			;else quit
		
	ldi	YL,low(str_buff)	;string destination
	ldi	YH,high(str_buff)

	cpi	stat_cnt,2		;'bad cmd'	
	breq	rsh2
	
	cpi	stat_cnt,1
	breq	rsh1			;'bad pass'

	ldi	ZL,low(smsg1*2)		;'OK'
	ldi	ZH,high(smsg1*2)
	rjmp	rsh3
	
rsh1:	ldi	ZL,low(smsg2*2)
	ldi	ZH,high(smsg2*2)
	rjmp	rsh3

rsh2:	ldi	ZL,low(smsg3*2)
	ldi	ZH,high(smsg3*2)

rsh3:	rcall	mov_str_ram		;move message to str_buff
rsh4:	rcall	send_msg		;send it

; Logout now if pending...

rsh5:	sbrs	cflags,log_me_out	;continue if logout pending
	rjmp	rsh6

	rcall	get_eeflags
	cbr	A,1<<log_stat		;do it
	rcall	put_eeflags		;update flags
	cbr	cflags,1<<log_me_out
	sbi	PORTB,in_use_led	;'in use' led off
rsh6:	ret


;**************************************************************************
;*
;* Send SMS
;*
;* Entry: {str_buff} = string to send
;*
;**************************************************************************

send_msg:
	rcall	get_eeflags
	sbrs	A,log_stat		;continue if user logged in
	rjmp	smg8			;else no action

; This hack turns on charger if battery level < 3

	lds	A,pstat_array+state	;get charge control state 
	cpi	A,4
	brsh	smg1			;permanent or fixed time charge

	lds	A,pstat_array+blevel	;get current battery level
	cpi	A,3
	brsh	smg1			;skip if 3->4

	cbi	PORTD,chg_ctrl		;charger on if <3
	ldi	A,0
	sts	pstat_array+state,A	;force charge control to init state

smg1:	ldi	stat_cnt,SMS_RETRY_COUNT ;max. retries
smg2:	sbrc	aflags,tx_buff_lok
	rjmp	smg2			;don't scramble the buffer!

	rcall	build_send_sms		;create message frame
	rcall	send_frame		;send it

	ldi	YH,high(rx_buff)	;pointer to receive buffer
	ldi	YL,low(rx_buff)

; Wait for phone response (or timeout)...

	ldi	C,high(SMS_SEND_TIMEOUT*1000) ;timeout
	add	C,sys_tmr_hi		;timeout tick
smg3:	rcall	wait_frame
	brts	smg6			;abort with error if timeout

; Frame received, look for expected response 
 
smg4:	ldd	A,Y+3
	cpi	A,2			;msg_type = SMS handling?
	brne	smg3			;keep looking if not

	ldd	A,Y+9
	cpi	A,2			;message sent?
	breq	smg7			;continue if it was

	cpi	A,3			;send failed?
	brne	smg3			;keep looking if not

; Send failed, discover reason...

	ldd	A,Y+11
	cpi	A,1			;normally 01 ..,
	brne	smg6

	ldd	A,Y+12
	cpi	A,$32			;reason = 'check operator services'?
	breq	smg5

; Retry on error code $6F only...

	cpi	A,$6F			;reason = rf section?
	brne	smg6

	dec	stat_cnt		;retry count
	breq	smg6			;exit when exhausted
	
	ldi	B,low(1000)
	ldi	C,high(1000)
	rcall	sleep_msecs		;wait one second longer...
	rjmp	smg2			;and go try again

smg5:	sbr	cflags,1<<serv_err	;account disabled...
	cbi	PORTB,serv_err_led	;light 'no service' led	

smg6:	cbi	PORTB,send_err_led	;turn on 'send error' led
	set				;error flag
	rjmp	smg9			;quit

smg7:	ldi	ZL,low(tx_count)
	ldi	ZH,high(tx_count)
	rcall	update_sms_cntr		;update sms sent counter
smg8:	clt				;no error
smg9:	ret


;**************************************************************************
;*
;* Wait for frame received
;*
;* Also looks for unsolicited 'network registration' frames and extracts
;* netstatus. If not on network, lights 'no service' led.
;*
;* Entry: C = timeout tick (1/4 sec resolution)
;*
;**************************************************************************

wait_frame:
	cbr	aflags,1<<rx_buff_lok|1<<rx_too_big|1<<tx_fatal_err
	
wfm1:	sbrs	aflags,rx_buff_lok	;skip if frame received
	rjmp	wfm4			;else go check for timeout

	ldd	A,Y+3
	cpi	A,$0A			;msg_type = net reg?
	brne	wfm3			;exit if not

	ldd	A,Y+14			;get netstatus byte
	cpi	A,2
	brlo	wfm2
	
	cbi	PORTB,serv_err_led	;no service, light error led
	rjmp	wait_frame
	
wfm2:	sbrs	cflags,serv_err
	sbi	PORTB,serv_err_led	;on network, turn off led
	rjmp	wait_frame

wfm3:	clt
	rjmp	wfm6			;exit

wfm4:	sbrc	aflags,tx_fatal_err
	rjmp	wfm5			;no response to transmit, abort

	cp	C,sys_tmr_hi
	breq	wfm5			;keep trying until timeout

	sbrs	aflags,tick_flag	;skip on 0.25 sec mark
	rjmp	wfm1
	
	wdr				;watchdog keep-alive
	cbr	aflags,1<<tick_flag
	rjmp	wfm1

wfm5:	set				;timeout/error flag
wfm6:	wdr
	ret	


;**************************************************************************
;*
;* Raise send message request & wait for pickup
;*
;**************************************************************************

send_frame:
	sbr	aflags,1<<tx_msg_req	;raise send request
sms1:	sbrs	aflags,tx_msg_req	;wait for pickup
	rjmp	sms2

	sbrs	aflags,tick_flag	;skip on 0.25 sec mark
	rjmp	sms1
	
	wdr				;watchdog keep-alive
	cbr	aflags,1<<tick_flag
	rjmp	sms1

sms2:	wdr
	ret


;**************************************************************************
;*
;* Extract message string from receive buffer & decode
;*
;**************************************************************************

get_msg:
	ldi	YH,high(rx_buff)	;pointer to receive buffer
	ldi	YL,low(rx_buff)
	ldd	A,Y+5			;sanity checks...
	cpi	A,46			;min frame length is 46 bytes...
	brlo	gem4			;abort if less

	ldd	A,Y+12			;requested location?
	cp	A,sms_mem_loc
	brne	gem4			;abort if not
	
; Compare this timestamp with previous timestamp. If they match, the
; previous message obviously wasn't deleted by the phone (unlikely,
; but possible).

	ldi	ZL,low(time_stamp)	;base pointer 
	ldi	ZH,high(time_stamp)

	dec	A			;zero base
	mov	B,A
	lsl	A			;*6 to get offset
	lsl	A
	add	A,B
	add	A,B
	add	ZL,A			;Y points to previous timestamp
	clr	A
	adc	ZH,A
	
	mov	XH,YH		
	mov	XL,YL
	ldi	A,42			;offset to message TS field
	add	XL,A
	clr	A
	adc	XH,A
	
	ldi	lcnt,6			;length (minus timezone)
	set				;becomes match flag
gem1:	ld	A,X+			
	ld	B,Z
	cp	A,B
	breq	gem2
	clt				;clear on any mismatch	
gem2:	st	Z+,A
	dec	lcnt
	brne	gem1
	brts	gem4			;abort if match - same message!				

; Unpack & save message...
	
	ldd	C,Y+29			;get (unpacked) message length
	tst	C			;sanity check...
	breq	gem4			;abort if zero length
	
	cpi	C,34+1
	brlo	gem3
	ldi	C,34			;max. allowed length minus terminator
	
gem3:	ldi	A,49			;offset to packed message
	add	YL,A
	clr	A
	adc	YH,A

	ldi	ZL,low(str_buff)	;put unpacked message here
	ldi	ZH,high(str_buff)

	rcall	decode_str		;decode & save
	clr	A
	st	Z,A			;null terminate
	
	ldi	ZL,low(rx_count)
	ldi	ZH,high(rx_count)
	rcall	update_sms_cntr		;update sms received counter
	clt
	rjmp	gem5
	
gem4:	ldi	stat_cnt,4		;don't ack this msg!
	set
gem5:	ret


;**************************************************************************
;*
;* Decode inbound message and execute appropriate handler
;*
;**************************************************************************	

cmd_handler:
	ldi	YH,high(str_buff)	;pointer to received message string
	ldi	YL,low(str_buff)
	rcall	compr_sys		;next part 'system' string?
	brtc	cmh1			;assume 'user' string if not

	cpi	C,2			;'in' or 'out' command?
	brsh	cmh2			;go if not

	sbis	PIND,prog_jmp		;skip if not in program mode	
	rjmp	cmh2
	
	ldi	YH,high(str_buff)	;reset to start of message
	ldi	YL,low(str_buff)
cmh1:	rcall	auth_sender		;authenticate sender
	rjmp	usr_str_cmd		;handle as 'user' string

cmh2:	cpi	C,7			;'login' command?
	breq	cmh3			;skip checks if so
	rcall	auth_sender		;else authenticate sender
cmh3:	ldi	ZH,high(sys_jmp_tbl)
	ldi	ZL,low(sys_jmp_tbl)
	add	ZL,C
	clr	A
	adc	ZH,A
	mov	D,C			;copy string index
	ijmp				;execute command handler

sys_jmp_tbl:
	rjmp	sys_in_cmd
	rjmp	sys_out_cmd
	rjmp	sys_charge_cmd
	rjmp	sys_pass_cmd
	rjmp	sys_ack_cmd
	rjmp	sys_ndis_cmd
	rjmp	sys_ndis_cmd
	rjmp	sys_login_cmd
	rjmp	sys_logout_cmd
	rjmp	sys_stat_cmd
	rjmp	sys_count_cmd


;**************************************************************************
;*
;* {string}
;*
;**************************************************************************

usr_str_cmd:
	ldi	ZH,high(out_tbl)	;start of 'OUT' string store
	ldi	ZL,low(out_tbl)
	ldi	C,16			;max. entries to scan
	rcall	compr_usr		;look for match
	brts	usc1			;go if found
	
	ldi	stat_cnt,2		;'bad cmd'
	rjmp	usc6			;abort if none found

usc1:	mov	lcnt,C			;index
	andi	lcnt,7			;remove low/high offset
	inc	lcnt			;for at least one rotate

	clr	B
	sec				;move the bit...
usc2:	rol	B			;...into position
	dec	lcnt
	brne	usc2

	ldi	ZH,high(puls_bits)
	ldi	ZL,low(puls_bits)
	rcall	ee_read			;get pulse bits
	and	A,B			;this bit pulse bit?
	breq	usc3			;go if not

	in	B,PORTC			;get current port state
	mov	D,B
	or	B,A
	out	PORTC,B			;turn on the bit	

	ldi	B,low(1000)
	ldi	C,high(1000)
	rcall	sleep_msecs		;one second delay

	out	PORTC,D			;turn off the bit
	rjmp	usc6			;done

usc3:	in	A,PORTC			;get current port state
	cpi	C,8
	brsh	usc4			;clear bit command (driver inverts)

	or	A,B			;set the bit
	rjmp	usc5

usc4:	ser	C
	eor	B,C			;invert mask
	and	A,B			;clear the bit

usc5:	rcall	write_out_port		;update port & save in EEPROM
usc6:	ret


;**************************************************************************
;*
;* IN{n,L|n,H}{string}
;*
;**************************************************************************

sys_in_cmd:
	rcall	next_char		;get input bit # (1 - 4)
	breq	sic3			;abort if buffer end

	subi	B,$30			;make binary
	cpi	B,5			;limit check
	brsh	sic3			;abort if invalid port input number

	mov	C,B
	rcall	next_char
	breq	sic3			;abort if unexpected buffer end

	ld	A,Y
	tst	A
	breq	sic3			;abort if zero string length
	
	andi	B,$5F			;force upper case
	cpi	B,'L'
	breq	sic1
	
	cpi	B,'H'		
	brne	sic3			;abort if neither

	subi	C,-4			;make index

; Create bit mask for state change notification

sic1:	mov	lcnt,C
	clr	B
	sec
sic2:	rol	B
	dec	lcnt
	brne	sic2

	ldi	ZH,high(in_mask)
	ldi	ZL,low(in_mask)	
	rcall	ee_read			;get current mask
	or	A,B			;update it
	rcall	ee_write

	dec	C			;adjust
	lsl	C			;*16
	lsl	C
	lsl	C
	lsl	C
	ldi	ZH,high(in_tbl)		;user 'IN' strings start here
	ldi	ZL,low(in_tbl)	
	add	ZL,C			;add index
	clr	C
	adc	ZH,C
	ldi	C,16			;max. string length
	rcall	write_ee_str		;save user string in EEPROM
	rjmp	sic4

sic3:	ldi	stat_cnt,2		;'bad cmd'
sic4:	ret


;**************************************************************************
;*
;* OUT{nL|nH}{string}
;*
;**************************************************************************

sys_out_cmd:
	rcall	next_char		;get output bit # (1 - 8)
	breq	soc1			;abort if buffer end

	subi	B,$30			;make binary
	cpi	B,9			;limit check
	brsh	soc1			;abort if invalid output number

	mov	C,B
	mov	lcnt,B
	
	rcall	next_char
	breq	soc1			;abort if unexpected buffer end

	ld	A,Y
	tst	A
	breq	soc1			;abort if zero {string} length
	
	andi	B,$5F			;force upper case
	cpi	B,'L'
	breq	soc3
	
	cpi	B,'H'		
	breq	soc2
	
	cpi	B,'P'
	breq	soc3
	
soc1:	ldi	stat_cnt,2		;'bad cmd'
	rjmp	soc6

soc2:	subi	C,-8			;make index
soc3:	dec	C			;adjust
	lsl	C			;*16
	lsl	C
	lsl	C
	lsl	C
	ldi	ZH,high(out_tbl)	;user 'OUT' strings start here
	ldi	ZL,low(out_tbl)	
	add	ZL,C			;add index
	clr	C
	adc	ZH,C
	ldi	C,16			;max. string length (truncates if longer)
	rcall	write_ee_str		;save user string in EEPROM

	clr	C
	sec
soc4:	rol	C			;C = bit mask		
	dec	lcnt
	brne	soc4

	ldi	ZH,high(puls_bits)
	ldi	ZL,low(puls_bits)
	rcall	ee_read			;A = pulse bits

	cpi	B,'P'			;'pulsed' mode requested?
	brne	soc5			;go disable if not
		
	or	A,C			;else enable for this bit
	rcall	ee_write		;update eeprom

	ser	A
	eor	C,A			;invert mask

	in	A,PORTC			;get current port state
	and	A,C			;output bit off (high) initially
	rcall	write_out_port		;update port
	rjmp	soc6			;done
	
soc5:	ser	B
	eor	C,B			;invert mask
	and	A,C			;turn off the bit
	rcall	ee_write		;update eeprom
soc6:	ret


;**************************************************************************
;*
;* CHARGE{n}
;*
;**************************************************************************

sys_charge_cmd:
	rcall	convert_dec_str		;convert {n} to hex
	brts	scd2			;conversion error

	tst	D
	brne	scd2			;number too big
	
	mov	A,C
	sbrs	cflags,got_pstat
	rjmp	scd1

; 5110/6110 attached, level 0->4 allowed
 
	cpi	A,5
	brsh	scd2
		
	ldi	ZL,low(eelevel)
	ldi	ZH,high(eelevel)		
	rcall	ee_write		;save new level in eeprom
	rjmp	scd3			;done

; 3210/3310 attached, 10 - 240 (minutes) allowed

scd1:	cpi	A,10
	brlo	scd2			;<10 minutes

	cpi	A,241
	brsh	scd2			;>240 minutes

	ldi	ZL,low(eecharge)
	ldi	ZH,high(eecharge)		
	rcall	ee_write		;save new charge time in eeprom
	rjmp	scd3

scd2:	ldi	stat_cnt,2		;'bad cmd'
scd3:	ret


;**************************************************************************
;*
;* Get & convert decimal string to hex
;*
;* Entry: Y points to start of decimal string
;* Exit : D,C = hex result
;*
;**************************************************************************

convert_dec_str:
	clr	lcnt
cds1:	ld	A,Y+			;find string length
	tst	A
	breq	cds2
	inc	lcnt
	rjmp	cds1

cds2:	tst	lcnt			;zero length?
	breq	cds3

	cpi	lcnt,5			;>4 characters?
	brlo	cds4
	
cds3:	set				;flag error
	rjmp	cds8			;and exit

cds4:	ldi	ZH,high((dec_con+4)*2)	;decimal conversion constants
	ldi	ZL,low((dec_con+4)*2)

	clr	C			;accumulator low byte
	clr	D			;            high byte
	ld	A,-Y			;dummy (back up)
		
cds5:	ld	A,-Y
	subi	A,'0'
	brlo	cds3			;<'0'
	
	cpi	A,10
	brsh	cds3			;>'9'

	lpm				;get constant in B,R0
	mov	B,R0
	adiw	ZL,1
	lpm

	tst	A			;zero case
	breq	cds7
	
cds6:	add	C,R0			;add constant recursively
	adc	D,B
	dec	A
	brne	cds6

cds7:	ldi	A,3			;back up in constant list
	sub	ZL,A
	clr	A
	sbc	ZH,A

	dec	lcnt			;digits to do
	brne	cds5

	clt
cds8:	ret


;**************************************************************************
;*
;* PASS{string}
;*
;**************************************************************************

sys_pass_cmd:
	push	YH
	push	YL

; Determine password string length...

	clr	C
sps1:	ld	A,Y+
	tst	A
	breq	sps2
	inc	C
	rjmp	sps1

sps2:	pop	YL
	pop	YH

	tst	C			;zero length password?
	breq	sps3

	cpi	C,9			;>8 characters?
	brlo	sps4
	
sps3:	ldi	stat_cnt,1		;'bad pass' error
	rjmp	sps5

; Save new password in EEPROM

sps4:	ldi	ZH,high(password)
	ldi	ZL,low(password)	
	rcall	write_ee_str		;save the new password
	rcall	get_eeflags
	sbr	A,1<<pwd_stat		;flag password defined
	rcall	put_eeflags
sps5:	ret


;**************************************************************************
;*
;* ACK{ON|OFF}
;*
;**************************************************************************

sys_ack_cmd:
	rcall	next_char		;get first char of argv
	breq	sac2			;abort if buffer end
	
	andi	B,$5F			;force upper case
	cpi	B,'O'
	brne	sac2			;abort if not

	rcall	get_eeflags		;get flags in A
	rcall	next_char		;get next char of argv
	breq	sac2			;abort if buffer end

	andi	B,$5F			;force upper case
	cpi	B,'N'
	brne	sac1
	
	sbr	A,1<<user_ack		;enable user command acks
	rjmp	sac4
	
sac1:	cpi	B,'F'
	breq	sac3
	
sac2:	ldi	stat_cnt,2		;'bad cmd'	
	rjmp	sac5

sac3:	cbr	A,1<<user_ack		;disable acks
sac4:	rcall	put_eeflags		;update in EEPROM
sac5:	ret


;**************************************************************************
;*
;* EN{string}
;* DIS{string}
;*
;**************************************************************************

sys_ndis_cmd:
	ldi	ZH,high(in_tbl)		;start of 'IN' user string store
	ldi	ZL,low(in_tbl)
	ldi	C,8			;max. entries to scan
	rcall	compr_usr		;look for match
	brts	sec1			;go if found

	ldi	stat_cnt,2		;'bad cmd'
	rjmp	sec5

sec1:	inc	C			;bump index
	clr	B
	sec
sec2:	rol	B			;shift the enabling bit
	dec	C
	brne	sec2

	ldi	ZH,high(in_mask)
	ldi	ZL,low(in_mask)	
	rcall	ee_read			;get mask in A

	ser	C
	eor	C,B			;invert mask

	cpi	D,5			;'ENable' command?
	breq	sec3

	and	A,C			;turn off specified bit
	rjmp	sec4

sec3:	or	A,B			;turn on specified bit
sec4:	rcall	ee_write		;update in_mask
sec5:	ret


;**************************************************************************
;*
;* LOGIN{password}
;*
;**************************************************************************

sys_login_cmd:
	sbis	PIND,prog_jmp		;continue if not in program mode
	rjmp	slc2			;else skip password check

; Verify password if set...

	rcall	get_eeflags
	sbrs	A,pwd_stat		;continue if password set
	rjmp	slc2			;else skip check
	
	ldi	ZH,high(password)	;pointer to password string
	ldi	ZL,low(password)
	ldi	C,8			;max. length
	rcall	compr_estr		;check password
	brtc	slc1			;abort if mismatch

	rcall	next_char		;next char should be a null...
	breq	slc2			;continue if length match

slc1:	ldi	stat_cnt,1		;'bad pass' code
	rjmp	slc3			;abort

slc2:	rcall	save_addr		;save dest. address (phone number)
	brts	slc3			;skip login if bad address

	rcall	get_eeflags
	sbr	A,1<<log_stat		;flag logged in
	rcall	put_eeflags		;update flags

	rcall	init_fifo		;init in_port fifo & state save
	cbi	PORTB,in_use_led	;'in use' led on
slc3:	ret


;**************************************************************************
;*
;* LOGOUT
;*
;**************************************************************************

sys_logout_cmd:
	sbr	cflags,1<<log_me_out	;logout pending (don't do it yet!)
	ret


;**************************************************************************
;*
;* STAT
;*
;* Send a snapshop of input & output ports
;*
;* Output format: {IIII} {OOOOOOOO} (I=input bit, O=output bit, msb first)
;*
;**************************************************************************

sys_stat_cmd:
	ldi	YL,low(str_buff)	;string destination
	ldi	YH,high(str_buff)

	ldi	lcnt,4			;bits to do
	in	B,PINA			;get input port state
	swap	B			;to high nibble
	rcall	hilo_out		;convert to 'HL' string
	
	ldi	A,$20
	st	Y+,A

	ldi	lcnt,8
	in	B,PORTC			;get output port state 
	ldi	A,$FF
	eor	B,A			;invert
	rcall	hilo_out		;convert to 'HL' string
	
	clr	A
	st	Y,A			;terminate	
	ldi	stat_cnt,3		;to send string in {str_buff}
	ret

hilo_out:
	lsl	B			;push out a bit
	ldi	A,'L'
	brcc	hlo1			;go if low

	ldi	A,'H'
hlo1:	st	Y+,A
	dec	lcnt
	brne	hilo_out
	ret					
	

;**************************************************************************
;*
;* COUNT
;*
;* Send sms counters & firmware version
;*
;* Output format:
;* r={rx_cnt} s={tx_cnt} v={VERS}
;*
;**************************************************************************

sys_count_cmd:
	ldi	YL,low(str_buff)	;string destination
	ldi	YH,high(str_buff)
	
	ldi	A,'r'
	st	Y+,A
	ldi	A,'='
	st	Y+,A

	ldi	ZL,low(rx_count)	;sms counters in eeprom
	ldi	ZH,high(rx_count)
	rcall	ee_read			;high byte {rx_cnt}
	mov	B,A
	adiw	ZL,1
	rcall	ee_read			;low byte {rx_cnt}
	rcall	hex_dec			;convert to decimal

	ldi	A,$20
	st	Y+,A
	ldi	A,'s'
	st	Y+,A
	ldi	A,'='
	st	Y+,A

	ldi	ZL,low(tx_count)
	ldi	ZH,high(tx_count)
	rcall	ee_read			;high byte {tx_cnt}
	mov	B,A
	adiw	ZL,1
	rcall	ee_read			;low byte {tx_cnt}
	rcall	hex_dec

	ldi	A,$20
	st	Y+,A
	ldi	A,'v'
	st	Y+,A
	ldi	A,'='
	st	Y+,A

	ldi	A,VERSION_MAJOR
	st	Y+,A			;write version number
	ldi	A,'.'
	st	Y+,A
	ldi	A,VERSION_MINOR
	st	Y+,A

	clr	A
	st	Y,A			;terminate	
	ldi	stat_cnt,3		;to send string in {str_buff}
	ret


;**************************************************************************
;*
;* Convert byte to ascii
;*
;**************************************************************************

hex_asc:
	push	A
	swap	A
	rcall	hex1
	pop	A
hex1:	andi	A,$0F
	subi	A,$0A
	brcs	hex2
	subi	A,-7
hex2:	subi	A,-$3A
	st	Y+,A
	ret


;**************************************************************************
;*
;* Convert hex word to decimal
;*
;* Entry: B,A = hex word
;*        Y   = pointer to output
;*
;**************************************************************************

hex_dec:
	ldi	ZH,high(dec_con*2)	;decimal conversion constants
	ldi	ZL,low(dec_con*2)
	ldi	lcnt,5			;digits to do

hxd1:	lpm
	mov	C,R0
	adiw	ZL,1
	lpm
	adiw	ZL,1

	ldi	D,-1			;recursively subtract constant
hxd2:	sub	A,R0			
	sbc	B,C
	inc	D
	brcc	hxd2

	add	A,R0			;restore remainder
	adc	B,C
	
	subi	D,-$30			;bcd digit to ascii
	st	Y+,D			;and save in buffer

	dec	lcnt
	brne	hxd1
	ret


;**************************************************************************
;*
;* Update 16-bit sms send/receive counter
;*
;* Entry: Z = pointer to counter (in eeprom)
;*
;**************************************************************************

update_sms_cntr:
	push	ZH
	push	ZL

	adiw	ZL,1
	rcall	ee_read			;get counter low byte
	inc	A			;bump for this receive
	rcall	ee_write		;write it back

	pop	ZL
	pop	ZH

	tst	A			;carry?
	brne	udc1			;skip high byte update if not
	
	rcall	ee_read			;get counter high byte
	inc	A			;bump
	rcall	ee_write		;write it back
udc1:	ret


;**************************************************************************
;*
;* Initialise in_port fifo
;*
;**************************************************************************

init_fifo:
	in	B,SREG
	cli
	ldi	A,low(s_buff)		;init port state fifo pointers
	sts	s_buff_tptr,A
	sts	s_buff_hptr,A
	
	ldi	A,high(s_buff)
	sts	s_buff_tptr+1,A
	sts	s_buff_hptr+1,A

	clr	A
	sts	s_buff_len,A		;zero depth
	sts	bit_temp,A		;and multi-bit save

	in	A,PINA			;read input port
	andi	A,$0F			;mask off unwanted bits
	sts	in_port,A		;establish initial state
	cbr	aflags,1<<in_sample
	out	SREG,B
	ret


;**************************************************************************
;*
;* Initialise send/receive subsystem
;*
;**************************************************************************

init_comms:
	in	B,SREG
	cli
	ldi	A,BAUD_115200
	out	UBRR,A			;set uart bit rate

	ldi	A,1<<TXEN
	out	UCR,A			;uart tx enabled

	ldi	lcnt,250
	ldi	A,$55			;sync pattern
icm1:	sbis	USR,UDRE
	rjmp	icm1
	out	UDR,A			;send a byte
	dec	lcnt
	brne	icm1			;do 250 of 'em

icm2:	sbis	USR,TXC			;wait for last byte to go...
	rjmp	icm2
	
	in	A,UDR			;flush data register
	ldi	A,1<<RXCIE|1<<RXEN|1<<TXEN ;enable tx & rx + rx interrupt 
	out	UCR,A

	clr	iflags
	clr	aflags
	clr	tx_byte_cnt		;(see tx_irq)
	clr	tx_seq_no		;first packet will be #1

	clr	A
	sts	rx_fto_cnt,A		;zero comms error counters
	sts	rx_ato_cnt,A
	sts	rx_fce_cnt,A

	sbi	PORTD,PD2		;clear possible pending request	
	ldi	A,1<<INT0
	out	GIMSK,A			;enable INT0 interrupts
	out	SREG,B
	ret


;**************************************************************************
;*
;* Build SMS transmit frame
;*
;**************************************************************************

build_send_sms:
	ldi	YL,low(tx_buff)
	ldi	YH,high(tx_buff)

	push	YH			;copy pointer
	push	YL

	ldi	ZL,low(msg_hdr*2)
	ldi	ZH,high(msg_hdr*2)
	ldi	lcnt,12			;length
	rcall	mov_flash_ram		;insert first 12 bytes

	ldi	ZL,low(smsc_addr)
	ldi	ZH,high(smsc_addr)
	ldi	lcnt,12			;length

bsm1:	ld	A,Z+			;[+12] insert SMSC
	st	Y+,A
	dec	lcnt
	brne	bsm1

	ldi	A,$15			;[+24] first octet of TPDU
	st	Y+,A
	ldi	A,0
	st	Y+,A			;null
	st	Y+,A			;PID
	st	Y+,A			;DCS (text)			
	st	Y+,A			;unpacked message length
		
	ldi	ZL,low(dest_addr)
	ldi	ZH,high(dest_addr)
	ldi	lcnt,12			;length
	rcall	mov_ee_ram		;[+29] insert dest. address

	ldi	ZL,low(vp_field*2)
	ldi	ZH,high(vp_field*2)
	ldi	lcnt,7			;length
	rcall	mov_flash_ram		;[+41] insert VP

	ldi	ZL,low(str_buff)
	ldi	ZH,high(str_buff)
	rcall	encode_str		;pack & insert message string

	pop	ZL			;pointer to buffer start
	pop	ZH
	std	Z+28,D			;unpacked message length
	
	ldi	A,1			;frames to go
	st	Y+,A			
	
	rcall	next_seq_no
	st	Y+,A			;sequence number

	mov	A,YL
	sub	A,ZL			;find length thus far
	mov	C,A
	subi	A,6			;minus header length
	std	Z+5,A			;save as block length

	clr	B
	sbrc	A,0
	st	Y+,B			;pad to make even
	
	rcall	xor_buff		;calculate check word
	st	Y+,A			;append it
	st	Y+,B
	ret				;done


;**************************************************************************
;*
;* Build 'delete SMS' frame
;*
;**************************************************************************

build_del_sms:
	ldi	YL,low(tx_buff)		;transmit buffer
	ldi	YH,high(tx_buff)
	
	push	YH			;copy pointer
	push	YL

	ldi	ZL,low(del_sms_frm*2)
	ldi	ZH,high(del_sms_frm*2)
	ldi	lcnt,14			;length
	rcall	mov_flash_ram		;insert first 14 bytes

	pop	ZL
	pop	ZH

	std	Z+11,sms_mem_loc	;message location

	rcall	next_seq_no
	std	Z+13,A			;frame sequence number

	ldi	C,14
	rcall	xor_buff		;generate check word
	st	Z+,A			;append it
	st	Z+,B
	ret


;**************************************************************************
;*
;* Build 'get SMS message' (from SIM) frame
;*
;**************************************************************************

build_get_sms:
	ldi	YL,low(tx_buff)		;transmit buffer
	ldi	YH,high(tx_buff)

	push	YH			;copy pointer
	push	YL

	ldi	ZL,low(get_sms_frm*2)
	ldi	ZH,high(get_sms_frm*2)
	ldi	lcnt,16			;length
	rcall	mov_flash_ram		;insert first 14 bytes

	pop	ZL
	pop	ZH

	std	Z+11,sms_mem_loc	;message location

	rcall	next_seq_no
	std	Z+15,A			;frame sequence number

	ldi	C,16
	rcall	xor_buff		;generate check word
	st	Z+,A			;append it
	st	Z+,B
	ret


;**************************************************************************
;*
;* Build 'get SMSC' frame
;*
;**************************************************************************

build_get_smsc:
	ldi	YL,low(tx_buff)		;transmit buffer
	ldi	YH,high(tx_buff)
	
	push	YH			;copy pointer
	push	YL

	ldi	ZL,low(get_smsc_frm*2)
	ldi	ZH,high(get_smsc_frm*2)
	ldi	lcnt,14			;length
	rcall	mov_flash_ram		;insert first 12 bytes

	pop	ZL
	pop	ZH

	rcall	next_seq_no
	std	Z+13,A			;insert sequence number

	ldi	C,14
	rcall	xor_buff		;generate check word
	st	Z+,A			;append it
	st	Z+,B
	ret
	

;**************************************************************************
;*
;* Build 'get status' frame
;*
;**************************************************************************

build_get_stat:
	ldi	YL,low(tx_buff)		;transmit buffer
	ldi	YH,high(tx_buff)

	push	YH			;copy pointer
	push	YL

	ldi	ZL,low(get_stat_frm*2)
	ldi	ZH,high(get_stat_frm*2)
	ldi	lcnt,12			;length
	rcall	mov_flash_ram		;insert first 12 bytes

	pop	ZL
	pop	ZH

	rcall	next_seq_no
	std	Z+11,A			;insert sequence number

	ldi	C,12
	rcall	xor_buff		;generate check word
	st	Z+,A			;append it
	st	Z+,B
	ret	


;**************************************************************************
;*
;* Build 'press key' frame
;*
;* Entry: A = key code
;*        B = press (1) or release (2)
;*
;**************************************************************************

build_press_key:
	ldi	YL,low(tx_buff)		;transmit buffer
	ldi	YH,high(tx_buff)

	push	YH			;copy pointer
	push	YL

	ldi	ZL,low(press_key_frm*2)
	ldi	ZH,high(press_key_frm*2)
	ldi	lcnt,16			;length
	rcall	mov_flash_ram		;insert first 16 bytes

	pop	ZL
	pop	ZH

	std	Z+10,B			;press/release
	std	Z+11,A			;key code

	rcall	next_seq_no
	std	Z+14,A			;insert sequence number

	ldi	C,16
	rcall	xor_buff		;generate check word
	st	Z+,A			;append it
	st	Z+,B
	ret


;**************************************************************************
;*
;* Generate next transmit frame sequence number
;*
;**************************************************************************

next_seq_no:
	mov	A,tx_seq_no
	inc	A
	andi	A,7
	mov	tx_seq_no,A
	ori	A,$40
	ret


;**************************************************************************
;*
;* Save SMSC address (from 'get_smsc' command response)
;*
;* Garbage at the end of the address is removed.
;*
;* Entry: Y = pointer to start of receive buffer 
;*
;**************************************************************************

save_smsc:
	push	YH
	push	YL
	ldi	A,27			;point to SMSC address
	add	YL,A
	clr	A
	adc	YH,A

	ld	lcnt,Y			;get SMSC length (in octets)
	inc	lcnt			;include length byte

	ldi	ZL,low(smsc_addr)	;save here
	ldi	ZH,high(smsc_addr)

	ldi	C,12			;fixed field length
	sub	C,lcnt

smc1:	ld	A,Y+			;copy it
	st	Z+,A
	dec	lcnt
	brne	smc1
	
	tst	C
	breq	smc3

	clr	A
smc2:	st	Z+,A
	dec	C
	brne	smc2

smc3:	pop	YL
	pop	YH
	ret


;**************************************************************************
;*
;* Save destination addresses (from SMS message frame) to EEPROM
;*
;* Garbage at the end of the address is removed.
;*
;**************************************************************************

save_addr:
	ldi	YL,low(rx_buff)
	ldi	YH,high(rx_buff)
	ldi	A,30			;point to destination address
	add	YL,A
	clr	A
	adc	YH,A

; Do some basic checks to detect an invalid number

	ld	lcnt,Y			;get length (in semi-octets)
	cpi	lcnt,10			;must be >10 digits...
	brlo	sad1			;not enough, quit

	cpi	lcnt,21			;must be 10-20 digits...
	brsh	sad1			;too many, quit

	ldd	A,Y+1			;get type_of_address octet
	cpi	A,$81			;unknown / telephone
	breq	sad2
	
	cpi	A,$91			;international / telephone
	breq	sad2

	cpi	A,$A1			;national / telephone
	breq	sad2

sad1:	set				;failed checks
	rjmp	sad5			;quit

sad2:	inc	lcnt			;make even
	lsr	lcnt			;convert to byte count
	inc	lcnt			;plus length & number type
	inc	lcnt
	
	ldi	C,12			;fixed field length
	sub	C,lcnt			;find any unused space
	ldi	ZL,low(dest_addr)	;save here
	ldi	ZH,high(dest_addr)
	rcall	mov_ram_ee		;write address field to EEPROM

	tst	C
	breq	sad4

	clr	A
sad3:	rcall	ee_write		;zero remaining space
	adiw	ZL,1
	dec	C
	brne	sad3

sad4:	clt	
sad5:	ret


;**************************************************************************
;*
;* Authenticate sender
;*
;**************************************************************************

auth_sender:
	rcall	get_eeflags
	sbrs	A,log_stat		;skip if logged in
	rjmp	aus4			;else quit

aus1:	push	YH
	push	YL
	ldi	YL,low(rx_buff)		;receive buffer
	ldi	YH,high(rx_buff)
	ldi	A,30			;offset to address field + 1
	add	YL,A
	clr	A
	adc	YH,A

	ldi	ZL,low(dest_addr)	;dest address in eeprom
	ldi	ZH,high(dest_addr)
	rcall	ee_read			;get first (length) byte
	inc	A			;round it up...
	lsr	A			;to get length in bytes
	inc	A			;plus length & number type byte
	inc	A
	mov	lcnt,A

aus2:	rcall	ee_read
	adiw	ZL,1
	rcall	next_char
	breq	aus3			;mismatch
	cp	A,B
	brne	aus3			;mismatch
	
	dec	lcnt
	brne	aus2			;do entire address

	pop	YL
	pop	YH
	rjmp	aus5

aus3:	ldi	stat_cnt,4		;kill response message
	pop	YL
	pop	YH
aus4:	pop	A			;dump caller
	pop	A
aus5:	ret


;**************************************************************************
;*
;* Encode 7-bit string into SMS 8-bit packed format
;*
;* Entry: Y = pointer to 8-bit string (destination)
;*        Z = pointer to 7-bit string (source)
;*
;* Exit:  C = packed string length
;*        D = unpacked string length
;*
;**************************************************************************

encode_str:
	clr	C
	clr	E

enc1:	ldi	lcnt,1
	ld	B,Z+
	tst	B
	breq	enc5

	inc	E
enc2:	mov	A,B
	ld	B,Z+
	tst	B
	breq	enc4

	inc	E
	clr	D
	push	lcnt
enc3:	lsr	B
	ror	D
	dec	lcnt
	brne	enc3
	pop	lcnt

	or	A,D
	st	Y+,A
	inc	C
	
	inc	lcnt
	cpi	lcnt,8
	brne	enc2
	rjmp	enc1

enc4:	st	Y+,A
	inc	C
enc5:	mov	D,E
	ret


;**************************************************************************
;*
;* Decode 8-bit packed SMS string into 7-bit format
;*
;* Entry: Y = pointer to 8-bit string (source)
;*        Z = pointer to 7-bit string (destination)
;*        C = unpacked string length
;*
;**************************************************************************

decode_str:
	clr	A
dec1:	ldi	lcnt,1
	
dec2:	ld	B,Y+
	clr	D
	push	lcnt

dec3:	lsl	B
	rol	D
	dec	lcnt
	brne	dec3

	pop	lcnt
	lsr	B
	or	B,A
	st	Z+,B
	mov	A,D
	
	inc	lcnt
	cpi	lcnt,8
	brne	dec4	

	dec	C
	breq	dec5
	
	st	Z+,A			;grow a char...
	clr	A
	ldi	lcnt,1
	
dec4:	dec	C
	brne	dec2
dec5:	ret


;**************************************************************************
;*
;* Generate 16-bit XOR of buffer contents
;*
;* Entry: C = buffer length
;*        Z = buffer pointer
;*
;* Exit:  B,A = result
;*
;**************************************************************************

xor_buff:
	push	D
	clr	A
	clr	B
xbf1:	ld	D,Z+	
	sbrc	ZL,0			;skip if even byte
	rjmp	xbf2			;must be odd byte

	eor	B,D
	rjmp	xbf3

xbf2:	eor	A,D
xbf3:	dec	C
	brne	xbf1
	pop	D
	ret


;**************************************************************************
;*
;* Copy string from flash memory to ram
;*
;* Entry: Z = source (flash memory) pointer
;*        Y = destination (ram) pointer
;*
;**************************************************************************

mov_str_ram:
	push	R0
msr1:	lpm
	adiw	ZL,1
	st	Y+,R0
	tst	R0			;zero terminated
	brne	msr1
	pop	R0
	ret


;**************************************************************************
;*
;* Copy flash memory data block to ram
;*
;* Entry: Y = destination (ram) pointer
;*        Z = source (flash memory) pointer
;*        lcnt = length
;*
;**************************************************************************

mov_flash_ram:
	push	R0
mfr1:	lpm
	adiw	ZL,1
	st	Y+,R0
	dec	lcnt
	brne	mfr1
	pop	R0
	ret


;**************************************************************************
;*
;* Copy ram memory block to eeprom
;*
;* Entry: Y = source (ram) pointer
;*        Z = destination (eeprom) pointer
;*        lcnt = length
;*
;**************************************************************************

mov_ram_ee:
	ld	A,Y+
	rcall	ee_write
	adiw	ZL,1
	dec	lcnt
	brne	mov_ram_ee
	ret


;**************************************************************************
;*
;* Copy eeprom block to ram
;*
;* Entry: Y = destination (ram) pointer
;*        Z = source (eeprom) pointer
;*        lcnt = length
;*
;**************************************************************************

mov_ee_ram:
	rcall	ee_read			;read a byte from EEPROM
	adiw	ZL,1			;bump pointer
	st	Y+,A			;write to buffer
	dec	lcnt
	brne	mov_ee_ram
	ret


;**************************************************************************
;*
;* Copy string from ram to eeprom
;*
;* Entry: Y = source (ram) 
;*        Z = destination (eeprom)
;*        C = string length (or null terminated)
;*
;**************************************************************************

write_ee_str:
	ld	A,Y+
	rcall	ee_write
	adiw	ZL,1			;bump write pointer
	tst	A
	breq	wes2

	dec	C
	brne	write_ee_str
wes2:	ret		


;**************************************************************************
;*
;* Search 'system' string table for match with received message
;*
;* Entry: Y = message buffer pointer
;*
;* Exit:  C = matching string 'index' number
;*	  T flag set = match
;*
;**************************************************************************

compr_sys:
	clr	C			;string index
	ldi	ZH,high(sys_tbl*2)	;sys string table pointer 
	ldi	ZL,low(sys_tbl*2)
cps1:	rcall	compr_cstr
	brts	cps2			;exit if match found
	
	inc	C
	cpi	C,11			;number of code memory strings
	brlo	cps1
cps2:	ret


;**************************************************************************
;*
;* Search 'user' string table for match with received message
;*
;* Entry: Y = message buffer pointer
;*        Z = EEPROM string pointer
;*        C = number of entries to search
;*
;* Exit:  C = matching string 'index' number
;*	  T flag set = match
;*
;**************************************************************************

compr_usr:
	push	C
	mov	lcnt,C

cpu1:	ldi	C,16			;max. string length
	rcall	compr_estr		;look for match
	brts	cpu2			;go if found

	adiw	ZL,16			;Z points to next EEPROM string
	dec	lcnt
	brne	cpu1

cpu2:	pop	C
	sub	C,lcnt			;create simple index (0-n)
	ret
	

;**************************************************************************
;*
;* Compare message string with code memory string
;*
;* Entry: Y = message buffer pointer
;*        Z = code memory string pointer
;*
;**************************************************************************	

compr_cstr:
	push	YL			;message string pointer
	push	YH
	set
cpc1:	lpm
	adiw	ZL,1
	tst	R0
	breq	cpc3			;exit if end of string

	brtc	cpc1			;skip test if mismatch already

	rcall	next_char
	breq	cpc2			;buffer end, no match

	andi	B,$5F			;force upper case
	cp	R0,B
	breq	cpc1			;test next if match

cpc2:	clt				;indicate no match
	rjmp	cpc1

cpc3:	brtc	cpc4			;skip if no match

	pop	B			;else keep current pointer
	pop	B 
	rjmp	cpc5

cpc4:	pop	YH			;restore message pointer
	pop	YL
cpc5:	ret


;**************************************************************************
;*
;* Compare message string with EEPROM string
;*
;* Entry: Y = message buffer pointer
;*        Z = EEPROM string pointer
;*	  C = EEPROM (max) string length
;*
;**************************************************************************

compr_estr:
	push	ZH
	push	ZL
	push	YH
	push	YL
	clt

cpe1:	rcall	ee_read
	adiw	ZL,1

	tst	A			;end of string?
	brne	cpe2

	brts	cpe3			;must be a complete match...	
	rjmp	cpe4			;else empty string 

cpe2:	rcall	next_char
	breq	cpe4			;buffer end, incomplete match

	cp	A,B
	brne	cpe4			;exit if mismatch

	set				;at least one char match...
	dec	C
	brne	cpe1
	
cpe3:	pop	A			;keep current message pointer
	pop	A 
	rjmp	cpe5

cpe4:	clt				;flag mismatch
	pop	YL			;restore message pointer
	pop	YH
cpe5:	pop	ZL
	pop	ZH
	ret	


;**************************************************************************
;*
;* Returns next char from message buffer
;*
;**************************************************************************

next_char:
	ld	B,Y+
	tst	B
	ret


;**************************************************************************
;*
;* Set output port state
;*
;**************************************************************************

write_out_port:
	out	PORTC,A
	ldi	ZH,high(out_port)
	ldi	ZL,low(out_port)	
	rcall	ee_write		;update EEPROM image
	ret


;**************************************************************************
;*
;* Get flags from EEPROM
;*
;**************************************************************************

get_eeflags:
	ldi	ZH,high(eeflags)
	ldi	ZL,low(eeflags)	
	rcall	ee_read			;read flags bits
	ret


;**************************************************************************
;*
;* Update flags in EEPROM
;*
;**************************************************************************

put_eeflags:
	ldi	ZH,high(eeflags)
	ldi	ZL,low(eeflags)	
	rcall	ee_write		;write flags bits	
	ret


;**************************************************************************
;*
;* Erase entire EEPROM
;*
;**************************************************************************

erase_eeprom:
	ldi	ZL,low(eestart)
	ldi	ZH,high(eestart)
	ldi	B,2			;to do 512 bytes
	clr	lcnt
	clr	A			;init value

epr1:	rcall	ee_write
	adiw	ZL,1
	dec	lcnt
	brne	epr1
	dec	B
	brne	epr1

	ldi	ZL,low(eemagic)		;write magic number
	ldi	ZH,high(eemagic)
	ldi	A,$5A
	rcall	ee_write
	
	ldi	ZL,low(eelevel)		;write charge defaults
	ldi	ZH,high(eelevel)
	ldi	A,DISCHARGE_LEVEL
	rcall	ee_write

	ldi	ZL,low(eecharge)
	ldi	ZH,high(eecharge)
	ldi	A,CHARGE_TIME
	rcall	ee_write
	ret


;**************************************************************************
;*
;* EEPROM random read/write routines
;*
;* Entry: Z = address, A = data (writes)
;* Exit : A = data (reads)
;*
;**************************************************************************

ee_read:
	sbic	EECR,EEWE
	rjmp	ee_read			;loop until last cycle complete

	out	EEARL,ZL		;write address
	out	EEARH,ZH
	sbi	EECR,EERE		;set read strobe
	in	A,EEDR			;get data
	ret
	
ee_write:
	sbic	EECR,EEWE
	rjmp	ee_write		;loop until last cycle complete

	out	EEARL,ZL		;write address
	out	EEARH,ZH
	out	EEDR,A			;write data
	cli				;interrupts off for the next bit...
	sbi	EECR,EEMWE		;set master write enable
	sbi	EECR,EEWE		;set write strobe
	sei
	ret
	

;**************************************************************************
;*
;* Sleep C, B millisecs (65535 max)
;*
;**************************************************************************

sleep_msecs:
	clr	sys_tmr_lo		;synchronise...
	clr	sys_tmr_hi

slp1:	mov	A,sys_tmr_lo
	sub	A,B
	mov	A,sys_tmr_hi
	sbc	A,C
	brsh	slp2	

	sbrs	aflags,tick_flag	;skip on 1/4 sec mark
	rjmp	slp1
	
	wdr				;watchdog keep-alive
	cbr	aflags,1<<tick_flag
	rjmp	slp1

slp2:	wdr
	ret


;**************************************************************************
;**************************************************************************
;*
;* Timer/Counter 0 Overflow Interrupt Handler (1ms interval)
;*
;**************************************************************************
;**************************************************************************

tmr0_irq:
	in	bstate,SREG		;save state
	push	A

	ldi	A,141			;reload timer 0 up count
	out	TCNT0,A

; Update charge cycle minutes timer

	sbrs	cflags,time_sync
	rjmp	tiq1
	
	ldi	A,1			;one second down counter
	sub	min_tmr_lo,A
	clr	A
	sbc	min_tmr_hi,A
	brcc	tiq3

tiq1:	ldi	A,low(60*1002)		;60*1002*((1/(7.3728MHz/64))*(256-141))
	mov	min_tmr_lo,A
	ldi	A,high(60*1002)
	mov	min_tmr_hi,A
	sbr	cflags,1<<time_sync

	lds	A,chg_tmr_lo		;minutes down counter
	subi	A,1
	sts	chg_tmr_lo,A
	brcc	tiq3

	lds	A,chg_tmr_hi
	tst	A
	brne	tiq2
	sbr	cflags,1<<time_out	;minutes counter expired
tiq2:	dec	A	
	sts	chg_tmr_hi,A

; Next bit increments system timers

tiq3:	inc	sys_tmr_lo		;bump millisecs count
	breq	tiq4
	
	mov	A,sys_tmr_lo
	cpi	A,128
	breq	in_detect		;do every 128ms...
	rjmp	timeouts

tiq4:	inc	sys_tmr_hi
	sbr	aflags,1<<tick_flag	;for watchdog keep-alive
	
;**************************************************************************
;*
;* Look for input port state change (every 128ms)
;*
;**************************************************************************

in_detect:
	push	B			;save used
	push	C
	push	D
	push	E
	push	YL
	push	YH
	
	in	B,PINA			;read input port
	andi	B,$0F			;mask off unwanted bits
	
	mov	D,B
	lds	A,in_port		;get previous saved state

	eor	A,B			;look for changed bits
	breq	idt3			;go if no change

	mov	C,A			;C,A = changed bit(s)
	and	C,B			;C = changed high bit(s)
	swap	C			;to high nibble
	
	ser	E
	eor	B,E			;invert input port bits
	and	A,B			;A = changed low bit(s)
	
	or	C,A			;merge changed high + changed low
	
	sbrc	aflags,in_sample	;skip if first sample
	rjmp	idt1			;else go compare to previous sample

	sbr	aflags,1<<in_sample	;flag to sample twice (noise filter) 
	sts	in_temp,C		;save changed bits for next sample
	rjmp	idt4
	
idt1:	lds	A,in_temp		;bits changed last sample
	and	C,A
	breq	idt3			;ignore if no bit(s) remain changed

; Save changed bits for foreground processing

	lds	A,s_buff_len
	cpi	A,16
	breq	idt3			;ignore change if buffer full

	inc	A
	sts	s_buff_len,A

	lds	YL,s_buff_hptr		;get fifo head pointer
	lds	YH,s_buff_hptr+1
	st	Y+,C			;'push' the changed state
	
	ldi	A,low(s_buff+16)	;physical end of buffer?
	cp	YL,A
	ldi	A,high(s_buff+16)
	cpc	YH,A
	brcs	idt2

	ldi	YL,low(s_buff)		;wrap pointer
	ldi	YH,high(s_buff)

idt2:	sts	s_buff_hptr,YL		;save fifo head pointer
	sts	s_buff_hptr+1,YH
	sts	in_port,D		;save port state

idt3:	cbr	aflags,1<<in_sample

idt4:	pop	YH
	pop	YL
	pop	E
	pop	D
	pop	C
	pop	B

;**************************************************************************
;*
;* Handle send/receive timeouts
;*
;**************************************************************************

timeouts:
	sbrs	iflags,rx_ack_pend	;skip if receive ack pending
	rjmp	tme2

	dec	rx_ack_tmr		;receive ack timer
	brne	tme2

; Receive acknowledge timeout...

	cbr	iflags,1<<rx_ack_pend	;reset pending
	
	lds	A,rx_ato_cnt		;bump error count
	inc	A
	sts	rx_ato_cnt,A

	dec	tx_retry_cnt		;bump retry count
	breq	tme1			;go if exhausted
	
	sbr	aflags,1<<tx_msg_retry	;else request buffer retransmit
	rjmp	tme2
	
tme1:	cbr	aflags,1<<tx_buff_lok	;release transmit buffer
	sbr	aflags,1<<tx_fatal_err	;flag serious error

tme2:	tst	rx_frame_tmr		;receive frame timer/delay timer
	breq	tme3
	
	dec	rx_frame_tmr
	rjmp	tme4

tme3:	sbrs	iflags,rx_frame_ip	;skip if frame receive in progress
	rjmp	tme4
	
; Receive frame timeout...

	cbr	iflags,1<<rx_frame_ip	;reset receive in progress
	lds	A,rx_fto_cnt		;bump timeout error count
	inc	A
	sts	rx_fto_cnt,A

tme4:	cbi	PORTD,PD2		;trigger an INT0 interrupt

	pop	A
	out	SREG,bstate
	reti


;**************************************************************************
;*
;* Interrupt 0 handler
;*
;**************************************************************************

int0_irq:
	in	astate,SREG
	push	A
	push	B
	push	C
	push	YH
	push	YL
	push	ZH
	push	ZL	

	clr	A
	out	GIMSK,A			;prevent reentry!
	sbi	PORTD,PD2		;set the INT0 bit to clear request
	sei				;enable nested interrupts

	sbrs	iflags,rx_frame_rdy	;skip if received frame
	rjmp	inq2			;else go check transmit stuff

; Complete frame received...

	sbr	aflags,1<<rx_buff_lok	;lock buffer (cleared by foreground)
	cbr	iflags,1<<rx_frame_rdy	;rx_irq handshake
	
	ldi	ZL,low(rx_buff)		;receive buffer pointer
	ldi	ZH,high(rx_buff)	

; Reconstitute start of frame header

	ldi	A,$1E			;frame ID
	st	Z,A
	ldi	A,$0C			;destination device (fbus)
	std	Z+1,A
	ldi	A,0			;source device (phone)
	std	Z+2,A
	std	Z+3,rx_msg_typ		;message type
	std	Z+4,A			;null

; Check frame

	push	ZH
	push	ZL
	ldd	C,Z+5			;get length byte
	subi	C,-8			;add header + check word length
	sbrc	C,0
	inc	C			;must be even
	rcall	xor_buff		;generate frame check word
	pop	YL
	pop	YH
	or	A,B			;result should be zero...
	breq	inq1			;go if OK
	
; Frame check error...

	cbr	aflags,1<<rx_buff_lok	;dump buffer
	lds	A,rx_fce_cnt		;bump error count
	inc	A
	sts	rx_fce_cnt,A
	rjmp	inq2			;go do transmit stuff	

; Received frame checks OK, build acknowledge frame

inq1:	ldd	C,Y+3			;message type
	ldd	A,Y+5			;block length
	subi	A,-5			;add header length (-1)
	add	YL,A			;add offset to frame sequence number			
	clr	A
	adc	YH,A

	ld	B,Y			;get the sequence number
	andi	B,7			;turn off 'block' bits
;	mov	rx_seq_no,B		;(not used)

	ldi	YL,low(ack_buff)	;build frame here
	ldi	YH,high(ack_buff)

; Copy first 6 bytes of acknowledge frame from code memory

	ldi	ZL,low(ack_hdr*2)
	ldi	ZH,high(ack_hdr*2)
	ldi	lcnt,6			;length
	rcall	mov_flash_ram		;insert first 6 bytes
	st	Y+,C			;insert message type
	st	Y+,B			;and sequence number		

; Generate & append check word

	ldi	ZL,low(ack_buff)
	ldi	ZH,high(ack_buff)
	ldi	C,8			;length
	rcall	xor_buff		;generate check word
	st	Y+,A			;append it
	st	Y+,B

	sbr	iflags,1<<tx_ack_pend	;flag send pending

; Check if frame transmit in progress...

inq2:	sbrc	iflags,tx_frame_ip
	rjmp	inq5			;exit if transmit in progress

	sbrc	iflags,tx_ack_pend	;skip if transmit acknowledge not pending
	rjmp	inq3			;else go prepare to send it

; Transmit idle, check for foreground (or timeout) send pending...

	sbrc	iflags,rx_frame_ip
	rjmp	inq5			;listen before transmit (defer to phone)
		
	sbrc	aflags,tx_msg_retry
	rjmp	inq2a			;go if retransmit pending

	sbrs	aflags,tx_msg_req	;skip if send pending
	rjmp	inq5			;else exit

	ldi	A,TX_RETRY_COUNT
	mov	tx_retry_cnt,A		;retry count

inq2a:	ldi	YL,low(tx_buff)		;transmit buffer pointer
	sts	tx_buff_ptr,YL
	ldi	YH,high(tx_buff)
	sts	tx_buff_ptr+1,YH

	ldd	A,Y+5			;get block length
	subi	A,-8			;add frame header length
	sbrc	A,0
	inc	A			;must be even number
	mov	tx_byte_cnt,A		;bytes to transmit

	ldi	A,RX_ACK_TIMEOUT	;receive ack timeout
	mov	rx_ack_tmr,A

	sbr	iflags,1<<rx_ack_pend	;flag acknowledge (will be) pending
	sbr	aflags,1<<tx_buff_lok	;cleared when ack received (or timeout)
	cbr	aflags,1<<tx_msg_req|1<<tx_msg_retry ;clear transmit requests
	rjmp	inq4
	
; Transmit acknowledge pending, prepare to send it...

inq3:	ldi	A,low(ack_buff)		;pass pointer to 'ack' buffer
	sts	tx_buff_ptr,A
	ldi	A,high(ack_buff)
	sts	tx_buff_ptr+1,A
	ldi	A,10			;frame length
	mov	tx_byte_cnt,A
	cbr	iflags,1<<tx_ack_pend

inq4:	sbr	iflags,1<<tx_frame_ip	;cleared by tx_irq when frame sent
	cli				;interrupts off for a moment...
	sbi	UCR,UDRIE		;enable uart tx interrupts
	
inq5:	cli
	ldi	A,1<<INT0
	out	GIMSK,A			;enable INT0 interrupts
	pop	ZL
	pop	ZH
	pop	YL
	pop	YH
	pop	C
	pop	B
	pop	A
	out	SREG,astate
	reti


;**************************************************************************
;*
;* UART receive complete interrupt handler
;*
;**************************************************************************

rx_irq:
	in	bstate,SREG		;save state
	push	A
	push	B
	push	YH
	push	YL

	in	B,UDR			;get the received byte	
	
	sbrc	iflags,rx_frame_ip	
	rjmp	rxq3			;frame receive in progress				

	sbrs	iflags,rx_delay		;skip if waiting
	rjmp	rxq0

	tst	rx_frame_tmr		;5ms expired?
	brne	rxq1			;just exit if not

	cbr	iflags,1<<rx_delay	;clear wait flag	
rxq0:	cpi	B,$1E			;frame ID?
	breq	rxq2			;go if it is

; Waiting for frame ID but got something else...

	sbr	iflags,1<<rx_delay	;flag wait before looking for next ID 
	ldi	A,5			;init 5ms timer
	mov	rx_frame_tmr,A
rxq1:	rjmp	rxq14			;exit

; Frame ID received, initialise flags & timer

rxq2:	sbr	iflags,1<<rx_frame_ip|1<<rx_header_ip
	cbr	iflags,1<<rx_to_null
	clr	rx_hdr_cnt		;init header byte counter
	ldi	A,RX_FRAME_TIMEOUT
	mov	rx_frame_tmr,A		;init frame receive timeout
	rjmp	rxq14			;exit for now

; Frame receive in progress, header or block?

rxq3:	sbrs	iflags,rx_header_ip	;skip if header receive
	rjmp	rxq10			;else go buffer the byte

; Receiving header...

	inc	rx_hdr_cnt		;update header byte count
	mov	A,rx_hdr_cnt

	cpi	A,1			;destination byte?
	brne	rxq4

	cpi	B,$0C			;destination = fbus?
	brne	rxq7			;dump frame if not
	rjmp	rxq14			;else exit
	
rxq4:	cpi	A,2			;source byte?
	brne	rxq5
	
	cpi	B,0			;source = phone?
	brne	rxq7			;go dump frame if not
	rjmp	rxq14			;else exit
	
rxq5:	cpi	A,3			;message type byte?
	brne	rxq8
	
	cpi	B,$7F			;acknowledge frame?
	breq	rxq6

	mov	rx_msg_typ,B		;save message type
	rjmp	rxq14			;exit for now

; Acknowledge frame detected...

rxq6:	sbrc	iflags,rx_to_null	;continue if preceding bytes were OK
	rjmp	rxq14			;else ignore it

	cbr	iflags,1<<rx_ack_pend	;clear pending flag (no other checks!)
	cbr	aflags,1<<tx_buff_lok	;release transmit buffer

rxq7:	sbr	iflags,1<<rx_to_null	;remaining bytes to bit bucket
	rjmp	rxq14			;exit for now	
	
rxq8:	cpi	A,5			;block length byte?
	brne	rxq14			;exit if not

; Block length byte received ...
	
	mov	A,B
	sbrc	A,0
	inc	A			;must be an even number!
	inc	A			;for length & check bytes
	inc	A
	inc	A
	mov	rx_byte_cnt,A		;save as frame (block) length

	cpi	A,208-5+1		;buffer size minus header length
	brlo	rxq8a

	sbr	aflags,1<<rx_too_big	;flag buffer overflow for foreground
	rjmp	rxq9

rxq8a:	sbrc	aflags,rx_buff_lok	;skip if receive buffer not locked
rxq9:	sbr	iflags,1<<rx_to_null	;...else send to bit bucket

	cbr	iflags,1<<rx_header_ip	;done header inspection
	
	ldi	YL,low(rx_buff+5)	;init receive buffer pointer
	ldi	YH,high(rx_buff+5)
	rjmp	rxq11			;go save length byte
	
; Receiving block, save byte in buffer

rxq10:	lds	YL,rx_buff_ptr		;get receive buffer pointer
	lds	YH,rx_buff_ptr+1

rxq11:	sbrc	iflags,rx_to_null
	rjmp	rxq13			;skip save if output to null	

rxq12:	st	Y+,B			;save it
	sts	rx_buff_ptr,YL
	sts	rx_buff_ptr+1,YH

; Check for end of frame...

rxq13:	dec	rx_byte_cnt		;update received byte count
	brne	rxq14			;exit if more to do

; Frame receive complete...

	sbrs	iflags,rx_to_null
	sbr	iflags,1<<rx_frame_rdy	;flag for int0_irq
	cbr	iflags,1<<rx_frame_ip	;frame receive complete
	cbi	PORTD,PD2		;trigger an INT0 interrupt

rxq14:	pop	YL			;restore state
	pop	YH
	pop	B
	pop	A
	out	SREG,bstate
	reti	


;**************************************************************************
;*
;* UART transmit data register empty interrupt handler
;*
;**************************************************************************

tx_irq:
	in	bstate,SREG		;save state
	push	A
	push	XH
	push	XL

	tst	tx_byte_cnt		;done 'em all?
	brne	txq1			;go if not

	cbr	iflags,1<<tx_frame_ip	;buffer send complete
	cbi	UCR,UDRIE		;disable further tx ints for now
	cbi	PORTD,PD2		;trigger an INT0 interrupt
	rjmp	txq2
		
txq1:	dec	tx_byte_cnt
	lds	XL,tx_buff_ptr		;get transmit buffer pointer
	lds	XH,tx_buff_ptr+1
	ld	A,X+			;get a byte from the buffer
	out	UDR,A			;write it to the tx register
	sts	tx_buff_ptr,XL		;save updated pointer
	sts	tx_buff_ptr+1,XH

txq2:	pop	XL
	pop	XH
	pop	A
	out	SREG,bstate		;restore state
	reti


;**************************************************************************
;**************************************************************************
;*
;* DATA
;*
;**************************************************************************
;**************************************************************************

smsg1:
.db	"OK",0,0

smsg2:
.db	"bad pass",0,0

smsg3:
.db	"bad cmd",0

ack_hdr:
.db	$1E,0,$0C,$7F,0,2			  ;'acknowledge' header

msg_hdr:
.db	$1E,0,$0C,2,0,0,0,1,0,1,2,0		  ;'send SMS' header 

get_sms_frm:
.db	$1E,0,$0C,$14,0,$0A,0,1,0,7,2,0,1,$64,1,0 ;'get SMS message' frame

del_sms_frm:
.db	$1E,0,$0C,$14,0,8,0,1,0,$0A,2,0,1,0	  ;'delete SMS' frame

get_smsc_frm:
.db	$1E,0,$0C,2,0,8,0,1,0,$33,$64,1,1,0	  ;'get SMSC' frame

get_stat_frm:
.db	$1E,0,$0C,4,0,6,0,1,0,1,1,0		  ;'get status' frame

press_key_frm:
.db	$1E,0,$0C,$0C,0,9,0,1,0,$42,1,$0D,1,1,0,0 ;'press key' frame

vp_field:
.db	$A7,0,0,0,0,0,0,0			  ;VP field (24 hours validity)
						  ;(pad byte added)
sys_tbl:
.db	"IN",0,"OUT",0,"CHARGE",0,"PASS",0,"ACK",0,"EN",0
.db	"DIS",0,"LOGIN",0,"LOGOUT",0,"STAT",0,"COUNT",0

dec_con:
.db	$27,$10,3,$E8,0,$64,0,$0A,0,1		  ;decimal conversion constants

copyright:
.db	"(C)2004 Silicon Chip Publications P/L",0


;**************************************************************************
;**************************************************************************
;*
;* DEFINE USED SRAM
;*
;**************************************************************************
;**************************************************************************

		.dseg

rx_fto_cnt:	.byte	1	;receive frame timeout error count
rx_ato_cnt:	.byte	1	;receive ack timeout error count
rx_fce_cnt:	.byte	1	;receive frame check error count

pstat_array:	.byte	7	;phone status array

chg_tmr_lo:	.byte	1	;charge minutes timer
chg_tmr_hi:	.byte	1

in_port:	.byte	1	;last input port state
in_temp:	.byte	1	;input port bit temp
bit_temp:	.byte	1	;input port bit temp	

s_buff_len:	.byte	1	;fifo depth
s_buff_tptr:	.byte	2	;fifo tail pointer
s_buff_hptr:	.byte	2	;fifo head pointer
s_buff:		.byte	16	;input port state change fifo

rx_buff_ptr:	.byte	2	;receiver buffer pointer
tx_buff_ptr:	.byte	2	;transmit buffer pointer
rx_buff_len:	.byte	1	;bytes in receive buffer
smsc_addr:	.byte	12	;SMSC length, type & number

time_stamp:	.byte	6*4	;timestamp save for 4 slots
str_buff:	.byte	35	;string buffer

; WARNING: the following buffers must start on an EVEN (word) boundary!

ack_buff:	.byte	10	;transmit ack buffer
tx_buff: 	.byte	128	;transmit buffer
rx_buff: 	.byte	208	;receive buffer

; Stack top is at $25F (allow 32 bytes)


;**************************************************************************
;**************************************************************************
;*
;* DEFINE USED EEPROM
;*
;**************************************************************************
;**************************************************************************

		.eseg
		.org	0

eestart:	.byte	1	;reserved
eemagic:	.byte	1	;magic number
eeflags:	.byte	1	;various flags
eelevel:	.byte	1	;battery discharge level (5110 & 6110)
eecharge:	.byte	1	;battery charge time (3210 & 3310)
rx_count:	.byte	2	;received sms count
tx_count:	.byte	2	;sent sms count
in_mask:	.byte	1	;input port bit change mask
puls_bits:	.byte	1	;output bit(s) to pulse
out_port:	.byte	1	;output port state save
password:	.byte	8	;password
dest_addr:	.byte	12	;destination address

		.org	$30
in_tbl:		.byte	8*16

		.org	$B0
out_tbl:	.byte	16*16
