;**************************************************************************
;**************************************************************************
;*
;*
;*                      RFID Security Module Firmware
;* 
;*                           Version 1.0 31/03/04
;*                           
;*                           Created by P.B.Smith
;*                   (C)2004 Silicon Chip Publications P/L
;*                           All Rights Reserved.
;*
;*                          www.siliconchip.com.au
;*
;*               +-------------------------------------------+
;*               | This source code is released as FREEWARE  |
;*               | and as such is NOT in the public domain.  |
;*               +-------------------------------------------+
;*
;**************************************************************************

.nolist
.include "2313ndef.inc"		;AVR definitions (non-standard!)
.list

;**************************************************************************
;*
;* REGISTERS
;*
;**************************************************************************

.def	status		=R2	;SREG save (used by interrupt)
.def	gen_timer	=R3	;general purpose timer
.def	mid_timer	=R4	;master id detected timer
.def	pzled_timer	=R5	;low frequency timer (for piezo & LED)
.def	tamp_timer	=R6	;tamper 'noise' timer
.def	lock_timer	=R7	;lock power timer
.def	rx_timer	=R8	;frame inter-char timer
.def	alarm_tmr_lo	=R9	;alarm timer low byte
.def	alarm_tmr_hi	=R10	;alarm timer high byte
.def	bad_id_cnt	=R11	;consecutive unknown ID count

.def	A		=R16	;scratch
.def	B		=R17	;scratch
.def	C		=R18	;scratch
.def	D		=R19	;scratch
.def	E		=R20	;scratch
.def	F		=R21	;scratch
.def	aflags		=R22	;various flags (see bit vars below)
.def	bflags		=R23	;ditto
.def	cflags		=R24	;ditto
.def	beep_cnt	=R25	;piezo beep count
.def	XL		=R26	;not used
.def	XH		=R27	;scratch
.def	YL		=R28	;not used
.def	YH		=R29	;not used
.def	ZL		=R30	;scratch pointer
.def	ZH		=R31	;scratch

; Register-bound bit variables

; aflags
.equ	ALARM_TRIG	=0	;set to trigger an alarm cycle
.equ	ALARM_ON	=1	;alarm state
.equ	ALARM_STAT	=2	;alarm state since last power up/rearm
.equ	TAMPER_POL	=3	;tamper switch input polarity (must be bit 3!)
.equ	TAMPER_STAT	=4	;tamper switch input state
.equ	LOCK_TRIG	=5	;set to trigger a lock power cycle
.equ	POR_STAT	=6	;power_on_reset indicator
.equ	GLOBAL_ARM	=7	;global arm

; bflags
.equ	ID_SAVED	=0	;ID saved successfully
.equ	ID_ERASED	=1	;ID erased successfully
.equ	ID_EMPTY	=2	;ID array empty
.equ	ID_FOUND	=3	;ID found
.equ	ID_MASTER	=4	;master ID detected
.equ	ID_UPDATE	=5	;ID add/remove function active
.equ	FRAME_IP	=7	;frame receive (from ID-12) in progress

; cflags
.equ	BLANK_LED	=0	;controls LED blanking after beep
.equ	DO_IT_NOW	=1	;piezo/LED timer preempt flag


;**************************************************************************
;*
;* HARDWARE SPECIFIC
;*
;**************************************************************************

; Port B bits

.equ	PIEZO_BIT	=7	;piezo output
.equ	IDRES_BIT	=5	;ID-12 reset output
.equ	LED_BIT		=1	;LED output
.equ	JP1_BIT		=0	;jumper input

; Port D bits

.equ	ARM_BIT		=6	;arm/disarm output
.equ	LOCK_BIT	=5	;door strike output
.equ	ALARM_BIT	=4	;alarm output
.equ	TAMPER_BIT	=3	;tamper switch input
.equ	JP2_BIT		=2	;JP2 input
.equ	JP3_BIT		=1	;JP3 input


;**************************************************************************
;*
;* CONSTANTS
;*
;**************************************************************************

.equ	LOCK_ON_TIME	=10	;lock 'on' time in 0.5 sec increments (32 max)
.equ	ALARM_ON_TIME	=295 	;alarm siren time in secs (4294 max)
;				;(5 minutes is max. time allowed by law)
.equ	ADD_REMOVE_TIME	=4	;add/remove timeout in secs (16 max)
.equ	MAX_BAD_IDS	=3	;number of consecutive bad IDs before alarm

.equ	BAUD_9600	=25	;UART baud rate divisor (4MHz crystal)

	.cseg
	.org	$0

;**************************************************************************
;*
;* INTERRUPT VECTORS
;*
;**************************************************************************
        
	rjmp	reset		;reset
	rjmp	bad_irq		;IRQ0
	rjmp	bad_irq		;IRQ1
	rjmp	bad_irq		;timer 1 capture
	rjmp    bad_irq		;timer 1 compare
	rjmp	bad_irq 	;timer 1 overflow
	rjmp	tmr0_irq	;timer 0 overflow
	rjmp	bad_irq		;UART rx complete
	rjmp	bad_irq		;UART data register empty
	rjmp	bad_irq		;UART tx complete
	rjmp	bad_irq		;analog comparator
bad_irq:
	reti


;**************************************************************************
;*
;* RESET ENTRY POINT
;*
;* Initialise I/O registers & flags
;*
;**************************************************************************
	
reset:	cli				;assume nothing
	ldi	A,low(RAMEND)		;set stack top
	out	SPL,A

	sbi	ACSR,ACD		;power down comparator
		
	ldi	A,0b10100010		;IDRES, PIEZO, LED outputs 
	out	DDRB,A
	ldi	A,0b11111111		;inputs pulled up, others inactive
	out	PORTB,A

	ldi	A,0b01110000		;ARM, ALARM, LOCK outputs
	out	DDRD,A
	ldi	A,0b10001110		;inputs pulled up, others inactive
	out	PORTD,A

	ldi	A,0b00000101		;CT0 clocked at CK/1024
	out	TCCR0,A
	ldi	A,1<<TOIE0		;enable CT0 overflow interrupt
	out	TIMSK,A

	clr	aflags			;init bit & byte variables
	clr	bflags
	clr	cflags
	clr	beep_cnt
	clr	pzled_timer
	clr	bad_id_cnt
	
	ldi	A,BAUD_9600
	out	UBRR,A			;set UART bit rate
	ldi	A,1<<RXEN		;no interrupts, receiver enabled 
	out	UCR,A
	in	A,UDR			;flush UART data register

	sei				;enable interrupts

;**************************************************************************
;*
;* Initialise the alarm system
;*
;**************************************************************************	

	sbis	PINB,JP1_BIT		;skip if JP1 not installed
	rcall	erase_all		;else erase EEPROM & set mode

	rcall	get_tamp		;determine tamper switch polarity
	rcall	get_alloc		;test for empty ID array
	breq	ini1			;don't arm if empty
	rcall	arm_me			;power up in armed state

ini1:	wdr				;reset watchdog...
	ldi	A,0b00001111
	out	WDTCR,A			;1.9 sec timeout, enable it

;**************************************************************************
;*
;* Main control loop
;*
;**************************************************************************

main:	rcall	get_id			;wait for valid code from ID-12
	rcall	compr_id		;see if it's in EEPROM
	sbrc	bflags,ID_FOUND		;skip if not found
	rjmp	mn3			;found, go decide what to do
	
; ID not found, check if add/remove function enabled  	

	sbrc	bflags,ID_UPDATE	;skip if add/remove not enabled
	rjmp	mn0			;...else go add it

	sbrs	bflags,ID_EMPTY		;skip if EEPROM empty, make master
	rjmp	mn1			;bad ID
	
mn0:	rcall	save_id			;attempt to add the ID
	sbrc	bflags,ID_SAVED		;skip if update failed
	rjmp	mn8			;update OK (1 beep)
	rjmp	mn11			;update failed (4 beeps)

; Unknown ID, bump the 'bad' count

mn1:	ldi	beep_cnt,3		;to beep 3 times
	inc	bad_id_cnt		;bump the bad count
	mov	A,bad_id_cnt
	cpi	A,MAX_BAD_IDS		;exceeded max bad IDs?
	brlo	mn12			;continue if not

mn2:	clr	bad_id_cnt
	sbr	aflags,1<<ALARM_TRIG	;trigger an alarm (if armed)
	cbr	bflags,1<<ID_UPDATE	;disable master add/remove
	rjmp	main			;loop

; ID found, check if it's the master ID
	
mn3:	clr	bad_id_cnt		;clear previous bad count
	sbrc	bflags,ID_MASTER	;continue if not master
	rjmp	mn4			;master, go toggle alarm state

	sbrs	bflags,ID_UPDATE	;skip if add/remove function active
	rjmp	mn6			;not active, go toggle alarm state
	
	sbrc	bflags,ID_ERASED	;should have been erased...
	rjmp	mn9			;erased OK (2 beeps)
	rjmp	mn11			;erase failed (4 beeps)
		
; Master ID detected, check if JP3 installed
 			
mn4:	sbic	PIND,JP3_BIT
	rjmp	mn5			;jumper out
	sbr	bflags,1<<ID_UPDATE	;jumper in, enable add/remove function
	rjmp	mn6
mn5:	cbr	bflags,1<<ID_UPDATE	;jumper out, disable add/remove function

; Toggle alarm state

mn6:	sbrc	aflags,GLOBAL_ARM	;skip to arm
	rjmp	mn7
	rcall	arm_me			;arm the system (1 beep)
	rjmp	mn12	
mn7:	rcall	disarm_me		;disarm the system (2 beeps)
	rjmp	mn12

; Sound piezo to indicate results

mn8:	ldi	beep_cnt,1		;1 beep: add ID OK
	rjmp	mn12
mn9:	ldi	beep_cnt,2		;2 beeps: remove ID OK
	rjmp	mn12
mn11:	ldi	beep_cnt,4		;4 beeps: add/remove ID failed

; Reload 'add/remove' function timer

mn12:	ldi	A,((ADD_REMOVE_TIME * 100000) / 6553)
	mov	mid_timer,A		;add/remove function timeout
	rjmp	main			;loop the loop


;**************************************************************************
;*
;* Erase EEPROM & set mode
;*
;**************************************************************************

erase_all:	
	ldi	A,id_alloc
	ldi	B,0
	ldi	C,(24*5)+3		;24 slots * 5 bytes + 3 id_alloc bytes

eea1:	rcall	ee_write		;wipe EEPROM
	inc	A
	dec	C
	brne	eea1

eea2:	sbis	PIND,TAMPER_BIT
	rjmp	eea2			;tamper input must be high initially

	ldi	B,4
	sbi	PORTD,LOCK_BIT
	nop
	sbis	PIND,TAMPER_BIT		;door strike ouput -> tamper input
	rjmp	eea3			;mode 4: lock powered on arm & disarm

	dec	B
	sbi	PORTD,ALARM_BIT
	nop
	sbis	PIND,TAMPER_BIT		;alarm ouput -> tamper input
	rjmp	eea3			;mode 3: lock powered on arm
	
	dec	B
	sbi	PORTD,ARM_BIT
	nop
	sbis	PIND,TAMPER_BIT		;arm/disarm ouput -> tamper input
	rjmp	eea3			;mode 2: lock powered on disarm
		
	dec	B			;mode 1: keyless entry (default)
eea3:	ldi	A,sys_mode
	rcall	ee_write		;save mode in EEPROM

	ldi	A,0b10001110		;reinit port D state
	out	PORTD,A
	mov	beep_cnt,B		;beep 'mode' times		

eea4:	rjmp	eea4			;wait for power off


;**************************************************************************
;*
;* Determine tamper switch input active state
;*
;**************************************************************************

get_tamp:
	ldi	A,3
	mov	gen_timer,A

gtp1:	tst	gen_timer		;wait for more than 130ms (settling time)
	brne	gtp1

	sbis	PIND,TAMPER_BIT		;skip if high, will be active low
	sbr	aflags,1<<TAMPER_POL	;set for active high
	ret


;**************************************************************************
;*
;* Arm the system
;*
;**************************************************************************

arm_me:
	ldi	A,sys_mode		;get mode byte
	rcall	ee_read
	cpi	B,2
	breq	arm1			;mode 2 = lock powered on disarm
	
	sbrs	aflags,POR_STAT		;if this a power-on arm?
	rjmp	arm1			;don't trigger lock if it is!	
	
	sbr	aflags,1<<LOCK_TRIG	;trigger lock power cycle

arm1:	cpi	B,1
	breq	arm4			;mode 1 = keyless, don't arm				

	sbis	PIND,JP2_BIT		;skip if no shunt on JP2
	rjmp	arm2

	sbi	PORTD,ARM_BIT		;active low ARM output
	rjmp	arm3

arm2:	cbi	PORTD,ARM_BIT		;active high ARM output

arm3:	cbr	aflags,1<<ALARM_STAT | 1<<ALARM_ON | 1<<ALARM_TRIG
	sbr	aflags,1<<GLOBAL_ARM	;set global arm
arm4:	sbr	aflags,1<<POR_STAT
	ldi	beep_cnt,1		;beep once
	ret


;**************************************************************************
;*
;* Disarm the system
;*
;**************************************************************************

disarm_me:
	ldi	A,sys_mode
	rcall	ee_read			;get mode byte
	cpi	B,3
	breq	dis1			;mode 3 = lock powered on arm
	
	sbr	aflags,1<<LOCK_TRIG	;trigger lock power cycle

dis1:	cpi	B,1
	brne	dis2			;skip if not keyless mode

	ldi	beep_cnt,1		;else beep once only & skip disarm
	rjmp	dis5

dis2:	sbis	PIND,JP2_BIT		;skip if no shunt on JP2
	rjmp	dis3

	cbi	PORTD,ARM_BIT
	rjmp	dis4

dis3:	sbi	PORTD,ARM_BIT
dis4:	cbr	aflags,1<<GLOBAL_ARM	;clear the global arm flag
	cbr	aflags,1<<ALARM_ON | 1<<ALARM_STAT ;clear alarm state flags
	cbi	PORTD,ALARM_BIT		;switch off ALARM output
	ldi	beep_cnt,2		;beep twice
dis5:	ret


;**************************************************************************
;*
;* Receive an ID code string from the ID-12 module
;*
;**************************************************************************

get_id:
	wdr				;kick the dog
	cbr	bflags,1<<FRAME_IP	;clear 'frame receive in progress'

gid1:	ldi	A,2
	mov	rx_timer,A		;frame inter-char timeout

gid2:	tst	rx_timer		;inter-char timeout?
	breq	get_id			;maybe, so reset 'in progress' flag

gid3:	rcall	rx_char			;get char from ID-12
	brcs	gid4			;go when char received
	
	sbrs	bflags,ID_UPDATE	;skip if add/remove function enabled
	rjmp	gid2

	tst	mid_timer		;add/remove function timer expired?
	brne	gid2			;nope, loop
	
	cbr	bflags,1<<ID_UPDATE	;yes, disable add/remove
	ldi	beep_cnt,1		;beep to indicate	
	rjmp	gid2			;loop
	
gid4:	sbrc	bflags,FRAME_IP		;skip if first char in frame
	rjmp	gid5			;frame receive already underway
	
	cpi	B,2			;first (start) char?
	brne	gid1			;no, so invalid, ignore

	ldi	ZL,id_buff		;put ID bytes & checksum here	
	ldi	C,15			;total chars in frame -1
	clr	R1			;init checksum
	sbr	bflags,1<<FRAME_IP	;flag frame receive underway
	rjmp	gid1			;loop for next char

gid5:	cpi	C,4			;go all data bytes & checksum?
	brlo	gid7			;yes, just count last 3 chars

	rcall	asc_hex			;convert to hex
	sbrc	C,0			;low bit means low nibble...
	rjmp	gid6
	
	ld	A,-Z			;get previous nibble
	swap	A			;swap to high
	or	B,A			;merge low nibble
	eor	R1,B			;keep checksum

gid6:	st	Z+,B			;save the char/byte
gid7:	dec	C			;done 'em all?
	brne	gid1			;loop if not

	tst	R1			;checksum OK?
	brne	get_id			;dump the ID if not
	sbr	cflags,1<<DO_IT_NOW	;to speed LED/piezo response
	ret


;**************************************************************************
;*
;* Convert ASCII to HEX
;*
;* Entry: B = ASCII character
;* Exit : B = hex nibble
;*
;**************************************************************************

asc_hex:
	subi	B,$30			;'0'
	brcs	ahx2			;bad HEX if < 0

	cpi	B,10
	brcs	ahx1			;done if 0-9

	andi	B,$1F			;to upper case
	subi	B,7
	cpi	B,$0A
	brcs	ahx2			;bad hex if < ASCII 'A'

	cpi	B,$10
	brcs	ahx2			;bad hex if > ASCII 'F'

ahx1:	clc
ahx2:	ret				;carry set if invalid (not tested)


;**************************************************************************
;*
;* Search for received ID code
;*
;**************************************************************************

compr_id:
	ldi	ZH,id_codes		;ID code array pointer
	cbr	bflags,1<<ID_FOUND | 1<<ID_MASTER | 1<<ID_ERASED | 1<<ID_EMPTY
	rcall	get_alloc		;get ID allocation bits
	brne	cp1			;skip if array not empty			

	sbr	bflags,1<<ID_EMPTY	;set the empty flag
	rjmp	cp6			;quit

cp1:	ldi	C,24			;bit count
cp2:	ldi	XH,5			;number of bytes to compare
	lsr	F			;test the next bit
	ror	E
	ror	D
	brcc	cp5			;unoccupied slot

; Occupied slot, do compare

	sbr	F,0b10000000		;fixup MSB of rotate
	ldi	ZL,id_buff		;received ID code pointer
cp3:	mov	A,ZH
	rcall	ee_read			;get ID byte from EEPROM
	ld	A,Z+			;get ID byte from buffer 
	cp	A,B			;match?
	brne	cp5			;no
	
	inc	ZH			;bump array pointer
	dec	XH			
	brne	cp3			;compare all 5 bytes

; ID found, flag & check if master

	sbr	bflags,1<<ID_FOUND
	cpi	C,24			;very first id entry?
	brne	cp4			;no
	
	sbr	bflags,1<<ID_MASTER	;yes, must be master
	rjmp	cp6			;exit (no erase)
	
; ID found, erase if requested

cp4:	sbrs	bflags,ID_UPDATE		
	rjmp	cp6			;found but no erase

	cbr	F,0b10000000		;clear the allocation bit to 'erase'	
cp5:	add	ZH,XH			;to next EEPROM slot
	dec	C			;scanned entire EEPROM?
	brne	cp2			;loop if not

	sbrs	bflags,ID_FOUND
	rjmp	cp6			;not found, exit
	
	sbrs	bflags,ID_UPDATE		
	rjmp	cp6			;no erase, skip {id_alloc} write
	
	ldi	A,id_alloc
	mov	B,D			;low byte
	rcall	ee_write_read		;write low byte {id_alloc}
	brne	cp6
	
	inc	A
	mov	B,E			;mid byte
	rcall	ee_write_read		;write mid byte {id_alloc}
	brne	cp6

	inc	A
	mov	B,F			;high byte
	rcall	ee_write_read		;write high byte {id_alloc}
	brne	cp6

	sbr	bflags,1<<ID_ERASED	;erase successful
cp6:	ret


;**************************************************************************
;*
;* Save ID code in first free EEPROM slot
;*
;**************************************************************************

save_id:
	ldi	ZH,id_codes		;ID code array pointer
	rcall	get_alloc		;get allocation bits	
	cbr	bflags,1<<ID_SAVED	;not saved yet!

	ldi	C,24			;bit count
sd1:	ldi	XH,5			;bytes to write
	lsr	F			;test the next bit
	ror	E
	ror	D
	brcs	sd4			;allocated slot

	sbrc	bflags,ID_SAVED
	rjmp	sd5			;already saved, skip write

; Unallocated slot, save ID code

	ldi	ZL,id_buff		;received ID code pointer
sd3:	ld	B,Z			;get ID byte from buffer 
	mov	A,ZH			;EEPROM address
	rcall	ee_write		;get ID byte from EEPROM
	rcall	ee_read			;read back to verify
	ld	A,Z+			;get ID byte again
	cp	A,B			;read = write?
	brne	sd5			;abort save if not

	inc	ZH			;bump array pointer
	dec	XH			
	brne	sd3			;save all 5 bytes

	sbr	bflags,1<<ID_SAVED	;ID saved successfully

sd4:	sbr	F,0b10000000		;allocated

sd5:	add	ZH,XH			;to next EEPROM slot
	dec	C			;scanned entire array?
	brne	sd1			;loop if not

; Update 'id allocation' bits

	sbrs	bflags,ID_SAVED
	rjmp	sd7
	
	ldi	A,id_alloc		;EEPROM address	
	mov	B,D			;low byte
	rcall	ee_write_read		;write low byte {id_alloc}
	brne	sd6

	inc	A
	mov	B,E			;mid byte
	rcall	ee_write_read		;write mid byte {id_alloc}
	brne	sd6	

	inc	A
	mov	B,F			;high byte
	rcall	ee_write_read		;write high byte {id_alloc}
	breq	sd7	

sd6:	cbr	bflags,1<<ID_SAVED	;id_alloc update failed
sd7:	ret


;**************************************************************************
;*
;* Get ID allocation bits from EEPROM
;*
;* Exit: D = low byte
;*	 E = mid byte
;*	 F = high byte
;* 
;**************************************************************************

get_alloc:
	ldi	A,id_alloc
	rcall	ee_read			;get ID allocation bits...
	mov	D,B			;low byte
	inc	A
	rcall	ee_read
	mov	E,B			;mid byte
	inc	A
	rcall	ee_read
	mov	F,B			;high byte
	or	B,E			;test for empty array
	or	B,D
	ret


;**************************************************************************
;*
;* EEPROM write, read & verify a byte
;* 
;**************************************************************************

ee_write_read:
	mov	C,B			;data
	rcall	ee_write		;write it, A = address
	rcall	ee_read			;read it back
	cp	C,B			;read = write?
	ret				;return to check


;**************************************************************************
;*
;* EEPROM random read/write routines
;*
;* Entry: A = address, B = data (writes)
;* Exit : B = data (reads)
;*
;**************************************************************************

ee_read:
	sbic	EECR,EEWE
	rjmp	ee_read			;loop until last cycle complete

	out	EEAR,A			;write address
	sbi	EECR,EERE		;set read strobe
	in	B,EEDR			;get data
	ret				;(read takes 4 clk cycles)
	
; Writes typically take 2.5ms @ 5V

ee_write:
	sbic	EECR,EEWE
	rjmp	ee_write		;loop until last cycle complete

	out	EEAR,A			;write address
	out	EEDR,B			;write data
	cli				;interrupts off for the next bit...
	sbi	EECR,EEMWE		;set master write enable
	sbi	EECR,EEWE		;set write strobe
	sei
	ret
	

;**************************************************************************
;*
;* Get character from UART if available
;*
;**************************************************************************

rx_char:
	clc
	sbis	USR,RXC
	rjmp	rcx1			;exit if no received char

	in	B,UDR			;get the received char	
	sec
rcx1:	ret	


;**************************************************************************
;**************************************************************************
;*
;* TIMER/COUNTER 0 OVERFLOW INTERRUPT HANDLER (65.5ms interval)
;*
;* Functions:
;*
;*  - serial receive inter-char timer
;*  - master id add/remove function timer
;*  - general timer
;*  - alarm timer
;*  - door lock handler
;*  - tamper detection
;*  - beeps piezo
;*  - flashes LED
;*
;**************************************************************************
;**************************************************************************

tmr0_irq:
	in	status,SREG		;save state
	push	A			;save used global
	push	B
	
	dec	rx_timer		;frame inter-char timer
	dec	mid_timer		;master id timer
	dec	gen_timer		;general purpose timer
	
;**************************************************************************
;*
;* Generate piezo beeps & LED flashes
;*
;*  if (pzled_timer = 0)
;*    if alarm_on | {beep_count} >0 then toggle piezo & LED outputs
;*    -elseif-
;*    alarm_prev then toggle LED output (indicates alarm occured)
;*    -elsif-
;*    global_arm then toggle LED output (indicates armed)
;*
;**************************************************************************

	tst	pzled_timer		;timer running?
	breq	tq1			;no, go check if it should be

	sbrc	cflags,DO_IT_NOW
	rjmp	tq1			;preempt expire to speed response		

	dec	pzled_timer
	brne	tq10			;exit if not expired

tq1:	cbr	cflags,1<<DO_IT_NOW
	sbrc	aflags,ALARM_ON
	rjmp	tq6			;alarm on

	tst	beep_cnt
	brne	tq6			;{beep_count} > 0

	sbi	PORTB,PIEZO_BIT		;piezo off
	sbrs	aflags,ALARM_STAT	;skip if previous alarm
	rjmp	tq2
	
	ldi	A,2			;armed indication
	sbis	PINB,LED_BIT		;0.13 secs on
	ldi	A,3			;2 secs off
	rjmp	tq5		

tq2:	sbrs	aflags,GLOBAL_ARM	;skip if armed
	rjmp	tq9			;else go switch LED off

	sbrs	cflags,BLANK_LED	;skip to enable blanking
	rjmp	tq3

	cbr	cflags,1<<BLANK_LED
	ldi	A,30			;blanking time
	mov	pzled_timer,A
	rjmp	tq10			;don't turn on LED yet

; (Re)load timer for armed indication (short on, long off)

tq3:	ldi	A,2			;armed indication
	sbis	PINB,LED_BIT		;0.13 secs on
tq4:	ldi	A,30			;2 secs off
tq5:	mov	pzled_timer,A
	rjmp	tq8			;go toggle LED state

; (Re)load timer for beep / alarmed indication

tq6:	ldi	A,2			;0.13 secs on
	sbis	PINB,PIEZO_BIT
	ldi	A,3			;0.2 secs off
	mov	pzled_timer,A		;reload timer
	sbr	cflags,1<<BLANK_LED	;to blank LED on arming  

; Toggle the piezo output

	sbis	PINB,PIEZO_BIT		;toggle the piezo bit
	rjmp	tq7
	cbi	PORTB,PIEZO_BIT
	sbi	PORTB,LED_BIT		;synchronise LED to PIEZO
	rjmp	tq8
tq7:	sbi	PORTB,PIEZO_BIT

	tst	beep_cnt
	breq	tq8
	dec	beep_cnt		;update count on rising edge

; Toggle the LED output

tq8:	sbis	PINB,LED_BIT		;toggle the output bit
	rjmp	tq9
	cbi	PORTB,LED_BIT
	rjmp	tq10
tq9:	sbi	PORTB,LED_BIT

tq10:	sbrs	aflags,GLOBAL_ARM
	rjmp	tq16			;skip alarm stuff if not armed

	sbrc	aflags,ALARM_STAT
	rjmp	tq16			;...or if already alarmed

;**************************************************************************
;*
;* Alarm 'on' timer
;*
;**************************************************************************

	sbrs	aflags,ALARM_ON
	rjmp	tq12			;no alarm

	dec	alarm_tmr_lo		;update alarm 'noise' timer
	brne	tq16			;no expire
	tst	alarm_tmr_hi
	breq	tq11			;expired
	dec	alarm_tmr_hi			
	rjmp	tq16			;no expire
	
; Clear alarm state when timer expires

tq11:	cbr	aflags,1<<ALARM_ON	;clear alarm condition
	sbr	aflags,1<<ALARM_STAT	;set previous alarm
	cbi	PORTD,ALARM_BIT		;switch off ALARM output
	rjmp	tq16

;**************************************************************************
;*
;* Test for alarm trigger
;*
;* If triggered, start timer and set enables
;*
;**************************************************************************

tq12:	sbrs	aflags,ALARM_TRIG
	rjmp	tq13			;no trigger

	cbr	aflags,1<<ALARM_TRIG	;reset trigger
	sbr	aflags,1<<ALARM_ON	;set alarm
	sbi	PORTD,ALARM_BIT		;switch on ALARM output
	ldi	A,low ((ALARM_ON_TIME * 100000) / 6553)
	mov	alarm_tmr_lo,A
	ldi	A,high ((ALARM_ON_TIME * 100000) / 6553)
	mov	alarm_tmr_hi,A
	rjmp	tq16

;**************************************************************************
;*
;* Tamper detection
;*
;**************************************************************************

tq13:	in	A,PIND			;get the current port D pin state
	andi	A,1<<TAMPER_BIT		;keep TAMPER bit
	mov	B,aflags	
	andi	B,1<<TAMPER_BIT		;determine if active
	eor	A,B
	brne	tq14			;inactive, go clear any previous	

	sbrs	aflags,TAMPER_STAT	;skip if previous detection
	rjmp	tq15

	dec	tamp_timer		;update timer
	brne	tq16
	
	sbr	aflags,1<<ALARM_TRIG	;timeout, trigger an alarm
	sbr	cflags,1<<DO_IT_NOW	;to speed LED/piezo response
tq14:	cbr	aflags,1<<TAMPER_STAT	;clear previous detect flag
	rjmp	tq16

tq15:	sbr	aflags,1<<TAMPER_STAT	;flag the detect
	ldi	A,1
	mov	tamp_timer,A		;start 65ms timer

;**************************************************************************
;*
;* Lock power control
;*
;**************************************************************************

tq16:	sbrs	aflags,LOCK_TRIG
	rjmp	tq17

	cbr	aflags,1<<LOCK_TRIG
	sbi	PORTD,LOCK_BIT		;lock output on
	ldi	A,((LOCK_ON_TIME * 100000) / 13106)
	mov	lock_timer,A
	rjmp	irq_exit

tq17:	sbis	PIND,LOCK_BIT		;skip if lock on
	rjmp	irq_exit

	dec	lock_timer		;update timer
	brne	irq_exit
	cbi	PORTD,LOCK_BIT		;lock output off

irq_exit:
	pop	B
	pop	A
	out	SREG,status		;and flags
	reti


;**************************************************************************
;**************************************************************************
;*
;* DATA
;*
;**************************************************************************
;**************************************************************************

copyright:
.db	"(C)2004 SC"			;copyright string


;**************************************************************************
;**************************************************************************
;*
;* DEFINE USED SRAM
;*
;**************************************************************************
;**************************************************************************
	
	.dseg

id_buff:	.byte	9		;ID code + checksum


;**************************************************************************
;**************************************************************************
;*
;* DEFINE USED EEPROM
;*
;**************************************************************************
;**************************************************************************

	.eseg
	.org	0

	.db	0			;never use first byte!
sys_mode:
	.db	0			;mode byte			
	.db	0,0,0			;reserved
id_alloc:
	.db	0,0,0			;ID array allocation bits

id_codes:
	.byte	24 * 5			;ID array
