; Watt Meter using LCD
; wattage is determined every 10.988 seconds
; kWh are accumulated at this rate 

	list P=16F628A
	#include p16f628A.inc

	__config _XT_OSC & _WDT_OFF & _PWRTE_ON & _BOREN_ON & _MCLRE_ON & _LVP_OFF

; Define variables at memory locations


EEPROM1		equ	H'00'	; non-volatile storage for line voltage sag cycles 
EEPROM2		equ	H'01'	; non-volatile storage for line voltage sag value
EEPROM3		equ	H'02'	; non-volatile storage for APOS MS byte
EEPROM4		equ	H'03'	; non-volatile storage for APOS LS btye
EEPROM5		equ	H'04'	; non-volatile storage for APGAIN MS byte	
EEPROM6		equ	H'05'	; non-volatile storage for APGAIN LS byte
EEPROM7		equ	H'06'	; non-volatile storage for Hysteresis of SAG
EEPROM8		equ	H'07'	; non-volatile storage for $/kWh (cost)
EEPROM9		equ	H'08'	; non-volatile storage for PHCAL
EEPROMA		equ	H'09'	; non-volatile storage for SAGCAL 

; Bank 0 RAM

EEIGHT		equ	H'20'	; 100 000, 000 (digit 9)
ESEVN		equ	H'21'	; 10 000,000 (digit 8)
ESIX		equ	H'22'	; 1 000,000s (digit 7 )
EFIVE		equ	H'23'	; 100,000s digit 6 store
EFOUR		equ	H'24'	; 10,000s digit 5 store
ETHREE		equ	H'25'	; 1 000s digit 4 store 
ETWO		equ	H'26'	; 100s digit 3 store
EONE		equ	H'27'	; 10s digit 2 store
EZERO		equ	H'28'	; 1's digit 1 store 
TIME_I		equ	H'29'	; delay store value
STORE1		equ	H'2A'	; storage 
STORE2		equ	H'2B'	; storage 
ADD_ADE		equ	H'2C'	; address/data value for ADE7756
ADD_ADEL	equ	H'2D'	; address/ data value (LS byte)
REG_A		equ	H'2E'	; ms byte of data, 40 bits
REG_B		equ	H'2F'	; 
REG_C		equ	H'30'	; 
REG_D		equ	H'31'
REG_E		equ	H'32'	; ls byte of 40 bit data
FLAG		equ	H'33'	; negative sign flag
BUS_TME		equ	H'34'	; busy time for display
UPDATE1		equ	H'35'	; seconds period counter
UPDATE2		equ	H'36'	; 6-minutes period LS byte
UPDATE3		equ	H'37'	; 6-minutes period MS byte
UPDATE4		equ	H'38'	; seconds period counter
HOURS1		equ	H'39'	; hours LS byte (hhhh.h)
HOURS2		equ	H'3A'	; hours
HOURS3		equ	H'3B'	; hours ms byte
FLAG1		equ	H'3C'	; data ready flag
WORK_VAL	equ	H'3D'	; working value
CNT4		equ	H'3E'	; 10 second counter
FIRST4		equ	H'3F'	; first 10 -seconds flag
NEG_SGN		equ	H'40'	; negative sign flag
ADDRESS		equ	H'41'	; display address position
D_STO		equ	H'42'	; data storage
COUNT		equ	H'43'	; 
BCD1		equ	H'44'	; overrange
BCD2		equ	H'45'	; MS decimal value
BCD3		equ	H'46'	; 
BCD4		equ	H'47'	; decimal value
BCD5		equ	H'48'	; 
UP_MODE1	equ	H'49'	; mode switch timer
BIN1		equ	H'4A'	; LS binary value
BIN2		equ	H'4B'	; 
BIN3		equ	H'4C'	; 
BIN4		equ	H'4D'	; 
UP_MODE		equ	H'4E'	; mode switch timer
TEMP		equ	H'4F'	; data storage
DSP_MODE	equ	H'50'	; display mode
SAG_CHNG	equ	H'51'	; sag level changed flag
SAG_X		equ	H'52' 	; sag level store
FG_SW		equ	H'53'	; switch closed flag
HYSTER		equ	H'54'	; Hysteresis for SAG
DOLLAR		equ	H'55'	; $/kWh value
WORK_VAL1	equ	H'56'	; working value ms byte
RSTLOOP		equ	H'57'	; reset delay counter
MD_CHFG		equ	H'58'	; mode changed flag
MD_STR		equ	H'59'	; storage of display mode value

; math routines
REMB0		equ	0x5A
TEMP1		equ 0x5B
TEMPB3		equ 0x5C
TEMPB2		equ	0x5D
TEMPB1      equ 0x5E
TEMPB0      equ 0x5F
AARGB5		equ	0x60
AARGB4		equ	0x61
AARGB3		equ	0x62
AARGB2      equ 0x63
AARGB1      equ 0x64
AARGB0      equ 0x65	; most significant byte of argument A
BARGB1      equ 0x66
BARGB0      equ 0x67	; most significant byte of argument B
LOOPCOUNT   equ 0x68   	; loop counter

REGA_S		equ	0x69	; REG_A store
REGB_S		equ	0x6A	; REG_B store
REGC_S		equ	0x6B	; REG_C store
SEC_FLG		equ	0x6C	; second flag

; All Banks RAM

W_TMP		equ	H'70'	; storage of w before interrupt
STATUS_TMP	equ	H'71'	; status storage before interrupt
BRNCNT		equ	H'72'	; brownout counter	
KWHOUR_A	equ	H'77'	; 9th byte of kWH storage (ms)
KWHOUR_B	equ	H'78'	; 8th byte of kWH storage
KWHOUR_C	equ	H'79'	; 7th byte of kWH storage
KWHOUR_D	equ	H'7A'	; 6th byte of kWH storage
KWHOUR_E	equ	H'7B'	; 5th byte of kWH storage
KWHOUR_F	equ	H'7C'	; 4th byte of kWH storage
KWHOUR_G	equ	H'7D'	; 3rd byte of kWH storage
KWHOUR_H	equ	H'7E'	; 2nd byte of kWH storage
KWHOUR_I	equ	H'7F'	; 1st byte of kWH storage (ls)


; preprogram EEPROM DATA 
	
	ORG     2100
	DE	D'100', H'00', D'00', D'07', H'0D', H'A8', D'00', D'100'
	DE	H'00', D'59'

; start at memory 0

	org	0
	goto	MAIN
	org     4		; interrupt vector 0004h, start interrupt routine here
;	goto	INTRUPT		

;******************************************************************************************* 

; INTERRUPT

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp 
	bcf 	STATUS,RP0	; bank 0 
	bcf	    INTCON,T0IF	; clear TMRO interrupt flag
	movf	TMR0,w		; timer value	
						; freq is 4MHz/4/16/(256(-8+2))=250Hz
	nop
	nop					; synchronise counter ie add two cycles
	addlw	0x8			; add to timer register and takes 2 cycles to start counting
	movwf	TMR0		; place in timer (the two cycles plus 2-cycle delay = 4cycles)
	
; increase mode switch counter (when UP_MODE1 reaches 2, UP_MODE is decreased giving about 2secs timer) 
	
	movf	UP_MODE,w
	btfsc	STATUS,Z	; if zero leave	(timed out)
	goto	CK_1s
	incf	UP_MODE1,f
	btfss	UP_MODE1,1	; decrease UP_MODE every 2nd count
	goto	CK_1s
	decf	UP_MODE,f	; decreased on 4th count of UP_MODE1
	clrf	UP_MODE1	; cleared on count of 2 (bit 1 high)

; check for 1 second period elapsed

CK_1s
	incf	UPDATE4,f	; seconds counter
	movf	UPDATE4,w
	xorlw	D'250'		; compare counter with 250 for 1 second
	btfss	STATUS,Z	; skip if equal
	goto	WAT_PER		; not 1 second
	incf	CNT4,f		; count to 10 counter
	bsf		SEC_FLG,0	; to ensure SAG indicated in cal modes

; load wattage value each 10.986328 seconds
WAT_PER
	movf	CNT4,w
	xorlw	0x0A		; is it 10
	btfss	STATUS,Z	; is it 10 (10 seconds)
	goto	ONE_SEC		; check seconds update
	movf	UPDATE4,w	; 247 is 0.988 seconds (0.01522% error from 10.986328s)
	xorlw	D'247'		; 
	btfss	STATUS,Z
	goto	ONE_SEC		; only update when at 10.988 seconds 
	clrf	CNT4
	clrf	UPDATE4		; reset counters
	bsf		PORTB,1		; cs high
	movlw	0x03		; RSTENERGY register address (ms bit low for read)
	call	ADD_RD		; load address

	movlw	D'40'		; 40-bit data
	call	DTA_RD		; read data, placed in REG_A, REG_B.... REG_E (MS....LS)
	bcf		FLAG,0
	btfss	FIRST4,0	; is it the first 10.988 seconds
	goto	CLR_REG
	btfsc	REG_A,7		; if set then negative
	bsf		FLAG,0		; negative flag	
	bsf		FLAG1,0		; data available flag
	goto	ADD_WATT	; total kWh

CLR_REG 
	bsf		FIRST4,0	; data ready on next 10.988 seconds
	clrf	REG_A
	clrf	REG_B
	clrf	REG_C
	clrf	REG_D
	clrf	REG_E

; total the energy into registers
ADD_WATT
	btfsc	REG_A,7		; if set then negative value
	goto	ONE_SEC		; bypass negative power tally

	movf	REG_E,w		; ls byte of energy
	addwf	KWHOUR_I,f	; add
	btfsc	STATUS,C
	call	OVER_H		; overflow to KWHOUR_H

	movf	REG_D,w		; next energy byte
	addwf	KWHOUR_H,f	; add
	btfsc	STATUS,C
	call	OVER_G		; overflow to KWHOUR_G

	movf	REG_C,w		; next energy byte
	addwf	KWHOUR_G,f	; add
	btfsc	STATUS,C
	call	OVER_F		; overflow to KWHOUR_F

	movf	REG_B,w		; next energy byte
	addwf	KWHOUR_F,f	; add
	btfsc	STATUS,C
	call	OVER_E		; overflow to KWHOUR_E

	movf	REG_A,w		; next energy byte
	addwf	KWHOUR_E,f	; add
	btfsc	STATUS,C
	call	OVER_D		; overflow into KWHOUR_D
	goto	ONE_SEC

; subroutine to roll carry through registers
OVER_H
	incfsz	KWHOUR_H,f		; add to next byte if overflow
	return
OVER_G
	incfsz	KWHOUR_G,f		; add to next byte if overflow
	return
OVER_F
	incfsz	KWHOUR_F,f		; add to next byte if overflow
	return
OVER_E
	incfsz	KWHOUR_E,f		; add to next byte if overflow
	return
OVER_D
	incfsz	KWHOUR_D,f		; add to next byte if overflow
	return
OVER_C
	incfsz	KWHOUR_C,f		; add to next byte if overflow
	return
OVER_B
	incfsz	KWHOUR_B,f		; add to next byte if overflow
	return
OVER_A
	incf	KWHOUR_A,f		; add to next byte if overflow
	return

; 1 second, now check for 6 minutes

ONE_SEC
	incf	UPDATE1,f	; increase counter 
	movf	UPDATE1,w
	xorlw	D'250'		; compare counter with 250 for 1 second
	btfss	STATUS,Z	; skip if equal
	goto	RECLAIM		; not 1 second
	clrf	UPDATE1		; seconds period count 
	incfsz	UPDATE2,f	; seconds counter LS byte for 6minutes
	goto	CK_360
	incf	UPDATE3,f	; seconds counter MS byte
						; ms byte of 168h or 360(Decimal) seconds = 6 minutes = 0.1hours 
CK_360
	btfss	UPDATE3,0	; if ls bit set (then 01) check LS byte
	goto	RECLAIM		; bypass hours update
	movlw	0x68		; 6 minutes
	xorwf	UPDATE2,w
	btfss	STATUS,Z
	goto	RECLAIM		; bypass increasing HOURS
	btfss	HOURS3,0	; if set then is the ms '1' of the 1869F for = 9999.9h or 1y 51days
	goto 	INC_HR		; increase HOURS
	movlw	0x86
	xorwf	HOURS2,w	; compare with 86h or 9999.9hours with HOURS3
	btfss	STATUS,Z	; if the same stop increasing
	goto	INC_HR
	movlw	0xA0
	xorwf	HOURS1,w	; final check for 1869Fh or 99999
	btfss	STATUS,Z
	goto	INC_HR		; not reached yet
	clrf	HOURS1		; cleared on 99999
	clrf	HOURS2
	clrf	HOURS3		; back to 0000.0 hours
	goto	RECLAIM

INC_HR

; check brownout counter first 

	movf	BRNCNT,w	; read value
	btfsc	STATUS,Z	; check for zero
	goto	ZROBRN
	decfsz	BRNCNT,f	; if not zero decrease
	goto	NOTZROBRN
ZROBRN
	btfsc	PORTA,1		; if SAG high can set RB0 high
	bsf		PORTB,0
NOTZROBRN

; increase hours		
	clrf	UPDATE2		; return to 0 minutes
	clrf	UPDATE3	
	incfsz	HOURS1,f
	goto	RECLAIM
	incfsz	HOURS2,f
	goto	RECLAIM
	incf	HOURS3,f

; end of interrupt reclaim w and status 

RECLAIM	

	bcf		SAG_CHNG,0	; reset flag for changed SAG level
	
; read SAG level at RA1

	btfsc	PORTA,1		; if clear increase SAG level
	goto	RES_SAG		; reset SAG level

; set voltage sag level higher (SAGLVL)
	
	bcf		PORTB,0		; clear (relay off) when porta,1 is low
	movlw	0x04		; set at 4 x 6 minutes delay 
	movwf	BRNCNT		; brown out counter

	btfss	SAG_X,1		; if last port read low then bypass
	goto	CK_Q
	bsf		SAG_CHNG,0	; set flag for changed level
	bsf		PORTB,1		; cs high
	movlw	0x11		; SAGLVL register
	call	ADD_LD

	movlw	EEPROM2		; sag level
	call	EEREAD		; get current value
	addwf	HYSTER,w	; add hysteresis
	movwf	ADD_ADE
	movlw	D'8'		; 8 bit data
	call	DAT_LD	
	goto	CK_Q
	
RES_SAG	
	btfsc	SAG_X,1		; if last port read high then bypass
	goto	CK_Q
	bsf		SAG_CHNG,0	; set flag for changed level
	bsf		PORTB,1		; cs high
	movlw	0x11		; SAGLVL register
	call	ADD_LD

	movlw	EEPROM2		; sag level
	call	EEREAD		; get current value
	movwf	ADD_ADE
	movlw	D'8'		; 8 bit data
	call	DAT_LD
	
CK_Q
	movf	PORTA,w		; read port
	movwf	SAG_X		; store	
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

; *********************************************************************************

; initialise ports

MAIN
	movlw	0x07		; comparators off
	movwf	CMCON		; I/O pins
	bcf		PORTB,0		; brownout relay drive off 
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00001000'	; port B outputs and inputs 
	movwf	TRISB		; port B data direction register
	movlw	B'11110010'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'10000011'	; TMR0 /16
	movwf	OPTION_REG	; port B pullups disabled
	bcf		STATUS,RP0	; memory bank 0
	
	call	INIT_RESET	; initial conditions
	bcf		FG_SW,0		; switch pressed flag 
	bsf		FLAG1,0		; data ready flag
	movlw	EEPROM7		; hysteresis store
	call	EEREAD		; get value	
	movwf	HYSTER		; hysteresis
	movlw	EEPROM8		; cost per kWh store
	call	EEREAD		; get value	
	movwf	DOLLAR		; value/kWh
	clrf	PORTB
	clrf	BRNCNT		; brownout counter cleared 

	movlw	0x05
	movwf	CNT4		; set as 5 seconds

; set up initial conditions for display

INIT_DISPLAY	
	clrf	PORTA		; RS, R/W, E low
 	movlw	0x10
	call	DELX
	
	movlw	B'00110010'	; initialise module
	movwf	PORTB
	nop
	bsf		PORTA,0		; enable high
	nop
	bcf		PORTA,0		; low
	call 	DELAYms

; set 4-bit operation

	bcf		PORTB,4		; display command
	bcf		PORTA,2		; register select low
	nop
	bsf		PORTA,0		; enable set
	nop
	bcf		PORTA,0		; enable clear
	call	DELAYms
	
	movlw	B'00101100'	; display function (4-bits, 2 lines, 5x10 dots)
	call	LOAD
	movlw	B'00001110'	; blinking off, cursor off
	call	LOAD
	movlw	B'00000001'	; display clear
	call	LOAD
	movlw	0x20
	call	DELX
	movlw	B'00000110'	; entry mode. cursor moves right, display not shifted
	call	LOAD
	movlw	B'00000001'	; display clear
	call	LOAD
	movlw	0x20
	call	DELX
	movlw	B'00001110'	; blinking off, cursor off
	call	LOAD
	
; set gain conditions for input amplifiers (GAIN)

	bcf		PORTB,2		; clock low
	bsf		PORTB,1		; cs high
	movlw	0x0A		; GAIN register address
	call	ADD_LD		; load address
	movlw	B'01010000'	; load CH2 gain =4, 0.25V FS, CH1 gain =1
	movwf	ADD_ADE
	movlw	0x08		; 8 bit data
	call	DAT_LD		; load gain data

; set mode register (MODE) 

	bsf		PORTB,1		; cs high
	movlw	0x06		; MODE register
	call	ADD_LD
	movlw	B'00000100'	; mode register values high pass filter enabled
	movwf	ADD_ADEL	; least sig byte
	movlw	B'00000000'	; ms byte
	movwf	ADD_ADE
	movlw	D'16'		; 16 bit data
	call	DAT_LD
	
RENEW					
	call	REN_DTA		; renew data for AD7756 SAG and Calibration values
	btfsc	PORTA,1		; if SAG high can set RB0 high 
	bsf		PORTB,0		; set relay on

; allow interrupts

SET_UP
	bsf		INTCON,T0IE	; set interrupt enable for TMR0 
	bsf		INTCON,GIE	; set global interrupt enable 
	goto	HOURS		; display values

; check display Mode

CK_MODE
	movf	DSP_MODE,w	; display mode 0 is normal kWh run mode
	btfsc	STATUS,Z
	goto	SG_LVL		; check SAG in mode 0
	movf	DSP_MODE,w	; display mode 1 is normal $ run mode
	xorlw	0x01
	btfss	STATUS,Z
	goto	CK_DTAFG	; check data flag
SG_LVL	
	btfsc	SAG_CHNG,0	; if set then SAG level changed
	goto	HOURS
RUN_MDE					; normal run and Dollar mode
	btfss	FLAG1,0		; wait for new data flag 
	goto	CK_SW		; check switch
	bcf		FLAG1,0		; clear flag
	goto	HOURS

; Check data ready and if APOS or APGAIN modes

CK_DTAFG
	btfsc	SAG_CHNG,0	; if set then SAG level changed
	goto	OTHER_MODES ; write SAG when in calibrate modes (SAG modes only)
	btfss	SEC_FLG,0	; wait for new seconds flag 
	goto	CK_SW		; check switch
	bcf		SEC_FLG,0	; clear flag
	goto	OTHER_MODES

; Display hours and wattage on line 1

HOURS
	movf	HOURS3,w	; ms hours
	movwf	BIN3
	movf	HOURS2,w
	movwf	BIN2
	movf	HOURS1,w
    movwf	BIN1
	clrf	BIN4		; ms byte
	call	BIN_BCD		; convert to BCD

	movlw	0x80		; address at first line far left
 	call	LOAD
	movlw	0x03		; first 3 characters
	movwf	STORE1
	movlw	EFOUR		; MSD of hours
	movwf	FSR
WAT4
	movlw	0x30		; check if 0
	subwf	INDF,w
	btfss	STATUS,Z
	goto	WAT2
	call	SPACE1		; leading 0's to space
	incf	FSR,f
	decfsz	STORE1,f
	goto	WAT4
	goto	WAT3
WAT2
	movf	INDF,w
	call	DRV_LCD
	incf	FSR,f
	decfsz	STORE1,f
	goto	WAT2
WAT3
	movf	INDF,w		
	call	DRV_LCD		; EONE
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	EZERO,w
	call	DRV_LCD
	movlw	A'h'		; h for hours
	call	DRV_LCD

WATTAGE 
	btfss	FIRST4,0	; is it the first 10.988 seconds
	goto	CL_REG
RET_WATT
	movf	REG_A,w
	movwf	REGA_S		; store
	movf	REG_B,w
	movwf	REGB_S		; store
	movf	REG_C,w
	movwf	REGC_S		; store
	btfss	FLAG,0		; if set then negative
	goto	CONV
	comf	REGA_S,f
	comf	REGB_S,f
	comf	REGC_S,f	; get reverse
	bsf		NEG_SGN,0
	goto	CONV
CL_REG
	clrf	REG_A
	clrf	REG_B
	clrf	REG_C
	clrf	REG_D
	clrf	REG_E
	goto	RET_WATT		
CONV
	movf	REGA_S,w	; ms byte of value
	movwf	BIN3		; next ms byte
	movf	REGB_S,w
	movwf	BIN2
	movf	REGC_S,w	; ls byte of value
	movwf	BIN1
	clrf	BIN4		; ms byte
	call	BIN_BCD		; convert to BCD

; check negative flag

;	call 	SPACE1
	btfss	NEG_SGN,0	; negative flag
	goto	POSSGN
	movlw	A'-'		; minus
	call	DRV_LCD
	goto	NO_PLUS
POSSGN
	call 	SPACE1		; space for +
NO_PLUS
	movlw	0x03		; first 3 characters
	movwf	STORE1
	clrf	NEG_SGN		; negative flag
	movlw	ESIX		; MSD of watts
	movwf	FSR
WAT8
	movlw	0x30		; check if 0
	subwf	INDF,w
	btfss	STATUS,Z
	goto	WAT10
	call	SPACE1		; leading 0's to space
	incf	FSR,f
	decfsz	STORE1,f
	goto	WAT8
	goto	WAT6
WAT10
	movf	INDF,w
	call	DRV_LCD
	incf	FSR,f
	decfsz	STORE1,f
	goto	WAT10
WAT6
	movf	ETHREE,w		
	call	DRV_LCD		; 
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	ETWO,w		
	call	DRV_LCD		;
	movf	EONE,w		
	call	DRV_LCD		;  
	movlw	A'W'		; Watts
	call	DRV_LCD

; line 2
; check for SAG
CK_SAG
	btfsc	PORTA,1		; if clear then SAG
	goto	OFF_SAG
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SAGDRV		; place SAG on display
	call	SPACE1
	goto	REDO
OFF_SAG	
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4		; position display

; kWh display (values in KWHOUR_I (ls) to KWHOUR_A (ms), 
; using KWHOUR_B, _C, _D, and _E divided by 65536 (65536 x 10.98632/2 = 360000/10) for number of Watts in 1Hr
; decimal point positioned for kWh

REDO
	movf	KWHOUR_B,w
	movwf	BIN4		; ms byte
	movf	KWHOUR_C,w
	movwf	BIN3
	movf	KWHOUR_D,w
	movwf	BIN2		; 
	movf	KWHOUR_E,w
	movwf	BIN1		; ls byte
	call	TIMES2		; double value
	call	BIN_BCD

; check overrange

	movf	BCD1,w
	andlw	0xF0		; get ms BCD digit
	btfsc	STATUS,Z	; if zero continue otherwise clear KWh storage
	goto	MD_KWCOST

;	clear 	KWHOUR_A to	KWHOUR_I

	movlw	0x77		; start of RAM addressing
	movwf	FSR			; pointer
KWH_CLEAR
	clrf	INDF		; clear values
	incf	FSR,f
	btfss	FSR,7		; if bit 7 set finished (77 to 7F cleared)
	goto	KWH_CLEAR

	goto	REDO		; redo BCD conversion

; mode check to display either kWh or $

MD_KWCOST
	movf	DSP_MODE,w	; display mode 0 is normal run mode
	xorlw	0x01
	btfss	STATUS,Z
	goto	VALID_DTA		; display kWh 	

; multiply dollar value by 1.28 (x 55 / 43)
	
	movf	DOLLAR,W	; current working Dollar value
	movwf	AARGB0		; 
	movlw	D'55'		; multiply by 55
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply by 55
	movlw	D'43'
	movwf	BARGB0
	call	DIV16_8		; divide 
	movf	AARGB1,W	; x 1.28 multiplied value
	movwf	BARGB1		; ls byte
	movf	AARGB0,W
	movwf	BARGB0		; ms byte
	
; multiply (dollar value x 1.28) x kWh value

	movf	KWHOUR_B,w	; ms byte
	movwf	AARGB0
	movf	KWHOUR_C,w
	movwf	AARGB1
	movf	KWHOUR_D,w
 	movwf	AARGB2
	movf	KWHOUR_E,w
	movwf	AARGB3		; ls byte
	call	MULT3216
	
; divide by 128 (actually 64 as kWh value is only half its value) (kWh is multiplied by 2 in kWh display)

	movlw	0x06		; divide by value
	movwf	LOOPCOUNT
DIV_128	
	bcf		STATUS,C	; carry cleared
	rrf		AARGB0,f
	rrf		AARGB1,f
	rrf		AARGB2,f
	rrf		AARGB3,f
	rrf		AARGB4,f
	rrf		AARGB5,f
	decfsz	LOOPCOUNT,f
	goto	DIV_128
		
; display cost kWh value 

	movf	AARGB2,w
	movwf	BIN4		; ms byte
	movf	AARGB3,w
	movwf	BIN3
	movf	AARGB4,w
	movwf	BIN2		; 
	movf	AARGB5,w
	movwf	BIN1		; ls byte

	call	BIN_BCD		; convert to BCD and ASCII
	
; display value

	movlw	A'$'		; Dollar sign
	call	DRV_LCD
	movlw	0x03		; first 3 characters
	movwf	STORE1
	movlw	EEIGHT		; MSD of Dollar
	movwf	FSR
DOLL8
	movlw	0x30		; check if 0
	subwf	INDF,w
	btfss	STATUS,Z
	goto	DOLL10
	call	SPACE1		; leading 0's to space
	incf	FSR,f
	decfsz	STORE1,f
	goto	DOLL8
	goto	DOLL6
DOLL10
	movf	INDF,w
	call	DRV_LCD
	incf	FSR,f
	decfsz	STORE1,f
	goto	DOLL10
DOLL6
	movf	EFIVE,w		
	call	DRV_LCD		; 
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	EFOUR,w		
	call	DRV_LCD		;
	movf	ETHREE,w
	call	DRV_LCD
	call	SPACE4
	btfss	FG_SW,0		; if set then a switch was pressed and so add delay 
	goto	CK_MODE
	movlw	0x50
	call	DELX		; add delay
	clrf	FG_SW		; reset flag
		
	goto	CK_SW
	
VALID_DTA
	movlw	0x04		; first 4 characters
	movwf	STORE1
	movlw	EEIGHT		; MSD of kWh
	movwf	FSR
KWAT8
	movlw	0x30		; check if 0
	subwf	INDF,w
	btfss	STATUS,Z
	goto	KWAT10
	call	SPACE1		; leading 0's to space
	incf	FSR,f
	decfsz	STORE1,f
	goto	KWAT8
	goto	KWAT6
KWAT10
	movf	INDF,w
	call	DRV_LCD
	incf	FSR,f
	decfsz	STORE1,f
	goto	KWAT10
KWAT6
	movf	EFOUR,w		
	call	DRV_LCD		; 
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	ETHREE,w		
	call	DRV_LCD		;
	movf	ETWO,w
	call	DRV_LCD
	movf	EONE,w
	call	DRV_LCD 
	movlw	A'k'		; write kWh
	call	DRV_LCD
	movlw	A'W'
	call	DRV_LCD
	movlw	A'h'
	call	DRV_LCD
	btfss	FG_SW,0		; if set then a switch was pressed and so add delay 
	goto	CK_MODE
	movlw	0x50
	call	DELX		; add delay
	clrf	FG_SW		; reset flag
	
; check switches
CK_SW
	btfsc	SAG_CHNG,0	; if SAG level changed
	goto	CK_MDES		; check Mode switch

	movlw	0xF0
	iorwf	PORTB,f		; set RB4-7 high
	
	call	ADD_CYC		; add cycles for delay
	btfss	PORTB,3		; if high then a switch pressed
	goto	CK_MODE		; return
CK_MDES
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,6		; mode switch
	call	ADD_CYC		; add cycles for delay
	btfss	PORTB,3		; if high then mode switch pressed
	goto	CK_RST		; check reset switch
	movlw	0x02
	call	DELX
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,6		; mode switch
	movlw	0x02
	call	DELX
	btfss	PORTB,3		; if high then switch pressed
	goto	CK_RST	
	bsf		FG_SW,0		; set switch closed flag
	bsf		MD_CHFG,0	; set mode change flag to extend delay
	incf	DSP_MODE,f	; next mode
	movf	DSP_MODE,W
	sublw	0x0A
	btfsc	STATUS,C	; if c zero then reset
	goto	MD_TMR
	clrf	DSP_MODE
MD_1
	btfsc	FG_SW,0		; if a switch was pressed 
	bsf		FLAG1,0		; set flag to rewrite display
	goto	CK_MODE

CK_RST
	clrf	MD_CHFG		; clear mode change flag
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,7		; reset switch
	call	ADD_CYC		; add cycles for delay
	btfss	PORTB,3		; if high then reset switch pressed
	goto	OTHER_MODES

	movlw	0x80		; delay time for reset
	movwf	RSTLOOP		; delay for reset to take effect
RST_LOOP
	movlw	0x05
	call	DELX
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,7		; reset switch
	call	ADD_CYC		; add cycles for delay	
	btfss	PORTB,3		; if high then reset switch pressed
	goto	OTHER_MODES	
	decfsz	RSTLOOP,f	; decrease delay loop counter
	goto	RST_LOOP

	call	INIT_RESET	; set initial conditions
	clrf	DSP_MODE
	bsf		FG_SW,0		; set switch closed flag
	bsf		FLAG1,0		; set flag to rewrite display
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPACE2
	movlw	A'R'		; write RESET
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'S'
	call	DRV_LCD
	movlw	A'E'		; 
	call	DRV_LCD
	movlw	A'T'
	call	DRV_LCD
	call	SPACE4
	call	SPACE4
	call	SPACE1
	movlw	0x50	
	call	DELX		; timer to display RESET
	goto	CK_MODE

; set mode timer when mode 01 selected

MD_TMR		
	movf	DSP_MODE,W	; mode
	xorlw	0x01		; cost mode
	btfss	STATUS,Z	; if cost mode set timer
	goto	CHK_MDE
	movlw	0x40		; set timer  adjust for best operation between normal and cal modes
	movwf	UP_MODE		; decreased 250 times/sec in interrupt
	goto	OTHER_MODES	
CHK_MDE
	movf	DSP_MODE,W
	xorlw	0x02
	btfss	STATUS,Z	; if mode 02 check timer
	goto	OTHER_MODES		
	movf	UP_MODE,W	; mode timer
	btfsc	STATUS,Z	; if reset clear mode
	clrf	DSP_MODE	; return to mode 0 if timed out
	
; check other display modes

OTHER_MODES
	movf	DSP_MODE,W
	movwf	MD_STR		; working DSP_MODE store
	btfsc	STATUS,Z	; if zero return
	goto	MD_1
	decf	MD_STR,f	; cost mode
	btfsc	STATUS,Z	; if zero return
	goto	MD_1
	decf	MD_STR,f	; cost cal mode
	btfsc	STATUS,Z	; if zero 
	goto	COST_MODE
	decf	MD_STR,f	; Power offset
	btfsc	STATUS,Z	; if zero 
	goto	APOS_CAL
	decf	MD_STR,f	; Power gain cal
	btfsc	STATUS,Z	; if zero 
	goto	APGAIN_CAL
	decf	MD_STR,f	; Phase CAL
	btfsc	STATUS,Z	; if zero 
	goto	PH_CAL
	decf	MD_STR,f	; SAG level
	btfsc	STATUS,Z	; if zero 
	goto	SAG_LVL
	decf	MD_STR,f	; SAG calibration
	btfsc	STATUS,Z	; if zero 
	goto	SAG_CAL	
	decf	MD_STR,f	; SAG Hysteresis
	btfsc	STATUS,Z	; if zero 
	goto	SAG_HYSTER
	decf	MD_STR,f	; SAG_CYCLES
	btfsc	STATUS,Z	; if zero 
	goto	SAG_CYCLE

	clrf	DSP_MODE
	goto	MD_1

; SAG LEVEL MODE
SAG_LVL	
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM2
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	WORK_VAL
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	SAG_DN
	movf	WORK_VAL,w	; current SAG level
	sublw	H'37'		; maximum value 
	btfss	STATUS,C	; if 0 then at max
	goto	MAX_SAG
	incf	WORK_VAL,f
	goto	WRITE_SAGLVL
MAX_SAG
	movlw	H'38'		; max sag value
	movwf	WORK_VAL		
WRITE_SAGLVL
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM2
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
		
	call 	REN_DTA		; renew data
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_SAGLVL	

SAG_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_SAGLVL
	movf	WORK_VAL,W	; check value
	btfsc	STATUS,Z	; if zero do not decrease
	goto	CK_MODE
	decf	WORK_VAL,f
	goto	WRITE_SAGLVL; write SAG level to eeprom
	
DISP_SAGLVL
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPACE1
	movlw	A'S'		; write SAG
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'G'
	call	DRV_LCD
	call	SPACE1	
	movlw	A'L'		; write LEVEL
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'V'
	call	DRV_LCD
	movlw	A'E'		; 
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	call	SPACE4
	call	SPACE2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	btfsc	PORTA,1		; if clear then SAG
	goto	CLR_SAG1	; SAG off

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SAGDRV		; write sag
	call	SPACE4
	goto	DIS_SAG		; display SAG LEVEL
CLR_SAG1
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4
	call	SPACE3		; position display
DIS_SAG	
	movf	WORK_VAL,W	; current working SAG Level
	movwf	AARGB0		; 
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROMA		; multiplier value
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply by EEPROMA value	
	movlw	D'11'
	movwf	BARGB0
	call	DIV16_8		; divide by 11 (calibrate SAG LEVEL for voltage level on display)
	movf	AARGB1,W	; multiplied value
	movwf	BIN1		; ls binary value
	movf	AARGB0,W	; multiplied value
	movwf	BIN2		; ls binary value
	call	BIN_BCD34	; binary to BCD (clears BIN2,3,4)
	movf	ETWO,W		; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLK
	movf	ETWO,W		; not zero so display	
	call	DRV_LCD
	goto	MID_SAGLVL
MS_BLK
	call	SPACE1		; space instead of 0
	movf	EONE,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_BLK
MID_SAGLVL
	movf	EONE,W		; not zero so display	
	call	DRV_LCD
	goto	LS_SAGLVL
LS_BLK
	call	SPACE1		; space instead of 0	
LS_SAGLVL
	movf	EZERO,W		; ls digit
	call	DRV_LCD
	call	SPACE1
	movlw	A'V'		; Volts
	call	DRV_LCD
	movlw	A'o'		; Volts
	call	DRV_LCD
	movlw	A'l'		; Volts
	call	DRV_LCD
	movlw	A't'		; Volts
	call	DRV_LCD
	movlw	A's'		; Volts
	call	DRV_LCD

END_CHNG
	movlw	0x50
	btfsc	MD_CHFG,0	; if mode changed flag set add extra delay	
	movlw	0x80
	call	DELX		; add delay
	clrf	FG_SW		; reset flag
	clrf	MD_CHFG		; mode change flag
	goto	CK_MODE

; SAG CYCLE MODE
SAG_CYCLE	
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM1
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	WORK_VAL
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	CYC_DN
	movf	WORK_VAL,w	; current SAG level
	sublw	D'254'		; maximum value 255 cycles
	btfss	STATUS,C	; if 0 then at max
	goto	MAX_CYC
	incf	WORK_VAL,f
	goto	WRITE_SAGCYC
MAX_CYC
	movlw	D'255'		; max Cycles value
	movwf	WORK_VAL		
WRITE_SAGCYC
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM1
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
		
	call 	REN_DTA		; renew data
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_SAGCYC	

CYC_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_SAGCYC
	movf	WORK_VAL,W	; check value
	btfsc	STATUS,Z	; if zero do not decrease
	goto	CK_MODE
	decfsz	WORK_VAL,f
	goto	WRITE_SAGCYC; write SAG CYCLEs to eeprom
	incf	WORK_VAL,f	; not zero
	goto	WRITE_SAGCYC
DISP_SAGCYC
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPACE1
	movlw	A'S'		; write SAG
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'G'
	call	DRV_LCD
	call	SPACE1	
	movlw	A'H'		; write Half
	call	DRV_LCD
	movlw	A'A'		; 
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	movlw	A'F'		; 
	call	DRV_LCD
	call	SPACE1
	movlw	A'C'
	call	DRV_LCD
	movlw	A'Y'		; write CYCLES
	call	DRV_LCD
	movlw	A'C'
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'S'
	call	DRV_LCD

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	btfsc	PORTA,1		; if clear then SAG
	goto	CLR_CYC1	; SAG off

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SAGDRV		; sag on display
	call	SPACE1
	call	SPACE3
	goto	DIS_CYC		; display SAG CYCLES
CLR_CYC1
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4
	call	SPACE3		; position display
DIS_CYC
	movf	WORK_VAL,W	; current working SAG Level
	movwf	BIN1		; ls binary value
	call	BIN_BCD24	; binary to BCD Clears BIN2,3,4
	movf	ETWO,W		; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLK1
	movf	ETWO,W		; not zero so display	
	call	DRV_LCD
	goto	MID_SAGCYC
MS_BLK1
	call	SPACE1		; space instead of 0
	movf	EONE,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_BLK1
MID_SAGCYC
	movf	EONE,W		; not zero so display	
	call	DRV_LCD
	goto	LS_SAGCYC
LS_BLK1
	call	SPACE1		; space instead of 0	
LS_SAGCYC
	movf	EZERO,W		; ls digit
	call	DRV_LCD
	call	SPACE4
	call	SPACE2
		
	goto	END_CHNG
	
; SAG HYSTERESIS
SAG_HYSTER
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM7
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	HYSTER
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	HYST_DN
	movf	HYSTER,w	; current SAG Hysteresis 
	sublw	H'37'		; maximum value 
	btfss	STATUS,C	; if 0 then at max
	goto	MAX_HYS
	incf	HYSTER,f
	goto	WRITE_HYSTER
MAX_HYS
	movlw	H'38'		; max value
	movwf	HYSTER		
WRITE_HYSTER
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM7
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	HYSTER,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_HYSTER	

HYST_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_HYSTER
	movf	HYSTER,W	; check value
	btfsc	STATUS,Z	; if zero do not decrease
	goto	CK_MODE
	decf	HYSTER,f
	goto	WRITE_HYSTER; write SAG level to eeprom
	
DISP_HYSTER
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPACE1
	movlw	A'S'		; write SAG
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'G'
	call	DRV_LCD
	call	SPACE1	
	movlw	A'H'		; write HYSTERESIS
	call	DRV_LCD
	movlw	A'Y'
	call	DRV_LCD
	movlw	A'S'
	call	DRV_LCD
	movlw	A'T'		; 
	call	DRV_LCD
	movlw	A'E'		; write HYSTERESIS
	call	DRV_LCD
	movlw	A'R'
	call	DRV_LCD
	movlw	A'E'		; write HYSTERESIS
	call	DRV_LCD
	movlw	A'S'
	call	DRV_LCD
	movlw	A'I'
	call	DRV_LCD
	movlw	A'S'		
	call	DRV_LCD
	call	SPACE1

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	btfsc	PORTA,1		; if clear then SAG
	goto	CLR_HYS1	; SAG off

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SAGDRV		; sag on display
	call	SPACE4
	goto	DIS_SAG2	; display SAG LEVEL
CLR_HYS1
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4
	call	SPACE3
DIS_SAG2	
	movf	HYSTER,W	; current working SAG Hysteresis
	movwf	AARGB0		; 
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROMA		; multiplier value
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply by EEPROMA value	
	movlw	D'11'
	movwf	BARGB0
	call	DIV16_8		; divide by 11 (calibrate SAG LEVEL for voltage level on display)
	movf	AARGB1,W	; multiplied value
	movwf	BIN1		; ls binary value
	movf	AARGB0,W	; multiplied value
	movwf	BIN2		; ls binary value
	call	BIN_BCD34
	movf	ETWO,W		; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLK3
	movf	ETWO,W		; not zero so display	
	call	DRV_LCD
	goto	MID_HYSTER
MS_BLK3
	call	SPACE1		; space instead of 0
	movf	EONE,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_BLK3
MID_HYSTER
	movf	EONE,W		; not zero so display	
	call	DRV_LCD
	goto	LS_HYSTER
LS_BLK3
	call	SPACE1		; space instead of 0	
LS_HYSTER
	movf	EZERO,W		; ls digit
	call	DRV_LCD
	call	SPACE1
	movlw	A'V'		; Volts
	call	DRV_LCD
	movlw	A'o'		; Volts
	call	DRV_LCD
	movlw	A'l'		; Volts
	call	DRV_LCD
	movlw	A't'		; Volts
	call	DRV_LCD
	movlw	A's'		; Volts
	call	DRV_LCD
	
	goto	END_CHNG

; $/kWh displayed instead of kWh

COST_MODE
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM8
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	DOLLAR
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	DOLL_DN
	movf	DOLLAR,w	; current Dollar value 
	sublw	D'254'		; maximum value $25.5
	btfss	STATUS,C	; if 0 then at max
	goto	MAX_DOLL
	incf	DOLLAR,f
	goto	WRITE_DOLLAR
MAX_DOLL
	movlw	D'255'		; max value
	movwf	DOLLAR	
WRITE_DOLLAR
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM8
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	DOLLAR,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_DOLLAR	

DOLL_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_DOLLAR
	movf	DOLLAR,W	; check value
	btfsc	STATUS,Z	; if zero do not decrease
	goto	CK_MODE
	decf	DOLLAR,f
	goto	WRITE_DOLLAR; write $value to eeprom
	
DISP_DOLLAR
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPACE1
	movlw	A'C'		; write CENTS
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'N'
	call	DRV_LCD
	movlw	A'T'
	call	DRV_LCD
	movlw	A'S'
	call	DRV_LCD
	call	SPACE1	
	movlw	A'/'		; write per
	call	DRV_LCD
	call	SPACE1
	movlw	A'k'		; write kWh
	call	DRV_LCD
	movlw	A'W'
	call	DRV_LCD
	movlw	A'h'		; 
	call	DRV_LCD
	call	SPACE4

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4
	call	SPACE2
	movf	DOLLAR,W	; current working Dollar value
	movwf	BIN1		; ls binary value
	call	BIN_BCD24
	movf	ETWO,W		; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLKA
	movf	ETWO,W		; not zero so display	
	call	DRV_LCD
	goto	MID_DOLL
MS_BLKA
	call	SPACE1		; space instead of 0
MID_DOLL
	movf	EONE,W		; ls digit
	call	DRV_LCD
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	EZERO,W		; after decimal points digit
	call	DRV_LCD
	call	SPACE1
	movlw	A'C'		; Cents
	call	DRV_LCD
	movlw	A'e'		; 
	call	DRV_LCD
	movlw	A'n'		; 
	call	DRV_LCD
	movlw	A't'		; 
	call	DRV_LCD
	movlw	A's'		; 
	call	DRV_LCD
	
	goto	END_CHNG

; phase calibration	

PH_CAL
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM9
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	WORK_VAL
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	PHCAL_DN

; increment value
; Check maximum
	btfsc	WORK_VAL,5	; if set then negative
	goto	BY_PH		; negative max check
	movf	WORK_VAL,w
	sublw	B'00011110'	; maximum
	btfss	STATUS,C	; if c is 0 then no increment
	goto	WRITE_PHCAL
BY_PH
	incf	WORK_VAL,w	
	andlw	B'00111111'	; reject 6 and 7 bits
	movwf	WORK_VAL

WRITE_PHCAL
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM9
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
		
	call 	REN_DTA		; renew data
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_PHCAL

PHCAL_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_PHCAL
	
; decrement value
; check minimum
	btfss	WORK_VAL,5	; if negative check minimum
	goto	VALU_PH
	movf	WORK_VAL,w	;
	andlw	B'00011111'	; check absolute value
	xorlw	0x01
	btfsc	STATUS,Z	; check for 0
	goto	DISP_PHCAL
VALU_PH
	decf	WORK_VAL,w
	andlw	B'00111111'	; reject 6 and 7 bits
	movwf	WORK_VAL
	goto	WRITE_PHCAL

DISP_PHCAL
	movlw	0x80		; address at first line far left
	call	LOAD
	call	SPACE1
	movlw	A'P'		; write PHASE
	call	DRV_LCD
	movlw	A'H'
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'S'		; write PHASE
	call	DRV_LCD
	movlw	A'E'		; write PHASE
	call	DRV_LCD
	call	SPACE2
	call	SPACE4
	call	SPACE4

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4
	call	SPACE2		; position display
	btfss	WORK_VAL,5	; if set negative
	goto	NN_PHASE
	movlw	A'-'		; minus sign
	call	DRV_LCD
	goto	N_PHASE
NN_PHASE
	call	SPACE1
	goto	PHCAL_V
N_PHASE
	comf	WORK_VAL,w	; complement value
	addlw	0x01		; add 1
	goto	PCAL_V
PHCAL_V
	movf	WORK_VAL,W	; current working 
PCAL_V
	andlw	B'00011111'	; remove sign
	movwf	AARGB0		; 
	movlw	D'252'		; multiply by 252
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply 

; shift left for multiply by 8
	bcf		STATUS,C	; carry cleared
	rlf		AARGB1,f	; shift left
	rlf		AARGB0,f
	rlf		AARGB1,f	; shift left
	rlf		AARGB0,f	
	rlf		AARGB1,f	; shift left
	rlf		AARGB0,f
	movlw	D'45'
	movwf	BARGB0
	call	DIV16_8		; divide (set display to show 4.47us/increment (x252 x8/45=44.8) for 4.47us
	movf	AARGB1,W	; multiplied value
	movwf	BIN1		; ls binary value
	movf	AARGB0,w	; ms value
	movwf	BIN2		; 
	call	BIN_BCD34
	movf	ETHREE,W	; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLKP
	movf	ETHREE,W	; not zero so display	
	call	DRV_LCD
	goto	MID_PHCAL
MS_BLKP
	call	SPACE1		; space instead of 0
	movf	ETWO,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_BLKP
MID_PHCAL
	movf	ETWO,W		; not zero so display	
	call	DRV_LCD
	goto	LS_PHCAL
LS_BLKP
	call	SPACE1		; space instead of 0	
LS_PHCAL
	movf	EONE,W		; ls digit
	call	DRV_LCD
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	EZERO,W		; digit after dp
	call	DRV_LCD
	call	SPACE1
	movlw	H'E4'		; micro symbol u
	call	DRV_LCD
	movlw	A's'		; s
	call	DRV_LCD
	call	SPACE1
	
	goto	END_CHNG	

; offset calibrate (EEPROM3,4 (ms ls) WORK_VAL1, WORK_VAL 
APOS_CAL
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM3		; ms byte of stored calibration number
	call	EEREAD		; get value	
	movwf	WORK_VAL1	; ms working value
	movlw	EEPROM4		; ls byte of stored calibration number
	call	EEREAD		; get value	
	movwf	WORK_VAL	; ls working value
	bsf		INTCON,GIE	; set global interrupt enable
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	APOS_DN

; increment value
; Check maximum
	btfsc	WORK_VAL1,3	; if set then negative
	goto	BY_APOS		; negative max check
	movf	WORK_VAL1,w
	xorlw	B'00000111'	; check max on ms byte
	btfss	STATUS,Z	; if z is 1 then check ls byte
	goto	BY_APOS		; not at max so can increase
	incf	WORK_VAL,w
	btfsc	STATUS,Z	; if zero do not increase
	goto	WRITE_APOS
BY_APOS
	movlw	0x01
	addwf	WORK_VAL,f	; add 1
	btfsc	STATUS,C	; if carry set increase ms byte
	
	incf	WORK_VAL1,f

WRITE_APOS
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM4
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL,w
	call	EWRITE
	movlw	EEPROM3
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL1,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
		
	call 	REN_DTA		; renew data
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_APOS

APOS_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_APOS
	
; decrement value
; check minimum
	btfss	WORK_VAL1,3	; if negative check minimum
	goto	VALU_APOS
	movf	WORK_VAL1,w	; ms byte of value
	andlw	B'00000111'	; check absolute value
	btfss	STATUS,Z	; check for 0
	goto	VALU_APOS
	movf	WORK_VAL,w	; ls byte
	btfsc	STATUS,Z	; if zero do not decrement
	goto	DISP_APOS
VALU_APOS
	movf	WORK_VAL,w	; ls byte
	movwf	AARGB2
	movf	WORK_VAL1,w	; ms byte
	movwf	AARGB1
	clrf	BARGB0
	movlw	0x01
	movwf	BARGB1
	call	SUBTRACT	; take 1 away
	movf    AARGB1,w
	movwf	WORK_VAL1
	movf	AARGB2,w
	movwf	WORK_VAL
	goto	WRITE_APOS

; display
DISP_APOS
	movlw	0x80		; address at first line far left
	call	LOAD
	movlw	A'O'		; write OFFSET
	call	DRV_LCD
	movlw	A'F'
	call	DRV_LCD
	movlw	A'F'
	call	DRV_LCD
	movlw	A'S'		; write OFFSET
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'T'		; 
	call	DRV_LCD
	call	SPACE1
RET_APGAIN
	movf	REG_A,w
	movwf	REGA_S		; store
	movf	REG_B,w
	movwf	REGB_S		; store
	movf	REG_C,w
	movwf	REGC_S		; store
	btfss	FLAG,0		; if set then negative
	goto	CONV_APOS
	
	comf	REGA_S,f
	comf	REGB_S,f
	comf	REGC_S,f	; get reverse
	bsf		NEG_SGN,0
CONV_APOS
	movf	REGA_S,w	; ms byte of value
	movwf	BIN3		; next ms byte
	movf	REGB_S,w
	movwf	BIN2
	movf	REGC_S,w	; ls byte of value
	movwf	BIN1
	clrf	BIN4		; ms byte
	call	BIN_BCD		; convert to BCD

; check negative flag

	btfss	NEG_SGN,0	; negative flag
	goto	POSS_APOS
	movlw	A'-'		; minus
	call	DRV_LCD
	goto	NO_APOS
POSS_APOS
	call 	SPACE1		; space for +
NO_APOS
	clrf	NEG_SGN		; negative flag
	movlw	0x03		; first 3 characters
	movwf	STORE1
	movlw	ESIX		; MSD of watts
	movwf	FSR
APOS8
	movlw	0x30		; check if 0
	subwf	INDF,w
	btfss	STATUS,Z
	goto	APOS10
	call	SPACE1		; leading 0's to space
	incf	FSR,f
	decfsz	STORE1,f
	goto	APOS8
	goto	APOS6
APOS10
	movf	INDF,w
	call	DRV_LCD
	incf	FSR,f
	decfsz	STORE1,f
	goto	APOS10
APOS6
	movf	ETHREE,w		
	call	DRV_LCD		; 
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	ETWO,w		
	call	DRV_LCD		;
	movf	EONE,w		
	call	DRV_LCD		;  
	movlw	A'W'		; Watts
	call	DRV_LCD

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4
	movlw	A'<'		; <
	call	DRV_LCD
	btfss	WORK_VAL1,3	; if set negative
	goto	NN_APOS
	movlw	A'-'		; minus sign
	call	DRV_LCD
	goto	N_APOS
NN_APOS
	call	SPACE1
	goto	APOS_V
N_APOS
	comf	WORK_VAL1,f	; complement value
	comf	WORK_VAL,w
	addlw	0x01		; add 1
	movwf	WORK_VAL
	btfsc	STATUS,Z	; if zero increment
	incf	WORK_VAL1,f
APOS_V
	movf	WORK_VAL1,W	; current working 
	andlw	B'00000111'	; remove sign

; convert to decimal
	
	movwf	BIN2		; ms binary value
	movf	WORK_VAL,w	; ls value
	movwf	BIN1		; 
	call	BIN_BCD34
; value to display
	
	movlw	0x03		; first 3 characters
	movwf	STORE1
	movlw	ETHREE		; MSD
	movwf	FSR
APOS_8
	movlw	0x30		; check if 0
	subwf	INDF,w
	btfss	STATUS,Z
	goto	APOS_10
	call	SPACE1		; leading 0's to space
	incf	FSR,f
	decfsz	STORE1,f
	goto	APOS_8
	goto	APOS_6
APOS_10
	movf	INDF,w
	call	DRV_LCD
	incf	FSR,f
	decfsz	STORE1,f
	goto	APOS_10
APOS_6
	movf	EZERO,w		
	call	DRV_LCD		; 
	movlw	A'>'		; >
	call	DRV_LCD
	call	SPACE3
	call	SPACE2

	goto	END_CHNG

; gain calibrate (EEPROM5,6 (ms,ls) WORK_VAL1 WORK_VAL	

APGAIN_CAL
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM5		; ms byte of stored calibration number
	call	EEREAD		; get value	
	movwf	WORK_VAL1	; ms working value
	movlw	EEPROM6		; ls byte of stored calibration number
	call	EEREAD		; get value	
	movwf	WORK_VAL	; ls working value
	bsf		INTCON,GIE	; set global interrupt enable
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	APGAIN_DN

; increment value
; Check maximum
	btfsc	WORK_VAL1,3	; if set then negative
	goto	BY_APGAIN	; negative max check
	movf	WORK_VAL1,w
	xorlw	B'00000111'	; check max on ms byte
	btfss	STATUS,Z	; if z is 1 then check ls byte
	goto	BY_APGAIN	; not at max so can increase
	incf	WORK_VAL,w
	btfsc	STATUS,Z	; if zero do not increase
	goto	WRITE_APGAIN
BY_APGAIN
	movlw	0x01
	addwf	WORK_VAL,f	; add 1
	btfsc	STATUS,C	; if carry set increase ms byte
	
	incf	WORK_VAL1,f

WRITE_APGAIN
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM6
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL,w
	call	EWRITE
	movlw	EEPROM5
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL1,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
		
	call 	REN_DTA		; renew data
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_APGAIN

APGAIN_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_APGAIN
	
; decrement value
; check minimum
	btfss	WORK_VAL1,3	; if negative check minimum
	goto	VALU_APGAIN
	movf	WORK_VAL1,w	; ms byte of value
	andlw	B'00000111'	; check absolute value
	btfss	STATUS,Z	; check for 0
	goto	VALU_APGAIN
	movf	WORK_VAL,w	; ls byte
	btfsc	STATUS,Z	; if zero do not decrement
	goto	DISP_APGAIN
VALU_APGAIN
	movf	WORK_VAL,w	; ls byte
	movwf	AARGB2
	movf	WORK_VAL1,w	; ms byte
	movwf	AARGB1
	clrf	BARGB0
	movlw	0x01
	movwf	BARGB1
	call	SUBTRACT	; take 1 away
	movf    AARGB1,w
	movwf	WORK_VAL1
	movf	AARGB2,w
	movwf	WORK_VAL
	goto	WRITE_APGAIN

; display
DISP_APGAIN
	movlw	0x80		; address at first line far left
	call	LOAD
	call	SPACE1
	movlw	A'P'		; write POWER
	call	DRV_LCD
	movlw	A'O'
	call	DRV_LCD
	movlw	A'W'
	call	DRV_LCD
	movlw	A'E'		; write 
	call	DRV_LCD
	movlw	A'R'
	call	DRV_LCD
	call	SPACE1
	goto	RET_APGAIN	; got to section of APOS where code is the same

; SAG level Calibration	

SAG_CAL
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROM2		; sag level setting
	call	EEREAD		; get value	
	movwf	WORK_VAL
	movlw	EEPROMA		; sag level calibration
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	WORK_VAL1
	call	UP_SWITCH	; check if up switch closed
	btfss	PORTB,3		; if high then up switch pressed
	goto	SAGCAL_DN
	movf	WORK_VAL1,w	; current SAG level calibration
	sublw	D'150'		; maximum value 
	btfss	STATUS,C	; if 0 then at max
	goto	MAX_SAGCAL
	incf	WORK_VAL1,f
	goto	WRITE_SAGCAL
MAX_SAGCAL
	movlw	D'151'		; max cal value
	movwf	WORK_VAL1		
WRITE_SAGCAL
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROMA
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	WORK_VAL1,w
	call	EWRITE
	bsf		INTCON,GIE	; set global interrupt enable 
	bsf		FG_SW,0		; set switch closed flag
	goto	DISP_SAGLVLCAL	

SAGCAL_DN	
	call	DOWN_SWITCH	; check if down switch pressed
	btfss	PORTB,3		; if high then down switch pressed
	goto	DISP_SAGLVLCAL
	movf	WORK_VAL1,W	; check value
	btfsc	STATUS,Z	; if zero do not decrease
	goto	CK_MODE
	decf	WORK_VAL1,f
	goto	WRITE_SAGCAL; write SAG level cal to eeprom
	
DISP_SAGLVLCAL
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPACE1
	movlw	A'S'		; write SAG
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'G'
	call	DRV_LCD
	call	SPACE1	
	movlw	A'L'		; write LEVEL
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'V'
	call	DRV_LCD
	movlw	A'E'		; 
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	call	SPACE1
	movlw	A'C'		; write CAL
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	call	SPACE2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	btfsc	PORTA,1		; if clear then SAG
	goto	CLR_SAGCAL	; SAG off

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SAGDRV		; write sag
	call	SPACE4
	goto	DIS_SAGCAL	; display SAG LEVEL
CLR_SAGCAL
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	call	SPACE4
	call	SPACE3		; position display
DIS_SAGCAL	
	movf	WORK_VAL,W	; current working SAG Level
	movwf	AARGB0		; 
	bcf		INTCON,GIE	; clear global interrupt enable 
	movlw	EEPROMA		; multiplier value
	call	EEREAD		; get value	
	bsf		INTCON,GIE	; set global interrupt enable 
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply by EEPROMA value	
	movlw	D'11'
	movwf	BARGB0
	call	DIV16_8		; divide by 11 (calibrate SAG LEVEL for voltage level on display)
	movf	AARGB1,W	; multiplied value
	movwf	BIN1		; ls binary value
	movf	AARGB0,W	; multiplied value
	movwf	BIN2		; ls binary value
	call	BIN_BCD34	; binary to BCD (clears BIN2,3,4)
	movf	ETWO,W		; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLKCAL
	movf	ETWO,W		; not zero so display	
	call	DRV_LCD
	goto	MID_SAGCAL
MS_BLKCAL
	call	SPACE1		; space instead of 0
	movf	EONE,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_BLKCAL
MID_SAGCAL
	movf	EONE,W		; not zero so display	
	call	DRV_LCD
	goto	LS_SAGLVLCAL
LS_BLKCAL
	call	SPACE1		; space instead of 0	
LS_SAGLVLCAL
	movf	EZERO,W		; ls digit
	call	DRV_LCD
	call	SPACE1
	movlw	A'V'		; Volts
	call	DRV_LCD
	movlw	A'o'		; Volts
	call	DRV_LCD
	movlw	A'l'		; Volts
	call	DRV_LCD
	movlw	A't'		; Volts
	call	DRV_LCD
	movlw	A's'		; Volts
	call	DRV_LCD

	goto	END_CHNG	



;***************************************************************************
; subroutines

; down switch check
DOWN_SWITCH
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,5		; down switch
DY_SW
	movlw	0x02
	call	DELX
	return

; Up switch check
UP_SWITCH
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,4		; up switch
	goto	DY_SW

; Initial conditions all banks

INIT_RESET
;	clear	KWHOUR_A to	KWHOUR_I
;	clear	UPDATE1 to	UPDATE4
;	clear 	HOURS1 to HOURS3
;	clear	DSP_MODE	; display mode
;	clear	FLAG		; negative flag
;	clear	FLAG1		; data ready flag
;	clear	NEG_SGN		; negative flag
;	clear	FIRST4		; first 10 -seconds flag
;	clear	CNT4		; 10 -seconds flag

	movlw	0x20		; start of RAM addressing
	movwf	FSR			; pointer
LOOP_CLEAR
	clrf	INDF		; clear values
	incf	FSR,f
	btfss	FSR,7		; if bit 7 set finished (20 to 7F cleared)
	goto	LOOP_CLEAR
	bsf		SAG_CHNG,0	; set flag for changed level
;	bsf		PORTB,1		; CS bar set
	return

; place SAG on display

SAGDRV
	movlw	A'S'		; write SAG
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'G'
	call	DRV_LCD
	return

; update sag and calibration values

REN_DTA

; set phase correction (PHCAL)

	bsf		PORTB,1		; cs high
	movlw	0x0C		; PHCAL register
	call	ADD_LD
	movlw	EEPROM9		; phase value
	call	COM_RUN		; common routine
	
; set cycle count for voltage sag (SAGCYC)

	bsf		PORTB,1		; cs high
	movlw	0x0F		; SAGCYC register
	call	ADD_LD
	movlw	EEPROM1		; sag cycles
	call	COM_RUN		; common routine

; set voltage sag level (SAGLVL)

	bsf		PORTB,1		; cs high
	movlw	0x11		; SAGLVL register
	call	ADD_LD
	movlw	EEPROM2		; sag level
	call	COM_RUN		; common routine

; set Active Power Gain (APGAIN)

	bsf		PORTB,1		; cs high
	movlw	0x0B		; APGAIN register address 
	call	ADD_LD		; load address
	movlw	EEPROM5
	call	EEREAD
	movwf	ADD_ADE		; ms byte of data
	movlw	EEPROM6
	call	COM_16
	
; set Active Power OffSet (APOS)

	bsf		PORTB,1		; cs high
	movlw	0x0D		; APOS register address 
	call	ADD_LD		; load address
	movlw	EEPROM3
	call	EEREAD
	movwf	ADD_ADE		; ms byte of data
	movlw	EEPROM4
COM_16
	call	EEREAD
	movwf	ADD_ADEL	; ls byte of data
	movlw	D'16'
	call	DAT_LD		; write data
	return

; common routine

COM_RUN 
	call	EEREAD		; get current value
	movwf	ADD_ADE
	movlw	D'8'		; 8 bit data
	call	DAT_LD
	return

; load address to ADE7756. 

ADD_LD					; address including command for data write (load)
	iorlw	B'10000000'	; set ms 3 bits to 100 for write operation
ADD_RD					; address including command for data read
	movwf	ADD_ADE
	movlw	0x08		; number of bits

; data load into ADE7756

DAT_LD
	movwf	STORE2
	bcf		PORTB,1		; CS low
NXT_BT	
	bcf		PORTA,3		; data bit low initially
	rlf		ADD_ADEL,f	; ls byte if used
	rlf		ADD_ADE,f	; get ms bit
	btfsc	STATUS,C	; check status bit
	bsf		PORTA,3		; set data bit if carry set
	bsf		PORTB,2		; clock high
	bcf		PORTB,2		; clock data
	decfsz	STORE2,f	; next bit
	goto	NXT_BT		; do all bits
	return

; read data

DTA_RD	
	movwf	STORE2		; store count value
NXT_RD
	bsf		PORTB,2		; clock high
	bcf		PORTB,2		; clock low
	bcf		STATUS,C	; clear carry
	btfsc	PORTA,4		; check data
	bsf		STATUS,C	; set carry if data set
	rlf		REG_E,f		; ls register
	rlf		REG_D,f
	rlf		REG_C,f		; 
	rlf		REG_B,f
	rlf		REG_A,f		; data to ms register

	decfsz	STORE2,f	; do all bits
	goto	NXT_RD
	return

; add delay in cycles for switches
ADD_CYC					; add cycles for delay
	nop
	nop
	return

; add space in display

SPACE4
	movlw	0x20		; space
	call	DRV_LCD
SPACE3
	movlw	0x20		; space
	call	DRV_LCD
SPACE2
	movlw	0x20		; space
	call	DRV_LCD
SPACE1
	movlw	0x20		; space
	call	DRV_LCD
	return

; delay timer

DELAY_TM
	movlw	0x0F
DELX	
	movwf	TIME_I		; initial timer 
TIM_CON
	call	DELAYms
	decfsz	TIME_I,f
	goto	TIM_CON		; continue timer

; nominal 10ms delay loop

DELAYms
	movlw	0x0A		; ~10ms delay with 4 MHz clock
	movwf	STORE1		; STORE1 is number of loops value
LOOP1
	movlw	0xA0
	movwf	STORE2		; STORE2 is internal loop value	
LOOP2
	decfsz	STORE2,f
	goto	LOOP2
	decfsz	STORE1,f
	goto	LOOP1		; decrease till STORE1 is zero
	return

; preload display commands (4-bit) 

LOAD
	call	LEV_UPPER	; upper bits	

	bcf		PORTA,2		; register select low
	
	call	LEV_LOWER	; get lower bits
	bcf		PORTA,2		; register select low
	goto	BUS_CK		; check busy flag
	
; driving the LCD module with display data

DRV_LCD	
	call	LEV_UPPER	; get upper bits
	bsf		PORTA,2		; register select
	
	call	LEV_LOWER	; get lower bits
	bsf		PORTA,2		; register select
	nop

; check busy flag

BUS_CK
	bsf		PORTA,0		; enable high
	nop
	bcf		PORTA,0		; enable low	
	movlw	D'100'
	movwf	BUS_TME
BUS_Y
	decfsz	BUS_TME,f
	goto	BUS_Y
	return

; upper bits

LEV_UPPER
	movwf	D_STO		; store data
	movf	PORTB,w		; get portB levels
	andlw	0x0F		; clear upper bits keep lower bits
	movwf	PORTB		; upper bits cleared
	movf	D_STO,w
	andlw	0xF0		; upper bits	
	iorwf	PORTB,f		; w to portb
	return
; lower bits

LEV_LOWER
	nop
	bsf		PORTA,0		; enable high
	nop
	bcf		PORTA,0		; enable low
	movf	PORTB,w		; get portB levels
	andlw	0x0F		; clear upper bits keep lower bits
	movwf	PORTB		; upper bits cleared
	swapf	D_STO,w
	andlw	0xF0		; get lower bits
	iorwf	PORTB,f		; place display commands in portB
	return

; subroutine to double display value
TIMES2
	bcf		STATUS,C	; clear carry
	rlf		BIN1,f		; LSB shift left binary registers
	rlf		BIN2,f
	rlf 	BIN3,f
	rlf		BIN4,f		; MSB
	return

; Subroutine BCD (to convert 32-bit binary to 10 digit BCD)
; Binary value is in BIN1, BIN2, BIN3 & BIN4. BIN1 is LSB, BIN4 is MSB
; Result in BCD is in BCD1, BCD2, BCD3, BCD4 & BCD5.
; BCD2 is MSB, BCD5 is LSB
BIN_BCD24
	clrf	BIN2		; clear more significant bytes
BIN_BCD34
	clrf	BIN3
	clrf	BIN4
BIN_BCD	
BINBCDX
	bcf		STATUS,C	; clear carry bit
	movlw	D'32'
	movwf	COUNT		; 32 in count
	clrf	BCD1		; set BCD registers to 0 
	clrf	BCD2
	clrf	BCD3
	clrf	BCD4
	clrf	BCD5

LOOPBCD	
	rlf		BIN1,f		; LSB shift left binary registers
	rlf		BIN2,f
	rlf 	BIN3,f
	rlf		BIN4,f		; MSB
	rlf		BCD5,f		; LSB shift left BCD registers
	rlf		BCD4,f
	rlf		BCD3,f
	rlf		BCD2,f
	rlf		BCD1,f		; MSB

	decfsz	COUNT,f		; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	
; result in BCD1-5. (BCD1 overrange, BCD2 MS byte)

	movf	BCD1,w
	andlw	0x0F
	iorlw	0x30
	movwf	EEIGHT

	swapf	BCD2,w		; get ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ESEVN		; ms digit
	movf	BCD2,w		; get 2nd ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ESIX

	swapf	BCD3,w		; get next nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EFIVE		; ms digit
	movf	BCD3,w		; get next nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EFOUR
	
	swapf	BCD4,w		; get ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ETHREE		; ms digit
	movf	BCD4,w		; get 2nd ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ETWO

	swapf	BCD5,w		; get ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EONE		; ms digit
	movf	BCD5,w		; get 2nd ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EZERO
	return				; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	
	movlw	BCD5		; BCD LSB address
	movwf	FSR			; pointer for BCD5
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD4
	movwf	FSR
	call 	ADJBCD
	movlw	BCD3
	movwf	FSR
	call 	ADJBCD
	movlw	BCD2
	movwf	FSR
	call 	ADJBCD
	movlw	BCD1
	movwf	FSR
	call 	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	
	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 1-5)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return				; end subroutine

; subroutine to read EEPROM memory

EEREAD	
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bsf		EECON1,RD	; read EEPROM
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP0	; select bank 0
	return

; subroutine to write to EEPROM

EWRITE
	bsf		STATUS,RP0	; select bank 1
	movwf	EEDATA		; data register
	bcf		INTCON,GIE	; disable interrupts

	bsf		EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bsf		INTCON,GIE	; enable interrupts
	bcf		EECON1,WREN	; clear WREN bit
WRITE	
	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank 0 
	return				; value written 


;**********************************************************************************************
        
; 8 x 8 multiply

EIGHTEIGHT      CLRF    AARGB1          ; clear partial product
UMUL0808L        
                MOVLW   0x08
                MOVWF   LOOPCOUNT
                MOVF    AARGB0,W

LOOPUM0808A
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    LUM0808NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808A

                CLRF    AARGB0
                RETLW   0x00

LUM0808NAP
                BCF     STATUS,C
                GOTO    LUM0808NA

LOOPUM0808
                RRF     BARGB0, F
                BTFSC   STATUS,C
                ADDWF   AARGB0, F
LUM0808NA       RRF    	AARGB0, F
                RRF    	AARGB1, F
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808

                return             



;**********************************************************************************************
        
;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8      	CLRF            REMB0
                MOVLW           0x08
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           0x08
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B          RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
                return

;**********************************************************************************************
        
;       32x16 Bit Unsigned Fixed Point Multiply 32x16 -> 48

;       Input:  32 bit unsigned fixed point multiplicand in AARGB0
;                       16 bit unsigned fixed point multiplier in BARGB0

;       Result: AARG  <--  AARG x BARG


MULT3216
                CLRF    AARGB4          ; clear partial product
                CLRF    AARGB5
                MOVF   AARGB0,W
                MOVWF   TEMPB0
                MOVF   AARGB1,W
                MOVWF   TEMPB1
                MOVF   AARGB2,W
                MOVWF   TEMPB2
                MOVF   AARGB3,W
                MOVWF   TEMPB3

                MOVLW   0x08
                MOVWF   LOOPCOUNT

LOOPUM3216A
                RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM3216NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM3216A

                MOVWF   LOOPCOUNT

LOOPUM3216B
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM3216NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM3216B

                CLRF    AARGB0
                CLRF    AARGB1
                CLRF    AARGB2
                CLRF    AARGB3
                RETLW   0x00

BLUM3216NAP
                BCF     STATUS,C
                GOTO    BLUM3216NA

ALUM3216NAP
                BCF     STATUS,C
                GOTO    ALUM3216NA

ALOOPUM3216
                RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM3216NA
                MOVF   	TEMPB3,W
                ADDWF   AARGB3, F
                MOVF    TEMPB2,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB2,W
                ADDWF   AARGB2, F
                MOVF    TEMPB1,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F

ALUM3216NA
                RRF    	AARGB0, F
                RRF    	AARGB1, F
                RRF    	AARGB2, F
                RRF     AARGB3, F
                RRF     AARGB4, F
                DECFSZ  LOOPCOUNT, F
                GOTO    ALOOPUM3216

                MOVLW   0x08
                MOVWF   LOOPCOUNT

BLOOPUM3216
                RRF     BARGB0, F
                BTFSS   STATUS,C
                GOTO    BLUM3216NA
                MOVF   	TEMPB3,W
                ADDWF   AARGB3, F
                MOVF    TEMPB2,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB2,W
                ADDWF   AARGB2, F
                MOVF    TEMPB1,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F

BLUM3216NA
                RRF   	AARGB0, F
                RRF    	AARGB1, F
                RRF    	AARGB2, F
                RRF     AARGB3, F
                RRF     AARGB4, F
                RRF     AARGB5, F
                DECFSZ  LOOPCOUNT, F
                GOTO    BLOOPUM3216

                return 

; subtract AARGB1 (ms) AARGB2 (ls) - BARGBO BARGB1 = AARGB1 AARGB2
SUBTRACT
	call	NEG_A		; complement of A
	movf	BARGB1,w 
	addwf	AARGB2,f 	; add lsb
	btfsc	STATUS,C	; add carry
	incf	AARGB1,f 
	movf	BARGB0,w 
	addwf	AARGB1,f 
	return

NEG_A
	comf	BARGB1,f 
	incf	BARGB1,f 
	btfsc	STATUS,Z
	decf	BARGB0,f 
	comf	BARGB0,f 
	return	                             
	end
