; 			ECGSAMPL.asm  -- Simple ADC controller using PIC16F84A
;			running at 4MHz (so 1 machine cycle or m/c = 1us)
;			written by Jim Rowe. Last revised 14/12/2004

	list	P=16F84A
	#include "p16f84A.inc"
	__config _HS_OSC & _WDT_OFF & _PWRTE_ON
	
;	define variable storage locations
ADCvalue		equ		h'10'		;ADC output value stored here
CmdChar			equ		h'11'		;command byte from PC stored here
BitCounter		equ		h'12'		;serial bit counter stored here
Counter1		equ		h'13'		;bit timing counter stored here
Counter2		equ		h'14'		;sampling count A stored here
Counter3		equ		h'15'		;sampling count B stored here
Counter4		equ		h'16'		;fourth timing counter here
Counter5		equ		h'17'		;fifth timing counter here
SampleRate		equ		h'18'		;sampling rate flag stored here

;	---------------------------------------------------------------
;	MAIN PROGRAM STARTS HERE

	ORG h'0'
	GOTO Start					;vector for start of prog
	ORG h'04'
	GOTO Start					;interrupt vector (not used here)
	
Start:
	BCF STATUS, RP0				;set for bank 0
	CLRF PORTB					;clear port latches
	CLRF PORTA
	CLRF INTCON					;& also disable interrupts
	BSF STATUS, RP0				;now swing to bank1
	CLRF TRISB					;and set port B for all outputs
	MOVLW h'15'					;also set RA4, RA2 & RA0 as inputs
	MOVWF TRISA
	BCF STATUS, RP0				;then return to bank 0
	BSF PORTA, 1				;set RA1 output to 'mark'
	BCF PORTA, 3				;& also set RA3 to low (CTS-bar)
	MOVLW h'01'					;also set default sampling rate
	MOVWF SampleRate			; to 1kS/s
	; this ends the initialisation sequence
	;--------------------------------------------------------------------
	; main program loop now starts
	
WaitForCommand:
	BTFSC PORTA,0				;check RA0 for arrival of a start bit (->0)
	GOTO $-1					;nothing yet, so keep looking
	CALL WaitHalfaBit			;was a 0, so delay half a bit time
	BTFSC PORTA,0				;and see if RA0 is still a 0
	GOTO $-4					;wasn't, so loop back
	CALL BonzaiBit				;was, so it's a start bit. Wait 18us
	CALL FetchaByte				;then strobe 8 bits into CmdChar
	MOVLW "C"					;now make sure it's a C
	XORWF CmdChar,0
	BTFSS STATUS,Z				;is it a C?
	GOTO WaitForCommand			;no, so abort and keep looking
	BTFSC PORTA,0				;yes, so now check for sample rate char
	GOTO $-1					;still waiting...
	CALL WaitHalfaBit			;found a 0, so delay half a bit time
	BTFSC PORTA,0				;and check again
	GOTO $-4					;false alarm, so keep looking
	CALL BonzaiBit				;still 0, so another start bit. Wait one bit
	CALL FetchaByte				;then strobe 8 bits into CmdChar
	CALL GetSampleRate			;go check & save sample rate
	MOVLW h'40'					;now set up loop counters for 8192 samples
	MOVWF Counter2				; (outer loop Counter2 now has 40h = 64d)
StartConversionLoop:
	MOVLW h'80'					;set inner loop Counter3
	MOVWF Counter3				; (Counter3 now has 80h = 128d)
	CALL DoConv					;go do a conversion (takes 48us)
	CALL SendItBack				;then send the data back to PC (takes 296us)
	CALL SampleTimingWait		;pause for correct sample rate timing
	DECFSZ Counter3,1			;now decrement inner loop counter
	GOTO $-4					;loop back if counter3 not zero yet
	DECFSZ Counter2,1			;must've done 128, so decrem outer loop ctr
	GOTO $-8					;and loop back, resetting counter3
	GOTO WaitForCommand			;must've done 8192, so loop back to start
	
	; this is the end of the main program loop
	; --------------------------------------------------------- 
	
	; delay routine for half a bit time less 2us at 38,400bps
	; (11mc or 11us total, including call & return)
WaitHalfaBit:
	NOP							;do 7 nops
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP							; delay = 9mc including call
	RETURN						;so return (+2 = 11mc)
	;----------------------------------------------------------------
	
	; delay routine for a full bit time (26us) less 8 machine cycles
	; (because FetchaByte takes 8mc to sample each incoming bit)
	; (26 - 8 = 18us, which includes call & return)
BonzaiBit:
	MOVLW h'04'					;set counter for 4 loops (= 9us)
	MOVWF Counter1				; (delay to this point = 4 mc)
	DECFSZ Counter1,1			;decrement counter, skip when 0 reached
	GOTO $-1					;otherwise keep looping
	NOP							;we get here after 11 mc more (-> 15 mc)
	RETURN						;we're done, so return (+3 = 18 mc)	
	;------------------------------------------------------------------
	
	;routine to input a serial byte via RA0, save it in CmdChar	
FetchaByte:
	CLRW						;first clear w reg
	CLRF CmdChar				;and CmdChar	
	CLRF STATUS					;also clear carry, etc
	MOVLW h'08'					;then set counter for strobing in 8 bits
	MOVWF BitCounter							
	BTFSS PORTA,0				;now check to see if next bit is a 1
	GOTO $+3					;no, so go add a zero to CmdChar
	BSF STATUS,C				;yes, so set carry bit for adding a one
	GOTO $+2
	BCF STATUS,C				;clearing carry to add a zero instead
	RRF CmdChar,1				;rotate CmdChar & C to add in bit to MSB
	CALL BonzaiBit				;bit has been updated, so wait one bit time
	DECFSZ BitCounter,1			;then see if we've done 8 data bits
	GOTO $-8					;if not, loop back & continue
	RETURN						;yes, done 8 so we can leave
	
	;----------------------------------------------------------------------
	;routine to check sample rate command char, set rate accordingly
GetSampleRate:
	MOVLW "1"					; check if cmd char is a 1
	XORWF CmdChar,0
	BTFSS STATUS,Z				; skip if Z=1, because it was a 1
	GOTO $+4					; it wasn't a 1, so keep looking
	MOVLW h'01'					; was a 1, so set sample rate
	MOVWF SampleRate			; for 1kS/s (SampleRate = 01h)
	RETURN						; and exit
	MOVLW "2"					; now check if it's a 2
	XORWF CmdChar,0
	BTFSS STATUS,Z				; skip if it was a 2
	GOTO $+3					; it wasn't, so keep looking
	CLRF SampleRate				; was a 2, so set sample rate for 2kS/s
	RETURN						; and exit (with SampleRate = 00h)
	MOVLW "5"					; now check if it's a 5
	XORWF CmdChar,0
	BTFSS STATUS,Z				; skip if was a 5
	GOTO $+3					; it wasn't, so ignore and exit
	MOVLW h'03'					; was a 5, so set sample rate
	MOVWF SampleRate 			; for 500S/s (SampleRate = 03h)
	RETURN						; and exit
	
	;----------------------------------------------------------------------
	;ADC-SAR routine using port B to drive resistive ladder DAC
	;takes 48us total per conversion (4MHz clock), incl call & return
DoConv:
	CLRF PORTB					;clear all port B bits
	BSF PORTB,7					;then set bit 7 (DAC=80h)
	NOP							;short wait for comp to settle
	BTFSS PORTA,4				;then check comparator output
	GOTO $+2					;not 1, so leave b7 set
	BCF PORTB,7					;did -> 1, so reset b7
	BSF PORTB,6					;now try bit 6
	NOP
	BTFSS PORTA,4				;check comp o/p again
	GOTO $+2					;not 1, so leave b6 set
	BCF PORTB,6					;did -> 1, so reset b6
	BSF PORTB,5					;then try bit 5
	NOP
	BTFSS PORTA,4
	GOTO $+2
	BCF PORTB,5
	BSF PORTB,4					;then try bit 4
	NOP
	BTFSS PORTA,4
	GOTO $+2
	BCF PORTB,4
	BSF PORTB,3					;then try bit 3
	NOP
	BTFSS PORTA,4
	GOTO $+2
	BCF PORTB,3
	BSF PORTB,2					;then try bit 2
	NOP
	BTFSS PORTA,4
	GOTO $+2
	BCF PORTB,2
	BSF PORTB,1					;then try bit 1
	NOP
	BTFSS PORTA,4
	GOTO $+2
	BCF PORTB,1
	BSF PORTB,0					;then try bit 0
	NOP
	BTFSS PORTA,4
	GOTO $+2
	BCF PORTB,0
	MOVF PORTB,0				;finished, so fetch final port B value
	MOVWF ADCvalue				;and save it for sending to PC
	CLRF PORTB					;then clear port B
	RETURN						;and return
	
	;----------------------------------------------------------------
	; routine to send a byte of data at 38,400bps to PC via RA1
	; total time taken = 25 + (8 x 27) + 25 = 266 m/c incl call & retn
	; this means 266us with our 4MHz clock rate
	 
SendItBack:
	CALL Senda0			; first off send a start bit (takes 23us)
	NOP					; NOP to pad eff start bit length to 26us
	BTFSS ADCvalue, 0	; skip if there's a 1 in bit0 of ADCvalue
	GOTO $+3			; no, so go send a zero bit
	CALL Senda1			; yes, so send a 1 (takes 23us) 
	GOTO $+3			; before moving to next bit
	CALL Senda0			; send a 0 (takes 23us)
	NOP					; NOP to make bit 0 send = 30 = bit 1 send	
	BTFSS ADCvalue, 1	; now process bit1 of ADCvalue the same way
	GOTO $+3
	CALL Senda1
	GOTO $+3
	CALL Senda0
	NOP
	BTFSS ADCvalue, 2	; and bit2
	GOTO $+3
	CALL Senda1
	GOTO $+3
	CALL Senda0
	NOP
	BTFSS ADCvalue, 3	; and bit3
	GOTO $+3
	CALL Senda1
	GOTO $+3
	CALL Senda0
	NOP
	BTFSS ADCvalue, 4	; and bit4
	GOTO $+3
	CALL Senda1
	GOTO $+3
	CALL Senda0
	NOP
	BTFSS ADCvalue, 5	; and bit5
	GOTO $+3
	CALL Senda1
	GOTO $+3
	CALL Senda0
	NOP
	BTFSS ADCvalue, 6	; and bit6
	GOTO $+3
	CALL Senda1
	GOTO $+3
	CALL Senda0
	NOP
	BTFSS ADCvalue, 7	; and bit7
	GOTO $+3
	CALL Senda1
	GOTO $+3
	CALL Senda0
	NOP
	CALL Senda1			; finally send a stop bit (takes 23us)
	RETURN				; and return (takes 2 m/c)

Senda0:
	; routine to send a '0' bit, taking a total of 23us
	; (1 bit time at 38,400bps less 4mc) including call & return
	BCF PORTA,1			; make RA1 a zero
	CALL BonzaiBit		; then pause for 18mc (18us)
	RETURN 				; before returning
	
Senda1:
	; routine to send a '1' bit, taking a total of 23us
	; (1 bit time at 38,400bps less 4mc) including call & return
	BSF PORTA,1			; make RA1 a one
	CALL BonzaiBit		; then pause for 18mc (18us)
	RETURN				; before returning
	
	;-------------------------------------------------------------------
	; routine to pause for correct time to achieve desired sampling rate
	; first loop gives total pause of 185us (including call & return)
	; which is all that is needed for 2kS/s sampling. Padding loop adds
	; either 500us or 1500us more, for either 1kS/s or 500S/s rates
	
SampleTimingWait:
	MOVLW h'3A'			;set Counter4 for 58 loops
	MOVWF Counter4		; after this instruction, delay so far = 4mc
	DECFSZ Counter4,1	;decrement & skip when 0 reached
	GOTO $-1
	NOP					;padding NOP to make delay = 181 mc so far
	MOVF SampleRate,1	;now check if SR = 0 (2000 S/s flag)
	BTFSC STATUS,Z		;skip if Z status not set (i.e., SR not 0)
	RETURN				;was 0, so return (total time = 185mc = 185us)
	MOVLW h'01'			;not 0, so check if SR=1 (1000 S/s)
	XORWF SampleRate,0
	BTFSS STATUS,Z		;skip if it was a 1 (i.e., we got a match)
	GOTO $+3			;wasn't, so set for 3
	MOVLW h'01'			;yes it was, so load 1 in w
	GOTO $+2			;and proceed
	MOVLW h'04'			;load 4 in w
	NOP					;padding NOP so each path equal time
	MOVWF Counter5		;set Counter5 with value in w (either 1 or 4)
	GOTO $+3			; (added time just after this GOTO = 9us)
PaddingLoop:
	MOVLW h'6D'			;set w for 108 padding loops (later loops)
	GOTO $+2
	MOVLW h'A1'			;set w for 160 padding loops (1st loop)
	NOP					;NOPs for timing adjustment
	NOP
	NOP
	MOVWF Counter4		;load Counter4 for required loops
	DECFSZ Counter4,1	;decrement & skip when 0 reached
	GOTO $-1			;loop until done
	DECFSZ Counter5,1	;done 160 or 108, so dec & check Counter5
	GOTO PaddingLoop	;not finished, so loop back for another set
	RETURN				;padding is done, so return 
	

	END
