
; automatic table lamp dimmer

; Philips RC5 code for infra red transmission (also used with Marantz, Grundig and Loewe)
; Comprises 2 start bits, 1 toggle bit (alternates high or low on successive same key press)
; 5-system address bits and 6 command bits. Data stream starts with the start bits. 
; biphase encoding with high to a low transition = a low and low to high = high
; each bit transmitted at 1.778ms rate. Whole code at 24.889ms and next code starts 113.778ms
; later. Can decode by detecting first start bit then waiting 1.778ms + 444.5us for next start
; bit. Successive bits are 1.778ms apart. Total of 14 bits. 

;Address
;0 TV1
;1 TV2
;2 Videotext
;3 Expansion for TV1 and TV2
;4 Laser vision player
;5 VCR1
;6 VCR2
;7 Reserved
;8 SAT1
;9 Expansion for VCR1 and VCR2
;10 SAT2
;11 Reserved
;12 CD Video
;13 Reserved
;14 CD Photo
;15 Reserved
;16 Audio preamplifier1
;17 Tuner
;18 Analogue cassetter recorder
;19 Audio Preamp2
;20 CD
;21 Audio rack
;22 Audio Satellite
;23 DCC Recorder
;24 Reserved
;25 Reserved
;26 Writable CD
;27-31 Reserved

;Keycode commands 
;0 0
;1 1
;2 2
;3 3
;4 4
;5 5
;6 6
;7 7
;8 8
;9 9
;16 Volume +
;17 Volume -
;18 Brightness +
;19 Brightness -
;20 Colour saturation +
;21 Colour Saturation -
;22 Bass +
;23 Bass -
;24 Treble +
;25 Treble -
;26 Balance right
;27 balance left
;63 System select
;71 Dim local display

; Special commands
;10 1/2/3 digits /10
;11 Freq/prog/ch11
;12 standby
;13 Mute
;14 Personal preference
;15 Display
;28 Contrast +
;29 Contrast -
;30 Search +
;31 Tint/hue
;32 CH/prog +
;33 CH/prog -
;34 Alternate CH

;37 stereo/mono
;39 Tint/hue +
;48 Pause
;50 rewind
;52 wind
;53 play
;54 stop
;55 record

	list P=16F628A
	#include p16f628A.inc

	__config _HS_OSC & _WDT_OFF & _PWRTE_ON & _BOREN_ON & _MCLRE_ON & _LVP_OFF
	
; Define variables at memory locations

EEPROM0		equ	H'00'	; non-volatile storage for down dimming value brightness
EEPROM1		equ	H'01'	; non-volatile storage for auto dimming rate
EEPROM2		equ	H'02'	; non-volatile storage for up dimming value brightness

; Bank 0 RAM

TIMER		equ	H'20'	; temp timer value
FLAG_Z		equ	H'21'	; zero crossing flag
PULSE_V		equ	H'22'	; pulse value 
PULSE_O		equ	H'23'	; operational pulse value
DIVID_O		equ	H'24'	; operational divide value
ON_OFFT		equ	H'25'	; on/off timer
SOFT_ST		equ	H'26'	; soft start rate counter
TOGGLE		equ	H'27'	; remote control toggle flag
R_FLAG		equ	H'28'   ; remote control flag
R_TMR		equ	H'29'	; remote control timer
RD_RMT		equ	H'2A'   ; remote input store
GLITCH		equ	H'2B'	; flag that determines a glitch for start of remote code
DIMRTE		equ	H'2C'	; dimming rate
FLG_HL		equ	H'2D'	; high or low remote control sign

BRIGHT_C	equ	H'2E'	; current lamp brightness 
BRIGHT_S	equ	H'2F'	; brightness setting for lookup table 
BRIGHT_V	equ	H'30'	; stored brightness
AUT_DIM_FLG	equ	H'31'	; AUT_DIM flag to detect when operate switch is pressed
TOUCH_TMR	equ	H'32'	; touch timer
POSITION	equ	H'33'	; position counter within half sine wave starting from zero crossing
RATE		equ	H'34'	; rate execution counter
FLAG		equ	H'35'	; bit 0, off flag, 1 on, 2 up, 3 down 
COUNTZ_Z	equ	H'36'	; counter from zero crossing to zero crossing
COUNTZ		equ	H'37'	; counter storage of z-z counter
TOUCH_C		equ	H'38'	; touch count
REMOTE_M	equ	H'39'	; remote control input ms byte
REMOTE_L	equ	H'3A'	; remote control input LS byte
REM_FLG		equ	H'3B'	; flag to indicate current remote control sequencing 
REM_CNT		equ	H'3C'	; interrupt count for polling remote control signal
REM_COD		equ	H'3D'	; remote control bit counter
FLAG_ON		equ	H'3E'	; 0, off flag. 1,on/off. 2,up/dn dimming mode.
						; 3, soft start. 4, stop dimming 
DIVIDE		equ	H'3F'	; divider of timer counter
SLO_RATE	equ	H'40'	; slower dimming rate timer
PGM_TMR		equ	H'41'	; program LED timer
BRIGHT_E	equ	H'42'	; working register for stored brightness
FLASHER_P	equ	H'43'	; prescaler for LED flasher
LED_FLASH	equ	H'44'	; LED flash counter during Auto Dim
SHIFT_CNT	equ	H'45'	; number shift counter
TEMP_VAL	equ	H'46'	; temporary register

H_BYTE		equ	H'47'	; high byte result of BCD to binary conversion
L_BYTE		equ	H'48'	; low byte BCD result
REG_0		equ	H'49'	; MSD of 5-digit BCD in right hand (low) nibble
REG_1		equ	H'4A'	; BCD number (packed BCD)
REG_2		equ	H'4B'	; BCD low order numbers
H_TEMP		equ	H'4C'	; temporary register of high order binary result
L_TEMP		equ	H'4D'	; temporary register of low order binary result	
PORTB_STO	equ	H'4E'	; portb storage after reading RB7,6,5 with pullups
BRIGHT_X	equ	H'4F'	; stored dimming up value

; interrupt storage
ST_ADR		equ	H'7C'	; storage of EEPROM Address during interrupt
EDTA_ST		equ	H'7D'	; EEPROM data store
W_TMP		equ	H'7E'	; storage of w before interrupt
STATUS_TMP	equ	H'7F'	; status storage before interrupt

; preprogram EEPROM DATA 
	
	ORG     2100
	DE	D'146'	; stored down brightness
	DE  D'02'	; auto dimming rate
	DE	D'36'	; stored up brightness

; define reset and interrupt vector start addresses

	org		0	  	; start at address 0000h
	goto	MAIN	; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT

; lookup table for lamp brightness versus phase

BRIGH_T 
	addwf	PCL,f	; add value to program counter
	retlw	D'20'	; 0
	retlw	D'20'	; 1 
	retlw	D'20'	; 2 
	retlw	D'20'	; 3 
	retlw	D'20'	; 4 
	retlw	D'20'	; 5 
	retlw	D'20'	; 6 
	retlw	D'20'	; 7 
	retlw	D'20'	; 8  
	retlw	D'20'	; 9 
	retlw	D'20'	; 10 
 	retlw	D'20'	; 11 
	retlw	D'20'	; 12 
 	retlw	D'20'	; 13 
	retlw	D'20'	; 14 
 	retlw	D'20'	; 15 
	retlw	D'20'	; 16 
 	retlw	D'20'	; 17 
	retlw	D'20'	; 18 
 	retlw	D'20'	; 19 
	retlw	D'20'	; 20 
 	retlw	D'20'	; 21 
	retlw	D'20'	; 22 
 	retlw	D'20'	; 23 
	retlw	D'20'	; 24 
	retlw	D'20'	; 25 
	retlw	D'20'	; 26 
	retlw	D'20'	; 27 
	retlw	D'20'	; 28 
	retlw	D'20'	; 29 
	retlw	D'20'	; 30 
	retlw	D'20'	; 31 
	retlw	D'20'	; 32 
	retlw	D'20'	; 33
	retlw	D'20'	; 34 
	retlw	D'22'	; 35 
	retlw	D'24'	; 36 
	retlw	D'26'	; 37 
	retlw	D'28'	; 38 
	retlw	D'30'	; 39
	retlw	D'32'	; 40 
	retlw	D'34'	; 41 
	retlw	D'36'	; 42 
	retlw	D'38'	; 43 
	retlw	D'40'	; 44 
	retlw	D'42'	; 45 
	retlw	D'44'	; 46 
	retlw	D'46'	; 47 
	retlw	D'48'	; 48 
	retlw	D'52'	; 49 
	retlw	D'56'	; 50 
	retlw	D'60'	; 51 
	retlw	D'64'	; 52 
	retlw	D'68'	; 53 
	retlw	D'71'	; 54 
	retlw	D'74'	; 55 	
	retlw	D'77'	; 56 
	retlw	D'80'	; 57 
	retlw	D'83'	; 58 
	retlw	D'86'	; 59 
	retlw	D'89'	; 60 
	retlw	D'92'	; 61 
	retlw	D'95'	; 62
	retlw	D'98'	; 63 
	retlw	D'100'	; 64 
	retlw	D'102'	; 65 	
	retlw	D'104'	; 66 
	retlw	D'106'	; 67 
	retlw	D'108'	; 68 
	retlw	D'110'	; 69 
	retlw	D'112'	; 70
	retlw	D'114'	; 71 
	retlw	D'116'	; 72 
	retlw	D'118'	; 73 
	retlw	D'120'	; 74 
	retlw	D'122'	; 75 
	retlw	D'124'	; 76 
	retlw	D'126'	; 77 	
	retlw	D'128'	; 78 
	retlw	D'130'	; 79 
	retlw	D'132'	; 80 
	retlw	D'134'	; 81 
	retlw	D'136'	; 82 
	retlw	D'138'	; 83 
	retlw	D'140'	; 84 
	retlw	D'142'	; 85 
	retlw	D'144'	; 86 
	retlw	D'146'	; 87 
	retlw	D'148'	; 88 
	retlw	D'150'	; 89 	
	retlw	D'151'	; 90 
	retlw	D'152'	; 91 
	retlw	D'153'	; 92 
	retlw	D'154'	; 93 
	retlw	D'155'	; 94
	retlw	D'156'	; 95 
	retlw	D'157'	; 96 
	retlw	D'158'	; 97 
	retlw	D'159'	; 98 
	retlw	D'160'	; 99
	retlw	D'161'	; 100 
	retlw	D'162'	; 101 	
	retlw	D'163'	; 102 
	retlw	D'164'	; 103 
	retlw	D'165'	; 104 
	retlw	D'166'	; 105 
	retlw	D'167'	; 106
	retlw	D'168'	; 107 
	retlw	D'169'	; 108 
	retlw	D'170'	; 109 
	retlw	D'171'	; 110 
	retlw	D'172'	; 111 
	retlw	D'173'	; 112 
	retlw	D'174'	; 113 	
	retlw	D'175'	; 114 
	retlw	D'176'	; 115 
	retlw	D'177'	; 116 
	retlw	D'178'	; 117 
	retlw	D'179'	; 118
	retlw	D'180'	; 119 
	retlw	D'181'	; 120 
	retlw	D'182'	; 121 
	retlw	D'183'	; 122 
	retlw	D'184'	; 123 
	retlw	D'185'	; 124 
	retlw	D'186'	; 125 	
	retlw	D'187'	; 126 
	retlw	D'188'	; 127 
	retlw	D'189'	; 128 
	retlw	D'190'	; 129 
	retlw	D'191'	; 130
   	retlw	D'192'	; 131 
	retlw	D'193'	; 132 
	retlw	D'194'	; 133 	
	retlw	D'195'	; 134 
	retlw	D'196'	; 135 
	retlw	D'197'	; 136 
	retlw	D'198'	; 137 
	retlw	D'199'	; 138
	retlw	D'200'	; 139 
	retlw	D'201'	; 140 
	retlw	D'202'	; 141 
	retlw	D'203'	; 142 
	retlw	D'204'	; 143 
	retlw	D'205'	; 144 
	retlw	D'206'	; 145 	
	retlw	D'207'	; 146 
	retlw	D'208'	; 147 
	retlw	D'209'	; 148 
	retlw	D'210'	; 149 
	retlw	D'211'	; 150
	retlw	D'212'	; 151
	retlw	D'213'	; 152 
	retlw	D'214'	; 153 
	retlw	D'215'	; 154 
	retlw	D'216'	; 155 
	retlw	D'217'	; 156 
	retlw	D'218'	; 157 
	retlw	D'219'	; 158 	
	retlw	D'220'	; 159 
	retlw	D'221'	; 160 
	retlw	D'222'	; 161 
	retlw	D'223'	; 162 
	retlw	D'224'	; 163 
	retlw	D'225'	; 164 
	retlw	D'226'	; 165 
	retlw	D'227'	; 166
	retlw	D'228'	; 167 
	retlw	D'229'	; 168 
	retlw	D'230'	; 169 
	retlw	D'231'	; 170 
	retlw	D'232'	; 171 
	retlw	D'233'	; 172 
	retlw	D'234'	; 173 	
	retlw	D'235'	; 174 
	retlw	D'236'	; 175 
	retlw	D'237'	; 176 
	retlw	D'238'	; 177 
	retlw	D'239'	; 178 
	retlw	D'240'	; 179 
	retlw	D'240'	; 180
 
; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT 
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0

; check which interrupt	
	
	btfsc	INTCON,INTF	; INT interrupt flag set then goto zero
	goto	ZERO		; zero crossing signal update
	btfsc	INTCON,T0IF	; TMRO overflow interrupt flag then goto counter
	goto	COUNTER		; TMRO overflow so increase counter 
	goto	RECLAIM		; end of interrupt reclaim w and status

; POSITION counter timer 250 counts = 10ms (count 20 is 14 degrees for max brightness and count 
; 220 is 158 degrees or min brightness start point. 

COUNTER	

	incf	POSITION,f	; position in half wave starting from zero crossing point

; adjust interrupt rate with counter

	movf	DIVIDE,w	; divide value
	movwf	DIVID_O		; operating divide value
	movf   	PULSE_O,f	; pulse value number of interrupt speedups
	btfsc	STATUS,Z	; if zero leave
	goto	NO_CHNG
	incf	DIVID_O,f
	decf	PULSE_O,f	; freq is 20MHz/4/4/50. =25kHz or 40us period
NO_CHNG
	movf	TMR0,w		; each change of DIVID_O speeds interrupt time to 39.2us	
	addwf	DIVID_O,w	; add to timer register and takes 2 cycles to start counting
	movwf	TMR0	
	bcf		INTCON,T0IF	; clear TMRO interrupt flag

; increase zero crossing counter
	
	incf	COUNTZ_Z,f	; increase the zero crossing to zero crossing counter
	btfsc	STATUS,Z
	bsf		FLAG_Z,0	; if zero set flag
	btfsc	FLAG_Z,0	; is flag set
	goto	POSN		; second half of sine wave
	movf	COUNTZ_Z,w
	xorwf	COUNTZ,w	; check if same (countZ = 250)
	btfsc	STATUS,Z	; if zero then half way through zero-zero crossing point
	goto	ZERO_H		; go to half way between zero crossings
	

; check if POSITION counter = BRIGHT_C counter then trigger @ 40us gate drive each
	
POSN
	movf	PORTB,w		; looking at RB4 
	movwf	RD_RMT		; remote input store
	movf	POSITION,w	; look at position in mains waveform
	xorwf	BRIGHT_C,w	; current brightness. If zero then drive Triac gate
	btfsc	STATUS,Z	; if zero bit high drive gate (is off, on at next timer interrupt)
	goto	FLG_TST
	incf	BRIGHT_C,w	; drive gate for second time so total of 80us
	xorwf	POSITION,w
	btfss	STATUS,Z
	goto	REMOTE
FLG_TST
	btfsc	FLAG,0		; is the off flag set
	goto	REMOTE		; do not drive gate if off
	movlw	B'11110001'
	andwf	PORTB,f		; RB1-RB3 low
	goto	D_PORTB
	
; check if RB4 has gone low for start of remote control sequence

REMOTE
	movlw	B'00001110'	; RB1-RB3 high
	iorwf	PORTB,f
D_PORTB	
	btfsc	REM_FLG,0	; has remote control sequence started		
	goto	REM_SEQ
	btfss	RD_RMT,4	; was RB4 low
	goto	CK_TMR
	bcf		GLITCH,0	; clear glitch bit as high RB4 
	goto	RECLAIM
CK_TMR
	movf	R_TMR,w		; remote control timer
	btfss	STATUS,Z
	goto	RECLAIM		; wait till timer is 0

	btfsc	GLITCH,0	; if high then was high at last interrupt
	goto	NO_GLC
	bsf		GLITCH,0	; set if was clear
	goto	RECLAIM		; bypass as may be a glitch
NO_GLC
	bcf		GLITCH,0	; clear as was set
	bsf		REM_FLG,0	; set beginning of remote control flag

	movlw	D'55'		; initial time period to wait till next start bit
	movwf	REM_CNT
	movlw	D'13'		; shift register counter
	movwf	REM_COD
	bsf		REMOTE_L,0	; set first bit in remote control sequence
	clrf	FLG_HL		; * clear bit 1 in high low flag
	bsf		REM_FLG,7	; * start bit flag set

	goto 	RECLAIM

REM_SEQ	
	btfsc	REM_FLG,1	; has it finished
	goto	RECLAIM
	decfsz	REM_CNT,f	; decrease interrupt counter for timing
	goto	CKEDGE		; * check change in level
	bcf		REM_FLG,7	; * clear start bit flag
	movlw	D'45'		; 45 x 40us = 1.8ms or period between valid bits
	movwf	REM_CNT
	rlf		REMOTE_L,f	; least sig byte in remote control code sequence
	rlf		REMOTE_M,f	; most sig byte 
	movf	RD_RMT,w	; * check bit 4 portB via RD_RMT
	andlw	B'00010000'	; * mask out bits except bit 4
	movwf	FLG_HL		; * place in flag
	btfsc	RD_RMT,4	; check if high or low
	goto	HI			; high so clear
	bsf		REMOTE_L,0	; if low set this bit
	bsf		PORTA,1		; acknowledge LED
	goto	BY_HI		; bypass high
HI	bcf		REMOTE_L,0	; clear 0 bit
	bcf		PORTA,1		; acknowledge LED
BY_HI
	decfsz	REM_COD,f	; decrease shift register count
	goto	RECLAIM
	bsf		REM_FLG,1	; set end of remote control decoding flag
	movlw	D'09'		; timer (9 gives 90ms and plus first 24.889ms =114.88ms)
						; so await for next code which occurs every 113.778ms
	movwf	R_TMR		; remote control timer
	bcf		PORTA,1		; acknowledge LED
	goto	RECLAIM

; align change in level (edge detect) with REM_CNT counter
 
CKEDGE
	btfsc	REM_FLG,7	; start bit flag bypass edge detect as AGC level setting
	goto	RECLAIM
	movlw	B'00010000'	; select bit 4 for PORTB,4
	andwf	RD_RMT,w	; bit 4
	xorwf	FLG_HL,w	; compare PORT with FLAG (previous level)		
	btfsc	STATUS,Z	; Z=1 if zero then no change
	goto	RECLAIM		; no change 
	movlw	B'00010000'	; select bit 4 for PORTB,4
	andwf	RD_RMT,w	; bit 4 select 
	movwf	FLG_HL		; reload new level
	movlw	D'6'		; count of 6 for REM_CNT
	subwf	REM_CNT,w	; is REM_CNT larger or equal to 6
	btfss	STATUS,C
	goto	RECLAIM		; smaller than 6 
	movf	REM_CNT,w
	sublw	D'16'		; check if 16 or less
	btfss	STATUS,C
	goto	RECLAIM		; larger than 16 
	movlw	D'11'
	movwf	REM_CNT		; set REM_CNT to 11 to align with change in level
	goto	RECLAIM

; zero crossing 

ZERO

	movf	TMR0,w		; timer value
	movwf	TIMER		; temp storage for TMR0	

; reset timer counter
; freq is 20MHz/4/4/50. [50 =(256(-208-2)] =25kHz or 40us period 
	
	movf	DIVIDE,w	; originally at D'208'	
	movwf	TMR0		; set timer
	nop					; make sure timer reset before clearing flag 
	nop
	nop
	bcf		INTCON,INTF	; clear INT flag
	bcf		INTCON,T0IF	; clear TMRO interrupt flag

; check lock
	
	movf	COUNTZ_Z,w	; zero - zero crossing count of 500=244 as (256+244=500)
	sublw	D'150'		; overrange into next count
	btfsc	STATUS,C	
	goto	SLOWER
	movf	COUNTZ_Z,w
	sublw	D'243'
	btfss	STATUS,Z	; if zero check timer status
	goto	FST_SLO

; check timer value to within acceptable range (ie. in middle of count between 205 and 255)

	movf	TIMER,w		; look at timer
	sublw	D'225'		; is it larger than 225
	btfsc	STATUS,C	; if c is 0 then >225 if 1 then <225 so speed up
	goto	FASTER
	movf	TIMER,w
	sublw	D'235'
	btfsc	STATUS,C	; if c is 1 then <= 235
	goto	RESET		; in lock so leave values as is
	goto	SLOWER
FST_SLO
	btfss	STATUS,C	; if 0 then count is larger so slow down
	goto	SLOWER
FASTER
	incf	PULSE_V,w
	sublw	D'250'
	btfsc	STATUS,C	; if c is 0 then do not increase
	goto	INC_PLS
	clrf	PULSE_V
	incf	DIVIDE,f	; increase divide as PULSE_V has exhausted its range
	goto	RESET		
INC_PLS
	incf	PULSE_V,f
	goto 	RESET
SLOWER
	movf	PULSE_V,w
	btfss	STATUS,Z
	goto	DEC_PLS
	decf	DIVIDE,f	; zero PULSE_V so reduce Divide value 
	movlw	D'250'
	movwf	PULSE_V
	goto	RESET
DEC_PLS	
	decf	PULSE_V,f

; zero crossing reset counter (also compensates for shift due to input filtering)

RESET
	bcf		FLAG_Z,0	; clear zero crossing sine flag (plus or minus half sine wave)
	clrf	COUNTZ_Z	; counter from zero crossing to zero crossing
	
ZERO_H
	movf	PULSE_V,w
	movwf	PULSE_O		; transfer value to operating counter
	movlw	D'00'		; offset compensates for filter phase change 
	movwf	POSITION	; reset position counter any offset is to counteract shift in zero
 						; voltage crossing detector due to filtering
	movf	BRIGHT_S,w	; update brightness
	call	BRIGH_T		; lookup table	
	movwf	BRIGHT_C	; update every zero crossing only (10ms to prevent erratic brightness during dimming up) 
	movf	R_TMR,w		; look at remote control timer
	btfss	STATUS,Z	; if zero do not decrease
	decf	R_TMR,f

; flash LED during auto dim
	movf	FLASHER_P,w	; prescaler
	btfss	STATUS,Z	; if zero then decrease LED_FLASH
	goto	DEC_FLASH
	movlw	0x02
	movwf	FLASHER_P	; start count again
	movf	LED_FLASH,w	; flash counter
	btfss	STATUS,Z	; if zero drive LED if in auto dim mode
	goto	DEC_LED_F
	btfss	AUT_DIM_FLG,1
	goto	SET_FLASH
	bsf		PORTA,0		; auto dim LED on
	bcf		PORTA,1		; clear ack LED

SET_FLASH
	movlw	0x10
	movwf	LED_FLASH	; start count again
	goto	DEC_FLASH
DEC_LED_F
	decf	LED_FLASH,f
	bcf		PORTA,0		; LEDs off
DEC_FLASH
	decf	FLASHER_P,f	; decrease

; decrease program timer
	movf	PGM_TMR,w
	btfss	STATUS,Z	; if zero bypass
	decf	PGM_TMR,f	; reduce to zero

; decrease touch timer
	btfss	AUT_DIM_FLG,1	; if set slower rate
	goto	TOUCH_FAST
	decfsz	SLO_RATE,f	; if zero reload
	goto	REMOTE
	movlw	D'16'
	movwf	SLO_RATE
TOUCH_FAST
	movf	TOUCH_TMR,w
	btfss	STATUS,Z	; if zero bypass
	decf	TOUCH_TMR,f	; reduce to zero
	goto	REMOTE

; end of interrupt reclaim w and status 

RECLAIM
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	
	retfie				; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN
	clrf	GLITCH		; glitch flag for start of remote
	clrf	R_FLAG		; remote control flags
	clrf	R_TMR		; remote timer
	clrf	ON_OFFT		; on/off timer
	clrf	PULSE_V		; pulse value
	clrf	PULSE_O		; operating pulse value
	clrf	REM_CNT		; remote control polling counter
	clrf	REM_FLG		; remote control flag status
	clrf	TOUCH_C		; touch counter
	clrf	DIMRTE		; dimming rate
	clrf	FLAG_ON		; on/off flag
	bsf		FLAG,0		; set off flag
	movlw	D'208'		; initial divider for timer counter
	movwf	DIVIDE		; freq is 20MHz/4/4/50. [50 =(256(-208-2)] =25kHz or 40us period 
	movlw	D'220'		; min brightness setting
	movwf	BRIGHT_C	; current brightness level
	movlw	D'159'		; min brightness
	movwf	BRIGHT_S	; brightness setting
	movlw	D'250'		; zero crossing
	movwf	COUNTZ		; counter for zero crossing - zero crossing length
	clrf	COUNTZ_Z	; clear countZ-Z zero crossing counter
	clrf	TOUCH_C		; touch time counter
	clrf	AUT_DIM_FLG	; auto dimming flag
	clrf	TOUCH_TMR	; touch timer with mute switch
	clrf	FLASHER_P	; LED flasher prescaler
	clrf	LED_FLASH	; LED flash counter

	movlw	EEPROM0
	call	EEREAD		; EEPROM address sets EEADR
	movwf	BRIGHT_V	; stored down brightness
	movlw	EEPROM2
	call	EEREAD		; EEPROM address sets EEADR
	movwf	BRIGHT_X	; stored up brightness

; setup port control

; set inputs/outputs
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	bsf		STATUS,RP0	; select memory bank 1

	movlw	B'11110001'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'11100000'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000001'	; settings (pullups enabled TMR0/4)
	movwf	OPTION_REG

	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'11111111'	; 1 for RB1-RB3
	movwf	PORTB		; portB outputs high
	movlw	B'00011100'
	movwf	PORTA		; portA initial settings

; determine remote code
	
	movf	PORTB,w
	movwf	PORTB_STO	; store portB levels
	movlw	B'00011111'
	movwf	PORTB		; set RB5,6,7 low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00010001'	; port B outputs/ inputs set. RB7.6,5 outputs and low
	movwf	TRISB		; port B data direction register
	movlw	B'10000001'	; settings (pullups disabled TMR0/4)
	movwf	OPTION_REG
	bcf		STATUS,RP0	; select memory bank 0

; interrupt enable 

	bsf		INTCON,T0IE	; set interrupt enable for TMR0 
	bsf		INTCON,INTE	; set interrupt enable for RB0
	bsf		INTCON,GIE	; set global interrupt enable for above

; check remote codes 
	
SWITCH
	movf	POSITION,w	; position counter
	xorlw	D'78'		; check once in positive 
	btfss	STATUS,Z
	goto	DEM_DEC		; remote control decoding

; auto dimming flag check

	btfsc	AUT_DIM_FLG,1	; if set then in auto dim mode
	goto	AUTO_DN
; soft start flag check	

	btfsc	FLAG_ON,3	; is soft start set
	goto	ON_SET
	btfsS	FLAG_Z,0	; is flag bit set (positive half of sine wave) 
	goto	SWITCH		; 

; remote control decoding

DEM_DEC
	movf	R_TMR,w		; is timer at zero
	btfsc	STATUS,Z	; if not zero check flags
	goto	CL_CKR		; if zero then clear flags and check remote
	movf	R_FLAG,w	; are any flags set
	btfsc	STATUS,Z	; if flags set then check position
	goto	CL_CKR

	movf	POSITION,w
	xorlw	D'120'		; once per cycle
	btfss	STATUS,Z	; if zero can update 
	goto	SWITCH
	
	btfsc	R_FLAG,0	; up flag
	goto	REM_UP
	btfsc	R_FLAG,1
	goto	REM_DN
		
CL_CKR
	clrf	R_FLAG
	
	btfss	REM_FLG,1	; is remote control entered flag set
	goto	SWITCH
	movf	REMOTE_M,w	; most significant remote code 
	andlw	B'00110111'	; mask out bit 7 and 6 and toggle bit

; compare with address code out if invalid clear REM_FLG
; if valid check command bits and ls 2 address bits then clear REM_FLG
 
	btfss	PORTB_STO,5		; check port
	goto	VCR_1		; VCR1 code 
	btfss	PORTB_STO,6
	goto	SAT_1		; satellite codes
	btfss	PORTB_STO,7
	goto	SAT_2		; satellite codes
; all high = TV

; code for television (TV1)

TV_1
	xorlw	B'00110000'	; compare with start bits and ms 3-bits of address
	btfss	STATUS,Z	; if zero then matching
	goto	CLR_RMF		; clear remote flag
TV_SAME
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'00001100'	; bits 0-5 keycode (12), bit 7 and bit 6 are ls address bits
	btfss	STATUS,Z
	goto	TV_LOOP
	clrf	SHIFT_CNT	; shift enter count
	bsf		AUT_DIM_FLG,1; set flag
	btfsc	FLAG_ON,0	; is off flag on
	goto	AUTO_DN1
	bsf		FLAG_ON,3	; dim up flag
	
	goto	AUTO_UP

TV_LOOP
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'00100000'	; bits 0-5 keycode (32), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_UP

	movf	REMOTE_L,w
	xorlw	B'00100001'	; bits 0-5 keycode (33), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_DN 
	movf	REMOTE_L,w
	xorlw	B'00010000'	; bits 0-5 keycode (16), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	UP_SET
	movf	REMOTE_L,w
	xorlw	B'00010001'	; bits 0-5 keycode (17), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	DN_SET  
	movf	REMOTE_L,w
	xorlw	B'00001101'	; bits 0-5 keycode (13), bit 7 and bit 6 are ls address bits
	btfss	STATUS,Z
	goto	CK_9
	btfsc	FLAG_ON,0	; is off flag on
	goto	OFF_SET
	bsf		FLAG_ON,3	; soft start dim up flag
	goto	ON_SET
CK_9
	movf	REMOTE_L,w
	andlw	B'11000000'	; ms bits are address
	xorlw	B'00000000'	; bits 7 and 6 are zero for TV and SAT1
	btfss	STATUS,Z
	goto	CLR_RMF

	movf	REMOTE_L,w
	andlw	B'00111111'	; mask address bits
	sublw	0x09		; if negative bypass
	btfss	STATUS,C
	goto	CLR_RMF
;	goto	AUTO_DIM_RATE_PGM

AUTO_DIM_RATE_PGM

; is auto dimming in operation
	btfss	AUT_DIM_FLG,1	; if set then in auto dim mode
	goto	CLR_RMF

; if shift counter set then bypass the zero checking
	btfsc	SHIFT_CNT,0
	goto	SET_RATE_2

; if remote_l is 0 then store BRIGHT_V and BRIGHT_X to EEPROM
	movf	REMOTE_L,w
	andlw	B'00001111'	; mask address bits
	btfss	STATUS,Z
	goto	SET_RATE_1
	movlw	EEPROM0
	call	EEREAD		; EEPROM address sets EEADR
	movf	BRIGHT_V,w	; stored brightness
	call	EWRITE		; EEPROM store
	movlw	EEPROM2
	call	EEREAD		; EEPROM address sets EEADR
	movf	BRIGHT_X,w	; stored brightness
	call	EWRITE		; EEPROM store
	goto	DEL_RESP
SET_RATE_1				; ls bits set for rate
	movlw	EEPROM1		; EEPROM store for rate of dimming
	call	EEREAD		; sets EEADR
	movf	REMOTE_L,w	; remote control number
	andlw	B'00001111'	; mask address bits
	bsf		SHIFT_CNT,0	; set shift bit
	goto	WRI_VAL

; set in EEPROM
SET_RATE_2
	movlw	EEPROM1		; EEPROM store for rate of dimming
	call	EEREAD		; sets EEADR
	movwf	TEMP_VAL
	swapf	TEMP_VAL,f	; rotate bits
	movf	REMOTE_L,w	; remote control number
	andlw	B'00001111'	; mask address bits
	iorwf	TEMP_VAL,w	; combine ms and ls bits 
	clrf	SHIFT_CNT	; return shift
; convert to binary from BCD
	movwf	REG_2		; ls byte BCD count (1 to 99)
	call	BCD_BIN		; convert
	movf	L_BYTE,w	; binary result
WRI_VAL
	call	EWRITE		; write value
DEL_RESP
	movlw	D'100'		; time 10ms x value
	movwf	PGM_TMR		; delay
PGM_LED
	bcf		PORTA,0		; Prgm LED off	
	btfss	PGM_TMR,2	; time for program LED to light
	bsf		PORTA,0		; program LED on
	movf	PGM_TMR,w
	btfss	STATUS,Z
	goto	PGM_LED
; when zero out
	bcf		PORTA,0	
	goto	CLR_RMF

; VCR decoding VCR1. Note VCR1 code for Mute is the same as TV1 mute. 
	
VCR_1
	xorlw	B'00110001'	; compare with start bits and ms 3-bits of address
	btfss	STATUS,Z	; if zero then matching
	goto	CLR_RMF		; clear remote flag

	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'01001100'	; bits 0-5 keycode (12), bit 7 and bit 6 are ls address bits
	btfss	STATUS,Z
	goto	VCR_LOOP
	clrf	SHIFT_CNT	; shift enter count
	bsf		AUT_DIM_FLG,1; set flag
	btfsc	FLAG_ON,0	; is off flag on
	goto	AUTO_DN1
	bsf		FLAG_ON,3	; dim up flag
	
	goto	AUTO_UP

VCR_LOOP
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'01100000'	; bits 0-5 keycode (32), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_UP

	movf	REMOTE_L,w
	xorlw	B'01100001'	; bits 0-5 keycode (33), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_DN 
	movf	REMOTE_L,w
	xorlw	B'01010000'	; bits 0-5 keycode (16), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	UP_SET
	movf	REMOTE_L,w
	xorlw	B'01010001'	; bits 0-5 keycode (17), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	DN_SET  
	movf	REMOTE_L,w
	xorlw	B'01001101'	; bits 0-5 keycode (13), bit 7 and bit 6 are ls address bits
	btfss	STATUS,Z
	goto	CK_9_VCR
	btfsc	FLAG_ON,0	; is off flag on
	goto	OFF_SET
	bsf		FLAG_ON,3	; soft start dim up flag
	goto	ON_SET
CK_9_VCR
	movf	REMOTE_L,w
	andlw	B'11000000'	; ms bits are address
	xorlw	B'01000000'	; bits 7 and 6 are zero for TV and SAT1, 01 for VCR1
	btfss	STATUS,Z
	goto	CLR_RMF

	movf	REMOTE_L,w
	andlw	B'00111111'	; mask address bits
	sublw	0x09		; if negative bypass
	btfss	STATUS,C
	goto	CLR_RMF
	goto	AUTO_DIM_RATE_PGM

; satellite decoding SAT1 & SAT2
	
SAT_COD
SAT_1
	xorlw	B'00110010'	; compare with start bits and ms 3-bits of address
	btfss	STATUS,Z	; if zero then matching
	goto	CLR_RMF		; clear remote flag
	goto	TV_SAME


SAT_2
	xorlw	B'00110010'	; compare with start bits and ms 3-bits of address
	btfss	STATUS,Z	; if zero then matching
	goto	CLR_RMF		; clear remote flag
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'10001100'	; bits 0-5 keycode (12), bit 7 and bit 6 are ls address bits
	btfss	STATUS,Z
	goto	SAT_LOOP
	clrf	SHIFT_CNT	; shift enter count
	bsf		AUT_DIM_FLG,1; set flag
	btfsc	FLAG_ON,0	; is off flag on
	goto	AUTO_DN1
	bsf		FLAG_ON,3	; dim up flag
	
	goto	AUTO_UP

SAT_LOOP
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'10100000'	; bits 0-5 keycode (32), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_UP

	movf	REMOTE_L,w
	xorlw	B'10100001'	; bits 0-5 keycode (33), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_DN 
	movf	REMOTE_L,w
	xorlw	B'10010000'	; bits 0-5 keycode (16), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	UP_SET
	movf	REMOTE_L,w
	xorlw	B'10010001'	; bits 0-5 keycode (17), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	DN_SET  
	movf	REMOTE_L,w
	xorlw	B'10001101'	; bits 0-5 keycode (13), bit 7 and bit 6 are ls address bits
	btfss	STATUS,Z
	goto	CK_9_SAT
	btfsc	FLAG_ON,0	; is off flag on
	goto	OFF_SET
	bsf		FLAG_ON,3	; soft start dim up flag
	goto	ON_SET
CK_9_SAT
	movf	REMOTE_L,w
	andlw	B'11000000'	; ms bits are address
	xorlw	B'10000000'	; bits 7 and 6 are zero for TV and SAT1, 10 for SAT2
	btfss	STATUS,Z
	goto	CLR_RMF

	movf	REMOTE_L,w
	andlw	B'00111111'	; mask address bits
	sublw	0x09		; if negative bypass
	btfss	STATUS,C
	goto	CLR_RMF
	goto	AUTO_DIM_RATE_PGM

CLR_RMF
	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,1		; acknowledge off
	goto	SWITCH

; switch function

; remote control mute (off)	
OFF_SET
	clrf	AUT_DIM_FLG ; auto dimming flag
	movlw	D'30'		; time 20ms x value
	movwf	TOUCH_TMR	
	clrf	R_FLAG		; remote control flags
	bcf		FLAG_ON,7
OFF_C
	clrf	TOUCH_C
	bcf		FLAG_ON,0	; on flag off
	bcf		FLAG_ON,1	; dim down flag
	bsf		FLAG,0		; set off flag
	movlw	D'159'
	movwf	BRIGHT_S

	goto	OUT_FLG1	; delay

; remote control mute (on with soft start turn on)
ON_SET
	clrf	AUT_DIM_FLG ; auto dimming flag
	movlw	D'01'		; time 20ms x value
	movwf	TOUCH_TMR	
	clrf	R_FLAG		; remote control flags
	bcf		FLAG_ON,1
BY_TM
	clrf	TOUCH_C
	bsf		FLAG_ON,0	; on flag set
	clrf	FLAG
	bsf		FLAG_ON,3	; soft start flag
RECYC_6
	movf	TOUCH_TMR,w
	btfss	STATUS,Z	; wait
	goto	RECYC_6	
	movlw	D'06'		; soft start rate (larger is faster)
	movwf	SOFT_ST
	decf	BRIGHT_S,w	; increase brightness setting 
SOFT_T
	sublw	D'34'		; min value
	btfsc	STATUS,C	; if c is 1 then BRIGHT_S at min value
	goto	CLR_DEC		; 
	decf	BRIGHT_S,f
	movf	BRIGHT_S,w
	decfsz	SOFT_ST,f	; decrease counter
	goto	SOFT_T		; continue dec bright_s till counter is zero
	incf	SOFT_ST,f
	movlw	D'1'		; time 20ms x value
	movwf	TOUCH_TMR
	goto	RECYC_6
OUT_FLG1
	movf	TOUCH_TMR,w	; touch TMR
	btfss	STATUS,Z	;
	goto	OUT_FLG1	
	goto	OUT_FLG
	
CLR_DEC
	movlw	D'30'		; time 20ms x value
	movwf	TOUCH_TMR
	bcf		FLAG_ON,3
	bsf		FLAG_ON,7	; has run once flag
	bsf		FLAG_ON,1	; set down dimming
	bcf		FLAG_ON,5	; clear the on/on flag
	bcf		FLAG_ON,4	; clear the stop dimming flag
	goto	OUT_FLG1
CLR_THR
	decf	BRIGHT_S,f
	movf	BRIGHT_S,w
	sublw	D'34'		; max brightness
	btfsc	STATUS,C
	goto	SET_1		; at max so set down dimming	
	movf	BRIGHT_S,w
	sublw	D'159'		; max value (min brightness)
	btfss	STATUS,C
	bcf		FLAG_ON,1	; up dimming
	goto	STET_1		; no change
SET_1
	bsf		FLAG_ON,1	; down dimming
STET_1
	bcf		FLAG_ON,3	; clear soft start flag
	decf	BRIGHT_S,f
	movf	BRIGHT_S,w
	sublw	D'34'
	btfss	STATUS,C
	goto	OUT_FLG
	bsf		FLAG_ON,1	; set down dimming
	incf	BRIGHT_S,f
	bsf		FLAG_ON,4	; set stop dimming flag
	bsf		FLAG_ON,7
	clrf	ON_OFFT
	goto	OUT_FLG

; remote control volume up (slow dimmimg up)	
UP_SET
	clrf	AUT_DIM_FLG ; auto dimming flag
	clrf	R_FLAG		; remote control flags
	bcf		FLAG_ON,1	; up/dn dimming flag
	bcf		FLAG_ON,7
	bsf		FLAG_ON,0	; on flag set
	clrf	FLAG
	bcf		FLAG_ON,5
	decf	BRIGHT_S,w	; increase brightness setting 
	sublw	D'34'		; min value
	btfsc	STATUS,C	; if c is 1 then BRIGHT_S at min value
	goto	TOGL_U		; toggle up flag
	decf	BRIGHT_S,f	;
	goto	STO_VU
TOGL_U
	bsf		FLAG_ON,1	; up/down dimming flag
STO_VU
	movf	BRIGHT_S,w	; stored brightness
	movwf	BRIGHT_X
	goto	OUT_FLG

; remote control volume down (slow dimming down)
DN_SET
	clrf	AUT_DIM_FLG ; auto dimming flag
	clrf	R_FLAG		; remote control flags
	bsf		FLAG_ON,1
	bcf		FLAG_ON,7
	bsf		FLAG_ON,0	; on flag set
	clrf	FLAG
	bcf		FLAG_ON,5
	bcf		FLAG_ON,4
	movf	BRIGHT_S,w	; current brightness 
	sublw	D'159'		; min brightness  
	btfss	STATUS,C	; light is brighter than setting if c is 0
	goto	TOGL_D		; toggle down flag & do nothing to brightness
	incf	BRIGHT_S,f	; decrease brightness
; store current brightness
STO_V
	movf	BRIGHT_S,w	; stored brightness
	movwf	BRIGHT_V
	goto	OUT_FLG
TOGL_D
	bcf		FLAG_ON,0
	bcf		FLAG_ON,1	; up/dn dimming flag
	bsf		FLAG,0		; set off flag	
	goto	STO_V

; remote control channel up (fast dimming up)
REM_UP
	clrf	AUT_DIM_FLG ; auto dimming flag
	movlw	D'01'		; time 20ms x value
	movwf	TOUCH_TMR
	clrf	R_FLAG
	bsf		R_FLAG,0	; up flag
	bcf		FLAG_ON,1	; up/dn dimming flag
	bcf		FLAG_ON,7
	btfss	FLAG_ON,0	; was on flag clear ie after an off 
	bsf		FLAG_ON,6
	bsf		FLAG_ON,0	; on flag set
	clrf	FLAG
	bcf		FLAG_ON,5
	decf	BRIGHT_S,w	; increase brightness setting 
	sublw	D'34'		; min value
	btfsc	STATUS,C	; if c is 1 then BRIGHT_S at min value
	goto	MAX_B		; max brightness
	decf	BRIGHT_S,f	;
	goto	OUT_FLG1

MAX_B
	bsf		FLAG_ON,1	; up/down dimming flag
	goto	OUT_FLG1

; remote control channel down (fast dimming down)
REM_DN
	clrf	AUT_DIM_FLG ; auto dimming flag
	movlw	D'1'		; time 20ms x value
	movwf	TOUCH_TMR
	clrf	R_FLAG		; remote control flags
	bsf		R_FLAG,1	; remote down flag
	bsf		FLAG_ON,1
	bcf		FLAG_ON,7
;	btfsc	FLAG_ON,4	; is dimming stop flag set
;	goto	OUT_FLG1
	bsf		FLAG_ON,0	; on flag set
	clrf	FLAG
	bcf		FLAG_ON,5
	bcf		FLAG_ON,4
	movf	BRIGHT_S,w	; current brightness 
	sublw	D'159'		; min brightness  
	btfss	STATUS,C	; light is brighter than setting if c is 0
	goto	TOG_D		; toggle down flag & do nothing to brightness
	incf	BRIGHT_S,f	; decrease brightness
	goto	OUT_FLG1
TOG_D
	bcf		FLAG_ON,0
	bcf		FLAG_ON,1	; up/dn dimming flag	
	bsf		FLAG_ON,4	; dimming stop flag
	bsf		FLAG,0		; set off flag
OUT_FLG
	goto	CLR_RMF

; automatic dimming
AUTO_DN	
	btfsc	REM_FLG,1	; is remote control entered flag set
	goto	CL_CKR
	btfss	FLAG_ON,0	; is off flag on
	goto	AUTO_UP
AUTO_DN1
	movf	TOUCH_TMR,w	; touch TMR
	btfss	STATUS,Z	;
	goto	SWITCH		; wait for timer

	clrf	R_FLAG		; remote control flags
	bsf		FLAG_ON,1
	bcf		FLAG_ON,3
	bcf		FLAG_ON,7
	bsf		FLAG_ON,0	; on flag set
	clrf	FLAG
	bcf		FLAG_ON,5
	bcf		FLAG_ON,4
	movlw	EEPROM0		; stored brightness
	call	EEREAD		
	movwf	BRIGHT_E	; store
	movf	BRIGHT_S,w	; current brightness 
	subwf	BRIGHT_E,w	; stored brightness  
	btfss	STATUS,C	; light is brighter than setting if c is 0
	goto	TOGL_A		; toggle down flag & do nothing to brightness
	incf	BRIGHT_S,f	; decrease brightness
	movlw	EEPROM1		; dimming rate
	call	EEREAD
	movwf	TOUCH_TMR	; delay
	goto	OUT_FLG
TOGL_A
	clrf	AUT_DIM_FLG ; auto dimming flag
	bcf		FLAG_ON,0
	bcf		FLAG_ON,1	; up/dn dimming flag	
	bsf		FLAG_ON,4	; dimming stop flag
	movf	BRIGHT_S,w	; current brightness 
	sublw	D'159'		; min brightness  
	btfss	STATUS,C	; light is brighter than setting if c is 0
	bsf		FLAG,0		; set off flag	
	goto	OUT_FLG

AUTO_UP	

	movf	TOUCH_TMR,w	; touch TMR
	btfss	STATUS,Z	;
	goto	SWITCH		; wait for timer

	clrf	R_FLAG		; remote control flags
	bcf		FLAG_ON,1
	bcf		FLAG_ON,7
	clrf	FLAG
	bcf		FLAG_ON,5
	bcf		FLAG_ON,3
	btfss	FLAG_ON,0	; was on flag clear ie after an off 
	bsf		FLAG_ON,6
	bcf		FLAG_ON,0	; on flag set

	movlw	EEPROM2		; stored brightness
	call	EEREAD		
	movwf	BRIGHT_E	; store
	decf	BRIGHT_S,w	; current brightness 
	subwf	BRIGHT_E,w	; stored brightness  
	btfsc	STATUS,C	; light is brighter than setting if c is 0
	goto	MAX_B_AUTO	; max brightnes

	decf	BRIGHT_S,w	; increase brightness setting 
	sublw	D'34'		; min value
	btfsc	STATUS,C	; if c is 1 then BRIGHT_S at min value
	goto	MAX_B_AUTO	; max brightness
	decf	BRIGHT_S,f	; dim up
	movlw	EEPROM1		; dimming rate
	call	EEREAD
	movwf	TOUCH_TMR	; delay
	goto	OUT_FLG
MAX_B_AUTO
	clrf	AUT_DIM_FLG ; auto dimming flag
	bsf		FLAG_ON,0
	bsf		FLAG_ON,1	; up/dn dimming flag	
	bsf		FLAG_ON,4	; dimming stop flag
	goto	OUT_FLG

; Subroutines
; subroutine to read EEPROM memory

EEREAD	
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bsf		EECON1,RD	; read EEPROM
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP0	; select bank 0
	return

; subroutine to write to EEPROM

EWRITE
	bsf		STATUS,RP0	; select bank 1
	movwf	EEDATA		; data register
	bcf		INTCON,GIE	; disable interrupts

	bsf		EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bsf		INTCON,GIE	; enable interrupts
	bcf		EECON1,WREN	; clear WREN bit
WRITE	
	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank 0 
	return				; value written 

; subroutine to convert from BCD to binary
BCD_BIN
; clear initial registers not using full capability of 5-digit BCD to 16-bit binary conversion
	clrf	REG_0		; high BCD number (not using this byte)
	clrf	REG_1		; middle BCD number (not using this byte)

; start conversion
	clrf	H_BYTE		; high order binary result
	movf	REG_0,w
	andlw	0x0F		; mask high nibble
	movwf	L_BYTE 
	call	TIMES_A
	swapf	REG_1,w
	call	TIMES_B
	movf	REG_1,w
	call	TIMES_B
	swapf	REG_2,w
	call	TIMES_B
	movf	REG_2,w
	andlw	0x0F
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf	H_BYTE,f
	return		

TIMES_B
	andlw	0x0F
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf	H_BYTE,f
TIMES_A
	bcf		STATUS,C
	rlf		L_BYTE,w	; times 2
	movwf	L_TEMP
	rlf		H_BYTE,w
	movwf	H_TEMP
; times 8
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
; to mult by 10
	movf	L_TEMP,w
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf	H_BYTE,f
	movf	H_TEMP,w
	addwf	H_BYTE,f
	return	

	end		 	
