;		PICTIMER.asm 	-- Countdown Timer program for PIC16F84A
;		written by Jim Rowe. Last revised 8/10/2005
;		(note that PIC clock is 4MHz, so 1mc = 1us)

 
	list	P=16F84A
	#include "p16f84A.inc"
	__config _HS_OSC & _WDT_OFF & _PWRTE_ON
	
;	define variables at data memory locations
Counter1 equ	h'0C'	; general purpose counter variable 1
Counter2 equ 	h'0D'	; general purpose counter variable 2
Counter3 equ	h'0E'	; general purpose counter variable 3
Keycode	 equ	h'0F'	; where keyswitch scanning code is stored

; storage registers for time values input from keyboard
Time10m	equ		h'10'	; tens of minutes
Time1m	equ		h'11'	; units of minutes
Time10s	equ		h'12'	; tens of seconds
Time1s	equ		h'13'	; units of seconds

; storage registers for time period TA and TB current values
TAmin	equ		h'14'	; TA minutes value
TAsec	equ		h'15'	; TA seconds value
TBmin	equ		h'16'	; TB minutes value
TBsec	equ		h'17'	; TB seconds value
T50ms	equ		h'18'	; timer current 50ms count

; storage regs for TimeA & TimeB settings
TAminset 	equ h'19'	; TA minutes setting (0-99)
TAsecset	equ h'1A'	; TA seconds setting (0-59)
TBminset	equ h'1B'	; TB minutes setting (0-99)
TBsecset	equ h'1C'	; TB seconds setting (0-59)

;storage register for current (TA + TB) cycle & Cycles setting
Cycles	equ		h'1D'	; number of cycles
CyclesSet	equ h'1E'	; cycles setting (1-99 or 00 for continuous)

;storage register for timer run flag, other flags
Flags	equ		h'1F'	; Bit 0 = run flag (1 = timer running)
						; bit 1 = IMode flag (1 = input entry mode)
						; bit 2 = Time A flag (1 = time A programmed)
						; bit 3 = Time B flag (1 = time B programmed)
						; bit 4 = AB flag (0 = time A,1 = time B)
						; bit 5 = cycles flag (1 = Cycles > 1)
						; bit 6 = display change flag (1 = changed)
						
;temporary storage registers
Temp	equ		h'20'	; working storage 1
Temp2	equ		h'21'	; working storage 2
Temp3	equ		h'22'	; working storage 3

;storage regs for the various display digits (ASCII code)
Ta10min		equ		h'23'	; Time A tens of mins
Ta1min		equ		h'24'	; Time A units of minutes
Ta10sec		equ		h'25'	; Time A tens of seconds
Ta1sec		equ		h'26'	; Time A units of seconds

Tb10min		equ		h'27'	; Time B tens of mins
Tb1min		equ		h'28'	; Time B units of minutes
Tb10sec		equ		h'29'	; Time B tens of seconds
Tb1sec		equ		h'2A'	; Time B units of seconds

Tin10min	equ		h'2B'	; time input tens of minutes
Tin1min		equ		h'2C'	; time input units of minutes
Tin10sec 	equ		h'2D'	; time input tens of seconds
Tin1sec		equ		h'2E'	; time input units of seconds

Cycles10	equ		h'2F'	; Tens of cycles (current)
Cycles1		equ		h'30'	; Units of cycles (current)
CycSet10	equ		h'31'	; Tens of cycles (setting)
CycSet1		equ		h'32'	; Units of cycles (setting)

; two more working data registers (for byte splitting)
Byte10	equ		h'33'	; tens of units digit
Byte1	equ		h'34'	; units digit

; regs for saving context during interrupt servicing
WSave	equ		h'35'	; w reg stored here
SSave	equ		h'36'	; status reg stored here

; -----------------------------------------------------------------
; now program begins

	ORG		h'00'		; normal start vector
	GOTO Initialise
	ORG		h'04'		; interrupt servicing vector
	GOTO IntService
	
Initialise:
	BCF STATUS, RP0		; set for Bank0
	CLRF PORTB			; clear PortB
	CLRF PORTA			; also PortA (turns off piezo, relays)
	CLRF INTCON			; disable all interrupts also
	BSF STATUS, RP0		; now set for Bank1
	MOVLW h'0F'			; set RB4-7 as outputs, RB0-3 as inputs
	MOVWF TRISB			; by loading into its config register
	MOVLW h'00'			; also set RA0-4 as outputs
	MOVWF TRISA			;
	MOVLW h'87'			; now disable PortB pullups,
	MOVWF OPTION_REG	; also set for TMR0, int clock & PS = 256
	BCF STATUS, RP0		; finally reset for Bank0
	CALL WipeSlate		; initialise all timer settings
	CALL CheckMem		; check EEPROM for saved settings, get 'em if so
	CALL ResetA			; reset both time periods and T50ms count
	CALL ResetB
	CALL GetDigits		; now work out display digits for Ta, Tb, Cycles
	CALL SetCycles		; also get Cycles setting digits
	CALL DispInit		; initialise LCD display module
	CALL DispGreeting	; then display startup message for about 3 secs
	BSF Flags,6			; set 'disp changed' flag so it shows init values
	
LoopStart:
	BTFSS Flags,6		; check if display has changed
	GOTO $+6			; otherwise don't bother to update
	BTFSS Flags,1		; changed, but now test if input mode flag bit
	GOTO $+3			; not set, so go for normal display mode
	CALL DispInput		; was set, so go for input display mode
	GOTO $+2			; and skip to do a key scan
	CALL DispUpdate		; update normal mode displays on LCD module
	CALL DoAScan		; now go scan keyboard
	MOVF Keycode,1		; check if a key was pressed
	BTFSC STATUS,Z 		; skip if Keycode wasn't zero (Z = 0)
	GOTO LoopStart		; Keycode = 0, so no key pressed. Back to start
	MOVLW h'7F'			; Key pressed, so see if code is a del (stop)
	XORWF Keycode,0		; w will now have 0 if it was a match...
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a del, and w -> 0)
	GOTO $+3			; Z=0 (because it wasn't a del), so continue
	CALL StopTimer		; it was a del, so go stop timer etc.
	GOTO LoopStart		; then back to start of loop
	MOVLW "G"			; not del, so check if it's a G (start)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a G)
	GOTO $+3			; Z=0, so keep looking
	CALL StartTimer		; w -> 0, so it is a G: go start timer
	GOTO LoopStart		; then back to start of loop 
	MOVLW "M"			; check if it's an M (tens of minutes)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was an M)
	GOTO $+3			; Z=0, so keep looking
	CALL Add10min		; w -> 0, so it was an M; go add 10 to mins
	GOTO LoopStart		; and back to start of loop
	MOVLW "m"			; check if it's an m (units of mins)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was an m)
	GOTO $+3			; Z=0, so keep looking
	CALL Add1min		; w -> 0, so it was an m; go add 1 to mins
	GOTO LoopStart		; and back to start of loop
	MOVLW "S"			; check if it's an S (tens of secs)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (because it was an S)
	GOTO $+3			; otherwise keep looking
	CALL Add10sec		; w -> 0, so it was an S; go add 10 to secs
	GOTO LoopStart		; and back to start of loop
	MOVLW "s"			; check if it's an s (units of seconds)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (because it was an s)
	GOTO $+3			; otherwise keep looking
	CALL Add1sec		; w ->0, so it was an s; go add 1 to secs
	GOTO LoopStart		; and back to start of loop
	MOVLW "A"			; check if it's an A ( to save time period A)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (because it was an A)
	GOTO $+3			; otherwise keep looking
	CALL TASave			; was an A, so set time for period A
	GOTO LoopStart		; and back to start of loop
	MOVLW "B"			; check if it's a B (time period B)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (because it was a B)
	GOTO $+3			; otherwise keep looking
	CALL TBSave			; was a B, so set time for period B
	GOTO LoopStart		; and back to start of loop
	MOVLW "C"			; check if it's a C (number of cycles)
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (because it was a C)
	GOTO $+3			; otherwise keep looking
	CALL CycSave		; was a C, so set mins for number of cycles
	GOTO LoopStart		; and back to start of loop
	MOVLW h'0D'			; finally check for an Enter
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (because it was an 0Dh)
	GOTO LoopStart		; otherwise back to start (unknown code)
	CALL SaveSettings	; was an 0Dh, so save all settings in EEPROM
	GOTO LoopStart		; and then back to start of loop
;	
;   Program's main loop ends here -- subroutines follow
;   ---------------------------------------------------------
;
Add10min:
	;routine to add 10min to current input time value
	BSF Flags,1			; first make sure we're now in input mode
	INCF Time10m, 1		; then add 1 to Time10m
	MOVF Time10m,0	    ; now check if over 9
	SUBLW h'09'			; w = 9 - Time10m
	BTFSC STATUS,C		; skip if C=0 (ie., Time10m >9)
	GOTO $+2			; no, so set flag & return
	CLRF Time10m		; reset Time10m to zero
	BSF Flags,6			; set 'display changed' flag
	RETURN				; before returning 
	
Add1min:
	;routine to add 1min to current input time value
	BSF Flags,1			; first make sure we're in input mode
	INCF Time1m, 1		; add 1 to Time1m
	MOVF Time1m,0	    ; now check if over 9
	SUBLW h'09'			; w = 9 - Time1m
	BTFSC STATUS,C		; skip if C=0 (ie., Time1m >9)
	GOTO $+2			; no, so set flag & return
	CLRF Time1m			; reset Time1m to zero
	BSF Flags,6			; set change flag
	RETURN				; before returning
	 	
Add10sec:
	;routine to add 10sec to current input time value
	BSF Flags,1			; first make sure we're in input mode
	INCF Time10s, 1		; add 1 to Time10s
	MOVF Time10s,0	    ; now check if over 5
	SUBLW h'05'			; w = 5 - Time10s
	BTFSC STATUS,C		; skip if C=0 (ie., Time10s >5)
	GOTO $+2			; no, so set flag & return
	CLRF Time10s		; reset Time10s to zero
	BSF Flags,6			; set change flag
	RETURN				; before returning 
		
Add1sec:
	;routine to add 1sec to current input time value
	BSF Flags,1			; first make sure we're in input mode
	INCF Time1s, 1		; add 10 to Time1s
	MOVF Time1s,0	    ; now check if over 9
	SUBLW h'09'			; w = 9 - Time1s
	BTFSC STATUS,C		; skip if C=0 (ie., Time1s >9)
	GOTO $+2			; no, so set flag & return
	CLRF Time1s			; reset Time1s to zero
	BSF Flags,6			; set change flag
	RETURN				; before returning
	
BeepDelay:
	MOVLW h'00'			; set Counter2 for 255 loops (255ms approx)
	MOVWF Counter2
	CALL Delay1ms		; go away for 1ms
	DECFSZ Counter2, 1	; then decrement Counter2 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; reached zero, so return
	
BigWait:
	;routine to pause for about 3 seconds, to display startup message
	MOVLW h'0C'			; set Counter3 for 12 outer loops
	MOVWF Counter3		; (12 x 255ms = approx 3.06 sec)
	CALL BeepDelay		; go away for 255ms
	DECFSZ Counter3,1	; then decrement Counter 2 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; did reach zero, so return
	
BounceWait:
	MOVLW h'C8'			; set Counter 2 for 200 loops (200ms approx)
	MOVWF Counter2
	CALL Delay1ms		; go away for 1ms
	DECFSZ Counter2, 1	; then decrement Counter2 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; reached zero, so return
	
ByteSplit:
	; routine to split a byte in w (value 0-99) into two decimal
	; digits which are placed into Byte10 and Byte1
	ANDLW h'7F'			; first make sure byte value is less than 128
	MOVWF Temp3			; then save it in Temp3
	CLRF Byte10			; initialise output digit values
	CLRF Byte1
Beg10:
	MOVLW h'0A'			; subtract 10d from input byte in Temp3
	SUBWF Temp3,0		; w = Temp3 - 0Ah
	BTFSS STATUS,C		;skip if we got a carry (Temp3 >= 10d)
	GOTO Beg1			;no carry, so Temp3 < 10d. Skip to Beg1
	BTFSS STATUS,Z		;carry was set, but check for zero
	GOTO $+3			;no zero, so continue
	INCF Byte10,1		;0 and C, so must have been 10 exactly
	RETURN				;so just increment 10s & leave
	INCF Byte10,1		;positive result, so increment 10s
	MOVWF Temp3			;save remainder in w back to Temp3
	GOTO Beg10			;and continue looping
Beg1:
	INCF Byte1,1		;increment units -- must have at least one
	DECF Temp3,1		;now decrement Temp3
	BTFSS STATUS,Z		;skip if zero bit is set now
	GOTO Beg1			;not zero yet, so continue
	RETURN				;zero, so we're done -- leave
	 
CheckMem:
	;routine to check if any settings in EEPROM, & if so restore 'em
	CLRW				; set w for first address in EE memory (00h)
	MOVWF Temp			; and save in Temp 
	CALL ReadEE			; then read back data (returns in w)
	XORLW h'0F'			; check if it's first flag (0Fh)
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a match)
	RETURN				; mustn't have been a match, so leave now
	INCF Temp,1			; now set for second flag address
	MOVF Temp,0			; copy into w
	CALL ReadEE			; and read data
	XORLW h'F0'			; check if it's second flag (F0h)
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a match)
	RETURN				; no second match, so leave now
	INCF Temp,1			; now set for first data address
	MOVF Temp,0			; copy into w
	CALL ReadEE			; read it
	MOVWF TAminset		; and save as TA minutes
	INCF Temp,1			; now repeat for next address
	MOVF Temp,0
	CALL ReadEE			; read it
	MOVWF TAsecset		; and save as TA seconds
	INCF Temp,1			; now repeat for next address
	MOVF Temp,0
	CALL ReadEE			; read it
	MOVWF TBminset		; and save as TB minutes
	INCF Temp,1			; now repeat for next address
	MOVF Temp,0
	CALL ReadEE			; read it
	MOVWF TBsecset		; and save as TB seconds
	INCF Temp,1			; now repeat for next address
	MOVF Temp,0
	CALL ReadEE			; read it
	MOVWF CyclesSet		; and save as Cycles setting	
	INCF Temp,1			; now repeat for next address
	MOVF Temp,0
	CALL ReadEE			; read it
	MOVWF Flags			; and save as flags setting
	RETURN				; before returning with settings restored
	
ClearLCD:
	;routine to clear LCD and reset address counter
	MOVLW h'01'			; clear display & reset addr ptr
	CALL DispAddress
	CALL Delay160ms		; pause 160ms to give it time to clear
	CALL Delay160ms		; and again, just for tardy LCDs
	RETURN				; then return	
	
CycSave:
	;routine to set and save number of (TA + TB) timing cycles
	BSF Flags,5			; set Cycles flag to show >1 programmed
	MOVF Time10m,0		; fetch tens of minutes input into w
	CALL Multby10		; and multiply by 10
	ADDWF Time1m,0		; now add in the units
	MOVWF CyclesSet		; and save in Cycles setting
	CALL SetCycles		; also save in cycles setting digits
	BCF Flags,1			; finally reset input mode flag
	CALL ClearLCD		; then clear LCD
	BSF Flags,6			; but set 'display changed' flag
	RETURN				; before returning
	
Delay1ms:
	;routine to delay approx 1ms before returning
	MOVLW h'00'			; set Counter1 for 255 loops
	MOVWF Counter1
	NOP
	DECFSZ Counter1, 1	; decrement Counter1 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; reached zero, so return
	
Delay160ms:
	;routine to delay approx 160ms before returning
	MOVLW h'A0'			; set Counter2 for 160 outer loops
	MOVWF Counter2
	GOTO $+3			; then skip to start of inner loop
Delay10ms:
	;routine to delay approx 10ms before returning
	MOVLW h'0A'			; set Counter2 for 10 outer loops
	MOVWF Counter2
	CALL Delay1ms		; then wait 1ms
	DECFSZ Counter2, 1	; then decrement Counter2 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; reached zero, so return	
	
DispAddress:
	;routine to load display address (in w reg) into LCD module
	BCF PORTA,4			; first set RS pin of LCD low, for instr/addr
	CALL Nibbles2LCD	; then send addr nibbles to LCD
	BCF PORTA,4			; make sure RS is low
	GOTO $+4 			; then jump to delay 160us before return
	
DisplayData:
	;routine to display a data byte in w reg at the current LCD address
	BSF PORTA,4			; RS pin of LCD high, for sending data
	CALL Nibbles2LCD	; then send data nibbles to LCD
	BCF PORTA,4			; RS pin of LCD low again
BusyWait:
	; routine to wait intil LCD module is not busy, after writing
	MOVLW h'28'			; set delay counter for 40 loops
	MOVWF Counter1		; (should give about 160us)
	NOP
	DECFSZ Counter1,1	; decrement counter & skip when zero
	GOTO $-2			; loop back until we reach zero
	RETURN				; then return

DispGreeting:
	;routine to display startup message for a short time
	MOVLW h'82'			; first set address to line 1, char 3
	CALL DispAddress
	MOVLW "S"			; then send "Silicon Chip"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "c"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW h'20'
	CALL DisplayData
	MOVLW "C"
	CALL DisplayData
	MOVLW "h"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "p"
	CALL DisplayData
	MOVLW h'C2'			; now move down to line 2, char 3
	CALL DispAddress
	MOVLW "F"			; and send "FlexiTimer 3"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "x"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "T"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "m"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW h'20'
	CALL DisplayData
	MOVLW "3"
	CALL DisplayData
	CALL BigWait		; then pause about 3 sec, so it's seen
	CALL ClearLCD		; finally clear LCD again
	RETURN				; before leaving
	
DispInit:
	;routine to initialise LCD display module
	CALL Delay160ms		; pause about 160ms before proceeding
	CLRF PORTA			; then set LCD's RS line low for instr/addr
	MOVLW h'30'			; load init code into RB4-7
	MOVWF PORTB
	CALL ToggleEN		; then toggle EN to write to LCD
	CALL Delay10ms		; then wait about 10ms
	BCF PORTA,4			; make sure RS is still low
	BCF PORTB,4			; now drop RB4 (instr code 30 -> 20h)
	CALL ToggleEN		; toggle EN to write
	CALL Delay10ms		; pause 10ms again	
	MOVLW h'28'			; now set LCD functions (after this, 4-bit i/f)
	CALL DispAddress	; (4 bit i/f, 2 line display, 5x7 chars)
	MOVLW h'0C'			; also set display mode (display on,
	CALL DispAddress	; blink off, cursor off)
	CALL ClearLCD		; then clear LCD screen
	MOVLW h'06'			; and finally set entry mode
	CALL DispAddress	; (increment address, no display shift)
	RETURN				; should now be set up & ready to go, so leave
	
DispInput:
	; routine to display input from time setting keys, in input mode
	CALL GetInDigits	; first update input mode digits
	CALL ShowInput		; then display "Input: "
	MOVF Tin10min,0		; followed by Input 10m and 1m digits
	CALL DisplayData
	MOVF Tin1min,0
	CALL DisplayData
	MOVLW ":"			; followed by a colon
	CALL DisplayData
	MOVF Tin10sec,0		; and Input 10s and 1s digits
	CALL DisplayData
	MOVF Tin1sec,0
	CALL DisplayData
	BCF Flags,6			; then clear 'display changed' flag
	RETURN				; and return
	
DispSetSave:
	;routine to display startup message for a short time
	CALL ClearLCD		; first clear LCD
	MOVLW h'81'			; then set address to line 1, char 2
	CALL DispAddress
	MOVLW "S"			; and send "Settings saved"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW "g"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW h'20'
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW "a"
	CALL DisplayData
	MOVLW "v"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "d"
	CALL DisplayData
	CALL BigWait		; then pause about 3 sec, so it's seen
	CALL ClearLCD		; finally clear LCD again
	RETURN				; before leaving
	
DispUpdate:
	;routine to update all LCD character displays (not input mode)
	CALL GetDigits		; first update Ta, Tb & Cycles display digits
	MOVLW h'80'			; and reset LCD to line1, char1
	CALL DispAddress
	CALL ShowTimeA		; then display "Ta"
	MOVF Ta10min,0		; now fetch Ta tens of mins digit into w
	CALL DisplayData	; and display
	MOVF Ta1min,0		; then do Ta units of mins
	CALL DisplayData
	MOVLW ":"			; followed by colon
	CALL DisplayData
	MOVF Ta10sec,0		; then Ta tens of secs
	CALL DisplayData
	MOVF Ta1sec,0		; and units of secs
	CALL DisplayData
	MOVLW h'20'			; then a space
	CALL DisplayData
	BTFSS Flags,3		; skip if TB flag set
	GOTO Blankend		; flag is zero (i.e., no TB set)
	CALL ShowTimeB		; flag is set, so display "Tb"
	MOVF Tb10min,0		; now fetch Tb tens of mins
	CALL DisplayData	; and display
	MOVF Tb1min,0		; then do Tb units of mins
	CALL DisplayData
	MOVLW ":"			; followed by a colon
	CALL DisplayData
	MOVF Tb10sec,0		; then Tb tens of secs
	CALL DisplayData
	MOVF Tb1sec,0		; and units of secs
	CALL DisplayData
	GOTO Line2
Blankend:
	CALL Show7Spaces	; Tb not programmed, so space to line end
Line2:	
	MOVLW h'C0'			; now set LCD for line2, char1
	CALL DispAddress
	MOVF CyclesSet,1	; check if set for continuous operation
	BTFSC STATUS,Z		; skip if Z=0, because CyclesSet not 00
	GOTO ShowContin		; Z=1, so CyclesSet = 0 (continuous mode)
	MOVLW "C"			; not continuous, so send "Cycle:"
	CALL DisplayData
	MOVLW "y"
	CALL DisplayData
	MOVLW "c"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW ":"
	CALL DisplayData
	MOVF Cycles10,0		; get tens of cycles
	CALL DisplayData	; and display
	MOVF Cycles1,0		; then get units of cycles
	CALL DisplayData	; and display
	MOVLW "/"			; followed by a slash
	CALL DisplayData
	MOVF CycSet10,0		; tens of cycles set
	CALL DisplayData
	MOVF CycSet1,0		; and units of cycles set
	CALL DisplayData
	GOTO LastFive		; then skip to end of line
ShowContin:
	MOVLW "C"			; we are in cintinuous mode, so show "Cont"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	CALL Show7Spaces	; followed by seven space filler
LastFive:
	MOVLW h'20'			; next show one space
	CALL DisplayData
	BTFSS Flags,0		; skip if runflag set (ie timer is running)
	GOTO ShowStop		; Z=1, so flag is zero. Go show stopped
	BTFSS Flags,4		; skip if ABflag set (means Time B)
	GOTO $+4			; ABflag = 0 (means Time A)
	CALL Show2Spaces	; in Time B, so blank next 2 chars
	CALL ShowTimeB		; and then display the Time B tag
	GOTO $+3			; before leaving
	CALL ShowTimeA		; Time A, so display its tag instead
	CALL Show2Spaces	; then blank out next 2 chars
	BCF Flags,6			; and finally clear 'display changed' flag
	RETURN				; before leaving
ShowStop:
	MOVLW "S"			; display "Stop" at end of line 2
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "o"
	Call DisplayData
	MOVLW "p"
	CALL DisplayData
	BCF Flags,6			; clear 'display changed' flag
	RETURN				; then return
	
DoAScan:
	; routine to scan keys, verify value for valid keypress
	; returns with code (or zero if no key pressed) in Keycode
	CALL Scankeys		; go scan keyboard
	MOVWF Keycode		; then save return value in w to Keycode
	MOVF Keycode,1		; now check if Keycode = 0
	BTFSC STATUS, Z		; skip if Z=0 (ie Keycode not 0)
	RETURN				; otherwise return, as no key pressed
	CALL BounceWait		; key was pressed, so wait for 100ms
	CALL Scankeys		; & scan again to verify
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS,Z		; skip if Z=0 (ie w not 0, has a keycode)
	RETURN				; was zero, so false alarm - exit 
	XORWF Keycode, 0	; not zero, so compare with first code
	BTFSC STATUS,Z		; skip if Z=0 (because codes didn't match)
	RETURN				; did match, so return with code in Keycode
	CALL BounceWait		; no match, so wait another 100ms
	GOTO DoAScan		; and back to scan again
	
GetASCII:
	; routine to accept numeral byte in w & convert it
	; to ASCII code for display. Returns with the code in w
	ANDLW h'0F'			; first mask off upper nibble, if any
	IORLW h'30'			; now make upper nibble h'30'
	RETURN				; that's all -- return with it in w
	
GetDigits:
	; routine to work out display digits from current TimeA, TimeB
	; & Cycles bytes in normal (non-input) mode
	MOVF TAmin,0		; first get TA minutes in w
	CALL ByteSplit		; and extract its digits to Byte10, Byte1
	MOVF Byte10,0		; then get 10s digit value in w
	CALL GetASCII		; convert to ASCII
	MOVWF Ta10min		; and save for display
	MOVF Byte1,0		; followed by the 1's digit
	CALL GetASCII
	MOVWF Ta1min
	MOVF TAsec,0		; now do the same for the TA seconds
	CALL ByteSplit
	MOVF Byte10,0
	CALL GetASCII
	MOVWF Ta10sec
	MOVF Byte1,0
	CALL GetASCII
	MOVWF Ta1sec
	MOVF TBmin,0		; and repeat process for the TB minutes
	CALL ByteSplit
	MOVF Byte10,0
	CALL GetASCII
	MOVWF Tb10min
	MOVF Byte1,0
	CALL GetASCII
	MOVWF Tb1min
	MOVF TBsec,0		; and the TB seconds
	CALL ByteSplit
	MOVF Byte10,0
	CALL GetASCII
	MOVWF Tb10sec
	MOVF Byte1,0
	CALL GetASCII
	MOVWF Tb1sec
	MOVF Cycles,0		; and the current cycles
	CALL ByteSplit
	MOVF Byte10,0
	CALL GetASCII
	MOVWF Cycles10
	MOVF Byte1,0
	CALL GetASCII
	MOVWF Cycles1
	RETURN				; before returning
	
GetInDigits:
	; routine to get input display digits when in input mode
	MOVF Time10m,0		; first get tens of minutes in w
	CALL GetASCII		; then get its digit code
	MOVWF Tin10min		; & save it
	MOVF Time1m,0		; then do likewise for the 1's digit
	CALL GetASCII
	MOVWF Tin1min
	MOVF Time10s,0		; & the tens of seconds
	CALL GetASCII
	MOVWF Tin10sec
	MOVF Time1s,0		; & the units of seconds
	CALL GetASCII
	MOVWF Tin1sec	
	RETURN				; now we're done, so return
	
Nibbles2LCD:
	; routine to separate nibbles of data byte in w, send to LCD
	MOVWF Temp2			; first save byte in Temp2
	ANDLW h'F0'			; also strip off lower nibble
	MOVWF PORTB			; & send hi nibble to LCD via RB4-7
	CALL ToggleEN		; then toggle EN line to write into LCD	
	SWAPF Temp2,0		; now get byte back in w, swapping nibbles
	ANDLW h'F0'			; strip off lower nibble
	MOVWF PORTB			; & send hi (low) nibble to LCD via RB4-7
	CALL ToggleEN		; toggle EN line again, to write into LCD
	RETURN				; then return
	
Multby10:
	; routine to multiply number N in w reg by 10d, then return
	; with result of 10*N in w reg
	MOVWF Temp			; save N in Temp
	BCF STATUS,C		; now clear carry bit
	RLF Temp,1			; rotate Temp L so it now has 2N
	MOVF Temp,0			; 2N is now in w reg also
	BCF STATUS,C		; clear carry bit again
	RLF Temp,1			; rotate Temp L so it now has 4N
	BCF STATUS,C		; clear carry bit again
	RLF Temp,1			; rotate Temp L so it now has 8N
	BCF STATUS,C		; clear carry bit again
	ADDWF Temp,0		; then add Temp to w, so w now has 10N
	RETURN				; then return
	
PutInEE:
	; routine to write data byte in w into current EEADR in
	; EE memory, also increment memory address in EEADR
	MOVWF EEDATA		; first move data into EEDATA
	CALL WriteEE		; then write it and
	INCF EEADR,1		; finally increment address
	RETURN				; before returning
	
ReadEE:
	; routine to read data byte from EE memory address (in w)
	MOVWF EEADR			; first, set up address register
	BSF STATUS,RP0		; next switch to Bank1
	BSF EECON1,RD		; so we can set the read control bit
	BCF STATUS,RP0		; then switch back to Bank0
	MOVF EEDATA,0		; and read the data into w reg
	RETURN				; before we leave
	
RelayOff:
	;routine to turn off relays at end of current time period
	BCF PORTA,0			; turn off Relay 1
	BCF PORTA,1			; and also Relay 2
	RETURN				; before returning
	
RelayOn:
	;routine to turn relay on at start of current time period
	BTFSC Flags,4		; check AB flag (0=TA, 1=TB)
	GOTO $+3			; flag set, so skip 2
	BSF PORTA,0			; flag clear, so TA -- turn on Relay 1
	RETURN				; before returning
	BSF PORTA,1			; flag set, so TB -- turn on Relay 2
	RETURN				; before returning
	
SaveSettings:
	;routine to save current settings in data EE memory
	CLRW				; first set starting address
	MOVWF EEADR
	MOVLW h'0F'			; then write first mem flag byte
	CALL PutInEE
	MOVLW h'F0'			; then write second mem flag byte
	CALL PutInEE
	MOVF TAminset,0		; fetch TA minutes setting
	CALL PutInEE		; and write it
	MOVF TAsecset,0		; now do the same for TA seconds
	CALL PutInEE
	MOVF TBminset,0		; and TB minutes
	CALL PutInEE
	MOVF TBsecset,0		; and TB seconds
	CALL PutInEE
	MOVF CyclesSet,0	; and Cycles
	CALL PutInEE
	MOVF Flags,0		; and flags
	CALL PutInEE
	CALL DispSetSave	; finally give confirmation message
	BSF Flags,6			; and set 'display changed' flag
	RETURN				; before leaving		
	
Scankeys:
	; routine to look for a pressed key, return with code in w
	; (or with 0 in w if no key is pressed)
	MOVLW h'80'			; enable top row output
	MOVWF PORTB
	NOP					; short delay
	NOP
	BTFSC PORTB, 0		; check if column0 key is pressed
	RETLW "s"			; yes, so return with 73h in w reg
	BTFSC PORTB, 1		; no, so what about column1?
	RETLW "S"			; yes, so return with 53h in w reg
	BTFSC PORTB, 2		; no, so how about column2?
	RETLW "m"			; yes, return with 6Dh in w reg
	BTFSC PORTB, 3 		; how about column3?
	RETLW "M" 			; yes, return with 4Dh in w reg
	MOVLW h'40'			; nothing yet, so now check next row
	MOVWF PORTB
	NOP
	NOP
	BTFSC PORTB, 0		; check if column0 key is pressed
	RETLW h'0D'			; yes, so return with 0Dh in w reg
	BTFSC PORTB, 1		; no, so what about column1?
	RETLW "C"			; yes, so return with 43h in w reg
	BTFSC PORTB, 2		; no, so how about column2?
	RETLW "B"			; yes, return with 42h in w reg
	BTFSC PORTB, 3 		; how about column3?
	RETLW "A" 			; yes, return with 41h in w reg
	MOVLW h'20'			; nothing yet, so now check bottom row
	MOVWF PORTB
	NOP
	NOP
	BTFSC PORTB, 0		; check if column0 key is pressed
	RETLW h'7F'			; yes, so return with del in w reg
	BTFSC PORTB, 3 		; nothing, but how about column3?
	RETLW "G" 			; yes, so return with 47h in w reg
	RETLW h'00'			; no key found, so return with zero in w reg
	
SetCycles:
	;routine to get CycSet10 & CycSet1 digit codes from CyclesSet
	MOVF CyclesSet,0	; get cycles setting in w
	CALL ByteSplit		; and go turn it into digit values
	MOVF Byte10,0		; then fetch 10's digit
	CALL GetASCII
	MOVWF CycSet10		; and save as Cycles setting 10's
	MOVF Byte1,0		; do likewise for 1's digit
	CALL GetASCII
	MOVWF CycSet1
	RETURN				; before returning
	
ShortBeep:
	;routine to give short beep on piezo (via RA2)
	BSF PORTA, 2		; set bit 2 of Port A, to turn on piezo
	CALL BeepDelay		; then wait about 255ms
	BCF PORTA, 2		; before turning it off again
	RETURN				; and then returning with job done
	
ShowInput:
	;routine to show "Input: " on line 1 of LCD, in input mode
	CALL ClearLCD		; first clear LCD screen
	MOVLW h'82'			; and reset LCD to line1, char3
	CALL DispAddress
	MOVLW "I"			; now send the chars to LCD
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW "p"
	CALL DisplayData
	MOVLW "u"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW ":"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	RETURN				; then done, so return	
	
Show7Spaces:
	; routine to show seven spaces on LCD, starting at curr addr
	MOVLW h'07'
	GOTO $+2
Show2Spaces:
	;routine to show two spaces at current LCD address
	MOVLW h'02'
	MOVWF Counter3
	MOVLW h'20'
	CALL DisplayData
	DECFSZ Counter3,1	; decrement counter, skip until done
	GOTO $-3			; otherwise keep looping & sending spaces
	RETURN
	
ShowTimeA:
	;routine to display label "Ta" at current LCD address
	MOVLW "T"
	CALL DisplayData
	MOVLW "a"
	CALL DisplayData
	RETURN
	
ShowTimeB:
	;routine to display label "Tb" at current LCD address
	MOVLW "T"
	CALL DisplayData
	MOVLW "b"
	CALL DisplayData
	RETURN		
	
StartTimer:
	;routine to start the timer and do associated housekeeping
	BCF Flags,4			; clear ABflag since we're starting
	CALL ResetA			; now reset TA to its setting value
	BTFSS Flags,3		; now check if TB flag is set
	GOTO $+2			; no, so jump to cycles reset
	CALL ResetB			; yes, so reset TB to its settings also
	MOVLW h'01'			; now reset Cycles to 1
	MOVWF Cycles
	CALL RelayOn		; turn on TA relay
	BSF Flags,0			; set run flag
	BSF Flags,6			; and 'display changed' flag
	BCF Flags,1			; but clear input mode flag
	CALL TimerSet		; go set timer TMR0
	BSF INTCON,GIE		; then turn on interrupts
	RETURN				; and return
	
StopTimer:
	;routine to stop the timer and do associated housekeeping
	BCF INTCON,GIE		; turn off all interrupts (stops timer)
	BCF Flags,0			; clear Run flag
	CALL DispUpdate		; update displays
	CALL RelayOff		; turn off relays
	CALL ShortBeep		; and go give a short beep on piezo
	RETURN
	
TASave:
	;routine to set time period TA from input regs
	BSF Flags,2			; set TA flag to show it's programmed
	MOVF Time10m,0		; fetch tens of minutes input into w
	CALL Multby10		; and multiply by 10
	ADDWF Time1m,0		; now add in the units
	MOVWF TAmin			; and save total in TAmin
	MOVWF TAminset		; also in Taminset 
	MOVF Time10s,0		; now fetch tens of seconds input into w
	CALL Multby10		; multiply by 10
	ADDWF Time1s,0		; add in units of seconds
	MOVWF TAsec			; and save total in TAsec
	MOVWF TAsecset		; as well as TAsecset
	BCF Flags,1			; finally clear input mode flag
	CALL ClearLCD		; clear LCD
	BSF Flags,6			; and set 'display changed' flag
	RETURN				; before returning
		
TBSave:
	;routine to set time period TB from input regs
	MOVF Time10m,0		; fetch tens of minutes input into w
	CALL Multby10		; and multiply by 10
	ADDWF Time1m,0		; now add in the units
	MOVWF TBmin			; and save total in TBmin
	MOVWF TBminset		; as well as TBminset 
	MOVF Time10s,0		; now fetch tens of seconds input into w
	CALL Multby10		; multiply by 10
	ADDWF Time1s,0		; add in units of seconds
	MOVWF TBsec			; and save total in TBsec
	MOVWF TBsecset		; as well as TBsecset
	BTFSS STATUS,Z		; now -- skip if Z=1 (ie., TBsecset was zero)
	GOTO $+6			; wasn't zero, so just set flags & go
	MOVF TBminset,1		; was zero, so check TBminset as well
	BTFSS STATUS,Z		; skip if it was zero too
	GOTO $+3			; wasn't zero, to just set flags & go
	BCF Flags,3			; it was zero too, so clear TB flag
	GOTO $+2			; then set other flags & go
	BSF Flags,3			; set TB flag to show it's programmed	
	BCF Flags,1			; finally reset input mode flag
	CALL ClearLCD		; and clear LCD
	BSF Flags,6			; then set 'display changed' flag
	RETURN				; before returning
	
TimerSet:
	;routine which preloads Timer0 for 49.92ms and enables T0IE
	MOVLW h'3D'			; preload Timer0 with 61d, for correct time
	MOVWF TMR0			; (also clears prescaler, ready to start)
	CLRF INTCON			; now clear all interrupt flags and
	BSF INTCON,5		; enable only T0IE interrupts
	RETURN				; timer is now set, so we can leave
	
ToggleEN:
	;routine to toggle EN line of LCD, to write an instr or data nibble
	BSF PORTA,3			; take LCD's EN line high (RA3)
	NOP					; pause 1us (1mc) to let it stabilise
	BCF PORTA,3			; then low again, to write into LCD contr
	RETURN				; then return
	
WipeSlate:
	; routine to clear all current settings
	CLRF TAminset		; first clear TA and TB setting regs
	MOVLW h'0A'			; (TA default setting = 10 secs)
	MOVWF TAsecset
	CLRF TBminset
	CLRF TBsecset
	CLRF Flags			; then clear all flags
	MOVLW h'01'			; and set for a single cycle only
	MOVWF Cycles		; as both current cycle and setting,
	MOVWF CyclesSet		; before doing last step
InputClear:
	CLRF Time10m		; now clear keyboard input regs
	CLRF Time1m
	CLRF Time10s
	CLRF Time1s
	RETURN				; before leaving
	
WriteEE:
	; routine to write a byte of data into EE memory address
	; (address already set in EEADR register, data in EEDATA)
	BCF INTCON, GIE		; first disable interrupts while we do it
	BSF STATUS, RP0		; then swing over to Bank1
	BSF EECON1, WREN	; enable EE writing
	MOVLW h'55'			; and carry out 'magic sequence'
	MOVWF EECON2
	MOVLW h'AA'
	MOVWF EECON2
	BSF EECON1, WR		; so it does write the byte
	BCF EECON1, WREN	; then disable EE writing again
	BTFSC EECON1, WR	; now check to see if write done
	GOTO $-1			; loop back until it is (WR->0)
	BCF EECON1, EEIF	; we're done, so clear EEIF bit
	BCF STATUS, RP0		; and swing back to Bank0
	RETURN				; before leaving
	
	
; ----------------------------------------------------------------	
	
	; end of main routines -- interrupt servicing routines follow
	
IntService:
	; main routine to service interrupts from
	; Timer 0 (every 49.92ms when timer is going)
	MOVWF WSave			; first save context (w and status regs)
	SWAPF STATUS,0		; using SWAPF here to avoid STATUS change
	MOVWF SSave
	
	BTFSC Flags,4		; now -- skip if ABflag is zero (ie Time A)
	GOTO BUpdate		; not Time A, so go update Time B
	CALL UpdateA		; Time A, so go update it
	BTFSS STATUS,Z		; skip if Z=1 (because TimeA must have ended)
	GOTO Restart		; still going, so just restart timer & return
	CALL RelayOff		; TimeA ended, so turn off relay
	CALL ResetA			; and reset TimeA ready for next time
	BTFSS Flags,3		; skip if TimeB flag set, ie TimeB is prog
	GOTO CheckCycles	; not set, so Time B not prog -- go check cycles
	BSF Flags,4			; TB is prog, so set ABflag for it
	BSF PORTA,1			; turn on Relay 2
	CALL ResetB			; go set up TimeB count
	GOTO Restart		; then restart timer & leave
BUpdate:
	CALL UpdateB		; Time B, so go update it 
	BTFSS STATUS,Z		; skip if Z=1 (Time B must have ended)
	GOTO Restart		; still going, so restart timer & return
	CALL RelayOff		; Time B ended, so turn off relay
	CALL ResetB			; and reset TimeB ready for next time
CheckCycles:
	BTFSS Flags,5		; skip if CycFlag = 1 (ie., Cycles > 1)
	GOTO Shutoff		; CycFlag = 0, so go stop timer
	MOVF CyclesSet,1	; check if CyclesSet = 0
	BTFSC STATUS, Z		; skip if Z=0, i.e., CyclesSet not 0
	GOTO Carryon		; Z=1, so CyclesSet = 0 (continuous mode); skip check
	MOVF Cycles,0		; not continuous, so check if done
	SUBWF CyclesSet,0	; w = CyclesSet - Cycles
	BTFSC STATUS,Z		; skip if Z=0 (means Cycles < CyclesSet)
	GOTO Shutoff		; Z=1, so Cycles = CyclesSet already. End up
Carryon:
	INCF Cycles,1		; not finished yet, so increment Cycles
	BCF Flags,4			; then swing AB flag back to Time A
	CALL RelayOn		; turn on Relay 1
	GOTO Restart		; restart timer & return
	
Shutoff:
	; routine to stop timer, because time & cycles are both up
	BCF INTCON,5		; turn off timer interrupt T0IE (stops timer)
	BCF Flags,0			; clear Run flag
	CALL RelayOff		; turn off relays
	CALL ShortBeep		; give a short beep on piezo
	GOTO $+2			; then return from int (without TMR0 reset)
	
Restart:
	; routine to reset TMR0, set disp changed flag & return from int
	CALL TimerSet		; reset timer TMR0 for 49.92ms, reset T0IE
	BSF Flags,6			; set display changed flag
	SWAPF SSave,0		; now restore context (status & w regs)
	MOVWF STATUS		;
	SWAPF WSave,1		; using SWAPFs to avoid changing STATUS
	SWAPF WSave,0		;
	RETFIE				; and return, enabling ints (TOS->PC, 1->GIE)
	
UpdateA:
	; routine to update (decrement) current Time A by 50ms,
	; check if it ends (->0) & if so return with 0 in w reg & Z=1
	DECFSZ T50ms,1		; decrement T50ms & skip if it reaches zero
	GOTO JustExit		; didn't reach 0, so just return with Z=0
	MOVF TAsec,1		; did reach 0, so now check TAsec
	BTFSC STATUS,Z		; skip if Z=0 (because TAsec <> 0)
	GOTO $+3			; TAsec already zero, so go check TAmins
	DECF TAsec,1		; wasn't zero, so decrement TAsec
	GOTO Reset50ms		; and leave after resetting 50ms counter etc
	MOVF TAmin,0		; TAsec was 0, so check TAmin
	BTFSC STATUS,Z		; skip if Z=0 (because TAmin <> 0)
	RETURN				; Z=1, so TAmin = w = 0. TA has ended, so leave
	DECF TAmin,1		; TAmin <> 0, so decrement it
	MOVLW h'3B'			; now reset TAsec to 59d
	MOVWF TAsec
Reset50ms:
	MOVLW h'14'			; reset T50ms to 20d
	MOVWF T50ms	
	BSF Flags,6		    ; a displayed value has changed, so set flag
JustExit:
	BCF STATUS,Z		; now clear Z so we don't flag end
	RETURN				; before returning
	
UpdateB:
	; routine to update (decrement) current Time B by 50ms,
	; check if it ends (->0) & if so return with 0 in w reg & Z=1
	DECFSZ T50ms,1		; decrement T50ms & skip if it reaches zero
	GOTO JustExit		; didn't reach 0, so just return with Z=0
	MOVF TBsec,1		; did reach 0 (Z=1), so now check TBsec
	BTFSC STATUS,Z		; skip if Z=0 (because TBsec <> 0)
	GOTO $+3			; TBsec already zero, so go check TBmins
	DECF TBsec,1		; wasn't zero, so decrement TBsec
	GOTO Reset50ms		; and leave after resetting 50ms counter etc
	MOVF TBmin,0		; TBsec was 0, so check TBmin
	BTFSC STATUS,Z		; skip if Z=0 (because TBmin <> 0)
	RETURN				; Z=1, so TBmin = w = 0. TB has ended, so leave
	DECF TBmin,1		; TBmin <> 0, so decrement
	MOVLW h'3B'			; now reset TBsec to 59d
	MOVWF TBsec
	GOTO Reset50ms		; and return after resetting 50ms counter etc
	
ResetA:
	; routine to reset Time A to programmed value, ready to go
	MOVF TAminset,0		; get TA minutes setting
	MOVWF TAmin			; and use to reset TAmin
	MOVF TAsecset,0		; now get TA seconds setting
	MOVWF TAsec			; and use to reset TAsec
	MOVLW h'14'			; now reset 50ms count as well
	MOVWF T50ms
	RETURN				; that's all, so leave
	
ResetB:
	; routine to reset Time B to programmed value, ready to go
	MOVF TBminset,0		; get TB minutes setting
	MOVWF TBmin			; and use to reset TBmin
	MOVF TBsecset,0		; now get TB seconds setting
	MOVWF TBsec			; and use to reset TBsec
	MOVLW h'14'			; now reset 50ms count as well
	MOVWF T50ms
	RETURN				; then leave
	 
	 	
	END
	
