/*  
 * USB CDC-ACM Demo
 *
 * This file may be used by anyone for any purpose and may be used as a
 * starting point making your own application using M-Stack.
 *
 * It is worth noting that M-Stack itself is not under the same license as
 * this file.
 *
 * M-Stack is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  For details, see sections 7, 8, and 9
 * of the Apache License, version 2.0 which apply to this file.  If you have
 * purchased a commercial license for this software from Signal 11 Software,
 * your commerical license superceeds the information in this header.
 *
 * Alan Ott
 * Signal 11 Software
 * 2014-05-12
 
 Modified Tim Blythman 2024 to behave as USB-serial bridge with DTR
 //DTR was RC3 in revA PCB
 RX     RC5
 TX     RC4
 DTR    RA5
 RX_LED RC0
 TX_LED RC1
 
 //using MPLAB X 6.15, XC8 1.45 and DFP 1.1.53   
 */

#define _XTAL_FREQ 16000000
#include "usb.h"
#include <xc.h>
#include <string.h>
#include <stdio.h>
#include "flash.h"
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"
//#include "io.h"

#ifdef MULTI_CLASS_DEVICE
static uint8_t cdc_interfaces[] = { 0 };
#endif

#define IN_EP_LIMIT 32

#define XC8_PRO

unsigned short J;
unsigned char K, S;
unsigned long cur_ratio;

static unsigned long calc_scaler(unsigned long desired_ratio, unsigned short* pJ, unsigned char* pK, unsigned char* pPrepostscale) {
  unsigned char J1_3, J4, J5_8, J9_12, J13_16;
  unsigned short temp;
  unsigned long actual_ratio;

  if( desired_ratio < 3 )
    desired_ratio = 3;
  else if( desired_ratio > 21327000 )
    desired_ratio = 21327000;
  if( desired_ratio <= 21327 && (desired_ratio < 20 || desired_ratio%10) ) {
    temp = desired_ratio;
    actual_ratio = desired_ratio;
    *pPrepostscale = 0;
  } else if( desired_ratio <= 213270 ) {
    temp = (desired_ratio + 5) / 10;
    actual_ratio = ((unsigned long)temp)*10;
    *pPrepostscale = 1;
  } else if( desired_ratio <= 2132700 ) {
    temp = (desired_ratio + 50) / 100;
    actual_ratio = ((unsigned long)temp)*100;
    *pPrepostscale = 2;
  } else {
    temp = (desired_ratio + 500) / 1000;
    actual_ratio = ((unsigned long)temp)*1000;
    *pPrepostscale = 3;
  }

  J1_3 = temp & 7;
  temp >>= 3;
  J4 = temp / 1000;
  if( J4 > 1 )
    J4 = 1;
  temp -= (unsigned short)(J4 * 1000);
  J13_16 = temp / 100;
  if( J13_16 > 15 )
    J13_16 = 15;
  temp -= (unsigned short)(J13_16 * 100);
  J9_12 = temp / 10;
  if( J9_12 > 15 )
    J9_12 = 15;
  temp -= (unsigned short)(J9_12 * 10);
  J5_8 = temp;

  *pJ = (unsigned short)((unsigned short)J1_3 | (unsigned short)(J4<<3) | (unsigned short)(J5_8<<4) | (((unsigned short)J9_12)<<8) | (((unsigned short)J13_16)<<12));
  *pK = 4;
  return actual_ratio;
}

#ifdef XC8_PRO
static unsigned long calc_ratio(unsigned short J, unsigned char K, unsigned char Prepostscale) {
  unsigned char J5_8, J9_12, J13_16;
  unsigned short divv, prescale;
  J5_8 = (J>>4)&15;
  J9_12 = (J>>8)&15;
  J13_16 = (J>>12)&15;
  divv = (unsigned short)J5_8 + (unsigned short)(J9_12 * 10) + ((unsigned short)J13_16) * 100;
  switch(Prepostscale) {
  case 0:
    prescale = 1;
    break;
  case 1:
    prescale = 10;
    break;
  case 2:
    prescale = 100;
    break;
  case 3:
    prescale = 1000;
    break;
  default:
    return 0;
  }
  switch(K) {
  case 7:
    divv += (unsigned short)((J>>1)&7) * 1000;
    return ((unsigned long)prescale) * (divv * 2 + (J&1));
  case 6:
    divv += (unsigned short)((J>>2)&3) * 1000;
    return ((unsigned long)prescale) * (divv * 4 + (J&3));
  case 5:
    if( (J&7) > 4 )
      return 0;
    if( J&8 )
      divv += 1000;
    return ((unsigned long)prescale) * (divv * 5 + (J&7));
  case 4:
    if( J&8 )
      divv += 1000;
    return ((unsigned long)prescale) * (divv * 8 + (J&7));
  case 3:
    if( (J&7) > 9 )
      return 0;
    return ((unsigned long)prescale) * (divv * 10 + (J&15));
  default:
    return 0;
  }
}
#endif

static const unsigned char serial_order[16] = { 12, 11, 10, 9, 8, 7, 6, 5, 1, 2, 3, 4, 16, 15, 14, 13 };
void UpdateJ(unsigned short J) {
    // serial data = RA5
    // bit clock = RA4 (clocked on low->high)
    // latch = RC4 (latches on low->high)
    unsigned char i, which_bit;
    LATCbits.LATC4 = 0;
    for( i = 0; i < 16; ++i ) {
        LATAbits.LATA4 = 0;
        which_bit = serial_order[i];
        LATAbits.LATA5 = (J>>(which_bit-1))&1;
        asm("nop");
        asm("nop");
        LATAbits.LATA4 = 1;
        asm("nop");
        asm("nop");
    }
    LATCbits.LATC4 = 1;
}
void UpdateK(unsigned char K) {
    LATCbits.LATC0 = (unsigned char)(K&1);
    LATCbits.LATC1 = (unsigned char)((K>>1)&1);
    LATCbits.LATC2 = (unsigned char)((K>>2)&1);
}
void UpdateS(unsigned char S) {
    LATCbits.LATC4 = (unsigned char)(S&1);
    LATCbits.LATC3 = (unsigned char)((S>>1)&1);
}

void show_status(char* buf) {
  unsigned short pps;
  switch(S) {
  case 0:
    pps = 1;
    break;
  case 1:
    pps = 10;
    break;
  case 2:
    pps = 100;
    break;
  case 3:
    pps = 1000;
    break;
  }
  sprintf(buf, "\r\nJ=%u K=%u P=%u ratio=%lu\r\n", J, (unsigned short)K, pps, cur_ratio/*calc_ratio(J, K, S)*/);
}

void UpdateJKS() {
    UpdateJ(J);
    UpdateK(K);
    UpdateS(S);
}

#define EEPROM_ADDR 0x1FE0
const unsigned char init_ratio[32] __at(0x1FE0);

static const unsigned char* help[] = {
        "Help\r\n",
        "\r\n",
        "l                 load ratio from flash memory\r\n",
        "m                 measure frequency\r\n",
        "p J<x> K<y> P<z>  set J/K/P to x/y/z\r\n",
        "r<x>              set ratio to x\r\n",
        "s                 show status\r\n",
        "w                 write ratio to flash memory\r\n"
};

void parse(const char* buf) {
    char* endptr;
    unsigned long ratio;
    unsigned char* out_buf = usb_get_in_buffer(2);
    if( (buf[0] == 'r' || buf[0] == 'R') && buf[1] && (ratio = strtoul(buf+1, &endptr, 10)) && !*endptr ) {
        cur_ratio = calc_scaler(ratio, &J, &K, &S);
        UpdateJKS();
        show_status(out_buf);
        usb_send_in_buffer(2, strlen(out_buf));
    } else if( (buf[0] == 'm' || buf[0] == 'M') ) {
        unsigned char i;
        unsigned char slow = 0;
        unsigned long TMR1val;

        // set ratio to 80
        UpdateJ(16);
        UpdateK(4);
        UpdateS(1);
    Restart:
        ratio = 0;
        for( i = 0; i < 16; ++i ) {
            unsigned char TMR1HH = 0;

            OPTION_REG = 0xEF;
            INTCONbits.TMR0IF = 0;
            INTCONbits.TMR0IE = 1;
            TMR0 = 0;
            TMR1H = 0;
            TMR1L = 0;
            PIR1bits.TMR1IF = 0;
            T1CON = 0x44;
            TMR0 = 0xFF;
            TMR1val = 0;
            while( !T1CONbits.TMR1ON ) {
                if( ++TMR1val == 1000000UL ) {
                    strcpy(out_buf, "No signal.\r\n");
                    goto SendBuf;
                }
            }
            if( slow == 2 )
                TMR0 = 0xFF;
            while( T1CONbits.TMR1ON ) {
                if( PIR1bits.TMR1IF ) {
                    PIR1bits.TMR1IF = 0;
                    if( ++TMR1HH == 0 ) {
                        if( slow < 2 ) {
                            ++slow;
                            UpdateJ(3);
                            UpdateS(0);
                            goto Restart;
                        } else {
                            strcpy(out_buf, "Freq. too low.\r\n");
                            goto SendBuf;
//                            usb_send_in_buffer(2, strlen(out_buf));
//                            UpdateJKS();
//                            return;
                        }
                    }
                }
            }
            INTCONbits.TMR0IE = 0;

            TMR1val = (TMR1L | ((unsigned short)TMR1H<<8) | ((unsigned long)TMR1HH<<16));
            if( slow == 2 )
                ratio += (144000000UL / TMR1val);
            else if( slow == 1 )
                ratio += (144000000UL / TMR1val) * 256;
            else
                ratio += (3840000000UL / TMR1val) * 256;
            if( i == 0 && ratio < 400000UL ) {
                ++i;
                break;
            }
            if( i == 3 && ratio < 4000000UL ) {
                ++i;
                break;
            }
        }
        ratio /= i;
        sprintf(out_buf, "Freq=%lu\r\n", ratio);
SendBuf:
        usb_send_in_buffer(2, strlen(out_buf));
        UpdateJKS();
    } else if( (buf[0] == 'w' || buf[0] == 'W') ) {
        cur_ratio = calc_scaler(ratio, &J, &K, &S);
        HEFLASH_writeBlock(EEPROM_ADDR, (char*)&ratio, 4);
        strcpy(out_buf, "Saved to flash.\r\n");
        usb_send_in_buffer(2, strlen(out_buf));
    } else if( (buf[0] == 's' || buf[0] == 'S') && !buf[1] ) {
    ShowStatus:
        show_status(out_buf);
        usb_send_in_buffer(2, strlen(out_buf));
#ifdef XC8_PRO
    } else if( (buf[0] == 'l' || buf[0] == 'L') ) {
        HEFLASH_readBlock((char*)&ratio, EEPROM_ADDR, 4);
        if( ratio < 3 || ratio > 21327000UL )
            ratio = 1000;

        cur_ratio = calc_scaler(ratio, &J, &K, &S);
        UpdateJKS();
        goto ShowStatus;
     } else if( (buf[0] == 'h' || buf[0] == 'H' || buf[0] == '?') ) {
        unsigned char i;
        for( i = 0; i < sizeof(help) / sizeof(*help); ++i ) {
            while(usb_in_endpoint_busy(2))
                ;
            strcpy(out_buf, help[i]);
            usb_send_in_buffer(2, strlen(out_buf));
        }
    } else if( (buf[0] == 'p' || buf[0] == 'P') && buf[1] == ' ' && (buf[2] == 'j' || buf[2] == 'J') ) {
        ratio = strtoul(buf+3, &endptr, 10);
        if( /*ratio >= 0UL &&*/ ratio <= 65535UL ) {
            unsigned short newJ = ratio;
            if( endptr[0] == ' ' && (endptr[1] == 'k' || endptr[1] == 'K') ) {
                ratio = strtoul(endptr+2, &endptr, 10);
                if( /*ratio >= 0 &&*/ ratio <= 7 ) {
                    unsigned char newK = ratio;
                    if( endptr[0] == ' ' && (endptr[1] == 'p' || endptr[1] == 'P') ) {
                        ratio = strtoul(endptr+2, &endptr, 10);
                        J = newJ;
                        K = newK;
                        if( ratio == 1 )
                            S = 0;
                        else if( ratio == 10 )
                            S = 1;
                        else if( ratio == 100 )
                            S = 2;
                        else
                            S = 3;
                        UpdateJKS();
                        cur_ratio = calc_ratio(J, K, S);
                        goto ShowStatus;
//                        show_status(out_buf);
//                        usb_send_in_buffer(2, strlen(out_buf));
//                        return;
                    }
                }
            }
        }
#endif
    }
}

char parse_buf[80];
int main(void){
    unsigned long ratio;
    unsigned char parse_buf_pos = 0, i;

    hardware_init();
    ANSELA = 0;
    ANSELC = 0;
    TRISA = 0xCF;
    TRISC = 0xE0;

    HEFLASH_readBlock((char*)&ratio, EEPROM_ADDR, 4);
    if( ratio < 3 || ratio > 21327000UL )
        ratio = 1000;

    cur_ratio = calc_scaler(ratio, &J, &K, &S);
    UpdateJKS();
    GIE=1;          //interrupts on

#ifdef MULTI_CLASS_DEVICE
	cdc_set_interface_list(cdc_interfaces, sizeof(cdc_interfaces));
#endif
	usb_init();
    while (1) {
		// Send data to the PC 
#if 0
        if (usb_is_configured() &&
		    !usb_in_endpoint_halted(2) &&
		    !usb_in_endpoint_busy(2)) {
		}
#endif
		// Handle data received from the host 
		if (usb_is_configured() &&
		    !usb_out_endpoint_halted(2) &&
		    usb_out_endpoint_has_data(2) ){     //stall if break pending
			const unsigned char* out_buf;
            unsigned char* buf;
			size_t out_buf_len;			

			out_buf_len = usb_get_out_buffer(2, &out_buf);
            if( out_buf_len > sizeof(parse_buf) - parse_buf_pos )
                out_buf_len = sizeof(parse_buf) - parse_buf_pos;
            memcpy(parse_buf+parse_buf_pos, out_buf, out_buf_len);
            i = parse_buf_pos;
            parse_buf_pos += out_buf_len;
            usb_arm_out_endpoint(2);

			buf = usb_get_in_buffer(2);
            memcpy(buf, out_buf, out_buf_len);
            usb_send_in_buffer(2, out_buf_len);
            
            while( i < parse_buf_pos ) {
                if( parse_buf[i] == '\r' || parse_buf[i] == '\n' ) {
                    parse_buf[i] = 0;
                    if( i > 0 && (parse_buf[i-1] == '\r' || parse_buf[i-1] == '\n') )
                        parse_buf[i-1] = '\0';
                    parse(parse_buf);
                    parse_buf_pos -= i+1;
                    i = 0;
                } else {
                    ++i;
                }
            }
		}
    }
}

/* Callbacks. These function names are set in usb_config.h. */
void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return -1;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{
	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
	return 0;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
	/* This is where baud rate, data, stop, and parity bits are set. */
	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
    return 0;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
    return 0;
}


#ifdef _PIC14E
void interrupt isr()
{
    if( INTCONbits.TMR0IF ) {
        T1CON ^= 1;
        INTCONbits.TMR0IF = 0;
    }
	usb_service();
//    uartISR();
//    tmr2ISR();
}
#elif _PIC18

#ifdef __XC8
void interrupt high_priority isr()
{
	usb_service();
}
#elif _PICC18
#error need to make ISR
#endif

#endif
