#include "io.h"

volatile char uart_rx_buffer[UART_BUFFER_SIZE];
volatile char uart_tx_buffer[UART_BUFFER_SIZE];
volatile unsigned char uart_rx_buffer_head,uart_rx_buffer_tail;
volatile unsigned char uart_tx_buffer_head,uart_tx_buffer_tail;
volatile unsigned char txFlicker=0;
volatile unsigned char rxFlicker=0;

void ioInit(void){
    ANSELA=0;       //no analog    
    ANSELC=0;
    TX_LED=0;   //off
    RX_LED=0;   //off
    DTR_PIN=1;  //idles high
    TRISC=0b11110000;   //UART pins inputs til connection happens, RC2,RC3 driven to avoid floating input   
    TRISA=0b11001111;   //RA4 outputs to avoid floating input, RA5 is DTR
}

void uartSetBR(uint32_t r){
    unsigned int d;    
    if(r<(BAUD_DIVIDEND_H/LIMIT_16BIT)){  //need to use low speed mode to avoid overflow
        TXSTA = 0x20;       	//enable TX, low speed baud rate
        if(r<BAUD_DIVIDEND_L/LIMIT_16BIT){        //avoid overflow in low speed
            SPBRGL = 0xFF;  //minimum rate; actual is ~47 baud for 48MHz clock
            SPBRGH = 0xFF;            
        }else{
            d=(BAUD_DIVIDEND_L+(r/2))/r;
            if(d){d=d-1;}       //counter needs -1
            SPBRGL = ((d>>0)&0xFF); //LSB
            SPBRGH = ((d>>8)&0xFF); //MSB
        }
    }else{
        TXSTA = 0x24;       	//enable TX, High speed baud rate    
        d=(BAUD_DIVIDEND_H+(r/2))/r;
        if(d){d=d-1;}       //counter needs -1
        SPBRGL = ((d>>0)&0xFF); //LSB
        SPBRGH = ((d>>8)&0xFF); //MSB
    }
}

void uartInit(){            //baudrate set elsewhere
    uart_rx_buffer_head=0;
    uart_rx_buffer_tail=0;    
    uart_tx_buffer_head=0;
    uart_tx_buffer_tail=0;    
    TXSTA = 0x24;       	//enable TX, High speed baud rate
    RCSTA = 0x90;       	//enable RX, continuous receive
    BAUDCON = 0x08;     	//16 bit BRG
	if (RCSTAbits.OERR){
		RCSTAbits.SPEN = 0;  // reset the port
		RCREG;
		RCSTAbits.SPEN = 1;  // and keep going.
	}
    PIE1bits.RCIE=1;         //receive interrupt, only enable transmit interrupt if buffer full
	INTCONbits.PEIE = 1;     //peripheral interrupt enable	
}

char uartReceive(void){                  //returns -1 on no data
  int c;
  if(uart_rx_buffer_head==uart_rx_buffer_tail){
      return -1;    //empty
  }
  c=uart_rx_buffer[uart_rx_buffer_tail];
  uart_rx_buffer_tail=(uart_rx_buffer_tail+1)%UART_BUFFER_SIZE;
  return c;
}

void uartReceiveN(char n, char* b){ //receive n bytes into b; no check done, use uartAvailable prior
    char i,j;
    j=uart_rx_buffer_tail;
    for(i=0;i<n;i++){
        b[i]=uart_rx_buffer[j];
        j=(j+1)%UART_BUFFER_SIZE;        
    }
    uart_rx_buffer_tail=(uart_rx_buffer_tail+n)%UART_BUFFER_SIZE;
}

char uartSend(char b){               //use arduino form
    while(((uart_tx_buffer_head+1)%UART_BUFFER_SIZE)==uart_tx_buffer_tail){} //blocking till buffer free
    PIE1bits.TXIE=0;        //disable while adding
    uart_tx_buffer[uart_tx_buffer_head]=b;                          //add to queue
    uart_tx_buffer_head=(uart_tx_buffer_head+1)%UART_BUFFER_SIZE;
    PIE1bits.TXIE=1;        //(re)enable        
    return 1;
}

void uartSendN(char n, const char* b){    //add n bytes from b to tx buffer, use uartTXBufferSpace before!
    char i,j;
    j=uart_tx_buffer_head;
    PIE1bits.TXIE=0;        //disable while adding
    for(i=0;i<n;i++){
        uart_tx_buffer[j]=b[i];
        j=(j+1)%UART_BUFFER_SIZE;        
    }
    uart_tx_buffer_head=(uart_tx_buffer_head+n)%UART_BUFFER_SIZE;    
    PIE1bits.TXIE=1;        //(re)enable        
}

char uartAvailable(void){                //    
    return (uart_rx_buffer_head+UART_BUFFER_SIZE-uart_rx_buffer_tail)%UART_BUFFER_SIZE;
}

char uartTXavailable(void){              //  bytes queued in tx 
    //return (uart_tx_buffer_head+UART_BUFFER_SIZE-uart_tx_buffer_tail)%UART_BUFFER_SIZE;
    //return uart_tx_buffer_head-uart_tx_buffer_tail;         //if we only care about 0 vs !0
    return uart_tx_buffer_head ^ uart_tx_buffer_tail;         //if we only care about 0 vs !0
}

char uartTXBufferSpace(void){       //room in tx to add to buffer
    return (uart_rx_buffer_tail+UART_BUFFER_SIZE-uart_rx_buffer_head-1)%UART_BUFFER_SIZE;
}

void rxInject(char c){  //inject a byte of debugging message in data to host
    if(((uart_rx_buffer_head+1)%UART_BUFFER_SIZE)!=uart_rx_buffer_tail){        //if space available
        uart_rx_buffer[uart_rx_buffer_head]=c;                                  //add to buffer
        uart_rx_buffer_head=(uart_rx_buffer_head+1)%UART_BUFFER_SIZE;           
    }   
}

inline void uartISR(void){
    char c;
    if(PIR1bits.TXIF && PIE1bits.TXIE){     //TX ready
        if(uartTXavailable()){  //TXIF set means buffer reg free
            TXREG=uart_tx_buffer[uart_tx_buffer_tail];
            uart_tx_buffer_tail=(uart_tx_buffer_tail+1)%UART_BUFFER_SIZE;
        }
        if(uartTXavailable()==0){PIE1bits.TXIE=0;}  //disable if empty
    }
    if(PIR1bits.RCIF && PIE1bits.RCIE){              //RX interrupt flagged
        //PIR1bits.RCIF=0;      //RCIF is read-only
        rxFlicker=LED_FLICKER_TIME;
    	if (RCSTAbits.OERR){        //overrun error, need to fix to continue
        	RCSTAbits.CREN = 0;     //reset the port
    		//RCREG;
        	RCSTAbits.CREN = 1;     // and keep going.
        }
        c = RCREG;                  //do a read anyway
        if(((uart_rx_buffer_head+1)%UART_BUFFER_SIZE)!=uart_rx_buffer_tail){        //if space available
            uart_rx_buffer[uart_rx_buffer_head]=c;                                  //add to buffer
            uart_rx_buffer_head=(uart_rx_buffer_head+1)%UART_BUFFER_SIZE;           
        }//otherwise lost
    }    
}

void tmr2Init(void){    
    TMR2IE=0;
    TMR2IF=0;    
    T2CON=0;    //reset    
    PR2=0xFF;   //max count
    TMR2=0;     //reset counter
    T2CONbits.T2CKPS=3;     //PS=64 => 732Hz
    T2CONbits.T2OUTPS=15;   //45Hz
    T2CONbits.TMR2ON=1;    
    PEIE=1;
    TMR2IE=1;        
}

inline void tmr2ISR(void){
    if(TMR2IE && TMR2IF){
        TMR2IF=0;           //clear flag
        if(txFlicker){
            txFlicker--;
            TX_LED=1;
        }else{
            TX_LED=0;
        }
        if(rxFlicker){
            rxFlicker--;
            RX_LED=1;
        }else{
            RX_LED=0;
        }
    }    
}