// Connects to WiFi, requests NTP data and sends out as valid GPS sentences (RMC, GSA and GGA), with offset added to millis()
// Also shows Wifi status via custom ESP82 sentence
// Implements 64bit millis() for long term installation
// Settings menu with save to EEPROM with defaults (baudrate, WIFI SSID, WIFI PASS, NTP Server, Dummy COORDS for TZ detecting clock)
// uses IP API to get approx latitude/longitude
// LED status- on solid while connecting to Wi-Fi, flickers during NMEA data transmit
// working with analog clock and HiVis Clock
// V9 Add 1PPS feature
// V10 change srollover to 64 bit to prevent glitching
// V11 general tidyup (variable location/scope)

// V12: 2021 changes for V3.0.0 Board profile:
//add WiFiClient wificlient;              //line 23
//replace   http.begin(ipapi);            //line 81
//with      http.begin(wificlient,ipapi); //line 82

//V13skt: 2022 - modifications by S.Keller-Tuberg
// added support for backspace upon string input. Bit of a kludge, but it works.
// added additional WiFi information upon initialisation and also within setup/configuration menu
// migrated from single to multiple SSIDs. Works around ESP / arduino failing to connect when the AP has guest/additional SSIDs (eg Telstra smartmodem)
// migrated EEPROM storage from hard-coded addresses to a structure
// changed arduino LED behaviour to reduce the time it is turned on (saves battery consumption)
// added SSID and wifi channel number into ESP82 sentence
// If no SSID/passwords yet configured, will not attempt to connect to wifi/get lon/lat or get NTP time
//--------------------------------------------------------------------------------------------------------------
#include <EEPROM.h>
#include <ESP8266WiFiMulti.h>
#include <ESP8266HTTPClient.h>
#include <WiFiUdp.h>

ESP8266WiFiMulti wifiMulti ;
//--------------------------------------------------------------------------------------------------------------
// Define a structure for the EEPROM that is trivial to change without having to reedit all over the place
//#define DEBUG 1               // Comment this line out to exclude the debugging statements

#define STRLEN  (80)
#define BAUDRATE_DEFAULT (9600)

#define STRING_DEFAULT "-"
#define NTPHOST_DEFAULT "au.pool.ntp.org"
#define DUMMY_COORDS_DEFAULT "3351.000,S,15112.000,E"     //somewhere in Sydney

struct _EEPROM_struct
{
  unsigned short baudrate ;

  char  ssid1[STRLEN] ;
  char  pass1[STRLEN] ;

  char  ssid2[STRLEN] ;
  char  pass2[STRLEN] ;

  char  ssid3[STRLEN] ;
  char  pass3[STRLEN] ;

  char  ntp_host[STRLEN] ;
  char  dummy_coords[STRLEN] ;

  unsigned short checksum ;
} ;

typedef struct _EEPROM_struct EEPROM_STRUCT ;

EEPROM_STRUCT eeprom_struct =
{
  BAUDRATE_DEFAULT,

  STRING_DEFAULT,
  STRING_DEFAULT,
  STRING_DEFAULT,
  STRING_DEFAULT,
  STRING_DEFAULT,
  STRING_DEFAULT,
  DUMMY_COORDS_DEFAULT,

  0 // default checksum value is just a space holder
} ;
//--------------------------------------------------------------------------------------------------------------
//GPIO4 is D2 on D1 Mini
#define PPSPIN 4

char ipapi[] = "http://ip-api.com/line?fields=lat,lon" ;
char hex[] = "0123456789ABCDEF" ;               //for writing out checksum

HTTPClient http ;
WiFiClient wificlient ;
WiFiUDP udp ;
IPAddress ntp_hostIP ;

unsigned long long boott = (43099ULL * 86400ULL) << 32 ;        //1/1/2018
unsigned long ts = 0 ;
unsigned long dsn ;                    //day serial number
int date, mon, yr ;                    //calculated from daysn
bool ntplock = false ;                 //set after we get a valid NTP return
bool ntphit = false ;                  //an attempt was made to get NTP
bool led_on = false ;
unsigned long long srollover ;         //for detecting when a new second rolls over
long lastadjust = 0 ;                  //track last adjustment, signed to see if +/-
unsigned long lastNTP ;                //last time NTP was attempted
unsigned long long tnow ;              //64bit time to work with
unsigned long tshort ;                 //32 bit time in seconds
//--------------------------------------------------------------------------------------------------------------
void (* Software_Reset) (void) = 0 ;        // function template to cause a software reset
//--------------------------------------------------------------------------------------------------------------
// This routine is called when the arduino comes out of reset

void
setup (void)
{
  int httpCode ;                  //for HTTP call
  int httptries ;
  int n ;
  unsigned short checksum ;
  unsigned long long ttemp ;      //for storing millis() while working
  char str[STRLEN<<1] ;

  pinMode(LED_BUILTIN, OUTPUT) ;  //active low on some devices
  digitalWrite(LED_BUILTIN, LOW) ; // LED On
  led_on = true ;
  pinMode(PPSPIN, OUTPUT) ;
  pinMode(PPSPIN, LOW) ;

  delay (1000) ;
  Serial.begin(9600) ;
  Serial.printf ("\nWelcome to the WiFi NTP-client GPS emulator!\n") ;

  //load config from EEPROM
  EEPROM.begin(sizeof(EEPROM_STRUCT)) ;
  EEPROM.get(0, eeprom_struct) ;

  // Does the checksum look OK? Does it look sane?
  if ((checksum = calc_eeprom_checksum()) != eeprom_struct.checksum)
  {
    Serial.printf ("\nEEPROM checksum error. Calculated checksum %#04X but checksum in EEPROM is %#04X\n", checksum, eeprom_struct.checksum) ;
    goto rewrite_eeprom ;
  }

  // Does the baud rate look OK?
  if ((eeprom_struct.baudrate != 4800) && (eeprom_struct.baudrate != 9600))
  {
    Serial.printf ("\nEEPROM baudrate error. Was set to %hu\n", eeprom_struct.baudrate) ;

  rewrite_eeprom:
    Serial.printf ("Re-initialising EEPROM...\n") ;
    eeprom_struct.baudrate = BAUDRATE_DEFAULT ;               //load sane defaults

    strcpy (eeprom_struct.ssid1, STRING_DEFAULT) ;
    strcpy (eeprom_struct.pass1, STRING_DEFAULT) ;

    strcpy (eeprom_struct.ssid2, STRING_DEFAULT) ;
    strcpy (eeprom_struct.pass2, STRING_DEFAULT) ;

    strcpy (eeprom_struct.ssid3, STRING_DEFAULT) ;
    strcpy (eeprom_struct.pass3, STRING_DEFAULT) ;

    strcpy (eeprom_struct.ntp_host, NTPHOST_DEFAULT) ;
    strcpy (eeprom_struct.dummy_coords, DUMMY_COORDS_DEFAULT) ;

    // Recalculate the checksum based on the default data
    eeprom_struct.checksum = calc_eeprom_checksum() ;

    EEPROM.put(0, eeprom_struct) ; // Write out the eeprom structure
  }

  EEPROM.end() ;

#ifdef DEBUG
  Serial.printf("Configuration loaded from EEPROM:\n") ;
  Serial.printf("Baudrate set to %d\n", eeprom_struct.baudrate) ;

  Serial.printf("SSID 1: '%s'\n", eeprom_struct.ssid1) ;
  Serial.printf("Password 1: '%s'\n\n", eeprom_struct.pass1) ;

  Serial.printf("SSID 2: '%s'\n", eeprom_struct.ssid2) ;
  Serial.printf("Password 2: '%s'\n\n", eeprom_struct.pass2) ;

  Serial.printf("SSID 3: '%s'\n", eeprom_struct.ssid3) ;
  Serial.printf("Password 3: '%s'\n\n", eeprom_struct.pass3) ;

  Serial.printf("NTP host: '%s'\n", eeprom_struct.ntp_host) ;
  Serial.printf("Dummy coordinate string: '%s'\n\n", eeprom_struct.dummy_coords) ;
#endif

  Serial.printf("Type '~' to configure\n") ;
  delay(300) ;
  Serial.end() ;
  delay(100) ;

  // Set the new baud rate based on the config
  Serial.begin (eeprom_struct.baudrate) ;

  // Initialise the WiFi based on the configured SSIDs and passwords
  WiFi.mode (WIFI_STA) ;
  WiFi.disconnect () ;

  // Register the multiple Wifi Networks. WiFi daemon will try and connect to the strongest signal
  n = 0 ;
  n += set_ssid (eeprom_struct.ssid1, eeprom_struct.pass1) ;
  n += set_ssid (eeprom_struct.ssid2, eeprom_struct.pass2) ;
  n += set_ssid (eeprom_struct.ssid3, eeprom_struct.pass3) ;

  if (n == 0)
  {
    // No data is yet configured - we should jump straight to configuration
    for (;;)
    {
      check_for_serial_input() ; // when configuration has been changed, the arduino will soft-reset and restart
      toggle_led() ;
      yield() ;
    }
  }

  // Weit in this next loop until we have connected to one of the wifi access points
  Serial.printf("\n") ;

  while (wifiMulti.run() != WL_CONNECTED) // Note: this call can take some time.... it doesn't return immediately
  {
    toggle_led() ;
    ttemp = millis64() ;

    if ((ttemp % 1000) < srollover)     //do this each second to let connected device know we're awake
    {
      togglePPS() ;
      do_time_update() ;
      doRMC() ;
      doGGA() ;
      doGSA() ;
      doESPstat() ;
    }

    check_for_serial_input() ;
    srollover = ttemp % 1000 ;
    yield() ;                           //to stop watchdog resets
  }

  Serial.printf ("%s %s\n\n", wifi_status_string(str), str) ;

  //use IP API to find lat/lon
  for (httptries=10 ; httptries > 0 ; httptries--)
  {
    toggle_led() ;
    ttemp = millis64() ;
    http.begin(wificlient, ipapi) ;     //URL
    httpCode = http.GET() ;             //fetch

    if (httpCode == 200)                //valid data returned?
    {
      setlatlon(http.getString()) ;     //set lat/lon from data
      httptries = 0 ;                   //got our data, so continue
    }

    http.end() ;

    if ((ttemp % 1000) < srollover)     //do this each second to let connected device know we're awake
    {
      togglePPS() ;
      do_time_update() ;
      doRMC() ;
      doGGA() ;
      doGSA() ;
      doESPstat() ;
    }

    srollover = ttemp % 1000 ;
    yield() ;                           //to stop watchdog resets
  }

  // Force fetching of network time as we exit setup
  udp.begin(2390) ;
  getNTP() ;                            //try to get network time using NTP
  lastNTP = millis() ;

  digitalWrite(LED_BUILTIN, HIGH) ;     //LED off now Wi-Fi connected, have IP address and NTP time
  led_on = false ;
}
//--------------------------------------------------------------------------------------------------------------
// This is the main programme loop
void
loop (void)
{
  unsigned long long ttemp ;

  // Once every hour, try to get the network time again
  // Otherwise, try every minute if not yet locked
  if ((millis() - lastNTP > 3600000UL) || (!ntplock && (millis() - lastNTP > 60000UL)))
  {
    getNTP() ;
    lastNTP = millis() ;
  }

  ttemp = millis64() ;

  if ((ttemp % 1000) < srollover)               //output sentences each second, will glitch to (1)296ms every 71 days
  {
    toggle_led() ;
    togglePPS() ;
    toggle_led() ;
    do_time_update() ;
    doRMC() ;
    doGGA() ;
    doGSA() ;
    doESPstat() ;
  }

  check_for_serial_input() ;
  srollover = ttemp % 1000 ;
}
//--------------------------------------------------------------------------------------------------------------
// check to see whether the SSID and password have been configured, and if so, set them
int
set_ssid (char *ssid, char *password)
{
  if ((strcmp (ssid, STRING_DEFAULT) == 0) || (strcmp (password, STRING_DEFAULT) == 0))
    return 0 ;

  Serial.printf ("Searching for SSID='%s', password='%s'\n", ssid, password) ;
  wifiMulti.addAP(ssid, password) ;
  return 1 ;
}
//--------------------------------------------------------------------------------------------------------------
void
getNTP (void)
{
  byte NTPpacket[48] = "" ;
  unsigned long long ts3, ts4 ;
  unsigned long long newboott, ts1, ts2 ;           //to measure adjustment, outgoing/return timestamps
  int i ;

  ntphit = true ;               //flag that we're trying to do an update

  for (i = 0 ; i < 48 ; i++)    //clear packet contents
    NTPpacket[i] = 0 ;

  NTPpacket[0] = 0xE3 ;    //set default values for outgoing packet to server
  NTPpacket[2] = 0x06 ;
  NTPpacket[3] = 0xEC ;
  NTPpacket[12] = 0x31 ;
  NTPpacket[13] = 0x4E ;
  NTPpacket[14] = 0x31 ;
  NTPpacket[15] = 0x34 ;
  udp.beginPacket(eeprom_struct.ntp_host, 123) ;          //NTP requests are to port 123
  udp.write(NTPpacket, 48) ;            //buffer 48 bytes
  udp.endPacket() ;                     //send 48 bytes
  ts1 = millis64() ;                    //outgoing timestamp

  while ((millis64() - ts1 < 500UL) & (!udp.parsePacket()))
  {
  }                                     //wait up to a 0.5s for packet to return

  ts2 = millis64() ;                    //return timestamp
  udp.read(NTPpacket, 48) ;             //save the returned packet
  ts3 = (((unsigned long long)NTPpacket[32]) << 56) | (((unsigned long long)NTPpacket[33]) << 48) | (((unsigned long long)NTPpacket[34]) << 40) | (((unsigned long long)NTPpacket[35]) << 32) | (((unsigned long long)NTPpacket[36]) << 24) | (((unsigned long long)NTPpacket[37]) << 16) | (((unsigned long long)NTPpacket[38]) << 8) | (((unsigned long long)NTPpacket[39])) ;
  ts4 = (((unsigned long long)NTPpacket[40]) << 56) | (((unsigned long long)NTPpacket[41]) << 48) | (((unsigned long long)NTPpacket[42]) << 40) | (((unsigned long long)NTPpacket[43]) << 32) | (((unsigned long long)NTPpacket[44]) << 24) | (((unsigned long long)NTPpacket[45]) << 16) | (((unsigned long long)NTPpacket[46]) << 8) | (((unsigned long long)NTPpacket[47])) ;

  if ((ts3 != 0) && (ts4 != 0))         //if valid packet returned
  {
    newboott = ts3 - NTPfromms(ts1 + (ts2 - ts1) / 2) ;                 //calculate time offset
    lastadjust = msfromNTP(((long long)newboott - (long long)boott)) ;  //cast to signed so we can check for +/- adjustment
    boott = newboott ;                                                  //update time offset
    ntplock = true ;                                                    //fix is good
  }
}
//--------------------------------------------------------------------------------------------------------------
void
dodsn (unsigned long dsn)
{
  byte dim[] = { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 } ;   //days in month

  yr = 1900 ;
  mon = 1 ;
  date = dsn + 1 ;                      //1/1/1900 has timestamp 0, ie date of 0 is 1

  while (date > 365)                    //count years
  {
    if (((yr % 4) == 0) && (yr != 1900))
    {
      date-- ;
    }

    yr++ ;
    date = date - 365 ;
  }

  if (((yr % 4) == 0) && (yr != 1900))
  {
    dim[2]++ ;
  }                                     //if a leap year, Feb has 29

  while (date > dim[mon])               //count months
  {
    date = date - dim[mon] ;
    mon++ ;
  }

  if ((date == 0) && (mon == 1))        //glitch on 31/12 of leap years
  {
    yr-- ;
    date = 31 ;
    mon = 12 ;
  }
}
//--------------------------------------------------------------------------------------------------------------
unsigned long long
NTPfromms (unsigned long long t)        //converts a time in ms to time in 64 bit NTP, max precision, will rollover after 194 years
{
  return ((t << 22) / 1000) << 10 ;
}
//--------------------------------------------------------------------------------------------------------------
unsigned long
msfromNTP (unsigned long long t)        //converts a 64 bit NTP time into ms
{
  return (t * 1000) >> 32 ;             //resolution will be lost however we do this
}
//--------------------------------------------------------------------------------------------------------------
int
seconds (unsigned long t)
{
  return t % 60 ;
}
//--------------------------------------------------------------------------------------------------------------
int
minutes (unsigned long t)
{
  return (t / 60) % 60 ;
}
//--------------------------------------------------------------------------------------------------------------
int
hours (unsigned long t)
{
  return (t / 3600) % 24 ;
}
//--------------------------------------------------------------------------------------------------------------
void
add_GPS_checksum_and_output (char *sentence, int n)
{
  int i ;
  byte checksum = 0x00 ;

  for (i=1 ; i < (n-1) ; i++)
    checksum = checksum ^ sentence[i] ;

  sentence[n++] = hex[(checksum >> 4) & 0xF] ;
  sentence[n++] = hex[checksum & 0xF] ;
  Serial.println (sentence) ;
}
//--------------------------------------------------------------------------------------------------------------
void
doESPstat (void)                        //prints a sentence that looks like GPS data, but gives us WIFI info
{
  char sentence[STRLEN<<1] = "" ;       //WiFi.status(),WiFi.localIP(),ntp data received, have we just tried to hit NTP?, last adjustment value in ms
  char str[STRLEN+2] ;
  int  i, n ;

  n = sprintf(sentence, "$ESP82,%s,%s,%d.%d.%d.%d,%01d,%01d,%01d*",
              wifi_status_string (str),
              str,
              WiFi.localIP()[0], WiFi.localIP()[1], WiFi.localIP()[2], WiFi.localIP()[3],
              ntplock,
              ntphit,
              lastadjust) ;

  add_GPS_checksum_and_output (sentence, n) ;
  ntphit = false ;                      //clear this flag
}
//--------------------------------------------------------------------------------------------------------------
void
do_time_update (void)                           //update sentence values
{
  tnow = boott + NTPfromms(millis64()) ;        //get current time
  tshort = (tnow) >> 32 ;                       //get integer seconds part
  dodsn(tshort / 86400) ;                       //calculate date
}
//--------------------------------------------------------------------------------------------------------------
void
doRMC (void)                                    //output an RMC sentence
{
  char sentence[STRLEN+2] = "" ;                //standard RMC: time, fix, coordinates, date, checksum
  int  n ;

  n = sprintf(sentence, "$GPRMC,%02d%02d%02d.%03d,%c,%s,0.00,000.00,%02d%02d%02d,,,*",
            hours(tshort), minutes(tshort), seconds(tshort),
            ((unsigned long)(tnow)) / 4294968UL,
            ntplock ? 'A':'V',
            eeprom_struct.dummy_coords,
            date, mon, yr % 100) ;

  add_GPS_checksum_and_output (sentence, n) ;
}
//--------------------------------------------------------------------------------------------------------------
void
doGSA (void)                                    //ouput a GSA sentence
{
  char sentence[STRLEN+2] = "" ;
  int  n ;

  n = sprintf(sentence, "$GPGSA,A,%c,,,,,,,,,,,,,1.00,1.00,1.00,*", ntplock ? '3':'1') ;
  add_GPS_checksum_and_output (sentence, n) ;
}
//--------------------------------------------------------------------------------------------------------------
void
doGGA (void)                                    //output a GGA sentence
{
  char sentence[STRLEN+2] = "" ;
  int  n ;

  n = sprintf(sentence, "$GPGGA,%02d%02d%02d.%03d,%s,%c,04,1.0,0.0,M,0.0,M,,*",
          hours(tshort), minutes(tshort), seconds(tshort),
          ((unsigned long)(tnow)) / 4294968UL,
          eeprom_struct.dummy_coords,
          ntplock ? '8':'0') ;

  add_GPS_checksum_and_output (sentence, n) ;
}
//--------------------------------------------------------------------------------------------------------------
unsigned long long
millis64 (void)
{
  static unsigned long lo, hi ;         //hi and lo 32 bits
  unsigned long newlo = millis() ;      //check millis()

  if (newlo < lo)
    hi++ ;

  lo = newlo ;                                                                  //save for next check
  return ((unsigned long long)(hi) << 32) | ((unsigned long long)(lo)) ;        //return 64 bit result
}
//--------------------------------------------------------------------------------------------------------------
// Read serial input and wait for a tilde character.
// If received, enter setup

void
purge_serial_input (void)
{
  // Read all characters (if any) so that none remain in the queue
  while (Serial.available())
    Serial.read() ;
}
//--------------------------------------------------------------------------------------------------------------
void
read_string (char *prompt, char *s)
{
  int i, d, len ;

  s[0] = '\0' ;

  Serial.printf("Enter %s: ", prompt) ;

  for (len=0 ;;)
  {
    if (Serial.available())
    {
      d = Serial.read() ;
      len = strlen(s) ;

      if ((d >= ' ') && (d < 127) && (len < STRLEN)) // Is it a printable character?
      {
        Serial.write(d) ;
        s[len++] = d ; // copy the character
        s[len] = '\0' ; // terminate string one character beyond
      }

      // Is it a backspace? The Arduino serial monitor doesn't handle backspace sensibly - so start a new line
      else if (((d == '\b') || (d == 127)) && (len > 0))       // backspace can be either char 8 or char 127
      {
        s[--len] = '\0' ;  // erase the last character in the string

        // Because the serial monitor doesn't backspace, just indicate the backspace and start a new line
        Serial.printf (" <BS>\n%s", s) ;
      }

      else if ((d == '\r') || (len >= STRLEN))  // Enter is carraige return
      {
        Serial.println() ;
        delay(100) ;
        purge_serial_input() ;
        break ;
      }
    }
  }
}
//--------------------------------------------------------------------------------------------------------------
char *
wifi_status_string (char *str)
{
  str[0] = '\0' ;

  switch (WiFi.status())
  {
    case WL_IDLE_STATUS:
      return "idle" ;

    case WL_NO_SSID_AVAIL:
      return "SSIDs unreachable" ;

    case WL_CONNECTED:

      sprintf (str, "SSID %s chan %d", WiFi.SSID().c_str(), WiFi.channel()) ;
      return "connected" ;

    case WL_CONNECT_FAILED:
      sprintf (str, "SSID %s", WiFi.SSID().c_str()) ;
      return "connection failed" ;

    case WL_WRONG_PASSWORD:
      sprintf (str, "SSID %s", WiFi.SSID().c_str()) ;
      return "wrong password" ;

    case WL_DISCONNECTED:
      return "disconnected" ;

    default:
      sprintf (str, "WiFi.status()==%d", WiFi.status()) ;
      return "?" ;
  }
}
//--------------------------------------------------------------------------------------------------------------
void
check_for_serial_input (void)                   //see if user wants to change settings
{
  int n, i ;
  bool changed, prompt ;
  char str[STRLEN<<1] ;

  // Outer loop - check for tilde character being pressed
  if (Serial.available() && (Serial.read() == '~'))
  {
    purge_serial_input() ;
    changed = false ;
    prompt = true ;

    // Loop until exit selected
    for (;;)
    {
      if (prompt == true)
      {
        Serial.printf ("\n\n\nWiFi NTP-client GPS emulator setup\n") ;

        Serial.printf ("Current IP Address: %d.%d.%d.%d\n", WiFi.localIP()[0], WiFi.localIP()[1], WiFi.localIP()[2], WiFi.localIP()[3]) ;
        Serial.printf ("Current Baudrate: %hu\n", eeprom_struct.baudrate) ;
        Serial.printf ("Wifi status: %s %s\n", wifi_status_string(str), str) ;

        Serial.printf ("\nScanning available WiFi networks...\n") ;
        WiFi.scanNetworks (true) ; // Kick off the scan

        // Wait for the scan to complete (the yield is to stop the watchdog timer expiring)
        for (; (n = WiFi.scanComplete()) <= 0 ;)
          yield() ;

        Serial.printf ("\n%d network(s) found\n", n) ;

        for (i=0 ; i < n ; i++)
          Serial.printf ("(%c) %-20s Channel:%2d (%3ddBm) %s\n",
                         'a'+i,
                         WiFi.SSID(i).c_str(),
                         WiFi.channel(i),
                         WiFi.RSSI(i),
                         (WiFi.encryptionType(i) == ENC_TYPE_NONE) ? "open" : "encrypted") ;

        WiFi.scanDelete() ;

        Serial.printf("\n1: Choose 4800 Baudrate\n") ;
        Serial.printf("2: Choose 9600 Baudrate\n") ;
        Serial.printf("3: Set SSID #1. [Current: '%s']\n", eeprom_struct.ssid1) ;
        Serial.printf("4: Set Password #1. [Current: '%s']\n", eeprom_struct.pass1) ;
        Serial.printf("5: Set SSID #2. [Current: '%s']\n", eeprom_struct.ssid2) ;
        Serial.printf("6: Set Password #2. [Current: '%s']\n", eeprom_struct.pass2) ;
        Serial.printf("7: Set SSID #3. [Current: '%s']\n", eeprom_struct.ssid3) ;
        Serial.printf("8: Set Password #3. [Current: '%s']\n", eeprom_struct.pass3) ;
        Serial.printf("N: Define NTP Server. [Current: '%s']\n", eeprom_struct.ntp_host) ;
        Serial.printf("C: Define dummy coord string (for GPRMC and GPGGA) [Current: '%s']\n", eeprom_struct.dummy_coords) ;
        Serial.printf("X: Exit and save\n\n") ;
        Serial.printf("Option: > ") ;

        prompt = false ;
      }

      // loop until input is received
      if ((Serial.available()) && ((i = Serial.read()) > ' '))
      {
        Serial.println((char)i) ;
        purge_serial_input() ;
        prompt = true ;

        switch (i)
        {
          case '1':
            if (eeprom_struct.baudrate != 4800)
            {
              changed = true ;
              eeprom_struct.baudrate = 4800 ;
            }

            break ;

          case '2':
            if (eeprom_struct.baudrate != 9600)
            {
              changed = true ;
              eeprom_struct.baudrate = 9600 ;
            }

            break ;

          case '3':
            read_string ("SSID 1", str) ;

            if (strcmp(str, eeprom_struct.ssid1) != 0)
            {
              strcpy (eeprom_struct.ssid1, str) ;
              changed = true ;
            }

            break ;

          case '4':
            read_string ("password 1", str) ;

            if (strcmp(str, eeprom_struct.pass1) != 0)
            {
              strcpy (eeprom_struct.pass1, str) ;
              changed = true ;
            }

            break ;

          case '5':
            read_string ("SSID 2", str) ;

            if (strcmp(str, eeprom_struct.ssid2) != 0)
            {
              strcpy (eeprom_struct.ssid2, str) ;
              changed = true ;
            }

            break ;

          case '6':
            read_string ("password 2", str) ;

            if (strcmp(str, eeprom_struct.pass2) != 0)
            {
              strcpy (eeprom_struct.pass2, str) ;
              changed = true ;
            }

            break ;

          case '7':
            read_string ("SSID 3", str) ;

            if (strcmp(str, eeprom_struct.ssid3) != 0)
            {
              strcpy (eeprom_struct.ssid3, str) ;
              changed = true ;
            }

            break ;

          case '8':
            read_string ("password 3", str) ;

            if (strcmp(str, eeprom_struct.pass3) != 0)
            {
              strcpy (eeprom_struct.pass3, str) ;
              changed = true ;
            }

            break ;

          case 'N':
          case 'n':
            read_string ("NTP server name", str) ;

            if (strcmp(str, eeprom_struct.ntp_host) != 0)
            {
              strcpy (eeprom_struct.ntp_host, str) ;
              changed = true ;
            }

            break ;

          case 'C':
          case 'c':
            read_string ("dummy Coordinates (in the form DDMM.SS,N/S,DDDMMSS,E/W)", str) ;

            if (strcmp(str, eeprom_struct.dummy_coords) != 0)
            {
              strcpy (eeprom_struct.dummy_coords, str) ;
              changed = true ;
            }

            break ;

          case 'X':
          case 'x':
            if (changed)
            {
              EEPROM.begin(sizeof(EEPROM_STRUCT)) ;
              eeprom_struct.checksum = calc_eeprom_checksum() ;
              EEPROM.put(0, eeprom_struct) ; // Write the config structure to EEPROM
              EEPROM.end() ;
              Serial.printf("\nExit config, changes saved\n\n") ;
              Software_Reset() ; // Cause a software reset to load the new configuration
            }

            Serial.printf("\nExit config, nothing to save!\n\n") ;
            return ; // we can return from this routine - back to normal operation
        }
      }
    }
  }
}
//--------------------------------------------------------------------------------------------------------------
// The eeprom checksum is calculated over all bytes EXCEPT the checksum bytes
unsigned long
calc_eeprom_checksum (void)
{
  int i, len ;
  unsigned short checksum ;
  unsigned char *p ;

  checksum = 0 ;
  p = (unsigned char *)&eeprom_struct ;
  len = sizeof(eeprom_struct) - sizeof(eeprom_struct.checksum) ;

  for (i=0 ; i < len ; i++)
    checksum += *p++ ;

  return checksum ;
}
//--------------------------------------------------------------------------------------------------------------
void
setlatlon (String a)  //extract latitude/longitude in RMC format from string in lat(CR)lon format. Works to nearest minute as this is <2km
{
  int lonindex ;
  float lat, lon ;
  int lati, latf, loni, lonf ;           //latitude/longitude integer/fractional parts
  char ns = 'N' ;
  char ew = 'E' ;                        //for positive values

  lat = a.toFloat() ;                    //latitude is first line
  lonindex = a.indexOf('\n') + 1 ;       //look for /n
  lon = a.substring(lonindex).toFloat() ;//longitude is on second line

  if ((lat == 0) || (lon == 0))
    return ;

  if (lat < 0)                          //check if negative and change compass direction
  {
    ns = 'S' ;
    lat = -lat ;
  }

  if (lon < 0)                          //check if negative and change compass direction
  {
    ew = 'W' ;
    lon = -lon ;
  }

  lati = lat ;
  latf = (lat - lati) * 60 ;            //counts up to 59 minutes
  loni = lon ;
  lonf = (lon - loni) * 60 ;            //counts up to 59 minutes
  sprintf (eeprom_struct.dummy_coords, "%02d%02d.000,%c,%03d%02d.000,%c", lati, latf, ns, loni, lonf, ew) ;
}
//--------------------------------------------------------------------------------------------------------------
void
togglePPS (void)
{
  pinMode(PPSPIN, HIGH) ;
  delay(10) ;
  pinMode(PPSPIN, LOW) ;
}
//--------------------------------------------------------------------------------------------------------------
void
toggle_led (void)
{
  if (led_on == true)
  {
    digitalWrite(LED_BUILTIN, HIGH) ;    //LED off
    led_on = false ;
  }
  else
  {
    digitalWrite(LED_BUILTIN, LOW) ;     //LED on
    led_on = true ;
  }
}
