;**************************************************************************
;**************************************************************************
;*
;*               Studio Series Preamp Control Module Firmware
;* 
;*                          Version 1.0 27/03/06
;*                          
;*                          Created by P.B.Smith
;*                 (C)2006 Silicon Chip Publications P/L
;*                          All Rights Reserved.
;*
;*                         www.siliconchip.com.au
;*
;*               +-------------------------------------------+
;*               | This source code is released as FREEWARE  |
;*               | and as such is NOT in the public domain.  |
;*               +-------------------------------------------+
;*
;**************************************************************************
;**************************************************************************

;************************************^*************************************
;*
;* Change record
;*
;* Version 1.1  22/01/07
;* - Fixed bug in select_source routine that prevented channel selection
;*   via switches until first infrared channel change.
;*
;**************************************************************************

.nolist
.include "2313ndef.inc"		;AVR definitions (non-standard!)
.list

;**************************************************************************
;*
;* REGISTERS
;*
;**************************************************************************

.def	ir_state	=R2	;ir receive state
.def	ir_bit_cnt	=R3	;ir bit count
.def	ir_sys		=R4	;ir received system bits
.def	ir_cmd		=R5	;ir received command bits
.def	ir_code		=R6	;last received command bits
.def	irate_tmr	=R7	;interrupt rate timer
.def	tick_tmr	=R8	;general purpose 1ms timer
.def	led_tmr		=R9	;LED on timer
.def	sw_tmr		=R10	;switch valid timer
.def	sw_bit		=R11	;last detected switch bit
.def	sw_tmp		=R12	;switch bit temp save
.def	mot_ctl		=R13	;motor control bit image
;.def			=R14
.def	status		=R15	;SREG save (used by interrupts)

.def	A		=R16	;scratch
.def	B		=R17	;scratch
.def	C		=R18	;scratch
.def	aflags		=R19	;various flags (see bit vars below)
.def	bflags		=R20	;ditto
.def	ir_timer	=R21	;ir bit timer
.def	dir_cnt		=R22	;motor direction change delay timer
.def	mot_cnt		=R23	;motor startup delay timer
.def	rcnt_lo		=R24	;motor run timer
.def	rcnt_hi		=R25
.def	XL		=R26
.def	XH		=R27
.def	YL		=R28
.def	YH		=R29
.def	ZL		=R30
.def	ZH		=R31

; Register-bound bit variables

; aflags

.equ	m_alps		=7	;1=alps pot installed
.equ	m_mute		=6	;1=mute function active
.equ	m_run		=5	;1=motor is running
.equ	m_cmd		=4	;1=request motor on
.equ	m_dir		=3	;1=motor right, 0=motor left
.equ	m_dly		=2	;1=motor start delay in progress
.equ	m_start		=1	;1=motor startup in progress
.equ	m_reset		=0	;1=motor reset required (at stops)

; bflags

.equ	c_tog		=6	;1=comparator toggle stopped motor
.equ	c_poll		=5	;1=enable comparator polling
.equ	wd_res		=4	;1=watchdog reset pending
.equ	sw_wait		=3	;1=switch valid timer running
.equ	sw_rdy		=2	;1=new switch pressed
.equ	ir_rpt		=1
.equ	ir_rdy		=0	;1=infrared data available


;**************************************************************************
;*
;* Hardware specific
;*
;**************************************************************************

; Port B bits

; Relay control outputs / push-button switch inputs

.equ	CD_CTL		=7	;CD
.equ	DVD_CTL		=6	;DVD
.equ	TUNER_CTL	=5	;Tuner
.equ	AUX_CTL		=4	;Aux
.equ	TAPE_CTL	=3	;Tape

; Config jumper & comparator inputs

.equ	JP1_SEL		=2	;setup select
.equ	M_REF		=1	;ref voltage (comparator -ve input)
.equ	M_SENSE		=0	;motor current sense (comparator +ve input)

; Port D bits

.equ	JP2_SEL		=6	;pot/motor type select (in=Alps)
.equ	M_HI_A		=5	;high side motor control 'A' (left)
.equ	M_LO_B		=4	;low side motor control 'B' (right)
.equ	M_LO_A		=3	;low side motor control 'A' (left)
.equ	M_HI_B		=2	;high side motor control 'B' (right)
.equ	ACK_LED		=1	;'ACK' LED output
.equ	IR_RXD		=0	;infrared receiver input


;**************************************************************************
;*
;* Constants
;*
;**************************************************************************

.equ	KEY_RPT		=114	;time between code repeats (ms)

;Supported infrared equipment addresses

.equ	TV_ADDR		=0
.equ	SAT1_ADDR	=8
.equ	SAT2_ADDR	=10
.equ	CD_ADDR		=20

; Used infrared remote keys

.equ	CH_1		=1
.equ	CH_2		=2
.equ	CH_3		=3
.equ	CH_4		=4		
.equ	CH_5		=5
.equ	VOL_UP		=16
.equ	VOL_DN		=17
.equ	CH_UP		=32
.equ	CH_DN		=33
.equ	MUTE_TV1	=13	;'punch-through' mute (sys = 0)
.equ	MUTE_ALT1	=18	;alt mute: 12 (Aifa), TV1 code 191
.equ	MUTE_ALT2	=43	;alt mute: 12 (Aifa), CD code 651

; Preamp control bit mask

.equ	C_MASK		=1<<CD_CTL|1<<DVD_CTL|1<<TUNER_CTL|1<<AUX_CTL|1<<TAPE_CTL

; Motor control bit masks

.equ	M_LEFT		=1<<M_LO_B|1<<M_HI_A
.equ	M_RIGHT		=1<<M_LO_A|1<<M_HI_B
.equ	M_STOP		=1<<M_HI_A|1<<M_HI_B
.equ	M_MASK		=1<<M_LO_A|1<<M_LO_B|1<<M_HI_A|1<<M_HI_B

; Motor constants

.equ	M_TIME		=12000	;max rotation time (ms) (actual = 8-9s)

.equ	M_ALPS_1D	=29	;time (ms) to pulse motor for 1 deg of travel (Alps)
.equ	M_ALT_1D	=20	; (Altronics)

	.cseg
	.org	$0

;**************************************************************************
;*
;* Interrupt vectors
;*
;**************************************************************************
        
	rjmp	reset		;reset
	rjmp	bad_irq		;IRQ0
	rjmp	bad_irq		;IRQ1
	rjmp	bad_irq		;timer 1 capture
	rjmp    bad_irq		;timer 1 compare
	rjmp	bad_irq 	;timer 1 overflow
	rjmp	tmr0_irq	;timer 0 overflow
	rjmp	bad_irq		;UART rx complete
	rjmp	bad_irq		;UART data register empty
	rjmp	bad_irq		;UART tx complete
	rjmp	bad_irq		;analog comparator

bad_irq:
	reti
	

;**************************************************************************
;*
;* Reset entry point
;*
;**************************************************************************
	
reset:	cli				;disable all interrupts
	ldi	A,low(RAMEND)		;set stack top
	out	SPL,A

	clr	aflags			;init variables
	clr	bflags
	clr	ir_state		;reset ir receive state
	
	ldi	A,0b11111100		;control & JP1 bits pulled up
	out	PORTB,A
	ldi	A,0b00000000		;all bits in initially
	out	DDRB,A

	ldi	A,0b01000011		;inputs pulled up, LED off
	out	PORTD,A
	ldi	A,0b00111110		;JP2 & ir bits in, motor & LED out
	out	DDRD,A

	ldi	A,1			;CT0 clocked at CK
	out	TCCR0,A
	ldi	A,1<<TOIE0		;enable CT0 overflow interrupt
	out	TIMSK,A
	sei				;enable interrupts


;**************************************************************************
;*
;* Initialisation 
;*
;**************************************************************************

init:	ldi	A,ee_magic
	rcall	ee_read
	cpi	B,$5A			;first power up?
	breq	in1			;skip next bit if not		
	rcall	ee_init			;else set eeprom defaults

; Check jumper settings...

in1:	sbic	PIND,JP2_SEL		;skip if Alps pot installed
	rjmp	in2			;else assume Altronics
	sbr	aflags,1<<m_alps

in2:	sbic	PINB,JP1_SEL		;skip if 'setup' jumper in
	rjmp	sel_last		;else continue init


;**************************************************************************
;*
;* User setup (JP1 installed)
;*
;**************************************************************************

; Indicate setup mode by flashing each select LED in turn

user_setup:
	ldi	C,1
us0:	mov	B,C
	rcall	select_source
	ldi	A,250			;250ms pause
	rcall	delay
	inc	C
	cpi	C,6
	brne	us0

	ldi	B,0			;now switch all LEDs off
	rcall	select_source		
	rcall	ee_init			;set eeprom-based defaults

; Wait for two consecutive identical infrared codes...

us1:	rcall	wait_key
	mov	C,ir_sys
	rcall	wait_key
	cp	C,ir_sys		;same sys address?
	brne	us1			;redo if not

	ldi	A,ee_ir_sys
	mov	B,ir_sys
	rcall	ee_write		;save as our address...

	ldi	B,5
	rcall	flash_it		;flash 'ack' LED 5 times

; Look for 686_ setup codes...

us2:	rcall	wait_key
	cpi	B,6
	brne	us2

	rcall	wait_key
	cpi	B,8
	brne	us2

	rcall	wait_key
	cpi	B,6
	brne	us2

	rcall	wait_key
	cpi	B,1			;[1] disable punch-through mute
	ldi	A,ee_punch
	breq	us4

	cpi	B,2			;[2] disable muting function
	brne	us2

us3:	ldi	A,ee_mute
us4:	ldi	B,0
	rcall	ee_write		;zero to disable the function
	
	ldi	B,5
	rcall	flash_it		;flash LED 5 times
	rjmp	us2			;loop until JP1 removed	

; Select last used source 

sel_last:
	ldi	A,ee_source
	rcall	ee_read			;get last source
	rcall	sel_src_bit		;and select it

	wdr				;reset watchdog
	ldi	A,0b00001000
	out	WDTCR,A			;timeout = 15ms, enable it


;**************************************************************************
;*
;* Foreground tasks
;*
;**************************************************************************

main:	sbr	bflags,1<<wd_res	;watchdog keep-alive (see tmr0_irq)
	sbrs	bflags,ir_rdy		;do ir stuff if code received
	rjmp	mn22			;else go look for switch press

	cbr	bflags,1<<ir_rdy
		
; Check if received infrared code is for us...

	ldi	A,ee_ir_sys
	rcall	ee_read			;get programmed address
	mov	A,ir_code
	andi	A,$7F			;remove the toggle bit
	cp	B,ir_sys		;same as received?
	breq	mn2			;go if it is

	tst	ir_sys			;address 0 (TV1)?
	brne	mn1			;ignore code if not

	ldi	A,ee_punch
	rcall	ee_read			;get punch-through mute status
	tst	B			;disabled?
	breq	mn1			;ignore if so

	mov	A,ir_code		;get received code again
	andi	A,$7F
	cpi	A,MUTE_TV1		;is it a punch-through mute?
	breq	mn4			;to mute handling
mn1:	rjmp	mn22			;else not for us
	
mn2:	cpi	A,MUTE_TV1		;mute/unmute?
	breq	mn4

	cpi	B,TV_ADDR
	brne	mn3
	cpi	A,MUTE_ALT1		;alternate 'mute' keys?
	breq	mn4
	rjmp	mn10

mn3:	cpi	B,CD_ADDR
	brne	mn10
	cpi	A,MUTE_ALT2
	brne	mn10

; Mute function

mn4:	ldi	A,ee_mute
	rcall	ee_read			;get muting status
	tst	B			;disabled?
	breq	mn1			;ignore if so

	sbrc	bflags,ir_rpt
	rjmp	mn21			;ignore if it's a repeat

	sbrs	aflags,m_mute		;skip if currently muted...
	rjmp	mn8

; Already muted, calculate return time to original position
		
	cli
	subi	rcnt_lo,low(M_TIME)	;calc mute return time
	sbci	rcnt_hi,high(M_TIME)

	sbrs	aflags,m_alps
	rjmp	mn5

; Alps pot 'allowances'

	subi	rcnt_lo,low(-30)	;minus stop/gearbox backlash time
	sbci	rcnt_hi,high(-30)

	sbrs	bflags,c_tog		;skip if comparator stopped motor	
	rjmp	mn6

	subi	rcnt_lo,low(-210)	;clutch loading time vs current ramp-up
	sbci	rcnt_hi,high(-210)
	rjmp	mn6

; Altronics pot 'allowances'

mn5:	subi	rcnt_lo,low(-60)	;minus stop/gearbox backlash time
	sbci	rcnt_hi,high(-60)

	sbrs	bflags,c_tog		;skip if comparator stopped motor	
	rjmp	mn6

	subi	rcnt_lo,low(-165)	;clutch loading time vs current ramp-up
	sbci	rcnt_hi,high(-165)

mn6:	com	rcnt_lo
	com	rcnt_hi

	cpi	rcnt_lo,low(M_TIME+1)	;check result against max rotation time
	ldi	A,high(M_TIME+1)
	cpc	rcnt_hi,A
	brcc	mn7			;don't move if 'overflow'

	sbr	aflags,1<<m_dir|1<<m_cmd ;rotate right, motor on
mn7:	sei
	rjmp	mn20

; Load max rotation time

mn8:	cli
	ldi	rcnt_lo,low(M_TIME)
	ldi	rcnt_hi,high(M_TIME)

	cbr	aflags,1<<m_dir		;rotate left
	sbr	aflags,1<<m_mute|1<<m_cmd ;mute active, motor on

mn9:	sei
	rjmp	mn21			;done

; Volume up/down function

mn10:	cpi	A,VOL_UP
	breq	mn11

	cpi	A,VOL_DN
	brne	mn14			;skip if not vol up/down

mn11:	cli
	ldi	rcnt_lo,low(KEY_RPT+20)	;RC5 key repeat time + margin (smooth!)
	ldi	rcnt_hi,high(KEY_RPT+20)
	sbr	aflags,1<<m_dir|1<<m_cmd ;rotate right, motor on

mn12:	cpi	A,VOL_UP
	breq	mn13			;skip if vol up request 
	cbr	aflags,1<<m_dir		;else rotate left for vol down

mn13:	sei
	rjmp	mn20			;done

; Channel up/down function (fine volume adjust)

mn14:	cpi	A,CH_UP
	breq	mn15

	cpi	A,CH_DN
	brne	mn19			;skip if not channel up/down

mn15:	cli
	ldi	rcnt_lo,low(M_ALPS_1D)	;time to rotate one degree (approx)
	ldi	rcnt_hi,high(M_ALPS_1D)
	sbrc	aflags,m_alps		;skip if Altronics pot installed
	rjmp	mn16

	ldi	rcnt_lo,low(M_ALT_1D)	;Altronics motor is a little faster...
	ldi	rcnt_hi,high(M_ALT_1D)	;...so we need less run time

mn16:	sbr	aflags,1<<m_dir|1<<m_cmd ;rotate right, motor on
mn17:	cpi	A,CH_UP
	breq	mn18			;skip if volume up request 
	cbr	aflags,1<<m_dir		;else clear bit to rotate left

mn18:	sei
	rjmp	mn20			;done

; Numeric key function (source select)

mn19:	cpi	A,1
	brlo	mn22			;ignore if key '0'
	
	cpi	A,6
	brsh	mn22			;ignore if > key '5'

	mov	B,A
	rcall	select_source

; Recognised key code, flash the 'ack' LED

mn20:	cbr	aflags,1<<m_mute
mn21:	rcall	flash_led
	rjmp	mn23

; Check if new switch pressed & update if needed

mn22:	sbrs	bflags,sw_rdy		;skip if new switch pressed
	rjmp	mn23

	cbr	bflags,1<<sw_rdy
	mov	B,sw_bit		;get the switch bit mask
	rcall	sel_src_bit		;select source
mn23:	rjmp	main			;loop


;**************************************************************************
;*
;* Select preamp source (drive interface select line low)
;*
;**************************************************************************

; B = desired source number (1-5)

select_source:
	tst	B
	breq	sel_src_bit

	ldi	A,0b00000100
sl1:	lsl	A			;turn the number into a bit mask
	dec	B
	brne	sl1

	mov	B,A

; B = desired source bit mask

sel_src_bit:
	in	A,DDRB
	andi	A,!C_MASK		;all control bits to inputs first...
	out	DDRB,A

	mov	XH,B			;save bit mask

	com	B			;invert it
	andi	B,C_MASK		;keep the control bits only

	in	A,PORTB			;get port (register) state
	andi	A,!C_MASK		;zero the control bits
	or	A,B			;include the control bit
	out	PORTB,A			;write chosen source bit low

	in	A,DDRB
	or	A,XH
	out	DDRB,A			;switch the source bit to an output

; Save source select bit in eeprom

	mov	sw_bit,XH		;save here (for ir codes - and first init!)
	ldi	A,ee_source
	rcall	ee_read			;read current source bit mask
	cp	B,XH
	breq	sl2			;skip update if not needed

	mov	B,XH			;else do it
	rcall	ee_write		;make it last (baby)
sl2:	ret


;**************************************************************************
;*
;* Wait for next infrared key code (used during startup)
;*
;* Exit: A = ir code
;*
;**************************************************************************

wait_key:
	sbis	PINB,JP1_SEL
	rjmp	wk1

	ldi	A,50			;setup jumper removed...
	rcall	delay			;short delay
	rjmp	reset			;then restart

wk1:	sbrs	bflags,ir_rdy
	rjmp	wait_key		;loop until key received

	cbr	bflags,1<<ir_rdy

	sbrc	bflags,ir_rpt
	rjmp	wait_key		;ignore if repeat code

	rcall	flash_led
	mov	B,ir_code
	andi	B,$7F		
	ret


;**************************************************************************
;*
;* Flash LED 'B' times
;*
;**************************************************************************

flash_it:
	rjmp	fi2

fi1:	rcall	flash_led

fi2:	sbis	PIND,ACK_LED
	rjmp	fi2			;wait for LED off

	ldi	A,250
	rcall	delay			;delay for about 250ms
	dec	B
	brne	fi1
	ret


;**************************************************************************
;*
;* Flash LED (timed by tmr0_irq)
;*
;**************************************************************************

flash_led:
	ldi	A,50			;about 50ms on time
	mov	led_tmr,A
	cbi	PORTD,ACK_LED		;switch on LED
	ret


;**************************************************************************
;*
;* Timed delay
;*
;**************************************************************************

delay:
	clr	tick_tmr
dy1:	cp	tick_tmr,A
	brlo	dy1			;A * 1.024ms
	ret


;**************************************************************************
;*
;* Initialise EEPROM
;*
;**************************************************************************

ee_init:
	ldi	A,ee_source		;set 'CD' as default source
	ldi	B,1<<CD_CTL
	rcall	ee_write

	ldi	A,ee_mute		;enable muting function
	ldi	B,-1
	rcall	ee_write

	ldi	A,ee_punch		;enable 'punch-through' mute
	rcall	ee_write

	ldi	A,ee_magic		;save the magic number
	ldi	B,$5A
	rcall	ee_write
	ret


;**************************************************************************
;*
;* EEPROM random read/write routines
;*
;* Entry: A = address, B = data (writes)
;* Exit : B = data (reads)
;*
;**************************************************************************

ee_read:
	sbic	EECR,EEWE
	rjmp	ee_read			;loop until last cycle complete

	out	EEAR,A			;write address
	sbi	EECR,EERE		;set read strobe
	in	B,EEDR			;get data
	ret				;(read takes 4 clk cycles)
	
; Writes typically take 2.5ms @ 5V

ee_write:
	sbic	EECR,EEWE
	rjmp	ee_write		;loop until last cycle complete

	out	EEAR,A			;write address
	out	EEDR,B			;write data
	cli				;interrupts off for the next bit...
	sbi	EECR,EEMWE		;set master write enable
	sbi	EECR,EEWE		;set write strobe
	sei
	ret
	

;**************************************************************************
;*
;* Timer/counter 0 overflow interrupt handler
;*
;* Tick interval is 64us (1/fOSC x 256)
;*
;**************************************************************************

tmr0_irq:
	in	status,SREG		;save state
	push	A			;save used global
	push	B
	push	ZH
	push	ZL
	
;**************************************************************************
;* Do full rate (64us) tasks
;**************************************************************************

tq1:	rcall	ir_receive		;run ir receive state machine

	dec	irate_tmr		;update 1/16 rate timer
	brne	tq4
	
;**************************************************************************
;* Do 1/16 rate (1.024ms) tasks
;**************************************************************************

	inc	tick_tmr		;update general-purpose timer

	ldi	A,16
	mov	irate_tmr,A		;reload rate timer

	sbrs	bflags,wd_res		;keep-alive set?
	rjmp	tq2

	cbr	bflags,1<<wd_res
	wdr				;kick the 'dog
	
tq2:	dec	led_tmr			;LED on timer
	brne	tq3
	sbi	PORTD,ACK_LED		;count expired, switch off

tq3:	rcall	motor_drive		;execute motor driver
	rcall	switch_scan		;and select switch scanner

tq4:	pop	ZL			;restore used registers
	pop	ZH
	pop	B
	pop	A
	out	SREG,status		;and flags
	reti


;**************************************************************************
;*
;* Infrared receive state machine (must be called every 64us)
;*
;**************************************************************************

ir_receive:
	inc	ir_timer		;bump ir protocol (bit) timer

; Start (resume) ir decode in the correct state.

	ldi	ZH,high(rc5_jmp_tbl)
	ldi	ZL,low(rc5_jmp_tbl)

	add	ZL,ir_state		;state number becomes index
	clr	A
	adc	ZH,A			;find the correct rountine...
	ijmp				; and run it
		
rc5_jmp_tbl:
	rjmp	rc5_s0
	rjmp	rc5_s1
	rjmp	rc5_s2
	rjmp	rc5_s3
	rjmp	rc5_s4
	rjmp	rc5_s5
	rjmp	rc5_s6
	rjmp	rc5_s7
	

;**************************************************************************
;
; RC5 State 0
;
; Wait for line idle (high)
;
;**************************************************************************

rc5_s0:
	sbis	PIND,IR_RXD		;wait for line high
	rjmp	exit_state
			
	clr	ir_timer		;init timer
	rjmp	next_state		;move to state 1 & exit


;**************************************************************************
;
; RC5 State 1
;
; Establish line is idle (high) for 5ms before waiting for a start bit
;
;**************************************************************************

rc5_s1:
	sbis	PIND,IR_RXD		;skip if line high
	rjmp	reset_state		;else reset & exit
	
	cpi	ir_timer,78		;idle high for 5ms?
	brlo	r1_0			;keep waiting if not
	rjmp	next_state		;else move to state 2 & exit

r1_0:	rjmp	exit_state
	

;**************************************************************************
;
; RC5 State 2
;
; Wait (forever) for start bit
;
;**************************************************************************

rc5_s2:
	sbic	PIND,IR_RXD		;skip if start bit detected
	rjmp	exit_state

r2_0:	clr	ir_timer		;init timer
	rjmp	next_state		;move to state 3 & exit


;**************************************************************************
;
; RC5 State 3
;
; Wait for rising edge of first start bit
;
;**************************************************************************

rc5_s3:
	sbis	PIND,IR_RXD		;skip if line returned high
	rjmp	r3_0			;else go check for timeout

	clr	ir_timer		;init timer
	rjmp	next_state		;move to state 4 & exit

r3_0:	cpi	ir_timer,31		;more than 2ms?
	brlo	exit_state		;exit to keep sampling
	rjmp	reset_state		;else reset


;**************************************************************************
;
; RC5 State 4
;
; Wait for trailing edge of second start bit to synchronise timing
;
;**************************************************************************

rc5_s4:
	sbic	PIND,IR_RXD		;skip when line goes low
	rjmp	r4_0			;else go check for timeout
	
	ldi	A,12			;init bit count
	mov	ir_bit_cnt,A
	clr	ir_sys			;zero the bit store
	clr	ir_cmd	

	clr	ir_timer		;init timer
	rjmp	next_state		;more to state 5 & exit

r4_0:	cpi	ir_timer,31		;more than 2ms?
	brlo	r4_1			;skip to keep sampling
	rjmp	reset_state		;timeout, reset
r4_1:	rjmp	exit_state


;**************************************************************************
;
; RC5 State 5
;
; Sample the line at 1/4 bit time and store result
;
;**************************************************************************

rc5_s5:
	cpi	ir_timer,21		;3/4 bit time from last edge (1.34ms)?
	brlo	r5_1			;keep waiting if not

	inc	ir_state		;assume will be low, next state = 6
	sbic	PIND,IR_RXD		;skip if sampled low
	rjmp	r5_2
	
	clc				;to shift in a low bit
r5_0:	rol	ir_cmd
	rol	ir_sys

	clr	ir_timer		;init timer
r5_1:	rjmp	exit_state

r5_2:	inc	ir_state		;sampled high, next state = 7
	sec
	rjmp	r5_0			;go shift in the high bit


;**************************************************************************
;
; RC5 State 6
;
; Bit was a low, so wait for rising edge to syncronize timing
;
;**************************************************************************

rc5_s6:
	sbis	PIND,IR_RXD		;skip if sampled high
	rjmp	r6_3			;else go check for timeout
	
r6_0:	dec	ir_bit_cnt		;done all bits?
	brne	r6_2			;not yet...
	
	mov	A,ir_cmd		;get all the bits...
	rol	A
	rol	ir_sys			;...in the right places
	rol	A
	rol	ir_sys

	bst	ir_sys,5		;move toggle bit...
	bld	ir_cmd,7		;to command byte MSB
	
	clt
	bld	ir_sys,5
	bld	ir_cmd,6		;clear unused bits

	cbr	bflags,1<<ir_rpt
	cp	ir_code,ir_cmd		;same code as last time?
	brne	r6_1			;skip if not
	sbr	bflags,1<<ir_rpt	;else flag key repeat
	
r6_1:	mov	ir_code,ir_cmd	
	sbr	bflags,1<<ir_rdy	;mail to the foreground
	rjmp	reset_state		;reset machine & exit
	
r6_2:	ldi	A,5			;more to do
	mov	ir_state,A		;loop back to get another bit
	clr	ir_timer
	rjmp	exit_state

r6_3:	cpi	ir_timer,35		;5/4 bit time from edge (2.2ms)?
	brlo	exit_state		;keep sampling if not
	rjmp	reset_state		;else timeout waiting for middle edge


;**************************************************************************
;
; RC5 State 7
;
; Bit was a high, so wait for falling edge to syncronize timing
;
;**************************************************************************

rc5_s7:
	sbic	PIND,IR_RXD		;skip if sampled low
	rjmp	r6_3			;else go check for timeout
	rjmp	r6_0


;**************************************************************************

; Common exit stuff...

next_state:
	inc	ir_state		;bump to next state
	rjmp	exit_state

reset_state:
	clr	ir_state		;reset machine

exit_state:
	ret


;**************************************************************************
;*
;* Motor drive routines
;*
;**************************************************************************

motor_drive:
	sbrs	aflags,m_cmd		;skip if motor run command
	rjmp	md9			; else go switch motor off

	sbrc	aflags,m_run		;skip if motor not already running	
	rjmp	md4

; Motor not running, check if a previous direction change
; has invoked a delay before starting it up again.

	sbrs	aflags,m_dly		;skip if delay in progress
	rjmp	md1			;else go start motor now

	dec	dir_cnt			;bump delay count
	brne	md11			;exit if more to wait
	
	cbr	aflags,1<<m_dly|1<<m_reset
	rjmp	md2

; Check if pot is at stops first (not currently used)

md1:	sbrs	aflags,m_reset		;skip if previous timeout (at stops)
	rjmp	md2

	rcall	get_dir			;get requested direction
	brtc	md11			;just exit if same direction

; Start the motor

md2:	in	A,PORTD			;get current port state
	andi	A,~M_MASK		;clear the motor control bits
	ldi	B,M_RIGHT
	sbrs	aflags,m_dir
	ldi	B,M_LEFT		;enable bits for chosen direction
md3:	or	A,B
	out	PORTD,A			;start motor
	mov	mot_ctl,B		;save control bits

; Load motor start -> comparator polling delay (for muting)
; Note: 120ms is tested min. for Altronics pot.

	ldi	mot_cnt,220
	sbr	aflags,1<<m_start|1<<m_run ;flag startup & running
	cbr	bflags,1<<c_tog
	rjmp	md11			;done for now

; Motor already running...

md4:	sbrs	aflags,m_start
	rjmp	md5

	dec	mot_cnt			;startup timer expired?
	brne	md6			;skip if not

	cbr	aflags,1<<m_start

md5:	sbrs	aflags,m_mute		;skip if mute in progress
	rjmp	md6

	sbis	ACSR,ACO		;trip
	rjmp	md6			;no trip

	sbr	bflags,1<<c_tog		;flag comparator stopped motor
	rjmp	md9

; Check if direction change requested

md6:	rcall	get_dir
	brtc	md8			;no change, go update timers

; Direction change requested, initiate delay

md7:	ldi	dir_cnt,100		;load delay count
	sbr	aflags,1<<m_dly
	rjmp	md10			;go stop motor

; Update run timer, switch motor off when it expires

md8:	subi	rcnt_lo,1	
	sbci	rcnt_hi,0
	brcc	md11

md9:	cbr	aflags,1<<m_cmd|1<<m_dly ;clear run request

; Switch motor off

md10:	cbr	aflags,1<<m_run|1<<m_start
	in	A,PORTD			;get current port state
	andi	A,~M_MASK		;clear the motor control bits	
	ori	A,M_STOP		;include the stop bits
	out	PORTD,A			;stop motor
md11:	ret


;**************************************************************************
;*
;* Determine if last saved motor direction = requested direction
;*
;* Exit : T=0 - same direction
;*        T=1 - opposite direction
;*
;**************************************************************************

get_dir:
	clt				;assume will be same
	mov	A,mot_ctl
	cpi	A,M_LEFT
	breq	gd1			;go if spinning left

	sbrs	aflags,m_dir		;skip if right (same)
	rjmp	gd2			;opposite
	rjmp	gd3

gd1:	sbrc	aflags,m_dir		;skip if left (same)
gd2:	set				;flag opposite
gd3:	ret


;**************************************************************************
;*
;* Selection switch scanning (make before break action).
;*
;* Inactive bits are pulled high, going low on a switch press. Noise
;* induced false detections are eliminated by sampling each detection
;* for 5 passes (5ms) before deeming it valid.
;*
;* Switch debouncing is unnecessary and slows things down - we must
;* have fast switching!
;*
;**************************************************************************

switch_scan:
	in	B,PINB
	com	B			;invert
	andi	B,C_MASK		;exclude all other port B bits

	mov	A,sw_bit		;get selected bit pattern
	com	A
	and	B,A			;exclude it from this read	
	breq	ss5			;no (new) closure, just exit

; For the case where more than one bit is active (high),
; ignore (clear) all but the most significant bit.

	clt
	ldi	A,9
ss1:	rol	B
	brcc	ss3			;bit not found yet...
	brtc	ss2			;found, skip if first
	clc				;second (or more) bit, clear it
ss2:	set				;to clear all following bits
ss3:	dec	A
	brne	ss1

	sbrc	bflags,sw_wait		;skip if timer not running
	rjmp	ss4	

	mov	sw_tmp,B		;save bit for next iternation
	ldi	A,5			;load 5ms timer
	mov	sw_tmr,A
	sbr	bflags,1<<sw_wait	;flag running
	rjmp	ss6

ss4:	cp	B,sw_tmp		;changed since last scan?
	brne	ss5			;abort if so
	
	dec	sw_tmr			;bump the timer
	brne	ss6			;exit if not expired

	sbrc	bflags,sw_rdy
	rjmp	ss5			;ignore if foreground yet to fetch
	
	mov	sw_bit,B		;save for foreground
	sbr	bflags,1<<sw_rdy	;and flag available

ss5:	cbr	bflags,1<<sw_wait
ss6:	ret


;**************************************************************************
;*
;* DATA
;*
;**************************************************************************

copyright:
.db	"(C)2006 Silicon Chip Publications P/L"	;copyright notice


;**************************************************************************
;*
;* Define used SRAM
;*
;**************************************************************************
	
	.dseg

; ... stack space only ...


;**************************************************************************
;*
;* Define used EEPROM
;*
;**************************************************************************

	.eseg
	.org	0

.db	0				;never use first byte!

ee_magic:
.db	0				;written at first power up

ee_ir_sys:
.db	0				;RC5 equipment address

ee_source:
.db	0				;last selected source (1-5)

ee_mute:
.db	0				;mute function status

ee_punch:
.db	0				;punch-through mute status
