;		CPRTrainer.asm 	-- CPR Training Beeper program for PIC16F628A
;		written by Jim Rowe. Last revised 23/10/2006
;		(note that PIC internal clock is used, 4MHz so 1mc = 1us)

 
	list	P=16F628A
	#include "p16f628A.inc"
	__config h'2139'
	
;	define variables at data memory locations
Counter1 equ	h'20'	; Timing counter variable 1
Counter2 equ 	h'21'	; Timing counter variable 2
Counter3 equ	h'22'	; Timing counter variable 3

; 	storage registers for compression beeps, cycles done
ComprBeeps	equ		h'23'	; number of compression beeps done
CyclesDone	equ		h'24'	; number of (30 beep + 2 breath) cycles done

; -----------------------------------------------------------------
; program begins

	ORG		h'00'		; normal start vector
	GOTO Initialise
	ORG		h'04'		; interrupt servicing vector
	GOTO Initialise		; dummy only, since we're not using interrupts
	
Initialise:
	CLRF INTCON			; disable all interrupts (we don't need them)
	BCF STATUS, RP0		; set for Bank0
	CLRF PORTA			; clear PortA (turns off piezo)
	MOVLW h'07'			; also set bits 0-2 of PortB to turn off LEDs 
	MOVWF PORTB
	BSF STATUS, RP0		; now set for Bank1
	MOVLW h'00'			; set RB0-7 as outputs
	MOVWF TRISB			; by loading into its config register
	MOVLW h'00'			; also set RA0-7 as outputs
	MOVWF TRISA			;
	BCF STATUS, RP0		; finally reset for Bank0

LoopStart:
	MOVLW h'06'			; reset cycles done counter for 6 cycles
	MOVWF CyclesDone
CycleStart:
	MOVLW h'1E'			; reset compression beeps counter to 30d
	MOVWF ComprBeeps
ContCycle:
	CALL Delay450ms		; first go wait for 450ms
	CALL ComBeepLED1	; then go do 151.2ms of 4kHz beep & LED1 flashing
	DECFSZ ComprBeeps,1	; now decrement comprbeeps ctr, skip if it reaches zero
	GOTO ContCycle		; otherwise keep looping
	CALL BreathLED2		; must have done 30 compr beeps, so signal for 2 breaths
	DECFSZ CyclesDone,1	; now decrement cycles done ctr, skip if it reaches zero
	GOTO CycleStart		; otherwise go begin another cycle
	CALL ChangeLED3		; must have done 6 cycles, so go signal changeover
	GOTO LoopStart		; then go back to begin all over again
;	
;   Main loop ends here -- subroutines follow
;   ---------------------------------------------------------
;
ComBeepLED1:
	; routine to light LED1 and beep at 3.96kHz for 151.2ms
	MOVLW h'04'			; first set Counter2 for 4 outer loops
	MOVWF Counter2
NextOuterLoop:
	MOVLW h'96'			; now set Counter3 for 150 periods
	MOVWF Counter3
NextPeriod:
	BSF PORTB, 4		; start by setting RB4, to start test pulse at TP2
	BCF PORTB, 0		; next turn on LED1 by resetting RB0
	BSF PORTA, 1		; then lift RA1, lower RA0
	BCF PORTA, 0		; to start first 1/2 of beep period
	CALL Delay126us	    ; and go wait for 126us
	BSF PORTB, 0		; now lift RB0 again to turn off LED1
	BCF PORTA, 1		; and drop RA1, raise RA0 to start second 1/2 of beep
	BSF PORTA, 0
	CALL Delay126us		; then go wait for 126us again, to end period
	DECFSZ Counter3, 1	; decrem Ctr3 and skip if it reaches zero
	GOTO NextPeriod		; otherwise keep looping
	DECFSZ Counter2, 1	; must have done 150 loops, so check if we've done four
	GOTO NextOuterLoop	; we haven't, so loop back for another
	BCF PORTB, 4		; must have done 4 outer loops, so reset RB4 to end TP2
	RETURN				; then return (having taken 4 x 150 x 0.252 = 151.2ms)
	
BreathLED2:
	; routine to light LED2 and beep at 4kHz for 2 secs, then silence for 1 sec
	; and finally light LED2 and beep at 4kHz for 2s again
	MOVLW h'28'			; first set Counter2 for 40 outer loops (40 x 50.4 = 2016ms)
	MOVWF Counter2
BreathLoop:
	MOVLW h'C8'			; now set Counter3 for 200 inner loops (periods)
	MOVWF Counter3
NextInLoop:
	BSF PORTB, 4		; begin by setting RB4 to start TP2 test pulse
	BCF PORTB, 1		; next reset RB1 to turn on LED2
	BSF PORTA, 1		; then lift RA1, lower RA0
	BCF PORTA, 0		; to start first 1/2 of beep period
	CALL Delay126us	    ; and go wait for 126us
	BSF PORTB, 1		; now lift RB1 again to turn off LED2
	BCF PORTA, 1		; and drop RA1, raise RA0 to start second 1/2 of beep
	BSF PORTA, 0
	CALL Delay126us		; then go wait for 126us again
	DECFSZ Counter3, 1	; decrem Ctr3 and skip if it reaches zero
	GOTO NextInLoop		; otherwise keep looping
	DECFSZ Counter2,1	; must have done 200 loops (50.4ms), so dec & test Ctr2
	GOTO BreathLoop		; not done 8 outer loops, so keep going
	BCF PORTB, 4		; must have done 40 outer loops (2016ms), so clear RB4

	CALL Delay1sec		; then wait a second for gap between beeps

SecondBeep:
	MOVLW h'28'			; now set Counter2 for 40 outer loops again
	MOVWF Counter2
BrethLoop2:
	MOVLW h'C8'			; and Counter3 again for 200 inner loops
	MOVWF Counter3
InLoop2:
	BSF PORTB, 4		; begin period 2 by setting RB4 to begin TP2 pulse
	BCF PORTB, 1		; then turn on LED2 by dropping RB1
	BSF PORTA, 1		; next lift RA1, lower RA0
	BCF PORTA, 0		; to start first 1/2 of beep period
	CALL Delay126us	    ; and go wait for 500us
	BSF PORTB, 1		; now lift RB1 again to turn off LED2
	BCF PORTA, 1		; and drop RA1, raise RA0 to start second 1/2 of beep
	BSF PORTA, 0
	CALL Delay126us		; then go wait for 500us again
	DECFSZ Counter3, 1	; decrem Ctr3 and skip if it reaches zero
	GOTO InLoop2		; otherwise keep looping, because not looped enough
	DECFSZ Counter2,1	; must have done 200 loops, so decrement & test Ctr2
	GOTO BrethLoop2		; can't have done 8 outer loops, so keep going
	BCF PORTB, 4		; must have done 40 x 200, so clear RB4 to end TP2
	RETURN				; and return
	 	
ChangeLED3:
	; routine to signal changeover of compressor and breather, after six cycles
	; (light LED3, give 3kHz/2kHz modulated beep for 2 sec)
	BSF PORTB, 4		; start by setting RB4 to begin TP2 pulse
	MOVLW h'08'			; now set Counter 2 for 10 outer loops
	MOVWF Counter2		; (10 times (225 x 0.33 + 230 x 0.504)ms = 20000.7ms)
OuterModLoop:
	MOVLW h'FF'			; next set Counter3 for 255 periods
	MOVWF Counter3		; of 3030Hz
New3kHzPeriod:
	BCF PORTB, 2		; then turn on LED3 by dropping RB2
	BSF PORTA, 1		; next lift RA1, lower RA0
	BCF PORTA, 0		; to start first 1/2 of beep period
	CALL Delay165us		; and go wait for 165us (1/2 cycle at 3030Hz)
	BSF PORTB, 2		; now lift RB2 again to turn off LED3
	BCF PORTA, 1		; and drop RA1, raise RA0 to start second 1/2 of beep
	BSF PORTA, 0
	CALL Delay165us		; then go wait for 165us again
	DECFSZ Counter3, 1	; decrem Ctr3 and skip if it reaches zero
	GOTO New3kHzPeriod	; otherwise keep looping (not done 255 loops)

	MOVLW h'E6'			; OK, we've done 3kHz beep, so now reset Counter3
	MOVWF Counter3		; for 230 periods of 1984Hz (= 115.9ms)
New2kHzPeriod:
	BCF PORTB, 2		; then turn on LED3 by dropping RB2
	BSF PORTA, 1		; next lift RA1, lower RA0
	BCF PORTA, 0		; to start first 1/2 of beep period
	CALL Delay252us		; and go wait for 252us (504us/2)
	BSF PORTB, 2		; now lift RB2 again to turn off LED3
	BCF PORTA, 1		; and drop RA1, raise RA0 to start second 1/2 of beep
	BSF PORTA, 0
	CALL Delay252us		; then go wait for 252us again
	DECFSZ Counter3, 1	; decrem Ctr3 and skip if it reaches zero
	GOTO New2kHzPeriod	; otherwise keep looping
	DECFSZ Counter2, 1	; decrem Ctr2 and skip if it reaches zero
	GOTO OuterModLoop	; not done eight loops yet, so keep looping
	BCF PORTB, 4		; must have done all loops, so end TP2 pulse
	RETURN				; and return

	; Delay timing subroutines follow
	
Delay126us:
	; routine to delay approx 126us (1/2 cycle of 4kHz) before returning
	MOVLW h'28'			; set Counter1 for 40 loops of 3us
	MOVWF Counter1		; (call and return add 6us more)
	DECFSZ Counter1, 1	; decrement Counter1 & skip when zero
	GOTO $-1			; not zero yet, so loop back
	RETURN				; done 40 loops (40 x 3us = 120) so return

Delay165us:
	; routine to delay approx 165us (1/2 cycle of 3030Hz) before returning
	MOVLW h'35'			; set Counter1 for 53 loops of 3us
	MOVWF Counter1		; (call and return add 6us more)
	DECFSZ Counter1, 1	; decrement Counter1 & skip when zero
	GOTO $-1			; not zero yet, so loop back
	RETURN				; done 53 loops (53 x 3us = 159) so return
	
Delay252us:
	; routine to delay approx 252us (1/2 cycle of 2kHz) before returning
	MOVLW h'29'			; set Counter1 for 41 loops
	MOVWF Counter1		; (call and return add 6us more)
	NOP					; loop start, has 3 NOPs for padding (6us per loop) 
	NOP
	NOP
	DECFSZ Counter1,1	; decrement Counter1 & skip when zero
	GOTO $-4			; not zero yet, so keep looping
	RETURN				; must have done 41 loops (41 x 6us = 246us)

Delay450ms:
	; routine to delay approx 450ms before returning
	MOVLW h'0A'			; first set Counter3 for 10 outer loops
	MOVWF Counter3
	MOVLW h'58'			; now set Counter2 for 88 loops
	MOVWF Counter2
	CALL Delay252us		; then wait 0.252ms
	CALL Delay252us		; and again, for a total of 0.504ms
	DECFSZ Counter2, 1	; then decrement Counter2 & skip when zero
	GOTO $-3			; not zero yet (88 loops), so loop back
	DECFSZ Counter3, 1	; done 88 x 0.504ms = 44.35ms, so check Ctr3
	GOTO $-7			; keep looping if not done 10 outer loops
	RETURN				; must have done 10 x 44.35ms = 443.5ms, so return
	
Delay1sec:
	; routine to delay approx 1 sec before returning
	MOVLW h'14'			; first set Counter3 for 20 outer loops
	MOVWF Counter3
	MOVLW h'C8'			; set Counter2 for 200 inner loops of 0.252ms
	MOVWF Counter2
	CALL Delay126us	  	; then wait 0.126ms
	CALL Delay126us		; and again, for a total of 0.252ms
	DECFSZ Counter2, 1	; then decrement Counter2 & skip when zero
	GOTO $-3			; not done 200 loops yet, so loop back
	DECFSZ Counter3, 1	; have done 200, so see if we've done 20
	GOTO $-7			; not done 20 outer loops, so keep going
	RETURN				; must have done 20 x 50.1ms = 1008ms, so return	

; ----------------------------------------------------------------	
	
	 	
	END
	
