; Air Conditioning Controller 


	list P=16F88
	#include p16f88.inc
	ERRORLEVEL -302
	ERRORLEVEL -306

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB3 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; Define variables at memory locations

EEPROM1			equ	H'01'	; non-volatile storage for period that duty cycle is measured over. DUTY_LENGTH 
; (5-minutes) 0-5V = 0-16 minutes (monitor period)
EEPROM2			equ	H'02'	; non-volatile storage for period that prevents retriggering of compressor
; RETRIGER_PERIOD 0-5V (255/8) for 0-15 seconds
EEPROM3			equ	H'03'	; storage of L-LOAD
EEPROM4			equ	H'04'	; storage of H_LOAD
EEPROM5			equ	H'05'	; storage of MULTIPLIER
EEPROM6			equ	H'06'	; storage of ADAPTATION
		
; Bank 0 RAM
; A/D values
VR1				equ	H'20'	; VR1 setting
VR2				equ	H'21'	; VR2 setting
VR3				equ	H'22'	; VR3 setting
THROTTLE		equ	H'23'	; Throttle value
DUTY_LENGTH		equ	H'24'	; duty cycle measurement period
RETRIGER_PERIOD	equ	H'25'	; retrigger period
MULTIPLIER		equ	H'26'	; multiplier of compressor drive duty cycle
L_LOAD			equ	H'27'	; low throttle setting
H_LOAD			equ	H'28'	; high throttle setting
; General
ADCOUNT			equ	H'29'	; address counter for A/D
TIMERH			equ	H'2A'	; ms byte of timer
TIMERL			equ	H'2B'	; ls byte of timer
FLASHER			equ	H'2C'	; LED flasher timer 
LOOP_COUNT		equ	H'2D'	; update A/D counter
STORE1			equ	H'2E'	; delay register
STORE2			equ	H'2F'	; delay register	

; compressor DEMAND duty cycle log (keep addresses for indirect addressing) 
DUTY0			equ	H'30'	; duty cycle log ms byte
DUTY1			equ	H'31'	; duty cycle log 
DUTY2			equ	H'32'	; duty cycle log 
DUTY3			equ	H'33'	; duty cycle log 
DUTY4			equ	H'34'
DUTY5			equ	H'35'	; duty cycle log 
DUTY6			equ	H'36'
DUTY7			equ	H'37'
DUTY8			equ	H'38'	; 
DUTY9			equ	H'39'
DUTY10			equ	H'3A'	; duty cycle log 
DUTY11			equ	H'3B'
DUTY12			equ	H'3C'
DUTY13			equ	H'3D'	; 
DUTY14			equ	H'3E'
DUTY15			equ	H'3F'	; duty cycle log ls byte

; compressor DRIVE duty cycle log for comparison (keep addresses for indirect addressing) 
DUTY_D0			equ	H'40'	; duty cycle log ms byte
DUTY_D1			equ	H'41'	; duty cycle log 
DUTY_D2			equ	H'42'	; duty cycle log 
DUTY_D3			equ	H'43'	; duty cycle log 
DUTY_D4			equ	H'44'
DUTY_D5			equ	H'45'	; duty cycle log 
DUTY_D6			equ	H'46'
DUTY_D7			equ	H'47'
DUTY_D8			equ	H'48'	; 
DUTY_D9			equ	H'49'
DUTY_D10		equ	H'4A'	; duty cycle log 
DUTY_D11		equ	H'4B'
DUTY_D12		equ	H'4C'
DUTY_D13		equ	H'4D'	; 
DUTY_D14		equ	H'4E'
DUTY_D15		equ	H'4F'	; duty cycle log ls byte

; compressor DRIVE duty cycle log for Fan relay on or off decision
DUTY_F0			equ	H'50'	; duty cycle log ms byte
DUTY_F1			equ	H'51'	; duty cycle log 
DUTY_F2			equ	H'52'	; duty cycle log 
DUTY_F3			equ	H'53'	; duty cycle log 
DUTY_F4			equ	H'54'
DUTY_F5			equ	H'55'	; duty cycle log 
DUTY_F6			equ	H'56'
DUTY_F7			equ	H'57'
DUTY_F8			equ	H'58'	; 
DUTY_F9			equ	H'59'
DUTY_F10		equ	H'5A'	; duty cycle log 
DUTY_F11		equ	H'5B'
DUTY_F12		equ	H'5C'
DUTY_F13		equ	H'5D'	; 
DUTY_F14		equ	H'5E'
DUTY_F15		equ	H'5F'	; duty cycle log ls byte

SAMPLE			equ	H'60'	; sampling counter
PORTB_STO		equ	H'61'	; portB storage
START_DELAY		equ	H'62'	; start delay counter
TEMP			equ	H'63'	; temporary register
ONES_DEM		equ	H'64'	; ones count for compressor demand
ONES_DRV		equ	H'65'	; ones count for compressor drive
THRESH_FLG		equ	H'66'	; threshold flag	
SPEED_FLG		equ	H'67'	; speed signal flag
SPEED_STO		equ	H'68'	; speed signal storage
SPEED_PERIOD	equ	H'69'	; period over which a speed signal must be present for validation of signal
ONES_DEM0		equ	H'6A'	; ones demand count after processing ms byts
ONES_DEM1		equ	H'6B'	; ones demand count after processing ls byte
RETRIG_COUNTER	equ	H'6C'	; retrigger counter
TEMP_VR3		equ	H'6D'	; temporary register for VR3 for retrigger period calculation
WIDTH			equ	H'6E'	; test signal of compressor duty overall width
ONES_SHOW		equ	H'6F'	; test signal of compressor, compressor high length within the width
			 
; All Banks RAM

W_TMP			equ	H'70'	; storage of w before interrupt
STATUS_TMP		equ	H'71'	; status storage before interrupt
ADAPT			equ	H'72'	; adaptation (control of recovery rate for cooldown)
COOL_FLG		equ	H'73'	; flag to pause logging with cooldown
STOP_FLG		equ	H'74'	; flag to pause logging when stopped
HI_FLG			equ	H'75'	; flag to pause logging with high load
START_LOG		equ	H'76'	; start logging after initial compressor run after starting

; Math routines registers
AARGB0			equ	H'78'	; ms multiplier
AARGB1			equ	H'79'	; ls multiplier
BARGB0			equ	H'7A'	; ms multiplier
BARGB1			equ	H'7B'	; ms multiplier
LOOPCOUNT		equ	H'7C'	; counter
AARGB3			equ	H'7D'	; divide routine
REMB0			equ	H'7E'	; remainder
TEMP1			equ	H'7F'	; temporary register

; preprogram EEPROM DATA 
	
	ORG     2101			; start at H01 (EEPROM1)
	
	DE 	D'80'	; 5 mins	; duty length
	DE	D'10'	; 5 seconds	; retrigger period
	DE	D'64'	; 1.25V		; low load
	DE	D'192'	; 3.75V		; high load
	DE	D'50'	; x 1.2		; multiplier (127 + MULTIPLIER/2)/127 range from 1 to 2
	DE	D'00'	; 0			; minimal control adaptation

; start at memory 0
	org	0
	goto	SETUP
	org	4
	goto	INTERRUPT

; **********************************************************************************************

SETUP
	clrf	PORTB		; port B outputs low
  	clrf	PORTA		; port A output low
	bsf		STATUS,RP0	; select memory bank 1

; set inputs/outputs
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'01001111'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'00111100'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000111'	; settings (pullups enabled, TMR0/256,negative edge interrupt)
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'00111100'	; AN2 to AN5 are analog inputs
	movwf	ANSEL
	movlw	B'01000000'	; left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'11010000'	; Fosc, channel etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00111000'	; 0011 for 500kHz
	movwf	OSCCON		; 500kHz osc
	bcf		STATUS,RP0	; select memory bank 0

	movlw	B'00000001'	; timer 1 fosc/4
	movwf	T1CON
	bsf		STATUS,RP0	; select memory bank 1		
	bcf		STATUS,RP0	; memory bank 0

; read EEPROM1 (measuremnt period for duty cycle)
	movlw	EEPROM1
	call	EEREAD_NO_INTERRUPT	; sets EEADR
	movwf	DUTY_LENGTH	; duty cycle measurement period

; read EEPROM2 (retrigger period)
	movlw	EEPROM2
	call	EEREAD_NO_INTERRUPT	; sets EEADR
	movwf	RETRIGER_PERIOD

; read EEPROM3 (low load)
	movlw	EEPROM3
	call	EEREAD_NO_INTERRUPT	; sets EEADR
	movwf	L_LOAD

; read EEPROM4 (high load)
	movlw	EEPROM4
	call	EEREAD_NO_INTERRUPT	; sets EEADR
	movwf	H_LOAD

; read EEPROM5 (multiplier)
	movlw	EEPROM5
	call	EEREAD_NO_INTERRUPT	; sets EEADR
	movwf	MULTIPLIER

; read EEPROM6 (adaptation)
	movlw	EEPROM6
	call	EEREAD_NO_INTERRUPT	; sets EEADR
	movwf	ADAPT

; initial conditions
	movlw	D'255'
	call	DELAYX			; delay to start
	movlw	D'10'
	movwf	START_DELAY		; delay after interrupt starts 5 seconds
	clrf	ADCOUNT			; address counter

; start duty cycle log at 0 (off)
	clrf	DUTY0			; duty cycle log ms byte					
	clrf	DUTY1			; duty cycle log
	clrf	DUTY2
	clrf	DUTY3			; 
	clrf	DUTY4			; duty cycle log 
	clrf	DUTY5			; duty cycle log 
	clrf	DUTY6
	clrf	DUTY7
	clrf	DUTY8			; 
	clrf	DUTY9
	clrf	DUTY10			; duty cycle log 
	clrf	DUTY11
	clrf	DUTY12
	clrf	DUTY13			; 
	clrf	DUTY14
	clrf	DUTY15			; duty cycle log ls byte

; start duty cycle log at 0 (off)
	clrf	DUTY_D0			; duty cycle log ms byte					
	clrf	DUTY_D1			; duty cycle log
	clrf	DUTY_D2
	clrf	DUTY_D3			; 
	clrf	DUTY_D4			; duty cycle log 
	clrf	DUTY_D5			; duty cycle log 
	clrf	DUTY_D6
	clrf	DUTY_D7
	clrf	DUTY_D8			; 
	clrf	DUTY_D9
	clrf	DUTY_D10		; duty cycle log 
	clrf	DUTY_D11
	clrf	DUTY_D12
	clrf	DUTY_D13		; 
	clrf	DUTY_D14
	clrf	DUTY_D15		; duty cycle log ls byte 

; start duty cycle log at 0 (off)for fan check
	clrf	DUTY_F0			; duty cycle log ms byte					
	clrf	DUTY_F1			; duty cycle log
	clrf	DUTY_F2
	clrf	DUTY_F3			; 
	clrf	DUTY_F4			; duty cycle log 
	clrf	DUTY_F5			; duty cycle log 
	clrf	DUTY_F6
	clrf	DUTY_F7
	clrf	DUTY_F8			; 
	clrf	DUTY_F9
	clrf	DUTY_F10		; duty cycle log 
	clrf	DUTY_F11
	clrf	DUTY_F12
	clrf	DUTY_F13		; 
	clrf	DUTY_F14
	clrf	DUTY_F15		; duty cycle log ls byte 

	movf	DUTY_LENGTH,w	; duty cycle tally length period
	clrf	AARGB0
	movwf	AARGB1
	movlw	D'17'
	movwf	BARGB0			; divide by 17 so 255 (max) gives 15. (15x0.5s=7.5s sampling rate)
; 							; sample rate x 128 = total recording of duty from compressor DEM 
;							; and drive signals. 7.5 x 128 = 16 minutes
	call	DIV16_8			; divide
	movf	AARGB1,w				
	movwf	SAMPLE			; sampling counter
	bcf		THRESH_FLG,0	; flag clear for reset
	movf	PORTB,w			; speed signal storage
	movwf	SPEED_STO		; both the same for no signal
	clrf	SPEED_FLG		; speed signal flag cleared
	clrf	TIMERH
	clrf	TIMERL			; override timers
	clrf	RETRIG_COUNTER	; retrigger counter
	clrf	PORTB_STO		; portB storage
	clrf	COOL_FLG		; logging pause flag with cooldown
	clrf	STOP_FLG		; logging pause flag when stopped 
	clrf	HI_FLG			; logging pause flag with high load
	clrf	START_LOG
	bsf		START_LOG,0		; starting log flag
	clrf	ONES_DEM		; on/off value for compressor demand flag
	clrf	ONES_DRV		; on/off value for compressor drive signal

; allow interrupts
	bsf		INTCON,INTE		; negative edge interrupt
	bcf		INTCON,INTF		; clear edge interrupt flag
	bsf		STATUS,RP0		; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 overflow interrupt
	bcf		STATUS,RP0		; select memory bank 0
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag
	bsf		INTCON,PEIE		; enable periperal interrupts 	 
	bsf 	INTCON,GIE		; set global interrupt enable 

CYCLE						; code cycle to do A/D, speed signal detect and threshold detect, logic etc

; speed signal
	btfss	PORTB,1			; speed input if set check if last time was set
	goto	CK_CLR_PORTB1
	btfss	SPEED_STO,1		; if bit 1 set no change if clear then change
	goto	DETECT_SPEED	; speed signal detected
	goto	OUT_SPEED		; no change
CK_CLR_PORTB1
	btfss	SPEED_STO,1		; if bit 1 set then a change in level
	goto	OUT_SPEED		; no change
DETECT_SPEED
	bsf		SPEED_FLG,0		; changed so set detect speed flag
	movlw	D'03'			; reset timer to 3 so it does not timeout and clear speed flag
	movwf	SPEED_PERIOD	;
	movf	PORTB,w			; store portB value to storage so wait for another change in value
	movwf	SPEED_STO		
OUT_SPEED	

; update A/D values
; A/D conversion
; set analog input address

	movlw	B'11000111'		; mask bits 5:3
	andwf	ADCON0,f		; set bits 5:3 to zero
	btfsc	ADCOUNT,0		; if bit 0 set then set bit 3 in ADCON0
	bsf		ADCON0,3
	btfsc	ADCOUNT,1		; if bit 1 set then set bit 4 in ADCON0
	bsf		ADCON0,4
	btfsc	ADCOUNT,2		; if bit 2 set,set bit 5
	bsf		ADCON0,5

	movf	ADCOUNT,w		; find address
	btfsc	STATUS,Z		; if zero
	goto	CH_0AD
	movf	ADCOUNT,w
	xorlw	0x01			; 1
	btfsc	STATUS,Z
	goto	CH_1AD
	movf	ADCOUNT,w
	xorlw	0x02			; 2
	btfsc	STATUS,Z
	goto	CH_2AD
	movf	ADCOUNT,w
	xorlw	0x03			; 3
	btfsc	STATUS,Z
	goto	CH_3AD
	movf	ADCOUNT,w
	xorlw	0x04			; 4
	btfsc	STATUS,Z
	goto	CH_4AD
	movf	ADCOUNT,w
	xorlw	0x05			; 5
	btfsc	STATUS,Z
	goto	CH_5AD

; Channel 0 A/D value
CH_0AD						; channels 0 and 1 not used
CH_1AD ; channel not used
	goto	NEW_ADDRESS		; end

; Channel 2 A/D value
CH_2AD
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV2
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV2
	movf	ADRESH,w
	movwf	VR1				; VR1 setting
	goto	NEW_ADDRESS		; end

; Channel 3 A/D value
CH_3AD
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV3
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV3
	movf	ADRESH,w
	movwf	VR2				; VR2 setting
	goto	NEW_ADDRESS	; end

; Channel 4 A/D value
CH_4AD
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV4
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV4
	movf	ADRESH,w
	movwf	VR3				; VR3 setting
	goto	NEW_ADDRESS		; end

; Channel 5 A/D value
CH_5AD
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV5
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV5
	movf	ADRESH,w
	movwf	THROTTLE		; Throttle value

; switch H-load, L_load LED on or off
; H_LOAD
	movf	H_LOAD,w
	subwf	THROTTLE,w		; if throttle > high load switch on LED
	btfss	STATUS,C		; if positive
	goto	H_LOAD_LED_OFF
	bsf		PORTB,7			; H_LOAD LED on
	goto	BY_H_LOAD_LED
H_LOAD_LED_OFF
	bcf		PORTB,7
BY_H_LOAD_LED

; L_LOAD
	movf	THROTTLE,w
	subwf	L_LOAD,w		; if throttle < low load switch on LED
	btfss	STATUS,C		; if positive
	goto	L_LOAD_LED_OFF
	bsf		PORTA,6			; L_LOAD LED on
	goto	BY_L_LOAD_LED
L_LOAD_LED_OFF
	bcf		PORTA,6
BY_L_LOAD_LED

; set new address
NEW_ADDRESS
	incf	ADCOUNT,f		; counter
	movf	ADCOUNT,w
	sublw	B'00000101'		; compare with 5,if positive or zero leave as is
	btfsc	STATUS,C
	goto	CYCLE			; do A/D for all until back to 0
	clrf	ADCOUNT			; back to zero

; if LK1 out (RB2 high)
	btfss	PORTB,2			; high 
	goto	LINK1

; VR1
	movf	VR1,w			; VR1 to multiplier
	xorwf	MULTIPLIER,w	; compare
	btfsc	STATUS,Z
	goto	BYPASS_EEPROM5	; do not write to EEPROM if the same
	
	movlw	EEPROM5
	call	EEREAD
	movf	VR1,w
	call	EWRITE
	movf	VR1,w			; VR1 to multiplier
	movwf	MULTIPLIER		; new value
BYPASS_EEPROM5
	
; VR2
	movf	VR2,w			; VR2 to low load
	xorwf	L_LOAD,w		; compare
	btfsc	STATUS,Z
	goto	BYPASS_EEPROM3	; do not write to EEPROM if the same
	
	movlw	EEPROM3
	call	EEREAD
	movf	VR2,w
	call	EWRITE
	movf	VR2,w			; VR2 to L_LOAD
	movwf	L_LOAD			; new value
BYPASS_EEPROM3

; VR3
	movf	VR3,w			; VR3 to high load
	xorwf	H_LOAD,w		; compare
	btfsc	STATUS,Z
	goto	BYPASS_EEPROM4	; do not write to EEPROM if the same
	
	movlw	EEPROM4
	call	EEREAD
	movf	VR3,w
	call	EWRITE
	movf	VR3,w			; VR3 to high load
	movwf	H_LOAD			; new value
BYPASS_EEPROM4
	goto	LOGIC

LINK1
; Link LK1 in, then place values into delays 
; VR1 
	movf	VR1,w			; VR1 to adptation
	xorwf	ADAPT,w			; compare
	btfsc	STATUS,Z
	goto	BYPASS_EEPROM6	; do not write to EEPROM if the same
	
	movlw	EEPROM6
	call	EEREAD
	movf	VR1,w
	call	EWRITE
	movf	VR1,w			; VR1 to adaptation
	movwf	ADAPT			; new value
BYPASS_EEPROM6 

; VR2
	movf	VR2,w			; VR2 to duty length
	xorwf	DUTY_LENGTH,w	; compare
	btfsc	STATUS,Z
	goto	BYPASS_EEPROM1	; do not write to EEPROM if the same
	
	movlw	EEPROM1
	call	EEREAD
	movf	VR2,w
	call	EWRITE
	movf	VR2,w			; VR2 to duty length
	movwf	DUTY_LENGTH		; new value
BYPASS_EEPROM1

; VR3
	movf	VR3,w			; VR3 /8 to retrigger period
	movwf	TEMP_VR3
	rrf		TEMP_VR3,f		; /2
	rrf		TEMP_VR3,f		; /4
	rrf		TEMP_VR3,w		; /8 (255max/8 = 32 x 0.5s)
	andlw	B'00011111'		; remove ms bits
	movwf	TEMP_VR3
	xorwf	RETRIGER_PERIOD,w		; compare
	btfsc	STATUS,Z
	goto	BYPASS_EEPROM2	; do not write to EEPROM if the same
	
	movlw	EEPROM2
	call	EEREAD
	movf	TEMP_VR3,w
	call	EWRITE
	movf	TEMP_VR3,w		; processed VR3 value to retrigger period
	movwf	RETRIGER_PERIOD	; new value
BYPASS_EEPROM2
	
LOGIC
; compressor fan on set condenser relay on
	btfsc	PORTA,0			; ## disable for logging test
	bsf		PORTA,1			; ## disable for logging test
	
; if over-ride then operate air con as normal with compressor drive following compressor demand
	btfss	PORTA,7			; if set then override
	goto	LOGIC0
	btfss	PORTB_STO,4		; compressor demand
	goto	NO_DEM
	bsf		PORTA,0			; compressor on
	goto	CYCLE
NO_DEM
	movf	RETRIGER_PERIOD,w; value is x 0.5s
	movwf	RETRIG_COUNTER	; load with retrigger period
	bcf		PORTA,0			; compressor off
	goto	CYCLE
LOGIC0
; HIGH LOAD SWITCH OFF
	btfss	PORTB,7			; if high load LED is on, switch off compressor
	goto	LOGIC1
	movf	RETRIG_COUNTER,w
	btfss	STATUS,Z		; if zero can switch off compressor
	goto	LOGIC1

	movf	RETRIGER_PERIOD,w; value is x 0.5s
	movwf	RETRIG_COUNTER	; load with retrigger period
	bcf		PORTA,0			; compressor off (retrig counter will prevent turn on until it is zero)

LOGIC1
; No high or low load condition (compressor drive follows compressor demand)
	btfsc	PORTB,7			; high load LED if set check next logic 
	goto	LOGIC2
	btfsc	PORTA,6			; low load LED if set check next logic
	goto	LOGIC2
; do not switch on until retrigger counter is zero
	btfss	PORTB_STO,4		; if set then compressor demand signal is high		
	goto	COMPRESSOR_OFF1
	movf	RETRIG_COUNTER,w; on demand
	btfsc	STATUS,Z		; if zero can switch on compressor
	bsf		PORTA,0			; compressor on
	goto	LOGIC2
COMPRESSOR_OFF1				; off demand
	bcf		PORTA,0			; compressor off
	
LOGIC2
; low load control
; check if Throttle level is below low load setting (RA6 high)
	btfss	PORTA,6			; if clear bypass
	goto	CYCLE			; bypass switch pressed?
; if compressor demand is on, run the compressor if speed signal is valid
	btfss	PORTB_STO,4		; if set then compressor demand signal is high	
	goto	NOT_DEMANDED	; low load and demand off 
SET_COMPRESS_ON
	btfss	PORTB,3			; link out
	goto	ON_WITH_LINK
	btfss	SPEED_FLG,0		; no speed then off
	goto	SET_COMPRESS_OFF
ON_WITH_LINK
	movf	RETRIG_COUNTER,w; on demand
	btfsc	STATUS,Z
	bsf		PORTA,0			; compressor on	when counter 0
	goto	CYCLE

NOT_DEMANDED				; no compressor demand
; decisions
; 1.if dem signal tally is <3 do not switch on 
	movf	ONES_DEM,w
	sublw	D'03'
	btfsc	STATUS,C		; if <4 bypass decisions
	goto	SET_COMPRESS_OFF; 

; 2.if the speed signal ceases or is off, stop compressor (LK2 out only)
; (wait till retrigger counter is zero)
	btfss	PORTB,3			; link 2 out
	goto	NEXT_DECISION
	btfss	SPEED_FLG,0		; no speed then off
	goto	SET_COMPRESS_OFF

NEXT_DECISION
; 3.if threshold is not reached (THRESH_FLG,0 set)and a speed signal(SPEED_FLG,0 set),start compressor 
; (set retrigger counter)
; 4.threshold is not reached (THRESH_FLG,0 set)and LK2 in, start compressor (set retrigger counter)
	btfss	THRESH_FLG,0	; if set can run compressor
	goto	SET_COMPRESS_OFF
	btfss	PORTB,3			; LK2 if in (low) can run compressor
	goto	SET_COMPRESS_ON
	btfsc	SPEED_FLG,0		; no speed then off
	goto	SET_COMPRESS_ON	; compressor on if speed

SET_COMPRESS_OFF
	movf	RETRIGER_PERIOD,w; value is x 0.5s
	movwf	RETRIG_COUNTER	; load with retrigger period
	bcf		PORTA,0			; compressor off

	goto	CYCLE

; ******************************************************************************************************
; INTERRUPT

; start interrupt by saving w and status registers 
	
INTERRUPT
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp 

	btfss	PIR1,TMR1IF		; timer 1 interrupt flag
	goto	CHECK_EDGE
	bcf		PIR1,TMR1IF		; clear flag


; on overflow set timer to 62500 so 4x 1/500kHz = 8us. 8us x 62500 = update every .5s 
; HFFFF - D6250 = D3035 so preload with this offset

	bcf		T1CON,TMR1ON	; stop timer
	movlw	H'0B'
	movwf	TMR1H			; timer high byte
	movlw	H'DB'			; 
	movwf	TMR1L			; timer low byte
	bsf		T1CON,TMR1ON	; restart timer	

; start delay

	movf	START_DELAY,w	; start delay
	btfsc	STATUS,Z		; when zero do not decrement
	goto	FLASH_RATE		; end of start delay 
	decf	START_DELAY,f	; decrease start delay
	goto	RECLAIM

; flasher rate counter	
FLASH_RATE
	incf	FLASHER,f		; LED flasher timer

; retrigger timer
	movf	RETRIG_COUNTER,w; retrigger counter
	btfsc	STATUS,Z		; if zero do not decrease
	goto	OVER_R_TIME
	decf	RETRIG_COUNTER,f; reduce to zero

; over-ride timer
OVER_R_TIME
	movf	TIMERL,w		; ls byte of timer
	btfss	STATUS,Z		; if zero check ms byte
	goto	DEC_LS_TIMER
	movf	TIMERH,w		; ms byte of timer
	btfsc	STATUS,Z		; if zero do not decrease
	goto	OVER_LED_OFF	; LED off when timers are 0
	
DEC_LS_TIMER
	bsf		PORTA,7			; override LED on when timer running
	movf	TIMERL,w
	btfsc	STATUS,Z		; if zero decrease timerH
	decf	TIMERH,f		; bypass decrement if timerL not zero
	decf	TIMERL,f		; decrease timerL
	goto	SPEEDTIME
OVER_LED_OFF
	bcf		PORTA,7			; if zero switch override LED off

; speed period decreased
SPEEDTIME
	movf	SPEED_PERIOD,w	; speed period counter
	btfsc	STATUS,Z		; if zero clear speed flag
	goto	CLEAR_SPEED_FLAG
	decf	SPEED_PERIOD,f
	btfsc	STATUS,Z		; if zero 
CLEAR_SPEED_FLAG
	clrf	SPEED_FLG
; if flag set flash speed LED
	bcf		PORTB,5			; clear and light if required
	btfss	SPEED_FLG,0		; if set flash
	goto	BY_FLSH_SPEED_LED
	btfss	FLASHER,0
	bsf		PORTB,5			; flash	
BY_FLSH_SPEED_LED

; test compressor demand flag signal 
; remove internal pullup, momentarily take RB4 low
COMP_DEM_FLAG_SIGNAL

	bcf		PORTB,4		; output low when selected as an output
	bsf		STATUS,RP0	; select memory bank 1
	bsf		OPTION_REG,7; internal pullups off	
	bsf		TRISB,4		; RB4 an input
	nop					; wait for capacitor charge
	nop
	nop
	nop
	nop
	nop
	nop					; 14us + 2us (next instruction)long enough for 100ohm to charge 100nF
	bcf		STATUS,RP0	; select memory bank 0
	movf	PORTB,w		; read portB
	movwf	PORTB_STO	; store
	bsf		STATUS,RP0	; select memory bank 1
	bcf		OPTION_REG,7; pullups on again
	bcf		TRISB,4		; RB4 output
	bcf		STATUS,RP0	; select memory bank 0

; if compressor demand flag goes high clear COOL_FLG (cooldown flag)
COOL_FLAG
	btfss	COOL_FLG,0		; check when set
	goto	HIGH_FLAG
	btfsc	PORTB_STO,4
	clrf	COOL_FLG		; cleared
HIGH_FLAG	
; if compressor demand flag goes low then high clear HI_FLG
	btfss	HI_FLG,0		; check when set
	goto	STOPPED_FLAG
	btfss	PORTB_STO,4		
	bsf		HI_FLG,1		; set bit 1 when compressor goes low
; if bit 1 is set and compressor demand flag goes high clear flag
	btfss	PORTB_STO,4
	goto	STOPPED_FLAG
	btfsc	HI_FLG,1		; compressor demand flag high and bit 1 set
	clrf	HI_FLG			; cleared when bit 1 high
STOPPED_FLAG
; if compressor demand flag goes low then high clear STOP_FLG
	btfss	STOP_FLG,0		; check when set
	goto	STARTING_LOG
	btfss	PORTB_STO,4		
	bsf		STOP_FLG,1		; set bit 1 when compressor goes low
; if bit 1 is set and compressor demand flag goes high clear flag
	btfss	PORTB_STO,4
	goto	STARTING_LOG
	btfsc	STOP_FLG,1		; compressor demand flag high
	clrf	STOP_FLG		; cleared when bit 1 high

; clear START_LOG when compressor flag goes high then low 
; (this flag is set at startup and when there is no record of 
; compressor flag and the compressor flag is high) 
;
STARTING_LOG
	btfss	START_LOG,0		; check when set
	goto	STO_DEM_SIG	 
	btfsc	PORTB_STO,4
	bsf		START_LOG,1		; set bit 1 if compressor drive goes high
; if START_LOG,1 is set and compressor drive goes low clear the drive flag
	btfsc	PORTB_STO,4		; if low
	goto	TEST_DRV_SIG_FOR_FAN; log drive signal for fan run
	btfsc	START_LOG,1		; if set clear flag
	clrf	START_LOG
	goto	TEST_DRV_SIG_FOR_FAN

; store compressor demand signal every time SAMPLE timer is zero
STO_DEM_SIG
	movf	SAMPLE,w
	btfsc	STATUS,Z		; if zero add log
	goto	ADD_SAMPLE		; add to log
	decf	SAMPLE,f
	btfss	STATUS,Z		; if zero add to log
	goto	CHECK_EDGE
ADD_SAMPLE
	movf	DUTY_LENGTH,w	; duty cycle tally length period
	sublw	D'17'
	btfsc	STATUS,C		; if less than 17 leave SAMPLE at 0
	goto	ADD_LOG	
	clrf	AARGB0
	movf	DUTY_LENGTH,w
	movwf	AARGB1
	movlw	D'17'
	movwf	BARGB0			; divide by 17 so 255 (max) gives 15. (15x0.5s=7.5s sampling rate)
; 							; sample rate x 128 = total recording of duty from compressor DEM 
;							; and drive signals. 7.5 x 128 = 16 minutes
	call	DIV16_8			; divide
	movf	AARGB1,w				
	movwf	SAMPLE			; sampling counter

ADD_LOG	
; logging of compressor demand signal
; compressor demand flag stored in PORTB_STO,4

; pausing logic

; if COOL_FLG,0 set, bypass recording
	btfsc	COOL_FLG,0
	goto	END_ONES_DEM_TALLY
; if hi_flg,0 set, bypass recording
	btfsc	HI_FLG,0
	goto	END_ONES_DEM_TALLY
; if stop_flg,0 set, bypass recording
	btfsc	STOP_FLG,0
	goto	END_ONES_DEM_TALLY

; if over ride bypass checks
	btfsc	PORTA,7
	goto	REC_DEM_FLG

; if high high load and compressor demand flag is high bypass logging (pause)
	btfss	PORTB,7			; high load when set
	goto	LOW_LOAD_L
	btfss	PORTB_STO,4		; compressor demand flag
	goto	REC_DEM_FLG		; record or log the compressor flag
	bsf		HI_FLG,0		; set flag to prevent logging until reset
	goto	END_ONES_DEM_TALLY

; check level of compressor demand flag when in low load
LOW_LOAD_L
; if demand flag is low and compressor is on then pause the record as it is running the extra cooldown
	btfss	PORTA,6		; if low load
	goto	REC_DEM_FLG	; record compressor demand flag
	btfsc	PORTB_STO,4	; demand flag, if set then record (if not stopped)
	goto	STOP_NO_FLAG; record if not stopped
	btfss	PORTA,0		; compressor, if on do not record status
	goto	REC_DEM_FLG	; record
	bsf		COOL_FLG,0	; set flag to stop recording (logging)
	goto	END_ONES_DEM_TALLY; bypass tally

; check compressor demand flag when stopped and LK2 out
; if demand flag is high pause the record
STOP_NO_FLAG
	btfss	PORTB,3		; link 2
	goto	REC_DEM_FLG
	btfsc	SPEED_FLG,0	; no speed then do not record
	goto	REC_DEM_FLG
	btfss	PORTB_STO,4	; if compressor demand flag set do not record
	goto	REC_DEM_FLG	; record
	bsf		STOP_FLG,0	; stopped flag stops recording 
	goto	END_ONES_DEM_TALLY; bypass tally (log)

; logging of compressor demand flag signal
; The value in each bit records demand flag status
;(1=compressor demand on, 0=DEM off)
REC_DEM_FLG
	bcf		STATUS,C	; clear carry
	btfsc	PORTB_STO,4	; if set then set carry
	bsf		STATUS,C	; place current status into DUTY15 ls bit

	rlf		DUTY15,f	; ls byte. 
	rlf		DUTY14,f	; duty cycle log 
	rlf		DUTY13,f	; 
	rlf		DUTY12,f	
	rlf		DUTY11,f	; duty cycle log 
	rlf		DUTY10,f
	rlf		DUTY9,f
	rlf		DUTY8,f		; 
	rlf		DUTY7,f
	rlf		DUTY6,f		; duty cycle log 
	rlf		DUTY5,f
	rlf		DUTY4,f
	rlf		DUTY3,f		; 
	rlf		DUTY2,f
	rlf		DUTY1,f		; 
	rlf		DUTY0,f		; ms byte 16 minutes tally

; count number of 1's 
	clrf	ONES_DEM	; 1's count
	movlw	D'16'
	movwf	TEMP		; 128 bits (16 bytes)to count
; count 1's

	movlw	H'3F'		; DUTY15 address
	movwf	FSR			; start address

COUNT_ONES_DEM
	btfsc	INDF,0		; if set increase ones
	incf	ONES_DEM,f
	btfsc	INDF,1		; if set increase ones
	incf	ONES_DEM,f
	btfsc	INDF,2		; if set increase ones
	incf	ONES_DEM,f
	btfsc	INDF,3		; if set increase ones
	incf	ONES_DEM,f
	btfsc	INDF,4		; if set increase ones
	incf	ONES_DEM,f
	btfsc	INDF,5		; if set increase ones
	incf	ONES_DEM,f
	btfsc	INDF,6		; if set increase ones
	incf	ONES_DEM,f
	btfsc	INDF,7		; if set increase ones
	incf	ONES_DEM,f

	decf	TEMP,f		; 16 to 15 to 14 ... down to 1
	btfsc	STATUS,Z	; if zero end
	goto	END_ONES_DEM_TALLY

	decf	FSR,f
	goto	COUNT_ONES_DEM	; continue counting 1's
END_ONES_DEM_TALLY

; if ones tally is zero, set START_LOG,0 to stop logging
; until compressor flag goes high and low for reset
	movf	ONES_DEM,w
	btfss	STATUS,Z
	goto	OUT_START_LOG; not zero
	bsf		START_LOG,0	; set when ones_dem is zero and compressor flag is set
	incf	DUTY15,f	; increase to a 1 so flag is not set once cleared until end of logging period 
; if ones_dem is zero again

OUT_START_LOG

; if override bypass checks
	btfsc	PORTA,7		; over ride LED
	goto	TEST_DRV_SIG
	
; pause compressor drive tally when COOL_FLG,0 is set and it is not L_LOAD
TEST_DRV_FLG
	btfsc	PORTA,6		; if not low load test drive flag
	goto	TEST_DRV_SIG; low load
	btfsc	COOL_FLG,0	; if set pause
	goto	END_ONES_DRV_TALLY; pause logging

; compressor drive signal logged
TEST_DRV_SIG
	bcf		STATUS,C	; clear carry
	btfss	PORTA,0		; compressor drive, if set then set carry
	goto	ROLL_DUTY
; if cooldown log high levels otherwise shift
	btfsc	COOL_FLG,0
	bsf		STATUS,C
; if low throttle and compressor drive on set carry bit 
	btfss	PORTA,6		; set when low throttle
	goto	ROLL_DUTY
	btfsc	PORTA,0		; high when compressor on
	bsf		STATUS,C
		
; the value in each bit records status
;(1=compressor drive on, 0=drv off)
ROLL_DUTY
	rlf		DUTY_D15,f	; ls byte. 
	rlf		DUTY_D14,f	; duty cycle log 
	rlf		DUTY_D13,f	; 
	rlf		DUTY_D12,f	
	rlf		DUTY_D11,f	; duty cycle log 
	rlf		DUTY_D10,f
	rlf		DUTY_D9,f
	rlf		DUTY_D8,f		; 
	rlf		DUTY_D7,f
	rlf		DUTY_D6,f	; duty cycle log 
	rlf		DUTY_D5,f
	rlf		DUTY_D4,f
	rlf		DUTY_D3,f		; 
	rlf		DUTY_D2,f
	rlf		DUTY_D1,f		; 
	rlf		DUTY_D0,f	; ms byte 

	movlw	D'16'		; 16 x 8 = number of bits to add
	movwf	TEMP		; storage

; count number of 1's 
	clrf	ONES_DRV	; 1's count

; count 1's

	movlw	H'4F'		; DUTY_D15 address
	movwf	FSR			; start address
COUNT_ONES_DRV
	btfsc	INDF,0		; if set increase ones
	incf	ONES_DRV,f
	btfsc	INDF,1		; if set increase ones
	incf	ONES_DRV,f
	btfsc	INDF,2		; if set increase ones
	incf	ONES_DRV,f
	btfsc	INDF,3		; if set increase ones
	incf	ONES_DRV,f
	btfsc	INDF,4		; if set increase ones
	incf	ONES_DRV,f
	btfsc	INDF,5		; if set increase ones
	incf	ONES_DRV,f
	btfsc	INDF,6		; if set increase ones
	incf	ONES_DRV,f
	btfsc	INDF,7		; if set increase ones
	incf	ONES_DRV,f

	decf	TEMP,f		; 16 to 15 to 14 ... down to 1
	btfsc	STATUS,Z	; if zero end
	goto	END_ONES_DRV_TALLY

	decf	FSR,f
	goto	COUNT_ONES_DRV	; continue counting 1's
END_ONES_DRV_TALLY

; condensor fan drive
; test compressor drive signal for fan run
; 
TEST_DRV_SIG_FOR_FAN
	bcf		STATUS,C	; clear carry
	btfsc	PORTA,0		; if compressor drive set then set carry
	bsf		STATUS,C
	
; the value in each bit records status
;(1=compressor drive on, 0=drv off)
	rlf		DUTY_F15,f	; ls byte. 
	rlf		DUTY_F14,f	; duty cycle log 
	rlf		DUTY_F13,f	; 
	rlf		DUTY_F12,f	
	rlf		DUTY_F11,f	; duty cycle log 
	rlf		DUTY_F10,f
	rlf		DUTY_F9,f
	rlf		DUTY_F8,f		; 
	rlf		DUTY_F7,f
	rlf		DUTY_F6,f	; duty cycle log 
	rlf		DUTY_F5,f
	rlf		DUTY_F4,f
	rlf		DUTY_F3,f		; 
	rlf		DUTY_F2,f
	rlf		DUTY_F1,f		; 
	rlf		DUTY_F0,f	; ms byte 

; test for zero
	movf	DUTY_F15,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F14,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F13,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F12,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F11,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F10,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F9,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F8,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F7,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F6,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F5,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F4,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F3,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F2,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F1,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on
	movf	DUTY_F0,w
	btfss	STATUS,Z	; if zero continue
	goto	SWITCH_FAN_ON; if not zero switch fan on

; all zero so switch fan off
	bcf		PORTA,1		; ## fan off enable if test off
	goto	RLY2_OUT
SWITCH_FAN_ON
; switch on relay 2 (fan driver relay)(porta,1 high) when there is at least one 1 in ONES_DRV
	bsf		PORTA,1		; ## fan relay on enable if test off	
RLY2_OUT

; run output test for ONES_DRV and ONES_DEM
; test programs to display demand and drive signals on porta,1
;	call	TEST_ONES_DEM; ## disable for test off
;	call	DELAYms		 ; ## disable for test off (gap between signals)
;	call	TEST_ONES_DRV; ## disable for test off
; disable fan drive when running this


; MULTIPLIER. alter compressor demand value by multiplier so compressor drive value must catch up
; multiply ONES_DEM (0 to 127) by MULTIPLIER (0 to 255)/2 + 127
MULT_CALC
	bcf		STATUS,C	; carry 0
	rrf		MULTIPLIER,w; divide by 2
	addlw	D'127'		; add 127
	movwf	BARGB0
	movf	ONES_DEM,w
	movwf	AARGB0
	call	EIGHTEIGHT	; multiply (result in AARGB0,AARGB1)
	movf	AARGB0,w
	movwf	ONES_DEM0	; ms byte	(result in ONES_DEM0,1)
	movf	AARGB1,w
	movwf	ONES_DEM1
; multiply ONES_DRV (0 to 127) by 127
	movlw	D'127'		; 127
	movwf	BARGB0
	movf	ONES_DRV,w
	movwf	AARGB0
	call	EIGHTEIGHT	; multiply (result in AARGB0,AARGB1)

; RESET flag for threshold

	btfss	PORTB_STO,4	; demand flag
	goto	COMPARE_16
	bsf		THRESH_FLG,0; only reset when compressor demand ls bit is a 1 (ie demand flag is set)
	
; transfer the log of flag values to drive values 
; most recent to least recent transfer depends on reset adaptation setting 
; <8 then no transfer, >248 then full transfer

	comf	ADAPT,w		; reverse sense
	sublw	D'248'		; if negative then >248
	btfss	STATUS,C	;
	goto	GREATER_248	; transfer all	
	
	comf	ADAPT,w
	sublw	D'232'		; if negative then >232
	btfss	STATUS,C	;
	goto	GREATER_232	; transfer some	

	comf	ADAPT,w
	sublw	D'216'		; if negative then >216
	btfss	STATUS,C	;
	goto	GREATER_216	; transfer some

	comf	ADAPT,w
	sublw	D'200'		; if negative then >200
	btfss	STATUS,C	;
	goto	GREATER_200	; transfer some	

	comf	ADAPT,w
	sublw	D'184'		; if negative then >184
	btfss	STATUS,C	;
	goto	GREATER_184	; transfer some

	comf	ADAPT,w
	sublw	D'168'		; if negative then >168
	btfss	STATUS,C	;
	goto	GREATER_168	; transfer some	

	comf	ADAPT,w
	sublw	D'152'		; if negative then >152
	btfss	STATUS,C	;
	goto	GREATER_152	; transfer some	

	comf	ADAPT,w
	sublw	D'136'		; if negative then >136
	btfss	STATUS,C	;
	goto	GREATER_136	; transfer some

	comf	ADAPT,w
	sublw	D'120'		; if negative then >120
	btfss	STATUS,C	;
	goto	GREATER_120	; transfer some	

	comf	ADAPT,w
	sublw	D'104'		; if negative then >104
	btfss	STATUS,C	;
	goto	GREATER_104	; transfer some

	comf	ADAPT,w
	sublw	D'88'		; if negative then >88
	btfss	STATUS,C	;
	goto	GREATER_88	; transfer some	

	comf	ADAPT,w
	sublw	D'72'		; if negative then >72
	btfss	STATUS,C	;
	goto	GREATER_72	; transfer some	

	comf	ADAPT,w
	sublw	D'56'		; if negative then >56
	btfss	STATUS,C	;
	goto	GREATER_56	; transfer some	

	comf	ADAPT,w
	sublw	D'40'		; if negative then >40
	btfss	STATUS,C	;
	goto	GREATER_40	; transfer some

	movf	ADAPT,w
	sublw	D'24'		; if negative then >24
	btfss	STATUS,C	;
	goto	GREATER_24	; transfer some

	comf	ADAPT,w
	sublw	D'8'		; if negative then >8
	btfss	STATUS,C	;
	goto	GREATER_8	; transfer some	
	goto	COMPARE_16	; no transfer

GREATER_248	
	movf	DUTY15,w	; demand value most recent
	movwf	DUTY_D15	; drive flag value
GREATER_232	
	movf	DUTY14,w	; duty cycle log 
	movwf	DUTY_D14
GREATER_216	
	movf	DUTY13,w
	movwf	DUTY_D13
GREATER_200	
	movf	DUTY12,w	; duty cycle log 
	movwf	DUTY_D12
GREATER_184		
	movf	DUTY11,w
	movwf	DUTY_D11
GREATER_168	
	movf	DUTY10,w	; duty cycle log 
	movwf	DUTY_D10
GREATER_152	
	movf	DUTY9,w
	movwf	DUTY_D9
GREATER_136	
	movf	DUTY8,w		; 	duty cycle log 
	movwf	DUTY_D8
GREATER_120	
	movf	DUTY7,w
	movwf	DUTY_D7
GREATER_104	
	movf	DUTY6,w		; duty cycle log 
	movwf	DUTY_D6
GREATER_88	
	movf	DUTY5,w
	movwf	DUTY_D5
GREATER_72	
	movf	DUTY4,w		; duty cycle log 
	movwf	DUTY_D4		; duty cycle log 
GREATER_56	
	movf	DUTY3,w
	movwf	DUTY_D3
GREATER_40	
	movf	DUTY2,w		; duty cycle log 
	movwf	DUTY_D2
GREATER_24
	movf	DUTY1,w
	movwf	DUTY_D1
GREATER_8	
	movf	DUTY0,w		; duty cycle log 
	movwf	DUTY_D0		; least recent
	
; 16-bit compare (ie. is ONES_DRV x 127 (AARGB0,1)>ONES_DEM0,1) if so set threshold flag	
COMPARE_16
	movf	AARGB0,w	; ms byte
	subwf	ONES_DEM0,w
	movwf	TEMP		; subtract and store value
	movf	AARGB1,w	; ls byte
	subwf	ONES_DEM1,w
	btfss	STATUS,C
	decf	TEMP,f		; decrease if ls byte subtraction is negative
	btfsc	TEMP,7		; if set then AARGB0,1>ONES_DEM0,1
	bcf		THRESH_FLG,0; flag clear when over threshold	
;
; 	if ones_drive > 120 reset
	movf	ONES_DRV,w
	sublw	D'120'
	btfss	STATUS,C	; if negative
	bcf		THRESH_FLG,0; clear threshold flag

; Switch input
CHECK_EDGE					; Override switch
	btfss	INTCON,INTF		; edge interrupt flag
	goto	RECLAIM
	bcf		INTCON,INTF		; clear edge interrupt flag
 	btfsc	PORTB,0
	goto	RECLAIM

; check if switch stays on for debounce period
	movlw	D'3'
	call	DELAYX
	btfsc	PORTB,0
	goto 	RECLAIM
	movlw	D'3'
	call	DELAYX
	btfsc	PORTB,0
	goto 	RECLAIM

; load timer add 2 minutes every time the switch is pressed to a max of 10-minutes
; set timer	to 2 minutes (240 x 0.5s =120s or 2 mins)

	movlw	D'240'			; 2 minutes
	addwf	TIMERL,f		; timer loaded
	btfss	STATUS,C		; if overrange increase timerH
	incf	TIMERH,f
	movf	TIMERH,w
	sublw	D'04'
	movlw	D'04'
	btfss	STATUS,C
	movwf	TIMERH			; max of 4B0 = 5 x 240	

; end of interrupt reclaim w and status 
RECLAIM

	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

; **************************************************************
; Subroutines

; delay
DELAYms
	movlw	D'20'		; delay 125ms
DELAYX
	movwf	STORE1		; STORE1 is number of loops value
LOOP1	
	movlw	D'2'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP2
	decfsz	STORE2,f
	goto	LOOP2
	decfsz	STORE1,f
	goto	LOOP1		; decrease till STORE1 is zero
	return


; subroutine to read EEPROM memory

EEREAD	
	bcf		INTCON,GIE	; clear global interrupt enable 
	bcf 	STATUS,RP0	; select memory bank 
	bsf		STATUS,RP1	; bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; pointer for data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,W	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	bsf		INTCON,GIE	; set global interrupt enable 
	return

; subroutine to write to EEPROM

EWRITE
	bcf		INTCON,GIE	; clear global interrupt enable 
	bsf	    STATUS,RP1	; select bank 
	bcf 	STATUS,RP0	; select memory bank 2
	movwf	EEDATA		; data register
;	bcf		INTCON,GIE	; disable interrupts
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; pointer for data memory
	bsf		EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
;	bsf		INTCON,GIE	; enable interrupts
	bcf		EECON1,WREN	; clear WREN bit
WRITE	
	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank select
	bcf 	STATUS,RP1	; select memory bank 0 
	bsf		INTCON,GIE	; set global interrupt enable 
	return				; value written 

; subroutine to read EEPROM memory

EEREAD_NO_INTERRUPT	
	bcf 	STATUS,RP0	; select memory bank 
	bsf		STATUS,RP1	; bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; pointer for data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,W	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM

EWRITE_NO_INTERRUPT
	bsf	    STATUS,RP1	; select bank 
	bcf 	STATUS,RP0	; select memory bank 2
	movwf	EEDATA		; data register
	bcf		INTCON,GIE	; disable interrupts
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; pointer for data memory
	bsf		EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITE1	
	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE1		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank select
	bcf 	STATUS,RP1	; select memory bank 0 
	return

; test program to show duty cycle of demand signal on porta,1
TEST_ONES_DEM
; oscilloscope trigger and indicator of length
	bsf		PORTA,1
	nop
	bcf		PORTA,1

	movlw	D'128'
	movwf	WIDTH		; width of tally
	movf	ONES_DEM,w
	movwf	ONES_SHOW	; count value
CYC_ONES
	movf	ONES_SHOW,w
	btfsc	STATUS,Z	; if zero no output	
	goto	DEC_WIDTH			
	bsf		PORTA,1
	decf	ONES_SHOW,f	; decrease
	decfsz	WIDTH,f
	goto	CYC_ONES
	bcf		PORTA,1
	goto	END_WIDTH
DEC_WIDTH
	bcf		PORTA,1		; output low
	movf	WIDTH,w
	btfsc	STATUS,Z	; if zero end
	goto	END_WIDTH
	decfsz	WIDTH,f
	goto	DEC_WIDTH
END_WIDTH
; indicator of length
	bsf		PORTA,1
	nop
	bcf		PORTA,1
	return

; test program to show the compressor drive duty cycle
TEST_ONES_DRV
; oscilloscope trigger and indicator of length
	bsf		PORTA,1
	nop
	bcf		PORTA,1

	movlw	D'128'
	movwf	WIDTH		; width of tally
	movf	ONES_DRV,w
	movwf	ONES_SHOW	; count value
CYC_ONES1
	movf	ONES_SHOW,w
	btfsc	STATUS,Z	; if zero no output	
	goto	DEC_WIDTH1			
	bsf		PORTA,1
	decf	ONES_SHOW,f	; decrease
	decfsz	WIDTH,f
	goto	CYC_ONES1
	bcf		PORTA,1
	goto	END_LENGTH
DEC_WIDTH1
	bcf		PORTA,1		; output low
	movf	WIDTH,w
	btfsc	STATUS,Z	; if zero end
	goto	END_LENGTH
	decfsz	WIDTH,f
	goto	DEC_WIDTH1
END_LENGTH
; indicator of width
	bsf		PORTA,1
	nop
	bcf		PORTA,1
	return

; 8 x 8 multiply

EIGHTEIGHT      CLRF    AARGB1          ; clear partial product
UMUL0808L        
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
                MOVF    AARGB0,W

LOOPUM0808A
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    LUM0808NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808A

                CLRF    AARGB0
                RETLW   H'00'

LUM0808NAP
                BCF     STATUS,C
                GOTO    LUM0808NA

LOOPUM0808
                RRF     BARGB0, F
                BTFSC   STATUS,C
                ADDWF   AARGB0, F
LUM0808NA       RRF    	AARGB0, F
                RRF    	AARGB1, F
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808

                return  

;**********************************************************************************************
        
;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8      	CLRF            REMB0
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B          RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
                return
           

 end
