;============================================================================
;	Program: Timetach.ASM
;============================================================================
;	Programmer: Alexander Radford
;============================================================================
;	Start Date: 15 September 2005
;============================================================================
;	Target Processor: PIC16F84A-20/P
;============================================================================
;	Compiler: MPLAB Version 7.20.00.00
;============================================================================
;	Release version: 1.0
;============================================================================
;	Revisions: NIL
;============================================================================
;	Program Description:
;
; This program is designed to:
;
; a) measure and display the instantaneous RPM of any single-cylinder petrol
;    engine which provides either one or two stimulus pulses per two output 
;    shaft revolutions to the tachometer input. Such engines are referred 
;    to as 4-stroke and 2-stroke respectively.
; b) display the Session Engine Run Time (sum of the time interval(s) engine
;    has been running since tachometer was last switched on)
; c) record and display the Total Engine Run Time (sum of all the Session
;    Engine Run Times including the present one).
;
; All of the above are displayed simultaneously using a Jaycar QP-5515 2-line 
; 16-Character Liquid Crystal Display module or similar.
;
; Engine Speed Range is 472 to 9999 RPM (for both 2 and 4-stroke engines). 
; Engine Speed display resolution is 1 RPM.
; Engine Speed inaccuracy (due to arithmetic rounding errors) should be
; better than +/- 0.24 % of reading. This figure does not include any
; inaccuracies due to the 4.000 MHz crystal or the engine speed sensor.
;
; Displayed run time range is 000 hours 00 minutes to 999 hours 59 minutes  
; (for both Session and Total Run Times).
; Run time resolution is one minute. 
; Run time inaccuracy (due to software) should be better than 0.0064% of
; reading. This figure does not include any inaccuracies due to the
; 4.000 MHz crystal.
;
; Display refresh rate is once per second.    
;============================================================================

	list	p=16f84A	; processor type
	radix	hex			; HEX is default language
	errorlevel	-302	; remove warning messages about bank 1/0
;----------------------------------------------------------------------------

;	CONFIGURATION BITS

	#include 	<P16F84A.inc>
	__CONFIG 	_CP_OFF & _WDT_OFF & _PWRTE_ON & _XT_OSC

; code protection=off, watch dog timer=off,
; power-up timer=on, oscillator=low speed crystal
;----------------------------------------------------------------------------

;PORT A
						
#define STR PORTA,0     ; 2 or 4 Stroke Select. 0 (sw ON) = 4 stroke.
#define RTR PORTA,1     ; Sess/Tot Run Time Reset. 0 (sw ON) = RESET times.
#define RS PORTA,2 		; LCD REGISTER SELECT signal. DISP CHAR=1, INSTR=0.
#define RW PORTA,3 		; LCD READ/WRITE signal. READ=1, WRITE=0.
#define	E PORTA,4		; LCD ENABLE signal. Falling edge of pos going pulse
						; strobes data.

;PORT B
					 
#define	STIM PORTB,0	; pos going pulse received here each time RPM sensor
						; is activated. One pulse must be received for every 
						; 2 (4-stroke) or 1 (2-stroke) engine shaft
						; revolutions. Each pulse initiates an interrupt.	
#define	LCD4 PORTB,4	; LSB of data nibble sent to LCD.
#define	LCD5 PORTB,5	
#define	LCD6 PORTB,6	
#define	LCD7 PORTB,7	; MSB of data nibble sent to LCD. Also Busy flag sent
						; from LCD 	
						
;---------------------------------------------------------------------------

;RAM Definitions

TMR1    equ 0x0C	; multiple (to 16) of TMR0 (32.768 ms) for .524288 s T/O.           
TMRT	equ 0x0D	; temp storage of latest value of TMR0 (units 128 us).
WTEMP   equ 0x0E	; temporary storage for W register during interrupt.
STEMP   equ 0x0F	; temporary storage for STATUS register during interrupt.
LCDTEMP equ 0x10    ; temporary storage for LCD data
N300U   equ 0x11    ; defines the multiple of 300 us used to generate delay.
N100M	equ 0x12    ; defines the multiple of 100 ms used to generate delay.
DLYCNT  equ 0x13	; counter used only in 300 us delay routine.
DLYCNT1 equ 0x14	; counter used in 100 ms delay routine and SUB Ewrite.
DLYCNT2 equ 0x15	; counter used in 100 ms delay routine and SUB Ewrite.
MINCNT  equ 0x16    ; counter indicates 1 m has elapsed for EEPROM refresh.   
STMCNT	equ 0x17    ; counter for number of stimulus pulses from sensor.
STMINT  equ 0x18    ; number of stimulus periods in Phase 1 (STMCNT-1). 
INDEX   equ 0x19    ; pointer to character in LCD message strings.
CONSLSB equ 0x1A    ; constant dividend/divisor LS byte. Also flag, storage.
CONSMID equ 0x1B    ; constant dividend/divisor middle byte. Also flag.
CONSMSB equ 0x1C    ; constant dividend/divisor MS byte.
PRODL   equ 0x1D    ; RPM product least significant byte.
PRODM   equ 0x1E    ; RPM product most significant byte.
QUOTL   equ 0x1F	; RPM quotient least significant byte.
QUOTM   equ 0x20	; RPM quotient most significant byte.
TMRTSL  equ 0x21    ; timer tacho start least significant byte (128 us).
TMRTSM  equ 0x22	; timer tacho start most significant byte (32.768 ms).
TMRTFL  equ 0x23    ; timer tacho finish least significant byte (128 us).
TMRTFM  equ 0x24	; timer tacho finish most significant byte (32.768 ms).
TMRTL   equ 0x25    ; timer tacho sample int least significant byte (128 us).
TMRTM   equ 0x26  	; timer tacho sample int most sig byte (32.768 ms).
RPMM    equ 0x27    ; final storage of the RPM thousands BCD.
RPMC    equ 0x28    ; final storage of the RPM hundreds BCD.
RPMX    equ 0x29    ; final storage of the RPM tens BCD.
RPMI    equ 0x2A    ; final storage of the RPM units BCD.
THC		equ 0x2B	; working storage in subroutine for the session and
THX		equ 0x2C	; total time BCDs STHC, STHX, STHI, STMX, STMI and 
THI		equ 0x2D	; TTHC, TTHX, TTHI, TTMX, TTMI.
TMX		equ 0x2E
TMI		equ 0x2F
STHC    equ 0x30    ; final storage of the session time hours hundreds BCD.
STHX    equ 0x31    ; final storage of the session time hours tens BCD.
STHI    equ 0x32    ; final storage of the session time hours units BCD.
STMX    equ 0x33    ; final storage of the session time minutes tens BCD.
STMI    equ 0x34    ; final storage of the session time minutes units BCD.
TTHC    equ 0x35    ; final storage of the total time hours hundreds BCD.
TTHX    equ 0x36    ; final storage of the total time hours tens BCD.
TTHI    equ 0x37    ; final storage of the total time hours units BCD.
TTMX    equ 0x38    ; final storage of the total time minutes tens BCD.
TTMI    equ 0x39    ; final storage of the total time minutes units BCD.

TMSB    equ 0x3A	; working storage in subroutine for the variables
TLSB	equ 0x3B	; STMSB, STLSB and TTMSB, TTLSB.   
STMSB	equ 0x3C    ; session time most significant byte (256 minutes).
STLSB   equ 0x3D    ; session time least significant byte (minutes).
				    ; the total time most significant byte (256 minutes) 
				    ; and the total time least significant byte (minutes)
				    ; are stored in EEPROM addresses 0x00 to 0x01
					; respectively.
	

;----------------------------------------------------------------------------

;EEPROM Definitions

EETLSB 	equ 0x01	; address of binary total time least significant byte 
					; (minutes).

;----------------------------------------------------------------------------

; General Definitions

SP	equ	0x20	; space


;*****************************  MAIN PROGRAM  *******************************

	org	0x000
	goto Powerup		; skip over interrupt vector 
	
	
;========================  INTERRUPT SERVICE ROUTINE  =======================	
; This routine handles both of the two possible interrupts:
;
; RB0 interrupt enabled only in the first (measurement) phase of operation.
; TMR0 interrupt enabled only in the second (display) phase of operation.

	
	org	0x004			; interrupt service routine starts here.
		movwf WTEMP		; store W register in WTEMP.
		swapf STATUS,W 	; move STATUS register into W register
		movwf STEMP 	; and then store it in STEMP.
		btfss INTCON,T0IE ; test for timeout or RB0 interrupt.
		goto Intrb0     ; branch to deal with a RB0 interrupt. 
		incf TMR1,F		; increment TMR1.
		bcf INTCON,T0IF	; clear TMR0 Interrupt flag bit. 
		goto Endint		

Intrb0	movf STMCNT,F   ; test if this is the first stimulus pulse to be 
		btfss STATUS,Z	; received in this particular measurement phase	
		goto Nextstim	; (0.508416 second interval).
		incf STMCNT,F
		movlw d'2'	    ; CONSMID is convenient register to store count
		movwf CONSMID	; to select which stimuli are recorded.
		movf TMRT,W	    ; record time at which first stimulus pulse has 
		movwf TMRTSL	; been received in this particular measurement	
		movf TMR1,W		; phase (0.508416 second interval). 
		movwf TMRTSM
		goto Skipstim

Nextstim movf CONSLSB,F ; CONSLSB was set to 1 if engine is 4-stroke,
		btfss STATUS,Z	;                    0 if engine is 2-stroke. 
		decf CONSMID,F  ; CONSMID was set to 2 above when first stimulus 
		decfsz CONSMID,F; received. For 2 stroke engines, only one in two
		goto Skipstim 	; stimuli pulses should be counted and timed and the
		movlw d'2'		; other one in two is simply ignored. For 4-stroke
		movwf CONSMID 	; engines every stimulus pulse should be counted and
		movf TMRT,W		; timed.
		movwf TMRTFL 	; re-load counter CONSMID with decimal 2.
	 	movf TMR1,W		; record time at which latest stimulus pulse has 
		movwf TMRTFM 	; been received in this particular measurement	
		incf STMCNT,F 	; phase (0.5 second interval). 
		btfsc STATUS,Z	; increment count of stimulus pulses in this
		decf STMCNT,F 	; particular measurement phase (500 ms interval).
Skipstim bcf INTCON,INTF; stops STMCNT increasing beyond d'255' and
						; causing arithmetic problems later.
						; clear RB0 Interrupt flag bit. 

Endint	swapf STEMP,W	; now restore W and STATUS registers to 
		movwf STATUS	; the state they were in before entering
		swapf WTEMP,F	; this interrupt routine.
		swapf WTEMP,W 
		
		retfie 			; status of GIE bit is handled automatically.
		
		
		
;========================  START OF MESSAGE TABLES  =========================
;Note: 	Tables all located in the same (first) 256 block of program memory to
;		avoid possible problem caused by the addwf command. 


		
Lcd_m1  addwf	PCL,F	; add offset POINTER to Program Counter (PCL).
		retlw   '1'
		retlw	SP
		retlw	'C'
		retlw	'Y'
		retlw	'L'
		retlw	SP
		retlw   '4'
		retlw   '/'
		retlw	'2'
		retlw 	SP 
		retlw	'S'
		retlw	'T'
		retlw	'R'
		retlw	'O'
		retlw	'K'
		retlw	'E'		; this is the 16th character (max 16).
		retlw	0x00		
		
Lcd_m2  addwf	PCL,F	; add offset POINTER to Program Counter (PCL).
		retlw	SP
		retlw   SP
		retlw	SP
		retlw	'T'
		retlw 	'A'
		retlw	'C'
		retlw	'H'
		retlw	'O'		
		retlw	'M'	
		retlw	'E'
		retlw	'T'
		retlw   'E' 
		retlw	'R'		; this is the 13th character (max 16).
		retlw	0x00

Lcd_m3  addwf	PCL,F	; add offset POINTER to Program Counter (PCL).
		retlw	'S'
		retlw	'/'
		retlw   'T'
		retlw   SP
		retlw	'='
		retlw   SP
		retlw	'S'
		retlw	'e'
		retlw	's'
		retlw	's'
		retlw	'/'
		retlw	'T'
		retlw	'o'
		retlw	't'
		retlw	'a'
		retlw	'l'		; this is the 16th character (max 16).
		retlw	0x00

Lcd_m4  addwf	PCL,F	; add offset POINTER to Program Counter (PCL).			
		retlw	'E'
		retlw 	'n'
		retlw	'g'
		retlw   'i'
		retlw	'n'
		retlw	'e'		
		retlw	SP	
		retlw	'R'
		retlw	'u'
		retlw   'n' 
		retlw	SP	
		retlw	'T'
		retlw	'i'		
		retlw	'm'	
		retlw	'e'		; this is the 15th character (max 16).
		retlw	0x00

Lcd_m5  addwf	PCL,F	; add offset POINTER to Program Counter (PCL).
		retlw	'A'
		retlw   'w'
		retlw	'a'
		retlw	'i'
		retlw	't'
		retlw	'i'
		retlw	'n'
		retlw	'g'
		retlw 	SP 
		retlw	'S'
		retlw	'i'
		retlw	'g'
		retlw	'n'
		retlw	'a'
		retlw	'l'		; this is the 15th character (max 16).
		retlw	0x00
		
Lcd_m6  addwf	PCL,F	; add offset POINTER to Program Counter (PCL).
		retlw	'S'
		retlw   'p'
		retlw	'e'
		retlw	'e'
		retlw	'd'
		retlw	':'
		retlw	SP
		retlw	'0'		; appropriate BCDs are added to all zeros later.
		retlw 	'0' 
		retlw	'0'
		retlw	'0'
		retlw	SP
		retlw	'R'
		retlw	'P'
		retlw	'M'		; this is the 15th character (max 16).
		retlw	0x00

Lcd_m7  addwf	PCL,F	; add offset POINTER to Program Counter (PCL).
		retlw	'S'
		retlw	'0'		; appropriate BCDs are added to all zeros later.
		retlw	'0'
		retlw	'0'
		retlw	':'
		retlw	'0'
		retlw 	'0' 
		retlw   SP
		retlw   SP
		retlw	'T'
		retlw	'0'
		retlw	'0'
		retlw	'0'
		retlw	':'	
		retlw   '0'
		retlw   '0'		; this is the 16th character (max 16).
		retlw	0x00



;===========================  POWER-UP ROUTINE  =============================

Powerup	clrf STATUS		; ensure we are in bank 0.
	movlw b'00000000'	; inhibit and disable all interrupts for now.
	movwf INTCON		; note GIE (Bit 7) is cleared.
	clrf PORTA			; clear port A latch.
	clrf PORTB			; clear port B latch.
	bsf	STATUS,RP0		; go to register bank 1.
	movlw b'00000011'	; set RA0, RA1 as inputs (used to set 1/2 stroke and
	movwf TRISA			; reset run times) and RA2, RA3, RA4 as outputs. 
	movlw b'00001111'	; set RB0-RB3 as inputs. RB1-RB3 are unused.
	movwf TRISB   	    ; set RB4-RB7 as outputs (data lines for LCD).
	movlw b'01010110'	; port B pullups enabled, rising edge RB0 int, 
	movwf OPTION_REG	; Tmr0 int clk, prescaler 1:128 and assigned to TMR0.
	bcf	STATUS,RP0		; go back to register bank 0.
	
	movlw 0x0B			; this routine clears all 68 SRAM registers from
	movwf FSR			; Address 0C to Address 4F using indirect addressing.
Nextram	incf FSR,F		
	clrf INDF			
	movlw 0x4F
	subwf FSR,W
	btfss STATUS,Z
	goto Nextram      	

;===========================  LCD INIT ROUTINE  =============================	

	movlw 0x30
	movwf PORTB
	movlw d'50'
	call Wait300u	 	; wait 15 ms after power-up for LCD
	call Lcd_en
	movlw d'16'
	call Wait300u	 	; wait 4.8 ms. 
	call Lcd_en			
	movlw d'1'
	call Wait300u	 	; wait 300 us.
	call Lcd_en
	movlw 0x20			; set LCD data interface to 4 bits (not 8).	
	movwf PORTB			
	movlw d'1'			; wait 300 us. 
	call Wait300u	 	; set LCD to two display lines.
	call Lcd_en			; set LCD to 5 X 7 character font. 
	movlw 0x28			; 0x28 and all subsequent WRITE operations to LCD 
	call Lcd_cmd		; require data to be sent as two separate nibbles.
	movlw 0x08			; display OFF, cursor OFF, cursor blink OFF.
	call Lcd_cmd
	movlw 0x10			; prevent shifting of display.
	call Lcd_cmd
	movlw 0x01			; clear the display and return cursor to top left.
	call Lcd_cmd
   	movlw 0x06			; increment cursor after each byte written to LCD.		
	call Lcd_cmd		; 0x06 is the LAST OF MANUFACTURER'S RECOMMENDATIONS. 
	movlw 0x0C			; display ON; cursor OFF; cursor blink OFF.
	call Lcd_cmd
	
			
;=======================  DISPLAY 2 INTRO MESSAGES  =========================
	
	movlw d'0'			; send first line of first introductory message to 
	movwf INDEX			; display and hold message for two seconds before 
M1_loop	call Lcd_m1		; clearing display.
	andlw 0xFF
	btfsc STATUS, Z
	goto M1_end
	call Lcd_chr
	incf INDEX,F
	movf INDEX,W
	goto M1_loop

M1_end movlw 0xC0 		; move cursor to second line of LCD.
	call Lcd_cmd	
	movlw d'0'			; send second line of first introductory message to 
	movwf INDEX			; display and hold message for two seconds before 
M2_loop	call Lcd_m2		; clearing display.
	andlw 0xFF
	btfsc STATUS, Z
	goto M2_end
	call Lcd_chr
	incf INDEX,F
	movf INDEX,W
	goto M2_loop

M2_end	movlw d'20'		; wait for 2 seconds.
	call Wait100m
	movlw 0x01			; clear the display and return cursor to top left.
	call Lcd_cmd
	movlw d'0'			; send first line of second introductory message to 
	movwf INDEX			; display and hold message for two seconds.
M3_loop	call Lcd_m3
	andlw 0xFF
	btfsc STATUS, Z
	goto M3_end
	call Lcd_chr
	incf INDEX,F
	movf INDEX,W
	goto M3_loop

M3_end movlw 0xC0 		; move cursor to second line of LCD.
	call Lcd_cmd	
	movlw d'0'			; send second line of second introductory message to 
	movwf INDEX			; display and hold message for two seconds. 
M4_loop	call Lcd_m4		
	andlw 0xFF
	btfsc STATUS, Z
	goto M4_end
	call Lcd_chr
	incf INDEX,F
	movf INDEX,W
	goto M4_loop

M4_end	movlw d'20'		; wait for 2 seconds.
	call Wait100m
	
	
;=======================  MEASUREMENT (FIRST) PHASE  ========================


	movlw d'60'
	movwf MINCNT    ; initialise run time counter for 1 minute resolution.	
 	btfsc RTR		; POLL RTR (Port A Bit 1) DIL SWITCH FOR POSSIBLE RESET.
	goto P1_start	; no reset desired.
	movlw EETLSB	
	movwf EEADR		; clear Total Time Count in EEPROM if RTR = 0.
	clrf EEDATA		; the Total Time counters, TTLSB and TTMSB are
	call Ewrite		; stored in EEPROM addresses 0x01 and 0x00 respectively.
	decf EEADR,F	; load EEPROM address of TTLSB byte.
	call Ewrite		; store cleared TTLSB and TTMSB in EEPROM.

P1_start movlw d'124'; ensures that each cycle lasts very close to 1000 ms.
	movwf TMR0		; 124x128 us = 15.872 ms and 31x32.768 ms = 1015.808 ms.
	bcf INTCON,T0IF	; See Line P2_loop below. (Error is 64 us or 0.0064%).
	clrf TMR1		; clear any prior setting of T0IF flag
	clrf CONSLSB 	; CONSLSB is convenient storage register for flag.
	btfss STR		; POLL STR (Port A Bit 0) DIL SWITCH FOR 4/2 STROKE.
	incf CONSLSB,F  ; CONSLSB set to 1 if DIL switch is CLOSED and 4-stroke 
	clrf STMCNT		; operation is desired.
	bcf INTCON,INTF ; clear any prior setting of the RB0/INT flag bit.  
	bsf INTCON,INTE	; in Phase 1, enable the RB0/INT interrupt only. 
	
P1_loop bcf INTCON,GIE
    nop
	btfss INTCON,T0IF	; it is important that TMRT and TMR1 are updated with 
	goto P1_lsb			; the latest time together and that no interrupts can 	
	incf TMR1,F			; occur without TMRT and TMR1 having BOTH been 
	bcf INTCON, T0IF	; modified. Therefore disable interrupts (using the
P1_lsb	movf TMR0,W		; bcf INTCON,GIE command) before executing these	
	movwf TMRT			; instructions as part of the P1_loop.	
	bsf INTCON,GIE	
	movlw d'16'		; 16 x 32.768 ms - 124 x 128 us = 508.416 ms 
	subwf TMR1,W	; as soon as TMR1 reaches d'16', 508.416 ms has elapsed
	btfss STATUS,C	; and it is time to enter the Display Phase 2 for the
	goto P1_loop	; remainder of the 1000 ms cycle.	
	bcf INTCON,INTE	; in Phase 2, disable the RB0/INT interrupt. 
	nop				; these nop instructions ensure that any interrupt				
	nop				; occurring in Phase 1 will be treated as RB0/INT.

;=========================  DISPLAY (SECOND) PHASE  =========================
	
	bsf INTCON,T0IE	; in Phase 2, enable the TMR0 interrupt. 
	movf STMCNT,F	; we have about 32 ms until the first interrupt occurs
    btfsc STATUS,Z	; and this is plenty of time for two EEPROM writes.
	goto Timestart	; see note below line Skiprt below.
	decfsz MINCNT,F ; initial value of MINCNT is d'60'. 
	goto Skiprt  	; only need to increment session and total run time 
	movlw d'60'		; counters in RAM and EEPROM memory every 60 seconds.
	movwf MINCNT
	incf STLSB,F	; increment Session Time Count (minutes).
	btfsc STATUS,Z	; increment most significant byte if necessary.
	incf STMSB,F    
	movlw EETLSB	; initialise EEPROM address to 0x01 (LSB).
	movwf EEADR		; get Total Time Count from EEPROM, increment it
	call Eread		; by one (minute) and store the new count.
	incf EEDATA,F	; the Total Time counters, TTLSB and TTMSB are
	call Ewrite		; stored in EEPROM addresses 0x01 and 0x00
	btfss STATUS,Z	; respectively.	
	goto Skiprt		; STATUS register is not affected by SUB Ewrite.	
	decf EEADR,F	; If least significant counter has rolled over then
	call Eread		; increment most significant counter byte.
	incf EEDATA,F	
	call Ewrite
Skiprt	decf STMCNT,W	
	movwf STMINT
	movwf STMCNT 	; if STMCNT is d'0' here then tacho sensor has not
	btfsc STATUS,Z	; received enough (at least 2) pulses during the previous
	goto Timestart	; 508.416 ms measurement phase to calculate RPM. In this
	movf TMRTSL,W	; case display "Awaiting Signal" message on top line of
	subwf TMRTFL,W	; LCD. CALCULATIONS BEGIN HERE.
	movwf TMRTL
	btfss STATUS,C
	incf TMRTSM,F
	movf TMRTSM,W
    subwf TMRTFM,W
	movwf TMRTM
	movf TMRTL,F    ; check that TMRTL and TMRTM are not both zero to avoid
	btfss STATUS,Z	; arithmetic problems later. If TMRTL and TMRTM are both
	goto Quot_start	; zero then we cannot calculate RPM and we must display
	movf TMRTM,F	; "Awaiting Signal" message on top line of LCD later.
	btfss STATUS,Z	
	goto Quot_start
	clrf STMCNT		
	goto Timestart

	
Quot_start	clrf QUOTL  ; Engine RPM = 60 X 10^6 X 2 X (STMCNT-1) / TMRT (us)
	clrf QUOTM
	movlw 0x1C			;			 = 937,500 X (STMCNT-1) / TMRT (128 us)
	movwf CONSLSB
	movlw 0x4E			;            = (0x0E4E1C / TMRT (128 us)) X STMINT 
	movwf CONSMID
	movlw 0x0E			; where STMINT = STMCNT-1	
	movwf CONSMSB		; and TMRT is expressed in units of 128 us.							
						 						
Quot_loop movf TMRTL,W	; TMRTL (LSB) and TMRTM (MSB) are a binary count of
	subwf CONSLSB,F		; the time interval (in units of 128 us) between when 
	btfsc STATUS,C		; the first and last stimulus pulse was detected 
	goto Quot_b2		; during the previous 508.416 ms measurement phase.
	movlw d'1'
	subwf CONSMID,F		; The above equation is used for both 4- and 2-stroke
	btfsc STATUS,C		; engines. In the case of 4- and 2-stroke engines,
	goto Quot_b2		; none and one respectively of every 2 incoming 
	movlw d'1'			; pulses is ignored in the interrupt handling routine 
	subwf CONSMSB,F		; which logs them. This technique improves the 
	btfsc STATUS,C		; accuracy of the quotient 937,500/TMRT (128 us) due 
	goto Quot_b2		; to the effective doubling of the numerator. As it 
	movf TMRTM,W		; stands, this quotient cannot be less than 236 
	subwf CONSMID,F		; (accuracy better than +/- 0.212% with rounding to 
	goto Quot_rnd		; the nearest whole number).
						
Quot_b2 movf TMRTM,W	 
	subwf CONSMID,F		
	btfsc STATUS,C
	goto Quot_inc
    movlw d'1'
	subwf CONSMSB,F
	btfss STATUS,C
	goto Quot_rnd

Quot_inc incf QUOTL,F
	btfsc STATUS,Z
	incf QUOTM,F
	goto Quot_loop
  
Quot_rnd bcf STATUS,C	; now round the result QUOTL/QUOTM to the nearest 
	rrf TMRTM,F			; unit. First divide binary TMRTL/TMRTM by 2 by
	rrf TMRTL,F			; rotating all 16 bits right through the carry bit.
	movf CONSLSB,W      ; Then, if adding half of the original TMRTL/TMRTM 
	addwf TMRTL,W		; to the (negative) number now stored in 
	btfsc STATUS,C		; CONSLSB/CONSMID results in a carry (zero or 
	incf CONSMID,F		; positive number) then round up the present quotient 
	movf CONSMID,W		; QUOTL/QUOTM by 1 to form the final quotient. This 
	addwf TMRTM,W 		; of course improves the accuracy of the RPM reading 
	btfss STATUS,C		; by a factor of 2.
	goto Prod_start
	incf QUOTL,F
	btfsc STATUS,Z
	incf QUOTM,F

Prod_start clrf PRODL	; now multiply the final QUOTL/QUOTM by STMINT to 
	clrf PRODM			; form a final binary number PRODL/PRODM which
						; represents the engine speed in RPM.
Prod_loop movf QUOTL,W
	addwf PRODL,F
	btfsc STATUS,C
	incf PRODM,F
	movf QUOTM,W
	addwf PRODM,F
	decfsz STMINT,F
    goto Prod_loop

 	movlw 0x0F		; if calculated RPM >= 9999 (0x270F) then there is no
	subwf PRODL,W	; need to calculate bcds RPMM, RPMC, RPMX and RPMI.
	movlw 0x27		; in this case we simply set all these bcds to d'9' 
	btfss STATUS,C	; and display '9999'.
	addlw d'1' 
	subwf PRODM,W
	btfsc STATUS,C
	goto Set9999
	
    clrf RPMM
	clrf RPMC
	clrf RPMX
	clrf RPMI

Looprm movlw 0xE8   ; evaluate the number of times d'1000'(0x03E8) will go
	subwf PRODL,W	; into the binary count of RPM contained in the 
	movlw 0x03		; registers PRODL and PRODM.
	btfss STATUS,C
	addlw d'1'
	subwf PRODM,W
	btfss STATUS,C
	goto Looprc
	incf RPMM,F
	movlw 0xE8
	subwf PRODL,F
	movlw 0x03
	btfss STATUS,C
	addlw d'1'
	subwf PRODM,F
	goto Looprm

Looprc movlw 0x64   ; evaluate the number of times d'100'(0x0064) will go
	subwf PRODL,W	; into the binary count of RPM contained in the 
	movlw 0x00		; registers PRODL and PRODM.
	btfss STATUS,C
	addlw d'1'
	subwf PRODM,W
	btfss STATUS,C
	goto Looprx
	incf RPMC,F
	movlw 0x64
	subwf PRODL,F
	movlw 0x00
	btfss STATUS,C
	addlw d'1'
	subwf PRODM,F
	goto Looprc 

Looprx movlw 0x0A   ; evaluate the number of times d'10'(0x000A) will go
	subwf PRODL,W	; into the binary count of RPM contained in the 
	movlw 0x00		; registers PRODL and PRODM.
	btfss STATUS,C
	addlw d'1'
	subwf PRODM,W
	btfss STATUS,C
	goto Loopri
	incf RPMX,F
	movlw 0x0A
	subwf PRODL,F
	movlw 0x00
	btfss STATUS,C
	addlw d'1'
	subwf PRODM,F
	goto Looprx 

Loopri movf PRODL,W		; store remaining (0-9) RPM in the bcd RPMI. 
	movwf RPMI
	goto Timestart
	
Set9999 movlw d'9'
	movwf RPMM
	movwf RPMC
	movwf RPMX
	movwf RPMI

Timestart movf STLSB,W
	movwf TLSB
	movf STMSB,W
	movwf TMSB
	call Getbcds
	movf THC,W
	movwf STHC
	movf THX,W
	movwf STHX
	movf THI,W
	movwf STHI
	movf TMX,W
	movwf STMX
	movf TMI,W
	movwf STMI

	movlw EETLSB	; initialise EEPROM address to 0x01 (LSB).
	movwf EEADR		; get Total Time Count (in minutes) from EEPROM and load
	call Eread		; it into the the temporary storage variables TLSB and
	movf EEDATA,W	; TMSB. TTLSB and TTMSB are permanently stored in 
	movwf TLSB		; EEPROM addresses 0x01 and 0x00 respectively.
	decf EEADR,F	
	call Eread		
	movf EEDATA,W	
	movwf TMSB
	call Getbcds
	movf THC,W
	movwf TTHC
	movf THX,W
	movwf TTHX
	movf THI,W
	movwf TTHI
	movf TMX,W
	movwf TTMX
	movf TMI,W
	movwf TTMI

	
	movlw 0x01		; clear the display and return cursor to top left.
	call Lcd_cmd	
 	movf STMCNT,F	; STMCNT=0 if no tacho signal was received from sensor
	btfss STATUS,Z	; during previous 508.416 ms measurement phase.
	goto Disp_rpm
	movlw d'0'		; send 'Awaiting Signal' message  to top line of display
	movwf INDEX		; while no tacho signal is being received from sensor.
Nosig_loop	call Lcd_m5	
	andlw 0xFF
	btfsc STATUS, Z
	goto Disp_times
	call Lcd_chr
	incf INDEX,F
	movf INDEX,W
	goto Nosig_loop

Disp_rpm	movlw '0' ; the literal for the ascii numeral zero. 
	movwf CONSLSB	; CONSLSB is convenient storage register for literal '0'
	movlw 0x27		; (zero). Initialise pointer to RAM locations which hold
	movwf FSR 		; BCD numbers (d'0-9') to be added to 0 in this message. 
	movlw d'0'		; send RPM count to top line of display and hold it
	movwf INDEX		; for one second until next update.
RPM_loop	call Lcd_m6	
	andlw 0xFF
	btfsc STATUS, Z
	goto Disp_times
	xorwf CONSLSB,F 
	btfss STATUS,Z	
	goto RPM_pos
	addwf INDF,W
	incf FSR,F
RPM_pos 	call Lcd_chr
	movlw '0'
	movwf CONSLSB
	incf INDEX,F
	movf INDEX,W
	goto RPM_loop

Disp_times movlw 0xC0 ; move cursor to second line of LCD.
	call Lcd_cmd
	movlw '0'		; the literal for the ascii numeral zero. 
	movwf CONSLSB	; CONSLSB is convenient storage register for literal '0'
	movlw 0x30		; (zero). Initialise pointer to RAM locations which hold
	movwf FSR 		; BCD numbers (d'0-9') to be added to 0 in this message. 
	movlw d'0'		; send run times to second line of display and hold it
	movwf INDEX		; for one second until next update.
Times_loop	call Lcd_m7	
	andlw 0xFF
	btfsc STATUS, Z
	goto P2_loop
	xorwf CONSLSB,F 
	btfss STATUS,Z	
	goto Times_pos	
	addwf INDF,W
	incf FSR,F
Times_pos	call Lcd_chr
	movlw '0'
	movwf CONSLSB
	incf INDEX,F
	movf INDEX,W
	goto Times_loop

P2_loop	movlw d'31'	; TMR1 is incremented in interrupt service routine.
	subwf TMR1,W	; as soon as TMR1 reaches d'31', 1000 ms has elapsed and
	btfss STATUS,C	; it is time to return to the Measurement Phase 1. 
	goto P2_loop		
	bcf INTCON,T0IE	; in subsequent Phase 1, disable the TMR0 interrupt. 
	goto P1_start	; Each complete cycle should take very close to 1 second
					; and is repeated continually until the unit is switched  
					; off. The display should be updated once every second.

;============================================================================
;	9 SUBROUTINES ONLY BELOW
;============================================================================	
	  
		 
		 	
;===========================  START Wait300u  ===============================
; Total delay = 300 us x W (>N300U) (actually about 4 us longer than 300 us).
; W must be loaded before calling this routine.
;----------------------------------------------------------------------------

Wait300u 	movwf N300U
Rept300u	movlw d'100'		; 300 us loop.
			movwf DLYCNT
Loop3u		decfsz DLYCNT,F	
			goto Loop3u
			decfsz N300U,F		; done enough (N300U) 300 us loops yet?
			goto Rept300u		; not yet.	
			return


;===========================  START Wait100m  ===============================
; Total delay = 100 ms x W (>N100M) (actually about 4 us longer than 100 ms).
; W must be loaded before calling this routine.
;----------------------------------------------------------------------------

Wait100m 	movwf N100M
Rept100m	movlw d'250'		; 250 x 400 us = 100 ms
			movwf DLYCNT1
Loop400u	movlw d'132'		; 400 us loop to crystal tolerance.	
			movwf DLYCNT2
Loop3m		decfsz DLYCNT2,F
			goto Loop3m			
			decfsz DLYCNT1,F	; done enough (250) 400 us loops yet?
			goto Loop400u
			decfsz N100M,F		; done enough (N100M) 100 ms loops yet?	
			goto Rept100m
			return	

	
;============================  START Ewrite  ================================
; Writes a single byte already placed in register EEDATA to the EEPROM memory 
; at the address (0x00 to 0x3F) already placed in register EEADR.
;----------------------------------------------------------------------------
Ewrite		bsf STATUS,RP0		; go to register bank 1.				
			bcf EECON1,EEIF		; clear flag after any previous operations.
			bcf INTCON, GIE     ; disable interrupts during EEPROM write. 
			bsf EECON1,WREN		; setting this bit enables erase/write
			movlw 0x55			; cycles.
			movwf EECON2
			movlw 0xAA			; refer to data sheet.
			movwf EECON2
			bsf EECON1,WR		; initiate EEPROM erase/write cycle.
			bcf EECON1,WREN		; preliminary data sheet indicates that at
			bsf INTCON,GIE		; Vdd=5V, this cycle takes about 5.5 ms. 
			movlw d'200'		; enable interrupts after EEPROM write.
			movwf DLYCNT1
Ewrlp1		movlw d'200'		; 200 x 5 us = 1 ms loop
			movwf DLYCNT2
Ewrlp2		btfsc EECON1,EEIF	; 5 us loop	
			goto Ewrend
			decfsz DLYCNT2,F	; allow maximum time of 200 ms for the flag
			goto Ewrlp2			; EEIF to be set. 
			decfsz DLYCNT1,F		
			goto Ewrlp1			
Ewrend      bcf STATUS,RP0		; go to register bank 0.						 	
			return
			
	
;============================  START Eread  =================================
; Reads a single byte to register EEDATA from the EEPROM memory at the
; address (0x00 to 0x3F) already placed in register EEADR.
;----------------------------------------------------------------------------	
Eread		bsf STATUS,RP0		; go to register bank 1.
			bsf EECON1,RD		; initiate	read cycle.
			bcf STATUS,RP0		; go to register bank 0.
			return				; data is available in register EEDATA
								; at the start of the instruction cycle			
								; immediately following bsf EECON1,RD.


;============================  START Getbcds  ===============================
; Takes a 16-bit non-negative binary integer (stored in two RAM locations
; TMSB and TLSB) which represents a number of minutes and converts it into
; 5 BCD digits THC, THX, THI, TMX and TMI each of which is stored in a single
; RAM location. The BCDs represent a run time (session or total) in hours and 
; minutes expressed as a decimal. 99959 is returned should the binary number
; be equal to or greater than 0xEA5F (d'59,999') which is the greatest number
; of minutes that can be meaningfully displayed.  
;----------------------------------------------------------------------------
Getbcds		clrf THC		; clear all five temporary storage registers for
			clrf THX		; BCDs.
			clrf THI
			clrf TMX
			clrf TMI
			movlw 0x5F		; first verify that the number of minutes 
			subwf TLSB,W	; contained in TMSB and TLSB is less than 0xEA5F. 
			movlw 0xEA		; (d'59,999) which is 999 hours 59 minutes. This 
			btfss STATUS,C	; is the maximum time which can be meaningfully
			addlw d'1'		; displayed. If the binary number contained in 
			subwf TMSB,W	; TMSB and TLSB >= 0xEA5F then bypass the
			btfsc STATUS,C	; calculations and simply set THC, THX, THI, TMX 
			goto St99959	; and TMI to 9, 9, 9, 5, 9 respectively.
	
Loopthc 	movlw 0x70  	; there are 6000 (0x1770) minutes in 100 hours.
			subwf TLSB,W	; this loop evaluates the number for the bcd THC.
			movlw 0x17
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,W
			btfss STATUS,C
			goto Loopthx
			incf THC,F
			movlw 0x70
			subwf TLSB,F
			movlw 0x17
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,F
			goto Loopthc

Loopthx		movlw 0x58 		; there are 600 (0x0258) minutes in 10 hours.
			subwf TLSB,W	; this loop evaluates the number for the bcd THX.		
			movlw 0x02
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,W
			btfss STATUS,C
			goto Loopthi
			incf THX,F
			movlw 0x58
			subwf TLSB,F
			movlw 0x02
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,F
			goto Loopthx

Loopthi 	movlw 0x3C 		; there are 60 (0x003C) minutes in 1 hour.
			subwf TLSB,W	; this loop evaluates the number for the bcd THI.		
			movlw 0x00
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,W
			btfss STATUS,C
			goto Looptmx
			incf THI,F
			movlw 0x3C
			subwf TLSB,F
			movlw 0x00
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,F
			goto Loopthi

Looptmx		movlw 0x0A	 	; there are 10 (0x000A) minutes/10 min interval.
			subwf TLSB,W	; this loop evaluates the number for the bcd TMX.		
			movlw 0x00
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,W
			btfss STATUS,C
			goto Looptmi
			incf TMX,F
			movlw 0x0A
			subwf TLSB,F
			movlw 0x00
			btfss STATUS,C
			addlw d'1'
			subwf TMSB,F
			goto Looptmx

Looptmi 	movf TLSB,W		; store remaining (0-9) minutes in the bcd TMI. 
			movwf TMI
			return

St99959		movlw d'9'
			movwf THC
			movwf THX
			movwf THI
			movwf TMI
			movlw d'5'
			movwf TMX
			return


;****************************************************************************
; 4 LCD Module Subroutines
;****************************************************************************



;===========================  START Lcd_en  =================================
; Sends 2 us wide pos going pulse to LCD Enable pin.
;----------------------------------------------------------------------------

Lcd_en		bsf	E			; sets LCD E-line High.
			nop
			bcf	E			; sets LCD E-line Low.
			return


;===========================  START Lcd_busy  ===============================
; Returns when LCD Busy Flag is Low (inactive) meaning next command or
; character can now be sent to LCD.
;----------------------------------------------------------------------------

Lcd_busy 	bsf	STATUS,RP0		; go to register bank 1.
			movlw 0xFF			; temp set RB7-RB4 as inputs to read Busy 
			movwf TRISB   	    ; flag and DDRAM address. RB0-RB3 remain as
			bcf	STATUS,RP0		; inputs. Go back to register bank 0. 
			bcf RS				; set R/S to 0 to read Busy Flag. 
			bsf	RW              ; set R/W signal to LCD to 1 (Read).
Busy_loop	bsf	E				; clock out wamted busy flag and unwanted  
			nop      			; AC6-AC4 data.
			movf PORTB, W		; store Busy flag (Bit 7 of Port B) in W.
			andlw 0x80			; Examine Busy flag, 1 = Busy
			bcf E				; set E signal to LCD back to 0.
			nop
			bsf E               ; clock out unwanted AC3-AC0 data. 
			nop
			bcf E
			btfss STATUS, Z
			goto Busy_loop
			bcf	RW				; set R/W signal to LCD to 0 (Write).
			bsf STATUS,RP0		; go to register bank 1.
			movlw 0x0F			; restore RB7-RB4 as outputs. 
			movwf TRISB   	    ; RB0-RB3 remain as inputs.
			bcf	STATUS,RP0		; go back to register bank 0.
			return

				
;===========================  START Lcd_cmd =================================
; Sends one control byte to LCD as two separate nibbles after LCD has been
; initially configured. Control byte must initially be in W register.
;----------------------------------------------------------------------------

Lcd_cmd		movwf	LCDTEMP			 	
			call 	Lcd_busy		; includes setting RS to 0.		    			
			bsf     E
			movf 	LCDTEMP, W 
			movwf   PORTB
			nop
			bcf     E				; strobe first nibble into LCD
			nop	
			bsf     E
			swapf	LCDTEMP, W		; swap nibbles of LCD_TEMP and place
			movwf   PORTB			; result in W register.		
			nop
			bcf 	E				; strobe second nibble into LCD
			return


;===========================  START Lcd_chr =================================
; Sends one character to LCD as two separate nibbles after LCD has been
; initially configured. Character must initially be in W register.
;----------------------------------------------------------------------------

				
Lcd_chr		movwf	LCDTEMP
			call 	Lcd_busy		    			
			bsf     RS				; ensure RS is set to 1 for characters.
			bsf     E
			movf 	LCDTEMP, W 
			movwf   PORTB
			nop
			bcf 	E				; strobe first nibble into LCD
			nop
			bsf     E
			swapf	LCDTEMP, W		; swap nibbles of LCD_TEMP and place
			movwf   PORTB			; result in W register.		
			nop
			bcf 	E				; strobe second nibble into LCD
			return


	end					; end of instructions.

