; UVPC.ASM: undervoltage protection circuit, with a Pic10F22x
; BENABADJI M. Salim - ORAN - Oct. 05th, 2022

;controller	equ	10F220
controller	equ	10F222

	if	controller == 10F220
		List P = 10F220
#include <P10F220.inc>
	endif
	
	if	controller == 10F222
		List P = 10F222
#include <P10F222.inc>
	endif

;WATCHDOG on (timeout # 2.3 s)
	__CONFIG _MCLRE_OFF & _CP_OFF & _WDT_ON & _MCPU_OFF & _IOFSCS_4MHZ

;----------------------------- define hardware -------------------------------
;internal 4MHz
	
#define	 MOSFET	 GPIO, 0
#define	 PIEZO	 GPIO, 1
#define	 PIEZOb	 GPIO, 2
#define	 BTN1	 GPIO, 3

;----------------------------- define constants ------------------------------

	CONSTANT  CBLOCK16 = 0x10 ; 16 bytes RAM for PIC10F200
	CONSTANT  CBLOCK23 = 0x09 ; 23 bytes RAM for PIC10F222

REF06V	EQU	.59			; internal band gap 0.6V
REFCST	EQU	.15045		; = 59 * 255 ; then do REFCST / meas => Vdd * 100

;meas thresholds
TRIPFLAT   EQU	.55			; correspond to Vdd = 2,73 [V]
TRIPFULL   EQU	.36			; correspond to Vdd = 4,18 [V]

;------------------------------ define variables -----------------------------

	;CBLOCK	CBLOCK16	; if PIC10F220	
	CBLOCK	CBLOCK23	; if PIC10F222	

;delay
	cpt1, cpt2			; counter for delays
;measure
	meas				; 8-bit value from internal ADC module
; siren
	frequency, counter
	cycles1, cycles2
	steps

	ENDC
;8 bytes used

;////////////////////////////////////////////////////////////////////////
;								MACROS
;////////////////////////////////////////////////////////////////////////

;------------------------------------------------------------------------
; if (file >= val) goto jmpOk ;
;------------------------------------------------------------------------
SIregGEval8b	macro	file, val, jmpOk
	MOVLW	val
	SUBWF	file, w		; w = file - val
	BTFSC	STATUS, C
	GOTO	jmpOk		; yes
	endm

;-------------------------- Delay 128 us (for 4 MHz)---------------------------
DELAY128us	macro
	movlw	.42
	movwf	cpt1		; 2 us
	decfsz	cpt1, f		; cpt1-- and test if nul
	goto	$-1			; no, then loop (3 us * 42 = 126 us)
	endm

;-------------------------- Delay 257 us (for 4 MHz)---------------------------
DELAY257us	macro
	movlw	.85
	movwf	cpt1		; 2 us
	decfsz	cpt1, f		; cpt1-- and test if nul
	goto	$-1			; no, then loop (3 us * 85 = 255 us)
	endm

;/////////////////////////////////////////////////////////////////////////////
	ORG		0			; address start of program memory
	GOTO	Setup
;/////////////////////////////////////////////////////////////////////////////

;-----------------------------------------------------------------------------
; precise delay of 1ms
;-----------------------------------------------------------------------------
Delay1ms				; 2 us
	MOVLW	.248		; 1 us
	MOVWF	cpt1		; 1 us
Delay
	NOP					; 1 us	\
	DECFSZ	cpt1, F		; 1 us	 |=> 4 * 248 = 992 us = 0.992 ms delay loop 
	GOTO	Delay		; 2 us	/
	GOTO	$+1			; 2 us
	RETLW	0			; 2 us

;-----------------------------------------------------------------------------
; analog measure, 8-bit resolution
;-----------------------------------------------------------------------------
meas8bit
	MOVWF	ADCON0

	MOVLW	.8
	MOVWF	cpt1		; cpt1 = 32 us delay before acquisition (quartz 4 MHz)
	NOP					; 1 us
	DECFSZ	cpt1, F		; 1 us
	GOTO	$-2			; internal loop = 4 * 8 = 32 us

	bsf		ADCON0, GO	; initiate one A/D conversion
	btfsc	ADCON0, GO	; A/D conversion complete ?
	goto	$-1			; no

	movf	ADRES, W	; yes, then retreive measure
	movwf	meas		; store it in meas

	CLRF	ADCON0		; AN0, AN1 digitals, ADC off => low power consumption
	RETLW	0

;-----------------------------------------------------------------------------
; ToneHo sound, approx. 2 kHz (never apply permanent DC voltage to piezo)
;-----------------------------------------------------------------------------
ToneHo
	clrf  cycles1
Ho
	bcf   PIEZO		; PIEZO LOW...
	bsf   PIEZOb	;...during 0.259 ms
	DELAY257us

	bsf   PIEZO		; PIEZO HIGH...
	bcf   PIEZOb	;...during 0.259 ms
	DELAY257us

	decfsz cycles1, f
	goto   Ho		; repeat Ho tone during 0.518 * 256 = 132,608 ms

	bcf   PIEZO		; turn off piezo
	RETLW	0

;-----------------------------------------------------------------------------
; ToneHi sound, approx. 4 kHz (never apply permanent DC voltage to piezo)
;-----------------------------------------------------------------------------
ToneHi
	clrf  cycles1
Hi
	bcf   PIEZO		; PIEZO LOW ...
	bsf   PIEZOb	;...during 0.130 ms
	DELAY128us
	bsf   PIEZO		; PIEZO HIGH ...
	bcf   PIEZOb	;...during 0.130 ms
	DELAY128us
	bcf   PIEZO		; PIEZO LOW ...
	bsf   PIEZOb	;...during 0.130 ms
	DELAY128us
	bsf   PIEZO		; PIEZO HIGH ...
	bcf   PIEZOb	;...during 0.130 ms
	DELAY128us
	decfsz cycles1, f
	goto   Hi		; repeat Hi tone during 0.520 * 256 = 133,12 ms

	bcf   PIEZO		; turn off piezo
	RETLW	0

;=============================================================================
Setup		        	; Beginning of the program

	BTFSS	STATUS, NOT_TO	; <4> cleared upon a WDT reset
	GOTO	fromWDTreset

;calibrate OSCCAL with the value inside the last flash address
	;MOVLW	0x1A		; particular calibration value for my PIC10F222 sample
	MOVWF	OSCCAL		; trim the internal oscillator accuracy to 1% at 4 MHz

;turn off ADC (and set AN0 & AN1 as ditigal pins)
	CLRF	ADCON0		; AN0, AN1 digitals, ADC off => low power consumption

fromWDTreset ;----------------------------------------
; after WDT reset, register OPTION = b'1111 1111', so:
;cfg OPTION
	MOVLW  b'10001111'	; no wake-up, pull-up ON, <5>=0:use GP2, not TOCKI
	OPTION				; and WDT prescaler 1:128 => timeout # 2.3 s

;cfg GPIO
	MOVLW	b'11111000'	; (GP2,GP1,GP0) outputs
	TRIS	GPIO
	CLRF	GPIO		; all outputs LOW

;test piezo
	btfss	BTN1		; BP1 idle ?
	call	ToneHi		; no, it has been pushed
	
;pick meas
	MOVLW	b'00001101'	; select internal 0.6V ref, ADC on
	CALL	meas8bit

;meas >= TRIPFLAT ? => Vdd < 2.73 [V]
	SIregGEval8b	meas, TRIPFLAT, batFlat
;meas >= TRIPFULL ?  => Vdd < 4.18 [V]
	SIregGEval8b	meas, TRIPFULL, doSleep

batFull
	CALL	ToneHi		; piezo BIP high
	GOTO	doSleep

batFlat
	BSF	MOSFET			; turn off mosfet T1
	CALL	ToneHo		; piezo BIP low
	
doSleep
;deadtime of #2.3 s
;after CLRWDT, prescaler is cleared ; so reconfig. OPTION for WDT 1:128
	MOVLW  b'11001111'	; no wake-up, no pull-up, <5>=0:use GP2, not TOCKI
	OPTION				; and WDT prescaler 1:128 => timeout # 2.3 s
;veilleuse
	sleep				; sleep clears WDT

;-----------------------------------------------------------------------------
	END
