;*******************************************************************
;* File: PID.asm
;*
;* Source code for 
;*
;* Ver: 1.0
;*
;* L. Lerner
;*
;* You are free to copy and modify this software provided you 
;* reference its origin.
;*
;*******************************************************************
.include "2313def.inc"
;*******************************************************************
;*  		REGISTERS FOR PASSING VARIABLES TO SUBROUTINES
;*******************************************************************
.def temp0	=r16
.def temp1	=r18
.def temp2	=r19
.def temp3	=r20
.def msb	=r21
.def sign	=r22

.def ovfll	=r23
.def ovflh	=r24
.def timel	=r7
.def timeh	=r8
.def u_data =r27
;* r28,r29 index for SRAM access.	  NOT OK for interim storage
;* r30,r31 index for FLASH exp table.     OK for interim storage
.def adcl   =r30
.def adch   =r31

.def vcl	= r1
.def vch	= r2
.def vintl	= r3
.def vinth	= r4
.def vcm1l	= r5
.def vcm1h	= r6
.def vsl	= r25
.def vsh	= r26
.def T_cl	= r9
.def T_ch	=r10
.def T_ambl	=r11
.def T_ambh	=r12
.def Step  =r13
.def T_sm1l	=r14
.def T_sm1h	=r15
;*******************************************************************
;*
;*					GLOBAL VARIABLES
;*
;*	Floating point(U) number = y * 2^yp stored as (y, yp), y > 127
;*
;*******************************************************************
.equ t_d	=  0	;t_d  * 2^t_dp chosen so Vi<Vmax, and Vsys>0 
.equ t_dp	=  1
.equ t_sdk	=  2	;t_sp * 2^t_spp = 1000 * (tau+tau1) / k
.equ t_sdkp	=  3
.equ t_pdk	=  4	;t_p  * 2^t_pp  = 1000 *  tau*tau1  / k
.equ t_pdkp	=  5
.equ k		=  6 
.equ kp		=  7
.equ Control=  8
.equ T_sl	=  9
.equ T_sh	= 10
.equ duratl	= 11
.equ durath	= 12
.equ nBase	= 13	;this is record size, not pointer to a variable
;*******************************************************************
;*  PB0 - input signal 					- input  	   (comp. +)
;*  PB1 - A/D Capacitor	  				- input/output (comp. -)
;*  PB2 - none
;*  PB3 - PWM							- output
;*  PB4 - none
;*  PB5 - none
;*  PB6 - none
;*  PB7 - none
;*******************************************************************
;*  PD0 - RXD (UART)		-	input,  set by TXEN in UCR regardless of DDRD
;*  PD1 - TXD (UART)		-	output, set by TXEN in UCR regardless of DDRD
;*  PD2 - none
;*  PD3 - none
;*  PD4 - TTL input 232		-	output
;*  PD5 - TTL output 232	-	input
;*  PD6 - Set sense			-	input
;*******************************************************************
.equ PDSET 		= (1<<PD4)
.equ PBSET 		= (1<<PB1)+(1<<PB3)
.equ NBAUD 		= 25					;BAUD = 9600 for 4MHz Clock
.equ exp_table	= 0x380	;256 values of R[256*Exp[-n*64]], placed at $380	

.equ nCountH	=  11	;256*256 - count of 4MHz/64 clock in 1 sec
.equ nCountL	= 220	;i.e. ( (255-11)*256+(256-220) )*64 = 4000000

.equ SRAMSTART	= 0x60

.equ CODE1  		= 125
.equ sABORT			=   9  ;must be < nCountH for Pulse to work right
.equ sHALT			=  10  ;must be < nCountH for Pulse to work right
.equ sRUNLoad  		=   1  ;runs Thermal Program
.equ sPCToController=   2  ;receives from PC and loads to EEPROM Thermal Program
.equ sControllerToPC=   3  ;send to PC from EEPROM current Thermal Program
.equ sWRITETAMB   	=   4  ;writes number sent from PC as ambient temperature
.equ sPULSE    		=   5  ;pulses load at specified duty cycle and sends temperature data
.equ sGETTAMB  		=   6  ;re-measures ambient temperature
.equ sTEST2     	=   7  ;simple test: displays T while toggling load

.equ sTemp			= $10	;data stamps
.equ sTempAmb		= $20
.equ sDuty			= $30

.equ MAXRECORD 		=  26

.eseg

;* le				=	record length (including byte le)
;* t_d   * 2^t_dp	=	drive time constant in seconds 
;* t_sdk * 2^t_sdkp	=	proportional drive const = 1000. * (tau1+tau2) / k
;* t_pdk * 2^t_pdkp	=	differential drive const = 1000. * (tau1*tau2) / k
;* k     * 2^kp		=	1 / integral drive const = k / 1000.

.org 0x0  ;le, t_d, t_dp, t_sdk, t_sdkp, t_pdk, t_pdkp,   k, kp, control, T_sl, T_sh, duratl, durath 
step1: .db 18, 150,    1,   141,    250,   154,      3, 222,  3,       0,   34,    1,    100,      0
;			T_sl,	T_sh,	duratl,	durath
step2: .db	  10,	   3,		20,		 1

.cseg

.org exp_table			
exp0: .db 255, 252, 248, 244, 240, 237, 233, 229, 226, 222, 219, 216, 212, 209, 206, 203, 199, 196, 193, 190, 187, 184, 182, 179, 176, 173, 171, 168, 165, 163
exp1: .db 160, 158, 155, 153, 150, 148, 146, 144, 141, 139, 137, 135, 133, 131, 129, 127, 125, 123, 121, 119, 117, 115, 114, 112, 110, 108, 107, 105, 103, 102
exp2: .db 100, 99, 97, 96, 94, 93, 91, 90, 88, 87, 86, 84, 83, 82, 81, 79, 78, 77, 76, 75, 73, 72, 71, 70, 69, 68, 67, 66, 65, 64
exp3: .db  63, 62, 61, 60, 59, 58, 57, 56, 55, 55, 54, 53, 52, 51, 50, 50, 49, 48, 47, 47, 46, 45, 44, 44, 43, 42, 42, 41, 41, 40
exp4: .db  39, 39, 38, 37, 37, 36, 36, 35, 35, 34, 34, 33, 33, 32, 32, 31, 31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 26, 26, 25, 25
exp5: .db  25, 24, 24, 23, 23, 23, 22, 22, 22, 21, 21, 21, 20, 20, 20, 19, 19, 19, 19, 18, 18, 18, 17, 17, 17, 17, 16, 16, 16, 16
exp6: .db  15, 15, 15, 15, 14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11, 10, 10, 10, 10, 10
exp7: .db  10, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6}
exp8: .db   6, 6, 6, 6, 6, 6, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5


;************************************************************
;*  Signed Multiplication Macro
;*
;*  @0|@1 = @1 * @2. 
;*
;*  r17 is loop counter
;*
;************************************************************
.MACRO mmul
	sub @0, @0
	ldi r17, 0x08
mmul1:
	brcc mmul2
	add @0, @2
mmul2:
	sbrc @1, 0
	sub @0, @2
	asr @0
	ror @1
	dec r17
	brne mmul1
.ENDM

;*********************************************************************
;*  Four interrupt vectors are used.  Program placement commences
;*  after last one, so they must be in ascending order.
;*  Reset -> T1 Capture -> T1 Compare -> T1 Overflow
;*********************************************************************

.org 0x0000
rjmp reset
.org ICP1addr
rjmp captured
.org OVF1addr
rjmp clearovf

;**********************************************************
;*  General initialisation.  Port B and D directions set.
;*  LCD display initialised, welcome message displayed, A/D
;*  converter initialised.
;**********************************************************
reset:
	ldi r17, RAMEND		;start placing calls below 0xdf
	out SPL, r17
	clr r17
	out PORTD, r17
	out PORTB, r17		;clear pullup on comparator
	ldi r17, PDSET
	out DDRD, r17
	ldi r17, PBSET
	out DDRB, r17
	rcall u_init

	ldi yl, SRAMSTART
	rcall setTamb
;**************************************************************
;*
;*  Instr_loop is the main switch routine of the PID controller,
;*	commanded by the value of a key sent by the controlling PC.
;*  To ensure spurious-free triggering, routines are initiated
;*  by the recption of CODE1 followed by a valid prodecure 
;*  request.
;*
;*  Valid codes are:
;*
;*  sRunLoad  		Steps load thru a set of temperatures and durations
;*  sPCToControler  Receives from UART thermal program
;*  sControlerToPC  Send to UART thermal program
;*  sPULSE   		pulses load at duty from UART, and returns temperature
;*  sSETTAMB  		Re-measures ambient temperature
;*  sTEST2    		Simple test of setup: displays T and toggles load
;* 
;*  If the PID controller is used without a controlling PC, the 
;*  controller will run sRunLoad for the default load stored in EEPROM.  
;*  This procedure is triggered by pressing the set temperature button 
;*  on the digital thermometer.  The set temperature is then the one 
;*  set on the thermometer. 
;*
;********************************************************************
instr_loop:
	rjmp setbutton	;button trigger of runsecene for PC-less operation
l0: rcall getcret	;same as getc, but returns immediatelly
	cpi u_data, CODE1
	brne instr_loop
l1:	rcall getc
	cpi u_data, CODE1
	breq l1
	cpi u_data, sRUNLoad
	brne l2
	rjmp runLoad
l2:	cpi u_data, sPCToController
	brne l3
	rcall PCToController
	rjmp instr_loop
l3:	cpi u_data, sControllerToPC
	brne l4
	rcall ControllerToPC
	rjmp instr_loop
l4:	cpi u_data, sPULSE
	brne l5
	rjmp Pulse
l5:	cpi u_data, sGETTAMB	;remeasures ambient temperature
	brne l6
	rcall setTamb
	rjmp instr_loop
l6:	cpi u_data, sWRITETAMB	;writes ambient temperature sent from PC
	brne l7
	rcall writeTamb
	rjmp instr_loop
l7:	cpi u_data, sTEST2
	brne l8
	rjmp test2
l8:	rjmp instr_loop		

;****************************************************************
;*
;*  Routine branched to by main switch loop.  Acquires ambient 
;*  temperature (equivalent to ground potential).  Displays the result.
;*  
;****************************************************************
setTamb:
	rcall get_T
	mov T_ambl, adcl
	mov T_ambh, adch
	rcall sendTempAmb
	ret

;****************************************************************
;*
;*  Routine branched to by main switch loop.  Writes ambient 
;*  temperature (equivalent to ground potential) sent by the
;*  PC into memory.
;*  
;****************************************************************
writeTamb:
	rcall getc
	mov T_ambl, u_data
	rcall getc
	mov T_ambh, u_data
	rcall sendTempAmb
	ret

;****************************************************************
;*
;*  Routine retrieves ambient temperature word, marks the MSB and
;*  send to UART.
;*
;****************************************************************
sendTempAmb:
	mov temp0, T_ambl
	mov temp1, T_ambh
	andi temp1, $0f
	ori temp1, sTempAmb	;mark this word as ambient temperature
	rcall w_out
	ret

;****************************************************************
;*
;*  Copies scenario from non-volatile storage in EEPROM to SRAM
;*
;*  Routine branched to by main switch loop.  copies its thermal 
;*  parameters from eeprom storage into the main memory.
;*
;*	During copying the set temperatures are adjusted by the current
;*  ambient temperature.
;*
;*****************************************************************
getLoad:
	ldi temp0, 1		;indexes records into table. Currently only 1
	ldi r17, 1			;EEPROM strobe
	clr temp1
gs1:out EEAR, temp1
	out EECR, r17
	in  temp2, EEDR
	dec temp0
	breq gs2
	add temp1, temp2	;temp2 is length of Load data including length byte
	rjmp gs1			;so temp2 > 0
gs2:dec temp2
	breq gs3
	inc temp1
	out EEAR, temp1
	out EECR, r17
	in  temp3, EEDR
	st y+, temp3
	rjmp gs2
gs3:cpi yl, SRAMSTART+64
	breq gs4
	clr temp1
	st y+, temp1
	rjmp gs3
gs4:ldi yl, SRAMSTART	;restore base pointer in yl
	ret

;****************************************************************
;*
;*  Routine branched to by main switch loop.  Runs the current load
;*  using the parameters in main memory.  The temperature, time
;*  and duty cycle are displayed on the screen.  If control
;*  becomes unstable for some reason the routine displays an
;*  error message and exits.
;*
;*  Possible control types are Control = 0	On/Off
;*									   = 1  PID
;*
;*  The procedure adjusts T_s for current ambient temperature,
;*  then records the current temperature T_sm1.  
;*
;*  For Control = 0, the load duty cycle vs is then calculated as
;*
;*  	vs = 1000	T_s > T_c
;*		   =    0	T_s < T_c
;*
;*  For Control = 1 function get_vc calculates the PID output
;*
;*  The key timing core of this procedure is formed by counter1
;*  overflow interrups, with the counter initial value programmed
;*  to produce a 1 second interval.  Duty cycle timing within this
;*  is performed by the compare register connected to this timer.
;*  When there is a match, a flag in TIFR is set and poling this
;*  turns the load off.
;*
;*	The routine output the current temperature and duty cycle to
;*  the PC.  Since 8-bytes take about 8ms, which is a substantial
;*  portion of 1s, the output takes place during dead time, either
;*  with load off (if duty <~50%) or load on (duty >~ 50%).
;*
;****************************************************************
runLoad:
	rcall sendTempAmb
	rcall getLoad		;PC-less Load is always default
	ldi r17, nBase
	mov Step, r17		;Step points to next program step
rn4:ldd temp0, y+T_sl	;adjust T_set for T_ambient
	ldd temp1, y+T_sh
	sub temp0, T_ambl	
	sbc temp1, T_ambh	
	brge rn5
	clr temp0			;should flag T_s < T_amb to PC
	clr temp1
rn5:std y+T_sl, temp0
	std y+T_sh, temp1
	rcall get_T			;get T_sm1, i.e. previous value of T_set
	mov T_sm1l, adcl
	mov T_sm1h, adch
	sub T_sm1l, T_ambl	
	sbc T_sm1h, T_ambh	
	brge rnm			;ensure that T_sm1 >= T_amb
	clr T_sm1l
	clr T_sm1h
rnm:ldd temp0, y+T_sl
	ldd temp1, y+T_sh
	cp  temp0, T_sm1l
	cpc temp1, T_sm1h
	brge rnp
	mov T_sm1l, temp0	;if T_sm1 > T_s -> T_sm1 = T_s
	mov T_sm1h, temp1
rnp:clr timel
	clr timeh
	clr ovfll
	clr ovflh
	clr vintl
	clr vinth		
	rcall set_vcm1		;set initial value = vcm1 of last vc
;****************************************************************
;*  				Loop starts here
;****************************************************************
rn0:
	rcall abort			;this takes no time if queue empty
	rcall get_T			;get current T_c
	mov T_cl, adcl
	mov T_ch, adch
	sub T_cl, T_ambl	;T_c -= T_ambient
	sbc T_ch, T_ambh
	brge rn1			;if T_c < T_ambient -> T_c = 0
	clr T_cl
	clr T_ch
rn1:ldd temp0, y+Control
	sbrs temp0, 0
;****************************************************************
;*  			Control = 0 (On/Off)
;****************************************************************	
	rjmp OnOff
;****************************************************************
;*  			Control = 1 (PDI)
;****************************************************************	
	rcall get_vc
rn7:
	ldd r17, y+durath
	cp timeh, r17
	brne rn2
	ldd r17, y+duratl
	cp timel, r17
	brne rn2
	
	mov zl, yl				;get data for next step of program
	add zl, Step
	ld temp0, z+
	std y+T_sl, temp0
	ld temp0, z+
	std y+T_sh, temp0
	ld temp0, z+
	std y+duratl, temp0
	ld temp1, z+
	std y+durath, temp1
	ldi r17, 4
	add Step, r17			;Step made to point to next record
	or temp1, temp0
	brne rn4				;next step acquired, -> continue
	rjmp instr_loop			;program over, go to instruction loop
rn2:
	inc timel				;temp1|temp0 + 1	
	brne rn3
	inc timeh
rn3:					
	rcall pwm
	rjmp rn0
OnOff:
	ldd temp0, y+T_sl
	ldd temp1, y+T_sh
	sub temp0, T_cl
	sbc temp1, T_ch
	breq oo1
	brlt oo1
	ldi vsl, 231			;=999
	ldi vsh, 3
	rjmp rn7
oo1:clr vsl
	clr vsh
	rjmp rn7
;*************************************************************
;*
;*	PWM routine generates a 1 second delay, based on the value
;*  nCountH, nCountL loaded into COUNTER1, which generates an
;*  overflow interrupt.  A duty cycle is generated by loading
;*  the COMPARE register with appropriate values. When the
;*  comparator signals a match by a flag in the TIFR register, 
;*  (polled) this is used to turn load OFF.  
;*
;*  Dead time during the duty cycle is used to send (or
;*  receive) data.  Since UART processing times are ~ few ms
;*  and could skew the on time if this is short, when duty <
;*  about 50% off time is used for output.
;*
;*  Software PWM precision is 1/1000 = 0.1%
;*
;*  Hardware PWM precision is 1/100 pulse - to pulse since a 
;*  triac will remain turned on for the full 10ms half period 
;*  if triggered, at the beginning.  Hence the actual duty pulse-
;*  to-pulse has a variability depending on the synchronisation
;*  of the microcontroller and the mains, which is particularly
;*  noticeble for small duty.  If these two phases are random
;*  over a large period the duty averages to 25%.
;*
;*  There is however no definite 0 duty cycle with this method,
;*  which introduces a heavy bias when the PWM is exactly 0.
;*  Hence when vs = 0, PWM is forced 0 by a different mechanism.
;*  
;*
;*  Input:	vsh|vsl = duty cycle * 1000
;*
;*  Output: PWM on PB3
;*
;*************************************************************
pwm:
	ldi r17, 0xff		;clear all interrupt flags.
	out TIFR, r17		;ICF1 still be set by portd activity.
	ldi r17, nCountH	;high byte written first! 
	out TCNT1H, r17					
	ldi r17, nCountL	;4MHz= (256^2-(nCountH*256+nCountL))*64
	out TCNT1L, r17					
	mov temp0, vsl
	mov temp1, vsh
	ldi temp3, 125		;nCountH|nCountL/1000 = 125/2
	rcall mul16by8		;could have incorporated into thermal param
	ror msb
	ror temp1
	ror temp0
	mov sign, temp0
	or sign, temp1
	subi temp0, (256-nCountL)
	sbci temp1, (255-nCountH)
	out OCR1AH, temp1			;high compare first!
	out OCR1AL, temp0
	mov temp2, temp1
	ldi r17, (1<<TOIE1) 		;pwm terminated by counter overflow
	out TIMSK,  r17				;overflow interrupt enabled
	sei							;enable global interrupts
	ldi r17, (1<<CS10)+(1<<CS11);start counting at clk/64 speed
	out TCCR1B, r17				;start counter
	cpi sign, 0					;in this case data_out in off period
	breq pw2
	sbi PORTB, PB3				;load ON. 
	sbrc temp2, 7				;if OCR1A >= 128 * 256
	rcall data_out				;-> use this dead time to output data
pw1:					;interrupt uses r17, but cant occur during data_out
	in r17, TIFR		;switching pin io/compare problematic
	sbrc r17, OCF1A
	rjmp pw2
	brie pw1
pw2:cbi PORTB, PB3		;load OFF
	sbrs temp2,7		;if OCR1A < 128 * 256
	rcall data_out		;-> use this dead time to output data
pw3:brie pw3			;wait for overflow interrupt to clear I	
	ret

;********************************************************************
;*
;*  Routine jumped to by main instruction loop.  Used in PC-mode to 
;*  load default Thermal Program into GUI
;*
;*  Reads EEPROM and sends UART, in predefined sequence, the Thermal
;*  Program, whose length in bytes is defined by very first byte of
;*  current nLoad.  First part as in getLoad for loading SRAM from 
;*  EEPROM.
;*
;********************************************************************
ControllerToPC:
	ldi temp0, 1		;indexes records into table. Currently only 1
	ldi r17, 1			;EEPROM strobe
	clr temp1
ct1:out EEAR, temp1
	out EECR, r17
	in  temp2, EEDR
	dec temp0
	breq ct2
	add temp1, temp2	;temp2 is length of Load data including length byte
	rjmp ct1			;so temp2 > 0
ct2:mov u_data, temp2	
	rcall putc			;length byte -> UART
ct4:dec temp2
	breq ct3
	inc temp1
	out EEAR, temp1
	out EECR, r17
	in  temp3, EEDR
	mov u_data, temp3
	rcall putc			;data byte -> UART
	rjmp ct4
ct3:ret

;*******************************************************************
;*
;*  Routine called by main switch loop. It receives Thermal Program
;*  data from UART and stores in SRAM.  The data is of fixed length
;*  MAXRECORD bytes, which limits the number of individual steps in
;*  the program to 4.  Subsequently the data in SRAM is given
;*  persistence in the EEPROM.  This roundabout way is used because
;*  the EEPROM write time is 3-5ms, which will result in the EEPROM
;*  receive buffer overflow if direct recording was attempted.
;*
;*******************************************************************

PCToController:
	mov zl, yl			;SRAM <- UART so it copes with send rate
	ldi temp0, MAXRECORD
tc0:rcall getc
	st z+, u_data
	dec temp0
	brne tc0
	mov zl, yl				;now SRAM -> EEPROM
	ldi temp0, (1<<EEWE)	;write flag
	ldi temp1, (1<<EEMWE)	;master write flag
	clr temp2				;address data
tc1:ld  temp3, z+
tc2:sbic EECR, EEWE		;this takes a few milliseconds
	rjmp tc2
	out EEAR, temp2		;address -> EEPROM
	out EEDR, temp3		;data    -> EEPROM
	out EECR, temp1		;master write enable, within 4 clocks must write
	out EECR, temp0		;write
	inc temp2
	cpi temp2, MAXRECORD
	brlo tc1			;loop must be short so UART copes with send rate
	ret

;********************************************************************
;*
;*  Routine jumped to by main switch loop.  The same controller routine
;*  is used in the 3 PC functions:
;*
;*  Run - PC Mode		- To run thermal program with timing from PC
;*  Impulse Responce	- To apply impulse and measure thermal parameters
;*	Pulse				- To continualy pulse at given duty cycle
;*
;*  The controller carries out the following loop:
;*
;*  	->  Receives duty from UART
;*			Sets up on interval using Overflow Interrupt on Counter1
;*			Sends temperature -> UART in the larger of on/off time
;*      <-  On counter overflow
;*		<-	If UART received data before Interrupt, reset counter.
;*
;*	The last step is needed because for duty cycles close to 100% the
;*  next duty could be received before the on period is over, due to
;*  timing errors between the Windows timer and the microcontroller.
;*  This will result synchronisation between the two deteriorating
;*  until a data cycle will be lost.
;*
;*	The PC send on-period data as a byte value, which therefore has
;*  a maximum 100/256 ~ 0.3% precision, which is adequate, as hardware
;*  precision is ~ 0.5% - 1%.  In actual fact software precision is
;*  less than 0.3% as TCNT ranges only 11 - 255.
;*
;*	A duty period corresponding to a u_data value of sAbort is 
;*  reserved for abort, and is never sent otherwise, since the minimum
;*  value of u_data corresponding to a 100% on period is 11.
;*
;********************************************************************
Pulse:
	rcall get_T			;T -> adch|adcl
	mov temp2, adcl		;if duty short, counter may overflow before T_out
	mov temp3, adch
	rcall getc			;PC sends duty as on-fraction*nCount/256.
	cpi u_data, sAbort
	brne pl0
	rjmp instr_loop
pl0:clr temp0
	ldi r17, 0xff		;clear all interrupt flags.
	out TIFR, r17		;ICF1 still be set by portd activity.
	cpi u_data, 0		;this special data point == 0% duty
	breq pl2
	out TCNT1H, u_data	;high-byte first
	out TCNT1L, temp0				
	ldi r17, (1<<TOIE1) 		;pwm terminated by counter overflow
	out TIMSK,  r17				;overflow interrupt enabled
	sei							;enable global interrupts
	ldi r17, (1<<CS10)+(1<<CS11);start counting at clk/64 speed
	out TCCR1B, r17				;start counter
	sbi PORTB, PB3				;load ON. 
	sbrs u_data, 7				;here the higher value -> shorter duty!
	rcall T_out					;-> use this dead time to output data
pl1:sbis USR,RXC		;if next duty arrived exit loop
	brie pl1
	cli					;in case pl1 is true
	cbi PORTB, PB3		;load OFF
	sbrc u_data,7		;if TCNT1 < 128 * 256
pl2:rcall T_out			;-> use this dead time to output data
						;pl1 wont be true if u_data, 7 set!
	clr r17
	out TCCR1B, r17		;stop counter
	rjmp pulse			;RXC wont be set at pl1 if u_data, 7 clear!

;************************************************************************
;*
;*  Routine carries out communication with PC during dead-time in runLoad.
;*
;*  	1)  Duty -> UART, T_c -> UART
;*
;*  Since both Duty and T_c < $1000, the upper nibble of the MSB is marked
;*  to indicate the parameter the word represents
;*
;************************************************************************
data_out:
	mov temp0, vsl
	mov temp1, vsh
	andi temp1, $0f
	ori temp1, sDuty	;mark this word as duty
	rcall w_out
	mov temp0, T_cl
	mov temp1, T_ch
	add temp0, T_ambl	;T_c -= T_ambient
	adc temp1, T_ambh
	andi temp1, $0f
	ori temp1, sTemp
	rcall w_out
	ret

;************************************************************************
;*
;*  Routine carries out communication with PC during dead-time in Pulse.
;*
;*		2)  T_c -> UART
;*
;*  Since T_c < $1000, the upper nibble of the MSB is marked to indicate 
;*  the word represents temperature.
;*
;************************************************************************
T_out:
	mov temp0, temp2
	mov temp1, temp3
	andi temp1, $0f
	ori temp1, sTemp
	rcall w_out
	ret
;************************************************************************
;*
;*  Function sends the number temp1|temp0, as a 4 digit hexidecimal,
;*  MSB first to UART.
;*
;*  Input:	temp1|temp0
;*
;*  Output: UART <- temp1(H), temp1(L), temp0(H), temp0(L)
;*
;************************************************************************
w_out:
	mov r17, temp1
	swap r17
	rcall nib_to_hex
	mov u_data, r17
	rcall putc

	mov r17, temp1
	rcall nib_to_hex
	mov u_data, r17
	rcall putc

	mov r17, temp0
	swap r17
	rcall nib_to_hex
	mov u_data, r17
	rcall putc

	mov r17, temp0
	rcall nib_to_hex
	mov u_data, r17
	rcall putc

	ldi u_data, ' '
	rcall putc
	ret

;*******************************************************************
;*
;*  Function converts the lower 4-bits of r17 into an equivalent
;*  hexidecimal value stored in the 8-bits of r17.
;*
;*  Input:	lower 4-bits r17
;*
;*  Output:	r17 (Hex)
;*
;*******************************************************************
nib_to_hex:
	andi r17, $0f
	cpi  r17, $0a
	brsh hx
	subi r17, $d0
	ret
hx:	subi r17, $a9 ;$57
	ret

;********************************************************************
;*
;* Function u_init from Atmel AP910
;*
;* Initialses the UART, and sets baud rate.
;*
;********************************************************************
u_init:
	ldi r17, NBAUD
	out UBRR, r17
	ldi r17, 1<<TXEN|1<<RXEN
	out UCR, r17
	in r17, UDR
	ret

;********************************************************************
;*
;* Function putc from Atmel AP910
;*
;* Sends a character to the UART
;*
;********************************************************************
putc:
	sbis USR, UDRE
	rjmp putc
	out  UDR, u_data
	ret

;********************************************************************
;*
;* Function getc from Atmel AP910
;*
;* Gets a character to the UART.  Does not return until this happens.
;*
;********************************************************************
getc:
	sbis USR,RXC
	rjmp getc

	in r17, USR					;read UART status
	in u_data, UDR				;read UART byte
	andi r17, (1<<FE)+(1<<OR)	;test for framing or overrun errors
	brne getc					;ignore data on error
	ret

;********************************************************************
;*
;* Same as getc, but returns immediately.  If there no data, or the
;* data is corrupt u_data = 0;
;*
;********************************************************************
getcret:
	sbis USR,RXC
	rjmp gc1
	in r17, USR					;read UART status
	in u_data, UDR				;read UART byte
	andi r17, (1<<FE)+(1<<OR)	;test for framing or overrun errors
	breq gc2
gc1:clr u_data					;u_data==0 -> error
gc2:ret

;*******************************************************************
;*
;*  Function tests whether the sABORT or sHALT control codes have 
;*  been received by UART, if not it returns immediately.
;*
;*  If sABORT has been received, the function dumps the stack, and
;*  returns to the instruction loop.
;*
;*  if sHALT has been received, the function waits until the next
;*  reception of sHALT (=resume) and then resumes operation.
;*
;*******************************************************************
abort:
	sbis USR, RXC
ar1:ret
	in r17, USR
	in u_data, UDR
	andi r17, (1<<FE)+(1<<OR)
	brne ar1
	cpi u_data, sABORT
	breq ar2			;Abort!
	cpi u_data, sHALT
	brne ar1			;Halt!
ar3:rcall getc
	cpi u_data, sHALT
	breq ar1			;Resume!
	rjmp ar3
ar2:ldi temp0, RAMEND		;dump all function calls -> reset stack
	out SPL, temp0
	rjmp instr_loop

;*******************************************************************
;*
;*  Routine acquires current temperature T_c.  On overflow or error 
;*  it retries, up to 5 attempts.
;*
;*  Success is determined by subtracting $f0 from the high ADC byte.
;*  Since ADC values are in the range $0000 -> $0fff, the counter
;*  is started at $f000, and overflow is indicated by adc < $0fff.
;*  
;*  On success the function returns.  On faulure it displays error
;*  and dumping the stack exits to the instruction loop.
;*
;******************************************************************
get_T:
	ldi temp0, 5
gt1:rcall atod
	subi adch, $f0
	brcc gt2
	dec temp0
	brne gt1
;	rjmp error
gt2:ret

;*****************************************************************
;*
;*  Function used to display error, dump stack, and exit to
;*  instruction loop.
;*
;*****************************************************************
error:
;	ldi temp0, RAMEND	;dump all function calls -> reset stack
;	out SPL, temp0
	rjmp instr_loop

;******************************************************************
;*  Interrupt handlers for timer1 capture (converted) and timer1
;*  overflow (clearovf).  In former case capture register saved.  In 
;*  the latter 0x0000 written, to adch|adcl regsiters.
;*  Timer1 is stopped, and capacitor set to discharge by writting 0!!! 
;*  to PB1 and toggling it as output.  Routine automatically clears TOV1 
;*  enabling resumption of counter operation.
;*  
;*  Interrupts automatically disabled via i of SREG.  This routine 
;*  can not be interrupted!  
;*
;*  Exit via ret not reti, so interrupts remain disabled.
;*
;*  No need to push SREG as this interrupt handler does'nt alter it.
;*******************************************************************
;*
captured:
	in   adcl, ICR1L	;low capture register read first! p34
	in   adch, ICR1H	;works, capture register < counter!
overflowed:
	ldi r17, 0x00		;r17 is nowehere sensitive -> no need to save
	out TCCR1B, r17		;stop timer1
	cbi PORTB, PB1		;Never write one.  Just toggle in/out
	sbi DDRB,  PB1		;discharge capacitor, takes only about 1.5us
	ret
clearovf:
	ldi adcl,  0x00
	ldi adch,  0x00
	rjmp overflowed

;****************************************************************
;*  Sets up A/D conversion using capture and overflow interrupts. 
;*  Clears the timer interrupts ICF1, TOV1, 0CF1A, TOV0 in TIFR.
;*  Sets ACIC to route comparator to capture of timer 1.
;*  Clears counter.
;*  Enables capture and overflow interrupts TICIE, TOIE1 in TIMSK.
;*  Toggles negative comparator pin PB1 as input.  This pin 
;*  previously had 0 written to it to discharge capacitor, so
;*  pull up not enabled on toggle.
;*  
;*  0.015uF with 68mV/150ohm source converts 4V in 120us.
;*
;*  Last counter is enabled at clock speed (CS10) with capture
;*  on falling edge of comparator transition.
;*
;*	Waits for conversion over (interrupt clear) before returning
;*
;*  Uses stack to depth of 1.
;*****************************************************************
;*  
atod:
	ldi r17, 0xff		;Clear all interrupt flags.  PD action sets ICF1.
	out TIFR, r17		;ICF1 still be set by portd activity.
	ldi r17, 0xf0
	out TCNT1H, r17		;high byte written first! $f0-> 12-bit overflow
	clr r17
	out TCNT1L, r17
	ldi r17, (1<<ACIC)	
	out ACSR, r17		;comparator -> capture.  16bits overflow in 13ms! 
	ldi r17, (1<<TICIE)+(1<<TOIE1) ;if no toie1 count ovfls. AVR hangs.
	out TIMSK,  r17				   ;overflow and capture interrupts
	sei							   ;enable global interrupts
	ldi r17, (1<<CS10)			   ;start counting at clock speed
;	ldi r17, (1<<CS11)		   ;start counting at clk/8 speed
	cbi DDRB,  PB1		;draws ~150uA current as output
	sbi PORTB, PB2		;start charging capacitor.  
	out TCCR1B, r17		;capture on fallin edge.
at:	brie at				;wait for conversion over before returning
	ret

;*****************************************************************
;* Exponential Function.
;*
;* Evaluates f(x) = Round[ 256 * Exp[-x] ], for x > 0,
;*
;*			 f(0) = 255 			   	  , for x = 0.
;*
;* Input : temp1.temp0
;*
;* Output: temp0 
;*
;* Uses look-up-table, stored as Exp[-n/64]    for x < 4
;*
;* and  bound checking when n > 255, i.e.      for x > 4
;*
;*****************************************************************
exp:
	ser r17
	subi temp0, $fe	;=add 2
	sbc temp1, r17	;temp1.temp0 + 2/256 is equivalent to rounding		
	lsr temp1		;faster to divide by 4 than multiply by 64
	ror temp0
	lsr temp1
	ror temp0		;temp1|temp0 = temp1.temp0 * 64
	or temp1, temp1
	brne e1			;if true evaluate by bounds, else look-up table
	ldi r30, low(exp_table<<1)
	ldi r31,high(exp_table<<1)
	add r30, temp0
	adc r31, temp1	;temp1 must equal zero
	lpm
	mov temp0, r0
	ret
e1: 
	cpi temp1, 1
	brne e6
	cpi temp0, 3
	brsh e2
	ldi temp0, 5
	ret
e2:	cpi temp0, 19
	brsh e3
	ldi temp0, 4
	ret
e3: cpi temp0, 41
	brsh e4
	ldi temp0, 3
	ret
e4: cpi temp0, 73
	brsh e5
	ldi temp0, 2
	ret
e5: cpi temp0, 144
	brsh e6
	ldi temp0, 1
	ret
e6: ldi temp0, 0
	ret

;****************************************************************
;*
;*  Function called by runLoad with Control = 1 to evaluate the
;*  -1 value of the control voltage vcm1
;*
;*  	vcm1(t=0) == vc(t=-1) = T_s - (T_s - T_sm1)/t_d
;*
;****************************************************************
set_vcm1:
	ldd temp0, y+T_sl
	ldd temp1, y+T_sh
	mov vcm1l, temp0
	mov vcm1h, temp1
	sub temp0, T_sm1l		;we had checked this is >=0
	sbc temp1, T_sm1h 
	ldd temp2, y+t_d
	clr msb
	rcall div24by8
	ldd temp3, y+t_dp
	neg temp3
	rcall mulby2p
	ori sign, 1
	rcall inv_abs
	sub vcm1l, temp0
	sbc vcm1h, temp1
	ret

;****************************************************************
;*
;*  PID function called by runLoad with Control = 1.
;*
;*  The load temperature is raised T_s -> T_s as
;*
;*  		T_s - (T_s - T_sm1) exp(-t/t_d)
;*
;*  where t_d is a thermal parameter of the load chosen for
;*  critical response (minimum overshoot).
;*
;*  The error voltage is then
;*
;*  	vc = T_s - (T_s - T_sm1) exp(-t/t_d) - T_c
;*
;*  and the inverse filtered PID voltage vs is
;*
;*		 PID = proportional + integral + differntial
;*
;* 		  vs = vc*t_sdk     + vint/k   + (vc - vcm1)*t_pdk
;*
;*
;*	where T_c is current temperature.
;*
;*  Such a filter on vc forms the ideal inverse response when
;*  combined with the control voltage of the form vc.  It also
;*  corresponds to simple PID control.
;*
;*	All values are treated as signed short (2 BYTES).  So there
;*  is a possibility of overflow.  This is mainly in situations
;*  where PID control is lax (ineffective).  In this case, the
;*  overflow is truncated.
;*
;*  Values of vsys < 0 or vsys > 100% are accumulated in an
;*  overflow register.  Overflow of the latter is prevented
;*  by simple cropping.
;*
;*	Input:	time, T_c, T_s, T_sm1, t_d, t_sdk, t_pdk, vint, vcm1
;*
;*  Output: 0 <= vsh|vsm < 1000, -32768 < vovflh|vovfll < 32768
;*
;****************************************************************
get_vc:
	ldd vcl, y+T_sl			;vc = T_s - T_c
	ldd vch, y+T_sh
	sub vcl, T_cl
	sbc vch, T_ch
	mov msb,  timeh
	mov temp1,timel
	clr temp0
	ldd temp2, y+t_d
	rcall div24by8
	ldd temp3, y+t_dp
	neg temp3
	rcall mulby2p
	rcall exp
	mov temp3, temp0
	ldd temp0, y+T_sl		;form T_s - T_sm1
	ldd temp1, y+T_sh
	sub temp0, T_sm1l
	sbc temp1, T_sm1h
	rcall mul16by8
	sub vcl, temp1
	sbc vch, msb
;*		 PID = vc*t_sdk     + vint/k   + (vc-vcm1)*t_pdk
	mov temp0, vcl
	mov temp1, vch		;vch < 1
	rcall abs
	ldd temp3, y+t_sdk
	rcall mul16by8
	ldd  temp3, y+t_sdkp
	rcall mulby2p		;this function also handles overflow
	rcall inv_abs
	mov vsl, temp0		;vs = t_s * vch|vcl * 250 / k * 2^(t_sp+2-k_p)
	mov vsh, temp1
	mov temp0, vintl	;vint could be negative
	mov temp1, vinth
	rcall abs	
	ldd temp2, y+k
	clr msb
	rcall div24by8
	ldd temp3, y+kp		;underflow possible if kp -ve
	neg temp3
	rcall mulby2p
	rcall inv_abs
	add vsl, temp0		;vs+= vih|vil * 31 / k * 2^(5-k_p)
	adc vsh, temp1	
	rcall ovfl_ctrl
	mov temp0, vcl
	mov temp1, vch
	sub temp0, vcm1l
	sbc temp1, vcm1h	;|vch-vcm1h| < 1
	rcall abs
	ldd temp3, y+t_pdk
	rcall mul16by8
	ldd  temp3, y+t_pdkp
	rcall mulby2p		;this function controls overflow
	rcall inv_abs
	add vsl, temp0
	adc vsh, temp1
	rcall ovfl_ctrl
	add vsl, ovfll		;pulse+=overflow.  Ovfl
	adc vsh, ovflh
	rcall ovfl_ctrl
	clr ovfll			;clear overflow
	clr ovflh
	add vintl, vcl		;accumulate vint
	adc vinth, vch		
	mov vcm1l, vcl		;store previous value of vc: vcm1
	mov vcm1h, vch
	cpi vsh, 128
	brlo gv1
	mov ovfll, vsl		;pulse < 0 -> overflow = pulse, pulse = 0
	mov ovflh, vsh
	clr vsl
	clr vsh
	ret
gv1:ldi r17, 3
	cpi vsl, 232		;1000 = 3 * 256 + 232
	cpc vsh, r17
	brlo gv2
	mov ovfll, vsl		;pulse > 999 -> overflow=pulse-999, pulse=999
	mov ovflh, vsh
	subi ovfll, 231
	sbci ovflh,   3
	ldi vsl, 231
	ldi vsh, 3
gv2:ret	
;*****************************************************************
;*
;*  Function overflow control is used to truncated overflowed
;*  vsys values which are treated as signed short.
;*
;*****************************************************************
ovfl_ctrl:
	brvc of2			;if |pulse| > 32767 overflow control
	brlt of1
	ldi vsl, 255		;vsl=32767
	ldi vsh, 127
	ret
of1:ldi vsl, 0
	ldi vsh, 128
of2:ret

;****************************************************************** 
;*
;*  Routine evaluates Abs(temp1|temp0) stores sign in sign register
;*
;*	Input:	temp1|temp0 (type signed short)
;*
;*  Output: temp1|temp0 = Abs(temp1|temp0) (type signed short)
;*
;*			sign(b=0)	= Sgn(temp1|temp0)
;*
;******************************************************************
abs:
	cpi temp1, 128
	brlo ab2
	ori sign, $1	;sets bit 1
	cpi temp0, 0
	breq ab1
	neg temp0
	com temp1
	ret
ab1:neg temp1
	ret
ab2:andi sign, $fe	;clears bit 1		
	ret

;******************************************************************
;* 
;*  Routine evaluates temp1|temp0 * -1^sign(b=0).
;*
;*  Input:	temp1|temp0 (>0)
;*
;*  Output: temp1|temp0 (*=-1^sign(b=0)
;*
;******************************************************************
inv_abs:
	sbrs sign, 0
	ret
	cpi temp0, 0
	breq ia1
	neg temp0
	com temp1
	ret
ia1:neg temp1
    ret

;**************************************************************
;*  
;*  Routine evaluates power of 2 multiplication and division on
;*  temp1|temp0 (which must be positive).  The power of 2 is 
;*  stored in temp3.
;*
;*  Input:	temp1|temp0, temp3
;*
;*  Output:	temp1|temp0 * 2^temp3 (shifting left/right)
;*
;**************************************************************
mulby2p:
	clr r17
	cpi temp3, 128
	brsh mu3
mu1:cpi temp3, 0
	brne mu2
	cpi r17, 0			;sets output to 127|255 on overflow
	breq mu7
	ldi temp1, 127
	ldi temp0, 255
mu7:ret
mu2:lsl temp0
	rol temp1
	brcc mu6
	ser r17
mu6:dec temp3
	rjmp mu1
mu3:neg temp3
mu4:cpi temp3, 0
	brne mu5
	ret
mu5:lsr temp1
	ror temp0
	dec temp3
	rjmp mu4

;*******************************************************************
;*  Routine for unsigned 24 bit by 8 bit division.  Since rem is at 
;*  most 8-bit but 9-bits are required for interim calculation T-bit
;*  used for this purpose, economising one register.
;*
;*  result = msb|temp1|temp0 = (msb|temp1|temp0)/temp2 + temp3
;*
;*  Input:   msb|temp1|temp0 dividend, temp2 divisor
;*
;*  Output:  msb|temp1|temp0 result,   temp3 remainder
;*
;*  Uses:   r17 as counter.
;*******************************************************************
;*
div24by8:
	sub temp3, temp3	;clear remainder
	ldi r17, 25			;load loop counter, 24+shift last result bit
d0: rol temp0			;shift left dividend
	rol temp1
	rol msb
	bld temp0, 0		;result bit from T-flag, k=25 drops initial T
	dec r17
	brne d1
	ret
d1:	bst temp3, 7		;store carry bit from forthcoming shift
	rol temp3			;shift latest temp1|temp0|msb bit to temp3
	sub temp3, temp2
	brcc d2				;if (C=0)||(T=1) result must be +ve
	brts d2
	add temp3, temp2	;if (C=1)&&(T=0) rem < divisor
	rjmp d0
d2: set					;must set T flag for case (C=0)&&(T=0)
	rjmp d0

;**************************************************************
;*  This routine multiplies 16bit multiplier in temp1|temp0 by
;*  8bit multiplicand in temp3.  Result is a 3 byte digit
;*  msb|temp1|temp0.
;*
;*  		    temp3
;*  	            *
;*  	  temp1|temp0
;*  -----------------
;*    msb|temp1|temp0
;*
;*  Uses r17 as counter
;**************************************************************
mul16by8:
	clr msb 			;MSB of result
	ldi r17, 0x10
	lsr temp1			;rotate right multiplicand to see LSB
	ror temp0
z0: brcc z1				;LSB clear no add
	add msb, temp3		;add multiplicand(8bit) to result
z1: ror msb				;ror retrieves any carry from overflow
	ror temp1
	ror temp0
	dec r17				;this has no effect on carry
	brne z0
	ret

;****************************************************************
;*
;*  Routine branched to by main switch loop.  It toggles load
;*  on/off and send ADC data to UART every wait period.
;*
;*  Used as simple test of system integrity.
;*
;****************************************************************
test2:
	rcall get_T
	mov temp0, adcl
	mov temp1, adch
	andi temp1, $0f
	ori temp1, sTemp
	rcall w_out
	sbi PORTB, PB3
	ldi temp2, 16		;wait ~ 16*64 = 1 sec
	rcall wait
	cbi PORTB, PB3
	ldi temp2, 16
	rcall wait
	rcall abort
	rjmp test2
;****************************************************************
;*
;*  Period timer for test2 routine.  Period ~ 64*temp2 milliseconds
;*
;****************************************************************
wait:
	ldi temp0, 255
	ldi temp1, 255
wt1:dec temp0
	brne wt1
	dec temp1
	brne wt1
	dec temp2
	brne wt1
	ret
;****************************************************************
;*
;*  Routine called from main switch loop.  It checks for set
;*  button press on the thermometer.  Four outcomes are possible:
;*
;*  1) No high level detected 			  -> returns immediately
;*
;* else
;*
;*  2) < 3 high levels follow in 0.64 sec -> returns in 0.64 sec
;*
;* else
;*
;*  3)  Low level of 0.64 sec follows less than 2 sec after 2)
;*										  
;*			-> thermal program run with previous preset temperature
;*
;* else
;*
;*  4) Preset temperature recorded in EEPROM and thermal program run	
;*
;******************************************************************
setbutton:
	sbis PIND, PD6		;+ve level detected
	rjmp l0
	clr msb
	clr temp3
sb1:ldi temp2, 1		;slice = 0.064 sec
	rcall wait			;count # +ve in 10 0.064 sec slices
	sbic PIND, PD6
	inc msb				;increment # +ve detected
	inc temp3			;increment # slice
	cpi temp3, 10
	brne sb1
	cpi msb, 3
	brsh sb2
	rjmp l0				;if # +ves < 3, false trigger
sb2:clr msb
	clr temp3
sb3:ldi temp2, 1		;slice = 0.064 sec
	rcall wait			;must reload temp2 each call to wait
	sbic PIND, PD6
	clr temp3			;if +ve detected reset counter
	inc temp3			;increment counter
	inc msb				;increment # 0.064 sec intervals
	cpi msb, 20
	brne sb7			;if msb > 42 
	rcall get_T			;		  -> this value will be used as T_s
sb7:cpi temp3, 11		;if counter = 11 exit loop
	brne sb3
	cpi msb, 42			;msb = time while +ve detected in loop / 0.064 + 11
	brlo sb8			;test2			
	ldi temp2, (T_sl+1)		;addresses in EEPROM are 1 up on SRAM
	ldi temp0, (1<<EEWE)	;write flag
	ldi temp1, (1<<EEMWE)	;master write flag
sb4:sbic EECR, EEWE			;this takes a few milliseconds
	rjmp sb4
	out EEAR, temp2		;address -> EEPROM
	out EEDR, adcl		;data    -> EEPROM
	out EECR, temp1		;master write enable, within 4 clocks must write
	out EECR, temp0		;write
	inc temp2
sb5:sbic EECR, EEWE			;this takes a few milliseconds
	rjmp sb5
	out EEAR, temp2		;address -> EEPROM
	out EEDR, adch		;data    -> EEPROM
	out EECR, temp1		;master write enable, within 4 clocks must write
	out EECR, temp0		;write
sb6:sbic EECR, EEWE		;so runLoad does not try to read EEPROM early
	rjmp sb6
sb8:ldi temp2, 10
	rcall wait
	rjmp runLoad	 	;rjmp test2
