; Ian Paterson's Spoke POV
; Version "h" copyright 2006 Ian Paterson
; and copyleft under the terms of the GNU General Public License
; http://www.gnu.org/licenses/gpl.txt
;
;    This program is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 2 of the License, or
;    (at your option) any later version.
;
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with this program; if not, write to the Free Software
;    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
;
; To contact Ian Paterson, or for more information about this program, visit http://www.ianpaterson.org 


	title "628h"
	LIST	P=16F628A, F=INHX8M
#include <p16f628a.inc>

   __CONFIG  _INTRC_OSC_NOCLKOUT & _WDT_OFF & _LVP_OFF & _CP_OFF

IMAGESIZE       equ 0xFF      ; Declare image size constant
POSTSCALERLIMIT equ 0x02      ; Declare upper limit of the "postscaler" for TIMER1

linecount       equ 0x20
byteout         equ 0x21
postscaler      equ 0x22
timebasevector  equ 0x23
timebasevalue   equ 0x24

   org 0x00
   GOTO Main
   org 0x04
; Interrupt handler code goes here
; NOTE: Since virtually every part of this program runs inside an interrupt handler, it's not necesssary
; to back up any critical registers when entering or exiting the handler.
   BTFSC PIR1,TMR1IF          ; Check to see if the interrupt was caused by a TIMER1 rollover Overflow)
   GOTO  Timer1Rollover
   BTFSC INTCON,INTF          ; Check to see if the interrupt was caused by the hall effect sensor
   GOTO  HallEffectTrigger
   BTFSC INTCON,T0IF          ; Check to see if the interrupt was caused by a TIMER0 rollover (overflow)
   GOTO  Timer0Rollover
   BTFSC PIR1,CMIF            ; Check to see if the interrupt was caused by a change in the comparator output
   GOTO  LowBattery
   RETFIE                     ; If none of the above is true, then just leave the interrupt handler

Timer1Rollover
   MOVLW POSTSCALERLIMIT
   SUBWF postscaler, W
   BTFSS STATUS, Z            ; If postscaler <> POSTSCALERLIMIT,
   INCF  postscaler, F        ; Increment the TIMER1 postscaler
   BCF   PIR1,TMR1IF          ; Clear TMR1IF (TIMER1 Interrupt Flag) before enabling interrupts
   RETFIE

HallEffectTrigger
   MOVFW TMR1H                ; Capture the value in the high byte of the TIMER1 register pair
   MOVWF timebasevector       ; Store it in timebasevector (after some manipulation, this will be used to reference the timebase lookup table)
   MOVLW POSTSCALERLIMIT
   SUBWF postscaler, W
   BTFSC STATUS, Z            ; If postscaler = POSTSCALERLIMIT,
   GOTO  TimerOverflowed
   GOTO  RollBits
TimerOverflowed
   MOVLW 0xFF
   MOVWF timebasevector
   GOTO  EndHallEffectTrigger
RollBits
   RRF   timebasevector
   BCF   timebasevector,7
   BTFSC postscaler,0
   BSF   timebasevector,7
EndHallEffectTrigger
   MOVLW 0x2                  ; Load memory page number into PCLATH: 
   MOVWF PCLATH               ; The TimingData table starts at 0x200,
   MOVFW timebasevector       ; therefore PCLATH must be loaded with 0x2.
   CALL  TimingData           ; (see example in the OutputImage subroutine)
   MOVWF timebasevalue        ; Store the value fetched from the TimingData table in timebasevalue
   CLRF  postscaler           ; Reset the postscaler
   CLRF  linecount            ; Reset the line counter
   CLRF  TMR1L                ; Reset TIMER1 low byte
   CLRF  TMR1H                ; Reset TIMER1 high byte
   BCF   INTCON,INTF          ; Clear INTF flag before enabling interrupts
   RETFIE

Timer0Rollover                ; (263 instruction cycles)
   CALL  OutputImage
   MOVLW IMAGESIZE
   SUBWF linecount, 0
   BTFSS STATUS, Z            ; if linecount <> IMAGESIZE
   INCF  linecount, 1         ; Increment the line counter
   BCF   INTCON,T0IF          ; Clear T0IF flag before enabling interrupts
   MOVFW timebasevalue
   MOVWF TMR0
   RETFIE

LowBattery
   BTFSC CMCON,C1OUT          ; See if supply voltage is lower than reference voltage
   GOTO  ShutDown             ; If so then goto ShutDown routine
   BCF   PIR1,CMIF            ; Clear comparator interrupt flag and return
   RETFIE
ShutDown                      ; Turn off all LEDs, disable timer and hall effect interupts and enter sleep mode
   CALL TurnOffLeds
   BCF   INTCON,INTE          ; Disable interrupt on RB0/INT pin (an External interrupt)
   BCF   INTCON,T0IE          ; Disable interrupt on TIMER0 overflow (when TIMER0 wraps around from 0xFF to 0x00)
   BCF   INTCON,PEIE          ; Disable PEIE (PEripheral Interrupt Enable - for TIMER1, the 16 bit timer)
   BSF   STATUS,RP0           ; Go to bank 1
   BCF   PIE1,TMR1IE          ; Disable interrupt on TIMER1 overflow (when the TMR1 register pair wraps around from 0xFFFF to 0x0000)
   BCF   PIE1,CMIE            ; Disable interrupt on comparator output change
   BCF   STATUS,RP0           ; Go to bank 0
   BCF   INTCON,GIE           ; Disable global interrupts
   SLEEP
   RETFIE

Main
   CLRF  postscaler
   MOVLW 0xFF
   MOVWF timebasevalue
   BCF   STATUS,RP0           ; Go to bank 0
   BCF   STATUS,RP1           ; Go to bank 0

   MOVLW 0x0A                 ; Turn comparators on
   MOVWF CMCON

   BSF   STATUS,RP0           ; Go to bank 1
   BCF   TRISA,6              ; Set port A6 as output (power on state of TRISA is 11111111)
   BCF   TRISA,7              ; Set port A7 as output
   CLRF  TRISB                ; Set port B as all outputs and then....
   BSF   TRISB,0              ; set bit 0 of port B as an input 
; OPTION_REG bit descriptions:
; Bit#:  Name:     If set to 1:
;  7     NOT_RBPU  Disable all PORTB pull-up resistors
;  6     INTEDG    Interrupt on rising edge of pulse to RB0/INT pin
;  5     T0CS      Select external clock source for TIMER0 (transition on RA4 pin as opposed to internal clock)
;  4     T0SE      TIMER0 increments on high to low transition of RA4
;  3     PSA       Prescaler counter is assigned to the watchdog timer (as opposed to TIMER0)
;  2     PS2   \
;  1     PS1    -> TIMER0 prescaler bits: 000=1:2 001=1:4 010=1:8 011=1:16 100=1:32 101=1:64 110=1:128 111=1:256
;  0     PS0   /
   MOVLW B'01000011'          ; According to the 16F628A data sheet, this is the way you must set bits in
   MOVWF OPTION_REG           ; OPTION_REG if you are switching the prescaler from watchdog timer to TIMER0.

   MOVLW 0x8D                 ; This sets voltage reference to 3.4375 volts (for 8.4V battery packs)
; If you plan to use a 7.2V battery pack, un-comment the following line:
;   MOVLW 0XAE                 ; This sets voltage reference to 2.917 volts (for 7.2V battery packs)
   MOVWF VRCON

   BCF   STATUS,RP0           ; Go back to bank 0

   BSF   T1CON,T1CKPS1        ; These set the TIMER1 prescaler. Here are the possible values:
   BSF   T1CON,T1CKPS0        ; 00=1:1 01=1:2 10=1:4 11=1:8

   BCF   T1CON,T1OSCEN        ; Turn off the TIMER1 oscillator to save power (we don't need it because we're using the internal oscillator)
   BCF   T1CON,TMR1CS         ; Select the internal oscillator for TIMER1
   BSF   T1CON,TMR1ON         ; Turn on TIMER1

   CALL TurnOffLeds

; Enable interrupts
   BCF   INTCON,INTF          ; Clear INTF flag before enabling interrupts
   BCF   INTCON,T0IF          ; Clear T0IF (TIMER0 Interrupt Flag) before enabling interrupts
   BCF   PIR1,TMR1IF          ; Clear TMR1IF (TIMER1 Interrupt Flag) before enabling interrupts
   BSF   INTCON,INTE          ; Enable interrupt on RB0/INT pin (an External interrupt)
   BSF   INTCON,T0IE          ; Enable interrupt on TIMER0 overflow (when TIMER0 wraps around from 0xFF to 0x00)
   BSF   INTCON,PEIE          ; Enable PEIE (PEripheral Interrupt Enable - for TIMER1, the 16 bit timer)
   BSF   STATUS,RP0           ; Go to bank 1
   BSF   PIE1,TMR1IE          ; Enable interrupt on TIMER1 overflow (when the TMR1 register pair wraps around from 0xFFFF to 0x0000)
   BSF   PIE1,CMIE            ; Enable interrpt on comparator output change
   BCF   STATUS,RP0           ; Go to bank 0
   BSF   INTCON,GIE           ; Enable global interrupts

WaitForInterrupt
   GOTO WaitForInterrupt

TurnOffLeds
   CLRF  PORTA                ; Clear port A (set all lines low)
   CLRF  PORTB                ; Clear port B (set all lines low)
ClearOutput macro
   MOVLW 0x00
   MOVWF byteout
   CALL  WriteByte
   endm
   ClearOutput
   ClearOutput
   ClearOutput
   ClearOutput
   BSF PORTB,5                ; Pulse the latch to send data to the LEDs
   BCF PORTB,5
   RETURN

OutputImage                   ; (252 instruction cycles)
GetData macro pagenum
   MOVLW pagenum              ; Load memory page number into PCLATH: 
   MOVWF PCLATH               ; The top four pages start at: 0x400, 0x500, 0x600 and 0x700 respectively.
                              ; The lower eight bits are handled by adding linecount to PCL,
                              ; therefore PCLATH only needs to be loaded with 0x4, 0x5, 0x6 and 0x7.
   MOVFW linecount            ; Move linecount into W (the lower eight bits of page address)
   CALL ImageData
   MOVWF byteout
   CALL WriteByte
   endm

   GetData 0x7;
   GetData 0x6;
   GetData 0x5;
   GetData 0x4;

   BSF PORTB,5                ; Pulse the latch to send data to the LEDs
   BCF PORTB,5
   RETURN

WriteByte                     ; (50 instruction cycles)
BitLoop macro
   BCF PORTA,6
   BTFSC byteout,0
   BSF PORTA,6
   BSF PORTA,7
   BCF PORTA,7
   RRF byteout
   endm
   BitLoop
   BitLoop
   BitLoop
   BitLoop
   BitLoop
   BitLoop
   BitLoop
   BitLoop
   RETURN

   org 0x1FF
TimingData
   ADDWF PCL,F
; Page boundary, timing data starts here
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFF
   RETLW 0xFE
   RETLW 0xFD
   RETLW 0xFC
   RETLW 0xFB
   RETLW 0xFA
   RETLW 0xF9
   RETLW 0xF8
   RETLW 0xF7
   RETLW 0xF6
   RETLW 0xF5
   RETLW 0xF4
   RETLW 0xF3
   RETLW 0xF2
   RETLW 0xF1
   RETLW 0xF0
   RETLW 0xEF
   RETLW 0xEE
   RETLW 0xED
   RETLW 0xEC
   RETLW 0xEB
   RETLW 0xEA
   RETLW 0xE9
   RETLW 0xE7
   RETLW 0xE6
   RETLW 0xE5
   RETLW 0xE4
   RETLW 0xE3
   RETLW 0xE2
   RETLW 0xE1
   RETLW 0xE0
   RETLW 0xDF
   RETLW 0xDE
   RETLW 0xDD
   RETLW 0xDC
   RETLW 0xDB
   RETLW 0xDA
   RETLW 0xD9
   RETLW 0xD8
   RETLW 0xD7
   RETLW 0xD6
   RETLW 0xD5
   RETLW 0xD4
   RETLW 0xD3
   RETLW 0xD2
   RETLW 0xD1
   RETLW 0xD0
   RETLW 0xCF
   RETLW 0xCE
   RETLW 0xCD
   RETLW 0xCC
   RETLW 0xCB
   RETLW 0xC9
   RETLW 0xC8
   RETLW 0xC7
   RETLW 0xC6
   RETLW 0xC5
   RETLW 0xC4
   RETLW 0xC3
   RETLW 0xC2
   RETLW 0xC1
   RETLW 0xC0
   RETLW 0xBF
   RETLW 0xBE
   RETLW 0xBD
   RETLW 0xBC
   RETLW 0xBB
   RETLW 0xBA
   RETLW 0xB9
   RETLW 0xB8
   RETLW 0xB7
   RETLW 0xB6
   RETLW 0xB5
   RETLW 0xB4
   RETLW 0xB3
   RETLW 0xB2
   RETLW 0xB1
   RETLW 0xB0
   RETLW 0xAF
   RETLW 0xAE
   RETLW 0xAC
   RETLW 0xAB
   RETLW 0xAA
   RETLW 0xA9
   RETLW 0xA8
   RETLW 0xA7
   RETLW 0xA6
   RETLW 0xA5
   RETLW 0xA4
   RETLW 0xA3
   RETLW 0xA2
   RETLW 0xA1
   RETLW 0xA0
   RETLW 0x9F
   RETLW 0x9E
   RETLW 0x9D
   RETLW 0x9C
   RETLW 0x9B
   RETLW 0x9A
   RETLW 0x99
   RETLW 0x98
   RETLW 0x97
   RETLW 0x96
   RETLW 0x95
   RETLW 0x94
   RETLW 0x93
   RETLW 0x92
   RETLW 0x91
   RETLW 0x90
   RETLW 0x8E
   RETLW 0x8D
   RETLW 0x8C
   RETLW 0x8B
   RETLW 0x8A
   RETLW 0x89
   RETLW 0x88
   RETLW 0x87
   RETLW 0x86
   RETLW 0x85
   RETLW 0x84
   RETLW 0x83
   RETLW 0x82
   RETLW 0x81
   RETLW 0x80
   RETLW 0x7F
   RETLW 0x7E
   RETLW 0x7D
   RETLW 0x7C
   RETLW 0x7B
   RETLW 0x7A
   RETLW 0x79
   RETLW 0x78
   RETLW 0x77
   RETLW 0x76
   RETLW 0x75
   RETLW 0x74
   RETLW 0x73
   RETLW 0x72
   RETLW 0x70
   RETLW 0x6F
   RETLW 0x6E
   RETLW 0x6D
   RETLW 0x6C
   RETLW 0x6B
   RETLW 0x6A
   RETLW 0x69
   RETLW 0x68
   RETLW 0x67
   RETLW 0x66
   RETLW 0x65
   RETLW 0x64
   RETLW 0x63
   RETLW 0x62
   RETLW 0x61
   RETLW 0x60
   RETLW 0x5F
   RETLW 0x5E
   RETLW 0x5D
   RETLW 0x5C
   RETLW 0x5B
   RETLW 0x5A
   RETLW 0x59
   RETLW 0x58
   RETLW 0x57
   RETLW 0x56
   RETLW 0x55
   RETLW 0x53
   RETLW 0x52
   RETLW 0x51
   RETLW 0x50
   RETLW 0x4F
   RETLW 0x4E
   RETLW 0x4D
   RETLW 0x4C
   RETLW 0x4B
   RETLW 0x4A
   RETLW 0x49
   RETLW 0x48
   RETLW 0x47
   RETLW 0x46
   RETLW 0x45
   RETLW 0x44
   RETLW 0x43
   RETLW 0x42
   RETLW 0x41
   RETLW 0x40
   RETLW 0x3F
   RETLW 0x3E
   RETLW 0x3D
   RETLW 0x3C
   RETLW 0x3B
   RETLW 0x3A
   RETLW 0x39
   RETLW 0x38
   RETLW 0x37
   RETLW 0x35
   RETLW 0x34
   RETLW 0x33
   RETLW 0x32
   RETLW 0x31
   RETLW 0x30
   RETLW 0x2F
   RETLW 0x2E
   RETLW 0x2D
   RETLW 0x2C
   RETLW 0x2B
   RETLW 0x2A
   RETLW 0x29
   RETLW 0x28
   RETLW 0x27
   RETLW 0x26
   RETLW 0x25
   RETLW 0x24
   RETLW 0x23
   RETLW 0x22
   RETLW 0x21
   RETLW 0x20
   RETLW 0x1F
   RETLW 0x1E
   RETLW 0x1D
   RETLW 0x1C
   RETLW 0x1B
   RETLW 0x1A
   RETLW 0x18
   RETLW 0x17
   RETLW 0x16
   RETLW 0x15
   RETLW 0x14
   RETLW 0x13
   RETLW 0x12
   RETLW 0x11
   RETLW 0x10
   RETLW 0xF
   RETLW 0xE
   RETLW 0xD
   RETLW 0xC
   RETLW 0xB
   RETLW 0xA

   org 0x3FF
ImageData
   ADDWF PCL,F
; Page boundary, image data starts here.
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000010'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00100000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'10000000'
   RETLW B'01000000'
   RETLW B'00100000'
   RETLW B'00000000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000011'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00000000'
   RETLW B'00100000'
   RETLW B'00000000'
   RETLW B'01000000'
   RETLW B'00000000'
   RETLW B'01000000'
   RETLW B'00000000'
   RETLW B'00100000'
   RETLW B'00000000'
   RETLW B'00010000'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00010000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'01000000'
   RETLW B'11000000'
   RETLW B'01100000'
   RETLW B'00000000'
   RETLW B'00010000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'01100000'
   RETLW B'00000000'
   RETLW B'00010000'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00010000'
   RETLW B'00110000'
   RETLW B'01000000'
   RETLW B'01100000'
   RETLW B'00010000'
   RETLW B'00011000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00010000'
   RETLW B'00100000'
   RETLW B'01000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00010000'
   RETLW B'00000000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000000'
; Page boundary
   RETLW B'00010100'
   RETLW B'00010100'
   RETLW B'00000010'
   RETLW B'00010010'
   RETLW B'00100010'
   RETLW B'00100010'
   RETLW B'00100000'
   RETLW B'00000000'
   RETLW B'01100001'
   RETLW B'01000001'
   RETLW B'01000001'
   RETLW B'11000001'
   RETLW B'10000001'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'10000001'
   RETLW B'10000001'
   RETLW B'00000001'
   RETLW B'01000001'
   RETLW B'01000011'
   RETLW B'00000010'
   RETLW B'00100010'
   RETLW B'00000010'
   RETLW B'00100110'
   RETLW B'00100100'
   RETLW B'00100100'
   RETLW B'00100100'
   RETLW B'00110100'
   RETLW B'00111000'
   RETLW B'00111000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00100000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00111000'
   RETLW B'00111000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00110100'
   RETLW B'00100100'
   RETLW B'00100100'
   RETLW B'00100100'
   RETLW B'00000000'
   RETLW B'01000000'
   RETLW B'01000010'
   RETLW B'01000010'
   RETLW B'10000010'
   RETLW B'10000000'
   RETLW B'10000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000001'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'10000010'
   RETLW B'10000010'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'01000100'
   RETLW B'01000100'
   RETLW B'01100100'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00101000'
   RETLW B'00111000'
   RETLW B'00111000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00111000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100100'
   RETLW B'00100000'
   RETLW B'00100100'
   RETLW B'01000100'
   RETLW B'01000000'
   RETLW B'01000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'10000010'
   RETLW B'10000010'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'10000001'
   RETLW B'00000001'
   RETLW B'01000000'
   RETLW B'01000010'
   RETLW B'01000010'
   RETLW B'00000010'
   RETLW B'00000000'
   RETLW B'00100010'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100100'
   RETLW B'00110000'
   RETLW B'00110100'
   RETLW B'00110100'
   RETLW B'00011000'
   RETLW B'00111000'
   RETLW B'00111000'
   RETLW B'00010000'
   RETLW B'00110000'
   RETLW B'00100000'
   RETLW B'00010000'
   RETLW B'00111000'
   RETLW B'00011000'
   RETLW B'00111000'
   RETLW B'00010000'
   RETLW B'00010100'
   RETLW B'00010100'
   RETLW B'00010110'
   RETLW B'00110010'
   RETLW B'00000010'
   RETLW B'00100010'
   RETLW B'00100010'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00000001'
   RETLW B'01000001'
   RETLW B'11000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'10000001'
   RETLW B'01000000'
   RETLW B'11000000'
   RETLW B'00000000'
   RETLW B'00100001'
   RETLW B'00100001'
   RETLW B'00100001'
   RETLW B'00000001'
   RETLW B'00000011'
   RETLW B'00010010'
   RETLW B'00010010'
   RETLW B'00010010'
   RETLW B'00000010'
   RETLW B'00001010'
   RETLW B'00011010'
   RETLW B'00001110'
   RETLW B'00011100'
   RETLW B'00011100'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00011000'
   RETLW B'00001100'
   RETLW B'00001100'
   RETLW B'00001100'
   RETLW B'00001100'
   RETLW B'00011010'
   RETLW B'00001010'
   RETLW B'00000010'
   RETLW B'00001010'
   RETLW B'00010000'
   RETLW B'00000000'
   RETLW B'00010001'
   RETLW B'00010001'
   RETLW B'00000001'
   RETLW B'00100001'
   RETLW B'10100001'
   RETLW B'00100000'
   RETLW B'01100000'
   RETLW B'00000001'
   RETLW B'01000001'
   RETLW B'10000000'
   RETLW B'00000001'
   RETLW B'10000000'
   RETLW B'00000000'
   RETLW B'01000000'
   RETLW B'10100000'
   RETLW B'10100001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00010001'
   RETLW B'00010001'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00001010'
   RETLW B'00001000'
   RETLW B'00001100'
   RETLW B'00001100'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00010000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001100'
   RETLW B'00001100'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000010'
   RETLW B'00010010'
   RETLW B'00010000'
   RETLW B'00010011'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00100001'
   RETLW B'00100001'
   RETLW B'01000000'
   RETLW B'00000000'
   RETLW B'10000000'
   RETLW B'10000001'
   RETLW B'00000001'
   RETLW B'10000001'
   RETLW B'00000001'
   RETLW B'01000001'
   RETLW B'10000000'
   RETLW B'10100001'
   RETLW B'00100001'
   RETLW B'00000001'
   RETLW B'00100000'
   RETLW B'00000001'
   RETLW B'00010010'
   RETLW B'00010010'
   RETLW B'00010010'
   RETLW B'00010010'
   RETLW B'00010000'
   RETLW B'00011100'
   RETLW B'00011100'
   RETLW B'00011100'
   RETLW B'00011000'
   RETLW B'00011000'
   RETLW B'00011000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00011000'
   RETLW B'00011000'
   RETLW B'00011000'
   RETLW B'00011100'
   RETLW B'00010100'
   RETLW B'00000000'
; Page boundary
   RETLW B'01100000'
   RETLW B'01000000'
   RETLW B'11000000'
   RETLW B'11100000'
   RETLW B'01000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'10000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'11100000'
   RETLW B'11000000'
   RETLW B'11100000'
   RETLW B'11100000'
   RETLW B'11100000'
   RETLW B'11100000'
   RETLW B'11100000'
   RETLW B'11100001'
   RETLW B'11100001'
   RETLW B'11100001'
   RETLW B'11100011'
   RETLW B'11000011'
   RETLW B'11100010'
   RETLW B'11100010'
   RETLW B'11100010'
   RETLW B'11000110'
   RETLW B'11000100'
   RETLW B'11000100'
   RETLW B'11000100'
   RETLW B'10000100'
   RETLW B'11000100'
   RETLW B'10000000'
   RETLW B'10001000'
   RETLW B'10001000'
   RETLW B'00001000'
   RETLW B'10001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00001000'
   RETLW B'00000000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'10000001'
   RETLW B'10000001'
   RETLW B'11000001'
   RETLW B'10000001'
   RETLW B'11000001'
   RETLW B'01000001'
   RETLW B'11000001'
   RETLW B'01100001'
   RETLW B'01000001'
   RETLW B'11000001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'11110001'
   RETLW B'01100001'
   RETLW B'11100001'
   RETLW B'11100001'
   RETLW B'11100001'
   RETLW B'01100011'
   RETLW B'11100010'
   RETLW B'01100010'
   RETLW B'11100010'
   RETLW B'11100010'
   RETLW B'01100010'
   RETLW B'11100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'11100010'
   RETLW B'01000010'
   RETLW B'01000100'
   RETLW B'11000100'
   RETLW B'11000100'
   RETLW B'01000100'
   RETLW B'01000100'
   RETLW B'00000100'
   RETLW B'10000100'
   RETLW B'10001100'
   RETLW B'10001000'
   RETLW B'00001000'
   RETLW B'00000100'
   RETLW B'00000100'
   RETLW B'10000100'
   RETLW B'10000100'
   RETLW B'11000100'
   RETLW B'01000110'
   RETLW B'01000010'
   RETLW B'01000010'
   RETLW B'01000010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01110001'
   RETLW B'01110001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01110000'
   RETLW B'01110000'
   RETLW B'01110000'
   RETLW B'01110000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01000000'
   RETLW B'00000000'
   RETLW B'11000000'
   RETLW B'10000000'
   RETLW B'11000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'00000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'00000000'
   RETLW B'10000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000001'
   RETLW B'00000011'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'00000010'
   RETLW B'10000010'
   RETLW B'10000010'
   RETLW B'10000010'
   RETLW B'00000010'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000100'
   RETLW B'11000000'
   RETLW B'11000100'
   RETLW B'01000010'
   RETLW B'01000000'
   RETLW B'01000010'
   RETLW B'01100010'
   RETLW B'01000010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'01100010'
   RETLW B'11100001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'01100001'
   RETLW B'11100001'
   RETLW B'11100000'
   RETLW B'11100000'
   RETLW B'01100000'
   RETLW B'11100000'
   RETLW B'11100000'
   RETLW B'00000000'
; Page boundary
   RETLW B'01000000'
   RETLW B'01000000'
   RETLW B'01000000'
   RETLW B'01000000'
   RETLW B'01100000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00110000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00010000'
   RETLW B'00110000'
   RETLW B'00110000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'00100000'
   RETLW B'01100000'
   RETLW B'01100000'
   RETLW B'01000000'
   RETLW B'01000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'00000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'10000000'
   RETLW B'11000000'
   RETLW B'11000000'
   RETLW B'00000000'
   end
