
; Infra red Rolling Code transmitter

	ERRORLEVEL -302
	list P=16F628A
	#include p16f628A.inc

	__config _INTOSC_OSC_NOCLKOUT & _WDT_OFF & _PWRTE_ON & _BOREN_ON & _MCLRE_ON & _LVP_OFF & _CP_ON & _DATA_CP_ON

; Define variables at memory locations

EEPROM0		equ	H'00'	; storage of seed 32-bit ms
EEPROM1		equ	H'01'	; storage of seed 32-bit
EEPROM2		equ	H'02'	; storage of seed 32-bit
EEPROM3		equ	H'03'	; storage of seed 32-bit ls

EEPROM4		equ	H'04'	; storage of multiplier 24-bit ms
EEPROM5		equ	H'05'	; storage of multiplier 24-bit
EEPROM6		equ	H'06'	; storage of multiplier 24-bit ls

EEPROM7		equ	H'07'	; storage of incrementer 24-bit
EEPROM8		equ	H'08'	; byte 5 of random code
EEPROM9		equ	H'09'	; byte 6 of random code
EEPROM10	equ	H'0A'	; scramble data code
	
; RAM

; Math
BARGB0		equ	H'20'	; random number generator
BARGB1		equ	H'21'	; random number gen
BARGB2		equ	H'22'	; random number gen
AARGB0		equ	H'23'	; random number gen
AARGB1		equ	H'24'	; random number gen
AARGB2		equ	H'25'	; random number gen
AARGB3		equ	H'26'	; random number gen
AARGB4		equ	H'27'	; random number gen
AARGB5		equ	H'28'	; random number gen
AARGB6		equ	H'29'	; random number gen

TEMPB0		equ	H'2A'	; random gen temp files
TEMPB1		equ	H'2B'	; random gen temp files
TEMPB2		equ	H'2C'	; random gen temp files
TEMPB3		equ	H'2D'	; random gen temp files
LOOPCOUNT	equ	H'2E'	; loop counter in random gen

; Seed
RANDB0		equ	H'2F'	; random number seed ms
RANDB1		equ	H'30'	; random number seed
RANDB2		equ	H'31'	; random number seed
RANDB3		equ	H'32'	; random number seed ls
RANDB4		equ	H'33'	; random number last byte
RANDB5		equ	H'34'	; random number last byte

; Multiplier
MULT0		equ	H'35'	; multiplier ms
MULT1		equ	H'36'	; multiplier 
MULT2		equ	H'37'	; multiplier ls

; Increment
INCREMENT	equ	H'38'	; increment value

; Counters
COUNT38kHz	equ	H'39'	; 38.46kHz counter
VALUE		equ	H'3A'	; Value of bit for sending data
CYCLE_COUNT	equ	H'3B'	; 38.46kHz cycles per data
IDENTIFIER	equ H'3C'	; S1/S2 transmit identifier
VALUE_1		equ	H'3D'	; delay counter
VALUE_2		equ	H'3E'	; delay counter	
PRIME_CONT	equ	H'3F'	; prime number counter

SCRAMBLE	equ	H'40'	; data code scramble
SCRAMBLE1	equ	H'41'	; data code scramble working register

; Other
TEMPX		equ	H'42'	; temporary register
INTER_COUNT	equ	H'43'	; interrupt count
RANDOM_BLNK equ	H'44'	; random blink rate
RNDM_BNK	equ	H'45'	; random blink rate
XTRA_DLY	equ	H'55'	; extra delay

; all banks 
FLAG		equ	H'70'	; timing flag	
		
; preprogram EEPROM DATA 
	
	ORG     H'2100'		; start at 00
	
	DE	H'72', H'2D', H'9B', H'75', H'00', H'A1', H'8E', H'1D', H'CD', H'17', H'00'

; define reset and interrupt vector start addresses

	org		0  			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector

	org     4			; interrupt vector 0004h, starts interrupt routine here

INTERRUPT
	bcf		INTCON,T0IF	; clear TMRO interrupt flag
	bsf		FLAG,0		; set timing flag at each interrupt (all banks register)
	retfie				; return from interrupt


; lookup for odd prime numbers less than D'255'
; use odd prime numbers (so not 2) (total 54 numbers)

PRIME_No
	addwf	PCL,f		; add value to program counter
	retlw	D'1'
	retlw	D'3'
	retlw	D'5'
	retlw	D'7'
	retlw	D'11'
	retlw	D'13'
	retlw	D'17'
	retlw	D'19'
	retlw	D'23'
	retlw	D'29'
	retlw	D'31'
	retlw	D'37'
	retlw	D'41'
	retlw	D'43'
	retlw	D'47'
	retlw	D'53'
	retlw	D'59'
	retlw	D'61'
	retlw	D'67'
	retlw	D'71'
	retlw	D'73'
	retlw	D'79'
	retlw	D'83'
	retlw	D'89'
	retlw	D'97'
	retlw	D'101'
	retlw	D'103'
	retlw	D'107'
	retlw	D'109'
	retlw	D'113'
	retlw	D'127'
	retlw	D'131'
	retlw	D'137'
	retlw	D'139'
	retlw	D'149'
	retlw	D'151'
	retlw	D'157'
	retlw	D'163'
	retlw	D'167'
	retlw	D'173'
	retlw	D'179'
	retlw	D'181'
	retlw	D'191'
	retlw	D'193'
	retlw	D'197'
	retlw	D'199'
	retlw	D'211'
	retlw	D'223'
	retlw	D'227'
	retlw	D'229'
	retlw	D'233'
	retlw	D'239'
	retlw	D'241'
	retlw	D'251'
	retlw	D'251'		; last number

MAIN
; initialise ports
	
	movlw	0x07		; comparators off
	movwf	CMCON		; I/O pins
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'11000000'	; port B outputs/inputs 
	movwf	TRISB		; port B data direction register
	movlw	B'11100011'	; RA outputs/inputs
	movwf	TRISA		; port A data direction register
	movlw	B'00000001'	; 1024us between bit changes in transmission (4MHz operation) 
	movwf	OPTION_REG	; port B pullups enabled

	bcf		STATUS,RP0	; memory bank 0
	clrf	PORTA		; ports clear or low
	clrf	PORTB

; read address to determine identifier and check for S2 closure

	movf	PORTA,w		; bits 0,1 and 6,7 are address
	andlw	B'11000011'	
	movwf	IDENTIFIER

; catch S2 closed if pressed
	bcf		PORTA,4		; shunt RA4 to ground
	call	DELAY		; debounce S2
	bsf		PORTA,4		; open drain to allow reading of S2
	nop					; time to rise
	btfsc	PORTA,4		; if bit 4 is set set bit 2 in identifier
	bsf		IDENTIFIER,2
	bcf		PORTA,4		; shunt RA4 to ground
	call	DELAY		; debounce S2
	bsf		PORTA,4		; open drain to allow reading of S2
	nop
	btfsc	PORTA,4		; if bit 4 is set set bit 2 in identifier
	bsf		IDENTIFIER,2
	bcf		PORTA,4		; shunt RA4 to ground
	call	DELAY		; debounce S2
	bsf		PORTA,4		; open drain to allow reading of S2
	nop					; time to rise
	btfsc	PORTA,4		; if bit 4 is set set bit 2 in identifier
	bsf		IDENTIFIER,2

; set RA4 low
	bcf		PORTA,4
	bsf		PORTA,2		; power on latch
	
; allow interrupts
	bsf		INTCON,T0IE	; set interrupt enable for TMR0 

; Recall seed/code
	movlw	EEPROM0		; seed
	call	EEREAD
	movwf	RANDB0

	movlw	EEPROM1		; seed
	call	EEREAD
	movwf	RANDB1

	movlw	EEPROM2		; seed
	call	EEREAD
	movwf	RANDB2

	movlw	EEPROM3		; seed
	call	EEREAD
	movwf	RANDB3

; Recall multiplier

	movlw	EEPROM4		; multiplier
	call	EEREAD
	movwf	MULT0

	movlw	EEPROM5		; multiplier
	call	EEREAD
	movwf	MULT1

	movlw	EEPROM6		; multiplier
	call	EEREAD
	movwf	MULT2

; Recall increment
	movlw	EEPROM7		; increment value
	call	EEREAD
	movwf	INCREMENT

; Recall byte 5 of code
	movlw	EEPROM8		; byte 5
	call	EEREAD
	movwf	RANDB4

; Recall byte 6 of code
	movlw	EEPROM9		; byte 6
	call	EEREAD
	movwf	RANDB5

; Recall scramble data code
	movlw	EEPROM10		; 
	call	EEREAD
	andlw	B'00011111'		; maximum of 32 
	movwf	SCRAMBLE

; check for sync or randomise of parameters
	
	btfss	PORTB,6		; if low then randomise
	goto	RANDOMISE
	btfss	PORTB,7		; if low then SYNC
	goto 	SYNCHRONISE

; Randomise and store new code. Send this latest code 

SET_NEW_CODE

	call	RANDOM		; calculate new code
; store
 	movlw	EEPROM0		; sets EEADR
	call	EEREAD
	movf	RANDB0,w
	call	EWRITE		; store

	movlw	EEPROM1		; seed
	call	EEREAD
	movf	RANDB1,w
	call	EWRITE		; store

	movlw	EEPROM2		; seed
	call	EEREAD
	movf	RANDB2,w
	call	EWRITE		; store

	movlw	EEPROM3		; seed
	call	EEREAD
	movf	RANDB3,w
	call	EWRITE		; store

	movlw	EEPROM8		; byte 5
	call	EEREAD
	movf	RANDB4,w
	call	EWRITE		; store

	movlw	EEPROM9		; byte 6
	call	EEREAD
	movf	RANDB5,w
	call	EWRITE		; store

; Four start bits 1,1,1,1.  Send identifier (bits 0,1,6,7 are address)
; dependent on RA0,1,6,7. Bit 2  =0 for main S1, bit 2 =1 for secondary, S2
; send RANDB0,1,2,3,4,5
; Four stop bits 1,0,1,1 (64 bits total)

; send data 

SEND_CODE
	clrf	TMR0		; start from zero	
	bcf		FLAG,0		; start at 0
	bcf		INTCON,T0IF	; clear interrupt flag
	bsf		INTCON,GIE	; allow interrupt
READ_FLAG
	btfss	FLAG,0		; wait for first interrupt to occur
	goto	READ_FLAG
	clrf	FLAG

; send a 1 for for first of start bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 1,1,1
	bsf		VALUE,0
	call	DRV_BIT		; 1
; wait for 15 interrupts before sending the 1 bit
	movlw	D'14'		; 14 for 15 interrupts
	movwf	INTER_COUNT
RECYCLE_FLG
	btfss	FLAG,0		; wait for flag in interrupt to maintain timing sync
	goto	RECYCLE_FLG
	clrf	FLAG
	decfsz	INTER_COUNT,f
	goto	RECYCLE_FLG
	
	bsf		VALUE,0
	call	DRV_BIT		; 1 (16th interrupt)
	bsf		VALUE,0
	call	DRV_BIT		; 1

; send address and switch identifier

	btfsc	IDENTIFIER,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,0
	bsf		VALUE,0	
	call	DRV_BIT

; send data ms bits first.
; send according to scramble
 
	movf	SCRAMBLE,w
	andlw	B'00011111' ; use 32 possibilities
	movwf	SCRAMBLE
	call	SEND_SCRAMBLE; send the dats

	goto	SEND_STOP

SUB_RANDB0
	btfsc	RANDB0,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB1
	btfsc	RANDB1,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB2
	btfsc	RANDB2,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB3
	btfsc	RANDB3,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB4
	btfsc	RANDB4,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB5
	btfsc	RANDB5,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,0
	bsf		VALUE,0
	call	DRV_BIT
	return

; send stop bits
; send a 1 for for first of stop bits
SEND_STOP
;	bsf		INTCON,GIE	; allow interrupt
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 0,1,1
	call	DRV_BIT		; 0
	bsf		VALUE,0
	call	DRV_BIT		; 1
	bsf		VALUE,0
	call	DRV_BIT		; 1

; data sent 	
	bcf		INTCON,GIE	; stop interrupt
LOOP_OFF
	bcf		PORTA,3		; ACK led off
	bcf		PORTA,2		; power latch off
	goto	LOOP_OFF	; end of transmission

; *********************************************************************			
SYNCHRONISE
	clrf	TMR0		; start from zero	
	bcf		FLAG,0		; start at 0
	bcf		INTCON,T0IF	; clear interrupt flag
	bsf		INTCON,GIE	; allow interrupt

READ_FLAG0
	btfss	FLAG,0		; wait for first interrupt to occur
	goto	READ_FLAG0
	clrf	FLAG

; sends the identifier, current random RANDB0,1,2,3 seed code, the MULTiplier (MULT0,1,2) code 
; the INCREMENT and scramble (80 bits)
; Four start bits 1,1,1,1. Four stop bits 1,0,1,1 (88 bits total)

; send a 1 for for first of start bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 1,1,1
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 1)
; wait for 16 interrupts before sending the 1 bit
	movlw	D'14'		; 14 for 15 interrupts
	movwf	INTER_COUNT
RECYCLE_FLG1
	btfss	FLAG,0		; wait for flag in interrupt to maintain timing sync
	goto	RECYCLE_FLG1
	clrf	FLAG
	decfsz	INTER_COUNT,f; when 16 interrupts
	goto	RECYCLE_FLG1
	
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 16)
	bsf		VALUE,0
	call	DRV_BIT		; 1

; send identifier
; send address identifier

	movf	PORTA,w		; bits 0,1 and 6,7 are address
	andlw	B'11000011'	; strip out unwanted bits
	movwf	IDENTIFIER
	btfsc	IDENTIFIER,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,0
	bsf		VALUE,0
	call	DRV_BIT

; send data ms bits first. Bytes: random seed, multiplier, increment value
; RANDB0 seed value
	btfsc	RANDB0,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,0
	bsf		VALUE,0
	call	DRV_BIT
; RANDB1
	btfsc	RANDB1,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,0
	bsf		VALUE,0
	call	DRV_BIT
; RANDB2
	btfsc	RANDB2,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,0
	bsf		VALUE,0
	call	DRV_BIT
; RANDB3
	btfsc	RANDB3,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,0
	bsf		VALUE,0
	call	DRV_BIT
; end of first block
	
; send stop bits
; send a 1 for for first of stop bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 0,1,1
	call	DRV_BIT		; 0
	bsf		VALUE,0
	call	DRV_BIT		; 1
	bsf		VALUE,0
	call	DRV_BIT		; 1
; data sent 
	bcf		INTCON,GIE	; stop interrupt

; add delay to allow EEPROMs within receiver to be stored before next block of data is sent

	call	DELAY		; delay gap 

; start of second block
	clrf	TMR0		; start from zero	
	bcf		FLAG,0		; start at 0
	bcf		INTCON,T0IF	; clear interrupt flag
	bsf		INTCON,GIE	; allow interrupt

READ_FLAGx
	btfss	FLAG,0		; wait for first interrupt to occur
	goto	READ_FLAGx
	clrf	FLAG

; sends the MULTiplier (MULT0,1,2) code 
; the INCREMENT and scramble (80 bits)
; Four start bits 1,1,1,1. Four stop bits 1,0,1,1 (88 bits total)

; send a 1 for for first of start bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 1,1,1
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 1)
; wait for 16 interrupts before sending the 1 bit
	movlw	D'14'		; 14 for 15 interrupts
	movwf	INTER_COUNT
RECYCLE_FLGX
	btfss	FLAG,0		; wait for flag in interrupt to maintain timing sync
	goto	RECYCLE_FLGX
	clrf	FLAG
	decfsz	INTER_COUNT,f; when 16 interrupts
	goto	RECYCLE_FLGX
	
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 16)
	bsf		VALUE,0
	call	DRV_BIT		; 1

; multiplier
; MULT0
CONTINUE_WITH_MULT
	btfsc	MULT0,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,0
	bsf		VALUE,0
	call	DRV_BIT
; MULT1
	btfsc	MULT1,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,0
	bsf		VALUE,0
	call	DRV_BIT
; MULT2
	btfsc	MULT2,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,0
	bsf		VALUE,0
	call	DRV_BIT
; increment value
; INCREMENT
	btfsc	INCREMENT,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,0
	bsf		VALUE,0
	call	DRV_BIT
; scramble value
; SCRAMBLE
	btfsc	SCRAMBLE,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,0
	bsf		VALUE,0
	call	DRV_BIT

; send stop bits
; send a 1 for for first of stop bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. 'VALUE' cleared at DRV_BIT
; then 0,1,1
	call	DRV_BIT		; 0
	bsf		VALUE,0
	call	DRV_BIT		; 1
	bsf		VALUE,0
	call	DRV_BIT		; 1
; data sent 
	bcf		INTCON,GIE	; stop interrupt

	bsf		PORTA,3		; Ack LED on
LOOP_2
	call	DELAY
	call	DELAY
	call	DELAY
	bcf		PORTA,3		; ACK LED off
	bcf		PORTA,2		; power latch off
	goto	LOOP_2		; program stopped till reset

; **************************************************************************************

RANDOMISE ;(RB6 low (3-5 link), S2 to power up) 
; alters the RANDB values, the MULTiplier values and the Increment values depending on S2 period closure

	bcf		PORTA,2		; power latch off until S2 closed
; wait for S2 closure if S1 pressed instead
; check switch
	bcf		PORTA,4		; ensure is low first
	nop
	bsf		PORTA,4		; open drain
	nop
	nop
	nop
	nop
	btfss	PORTA,4		; if high then S2 closed
	goto 	RANDOMISE	; if low then S2 not closed yet
	bsf		PORTA,2		; power latch on
	call	DELAY
	bcf		PORTA,4
	bsf		PORTA,3		; Ack LED on

; randomise values (randomness assured by the rapid and numerous changes 
; and the much greater and indeterminate time of S2 pressed)
	movlw	D'54'		; maximum value for Prime_counter
	movwf	PRIME_CONT	
	clrf	RANDOM_BLNK ; ack flash rate main counter
	movlw	D'12'
	movwf	RNDM_BNK	; ack flash rate secondary counter

CONTINUE_RANDOM ; (change and jumble values)
	decfsz	RANDOM_BLNK,f
	goto	RNDM_SEED
	decfsz	RNDM_BNK,f
	goto	RNDM_SEED
	movlw	D'12'
	movwf	RNDM_BNK	; ack flash rate	
	btfss	PORTA,3		; ACK LED
	goto	RNDM_ACK
	bcf		PORTA,3		; off
	goto	RNDM_SEED
RNDM_ACK
	bsf		PORTA,3

RNDM_SEED
; seed values

	movlw	D'77'		; add 77
	addwf	RANDB3,f
	btfsc	STATUS,C	; if carry the increment next value
	incf	RANDB2,f
	btfsc	STATUS,Z	; if over increase next
	incf	RANDB1,f
	btfsc	STATUS,Z
	incf	RANDB0,f

	movlw	D'02'		; add 2 to the more significant bytes
	addwf	RANDB1,f
	btfsc	STATUS,C	; if over increase next
	incf	RANDB0,f
	
; scramble value
	incf	SCRAMBLE,f	; even number increment
	incf	SCRAMBLE,f

; multiplier values
;	MULT0, MULT1, MULT2

	movlw	D'31'		; add 31
	addwf	MULT2,f
	btfsc	STATUS,C	; if carry the increment next value
	incf	MULT1,f
	btfsc	STATUS,Z	; if over increase next
	incf	MULT0,f
	
NEW_PRIME
	movf	PRIME_CONT,w; 
	call	PRIME_No	; get prime number from table
	movwf	INCREMENT
	decfsz	PRIME_CONT,f; next prime
	goto	CHECK_S2_RANDOM
	movlw	D'54'		; maximum value for Prime_counter
	movwf	PRIME_CONT

; check S2
CHECK_S2_RANDOM

; check switch
	bcf		PORTA,4		; ensure is low first
	nop
	nop
	bsf		PORTA,4		; open drain
	nop
	nop
	nop
	nop
	btfsc	PORTA,4		; if high then S2 closed
	goto 	CONTINUE_RANDOM	; if high then continue to randomise 
	bcf		PORTA,4		; low RA4

	bsf		PORTA,3		; Ack LED on

; store values when S2 reopened
; random values
	movlw	EEPROM0		; sets EEADR
	call	EEREAD
	movf	RANDB0,w
	call	EWRITE		; store

	movlw	EEPROM1		; seed
	call	EEREAD
	movf	RANDB1,w
	call	EWRITE		; store

	movlw	EEPROM2		; seed
	call	EEREAD
	movf	RANDB2,w
	call	EWRITE		; store

	movlw	EEPROM3		; seed
	call	EEREAD
	movf	RANDB3,w
	call	EWRITE		; store

; multiplier values
	movlw	EEPROM4		; multiplier
	call	EEREAD		; sets EEADR
	movf	MULT0,w
	call	EWRITE		; store

	movlw	EEPROM5		; multiplier
	call	EEREAD
	bsf		MULT1,5		; minimum multiplier value
	movf	MULT1,w
	call	EWRITE		; store

	movlw	EEPROM6		; multiplier
	call	EEREAD
	movf	MULT2,w
	call	EWRITE		; store

; increment value
	movlw	EEPROM7		; incrementer
	call	EEREAD
	movf	INCREMENT,w
	call	EWRITE		; store

; scramble value
	movlw	EEPROM10	; scramble
	call	EEREAD
	movf	SCRAMBLE,w
	call	EWRITE		; store

LOOP_1
	call	DELAY
	call	DELAY
	call	DELAY
	bcf		PORTA,3		; ack LED off
	bcf		PORTA,2		; power latch off
	goto	LOOP_1		; program stopped till reset

; .......................................................................

; subroutines

; delay subroutine

DELAY
	movlw	D'70'		; set delay period 
	movwf	VALUE_2		; 
LP_1
	movlw	D'255'		; set delay period value 1 
	movwf	VALUE_1		; VALUE_2 = w
LP_2
	decfsz	VALUE_1,f
	goto	LP_2
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_1
	return

; subroutine to drive bits with 38.46kHz for a 1 and a 0 
; NOTE(35.7kHz if the 'nop' instructions are included. @4-places)

; VALUE,0 sets the level (0 = low, 1 = high)
; 1 is signal burst and then no signal
; 0 is no signal and then signal burst

DRV_BIT 				; drive bits
	btfss	FLAG,0		; wait for flag in interrupt to maintain timing sync
	goto	DRV_BIT
	clrf	FLAG
	btfss	VALUE,0
	goto	SEND_LOW
	clrf	VALUE		; zero
	bsf		PORTA,3		; acknowledge LED on
	movlw	D'14'		; for a 1
	movwf	CYCLE_COUNT
	call	SEND_SIGNAL ; send signal
	clrf	PORTB
LOOP_FLG
	btfss	FLAG,0
	goto	LOOP_FLG	; loop till interrupt flag
	return

SEND_LOW
	bcf		PORTA,3		; ack LED off		
	movlw	D'14'		; for a 0
	movwf	CYCLE_COUNT
	call	SEND_NO_SIGNAL ; send no signal
; now drive signal	
SEND_SIGNAL_1
	movlw	B'00111111'
	movwf	PORTB
;	nop
	movlw	D'03'		; load 38.46kHz counter
	movwf	COUNT38kHz

LOOP_Z
	decfsz	COUNT38kHz,f	; decrease until 0
	goto	LOOP_Z
	movlw	D'03'
	movwf	COUNT38kHz
	clrf	PORTB
;	nop
LOOP_Y
	decfsz	COUNT38kHz,f
	goto	LOOP_Y
	btfss	FLAG,0		; continue till flag set in interrupt
	goto	SEND_SIGNAL_1
	return

; send 38.46kHz SIGNAL
SEND_SIGNAL
	movlw	B'00111111'
	movwf	PORTB
;	nop
	movlw	D'03'		; load 38.46kHz counter
	movwf	COUNT38kHz

LOOP2
	decfsz	COUNT38kHz,f	; decrease until 0
	goto	LOOP2
	movlw	D'03'
	movwf	COUNT38kHz
	clrf	PORTB
;	nop
LOOP3
	decfsz	COUNT38kHz,f
	goto	LOOP3
	decfsz	CYCLE_COUNT,f	; decrease till zero
	goto	SEND_SIGNAL
	return

SEND_NO_SIGNAL ; send 
	clrf	PORTB
	movlw	D'14'
	movwf	COUNT38kHz
	
LOOP4
	decfsz	COUNT38kHz,f
	goto	LOOP4
	decfsz	CYCLE_COUNT,f
	goto	SEND_NO_SIGNAL
	return

; subroutines
; subroutine to read EEPROM memory

EEREAD	
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bsf		EECON1,RD	; read EEPROM
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP0	; select bank 0
	return

; subroutine to write to EEPROM

EWRITE
	bsf		STATUS,RP0	; select bank 1
	movwf	EEDATA		; data register

	bsf		EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITE	
	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank 0 
	return				; value written 

; scramble data send

SEND_SCRAMBLE
	movf	SCRAMBLE,w
	andlw	B'00011111'	; max of 32
	movwf	SCRAMBLE1	; working value
	btfsc	STATUS,Z	; if zero
	goto	ZERO_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 1	
	goto	ONE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 2	
	goto	TWO_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 3	
	goto	THREE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 4	
	goto	FOUR_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 5	
	goto	FIVE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 6	
	goto	SIX_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 7	
	goto	SEVEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 8	
	goto	EIGHT_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 9	
	goto	NINE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 10	
	goto	TEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 11	
	goto	ELEVEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 12	
	goto	TWELVE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 13	
	goto	THIRTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 14	
	goto	FOURTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 15	
	goto	FIVETEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 16	
	goto	SIXTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 17	
	goto	SEVENTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 18
	goto	EIGHTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 19	
	goto	NINETEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 20	
	goto	TWENTY_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 21	
	goto	TWENTYONE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 22	
	goto	TWENTYTWO_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 23	
	goto	TWENTYTHREE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 24	
	goto	TWENTYFOUR_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 25	
	goto	TWENTYFIVE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 26	
	goto	TWENTYSIX_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 27	
	goto	TWENTYSEVEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 28	
	goto	TWENTYEIGHT_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 29	
	goto	TWENTYNINE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 30	
	goto	THIRTY_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 31	
	goto	THIRTYONE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z	; if 32	
	goto	THIRTYTWO_SCRAMBLE

ZERO_SCRAMBLE
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
ONE_SCRAMBLE
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWO_SCRAMBLE
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
THREE_SCRAMBLE
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FOUR_SCRAMBLE
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FIVE_SCRAMBLE
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SIX_SCRAMBLE
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SEVEN_SCRAMBLE
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
EIGHT_SCRAMBLE
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
NINE_SCRAMBLE
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TEN_SCRAMBLE
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
ELEVEN_SCRAMBLE
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWELVE_SCRAMBLE
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
THIRTEEN_SCRAMBLE
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FOURTEEN_SCRAMBLE
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FIVETEEN_SCRAMBLE
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SIXTEEN_SCRAMBLE
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SEVENTEEN_SCRAMBLE
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
EIGHTEEN_SCRAMBLE
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
NINETEEN_SCRAMBLE
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTY_SCRAMBLE
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYONE_SCRAMBLE
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYTWO_SCRAMBLE
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYTHREE_SCRAMBLE
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYFOUR_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	return
TWENTYFIVE_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB2
	return
TWENTYSIX_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB3
	return
TWENTYSEVEN_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB1
	return
TWENTYEIGHT_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB2
	return
TWENTYNINE_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB1
	return
THIRTY_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB0
	return
THIRTYONE_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB3
	return
THIRTYTWO_SCRAMBLE
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB2
	return

; number generator

;	Input:	32 bit initial integer seed in AARGB0, AARGB1, AARGB2, AARGB3

;	X  = (a * X(previous) + c) 

;	X is the result (2**48 bits used) and becomes the seed for next calculation 
;	with multiplier a (24 bit), increment c (8-bit) 	

RANDOM

		movf		RANDB0,w		; random numbers
		movwf		AARGB0
		movf		RANDB1,w
		movwf		AARGB1
		movf		RANDB2,w
		movwf		AARGB2
		movf		RANDB3,w
		movwf		AARGB3

		movf		MULT2,w			; multiplier 
		movwf		BARGB2
		movf		MULT1,w
		movwf		BARGB1
		movf		MULT0,w
		movwf		BARGB0

		call		FXM3224U		; multiply

		movf		INCREMENT,w		; increment value
        addwf       AARGB6,F		; c = incrementer
        btfsc       STATUS,Z		; carry to more significant bytes
        incf        AARGB5,F
		btfsc		STATUS,Z
		incf		AARGB4,F
		btfsc		STATUS,Z
		incf		AARGB3,F
		btfsc       STATUS,Z
        incf        AARGB2,F
		btfsc		STATUS,Z
		incf		AARGB1,F
		btfsc		STATUS,Z
		incf		AARGB0,F

; transfer values to RAND registers
	
		movf		AARGB1,w
		movwf		RANDB5
		movf		AARGB2,w
		movwf		RANDB4

		movf		AARGB3,w
		movwf		RANDB0
		movf		AARGB4,W
		movwf		RANDB1
		movf		AARGB5,W
		movwf		RANDB2
		movf		AARGB6,W
		movwf		RANDB3

; check if value the same as last time
		movlw		EEPROM3		; seed
		call		EEREAD
		xorwf		RANDB3,w
		btfsc		STATUS,Z
		incf		RANDB3,f
; check if last 4-ls bits were the same
		movlw		EEPROM3		; seed
		call		EEREAD
		andlw		H'0F'		; least 4-bits
		movwf		TEMPX
		movf		RANDB3,w
		andlw		H'0F'	
		xorwf		TEMPX,w
		btfsc		STATUS,Z
		incf		RANDB3,f		
		return


;       32x24 Bit Unsigned Fixed Point Multiply 32x24 -> 56

;       Input:  32 bit unsigned fixed point multiplicand in AARGB0, AARGB1,
;               AARGB2, AARGB3

;               24 bit unsigned fixed point multiplier in BARGB0, BARGB1,
;               BARGB2

;       Use:    CALL    FXM3224U

;       Output: 56 bit unsigned fixed point product in AARGB0

;       Result: AARG  <--  AARG x BARG

;       Max Timing:     11+617+2 = 630 clks

;       Min Timing:     11+151 = 162 clks

;       PM: 11+139+1 = 151              DM: 15

FXM3224U
               	clrf   		 	AARGB4          ; clear partial product
               	clrf    		AARGB5
               	clrf    		AARGB6
                movf   			AARGB0,W
                movwf   		TEMPB0
                movf   			AARGB1,W
                movwf   		TEMPB1
                movf   			AARGB2,W
                movwf   		TEMPB2
                movf   			AARGB3,W
                movwf   		TEMPB3

               	call 			UMUL3224L

                retlw           D'00'

; UMUL3224L        macro

;       Max Timing:     2+15+6*25+24+2+7*26+25+2+7*27+26 = 617 clks

;       Min Timing:     2+7*6+5+1+7*6+5+1+7*6+5+6 = 151 clks

;       PM: 31+24+2+25+2+26+2+27 = 139            DM: 15

UMUL3224L       movlw   		D'8'
                movwf   		LOOPCOUNT

LOOPUM3224A
                rrf    		 	BARGB2,F
                btfsc   		STATUS,C
               	goto   			ALUM3224NAP
                decfsz  		LOOPCOUNT,F
               	goto   			LOOPUM3224A

                movwf   		LOOPCOUNT

LOOPUM3224B
                rrf     		BARGB1,F
                btfsc   		STATUS,C
               	goto   			BLUM3224NAP
                decfsz  		LOOPCOUNT,F
               	goto   			LOOPUM3224B

                movwf  			LOOPCOUNT

LOOPUM3224C
                rrf     		BARGB0,F
                btfsc   		STATUS,C
              	goto  	 		CLUM3224NAP
                decfsz  		LOOPCOUNT,F
               	goto  	 		LOOPUM3224C

              	clrf    		AARGB0
               	clrf    		AARGB1
               	clrf    		AARGB2
               	clrf    		AARGB3
                retlw   		D'00'
                
ALUM3224NAP     bcf     		STATUS,C
               	goto  	 		ALUM3224NA
                
BLUM3224NAP     bcf     		STATUS,C
               	goto 	  		BLUM3224NA
                
CLUM3224NAP     bcf     		STATUS,C
              	goto  	 		CLUM3224NA

ALOOPUM3224
                rrf     		BARGB2,F
                btfss   		STATUS,C
                goto   			ALUM3224NA
                movf    		TEMPB3,W
                addwf   		AARGB3,F
                movf            TEMPB2,W
                btfsc           STATUS,C
                incfsz          TEMPB2,W
                addwf           AARGB2,F
                movf            TEMPB1,W
                btfsc           STATUS,C
                incfsz          TEMPB1,W
                addwf           AARGB1,F
                movf            TEMPB0,W
                btfsc           STATUS,C
                incfsz          TEMPB0,W
                addwf           AARGB0,F

ALUM3224NA
                rrf    			AARGB0,F
                rrf    			AARGB1,F
                rrf    			AARGB2,F
                rrf             AARGB3,F
                rrf             AARGB4,F
                decfsz  		LOOPCOUNT,F
                goto    		ALOOPUM3224

                movlw   		D'8'
                movwf   		LOOPCOUNT

BLOOPUM3224
                rrf     		BARGB1,F
                btfss   		STATUS,C
              	goto  	 		BLUM3224NA
                movf    		TEMPB3,W
                addwf   		AARGB3,F
                movf            TEMPB2,W
                btfsc           STATUS,C
                incfsz          TEMPB2,W
                addwf           AARGB2,F
                movf            TEMPB1,W
                btfsc           STATUS,C
                incfsz          TEMPB1,W
                addwf           AARGB1,F
                movf            TEMPB0,W
                btfsc           STATUS,C
                incfsz          TEMPB0,W
                addwf           AARGB0,F

BLUM3224NA
                rrf    			AARGB0,F
                rrf    			AARGB1,F
                rrf    			AARGB2,F
                rrf             AARGB3,F
                rrf             AARGB4,F
                rrf             AARGB5,F
                decfsz  		LOOPCOUNT,F
              	goto   			BLOOPUM3224

                movlw   		D'8'
                movwf   		LOOPCOUNT

CLOOPUM3224
                rrf     		BARGB0,F
                btfss   		STATUS,C
               	goto 	  		CLUM3224NA
                movf    		TEMPB3,W
                addwf   		AARGB3,F
                movf            TEMPB2,W
                btfsc           STATUS,C
                incfsz          TEMPB2,W
                addwf           AARGB2,F
                movf            TEMPB1,W
                btfsc           STATUS,C
                incfsz          TEMPB1,W
                addwf           AARGB1,F
                movf            TEMPB0,W
                btfsc           STATUS,C
                incfsz          TEMPB0,W
                addwf           AARGB0,F

CLUM3224NA
                rrf  			AARGB0,F
                rrf    			AARGB1,F
                rrf    			AARGB2,F
                rrf    			AARGB3,F
                rrf             AARGB4,F
                rrf             AARGB5,F
                rrf             AARGB6,F
                decfsz  		LOOPCOUNT,F
                goto   			CLOOPUM3224
				return	
    	            
; end of program
	end		 	
