;*******************************************************************************
;* File: cdata.asm   
;*******************************************************************************
; Written by Mauro Grassi Copyright Mauro Grassi 2005-2007
;
;
;
;
; begun circa 2001
; this update 7 July 2007 
; this update 3 August 2007 
; this update 8 August 2007 for SiliconChip, by Mauro Grassi.
; latest update 8 November 2007, fixed problem with Jaycar LCD.
;
; major bug in writing to IDE registers fixed 5 August 2007
; basically you must write in the proper sequence CS must come low first 
; before RD/WR come low then these must go high before CS goes high 
; basically some drives don't like any other sequence of events.
; CD player using AVR and PC CD rom
; ATAPI interface
; fixed IR system now works, Mauro Grassi 7 August 2007
; the IR module must be connected through a 100R resistor and a 10uF capacitor
; must be placed between its power terminals, ie pins 2 and 3
; pin 1 is the output which goes straight through to the input of AVR controller
;------------------register definitions-----------------------------------------
.include "8515def.inc"  ; chip definitions...

.def volume     =r11    ; current volume level (up to 16 volume levels?)
.def sstate     =r12    ; slave state of the machine...
.def mstate     =r9     ; master state of the machine...

.def mnexttrack =r8     ; master next track register
.def snexttrack =r4     ; slave next track register
.def drive      =r10    ; bit 1=0 master selected 1=slave selected...
                        ; drive is either 00=Master selected
                        ; or 02=slave selected
.def config     =r13    ; configuration register...
                        ; Bits 1&0 = 00 no drives detected
                        ;            01 one ATAPI detected master
                        ;            10 two ATAPI detected master and slave...
                        ;            11 reserved...
                        ; Bit 2 = 1 when MASTER is present
                        ; Bit 3 = 1 when SLAVE  is present

.def mode       =r14    ; when bit0=0 messages are displayed for commands...
                        ; when bit0=1 messages are not displayed for commands...
                        ;      bit7 selects ASCII or Hex mode for debugging.....
                        ; when bit1=0 repeat is off...
                        ; when bit1=1 repeat is on....
                        ; when bit2=0 random is off =1 random is on...
.def error      =r15    ; this is the last error register...

.def temp0      =r16
.def temp1      =r17
.def temp2      =r18
.def pointerlo  =r23
.def pointerhi  =r24
.def key        =r20
.def balance    =r5     ; holds the (volume) balance
.def random     =r3
;------------------ IR registers------------------------------------------------
.def ir_state	  =R1	    ;ir receive state
.def ir_bit_cnt =R2     ;ir bit count
.def ir_sys     =R3     ;ir received system bits
.def ir_cmd     =R19    ;ir received command bits
.def ir_code	  =R25	  ;last received command bits
.def irate_tmr	=R6	    ;interrupt rate timer
.def tick_tmr   =R7     ;general purpose 1ms timer
.def A          =R16    ;scratch
.def B          =R17    ;scratch
.def bflags     =R20    ;holds status
.def ir_timer	  =R21    ;ir bit timer
.equ IR_RXD     =2      ;infrared receiver input (bit 2 of PORTD)
;----------------- fpudiv registers (multiplexed)-------------------------------
.def rem1       =r28
.def rem2       =r29    ; 16-bit remainder
.def icounter   =r25    ; loop counter
.def byte1      =r21
.def byte2      =r22
.def byte3      =r23
.def byte4      =r24    ; 32-bit dividend
.def div1       =r19
.def div2       =r20    ; 16-bit divisor
;-------------------------------------------------------------------------------
; bflags
.equ	ir_rpt		=1
.equ	ir_rdy		=0	;1=infrared data available
;-------------------------------------------------------------------------------
; System PIN definitions... PORTS C and D are used PORTA=DATA PORT
;-------------------------------------------------------------------------------
;--- port D --------------------------------------------------------------------

.equ RD  =$7f                   ; RD signal
.equ WR  =$bf                   ; WR signal
.equ LCD =$df                   ; E of LCD
.equ RS  =$ef                   ; RS of LCD 
.equ RES =$f7                   ; IDE RESET line

;-------------------------------------------------------------------------------
.equ ALLHI =$ff                 ; universal all high mask
.equ PDMASK=$fd                 ; port D mask for output LEDs and so on...
;-------------------------------------------------------------------------------

;--- port B --------------------------------------------------------------------
.equ A0  =$fe
.equ A1  =$fd
.equ A2  =$fb
.equ CS0 =$f7
.equ CS1 =$ef                   ; IDE addressing lines...

.equ MAXSTATES=$20              ; maximum number of machine states
.equ EMPTY  =$01
.equ ERROR  =$02
.equ OK     =$03
.equ OK2    =$04
.equ NOTRANSITION=$7f
.equ LASTSTATE=$7e
;-------------------------------------------------------------------------------
; code segment begins here
;-------------------------------------------------------------------------------
.cseg
.org $0000
        rjmp    reset    ; reset vector
        reti
        reti
        reti
        reti
        reti
        reti
        reti
        reti
        reti
        reti
        reti
        reti
        reti
;-------------------------------------------------------------------------------
;-------------------------- UART section ---------------------------------------
inituart:
        ; initialize the UART to 8 bit no parity 1 stop bit and 115,200 baud
        ldi     temp0,$10
        out     UCR,temp0       ; bit 4=enable receiver
                                ; bit 3=enable transmitter...
        ldi     temp0,$03
        out     UBRR,temp0      ; 115200 baud rate @ 7.3728 MHz
        in      temp0,UDR
        ret

_igetchar:
        in      temp0,UDR       ; flush
getchar:
getkey:
        ; block till char received and return in temp0
        in      temp0,USR
        bst     temp0,3         ; overrun detection...
        brts    _igetchar       ; flush and receive new character on overrun.
        bst     temp0,7
        brtc    getchar
        in      temp0,UDR
        ret
;-------------------------------------------------------------------------------
;      RESET bootstrap - execution begins here
;-------------------------------------------------------------------------------
getinput:
        rcall   get_ir_key
        brcs    getinp0
        ldi     temp0,NOTRANSITION
        clc
        ret
getinp0:
        ; always present key functions must be processed here, the state thus
        ; does not change...
        ; Everpresent keys:
        ; Vol Up and Down and Line=swap drives.
        ;***********************************************************************
        cpi     temp0,$1b       ; line button swaps the drives...
        brne    getinp1
        
        rcall   swapdrivesstate
        rcall   shortdelay
        ldi     temp0,NOTRANSITION
        sec
        ret
getinp1:
        cpi     temp0,$4b
        brne    getinp2
        
        rcall   volupcommand
        brcs    getinp11
        ldi     temp0,ERROR
        sec
        ret
getinp11:
        ldi     temp0,NOTRANSITION
        sec
        ret
getinp2:
        cpi     temp0,$4d
        brne    getinp3
        rcall   voldowncommand
        brcs    getinp11
        ldi     temp0,ERROR
        sec
        ret
getinp3:
        cpi     temp0,$48
        brne    getinp4
        
        rcall   balupcommand
        brcs    getinp11
        ldi     temp0,ERROR
        sec
        ret
getinp4:
        cpi     temp0,$50
        brne    getinp5
        
        rcall   baldowncommand
        brcs    getinp11
        ldi     temp0,ERROR
        sec
        ret
getinp5:
        cpi     temp0,'G'
        brne    getinp6
        
        mov     temp0,mode
        ldi     temp1,$F9       ; bits 1 and 2 are the relevant bits...
        and     mode,temp1
        ldi     temp1,$06
        and     temp0,temp1
        lsr     temp0
        inc     temp0
        cpi     temp0,$03
        brne    getinp59
        clr     temp0
getinp59:
        lsl     temp0
        or      mode,temp0
        rcall   shortdelay
        ldi     temp0,NOTRANSITION
        sec
        ret        
getinp6:
        cpi     temp0,'m'
        brne    getinp7
        rcall   mutecommand
        brcs    getinp11
        ldi     temp0,ERROR
        sec
        ret
getinp7:
        sec
        ret
;-------------------------------------------------------------------------------
reset:
        cli
        ldi     temp0, low(RAMEND)
        out     SPL,temp0
        ldi     temp0, high(RAMEND)
        out     SPH,temp0            ; initialise stack pointer
        ldi     temp0,ALLHI
        out     PORTB,temp0
        out     PORTA,temp0
        out     PORTC,temp0
        ldi     temp0,ALLHI& PDMASK
        or      temp0,drive
        out     PORTD,temp0
        ldi     temp0,$fa
        out     DDRD,temp0           ; PORTD is (0) IOIOOOOO (7)
        ldi     temp0,$1f
        out     DDRB,temp0           ; PORTB is (0) OOOOOIII (7)
        ldi     temp0,$00
        out     DDRA,temp0
        out     DDRC,temp0

        ; we now setup counter 1 to count up clocks assume clock 7.3728MHz...
        ldi     temp0,$00 
        out     TCCR1A,temp0
        ldi     temp0,$02
        out     TCCR1B,temp0        

        ; write to set clock source CK/8 see page 120 of datasheet ATMEGA8515
        ; global variables need to be initialized now...
        ldi     temp0,$01
        out     TCCR0,temp0     ; set up the TIMER0 for normal mode
        
        clr     mode
        clr     drive
        clr     error
        
        clr     temp0
        ldi     XH,$02
        ldi     XL,$2D
        st      X,temp0         ; this is for the muting...
        
        ldi     temp0,$32
        mov     balance,temp0     ; $32=50%
        
        ldi     temp0,$80
        mov     volume,temp0    ; default volume level at 50%
      
        ldi     temp0,$01
        mov     mnexttrack,temp0
        mov     snexttrack,temp0
        rcall   ee_init         ; initialise EEPROM module
        rcall   inituart        ; initialise UART        
        rcall   initide         ; initialise IDE system
        rcall   initlcd         ; initialise LCD module
        
        ldi     temp0,$01
        rcall   writelcdcontrol
        ldi     ZL,low(messagewelcome)
        ldi     ZH,high(messagewelcome)
        rcall   printmessage
        ldi     temp0,$C0
        rcall   writelcdcontrol
        ldi     ZL,low(messagewelcome2)
        ldi     ZH,high(messagewelcome2)
        rcall   printmessage
        rcall   longdelay       ; print welcome screen...

        rcall   setupremote     ; setup remote control if enabled 
                                ; by pushbutton on start up
        rcall   setupide        ; setup drives connected auto detect
        
        clr     mstate          ; initial master state=0x00
        clr     sstate          ; initial slave state =0x00      
        
        rjmp    machine         ; go to main finite state machine...

messagewelcome:
.db     'C','D',' ','P','l','a','y','B','a','c','k',' ','v','2','.','9',$80,$80
messagewelcome2:
.db     'W','e','l','c','o','m','e',' ',' ',' ',' ',' ',' ',' ',' ',' ',$80,$80

;-------------------------------------------------------------------------------
;---------------------------begin delay system----------------------------------
longdelay:
        ; delay for 1.4 seconds...
        push    temp0
        push    temp1
        push    temp2
        
        clr     temp0
longd1:
        
        push    temp0
        in      temp2,SREG
        cli
        ldi     temp0,$00
        out     TCNT1H,temp0
        out     TCNT1L,temp0
        out     SREG,temp2

longd2:
        in      temp2,SREG
        cli 
        in      temp0,TCNT1L
        in      temp1,TCNT1H
        out     SREG,temp2
        
        cpi     temp1,$7F           ; 8000hex at 7.3728Mhz/8 is 35.5ms
        brlo    longd2
        
        pop     temp0
        inc     temp0
        cpi     temp0,$28        ; 1.4sec= 40 x 35.5 ms roughly and 40dec=28 hex
        brlo    longd1
        pop     temp2
        pop     temp1
        pop     temp0
        ret

shortdelay:
        ; delay for 0.4 second...
        push    temp0
        push    temp1
        push    temp2
        
        clr     temp0
shortd1:
        push    temp0
        in      temp2,SREG
        cli
        ldi     temp0,$00
        out     TCNT1H,temp0
        out     TCNT1L,temp0
        out     SREG,temp2

shortd2:
        in      temp2,SREG
        cli 
        in      temp0,TCNT1L
        in      temp1,TCNT1H
        out     SREG,temp2
        
        cpi     temp1,$7F           ; 8000hex at 7.3728Mhz/8 is 35.5ms
        brlo    shortd2
        
        pop     temp0
        inc     temp0
        cpi     temp0,$0B        ; 0.4s =11 x 35.5 ms roughly and 14dec=0Ehex
        brlo    shortd1
        pop     temp2
        pop     temp1
        pop     temp0
        ret

shortestdelay:
        ; delay for 0.2 second...
        push    temp0
        push    temp1
        push    temp2
        
        clr     temp0
shortestd1:
        push    temp0
        in      temp2,SREG
        cli
        ldi     temp0,$00
        out     TCNT1H,temp0
        out     TCNT1L,temp0
        out     SREG,temp2

shortestd2:
        in      temp2,SREG
        cli 
        in      temp0,TCNT1L
        in      temp1,TCNT1H
        out     SREG,temp2
        
        cpi     temp1,$7F           ; 8000hex at 7.3728Mhz/8 is 35.5ms
        brlo    shortestd2
        
        pop     temp0
        inc     temp0
        cpi     temp0,$05        ; 0.2s =5 x 35.5 ms roughly and 5dec=05hex
        brlo    shortestd1
        pop     temp2
        pop     temp1
        pop     temp0
        ret

delay10ms:
        ; delay for 0.01 second...
        push    temp0
        push    temp1
        push    temp2
        
        clr     temp0
del10msd1:
        push    temp0
        in      temp2,SREG
        cli
        ldi     temp0,$00
        out     TCNT1H,temp0
        out     TCNT1L,temp0
        out     SREG,temp2

del10msd2:
        in      temp2,SREG
        cli 
        in      temp0,TCNT1L
        in      temp1,TCNT1H
        out     SREG,temp2
        
        cpi     temp1,$20           ; 20hex at 7.3728Mhz/8 is 8.5ms
        brlo    del10msd2
        
        pop     temp0
        inc     temp0
        cpi     temp0,$01        ; 0.01s =1 x 8.5 ms roughly and 1dec=01hex
        brlo    del10msd1
        pop     temp2
        pop     temp1
        pop     temp0
        ret

;-------end delay system--------------------------------------------------------
; subroutine mul88
; multiply unsigned 8 bit by 8 bit giving 16 bit result
; input: div1, div2 and result in byte3, byte 4
;-------------------------------------------------------------------------------
mul88:
        ldi   icounter,$08
        clr   byte1
        clr   byte2
        clr   byte3
        clr   byte4
        clr   rem2
        clc
mul881:
        mov   temp0,div1
        andi  temp0,$01
        breq  mul882
        ; add rem2:div2 to byte3,byte4
        add   byte4,div2
        adc   byte3,rem2
mul882:
        ; now multiply by 2
        lsl   div2
        rol   rem2
        
        ror   div1
        dec   icounter
        brne  mul881
        ret
;-------------------------------------------------------------------------------
; subroutine = fpudiv3216
;              to perform a 32-bit by 16-bit unsigned binary division.
; input =  byte 1,2,3,4 = 32-bit dividend
;          div  1,2     = 16-bit divisor
; output = rem  1,2     = 16-bit remainder 
;          byte 1,2,3,4 = 32-bit quotient
; carry flag
;                       = 1 if overflow
;                       = 0 if no overflow
;-------------------------------------------------------------------------------
fpudiv3216:
        ldi     icounter,$20    ; loop counter (32-bits)...
        clr     rem1
        clr     rem2
        clc        
fpu_loop:
        rol     byte4
        rol     byte3
        rol     byte2
        rol     byte1
        rol     rem2
        rol     rem1
        brcs    fpu_subt
        rcall   compare16
        brcs    fpu_subt        
fpu_n1:
        dec     icounter        ; carry = 0
        brne    fpu_loop
        rjmp    fpu_next
fpu_subt:
        rcall   subtract16
        sec                     ; to be shifted into result
        rjmp    fpu_n1
fpu_next:
        rol     byte4
        rol     byte3
        rol     byte2
        rol     byte1
        rcall   compare16
        brcc    fpu_next2
                               ; carry = 0
fpu_skip2:
        rcall   subtract16
        inc     byte4
        brne    fpu_next2
        inc     byte3
        brne    fpu_next2
        inc     byte2
        brne    fpu_next2
        inc     byte1
fpu_next2:
        clc
        ret
compare16:
        ; carry=1 means >=0 carry=0 means <0
        cp      rem1,div1
        brlo    comp161
        brne    comp162
        cp      rem2,div2
        brlo    comp161
comp162:
        sec
        ret
comp161:
        clc
        ret
subtract16:
        sub     rem2,div2
        sbc     rem1,div1
        ret
disadecimal:
        push    temp1
        mov     byte4,temp0
        clr     byte3
        clr     byte2
        clr     byte1
        clr     div1
        ldi     div2,$0a
        ldi     temp1,$03
        ldi     ZL,$00
        ldi     ZH,$01
disadec1:
        push    temp1
        rcall   fpudiv3216
        mov     temp0,rem2
        rcall   displayfix
        st      -Z,temp0
        pop     temp1
        dec     temp1
        brne    disadec1
        
        pop     temp1
        adiw    ZL,$03
        sub     ZL,temp1
        clr     temp0
        sbc     ZH,temp0
_disad2:
        ld      temp0,Z+
        push    temp1
        rcall   writelcd
        pop     temp1
        dec     temp1
        brne    _disad2
        ret
;--------------------------- end of fpudiv routines ----------------------------
;--------------------------- LCD drivers ---------------------------------------
pause160ms:
        push    temp0
        ldi     temp0,$08
pause16ms0:        
        push    temp0
        rcall   delay10ms
        pop     temp0
        dec     temp0
        brne    pause16ms0
        pop     temp0
        ret

initlcd:
        ; initialize the LCD screen
        rcall   pause160ms
        ldi     temp0,$ff
        out     PORTB,temp0
        ldi     temp0,RS
        out     PORTD,temp0
        ldi     temp0,$38
        out     PORTA,temp0
        ldi     temp0,$ff
        out     DDRA,temp0
        ldi     temp0,$fa
        out     DDRD,temp0
        ldi     temp0, RS & LCD & WR
        out     PORTD,temp0
        nop
        ldi     temp0,RS & WR
        out     PORTD,temp0
        rcall   delay10ms
        ldi     temp0,$0C
        out     PORTA,temp0
        ldi     temp0,RS & WR & LCD
        out     PORTD,temp0
        nop
        ldi     temp0,RS & WR
        out     PORTD,temp0
        rcall   delay10ms
        ldi     temp0,$01
        out     PORTA,temp0
        ldi     temp0, RS & WR & LCD
        out     PORTD,temp0
        nop
        ldi     temp0,RS & WR
        out     PORTD,temp0
        rcall   delay10ms
        ldi     temp0,$06
        out     PORTA,temp0
        ldi     temp0,RS & WR & LCD
        out     PORTD,temp0
        nop
        ldi     temp0,$ff
        out     PORTD,temp0
        ldi     temp0,$00
        out     DDRA,temp0
        rcall   longdelay
        ret

writelcdcontrol:
        ; write temp0 control code to LCD screen
        push    temp2
        ldi     temp2,ALLHI
        out     PORTB,temp2
        rcall   waitlcd
        ldi     temp2,RS
        rjmp    _wlcd1
writelcd:
        ; write temp0 data code to LCD screen
        push    temp2
        cli
        ldi     temp2,ALLHI
        out     PORTB,temp2
        rcall   waitlcd
        ldi     temp2,ALLHI
_wlcd1:
        out     PORTA,temp0
        ldi     temp1,$ff
        out     DDRA,temp1      ; set PORTA as output
        andi    temp2,WR & PDMASK
        or      temp2,drive
        out     PORTD,temp2     ; send enable, write and rs signals...
        cbi     PORTD,5         ; put LCD enable low
        sbi     PORTD,5         ; put it high
        ldi     temp1,$00
        out     DDRA,temp1      ; set PORTA as input
        pop     temp2
        ret

readlcdcontrol:
        ; read temp0 control code from LCD screen
        push    temp2
        ldi     temp2,ALLHI
        out     PORTB,temp2
        ldi     temp2,RS
        rjmp    _rlcd1
readlcd:
        push    temp2
        ; read temp0 data code from LCD screen
        ldi     temp2,ALLHI
        out     PORTB,temp2
        ldi     temp2,ALLHI
_rlcd1:
        andi    temp2,LCD & PDMASK
        or      temp2,drive
        out     PORTD,temp2     ; send enable, write and rs signals...
        nop
        nop
        nop
        nop
        in      temp0,PINA      ; read in data byte
        nop
        nop
        nop
        nop
        ldi     temp2,ALLHI & PDMASK
        or      temp2,drive
        out     PORTD,temp2     ; done!
        pop     temp2
        ret
        
waitlcd:
        ; wait till LCD is not busy.
        push    temp2
        ldi     temp2,ALLHI
        out     PORTB,temp2
        push    temp1
        push    temp0
_waitlcd1:
        rcall   readlcdcontrol
        andi    temp0,$80
        brne    _waitlcd1
        pop     temp0
        pop     temp1
        pop     temp2
        ret
;------------------- General Kernel Routines -----------------------------------
disa:
        ; display hexadecimal byte to LCD screen, data in temp0
        push    temp1
        push    temp0
        push    temp0
        swap    temp0
        rcall   displayfix
        rcall   writelcd
        pop     temp0
        rcall   displayfix
        rcall   writelcd
        pop     temp0
        pop     temp1
        ret
displayfix:
        ; internal use to convert 0x00-0x0F to ASCII digit operates on temp0
        andi    temp0,$0f
        ldi     temp1,$30
        add     temp0,temp1
        cpi     temp0,$3a
        brlo    _disfix1
        ldi     temp1,$07
        add     temp0,temp1
_disfix1:
        ret
;-------------------------------------------------------------------------------
;*******************************************************************************
;*
;* Initialise EEPROM
;*
;*******************************************************************************
ee_init:
	ret
;*******************************************************************************
;*
;* EEPROM random read/write routines
;*
;* Entry: XH:XL = address and temp0=data returned... address is 0000-0200hex
;* by Mauro Grassi August 2007
;*******************************************************************************
ee_read:
	sbic	EECR,EEWE
	rjmp	ee_read			;loop until last cycle complete

  out EEARH,XH      ;write address
  out EEARL,XL
	
	sbi	EECR,EERE	  	;set read strobe
	
  in  temp0,EEDR    ;get data
	ret    				    ;(read takes 4 clk cycles)
	
; Writes typically take 2.5ms @ 5V
; XH:XL=address and temp0=data address is 0000-0200hex

ee_write:
  push  temp2
ee_wr0:
	sbic	EECR,EEWE
	rjmp	ee_wr0   	;loop until last cycle complete

	out	EEARH,XH		;write address
  out EEARL,XL
	out	EEDR,temp0  ;write data
	
  in  temp2,SREG
	cli	       			;interrupts off for the next bit...
	sbi	EECR,EEMWE	;set master write enable
	sbi	EECR,EEWE		;set write strobe
  out SREG,temp2
  pop  temp2
	ret
;*******************************************************************************
;* Infrared receive state machine (must be called every 64us)
;* by Peter Smith, borrowed by Mauro Grassi (6 August 2007)
;* from April 2006 Studio Series Preamp remote control routines...
;*******************************************************************************
ir_receive:
	inc	ir_timer		;bump ir protocol (bit) timer

; Start (resume) ir decode in the correct state.

	ldi	ZH,high(rc5_jmp_tbl)
	ldi	ZL,low(rc5_jmp_tbl)

	add	ZL,ir_state		;state number becomes index
	clr	A
	adc	ZH,A		     	;find the correct rountine...
	ijmp			  	    ; and run it
		
rc5_jmp_tbl:
	rjmp	rc5_s0
	rjmp	rc5_s1
	rjmp	rc5_s2
	rjmp	rc5_s3
	rjmp	rc5_s4
	rjmp	rc5_s5
	rjmp	rc5_s6
	rjmp	rc5_s7
;*******************************************************************************
;
; RC5 State 0
;
; Wait for line idle (high)
;
;*******************************************************************************

rc5_s0:
	sbis	PIND,IR_RXD		 ;wait for line high
	rjmp	exit_state
			
	clr	ir_timer		     ;init timer
	rjmp	next_state		 ;move to state 1 & exit

;*******************************************************************************
;
; RC5 State 1
;
; Establish line is idle (high) for 5ms before waiting for a start bit
;
;*******************************************************************************

rc5_s1:
	sbis	PIND,IR_RXD		;skip if line high
	rjmp	reset_state		;else reset & exit
	
	cpi	ir_timer,78  		;idle high for 5ms?
	brlo	r1_0    			;keep waiting if not
	rjmp	next_state		;else move to state 2 & exit

r1_0:	
    rjmp	exit_state
	
;*******************************************************************************
;
; RC5 State 2
; 
; Wait (forever) for start bit
;
;*******************************************************************************

rc5_s2:
	sbic	PIND,IR_RXD		;skip if start bit detected
	rjmp	exit_state

r2_0:	clr	ir_timer		;init timer
	rjmp	next_state		;move to state 3 & exit

;**************************************************************************
;
; RC5 State 3
;
; Wait for rising edge of first start bit
;
;**************************************************************************

rc5_s3:
	sbis	PIND,IR_RXD		;skip if line returned high
	rjmp	r3_0			    ;else go check for timeout

	clr	ir_timer		    ;init timer
	rjmp	next_state		;move to state 4 & exit

r3_0:	
  cpi	ir_timer,31   	;more than 2ms?
	brlo	exit_state		;exit to keep sampling
	rjmp	reset_state		;else reset

;**************************************************************************
;
; RC5 State 4
;
; Wait for trailing edge of second start bit to synchronise timing
;
;**************************************************************************

rc5_s4:
	sbic	PIND,IR_RXD		;skip when line goes low
	rjmp	r4_0			    ;else go check for timeout
	
	ldi	A,12		      	;init bit count
	mov	ir_bit_cnt,A
	clr	ir_sys			    ;zero the bit store
	clr	ir_cmd	

	clr	ir_timer		    ;init timer
	rjmp	next_state		;more to state 5 & exit

r4_0:	
  cpi	ir_timer,31	    ;more than 2ms?
	brlo	r4_1			    ;skip to keep sampling
	rjmp	reset_state		;timeout, reset
r4_1:	rjmp	exit_state

;**************************************************************************
;
; RC5 State 5
;
; Sample the line at 1/4 bit time and store result
;
;**************************************************************************

rc5_s5:
	cpi	ir_timer,21	 	;3/4 bit time from last edge (1.34ms)?
	brlo	r5_1			  ;keep waiting if not

	inc	ir_state		  ;assume will be low, next state = 6
	sbic	PIND,IR_RXD	;skip if sampled low
	rjmp	r5_2
	
	clc				        ;to shift in a low bit
r5_0:	
  rol	ir_cmd
	rol	ir_sys

	clr	ir_timer		  ;init timer
r5_1:	
  rjmp	exit_state

r5_2:	
  inc	ir_state		  ;sampled high, next state = 7
	sec
	rjmp	r5_0			  ;go shift in the high bit

;**************************************************************************
;
; RC5 State 6
;
; Bit was a low, so wait for rising edge to syncronize timing
;
;**************************************************************************

rc5_s6:
	sbis	PIND,IR_RXD		;skip if sampled high
	rjmp	r6_3			    ;else go check for timeout
	
r6_0:	
  dec	ir_bit_cnt	;done all bits?
	brne	r6_2			;not yet...
	
	mov	A,ir_cmd		;get all the bits...
	rol	A
	rol	ir_sys			;...in the right places
	rol	A
	rol	ir_sys

	bst	ir_sys,5		;move toggle bit...
	bld	ir_cmd,7		;to command byte MSB
	
	clt
	bld	ir_sys,5
	bld	ir_cmd,6		       ;clear unused bits

	cbr	bflags,1<<ir_rpt
	cp	ir_code,ir_cmd		;same code as last time?
	brne	r6_1			      ;skip if not
	sbr	bflags,1<<ir_rpt	;else flag key repeat
	
r6_1:	
  mov	ir_code,ir_cmd	
	sbr	bflags,1<<ir_rdy	;mail to the foreground
	rjmp	reset_state		  ;reset machine & exit
	
r6_2:	
  ldi	A,5		     	      ;more to do
	mov	ir_state,A		    ;loop back to get another bit
	clr	ir_timer
	rjmp	exit_state

r6_3:	
  cpi	ir_timer,35	    ;5/4 bit time from edge (2.2ms)?
	brlo	exit_state		;keep sampling if not
	rjmp	reset_state		;else timeout waiting for middle edge
;**************************************************************************
;
; RC5 State 7
;
; Bit was a high, so wait for falling edge to syncronize timing
;
;**************************************************************************

rc5_s7:
	sbic	PIND,IR_RXD		;skip if sampled low
	rjmp	r6_3			    ;else go check for timeout
	rjmp	r6_0
;**************************************************************************
; Common exit stuff...

next_state:
	inc	ir_state	    	;bump to next state
	rjmp	exit_state

reset_state:
	clr	ir_state		   ;reset machine

exit_state:
	ret
;-------------------------------------------------------------------------------
; IR receive routine by Mauro Grassi 7 August 2007
; timed for a 7.3728MHz crystal
;-------------------------------------------------------------------------------
iget_ir_key:
        ; wait for an IR command to be received...
        ; timeout after 0.75 second...
        ; returns Carry if there is a valid character in ir_code and return
        ; translated code in temp0
        ; returns No Carry when it timed out after 0.5s of inactivity on IR line
        clr     XL             ; clear the timeout count
        clr     XH             ; clear the timeout count 
        clr     bflags
        clr     ir_state       ; initialize IR state machine
_gik0:        
        push    temp2
        in      temp2,SREG
        cli
        clr     temp0
        out     TCNT1H,temp0
        out     TCNT1L,temp0
        out     SREG,temp2      ; atomic write of TIMER1 value (16 bit)
                                ; NB: when writing to TIMER1
                               ; you must write High byte first
        pop     temp2
_gik1:        
        push    temp2
        in      temp2,SREG
        cli
        in      temp0,TCNT1L
        in      temp1,TCNT1H
        out     SREG,temp2      ; atomic read of TIMER1 value (16 bit)
                                ; NB: when reading from TIMER1
                                ; you must read Low byte first
        pop     temp2
        cpi     temp1,$00
        brlo    _gik1
        cpi     temp0,$3B
        brlo    _gik1           ;at 7.3728MHz, we get a period of around 64us
                                ; with this value 003B=59decimal
                                ; or 59*8/7372800= 64us (prescaler setting x8)
        push    XL
        push    XH
        rcall   ir_receive      ; must be called every 64us
        pop     XH
        pop     XL
        adiw    XL,$01          ; add one to the timeout count in slots of 64us
                                ; so 0.75 sec = 11718 decimal = 2DC6 hex
        cpi     XH,$2D
        brlo    _gik2
        cpi     XL,$6C
        brlo    _gik2           ; continue if not timed out after 1 second...
        
        ; here we have timed out... so wrap up the routine, abort with no carry
        clr     bflags
        clc
        ret
_gik2:        
        mov     temp0,bflags
        andi    temp0,$01       ; check if character received through IR...
        breq    _gik0           ; if not then loop again...
        
        ; here we have completed receiving a character so abort with CARRY set
        ; and return code in temp0 too
        
        mov     temp0,ir_code
        andi    temp0,$7F       ; remove toggle bit from ir_code
                                ; and now translate the code from
                                ; the table below and put result in temp0
        sec
        ret

get_ir_key:
        rcall   iget_ir_key
        brcs    getir0
        clc
        ret        
getir0:
        mov     XL,temp0
        clr     XH
        rcall   ee_read         ; convert key code to remote control table codes...
        sec
        ret

printmessage:
        ; print message at FLASH ZH:ZL terminated by $80 character
        lsl     ZL
        rol     ZH      ; convert to word address.
_pm1:
        lpm
        mov     temp0,r0
        cpi     temp0,$80
        breq    _pmexit
        rcall   writelcd
        adiw    ZL,$01
        rjmp    _pm1
_pmexit:
        ret

;*******************************************************************************
; setupremote subroutine
; this either returns immediately when port d pin IRtx =1 
; or if port d pin IRTx=0 then setups the remote control codes into EEPROM......
; this is a function to learn the remote control...
;*******************************************************************************
setupremote:
       sbic     PIND,IR_RXD
       ret                      ; do nothing if line is high
       ; do something if line low
       ldi      temp0,$01
       rcall    writelcdcontrol
       ldi      ZL,low(messagesetupremote)
       ldi      ZH,high(messagesetupremote)
       rcall    printmessage
       rcall    longdelay
    
       ; first clear the EEPROM to the code FFhex which is nop code...
       ; we must clear 128 bytes to FFhex which is nop code...
       ldi      XH,$00
       ldi      XL,$00
       ldi      temp0,$FF
srem0:       
       rcall    ee_write
       adiw     XL,$01
       cpi      XL,$80
       brlo     srem0
       ; done!
       
       ; now we ask the user to press the buttons as they are displayed...
       ldi      temp0,$01
       rcall    writelcdcontrol
       ldi      ZL,low(messagepress)
       ldi      ZH,high(messagepress)
       rcall    printmessage
       
       ldi      temp2,$19         ; =25 commands to be defined...
       ldi      ZL,low(messagebuttons)
       ldi      ZH,high(messagebuttons)
       lsl      ZL
       rol      ZH                  ; convert to word address
srem00:       
       push     temp2
       ldi      temp2,$06         ; write 6 bytes to screen now
       ldi      temp0,$86
       rcall    writelcdcontrol
srem1:
       lpm
       mov      temp0,r0
       push     ZL
       push     ZH
       rcall    writelcd
       pop      ZH
       pop      ZL
       adiw     ZL,$01
       dec      temp2
       brne     srem1
       
       push     ZL
       push     ZH
srem2:       
       rcall    iget_ir_key
       brcc     srem2
       ; now temp0= $7F and ir_code pressed...
       ; so now we must use temp0 as the address to write the code of the key
       pop      ZH
       pop      ZL
       mov      XL,temp0
       clr      XH
       lpm
       mov      temp0,r0
       rcall    ee_write        ; write the code...
       rcall    longdelay
       adiw     ZL,$02          ; go to next code...
       pop      temp2
       dec      temp2
       brne     srem00
       ; finished...
       
       ldi      temp0,$01
       rcall    writelcdcontrol
       ldi      ZL,low(messagedone)
       ldi      ZH,high(messagedone)
       rcall    printmessage
       rcall    longdelay
       ret
messagesetupremote:
.db    'S','e','t','u','p',' ','R','e','m','o','t','e',' ',' ',' ',' ',$80,$80
messagedone:
.db    'S','e','t','u','p',' ','F','i','n','i','s','h','e','d',' ',' ',$80,$80
messagepress:
.db    'P','r','e','s','s',' ',$80,$80
messagebuttons:
.db     'U','N','U','S','E','D','T',' '
.db     'C','L','S','O','P','N','t',' '
.db     'V','o','l',' ','U','p',$4b,' '
.db     'V','o','l',' ','D','n',$4d,' '
.db     'C','h','n',' ','U','p',$48,' '
.db     'C','h','n',' ','D','n',$50,' '
.db     'M','u','t','e',' ',' ','m',' '
.db     'P','l','a','y',' ',' ','Z',' '
.db     'F','a','s','t',' ','F','F',' '
.db     'R','e','w','i','n','d','R',' '
.db     'S','t','o','p',' ',' ','S',' '
.db     'P','a','u','s','e',' ','p',' '
.db     'R','e','c','o','r','d','G',' '
.db     'M','A','/','S','L',' ',$1B,' '
.db     'P','o','w','e','r',' ','O',' '
.db     'C','h','n',' ','0',' ','0',' '
.db     'C','h','n',' ','1',' ','1',' '
.db     'C','h','n',' ','2',' ','2',' '
.db     'C','h','n',' ','3',' ','3',' '
.db     'C','h','n',' ','4',' ','4',' '
.db     'C','h','n',' ','5',' ','5',' '
.db     'C','h','n',' ','6',' ','6',' '
.db     'C','h','n',' ','7',' ','7',' '
.db     'C','h','n',' ','8',' ','8',' '
.db     'C','h','n',' ','9',' ','9',' '
; comments about the codes
; 0-9 are the channel keys
; 4b=volume up
; 4d=volume down
; 48=channel up
; 50=channel down
; T=+10 channels button
; t=+20 channels button
; Z=play
; p=pause
; R=rewind
; F=fast forward
; G=record
; S=stop
; m=mute
; $1B=line in
;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------
; ATA/PI drive interface section
; modified as of 3 August 2007 for correct timing as per ATAPI3.pdf
; by Mauro Grassi.
;-------------------------------------------------------------------------------
detectdrives:
        ; this subroutine detects the connected drives...
        ; it follows the protocol as specified in cd_ata.pdf pg 47 section 5.18.4
        ; the procedure is as follows:
        ; for each of MASTER and SLAVE do:
        ; (1) check device BSY=0, DRDY=1
        ; (2) check signature bytes $14, $EB indicating an ATAPI device...
        ; (3) if signature present, issue ATAPI INQUIRY packet command 
        ;     (which is mandatory for all ATAPI devices so safe)...
        ; (4) if signature not present, issue ATA identify drive command to 
        ;     force signature and recheck signature, if still not present
        ;     conclude that no ATAPI device is there present...
        ; (5) Finished so update configuration variable...
        push    drive
        ldi     temp0,$01
        rcall   writelcdcontrol         ; clear the screen
        ldi     ZL,low(messagedetect)
        ldi     ZH,high(messagedetect)
        rcall   printmessage
        rcall   longdelay
ddvs0:  
        clr     config    
        clr     drive
ddvs1:
        rcall   selectdrive
        
        ldi     ZL,$14        ; here we do a test if we can write and read a register
        ldi     temp0,$55     ; if not, then the device is not there or not
        rcall   writeide      ; responding properly, so act as if not detected
        clr     temp0
        ldi     ZL,$14        ; this will avoid locking up in auto detect
        rcall   readide
        cpi     temp0,$55
        breq    ddvs10001
        rjmp    ddvs3     
ddvs10001:
        rjmp    ddvs2
ddvs12:
        ; here the signature has been detected so all good...
        ; we now issue an ATAPI identify device command......
        inc     config       ; add one to the number of detected drives..
        push    mode
        ldi     temp0,$01
        or      mode,temp0          ; set silent mode on
        rcall   inquirycommand
        pop     mode
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$01
        breq    ddvs1ok             ; all ok now...
        ; an error occurred when running the INQUIRY command so while
        ; the drive is present report "UNKNOWN" in model number...
        ldi     XH,$00
        ldi     XL,$68
        ldi     ZL,low(ddvsdefault)
        ldi     ZH,high(ddvsdefault)
        lsl     ZL
        rol     ZH
        ldi     temp2,$18
ddvs121:
        lpm
        st      X+,r0
        adiw    ZL,$01
        dec     temp2
        brne    ddvs121
ddvs1ok:
        ; drive has been detected so update bits 2 and 3 of configuration register
        mov     temp0,drive
        ldi     temp1,$04
        cpi     temp0,$00
        breq    ddvs1ok1
        add     temp1,temp1
ddvs1ok1:
        or      config,temp1
        ; all gone ok detection success so print out the model number...
        ldi     temp0,$01
        rcall   writelcdcontrol
        mov     temp0,config
        andi    temp0,$03
        rcall   displayfix
        rcall   writelcd
        ldi     temp0,':'
        rcall   writelcd
        ldi     temp0,' '
        rcall   writelcd
        ldi     XH,$00
        ldi     XL,$68
        ldi     temp2,$08
ddvs1ok11:
        ld      temp0,X+
        rcall   writelcd
        dec     temp2
        brne    ddvs1ok11
        ldi     temp0,$C0
        rcall   writelcdcontrol
        ldi     XH,$00
        ldi     XL,$70
        ldi     temp2,$10
ddvs1ok2:
        ld      temp0,X+
        rcall   writelcd
        dec     temp2
        brne    ddvs1ok2
        rcall   longdelay
        rjmp    ddvs3
ddvs2:
        ; no signature was detected so now we must issue ATA identify drive
        ; 4 and recheck for signature bytes...
        ; the ATA command code for IDENTIFY DEVICE command is $EC
        ldi     ZL,$16
        mov     temp0,drive
        cpi     temp0,$00
        breq    ddvs20001
        ldi     temp0,$10
ddvs20001:
        ldi     ZL,$16
        rcall   writeide
        ldi     ZL,$17
        ldi     temp0,$EC
        rcall   writeide            ; write the command to command register...
ddvs22:        
        rcall   longdelay
        ldi     ZL,$14
        rcall   readide
        cpi     temp0,$14
        brne    ddvs3
        ldi     ZL,$15
        rcall   readide
        cpi     temp0,$EB
        brne    ddvs3
        rjmp    ddvs12              ; detected!
ddvs3:
        ; here we have established it is not an ATAPI device at all... so...
        inc     drive
        inc     drive
        mov     temp0,drive
        cpi     temp0,$03
        brsh    ddvs4               ; repeat this loop twice...
        rjmp    ddvs1
ddvs4:
        ; now configuration = 0 or 1 or 2 = number of ATAPI devices present...
        pop     drive
        rcall   selectdrive
        mov     temp0,config
        andi    temp0,$03
        brne    ddvs5
        ldi     temp0,$C0
        rcall   writelcdcontrol
        ldi     ZL,low(messagenonedetected)
        ldi     ZH,high(messagenonedetected)
        rcall   printmessage  
        rcall   longdelay    
ddvs5:      
        ret

messagedetect:
.db     'D','e','t','e','c','t','i','n','g',' ','D','r','i','v','e','s',$80
messagenonedetected:
.db     'N','o','n','e',' ','D','e','t','e','c','t','e','d',' ',' ',' ',$80

ddvsdefault:
.db     'U','n','k','n','o','w','n',' '
.db     ' ',' ',' ','D','r','i','v','e'
.db     ' ',' ',' ',' ',' ',' ',' ',' '
    
initide:
        ; initialize the drive
        ; a hardware reset of duration about 814ns
        clr     drive
        ldi     temp0,ALLHI
        out     PORTB,temp0
        andi    temp0,PDMASK
        or      temp0,drive
        out     PORTD,temp0
        nop
        nop
        nop
        nop
        ldi     temp0,RES & PDMASK
        or      temp0,drive
        out     PORTD,temp0
        
        ldi     temp0,$80
initide1:
        nop
        dec     temp0
        brne    initide1
        
        ldi     temp0,ALLHI
        out     PORTB,temp0
        andi    temp0,PDMASK
        or      temp0,drive
        out     PORTD,temp0
        ret

setupide:         
        ; at this stage the device is not busy and we may write commands to it
        clr     drive
initide2:
        rcall   selectdrive
        ldi     ZL,$0e
        ldi     temp0,$02
        ldi     temp1,$00
        rcall   writeide            ; write to Device Control Register
                                    ; 02 disables the nIEN bit no INTRQ used now
                                    ; 00 enables the nIEN bit INTRQ used now...
        ldi     ZL,$11
        ldi     temp0,$03
        ldi     temp1,$00
        rcall   writeide
        ldi     ZL,$12
        ldi     temp0,$01
        ldi     temp1,$00
        rcall   writeide
        ldi     temp0,$ef
        rcall   sendcommand         ; SET FEATURES command set transfer mode to PIO mode 0 with disabled IORDY       
        
        inc     drive
        inc     drive
        mov     temp0,drive
        cpi     temp0,$04
        brlo    initide2
        
        ; automatically detect all connected drives...
        rcall   longdelay           ; this is required for proper detection...
        
        rcall   detectdrives
        mov     temp0,config
        andi    temp0,$03
        brne    initide3
        ; no drives are connected so sleep forever more...
initidedead:        
        sleep
        rjmp    initidedead
initide3:
        ; at least one drive has been detected so initialize drive
        ; register to point to MASTER if present or if not to SLAVE
        clr     drive
        rcall   protectselectdrive
        ldi     temp0,$01
        rcall   writelcdcontrol
        ret

protectselectdrive:
        ; make sure that the drive register holds a valid value
        ; according to configuration register, that is a valid
        ; detected drive! If not, change to a valid drive,  and
        ; if no valid drives we shouldn't be here to begin with!
        mov     temp0,drive
        andi    temp0,$02
        brne    selslave
selmaster:
        ; check that bit 2 of configuration register is set...
        mov     temp0,config
        andi    temp0,$04
        brne    selectdrive         ; all ok...
        ; master drive is selected but doesn't exist! so change to slave
        ldi     temp0,$02
        mov     drive,temp0
        rjmp    selectdrive
selslave:
        mov     temp0,config
        andi    temp0,$08
        brne    selectdrive
        ; slave drive is selected but doesn't exist! so change to master
        ldi     temp0,$00
        mov     drive,temp0
selectdrive:
        ; select the drive given in drive register
        ; if drive register is 0 it selects MASTER
        ; if drive register is not 0 it selects SLAVE
        ldi     temp1,$00
        ldi     ZL,$16      ; address of DEV/HEAD register...
        ldi     temp0,$00
        mov     temp1,drive
        cpi     temp1,$00
        breq    selectd0
        ldi     temp0,$10   ;   bit 4=0 for MASTER bit4=1 for SLAVE
selectd0:
        rcall   writeide
        ret

readide:
        ; the register ZL contains the address (5-bits)
        ; the data is returned in temp1:temp0 (for 16-bit transfers)...
ireadide:
        ; internal call version
        push    temp2
        ldi     temp0,ALLHI & PDMASK
        or      temp0,drive
        out     PORTD,temp0         ; initial condition
        clr     temp0
        out     DDRC,temp0
        out     DDRA,temp0
        mov     ZH,ZL
        ori     ZL,$18
        out     PORTB,ZL            ; output the address...
        
        mov     ZL,ZH
        out     PORTB,ZL

        ldi     temp0,RD & PDMASK
        or      temp0,drive
        out     PORTD,temp0         ; enable READ
        nop
        in      temp1,PINC
        in      temp2,PINA          ; read in data...
        nop
        ldi     temp0,ALLHI
        out     PORTB,temp0        ; B must go high first this is IDE CS!
        andi    temp0,PDMASK
        or      temp0,drive
        out     PORTD,temp0        ; done!
        
        mov     temp0,temp2
        pop     temp2
        ret
writeide:
iwriteide:
        ; the register ZL contains the address (5-bits)
        ; the data is in temp1:temp0 (for 16-bit transfers)...
        push    temp2
        ldi     temp2,ALLHI & PDMASK
        or      temp2,drive
        out     PORTD,temp2         ; initial condition
        mov     ZH,ZL
        ori     ZL,$18
        out     PORTB,ZL            ; output the address...
        
        out     PORTA,temp0
        out     PORTC,temp1         ; output the data...
        ldi     temp2,$ff
        out     DDRA,temp2
        out     DDRC,temp2          ; PORTS A and C are outputs
        
        mov     ZL,ZH
        out     PORTB,ZL
 
        ldi     temp2,WR & PDMASK
        or      temp2,drive
        out     PORTD,temp2         ; enable WRITE
        
        nop
        nop                         ; now writing...
                
        ldi     temp2,ALLHI
        out     PORTB,temp2        ; B must go high first this is IDE CS!
        andi    temp2,PDMASK
        or      temp2,drive
        out     PORTD,temp2        ; done!
        
        ldi     temp2,$00
        out     DDRA,temp2
        out     DDRC,temp2         ; clean up...
        pop     temp2
        ret

sendcommand:
        ; send command in temp0 to drive.
        ldi     ZL,$17
        ldi     temp1,$00
        rcall   writeide
        ; returns status register in temp0
        ret

preparepacket:
        ; set-up packet data...
        ; reads six words of FLASH at Z to $60...
        rcall   clearmemory
ipreparepacket:
        ldi     temp2,$0c       ; byte counter
        lsl     ZL
        rol     ZH              ; convert to word address.
        ldi     XL,$20
        ldi     XH,$02
_pp1:
        lpm
        st      X+,r0
        adiw    ZL,$01
        dec     temp2
        brne    _pp1

        mov     temp0,mode
        andi    temp0,$01
        breq    _pp12
        ret
_pp12:
        ; print the command message since silent mode is OFF
        ldi     temp0,$C0
        rcall   writelcdcontrol
        ldi     temp2,$0c     ; byte counter
_pp2:
        lpm
        mov     temp0,r0
        rcall   writelcd
        adiw    ZL,$01
        dec     temp2
        brne    _pp2
        ldi     temp0,' '
        rcall   writelcd
        rcall   shortdelay
        ret

;-------------------------------------------------------------------------------
; sendpacket...
; modified 7 August 2007 for silicon chip
; to bring into line with the specifications
; as per file cd_ata.pdf
; on page 26(40) of that document...
sendpacket:
        ; send a packet command at FLASH ZL
        ; the packet command (12 bytes) is assumed to be
        ; written at $60 onwards in memory...
        rcall   preparepacket
isendpacket:
pk00:
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$80
        brne    pk00            ; wait for device BSY=0 

        ; now we initialize FEATURES, BYTE COUNT and DRIVE HEAD registers...   
        
        ldi     ZL,$14
        ldi     temp0,$C0       ; byte count Lo
        rcall   writeide
        ldi     ZL,$15
        ldi     temp0,$01       ; byte count Hi
        rcall   writeide        ; so we set maximum data transfer in bytes
        
        ldi     ZL,$11
        ldi     temp0,$00
        rcall   writeide        ; clear DMA & OVERLAPPED modes 
                                ; in FEATURES register
        ldi     temp0,$a0
        rcall   sendcommand     ; send packet command...
        ; packet command protocol (PIO version)
        ; see cd_ata.pdf= pg 26(40).
ipacketcommand:
pk0:
pk1:
        nop
        nop                   ; these two nops are required for slower drives to work properly... (do not remove)
        ldi     ZL,$12
        rcall   readide       ; read the ATAPI Interrupt Reason register
                              ; the device sets CoD and clears IO bits
        andi    temp0,$03
        cpi     temp0,$01
        brne    pk1           ; wait till sets CoD (bit 0) and 
                              ; clears IO(bit 1) bits
pk11:
        nop
        nop                   ; these two nops are required for slower drives to work properly... (do not remove)
        ldi     ZL,$17
        rcall   readide       ; read status register.
        andi    temp0,$88
        cpi     temp0,$08
        brne    pk11          ; wait until BSY=0 and DRQ=1

        ldi     temp2,$06   ; 12 bytes = 6 words
        ldi     XL,$20
        ldi     XH,$02        ; initialize pointer to memory.
pk2:                              
        ld      temp0,X+
        ld      temp1,X+      ; read a word from memory.
        ldi     ZL,$10
        rcall   writeide      ; send packet to the drive through data register...
        nop
        nop                   ; these nops are required for slower drives to work properly... (do not remove)
        nop
        nop
        nop
        nop
        nop
        nop
        nop
        dec     temp2
        brne    pk2           ; 12 bytes of packet have been sent now...
                              ; the device will now negate DRQ set BSY
                              ; and perform command...
pk3:
pk4:
        nop
        nop                   ; these two nops are required for slower drives to work properly... (do not remove)
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$80
        brne    pk4           ; wait till BSY=0
pk5:
        ;first we must determine the direction of the transfer
        ;by reading the IO CoD bits in $12 register, and
        ;the no. of bytes in the cylinder low:high registers...
        ldi     ZL,$14
        rcall   readide
        push    temp0
        ldi     ZL,$15
        rcall   readide
        mov     YH,temp0
        pop     pointerlo
        mov     pointerhi,YH
        clr     YL
        clr     YH

        ; now YH:YL count the number of words for transfer 1 word=16 bits...
pk6:        
        ldi     ZL,$12
        rcall   readide
        andi    temp0,$03
        push    temp0
pk7:
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$88
        cpi     temp0,$08
        breq    pkextra
        cpi     temp0,$80
        breq    pk7
        cpi     temp0,$88       ;loop if still busy
        breq    pk7
        pop     temp0
        rjmp    pkexcontinue    ; exit if no extra transfer is required...    
pkextra:
        pop     temp0           ; this now holds the CoD and IO bits
        cpi     temp0,$00
        breq    pkexwrite
        cpi     temp0,$02
        breq    pkexread
        rjmp    pkexcontinue
pkexread:
        ldi     XL,$60
        ldi     XH,$00        ; initialize pointer to memory.
        clr     temp2         ; page counter...
pkexr122:        
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$88
        cpi     temp0,$80
        breq    pkexr122
        cpi     temp0,$88
        breq    pkexr122
        cpi     temp0,$00
        brne    pkexr1222
        rjmp    pkexcontinue
        ; if drive is busy the contents of DATA register won't be defined...
        ; so we must pause while drive is BSY=1
pkexr1222:
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$80
        brne    pkexr1222
        
        ldi     ZL,$10
        rcall   readide

        adiw    YL,$01          ; word count...

        cpi     XH,$02
        brlo    pkexr1221
        cpi     XL,$20
        brsh    pkexr124
pkexr1221:
        st      X+,temp0
        st      X+,temp1
        rjmp    pkexr122
pkexr124:        
        inc     temp2
        rjmp    pkexr122
pkexwrite:
        ; here we write extra packet to device (as I/O bit 2 is 0)
      
        ldi     XL,$60
        ldi     XH,$00          ; initialize pointer to memory
        clr     temp2           ; clear page counter
        ; continue until all requested data transferred.   
        ; now although we've transferred all the bytes reported in $14,$15
        ; sometimes this is badly reported, especially if it exceeds the
        ; drive's capabilities, so DRQ may still be asserted at this point...
        ; hence we must keep reading until it is cleared...
pkexw122:         
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$88
        cpi     temp0,$80
        breq    pkexw122
        cpi     temp0,$88
        breq    pkexw122
        cpi     temp0,$00
        breq    pkexcontinue
pkexw1220:
        cpi     XH,$02
        brlo    pkexw1221
        cpi     XL,$20
        brlo    pkexw1221
        inc     temp2
        rjmp    pkexw1222
pkexw1221:
        ld      temp0,X+
        ld      temp1,X+
pkexw1222:
        push    temp0
        push    temp1
        push    temp2
pkexw888:        
        ; if drive is busy the contents of DATA register won't be defined...
        ; so we must pause while drive is BSY=1
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$80
        brne    pkexw888
        pop     temp2
        pop     temp1
        pop     temp0
        
        ldi     ZL,$10
        rcall   writeide
        adiw    YL,$01

        rjmp    pkexw122
pkexcontinue:
        ldi     ZL,$0e
        rcall   readide     ; read alternate status and ignore results (pg. 238(254) of ATAPI-4)
        ldi     ZL,$17
        rcall   readide     ; read status register...
        push    temp0
        andi    temp0,$01
        breq    pkexend
        ; error has occured so we must read the error register to temp1
        ldi     ZL,$11
        rcall   readide
        push    temp0
        ; now there might still be a valid DRQ=1 here which we must fix...
        ; especially for reading...
pkexco98:
        ldi     ZL,$17
        rcall   readide
        andi    temp0,$08
        breq    pkexco99
        ldi     ZL,$10
        rcall   readide
        rjmp    pkexco98
pkexco99:     
        pop     temp1    
        pop     temp0
        clc
        ret                 ; no carry means error has occurred...
pkexend:        
        pop     temp0
        sec                 ; carry means no error
        ret                 ; done, return with status register in temp0
;-------------------------------------------------------------------------------
;*******************************************************************************
placeparameter:
        ; goto the parameter location in XH:XL given by temp2=code page
        ; and temp1=code page offset (in bytes)
        ; this subroutine parses the MODE SENSE information at 00:60 SRAM (hex)
        ; and returns a pointer in XH:XL as requested...
        ; returns CARRY if all ok...
        ; returns NO CARRY if not found...
        ldi     XH,$00
        ldi     XL,$60
        ld      YH,X+
        ld      YL,X+ 
        adiw    YL,$02
        adiw    YL,$30
        adiw    YL,$30
        ; now YH:YL hold the end byte of the whole data...
        ; and pointerhi:lo holds number of bytes to transfer...
        adiw    XL,$06          ; to start of first page...
ppar0:
        cp      XH,YH
        brlo    ppar01
        cp      XL,YL
        brlo    ppar01
        rjmp    pparnotfound
ppar01:
        ; still more to search...
        ld      temp0,X+
        andi    temp0,$3F
        cp      temp0,temp2         ; compare page code...
        breq    pparfound
        ; code pages not equal so skip...
        ld      temp0,X+            ; now temp0 is the page length
        add     XL,temp0
        clr     temp0
        adc     XH,temp0            ; added to the index...
        rjmp    ppar0               ; keep going...
pparfound:
        sbiw    XL,$01
        add     XL,temp1
        clr     temp1
        adc     XH,temp1            ; add offset... and return...
        sec
        ret        
pparnotfound:
        ; we got to the end and not found...
        clc
        ret
        
putparameter:
        push    temp0
        rcall   placeparameter
        pop     temp0
        st      X,temp0
        ret
getparameter:
        rcall   placeparameter
        ld      temp0,X
        ret
;*******************************************************************************
getallide:
        ldi     ZL,$10
        rcall   readide
        ldi     ZL,$17
        rcall   readide
        ret

printide:
        ldi     temp0,$88
        rcall   writelcdcontrol
        ldi     ZL,$10
_pide1:
        rcall   readide
        rcall   disa
        cpi     ZL,$13
        brne    _pide2
        ldi     temp0,$c8
        rcall   writelcdcontrol
_pide2:
        inc     ZL
        cpi     ZL,$18
        brne    _pide1
        
        ldi     temp0,$C2
        rcall   writelcdcontrol
        mov     temp0,pointerhi
        rcall   disa
        mov     temp0,pointerlo
        rcall   disa
        ret
;----------------------------Some common IDE/ATAPI command packets--------------
softreset:
        ldi     temp0,$c0
        rcall   writelcdcontrol
        ldi     ZL,low(messagesoftreset)
        ldi     ZH,high(messagesoftreset)
        rcall   printmessage
        
        ldi     ZL,$0e
        ldi     temp0,$06
        ldi     temp1,$00
        rcall   writeide            ; software reset and disable INTRQ
                                    ; write to the Device Control Register...
        ret
messagesoftreset:
.db     'R','e','s','e','t','.',$80
;*******************************************************************************
ejectcommand:
        push    mode
        ldi     temp0,$01
        or      mode,temp0
        rcall   unlockcdcommand
        pop     mode
        ldi     ZL,low(peject)
        ldi     ZH,high(peject)
        rcall   sendpacket
        ret
peject:
.db     $1b,$00,$00,$00,$02,$00,$00,$00,$00,$00,$00,$00
.db     'E','j','e','c','t',' ','C','D',' ',' ',' ',' '
;*******************************************************************************
closetraycommand:
        ldi     ZL,low(pclose)
        ldi     ZH,high(pclose)
        rcall   sendpacket
        ret
pclose:
.db     $1b,$00,$00,$00,$03,$00,$00,$00,$00,$00,$00,$00
.db     'C','l','o','s','e',' ','T','r','a','y',' ',' '
;*******************************************************************************
stopdisccommand:
        ldi     ZL,low(pstopdisc)
        ldi     ZH,high(pstopdisc)
        rcall   sendpacket
        ret
pstopdisc:
.db     $1b,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'S','t','o','p',' ','D','i','s','c',' ',' ',' '
;*******************************************************************************
starttoccommand:
        ldi     ZL,low(pstarttoc)
        ldi     ZH,high(pstarttoc)
        rcall   sendpacket
        ret
pstarttoc:
.db     $1b,$01,$00,$00,$01,$00,$00,$00,$00,$00,$00,$00
; start the disc and read toc command...
.db     'S','t','a','r','t',' ','D','/','T','O','C',' '
;*******************************************************************************
scancommand:
        ldi     ZL,low(pscan)
        ldi     ZH,high(pscan)
        rcall   sendpacket
        ret
pscan:
.db     $ba,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'S','c','a','n',' ',' ',' ',' ',' ',' ',' ',' '
;*******************************************************************************
capacitycommand:
        ldi     ZL,low(pcapacity)
        ldi     ZH,high(pcapacity)
        rcall   sendpacket
        ret
pcapacity:
.db     $25,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'C','a','p','a','c','i','t','y',' ',' ',' ',' '
;*******************************************************************************
inquirycommand:
        ldi     ZL,low(pinquiry)
        ldi     ZH,high(pinquiry)
        rcall   sendpacket
        ret
pinquiry:
.db     $12,$00,$00,$00,$60,$00,$00,$00,$00,$00,$00,$00
.db     'I','n','q','u','i','r','y',' ',' ',' ',' ',' '
;*******************************************************************************
playwithtoccommand:
        rcall   toccommand
playcommand:
        ldi     ZL,low(pplaycd)
        ldi     ZH,high(pplaycd)
        rcall   sendpacket
        ret
pplaycd:
.db     $45,$ff,$00,$00,$00,$00,$ff,$ff,$ff,$00,$00,$00
.db     'P','l','a','y',' ','C','D',' ',' ',' ',' ',' '
;*******************************************************************************

playnexttrackcommand:
;*******************************************************************************
;* plays the next track as given in either mnexttrack or snexttrack registers
;* only plays that track then should stop.
;*******************************************************************************
        rcall   toccommand
        brcs    pnok1
        
        rcall   shortdelay
        ldi     temp0,ERROR
        clc
        ret             ; error reading the TOC (abort)...
pnok1:
        ;*** continue since no error with reading the TOC...

        ldi     ZL,low(pplayaudio)
        ldi     ZH,high(pplayaudio)
        rcall   ipreparepacket
        ;**** now TOC should be starting at 0060...
        ; first compare the nexttrack register with the limits at
        ; $62=first track $63=last track...
        
        mov     temp0,drive
        cpi     temp0,$00
        brne    pnok2
        mov     temp1,mnexttrack
        rjmp    pnok3
pnok2:        
        mov     temp1,snexttrack
pnok3:
        ;** now temp1 holds the next track according to the drive we're playing...
        clr     XH
        ldi     XL,$62
        ld      temp0,X+
        mov     temp2,temp0
        cp      temp1,temp0
        brsh    pnok4
        clr     temp1
        
        mov     temp0,drive
        andi    temp0,$02
        brne    pnok1000
        mov     mnexttrack,temp1
        rjmp    pnok6
pnok1000: 
        mov     snexttrack,temp1
        rjmp    pnok6
pnok4:
        ld      temp0,X
        cp      temp1,temp0
        brlo    pnok5   
        breq    pnok5
        mov     temp1,temp2        
        mov     temp0,drive
        andi    temp0,$02
        brne    pnok1001
        mov     mnexttrack,temp1
        rjmp    pnok5
pnok1001: 
        mov     snexttrack,temp1
pnok5:
        ldi     XL,$62
        ld      temp0,X
        sub     temp1,temp0
pnok6:
        ; now temp1 holds a valid relative track number, 0 being the first track
        ; so look for its address...
        ; the offset should be
        ; 0064 + temp1*8...
        clr     temp0
        lsl     temp1
        rol     temp0
        lsl     temp1
        rol     temp0
        lsl     temp1
        rol     temp0
        ldi     XL,$68
        add     XL,temp1
        adc     XH,temp0
        ; now XH:XL points to the beginning of the address of the track...
        ldi     YH,$02
        ldi     YL,$22  
        ld      temp0,X+
        st      Y+,temp0
        ld      temp0,X+
        st      Y+,temp0
        ld      temp0,X+
        st      Y+,temp0
        ld      temp0,X
        push    XL
        push    XH
        st      Y+,temp0
        adiw    YL,$01
        adiw    XL,$05
        ld      byte1,X+
        ld      byte2,X+
        ld      byte3,X+
        ld      byte4,X+
        pop     XH
        pop     XL
        adiw    XL,$01
        ld      temp0,-X
        sub     byte4,temp0
        ld      temp0,-X
        sbc     byte3,temp0
        ld      temp0,-X
        sbc     byte2,temp0
        ld      temp0,-X
        sbc     byte1,temp0
        mov     temp0,byte3
        st      Y+,temp0
        mov     temp0,byte4
        st      Y+,temp0
        rcall   isendpacket     ;* try to play the track...
        ret
pplayaudio:
.db     $45,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'P','l','a','y',' ','A','u','d','i','o',' ',' '
;*******************************************************************************
readsubchannelcommand:
ireadsubchannelcommand:         ; internal calls
        ldi     ZL,low(preadsubchannel)
        ldi     ZH,high(preadsubchannel)
        rcall   sendpacket
        ret
preadsubchannel:
.db     $42,$02,$40,$01,$00,$00,$00,$01,$C0,$00,$00,$00
; allocation data length=1C0 bytes and select current position in MSF format...
.db     'P','l','a','y','i','n','g','.','.','.',' ',' '
;*******************************************************************************
stopplaycommand:
        ldi     ZL,low(pstopplaycd)
        ldi     ZH,high(pstopplaycd)
        rcall   sendpacket
        ret
pstopplaycd:
.db     $4e,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'S','t','o','p',' ','P','l','a','y',' ','C','D'
;*******************************************************************************
resumecommand:
        ldi     temp0,$c0
        rcall   writelcdcontrol
        ldi     ZL,low(presume)
        ldi     ZH,high(presume)
        rcall   sendpacket
        ret
presume:
.db     $4b,$00,$00,$00,$00,$00,$00,$00,$01,$00,$00,$00
.db     'R','e','s','u','m','e',' ','P','l','a','y',' '
;*******************************************************************************
pausecommand:
        ldi     temp0,$c0
        rcall   writelcdcontrol
        ldi     ZL,low(ppause)
        ldi     ZH,high(ppause)
        rcall   sendpacket
        ret
ppause:
.db     $4b,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'P','a','u','s','e',' ','P','l','a','y',' ',' '
;*******************************************************************************
toccommand:
        push    mode
        ldi     temp0,$01
        or      mode,temp0
        ldi     ZL,low(ptoc)
        ldi     ZH,high(ptoc)
        rcall   sendpacket
        pop     mode
        ret
ptoc:
.db     $43,$00,$00,$00,$00,$00,$00,$03,$24,$00,$00,$00
; note 0324hex=804 bytes user RAM allocate up to that amount only...
.db     'R','e','a','d',' ','T','O','C',' ',' ',' ',' '
;*******************************************************************************
mechanismstatuscommand:
        ldi     ZL,low(pstatus)
        ldi     ZH,high(pstatus)
        rcall   sendpacket
        ret
pstatus:
.db     $bd,$10,$00,$00,$00,$00,$00,$00,$60,$00,$00,$00
.db     'R','e','a','d',' ','S','t','a','t','u','s',' '
;*******************************************************************************
sleepcommand:
        ldi     ZL,low(messagesleep)
        ldi     ZH,high(messagesleep)
        rcall   printmessage
        ldi     temp0,$e6
        rcall   sendcommand     ; sleep command = $e6
        ret
messagesleep:
.db     'S','l','e','e','p',$80
;*******************************************************************************
lockcdcommand:
        ldi     ZL,low(plockcd)
        ldi     ZH,high(plockcd)
        rcall   sendpacket
        ret
plockcd:
.db     $1e,$ff,$00,$00,$01,$00,$00,$00,$00,$00,$00,$00
.db     'L','o','c','k',' ','D','o','o','r',' ',' ',' '
;*******************************************************************************
unlockcdcommand:
        ldi     ZL,low(punlockcd)
        ldi     ZH,high(punlockcd)
        rcall   sendpacket
        ret
punlockcd:
.db     $1e,$ff,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'U','n','l','o','c','k',' ','D','o','o','r',' '
;*******************************************************************************
checkpowermodecommand:
        ldi     ZL,low(messagepowermode)
        ldi     ZH,high(messagepowermode)
        rcall   printmessage
        ldi     temp0,$e5
        rcall   sendcommand     ; check power mode = $e5
        ret
messagepowermode:
.db     'P','o','w','e','r',' ','M','o','d','e',$80
;*******************************************************************************
diagnosticscommand:
        ldi     ZL,low(messagediags)
        ldi     ZH,high(messagediags)
        rcall   printmessage
        ldi     temp0,$90
        rcall   sendcommand     ; diagnostics command = $90
        ret
messagediags:
.db     'D','i','a','g','n','o','s','t','i','c','s',$80
;*******************************************************************************
identifydevicecommand:
        ldi     ZL,low(messageid)
        ldi     ZH,high(messageid)
        rcall   printmessage
        ldi     temp0,$A1
        rcall   sendcommand     ; inquiry command = $a1
        rcall   clearmemory
        rcall   ipacketcommand
        ret

messageid:
.db     'I','d','e','n','t','i','f','y',' ','D','e','v','i','c','e',$80
;*******************************************************************************
postmodeselectcommand:
        push    pointerlo
        push    pointerhi
        push    mode
        ldi     temp0,$01
        or      mode,temp0      ; silent mode on.
        ldi     ZL,low(messagemselect)
        ldi     ZH,high(messagemselect)
        rcall   ipreparepacket
        rcall   shortestdelay  ; this delay is absolutely necessary for proper functioning 
        pop     mode
        pop     pointerhi
        pop     pointerlo
        ldi     XH,$02
        ldi     XL,$27
        st      X+,pointerhi
        st      X+,pointerlo        ; put in the parameter length list...
        rcall   isendpacket
        ret                         ; done!
messagemselect:
.db     $55,$10,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db     'M','o','d','e',' ','S','e','l','e','c','t',' '
;*******************************************************************************
mymodeselectcommand:
        ; assumes that data from mode sense is there
        ldi     XH,$00
        ldi     XL,$60
        ld      YH,X+
        ld      YL,X+
        adiw    YL,$02
        mov     pointerlo,YL
        mov     pointerhi,YH
        rcall   postmodeselectcommand
        ret
 
voldowncommand:
        ldi     temp0,$10
        sub     volume,temp0
        brcc    vold1
        clr     volume        
vold1:
        rcall   setvolumecommand
        ldi     temp0,$fb
        and     mode,temp0      ; reset the muting...
        sec
        ret

mutecommand:
        ldi     XH,$02
        ldi     XL,$2D        ; location of mute (volume register)
        mov     temp0,volume
        cpi     temp0,$00
        breq    muterestore
        st      X,volume
        clr     volume
mute000:
        rcall   setvolumecommand
        sec
        ret
muterestore:
        ld      volume,X
        rjmp    mute000

volupcommand:
        ldi     temp0,$10
        add     volume,temp0
        brcc    volu1
        ldi     temp0,$F8
        mov     volume,temp0  
volu1:
        rcall   setvolumecommand
        ldi     temp0,$fb
        and     mode,temp0      ; reset the muting...
        sec
        ret
        
baldowncommand:
        ldi     temp0,$05
        sub     balance,temp0
        brcc    bald1
        clr     balance       
bald1:
        rcall   setvolumecommand
        sec
        ret

balupcommand:
        ldi     temp0,$05
        add     balance,temp0
        mov     temp0,balance
        cpi     temp0,$64
        brlo    balu1
        ldi     temp0,$64
        mov     balance,temp0  
balu1:
        rcall   setvolumecommand
        sec
        ret

setvolumecommand:
        rcall   modesensecommand
        brcs    svol0
        ldi     temp0,NOTRANSITION
        ret
svol0:
        ; now search for the volume control byte and change it to volume on both channels...
        ldi     temp1,$00
        ldi     temp2,$0E
        rcall   getparameter     ; this requests to get parameter offset 0 from code page 0e in temp0
        ; now XH:XL points to the page 0E.
        push    XH
        push    XL
        ldi     YH,$00
        ldi     YL,$68
        ldi     temp2,$10
svol1:        
        ld      temp0,X+
        st      Y+,temp0
        dec     temp2
        brne    svol1
        pop     XL
        pop     XH
        ; so now it has been copied to the start of 0068hex
        ldi     temp0,$00
        ldi     YH,$00
        ldi     YL,$60
        st      Y+,temp0
        ldi     temp0,$16
        st      Y+,temp0
        ; set the page length in header
        ldi     pointerlo,$18
        ldi     pointerhi,$00
        ; number of bytes to write  
        push    pointerhi
        push    pointerlo
        mov     div1,volume
        mov     div2,balance
        rcall   mul88
        clr     byte1
        clr     byte2
        ldi     div2,$64
        ldi     div1,$00
        rcall   fpudiv3216
        clr     XH
        ldi     XL,$71              ;71hex is first channel volume offset
        st      X,byte4
        mov     temp0,volume
        sub     temp0,byte4
        clr     XH
        ldi     XL,$73
        st      X,temp0             ;73hex is second channel volume offset        
        pop     pointerlo
        pop     pointerhi
        rcall   postmodeselectcommand
        brcs    svol2
        ldi     temp0,NOTRANSITION
        ret
svol2:
        ldi     temp0,$01
        rcall   writelcdcontrol
        ldi     ZL,low(messagevol)
        ldi     ZH,high(messagevol)        
        rcall   printmessage
        mov     temp0,volume
        lsr     temp0
        lsr     temp0
        lsr     temp0
        lsr     temp0
        cpi     temp0,$00
        brne    svol200
        ldi     ZL,low(messagemute)
        ldi     ZH,high(messagemute)
        rcall   printmessage
        rjmp    svol300
svol200:
        ldi     temp1,$02
        rcall   disadecimal
svol300:
        ldi     temp0,$C0
        rcall   writelcdcontrol
        ldi     ZL,low(messagebal)
        ldi     ZH,high(messagebal)
        rcall   printmessage
        mov     temp0,balance
        ldi     temp1,$03
        rcall   disadecimal
        ldi     temp0,':'
        rcall   writelcd
        ldi     temp0,$64
        sub     temp0,balance
        ldi     temp1,$03
        rcall   disadecimal
        rcall   shortdelay
        ldi     temp0,$01
        rcall   writelcdcontrol
        ldi     temp0,NOTRANSITION        
        ret
        
messagevol:
.db     'V','o','l','u','m','e',' ',':',' ',$80
messagebal:
.db     'B','a','l','a','n','c','e',':',' ',$80
messagemute:
.db     'M','U','T','E',$80,$80
;*******************************************************************************
modesensecommand:
        push    mode
        ldi     temp0,$01
        or      mode,temp0
        ldi     ZL,low(pmodesense)
        ldi     ZH,high(pmodesense)
        rcall   sendpacket
        pop     mode
        ret
pmodesense:
.db     $5a,$00,$3F,$00,$00,$00,$00,$01,$C0,$00,$00,$00 
; request all CD-Rom pages and allocate up to 1C0 bytes of data returned...
.db     'M','o','d','e',' ','S','e','n','s','e',' ',' '
;*******************************************************************************
clearmemory:
        ldi     XL,$60
        ldi     XH,$00
        ldi     YL,$E0          ; clear 448 bytes or 224=0xE0hex words...
clearmem:
        ldi     temp0,$00
        st      X+,temp0
        st      X+,temp0
        dec     YL
        brne    clearmem
        ret
;*******************************************************************************
currentcdposition:
        rcall   readsubchannelcommand
        brcs    curr00
        clc             ; could not read subchannel information, abort!
        ret        
curr00:
        ldi     temp0,$80
        rcall   writelcdcontrol
        ldi     ZL,low(messagetrack)
        ldi     ZH,high(messagetrack)
        rcall   printmessage
        clr     XH
        ldi     XL,$61
        ld      temp0,X
        cpi     temp0,$11       ; play in progress code=$11
        breq    curr0
        ; if play is not in progress go to "next" track...
        rcall   getnexttrack
        clc
        ret
curr0:        
        ldi     XL,$66
        ld      temp0,X
        ldi     temp1,$02
        rcall   disadecimal
        ldi     temp0,' '
        rcall   writelcd
        clr     XH
        ldi     XL,$6d
        ld      temp0,X
        ldi     temp1,$02
        rcall   disadecimal
        ldi     temp0,':'
        rcall   writelcd
        clr     XH
        ldi     XL,$6e
        ld      temp0,X
        ldi     temp1,$02
        rcall   disadecimal
        ldi     temp0,' '        
        rcall   writelcd
        mov     temp0,mode
        andi    temp0,$02
        breq    curr998
        ldi     temp0,'1'
        rcall   writelcd
        rjmp    curr999
curr998:
        mov     temp0,mode
        andi    temp0,$04
        breq    curr990
        ldi     temp0,'R'
        rcall   writelcd
        rjmp    curr999
curr990:
        ldi     temp0,' '
        rcall   writelcd
curr999:
        sec
        ret
messagetrack:
.db     'T','r','a','c','k',' ',$80,$80
;*******************************************************************************
;*******************************************************************************
;* CD Player State Machine, by Mauro Grassi 23 August 2007
;* For Silicon Chip
;*******************************************************************************
;* Translates the code in temp0 (a key pressed)
;* into the next state...
;*******************************************************************************
nextstate:
        cpi   temp0,NOTRANSITION
        brne  nex99
        ret        
nex99:
        push  temp0
        ldi   ZL,low(nstatetable)
        ldi   ZH,high(nstatetable)
        ; now add to it the current state * 16
        mov   temp0,drive
        cpi   temp0,$00
        brne  nex0
        mov   temp0,mstate
        rjmp  nex1
nex0:
        mov   temp0,sstate
nex1:
        clr   temp1
        lsl   temp0
        rol   temp1
        lsl   temp0
        rol   temp1
        lsl   temp0
        rol   temp1
        lsl   temp0
        rol   temp1

        add   ZL,temp0
        adc   ZH,temp1
        lsl   ZL
        rol   ZH            ; now ZH:ZL points to the nstatetable
        
        pop   temp0
        andi  temp0,$7f
        cpi   temp0,'T'
        brne  nst1
        ldi   temp0,$00
        rjmp  nstend
nst1:
        cpi   temp0,'t'
        brne  nst2
        ldi   temp0,$01
        rjmp  nstend
nst2:
        cpi   temp0,$4b
        brne  nst3
        ldi   temp0,$02
        rjmp  nstend
nst3:
        cpi   temp0,$4d
        brne  nst4
        ldi   temp0,$03
        rjmp  nstend
nst4:
        cpi   temp0,$48
        brne  nst5
        ldi   temp0,$04
        rjmp  nstend
nst5:
        cpi   temp0,$50
        brne  nst6
        ldi   temp0,$05
        rjmp  nstend
nst6:
        cpi   temp0,'m'
        brne  nst7
        ldi   temp0,$06
        rjmp  nstend
nst7:
        cpi   temp0,'Z'
        brne  nst8
        ldi   temp0,$07
        rjmp  nstend
nst8:
        cpi   temp0,'F'
        brne  nst9
        ldi   temp0,$08
        rjmp  nstend
nst9:
        cpi   temp0,'R'
        brne  nst10
        ldi   temp0,$09
        rjmp  nstend
nst10:
        cpi   temp0,'S'
        brne  nst11
        ldi   temp0,$0A
        rjmp  nstend
nst11:
        cpi   temp0,'p'
        brne  nst12
        ldi   temp0,$0B
        rjmp  nstend
nst12:
        cpi   temp0,'G'
        brne  nst13
        ldi   temp0,$0C
        rjmp  nstend
nst13:
        cpi   temp0,$1b
        brne  nst14
        ldi   temp0,$0D
        rjmp  nstend
nst14:
        cpi   temp0,'O'
        brne  nst15
        ldi   temp0,$0E
        rjmp  nstend
nst15:
        cpi   temp0,'0'
        brne  nst16
        ldi   temp0,$0F
        rjmp  nstend
nst16:
        cpi   temp0,'1'
        brne  nst17
        ldi   temp0,$10
        rjmp  nstend
nst17:
        cpi   temp0,'2'
        brne  nst18
        ldi   temp0,$11
        rjmp  nstend
nst18:
        cpi   temp0,'3'
        brne  nst19
        ldi   temp0,$12
        rjmp  nstend
nst19:
        cpi   temp0,'4'
        brne  nst20
        ldi   temp0,$13
        rjmp  nstend
nst20:
        cpi   temp0,'5'
        brne  nst21
        ldi   temp0,$14
        rjmp  nstend
nst21:
        cpi   temp0,'6'
        brne  nst22
        ldi   temp0,$15
        rjmp  nstend
nst22:
        cpi   temp0,'7'
        brne  nst23
        ldi   temp0,$16
        rjmp  nstend
nst23:
        cpi   temp0,'8'
        brne  nst24
        ldi   temp0,$17
        rjmp  nstend
nst24:
        cpi   temp0,'9'
        brne  nst25
        ldi   temp0,$18
        rjmp  nstend
nst25:
        cpi   temp0,EMPTY
        brne  nst26
        ldi   temp0,$19
        rjmp  nstend
nst26:
        cpi   temp0,ERROR
        brne  nst27
        ldi   temp0,$1A
        rjmp  nstend
nst27:
        cpi   temp0,OK
        brne  nst28
        ldi   temp0,$1B
        rjmp  nstend
nst28:
        cpi   temp0,LASTSTATE
        brne  nst29
        mov   r0,error
        rjmp  nst292        
        
nst29:
        cpi   temp0,OK2
        brne  nst30
        ldi   temp0,$1C
        rjmp  nstend
nst30:
        ldi   temp0,$1D     ; this is the default look up if no keys/tokens matched...
nstend:
        clr   temp1
        add   ZL,temp0
        adc   ZH,temp1      ; now ZH:ZL points to the right place
        lpm
nst292:        
        mov   temp0,r0
        cpi   temp0,MAXSTATES
        brlo  nstend0
        ret
nstend0:
        mov   temp0,drive
        cpi   temp0,$00
        brne  nex2
        mov   error,mstate
        mov   mstate,r0
        ret        
nex2:
        mov   error,sstate
        mov   sstate,r0
        ret                 ; done!
;*******************************************************************************
;* This is the transition table
;* for each state
;* it lists the next state in the order...
;* Key pressed= T,t,4b,4d,48,50,m,Z,F,R,S,p,G,1b,O,0,1,2,3,4,5,6,7,8,9,Default (26 keys)
;* the code at that location is the next state or FF to stay in the same state
;*
;*******************************************************************************
;*******************************************************************************
;* These are the definitions of the states...
;*******************************************************************************
nstatetable:
;*****+10,+20,VlU,VlD,ChU,ChD,Mut,Ply,FFo,Rew,Stp,Pau,Rec,Lne,Pow,000,001,002,003,004,005,006,007,008,009,Emp,ERR,OK ,OK2,DEF,X.5,X.6
;**** State:0 Neutral State Transition Table
.db   $00,$0C,$00,$00,$00,$00,$00,$01,$00,$00,$00,$00,$00,$00,$00,$0D,$0E,$0F,$10,$11,$12,$13,$14,$15,$16,$00,$05,$00,$00,$00,$00,$00
.db   $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$00,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$05,$03,$04,$02,$02,$02
.db   $02,$02,$02,$02,$02,$02,$02,$02,$0A,$0B,$08,$07,$00,$00,$02,$0D,$0E,$0F,$10,$11,$12,$13,$14,$15,$16,$01,$05,$02,$00,$02,$02,$02
.db   $00,$03,$00,$00,$00,$00,$00,$01,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$05,$00,$00,$00,$00,$00
.db   $00,$03,$00,$00,$00,$00,$00,$01,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$05,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $08,$08,$08,$08,$08,$08,$08,$01,$08,$08,$08,$08,$08,$08,$08,$0D,$0E,$0F,$10,$11,$12,$13,$14,$15,$16,$00,$05,$00,$00,$00,$00,$00
.db   $07,$07,$07,$07,$07,$07,$07,$09,$0A,$0B,$08,$09,$00,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$05,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$05,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$01,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$02,$05,$00,$00,$00,$00,$00
.db   $0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$00,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$01,$05,$00,$00,$00,$00,$00
.db   $0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$00,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$01,$05,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$05,$03,$04,$00,$00,$00
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
.db   $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
;*******************************************************************************
;* jump to the current state's actual routine as defined in the table below
;*******************************************************************************
;* State     Means
;*  $00      Neutral State
;*  $01      Init Play State
;*  $02      (Current) Play(ing) State
;*  $03      OpenTray State                 
;*  $04      CloseTray State
;*  $05      Error State
;*  $06      Enter track no. state
;*  $07      Pause State
;*  $08      Stop State
;*  $09      Resume state      
;*  $0A      fastf state
;*  $0B      rewind state
;*  $0C
;*  $0D
;*  $0E
;*  $0F
actualstate:
      ldi   ZL,low(actualstatetable)
      ldi   ZH,high(actualstatetable)
istate:      
      mov   temp0,drive
      cpi   temp0,$00
      brne  psst0
      mov   temp0,mstate
      rjmp  psst1
psst0:
      mov   temp0,sstate
psst1:
      add   ZL,temp0              ;state number becomes index
      clr   temp0
      adc   ZH,temp0              ;find the correct rountine...
      ijmp                        ; and run it
actualstatetable:
;*******************************************************************************
;This is the actual state table from state 0 upwards...
;*******************************************************************************
      rjmp   neutralstate        ;* state 0
      rjmp   initplaystate       ;* state 1
      rjmp   playstate           ;* state 2
    	rjmp   opentraystate       ;* state 3
    	rjmp   closetraystate      ;* state 4
    	rjmp   errorstate          ;* state 5
      rjmp   entertracknumstate  ;* state 6
      rjmp   pausestate          ;* state 7
      rjmp   stopstate           ;* state 8
      rjmp   resumestate         ;* state 9
      rjmp   fastfstate          ;* state 10  
      rjmp   rewindstate         ;* state 11
      rjmp   openclosestate      ;* state 12
      rjmp   enter0state         ;* state 13
      rjmp   enter1state         ;* state 14
      rjmp   enter2state         ;* state 15
      rjmp   enter3state         ;* state 16
      rjmp   enter4state         ;* state 17
      rjmp   enter5state         ;* state 18
      rjmp   enter6state         ;* state 19
      rjmp   enter7state         ;* state 20
      rjmp   enter8state         ;* state 21
      rjmp   enter9state         ;* state 22
      ret
      ret
      ret
      ret
      ret
      ret
    	ret
    	ret
;*******************************************************************************
;*******************************************************************************
;* this is the state machine driving loop
;* prestate calls the prestate function of the current state
;* actualstate calls the actual state function of the current state
;* and should return a code in temp0 that corresponds to key pressed to
;* exit the state
;* poststate calls the post state function of the current state
;* nextstate uses the value in temp0 to go to the next state
;*******************************************************************************      
machine:
      rcall   actualstate
      rcall   nextstate
      rjmp    machine
;*******************************************************************************
;*******************************************************************************
;* Neutral State is the Idle Mode state...
neutralstate:
nstate9:
      rcall   getallide
      mov     temp0,random
      andi    temp0,$03
      cpi     temp0,$00
      breq    nstate1
      ldi     temp0,$01
      rcall   writelcdcontrol
      ldi     ZL,low(messageneutral)
      ldi     ZH,high(messageneutral)
      rcall   printmessage
      rjmp    nstate2
nstate1:
      ldi     temp0,$01
      rcall   writelcdcontrol
      
      rcall  modesensecommand
      brcc   nstate2 
      ; now mode sense was successful so read the medium byte in the header
      ; at location 0062
      clr   XH
      ldi   XL,$62
      ld    temp0,X
      ; if this byte is 0x70= door closed and no disc...
      ; if this byte is 0x71= door open...
      cpi   temp0,$70
      brne  nst9901
nst9902:
      ; here we've detected that the door is closed and no disc
      ; so open the door and display "insert disc"
      ldi   ZL,low(messageinsertdisc)
      ldi   ZH,high(messageinsertdisc)
      rcall printmessage
      rcall shortdelay
      rjmp  nstate2
nst9901:
      cpi   temp0,$71
      brne  nst9903
      rjmp  nst9902
nst9903:
      cpi   temp0,$10
      brsh  nst9904
      
      ldi   ZL,low(messagedisc)
      ldi   ZH,high(messagedisc)
      rcall printmessage
      rcall shortdelay
      rjmp  nstate2
nst9904:
      cpi   temp0,$20
      brsh  nst9905

      ldi   ZL,low(messagedisc)
      ldi   ZH,high(messagedisc)
      rcall printmessage
      rcall shortdelay
      
      rjmp  nstate2
nst9905:      
      cpi   temp0,$30
      brsh  nst9906
      
      ldi   ZL,low(messagedisc)
      ldi   ZH,high(messagedisc)
      rcall printmessage
      rcall shortdelay
      rjmp  nstate2      
nst9906:
nstate2:
      inc     random
nstate0:
      rcall   getinput
      brcc    nstate9
      push    temp0
      ldi     temp0,$01
      rcall   writelcdcontrol
      pop     temp0
      sec
      ret

messageneutral:
.db   'R','e','a','d','y',' ',$80,$80
messagedisc:
.db   'D','i','s','c',' ','R','e','a','d','y',' ',$80

errorhandler:
      ; an error has occurred
      ldi     temp0,ERROR
      clc
      ret                       

getnexttrack:
      ; work out the next track to play...
      mov   temp0,drive
      andi  temp0,$02
      brne  gnex0
      mov   temp1,mnexttrack
      rjmp  gnex1      
gnex0:
      mov   temp1,snexttrack
gnex1:
      mov   temp0,mode
      andi  temp0,$02
      brne  gnex100
gnex90:
      mov   temp0,mode
      andi  temp0,$04
      breq  gnex200
      ; here we must select a random track... To do this we must know
      ; how many total tracks there are on the CD!
      ; this number is obtained from the TOC...
      rcall toccommand
      brcc  gnex200         ; an error has occurred so just increment...
      ; otherwise we've read the TOC and the information is sitting
      ; at location $0062=first track 0063=last track...
      
      in    temp1,TCNT0
      
      ; now temp1 is a random number from 00 to FF
      ; given by reading the timer at a (random) time...
      
      ; we must now scale this random number to fit with the total tracks...
      ; as follows:
      ; randomtrack= remainder(timer/(1+last-first)) + first
      
      clr     XH
      ldi     XL,$63
      ld      temp0,X
      inc     temp0
      ldi     XL,$62
      ld      temp2,X
      push    temp2
      sub     temp0,temp2
      
      mov     div2,temp0
      clr     div1
      clr     byte1
      clr     byte2
      clr     byte3
      in      byte4,TCNT0
      rcall   fpudiv3216
      
      mov     temp0,rem2
      pop     temp2
      add     temp0,temp2
      mov     temp1,temp0       ; now temp1 is the random track
      rjmp    gnex100
gnex200:
      inc     temp1
gnex100:
      mov   temp0,drive
      andi  temp0,$02
      brne  gnex91
      mov   mnexttrack,temp1
      ret
gnex91:
      mov   snexttrack,temp1
      ret
      
getprevtrack:
      ; work out the (next) previous track to play...
      mov   temp0,drive
      andi  temp0,$02
      brne  gprev0
      mov   temp1,mnexttrack
      rjmp  gprev1      
gprev0:
      mov   temp1,snexttrack
gprev1:
      mov   temp0,mode
      andi  temp0,$02
      brne  gprev90
      dec   temp1
gprev90:
      mov   temp0,drive
      andi  temp0,$02
      brne  gprev91
      mov   mnexttrack,temp1
      ret
gprev91:
      mov   snexttrack,temp1
      ret

fastfstate:
      rcall getnexttrack
      ldi   temp0,EMPTY
      sec
      ret
      

rewindstate:
      rcall getprevtrack
      ldi   temp0,EMPTY
      sec
      ret
      
initplaystate:
;*** this state initializes playing from a non-playing state
      rcall  modesensecommand
      brcc   ipst00      ; if error in sensing at least try to play it!
      ; now mode sense was successful so read the medium byte in the header
      ; at location 0062
      clr   XH
      ldi   XL,$62
      ld    temp0,X
      ; if this byte is 0x70= door closed and no disc...
      ; if this byte is 0x71= door open...
      cpi   temp0,$70
      brne  ipst01
      ; here we've detected that the door is closed and no disc
      ; so open the door and display "insert disc"
      ldi   temp0,$01
      rcall writelcdcontrol
      ldi   ZL,low(messageinsertdisc)
      ldi   ZH,high(messageinsertdisc)
      rcall printmessage
      rcall longdelay
      ldi   temp0,OK
      sec
      ret                                ; goto eject cd state
ipst01:
      cpi   temp0,$71
      brne  ipst02
      
      ldi   temp0,$01
      rcall writelcdcontrol
      ldi   ZL,low(messageclosedoor)
      ldi   ZH,high(messageclosedoor)
      rcall printmessage
      rcall longdelay
      ldi   temp0,OK2                   ; goto close tray state
      sec
      ret
ipst02:
ipst00:
      push    mode
      ldi     temp0,$01
      or      mode,temp0
      rcall   lockcdcommand
      brcs    ipst000
      pop     mode
      rjmp    errorhandler
ipst000:
      pop     mode
      rcall   playnexttrackcommand
      brcs    ipst0
      rjmp    errorhandler
ipst0:
      ldi     temp0,EMPTY       ; and immediately return to the playing state...
      sec
      ret

messageinsertdisc:
.db   'I','n','s','e','r','t',' ','D','i','s','c',' ',$80,$80
messageclosedoor:
.db   'T','r','a','y',' ','O','p','e','n',' ',' ',' ',$80,$80

playstate:
playst0:
      rcall   currentcdposition
      brcc    playst1
      rcall   getinput
      brcc    playst0
      ret      
playst1:
      ldi     temp0,EMPTY
      sec
      ret
      
swapdrivesstate:
      ldi     temp0,$02
      eor     drive,temp0
      rcall   protectselectdrive
      ldi     temp0,EMPTY
      ret
      
opentraystate:
      rcall   ejectcommand
      brcs    opst0
      rjmp    errorhandler
opst0:
      ldi     temp0,EMPTY
      ret
      
closetraystate:
      rcall   closetraycommand
      brcs    ctst0
      rjmp    errorhandler
ctst0:
      ldi     temp0,EMPTY
      ret


openclosestate:
      rcall   modesensecommand
      brcc    oc0
      ldi     XH,$00
      ldi     XL,$62
      ld      temp0,X
      cpi     temp0,$71
      brne    oc0
      ldi     temp0,OK2
      sec
      ret
oc0:
      ldi     temp0,OK
      sec
      ret

errorstate:
      ldi     temp0,$01
      rcall   writelcdcontrol
      ldi     ZH,high(messageerror)
      ldi     ZL,low(messageerror)    
      rcall   printmessage
      ldi     temp0,$C0
      rcall   writelcdcontrol
      mov     temp0,error
      rcall   disa
      rcall   printide
errorwait:
      rcall   longdelay
      rcall   longdelay
      rcall   longdelay
      rcall   getallide
      ldi     temp0,ERROR
      clc
      ret
      
messageerror:
.db   'E','r','r','o','r',':',$80,$80

pausestate:
      rcall   pausecommand
      brcs    ppst0
      rjmp    errorhandler
ppst0:
      rcall   getinput
      brcc    ppst0
      sec
      ret

resumestate:
      rcall   resumecommand
      brcs    rsmst0
      rjmp    errorhandler
rsmst0:
      ldi     temp0,EMPTY
      sec
      ret

stopstate:
      rcall   stopplaycommand
      brcs    stst0
      rjmp    errorhandler
stst0:
      ldi     temp0,EMPTY
      sec
      ret

entertracknumstate:
      ldi     temp0,$01
      rcall   writelcdcontrol
      ldi     ZL,low(messageentertrack)
      ldi     ZH,high(messageentertrack)
      rcall   printmessage
      
      mov     temp0,drive
      andi    temp0,$02
      brne    etst0
      mov     temp1,mnexttrack
      rjmp    etst1
etst0:
      mov     temp1,snexttrack
etst1:
      mov     temp0,temp1
      ldi     temp1,$02
      rcall   disadecimal
      rcall   shortdelay
etst2:
      rcall   getinput
      brcc    etst2
      ret

enter0state:
      ldi     temp1,$00
      rjmp    est0 
enter1state:
      ldi     temp1,$01
      rjmp    est0
enter2state:
      ldi     temp1,$02
      rjmp    est0
enter3state:
      ldi     temp1,$03
      rjmp    est0
enter4state:
      ldi     temp1,$04
      rjmp    est0
enter5state:
      ldi     temp1,$05
      rjmp    est0
enter6state:
      ldi     temp1,$06
      rjmp    est0
enter7state:
      ldi     temp1,$07
      rjmp    est0
enter8state:
      ldi     temp1,$08
      rjmp    est0
enter9state:
      ldi     temp1,$09
est0:
      mov     temp0,drive
      andi    temp0,$02
      brne    est1
    
      push    temp1
      mov     div2,mnexttrack
      ldi     temp0,$0a
      mov     div1,temp0
      rcall   mul88
      pop     temp1
      add     byte4,temp1
      ldi     temp1,$00
      adc     byte3,temp1
      clr     byte2
      clr     byte1
      ldi     temp0,$64
      mov     div2,temp0
      clr     div1
      rcall   fpudiv3216
      mov     mnexttrack,rem2
      ldi     temp0,EMPTY
      sec
      ret
est1:
      push    temp1
      mov     div2,snexttrack
      ldi     temp0,$0a
      mov     div1,temp0
      rcall   mul88
      pop     temp1
      add     byte4,temp1
      ldi     temp1,$00
      adc     byte3,temp1
      clr     byte2
      clr     byte1
      ldi     temp0,$64
      mov     div2,temp0
      clr     div1
      rcall   fpudiv3216
      mov     snexttrack,rem2
      ldi     temp0,EMPTY
      sec
      ret

messageentertrack:
.db   'E','n','t','e','r',' ','T','r','a','c','k',':',' ',$80
;*******************************************************************************
