
; Shift lights and limiter

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB0  & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF
; Define variables at memory locations


EEPROM0		equ	H'00'	; non-volatile storage for rpm1 ms shift 1
EEPROM1		equ	H'01'	; non-volatile storage for rpm1 ls 
EEPROM2		equ	H'02'	; non-volatile storage for rpm2 ms shift 2
EEPROM3		equ	H'03'	; non-volatile storage for rpm2 ls
EEPROM4		equ	H'04'	; non-volatile storage for rpm3 ms shift 3
EEPROM5		equ	H'05'	; non-volatile storage for rpm3 ls
EEPROM6		equ	H'06'	; non-volatile storage for rpm4 ms (limit)
EEPROM7		equ	H'07'	; non-volatile storage for rpm4 ls
EEPROM8		equ	H'08'	; non-volatile storage for hysteresis for rpm1 ms
EEPROM9		equ	H'09'	; non-volatile storage for hysteresis for rpm1 ls
EEPROMA		equ	H'0A'	; non-volatile storage for hysteresis for rpm2 ms
EEPROMB		equ	H'0B'	; non-volatile storage for hysteresis for rpm2 ls
EEPROMC		equ	H'0C'	; non-volatile storage for hysteresis for rpm3 ms
EEPROMD		equ	H'0D'	; non-volatile storage for hysteresis for rpm3 ls
EEPROME		equ	H'0E'	; non-volatile storage for hysteresis for rpm4 ms
EEPROMF		equ	H'0F'	; non-volatile storage for hysteresis for rpm4 ls
EEPROMG		equ	H'10'	; non-volatile storage for dimming minimum
EEPROMH		equ	H'11'	; non-volatile storage for PWM update rate
EEPROMI		equ	H'12'	; non-volatile storage for RPM Averaging value

; Bank 0 RAM

CYL_SET		equ	H'20'	; cylinder setting
COUNT0		equ	H'21'	; 32-bit rpm counter ms
COUNT1		equ	H'22'	; 32-bit rpm counter
COUNT2		equ	H'23'	; 32-bit rpm counter
COUNT3		equ	H'24'	; 32-bit rpm counter ls
RPM_VAL0	equ	H'25'	; ms byte calculated RPM
RPM_VAL1	equ	H'26'	; ls byte calculated RPM
COUNTER0	equ	H'27'	; ms 32-bit timer1 overflow counter
COUNTER1	equ	H'28'	; 32-bit timer1 overflow counter
NIL_SIG		equ	H'29'	; flag indicating if timer1 continues to overflows beyond COUNTER0
DELCNT		equ	H'2A'	; delay counter
VALUE_1		equ	H'2B'	; delay value
VALUE_2		equ	H'2C'	; delay value
SHIFT1_MS	equ	H'2D'	; current rpm for shift light1 ms
SHIFT1_LS	equ	H'2E'	; current rpm for shift light1 ls
SHIFT2_MS	equ	H'2F'	; current rpm for shift light2 ms
SHIFT2_LS	equ	H'30'	; current rpm for shift light2 ls
SHIFT3_MS	equ	H'31'	; current rpm for shift light3 ms
SHIFT3_LS	equ	H'32'	; current rpm for shift light3 ls
SHIFT4_MS	equ	H'33'	; current rpm for shift light4 ms
SHIFT4_LS	equ	H'34'	; current rpm for shift light4 ls
HYSTER1_MS	equ	H'35'	; current hysteresis for shift 1 ms
HYSTER1_LS	equ	H'36'	; current hysteresis for shift 1 ls
HYSTER2_MS	equ	H'37'	; current hysteresis for shift 2 ms
HYSTER2_LS	equ	H'38'	; current hysteresis for shift 2 ls
HYSTER3_MS	equ	H'39'	; current hysteresis for shift 3 ms
HYSTER3_LS	equ	H'3A'	; current hysteresis for shift 3 ls
HYSTER4_MS	equ	H'3B'	; current hysteresis for shift 4 ms
HYSTER4_LS	equ	H'3C'	; current hysteresis for shift 4 ls
STORE0		equ	H'3D'	; rpm storage
STORE1		equ	H'3E'	; rpm storage
DELY		equ	H'3F'	; delay for A/D input
DIM_VAL		equ	H'40'	; dimming value
DIM_COUNT	equ	H'41'	; dimming counter
SHIFT1_ON	equ	H'42'	; shift1 LED flag
SHIFT2_ON	equ	H'43'	; shift2 LED flag
SHIFT3_ON	equ	H'44'	; shift3 LED flag
SHIFT4_ON	equ	H'45'	; shift4 flag
TEMP1		equ	H'46'	; temporary register
TEMP2		equ	H'47'	; temporary register
PWM_COUNT	equ	H'48'	; pwm counter
CONV_RATE	equ	H'49'	; A/D conversion rate
DIM_MIN		equ	H'4A'	; dimming minimum 
WITH_RANG	equ	H'4B'	; within range flag off
PWM_RANGE	equ	H'4C'	; range of RPM for PWM
TEMP_HYST	equ	H'4D'	; modified hysteresis
PWM_RANGE_STO equ H'4E'	; pwm range storage
UP_RANG_COUNT equ H'4F'	; counter for RPM range update
RANGE_COUNT	equ	H'50'	; stored value for UP_RANG_COUNT
TOTAL0		equ	H'51'	; rpm addition value for averaging ms byte
TOTAL1		equ	H'52'	; rpm addition value for averaging mid byte
TOTAL2		equ	H'53'	; rpm addition value for averaging ls byte
TALLY_CNT	equ	H'54'	; rpm addition tally counter (divide by this for average)
RPM_AVG0	equ	H'55'	; rpm average ms byte
RPM_AVG1	equ	H'56'	; rpm average ls byte
AVG_COUNT	equ	H'57'	; stored average total value
SETTINGS	equ	H'58'	; settings mode
FLASH		equ	H'59'	; LED flash counter
DIMVAL_TEMP	equ	H'5A'	; temporary dimming value
TRANSFER	equ	H'5B'	; rpm data transferred flag
			
; math routines
TEMPB0		equ	H'5D'
TEMPB1		equ	H'5E'
TEMPB2		equ	H'5F'
TEMP		equ H'60'
REMB3		equ H'61'
REMB2		equ	H'62'
REMB1      	equ H'63'
REMB0		equ	H'64'
AARGB5		equ	H'65'
AARGB4      equ H'66'
AARGB3		equ	H'67'
AARGB2      equ H'68'
AARGB1      equ H'69'
AARGB0      equ H'6A'	; most significant byte of argument A
BARGB3      equ H'6B'
BARGB2      equ H'6C'
BARGB1      equ H'6D'
BARGB0      equ H'6E'	; most significant byte of argument B
LOOPCOUNT   equ H'6F'  	; division counter

; All Banks RAM (70 to 7F)

W_TMP		equ	H'70'	; storage of w before interrupt
STATUS_TMP	equ	H'71'	; status storage before interrupt


NUMER_0		equ	H'73'	; rpm numerator value ms byte
NUMER_1		equ	H'74'	; rpm numerator value mid byte
NUMER_2		equ	H'75'	; rpm numerator value mid byte
NUMER_3		equ	H'76'	; rpm numerator value ls byte

	
; preprogram EEPROM DATA 
; initial values	
	ORG     H'2100'
	DE		H'03', H'E8'; shift 1 1000rpm
	DE		H'07', H'D0'; shift 2 2000rpm
	DE		H'0B', H'B8'; shift 3 3000rpm
	DE		H'0F', H'A0'; shift 4 4000rpm
	DE		H'00', H'C8'; hysteresis 1 200rpm
	DE		H'01', H'F4'; hysteresis 2 500rpm
	DE		H'01', H'F4'; hysteresis 3 500rpm
	DE		H'01', H'F4'; hysteresis 4 500rpm
	DE		D'0'		; dimming minimum value
	DE		D'2' 		; PWM update rate
	DE		D'6'		; rpm averaging for PWM limiting

; start at memory 0

	org		H'0'	; main program start
	goto	MAIN
	org 	H'4'	; interrupt vector 0004h, start interrupt routine here
;	goto	INTRUPT		

;******************************************************************************************* 


; Calculations

; 1.
; 1/2 pulse per rpm (1cyl 4-stroke)
; Timer 1 internal clock count /1
; capture every + edge
; 1Hz (120rpm) has 2M count (numerator needs to be 120 x 2 x 10^6)
; Calculations: min 240M/2M = 120rpm 

; 2.
; 1 pulse per rpm (2cyl 4-stroke, 1 cyl 2-stroke)
; Timer 1 internal clock count /1
; capture every + edge
; 1Hz (60rpm) has 2M count (numerator needs to be 60 x 2 x 10^6)
; Calculations: min 120M/2M = 60rpm 

; 3.
; 1.5 pulse per rpm (3cyl 4-stroke)
; Timer 1 internal clock count /1
; capture every + edge
; 1Hz (40rpm) has 2M count (numerator needs to be 40 x 2 x10^6)
; Calculations: min 80M/2M = 40rpm 

; 4.
; 2 pulses per rpm (4cyl 4-stroke, 2 cyl 2-stroke)
; Timer 1 internal clock count /1
; capture every positive edge 
; 1Hz (30rpm) has 2MHz count (numerator needs to be 30 x 2 x 10^6)
; Calculations: min 60M/2M = 30rpm 

; 5.
; 2.5 pulses per rpm (5cyl 4-stroke)
; Timer 1 internal clock count /1
; capture every + edge
; 1Hz (24rpm) has 2M count (numerator needs to be 24 x 2 x 10^6)
; Calculations: min 48M/2M = 24rpm 

; 6.
; 3 pulses per rpm (6cyl 4-stroke, 3 cyl 2-stroke)
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (20rpm) has 4M count (numerator needs to be 20 x 4 x 10^6)
; Calculations: min 80M/4M = 20rpm 

; 7 (and 11) (caters for asymmetrical twin engine)
; 1 pulse per rpm (2cyl 4-stroke, 1 cyl 2-stroke)
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (60rpm) has 4M count (numerator needs to be 60 x 4 x 10^6)
; Calculations: min 240M/4M = 60rpm 

; 8.
; 4 pulses per rpm (8cyl 4-stroke, 4 cyl 2-stroke)
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (15rpm) has 4M count (numerator needs to be 15 x 4 x 10^6)
; Calculations: min 60M/4M = 15rpm 

; 9 caters for asymmetrically firing 3-cylinder 4-stroke
; 1.5 pulse per rpm (3cyl 4-stroke)
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (40rpm) has 4M count (numerator needs to be 40 x 4 x10^6)
; Calculations: min 160M/4M = 40rpm 
 
; 11 see 7

; 10.
; 5 pulses per rpm (10 cyl 2-stroke)
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (12rpm) has 4M count (numerator needs to be 12 x 4 x 10^6)
; Calculations: min 48M/4M = 12rpm 

; 12
; 6 pulses per rpm (12cyl 4-stroke, 6 cyl 2-stroke)
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (10rpm) has 4M count (numerator needs to be 10 x 4 x 10^6)
; Calculations: min 40M/4M = 10rpm 		

; INTERRUPT

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp 
	bcf 	STATUS,RP0	; bank 0
	bcf		STATUS,RP1 
	btfss	PIR1,CCP1IF	; if capture place into Count2,3, clear NIL_SIG 
	goto	CK_TMR_O
	bcf		T1CON,0		; stop timer
	
; check for capture before 
; overflow flag set 
	
	btfss	PIR1,TMR1IF	; if timer overflow flag set, ignore values 
	goto	XFER_VAL	; capture and no overflow then values are valid
TMR_CLFLG
	bcf		PIR1,TMR1IF	; clear flag
	bcf		PIR1,CCP1IF	; clear flag
	clrf	NIL_SIG		; signal captured so clear value 
	
	goto	CLR_COUNT

; check timer overflow
CK_TMR_O	
	btfss	PIR1,TMR1IF	; if timer overflow flag set, increase COUNTER1
	goto	TIMER2		; 
	bcf		PIR1,TMR1IF	; clear flag
	incfsz	COUNTER1,f	; overflow counter FFFF to 0000
	goto	TIMER2
	incf	COUNTER0,f	; increase ms byte if ls byte overflows
	btfsc	COUNTER0,0	; if past count of ~ 67 000 000 clear
	bsf		NIL_SIG,0	; set flag if overflows beyond COUNTER0 
	goto	TIMER2
XFER_VAL
	bcf		PIR1,CCP1IF	; clear flag
	clrf	NIL_SIG		; signal captured so clear value 

; transfer COUNTER0,1 to COUNT0,1
; CCPR1L and H to COUNT2,3
	movf	COUNTER0,w
	movwf	COUNT0		; ms 32-bit rpm counter
	movf	COUNTER1,w
	movwf	COUNT1		; 32-bit rpm counter
	movf	CCPR1H,w	; capture compare value
	movwf	COUNT2		; 32-bit rpm counter
	movf	CCPR1L,w
	movwf	COUNT3		; ls 32-bit rpm counter

; clear counters
CLR_COUNT
	movf	CCP1CON,w	; store value
	clrf	CCP1CON		; clears prescaler
	movwf	CCP1CON		; restore value capture compare (bank 0)
	clrf	COUNTER0
	clrf	COUNTER1
	clrf	CCPR1L
	clrf	CCPR1H

	clrf	TMR1L
	clrf	TMR1H
	bsf		TRANSFER,0	; transferred flag
	bsf		T1CON,0		; restart timer

TIMER2	
; check timer 2 overflow for dimming

	btfss	PIR1,TMR2IF		; if timer overflow flag set, increase COUNTER1
	goto	RECLAIM			; not timer 2
	bcf		PIR1,TMR2IF		; clear flag
LEDS_SHOW
	movf	DIM_VAL,w
	btfsc	STATUS,Z		; if zero light
	goto	LIGHT
	subwf	DIM_COUNT,w		; light when counter > value 	
	btfss	STATUS,C
	goto	NO_LIGHT
LIGHT
	btfss	SHIFT1_ON,0		; when set set port
	goto	OFF_1
	bsf		PORTA,6			; shift LED1	
S_2
	btfss	SHIFT2_ON,0		; when set set port
	goto	OFF_2
	bsf		PORTA,7			; shift LED2	
S_3
	btfss	SHIFT3_ON,0		; when set set port
	goto	OFF_3
	bsf		PORTA,0			; shift LED3	
S_4
	btfss	SHIFT4_ON,0		; when set set port
	goto	OFF_4
	bsf		PORTA,2			; limit relay
	goto	BY_NO_LIGHT

OFF_1
	bcf		PORTA,6
	goto	S_2
OFF_2
	bcf		PORTA,7
	goto	S_3
OFF_3
	bcf		PORTA,0
	goto	S_4
OFF_4
	bcf		PORTA,2
	goto	BY_NO_LIGHT

NO_LIGHT
	bcf		PORTA,6			; LED off for dimming
	bcf		PORTA,7
	bcf		PORTA,0
	btfss	SHIFT4_ON,0		; if set leave on
	bcf		PORTA,2			; limit relay off

BY_NO_LIGHT
	incf	DIM_COUNT,f
	movlw	D'64'
	subwf	DIM_COUNT,w
	btfsc	STATUS,C		; if >64 return to zero
	clrf	DIM_COUNT

; generate limit PWM
PWM_LIMIT
	btfss	WITH_RANG,0		; if set then PWM limiter required
	goto	RESET_COUNTER
	incf	PWM_COUNT,f
	movf	PWM_COUNT,w		; get value
	subwf	TEMP_HYST,w		; if over hysteresis value then reset
	btfsc	STATUS,C
	goto	COMP_PWM_DRV
	clrf	PWM_COUNT		; PWM counter cleared at end of count
; update rpm range value after x cycles
	decfsz	UP_RANG_COUNT,f	; update counter
	goto	COMP_PWM_DRV
	movf	RANGE_COUNT,w	; number of cycles before update
	movwf	UP_RANG_COUNT
UPDATE_RNG	
	movf	PWM_RANGE,w		; rpm range
	movwf	PWM_RANGE_STO	; stored each cycle
; compare with pwm counter
COMP_PWM_DRV
	movf	PWM_RANGE_STO,w		; rpm range for PWM
	subwf	PWM_COUNT,w
	btfsc	STATUS,C
	goto	SET_PWM
	bcf		PORTA,1
	goto	RECLAIM
SET_PWM
	bsf		PORTA,1
	goto	RECLAIM

RESET_COUNTER ; PWM not required so set up ready
	movf	RANGE_COUNT,w	; number of cycles before update
	movwf	UP_RANG_COUNT
	movf	PWM_RANGE,w		; rpm range
	movwf	PWM_RANGE_STO	; stored each cycle

; end of interrupt reclaim w and status 
RECLAIM
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

; *********************************************************************************

; initialise ports

MAIN

; set inputs/outputs
	clrf	PORTA
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'11111111'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'00111000'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000111'	; settings (pullups enabled TMR0/256)
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'00111000'	; AN3 to AN5 are analog inputs
	movwf	ANSEL
	movlw	B'01000000'	; left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	bsf		T2CON,2		; timer 2 on
		
	movlw	B'01101000'	; Fosc, channel 5 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	clrf	PORTA
	bsf		STATUS,RP0	; select memory bank 1
	movlw	H'FF'
	movwf	PR2			; timer 2 preload (for frequency adjust)
	movlw	B'01111000'	; 8MHz
	movwf	OSCCON		; 
	bcf		STATUS,RP0	; memory bank 0
	clrf	PORTA
	bsf		PORTA,1		; injector drive on

; Read EEPROM values and store in registers

	movlw	EEPROM0 	; non-volatile storage for rpm1 ms shift 1
	call	EREAD		; get value
	movwf	SHIFT1_MS 	; current rpm for shift light1 ms
	movlw	EEPROM1 	; non-volatile storage for rpm1 ls
	call	EREAD		; get value
	movwf	SHIFT1_LS 	; current rpm for shift light1 ls 

	movlw	EEPROM2 	; non-volatile storage for rpm2 ms shift 2
	call	EREAD		; get value
	movwf	SHIFT2_MS 	; current rpm for shift light2 ms
	movlw	EEPROM3 	; non-volatile storage for rpm2 ls
	call	EREAD		; get value
	movwf	SHIFT2_LS 	; current rpm for shift light2 ls 

	movlw	EEPROM4 	; non-volatile storage for rpm3 ms shift 3
	call	EREAD		; get value
	movwf	SHIFT3_MS 	; current rpm for shift light3 ms
	movlw	EEPROM5 	; non-volatile storage for rpm3 ls
	call	EREAD		; get value
	movwf	SHIFT3_LS 	; current rpm for shift light3 ls 

	movlw	EEPROM6 	; non-volatile storage for rpm4 ms shift 4
	call	EREAD		; get value
	movwf	SHIFT4_MS 	; current rpm for shift light4 ms
	movlw	EEPROM7 	; non-volatile storage for rpm4 ls
	call	EREAD		; get value
	movwf	SHIFT4_LS 	; current rpm for shift light4 ls 

	movlw	EEPROM8 	; non-volatile storage for hysteresis for rpm1 ms
	call	EREAD		; get value
	movwf	HYSTER1_MS	; current hysteresis for shift 1 ms
	movlw	EEPROM9		; non-volatile storage for hysteresis for rpm1 ls	
	call	EREAD		; get value
	movwf	HYSTER1_LS	; current hysteresis for shift 1 ls

	movlw	EEPROMA 	; non-volatile storage for hysteresis for rpm2 ms
	call	EREAD		; get value
	movwf	HYSTER2_MS	; current hysteresis for shift 2 ms
	movlw	EEPROMB		; non-volatile storage for hysteresis for rpm2 ls	
	call	EREAD		; get value
	movwf	HYSTER2_LS	; current hysteresis for shift 2 ls

	movlw	EEPROMC 	; non-volatile storage for hysteresis for rpm3 ms
	call	EREAD		; get value
	movwf	HYSTER3_MS	; current hysteresis for shift 3 ms
	movlw	EEPROMD		; non-volatile storage for hysteresis for rpm3 ls	
	call	EREAD		; get value
	movwf	HYSTER3_LS	; current hysteresis for shift 3 ls

	movlw	EEPROME 	; non-volatile storage for hysteresis for rpm4 ms
	call	EREAD		; get value
	movwf	HYSTER4_MS	; current hysteresis for shift 4 ms
	movlw	EEPROMF		; non-volatile storage for hysteresis for rpm4 ls	
	call	EREAD		; get value
	movwf	HYSTER4_LS	; current hysteresis for shift 4 ls

	bcf		STATUS,C
	rrf		HYSTER4_MS,w; divide 16-bits/2
	rrf		HYSTER4_LS,w
	movwf	TEMP_HYST	; used in interrupt to drive PWM

	movlw	EEPROMG		; non-volatile storage for dimming
	call	EREAD		; get value
	movwf	DIM_MIN		; dimming minimum value

	movlw	EEPROMH		; non-volatile storage for PWM update rate	
	call	EREAD		; get value
	movwf	RANGE_COUNT
	
	movlw	EEPROMI		; non-volatile storage for PWM update rate	
	call	EREAD		; get value
	movwf	AVG_COUNT	; stored average total value

	call	DELAY

; ***********************************************************

; check cylinders and set required timer settings and numerator
CYL_VALUES

	clrf	CYL_SET		; start at 0
	btfss	PORTB,4		; if low set bit 0
	bsf		CYL_SET,0
	btfss	PORTB,3		; if low set bit 1
	bsf		CYL_SET,1
	btfss	PORTB,1		; if low set bit 2
	bsf		CYL_SET,2
	btfss	PORTB,2		; if low set bit 3
	bsf		CYL_SET,3	
; cyl_set has BCD switch value

	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'01'		; 1 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_1
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'02'		; 2 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_2
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'03'		; 3 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_3
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'04'		; 4 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_4
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'05'		; 5 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_5
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'06'		; 6 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_6
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'08'		; 8 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_8
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'09'		; 3 cyl 4-stroke (asymmetrically fired)
	btfsc	STATUS,Z
	goto	NO_9
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'10'		; 10 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_10
	movf	CYL_SET,w	; cylinder setting value 1 to 12
	xorlw	D'12'		; 12 cyl 4-stroke
	btfsc	STATUS,Z
	goto	NO_12

; numbers 7 and 11 cater for assymetrical twin 4-stroke
NO_7_11					; numerator is 240M
	movlw	H'0E'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'4E'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'1C'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	SIX_TWELVE_SETTING
NO_1					; numerator is 240M
	movlw	H'0E'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'4E'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'1C'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	ONE_FIVE_SETTING
NO_2					; numerator is 120M
	movlw	H'07'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'27'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'0E'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	ONE_FIVE_SETTING
NO_3					; numerator is 80M
	movlw	H'04'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'C4'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'B4'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	ONE_FIVE_SETTING
NO_4					; numerator is 60M
	movlw	H'03'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'93'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'87'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	ONE_FIVE_SETTING
NO_5					; numerator is 48M
	movlw	H'02'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'DC'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'6C'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	ONE_FIVE_SETTING
NO_6					; numerator is 80M
	movlw	H'04'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'C4'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'B4'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	SIX_TWELVE_SETTING
NO_8					; numerator is 60M
	movlw	H'03'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'93'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'87'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	SIX_TWELVE_SETTING
NO_9					; numerator is 160M
	movlw	H'09'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'89'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'68'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	SIX_TWELVE_SETTING
NO_10					; numerator is 48M
	movlw	H'02'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'DC'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'6C'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	SIX_TWELVE_SETTING
NO_12					; numerator is 40M
	movlw	H'02'
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'62'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'5A'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
	goto	SIX_TWELVE_SETTING
ONE_FIVE_SETTING	
	movlw	B'00000001'	; timer 1 prescaler /1 fosc/4
	movwf	T1CON		; bank 0
	movlw	B'00000101'	; every positive edge
	movwf	CCP1CON		; capture compare (bank 0)
	goto	INIT
SIX_TWELVE_SETTING	
	movlw	B'00010001'	; timer 1 prescaler /2 fosc/4
	movwf	T1CON		; bank 0
	movlw	B'00000110'	; every 4th edge
	movwf	CCP1CON		; capture compare (bank 0)

; initial conditions
INIT
	bsf		NIL_SIG,0	; set bit to cause overflow condition
	movlw	H'FF'		; set high the last rpm
	movwf	STORE0
	movwf	STORE1

	clrf	SHIFT1_ON	; shift1 LED flag
	clrf	SHIFT2_ON	; shift2 LED flag
	clrf	SHIFT3_ON	; shift3 LED flag
	clrf	SHIFT4_ON	; shift4 flag
	clrf	WITH_RANG	; within range flag off
	clrf	PWM_RANGE	; range of RPM for PWM
 	clrf	PWM_RANGE_STO; PWM range for PWM limiter
	movlw	D'01'
	movwf	UP_RANG_COUNT; counter for rpm range

; clear rpm and averaging values
	clrf	TRANSFER	; transferred flag
	clrf	TOTAL0		; rpm totals
	clrf	TOTAL1
	clrf	TOTAL2
	clrf	TALLY_CNT
	clrf	RPM_VAL0	; rpm value
	clrf	RPM_VAL1
	clrf	RPM_AVG0	; averaged value
	clrf	RPM_AVG1
	clrf	SETTINGS	; settings mode zero (normal run)

; enable interrupts
	bsf		STATUS,RP0	; select memory bank 1
	bsf		PIE1,TMR1IE	; timer1 interrupt enable for overflow (bank 1)
	bsf		PIE1,TMR2IE	; timer 2 Interrupt Enable
	bsf		PIE1,CCP1IE	; capture compare interrupt
	bcf		STATUS,RP0	; select memory bank 0
	bsf		INTCON,PEIE	; enable peripheral interrupts
	bsf		INTCON,GIE	; global interrupt enable

; measure AN5 for LDR level for dimming LEDs
CH_5AD
	bsf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5		; channel 5
	call	DELAYX			; wait
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV5
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV5
	movf	DIM_MIN,w		; subtract min value if <0 set at 0 (range is between 0 and D255-DIM_VAL)
	subwf	ADRESH,w
	btfss	STATUS,C		; check if over
	movlw	H'00'				
	movwf	DIMVAL_TEMP		; dimming value

; divide by 4 for 64 steps
	bcf		STATUS,C
	rrf		DIMVAL_TEMP,f		; /2
	bcf		STATUS,C
	rrf		DIMVAL_TEMP,w		; /4
	movwf	DIM_VAL				; value ued for dimming

SHIFT_RUN
; shift lights run mode
	decfsz	CONV_RATE,f
	goto 	FOUR_SHIFTS
;	movlw	D'60'
;	movwf	CONV_RATE		; dimming response
	goto	CH_5AD			; measure LDR

FOUR_SHIFTS
	btfss	TRANSFER,0		; when set new rpm data
	goto	SWITCH_PRESS	; check switch pressed
; for the 4-shifts
; if rpm above shift value set flag on (rpm > SHIFT1_MS, SHIFT1_LS, set SHIFT1_ON,0)
; SHIFT1
	movf	SHIFT1_MS,w		; ms of shift value
	subwf	RPM_VAL0,w		; ms rpm value 	
	movwf	TEMP1			; store
	movf	SHIFT1_LS,w		; ls of shift value
	subwf	RPM_VAL1,w		; ls rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfss	TEMP1,7			; if set then not greater
	bsf		SHIFT1_ON,0		; set flag when rpm > shift value
; SHIFT2
	movf	SHIFT2_MS,w		; ms of shift value
	subwf	RPM_VAL0,w		; ms rpm value 	
	movwf	TEMP1			; store
	movf	SHIFT2_LS,w		; ls of shift value
	subwf	RPM_VAL1,w		; ls rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfss	TEMP1,7			; if set then not greater
	bsf		SHIFT2_ON,0		; set flag when rpm > shift value
; SHIFT3
	movf	SHIFT3_MS,w		; ms of shift value
	subwf	RPM_VAL0,w		; ms rpm value 	
	movwf	TEMP1			; store
	movf	SHIFT3_LS,w		; ls of shift value
	subwf	RPM_VAL1,w		; ls rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfss	TEMP1,7			; if set then not greater
	bsf		SHIFT3_ON,0		; set flag when rpm > shift value
; SHIFT4
	movf	SHIFT4_MS,w		; ms of shift value
	subwf	RPM_AVG0,w		; ms averaged rpm value 	
	movwf	TEMP1			; store
	movf	SHIFT4_LS,w		; ls of shift value
	subwf	RPM_AVG1,w		; ls averaged rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfss	TEMP1,7			; if set then not greater
	bsf		SHIFT4_ON,0		; set flag when rpm > shift value

; if rpm less than shift value - hysteresis then clear SHIFT_ON 
; ie (rpm (RPM_VAL0, RPM_VAL1 < SHIFT1_MS,LS- HYSTER1_MS,LS then clear SHIFT1_ON)
; SHIFT1, subtract hysteresis
	movf	HYSTER1_MS,w	; ms of hysteresis value
	subwf	SHIFT1_MS,w		; ms shift value 	
	movwf	TEMP1			; store
	movf	HYSTER1_LS,w	; ls of hysteresis value
	subwf	SHIFT1_LS,w		; ls shift value 
	movwf	TEMP2			; store
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
; compare with rpm
	movf	TEMP1,w			; ms of shift - hysteresis
	subwf	RPM_VAL0,w		; ms rpm value 	
	movwf	TEMP1			; store
	movf	TEMP2,w			; ls of shift - hysteresis
	subwf	RPM_VAL1,w		; ls rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfsc	TEMP1,7			; if set then not greater
	bcf		SHIFT1_ON,0		; clear flag when rpm < shift - hysteresis

; SHIFT2, subtract hysteresis
	movf	HYSTER2_MS,w	; ms of hysteresis value
	subwf	SHIFT2_MS,w		; ms shift value 	
	movwf	TEMP1			; store
	movf	HYSTER2_LS,w	; ls of hysteresis value
	subwf	SHIFT2_LS,w		; ls shift value 
	movwf	TEMP2			; store
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
; compare with rpm
	movf	TEMP1,w			; ms of shift - hysteresis
	subwf	RPM_VAL0,w		; ms rpm value 	
	movwf	TEMP1			; store
	movf	TEMP2,w			; ls of shift - hysteresis
	subwf	RPM_VAL1,w		; ls rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfsc	TEMP1,7			; if set then not greater
	bcf		SHIFT2_ON,0		; clear flag when rpm < shift - hysteresis

; SHIFT3, subtract hysteresis
	movf	HYSTER3_MS,w	; ms of hysteresis value
	subwf	SHIFT3_MS,w		; ms shift value 	
	movwf	TEMP1			; store
	movf	HYSTER3_LS,w	; ls of hysteresis value
	subwf	SHIFT3_LS,w		; ls shift value 
	movwf	TEMP2			; store
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
; compare with rpm
	movf	TEMP1,w			; ms of shift - hysteresis
	subwf	RPM_VAL0,w		; ms rpm value 	
	movwf	TEMP1			; store
	movf	TEMP2,w			; ls of shift - hysteresis
	subwf	RPM_VAL1,w		; ls rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfsc	TEMP1,7			; if set then not greater
	bcf		SHIFT3_ON,0		; clear flag when rpm < shift - hysteresis

; SHIFT4, subtract hysteresis
	movf	HYSTER4_MS,w	; ms of hysteresis value
	subwf	SHIFT4_MS,w		; ms shift value 	
	movwf	TEMP1			; store
	movf	HYSTER4_LS,w	; ls of hysteresis value
	subwf	SHIFT4_LS,w		; ls shift value 
	movwf	TEMP2			; store
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
; compare with rpm
	movf	TEMP1,w			; ms of shift - hysteresis
	subwf	RPM_AVG0,w		; ms average rpm value 	
	movwf	TEMP1			; store
	movf	TEMP2,w			; ls of shift - hysteresis
	subwf	RPM_AVG1,w		; ls average rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfsc	TEMP1,7			; if set then not greater
	bcf		SHIFT4_ON,0		; clear flag when rpm < shift - hysteresis

; PWM limiting
; first check if over SHIFT_4 plus hysteresis use fully off signal
; add hysteris to SHIFT4
	movf	HYSTER4_MS,w	; ms of hysteresis value
	addwf	SHIFT4_MS,w		; ms shift value 	
	movwf	TEMP1			; store
	movf	HYSTER4_LS,w	; ls of hysteresis value
	addwf	SHIFT4_LS,w		; ls shift value 
	movwf	TEMP2			; store
	btfsc	STATUS,C		; 
	incf	TEMP1,f			; increase if over
; compare with rpm
	movf	TEMP1,w			; ms of shift + hysteresis
	subwf	RPM_AVG0,w		; ms averaged rpm value 	
	movwf	TEMP1			; store
	movf	TEMP2,w			; ls of shift + hysteresis
	subwf	RPM_AVG1,w		; ls averaged rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfsc	TEMP1,7			; if set then not greater
	goto	CHECK_LOWER
	bcf		PORTA,1			; injectors off
	bcf		WITH_RANG,0		; within range flag off
	goto	SWITCH_PRESS

CHECK_LOWER
; compare shift4 with rpm	

	movf	SHIFT4_MS,w		; ms of shift value
	subwf	RPM_AVG0,w		; ms averaged rpm value 	
	movwf	TEMP1			; store
	movf	SHIFT4_LS,w		; ls of shift value
	subwf	RPM_AVG1,w		; ls averaged rpm value 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
	btfsc	TEMP1,7			; if set then not greater
	goto	UNDER
	goto	CHECK_IN
UNDER
	bsf		PORTA,1			; PWM drive on
	bcf		WITH_RANG,0		; within range flag off
	goto	SWITCH_PRESS

CHECK_IN
; within range PWM varies from 0 to full between SHIFT4 and SHIFT4 plus hysteresis
	bsf		WITH_RANG,0		; within range flag

; take SHIFT4 from rpm and divide by 2 to convert the 0-500 hysteresis range to 0-250
; compare with counter for PWM 

	movf	SHIFT4_MS,w		; ms of shift value
	subwf	RPM_AVG0,w		; ms averaged rpm value 	
	movwf	TEMP1			; store
	movf	SHIFT4_LS,w		; ls of shift value
	subwf	RPM_AVG1,w		; ls averaged rpm value
	movwf	TEMP2 
	btfss	STATUS,C		; 
	decf	TEMP1,f			; reduce if negative
; /2
	bcf		STATUS,C
	rrf		TEMP1,w
	rrf		TEMP2,w			; 16-bit/2 value 
	movwf	PWM_RANGE		; value used in interrupt for PWM 
	
SWITCH_PRESS	
; Check for pressed switch. 
	movf	SETTINGS,w
	btfss	STATUS,Z		; when zero normal run mode
	goto	MODE
	btfsc	PORTB,5			; select switch pressed?
	goto	NORMAL_RUN

; Select switch selects set mode 
MODE
	btfss	PORTB,5			; select switch pressed?
	incf	SETTINGS,f		; next mode if pressed
	movf	SETTINGS,w
	sublw	D'6'			; stop at 6	
	btfss	STATUS,C
	clrf	SETTINGS		; cleared if over 6
; find settings
	movf	SETTINGS,w
	btfsc	STATUS,Z		; when zero got to normal run
	goto	NORMAL_RUN_DEL	; normal run after switch delay
	movf	SETTINGS,w
	xorlw	D'01'			; shift 1 mode
	btfsc	STATUS,Z
	goto	SHIFT1
	movf	SETTINGS,w
	xorlw	D'02'			; shift 2 mode
	btfsc	STATUS,Z
	goto	SHIFT2
	movf	SETTINGS,w
	xorlw	D'03'			; shift 3 mode
	btfsc	STATUS,Z
	goto	SHIFT3
	movf	SETTINGS,w
	xorlw	D'04'			; limit 4 mode
	btfsc	STATUS,Z
	goto	LIMIT4
	movf	SETTINGS,w
	xorlw	D'05'			; PWM mode
	btfsc	STATUS,Z
	goto	PWM_4
	movf	SETTINGS,w
	xorlw	D'06'			; DIMMING mode
	btfsc	STATUS,Z
	goto	DIMMING
	clrf	SETTINGS
NORMAL_RUN_DEL
	bcf		SHIFT1_ON,0	; shift off
	bcf		SHIFT2_ON,0	; shift off
	bcf		SHIFT3_ON,0	; shift off
	bcf		SHIFT4_ON,0	; shift off
	movlw	D'9'		; delay length for switch debounce
	call	DELAY_1
	goto	SWITCH_PRESS; check switches and mode

; LED1,2,3,4 light to show which limit is available to set the rpm and hysteresis.
; VR1 and VR2 then are adjusted and the set switch pressed to store values.
; No settings done unless Set is pressed. 
	
SHIFT1
; monitor AN4 and AN3, update limit and hysteresis if SET switch is pressed.
; set LED1, clear others
	bsf		SHIFT1_ON,0	; shift on to show LED
	bcf		SHIFT2_ON,0	; shift off
	bcf		SHIFT3_ON,0	; shift off
	bcf		SHIFT4_ON,0	; shift off
	movlw	D'6'		; delay length for switch debounce
	call	DELAY_1
CHECK_SET_SHIFT1
; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_SHIFT1_VAL
; check for SELECT switch
	btfss	PORTB,5		; set switch
	goto	MODE	; delay and check mode
	goto	CHECK_SET_SHIFT1

LOAD_SHIFT1_VAL
; get AN4 value where 5V (set to D500 max) = 12500rpm so x 25
	bcf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5	; channel 4
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV4_1
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV4_1
	call 	MULT25		; multiply by 25
	movf	AARGB0,w
	movwf	SHIFT1_MS	; shift point
	movf	AARGB1,w
	movwf	SHIFT1_LS
	movlw	EEPROM0 	; non-volatile storage for rpm1 ms shift 1
	call	EEREAD		; sets EEADR
	movf	SHIFT1_MS,w	; current rpm for shift light1 ms
	call	EWRITE		; store value in EEPROM
	movlw	EEPROM1 	; non-volatile storage for rpm1 ls
	call	EEREAD		; sets EEADR
	movf	SHIFT1_LS,w	; current rpm for shift light1 ls 
 	call	EWRITE		; store

; get AN3 value where 5V (D255-5) is = 500rpm so x 2 for 500rpm
	
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5	; channel 3
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV3_1
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV3_1
; take away 5 (for 0-250 range)
	movlw	D'05'
	subwf	ADRESH,f
	btfss	STATUS,C	; if negative (less than 0)clear to zero
	clrf	ADRESH
	movf	ADRESH,w	; A/D value
	movwf	HYSTER1_LS	; hysteresis ls byte
	clrf	HYSTER1_MS
; x2
	bcf		STATUS,C
	rlf		HYSTER1_LS,f; x2
	rlf		HYSTER1_MS,f
; store	
	movlw	EEPROM8 	; non-volatile storage for hysteresis for rpm1 ms
	call	EEREAD		; sets EEADR
	movf	HYSTER1_MS,w; current hysteresis for shift 1 ms
	call	EWRITE
	movlw	EEPROM9		; non-volatile storage for hysteresis for rpm1 ls	
	call	EEREAD		; sets EEADR
	movf	HYSTER1_LS,w; current hysteresis for shift 1 ls
	call	EWRITE	
; flash LED1
	movlw	D'5'
	movwf	FLASH		; number of flashes
FLSH1
	bcf		SHIFT1_ON,0	; shift off
	movlw	D'4'
	call	DELAY_1
	bsf		SHIFT1_ON,0	; shift on
	movlw	D'4'
	call	DELAY_1
	decfsz	FLASH,f		; decrease till zero
	goto	FLSH1		; flash
	bcf		SHIFT1_ON,0	; shift off
	goto	MODE

SHIFT2
; monitor AN4 and AN3, update limit and hysteresis if SET switch is pressed.
; set LED2, clear others
	bcf		SHIFT1_ON,0	; shift off
	bsf		SHIFT2_ON,0	; shift on to show LED
	bcf		SHIFT3_ON,0	; shift off
	bcf		SHIFT4_ON,0	; shift off
	movlw	D'6'		; delay length for switch debounce
	call	DELAY_1
CHECK_SET_SHIFT2
; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_SHIFT2_VAL
; check for SELECT switch
	btfss	PORTB,5		; set switch
	goto	MODE		; delay and check mode
	goto	CHECK_SET_SHIFT2

LOAD_SHIFT2_VAL
; get AN4 value where 5V (set to D500 max) = 12500rpm so x 25

	bcf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5	; channel 4
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV4_2
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV4_2
	call 	MULT25		; multiply by 25
	movf	AARGB0,w
	movwf	SHIFT2_MS	; shift point
	movf	AARGB1,w
	movwf	SHIFT2_LS
	movlw	EEPROM2 	; non-volatile storage for rpm2 ms shift 2
	call	EEREAD		; sets EEADR
	movf	SHIFT2_MS,w	; current rpm for shift light2 ms
	call	EWRITE		; store value in EEPROM
	movlw	EEPROM3 	; non-volatile storage for rpm2 ls
	call	EEREAD		; sets EEADR
	movf	SHIFT2_LS,w	; current rpm for shift light2 ls 
 	call	EWRITE		; store

; get AN3 value where 5V (D255-5) is = 500rpm so x 2 for 500rpm
	
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5	; channel 3
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV3_2
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV3_2
; take away 5 (for 0-250 range)
	movlw	D'05'
	subwf	ADRESH,f
	btfss	STATUS,C	; if negative (less than 0)clear to zero
	clrf	ADRESH
	movf	ADRESH,w	; A/D value
	movwf	HYSTER2_LS	; hysteresis ls byte
	clrf	HYSTER2_MS
; x2
	bcf		STATUS,C
	rlf		HYSTER2_LS,f; x2
	rlf		HYSTER2_MS,f
; store	
	movlw	EEPROMA 	; non-volatile storage for hysteresis for rpm2 ms
	call	EEREAD		; sets EEADR
	movf	HYSTER2_MS,w; current hysteresis for shift 2 ms
	call	EWRITE
	movlw	EEPROMB		; non-volatile storage for hysteresis for rpm2 ls	
	call	EEREAD		; sets EEADR
	movf	HYSTER2_LS,w; current hysteresis for shift 2 ls
	call	EWRITE	
; flash LED2
	movlw	D'5'
	movwf	FLASH		; number of flashes
FLSH2
	bcf		SHIFT2_ON,0	; shift off
	movlw	D'4'
	call	DELAY_1
	bsf		SHIFT2_ON,0	; shift on
	movlw	D'4'
	call	DELAY_1
	decfsz	FLASH,f		; decrease till zero
	goto	FLSH2		; flash
	bcf		SHIFT2_ON,0	; shift off
	goto	MODE

SHIFT3
; monitor AN4 and AN3, update limit and hysteresis if SET switch is pressed.
; set LED3, clear others
	bcf		SHIFT1_ON,0	; shift off
	bcf		SHIFT2_ON,0	; shift off
	bsf		SHIFT3_ON,0	; shift on to show LED
	bcf		SHIFT4_ON,0	; shift off
	movlw	D'6'		; delay length for switch debounce
	call	DELAY_1
CHECK_SET_SHIFT3
; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_SHIFT3_VAL
; check for SELECT switch
	btfss	PORTB,5		; set switch
	goto	MODE		; delay and check mode
	goto	CHECK_SET_SHIFT3

LOAD_SHIFT3_VAL
; get AN4 value where 5V (set to D500 max) = 12500rpm so x 25

	bcf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5	; channel 4
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV4_3
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV4_3
	call 	MULT25		; multiply by 25
	movf	AARGB0,w
	movwf	SHIFT3_MS	; shift point
	movf	AARGB1,w
	movwf	SHIFT3_LS
	movlw	EEPROM4 	; non-volatile storage for rpm3 ms shift 3
	call	EEREAD		; sets EEADR
	movf	SHIFT3_MS,w	; current rpm for shift light3 ms
	call	EWRITE		; store value in EEPROM
	movlw	EEPROM5 	; non-volatile storage for rpm3 ls
	call	EEREAD		; sets EEADR
	movf	SHIFT3_LS,w	; current rpm for shift light3 ls 
 	call	EWRITE		; store

; get AN3 value where 5V (D255-5) is = 500rpm so x 2 for 500rpm
	
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5	; channel 3
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV3_3
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV3_3
; take away 5 (for 0-250 range)
	movlw	D'05'
	subwf	ADRESH,f
	btfss	STATUS,C	; if negative (less than 0)clear to zero
	clrf	ADRESH
	movf	ADRESH,w	; A/D value
	movwf	HYSTER3_LS	; hysteresis ls byte
	clrf	HYSTER3_MS
; x2
	bcf		STATUS,C
	rlf		HYSTER3_LS,f; x2
	rlf		HYSTER3_MS,f
; store	
	movlw	EEPROMC 	; non-volatile storage for hysteresis for rpm3 ms
	call	EEREAD		; sets EEADR
	movf	HYSTER3_MS,w; current hysteresis for shift 3 ms
	call	EWRITE
	movlw	EEPROMD		; non-volatile storage for hysteresis for rpm3 ls	
	call	EEREAD		; sets EEADR
	movf	HYSTER3_LS,w; current hysteresis for shift 3 ls
	call	EWRITE	
; flash LED3
	movlw	D'5'
	movwf	FLASH		; number of flashes
FLSH3
	bcf		SHIFT3_ON,0	; shift off
	movlw	D'4'
	call	DELAY_1
	bsf		SHIFT3_ON,0	; shift on
	movlw	D'4'
	call	DELAY_1
	decfsz	FLASH,f		; decrease till zero
	goto	FLSH3		; flash
	bcf		SHIFT3_ON,0	; shift off
	goto	MODE

LIMIT4
; monitor AN4 and AN3, update limit and hysteresis if SET switch is pressed.
; set LED4, clear others
	bcf		SHIFT1_ON,0	; shift off
	bcf		SHIFT2_ON,0	; shift off
	bcf		SHIFT3_ON,0	; shift off
	bsf		SHIFT4_ON,0	; shift on to show LED
	movlw	D'6'		; delay length for switch debounce
	call	DELAY_1
CHECK_SET_SHIFT4
; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_SHIFT4_VAL
; check for SELECT switch
	btfss	PORTB,5		; set switch
	goto	MODE	; delay and check mode
	goto	CHECK_SET_SHIFT4

LOAD_SHIFT4_VAL
; get AN4 value where 5V (set to D500 max) = 12500rpm so x 25
	bcf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5	; channel 4
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV4_4
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV4_4
	call 	MULT25		; multiply by 25
	movf	AARGB0,w
	movwf	SHIFT4_MS	; shift point
	movf	AARGB1,w
	movwf	SHIFT4_LS
	movlw	EEPROM6 	; non-volatile storage for rpm4 ms shift 4
	call	EEREAD		; sets EEADR
	movf	SHIFT4_MS,w	; current rpm for shift light4 ms
	call	EWRITE		; store value in EEPROM
	movlw	EEPROM7 	; non-volatile storage for rpm4 ls
	call	EEREAD		; sets EEADR
	movf	SHIFT4_LS,w	; current rpm for shift light4 ls 
 	call	EWRITE		; store

; get AN3 value where 5V (D255-5) is = 500rpm so x 2 for 500rpm
	
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5	; channel 3
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV3_4
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV3_4
; take away 5 (for 0-250 range)
	movlw	D'05'
	subwf	ADRESH,f
	btfss	STATUS,C	; if negative (less than 0)clear to zero
	clrf	ADRESH
	movf	ADRESH,w	; A/D value
	movwf	HYSTER4_LS	; hysteresis ls byte
	clrf	HYSTER4_MS
; x2
	bcf		STATUS,C
	rlf		HYSTER4_LS,f; x2
	rlf		HYSTER4_MS,f
; store	
	movlw	EEPROME 	; non-volatile storage for hysteresis for rpm4 ms
	call	EEREAD		; sets EEADR
	movf	HYSTER4_MS,w; current hysteresis for shift 4 ms
	call	EWRITE
	movlw	EEPROMF		; non-volatile storage for hysteresis for rpm4 ls	
	call	EEREAD		; sets EEADR
	movf	HYSTER4_LS,w; current hysteresis for shift 4 ls
	call	EWRITE	
; flash LED4
	movlw	D'5'
	movwf	FLASH		; number of flashes
FLSH4
	bcf		SHIFT4_ON,0	; shift off
	movlw	D'4'
	call	DELAY_1
	bsf		SHIFT4_ON,0	; shift on
	movlw	D'4'
	call	DELAY_1
	decfsz	FLASH,f		; decrease till zero
	goto	FLSH4		; flash
	bcf		SHIFT4_ON,0	; shift off

; For Hyster4 changes divide HYSTER4 by 2 (convert 0-500 to 0-250)
	bcf		STATUS,C
	rrf		HYSTER4_MS,w	; divide 16-bits/2
	rrf		HYSTER4_LS,w
	movwf	TEMP_HYST		; used in interrupt to drive PWM
	goto	MODE

PWM_4
; monitor AN4 and AN3, update averaging and PWM update when SET switch is pressed.
; flash LEDs, clear others
	bsf		SHIFT1_ON,0	; shift on 
	bcf		SHIFT2_ON,0	; shift off 
	bcf		SHIFT3_ON,0	; shift off 
	bsf		SHIFT4_ON,0	; LED on
	movlw	D'1'		; delay length for switch update
	call	DELAY_1
	bcf		SHIFT1_ON,0	; shift off 
	bsf		SHIFT2_ON,0	; shift on 
	bcf		SHIFT3_ON,0	; shift off 
	movlw	D'1'		; delay length for switch update
	call	DELAY_1
	bcf		SHIFT1_ON,0	; shift off 
	bcf		SHIFT2_ON,0	; shift off 
	bsf		SHIFT3_ON,0	; shift on 
	movlw	D'1'		; delay length for switch update
	call	DELAY_1
	bsf		SHIFT1_ON,0	; shift on 
	bcf		SHIFT2_ON,0	; shift off 
	bcf		SHIFT3_ON,0	; shift off 
	movlw	D'1'		; delay length for switch update
	call	DELAY_1
	bcf		SHIFT1_ON,0	; shift off 
	bsf		SHIFT2_ON,0	; shift on 
	bcf		SHIFT3_ON,0	; shift off 
	movlw	D'1'		; delay length for switch update
	call	DELAY_1
	bcf		SHIFT1_ON,0	; shift off 
	bcf		SHIFT2_ON,0	; shift off 
	bsf		SHIFT3_ON,0	; shift on 
	movlw	D'1'		; delay length for switch update
	call	DELAY_1

CHECK_SET_PWM
; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_PWM_VAL
; check for SELECT switch
	btfss	PORTB,5		; set switch
	goto	MODE		; delay and check mode

; PWM indication with 1,2,3 LEDs
	bsf		SHIFT1_ON,0	; on
	bcf		SHIFT2_ON,0	; off
	bcf		SHIFT3_ON,0	; off
	movlw	D'1'
	call	DELAY_1
; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_PWM_VAL
; check for SELECT switch
	btfss	PORTB,5		; set switch
	goto	MODE		; delay and check mode
	bcf		SHIFT1_ON,0	; other LEDs off
	bsf		SHIFT2_ON,0
	bcf		SHIFT3_ON,0
	movlw	D'1'
	call	DELAY_1
; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_PWM_VAL
; check for SELECT switch
	btfss	PORTB,5		; set switch
	goto	MODE		; delay and check mode
	bcf		SHIFT1_ON,0	; other LEDs off
	bcf		SHIFT2_ON,0
	bsf		SHIFT3_ON,0
	movlw	D'1'
	call	DELAY_1
	goto	CHECK_SET_PWM

LOAD_PWM_VAL
; get AN4 value 
	bcf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5	; channel 4
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV4_PWM
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV4_PWM
	movf	ADRESH,w
	movwf	AVG_COUNT
	bcf		STATUS,C	; divide by 2
	rrf		AVG_COUNT,f
	bcf		STATUS,C	; divide by 4
	rrf		AVG_COUNT,f
; set minimum to 1
	movf	AVG_COUNT,w	; move to set or clear Z status bit
	movlw	D'1'		; ready for loading if zero
	btfsc	STATUS,Z	; if zero set at 1
	movwf	AVG_COUNT	; set at 1

; for Averaging total
	movlw	EEPROMI		; non-volatile storage for averaging total	
	call	EEREAD		; get value
	movf	AVG_COUNT,w	; stored average total value
	call	EWRITE

; get AN3 value where 5V (D255) is = 16 PWM before change so divide by 16
	
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5	; channel 3
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV3_PWM
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV3_PWM
; divide by 16
	bcf		STATUS,C
	rrf		ADRESH,f	; A/D value /2
	bcf		STATUS,C
	rrf		ADRESH,f	; A/D value /4
	bcf 	STATUS,C
	rrf		ADRESH,f	; A/D value /8
	bcf 	STATUS,C
	rrf		ADRESH,f	; A/D value /16

; set minimum to 1
	movf	ADRESH,w	; move to set or clear Z status bit
	movlw	D'1'		; ready for loading if zero
	btfsc	STATUS,Z	; if zero set at 1
	movwf	ADRESH		; set at 1
	movf	ADRESH,w
	movwf	RANGE_COUNT	; pwm range
; store
	movlw	EEPROMH		; non-volatile storage for PWM update rate	
	call	EEREAD		; get value
	movf	RANGE_COUNT,w
	call	EWRITE
; flash LED4
	movlw	D'5'
	movwf	FLASH		; number of flashes
FLSH4_PWM
	bsf		SHIFT1_ON,0	; other LEDs on
	bsf		SHIFT2_ON,0
	bsf		SHIFT3_ON,0
	bsf		SHIFT4_ON,0	; shift on
	movlw	D'4'
	call	DELAY_1
	bcf		SHIFT4_ON,0	; shift off
	bcf		SHIFT1_ON,0	; other LEDs off
	bcf		SHIFT2_ON,0
	bcf		SHIFT3_ON,0
	movlw	D'4'
	call	DELAY_1
	decfsz	FLASH,f		; decrease till zero
	goto	FLSH4_PWM	; flash
	goto	MODE

DIMMING
; Change DIM_MIN according to A/D
; store in EEPROM 
; AN5 used for dimming minimum. Use complement value 

; set LEDs
	bsf		SHIFT1_ON,0	; shift on to show LEDs
	bsf		SHIFT2_ON,0	; shift on
	bsf		SHIFT3_ON,0	; shift on
	bcf		SHIFT4_ON,0	; shift off 
	movlw	D'6'		; delay length for switch debounce
	call	DELAY_1
	
CHECK_SET_DIM
; get AN5 value for dimming
	bsf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5	; channel 5
	call	DELAYX		; allow time to charge input capacitance
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV5_D
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV5_D
	movf	ADRESH,w
	movwf   DIMVAL_TEMP	; dimming value
	
; divide by 8 for 32 steps
	bcf		STATUS,C
	rrf		DIMVAL_TEMP,f		; /2
	bcf		STATUS,C
	rrf		DIMVAL_TEMP,w		; /4
	movwf	DIM_VAL				; value used for dimming

; check for SET switch
	btfss	PORTB,7		; if closed load new values	
	goto	LOAD_DIM_VAL
; check for SELECT switch
	call	DELAYX
	call	DELAYX		
	btfss	PORTB,5		; set switch
	goto	MODE		; delay and check mode
	goto	CHECK_SET_DIM

LOAD_DIM_VAL
	comf	ADRESH,w	; complement for minimum setting
	movwf	DIM_MIN
	movlw	EEPROMG		; non-volatile storage for dimming
	call	EEREAD		; sets EEADR
	movf	DIM_MIN,w	; dimming minimum value
 	call	EWRITE		; write to EEPROM

; flash LEDs
	movlw	D'5'
	movwf	FLASH		; number of flashes
FLSH_ALL
	bcf		SHIFT1_ON,0	; shift off LEDs off
	bcf		SHIFT2_ON,0	; shift off
	bcf		SHIFT3_ON,0	; shift off
	bcf		SHIFT4_ON,0	; shift off 
	movlw	D'4'
	call	DELAY_1
	bsf		SHIFT1_ON,0	; shift on to show LEDs
	bsf		SHIFT2_ON,0	; shift on
	bsf		SHIFT3_ON,0	; shift on
	bcf		SHIFT4_ON,0	; shift off 
	movlw	D'4'
	call	DELAY_1
	decfsz	FLASH,f		; decrease till zero
	goto	FLSH_ALL	; flash
	bcf		SHIFT1_ON,0	; shift off LEDs off
	bcf		SHIFT2_ON,0	; shift off
	bcf		SHIFT3_ON,0	; shift off
	bcf		SHIFT4_ON,0	; shift off
	goto	MODE

; end of setup

; **********************************************************************
; normal run mode

NORMAL_RUN
	movf	NIL_SIG,w	; check if no signal
	btfss	STATUS,Z	; if zero can calculate
	goto	CLEAR_VALUES
	btfss	TRANSFER,0	; when set calcalculate
	goto	SHIFT_RUN
	bcf		TRANSFER,0	; clear value ready for next data
; divide to get rpm  values in 	NUMER_x
	movf	NUMER_0,w		
	movwf	AARGB0		; numerator ms byte
	movf	NUMER_1,w	;
	movwf	AARGB1
	movf	NUMER_2,w	;
	movwf	AARGB2		; 
	movf	NUMER_3,w	; numerator ls byte
	movwf	AARGB3

; stop interrupt
	bcf		INTCON,PEIE
	nop
	movf	COUNT0,w	; denominator ms byte
	movwf	BARGB0
	movf	COUNT1,w	; denominator 
	movwf	BARGB1
	movf	COUNT2,w	; denominator
	movwf	BARGB2
	movf	COUNT3,w	; denominator ls byte
	movwf	BARGB3

; allow interrupts
	bsf		INTCON,PEIE
	call	FXD3232U	; divide to get rpm
; AARGB2/AARGB3 has rpm
	movf	AARGB2,w
	movwf	RPM_VAL0	; ms byte of rpm
	movf	AARGB3,w
	movwf	RPM_VAL1	; ls byte
	goto	AVERAGE
	
CLEAR_VALUES
	clrf	RPM_VAL0
	clrf	RPM_VAL1
	clrf	RPM_AVG0
	clrf	RPM_AVG1
	bcf		SHIFT1_ON,0	; shift off 
	bcf		SHIFT2_ON,0	; shift off
	bcf		SHIFT3_ON,0	; shift off
	bcf		SHIFT4_ON,0	; shift off 
	bsf		PORTA,1		; injectors on
	bcf		PORTA,2		; LEDs off
	bcf		PORTA,0
	bcf		PORTA,6
	bcf		PORTA,7
	goto	CLEAR_TALLY1

AVERAGE
; average values for limiting
; setup average count
	movf	AARGB3,w
	addwf	TOTAL2,f
	btfss	STATUS,C	; if carry set increase
	goto	TALLY1		; next byte
	incfsz	TOTAL1,f	; increase if over
	goto	TALLY1
	incfsz	TOTAL0,f	; increase if over
	goto	TALLY1
	goto	CLEAR_TALLY	; clear if overrange
TALLY1
	movf	AARGB2,w
	addwf	TOTAL1,f
	btfss	STATUS,C	; if over increase next byte
	goto	REMEMBER_TALLY_COUNT
	incfsz	TOTAL0,f	; if overrange clear tally
	goto	REMEMBER_TALLY_COUNT
	
CLEAR_TALLY
	clrf	TOTAL0
	clrf	TOTAL1
	clrf	TOTAL2

REMEMBER_TALLY_COUNT
	incf	TALLY_CNT,f

	movf	TALLY_CNT,w	; check if average calculation is required after tallying
	subwf	AVG_COUNT,w
	btfsc	STATUS,C
	goto	SHIFT_RUN	; 
; divide to get average rpm  
	movlw	H'00'		; clear
	movwf	AARGB0		; numerator 0 
	movf	TOTAL0,w	; ms of total rpm
	movwf	AARGB1
	movf	TOTAL1,w	;
	movwf	AARGB2		; 
	movf	TOTAL2,w	;
	movwf	AARGB3

	clrf	BARGB0
	clrf	BARGB1
	clrf	BARGB2
	movf	TALLY_CNT,w	; average tally count
	movwf	BARGB3

	call	FXD3232U	; divide to get rpm

; AARGB2,AARGB3 have averaged rpm
	movf	AARGB2,w	; transfer average values to usable registers
	movwf	RPM_AVG0
	movf	AARGB3,w
	movwf	RPM_AVG1

; clear tally values
CLEAR_TALLY1
	clrf	TOTAL0
	clrf	TOTAL1
	clrf	TOTAL2
	clrf	TALLY_CNT
	
	goto	SHIFT_RUN	; tacho routines


; ************************************************************************


; subroutine to read EEPROM memory

EEREAD
	bcf		INTCON,GIE	; disable interrupts	
	bcf 	STATUS,RP0	; select memory bank 
	bsf		STATUS,RP1	; bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; pointer for data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,W	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	bsf		INTCON,GIE	; enable interrupts
	return

EREAD ;(BEFORE interrupt enabled)
	bcf		INTCON,GIE	; disable interrupts	
	bcf 	STATUS,RP0	; select memory bank 
	bsf		STATUS,RP1	; bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; pointer for data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,W	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	bsf		INTCON,GIE	; enable interrupts
	return

; subroutine to write to EEPROM
EEWRITE
EWRITE
	bcf		INTCON,GIE	; disable interrupts
	bsf	    STATUS,RP1	; select bank 
	bcf 	STATUS,RP0	; select memory bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; pointer for data memory
	bsf		EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITE	
	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank select
	bcf 	STATUS,RP1	; select memory bank 0 
	bsf		INTCON,GIE	; enable interrupts
	return				; value written 


DELAY1
	movlw	D'1'
	goto	DELAY_1
DELAY
	movlw	D'10'		; number of delay cycles
DELAY_1	
	movwf	DELCNT

DELAY_M
	movlw	D'255'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
LP_1
	movlw	D'255'		; set delay period value 2 
	movwf	VALUE_2		; VALUE_2 = w
LP_2
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1
	decfsz	DELCNT,f
	goto	DELAY_M
	return	

DELAYX
	movlw	D'2'
DELAYZ
	movwf	VALUE_2
X_LOOP1
	movlw	D'10'
	movwf	VALUE_1
X_LOOP
	decfsz	VALUE_1,f
	goto	X_LOOP	
	decfsz	VALUE_2,f
	goto	X_LOOP1
	return

;**********************************************************************************************

MULT25
; multiply 9-bit A/D (ADRESH, ADRESHL) by 25 to convert from 0-D500 to 0-12500rpm in 25rpm steps 

	clrf	AARGB1		; initially clear
	bsf		STATUS,RP0	; bank 1
	rlf		ADRESL,w	; get bit 9 into carry
	bcf		STATUS,RP0	; bank 0
	rlf		ADRESH,w	; A/D value
	movwf	AARGB2
	btfsc	STATUS,C	; if carry set, set bit 0 in ms byte
	bsf		AARGB1,0
; subtract 11 from value for 0-500 range rather than 0-511
; check if AARGB1,0 is set 
	btfss	AARGB1,0	; if set can continue	
	goto	NOT_SET
	movlw	D'11'
	subwf	AARGB2,f
	btfss	STATUS,C
	decf	AARGB1,f	; decrease ms byte if carry 
	goto	DO_TIMES25
NOT_SET	
	movlw	D'11'
	subwf	AARGB2,f
	btfss	STATUS,C	; if negative (less than 11 set at 0)
	clrf	AARGB2
DO_TIMES25
	movlw	D'25'		; x 25
	movwf	BARGB2
	clrf	BARGB0
	clrf	BARGB1
	clrf	AARGB0

	call	FXM2424U	; multiply
; value in AARGB0,1,2,3,4,5. Shift bytes left
	movf	AARGB4,w
	movwf	AARGB0
	movf	AARGB5,w
	movwf	AARGB1	
	return

; 24x24 Bit Unsigned Fixed Point Multiply 24x24 -> 48
; Input: 24 bit unsigned fixed point multiplicand in AARGB0,1,2
; 24 bit unsigned fixed point multiplier in BARGB0,1,2
; Use: CALL FXM2424U
; Output: 48 bit unsigned fixed point product in AARGB0
; Result: AARG <-- AARG x BARG
; Max Timing: 9+501+2 = 512 clks
; Min Timing: 9+150 = 159 clks

FXM2424U
	CLRF 	AARGB3 ; clear partial product
	CLRF 	AARGB4
	CLRF 	AARGB5
	MOVF 	AARGB0,W
	MOVWF 	TEMPB0
	MOVF 	AARGB1,W
	MOVWF 	TEMPB1
	MOVF 	AARGB2,W
	MOVWF 	TEMPB2

	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
LOOPUM2424A
	RRF 	BARGB2,F
	BTFSC 	STATUS,C
	GOTO 	ALUM2424NAP
	DECFSZ 	LOOPCOUNT,F
	GOTO	LOOPUM2424A
	MOVWF 	LOOPCOUNT
LOOPUM2424B
	RRF 	BARGB1,F
	BTFSC 	STATUS,C
	GOTO	BLUM2424NAP
	DECFSZ	LOOPCOUNT,F
	GOTO	LOOPUM2424B
	MOVWF	LOOPCOUNT
LOOPUM2424C
	RRF		BARGB0,F
	BTFSC 	STATUS,C
	GOTO 	CLUM2424NAP
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPUM2424C
	CLRF 	AARGB0
	CLRF 	AARGB1
	CLRF 	AARGB2
	RETLW 	0x00
CLUM2424NAP
	BCF 	STATUS,C
	GOTO 	CLUM2424NA
BLUM2424NAP
	BCF 	STATUS,C
	GOTO 	BLUM2424NA
ALUM2424NAP
	BCF 	STATUS,C
	GOTO 	ALUM2424NA
ALOOPUM2424
	RRF 	BARGB2,F
	BTFSS 	STATUS,C
	GOTO 	ALUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF	AARGB0,F
ALUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	ALOOPUM2424
	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
BLOOPUM2424
	RRF 	BARGB1,F
	BTFSS 	STATUS,C
	GOTO 	BLUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF 	AARGB0,F
BLUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	RRF 	AARGB4,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	BLOOPUM2424
	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
CLOOPUM2424
	RRF 	BARGB0,F
	BTFSS 	STATUS,C
	GOTO 	CLUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF 	AARGB0,F
CLUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	RRF 	AARGB4,F
	RRF 	AARGB5,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	CLOOPUM2424
	return

; 32/32 Bit Unsigned Fixed Point Divide 32/32 -> 32.32
; Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point divisor in BARGB0, BARGB1, BARGB2, BARGB3
; Use: CALL FXD3232U
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point remainder in REMB0, REMB1, REMB2, REMB3
; Result: AARG, REM <-- AARG / BARG
; Max Timing: 4+1025+2 = 1031 clks
; Max Timing: 4+981+2 = 987 clks
; PM: 4+359+1 = 364 DM: 13
FXD3232U
	CLRF 	REMB0
	CLRF	REMB1
	CLRF 	REMB2
	CLRF 	REMB3
	call	UDIV3232L
	return

UDIV3232L 
; Max Timing: 24+6*32+31+31+6*32+31+31+6*32+31+31+6*32+31+16 = 1025 clks
; Min Timing: 24+6*31+30+30+6*31+30+30+6*31+30+30+6*31+30+3 = 981 clks
; PM: 359 DM: 13
	CLRF 	TEMP
	RLF 	AARGB0,W
	RLF 	REMB3,F
	MOVF 	BARGB3,W
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	RLF 	AARGB0,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232A 
	RLF 	AARGB0,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22LA
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LA

UADD22LA 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LA 
	RLF		AARGB0,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232A
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22L8
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L8

UADD22L8 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L8 
	RLF 	AARGB1,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232B 
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB1,0
	GOTO 	UADD22LB
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LB

UADD22LB 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LB 
	RLF 	AARGB1,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232B
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS	AARGB1,0
	GOTO 	UADD22L16
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L16

UADD22L16 	
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF	TEMP,F

UOK22L16
	RLF		AARGB2,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232C 
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22LC
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LC

UADD22LC
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LC 
	RLF 	AARGB2,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232C
	RLF 	AARGB3,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22L24
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L24

UADD22L24 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L24 
	RLF 	AARGB3,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232D 
    RLF  	AARGB3,W
    RLF    	REMB3,F
    RLF    	REMB2,F
    RLF    	REMB1,F
    RLF    	REMB0,F
    RLF    	TEMP,F
    MOVF   	BARGB3,W
 	BTFSS  	AARGB3,0
   	GOTO   	UADD22LD

    SUBWF  	REMB3,F
  	MOVF   	BARGB2,W
    BTFSS   STATUS,C
    INCFSZ  BARGB2,W
    SUBWF   REMB2,F
    MOVF    BARGB1,W
    BTFSS   STATUS,C
    INCFSZ  BARGB1,W
    SUBWF   REMB1,F
    MOVF    BARGB0,W
    BTFSS   STATUS,C
    INCFSZ  BARGB0,W
    SUBWF   REMB0,F
    CLRW
    BTFSS   STATUS,C
    MOVLW   H'1'
    SUBWF   TEMP,F
    GOTO    UOK22LD

UADD22LD      
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F
    CLRW
    BTFSC   STATUS,C
    MOVLW   H'1'
    ADDWF   TEMP,F
        
UOK22LD
	RLF     AARGB3,F

   	DECFSZ  LOOPCOUNT, F
    GOTO    LOOPU3232D

    BTFSC   AARGB3,0
    GOTO    UOK22L
    MOVF    BARGB3,W
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F

UOK22L

	RETURN


	end
