;*******************************************************************************************
	title	"DC Speed Controller: PIC16F88 program"
	subtitle	"version 2.8 December 2007 Mauro Grassi"
;	For Silicon Chip Magazine
;	v1.0- October 2007	Mauro Grassi
;	v2.0- November 2007 Mauro Grassi
;   v2.7- December 2007 Mauro Grassi.
;	v2.8- 10 December 2007 Mauro Grassi.
;   v2.9- 11 December 2007 Mauro Grassi.
;	v3.0- 12 December 2007 Mauro Grassi. (got rid of H/L select. One voltage divider for 12V-36V operation).
;	the only difference in the boards, appears below or above 16V.
;   for below 16V use R1=100 Ohms and omit zener Z1.
;	for above 16V use R1=1k Ohms and Z1=16V zener.
;	Designed for 16F88 @ 8 MHz
;*******************************************************************************************
	list	p=16F88
	
	#include	<p16F88.inc>
	#define		P16CXX		1

	list
	cblock  	0xC0                        ;set cblock start address for (in bank 1)
    endc                                    ;float_ascii routine	
	__CONFIG	_CONFIG1, _CP_OFF & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO
	ERRORLEVEL -302
;********************************************************************************************
;*************************Basic Registers****************************************************
W_TEMP		equ 0x70
STATUS_TEMP equ 0x71
PCLATH_TEMP equ 0x72
;*********************Some temporary registers for General Use (not used in Interrupt)*******
ALPERIOD equ 0x10
POTSTABLECONSECUTIVE equ 0xA0	; threshhold for recognising stable pot...
MAXLOCK equ 0x40
LOCKLIMIT equ 0x20
z0  	equ 0x40
z1		equ	0x41
z2		equ	0x42
zerocounter	equ	0x43
cursor	equ	0x44
;*********************** Interrupt Variables used for Display Refresh ***********************
; To write to the display simply write ASCII values to dis0, dis1, dis2, dis3 (Left to Right)
;********************************************************************************************
dis0	equ 0x45
dis1	equ	0x46
dis2	equ	0x47
dis3	equ 0x48	; these are display registers interpreted as ASCII!
disptr	equ	0x49	; display pointer used internally for interrupt driven display refresh
t0  	equ 0x4A	; internal temporary used by Interrupt for display refresh
t1  	equ 0x4B	; internal temporary used by Interrupt for display refresh
;********************************************************************************************
resl 	equ 0x4E	; holds low digit (in ASCII) of DisA output
resh 	equ 0x4F	; holds high digit (in ASCII) of DisA output
;**************************** ADC system variables ******************************************
anbatteryh 	equ 0x50	; high byte of digital converted value of analog BATTERY input
anbatteryl 	equ 0x51	; low  byte of digital converted value of analog BATTERY input
anbackemfh 	equ 0x52	; high byte of digital converted value of analog BACKEMF input
anbackemfl 	equ 0x53	; low  byte of digital converted value of analog BACKEMF input
anspeedpoth	equ	0x54	; high byte of digital converted value of analog SPEEDPOT input
anspeedpotl	equ	0x55	; low  byte of digital converted value of analog SPEEDPOT input
;********************************************************************************************
;*************** Some System variables ******************************************************
PORADDRESS  equ 0x01		; POR register in data EEPROM memory for detecting POR condition...
PROGVER		equ 0x30		; should NOT equal 0xFF! this is the program version!!
BASEMEMORY	equ 0x10		; base memory for memory settings (16 bytes) in data EEPROM...
settings 	equ	0x56		; used for system settings
							; bit 0,1 indicate voltage selection
							; bit 2=1 disables low priority beeps (cf flashing,4)
							; bit 3=1 disables high priority beeps (cf flashing,5)
SETTINGSDEFAULT equ 0x01	; default value for settings
VOLTIN1		equ	1			; bit 1 of settings variable used for bit 1 of voltage selection
VOLTIN0		equ	0			; bit 0 of settings variable used for bit 0 of voltage selection
							; VOLTIN1 VOLTIN0     Input Voltage
							;     0      0          VIN00
							;	  0      1          VIN01
							;     1      0          VIN10
							;	  1      1			VIN11
mode		equ	0x57		; used for mode
							;
							; mode
							;	bits 1,0 = 00 normal mode (source is instant pot value)
							;			 = 01 memory mode (source is memory value)
							;    		 = 1X frozen mode (source is last saved pot value)
							; bit 2=0 freeze pot for memory select 1=accept value for memory select from pot (into currentmemory)
							; bit 3=0 normal 1=turn off PWM ...
;*************************************************************************************************************************************
; This is the MENU System...
;*************************************************************************************************************************************
; a finite state machine...
;*************************************************************************************************************************************
; State    Name      Display         Press                         NextState
;************************************************************************************************************************************
VIN00R2		equ	33000	; value of the resistor R2 in case 00
VIN00R3		equ	4700	; value of the resistor R3 in case 00
VIN01R2		equ	33000	; value of the resistor R2 in case 01
VIN01R3		equ	4700	; value of the resistor R3 in case 01
VIN10R2		equ	33000	; value of the resistor R2 in case 10
VIN10R3		equ	4700	; value of the resistor R3 in case 10
VIN11R2		equ	33000	; value of the resistor R2 in case 11
VIN11R3		equ	4700	; value of the resistor R3 in case 11
; some floating point constants for each case
; VINxxDIVEXP, B0, B1, B2 should be the floating point representations of the number f
; where f is calculated as follows:
;	f= (5.00/1023) * (VINxxR2+VINxxR3)/VINxxR3
VIN00DIVEXP	equ 0x7A
VIN00DIVB0	equ	0x20
VIN00DIVB1	equ	0x95
VIN00DIVB2	equ	0x15	
VIN01DIVEXP	equ	0x7A	
VIN01DIVB0	equ	0x20
VIN01DIVB1	equ	0x95
VIN01DIVB2	equ	0x15	; represents f=0.039204675 (24V case with dividers 33k and 4.7k Ohms)
VIN10DIVEXP	equ 0x7A
VIN10DIVB0	equ	0x20
VIN10DIVB1	equ	0x95
VIN10DIVB2	equ	0x15	
VIN11DIVEXP	equ	0x7A	
VIN11DIVB0	equ	0x20
VIN11DIVB1	equ	0x95
VIN11DIVB2	equ	0x15	; represents f=0.039204675 (24V case with dividers 33k and 4.7k Ohms)
;****************************Input system variables *****************************************
DEBOUNCEINTERVAL equ 0x04	; for debouncing... pulses lower than this will be ignored...
button0		equ	0x58		; holds a time value that indicates how long the button 0 has been pressed down for
							; must be cleared to zero to initiate next button press capture
button1		equ	0x59		; holds a time value that indicates how long the button 1 has been pressed down for
							; must be cleared to zero to initiate next button press capture
oldbutton0	equ	0x5A		; internal use
oldbutton1	equ	0x5B		; internal use
;*************************** PWM system variables *******************************************
setfrequency equ	0x5C	; sets the frequency of PWM (can be set by user)
							; a number between 00 and FF
							; 0x00 corresponds to 488 Hz (PR2=0xFF) and 0xFF corresponds to 7.8kHz (PR2=0x0F)
							; upper limit of 7.8kHz for 6bit (lowest) resolution of PWM.
pr2value	 equ	0x5D	; internal value of PR2 for PWM period
							; this is computed from the setfrequency register via the formula
							; see page 86 of datasheet for PIC16F88
							; assuming 8MHz system clock and 1:16 TMR2 prescaler, we have:
							; pr2value= -1 + 125000/(488+28.6745098*setfrequency) where 28.6745098=(1/255)*(7800-488)
pwmmaxh		equ		0x5E	;
pwmmaxl		equ		0x5F	; point at which duty cycle equals pwm period (maximum possible value for ipwmh:ipwml)
							; this is computed as follows:
							; pwmmax= 4*(pr2value+1)
ipwml equ 0x61				; used internally by PWM system
ipwmh equ 0x60				; used internally by PWM system
pwml equ 0x62				; low byte of 10bit PWM duty cycle
pwmh equ 0x63				; high byte of 10bit PWM duty cycle (only two LSBs are significant)
pdeltah	equ	0x64			; homing delta high
pdeltal	equ	0x65			; homing delta low
flashing equ 0x66			; bit =1 (bit 0,1,2,3) indicate flash that digit...
							; bit 4= request beep (low priority) - low priority beeps can be disabled by settings(bit 2)
							; bit 5= request beep (high priority) - high priority beeps can be disabled by settings (bit 3)
							; bit 6=1 (disable flashing) 0=normal
							; bit 7=1 (request flash off next cycle) 0=normal
flashspeed equ 0x67			; this controls the speed of flashing digits and LED output for low battery...
							; this value is ANDed with disptr and if the result is non zero we blank the digit(s)...
f0	equ 0x68
f1	equ	0x69
f2	equ	0x6A
f3	equ 0x6B
batterylevel equ 0x6C		; contains the percentage level of the battery voltate 00-100%
loopcounter equ 0x6D		; used to update various variables...
backemfnorml equ 0x6E		; low byte of 10bit back emf reading which has been normalized to be within 000-3ff
backemfnormh equ 0x6F		; high byte of 10bit back emf reading which has been normalized to be within 000-3ff range.
currentmemory equ 0x73		; holds current memory ... (0-7)
anspeedsavel equ 0x74
anspeedsaveh equ 0x75
state equ 0x76
potmovement equ 0x77		; used internally to detect movements of the pot!
currentmemorylast equ 0x78
currentmemorylast2 equ 0x79
key	  equ 0x7A				; bits 0-3 = 1 when button pressed Short or Long
							; bit 7=1 if at least one button is pressed 0 otherwise
							; bit 6=1 if pot has been stable for a while 0 if moving about (default)
address equ 0x7B			; for use in EEPROM read/write routines.
locked equ 0x7C				; keeps track of lockings...
status equ 0x7D				; status bit 0=1 when locked 0= not locked...
alarmlevel equ 0x7E			; alarm battery level (alarm sounds if batterylevel goes below this level).
alarmperiod equ 0x7F
;***************SYSTEM pin definitions*******************************************************
; PORT definitions
;********************************************************************************************
; The PORTS are as follows on the speed controller board.
;
; TRISA= 00101111= 0x2F PORTA(default)=00101111=0x2F
;
; PORTA (7)= (Output) (Default=0) used for digit cathode
; PORTA (6)= (Output) (Default=0) used for digit cathode
; PORTA (5)= (Input)  (Default=X) MCLR (reset) unused
; PORTA (4)= (Output) (Default=0) LEDBIT (=1 turns on LED and PIEZO) (=0 turns off LED and PIEZO)
; PORTA (3)= (Input)  (Default=X) Analog Input AN3 for SPEED setting POT
; PORTA (2)= (Input)  (Default=X) Analog Input AN2 for BACK EMF reading
; PORTA (1)= (Input)  (Default=X) Analog Input AN1 for measuring Battery Voltage Supply
; PORTA (0)= (Input)  (Default=X) Digital Input (=0 if display board is connected) (=1 if display board is NOT connected)
;
; TRISB= 00011000= 0x18 PORTB(default)=11111001=0xF9
;
; PORTB (7)= (Output) (Default=1) used for display board shift register control controls DATA to shift register
; PORTB (6)= (Output) (Default=1) used for display board shift register control controls G enable of shift register
; PORTB (5)= (Output) (Default=1) used for display board shift register control controls CLK of shift register
; PORTB (4)= (Input)  (Default=X) Digital Input of pushbutton 1 on display board (=0 button pressed) *MUST be weakly pulled up*
; PORTB (3)= (Input)  (Default=X) Digital Input of pushbutton 2 on display board (=0 button pressed) *MUST be weakly pulled up*
; PORTB (2)= (Output) (Default=0) used for digit cathode
; PORTB (1)= (Output) (Default=0) used for digit cathode
; PORTB (0)= (Output) (Default=1) PWM output to control MOSFETs (0=turns ON mosfets) (=1 turns OFF mosfets).
;
; PIN definitions as follows:
; Defaults=
DEFAULTA 		equ 0x2F
DEFAULTTRISA 	equ 0x2F
DEFAULTB		equ	0xF9
DEFAULTTRISB	equ	0x18
; PORTA=
DIG0	 	equ	7		; digit bit position in PORTA
DIG1	 	equ	6		; digit bit position in PORTA
LEDBIT	 	equ	4		; bit for the LED and PIEZO in PORTA
SPEEDPOT 	equ	3		; bit for AN3 input = SPEED POT reading 0-5V in PORTA
BACKEMF	 	equ	2		; bit for AN2 input = BACK EMF reading 0-5V in PORTA
BATTERY	 	equ	1		; bit for AN1 input measures BATTERY level 0-5V in PORTA
BOARD	 	equ	0		; bit for digital input in PORTA (=0 means Display board is connected) (=1 display board not connected)
; PORTB=
SRDAT	equ 7			; Shift register Data bit of port B
SRGEN	equ 6			; Shift register G enable bit of port B
SRCLK	equ 5			; Shift register Clock bit of port B
BUTTON0 equ 4			; digital input for pushbutton 1 on PORTB
BUTTON1	equ	3			; digital input for pushbutton 2 on PORTB
DIG3	equ	2			; digit bit position in PORTB
DIG2	equ	1			; digit bit position in PORTB
PWMOUT	equ 0			; bit in PORTB controls PWM output (=0 means MOSFETs are ON) (=1 means MOSFETS are OFF)
;*************************************************************************************************************************************
	org	0x0000
	goto	Reset		; this is the reset vector
	nop
	nop
	nop
	movwf 	W_TEMP 				; Copy W to TEMP register
	movfw	PCLATH  			; Only required if using
	movwf 	PCLATH_TEMP 		; Save PCLATH into W
	bcf		PCLATH,3
	goto	Interrupt	; this is the interrupt vector for the display refresh...
;*********************************** Begin display routines (interrupt driven) ********************************************************
TranslateASCII
	; translate w (ASCII) into display code in w
	; do NOT relocate this code- it could cause problems with overflow of PCL (program counter)
	movwf	t1
	andlw	0x7F			; ignore decimal point
	call	InternalTranslateASCII
	btfsc	t1,7
	iorlw	0x10			; add decimal point
	return

InternalTranslateASCII
	movwf	t0				; temporary storage
	movlw	0x30
	subwf	t0,0			; now w=input-0x30
	btfsc	STATUS,C
	goto	transn0			; result is > 0x30
	retlw	0x00
transn0
	movlw	0x3A
	subwf	t0,0			; now w=input - 0x39
	btfss	STATUS,C
	goto	transdigit
transn1
	movlw	0x41
	subwf	t0,0			; now w=input - 0x41
	btfsc	STATUS,C
	goto	transn2
	retlw	0x00
transn2
	movlw	0x5B
	subwf	t0,0			; now w=input - 0x5A
	btfsc	STATUS,C
	retlw 	0x08			; dash
transalpha
	movlw	0x37
	subwf	t0,f
	goto	transdecode
transdigit
	movlw	0x30
	subwf	t0,f
transdecode
	movfw	t0
	addwf	PCL,f
; the segments are as follows:
;
;     ---1---
;    |       |
;    4       2
;    |       |
;     ---8---
;    |       |
;    80      20
;    |       |
;     --40---
;              @ = 10 (decimal point)
;
	retlw	0xE7			; '0'
	retlw	0x22			; '1'
	retlw	0xCB			; '2'
	retlw	0x6B			; '3'
	retlw	0x2E			; '4'
	retlw	0x6D			; '5'
	retlw	0xED			; '6'
	retlw	0x23			; '7'
	retlw	0xEF			; '8'
	retlw	0x6F			; '9'
	retlw	0xAF			; 'A'
	retlw	0xEC			; 'B'
	retlw	0xC5			; 'C'
	retlw	0xEA			; 'D'
	retlw	0xCD			; 'E'
	retlw	0x8D			; 'F'
	retlw	0xED			; 'G'
	retlw   0xAE			; 'H'
	retlw	0x22			; 'I'
	retlw 	0xE2			; 'J'
	retlw	0xEB			;  small A='K'
	retlw	0xC4			; 'L'
	retlw	0x00			; M is missing
	retlw	0xA8			; 'N'
	retlw	0xE8			; 'O'
	retlw	0x8F			; 'P'
	retlw	0x00			; Q is missing
	retlw	0x88			; 'R'
	retlw	0x6D			; 'S'
	retlw   0xCC			; 'T'
	retlw   0xE0			; 'U'
	retlw	0x00			; V is missing
	retlw	0x00			; W is missing
	retlw	0x00			; X is missing
	retlw   0x6E			; 'Y'
	retlw   0xCB			; 'Z'
LoadSR
	; loads the Shift register with value in w
	; low level routine used for displaying digits
	bcf		STATUS,RP0
	bcf		STATUS,RP1		; select Bank 0
	bsf		PORTB,SRGEN		; turn G enable off for 74595
	bsf		PORTB,SRCLK		; put clock high
	movwf	t1				; store value
	movlw	0x08			; number of bits
	movwf	t0				; store in counter
LoadSR0
	rlf		t1,f			; rotate left through carry
	bsf		PORTB,SRDAT		; defaults to Data high (flags not affected)
	btfss	STATUS, C
	bcf		PORTB,SRDAT		; put line low if carry was low
	bcf		PORTB,SRCLK		; clock line low
	bsf		PORTB,SRCLK		; clock line high
	decfsz	t0,f			; decrement bit count skip if zero
	goto 	LoadSR0			; loop again as many times as there are bits
	bcf		PORTB,SRCLK
	bsf		PORTB,SRCLK
	bcf		PORTB,SRGEN		; enable outputs...
	return					; finished

ScanDisplay
	; scans one digit of the display according to the
	; register disptr
	bcf		STATUS,RP0
	bcf		STATUS,RP1		; choose bank 0
	bsf		PORTB,SRGEN		; turn off all	
	bcf		PORTA,DIG0		
	bcf		PORTA,DIG1
	bcf		PORTB,DIG2
	bcf		PORTB,DIG3		; turn off all cathodes
	bcf		t0,7
	movlw	0x03			; 4 digit mask
	andwf	disptr,0
	btfsc	STATUS,Z
	goto	scan0
	xorlw	0x01
	btfsc	STATUS,Z
	goto	scan1
	xorlw	0x03
	btfsc	STATUS,Z
	goto	scan2
scan3
	movfw	dis3
	bsf		PORTB,DIG3
	btfsc	flashing,3
	bsf		t0,7
	goto	scancont
scan2
	movfw	dis2
	bsf		PORTB,DIG2
	btfsc	flashing,2
	bsf		t0,7
	goto	scancont
scan1
	movfw	dis1
	bsf		PORTA,DIG1
	btfsc	flashing,1
	bsf		t0,7
	goto	scancont
scan0
	movfw	dis0
	bsf		PORTA,DIG0
	btfsc	flashing,0
	bsf		t0,7
scancont
	movwf	t1
	btfsc	flashing,6
	goto	dontflashit
	btfss	t0,7			; bit7=1 indicates flash...
	goto	dontflashit
	movfw	flashspeed
	andwf	disptr,0
	btfsc	STATUS,Z		; this controls the speed of the flashing...
	goto	dontflashit
	btfss	flashing,7
	goto	here11
	movlw	0x70
	andwf	flashing,f
	goto	dontflashit
here11
	clrf	t1
dontflashit
	movfw	t1
	call	TranslateASCII
	call	LoadSR
	incf	disptr,f
dontclearflash
	btfsc	settings,2		; beep disabled?
	bcf		flashing,4
	btfsc	settings,3
	bcf		flashing,5
	btfsc	flashing,5
	goto	beepon
	btfss	flashing,4
	return
beepon
	bsf		PORTA,LEDBIT		; turn beep on...
	movfw	flashspeed
	andwf	disptr,0
	btfsc	STATUS,Z
	return						; nothing yet...
noalarm
	bcf		PORTA,LEDBIT
	bcf		flashing,4			; a one off...
	bcf		flashing,5
	return

Interrupt
	; this is the interrupt service routine for the display refresh and button state gathering
	swapf 	STATUS, W 			; Swap status to be saved
	movwf 	STATUS_TEMP 		; Save status to bank zero
	clrf 	STATUS 				; bank 0, regardless of current
	clrf 	PCLATH 				; Page zero, regardless of
interruptcontinue
	bsf		STATUS,RP0
	movfw	CMCON
	bcf		STATUS,RP0
	bcf		PIR1,TMR1IF
	call	ScanDisplay			; refresh the display (move on to next digit)
	movlw	0xF0				; the value 0xD0 gives refresh of about 60Hz when running at 8MHz
	movwf	TMR1H				; write to TIMER 1 to get it to overflow faster now (it counts UP)
endinterrupt
	bcf		STATUS,RP0			; select bank 0
endint
	movfw	button0
	xorlw	0x00
	btfsc	STATUS,Z			; is button0 register 0?
	call	processbutton0		; if 0 then listen to button 0
	movfw	button1
	xorlw	0x00
	btfsc	STATUS,Z			; is button1 register 0?
	call	processbutton1		; if 0 then listen to button 1
endint2
	movf 	PCLATH_TEMP, W 		; Restore PCLATH
	movwf 	PCLATH				; Move W into PCLATH
	swapf 	STATUS_TEMP, W 		; Swap STATUS_TEMP register
								; sets bank to original
	movwf 	STATUS 				; Move W into STATUS register
	swapf 	W_TEMP, F 			; Swap W_TEMP
	swapf 	W_TEMP, W 			; Swap W_TEMP into W
	retfie						; return from the interrupt

processbutton0
	; so the button0 register is 0 which means we can listen to button 0
	btfsc	PORTB,BUTTON0		; if the button pressed?
	goto	notpressedbutton0   ; do if the button is not pressed
pressedbutton0
	movfw	oldbutton0
	xorlw	0xFF
	btfsc	STATUS,Z			; do not increment past 0xFF
	goto	button0release
	incf	oldbutton0,f
	return
button0release
	movfw	oldbutton0
	movwf	button0
	return

notpressedbutton0
	movfw	oldbutton0
	xorlw	0x00
	btfsc	STATUS,Z			; is oldbutton0 register 0?
	return						; do nothing...
	movfw	oldbutton0
	movwf	button0				; store the pressed time...
	clrf	oldbutton0			; clear the internal counter
	return						; finished

processbutton1
	; so the button1 register is 0 which means we can listen to button 1
	btfsc	PORTB,BUTTON1		; if the button pressed?
	goto	notpressedbutton1   ; do if the button is not pressed
pressedbutton1
	movfw	oldbutton1
	xorlw	0xFF
	btfsc	STATUS,Z			; do not increment past 0xFF
	goto	button1release
	incf	oldbutton1,f
	return
button1release
	movfw	oldbutton1
	movwf	button1
	return

notpressedbutton1
	movfw	oldbutton1
	xorlw	0x00
	btfsc	STATUS,Z			; is oldbutton0 register 0?
	return						; do nothing...
	movfw	oldbutton1
	movwf	button1				; store the pressed time...
	clrf	oldbutton1			; clear the internal counter
	return						; finished

;************************************** end of display routines ***************************************************
;******************************************************************************************************************
Reset							; Program Execution begins here.
	clrf	PCLATH
	bcf		STATUS,RP0
	bcf		STATUS,RP1			; select bank 0
	movlw	DEFAULTA
	movwf	PORTA
	movlw	DEFAULTB
	movwf	PORTB
	bsf		STATUS,RP0			;Select Bank 1
	bcf		OPTION_REG,7		; enable pull ups on PORTB (used for BUTTON inputs)
	movlw	DEFAULTTRISA
	movwf	TRISA		
	movwf	ANSEL
	movlw	0x0D
	movwf	CMCON				; in bank 1
	bsf		OSCCON,6			;Set oscilator to 8MHz
	bsf		OSCCON,5
	bsf 	OSCCON,4
	movlw	DEFAULTTRISB	 
	movwf	TRISB				; set direction of bits in PortB
	bcf		STATUS,RP0			; Select Bank 0
;*************************Start Initializing Values*****************************************************************	
	movlw	SETTINGSDEFAULT		; start initializing some values...
	movwf	settings			; initialize the settings variable
	clrf	mode				; initialize mode
	clrf	state				; initial state
	clrf	loopcounter
	clrf	dis0
	clrf	dis1
	clrf	dis2
	clrf	dis3				; blank the display
	clrf	disptr
	clrf	currentmemory
	clrf	potmovement
	clrf	cursor
	clrf	key
	clrf	button0
	clrf	button1				; initialize button system (interrupt driven) to accept new presses
	clrf	oldbutton0
	clrf	oldbutton1
	clrf	flashing
	clrf	locked
	clrf	status
	movlw	0x84
	movwf	alarmlevel			; default is 132 which represents 22.0V
	clrf	alarmperiod
	movlw	0xFF
	movwf	pwml
	movwf	pwmmaxl
	movwf	anspeedsavel
	movlw	0x03
	movwf	pwmh
	movwf	pwmmaxh
	movwf	anspeedsaveh
	movlw	0x20
	movwf	flashspeed
	clrf	setfrequency
;***************End Initializing values***************************************************************
	movlw	PORADDRESS
	movwf	address
	call	readeeprom			; read POR register
	xorlw	PROGVER				; is it 55?
	btfsc	STATUS,Z			
	goto	retrieveporvalues	; YES, so values have been previously stored in EEPROM, so restore them...
setporvalues					; NO so set initial values in EEPROM
	movlw	PROGVER
	call	writeeeprom			; write POR value
	movlw	settings
	movwf	address
	movfw	settings
	call	writeeeprom
	movlw	setfrequency
	movwf	address
	movfw	setfrequency
	call	writeeeprom
	movlw	BASEMEMORY
	movwf	address
	movlw	0x08
	movwf	z2
lp0set
	movlw	0x03
	call	writeeeprom
	incf	address,f
	movlw	0xFF
	call	writeeeprom
	incf	address,f
	decfsz	z2,f
	goto	lp0set

	movlw	alarmlevel
	movwf	address
	movfw	alarmlevel
	call	writeeeprom
	goto	initcont
retrieveporvalues
	movlw	settings
	movwf	address
	call	readeeprom
	movwf	settings
	movlw	setfrequency
	movwf	address
	call	readeeprom
	movwf	setfrequency
	movlw	alarmlevel
	movwf	address
	call	readeeprom
	movwf	alarmlevel
initcont
	movlw	0x01
	movwf	T1CON				; set up timer 1 enable it and clock source is FOSC/4 (2MHz)
	bsf		STATUS,RP0			; select bank 1
	bsf		PIE1,TMR1IE			; enable interrupt on Timer 1 overflow (used for display refresh)
	bcf		STATUS,RP0			; select bank 0
	bcf		STATUS,RP1
	bsf		PCLATH,3
	call	ComputeFrequency	; compute and set the frequency...
	call	WritePWM
	bcf		PCLATH,3			; initially off...
	bsf		INTCON,PEIE			; enable peripheral interrupts
	bsf		INTCON,GIE			; enable interrupts (for display refresh)
main
mainloop
	incf	loopcounter,f
	bcf		STATUS,RP1
	bsf		STATUS,RP0			; select bank 1
	bsf		TRISB,PWMOUT		; turn off PWM signal
	bcf		STATUS,RP0
	bsf		PCLATH,3
	call	ReadAnalogInputs	; read all analog values
	bcf		PCLATH,3
	btfsc	mode,3
	goto	mainlo1
	bsf		STATUS,RP0
	bcf		TRISB,PWMOUT		; turn on PWM signal
mainlo1
docheckpot
	bcf		STATUS,RP0
	movfw	currentmemorylast
	subwf	currentmemorylast2,0
	xorlw	0x00
	btfss	STATUS,Z
	goto	notequal
equal
	movlw	POTSTABLECONSECUTIVE			; number of consecutive small deviations...
	subwf	potmovement,0
	btfsc	STATUS,C
	goto	reachednumber
	incf	potmovement,f
	goto	potcheckexit
reachednumber
	bsf		key,6			; it is now stable!
	goto	potcheckexit	; done!	
notequal					; the movement is higher than threshhold...
	bcf		key,6
	clrf	potmovement
potcheckexit
	rlf		anspeedpotl,0
	rlf		anspeedpoth,0		; now w has 0-7 in the least sig three bits according to the memory selected...
	; now we look up the memory location...
	movwf	z0
	movfw	currentmemorylast2
	movwf	currentmemorylast
	movfw	z0
	movwf	currentmemorylast2
	btfss	mode,2
	goto	ptch1
	movwf	currentmemory
ptch1
	btfsc	mode,1
	goto	freezeit
	btfss	mode,0
	goto	continuematch
memorymode
	bcf		STATUS,C
	rlf		currentmemory,0		; now *2
	addlw	BASEMEMORY
	movwf	address
	call	readeeprom
	andlw	0x03
	movwf	anspeedpoth			; get memory value (high byte)
	incf	address,f
	call	readeeprom
	movwf	anspeedpotl			; get memory value (low byte)...
	goto	continuematch
freezeit
	movfw	anspeedsavel
	movwf	anspeedpotl
	movfw	anspeedsaveh
	andlw	0x03
	movwf	anspeedpoth
continuematch
	bsf		PCLATH,3
	call	ComputeBackEmfNormalized
	bcf		PCLATH,3
	movfw	backemfnormh
	iorwf	backemfnorml,0		; if backemf=0 then motor is not connected
	btfss	STATUS,Z			; switch is open, so for automatic soft start we must
	goto	golock				; bring the current speed to off too.
mainlo2
	incf	zerocounter,f
	movfw	zerocounter
	xorlw	0x08
	btfss	STATUS,Z
	goto	golockz
	decf	zerocounter,f
	movlw	0x03
	movwf	pwmh
	movlw	0xFF
	movwf	pwml				; put speed to off for automatic soft start
	;bsf		flashing,7			; request flashing off...
	clrf	pdeltah
	clrf	pdeltal
	goto	extracontadjust
golock
	clrf	zerocounter			; if backemf not 0 then try to lock on speed
golockz
	movfw	anspeedpoth
	xorlw	0x03
	btfss	STATUS,Z
	goto	skipbtz
	movfw	anspeedpotl
	xorlw	0xFF
	btfsc	STATUS,Z
	goto	mainlo2
skipbtz
	clrf	z1
	movfw	anspeedpotl
	subwf	backemfnorml,0
	movwf	z0
	btfss	STATUS,C
	decf	z1,f
	movfw	anspeedpoth
	subwf	backemfnormh,0
	addwf	z1,f					; now z1:z0 is equal to = backemfnorm - anspeedpot
	movfw	z0
	movwf	f0
	movfw	z1
	movwf	f1						; now f1:f0=z1:z0
	btfss	f1,7					; is it negative?
	goto	skipinverse
	comf	f1,f
	comf	f0,f
	incf	f0,f
	btfsc	STATUS,Z
	incf	f1,f
skipinverse
	movfw	f1
	xorlw	0x00
	btfss	STATUS,Z
	goto	lockcont
	movlw	LOCKLIMIT
	subwf	f0,0
	btfsc	STATUS,C
	goto	lockcont
	; it is locked, so
	movlw	MAXLOCK
	subwf	locked,0
	btfsc	STATUS,C
	goto	isatlimitmax
	incf	locked,f
	goto	contmax
isatlimitmax
	btfsc	status,0
	goto	isatlimitmax2
	clrf	disptr
	bsf		flashing,4			; request beep...
isatlimitmax2
	bsf		status,0			; is locked.
contmax
	bsf		flashing,7			; request flashing off...
	clrf	pdeltal
	clrf	pdeltah
	goto	extracontadjust
lockcont
	movfw	locked
	xorlw	0x00
	btfsc	STATUS,Z
	goto	isatlimit
	decf	locked,f
	goto	lockcont1
isatlimit
	bcf		status,0				; so not locked...
lockcont1
	btfss	z1,7
	goto	increasespeed
decreasespeed
	movlw	0x01
	movwf	pdeltal
	clrf	pdeltah
	goto	extracont
increasespeed
	movlw	0xFF
	movwf	pdeltal
	movlw	0xFF
	movwf	pdeltah
extracont
extracontadjust
	bsf		PCLATH,3	
	call	adjustspeed
	bcf		PCLATH,3
gocont
	btfss	PORTB,BUTTON0
	goto	gocont1000	
	btfss	PORTB,BUTTON1
	goto	gocont1000
	btfss	key,5
	goto	gocont1000
	clrf	oldbutton0
	clrf	oldbutton1
	clrf	button0
	clrf	button1
	bcf		key,5
gocont1000
	btfsc	key,5			; if key(5)=1 ignore buttons...
	goto	statebranch
gocont999
	movfw	button0			; check if any buttons are pressed?
	xorlw	0x00
	btfsc	STATUS,Z
	goto	checknextbutton
	btfss	button0,7		; long or short press?
	goto	but0short
but0long
	btfss	button0,6
	goto	but0short
	bsf		key,0
	bcf		key,1
	bsf		key,7
	bsf		key,5			; block
	clrf	button0
	goto	potcheck		; continue...
but0short
	movlw	DEBOUNCEINTERVAL
	subwf	button0,0
	btfss	STATUS,C
	goto	but0skip		; debounce short pulses...
	bsf		key,7
	bcf		key,0
	bcf		key,1			
but0skip
	clrf	button0
	goto	potcheck		; continue...
checknextbutton
	movfw	button1			; check other button?
	xorlw	0x00
	btfsc	STATUS,Z
	goto	potcheck		; nothing, so go on to do display...
	btfss	button1,7		; short or long press?
	goto	but1short
but1long
	btfss	button1,6
	goto	but1short
	bsf		key,1
	bsf		key,0
	bsf		key,7	
	bsf		key,5			; block
	clrf	button1
	goto	potcheck
but1short
	movlw	DEBOUNCEINTERVAL
	subwf	button1,0
	btfss	STATUS,C
	goto	but1skip
	bsf		key,1
	bcf		key,0
	bsf		key,7
but1skip
	clrf	button1
potcheck					; check for activity on the pot bit 6 of key =1 when pot is stable!
	goto	statebranch

	org 0x400
statebranch
	call	Delay
	movfw	loopcounter
	xorlw	0x28
	btfss	STATUS,Z
	goto	mainloop
	clrf	loopcounter
checkbatalarm
	incf	alarmperiod,f
	bsf		PCLATH,3
	call	ComputeBatteryVoltage
	bcf		PCLATH,3					; for the battery alarm...
	movfw	alarmlevel
	subwf	batterylevel,0
	btfsc	STATUS,C
	goto	nobatalarm
batalarm
	call	gettablealarmperiod
	subwf	alarmperiod,0
	btfss	STATUS,C
	goto	nobatalarm
	bsf		flashing,6				; disable flashing...
	movlw	0x00
	movwf	dis0
	movlw	'L'
	movwf	dis1					; small 'A'
	movlw	'O'
	movwf	dis2
	movlw	0x00
	movwf	dis3
	call	gettablealarmperiod
	addlw	0x05
	subwf	alarmperiod,0
	btfss	STATUS,C
	goto	mainloop
	clrf	disptr
	bsf		flashing,5
	clrf	alarmperiod
	goto	mainloop
nobatalarm
	movlw	0x0F
	andwf	state,0
	movwf	state
	bsf		PCLATH,2
	addwf	PCL,f
	goto	state0
	goto	state1
	goto	state2
	goto	state3
	goto	state4
	goto	state5
	goto	state6
	goto	state7
	goto	state8
	goto	state9
	goto	stateA
	goto	stateB
	goto	stateC
	goto	stateD
	goto	stateE
	goto	stateF
;************************
gettablealarmperiod
	movfw	batterylevel
	movwf	z0
	rrf		z0,f
	rrf		z0,f
	rrf		z0,f
	rrf		z0,0
	andlw	0x0F
	bsf		PCLATH,2
	addwf	PCL,f
	retlw	0x08
	retlw	0x08
	retlw	0x10
	retlw	0x10
	retlw	0x20
	retlw	0x20
	retlw	0x40
	retlw	0x40	
	retlw	0x80	
	retlw	0x80	
	retlw	0x80
	retlw	0x80
	retlw	0x80
	retlw	0x80
	retlw	0x80
	retlw	0x80
poststate
	btfss	key,7
	goto	mainloop
	call	getnextstate	
	movwf	state
	goto	mainloop
getnextstate
	rlf		state,f
	rlf		state,0
	andlw	0x3C
	movwf	state
	movfw	key
	andlw	0x03
	bcf		key,7	
	addwf	state,0
	bsf		PCLATH,2
	btfsc	PORTA,0				; is display board connected?
	retlw	0x00				
	addwf	PCL,f
	retlw	0x01
	retlw	0x0A
	retlw	0x02
	retlw	0x0F	;0
	retlw	0x0E
	retlw	0x00
	retlw	0x00
	retlw	0x00	;1
	retlw	0x04
	retlw	0x00
	retlw	0x00
	retlw	0x00	;2
	retlw	0x00
	retlw	0x03
	retlw	0x00
	retlw	0x03	;3
	retlw	0x05
	retlw	0x05
	retlw	0x05
	retlw	0x02	;4
	retlw	0x00	; must be zero
	retlw	0x00	; must be zero
	retlw	0x00	; must be zero
	retlw	0x00	;5
	retlw	0x08
	retlw	0x08
	retlw	0x08
	retlw	0x08	;6
	retlw	0x08	
	retlw	0x08	
	retlw	0x08	
	retlw	0x08	;7
	retlw	0x00
	retlw	0x00
	retlw	0x00
	retlw	0x00	;8
	retlw	0x0D
	retlw	0x0D
	retlw	0x0D
	retlw	0x07	;9
	retlw	0x0B
	retlw	0x00
	retlw	0x00
	retlw	0x00	;A
	retlw	0x00
	retlw	0x00
	retlw	0x00
	retlw	0x00	;B
	retlw	0x01
	retlw	0x01
	retlw	0x01
	retlw	0x01	;C
	retlw	0x00	
	retlw	0x00
	retlw	0x00
	retlw	0x00	;D
	retlw	0x09
	retlw	0x0D
	retlw	0x0D
	retlw	0x06	;E
	retlw	0x0F
	retlw	0x00
	retlw	0x0F
	retlw	0x03	;F
;*************************
stateF
	bsf		flashing,6
	movlw	'A'
	movwf	dis0
	movlw	'0'
	btfss	settings,3
	movlw	'1'
	movwf	dis1
	movlw	'B'
	movwf	dis2
	movlw	'0'
	btfss	settings,2
	movlw	'1'
	movwf	dis3
	btfss	key,7
	goto	poststate
	btfsc	key,0
	goto	poststate
	movlw	0x08
	btfss	key,1
	movlw	0x04
	xorwf	settings,f
	movlw	settings
	movwf	address
	movfw	settings
	call	writeeeprom
	goto	poststate
state0
	clrf	mode
	bsf		PCLATH,3
	call	DisplaySpeedPercent
	bcf		PCLATH,3
	bcf		key,6
	clrf	potmovement
	goto	poststate
state1
	bsf		PCLATH,3
	call	DisplayBatteryVoltage
	bcf		PCLATH,3	
	btfss	key,7
	goto	poststate
	btfss	key,1
	goto	poststate
	btfss	key,0
	goto	poststate
	bsf		PCLATH,3
	call	Settopercentabove
	bcf		PCLATH,3
	clrf	disptr
	bsf		flashing,4					; request beep.
	goto	poststate
state2
	btfsc	mode,1							; if not frozen...
	goto	st2skip
	movfw	anspeedpoth
	movwf	anspeedsaveh
	movfw	anspeedpotl
	movwf	anspeedsavel
	bsf		mode,1							; freeze pot!
st2skip
	bsf		mode,2							; accept memory values from pot
	movlw	0x01
	addwf	currentmemory,0
	bsf		PCLATH,3
	call	DisA
	bcf		PCLATH,3
	movfw	resl
	movwf	dis1
	movlw	'C'
	movwf	dis0
	movlw	0x60				; dash
	movwf	dis2
	movwf	dis3
	bsf		flashing,6			; disable flashing...
	btfss	key,7				; any key pressed?
	goto	st22
	bcf		key,1
	bsf		key,0
	goto	poststate			; abort!
st22
	btfss	key,6
	goto	poststate
	clrf	disptr
	bsf		flashing,4
	bsf		key,7
	bcf		key,0
	bcf		key,1
	goto	poststate

state3
	bsf		mode,3
	bcf		flashing,6
	bsf		flashing,0
	bsf		flashing,1
	movlw	'C'
	movwf	dis0
	movlw	'L'
	movwf	dis1
	movlw	PROGVER
	bsf		PCLATH,3
	call	DisA
	bcf		PCLATH,3
	movfw	resh
	iorlw	0x80				; add decimal point...
	movwf	dis2
	movfw	resl
	movwf	dis3
	btfss	key,7
	goto	poststate
	movfw	key
	andlw	0x03
	btfsc	STATUS,Z
	goto	poststate
	xorlw	0x02
	btfsc	STATUS,Z
	goto	poststate
	bsf		flashing,6
	movlw	PORADDRESS
	movwf	address
	movlw	0xFF
	call	writeeeprom				; erase POR!
allexecstop
	goto	allexecstop
state4
	bsf		mode,0
	bcf		mode,1
	bcf		mode,2
	bsf		PCLATH,3
	call	DisplayMemorySpeedPercent
	bcf		PCLATH,3
	btfss	key,7				; key pressed?
	goto	st41
	bcf		key,0
	bcf		key,1
	goto	poststate			; transition to 0=key
st41
	btfsc	key,6				; not stable?
	goto	poststate
	bsf		key,0
	bsf		key,1				; transition 11
	movfw	anspeedpoth
	movwf	anspeedsaveh
	movfw	anspeedpotl
	movwf	anspeedsavel
	bsf		mode,1				; freeze pot!
	bsf		key,7
	goto	poststate			; transition...

state5
	bcf		mode,0
	bcf		mode,1
	bsf		key,7			; empty transition...
	goto	poststate

state6
	bcf		flashing,6
	bsf		flashing,0
	bcf		flashing,1
	bsf		mode,3			; turn off PWM for good...
	bcf		mode,2
	bcf		mode,1
	bcf		mode,0
	movlw	'A'
	movwf	dis0
	bsf		PCLATH,3
	call	SetAlarmLevel
	call	DisplayAlarmLevel
	bcf		PCLATH,3
	goto	poststate
state7
	bcf		flashing,6		; stop flashing...
	bsf		flashing,0
	bcf		flashing,1
	bsf		mode,3			; turn off PWM for good...
	bcf		mode,2
	bcf		mode,1
	bcf		mode,0
	movlw	'F'
	movwf	dis0
	bsf		PCLATH,3
	call	SetFrequency
	call	DisplayFrequency
	bcf		PCLATH,3
	goto	poststate
stateD
	clrf	mode
	movlw	0x03
	movwf	pwmh
	movlw	0xFF
	movwf	pwml
	bsf		key,7
	goto	poststate
state8
	clrf	mode
	movlw	0x03
	movwf	pwmh
	movlw	0xFF
	movwf	pwml	
	clrf	disptr
	bsf		flashing,4
	bsf		key,7
	goto	poststate		; empty transition!
state9
	bsf		flashing,6
	bsf		mode,3
	bcf		mode,2
	bcf		mode,1
	bcf		mode,0
	movlw	'F'
	movwf	dis0
	bsf		PCLATH,3
	call	DisplayFrequency
	bcf		PCLATH,3
	goto	poststate
stateA
	btfsc	mode,1							; if not frozen...
	goto	stAskip
	movfw	anspeedpoth
	movwf	anspeedsaveh
	movfw	anspeedpotl
	movwf	anspeedsavel
	bsf		mode,1							; freeze pot!
stAskip
	bsf		mode,2							; accept memory values from pot
	movlw	0x01
	addwf	currentmemory,0
	bsf		PCLATH,3
	call	DisA
	bcf		PCLATH,3
	movfw	resl
	movwf	dis1
	movlw	'C'
	movwf	dis0
	movlw	0x60				; dash
	movwf	dis2
	movwf	dis3
	bsf		flashing,6			; disable flashing...
	btfss	key,7				; any key pressed?
	goto	stA2
	bsf		key,0
	goto	poststate			; abort!
stA2
	btfss	key,6
	goto	poststate
	clrf	key
	bsf		key,7
	goto	poststate
stateB
	bcf		STATUS,C
	rlf		currentmemory,0
	addlw	BASEMEMORY
	movwf	address
	movfw	anspeedsaveh
	call	writeeeprom			; save the value!
	incf	address,f
	movfw	anspeedsavel
	call	writeeeprom			; save the value!
	bsf		key,7
	clrf	disptr				; start beep timer...
	bsf		flashing,4			; beep on...
	goto	poststate			; done!
stateC
	; this state is not used as of version 3.0!
	movlw	0x01
	xorwf	settings,f
	movlw	settings
	movwf	address
	movfw	settings
	call	writeeeprom
	clrf	disptr
	bsf		flashing,4
	bsf		key,7
	goto	poststate
stateE
	bsf		flashing,6
	bsf		mode,3
	bcf		mode,2
	bcf		mode,1
	bcf		mode,0
	movlw	'A'
	movwf	dis0
	bsf		PCLATH,3
	call	DisplayAlarmLevel
	bcf		PCLATH,3
	goto	poststate
;**********************************************************************************************************************************************
readeeprom
; read into W from address address
	bcf		STATUS,RP0
	bcf		STATUS,RP1
	movfw 	address
	bsf		STATUS,RP1
	movwf	EEADR
	bsf		STATUS,RP0
	bcf 	EECON1, EEPGD	; Point to Data memory
	bsf 	EECON1, RD 		; EE Read
	bcf		STATUS,RP0
	movfw	EEDATA
	bcf		STATUS,RP1
	return

writeeeprom
; write W to address address
	movwf	z1
	bsf		STATUS,RP0
	bsf		STATUS,RP1		; EECON1
	btfsc 	EECON1, WR 		; Wait for write
	goto	$-1
	bcf		STATUS,RP0		; EEADR
	bcf		STATUS,RP1
	movfw	address
	bsf		STATUS,RP1
	movwf 	EEADR 			; Data Memory
	bcf		STATUS,RP1
	movfw	z1
	bsf		STATUS,RP1
	movwf 	EEDATA 			; Data Memory Value
	bsf		STATUS,RP0		; EECON1
	bcf 	EECON1, EEPGD 	; Point to DATA
	bsf 	EECON1, WREN 	; Enable writes
	bcf 	INTCON, GIE 	; Disable INTs.
	movlw	0x55
	movwf 	EECON2 			; Write 55h
	movlw	0xAA
	movwf	EECON2 			; Write AAh
	bsf 	EECON1, WR 		; Set WR bit to
	bsf 	INTCON, GIE 	; Enable INTs.
	bsf 	EECON1, WREN 	; Disable writes
	bcf		STATUS,RP0
	bcf		STATUS,RP1
	return

scrolldelay
	movlw	0x20
	movwf	z2
	clrf	z1
	decfsz	z1,f
	goto	$-1
	decfsz	z2,f
	goto	$-4
	return

writechar
	movwf	z0
	movlw	0x04
	subwf	cursor,0
	btfsc	STATUS,C
	goto	casescroll
	btfss	cursor,0
	goto	cases02
	btfss	cursor,1
	goto	case1
case3
	movfw	z0
	movwf	dis3
	goto	casecont
cases02
	btfss	cursor,1
	goto	case0
case2
	movfw	z0
	movwf	dis2
	goto	casecont
case1
	movfw	z0
	movwf	dis1
	goto	casecont
case0
	movfw	z0
	movwf	dis0
casecont
	incf	cursor,f
	return
casescroll
	call	scrolldelay
	movfw	dis1
	movwf	dis0
	movfw	dis2
	movwf	dis1
	movfw	dis3
	movwf	dis2
	movfw	z0
	movwf	dis3
	return

Delay
	movlw	0x0E
	movwf	z2
	clrf	z1
	decfsz	z1,f
	goto	$-1
	decfsz	z2,f
	goto	$-4
	return
;******************************************************************************************************************************
; Some Input Buttons Routines...
;******************************************************************************************************************************
	org 0x800
#include <math16.inc>                ;constants and varible definitions
#include <floasc.inc>
;******************************************************************************************************************************
SetFrequency
	; set the frequency according to the pot level
	bcf		STATUS,RP1
	bcf		STATUS,RP0
	movfw	anspeedpoth
	movwf	z1
	movfw	anspeedpotl
	movwf	z0
	bcf		STATUS,C
	rrf		z1,f
	rrf		z0,f
	bcf		STATUS,C
	rrf		z1,f
	rrf		z0,f
	movfw	z0
	movwf	setfrequency		; the setting from the pot is divided by 4 to get 8 bit resolution 
	movlw	setfrequency
	movwf	address
	movfw	setfrequency
	bcf		PCLATH,3
	call	writeeeprom
	bsf		PCLATH,3
	call	ComputeFrequency
	return

SetAlarmLevel
	bcf		STATUS,RP0
	bcf		STATUS,RP1
	movfw	anspeedpoth
	movwf	AARGB2
	movfw	anspeedpotl
	movwf	AARGB3
	clrf	AARGB1
	clrf	AARGB0
	call	FLO3232
	movlw	0x7C
	movwf	BEXP
	movlw	0x79
	movwf	BARGB0
	movlw	0xDB
	movwf	BARGB1
	movlw	0x23
	movwf	BARGB2				; 
	call	FPM32				; A=A*0.244
	call	INT3232
	movfw	AARGB3
	movwf	alarmlevel
	movlw	alarmlevel
	movwf	address
	movfw	alarmlevel
	bcf		PCLATH,3
	call	writeeeprom
	bsf		PCLATH,3
	return

DisplayAlarmLevel
	bcf		STATUS,RP0
	bcf		STATUS,RP1
	movfw	alarmlevel
	movwf	AARGB3
	clrf	AARGB2
	clrf	AARGB1
	clrf	AARGB0
	call	FLO3232
	movlw	0x81
	movwf	BEXP
	movlw	0x40
	movwf	BARGB0
	movlw	0x00
	movwf	BARGB1
	movlw	0x00
	movwf	BARGB2				; 
	call	FPD32				; A=A/6
	goto	displaytwodigits

ComputeFrequency
	movfw	setfrequency
	movwf	AARGB3
	clrf	AARGB0
	clrf	AARGB1
	clrf	AARGB2
	call	FLO3232				; convert to float
	movlw	0x83
	movwf	BEXP
	movlw	0x65
	movwf	BARGB0
	movlw	0x65
	movwf	BARGB1
	movlw	0x65
	movwf	BARGB2				; B=28.6745098
	call	FPM32				; A=A*B
	movlw	0x87
	movwf	BEXP
	movlw	0x74
	movwf	BARGB0
	movlw	0x00
	movwf	BARGB1
	movlw	0x00
	movwf	BARGB2				; B=488.00
	call	FPA32				; A=A+B
	movfw	AEXP
	movwf	BEXP
	movfw	AARGB0
	movwf	BARGB0
	movfw	AARGB1
	movwf	BARGB1
	movfw	AARGB2
	movwf	BARGB2				; copy to B register
	movlw	0x8F
	movwf	AEXP
	movlw	0x74
	movwf	AARGB0
	movlw	0x24
	movwf	AARGB1
	movlw	0x00
	movwf	AARGB2				; now A=125000 using fprep.exe
	call	FPD32				; now A=A/B
	call	INT3232				; convert to integer
	movfw	AARGB3
	movwf	z0
	movfw	AARGB2
	movwf	z1
	decfsz  AARGB3,0			; minus 1
	movwf	pr2value
	bcf		STATUS,C
	rlf		z0,f
	rlf		z1,f
	bcf		STATUS,C
	rlf		z0,f
	rlf		z1,f				; *4
	movfw	z1
	xorlw	0x04				; correct by -1 for the 488Hz frequency case
	btfss	STATUS,Z
	goto	compexit
	decf	z1,f
	decf	z0,f
compexit
	movfw	z0
	movwf	pwmmaxl
	movfw	z1
	movwf	pwmmaxh
	return

DisplayFrequency
; this displays the current frequency setting from the actual value of PR2 register
; as frequency= 125000/(PR2+1)
; displays three digits in kHz...
	clrf	AARGB0
	clrf	AARGB1
	clrf	AARGB2
	bcf		STATUS,RP0			; select bank 0
	movfw	pr2value 			; get the actual value
	bcf		STATUS,RP0
	addlw	0x01				; w=PR2+1
	movwf	AARGB3
	btfsc	STATUS,C			; carry?
	bsf		AARGB2,0			; set bit 0 if carry...
	call	FLO3232				; convert integer to float...
	movfw	AEXP
	movwf	BEXP
	movfw	AARGB0
	movwf	BARGB0
	movfw	AARGB1
	movwf	BARGB1
	movfw	AARGB2
	movwf	BARGB2				; copy to B register
	movlw	0x8F
	movwf	AEXP
	movlw	0x74
	movwf	AARGB0
	movlw	0x24
	movwf	AARGB1
	movlw	0x00
	movwf	AARGB2				; now A=125000 using fprep.exe
	call	FPD32				; now A=A/B=125000/(PR2+1)
	call	float_ascii4		; show 4 figures
	movlw	ones
	movwf	FSR
	movfw	INDF				; get the ones
	iorlw	0x80				; add decimal point...
	movwf	dis1
	incf	FSR,f
	movfw	INDF
	movwf	dis2
	incf	FSR,f
	movfw	INDF
	movwf	dis3
	return

DisplayMemorySpeedPercent
	bsf		flashing,0
	bsf		flashing,1
	movlw	'C'
	movwf	dis0
	movfw	currentmemory
	addlw	0x01
	call	DisA
	movfw	resl
	iorlw	0x80
	movwf	dis1
	bcf		flashing,6			; enable flashing...
	call	dispspinternal
	call	float_ascii2
	movlw	tenths				; display it.
	movwf	FSR
	movfw	INDF
	movwf	dis3
	decf	FSR,f
	movfw	INDF
	movwf	dis2
	return

DisplaySpeedPercent
; this routine displays the current setting of the backemf
; as a percentage 0-100%
; based on the value in the registers backemfnormh backemfnorml
; it does NOT actually convert the analog voltage on the pin!
	bsf		flashing,0
	bcf		flashing,1
	bcf		STATUS,RP1
	bcf		STATUS,RP0			; select bank 0
	bcf		flashing,6			; enable flashing...
	movlw	'P'
	movwf	dis0
	call	dispspinternal
	call	displaytwodigits
	return

dispspinternal
	clrf	AARGB0
	clrf	AARGB1
	movfw	backemfnormh
	movwf	AARGB2
	movfw	backemfnorml
	movwf	AARGB3
	call	FLO3232				; convert 32 bit integer to 32 bit float
	movlw	0x7B
	movwf	BEXP
	movlw	0xC8
	movwf 	BARGB0
	movlw	0x32
	movwf 	BARGB1
	movlw	0x0C				; fprep.exe was used here
	movwf 	BARGB2				; put the multiplying constant= -100/1023=-0.09775171
	call	FPM32				; floating point multiply...
	movlw	0x85
	movwf	BEXP
	movlw	0x48
	movwf	BARGB0
	movlw	0x00
	movwf	BARGB1
	movlw	0x00
	movwf	BARGB2
	call	FPA32				; A=A+100.00
	return

DisplayBackEmfVoltage
; this routine displays the current voltage level of the backemf
; based on the values in the registes anbackemfh and anbackemfl
; it does NOT actually convert the analog voltage on the pin!
; it does take into account the current settings in settings register.
; The main assumption is that the same voltage divider is used in the
; back emf divider as in the battery level divider
	bcf		STATUS,RP1
	bcf		STATUS,RP0
	movlw	'E'
	movwf	dis0
	clrf	AARGB0
	clrf	AARGB1
	movfw	anbackemfh
	movwf	AARGB2
	movfw	anbackemfl
	movwf	AARGB3
	call	icomputevoltage
	call	displaytwodigits
	return

ComputeBatteryVoltage
	bcf		STATUS,RP1
	bcf		STATUS,RP0			; select bank 0
	clrf	AARGB0
	clrf	AARGB1
	movfw	anbatteryh
	movwf	AARGB2
	movfw	anbatteryl
	movwf	AARGB3
	call	icomputevoltage
	call	icomputepercentage
	return

icomputevoltage
; internal entry point
	call	FLO3232				; convert 32 bit integer to 32 bit float
	btfsc	settings,VOLTIN0
	goto	disbat13
	btfsc	settings,VOLTIN1
	goto	disbat2
disbat0
	movlw	VIN00DIVEXP
	movwf	BEXP
	movlw	VIN00DIVB0
	movwf 	BARGB0
	movlw	VIN00DIVB1
	movwf 	BARGB1
	movlw	VIN00DIVB2
	movwf 	BARGB2				; put the multiplying constant
	goto 	disbatend
disbat13
	btfsc	settings,VOLTIN1
	goto	disbat3
disbat1
	movlw	VIN01DIVEXP
	movwf	BEXP
	movlw	VIN01DIVB0
	movwf 	BARGB0
	movlw	VIN01DIVB1
	movwf 	BARGB1
	movlw	VIN01DIVB2
	movwf 	BARGB2				; put the multiplying constant	
	goto	disbatend
disbat2
	movlw	VIN10DIVEXP
	movwf	BEXP
	movlw	VIN10DIVB0
	movwf 	BARGB0
	movlw	VIN10DIVB1
	movwf 	BARGB1
	movlw	VIN10DIVB2
	movwf 	BARGB2				; put the multiplying constant
	goto	disbatend
disbat3
	movlw	VIN11DIVEXP
	movwf	BEXP
	movlw	VIN11DIVB0
	movwf 	BARGB0
	movlw	VIN11DIVB1
	movwf 	BARGB1
	movlw	VIN11DIVB2
	movwf 	BARGB2				; put the multiplying constant
disbatend
	call	FPM32				; floating point multiply...
	return

icomputepercentage
	movfw	AEXP
	movwf	f0
	movfw	AARGB0
	movwf	f1
	movfw	AARGB1
	movwf	f2
	movfw	AARGB2
	movwf	f3					; save it
	movlw	0x81
	movwf	BEXP
	movlw	0x40
	movwf	BARGB0
	movlw	0x00
	movwf	BARGB1
	movlw	0x00
	movwf	BARGB2
	call	FPM32				; multiply A=A*6
	call	INT3232
	movfw	AARGB3
	movwf	batterylevel
	movfw	f0
	movwf	AEXP
	movfw	f1
	movwf	AARGB0
	movfw	f2
	movwf	AARGB1
	movfw	f3
	movwf	AARGB2
	return

DisplayBatteryVoltage
; this routine displays the current voltage level of the battery
; based on the values in the registers anbatteryh and anbatteryl
; it does NOT actually convert the analog voltage on the pin!
; it does take into account the current settings in settings register.
	bsf		flashing,6						; disable flashing...
	movlw	'H'				; high voltage version
	movwf	dis0
	movfw	f0
	movwf	AEXP
	movfw	f1
	movwf	AARGB0
	movfw	f2
	movwf	AARGB1
	movfw	f3
	movwf	AARGB2				; restore the value...
	call	displaytwodigits
	return

displaytwodigits
	call	float_ascii2		; convert float to ASCII (two figures)
	movlw	last_digit			; display it.
	movwf	FSR
	decfsz	FSR,f
	movfw	INDF
	movwf	dis3
	decfsz	FSR,f
	movfw	INDF
	iorlw	0x80
	movwf	dis2
	decfsz	FSR,f
	movfw	INDF
	movwf	dis1
	return

adjustspeed
	movfw	pdeltal	
	addwf	pwml,f
	btfsc	STATUS,C
	incf	pwmh,f
	movfw	pdeltah
	addwf	pwmh,f
	btfss	pwmh,7			; is it negative?
	goto	skipzeroit
	clrf	pwmh
	clrf	pwml
	goto	gofastexit
skipzeroit
	movlw	0x04
	subwf	pwmh,0
	btfss	STATUS,C
	goto	gofastexit
	movlw	0x03
	movwf	pwmh
	movlw	0xFF
	movwf	pwml
gofastexit
	call	WritePWM	; writes the pvalh value to the PWM duty
	return

AnalogDelay
	movlw	0x20
	movwf	z1
	decfsz	z1,f
	goto	$-1						; small delay for analog to digital conversions
	return

ReadAnalogInputs
	; real all analog Inputs
	bcf		INTCON,GIE				; disable interrupts
	bcf		STATUS,RP1
	bsf		STATUS,RP0				; select bank 1
	movlw	(1<<BATTERY)|(1<<BACKEMF)|(1<<SPEEDPOT)
	movwf	ANSEL					; ANSEL is in bank 1
	movlw	0x80
	movwf	ADCON1					; ADCON1 is in bank 1 (set RIGHT justification, 0=V-ref 5V=V+ref) see page 117 of datasheet for PIC 16f88
	bcf		STATUS,RP0				; select bank 0
	movlw	0x01 | (BATTERY << 3)	; turn ADC module on and select BATTERY voltage pin 
	movwf	ADCON0					; set up values see page 116 of datasheet for PIC 16f88
	call	AnalogDelay
	bsf		ADCON0,2				; start the conversion
	btfsc	ADCON0,2
	goto	$-1						; wait till done...
	bcf		STATUS,RP0				; select bank 0
	movfw	ADRESH
	movwf	anbatteryh				; store high byte of battery voltage reading...
	bsf		STATUS,RP0				; select bank 1
	movfw	ADRESL
	bcf		STATUS,RP0				; select bank 0
	movwf	anbatteryl				; store low byte of battery voltage reading...
	movlw	0x01 | (BACKEMF << 3)	; turn ADC module on and select BACKEMF voltage pin 
	movwf	ADCON0
	call	AnalogDelay
	bsf		ADCON0,2				; start the conversion
	btfsc	ADCON0,2
	goto	$-1						; wait till done...
	movfw	ADRESH
	movwf	anbackemfh				; store high byte of backemf voltage reading...
	bsf		STATUS,RP0				; select bank 1
	movfw	ADRESL
	bcf		STATUS,RP0				; select bank 0
	movwf	anbackemfl				; store low byte of backemf voltage reading...
	movlw	0x01 | (SPEEDPOT << 3)	; turn ADC module on and select SPEED POT reading...
	movwf	ADCON0
	call	AnalogDelay
	bsf		ADCON0,2				; start the conversion
	btfsc	ADCON0,2
	goto	$-1						; wait till done...
	movfw	ADRESH
	movwf	anspeedpoth				; store high byte of speed pot reading...
	bsf		STATUS,RP0				; select bank 1
	movfw	ADRESL
	bcf		STATUS,RP0				; select bank 0
	movwf	anspeedpotl				; store low byte of speed pot reading...
	bsf		INTCON,GIE				; enable interrupts again...
	return							; done.
;******************************* End ADC conversion routines ****************************************************************
WritePWM
	; writes the pwmh:pwml values to duty of PWM...
	bcf		STATUS,RP0
	bcf		STATUS,RP1
	clrf	AARGB0
	clrf	AARGB1
	movfw	pwmh
	movwf	AARGB2
	movfw	pwml
	movwf	AARGB3
	call	FLO3232				; convert 32 bit integer to 32 bit float
	movfw	AEXP
	movwf	BEXP
	movfw	AARGB0
	movwf	BARGB0
	movfw	AARGB1
	movwf	BARGB1
	movfw	AARGB2
	movwf	BARGB2
	clrf	AARGB0
	clrf	AARGB1
	movfw	pwmmaxh
	movwf	AARGB2
	movfw	pwmmaxl
	movwf	AARGB3
	call	FLO3232				; convert 32 bit integer to 32 bit float
	call	FPM32
	movlw	0x75
	movwf	BEXP
	movlw	0x00
	movwf 	BARGB0
	movlw	0x20
	movwf 	BARGB1
	movlw	0x08				; fprep.exe was used here to get
	movwf 	BARGB2				; put the multiplying constant= 1/1023=0.0009775171
	call	FPM32				; floating point multiply...
	call	INT3232				; convert to integer A=INT(A)
	movfw	AARGB2
	movwf	ipwmh
	movfw	AARGB3
	movwf	ipwml
	movfw	pr2value
	bcf		INTCON,GIE
	bsf		STATUS,RP0		; select bank 1
	movwf	PR2				; set the period...
	bcf		STATUS,RP0
	movfw	ipwml
	movwf	z1
	movfw	ipwmh
	movwf	z2
	rrf		ipwmh,f
	rrf		ipwmh,f
	rrf		ipwmh,0
	andlw	0xC0			; put the two MSBs first
	movwf	ipwmh
	movfw	ipwml
	movwf	z0
	rrf		z0,f
	rrf		z0,0
	andlw	0x3F			; get the other 6 bits
	iorwf	ipwmh,0
	movwf	CCPR1L			; write the 8 MSBs
	btfsc	ipwml,0			; test bit 0 of pwml
	goto	writepwm0
	bcf		CCP1CON,4
	goto	writepwm1
writepwm0
	bsf		CCP1CON,4
writepwm1
	btfsc	ipwml,1			; test bit 1 of pwml
	goto	writepwm2
	bcf		CCP1CON,5
	goto	writepwm3
writepwm2
	bsf		CCP1CON,5
writepwm3
	movlw   0x17
	movwf	T2CON			; enable timer 2 with 1:16 prescaler...
	bsf		CCP1CON,3
	bsf		CCP1CON,2		; enable PWM
	movfw	z1
	movwf	ipwml
	movfw	z2
	movwf	ipwmh
	bsf		INTCON,GIE
	return

ComputeBackEmfNormalized
	bcf		STATUS,RP0
	bcf		STATUS,RP1
	clrf	AARGB0
	clrf	AARGB1
	movfw	anbatteryh
	movwf	AARGB2
	movfw	anbatteryl
	movwf	AARGB3
	call	FLO3232				; convert 32 bit integer to 32 bit float
	movfw	AEXP
	movwf	BEXP
	movfw	AARGB0
	movwf	BARGB0
	movfw	AARGB1
	movwf	BARGB1
	movfw	AARGB2
	movwf	BARGB2
	clrf	AARGB0
	clrf	AARGB1
	movfw	anbackemfh
	movwf	AARGB2
	movfw	anbackemfl
	movwf	AARGB3
	call	FLO3232				; convert 32 bit integer to 32 bit float
	call	FPD32				; A=A/B
	movlw	0x88
	movwf	BEXP
	movlw	0x7F
	movwf 	BARGB0
	movlw	0xC0
	movwf 	BARGB1
	movlw	0x00   				; fprep.exe was used here to get 1023.00
	movwf 	BARGB2				; put the multiplying constant=1023
	call	FPM32				; floating point multiply...
	bcf		AARGB0,7			; force positive
	call	INT3232				; convert to integer A=INT(A)
	movlw	0x04
	subwf	AARGB2,0
	btfsc	STATUS,C
	goto	compfix
	movfw	AARGB2
	movwf	backemfnormh
	movfw	AARGB3
	movwf	backemfnorml
	return
compfix
	movlw	0x03
	movwf	backemfnormh
	movlw	0xFF
	movwf	backemfnorml
	return

DisA
	; display w as two digit hexadecimal number
	; output two ASCII digits in resh:resl
	; uses t0 for temporary storage of w (unchanged)
	movwf	z0			; temporary storage
	swapf	z0,0		; swap nibbles...
	andlw	0x0F		; and mask
	addlw	0x30		; add to make ASCII
	movwf	resh
	movlw	0x3A
	subwf	resh,0		; now w=digit - 0x39
	btfss	STATUS,C
	goto	disa1
	movlw	0x07
	addwf	resh,f
disa1
	movfw	z0
	andlw	0x0F
	addlw	0x30
	movwf	resl
	movlw	0x3A
	subwf	resl,0
	btfss	STATUS,C
	goto	disa2
	movlw	0x07
	addwf	resl,f
disa2
	movfw	z0			; restore accumulator
	return

Settopercentabove
	movfw	batterylevel
	movwf	AARGB3
	clrf	AARGB2
	clrf	AARGB1
	clrf	AARGB0
	call	FLO3232			; convert to float
	movlw	0x7E
	movwf	BEXP
	movlw	0x6A
	movwf	BARGB0
	movlw	0xAA
	movwf	BARGB1
	movlw	0xAB
	movwf	BARGB2			; B=0.91666666 or 91.66%
	call	FPM32
	call	INT3232
	movfw	AARGB3
	movwf	alarmlevel
	movlw	alarmlevel
	movfw	alarmlevel
	bcf		PCLATH,3
	call	writeeeprom
	bsf		PCLATH,3
	return


	END
