;**************************************************************
;	LCmeter.asm -- firmware for Digital LC Meter, derived from
;	earlier versions written initially by Neil Heckt of Almost
;	All Digital Electronics (Auburn, WA USA) for his original
; 	L-C meter design published in Electronics Now magazine, &
;	since then modified/enhanced by various people, including:
;	Phil Rice, VK3BHR
;	Egbert Jarings, PA0EJH
;	Andreas Winter
;	Christi Morariu
;	Mike Keitz
;	It also makes use of 24-bit floating point maths routines
; 	for Microchip Technology Inc's 8-bit PIC processors,
;	written by Frank J. Testa and described in MTI's App Note
; 	AN575 (available on the MTI website, www.microchip.com)
;	The FP routines are in the separate file FP.TXT, which is
; 	taken from the MTI library file FP24.A16, but with the 
;	routines FLO1624 and INT2416 removed because they are not
; 	needed here.
;
;	This version adapted for the PIC16F628A processor by
;	Jim Rowe for Silicon Chip. Last revised 26/2/2008.
; 	Uses a 4MHz crystal, so one machine cycle (mc) = 1us
;
;************************************************************
;
;	First define macros for frequently used code fragments
;
;	Select Register Bank 0
bank0	macro
	errorlevel	+302		; Re-enable bank warning
	BCF		STATUS,RP0		; Select Bank 0
	endm

;	Select Register Bank 1
bank1	macro
	BSF		STATUS,RP0		; Select Bank 1
	errorlevel	-302		; disable warning
	endm

;	Swap bytes in register file via W
swap	macro	this,that
	MOVF	this,w		; get this
	XORWF	that,f		; Swap using Microchip
	XORWF	that,w		; Tips'n Tricks
	XORWF	that,f		; #18
	MOVWF	this
	endm

;	Copy bytes in register file via W
copy	macro	from,to
	MOVF	from,W
	MOVWF	to
	endm
;
;	**********************************************************
;     STATUS bit definitions (used in FP.TXT)

#define	_C	STATUS,0
#define	_Z	STATUS,2

;	**********************************************************
;
;	CPU configuration
;
	list p=PIC16f628A
	#include		<p16f628A.inc>
	__CONFIG        _CP_OFF & _WDT_OFF & _PWRTE_ON & _HS_OSC & _BOREN_ON & _LVP_OFF
;
;**********************************************************
;
;	I/O Assignments
;
#define	LCD0	PORTB,3	; these bits used to write data nibbles
#define	LCD1	PORTB,2	; to LCD module DB7-DB4 (RB3->DB4, etc)
#define	LCD2	PORTB,1
#define	LCD3	PORTB,0
#define	ENA		PORTB,4	; LCD module 'Enable' control line
#define	RS		PORTB,5	; LCD module 'RS' control line
#define	functn	PORTB,6	; 0 = Inductor, 1 = Capacitor (S1 posn)
#define relay	PORTB,7	; 1 = activate RLY, switching in Ccal

;**************************************************************
;
;	file register declarations: uses only registers in bank0
;	bank 0 file registers begin at 0x20 in the 16F8628

	cblock	0x20
;
;   Floating Point Stack & other locations used by FP.TXT rtns
;
;	FP Stack: TOS	A = 	AEXP:AARGB0:AARGB1:AARGB3:AARGB4
;					B = 	BEXP:BARGB0:BARGB1:BARGB2
;					C = 	CEXP:CARGB0:CARGB1
	AARGB4
	AARGB3
	AARGB2
	AARGB1
	AARGB0
	AEXP			; 8 bit biased exponent for argument A
	SIGN			; save location for sign in MSB

	FPFLAGS			; floating point library exception flags
	
	BARGB2
	BARGB1
	BARGB0
	BEXP			; 8 bit biased exponent for argument B

	TEMPB3			; 1 Unused byte
	TEMPB2			; 1 Unused byte
	TEMPB1			; Used
	TEMPB0			; 1 Unused byte

	CARGB1
	CARGB0			; most significant byte of argument C
	CEXP			; 8 bit biased exponent for argument C
;
;	Main program storage locations
;
 	COUNT			; Bin to BCD convert (bit count)
	cnt				;                    (BCD BYTES)
	CHR 
	F1:2
	F2:2
	F3:2
	bcd:4			; BCD, MSD first 
	TabStop			; used to fix slow displays.
	TabTemp
	FPE				; we collect FP errors in here
	R_sign			; Holds "+" or " " (sign)
	EEflag:1		; Cal adjust flag
	endc

	cblock	0x70		; Common RAM
	cal_t:2			; Ccal temporary value
	PB_data:1		; LCD output munger temp
	links:1			; user test links copy
	COUNT1			; used by delay routines & prescaler flush
	COUNT2			; Timing (100ms)
	endc

EXP		equ	AEXP	; Used by FP.TXT
TEMP	equ	TEMPB0

;**************************************************************
;
;   GENERAL MATH LIBRARY DEFINITIONS

;	define assembler constants
B0		equ	0
B1		equ	1
B2		equ	2
B3		equ	3
B4		equ	4
B5		equ	5
B6		equ	6
B7		equ	7

MSB		equ	7
LSB		equ	0

;**************************************************************
;
;       FLOATING POINT literal constants
;
EXPBIAS         equ     D'127'
;
;       floating point library exception flags
;
IOV     equ     0	; bit0 = integer overflow flag
FOV     equ     1   ; bit1 = FP overflow flag
FUN     equ     2   ; bit2 = FP underflow flag
FDZ     equ     3   ; bit3 = FP divide by zero flag
NAN		equ		4	; bit4 = not-a-number exception flag
DOM		equ		5	; bit5 = domain error exception flag
RND     equ     6   ; bit6 = FP rounding flag
					; 0 = truncation
                    ; 1 = unbiased rounding to nearest LSB
SAT     equ     7   ; bit7 = FP saturate flag
					; 0 = term on exception w/out saturation
					; 1 = term on exception with saturation
					; to appropriate value

;*************************************************************

;	Program now begins
;
	org	0

GO:
	CLRWDT			; clear the WDT (0 << Reset)
	CALL InitIO		; initialise MPU's ports, etc
	BCF	relay		; make sure relay is off, to remove Ccal

START:
	CALL	LCDINIT	 	; initialise the LCD module	
	CALL	EE_RD		; retrieve Ccal integer value

cmdloop:
	CALL	HOME		; now clr LCD & 'home' its cursor

;	Next 'zero' the meter
;
Chk4Z:
	MOVLW   Calibr-0x2100	; Displays " Calibrating "
	CALL	pmsg			; message to inform the user
	CALL	Measure			; Dummy run to stabilise oscillator.
	CALL	MS200			; 200ms delay
	CALL	Measure			; Get freq in F3
	copy	F3+0,F1+0		; Copy F3 to F1
	copy	F3+1,F1+1
	BSF	relay				; Add standard cap (Ccap) via relay
	CALL	MS200			; 200ms delay again
	CALL	Measure			; Get freq in F3
	copy	F3+0,F2+0		; Copy F3 to F2
	copy	F3+1,F2+1
	BCF	relay				; Remove Ccap again - turn off relay
	CALL	MS200			; 200ms delay again
	CALL	Measure			; Dummy run to stabilise oscillator.
;
;	Now we read state of user test links on LCD bus
M_F3:
	bank1			; move up to mem bank 1 to specify
	MOVLW	b'01001111'	; RB data bits to read
	MOVWF	TRISB		; 1 = input, 0 = output
	bank0				; now back to bank 0
	CALL	MS2			; settling time delay
	copy	PORTB,links	; save link settings
	bank1				; then back up to bank 1 to
	MOVLW	b'01000000'	; restore RB bit data directions
	MOVWF	TRISB
	bank0				; then back down to bank 0
;
;	now check links, in case any tests or changes are wanted
	BTFSS	links,0		; skip if LK4 is high (i.e., no link)
	GOTO	cal_up		; otherwise go to 'Increase C reading'
	BTFSS	links,1		; now check if LK3 is high, skip if so
	GOTO	cal_dn		; otherwise go to 'reduce C reading'
	BTFSS	links,2		; skip if LK2 is high
	GOTO	osc1		; otherwise go check osc freq F1
	BTFSS	links,3		; skip if LK1 is high
	GOTO	osc2		; otherwise go check osc freq F2
	BCF	relay			; no links set, so ensure relay is off
	BTFSS	EEflag,0	; need to save Ccal value?
	GOTO	cont		; no, so just continue
	BCF	EEflag,0		; yes, so clear flag to show it's done
	CALL	EE_WR		; then save it
	GOTO	cont		; and continue
;
;	routine to add +10 to cal_t:2
cal_up:
	BSF	EEflag,0		; Say "we're adjusting"
	MOVLW	0x0a		; +10
	ADDWF	cal_t+1,f
	BTFSS STATUS,C
	GOTO	cont
	INCF	cal_t+0,f
	GOTO cont			; then continue
;
;	routine to add -10 to cal_t:2
cal_dn:
	BSF	EEflag,0	; Say "we're adjusting"
	MOVLW	0xf6		; -10
	ADDWF	cal_t+1,f
	BTFSS STATUS,C
	GOTO	hi_byte
	INCF	cal_t+0,f
hi_byte:
	MOVLW	0xff
	ADDWF	cal_t+0,f
	GOTO	cont		; then continue		
;
;	Measure & display osc freq for initial setup
osc2:
	BSF	relay			; turn on relay to connect Ccal
osc1:
	CALL	HOME		; clear LCD & home cursor
	CALL	Measure		; measure osc frequency.
	CALL	CLEAR
	BTFSS	INTCON,T0IF	; did counter overflow (TOIF=1)?
	GOTO	Do_Disp		; if not set, go display reading
	MOVLW	ovr-0x2100	; but if it is, give over-range msg
	CALL	pmsg
	GOTO	M_F3		; and back to check again
Do_Disp:
	CLRF	AARGB0		; Copy to 24 bit number
	MOVF	F3,W		; in AARGB0, 1, 2
	MOVWF	AARGB1		; for display
	MOVF	F3+1,W
	MOVWF	AARGB2
	CALL	Display
	GOTO	M_F3		; then back to check again
;
;	---------------------------------------------------------------
;	now we can continue
cont:
	CALL	HOME			; clear LCD & home cursor
	CALL	MS200			; 200ms delay
	CALL	Measure			; measure F3 & leave it there
	MOVF	F3,w			; test for too low frequency
	BTFSC 	STATUS,Z		; F < 2560Hz?
	GOTO 	OORange			; if yes, go display
	BTFSS	INTCON,T0IF		; otherwise test for too high F
	GOTO	OK2GO			; F > 655359Hz ?
OORange:
	MOVLW	ovr-0x2100		; give over/under range message
	CALL	pmsg
	GOTO	M_F3			; then back to check again
;
;	next precompute main bracketed terms needed for all calc's
OK2GO:
	CLRF	FPE				; clear FPE first
	CALL	F1_F2
	CALL	F1_F3
;
;	See what mode we're in (C or L) - check S1 pos via RB6
	BTFSS	functn			; 0 = L, 1 = C
	GOTO	Do_Ind			; if it's zero, go to L mode
Do_Cap:
	CALL	C_calc			; otherwise, go calculate C
	MOVF	FPE,f			; Any FP errors?
	BTFSS STATUS,Z
	GOTO	complain		; if yes, go and complain
	MOVLW	Cintro-0x2100	; otherwise show "C = " string
	CALL	pmsg
	CALL	C_disp			; then display value of C
	GOTO	M_F3			; and back to try again
Do_Ind:
	CALL	L_calc			; S1 was in L pos, so calculate L
	MOVF	FPE,f			; Any FP errors?
	BTFSS STATUS,Z
	GOTO	complain		; if yes, go complain
	MOVLW	Lintro-0x2100	; otherwise show "L = " string
	CALL	pmsg
	CALL	L_disp			; then display value of L
	GOTO	M_F3			; and back to try again
;
;	We got an FP error of some sort, so... 
complain:
	MOVLW	ovr-0x2100		; start of "Over range" msg
	CALL	pmsg			; so display it
	GOTO	M_F3			; and then back to try again

;**********************************************************
;
;	Subroutine to display a string with start address in W
;	(strings are in EEPROM) Quite a bit of bank switching here

pmsg:
	bank1
	MOVWF	EEADR			; pointer
pm1:
	BSF     EECON1,RD       ; EE Read
    MOVF    EEDATA,W        ; W = EEDATA, affects Z bit
    bank0					; Does not change Z bit
	BTFSC	STATUS,Z		; zero (Z=1) means all done
	RETURN					; so leave
	CALL	DATS			; continue (byte -> display)
	bank1
	INCF    EEADR,F         ; bump address
	GOTO	pm1				; loop back

;**********************************************************
;
;	Routine to delay for 2ms (untrimmed)
;
MS2:
	MOVLW	0xFD		; DELAY 2ms
	MOVWF	COUNT1
	MOVLW	0x66
	MOVWF	COUNT2
	GOTO	L3		

;**********************************************************
;
;	Delay for about 300ms, 200ms or 100ms (untrimmed)
;
MS300:
	CALL	MS100
MS200:
	CALL	MS100
MS100:
	MOVLW	0x7e		; Count up
	MOVWF	COUNT1		; to roll-over
	MOVLW	0x20
	MOVWF	COUNT2			
L3:
	INCFSZ	COUNT2,F
	GOTO	L3
	INCFSZ	COUNT1,F
	GOTO	L3
	RETLW	0			; leaves at end with 0 in w

;**********************************************************
;
; 	Routine to send a BCD nibble to display
PutNyb:
	ANDLW	0x0F		; mask off other BCD digit
	ADDLW	0x30		; Convert BIN to ASCII, fall thru

;**********************************************************
;
;	Send a byte to display
DATS:
	DECF	TabStop,F	; Time to tickle bad display?
	BTFSS STATUS,Z
	GOTO	DAT1		; Not yet, so go straight to DAT1
	MOVWF	TabTemp		; Save character
	
;	BTFSS	FIXIT		; Check if we got a crook one.
;	CALL	LINE2		; Skip this if good

	MOVF	TabTemp,W	; Restore character

DAT1:
	BSF	RS			; select LCD data register
CM:
	MOVWF	CHR		; save char to display
	SWAPF	CHR,W	; swap upper & lower nibbles (4b mode)
	CALL	PB_dly	; and send upper nibble
	MOVF	CHR,W	; then get lower nibble & send it 
						; by falling through (this time)
;**********************************************************
;
;	routine to send 4-bit nibble to LCD & wait (untrimmed)
PB_dly:
	MOVWF	PB_data		; save nibble for bit testing
	BTFSS	PB_data,0	; copy LSbit
	BCF	LCD0
	BTFSC	PB_data,0
	BSF	LCD0
	
	BTFSS	PB_data,1
	BCF	LCD1
	BTFSC	PB_data,1
	BSF	LCD1
	
	BTFSS	PB_data,2
	BCF	LCD2
	BTFSC	PB_data,2
	BSF	LCD2
	
	BTFSS	PB_data,3	; copy MSbit
	BCF	LCD3
	BTFSC	PB_data,3
	BSF	LCD3
	
	BSF	ENA		; now all 4 are set, so toggle ENA line high
	NOP			
	BCF	ENA		; then low again, before falling into 200us delay 
D200us:
	MOVLW	0x42		; DELAY  200us
	MOVWF	COUNT1	
NXT5:
	DECFSZ	COUNT1,F
	GOTO	NXT5	
	RETLW	0			; and finally depart with zero in w

;**************************************************************
;
;	Convert 24-bit binary number in <AARGB0,1,2> into BCD no.
;	in <bcd>
;
B2_BCD:

b2bcd:
   	MOVLW   d'24'		; Load number of bits (24) into
   	MOVWF   COUNT		; cycle counter, then
   	CLRF    bcd+0		; clear result area
   	CLRF    bcd+1
   	CLRF    bcd+2
   	CLRF    bcd+3
b2bcd2:
  	MOVLW   bcd 		; make pointer for indir addressg
    MOVWF   FSR
    MOVLW   d'4'		; also set cnt for four loops
    MOVWF   cnt
b2bcd3:
 	MOVLW   0x33            
    ADDWF   INDF,f          ; add to both nibbles
    BTFSC   INDF,3          ; test if low result > 7
    ANDLW   0xf0            ; low result >7 so take the 3 out
    BTFSC   INDF,7          ; test if high result > 7
    ANDLW   0x0f            ; high result > 7 so ok
    SUBWF   INDF,f          ; any results <= 7, subtract back
    INCF    FSR,f           ; point to next
    DECFSZ  cnt,f			; check if done four
    GOTO    b2bcd3
        
	RLF     AARGB2,f		; get another bit
    RLF     AARGB1,f
    RLF     AARGB0,f

    RLF     bcd+3,f         ; put it into bcd
    RLF     bcd+2,f
    RLF     bcd+1,f
    RLF     bcd+0,f

    DECFSZ  COUNT,f         ; all done?
    GOTO    b2bcd2          ; no, so loop back
    RETURN                  ; yes, so leave

;*************************************************************
;	routine to initialise LCD module, in 4-bit interface mode

LCDINIT:
 	CALL	MS200		; wait for LCD module hardware reset
	BCF	RS				; pull register select line low
	BCF	ENA				; also enable line low
	MOVLW	0x03		; send init code the first time
	CALL	PB_dly
	CALL	MS100		; wait for LCD to catch up

	MOVLW	0x03		; send init code the second time
	CALL	PB_dly

	MOVLW	0x03		; and then a third time
	CALL	PB_dly

	MOVLW	0x02		; then send Fn set code for 4 bits
	CALL	PB_dly
	
	MOVLW	0x0C		; and code for display on
	CALL	ST200us

	MOVLW	0x28		; and code for 2 lines, 5x7 dots
	CALL	ST200us
	
	MOVLW	0x06		; then code for entry mode
	CALL	ST200us		; (fall into CLEAR after short delay)

;************ CLEAR DISPLAY ***************************

CLEAR:
	MOVLW	0x01	; clear display
	GOTO	Home2	; (longer delay needed when clearing)

;*********** MOVE TO HOME *****************************

HOME:
	MOVLW	0x09		; Count characters
	MOVWF	TabStop		; before tickling display.
	MOVLW	0x02		; home display's cursor
Home2:
	CALL	STROBE
	GOTO	MS2

;**********************************************************
;
;	SENDS DATA TO LCD DISPLAY MODULE (4 BIT MODE)	
;

STROBE:
	BCF	RS		; select LCD's command register
	GOTO	CM	; jump back to CM

;************ MOVE TO START OF LINE 2 *****************

LINE2:
	MOVLW	0xC0	; address for start of 2nd line of disp

ST200us:
	CALL	STROBE
	GOTO	D200us

;*************************************************************
;       Initialise input & output ports, etc
;
InitIO:
	MOVLW	b'00000110'	; select mode for comparators
	MOVWF	CMCON 	   	; AN0(p17)= C1Vin-, AN3(p2)=C1out,
 						; AN1(p18)= C2Vin-
						; AN2(p1) = C1Vin+,C2Vin+
						; RA4(p3) = C2out (for pullup)
	bank1				; now up to bank1
	MOVLW	b'00000000'
	MOVWF	VRCON 	   	; to set Volt ref module to OFF
	MOVLW	0x37		; also set Option register
	MOVWF	OPTION_REG	; b7 = Port B weak pull-ups enabled
						; b6 = INTEDG falling edge (don't care)
						; b5 = Tmr0 clock from RA4/T0CKI
						; b4 = Tmr count on falling edge
						; b3 = Prescaler -> Tmr0
						; b2-0 = Prescale Tmr0 by 256
;
						; next set port A data directions
	MOVLW	b'11110111'	; 1 = input, 0 = output
	MOVWF	TRISA		; by loading TRISA reg
						; RA0 = input (C1Vin-)
						; RA1 = input (C2Vin-)
						; RA2 = input (C1Vin+,C2Vin+)
						; RA3 = output (C1out)
						; RA4 = input (C2out, but T0CKI in)
						; RA5-7 = inputs (unused)
;
						; and port B data directions
	MOVLW	b'01000000'	; 1 = input, 0 = output
	MOVWF	TRISB		; RB0 = output (LCD DB4)
						; RB1 = output (LCD DB5)
						; RB2 = output (LCD DB6)
						; RB3 = output (LCD DB7)
						; RB4 = output (LCD EN line)
						; RB5 = output (LCD RS line)
						; RB6 = input (S1 function sw in)
						; RB7 = output (Ccal relay coil)
	bank0				; finally back down to bank 0
	RETURN				; and leave

;**********************************************************
;
;	routine to measure frequency, save in F3 & F3+1
;
Measure:
	BCF	INTCON,T0IF		; first clear timer overflow flag bit
	CLRF	TMR0		; clear Tmr0 and prescaler
	CLRF	F3			; also clear F3 & F3+1
	CLRF	F3+1		; ready to receive 16-bit number
	bank1				; now up to bank 1
	MOVLW	b'11100111'	; to enable RA4 output to T0CKI
	MOVWF	TRISA		; (0 = output)
	CALL	MS100		; then go for 100ms delay
	MOVLW	b'11110111'	; before disabling RA4 output to T0CKI
	MOVWF	TRISA		; to close gate again (1 = input)
	bank0				; then swing down to bank 0
	MOVF	TMR0,W		; get high byte into w		
	MOVWF	F3			; then copy to big end of 16 bit result
						; (the comparator output is 1 because
						; we've forced it high, so T0CKI=1)
PSC1:
	bank1				; so swing up to bank 1
	BSF	OPTION_REG,T0SE	; clock the prescaler by toggling TOSE
	NOP
	BCF	OPTION_REG,T0SE
	bank0				; then back down to bank 0
	DECF	F3+1,F		; decrement the counter
	MOVF	TMR0,W		; Has TMR0 changed?
	XORWF	F3,W		; if unchanged, XOR -> 0
	BTFSC	STATUS,Z
	GOTO	PSC1
	RETURN				; F3 & F3+1 now hold 16 bit result

;**********************************************************
;
;	Display contents of AARGB0,1,2 on LCD
;	First convert to BCD, then to ASCII (nibble at a time)
;
Display:
	CALL	B2_BCD		; convert count to BCD		
	CALL	Swap0		; then get next digit
	CALL	Move0		; get another BCD digit
	CALL	Swap1
	CALL	Move1
	CALL	Swap2
	CALL	Move2
	CALL	Swap3
	GOTO	Move3		; includes RETURN

;**********************************************************
;
;	Formatted display of BCD work area for capacitor
;

C_disp:
	MOVF	R_sign,w	; Sign
	CALL	DATS

F_C1:
	MOVF	bcd+0,W
	ANDLW	0x0F
	BTFSC STATUS,Z
	GOTO	F_C2
	CALL	PutNyb
	CALL	Swap1
	CALL	Move1
	CALL	DoDP		; Print DP
	CALL	Swap2
	GOTO	F_C3U

;--------------------------------------------------

F_C2	SWAPF	bcd+1,W
	ANDLW	0x0F
	BTFSC STATUS,Z
	GOTO	F_C3
	CALL	PutNyb
	CALL	Move1
	CALL	DoDP		; Print DP
	CALL	Swap2
	CALL	Move2
	GOTO	F_C3U		; print nF (includes return)

;--------------------------------------------------

F_C3:
	MOVF	bcd+1,W
	ANDLW	0x0F
	BTFSC STATUS,Z
	GOTO	F_C4
	CALL	PutNyb
	CALL	DoDP		; Print DP
	CALL	Swap2
	CALL	Move2
	CALL	Swap3
F_C3U:
	MOVLW	Unit1-0x2100	; nF
	GOTO	pmsg		; (includes return)

;--------------------------------------------------

F_C4:
	SWAPF	bcd+2,W		; Digit1 == 0 ?
	ANDLW	0x0F
	BTFSS STATUS,Z
	GOTO	NoB1_C
	MOVLW	0x20		; YES PRINT A SPACE
	CALL	DATS

	MOVF	bcd+2,W		; Digit2 == 0 ?
	ANDLW	0x0F
	BTFSS STATUS,Z
	GOTO	NoB2_C

	MOVLW	0x20		; YES PRINT A SPACE
	CALL	DATS
	GOTO	NoB3_C

NoB1_C:
	CALL	Swap2			; 1
NoB2_C:
	CALL	Move2			; 2
NoB3_C:
	CALL	Swap3			; 3
	CALL	DoDP			; Print DP
	CALL	Move3			; 4
	MOVLW	Unit2-0x2100	; pF
	GOTO	pmsg			; (includes return)

;**********************************************************
;
;	Formatted display of BCD work area for Inductor
;

L_disp:
	MOVF	R_sign,w	; Sign
	CALL	DATS
F_L1:
	MOVF	bcd+0,W
	ANDLW	0x0F
	BTFSC STATUS,Z
	GOTO	F_L2
	CALL	PutNyb
	CALL	Swap1
	CALL	DoDP		; Print DP
	CALL	Move1
	CALL	Swap2
	GOTO	F_L2U		; Print mH includes RETURN

;--------------------------------------------------

F_L2:
	SWAPF	bcd+1,W
	ANDLW	0x0F
	BTFSC STATUS,Z
	GOTO	F_L3
	CALL	PutNyb
	CALL	DoDP			; Print DP
	CALL	Move1
	CALL	Swap2
	CALL	Move2
F_L2U:
	MOVLW	Unit3-0x2100	; mH
	GOTO	pmsg			; includes RETURN

;--------------------------------------------------

F_L3:
	MOVF	bcd+1,W
	ANDLW	0x0F
	BTFSC STATUS,Z
	GOTO	F_L4
	CALL	PutNyb
	CALL	Swap2
	CALL	Move2
	CALL	DoDP		; Print DP
	CALL	Swap3
	GOTO	F_L4U		; Print uH includes RETURN

;--------------------------------------------------

F_L4:
	SWAPF	bcd+2,W		; Digit1 == 0 ?
	ANDLW	0x0F
	BTFSS STATUS,Z
	GOTO	NoB1_L
	MOVLW	0x20		; yes, so show a space
	CALL	DATS
	GOTO	NoB2_L
NoB1_L:
	CALL	Swap2		; 1
NoB2_L:
	CALL	Move2		; 2
	CALL	DoDP		; Print DP
	CALL	Swap3		; 3
	CALL	Move3		; 4
F_L4U:
	MOVLW	Unit4-0x2100	; uH
	GOTO	pmsg		; includes RETURN

;--------------------------------------------------
;
;	Shared subroutines for formatted output
;
DoDP:
	MOVLW	"."			; Print DP
	GOTO	DATS		; Return from DATS

Swap0:
	SWAPF	bcd+0,W		; GET NEXT DIGIT
	GOTO	PutNyb		; DISPLAY IT

Move0:
	MOVF	bcd+0,W		; GET OTHER BCD DIGIT
	GOTO	PutNyb

Swap1:
	SWAPF	bcd+1,W
	GOTO	PutNyb

Move1:
	MOVF	bcd+1,W
	GOTO	PutNyb

Swap2:
	SWAPF	bcd+2,W
	GOTO	PutNyb

Move2:
	MOVF	bcd+2,W
	GOTO	PutNyb

Swap3:
	SWAPF	bcd+3,W
	GOTO	PutNyb

Move3:
	MOVF	bcd+3,W
	GOTO	PutNyb

;********************************************************************
;
;	Floating point stack operations
;
;	add		CALL	FPA24
;		GOTO	S_fix

subtract:
	CALL	FPS24	; in FP.TXT
	GOTO	S_fix

divide:
	CALL	FPD24	; in FP.TXT
	GOTO	S_fix

multiply:
	CALL	FPM24	; in FP.TXT (falls thru into S_fix)
;
;	Fix stack after add, subtract, divide & multiply
;	(also collects any floating point errors in FPE)

S_fix:
	IORWF	FPE,f				; W may hold error (0xff)
	copy	CARGB1,BARGB1		; C -> B
	copy	CARGB0,BARGB0
	copy	CEXP,BEXP
	RETURN
;
;	Push stack (duplicates TOS)
;
S_push:
	copy	BARGB1,CARGB1	; B -> C
	copy	BARGB0,CARGB0
	copy	BEXP,CEXP
	copy	AARGB1,BARGB1	; A -> B
	copy	AARGB0,BARGB0
	copy	AEXP,BEXP
	RETURN
;
;	Swap A and B
S_swap:
	swap	AARGB1,BARGB1	; A <-> B
	swap	AARGB0,BARGB0
	swap	AEXP,BEXP
	RETURN

;********************************************************************
;
;	Calculate unknown capacitance or inductance
;   Output is a 24 bit positive integer (scaled)
;	right justified in AARGB0, AARGB1, AARGB2
;	also as BCD in bcd:bcd+1:bcd+2:bcd+3
;
C_calc:
	CALL	divide
	CALL	Get_Ccal	; Times 10,000 ( = 1000.0pF)
	CALL	multiply
	GOTO	PorM		; includes RETURN

;--------------------------------------------------------------------
L_calc:
	CALL	multiply
	CALL	Get_Lscale	; Precomputed Scale_factor/(4*PI*PI)
	CALL	multiply
	CALL	Get_Ccal
	CALL	S_swap
	CALL	divide

L_divF1:
	CALL	Get_F1		; Divide by F1^2
	CALL	S_push
	CALL	multiply
	CALL	S_swap
	CALL	divide
;
;	Handle Space or Minus in front of FP number
;	

PorM:
	BTFSS	AARGB0,7	; test sign
	GOTO	Pplus
Pminus:
	MOVLW	0x2d		; minus
	GOTO	PMdisp
Pplus:
	MOVLW	0x20		; plus
PMdisp:
	MOVWF	R_sign		; save for later display
	BCF	AARGB0,7		; make plus anyway
;
;	Format as raw BCD string in bcd:bcd+1:bcd+2:bcd+3
;
	CALL	INT2424		; To INT in AARGB0 etc.
	IORWF	FPE,f		; W may hold Error (0xff)
	GOTO	B2_BCD		; includes RETURN

;********************************************************************
;
;	Calculate (F1/F3)^2-1, leave result on stack
;
F1_F3:
	CALL	Get_F3
	GOTO	F1_F1

;**************************************************************
;
;	Calculate (F1/F2)^2-1, leave result on stack
;
F1_F2:
	CALL	Get_F2
F1_F1:
	CALL	Get_F1
	CALL	divide		; F1/Fx
	CALL	S_push
	CALL	multiply	; (F1/Fx)^2
	CALL	Get_One
	CALL	S_swap
	GOTO	subtract	; (F1/Fx)^2-1 (includes RETURN)

;**************************************************************
;	Fetch assorted things used for the calculation
;	of unknown L and C
;
Get_Lscale:
	CALL	S_push		; make room first
	MOVLW	0xB8		; 2.53303e+17
	MOVWF	AEXP		; create FP version of
	MOVLW	0x60		; precomputed 1/(4*PI*PI)
	MOVWF	AARGB0		; times any needed
	MOVLW	0xFA		; fiddle factor (1/100)
	MOVWF	AARGB1
	RETURN

Get_One:
	CALL	S_push		; make room first
	CLRF	AEXP		; create a binary 1
	CLRF	AARGB0
	CLRF	AARGB1
	MOVLW	0x01
	GOTO	LSB2stak
	
Get_Ccal:
	MOVLW	cal_t		; get integer value
	GOTO	W2stak		; includes stack push

Get_F1:
	MOVLW	F1			; includes stack push
	GOTO	W2stak

Get_F2:
	MOVLW	F2			; includes stack push
	GOTO	W2stak

Get_F3:
	MOVLW	F3		; Includes stack push
					; (falls thru to W2stak)

;**************************************************************
;	Copy 16 bit number, pointed to by W, to stack
;	and convert to FP (positive value only)
;	via a 24 bit number in AARGB0,1,2

W2stak:
	MOVWF	FSR
	CALL	S_push		; make room first
	CLRF	AEXP
	CLRF	AARGB0
	MOVF	INDF,W		; most signif byte first
	MOVWF	AARGB1
	INCF	FSR,F		; then least signif byte
	MOVF	INDF,W
LSB2stak:
	MOVWF	AARGB2
	CALL	FLO2424		; turns 24 bit int -> 24 bit FP
	IORWF	FPE,f		; W may hold Error (0xff)
	RETURN

;**************************************************************
;	Routine to read cal data from EEPROM back into into "cal_t"
;	
EE_RD:
	bank1
	MOVLW	cal_p-0x2100	; load address to read
	MOVWF	EEADR			; and save in EEADR
	bank0
	CALL	EE_R			; now read back byte
	MOVWF	cal_t+0			; and save in cal_t
	CALL	EE_Rinc			; now read back byte from next adr
	MOVWF	cal_t+1			; and save in cal_t+1
	RETURN					; then depart

EE_Rinc:
	bank1
	INCF	EEADR,F		; bump address

EE_R:
	bank1
	BSF	EECON1,RD		; EE read
	MOVF	EEDATA,W	; W = EEDATA
	bank0
	RETURN

;*************************************************************
;	Routine to write cal data from "cal_t" into EEPROM
;
EE_WR:
	bank1
	MOVLW	cal_p-0x2100	; load EEPROM address to write
	MOVWF	EEADR			; and save in EEADR
	MOVF	cal_t+0,W		; then fetch data byte #0
	CALL	EE_W			; and write to EEPROM
	MOVF	cal_t+1,W		; then fetch data byte #1
	CALL	EE_Winc			; and write to EEPROM
	bank0
	RETURN					; then depart

	errorlevel	-302	; In Bank 2

EE_Winc	INCF	EEADR,F		; bump address
	
EE_W:
	MOVWF	EEDATA
	BSF	EECON1,WREN		; Enable Write
	MOVLW	0x55		;
	MOVWF	EECON2		; Write 0x55
	MOVLW	0xAA		;
	MOVWF	EECON2		; Write 0xAA
	BSF	EECON1,WR		; Set WR bit (begin write)

EE_W2:
	BTFSC	EECON1,WR	; Wait for write to finish
	GOTO	EE_W2		; loop back until it has
	bank0
	BCF	PIR1,EEIF		; clear interrupts
	bank1
	RETURN	

	errorlevel	+302

;	**********************************************************
;	include floating point routines
;	(in FP.TXT, extracted from FP24.A16)
;
	#include 	<FP.TXT>

;*************************************************************	
;
;	Display text strings & cal constant (stored in data EEPROM)
;
        ORG 0x2100

ovr		de	"   Over Range   ",0
Unit1	de	" nF",0
Unit2	de	" pF",0
Unit3	de	" mH",0
Unit4	de	" uH",0
Cintro	de	" C = ",0
Lintro	de	" L = ",0
Calibr  de	"   Calibrating  ",0

cal_p	de	0x27,0x10		; Initial value = 10000

 	END

