;	MusicTune.asm -- Control program for Silicon Chip Music Tuning Aid,
;	based on PIC16F877A micro running at 16.00MHz (so Fosc/4 = 4MHz)
;	Written by Jim Rowe. Last revised 5/05/2008, 11:15am
;	Note that one machine instruction cycle (1mc) = 250ns

	list P=16F877A
	#include "p16f877A.inc"
	__config h'3F3A'	; HS osc selected, WDT disabled, PWRT enabled,
						; BOR disabled, RB3 is digital I/O, code prot off,
						; Flash prog mem write protection off, RB6 & RB7 are
						; general purpose I/O pins (not used for debugging)

;  storage for counters & other variables used by program, in data memory
Counter1	equ	h'20'	; gp counter variable 1
Counter2	equ h'21'	; gp counter variable 2
Counter3	equ h'22'	; gp counter variable 3
Counter4	equ h'23'	; counter for strobe clock timing 

Octave		equ h'24'	; code for current octave setting
						; b0 = 1 (01h) for octave 1,
						; b1 = 1 (02h) for octave 2,
						; b2 = 1 (04h) for octave 3,
						; b3 = 1 (08h) for octave 4,
						; b4 = 1 (10h) for octave 5,
						; b5 = 1 (20h) for octave 6,
						; b6 = 1 (40h) for octave 7,
						; b7 = 1 (80h) for octave 8

CurrNote	equ h'25'	; code for current note setting
						; 01 = C, 02 = C#, 03 = D, 04 = D#,
						; 05 = E, 06 = F, 07 = F#, 08 = G,
						; 09 = G#, 0A = A, 0B = A#, 0C = B,
						; 0D = C'

HNoteTime	equ h'26'	; H byte of curr note period time
LNoteTime	equ h'27'	; and L byte (for loading into Tmr1)

WaveCtr		equ h'28'	; output wave stepping counter
						; (which determines output octave)
CurrWavePtr	equ h'29'	; and current output wave pointer
StrobeCtr	equ h'2A'	; and strobe pulse clock counter
						; (values change for various octaves:
						; 20h for octave 1, 10h for octave 2,
						; 08h for octave 3, 04h for octave 4,
						; 02h for octave 5, 01h for octaves 6-7-8

; 	regs for saving context during interrupt servicing
WSave	equ		h'70'	; w reg stored here
SSave	equ		h'71'	; status reg stored here
PSave	equ 	h'72'	; PCLATH reg stored here

; --------------------------------------------------------------------------------

; now program begins

	ORG		h'00'		; normal start vector
	GOTO Initialise
	ORG		h'04'		; interrupt service vector
	GOTO Intserv		; 

Initialise:
; first we initialise the PIC itself
	BCF STATUS, RP0		; set for Bank0
	BCF STATUS, RP1
	BCF STATUS, IRP		; also set indir addressing for Banks0&1
	CLRF PORTA			; clear PortA
	CLRF PORTB			; and PortB
	CLRF PORTC			; and PortC
	CLRF PORTD			; and PortD
	CLRF PORTE			; and PortE
	CLRF INTCON			; disable all interrupts also
	CLRF RCSTA			; also clear RCSTA register to disable UART
	CLRF T1CON			; and clear T1 timer control register
	CLRF TMR1H			; also its H & L registers
	CLRF TMR1L 
	BCF ADCON0,0		; and shut down A/D module
	BSF STATUS, RP0		; now set for Bank1, so we can...
	MOVLW h'07'			; disable the comparators
	MOVWF CMCON
	MOVLW h'06'			; also the analog inputs, so RA0-3 can
	MOVWF ADCON1		; work as digital I/O pins
	CLRF CVRCON			; and also the comp voltage reference
	CLRF PIE1			; clear all interrupt mask bits for clean start
	BSF PIE1,0			; then set TMR1 overflow interrupt enable bit
	CLRF TRISA			; set RA0-RA5 as outputs by clearing its config reg
	CLRF TRISB			; do the same for RB0-RB7
	MOVLW h'0F'			; also set RC0 - RC3 as inputs, others as outputs
	MOVWF TRISC			; by loading its config register
	CLRF TRISD			; then set RD0 - RD7 as outputs also
	CLRF TRISE			; also disable PSP mode & set RE0-2 as outputs
	BCF STATUS, RP0		; now back to Bank 0
	CLRF PIR1			; where we clear all periph int flags

;  	next we set up for default setting of A = 440Hz, in octave 4
;	(Octave = 08h, CurrNote = 0Ah)		
	MOVLW h'08'
	MOVWF Octave
	MOVLW h'0A'
	MOVWF CurrNote
	CALL ClearLEDs		; now turn off all note & octave LEDs
	CALL SetWaveCtr		; now set output wave counter for octave
						; (also turns on current octave LED)
	CALL SetTmr1HnL		; also set TMR1 H & L values for the
						; current note (also turns on note LED)
	CLRF CurrWavePtr	; then clear output wave current pointer
	CALL LoadTmr1		; load Tmr1 H&L regs from current value,
						; and set it going... 
	BSF INTCON, PEIE	; then enable peripheral int's (for Tmr1)
	BSF INTCON, GIE		; & finally set GIE to get going in earnest
	
LoopStart:
;	now we begin main loop by scanning S1 - S4, in case one pressed
	BTFSS PORTC,0		; check if S2 is pressed, skip if not
	GOTO NoteDown		; if it was, go to drop a semitone
	BTFSS PORTC,1		; check if S1 is pressed, skip if not
	GOTO NoteUp			; if it was, go to raise a semitone
	BTFSS PORTC,2 		; check if S4 is pressed, skip if not
	GOTO OctaveDown		; if it was, go to drop an octave
	BTFSS PORTC,3		; check if S3 is pressed, skip if not
	GOTO OctaveUp		; if it was, go to raise an octave
	GOTO LoopStart		; no buttons pressed, so just loop back

NoteDown:
;	routine to drop current note a semitone
	DECFSZ CurrNote,1	; decrement CurrNote, skip if -> 0
	GOTO $+2			; then continue
	INCF CurrNote,1		; it did hit 0, so make it 01 again
	CALL ClearLEDs		; turn off all indicator LEDs
	CALL SetTmr1HnL		; now reset timing for new note,
						; also turn on its LED
	CALL LoadTmr1		; and reload Tmr1 regs for it
	CALL SetWaveCtr		; also reset wave counter & octave LED
	GOTO PausenBack		; then pause and loop back

NoteUp:
;	routine to raise current note a semitone
	INCF CurrNote,F		; increment current note
	MOVLW h'0E'			; now make sure it didn't -> 0E
	XORWF CurrNote,0	; Z will now = 1 if it did
	BTFSC STATUS,Z		; so skip if Z=0
	DECF CurrNote,F		; otherwise make it 0Dh again
	CALL ClearLEDs		; turn off all indicator LEDs
	CALL SetTmr1HnL		; now reset timing params for new note,
						; turn on its LED also
	CALL LoadTmr1		; and reload Tmr1 regs for it
	CALL SetWaveCtr		; also reset wave counter & octave LED
	GOTO PausenBack		; then pause and loop back

OctaveDown:
;	routine to drop an octave (unless Octave = 1 already)
	BCF STATUS,0		; first clear the carry bit
	RRF Octave, F		; then rotate Octave right by 1
	BTFSS STATUS,C		; now check if we went too far
	GOTO $+2			; otherwise just continue
	BSF Octave,0		; C=1, so we overshot. Reset to 01h
	CALL ClearLEDs		; turn off all indicator LEDs
	CALL SetTmr1HnL		; now reset timing params for current
						; note, turn its LED on again
	CALL LoadTmr1		; and reload Tmr1 regs for it
	CALL SetWaveCtr		; also reset wave counter and
						; octave indicator LED
	GOTO PausenBack		; then pause and loop back

OctaveUp:
;	routine to move up an octave (unless Octave = 80h already)
	BCF STATUS,0		; first clear the carry bit
	RLF Octave, F		; then rotate Octave left by 1
	BTFSS STATUS,C		; now check if we went too far
	GOTO $+2			; otherwise just continue
	BSF Octave,7		; C=1, so we overshot. Reset to 80h
	CALL ClearLEDs		; turn off all indicator LEDs
	CALL SetTmr1HnL		; now reset timing params for current
						; note, turn its LED on again
	CALL LoadTmr1		; and reload Tmr1 regs for it
	CALL SetWaveCtr		; also reset wave counter and
						; octave indicator LED

PausenBack:
	CALL Pause200ms		; then pause for about 200ms
	GOTO LoopStart		; then back to start of loop
	
;	end of main loop -- subroutines follow

;  ---------------------------------------------------------------------

ClearLEDs:
;	Subroutine to turn off all note & octave indicator LEDs
	CLRF PORTB
	CLRF PORTD
	BCF PORTC,4
	BCF PORTC,5
	BCF PORTC,6
	BCF PORTC,7
	BCF PORTE,1
	RETURN

Delay250us:
;   routine to delay approx 250us before return
;	delay is 4mc for entry & setup, 4mc for exit & return
;	plus 4mc for each loop. We use 248 loops, so total delay
;	is (4 x 248)mc + 8mc = 1000mc = 250us
	MOVLW h'F8'			; set Counter1 for 248 loops
	MOVWF Counter1
	NOP					; 1mc delay for timing
	DECFSZ Counter1, F	; decrement Ctr1 & skip when zero
	GOTO $-2			; didn't hit zero, so loop back
	RETURN				; reached zero, so return

Delay10ms:
;	routine to delay approx 10ms before returning
	MOVLW h'28'			; set Counter2 for 40d outer loops
	MOVWF Counter2
	CALL Delay250us		; then wait 250us
	DECFSZ Counter2, F	; then decrement Counter2 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return

LoadTmr1:
;	routine to load Timer1 registers from HNoteTime, LNoteTime
;	to begin a new note timing period
	BCF T1CON, 0		; first stop Tmr1, to avoid contention
	MOVF HNoteTime, W	; then fetch high byte into W
	MOVWF TMR1H			; and save in TMR1H
	MOVF LNoteTime, W	; now do likewise with low byte
	MOVWF TMR1L
	BSF T1CON, 0		; then restart Tmr1 again
	RETURN				; and leave	
	
Pause200ms:
;	routine to delay approx 200ms before returning
	MOVLW h'14'			; set Counter3 for 20 outer loops
	MOVWF Counter3
	CALL Delay10ms		; then wait 10ms
	DECFSZ Counter3, F	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return

SetTmr1HnL:
;	Subroutine to save Timer1 H & L bytes for current note,
;	also turn on appropriate note indicator LED
	MOVLW h'01'			; first check if CurrNote = 01h (= C)
	XORWF CurrNote,W	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for C
	MOVWF HNoteTime		; (Tmr1 preset value = FE44h)
	MOVLW h'44'
	MOVWF LNoteTime
	BSF PORTC,4			; then turn on note C LED
	RETURN				; and leave

	MOVLW h'02'			; next check if CurrNote = 02h (= C#)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for C#
	MOVWF HNoteTime		; (Tmr1 preset value = FE5Fh)
	MOVLW h'5F'
	MOVWF LNoteTime
	BSF PORTC,5			; then turn on note C# LED
	RETURN				; and leave

	MOVLW h'03'			; next check if CurrNote = 03h (= D)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for D
	MOVWF HNoteTime		; (Tmr1 preset value = FE78h)
	MOVLW h'78'
	MOVWF LNoteTime
	BSF PORTC,6			; then turn on note D LED
	RETURN				; and leave

	MOVLW h'04'			; next check if CurrNote = 04h (= D#)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for D#
	MOVWF HNoteTime		; (Tmr1 preset value = FE90h)
	MOVLW h'90'
	MOVWF LNoteTime
	BSF PORTC,7			; then turn on note D# LED
	RETURN				; and leave

	MOVLW h'05'			; next check if CurrNote = 05h (= E)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for E
	MOVWF HNoteTime		; (Tmr1 preset value = FEA7h)
	MOVLW h'A7'
	MOVWF LNoteTime
	BSF PORTD,4			; then turn on note E LED
	RETURN				; and leave

	MOVLW h'06'			; next check if CurrNote = 06h (= F)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for F
	MOVWF HNoteTime		; (Tmr1 preset value = FEBCh)
	MOVLW h'BC'
	MOVWF LNoteTime
	BSF PORTD,5			; then turn on note F LED
	RETURN				; and leave

	MOVLW h'07'			; next check if CurrNote = 07h (= F#)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for F#
	MOVWF HNoteTime		; (Tmr1 preset value = FED0h)
	MOVLW h'D0'
	MOVWF LNoteTime
	BSF PORTD,6			; then turn on note F# LED
	RETURN				; and leave

	MOVLW h'08'			; next check if CurrNote = 08h (= G)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for G
	MOVWF HNoteTime		; (Tmr1 preset value = FEE3h)
	MOVLW h'E3'
	MOVWF LNoteTime
	BSF PORTD,7			; then turn on note G LED
	RETURN				; and leave

	MOVLW h'09'			; next check if CurrNote = 09h (= G#)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FE'			; did match, so set Tmr1H&L for G#
	MOVWF HNoteTime		; (Tmr1 preset value = FEF5h)
	MOVLW h'F5'
	MOVWF LNoteTime
	BSF PORTB,0			; then turn on note G# LED
	RETURN				; and leave

	MOVLW h'0A'			; next check if CurrNote = 0Ah (= A)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FF'			; did match, so set Tmr1H&L for A
	MOVWF HNoteTime		; (Tmr1 preset value = FF06h)
	MOVLW h'06'
	MOVWF LNoteTime
	BSF PORTB,1			; then turn on note A LED
	RETURN				; and leave

	MOVLW h'0B'			; next check if CurrNote = 0Bh (= A#)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FF'			; did match, so set Tmr1H&L for A#
	MOVWF HNoteTime		; (Tmr1 preset value = FF16h)
	MOVLW h'16'
	MOVWF LNoteTime
	BSF PORTB,2			; then turn on note A# LED
	RETURN				; and leave

	MOVLW h'0C'			; next check if CurrNote = 0Ch (= B)
	XORWF CurrNote,0	; if it is, W = 0 and Z = 1
	BTFSS STATUS,Z		; skip if Z = 1, because it did match
	GOTO $+7			; otherwise keep looking
	MOVLW h'FF'			; did match, so set Tmr1H&L for B
	MOVWF HNoteTime		; (Tmr1 preset value = FF25h)
	MOVLW h'25'
	MOVWF LNoteTime
	BSF PORTB,3			; then turn on note B LED
	RETURN				; and leave

	MOVLW h'FF'			; must be C', so set Tmr1H&L for C'
	MOVWF HNoteTime		; (Tmr1 preset value = FF33h)
	MOVLW h'33'
	MOVWF LNoteTime
	BSF PORTB,4			; then turn on note C' LED
	RETURN				; and leave

SetWaveCtr:
;	Subroutine to set wave counter for current octave,
;	set StrobeCtr value for clear strobing and
;	also turn on appropriate octave indicator LED
	BTFSS Octave,0		; now check bit0 of Octave, skip if set
	GOTO $+7			; not set, so keep looking
	MOVLW h'01'			; was set, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 1
	MOVLW h'20'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 1
	BSF PORTD,0			; then turn on O1 LED
	RETURN				; and depart

	BTFSS Octave,1		; now check bit1 of Octave
	GOTO $+7			; not set, so keep looking
	MOVLW h'02'			; was set, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 2
	MOVLW h'10'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 2	
	BSF PORTD,1			; then turn on O2 LED
	RETURN				; and depart

	BTFSS Octave,2		; now check bit2 of Octave
	GOTO $+7			; not set, so keep looking
	MOVLW h'04'			; was set, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 3
	MOVLW h'08'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 3	
	BSF PORTD,2			; then turn on O3 LED
	RETURN				; and depart

	BTFSS Octave,3		; now check bit3 of Octave
	GOTO $+7			; not set, so keep looking
	MOVLW h'08'			; was set, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 4
	MOVLW h'04'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 4	
	BSF PORTD,3			; then turn on O4 LED
	RETURN				; and depart

	BTFSS Octave,4		; now check bit4 of Octave
	GOTO $+7			; not set, so keep looking
	MOVLW h'10'			; was set, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 5
	MOVLW h'02'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 5	
	BSF PORTB,5			; then turn on O5 LED
	RETURN				; and depart

	BTFSS Octave,5		; now check bit5 of Octave
	GOTO $+7			; not set, so keep looking
	MOVLW h'20'			; was set, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 6
	MOVLW h'01'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 6	
	BSF PORTB,6			; then turn on O6 LED
	RETURN				; and depart

	BTFSS Octave,6		; now check bit6 of Octave
	GOTO $+7			; not set, so keep looking
	MOVLW h'40'			; was set, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 7
	MOVLW h'01'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 7	
	BSF PORTB,7			; then turn on O7 LED
	RETURN				; and depart

	MOVLW h'80'			; must be O8, so load WaveCtr with value
	MOVWF WaveCtr		; for octave 8
	MOVLW h'01'			; also set StrobeCtr
	MOVWF StrobeCtr		; for octave 8	
	BSF PORTE,1			; then turn on O8 LED
	RETURN				; and depart

;	-----------------------------------------------------------

Intserv:
;	routine to service interrupts from Timer 1, which will occur
;	at the end of each 'top octave' note timing period

	MOVWF WSave			; first save context - w first, then
	SWAPF STATUS,W		; status (using SWAPF to avoid change)
	MOVWF SSave
	MOVF PCLATH, W		; then save PCLATH, just in case
	MOVWF PSave

	BCF T1CON, 0		; first stop Tmr1
	BCF PIR1,0			; next clear Tmr1 o/f interrupt flag
						; ready for next note time period...
	MOVF WaveCtr,W		; now fetch WaveCtr into W
	ADDWF CurrWavePtr,W	; add it to CurrWavePtr
	MOVWF CurrWavePtr	; and resave CurrWavePtr
	MOVLW h'04'			; now preload PCLATH with MSB of
	MOVWF PCLATH		; first data item in CosTable (0400h)
	MOVF CurrWavePtr,W	; then fetch CurrWavePtr in w as offset	
	CALL CosTable		; now go get output wfm sample value
	MOVWF PORTA			; and send it to PortA
	CALL LoadTmr1		; now reload Tmr1 for current note
						; (from HNoteTime, LNoteTime)
						; and restart it again (33mc after int)
 	DECFSZ Counter4,F	; now decrement Counter 4, skip if zero
	GOTO Restcon		; otherwise just restore context & go
	BSF PORTE,0			; did hit 0, so send clock strobe pulse
	NOP					; pausing to ensure clock pulse width
	BCF PORTE,0			; then reset RE0 again
	MOVF StrobeCtr,W	; now reset Counter 4 from StrobeCtr
	MOVWF Counter4

Restcon:
	MOVF PSave,W		; fetch saved PCLATH contents in w
	MOVWF PCLATH		; and restore them
	SWAPF SSave,W		; then restore status & w regs
	MOVWF STATUS		;
	SWAPF WSave,F		; using SWAPFs to avoid changing STATUS
	SWAPF WSave,W		;
	RETFIE				; & return with TOS->PC, 1->GIE
	 	
;	-----------------------------------------------------------	
	ORG h'03FF'
CosTable:
					; start of lookup table for cosine shaping
					; Note: 256 samples, so each sample is for
					; an incremental angle of 360/256 = 1.40625
					; degrees. Sample values are 5-bit equiv's
					; of 16*(1 - cosA), where A = N * 1.40625
					; and N varies from 0 to 255. Note also that
					; sample's sixth bit mirrors fifth bit,
					; since we use RA5 for msb output, not RA4
	ADDWF PCL, F	; add offset (in w) to PC, for a computed GOTO
	RETLW h'00'		; first 5-bit sample value (#0, in 400h)
	RETLW h'00'		; next value 
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value

	RETLW h'00'		; next value (#8)
	RETLW h'00'		; next value 
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'01'		; next value

	RETLW h'01'		; next value (#16)
	RETLW h'01'		; next value
	RETLW h'01'		; next value
	RETLW h'01'		; next value
	RETLW h'01'		; next value
	RETLW h'02'		; next value
	RETLW h'02'		; next value
	RETLW h'02'		; next value

	RETLW h'02'		; next value (#24)
	RETLW h'02'		; next value
	RETLW h'03'		; next value
	RETLW h'03'		; next value
	RETLW h'03'		; next value
	RETLW h'03'		; next value
	RETLW h'04'		; next value
	RETLW h'04'		; next value

	RETLW h'04'		; next value (#32)
	RETLW h'04'		; next value
	RETLW h'05'		; next value
	RETLW h'05'		; next value
	RETLW h'05'		; next value
	RETLW h'06'		; next value
	RETLW h'06'		; next value
	RETLW h'06'		; next value

	RETLW h'07'		; next value (#40)
	RETLW h'07'		; next value
	RETLW h'07'		; next value
	RETLW h'08'		; next value
	RETLW h'08'		; next value
	RETLW h'08'		; next value
	RETLW h'09'		; next value
	RETLW h'09'		; next value

	RETLW h'09'		; next value (#48)
	RETLW h'0A'		; next value
	RETLW h'0A'		; next value
	RETLW h'0B'		; next value
	RETLW h'0B'		; next value
	RETLW h'0B'		; next value
	RETLW h'0C'		; next value
	RETLW h'0C'		; next value

	RETLW h'0C'		; next value (#56)
	RETLW h'0D'		; next value
	RETLW h'0D'		; next value
	RETLW h'0E'		; next value
	RETLW h'0E'		; next value
	RETLW h'0E'		; next value
	RETLW h'0F'		; next value
	RETLW h'0F'		; next value

	RETLW h'30'		; next value (#64) (note 10 -> 30 for
	RETLW h'30'		; next value       (bit mirroring)
	RETLW h'30'		; next value
	RETLW h'31'		; next value
	RETLW h'31'		; next value
	RETLW h'32'		; next value
	RETLW h'32'		; next value
	RETLW h'32'		; next value

	RETLW h'33'		; next value (#72)
	RETLW h'33'		; next value
	RETLW h'33'		; next value
	RETLW h'34'		; next value
	RETLW h'34'		; next value
	RETLW h'35'		; next value
	RETLW h'35'		; next value
	RETLW h'35'		; next value

	RETLW h'36'		; next value (#80)
	RETLW h'36'		; next value
	RETLW h'36'		; next value
	RETLW h'37'		; next value
	RETLW h'37'		; next value
	RETLW h'37'		; next value
	RETLW h'38'		; next value
	RETLW h'38'		; next value

	RETLW h'38'		; next value (#88)
	RETLW h'39'		; next value
	RETLW h'39'		; next value
	RETLW h'39'		; next value
	RETLW h'3A'		; next value
	RETLW h'3A'		; next value
	RETLW h'3A'		; next value
	RETLW h'3B'		; next value

	RETLW h'3B'		; next value (#96)
	RETLW h'3B'		; next value
	RETLW h'3B'		; next value
	RETLW h'3C'		; next value
	RETLW h'3C'		; next value
	RETLW h'3C'		; next value
	RETLW h'3C'		; next value
	RETLW h'3D'		; next value

	RETLW h'3D'		; next value (#104)
	RETLW h'3D'		; next value
	RETLW h'3D'		; next value
	RETLW h'3D'		; next value
	RETLW h'3E'		; next value
	RETLW h'3E'		; next value
	RETLW h'3E'		; next value
	RETLW h'3E'		; next value

	RETLW h'3E'		; next value (#112)
	RETLW h'3E'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value

	RETLW h'3F'		; next value (#120)
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value

	RETLW h'3F'		; next value (#128)
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value

	RETLW h'3F'		; next value (#136)
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3F'		; next value
	RETLW h'3E'		; next value
	RETLW h'3E'		; next value
 
	RETLW h'3E'		; next value (#144)
	RETLW h'3E'		; next value
	RETLW h'3E'		; next value
	RETLW h'3E'		; next value
	RETLW h'3D'		; next value
	RETLW h'3D'		; next value
	RETLW h'3D'		; next value
	RETLW h'3D'		; next value

	RETLW h'3D'		; next value (#152)
	RETLW h'3C'		; next value
	RETLW h'3C'		; next value
	RETLW h'3C'		; next value
	RETLW h'3C'		; next value
	RETLW h'3B'		; next value
	RETLW h'3B'		; next value
	RETLW h'3B'		; next value

	RETLW h'3B'		; next value (#160)
	RETLW h'3A'		; next value
	RETLW h'3A'		; next value
	RETLW h'3A'		; next value
	RETLW h'39'		; next value
	RETLW h'39'		; next value
	RETLW h'39'		; next value
	RETLW h'38'		; next value

	RETLW h'38'		; next value (#168)
	RETLW h'38'		; next value
	RETLW h'37'		; next value
	RETLW h'37'		; next value
	RETLW h'37'		; next value
	RETLW h'36'		; next value
	RETLW h'36'		; next value
	RETLW h'36'		; next value

	RETLW h'35'		; next value (#176)
	RETLW h'35'		; next value
	RETLW h'35'		; next value
	RETLW h'34'		; next value
	RETLW h'34'		; next value
	RETLW h'33'		; next value
	RETLW h'33'		; next value
	RETLW h'33'		; next value

	RETLW h'32'		; next value (#184)
	RETLW h'32'		; next value
	RETLW h'32'		; next value
	RETLW h'31'		; next value
	RETLW h'31'		; next value
	RETLW h'30'		; next value
	RETLW h'30'		; next value
	RETLW h'30'		; next value

	RETLW h'0F'		; next value (#192)
	RETLW h'0F'		; next value
	RETLW h'0E'		; next value
	RETLW h'0E'		; next value
	RETLW h'0E'		; next value
	RETLW h'0D'		; next value
	RETLW h'0D'		; next value
	RETLW h'0C'		; next value

	RETLW h'0C'		; next value (#200)
	RETLW h'0C'		; next value
	RETLW h'0B'		; next value
	RETLW h'0B'		; next value
	RETLW h'0B'		; next value
	RETLW h'0A'		; next value
	RETLW h'0A'		; next value
	RETLW h'09'		; next value

	RETLW h'09'		; next value (#208)
	RETLW h'09'		; next value
	RETLW h'08'		; next value
	RETLW h'08'		; next value
	RETLW h'08'		; next value
	RETLW h'07'		; next value
	RETLW h'07'		; next value
	RETLW h'07'		; next value

	RETLW h'06'		; next value (#216)
	RETLW h'06'		; next value
	RETLW h'06'		; next value
	RETLW h'05'		; next value
	RETLW h'05'		; next value
	RETLW h'05'		; next value
	RETLW h'04'		; next value
	RETLW h'04'		; next value

	RETLW h'04'		; next value (#224)
	RETLW h'04'		; next value
	RETLW h'03'		; next value
	RETLW h'03'		; next value
	RETLW h'03'		; next value
	RETLW h'03'		; next value
	RETLW h'02'		; next value
	RETLW h'02'		; next value

	RETLW h'02'		; next value (#232)
	RETLW h'02'		; next value
	RETLW h'02'		; next value
	RETLW h'01'		; next value
	RETLW h'01'		; next value
	RETLW h'01'		; next value
	RETLW h'01'		; next value
	RETLW h'01'		; next value

	RETLW h'01'		; next value (#240)
	RETLW h'00'		; next value 
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value

	RETLW h'00'		; next value (#248)
	RETLW h'00'		; next value 
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; next value
	RETLW h'00'		; last value (#255, in 4FFh)

; ----------------------------------------------------------------	
	
	END
