; LED Strobe


	list P=16F88
	#include p16f88.inc
	ERRORLEVEL -302
	ERRORLEVEL -306

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _HS_OSC

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; Define variables at memory locations

EEPROM1		equ	H'00'	; non-volatile storage for division ratio (DIV_SET) 
EEPROM2		equ	H'01'	; non-volatile storage for flash on period (FLASH_ON)
EEPROM3		equ	H'02'	; non-volatile storage for MODE
EEPROM4		equ	H'03'	; non volatile storage for option register (edge input + or -)
EEPROM5		equ	H'04'	; non volatile storage for rpm value ms byte
EEPROM6		equ	H'05'	; non volatile storage for rpm value ls byte (65536rpm max)
EEPROM7		equ	H'06'	; non volatile storage for flash mode (auto/fixed bit 7) and auto % value (bits 0,1,2,3)
EEPROM8		equ	H'07'	; non volatile storage for averaging number (trigger mode only)
	
; Bank 0 RAM
; A/D value
PCLATH_STO	equ	H'20'	; store PCLATH
AUTO		equ	H'21'	; automatic flash period
MODE		equ	H'22'	; mode

D_STO		equ	H'23'	; data store for LCD drive
PORTA_STO	equ	H'24'	; data store for LCD drive
DIV_SET		equ	H'25'	; division ratio
FLASH_ON	equ	H'26'	; flash on period	

NUMER_0		equ	H'27'	; rpm numerator value ms byte
NUMER_1		equ	H'28'	; rpm numerator value mid byte
NUMER_2		equ	H'29'	; rpm numerator value mid byte
NUMER_3		equ	H'2A'	; rpm numerator value ls byte

NUMER_4		equ	H'2B'	; frequency numerator value ms byte
NUMER_5		equ	H'2C'	; frequency numerator value mid byte
NUMER_6		equ	H'2D'	; frequency numerator value mid byte
NUMER_7		equ	H'2E'	; frequency numerator value ls byte

NIL_SIG		equ	H'2F'	; no signal flag

COUNTER0	equ	H'30'	; overflow counter
COUNTER1	equ	H'31'	; overflow counter
COUNT0		equ	H'32'	; 32 bit rpm counter
COUNT1		equ	H'33'	; 32 bit rpm counter
COUNT2		equ	H'34'	; 32 bit rpm counter
COUNT3		equ	H'35'	; 32 bit rpm counter

STORE1		equ	H'36'	; delay counter
STORE2		equ	H'37'	; delay counter
STORE3		equ	H'38'	; delay counter
TOTAL0		equ	H'39'	; totals for capture counts
TOTAL1		equ	H'3A'	; totals for capture counts
TOTAL2		equ	H'3B'	; totals for capture counts
RPM_FLG		equ	H'3C'	; flag for rpm decimal point
DIV_SET_CNT	equ	H'3D'	; counter for flashing LED on 1st/second,third etc count
DIV_SET_POS	equ	H'3E'	; counter to match div_set_cnt so the firing count can be altered

COUNTX0		equ	H'3F'	; 32 bit counter
COUNTX1		equ	H'40'	; 32 bit counter
COUNTX2		equ	H'41'	; 32 bit counter
COUNTX3		equ	H'42'	; 32 bit counter

RPM0		equ	H'43'	; rpm value (generator mode) ms byte
RPM1		equ	H'44'	; rpm value (generator mode) ls byte
COUNT_REQ0	equ	H'45'	; 32 bit required rpm counter (generator mode)
COUNT_REQ1	equ	H'46'	; 32 bit required rpm counter
COUNT_REQ2	equ	H'47'	; 32 bit required rpm counter
COUNT_REQ3	equ	H'48'	; 32 bit required rpm counter
TEMPOR0		equ	H'49'	; temp value ms byte
TEMPOR1		equ	H'4A'	; temp value ls byte
GEN_FLG		equ	H'4B'	; flag to update display in GEN mode when rpm changed with a switch and include delay
LAST_REQ0	equ	H'4C'	; store new count_required value ms byte
LAST_REQ1	equ	H'4D'	; store new count_required value 
LAST_REQ2	equ	H'4E'	; store new count_required value 
LAST_REQ3	equ	H'4F'	; store new count_required value ls byte
COMPARE_FG	equ	H'50'	; compare interrupt flag
FLASHCON	equ H'51'	; flash Auto/fixed

; math routines

OUT0		equ	H'52'	; output value
OUT1		equ	H'53'	; output value
OUT2		equ	H'54'	; output value
OUT3		equ	H'55'	; output value
OUT4		equ	H'56'	; output value
OUT5		equ	H'57'	; output value
OUT6		equ	H'58'	; output value
OUT7		equ	H'59'	; output value
BIN_0		equ	H'5A'	; binary value ls
BIN_1		equ	H'5B'	; binary value 
BIN_2		equ	H'5C'	; binary value ms
BCD_0		equ	H'5D'	; BCD MS
BCD_1		equ	H'5E'	; display value MS
BCD_2		equ	H'5F'	; display value LS
BCD_3		equ	H'60'	; display value LS
CNT_24		equ	H'61'	; counter for BCD conversion

TEMP		equ H'62'
REMB3		equ H'63'
REMB2		equ	H'64'
REMB1      	equ H'65'
REMB0		equ	H'66'
AARGB3		equ	H'67'
AARGB2      equ H'68'
AARGB1      equ H'69'
AARGB0      equ H'6A'	; most significant byte of argument A
BARGB3      equ H'6B'
BARGB2      equ H'6C'
BARGB1      equ H'6D'
BARGB0      equ H'6E'	; most significant byte of argument B
LOOPCOUNT   equ H'6F'  	; division counter

; All Banks RAM

W_TMP		equ	H'70'	; storage of w before interrupt
STATUS_TMP	equ	H'71'	; status storage before interrupt
OPTIONS		equ	H'72'	; initial options store
COUNTREQ0	equ	H'73'	; 32 bit required rpm counter (generator mode)
COUNTREQ1	equ	H'74'	; 32 bit required rpm counter
COUNTREQ2	equ	H'75'	; 32 bit required rpm counter
COUNTREQ3	equ	H'76'	; 32 bit required rpm counter
AVERAG		equ	H'77'	; averaging
EDGE_FG		equ	H'78'	; edge flag
AV_CNT		equ	H'79'	; averaging counter
TALLY0		equ	H'7A'	; average tally ms byte
TALLY1		equ	H'7B'	; average tally 
TALLY2		equ	H'7C'	; average tally 
TALLY3		equ	H'7D'	; average tally ls byte
UPDATE1		equ	H'7E'	; display update counter
UPDATE2		equ	H'7F'	; display update counter

; preprogram EEPROM DATA 
	
 	ORG     H'2100'
 	DE		D'01'		; DIV_SET, EEPROM1
	DE 		D'50'		; FLASH_ON, EEPROM2 
	DE 		H'80'		; MODE, EEPROM3
	DE		B'01000110'	; OPTIONS, (edge trigger) EEPROM4 (set bit 6 for rising edge)
	DE		H'03', H'E8'; 1000rpm. extra 0-100rpm (selected on potentiometer) 
	DE		H'05'		; flash auto 5% (set bit 7 for fixed) 
	DE		H'02'		; averaging

; start at memory 0
	org	0	; reset vector
	goto	SETUP
	org 4	; interrupt vector
; start interrupt save w and status register
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp 
	bcf		STATUS,RP0	; select bank 0
	bcf		STATUS,RP1	; bank 0
	movf	PCLATH,w
	movwf	PCLATH_STO	; keep PCLATH
	bsf		PCLATH,3	; page 1
	goto	INTERRUPT	; interrupt at page 1 address

; ******************************************************************************************************
; capture/compare LED firing
	org		H'800'		; start at page 1

INTERRUPT
	
	movlw	D'14'
	movwf	TMR2		; compensation of timer1 counter for compare counter 
; mode
	btfsc	MODE,7		; if set generator mode
	goto	GEN_COMPARE

; check edge interrupt
	btfss	INTCON,INTF	; if set edge
	goto	CHECK_CAPT	; check capture
	bsf		EDGE_FG,0	; set edge flag to show on display
	bcf		EDGE_FG,1	; clear display zero flag when edge reappears 
	bcf		INTCON,INTF
; if DIV_SET is 0 or 1 then fire on every edge interrupt
	movf	DIV_SET,w
	btfsc	STATUS,Z	; if zero fire LED
	goto	FIRE_LED
	decf	DIV_SET,w	; reduce by 1
	btfsc	STATUS,Z	; if was a 1 fire LED
	goto	FIRE_LED	
; decrease counter
	movf	DIV_SET_CNT,w	; if counter is zero load DIV_SET
	btfsc	STATUS,Z		
	goto	TRANS		; load division
	decf	DIV_SET_CNT,w	; decrease and if zero load DIV_SET
	btfsc	STATUS,Z	
	goto	TRANS
	decf	DIV_SET_CNT,f	; decrease value
	
COMPARE_DIV
	xorwf	DIV_SET_POS,w	; compare with fire position
	btfss	STATUS,Z		; fire when the same value
	goto	CHECK_CAPT
FIRE_LED
	movlw	B'00110000'		; set LED on
	iorwf	PORTB,f
; preload timer0 switch off when timer expires
; if fixed use FLASH_ON
	btfss	FLASHCON,7
	goto	USE_AUTO
	clrf	AUTO			; minimum auto flash set in fixed mode		
	comf	FLASH_ON,w		; flash on period
MOV_TIME
	movwf	TMR0			; set timer
	nop
	nop
	bcf		INTCON,TMR0IF	; clear flag
	goto	CHECK_CAPT
USE_AUTO
	comf	AUTO,w			; calculated automatic flash period
	goto	MOV_TIME

TRANS ;  transfer
	movf	DIV_SET,w
	movwf	DIV_SET_CNT		; transfer division value to counter
	xorwf	DIV_SET_POS,w	; compare with fire position
	btfsc	STATUS,Z		; fire when the same value
	goto	FIRE_LED
				
CHECK_CAPT ; check capture
	btfss	PIR1,CCP1IF	; if capture place into Count2,3, clear NIL_SIG 
	goto	CK_TMR_1
	bcf		T1CON,0		; stop timer
	
; check for capture before 
; overflow flag set 
	
	btfss	PIR1,TMR1IF	; if timer overflow flag set, ignore values 
	goto	XFER_VAL	; capture and no overflow then values are valid
TMR_CLFLG
	bcf		PIR1,TMR1IF	; clear flag
	bcf		PIR1,CCP1IF	; clear flag
	clrf	NIL_SIG		; signal captured so clear value 
	goto	CLR_COUNT

; check timer overflow
CK_TMR_1	
	btfss	PIR1,TMR1IF	; if timer overflow flag set, increase COUNTER1
	goto	CK_TMR0		; 
	bcf		PIR1,TMR1IF	; clear flag
	incfsz	COUNTER1,f	; overflow counter FFFF to 0000
	goto	CK_TMR0
	incf	COUNTER0,f	; increase ms byte if ls byte overflows
	btfsc	COUNTER0,5	; 107 s timeout for 1rpm
	bsf		NIL_SIG,0	; set flag if overflows beyond COUNTER0 
	goto	CK_TMR0
XFER_VAL
	bcf		PIR1,CCP1IF	; clear flag
	clrf	NIL_SIG		; signal captured so clear value 

; transfer COUNTER0,1 to COUNT0,1
; CCPR1L and H to COUNT2,3
	movf	COUNTER0,w
	movwf	COUNT0		; ms 32-bit rpm counter
	movf	COUNTER1,w
	movwf	COUNT1		; 32-bit rpm counter
	movf	CCPR1H,w	; capture compare value
	movwf	COUNT2		; 32-bit rpm counter
	movf	CCPR1L,w
	movwf	COUNT3		; ls 32-bit rpm counter

; clear counters
CLR_COUNT
	movf	CCP1CON,w	; store value
	clrf	CCP1CON		; clears prescaler
	movwf	CCP1CON		; restore value capture compare (bank 0)
	clrf	COUNTER0
	clrf	COUNTER1
	clrf	CCPR1L
	clrf	CCPR1H
	movf	TMR2,w		; compensate for the time timer1 is off
	movwf	TMR1L
	clrf	TMR1H
	bsf		T1CON,0		; restart timer

CK_TMR0

; check timer 0 for overflow
	btfss	INTCON,TMR0IF	; if flag set clear LED
	goto	RECLAIM
	movlw	B'11001111'		
	andwf	PORTB,f			; off again 
	bsf		EDGE_FG,3		; set to indicate that flash goes off
	bcf		INTCON,TMR0IF	; flag clear

RECLAIM
	movf	PCLATH_STO,w
	movwf	PCLATH		; restore PCLATH	
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

GEN_COMPARE
	bcf		INTCON,INTE	; clear edge interrupt and flag
	bcf		INTCON,INTF
	btfss	PIR1,CCP1IF	; if compare flag
	goto	CK_TMR_1 
	bcf		PIR1,CCP1IF	; clear flag
	clrf	NIL_SIG		; signal captured so clear value 

; compare 
	movf	COUNTER0,w	; ms counter of timer1
	xorwf	COUNT_REQ0,w
	btfss	STATUS,Z
	goto	COMP_LARGER0; check if larger	
	movf	COUNTER1,w	; counter of timer1
	xorwf	COUNT_REQ1,w
	btfss	STATUS,Z
	goto	COMP_LARGER1; ms is equal so check LS byte	
	goto	FIRE_LED1
; compare if ms byte is larger
COMP_LARGER0
; if COUNTER0 is larger than COUNT_REQ0 
	movf	COUNTER0,w
	subwf	COUNT_REQ0,w
	btfss	STATUS,C
	goto	RELOAD		
	goto	CK_TMR_1
COMP_LARGER1
; if COUNTER1 is larger than COUNT_REQ1 when ms byte is equal
	movf	COUNTER1,w
	subwf	COUNT_REQ1,w
	btfss	STATUS,C
	goto	RELOAD		
	goto	CK_TMR_1

FIRE_LED1
; fire LED	
	movlw	B'00110000'		; set LED on
	iorwf	PORTB,f
;
; preload timer0 switch off when timer expires
; if fixed use FLASH_ON
	btfss	FLASHCON,7
	goto	USE_AUTO1
	clrf	AUTO			; minimum auto flash set in fixed mode		
	comf	FLASH_ON,w		; flash on period
MOV_TIME1
	movwf	TMR0			; set timer
	nop
	nop
	bcf		INTCON,TMR0IF	; clear flag
	goto	NEW
USE_AUTO1
	comf	AUTO,w			; calculated automatic flash period
	goto	MOV_TIME1

; if new values ready load
NEW
	btfss	COMPARE_FG,0	; if set, load new values
	goto	OLD_RELOAD
RELOAD
	movf	COUNTREQ0,w
	movwf	COUNT_REQ0
	movf	COUNTREQ1,w
	movwf	COUNT_REQ1
	movf	COUNTREQ2,w
	movwf	COUNT_REQ2
	movf	COUNTREQ3,w
	movwf	COUNT_REQ3
	bcf		COMPARE_FG,0
OLD_RELOAD
; reload/reset required counters
	clrf	COUNTER0		; timer1 overflow ms byte
	clrf	COUNTER1
	movf	COUNT_REQ2,w
	movwf	CCPR1H
	movf	COUNT_REQ3,w
	movwf	CCPR1L
	bcf		T1CON,0			; stop timer
	clrf	TMR1H			; ms byte of timer 1
	movf	TMR2,w			; timer compensation
	movwf	TMR1L			; timer ls byte
	bsf		T1CON,0			; restart timer
	goto	CK_TMR0


; **********************************************************************************************
; Calculations

; 0.
; 1/2 pulse per rpm 
; Timer 1 internal clock count /1
; capture every + edge
; 1Hz (120rpm) has 5M count (numerator needs to be 120 x 5 x 10^6)
; Calculations: min 600M/5M = 120rpm 
; Calculation: 1000M/5M = 200. Shift DP for 2.00Hz

; 1.
; 1 pulse per rpm 
; Timer 1 internal clock count /1
; capture every + edge
; 1Hz (60rpm) has 5M count (numerator needs to be 60 x 5 x 10^6)
; Calculations: min 300M/5M = 60rpm 
; Calculation: 500M/5M = 100. Shift DP for 1.00Hz

; 2.
; 2 pulses per rpm 
; Timer 1 internal clock count /1
; capture every positive edge 
; 1Hz (30rpm) has 5MHz count (numerator needs to be 30 x 5 x 10^6)
; Calculations: min 150M/5M = 30rpm 
; Calculation: 250M/5M = 50. Shift DP for 0.50Hz

; 3.
; 3 pulses per rpm 
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (20rpm) has 10M count (numerator needs to be 20 x 10 x 10^6)
; Calculations: min 200M/10M = 20rpm 
; Calculation: 333,333,333/10M = 33 Shift DP for 0.33Hz

; 4.
; 4 pulses per rpm
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (15rpm) has 10M count (numerator needs to be 15 x 10 x 10^6)
; Calculations: min 150M/10M = 15rpm 
; Calculation: 250M/10M = 25. Shift DP for 0.25Hz

; 5.
; 5 pulses per rpm 
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (12rpm) has 10M count (numerator needs to be 12 x 10 x 10^6)
; Calculations: min 120M/10M = 12rpm 
; Calculation: 200M/10M = 20. Shift DP for 0.20Hz

; 6
; 6 pulses per rpm 
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (10rpm) has 10M count (numerator needs to be 10 x 10 x 10^6)
; Calculations: min 100M/10M = 10rpm 
; Calculation: 166,666,666/10M = 16.6. Shift DP for 0.166Hz

; 7
; 7 pulses per rpm 
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (8.5714285rpm) has 10M count (numerator needs to be 10 x 8.57... x 10^6)
; Calculations: min 85,714,285/10M = 8.57rpm 
; Calculation: 142,857,142/10M = 14.28. Shift DP for 0.1428571Hz

; 8
; 8 pulses per rpm 
; Timer 1 internal clock count /2
; capture every 4th edge
; 1Hz (7.5rpm) has 10M count (numerator needs to be 10 x 7.5 x 10^6)
; Calculations: min 75M/10M = 7.5rpm 
; Calculation: 125M/10M = 12.5. Shift DP for 0.125Hz

	org H'0D'	; page0	

SETUP
	movlw	EEPROM4
	call	EEREAD		; sets EEADR
	movwf	OPTIONS		; option reg store

	clrf	PORTB		; port B outputs low
	clrf	PORTA		; port A output low
	bsf		STATUS,RP0	; select memory bank 1

; set inputs/outputs
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'00001111'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'11110000'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movf	OPTIONS,w	; options in storage
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'00010000'	; AN4 is an analog input
	movwf	ANSEL
	movlw	B'01000000'	; left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'10100000'	; Fosc, channel4 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	bsf		T2CON,2		; timer 2 on no pre/post scaling
	
; initial conditions
	clrf	EDGE_FG		; clear edge flag
	clrf	DIV_SET_CNT	; counter for flashing LED on 1st/second,third etc count
	movlw	D'01'
	movwf	DIV_SET_POS	; counter to match div_set_cnt so the firing count can be altered
 	
; read EEPROM 
READ_EEP	
; division
	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movwf	DIV_SET		; division
; flash on period
	movlw	EEPROM2
	call	EEREAD		; sets EEADR
	movwf	FLASH_ON	; flash on period
; Mode
; mode. 00 is rpm/Hz display and triggered
; mode. H80 is rpm/Hz generated
	movlw	EEPROM3
	call	EEREAD		; sets EEADR
	movwf	MODE		; display MODE
; generator rpm
	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movwf	RPM0		; ms byte rpm
	movlw	EEPROM6
	call	EEREAD		; sets EEADR
	movwf	RPM1		; ls byte rpm
; flash or continuous
	movlw	EEPROM7
	call	EEREAD		; sets EEADR
	movwf	FLASHCON	; flash or continuous flag
; Averaging
	movlw	EEPROM8
	call	EEREAD		; sets EEADR
	movwf	AVERAG		; averaging value

; startup
	movlw	H'FF'		; start up delay
	call	DELAYX
	call	INIT_LC
	call	DELAYms
	call	INIT_LC
	call	DELAYms

	movlw	B'00101100'	; display function (4-bits, 2 lines, 5x10 dots)
	call	LOAD
	movlw	B'00001110'	; blinking off, cursor off
	call	LOAD
	movlw	B'00000001'	; display clear
	call	LOAD
	movlw	H'FF'		; delay 
	call	DELAYX		; 
	movlw	B'00000110'	; entry mode. cursor moves right, display not shifted
	call	LOAD
	movlw	B'00001110'	; blinking off, cursor off
	call	LOAD
	movlw	B'00101100'	; display function (4-bits, 2 lines, 5x10 dots)
	call	LOAD		; 
	movlw	B'00000001'	; display clear
	call	LOAD
	movlw	H'FF'		; delay 
	call	DELAYX
	bsf		PCLATH,3	; page 1
	call	CHARACTER_GEN; up/dn arrows

; initialise
	bsf		NIL_SIG,0	; set bit to cause overflow condition
	clrf	TOTAL0		; totals for capture counts
	clrf	TOTAL1
	clrf	TOTAL2
	clrf	AUTO		; auto flash period
; clear average values
	clrf	AV_CNT			; averaging counter
	clrf	TALLY0			; average tally ms byte
	clrf	TALLY1			; average tally 
	clrf	TALLY2			; average tally 
	clrf	TALLY3			; average tally ls byte
	bsf		EDGE_FG,1		; set flag when new display
	bcf		EDGE_FG,0		; edge trigger flag off
	bsf		EDGE_FG,2		; display flag

; TIMER1 mode resets
	bcf		T1CON,0			; stop timer
	clrf	COUNTER0		; timer1 overflow ms byte
	clrf	COUNTER1
	clrf	COUNT_REQ0
	movlw	D'60'
	movwf	COUNT_REQ1
	movlw	H'FF'
	movwf	COUNT_REQ2
	movwf	COUNT_REQ3	
	movwf	CCPR1H
	movwf	CCPR1L
			
	clrf	TMR1H			; ms byte of timer 1
	clrf	TMR1L			; timer ls byte
	bsf		T1CON,0			; restart timer

; enable interrupts
	bcf		INTCON,INTF	; clear edge flag
	bsf		INTCON,TMR0IE; timer 0 interrupt enable
	bcf		INTCON,TMR0IF; timer 0 interrupt flag
	bsf		STATUS,RP0	; select memory bank 1
	bsf		PIE1,TMR1IE	; timer1 interupt enable for overflow (bank 1)
	bsf		PIE1,CCP1IE	; capture compare interrupt
	bcf		STATUS,RP0	; select memory bank 0
	
	bsf		INTCON,PEIE	; enable peripheral interrupts
	bsf		INTCON,GIE	; global interrupt enable

CYCLE	
; code cycle
	movf	MODE,w			; if zero or H80 run a/d and rpm/Hz display
	andlw	B'01111111'		; strip ms bit
	btfss	STATUS,Z
	goto	CK_SW			; check switches

; run display update when all counters are 0 
	decfsz	UPDATE1,f
	goto	CK_SW
	incfsz	UPDATE2,f
	goto	CK_SW

CYCLE2
	
; check mode and alter edge triggering 

	btfss	MODE,7		; when set Generator mode so no edge interrupt
	goto 	EDGE
	bcf		INTCON,INTE	; no edge interrupt
	
	goto	CK_DIV
EDGE
	bsf		INTCON,INTE	; edge interrupt

; check division setting and set required timer settings and numerator
CK_DIV
	btfsc	MODE,7		; when set use Generator mode division of 1
	goto	NO_1
	movf	DIV_SET,w
	btfsc	STATUS,Z
	goto	NO_0		; 1/2 pulse / rpm
	movf	DIV_SET,w	; 
	xorlw	D'01'		; 1 pulse per rpm
	btfsc	STATUS,Z
	goto	NO_1
	movf	DIV_SET,w	; 
	xorlw	D'02'		; 2 pulses per rpm
	btfsc	STATUS,Z
	goto	NO_2
	movf	DIV_SET,w	; 
	xorlw	D'03'		; 3 pulses per rpm
	btfsc	STATUS,Z
	goto	NO_3
	movf	DIV_SET,w	; 
	xorlw	D'04'		; 4 pulses per rpm
	btfsc	STATUS,Z
	goto	NO_4
	movf	DIV_SET,w	; 
	xorlw	D'05'		; 5 pulses per rpm
	btfsc	STATUS,Z
	goto	NO_5
	movf	DIV_SET,w	
	xorlw	D'06'		; 6 pulses per rpm 
	btfsc	STATUS,Z
	goto	NO_6
	movf	DIV_SET,w	
	xorlw	D'07'		; 7 pulses per rpm 
	btfsc	STATUS,Z
	goto	NO_7
	movf	DIV_SET,w	
	xorlw	D'08'		; 8 pulses per rpm 
	btfsc	STATUS,Z
	goto	NO_8
NO_0
; rpm values
	movlw	H'23'		; 600M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'C3'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'46'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'3B'		; 1000M
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'9A'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'CA'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'00'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	ZERO_TWO_SETTING ; for div by 5M

NO_1
; rpm values
	movlw	H'11'		; 300M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'E1'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'A3'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'1D'		; 500M
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'CD'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'65'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'00'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	ZERO_TWO_SETTING ; for div by 5M

NO_2
; rpm values
	movlw	H'08'		; 150M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'F0'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'D1'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'80'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'0E'		; 250M
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'E6'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'B2'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'80'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	ZERO_TWO_SETTING ; for div by 5M

NO_3
; rpm values
	movlw	H'0B'		; 200M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'EB'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'C2'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'13'		; 333,333,333
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'DE'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'43'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'55'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	THREE_SIX_SETTING ; every 4th edge and for div by 10M

NO_4
; rpm values
	movlw	H'08'		; 150M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'F0'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'D1'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'80'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'0E'		; 250M
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'E6'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'B2'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'80'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	THREE_SIX_SETTING ; every 4th edge and for div by 10M

NO_5
; rpm values
	movlw	H'07'		; 120M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'27'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'0E'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'0B'		; 200M
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'EB'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'C2'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'00'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	THREE_SIX_SETTING ; every 4th edge and for div by 10M

NO_6
; rpm values
	movlw	H'05'		; 100M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'F5'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'E1'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'00'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'09'		; 166,666,666
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'EF'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'21'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'AA'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	THREE_SIX_SETTING ; every 4th edge and for div by 10M

NO_7
; rpm values
	movlw	H'05'		; 85,714,285
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'1B'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'E5'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'6D'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'08'		; 142,857,142
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'83'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'D3'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'B5'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	THREE_SIX_SETTING ; every 4th edge and for div by 10M

NO_8
; rpm values
	movlw	H'04'		; 75M
	movwf	NUMER_0		; rpm numerator value ms byte
	movlw	H'78'
	movwf	NUMER_1		; rpm numerator value mid byte
	movlw	H'68'
	movwf	NUMER_2		; rpm numerator value mid byte
	movlw	H'C0'
	movwf	NUMER_3		; rpm numerator value ls byte
; Frequency values
	movlw	H'07'		; 125M
	movwf	NUMER_4		; frequency numerator value ms byte
	movlw	H'73'
	movwf	NUMER_5		; frequency numerator value mid byte
	movlw	H'59'
	movwf	NUMER_6		; frequency numerator value mid byte
	movlw	H'40'
	movwf	NUMER_7		; frequency numerator value ls byte
	goto	THREE_SIX_SETTING ; every 4th edge and for div by 10M

ZERO_TWO_SETTING ; settings from 0 to 2	
	
	movlw	B'00000001'	; timer 1 prescaler /1 fosc/4
	movwf	T1CON		; bank 0
; set capture or compare depending on triggered or generator mode
	movlw	B'00001010'	; compare
	btfss	MODE,7		; when set, generator mode so keep compare
	movlw	B'00000101'	; triggered mode so set capture every positive edge
	movwf	CCP1CON		; capture compare (bank 0)
	goto	STROBE_RUN

THREE_SIX_SETTING ; settings from 3 to 6	
	movlw	B'00010001'	; timer 1 prescaler /2 fosc/4
	movwf	T1CON		; bank 0
	movlw	B'00000110'	; every 4th edge
	movwf	CCP1CON		; capture compare (bank 0)

STROBE_RUN
; tachometer run
	bcf		GEN_FLG,0	; generator display update flag
	btfss	MODE,7		; when set use Generator mode division of 1
	goto	TRIG_RUN
	goto	GEN_RUN

DRV_GEN_DSP
	bsf		GEN_FLG,0	; set to include a switch delay if coming from a switch value change

GEN_RUN ; generator run
; RPM0, RPM1 have rpm in 100rpm steps

; Channel 4 A/D value 
; 0-100rpm from the potentiometer (ADRESH)
; ADRESH 0 to 26 = 0
; ADRESH 228 to 255 =100
	call	ACQUIRE_AD 
	comf	ADRESH,f	; reverse sense to suit potentiometer connection on circuit
	movlw	D'227'
	subwf	ADRESH,w	
	btfsc	STATUS,C	; check for >227
	goto	PHASE_MAX
	movlw 	D'26'
	subwf	ADRESH,w
	btfss	STATUS,C	; check if <27
	goto	ZRO_PHASE_M	; set at 0
; use w value but divide by 2
	movwf	ADRESH
	bcf		STATUS,C	; carry clear
	rrf		ADRESH,f	; this is the 0-100rpm range setting from potentiometer
	goto	ADD_RPM
ZRO_PHASE_M
	clrf	ADRESH		; set at 0
	goto	ADD_RPM
PHASE_MAX
	movlw	D'100'
	movwf	ADRESH		; set at 100
ADD_RPM ; add to rpm + move into binary registers
	clrf	BIN_2
	movf	RPM0,w
	movwf	BIN_1		; get value
	movf	ADRESH,w
	addwf	RPM1,w		; add to ls byte
	movwf	BIN_0
	btfsc	STATUS,C	; if carry increase ms byte
	incf	BIN_1,f

; if both zero set at 1
	movf	BIN_1,w
	btfss	STATUS,Z
	goto	STO_TEMP	; not zero
	movf	BIN_0,w
	btfsc	STATUS,Z
	incf	BIN_0,f		; a 1

STO_TEMP
; store in temp for frequency measurement
	movf	BIN_0,w
	movwf	TEMPOR1
	movf	BIN_1,w
	movwf	TEMPOR0
	clrf	BIN_2

; calculate automatic pulse width if auto mode
	btfss	FLASHCON,7
	call	CALC_AUTO

	call	BCD			; convert to packed BCD with ASCII encoding
; values in OUT0-OUT7
	movlw	H'80'		; address line 1
	call	LOAD

	clrf	RPM_FLG		; clear so no decimal point
	call	NUMBER_SHOW
; rpm
	movlw	A'r'
	call	DRV_LCD
	movlw	A'p'
	call	DRV_LCD
	movlw	A'm'
	call	DRV_LCD
	call	SPACE1
	movlw	A'G'		; generated
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	A'n'
	call	DRV_LCD
	movlw	D'4'
	call	SPACEX

; divide rpm value into numerator (300M) to get counter value
	movf	NUMER_0,w		
	movwf	AARGB0		; numerator ms byte
	movf	NUMER_1,w	;
	movwf	AARGB1
	movf	NUMER_2,w	;
	movwf	AARGB2		; 
	movf	NUMER_3,w	; numerator ls byte
	movwf	AARGB3

; rpm value in denominator
	clrf	BARGB0
	clrf	BARGB1
	movf	TEMPOR0,w		; denominator
	movwf	BARGB2
	movf	TEMPOR1,w		; denominator ls byte
	movwf	BARGB3
	bsf		PCLATH,3	; page 1
	call	FXD3232U	; divide to get counter value

; place required counter value into registers
	
	movf	AARGB0,w		
	movwf	COUNTREQ0	; required counter value ms byte
	movf	AARGB1,w	;
	movwf	COUNTREQ1
	movf	AARGB2,w	;
	movwf	COUNTREQ2	; 
	movf	AARGB3,w	; ls byte
	movwf	COUNTREQ3
	bsf		INTCON,GIE
	bsf		COMPARE_FG,0	; set to indicate new values are ready

; compare with last COUNT_REQ values
	movf	COUNTREQ0,w
	xorwf	LAST_REQ0,w
	btfss	STATUS,Z
	goto	RESET_COMPARE	; different
	movf	COUNTREQ1,w
	xorwf	LAST_REQ1,w
	btfss	STATUS,Z
	goto	RESET_COMPARE	; different
	movf	COUNTREQ2,w
	xorwf	LAST_REQ2,w
	btfss	STATUS,Z
	goto	RESET_COMPARE	; different
	movf	COUNTREQ3,w
	xorwf	LAST_REQ3,w
	btfss	STATUS,Z
	goto	RESET_COMPARE	; different

	goto	DIV_FREQ_GEN	; all the same

; when different to last time load new values
RESET_COMPARE
	movf	COUNTREQ2,w
	movwf	LAST_REQ2		; store new count
	movf	COUNTREQ3,w
	movwf	LAST_REQ3		; store new count
	movf	COUNTREQ0,w
	movwf	LAST_REQ0		; store new count
	movf	COUNTREQ1,w
	movwf	LAST_REQ1		; store new count
; different so check if rpm <512 if so reload COUNTER_REQ0,1
	movf	TEMPOR0,w
	btfsc	STATUS,Z
	goto	RELOAD_REQ		; zero so TEMPOR1 is 255 or less 
	decfsz	TEMPOR0,f		; if 0 then 512 or less	
	goto	DIV_FREQ_GEN
; <512 so reload
RELOAD_REQ
	bcf		INTCON,GIE		; prevent interrupt
	movf	COUNTREQ0,w
	movwf	COUNT_REQ0
	movf	COUNTREQ1,w
	movwf	COUNT_REQ1
	bsf		INTCON,GIE		; enable interrupt 

DIV_FREQ_GEN
; divide to get frequency values 
	movf	NUMER_4,w		
	movwf	AARGB0		; numerator ms byte
	movf	NUMER_5,w	;
	movwf	AARGB1
	movf	NUMER_6,w	;
	movwf	AARGB2		; 
	movf	NUMER_7,w	; numerator ls byte
	movwf	AARGB3

	movf	COUNTREQ0,w	; denominator ms byte
	movwf	BARGB0
	movf	COUNTREQ1,w	; denominator 
	movwf	BARGB1
	movf	COUNTREQ2,w	; denominator
	movwf	BARGB2
	movf	COUNTREQ3,w	; denominator ls byte
	movwf	BARGB3
	bsf		PCLATH,3	; page 1
	call	FXD3232U	; divide to get rpm
; result in AARGB0,1,2,3
; move into binary registers
	movf	AARGB3,w
	movwf	BIN_0
	movf	AARGB2,w
	movwf	BIN_1
	movf	AARGB1,w
	movwf	BIN_2
	call	BCD			; convert to packed BCD with ASCII encoding

; values in OUT0 to OUT7	
	movlw	H'C0'		; address line 2
	call	LOAD

	bsf		RPM_FLG,0	; set so decimal point can be added
	call	NUMBER_SHOW	; display numbers

	movlw	A'H'
	call	DRV_LCD
	movlw	A'z'
	call	DRV_LCD
	movlw	D'6'
	call	SPACEX

; check if flash goes off
	btfss	EDGE_FG,3	; if set goes off
	goto	BY_FLSH_OFF
	bcf		EDGE_FG,3
	movlw	A'*'
	call	DRV_LCD
BY_FLSH_OFF
	call	SPACE1

	movf	GEN_FLG,w	; if not zero then add switch delay
	btfsc	STATUS,Z
	goto	CK_SW
	clrf	GEN_FLG		; clear when delay runs
	movlw	D'06'		; delay
	call	DELAY_SWITCH1
	goto	CK_SW

; trigger input
TRIG_RUN
; divide to get rpm  values in 	NUMER_x
	movf	NUMER_0,w		
	movwf	AARGB0		; numerator ms byte
	movf	NUMER_1,w	;
	movwf	AARGB1
	movf	NUMER_2,w	;
	movwf	AARGB2		; 
	movf	NUMER_3,w	; numerator ls byte
	movwf	AARGB3

; stop interrupt
	bcf		INTCON,GIE
	nop
	movf	COUNT0,w	; denominator ms byte
	movwf	COUNTX0		; store for frequency
	movf	COUNT1,w	; denominator 
	movwf	COUNTX1
	movf	COUNT2,w	; denominator
	movwf	COUNTX2
	movf	COUNT3,w	; denominator ls byte
	movwf	COUNTX3		; store for frequency

; allow interrupt
	bsf		INTCON,GIE

	movf	COUNTX3,w	; ls bytes addition
	addwf	TALLY3,f
	btfss	STATUS,C	; if carry set increase
	goto	TALLYX		; next byte
	incfsz	TALLY2,f	; increase if over
	goto	TALLYX
	incfsz	TALLY1,f	; increase if over
	goto	TALLYX
	incfsz	TALLY0,f	; increase if over
	goto	TALLYX
	goto	CLEAR_TALLY	; clear if overrange
TALLYX
; TALLY2
	movf	COUNTX2,w	;  bytes addition
	addwf	TALLY2,f
	btfss	STATUS,C	; if carry set increase
	goto	TALLYY		; next byte
	incfsz	TALLY1,f	; increase if over
	goto	TALLYY
	incfsz	TALLY0,f	; increase if over
	goto	TALLYY
	goto	CLEAR_TALLY	; clear if overrange
TALLYY
; TALLY1
	movf	COUNTX1,w	;  bytes addition
	addwf	TALLY1,f
	btfss	STATUS,C	; if carry set increase
	goto	TALLYZ		; next byte
	incfsz	TALLY0,f	; increase if over
	goto	TALLYZ
	goto	CLEAR_TALLY	; clear if overrange

TALLYZ
; TALLY0
	movf	COUNTX0,w	; ms byte addition
	addwf	TALLY0,f
	btfss	STATUS,C	; if carry then overrange
	goto	REMEMBER_TALLY_COUNT
		
CLEAR_TALLY
	clrf	TALLY0
	clrf	TALLY1
	clrf	TALLY2
	clrf	TALLY3
	clrf	AV_CNT
	goto	END_OF_TALLY

REMEMBER_TALLY_COUNT
; check if averaging is complete
	incf	AV_CNT,f	; next average tally
	movf	AV_CNT,w
	subwf	AVERAG,w	; if negative, end of tally
	btfss	STATUS,C	
	goto	END_OF_TALLY
	btfsc	STATUS,Z	; if the same value
	goto	END_OF_TALLY
	btfsc	EDGE_FG,2	; when set run 00 display
	goto	RESET1_RPM	; show 0rpm 0Hz 
	goto	END_DISP	; bypass update display

END_OF_TALLY
	clrf	AV_CNT		; clear average counter
	
	bcf		EDGE_FG,2
; calculate average (TALLY/AVERAG)
	movf	TALLY0,w		
	movwf	AARGB0		; numerator ms byte
	movf	TALLY1,w	;
	movwf	AARGB1
	movf	TALLY2,w	;
	movwf	AARGB2		; 
	movf	TALLY3,w	; numerator ls byte
	movwf	AARGB3
	clrf	TALLY0		; clear for next average
	clrf	TALLY1
	clrf	TALLY2
	clrf	TALLY3
	clrf	AV_CNT
	
	clrf	BARGB0
	clrf	BARGB1
	clrf	BARGB2
	movf	AVERAG,w	; denominator ls byte
	movwf	BARGB3
	bsf		PCLATH,3	; page 1
	call	FXD3232U	; divide to get average rpm

; result in AARGB0,1,2,3
	movf	AARGB0,w	; denominator ms byte
	movwf	BARGB0
	movwf	COUNTX0		; store for frequency
	movf	AARGB1,w	; denominator 
	movwf	BARGB1
	movwf	COUNTX1		; store for frequency
	movf	AARGB2,w	; denominator
	movwf	BARGB2
	movwf	COUNTX2		; store for frequency
	movf	AARGB3,w	; denominator ls byte
	movwf	BARGB3
	movwf	COUNTX3		; store for frequency

; divide to get rpm values in NUMER_x
	movf	NUMER_0,w		
	movwf	AARGB0		; numerator ms byte
	movf	NUMER_1,w	;
	movwf	AARGB1
	movf	NUMER_2,w	;
	movwf	AARGB2		; 
	movf	NUMER_3,w	; numerator ls byte
	movwf	AARGB3

RPM_VALUE
	bsf		PCLATH,3	; page 1
	call	FXD3232U	; divide to get rpm
; result in AARGB0,1,2,3

; move into binary registers
	movf	AARGB3,w
	movwf	BIN_0
	movf	AARGB2,w
	movwf	BIN_1
	movf	AARGB1,w
	movwf	BIN_2

; calculate automatic pulse width if auto mode
	btfss	FLASHCON,7
	call	CALC_AUTO

	call	BCD			; convert to packed BCD with ASCII encoding
; values in OUT0-OUT7

RESET1_RPM
	movlw	H'80'		; address line 1
	call	LOAD

; if new display clear
	btfsc	EDGE_FG,2
	goto	RESET_RPM
	btfsc	EDGE_FG,1
	goto	RESET_RPM
; if overflow load 0
	movf	NIL_SIG,w	; check if no signal
	btfsc	STATUS,Z	; if zero can calculate
	goto	VALUE_RPM
; reset values
RESET_RPM
	bsf		EDGE_FG,1
	movlw	H'30'		; set at ASCII 0
;	movwf	OUT0
;	movwf	OUT1
	movwf	OUT2
	movwf	OUT3
	movwf	OUT4
	movwf	OUT5
	movwf	OUT6
	movwf	OUT7

VALUE_RPM
	clrf	RPM_FLG		; clear so no decimal point
	call	NUMBER_SHOW
	goto	RPM_DISP

NUMBER_SHOW
; places the numbers (up to 6-digits on display)
; ignore out0, out1 as these are beyond max values
; test OUT_0
;	movf	OUT0,w		; rpm
;	xorlw	H'30'		; is it 0
;	btfss	STATUS,Z
;	goto	LOADR0		; not zero so load value
; leading is zero
;	call	SPACE1 

; test OUT_1
;	movf	OUT1,w		; rpm
;	xorlw	H'30'		; is it 0
;	btfss	STATUS,Z
;	goto	LOADR1		; not zero so load value
; leading is zero
;	call	SPACE1 

; test OUT_2
	movf	OUT2,w		; rpm
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADR2		; not zero so load value
; leading is zero
	call	SPACE1 

; test OUT_3
	movf	OUT3,w		; rpm
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADR3		; not zero so load value
; leading is zero
	call	SPACE1 

; test OUT_4
	movf	OUT4,w		; rpm
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADR4		; not zero so load value
; leading is zero
	call	SPACE1 

; test OUT_5
	btfsc	RPM_FLG,0	; if set then rpm mode so show this digit and add decimal point after
	goto	LOADR5
	movf	OUT5,w		; rpm
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADR5		; not zero so load value
; leading is zero
	call	SPACE1 

; test OUT_6
	movf	OUT6,w		; rpm
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADR6		; not zero so load value
; leading is zero
	call	SPACE1 
	goto	LOADR7

;LOADR0
;	movf	OUT0,w		; not zero so load value
;	call 	DRV_LCD
;LOADR1	
;	movf	OUT1,w		; not zero so load value
;	call 	DRV_LCD
LOADR2
	movf	OUT2,w
	call	DRV_LCD	
LOADR3
	movf	OUT3,w
	call	DRV_LCD	
LOADR4
	movf	OUT4,w
	call	DRV_LCD	
LOADR5
	movf	OUT5,w
	call	DRV_LCD	
	btfss	RPM_FLG,0	; if set then rpm mode, add decimal point 
	goto	LOADR6
	movlw	A'.'
	call	DRV_LCD
LOADR6
	movf	OUT6,w
	call	DRV_LCD	
LOADR7
	movf	OUT7,w
	call	DRV_LCD	
	return

RPM_DISP
	movlw	A'r'
	call	DRV_LCD
	movlw	A'p'
	call	DRV_LCD
	movlw	A'm'
	call	DRV_LCD
	call	SPACE1

; SHOW_TRIG
	movlw	A'T'		; triggered
	call	DRV_LCD
	movlw	A'r'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD	
	movlw	A'g'
	call	DRV_LCD

; if edge flag set then show trigger
; alternate trigger flag on and off
	btfss	EDGE_FG,6
	goto	SET_EDGE_FG
	bcf		EDGE_FG,6		; when set clear	
	btfss	EDGE_FG,0
	goto	NO_EDGE
;	btfss	EDGE_FG,5		; flag 5 slows down arrow flash by 2
;	goto	SET_EDGE_5
;	bcf		EDGE_FG,5
	call	SPACE1
	movlw	H'01'			; down arrow
	btfsc	OPTIONS,6		; if set rising edge trigger
	movlw	H'00'			; up arrow
	call	DRV_LCD
	bcf		EDGE_FG,0
	goto	NO_EDGE
;SET_EDGE_5
;	bsf		EDGE_FG,5
;	goto	NO_EDGE	
SET_EDGE_FG
	bsf		EDGE_FG,6	
NO_EDGE
	movlw	D'2'
	call	SPACEX	
	
DIV_FREQ
; divide to get frequency values in	NUMER_x
	movf	NUMER_4,w		
	movwf	AARGB0		; numerator ms byte
	movf	NUMER_5,w	;
	movwf	AARGB1
	movf	NUMER_6,w	;
	movwf	AARGB2		; 
	movf	NUMER_7,w	; numerator ls byte
	movwf	AARGB3

	movf	COUNTX0,w	; denominator ms byte
	movwf	BARGB0
	movf	COUNTX1,w	; denominator 
	movwf	BARGB1
	movf	COUNTX2,w	; denominator
	movwf	BARGB2
	movf	COUNTX3,w	; denominator ls byte
	movwf	BARGB3
	bsf		PCLATH,3	; page 1
	call	FXD3232U	; divide to get rpm
; result in AARGB0,1,2,3
; move into binary registers
	movf	AARGB3,w
	movwf	BIN_0
	movf	AARGB2,w
	movwf	BIN_1
	movf	AARGB1,w
	movwf	BIN_2
	call	BCD			; convert to packed BCD with ASCII encoding

; values in OUT0 to OUT7	
	movlw	H'C0'		; address line 2
	call	LOAD

; if new display clear
	btfsc	EDGE_FG,2
	goto	RESET_HZ
	btfsc	EDGE_FG,1
	goto	RESET_HZ
; if overflow load 0
	movf	NIL_SIG,w	; check if no signal
	btfsc	STATUS,Z	; if zero can calculate
	goto	VALUE_Hz
; reset values
RESET_HZ
	movlw	H'30'		; set at ASCII 0
	movwf	OUT0
	movwf	OUT1
	movwf	OUT2
	movwf	OUT3
	movwf	OUT4
	movwf	OUT5
	movwf	OUT6
	movwf	OUT7

VALUE_Hz
	bsf		RPM_FLG,0	; set so decimal point can be added
	call	NUMBER_SHOW	; display numbers

FREQUENCY_DISPLAY
	movlw	A'H'
	call	DRV_LCD
	movlw	A'z'
	call	DRV_LCD
	call 	SPACE1

	movlw	A'/'
	call	DRV_LCD
	movf	DIV_SET,w
	btfss	STATUS,Z	; division setting if zero then 0.5p/rev
	goto	NOT_0.5
	movlw	A'0'
	call	DRV_LCD
	movlw	A'.'		; dp
	call	DRV_LCD
	movlw	A'5'
	call	DRV_LCD
	goto	END_LINE
NOT_0.5
	movf	DIV_SET,w	; get division ratio
	iorlw	H'30'		; convert to ASCII
	call	DRV_LCD
	movlw	D'2'
	call	SPACEX
END_LINE
	call	SPACE1

; check if flash goes off
	btfss	EDGE_FG,3	; if set goes off
	goto	BY_FLSH_OFF1
	bcf		EDGE_FG,3
	movlw	A'*'
	call	DRV_LCD
BY_FLSH_OFF1
	call	SPACE1

END_DISP
	movf	GEN_FLG,w	; if not zero then add switch delay
	btfsc	STATUS,Z
	goto	CK_SW
	clrf	GEN_FLG		; clear when delay runs
	movlw	D'06'		; delay
	call	DELAY_SWITCH1
	goto	CK_SW

; 
CALC_AUTO ; calculate automatic flash period
; 23437 x flashcon/rpm (200us x 128 (divider)per timer0 count) so 25.6us/count
; 1 count = 1% so 1% or 1/100 of frequency is required value. 
; eg 1000rpm into 23437 is 23.437 and so 
; when times 25.6us = 600us. 1000rpm has 60ms period, and 600us is 1% of 60ms. 

	clrf	BARGB0		; denominator ms byte
	movf	BIN_2,w		; denominator 
	movwf	BARGB1
	movf	BIN_1,w		; denominator 
	movwf	BARGB2
	movf	BIN_0,w		; denominator ls byte
	movwf	BARGB3

; check flashcon value and set numerator at 23437 for 1%, 46874 for 2% etc

	movf	FLASHCON,w
	xorlw	D'01'		; 1%
	btfsc	STATUS,Z
	goto	FLASH_1
	movf	FLASHCON,w
	xorlw	D'02'		; 2%
	btfsc	STATUS,Z
	goto	FLASH2
	movf	FLASHCON,w
	xorlw	D'03'		; 3%
	btfsc	STATUS,Z
	goto	FLASH3
	movf	FLASHCON,w
	xorlw	D'04'		; 4%
	btfsc	STATUS,Z
	goto	FLASH4
	movf	FLASHCON,w
	xorlw	D'05'		; 5%
	btfsc	STATUS,Z
	goto	FLASH5
	movf	FLASHCON,w
	xorlw	D'06'		; 6%
	btfsc	STATUS,Z
	goto	FLASH6
	movf	FLASHCON,w
	xorlw	D'07'		; 7%
	btfsc	STATUS,Z
	goto	FLASH7
	movf	FLASHCON,w
	xorlw	D'08'		; 8%
	btfsc	STATUS,Z
	goto	FLASH8
	movf	FLASHCON,w
	xorlw	D'09'		; 9%
	btfsc	STATUS,Z
	goto	FLASH9
	movf	FLASHCON,w
	xorlw	D'10'		; 10%
	btfsc	STATUS,Z
	goto	FLASH10

FLASH_1
; H5B8D = 23437
	clrf	AARGB0		; numerator ms byte
	clrf	AARGB1
	movlw	H'5B'
	movwf	AARGB2		; 
	movlw	H'8D'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH2
; 46874 = HB71A
	clrf	AARGB0		; numerator ms byte
	clrf	AARGB1
	movlw	H'B7'
	movwf	AARGB2		; 
	movlw	H'1A'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH3
; 70311 = H112A7
	clrf	AARGB0		; numerator ms byte
	movlw	H'01'
	movwf	AARGB1
	movlw	H'12'
	movwf	AARGB2		; 
	movlw	H'A7'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH4
; 93748 = H16E34
	clrf	AARGB0		; numerator ms byte
	movlw	H'01'
	movwf	AARGB1
	movlw	H'6E'
	movwf	AARGB2		; 
	movlw	H'34'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH5
; 117185 = H1C9C1
	clrf	AARGB0		; numerator ms byte
	movlw	H'01'
	movwf	AARGB1
	movlw	H'C9'
	movwf	AARGB2		; 
	movlw	H'C1'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH6
; 140622 = H2254E 
	clrf	AARGB0		; numerator ms byte
	movlw	H'02'
	movwf	AARGB1
	movlw	H'25'
	movwf	AARGB2		; 
	movlw	H'4E'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH7
; 164059 = H280DB
	clrf	AARGB0		; numerator ms byte
	movlw	H'02'
	movwf	AARGB1
	movlw	H'80'
	movwf	AARGB2		; 
	movlw	H'DB'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH8
; 187496 = H2DC68
	clrf	AARGB0		; numerator ms byte
	movlw	H'02'
	movwf	AARGB1
	movlw	H'DC'
	movwf	AARGB2		; 
	movlw	H'68'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH9
; 210933 = H337F5
	clrf	AARGB0		; numerator ms byte
	movlw	H'03'
	movwf	AARGB1
	movlw	H'37'
	movwf	AARGB2		; 
	movlw	H'F5'
	movwf	AARGB3
	goto	DIV_FLASH

FLASH10
; 234370 = H39382
	clrf	AARGB0		; numerator ms byte
	movlw	H'03'
	movwf	AARGB1
	movlw	H'93'
	movwf	AARGB2		; 
	movlw	H'82'
	movwf	AARGB3

; divide 
DIV_FLASH
	bsf		PCLATH,3	; page 1
	call	FXD3232U	; divide 
; result in AARGB0,1,2,3
; if AARGB0, AARGB1, AARGB2 not zero then use FF for AUTO
	movf	AARGB0,w
	btfss	STATUS,Z
	goto	LD_AUTO_MAX	; not zero so load max value
 	movf	AARGB1,w
	btfss	STATUS,Z
	goto	LD_AUTO_MAX	; not zero so load max value
	movf	AARGB2,w
	btfss	STATUS,Z
	goto	LD_AUTO_MAX	; not zero so load max value

	movf	AARGB3,w
	movwf	AUTO
	return
LD_AUTO_MAX
	movlw	H'FF'		; load max value
	movwf	AUTO
	return		
; 


; check switches
CK_SW
	movf	PORTB,w			; switches at RB1, 2 & 3
	iorlw	B'11110001'		; or with 1,s and switch inputs with 0's
	xorlw	H'FF'			; if all 1's then no switch pressed
	btfsc	STATUS,Z
	goto	CYCLE			; all 1's so bypass switch check
	
; which switch
	
	btfss	PORTB,1
	goto	MODE_DSP
	btfss	PORTB,2
	goto	UP
	btfss	PORTB,3
	goto	DN
	goto	CYCLE

UP
; for Mode 0 and if mode,7 bit clear then can rotate the position for triggering when triggering uses > 1pulse per rev
	movf	MODE,w
	btfss	STATUS,Z		; if zero can increase DIV_SET_POS that determines the actual 
							; firing position in a series of pulses
	goto	BY_UP_MODE0
	incf	DIV_SET_POS,w
	subwf	DIV_SET,w		; if greater than DIV_SET return to a 1
	btfss	STATUS,C
	goto	RET_1
	incf	DIV_SET_POS,f	; increase
	goto	END_SW			; finished with switch	
RET_1
	movlw	D'01'
	movwf	DIV_SET_POS
	goto	END_SW

BY_UP_MODE0 
; if MODE0 and bit 7 set increase rpm0,1
	movf	MODE,w
	xorlw	B'10000000'		;
	btfss	STATUS,Z
	goto	NOT_MODE0_UP; other modes
; increase rpm in 100rpm steps
	movf	RPM0,w		; ms rpm
	movwf	TEMPOR0 
	movlw	D'100'
	addwf	RPM1,w
	movwf	TEMPOR1		; temporary
	movlw	D'01'		; get 1 for addition if required
	btfsc	STATUS,C	; if carry then increase ms byte
	addwf	TEMPOR0,f	; add 1 temporary 
	btfsc	STATUS,C	; if carry ignore addition
	goto	DRV_GEN_DSP	
; transfer temporary values to RPM0,1
TRANS_TEMP_RPM
	movf	TEMPOR0,w
	movwf	RPM0
	movf	TEMPOR1,w
	movwf	RPM1
; store in EEPROM
	movlw 	EEPROM5
	call	EEREAD
	movf	RPM0,w
	call	EEWRITE
	movlw 	EEPROM6
	call	EEREAD
	movf	RPM1,w
	call	EEWRITE
	goto	DRV_GEN_DSP

NOT_MODE0_UP
; if mode 1 then can alter triggering or generator flag bit 7 in Mode
	movf	MODE,w
	andlw	B'01111111'		; remove ms bit
	xorlw	B'00000001'		; 1
	btfss	STATUS,Z
	goto	BY_UP_MODE1
MODE_1_CHNG
	btfsc	MODE,7			; alter bit 7
	goto	CLR_7
	bsf		MODE,7
	goto	STO_7
CLR_7
	bcf		MODE,7
STO_7
	movlw	EEPROM3
	call 	EEREAD		; sets eeadr
	movf	MODE,w
	andlw	B'10000000'	; only store bit 7 for mode
	call	EEWRITE
; clear average values
	clrf	AV_CNT			; averaging counter
	
	clrf	TALLY0			; average tally ms byte
	clrf	TALLY1			; average tally 
	clrf	TALLY2			; average tally 
	clrf	TALLY3			; average tally ls byte
	goto	DRV_MODE_DSP

BY_UP_MODE1
	movf	MODE,w
	xorlw	B'00000010'		; mode 2 and trigger mode Mode,7 clear
	btfss	STATUS,Z
	goto	BY_UP_MODE2
	btfss	OPTIONS,6		; change bit 6 (edge)
	goto	OPT6
	bcf		OPTIONS,6
	goto	CHNG_OPT6
OPT6
	bsf		OPTIONS,6	
CHNG_OPT6
	bsf		STATUS,RP0	; select memory bank 1
	movf	OPTIONS,w	; options in storage
	movwf	OPTION_REG
	bcf		STATUS,RP0	; bank 0
; write
	movlw	EEPROM4
	call	EEREAD
	movf	OPTIONS,w
	call 	EEWRITE
	goto	DRV_MODE_DSP

BY_UP_MODE2
; If mode 3 then division but only for Triggered mode (Mode,7 clear)

	movf	MODE,w
	xorlw	B'00000011'		; mode 3 and trigger mode Mode,7 clear
	btfss	STATUS,Z
	goto	BY_UP_MODE3
; increase DIV_SET to 8
	incf	DIV_SET,w
	sublw	D'08'			; if over 8 return to 0
	incf	DIV_SET,w		; ready next value if required 
	btfss	STATUS,C
	clrw					; return to 0
	movwf	DIV_SET			; load 0 or next value
WRI_DIV_SET
; write
	movlw	EEPROM1
	call	EEREAD
	movf	DIV_SET,w
	call 	EEWRITE
; clear average values
	clrf	AV_CNT			; averaging counter
	
	clrf	TALLY0			; average tally ms byte
	clrf	TALLY1			; average tally 
	clrf	TALLY2			; average tally 
	clrf	TALLY3			; average tally ls byte
	goto	DRV_MODE_DSP

BY_UP_MODE3

; if mode 4 then can alter flash as fixed or automatic
	movf	MODE,w
	andlw	B'01111111'		; remove ms byte
	xorlw	B'00000100'		; 4
	btfss	STATUS,Z
	goto	BY_UP_MODE4
UP_MODE5
	btfsc	FLASHCON,7		; alter bit 0
	goto	CLR_F7
	bsf		FLASHCON,7
	goto	STO_F7
CLR_F7
	bcf		FLASHCON,7
STO_F7
	movlw	EEPROM7
	call 	EEREAD		; sets eeadr
	movf	FLASHCON,w
	call	EEWRITE
	goto	DRV_MODE_DSP	

BY_UP_MODE4
; if mode 5 can alter flash period either % or actual value
	movf	MODE,w
	andlw	B'01111111'		; remove ms byte
	xorlw	B'00000101'		; mode 5 
	btfss	STATUS,Z
	goto	BY_UP_MODE5

; if fixed increase FLASH_ON
	btfss	FLASHCON,7
	goto	INC_FLASHCON	; auto
	incf	FLASH_ON,f
	movf	FLASH_ON,w
	xorlw	D'255'			; clear when up to 255
	btfsc	STATUS,Z
	clrf	FLASH_ON
	movf	FLASH_ON,w
	xorlw	D'254'			; clear when up to 254
	btfsc	STATUS,Z
	clrf	FLASH_ON
WRI_EEP2
	movlw	EEPROM2
	call	EEREAD
	movf	FLASH_ON,w
	call 	EEWRITE
	goto	DRV_MODE_DSP

INC_FLASHCON	; auto
; increase FLASHCON to 10
	incf	FLASHCON,w
	sublw	D'10'			; if over 10 return to 0
	btfss	STATUS,C
	clrf	FLASHCON		; return to 0
	incf	FLASHCON,f		; load 1 or next value
; write
WRI_FLASHCON_SET
	movlw	EEPROM7
	call	EEREAD
	movf	FLASHCON,w
	call 	EEWRITE
	goto	DRV_MODE_DSP

BY_UP_MODE5
	movf	MODE,w
	xorlw	B'00000110'		; mode 6 and trigger mode 
	btfss	STATUS,Z
	goto	BY_UP_MODE6
; increase AVERAG to 10
	incf	AVERAG,w
	sublw	D'10'			; if over 10 return to 0
	btfss	STATUS,C
	clrf	AVERAG			; return to 0
	incf	AVERAG,f		; load 1 or next value
; write
WRI_AVG_SET
	movlw	EEPROM8
	call	EEREAD
	movf	AVERAG,w
	call 	EEWRITE
; clear average values
	clrf	AV_CNT			; averaging counter
	
	clrf	TALLY0			; average tally ms byte
	clrf	TALLY1			; average tally 
	clrf	TALLY2			; average tally 
	clrf	TALLY3			; average tally ls byte
	bsf		EDGE_FG,1
	bsf		EDGE_FG,2
	goto	DRV_MODE_DSP

BY_UP_MODE6	
	goto	END_SW
	
DN
; for Mode 0 and if mode,7 bit clear then can rotate the position for triggering when triggering uses > 1pulse per rev
	movf	MODE,w	
	btfss	STATUS,Z		; if zero can decrease DIV_SET_POS that determines the actual 
							; firing position in a series of pulses
	goto	BY_DN_MODE0
	decf	DIV_SET_POS,w	; if zero load DIV_SET
	btfsc	STATUS,Z
	goto	RET_VAL
	decf	DIV_SET_POS,f	; decrease
	goto	END_SW			; finished with switch	
RET_VAL
	movf	DIV_SET,w
	movwf	DIV_SET_POS
	goto	END_SW

BY_DN_MODE0 
; if MODE0 and bit 7 set decrease rpm0,1
	movf	MODE,w
	xorlw	B'10000000'		;
	btfss	STATUS,Z
	goto	NOT_MODE0_DN; other modes
; decrease rpm in 100rpm steps
	movf	RPM0,w		; ms rpm
	movwf	TEMPOR0 
	movlw	D'100'
	subwf	RPM1,w
	movwf	TEMPOR1		; temporary
	movlw	D'01'		; get 1 for subtraction if required
	btfss	STATUS,C	; if carry then increase ms byte
	subwf	TEMPOR0,f	; subtract 1 temporary 
	btfss	STATUS,C	; if carry ignore addition
	goto	DRV_GEN_DSP
	goto	TRANS_TEMP_RPM

NOT_MODE0_DN
; other modes
; if mode 1 then can alter triggering or generator flag bit 7 in Mode
	movf	MODE,w
	andlw	B'01111111'		; remove ms bit
	xorlw	B'00000001'		; 1
	btfsc	STATUS,Z
	goto	MODE_1_CHNG		; change bit 7

BY_DN_MODE1
	movf	MODE,w
	xorlw	B'00000010'		; mode 2
	btfsc	STATUS,Z
	goto	BY_UP_MODE1

; If mode 3 then division but only for Triggered mode (Mode,7 clear)

	movf	MODE,w
	xorlw	B'00000011'		; mode 3 and trigger mode Mode,7 clear
	btfss	STATUS,Z
	goto	BY_DN_MODE3
; decrease DIV_SET to 0 return to 8
	movf	DIV_SET,w		; if 0 return to 8
	btfsc	STATUS,Z
	goto 	RETURN_8		
	decf	DIV_SET,f		; decrease to next value 
	goto	WRI_DIV_SET		; store value in EEPROM
RETURN_8
	movlw	D'08'			; return to 8
	movwf	DIV_SET			; load 0 or next value
	goto	WRI_DIV_SET

BY_DN_MODE3
; if mode 4 then can alter flash as fixed or auto
	movf	MODE,w
	andlw	B'01111111'
	xorlw	B'00000100'		; 4
	btfss	STATUS,Z
	goto	BY_DN_MODE4
	goto	UP_MODE5		; same routine as up mode

BY_DN_MODE4
	movf	MODE,w
	andlw	B'01111111'		; remove ms byte
	xorlw	B'00000101'		; mode 5 
	btfss	STATUS,Z
	goto	BY_DN_MODE5
; if fixed or auto flash
	btfss	FLASHCON,7
	goto	FLASH_CON_DN

	decf	FLASH_ON,f
	movf	FLASH_ON,w
	xorlw	D'255'
	btfss	STATUS,Z
	goto	WRI_EEP2
	movlw	D'253'			; 253 max
	movwf	FLASH_ON
	goto	WRI_EEP2

FLASH_CON_DN
	movf	FLASHCON,w		; if 1 return to 10
	xorlw	D'01'
	btfsc	STATUS,Z
	goto 	RETURN_10_FLSH		
	decf	FLASHCON,f		; decrease to next value 
	goto	WRI_FLASHCON_SET; store value in EEPROM
RETURN_10_FLSH
	movlw	D'10'			; return to 10
	movwf	FLASHCON		; load 10 
	goto	WRI_FLASHCON_SET

BY_DN_MODE5
	movf	MODE,w
	xorlw	B'00000110'		; mode 6 and trigger mode Mode,7 clear
	btfss	STATUS,Z
	goto	END_SW
; decrease AVERAG to 0 return to 10
	movf	AVERAG,w		; if 1 return to 10
	xorlw	D'01'
	btfsc	STATUS,Z
	goto 	RETURN_10		
	decf	AVERAG,f		; decrease to next value 
	goto	WRI_AVG_SET		; store value in EEPROM
RETURN_10
	movlw	D'10'			; return to 10
	movwf	AVERAG			; load 10 
	goto	WRI_AVG_SET
	
MODE_DSP
; shows current mode on display

	incf	MODE,f			; mode switch increases by 1 each time
	movf	MODE,w
	andlw	B'01111111'		; remove ms bit (TRIGgered or GENerated flag)
	sublw	D'6'
		
	btfss	STATUS,C		; when >6 set at 0
	goto	CLEAR_MODE
	goto	DRV_MODE_DSP
CLEAR_MODE
	movf	MODE,w
	andlw	B'10000000'		; keep ms bit
	movwf	MODE			; clear bits 0-6
	bsf		GEN_FLG,1		; set to add delay after display update
	goto	CYCLE2

DRV_MODE_DSP
; display 
; mode 0 = trigger or generator (bit 7 selects gen when set)
; mode 1 = trigger or generator select	(EEPROM3) (MODE)
; mode 2 = +/- trigger (EEPROM4)(OPTIONS)set bit 6 for rising edge store and place in OPTION_REG
; mode 3 = pulses/rev (EEPROM1)(DIV_SET)
; mode 4 = flash fixed or automatic EEPROM7, FLASHCON
; mode 5 = FLASH_ON period (EEPROM2)
; mode 6 = averaging value EEPROM8, AVERAG
	
CK_MODE1
; mode 1 = trigger or generator select	(EEPROM3) (MODE)
	movf	MODE,w
	andlw	B'01111111'	; 
	xorlw	H'01'		; if a 1 then mode 1
	btfss	STATUS,Z
	goto	CK_MODE2

; write trigger or generator, then selected (Gen or Trig)
		
	movlw	H'80'		; address line 1
	call	LOAD

	movlw	A'T'
	call	DRV_LCD
	movlw	A'r'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'g'
	call	DRV_LCD
	movlw	A'/'
	call	DRV_LCD
	movlw	A'G'
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	A'n'
	call	DRV_LCD
	movlw	D'8'
	call	SPACEX

	movlw	H'C0'		; address line 2
	call	LOAD
	btfss	MODE,7	; if set then generator
	goto	TRIG_WRI
	movlw	A'G'
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	A'n'
	call	DRV_LCD
	goto	FILL_DISP
TRIG_WRI
	movlw	A'T'
	call	DRV_LCD
	movlw	A'r'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'g'
	call	DRV_LCD
FILL_DISP
	movlw	D'13'
	call	SPACEX
	goto	END_SW

CK_MODE2
; mode 2 = +/- trigger (EEPROM4)(OPTIONS)set bit 6 for rising edge store and place in OPTION_REG
	movf	MODE,w
	andlw	B'01111111'	; 
	xorlw	H'02'		; if a 2 then mode 2
	btfss	STATUS,Z	; if mode 2
	goto	CK_MODE3
	btfsc	MODE,7		; if set (Gen mode)bypass mode 2 
	goto	MODE_DSP	; increment mode

	movlw	H'80'		; address line 1
	call	LOAD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'd'
	call	DRV_LCD
	movlw	A'g'
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	D'12'
	call	SPACEX
	movlw	H'C0'		; address line 2
	call	LOAD
	btfss	OPTIONS,6	; if set then rising edge, clear falling edge
	goto	FALLING
	movlw	A'R'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A's'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'n'
	call	DRV_LCD
	movlw	A'g'
	call	DRV_LCD
	call	SPACE1
	movlw	H'00'		; up arrow
	call	DRV_LCD
	goto	END2
FALLING
	movlw	A'F'
	call	DRV_LCD
	movlw	A'a'
	call	DRV_LCD
	movlw	A'l'
	call	DRV_LCD
	movlw	A'l'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'n'
	call	DRV_LCD
	movlw	A'g'
	call	DRV_LCD
	call	SPACE1
	movlw	H'01'		; down arrow
	call	DRV_LCD

END2
	movlw	D'9'
	call	SPACEX
	goto	END_SW

CK_MODE3
; mode 3 division
	movf	MODE,w
	andlw	B'01111111'	; 
	xorlw	H'03'		; if a 3 then mode 3
	btfss	STATUS,Z	; if mode 3
	goto	CK_MODE4
	btfsc	MODE,7		; if set (Gen mode)bypass mode 3 
	goto	MODE_DSP	; increment mode

	movlw	H'80'		; address line 1
	call	LOAD
	movlw	A'D'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'v'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A's'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'o'
	call	DRV_LCD
	movlw	A'n'
	call	DRV_LCD
	movlw	D'8'
	call	SPACEX

	movlw	H'C0'		; address line 2
	call	LOAD
	movlw	A'/'
	call	DRV_LCD
	movf	DIV_SET,w
	btfss	STATUS,Z	; division setting if zero then 0.5p/rev
	goto	NO_0.5
	movlw	A'0'
	call	DRV_LCD
	movlw	A'.'		; dp
	call	DRV_LCD
	movlw	A'5'
	call	DRV_LCD
	goto	END_3
NO_0.5
	movf	DIV_SET,w	; get division ratio
	iorlw	H'30'		; convert to ASCII
	call	DRV_LCD
	call	SPACE1
	call	SPACE1
END_3
	movlw	D'12'
	call	SPACEX
	goto	END_SW

CK_MODE4
; mode 4 = flash fixed or auto (EEPROM7)(FLASHCON,7)
	movf	MODE,w
	andlw	B'01111111'	; 
	xorlw	H'04'		; if a 4 then mode 4
	btfss	STATUS,Z	; if mode 4
	goto	CK_MODE5

	movlw	H'80'		; address line 1
	call	LOAD
	movlw	A'F'
	call	DRV_LCD
	movlw	A'l'
	call	DRV_LCD
	movlw	A'a'
	call	DRV_LCD
	movlw	A's'
	call	DRV_LCD
	movlw	A'h'
	call	DRV_LCD
	call	SPACE1
	movlw	A'M'
	call	DRV_LCD
	movlw	A'o'
	call	DRV_LCD
	movlw	A'd'
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	D'6'
	call	SPACEX

	movlw	H'C0'		; address line 2
	call	LOAD
	btfsc	FLASHCON,7	; if clear then auto
	goto	FLASH1
	movlw	A'A'
	call	DRV_LCD
	movlw	A'u'
	call	DRV_LCD
	movlw	A't'
	call	DRV_LCD
	movlw	A'o'
	call	DRV_LCD
	movlw	A'm'
	call	DRV_LCD
	movlw	A'a'
	call	DRV_LCD
	movlw	A't'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'c'
	call	DRV_LCD
	goto	END3
FLASH1
	movlw	A'F'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'x'
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	A'd'
	call	DRV_LCD
	
	movlw	D'4'
	call	SPACEX
END3
	movlw	D'9'
	call	SPACEX
	goto	END_SW

CK_MODE5
; flash period
	movf	MODE,w
	andlw	B'01111111'	; 
	xorlw	H'05'		; if a 5 then mode 5
	btfss	STATUS,Z	; if mode 5
	goto	CK_MODE6
	movlw	H'80'		; address line 1
	call	LOAD
	movlw	A'F'
	call	DRV_LCD
	movlw	A'l'
	call	DRV_LCD
	movlw	A'a'
	call	DRV_LCD
	movlw	A's'
	call	DRV_LCD
	movlw	A'h'
	call	DRV_LCD
	call	SPACE1
	movlw	A'P'
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	A'r'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'o'
	call	DRV_LCD
	movlw	A'd'
	call	DRV_LCD
	movlw	D'4'
	call	SPACEX

	movlw	H'C0'		; address line 2
	call	LOAD

; if fixed show us or ms values
	btfss	FLASHCON,7
	goto	AUTO_VAL

; multiply flash period x 256 for micro seconds display
	movlw	H'FF'
	movwf	AARGB0
	movf	FLASH_ON,w
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply
; add FLASH_ON to make it x 256
	movf	FLASH_ON,w
	addwf	AARGB1,f
	btfsc	STATUS,C	; if carry increment ms byte
	incf	AARGB0,f
		
; add D325 for initial 32.5us
	movlw	H'45'		; add H'145'
	addwf	AARGB1,f
	btfsc	STATUS,C	; if carry increment ms byte
	incf	AARGB0,f
	incf	AARGB0,f	; add H'1'
	clrf	BIN_2		; binary ms byte cleared
	movf	AARGB0,w	; ms of multiply
	movwf	BIN_1
	movf	AARGB1,w	; ls of multiply
	movwf	BIN_0
	call	BCD			; convert to BCD

; test OUT_3
	movf	OUT3,w		; 
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADF3		; not zero so load value
; leading is zero
	movlw	D'2'
	call	SPACEX
	bcf		EDGE_FG,7	; clear to indicate microseconds (us)
; test OUT_4
	movf	OUT4,w		; 
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADF4		; not zero so load value
; leading is zero
	call	SPACE1 

; test OUT_5
	movf	OUT5,w		; 
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADF5		; not zero so load value
; leading is zero
	call	SPACE1 
	goto	LOADF6		; not zero so load value

LOADF3
	movf	OUT3,w
	call	DRV_LCD	
	movlw	A'.'
	call	DRV_LCD
	bsf		EDGE_FG,7	; set to indicate ms
LOADF4
	movf	OUT4,w
	call	DRV_LCD	
LOADF5
	movf	OUT5,w
	call	DRV_LCD	
LOADF6
	btfsc	EDGE_FG,7	; if set then ms
	goto	MS	
	movf	OUT6,w
	call	DRV_LCD
	movlw	H'E4'	; u symbol
	call	DRV_LCD
DRV_S
	movlw	A's'	; seconds
	call	DRV_LCD
	movlw	D'10'
	call	SPACEX
	goto	END_SW
MS
	movlw	A'm'	; m (milli)
	call	DRV_LCD
	goto	DRV_S

AUTO_VAL
	movf	FLASHCON,w
	xorlw	D'10'	; if 10 the write 10
	btfsc	STATUS,Z
	goto	WRI_10
	call	SPACE1
	movf	FLASHCON,w
	andlw	B'0111111'	; remove ms bit as could be set
	iorlw	H'30'		; convert to ASCII
	call	DRV_LCD
PERC
	movlw	A'%'		; percent
	call	DRV_LCD
	movlw	D'14'
	call	SPACEX
	goto	END_SW
WRI_10
	movlw	A'1'
	call	DRV_LCD
	movlw	A'0'
	call	DRV_LCD
	goto	PERC

CK_MODE6
; averaging

	movf	MODE,w
	andlw	B'01111111'	; 
	xorlw	H'06'		; if a 6 then mode 6
	btfss	STATUS,Z	; if mode 6
	goto	CK_MODE6
	btfsc	MODE,7		; if set (Gen mode)bypass mode 6 
	goto	MODE_DSP	; increment mode
	
	movlw	H'80'		; address line 1
	call	LOAD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'v'
	call	DRV_LCD
	movlw	A'e'
	call	DRV_LCD
	movlw	A'r'
	call	DRV_LCD
	movlw	A'a'
	call	DRV_LCD
	movlw	A'g'
	call	DRV_LCD
	movlw	A'i'
	call	DRV_LCD
	movlw	A'n'
	call	DRV_LCD
	movlw	A'g'
	call	DRV_LCD
	movlw	D'7'
	call	SPACEX
	movlw	H'C0'		; address line 2
	call	LOAD

	clrf	BIN_2		; binary ms byte cleared
	clrf	BIN_1
	movf	AVERAG,w	; average value
	movwf	BIN_0
	call	BCD			; convert to BCD

	movf	OUT5,w		; 
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADA5		; not zero so load value
; leading is zero
	call	SPACE1 
	movf	OUT6,w	
	xorlw	H'30'		; is it 0
	btfss	STATUS,Z
	goto	LOADA6		; not zero so load value
; leading is zero
	call	SPACE1 
	goto	LOADA7
LOADA5
	movf	OUT5,w
	call	DRV_LCD	
LOADA6
	movf	OUT6,w
	call	DRV_LCD	
LOADA7
	movf	OUT7,w
	call	DRV_LCD	
	movlw	D'13'
	call	SPACEX
	
;	clrf	AV_CNT
	bsf		EDGE_FG,1	; set when new display 
	bsf		EDGE_FG,2

END_SW
; switch delay

	call	DELAY_SWITCH	; switch delay
	goto	CYCLE
 
; ***************************************************************
; subroutines


; subroutine to wait for conversion

ACQUIRE_AD
; wait for >20us
	movlw	D'50'
	movwf	STORE3
LOOP_CONV
	decfsz	STORE3,f	; decrease 
	goto	LOOP_CONV	
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return


; add space in display

SPACEX
	movwf	STORE3
SP_CONT	
	movlw	H'20'		; space
	call	DRV_LCD
	decfsz	STORE3,f
	goto	SP_CONT		; space continue
	return

SPACE1
	movlw	H'20'		; space
	call	DRV_LCD
	return

; switch delay loop
DELAY_SWITCH
	movlw	H'08'
DELAY_SWITCH1
	movwf	STORE3
LOOP_SWITCH
	movlw	H'FF'		; delay 
	call	DELAYX
	decfsz	STORE3,f	; next store3
	goto	LOOP_SWITCH	
	return

; delay loop 

DELAYms
	movlw	D'23'		; delay value
DELAYX
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	H'B0'
DELDSP
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8		; decrease till STORE1 is zero
	return

; initialise display

INIT_LC
	movlw	B'00001001'	; initialise module
	movwf	PORTA
	nop
	bsf		PORTB,7		; enable high
	nop
	bcf		PORTB,7		; low
	return

; preload display commands (4-bit) 

LOAD
	movwf	D_STO		; store data	
	movf	D_STO,w
	andlw	H'F0'		; get upper bits
; place display commands in portB
	movwf	PORTA_STO
	call	COMMAND
	goto	SECOND_BYTE
COMMAND
	bcf		PORTA,0
	bcf		PORTA,1
	bcf		PORTA,2
	bcf		PORTA,3
	btfsc	PORTA_STO,7
	bsf		PORTA,1	
	btfsc	PORTA_STO,6
	bsf		PORTA,2	
	btfsc	PORTA_STO,5
	bsf		PORTA,0	
	btfsc	PORTA_STO,4
	bsf		PORTA,3	

	bcf		PORTB,6		; register select low
	nop
	bsf		PORTB,7		; enable set
	nop
	bcf		PORTB,7		; enable clear
	return
SECOND_BYTE
	swapf	D_STO,w
	andlw	H'F0'		; get lower bits
; place display commands in PORTA
	movwf	PORTA_STO
	call	COMMAND
	goto	BUS_CK		; check busy flag
	
; driving the LCD module with display data

DRV_LCD	
	movwf	D_STO		; store data
	andlw	H'F0'		; upper bits
	movwf	PORTA_STO
	call	COMMAND2
	goto	SECOND_BYTE2
COMMAND2
	bcf		PORTA,0
	bcf		PORTA,1
	bcf		PORTA,2
	bcf		PORTA,3
	btfsc	PORTA_STO,7
	bsf		PORTA,1	
	btfsc	PORTA_STO,6
	bsf		PORTA,2	
	btfsc	PORTA_STO,5
	bsf		PORTA,0	
	btfsc	PORTA_STO,4
	bsf		PORTA,3	
	bsf		PORTB,6		; register select
	bsf		PORTB,7		; enable high
	nop
	bcf		PORTB,7		; enable low
	return
SECOND_BYTE2
	swapf	D_STO,w
	andlw	H'F0'		; lower bits
	movwf	PORTA_STO
	call	COMMAND2

BUS_CK
	movlw 	D'2'		; was 30
	movwf	STORE1		; delay values
	movlw	D'255'		; delay for busy flag to clear
	goto	DELDSP

; subroutine to read EEPROM memory 

EEREAD
	bcf		STATUS,RP1	; select bank 0
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE
EEWRITE	
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
	bsf 	INTCON,GIE	; enable interrupts
WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	return				; value written 

; Subroutine to convert from 24-bit binary (D16777216) to 4-digit BCD (packed)
; Binary value is in BIN0 BIN1 & BIN2. Ms byte is BIN2
; Result in BCD is in BCD0, BCD1, BCD2 & BCD3.  
; BCD0 is MSB, BCD2 is LSB

BCD	
	bcf		STATUS,C	; clear carry bit
	movlw	D'24'
	movwf	CNT_24		; 24 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
	clrf	BCD_2
	clrf	BCD_3
	
LOOPBCD
	rlf		BIN_0,f
	rlf		BIN_1,f
	rlf		BIN_2,f		; LSB shift left binary registers
	rlf		BCD_3,f
	rlf		BCD_2,f		; LSB shift left BCD registers
	rlf		BCD_1,f
	rlf		BCD_0,f

	decfsz	CNT_24,f	; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust

; convert to unpacked ASCII
; convert BCD0,1,2,3 to OUT0,1,2,3,4,5,6,7
; BCD3 to OUT6,7
	movf	BCD_3,w		; ls decimal
	andlw	H'0F'		; 
	addlw	H'30'		; convert to ASCII
	movwf	OUT7
	swapf	BCD_3,w 	; decimal value
	andlw	H'0F'
	addlw	H'30'		; convert to ASCII
	movwf	OUT6
; BCD2 to OUT4,5
	movf	BCD_2,w		; decimal
	andlw	H'0F'		;
	addlw	H'30'		; convert to ASCII
	movwf	OUT5
	swapf	BCD_2,w 	; decimal value
	andlw	H'0F'
	addlw	H'30'		; convert to ASCII
	movwf	OUT4
; BCD1 to OUT2,3
	movf	BCD_1,w		; decimal
	andlw	H'0F'		; 
	addlw	H'30'		; convert to ASCII
	movwf	OUT3
	swapf	BCD_1,w 	; decimal value
	andlw	H'0F'
	addlw	H'30'		; convert to ASCII
	movwf	OUT2
; BCD0 to OUT0,1
	movf	BCD_0,w		; ms decimal
	andlw	H'0F'		; 
	addlw	H'30'		; convert to ASCII
	movwf	OUT1
	swapf	BCD_0,w 	; decimal value
	andlw	H'0F'
	addlw	H'30'		; convert to ASCII
	movwf	OUT0

	return				; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ
	movlw	BCD_3		; BCD LSB address
	movwf	FSR			; pointer for BCD3
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_2		; BCD address
	movwf	FSR			; pointer for BCD2
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_1
	movwf	FSR
	call 	ADJBCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR			; pointer for BCD_0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD
	movlw	H'03'		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-3)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	H'30'		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return				; end subroutine


;**********************************************************************************************

; 8 x 8 multiply

EIGHTEIGHT	    CLRF    AARGB1          ; clear partial product
UMUL0808L        
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
                MOVF    AARGB0,W

LOOPUM0808A
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    LUM0808NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808A

                CLRF    AARGB0
                RETLW   H'00'

LUM0808NAP
                BCF     STATUS,C
                GOTO    LUM0808NA

LOOPUM0808
                RRF     BARGB0, F
                BTFSC   STATUS,C
                ADDWF   AARGB0, F
LUM0808NA       RRF    	AARGB0, F
                RRF    	AARGB1, F
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808

                return             


; 32/32 Bit Unsigned Fixed Point Divide 32/32 -> 32.32
; Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point divisor in BARGB0, BARGB1, BARGB2, BARGB3
; Use: CALL FXD3232U
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point remainder in REMB0, REMB1, REMB2, REMB3
; Result: AARG, REM <-- AARG / BARG
; Max Timing: 4+1025+2 = 1031 clks
; Max Timing: 4+981+2 = 987 clks
; PM: 4+359+1 = 364 DM: 13

	org	H'A00'	; page 1
FXD3232U
	CLRF 	REMB0
	CLRF	REMB1
	CLRF 	REMB2
	CLRF 	REMB3
	call	UDIV3232L
	bcf		PCLATH,3	; page 0
	return

UDIV3232L 
; Max Timing: 24+6*32+31+31+6*32+31+31+6*32+31+31+6*32+31+16 = 1025 clks
; Min Timing: 24+6*31+30+30+6*31+30+30+6*31+30+30+6*31+30+3 = 981 clks
; PM: 359 DM: 13
	CLRF 	TEMP
	RLF 	AARGB0,W
	RLF 	REMB3,F
	MOVF 	BARGB3,W
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	RLF 	AARGB0,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232A 
	RLF 	AARGB0,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22LA
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LA

UADD22LA 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LA 
	RLF		AARGB0,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232A
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22L8
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L8

UADD22L8 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L8 
	RLF 	AARGB1,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232B 
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB1,0
	GOTO 	UADD22LB
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LB

UADD22LB 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LB 
	RLF 	AARGB1,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232B
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS	AARGB1,0
	GOTO 	UADD22L16
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L16

UADD22L16 	
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF	TEMP,F

UOK22L16
	RLF		AARGB2,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232C 
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22LC
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LC

UADD22LC
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LC 
	RLF 	AARGB2,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232C
	RLF 	AARGB3,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22L24
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L24

UADD22L24 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L24 
	RLF 	AARGB3,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232D 
    RLF  	AARGB3,W
    RLF    	REMB3,F
    RLF    	REMB2,F
    RLF    	REMB1,F
    RLF    	REMB0,F
    RLF    	TEMP,F
    MOVF   	BARGB3,W
 	BTFSS  	AARGB3,0
   	GOTO   	UADD22LD

    SUBWF  	REMB3,F
  	MOVF   	BARGB2,W
    BTFSS   STATUS,C
    INCFSZ  BARGB2,W
    SUBWF   REMB2,F
    MOVF    BARGB1,W
    BTFSS   STATUS,C
    INCFSZ  BARGB1,W
    SUBWF   REMB1,F
    MOVF    BARGB0,W
    BTFSS   STATUS,C
    INCFSZ  BARGB0,W
    SUBWF   REMB0,F
    CLRW
    BTFSS   STATUS,C
    MOVLW   H'1'
    SUBWF   TEMP,F
    GOTO    UOK22LD

UADD22LD      
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F
    CLRW
    BTFSC   STATUS,C
    MOVLW   H'1'
    ADDWF   TEMP,F
        
UOK22LD
	RLF     AARGB3,F

   	DECFSZ  LOOPCOUNT, F
    GOTO    LOOPU3232D

    BTFSC   AARGB3,0
    GOTO    UOK22L
    MOVF    BARGB3,W
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F

UOK22L

	RETURN
; *********************************************************

; character generation for trigger arrows	
CHARACTER_GEN
	bcf		PCLATH,3	; page 0 call
; character 1, up arrow
	movlw	B'01000000'	; address 0
	call	LOAD		; character gen 1 
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00001110'
	call	DRV_LCD
	movlw	B'00010101'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
; character 2
; down arrow
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	movlw	B'00010101'
	call	DRV_LCD
	movlw	B'00001110'
	call	DRV_LCD
	movlw	B'00000100'
	call	DRV_LCD
	return

; ******************************************************************************************************

 end
