;
; One Shot Timer revised for PIC16F628A
; time durations from 
; 0.1 to 9.9 seconds x 1 range (0.1 second steps)
; 1 to 99 seconds in x10 range (1 second steps)
; 10 seconds to 990 seconds in x 100 range (10 second steps)
; One shot (single timeout) and Alternate mode (switches on and of at the timer set rate)
; input options are + or - edge triggering for one shot mode
; + or ground level operation for alternate mode
; Alternative mode can be also a duty cycle output where the 1's BCD switch sets the
; off time and the 10's switch the on time for the relay.
; If the 1's switch is zero relay will be always on
; If the 10's switch is at zero the relay will be always off
; both 0's will set relay as off.
; Duty cycle mode initiated with momentary connection of RB4-RB7 to ground.
; Easily done by setting the 1's switch to a 7 and then momentarily connecting the TP1 pin
; to ground. Each time this is done it alternates between the duty cycle operation
; or standard alternate 50% duty. Mode is remembered in EEPROM.

; CPU configuration
; 	
	list P=16F628A
	#include p16f628A.inc

	__config _XT_OSC & _WDT_OFF & _PWRTE_ON & _BOREN_ON & _MCLRE_ON & _LVP_OFF

; Define variables at memory locations

; EEPROM DATA

EEPROM0		equ	H'00'	; non-volatile storage for pwm option

; RAM

TMR_RUN		equ	H'20'	; timer running
UPDATE1		equ	H'21'	; 1 second counter ls byte
UPDATE2		equ	H'22'	; 1 second counter ms byte

TME_DUR		equ	H'26'	; timer duration	
TME_ELP		equ	H'27'	; timer elapsed
TMS_HRD		equ	H'28'	; times one hundred delay
TEMP_1		equ	H'29'	; switch temporary decoder  	
STORE1		equ	H'2A'	; delay store
STORE2		equ	H'2B'	; delay store
STORE3		equ	H'2C'	; delay store
PWM_REG		equ	H'2D'	; alternate PWM mode flag
TME_PWM		equ	H'2E'	; PWM timing
T_STO		equ	H'2F'	; timing storage

H_BYTE		equ	H'30'	; high byte
L_BYTE		equ	H'31'	; low byte
REG_0		equ	H'32'	; MSD of 5-digit BCD in right hand (low) nibble
REG_1		equ	H'33'	; BCD number (packed BCD)
REG_2		equ	H'34'	; BCD low order numbers
H_TEMP		equ	H'35'	; temporary register of high order binary result
L_TEMP		equ	H'36'	; temporary register of low order binary result	

; All banks ram
STATUS_TMP 	equ H'70'	; temp storage for status during interrupt
W_TMP		equ	H'71'	; temporary storage for w during interrupt

; preprogram EEPROM DATA
	
	ORG     H'2100'		; start of address
	DE		D'00'		; preset 0

; define reset and interrupt vector start addresses

	org		0	  		; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4			; interrupt vector 0004h, start interrupt routine here

;******************************************************************************************* 


; INTERRUPT

; start interrupt by saving w and status registers before altered by interrupt routine

	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0

	btfsc	INTCON,T0IF	; TMRO overflow interrupt flag 
	goto	TMR			; TMRO overflow so increase millisecond counter
	goto	RECLAIM		; end of interrupt reclaim w and status

; increment update counter 

TMR	movf	TMR0,w		; timer value	
						; freq is 4MHz/4/4/(256(-8+2))=1kHz
	nop
	nop					; synchronise counter read with divide by 4 ie add two cycles
	addlw	D'8'		; add to timer register and takes 2 cycles to start counting
	movwf	TMR0		; place in timer (the two cycles plus 2-cycle delay = 4cycles)
	
	bcf		INTCON,T0IF	; clear TMRO interrupt flag

; check for alternate mode

	btfss	PORTA,0		; one shot or alternate mode input
	goto	ALTRCK
	goto	ONE_EDG		; *T_RUN* check edge again and timer run check

ALTRCK
	btfss	PORTA,1		; if high then timer runs when input at RB0 is high
	goto	LW_RN		; run on low
	btfss	PORTB,0		; if high then run, if low stop
	goto	ONE_S		; end of timer
	goto	T_RUN		; check if timer started
LW_RN	
	btfsc	PORTB,0		; if low then run, if high stop
	goto	ONE_S		; end of timer
	goto	T_RUN		; has timer started *	

; oneshot timer check for edge and start/ restart timer

ONE_EDG 
	movf	TMR_RUN,w	; is timer already running
	btfsc 	STATUS,Z	; timer off if zero
	goto	EDGEL		; check RB0 for edge change 
	btfss	INTCON,INTF	; Intcon INTF flag set on edge detect on RB0 (see option register)
	goto	ZRO_TM		; check for zero time setting
	goto	EDGEL		; check edge changes

; check for timer running

T_RUN
	movf	TMR_RUN,w	; timer running flag
	btfsc	STATUS,Z	; if zero then timer off
	goto	EDGEL		; check RB0 for edge or level change

; check for zero time duration

ZRO_TM
	btfsc	PORTA,0		; check alternate/oneshott mode
	goto	ZRO_T		; standard mode 
	btfss	PWM_REG,0	; if set then pwm /alternate mode
	goto	ZRO_T		; standard timer

; check for zero on either duty cycle timer

	movf	TME_PWM,w
	btfss	STATUS,Z	; if zero, off
	goto	CK_N		; check nibble timing
	bcf		PORTA,3		; relay off
	goto 	RECLAIM

CK_N
	movf	TME_PWM,w	; duty cycle timer
	andlw	H'F0'		; ms nibble is time relay is on
	btfss	STATUS,Z	; if zero relay stays off
	goto	CK_LS
CLR3
	bcf		PORTA,3		; relay off
	goto	RECLAIM

CK_LS
	movf	TME_PWM,w	; duty cycle timer
	andlw	H'0F'		; ls nibble is time relay is off
	btfss	STATUS,Z	; if zero relay stays on
	goto	TMER
	bsf		PORTA,3
	goto	RECLAIM

ZRO_T
	movf	TME_DUR,w
	btfsc	STATUS,Z	; if zero timeout
	goto	EOT			; end of timing

; increase 0.1 second counter when timer running
	
TMER
	incfsz	UPDATE1,f	; increase counter 1 skip if overflowed
	goto 	CHK_UPD		; goto check update time
	incf	UPDATE2,f	; increase counter 2 when counter 1 overflows

CHK_UPD	
	movf	UPDATE2,w	; MS Byte of update counter to w
	xorlw	H'00'		; 00 compare MS Byte counter  (100 decimal for 0.1 second) 
	btfss	STATUS,Z	; skip if equal
	goto	RECLAIM		; not, so continue
	movf	UPDATE1,w	; LS Byte of update counter
	xorlw	H'64'		; 64=100 compare LS byte counter with (100) 
	btfss	STATUS,Z	; skip if equal
	goto	RECLAIM		; not 0.1 second yet
	clrf	UPDATE1		; clear timer counter 1
	clrf	UPDATE2		; and timer 2
	incf	TME_ELP,f	; elapsed time

; check if standard or pwm timing when alternate mode
	
	btfsc	PORTA,0		; check alternate/oneshott mode
	goto	STANDRD
	btfss	PWM_REG,0	; if set then pwm /alternate mode
	goto	STANDRD		; standard timer
	btfss	PORTA,3
	goto	LOW_3		; use TME_PWM low nibbles
	swapf	TME_PWM,w	; use high nibbles
PWM_TM
	andlw	H'0F'		; delete high nibbles
	movwf	T_STO		; time store
	incf	TME_ELP,w
	subwf	T_STO,w
	goto	CK_VAL
LOW_3
	movf	TME_PWM,w
	goto	PWM_TM

STANDRD
	incf	TME_ELP,w
	subwf	TME_DUR,w	; compare elapsed time with duration setting
CK_VAL
	btfsc	STATUS,C
	goto	RECLAIM	
	decfsz	TMS_HRD,f	; if zero then end of timing period
	goto	MORE_T		; more time

; end of timing
	
EOT
	bcf		INTCON,INTF	; clear bit to wait for next edge

; check if oneshot or alternate mode

	btfss	PORTA,0		; one shot or alternate mode input
	goto	ALTERN
	
ONE_S
	clrf	TMR_RUN		; timer running flag off
	bcf		PORTA,3		; relay and LED off
	goto	RECLAIM

ALTERN
	btfss	PWM_REG,0
	goto	RL_ON
	movf	TME_PWM,w
	btfsc	STATUS,Z	; if zero end	
	goto	ONE_S	

RL_ON	
	btfss	PORTA,3		; relay on or off
	goto	SET_RLY
	bcf		PORTA,3		; relay off
	goto	STRT2		; timer start again
SET_RLY	
	movf	TME_DUR,w
	btfsc	STATUS,Z	; if zero timeout
	goto	ONE_S		; end of timing
	bsf		PORTA,3		; relay on
	goto	STRT2		; timer start again

; x ten, x100 range
	
MORE_T
	clrf	TME_ELP		; elapsed time cleared
	goto	RECLAIM

; check for edge or level change

EDGEL
	btfss	PORTA,0		; one shot or alternate mode input
	goto	ALTERCK
	goto	EDGE2		; check for edge change on RB0

ALTERCK
	btfss	PORTA,1		; if high then timer runs on high
	goto	LOW_RN		; run on low
	btfss	PORTB,0		; if high then run, if low stop
	goto	ONE_S		; end of timer
	movlw	H'01'
	call	DELAY1		; deglitch
	btfss	PORTB,0		; if high then run, if low stop
	goto	ONE_S		; end of timer
	goto	STRT		; start of timer
LOW_RN
	btfsc	PORTB,0		; if low then run, if high stop
	goto	ONE_S		; end of timer
	movlw	H'01'
	call	DELAY1		; deglitch
	btfsc	PORTB,0		; if low then run, if high stop
	goto	ONE_S
	goto	STRT		; start of timer

EDGE2
	btfss	INTCON,INTF	; Intcon INTF flag set on edge detect on RB0 (see option register)
	goto	RECLAIM		; not so bypass
	bcf		INTCON,INTF	; clear bit to wait for next edge

; edge detected. Deglitch the edge
	
	movlw	H'01'
	call	DELAY1		; delay for edge check if still low or high after delay
	btfsc	PORTA,1		; if porta,1 is low then deglitch 
	goto	CK_HI		; rising trigger 
	
	btfsc	PORTB,0		; portb,0 should be low if there was a low edge
	goto	NO_TIM		; portb,0 high again so was a glitch
	goto	STRT

CK_HI
	btfss	PORTB,0		; portb,0 should be high if there was a high edge
	goto	NO_TIM

STRT
	bsf		TMR_RUN,0	; timer started

; relay on only if time duration not zero

; check for zero time duration

	btfsc	PORTA,0		; check alternate/oneshott mode
	goto	ZRO_S		; standard mode 
	btfss	PWM_REG,0	; if set then pwm /alternate mode
	goto	ZRO_S		; standard timer

; pwm timer
	movf	TME_PWM,w
	btfsc	STATUS,Z	; if zero end	
	goto	ONE_S		
	goto	S_RLY
	
ZRO_S
	movf	TME_DUR,w
	btfsc	STATUS,Z	; if zero timeout
	goto	STRT2		; bypass relay on

S_RLY
	bsf		PORTA,3		; relay and LED on
STRT2
	clrf	UPDATE1
	clrf	UPDATE2		; 0.1 second counters cleared
	clrf	TME_ELP		; elapsed time cleared
	
; check whether x 1, x 10 or x 100 time selected. Equates to x0.1, x1 and x10 on PC board 
	
	bsf		PORTA,2		; set high
 
	bsf		STATUS,RP0	; select memory bank 1
	bsf		TRISA,2		; RA2 input
	bcf		STATUS,RP0	; select memory bank 0
	
	btfss	PORTA,2	; if low then is tied low
	goto	TMS_1	; times one

; Drive RA2 low

	bsf		STATUS,RP0	; select memory bank 1
	bcf		TRISA,2		; RA2 output
	bcf		STATUS,RP0	; select memory bank 0

	bcf		PORTA,2		; set low

	bsf		STATUS,RP0	; select memory bank 1
	bsf		TRISA,2		; RA2 input
	bcf		STATUS,RP0	; select memory bank 0

	btfsc	PORTA,2		; if high then is tied high
	goto	TMS_100		; times hundred

	movlw 	H'A'
	movwf	TMS_HRD		; times 10 default
	goto	RECLAIM
TMS_1
	movlw	D'1'		; 
	movwf	TMS_HRD
	goto	RECLAIM
TMS_100	
	movlw	H'64'		; 100
	movwf	TMS_HRD
	goto	RECLAIM

NO_TIM	nop

; end of interrupt reclaim w and status 

RECLAIM
	bsf		STATUS,RP0	; select memory bank 1
	bcf		TRISA,2		; RA2 output
	bcf		STATUS,RP0	; select memory bank 0
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

;********************************************************************************************** 
  	
; RESET	
	
; Set ports A & B

MAIN
	clrf	TMR_RUN		; timer running flag
	movlw	H'07'		; comparators off
	movwf	CMCON		; I/O pins
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'11111111'	; RB0 - RB7 inputs
	movwf	TRISB		; port B data direction register
	movlw	B'01000001'	; rising edge for RB0
	movwf	OPTION_REG	; TMRO prescaler 4 division, PORTB pullups enabled
	movlw   B'00010011'	; RA0,1 & RA4 inputs, RA2, RA3 ouput
	movwf   TRISA		; A port data direction register
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'00000000'
	movwf	PORTA		; portA outputs low 

	call	DELAY		; power up delay

	bcf		INTCON,INTF	; clear bit to wait for next edge

; check PWM/ alternate mode

RD_PWM
	movf	PORTB,w		; check portB
	andlw	H'F0'		; check RB4-RB7
	btfss	STATUS,Z	; if all low then PWM alternate mode change	
	goto	EEPRD		; not all low so read eeprom mode

	call	DELAY
	movf	PORTB,w		; check portB
	andlw	H'F0'		; check RB4-RB7
	btfss	STATUS,Z	; if all low then PWM alternate mode change	
	goto	EEPRD		; not all low so read eeprom mode

	call	DELAY
	movf	PORTB,w		; check portB
	andlw	H'F0'		; check RB4-RB7
	btfss	STATUS,Z	; if all low then PWM alternate mode change	
	goto	EEPRD		; not all low so read eeprom mode
SET_PWM
	movlw	EEPROM0
	call	EEREAD
	movwf	PWM_REG		; value ready
	btfss	PWM_REG,0	; if high
	goto	ST_0
	bcf		PWM_REG,0	; clear if set
	goto	EEPWR
ST_0
	bsf		PWM_REG,0	
EEPWR
	movlw	EEPROM0
	call	EEREAD		; sets EEADR
	movf	PWM_REG,w	
	call	EWRITE

; wait till all open again

W_OPN
	movlw	H'1'
	call	DELAY1
	movf	PORTB,w		; check portB
	andlw	H'F0'		; check RB4-RB7
	btfsc	STATUS,Z	; if all low then PWM alternate mode	
	goto	W_OPN		; wait till open
	
EEPRD
	movlw	EEPROM0
	call	EEREAD
	movwf	PWM_REG		; value ready

; check positive or negative edge triggering
	
RD_INP
	btfsc	PORTA,1		; is porta,1 high or low
	goto	HI_EDG

	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; falling edge for RB0
	movwf	OPTION_REG	; TMRO prescaler 4 division, PORTB pullups enabled
	bcf		STATUS,RP0	; select memory bank 0
	goto	RD_SW
	
HI_EDG
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01000001'	; rising edge for RB0
	movwf	OPTION_REG	; TMRO prescaler 4 division, PORTB pullups enabled
	bcf		STATUS,RP0	; select memory bank 0

; read switch inputs 

RD_SW 
	clrf	TEMP_1		; start by clearing TEMP_1
; bit 0
	btfss	PORTB,6		; for ls bit of switches
	bsf		TEMP_1,0	; if low set bit 0 in TEMP_1
; bit 1	
	btfss	PORTB,4		; for bit 1 of switches
	bsf		TEMP_1,1	; if low set bit1 in TEMP_1
; bit 2	
	btfss	PORTB,5		; for bit 2 of switches
	bsf		TEMP_1,2	; if low set bit2 in TEMP_1
; bit 3	
	btfss	PORTB,7		; for bit 3 of switches
	bsf		TEMP_1,3	; if low set bit3 in TEMP_1
; bit 4	
	btfss	PORTB,2		; for bit 4 of switches
	bsf		TEMP_1,4	; if low set bit4 in TEMP_1
; bit 5	
	btfss	PORTB,1		; for bit 5 of switches
	bsf		TEMP_1,5	; if low set bit5 in TEMP_1
; bit 6	
	btfss	PORTA,4		; for bit 6 of switches
	bsf		TEMP_1,6	; if low set bit6 in TEMP_1
; bit 7	
	btfss	PORTB,3		; for bit 7 of switches
	bsf		TEMP_1,7	; if low set bit7 in TEMP_1

	movf	TEMP_1,w	;
	movwf	TME_PWM		; pwm store 
	movwf	REG_2		; BCD value
	call	BCD_BIN		; convert to binary value for counter
	movf	L_BYTE,w	; binary value
	movwf	TME_DUR		; store time duration

; allow interrupts

	bsf		INTCON,T0IE	; set interrupt enable for timer
	bsf		INTCON,GIE	; set global interrupt enable 
	goto	RD_PWM		; read inputs
;
; ********************************************************************************

DELAY
	movlw	H'A'		; ten times delay
DELAY1 
	movwf	STORE3
DELAYms
	movlw	H'AF'		; delay with 4 MHz clock
	movwf	STORE1		; STORE1 is number of loops value
LOOP1
	movlw	H'A0'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP2
	decfsz	STORE2,f
	goto	LOOP2
	decfsz	STORE1,f
	goto	LOOP1		; decrease till STORE1 is zero
	decfsz	STORE3,f
	goto	DELAYms
	return

; subroutine to read EEPROM memory

EEREAD
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bsf		EECON1,RD	; read EEPROM
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP0	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE
EEWRITE	
	bsf		STATUS,RP0	; select bank 1
	movwf	EEDATA		; data register
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
	bsf		INTCON,GIE	; enable interrupts
WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank 0 
	return				; value written 

; subroutine to convert from BCD to binary

BCD_BIN

; clear initial registers as not using full capability of 5-digit BCD to 16-bit binary conversion

	clrf	REG_0		; high BCD number (not using this byte)
	clrf	REG_1		; middle BCD number (not using this byte)

; start conversion

	clrf	H_BYTE		; high order binary result
	movf	REG_0,w
	andlw	H'F'		; mask high nibble
	movwf	L_BYTE 
	call	TIMES_A
	swapf	REG_1,w
	call	TIMES_B
	movf	REG_1,w
	call	TIMES_B
	swapf	REG_2,w
	call	TIMES_B
	movf	REG_2,w
	andlw	H'F'
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf	H_BYTE,f
	return		

TIMES_B
	andlw	H'0F'
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf	H_BYTE,f
TIMES_A
	bcf		STATUS,C
	rlf		L_BYTE,w	; times 2
	movwf	L_TEMP
	rlf		H_BYTE,w
	movwf	H_TEMP
; times 8
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
; to mult by 10
	movf	L_TEMP,w
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf	H_BYTE,f
	movf	H_TEMP,w
	addwf	H_BYTE,f
	return	

	
	end	
