; 
; Uses assembly code for PIC16F88 microcontroller
; A 3-digit oxygen sensor display plus 7-segment bar graph 
; Displays air/fuel ratio on three digit display and also on bargraph. 
; optional S curve display for 0-1V narrowband sensors
; normally uses 0-5V display for wideband linear response

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_ON & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; Define variables at memory locations

EEPROM0		equ	H'00'	; for display brightness threshold (BRIGHT_DIM)
EEPROM1		equ	H'01'	; for display brightness minumum (BRIGHT_MIN)
EEPROM2		equ	H'02'	; dot/bargraph (DOT_BAR)or bargraph centred
EEPROM3		equ	H'03'	; regulator voltage MS byte REG_V0
EEPROM4		equ	H'04'	; regulator voltage LS byte REG_V1
EEPROM5		equ	H'05'	; S curve or linear (S_LIN)
EEPROM6		equ	H'06'	; Upper limit value for 5V (UPPER_LIMA0) ms byte
EEPROM7		equ	H'07'	; Upper limit value for 5V (UPPER_LIMA1) ls byte
EEPROM8		equ	H'08'	; Lower limit value for 0V (LOWER_LIMA0) ms byte
EEPROM9		equ	H'09'	; Lower limit value for 0V (LOWER_LIMA1) ls byte
EEPROMA		equ	H'0A'	; Decimal Point for A value(DECIMALA)
EEPROMB		equ	H'0B'	; Decimal Point for B value(DECIMALB)
EEPROMC		equ	H'0C'	; A or B value (AB_VALUE)
EEPROMD		equ	H'0D'	; Upper limit value for 5V (UPPER_LIMB0) ms byte
EEPROME		equ	H'0E'	; Upper limit value for 5V (UPPER_LIMB1) ls byte
EEPROMF		equ	H'0F'	; Lower limit value for 0V (LOWER_LIMB0) ms byte
EEPROM10	equ	H'10'	; Lower limit value for 0V (LOWER_ LIMB1) ls byte
EEPROM11	equ	H'11'	; invert flag for A bargraph (INVERTA)
EEPROM12	equ	H'12'	; invert flag for B bargraph (INVERTB)

; Bank 0 RAM

DISP1		equ	H'20'	; working storage for Display1 
DISP2		equ	H'21'	; working storage for Display2 
DISP3		equ	H'22'	; working storage for Display3 
DISP4		equ	H'23'	; working storage for Display4
BIN_0		equ	H'24'	; binary value ms byte
BIN_1		equ	H'25'	; binary value ls byte
BCD_0		equ	H'26'	; BCD value MS
BCD_1		equ	H'27'	; 
BCD_2		equ	H'28'	; BCD value LS
TEMP		equ	H'29'	; temporary register
CNT_16		equ	H'2A'	; counter for BCD conversion
BRIGHT		equ	H'2B'	; current brightness level
BRIGHT_DIM	equ	H'2C'	; brightness threshold
BRIGHT_MIN	equ	H'2D'	; brightness minimum 
STORE		equ	H'2E'	; delay counter for A/D conversion
INPUT0		equ	H'2F'	; input value ms
INPUT1		equ	H'30'	; input value ls
TIMER0		equ	H'31'	; timer ms
TIMER1		equ	H'32'	; timer ls
SETTINGS	equ	H'33'	; display settings
FLAG_1		equ	H'34'	; alternate interrupt rate for brightness control
REG_V0		equ	H'35'	; regulator voltage ms
REG_V1		equ	H'36'	; regulator voltage ls
PORTA_STO	equ	H'37' 	; switch closed detect
DOT_BAR		equ	H'38'	; dot or bar
MODE		equ	H'39'	; display mode
DISP1X		equ	H'3A'	; multiplexing storage for Display1 
DISP2X		equ	H'3B'	; multiplexing storage for Display2 
DISP3X		equ	H'3C'	; multiplexing storage for Display3 
DISP4X		equ	H'3D'	; multiplexing storage for Display4
UPDATE0		equ	H'3E'	; update counter
UPDATE1		equ	H'3F'	; digits update counter

UPPER_LIMA0	equ	H'40'	; Upper limit value ms byte A values
UPPER_LIMA1	equ	H'41'	; Upper limit value ls byte
LOWER_LIMA0	equ	H'42'	; Lower limit value ms byte
LOWER_LIMA1	equ	H'43'	; Lower limit value ls byte
S_LIN		equ	H'44'	; S curve or linear
ZRO_FLG		equ	H'45'	; leading zero flag
DECIMALA	equ	H'46'	; decimal point A value
UPPER_LIMB0	equ	H'47'	; Upper limit value ms byte B values
UPPER_LIMB1	equ	H'48'	; Upper limit value ls byte
LOWER_LIMB0	equ	H'49'	; Lower limit value ms byte
LOWER_LIMB1	equ	H'4A'	; Lower limit value ls byte
DECIMALB	equ	H'4B'	; decimal point B value
AB_VALUE	equ	H'4C'	; A or B value
SET_B		equ	H'4D'	; A or B display flag set by S4.
SW_FLG		equ	H'4E'	; switch flag
RATIO		equ	H'4F'	; S curve ratio value
TRAN_FLG 	equ	H'50'	; allow transfer in interrupt flag
NEGATIVE	equ	H'51'	; negative flag 
SPAN0		equ	H'52'	; value span ms byte
SPAN1		equ	H'53'	; value span ls byte
INVERTA		equ	H'54'	; A bargraph invert	flag
INVERTB		equ	H'55'	; B bargraph invert flag
INVERT_AB	equ	H'56'	; A and B invert for bargraph
RATE0		equ	H'57'	; change rate for S1,S2,S3 settup values at startup when pressed
RATE1		equ	H'58'	; change rate for S1,S2,S3 settup values at startup when pressed
		
; math routines
REMB0		equ	H'63'	; remainder
REMB1		equ	H'64'
TEMP1		equ H'65'	; temporary
TEMPD		equ	H'66'
AARGB3		equ	H'67'	; ls of argument A
AARGB2      equ H'68'
AARGB1      equ H'69'
AARGB0      equ H'6A'	; most significant byte of argument A
TEMPB1      equ H'6B'	; temporary
TEMPB0      equ H'6C'
BARGB1      equ H'6D'	; 
BARGB0      equ H'6E'	; most significant byte of argument B
LOOPCOUNT   equ H'6F'  	; loop counter

; All Banks RAM
STATUS_TMP 	equ H'70'	; temp storage for status during interrupt
W_TMP		equ	H'71'	; temporary storage for w during interrupt
FLASH0		equ	H'72'	; flash data address ms byte
FLASH1		equ	H'73'	; flash data address ls byte
DATA0		equ	H'74'	; Flash data ms byte
DATA1		equ	H'75'	; flash data ls byte

; preprogram EEPROM DATA 
	
 	ORG     H'2100'
 	DE		D'200'		; 0, for display brightness threshold (BRIGHT_DIM)
	DE		D'14'		; 1, for display brightness minimum (BRIGHT_MIN)
	DE		H'00'		; 2, DOT_BAR select
	DE		H'1'		; 3, regulator Voltage ms
	DE		H'F4'		; 4, regulator Voltage ls 500 for 5.00V
	DE		H'00'		; 5, S curve or linear 
	DE		H'00'		; 6, Upper limit A value ms byte (22.0)220
	DE		H'DC'		; 7, Upper limit A value ls byte
	DE		H'00'		; 8, Lower limit A value ms byte
	DE		H'4A'		; 9, Lower limit A value ls byte (7.4)

	DE		H'02'		; A, Decimal Point for A value(DECIMALA)
	DE		H'01'		; B, Decimal Point for B value(DECIMALB)
	DE		H'00'		; C, A or B value (AB_VALUE)
	DE		H'00'		; D, Upper limit value (UPPER_LIMB0) ms byte (lambda 1.50)
	DE		H'96'		; E, Upper limit value (UPPER_LIMB1) ls byte
	DE		H'00'		; F, Lower limit value (LOWER_LIMB0) ms byte (lambda 0.51)
	DE		H'33'		; 10,Lower limit value (LOWER_LIMB1) ls byte
	DE		H'00'		; 11,A bargraph invert
	DE		H'00'		; 12,B bargraph invert

; define reset and interrupt vector start addresses

	org		0  			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4			; interrupt vector 0004h, start interrupt routine here
	goto	INTERRUPT	; start of interrupt routine

; **************************************************************************
; lookup table subroutine to get seven segment display data. 
;  			RB76543210
;             fgae.cdb
;				  |-------------------------------- decimal point RB3	
;             01001000 ; for a '0'. other numbers as per below in SEVEN_SEG
; 0 is segment on, 1 is segment off

SEVEN_SEG
	addwf	PCL,f		; add value of display to program counter
; numbers 0-9
;             fgae.cdb	
	retlw 	B'01001000'	; 7-segment code 0 
	retlw 	B'11111010'	; 7-segment code 1
	retlw 	B'10001100'	; 7-segment code 2
	retlw 	B'10011000'	; 7-segment code 3
	retlw 	B'00111010'	; 7-segment code 4
	retlw 	B'00011001'	; 7-segment code 5 
	retlw 	B'00001001'	; 7-segment code 6
	retlw 	B'11011010'	; 7-segment code 7
	retlw 	B'00001000'	; 7-segment code 8
	retlw 	B'00011010'	; 7-segment code 9

; bargraph for linear response dot mode
LED_DOT_LINEAR
	addwf	PCL,f		; add value to program counter
;  		      fgae.cdb	
	retlw	B'11110111'	; 0 lower bar on (dp)
	retlw	B'11110111'	; 1 lower bar on (dp)
	retlw	B'11110101'	; 2 dp and D
	retlw	B'11110101'	; 3 dp and D
	retlw	B'11111101'	; 4 D
	retlw	B'11111101'	; 5 D
	retlw	B'11101101'	; 6 D & E
	retlw	B'11101101'	; 7 D & E
	retlw	B'11101111'	; 8 E
	retlw	B'11101111'	; 9 E
	retlw	B'11001111'	; 10 E & A
	retlw	B'11001111'	; 11 E & A
	retlw	B'11011111'	; 12 A
	retlw	B'11011111'	; 13 A
 	retlw	B'01011111'	; 14 A & F
	retlw	B'01011111'	; 15 A & F
 	retlw	B'01111111'	; 16 F
	retlw	B'01111111'	; 17 F
	retlw	B'00111111'	; 18 F & G
	retlw	B'00111111'	; 19 F & G
 	retlw	B'10111111'	; 20 G
	retlw	B'10111111'	; 21 G
	retlw	B'10111110'	; 22 G & B
	retlw	B'10111110'	; 23 G & B
 	retlw	B'11111110'	; 24 B
	retlw	B'11111110'	; 25 B

; LED bargraph linear response bargraph (0-7 bargraph)
LED_BAR_LINEAR
	addwf	PCL,f		; add value to program counter
;  		      fgae.cdb	
	retlw	B'11110111'	; 0 (dp)
	retlw	B'11110101'	; 1 + d
	retlw	B'11110101'	; 2 + d
	retlw	B'11100101'	; 3 + E
	retlw	B'11100101'	; 4 + E
	retlw	B'11000101'	; 5 + A
	retlw	B'11000101'	; 6 + A
 	retlw	B'01000101'	; 7 + F
	retlw	B'01000101'	; 8 + F
 	retlw	B'00000101'	; 9 + G
	retlw	B'00000101'	; 10 + G
 	retlw	B'00000100'	; 11 + B
	retlw	B'00000100'	; 11 + B
	 

; LED bargraph linear response bargraph (Centred response)
LED_BAR_LINEARC
	addwf	PCL,f		; add value to program counter
;  		      fgae.cdb	
	retlw	B'11000101'	; 0 A,E,D,dp
	retlw	B'11001101'	; 1 A,E,D
	retlw	B'11001111'	; 2 A,E
	retlw	B'11011111'	; 3 A
	retlw	B'01011111'	; 4 A,F
	retlw	B'00011111'	; 5 A,F,G
	retlw	B'00011110'	; 6 A,F,G,B
 	retlw	B'00011110'	; 7 A,F,G,B
 
; ******************************************************************************

; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3, Disp4 in sequence 

; start interrupt by saving w and status registers before altered by interrupt routine
INTERRUPT	
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; bank 0
	bcf		STATUS,RP1

INT_BY
	bcf		INTCON,TMR0IF
; update timer
	btfss	TRAN_FLG,0 	; if set allow transfer 
	goto	DELAY_TIMERS
	decfsz	UPDATE0,f	; update timer ms
	goto	BAR_TIMER
	decfsz	UPDATE1,f	; update timer for digits 
	goto	BAR_TIMER
	movlw	D'02'		; digits update time period
	movwf	UPDATE1	
; transfer values at each update
	movf	DISP1,w
	movwf	DISP1X		; multiplexing display value
	movf	DISP2,w
	movwf	DISP2X		; multiplexing display value
	movf	DISP3,w
	movwf	DISP3X		; multiplexing display value

BAR_TIMER
	movf	UPDATE0,w
	andlw	B'00111111'	; update rate is 30ms
	xorlw	B'00111111'
	btfss	STATUS,Z
	goto	DELAY_TIMERS
	movf	DISP4,w
	movwf	DISP4X		; multiplexing display value

DELAY_TIMERS
; delay timers
	decfsz	TIMER0,f	; ms byte timer
	goto	BRIGHTNESS
	movf	TIMER1,w
	btfss	STATUS,Z	; if zero do not decrease
	decf	TIMER1,f	; decrease to 0

BRIGHTNESS
; brightness control
	comf	BRIGHT,w	; if brightness at maximum run full drive
	btfsc	STATUS,Z
	goto	MULTIPLEX
; if not full brightness alternate
	btfsc	FLAG_1,0	; check bit 0
	goto	OFF_OUT
	bsf		FLAG_1,0	; set if clear
	movf	BRIGHT,w	; modulation off time to w
	movwf	TMR0		; 4MHz so off time is inverse of 4MHz/4/2/(FF-TMR0) 
	movlw	H'FF'
	movwf	PORTB		; displays off
	goto	RECLAIM
OFF_OUT
	comf	BRIGHT,w	; on time
	movwf	TMR0		; 4MHz so on time is inverse of 4MHz/4/2/(FF-TMR0) 
	bcf		FLAG_1,0	; clear if set

; multiplex display routine
MULTIPLEX
	btfss	PORTA,6		; skip if display 1 not lit
	goto	ALL_OFF		; all displays off for short period
	btfss	PORTA,1		; skip if display 2 not lit
	goto	LIT2		; display 2 lit
	btfss	PORTA,0		; skip if display 3 not lit				
	goto	LIT3		; display 3 lit
	btfss	PORTA,7		; skip if display 4 not lit
	goto	LIT4
LIT1
	bsf		PORTA,6		; disp1 off
	movf	DISP2X,w	; disp2 to w
	movwf	PORTB		; seven segment value to portB
	bcf		PORTA,1		; disp2 powered
	bsf		FLAG_1,6	; new multiplex flag
	goto	RECLAIM		; end of multiplex
LIT4
	bsf		PORTA,7		; disp4 off
	movf	DISP1X,w	; disp1	to w
	movwf	PORTB
	bcf		PORTA,6		; disp1 powered
	bsf		FLAG_1,5	; new multiplex flag
	goto 	RECLAIM
LIT2
	bsf		PORTA,1		; disp2 off
	movf	DISP3X,w	; disp3 to w
	movwf	PORTB		; portB has seven-segment code 
	bcf		PORTA,0		; disp3 powered
	bsf		FLAG_1,4	; new multiplex flag
	goto	RECLAIM		; end of multiplex
LIT3
	bsf		PORTA,0		; disp3 off
	movf	DISP4X,w	; disp4 to w
	movwf 	PORTB
	bcf		PORTA,7		; disp4 powered
	bsf		FLAG_1,3	; new multiplex flag
	goto	RECLAIM
ALL_OFF
	bsf		PORTA,6		; disp1 off
	movlw	D'50'
	movwf	TMR0		; 4MHz so on time is inverse of 4MHz/4/2/(HFF-D200) 
	bcf		FLAG_1,0	; clear so multiplex dimming runs next
	bsf		FLAG_1,7	; displays are all off flag

; end of interrupt reclaim w and status 

RECLAIM
	btfsc	INTCON,TMR0IF; if interrupt flag set redo interrupt
	goto	INT_BY	
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

;***************************************************************************************

; RESET		
; Set ports A & B

MAIN
; initial values
	clrf	FLAG_1		; multiplex flag
	clrf	MODE		; display mode
	clrf	SETTINGS	; settings
	clrf	TIMER1
	clrf	TIMER0		; delay timers
	clrf	UPDATE0		; update timer
	movlw	D'01'		; update set at startup
	movwf	UPDATE1
	movlw	B'11111111'	; code for 7-segment displays off
	movwf 	DISP1		; display values
	movwf	DISP2		; display value
	movwf	DISP3		; display value
	movwf	DISP4		; bar graph value
	movwf 	DISP1X		; multiplex initial display value
	movwf	DISP2X		; multiplex initial display value
	movwf	DISP3X		; multiplex initial display value
	movwf	DISP4X		; multiplex bar graph initial value
	clrf	BRIGHT		; display brightness
	clrf	SW_FLG		; switch flag
	bsf		TRAN_FLG,0 	; allow transfer in interrupt

; set inputs/outputs
	movlw	H'FF'		; set all outputs high
	movwf	PORTA
	movwf	PORTB
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'00000000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'00000000'	; 
	movwf	OPTION_REG	; TMRO prescaler is 2, 
	movlw   B'00111100'	; I/O RA0,1,6,7 outputs
	movwf   TRISA		; port A data direction register

; analog inputs, A/D
	movlw	B'00011100'	;  AN2,3,4 are analog inputs 
	movwf	ANSEL
	movlw	B'10000000'	; right justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0

	movlw	B'01010000'	; Fosc, channel 2 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on

	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01101000'	; for 4MHz
	movwf	OSCCON		; osc
	bcf		STATUS,RP0	; select memory bank 0

; set Linear or S curve with S1 closed at power up
	bcf		PORTA,6		; set RA6 low to test for S1

; setup A/D for Switch read
	bcf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5

	clrf	RATE0		; delay rate
	movlw	D'4'
	movwf	RATE1		; delay rate

; wait to charge input capacitance
	call	DELAY	
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	S2			; switch S1 not closed check S2

; S curve or linear 
	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movwf	S_LIN

CHNG_RATE_S
	decfsz	RATE0,f
	goto	SHOW_S_LIN
	decfsz	RATE1,f
	goto	SHOW_S_LIN
	movlw	D'4'	
	movwf	RATE1

	incf	S_LIN,f		; change from S to Linear to S... etc. S._P is petrol A/F, S._L is LPG A/F
; if both bits 0 and 1 are set clear
	btfss	S_LIN,0
	goto	KEEP_S_LIN
	btfsc	S_LIN,1
	clrf	S_LIN
KEEP_S_LIN
; write S curve or linear 
	movf	S_LIN,w
	call	EEWRITE_X

; show on display
SHOW_S_LIN
	btfsc	S_LIN,0		; if set then S	
	goto	SHOW_S		; petrol A/F S curve
	btfsc	S_LIN,1		; if set then S	
	goto	SHOW_S		; LPG A/F S curve

SHOW_LINEAR
; display 1
	movlw 	B'01101101'	; 7-segment code L
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 2
	bsf		PORTA,6
	bcf		PORTA,1
	movlw 	B'11101111'	; 7-segment code i
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 3
	bsf		PORTA,1
	bcf		PORTA,0
	movlw	B'10100011'	; 7-segment code n.
	movwf	PORTB		; show n.
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB
; display 4 period
	bsf		PORTA,0
	bcf		PORTA,6		; set RA6 low to test for S1
	call	DELAY
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	DELAY_END_S1; switch open
	goto	CHNG_RATE_S

; ********************************
DELAAY
	movlw	H'FF'
	movwf	LOOPCOUNT
DELAAY1
	call	DELAY
	decfsz	LOOPCOUNT,f
	goto	DELAAY1
	return
DELAY
	movlw	H'FF'
	movwf	STORE
DELAY1
	decfsz	STORE,f
	goto 	DELAY1
	return
; ***********************************
SHOW_S
; display 1
	movlw 	B'00010001'	; 7-segment code S. 
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 2
	bsf		PORTA,6
	bcf		PORTA,1
	movlw 	B'01101101'	; 7-segment code L
	btfsc	S_LIN,0		; if set then petrol A/F S curve
;			  fgae.cdb	
	movlw	B'01101000'	; 7-segment code U
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 3
	bsf		PORTA,1
	bcf		PORTA,0
	movlw 	B'01101101'	; 7-segment code L
	btfss	S_LIN,0		; if set then petrol A/F S curve
;			  fgae.cdb	
	movlw	B'00001110'	; 7-segment code P
	movwf	PORTB		; show
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB
; display 4 period
	bsf		PORTA,0
	bcf		PORTA,6		; set RA6 low to test for S1
	call	DELAY
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfsc	STATUS,C
	goto	CHNG_RATE_S
DELAY_END_S1
	call	DELAAY		; time for switch to open

S2 ; check S2 
	bsf		PORTA,0
	bsf		PORTA,7
	bsf		PORTA,6		; set RA6 high
	bcf		PORTA,1		; check S2

; wait to charge input capacitance
	call	DELAY	
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	S3			; switch S2 not closed check S3

; INVERT A
	movlw	EEPROM11
	call	EEREAD		; sets EEADR
	movwf	INVERTA
CHNG_RATE_A
	decfsz	RATE0,f
	goto	SHOW_INVERTA
	decfsz	RATE1,f
	goto	SHOW_INVERTA
	movlw	D'4'	
	movwf	RATE1

	incf	INVERTA,f	; change from non invert to invert
	movf	INVERTA,w	; store
	call	EEWRITE_X

; show on display
; Show Invert or Non Invert
SHOW_INVERTA	
; display 1
	bsf		PORTA,1
	bcf		PORTA,6
	movlw 	B'00000010'	; 7-segment code A.
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 2
	bsf		PORTA,6
	bcf		PORTA,1
	movlw 	B'10101011'	; 7-segment code n
	btfsc	INVERTA,0	; if set then Invert
	movlw	B'11111111'	; no display	
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 3
	bsf		PORTA,1
	bcf		PORTA,0
	movlw	B'11101111'	; 7-segment code i.
	movwf	PORTB		; 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB
; display 4 period
	bsf		PORTA,0
	bcf		PORTA,1		; set RA1 low to test for S2
	call	DELAY
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfsc	STATUS,C
	goto	CHNG_RATE_A
	call	DELAAY		; time for switch to open
	
; check S3
S3
	bsf		PORTA,7
	bsf		PORTA,6		; set RA6 high
	bsf		PORTA,1		; set RA1 high
	bcf		PORTA,0		; check S3

; wait to charge input capacitance
	call	DELAAY	
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	EEPROM_LOADING	; switch S3 not closed 

; INVERT B
	movlw	EEPROM12
	call	EEREAD		; sets EEADR
	movwf	INVERTB

CHNG_RATE_B
	decfsz	RATE0,f
	goto	SHOW_INVERTB
	decfsz	RATE1,f
	goto	SHOW_INVERTB
	movlw	D'4'	
	movwf	RATE1

	incf	INVERTB,f	; change from non invert to invert
	movf	INVERTB,w	; store
	call	EEWRITE_X

; show on display
;Show Invert or Non Invert
SHOW_INVERTB	
; display 1
	bsf		PORTA,0
	bcf		PORTA,6
	movlw 	B'00100001'	; 7-segment code b, 
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 2
	bsf		PORTA,6
	bcf		PORTA,1
	movlw 	B'10101011'	; 7-segment code n
	btfsc	INVERTB,0	; if set then Invert
	movlw	B'11111111'	; no display	
	movwf	PORTB		; show 
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB	
; display 3
	bsf		PORTA,1
	bcf		PORTA,0
	movlw	B'11101111'	; 7-segment code i.
	movwf	PORTB		; show n.
	call	DELAY
	movlw	H'FF'		; display off
	movwf	PORTB
; display 4 period
;	bcf		PORTA,0		; set RA6 low to test for S3
	call	DELAY
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfsc	STATUS,C
	goto	CHNG_RATE_B
	call	DELAAY		; time for switch to open
	
; obtain EEPROM values

EEPROM_LOADING
; set display off first
	movlw	H'FF'
	movwf	PORTA
	movwf	PORTB		; set high for after S. or Lin changeover shown on display

; threshold
	movlw	EEPROM0
	call	EEREAD		; sets EEADR
	movwf	BRIGHT_DIM

; Brightness minimum 
	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movwf	BRIGHT_MIN	; 

; dot or bargraph
	movlw	EEPROM2
	call	EEREAD		; sets EEADR
	movwf	DOT_BAR	  	; bit0 selects

; regulator Voltage
	movlw	EEPROM3
	call	EEREAD		; sets EEADR
	movwf	REG_V0	  	; ms btye
	movlw	EEPROM4
	call	EEREAD		; sets EEADR
	movwf	REG_V1	  	; ls btye

; S curve or linear 
	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movwf	S_LIN

; Upper limit value
	movlw	EEPROM6
	call	EEREAD		; sets EEADR
	movwf	UPPER_LIMA0	; ms btye
	movlw	EEPROM7
	call	EEREAD		; sets EEADR
	movwf	UPPER_LIMA1	; ls btye

; Lower limit value
	movlw	EEPROM8
	call	EEREAD		; sets EEADR
	movwf	LOWER_LIMA0	; ms btye
	movlw	EEPROM9
	call	EEREAD		; sets EEADR
	movwf	LOWER_LIMA1	; ls btye

; decimal point A value
	movlw	EEPROMA
	call	EEREAD		; sets EEADR
	movwf	DECIMALA

; decimal point B value
	movlw	EEPROMB
	call	EEREAD		; sets EEADR
	movwf	DECIMALB

; A or B value
	movlw	EEPROMC
	call	EEREAD		; sets EEADR
	movwf	AB_VALUE

; Upper limit value B
	movlw	EEPROMD
	call	EEREAD		; sets EEADR
	movwf	UPPER_LIMB0	; ms btye
	movlw	EEPROME
	call	EEREAD		; sets EEADR
	movwf	UPPER_LIMB1	; ls btye

; Lower limit value B
	movlw	EEPROMF
	call	EEREAD		; sets EEADR
	movwf	LOWER_LIMB0	; ms btye
	movlw	EEPROM10
	call	EEREAD		; sets EEADR
	movwf	LOWER_LIMB1	; ls btye

; A and B bargraph inversion
	movlw	EEPROM11
	call	EEREAD		; sets EEADR
	movwf	INVERTA
	movlw	EEPROM12
	call	EEREAD		; sets EEADR
	movwf	INVERTB

; interrupt enable and program now runs
	bsf		INTCON,TMR0IE	; set interrupt enable for TMR0 
	bsf		INTCON,GIE		; set global interrupt enable for above

RUN ; program run
	movf	TIMER1,w	; switch delay timer if zero can recheck switches
	btfss	STATUS,Z
	goto	NO_SWITCH

; wait for new multiplex flag
; Switch 1
	bcf		FLAG_1,5
WAIT_SET5
	btfss	FLAG_1,5	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET5
	bcf		INTCON,GIE	; stop interrupt

; A/D conversion
; channel 2
; CH2 bits 5,4,3 are 0,1,0 switch
	bcf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_5
	decfsz	STORE,f
	goto	WAIT20_5	
		
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	FLG6
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	goto	SWITCH_ACT

FLG6; Switch 2
	btfss	MODE,0		; if mode is clear then bypass this switch
	goto	FLG3
; wait for new multiplex flag
	bcf		FLAG_1,6
WAIT_SET6
	btfss	FLAG_1,6	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET6
	bcf		INTCON,GIE	; stop interrupt
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_6
	decfsz	STORE,f
	goto	WAIT20_6
	
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt

	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	FLG4
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	goto	SWITCH_ACT

FLG4; Switch 3
; wait for new multiplex flag
	bcf		FLAG_1,4
WAIT_SET4
	btfss	FLAG_1,4	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET4
	bcf		INTCON,GIE	; stop interrupt
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_4
	decfsz	STORE,f
	goto	WAIT20_4	
		
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt

	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	FLG3
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	goto	SWITCH_ACT

FLG3; Switch 4
; wait for new multiplex flag
	bcf		FLAG_1,3
WAIT_SET3
	btfss	FLAG_1,3	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET3
	bcf		INTCON,GIE	; stop interrupt
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_3
	decfsz	STORE,f
	goto	WAIT20_3	
		
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	CLEARB
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	btfsc	MODE,0		; check mode
	goto	SWITCH_ACT	; act on switch closure
	bsf		SET_B,0		; set flag when in mode 0 for A/B display 
	goto	NO_SWITCH
CLEARB
	clrf	SET_B		; flag for A or B display

NO_SWITCH

; BRIGHTNESS level channel 3
; CH3 bits 5,4,3 are 0,1,1 LDR

	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait > 20us to charge input capacitance
	movlw	D'20'
	movwf	STORE
WAIT20_CH3
	decfsz	STORE,f
	goto	WAIT20_CH3	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	STORE		; store in bank0 RAM
	rrf		ADRESH,f	; ms byte. shift to ms bits in STORE 
	rrf		STORE,f
	rrf		ADRESH,f
	rrf		STORE,f		; shift to get ms 8-bits
	comf	STORE,f		; reverse sense as LDR gives low value at high brightness 	

; compare with threshold value

; if STORE > BRIGHT_DIM set at FF		
	movf	STORE,w
	subwf	BRIGHT_DIM,w; brightness threshold
	btfsc	STATUS,C
	goto	CALC_BRIGHT
SET_FF
	movlw	H'FF'		; or set at maximum
	goto	TRANSF

; calculate (STORE/BRIGHT_DIM) x FF for BRIGHT value for display
CALC_BRIGHT
	btfsc	STATUS,Z	; if STORE = BRIGHT_DIM set at FF 
	goto	SET_FF
; Multiply Store x FF (shift left by 1 byte (8-bits))
	clrf	AARGB0
	movf	STORE,w
	movwf	AARGB1
	clrf	AARGB2
; Divide by BRIGHT_DIM
	clrf	BARGB0		; 
	movf	BRIGHT_DIM,w
	movwf	BARGB1
	call	FXD2416U
; result in AARGB0,1,2
	movf	AARGB2,w
; compare with minimum. If under, set at minimum 	
	
	subwf	BRIGHT_MIN,w; minimum brightness
	movf	AARGB2,w	; keep value
	btfsc	STATUS,C
	movf	BRIGHT_MIN,w; or set at minimum
; transfer to bright register for interrupt multiplexing
TRANSF
	movwf	BRIGHT		; display brightness

INPUT
; check mode
	movf	MODE,w		; if zero normal display if not zero settings display
	btfss	STATUS,Z
	goto	SETTINGS_RUN

; INPUT value channel 4
; CH4 bits 5,4,3 are 1,0,0 input value
	bcf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5

; wait for all displays off flag to be set in interrupt for a quiet reading	of value
	bcf		FLAG_1,7
WAIT_SET
	btfss	FLAG_1,7	; flag set in multiplex routine in interrupt signifying displays all off
	goto	WAIT_SET
	
; wait > 20us to charge input capacitance after quiet period
	movlw	D'100'
	movwf	STORE
WAIT20_CH4
	decfsz	STORE,f
	goto	WAIT20_CH4	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls byte	
	bcf		STATUS,RP0	; select memory bank 0
	movwf	INPUT1		; store in bank0 RAM
	movf	ADRESH,w	; ms byte.  
	movwf	INPUT0

DISPLAY
; check if S curve or Linear
	movf	S_LIN,w
	btfss	STATUS,Z	; if zero then linear A/F
	goto	S_DISPLAY	; A/F S curve

LINEAR_AB_DISPLAY
; Linear display
; adjust range depending on actual regulator Voltage

; an A/D value of 1023 is the regulator Voltage, 0 is zero. 3-digit display, 8-bars but 7-used for bargraph
; Span is altered by regulator voltage REG_V0,REG_V1 set at nominally 5V (D500) 
; (4.85V is D485; 5.15 is 515)
; SPAN= |(Upper limit - Lower Limit)| x REG_V0,REG_V1/D500

; test for A or B display

; Check if AB_VALUE and setting of S4 for alternate display
	movf	AB_VALUE,w	; if zero then A
	btfsc	STATUS,Z
	goto	A_REV
; B set but reversed if SET_B,0 set
	btfss	SET_B,0
	goto	B_DISPLAY_CALC
	goto	A_DISPLAY_CALC
A_REV
; A set but reversed if SET_B,0 set
	btfsc	SET_B,0
	goto	B_DISPLAY_CALC

A_DISPLAY_CALC
	movf	INVERTA,w	; invert values transfer to AB invert
	movwf	INVERT_AB
	movf	DECIMALA,w
	movwf	ZRO_FLG		; set decimal point and leading zero blanking

; Display calculation

; Positive (When upper > lower)
; SPAN = (Upper Limit - Lower Limit) x regulator Voltage/D500
; Display value = (SPAN x Input Value/H3FF) + Lower limit 

; Negative (When Lower > upper)
; SPAN = (Lower Limit - Upper Limit) x regulator Voltage/D500
; Display value = (SPAN x Complement of Input Value/H3FF) + Upper limit 

SUBTRACTA
; subtract Lower Limit from Upper Limit
	clrf	NEGATIVE		; negative flag 
	movf	LOWER_LIMA0,w	; ms lower limit
	subwf	UPPER_LIMA0,w	; ms upper limit 
	movwf	SPAN0
	movf	LOWER_LIMA1,w	; ls byte lower limit
	subwf	UPPER_LIMA1,w	; ls upper limit 
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry
	btfss	SPAN0,7			; if set then 0V Lower limit > 5V upper limit so rearrange calculation
	goto	MULT_CORR_A

; subtract Upper Limit from Lower Limit
	movf	UPPER_LIMA0,w	; ms upper limit 
	subwf	LOWER_LIMA0,w	; ms lower limit
	movwf	SPAN0
	movf	UPPER_LIMA1,w	; ls upper limit corrected
	subwf	LOWER_LIMA1,w	; ls byte lower limit
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry	
	bsf		NEGATIVE,0		; set negative flag

; Multiply (regulator correction)
MULT_CORR_A
; REG_V0,REG_V1 x SPAN0,SPAN1
	movf	REG_V0,w	; regulator voltage ms byte
	movwf	AARGB0
	movf	REG_V1,w
	movwf	AARGB1
	movf	SPAN0,w
	movwf	BARGB0
	movf	SPAN1,w
	movwf	BARGB1
	call	FXM1616U	; multiply
;result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
; Divide
	movlw	H'01'
	movwf	BARGB0		; for 500
	movlw	H'F4'
	movwf	BARGB1
	call	FXD2416U
; regulator corrected span result in AARGB0,1,2
; if AARGB0 is not zero (overrange) use 03 FF in AARGB0 and 1
	movf	AARGB0,w
	btfsc	STATUS,Z
	goto	CK_3A
SETA_VAL
	movlw	H'03'
	movwf	AARGB0
	movlw	H'FF'
	movwf	AARGB1
	goto	SPAN_INA
CK_3A
	movf	AARGB1,w
	andlw	B'11111100'	; test for set bits over 3
	btfss	STATUS,Z
	goto	SETA_VAL
; shift divide bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
SPAN_INA
; SPAN x input value
	btfss	NEGATIVE,0	; if negative flag use complement
	goto	USE_INPUT
USE_COMPLEMENT
; subtract INPUT0,INPUT1 from 3FF
	movf	INPUT0,w	; ms INPUT 
	sublw	H'3'		; ms maximum
	movwf	BARGB0
	movf	INPUT1,w	; ls INPUT
	sublw	H'FF'		; ls 
	movwf	BARGB1
	btfss	STATUS,C
	decf	BARGB0,f	; decrease if carry	
	movlw	B'00000011'
	andwf	BARGB0,f	; remove bits 2-7
	goto	MULT_SPAN_INPUTA		
USE_INPUT
	movf	INPUT1,w
	movwf	BARGB1
	movf	INPUT0,w
	movwf	BARGB0
MULT_SPAN_INPUTA
	call	FXM1616U	; multiply
;result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2	
; Divide
	movlw	H'03'
	movwf	BARGB0		; for 1023
	movlw	H'FF'
	movwf	BARGB1
	call	FXD2416U
; result in AARGB0,1,2

	btfsc	NEGATIVE,0	; if negative flag add Upper limit	
	goto	ADD_UPPERA
; add lower limit
	movf	AARGB1,w	; ms byte 
	addwf	LOWER_LIMA0,w	; lower limit ms value
	movwf	BIN_0
	movf	AARGB2,w	; ls byte 
	addwf	LOWER_LIMA1,w
	movwf	BIN_1
	btfsc	STATUS,C
	incf	BIN_0,f		; increase if carry
	goto	A_B_VAL_CONV

; Upper Limit < Lower limit so add upper limit
ADD_UPPERA
; add lower limit
	movf	AARGB1,w	; ms byte 
	addwf	UPPER_LIMA0,w	; upper limit ms value
	movwf	BIN_0
	movf	AARGB2,w	; ls byte 
	addwf	UPPER_LIMA1,w
	movwf	BIN_1
	btfsc	STATUS,C
	incf	BIN_0,f		; increase if carry
	goto	A_B_VAL_CONV

B_DISPLAY_CALC
	movf	INVERTB,w	; invert values transfer to AB invert
	movwf	INVERT_AB
	movf	DECIMALB,w
	movwf	ZRO_FLG		; set decimal point and leading zero blanking

; Display calculation

; Positive (When upper > lower)
; SPAN = (Upper Limit - Lower Limit) x regulator Voltage/D500
; Display value = (SPAN x Input Value/H3FF) + Lower limit 

; Negative (When Lower > upper)
; SPAN = (Lower Limit - Upper Limit) x regulator Voltage/D500
; Display value = (SPAN x Complement of Input Value/H3FF) + Upper limit 

SUBTRACTB
; subtract Lower Limit from Upper Limit
	clrf	NEGATIVE		; negative flag 
	movf	LOWER_LIMB0,w	; ms lower limit
	subwf	UPPER_LIMB0,w	; ms upper limit 
	movwf	SPAN0
	movf	LOWER_LIMB1,w	; ls byte lower limit
	subwf	UPPER_LIMB1,w	; ls upper limit 
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry
	btfss	SPAN0,7			; if set then 0V Lower limit > 5V upper limit so rearrange calculation
	goto	MULT_CORR_B

; subtract Upper Limit from Lower Limit
	movf	UPPER_LIMB0,w	; ms upper limit 
	subwf	LOWER_LIMB0,w	; ms lower limit
	movwf	SPAN0
	movf	UPPER_LIMB1,w	; ls upper limit corrected
	subwf	LOWER_LIMB1,w	; ls byte lower limit
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry	
	bsf		NEGATIVE,0		; set negative flag

; Multiply (regulator correction)
MULT_CORR_B
; REG_V0,REG_V1 x SPAN0,SPAN1
	movf	REG_V0,w	; regulator voltage ms byte
	movwf	AARGB0
	movf	REG_V1,w
	movwf	AARGB1
	movf	SPAN0,w
	movwf	BARGB0
	movf	SPAN1,w
	movwf	BARGB1
	call	FXM1616U	; multiply
;result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
; Divide
	movlw	H'01'
	movwf	BARGB0		; for 500
	movlw	H'F4'
	movwf	BARGB1
	call	FXD2416U
; regulator corrected span result in AARGB0,1,2
; if AARGB0 is not zero (overrange) use 03 FF in AARGB0 and 1
	movf	AARGB0,w
	btfsc	STATUS,Z
	goto	CK_3B
SETB_VAL
	movlw	H'03'
	movwf	AARGB0
	movlw	H'FF'
	movwf	AARGB1
	goto	SPAN_INB
CK_3B
	movf	AARGB1,w
	andlw	B'11111100'	; test for set bits over 3
	btfss	STATUS,Z
	goto	SETB_VAL
; shift divide bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
SPAN_INB
; SPAN x input value
	btfss	NEGATIVE,0	; if negative flag use complement
	goto	USE_INPUTB
USE_COMPLEMENTB
; subtract INPUT0,INPUT1 from 3FF
	movf	INPUT0,w	; ms INPUT 
	sublw	H'3'		; ms maximum
	movwf	BARGB0
	movf	INPUT1,w	; ls INPUT
	sublw	H'FF'		; ls 
	movwf	BARGB1
	btfss	STATUS,C
	decf	BARGB0,f	; decrease if carry	
	movlw	B'00000011'
	andwf	BARGB0,f	; remove bits 2-7
	goto	MULT_SPAN_INPUTB		
USE_INPUTB
	movf	INPUT1,w
	movwf	BARGB1
	movf	INPUT0,w
	movwf	BARGB0
MULT_SPAN_INPUTB
	call	FXM1616U	; multiply
;result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2	
; Divide
	movlw	H'03'
	movwf	BARGB0		; for 1023
	movlw	H'FF'
	movwf	BARGB1
	call	FXD2416U
; result in AARGB0,1,2

	btfsc	NEGATIVE,0	; if negative flag add Upper limit	
	goto	ADD_UPPERB
; add lower limit
	movf	AARGB1,w	; ms byte 
	addwf	LOWER_LIMB0,w	; lower limit ms value
	movwf	BIN_0
	movf	AARGB2,w	; ls byte 
	addwf	LOWER_LIMB1,w
	movwf	BIN_1
	btfsc	STATUS,C
	incf	BIN_0,f		; increase if carry
	goto	A_B_VAL_CONV

; Upper Limit < Lower limit so add upper limit
ADD_UPPERB
; add lower limit
	movf	AARGB1,w	; ms byte 
	addwf	UPPER_LIMB0,w	; lower limit ms value
	movwf	BIN_0
	movf	AARGB2,w	; ls byte 
	addwf	UPPER_LIMB1,w
	movwf	BIN_1
	btfsc	STATUS,C
	incf	BIN_0,f		; increase if carry

A_B_VAL_CONV
; check overrange
	btfss	BIN_0,2		; if set over 3FF
	goto	A_B_VAL_CONV_NO_OVER
	movlw	H'03'
	movwf	BIN_0
	movlw	H'FF'
	movwf	BIN_1

A_B_VAL_CONV_NO_OVER
; convert to packed BCD	
; do not increase past D999, H3E7
	movf	BIN_0,w			; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	NOT_999
	movf	BIN_1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	BIN_1			; E7 to BIN_1
NOT_999
	call	BCD_AND_LOAD	; convert and load
	bsf		TRAN_FLG,0 		; allow transfer in interrupt

; Bargraph
; if invert use complementary values
	btfss	INVERT_AB,0
	goto	DOT_BAR_SELECT
; subtract INPUT0,INPUT1 from 3FF
	movf	INPUT0,w	; ms INPUT 
	sublw	H'3'		; ms maximum
	movwf	INPUT0
	movf	INPUT1,w	; ls INPUT
	sublw	H'FF'		; ls 
	movwf	INPUT1
	btfss	STATUS,C
	decf	INPUT0,f	; decrease if carry	
	movlw	B'00000011'
	andwf	INPUT0,f	; remove bits 2-7
; dot or bar
DOT_BAR_SELECT
	movf	DOT_BAR,w
	btfss	STATUS,Z		; if zero dot mode
	goto	BAR_LINEAR

DOT_LINEAR
; divide by 39.35 to get the 26 possible bargraph selections 
; multiply by 100 and divide by 3935 to get correct division
	movf	INPUT0,w
	movwf	AARGB0
	movf	INPUT1,w
	movwf	AARGB1
	clrf	BARGB0
	movlw	D'100'
	movwf	BARGB1
	call	FXM1616U		; multiply
;result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2

	movlw	H'F'		; D3935 = HF5F
	movwf	BARGB0
	movlw	H'5F'
	movwf	BARGB1
	call	FXD2416U	; divide by 39.35 to get 24 different parts for dot mode
	movf	AARGB2,w	; result
	call	LED_DOT_LINEAR; get bargraph segments
	movwf	DISP4
	goto	RUN

BAR_LINEAR
; check which type of bargraph
	btfsc	DOT_BAR,1
	goto	LED_BAR_LIN_CENTRED; bargraph centred about half way

; Bargraph from 0 up
; divide by 85.25 to get the 12 possible bargraph selections 
; multiply by 100 and divide by 8525 to get correct division
	movf	INPUT0,w
	movwf	AARGB0
	movf	INPUT1,w
	movwf	AARGB1
	clrf	BARGB0
	movlw	D'100'
	movwf	BARGB1
	call	FXM1616U	; multiply
; result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2

	movlw	H'21'		; D8525 = H214D
	movwf	BARGB0
	movlw	H'4D'
	movwf	BARGB1
	call	FXD2416U	; divide 
	movf	AARGB2,w	; result
	call	LED_BAR_LINEAR; get bargraph segments
	movwf	DISP4
	goto	RUN

LED_BAR_LIN_CENTRED
; divide by 146.14 to get the 7 possible bargraph selections 
; multiply by 100 and divide by 14614 to get the correct division
	movf	INPUT0,w
	movwf	AARGB0
	movf	INPUT1,w
	movwf	AARGB1
	clrf	BARGB0
	movlw	D'100'
	movwf	BARGB1
	call	FXM1616U	; multiply
; result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2

	movlw	H'39'		; D14614 = H3916
	movwf	BARGB0
	movlw	H'16'
	movwf	BARGB1
	call	FXD2416U	; divide 
	movf	AARGB2,w	; result
	call	LED_BAR_LINEARC; get bargraph segments
	movwf	DISP4
	goto	RUN

S_DISPLAY
; S curve
; correct for regulator Voltage
; REG_V0,REG_V1 x INPUT0, INPUT1
	movf	REG_V0,w	; regulator voltage ms byte
	movwf	AARGB0
	movf	REG_V1,w
	movwf	AARGB1
	movf	INPUT0,w
	movwf	BARGB0
	movf	INPUT1,w
	movwf	BARGB1
	call	FXM1616U	; multiply
	
;result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
; Divide
	movlw	H'01'
	movwf	BARGB0		; for 500
	movlw	H'F4'
	movwf	BARGB1
	call	FXD2416U
	
; regulator corrected result in AARGB0,1,2
; if AARGB0 is not zero (overrange) use 03 FF in AARGB1 and 2
	movf	AARGB0,w
	btfss	STATUS,Z
	goto	WRI_RICH
	movf	AARGB1,w
	btfss	STATUS,Z
	goto	WRI_RICH	; rich

CORRECTED_S
	bcf		ZRO_FLG,0
	bsf		ZRO_FLG,1	; decimal point at XX.X

	movf	AARGB2,w
	sublw	D'181'		; if more than 181 is rich
	btfss	STATUS,C
	goto	WRI_RICH
	movf	AARGB2,w
	movwf	INPUT1

; read data lookup table
; get A/F ratio
	movlw	H'8'
	movwf	FLASH0		; ms byte address
	movf	INPUT1,w
	movwf	FLASH1
	call	FLASHREAD
	movf	DATA1,w	
	movwf	RATIO
A_F_VALUE
	movf	RATIO,w
	btfsc	STATUS,Z
	goto	WRI_LEAN	; if zero value write Lean
; add 9 to lookup if S_LIN,1 is set for LPG
	btfss	S_LIN,1		; if set add a 8
	goto	CK_REV1
; but do not add 8 if SET_B,0 is set
	btfsc	SET_B,0
	goto	TO_BINARY
	movlw	D'8'
	addwf	RATIO,f
	goto	TO_BINARY
CK_REV1	
; add 8 if SET_B,0 bit is set for LPG
	btfss	SET_B,0
	goto	TO_BINARY
	movlw	D'8'
	addwf	RATIO,f
TO_BINARY
	movf	RATIO,w
	movwf	BIN_1
	clrf	BIN_0
	call	BCD_AND_LOAD	; convert and load
	bsf		TRAN_FLG,0 		; allow transfer in interrupt
	goto	S_DOT_BAR
WRI_RICH
	movlw	H'FF'
	movwf	DISP1
	movwf	DISP2
	movlw	B'01001110'	; 7-segment code r
	movwf	DISP3		; r
	movlw	D'200'		; maximum value
	movwf	INPUT1
	goto	S_DOT_BAR
WRI_LEAN
	movlw	H'FF'
	movwf	DISP1
	movwf	DISP2
	movlw 	B'01101101'	; 7-segment code L
	movwf	DISP3		; 

S_DOT_BAR
; bar/dot graph
	movf	DOT_BAR,w
	btfss	STATUS,Z	; if zero dot mode
	goto	BAR_S

; read data lookup table
; get dot display
	movlw	H'9'
	movwf	FLASH0		; ms byte address
	movf	INPUT1,w
	movwf	FLASH1
	call	FLASHREAD
	movf	DATA1,w	

	goto	CONVERT
BAR_S
; get bar display
	movlw	H'A'
	movwf	FLASH0		; ms byte address
	movf	INPUT1,w
	movwf	FLASH1
	call	FLASHREAD
	movf	DATA1,w	

CONVERT
; convert lookup table data from a previous old version of the Air/Fuel ratio meter
; to this latest wideband version
	movwf	RATIO
	bcf		TRAN_FLG,0	; stop  transfer in interrupt 
	movlw	H'FF'
	movwf	DISP4
	btfss	RATIO,1		; original A/F meter c segment
	bcf		DISP4,3		; wideband A/F meter dp segment

	btfss	RATIO,2		; original A/F meter d segment
	bcf		DISP4,1		; wideband A/F meter d segment

	btfss	RATIO,3		; original A/F meter e segment
	bcf		DISP4,4		; wideband A/F meter e segment

	btfss	RATIO,4		; original A/F meter f segment
	bcf		DISP4,7		; wideband A/F meter f segment

	btfss	RATIO,5		; original A/F meter a segment
	bcf		DISP4,6		; wideband A/F meter g segment

	btfss	RATIO,6		; original A/F meter b segment
	bcf		DISP4,0		; wideband A/F meter b segment

	btfss	RATIO,7		; original A/F meter g segment
	bcf		DISP4,5		; wideband A/F meter a segment

	bsf		TRAN_FLG,0	; allow transfer
	goto	RUN	

; ************************************************************
SWITCH_ACT
; switch closed. 
; is TIMER1 clear if not. out
	movf	TIMER1,w	; switch timer is clear after switch delay
	btfss	STATUS,Z
	goto	RUN
	bsf		SW_FLG,0

; Check which switch is closed 	
; PORTA_STO has closed switch. S1 is RA6 low, S2 is RA1 low, S3 is RA0 low and S4 is RA7 low
	btfss	PORTA_STO,6
	goto	SWITCH1		; Mode
	btfss	PORTA_STO,1
	goto	SWITCH2		; settings
	btfss	PORTA_STO,0
	goto	SWITCH3		; down
	btfsc	PORTA_STO,7	; up
	goto	RUN			; no switch closed

SWITCH4; up
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'3'		; ls timer
	movwf	TIMER1

; check settings
; 0
	movf	SETTINGS,w
	xorlw	D'0'
	btfsc	STATUS,Z
	goto	DIMMING_MIN_UP
; 1
	movf	SETTINGS,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	DIMMING_THRESHOLD_UP
; 2
	movf	SETTINGS,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	REG_VOLT_UP

; 3
	movf	SETTINGS,w
	xorlw	D'3'
	btfsc	STATUS,Z
	goto	A_B_UP
; 4
	movf	SETTINGS,w
	xorlw	D'4'
	btfsc	STATUS,Z
	goto	DP_UP
; 5
	movf	SETTINGS,w
	xorlw	D'5'
	btfsc	STATUS,Z
	goto	LOWER_UP
; 6
	movf	SETTINGS,w
	xorlw	D'6'
	btfsc	STATUS,Z
	goto	UPPER_UP
; 7
	movf	SETTINGS,w
	xorlw	D'7'
	btfsc	STATUS,Z
	goto	BAR_DOT_UP

DIMMING_MIN_UP
; EEPROM1 for display brightness minimum 
	incfsz	BRIGHT_MIN,f
	goto	LOAD_MIN
	movlw	H'FF'
	movwf	BRIGHT_MIN	; stop at FF	
LOAD_MIN
	movlw	EEPROM1
	call	EEREAD
	movf	BRIGHT_MIN,w
	call	EEWRITE		; store in eeprom
	goto	RUN

DIMMING_THRESHOLD_UP
; EEPROM0 for display brightness threshold 
	incfsz	BRIGHT_DIM,f
	goto	LOAD_DIM
	movlw	H'FF'
	movwf	BRIGHT_DIM	; stop at FF	
LOAD_DIM
	movlw	EEPROM0
	call	EEREAD
	movf	BRIGHT_DIM,w
	call	EEWRITE		; store in eeprom
	goto	RUN

REG_VOLT_UP
; EEPROM3 regulator voltage MS byte REG_V0
; EEPROM4 regulator voltage LS byte REG_V1
	incf	REG_V1,f
	btfsc	STATUS,Z
	incf	REG_V0,f
LOAD_REG
	movlw	EEPROM3
	call	EEREAD
	movf	REG_V0,w
	call	EEWRITE		; store in eeprom
	movlw	EEPROM4
	call	EEREAD
	movf	REG_V1,w
	call	EEWRITE		; store in eeprom
	goto	RUN

A_B_UP
; check linear/S
	movf	S_LIN,w		; linear or S
	btfsc	STATUS,Z
	goto	LIN_INC		; linear

; in S mode changes from S Petrol to S LPG
	btfss	S_LIN,0		; if bit 0 set set bit 1 (clear 0)
	goto	SET1_S_LIN
	bcf		S_LIN,0
	bsf		S_LIN,1
	goto	WRI_S
SET1_S_LIN
	bsf		S_LIN,0
	bcf		S_LIN,1
WRI_S
; write to eeprom
	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movf	S_LIN,w
	call	EEWRITE
	goto	RUN

; Linear
;EEPROMC A or B value (AB_VALUE)
LIN_INC
	incf	AB_VALUE,w
	andlw	B'00000001'	; keep ls bit only
	movwf	AB_VALUE
	movlw	EEPROMC
	call	EEREAD
	movf	AB_VALUE,w
	call	EEWRITE		; store in eeprom
	goto	RUN

DP_UP
	btfsc	AB_VALUE,0		; clear is A set is B
	goto	B_DP
;EEPROMA Decimal Point for A value(DECIMALA)
	incf	DECIMALA,f
	btfss	DECIMALA,0		; when bits 0 and 1 set clear register
	goto	BY_CLEAR_DECA	; do not clear 
	btfsc	DECIMALA,1		; when bits 0 and 1 set clear register
	clrf	DECIMALA
BY_CLEAR_DECA
	movlw	EEPROMA
	call	EEREAD
	movf	DECIMALA,w
	call	EEWRITE		; store in eeprom
	goto	RUN
B_DP
;EEPROMB Decimal Point for B value(DECIMALB)
	incf	DECIMALB,f
	btfss	DECIMALB,0		; when bits 0 and 1 set clear register
	goto	BY_CLEAR_DECB	; do not clear 
	btfsc	DECIMALB,1		; when bits 0 and 1 set clear register
	clrf	DECIMALB
BY_CLEAR_DECB
	movlw	EEPROMB
	call	EEREAD
	movf	DECIMALB,w
	call	EEWRITE		; store in eeprom
	goto	RUN

LOWER_UP
	btfsc	AB_VALUE,0		; clear is A set is B
	goto	B_LOWER_UP
;EEPROM8 Lower limit value (LOWER_LIMA0) ms byte
;EEPROM9 Lower limit value (LOWER_LIMA1) ls byte
	incf	LOWER_LIMA1,f
	btfsc	STATUS,Z
	incf	LOWER_LIMA0,f

; do not increase past D999, H3E7
	movf	LOWER_LIMA0,w	; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	LOAD_LOWER_A
	movf	LOWER_LIMA1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	LOWER_LIMA1

LOAD_LOWER_A
	movlw	EEPROM8
	call	EEREAD
	movf	LOWER_LIMA0,w
	call	EEWRITE		; store in eeprom
	movlw	EEPROM9
	call	EEREAD
	movf	LOWER_LIMA1,w
	call	EEWRITE		; store in eeprom
	goto	RUN

B_LOWER_UP
;EEPROMF Lower limit value (LOWER_LIMB0) ms byte
;EEPROM10 Lower limit value (LOWER_LIMB1) ls byte
	incf	LOWER_LIMB1,f
	btfsc	STATUS,Z
	incf	LOWER_LIMB0,f

; do not increase past D999, H3E7
	movf	LOWER_LIMB0,w	; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	LOAD_LOWER_B
	movf	LOWER_LIMB1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	LOWER_LIMB1

LOAD_LOWER_B
	movlw	EEPROMF
	call	EEREAD
	movf	LOWER_LIMB0,w
	call	EEWRITE		; store in eeprom
	movlw	EEPROM10
	call	EEREAD
	movf	LOWER_LIMB1,w
	call	EEWRITE		; store in eeprom
	goto	RUN

UPPER_UP
	btfsc	AB_VALUE,0		; clear is A set is B
	goto	B_UPPER_UP
;EEPROM6 Upper limit value (UPPER_LIMA0) ms byte
;EEPROM7 Upper limit value (UPPER_LIMA1) ls byte
	incf	UPPER_LIMA1,f
	btfsc	STATUS,Z
	incf	UPPER_LIMA0,f

; do not increase past D999, H3E7
	movf	UPPER_LIMA0,w	; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	LOAD_UPPER_A
	movf	UPPER_LIMA1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	UPPER_LIMA1

LOAD_UPPER_A
	movlw	EEPROM6
	call	EEREAD
	movf	UPPER_LIMA0,w
	call	EEWRITE		; store in eeprom
	movlw	EEPROM7
	call	EEREAD
	movf	UPPER_LIMA1,w
	call	EEWRITE		; store in eeprom
	goto	RUN

B_UPPER_UP
;EEPROMD Upper limit value (UPPER_LIMB0) ms byte
;EEPROME Upper limit value (UPPER_LIMB1) ls byte

	incf	UPPER_LIMB1,f
	btfsc	STATUS,Z
	incf	UPPER_LIMB0,f

; do not increase past D999, H3E7
	movf	UPPER_LIMB0,w	; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	LOAD_UPPER_B
	movf	UPPER_LIMB1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	UPPER_LIMB1

LOAD_UPPER_B
	movlw	EEPROMD
	call	EEREAD
	movf	UPPER_LIMB0,w
	call	EEWRITE		; store in eeprom
	movlw	EEPROME
	call	EEREAD
	movf	UPPER_LIMB1,w
	call	EEWRITE		; store in eeprom
	goto	RUN

BAR_DOT_UP
;EEPROM2 dot/bargraph (DOT_BAR)or bargraph centred
; if S curve only allow dot and centred bargraph
	
	movf	S_LIN,w
	btfsc	STATUS,Z
	goto	SELECT_3
	movf	DOT_BAR,w
	btfsc	STATUS,Z	; if zero set bit 1
	goto	SET1
	clrf	DOT_BAR
	goto	BY_CLEAR_DB
SET1
	bsf		DOT_BAR,1
	goto	BY_CLEAR_DB

SELECT_3
	incf	DOT_BAR,f
	btfss	DOT_BAR,0		; when bits 0 and 1 set clear register
	goto	BY_CLEAR_DB		; do not clear DOT_BAR
	btfsc	DOT_BAR,1		; when bits 0 and 1 set clear register
	clrf	DOT_BAR
BY_CLEAR_DB
	movlw	EEPROM2
	call	EEREAD
	movf	DOT_BAR,w
	call	EEWRITE		; store in eeprom
	goto	RUN
	
SWITCH3; down
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'3'		; ls timer
	movwf	TIMER1

; check settings
; 0
	movf	SETTINGS,w
	xorlw	D'0'
	btfsc	STATUS,Z
	goto	DIMMING_MIN_DN
; 1
	movf	SETTINGS,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	DIMMING_THRESHOLD_DN
; 2
	movf	SETTINGS,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	REG_VOLT_DN
; 3
	movf	SETTINGS,w
	xorlw	D'3'
	btfsc	STATUS,Z
	goto	A_B_UP			; up is the same as dn
; 4
	movf	SETTINGS,w
	xorlw	D'4'
	btfsc	STATUS,Z
	goto	DP_UP 			; up is the same as down
; 5
	movf	SETTINGS,w
	xorlw	D'5'
	btfsc	STATUS,Z
	goto	LOWER_DN
; 6
	movf	SETTINGS,w
	xorlw	D'6'
	btfsc	STATUS,Z
	goto	UPPER_DN
; 7
	movf	SETTINGS,w
	xorlw	D'7'
	btfsc	STATUS,Z
	goto	BAR_DOT_UP		; ; up is the same as down

DIMMING_MIN_DN
	decf	BRIGHT_MIN,f
	movf	BRIGHT_MIN,w
	xorlw	H'FF'			; test for past 00
	btfsc	STATUS,Z
	clrf	BRIGHT_MIN		; stop at 00
	goto	LOAD_MIN		; store in EEPROM

DIMMING_THRESHOLD_DN
	decf	BRIGHT_DIM,f
	movf	BRIGHT_DIM,w
	xorlw	H'FF'			; test for past 00
	btfsc	STATUS,Z
	clrf	BRIGHT_DIM		; stop at 00
	goto	LOAD_DIM		; Store in EEPROM

REG_VOLT_DN
	movf	REG_V1,w
	btfsc	STATUS,Z		; if ls byte zero decrease ms byte
	decf	REG_V0,f
	decf	REG_V1,f
	goto	LOAD_REG

LOWER_DN
	btfsc	AB_VALUE,0		; clear is A set is B
	goto	B_LOWER_DN
; if both zero do not decrease
	movf	LOWER_LIMA1,w
	btfss	STATUS,Z
	goto	DECLA
	movf	LOWER_LIMA0,w
	btfsc	STATUS,Z
	goto	LOAD_LOWER_A
DECLA
	movf	LOWER_LIMA1,w
	btfsc	STATUS,Z		; if ls byte zero decrease ms byte
	decf	LOWER_LIMA0,f
	decf	LOWER_LIMA1,f
	goto	LOAD_LOWER_A
B_LOWER_DN
; if both zero do not decrease
	movf	LOWER_LIMB1,w
	btfss	STATUS,Z
	goto	DECLB
	movf	LOWER_LIMB0,w
	btfsc	STATUS,Z
	goto	LOAD_LOWER_B
DECLB
	movf	LOWER_LIMB1,w
	btfsc	STATUS,Z		; if ls byte zero decrease ms byte
	decf	LOWER_LIMB0,f
	decf	LOWER_LIMB1,f
	goto	LOAD_LOWER_B

UPPER_DN
	btfsc	AB_VALUE,0		; clear is A set is B
	goto	B_UPPER_DN
; if both zero do not decrease
	movf	UPPER_LIMA1,w
	btfss	STATUS,Z
	goto	DECUA
	movf	UPPER_LIMA0,w
	btfsc	STATUS,Z
	goto	LOAD_UPPER_A
DECUA
	movf	UPPER_LIMA1,w
	btfsc	STATUS,Z		; if ls byte zero decrease ms byte
	decf	UPPER_LIMA0,f
	decf	UPPER_LIMA1,f
	goto	LOAD_UPPER_A
B_UPPER_DN
; if both zero do not decrease
	movf	UPPER_LIMB1,w
	btfss	STATUS,Z
	goto	DECUB
	movf	UPPER_LIMB0,w
	btfsc	STATUS,Z
	goto	LOAD_UPPER_B
DECUB
	movf	UPPER_LIMB1,w
	btfsc	STATUS,Z		; if ls byte zero decrease ms byte
	decf	UPPER_LIMB0,f
	decf	UPPER_LIMB1,f
	goto	LOAD_UPPER_B

SWITCH2; setting
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'5'		; ls timer
	movwf	TIMER1
	incf	SETTINGS,f
	movlw	D'8'		; number of settings +1
	subwf	SETTINGS,w
	btfsc	STATUS,C	; 
	clrf	SETTINGS	; clear when reach end of settings 

; if S-curve only use 0, 1, 2, 3 and 7 values
	movf	S_LIN,w
	btfsc	STATUS,Z
	goto	RUN

; check for 4 and set at 7 so skip 4 (decimal point), 5 (0V value) and 6 (5V value)
	movf	SETTINGS,w
	xorlw	D'4'		; if settings has incremented to 4 set at 7
	btfss	STATUS,Z	; if 4? set at 7
	goto	RUN			; not a 4
	movlw	D'7'
	movwf	SETTINGS
	goto	RUN

SWITCH1; mode
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'5'		; ls timer
	movwf	TIMER1
	incf	MODE,f
	movlw	B'00000001'
	andwf	MODE,f		; keep ls bit only
	goto	RUN

; *****************************************************
SETTINGS_RUN
; settings adjust display  
; 0 dimming minimum bar segments C and dp, portb 2 and 3
; 1 dimming threshold bar segments C and d, portb 2 and 1
; 2 regulator voltage bar segments C and E, portb 2 and 4
; 3 A or B values bar segments C and A, portb 2 and 5
; 4 Decimal point X,XX or XX.X bar segments C and F, portb 2 and 7
; 5 Lower limit value bar segments C and G, portb 2 and 6
; 6 upper limit value segments C and B, portb 2 and 0
; 7 dot or bar (doT or bAr or bCn) all segments

	btfsc	SW_FLG,0	; switch flag only update at each switch press
	goto	DO_SET
	
	movf	TIMER1,w	; switch timer is clear after switch delay
	btfss	STATUS,Z
	goto	RUN
	btfss	SW_FLG,0	; switch flag only update at each switch press
	goto	RUN
DO_SET
	bcf		SW_FLG,0

; find settings
; '0'
	movf	SETTINGS,w
	btfsc	STATUS,Z
	goto	MIN_DIMMING	

	movf	SETTINGS,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	THRESH_DIMMING

	movf	SETTINGS,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	REG_VALUE

	movf	SETTINGS,w
	xorlw	D'3'
	btfsc	STATUS,Z
	goto	A_B_VALUE_SHOW

	movf	SETTINGS,w
	xorlw	D'4'
	btfsc	STATUS,Z
	goto	DP_SHOW

	movf	SETTINGS,w
	xorlw	D'5'
	btfsc	STATUS,Z
	goto	LOW_LIMIT_SHOW

	movf	SETTINGS,w
	xorlw	D'6'
	btfsc	STATUS,Z
	goto	UPPER_LIMIT_SHOW

	movf	SETTINGS,w
	xorlw	D'7'
	btfsc	STATUS,Z
	goto	DOT_BAR_SHOW

MIN_DIMMING
; 0. dimming minimum bar segments C and dp, portb 2 and 3
	bcf		TRAN_FLG,0 	; prevent transfer in interrupt
	movlw	B'11110011'	; lowest bar (C) dp bar for min dimming indication
	movwf	DISP4
	clrf	BIN_0
	movf	BRIGHT_MIN,w
DIM_LOAD
	movwf	BIN_1		; for BCD conversion
	bcf		ZRO_FLG,0	; clear flag for leading 0 suppression
	bcf		ZRO_FLG,1	; digit 2 '0' suppression when 0
	call	BCD_AND_LOAD; convert and load
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

THRESH_DIMMING		
; 1. dimming threshold bar segments C and d, portb 2 and 1
	bcf		TRAN_FLG,0 	; prevent transfer in interrupt
	movlw	B'11111001'	; 
	movwf	DISP4
	clrf	BIN_0
	movf	BRIGHT_DIM,w
	goto	DIM_LOAD

REG_VALUE
; 2. regulator voltage bar segments C and E, portb 2 and 4
	bcf		TRAN_FLG,0 	; prevent transfer in interrupt
	movlw	B'11101011'	; 
	movwf	DISP4
	movf	REG_V0,w	; ms byte
	movwf	BIN_0
	movf	REG_V1,w	; ls byte
	movwf	BIN_1
	bsf		ZRO_FLG,0	; set flag for no leading 0 suppression and DISP1 DP
	bcf		ZRO_FLG,1	;  
	call	BCD_AND_LOAD; convert and load
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

A_B_VALUE_SHOW
;3.
; show S Petrol or S LPG for S curve or in linear mode show:
; A or B values bar segments C and A, portb 2 and 5
	bcf		TRAN_FLG,0 	; prevent transfer in interrupt
	movlw	B'11011011'	; 
	movwf	DISP4

	movf	S_LIN,w		; S or linear curve
	btfss	STATUS,Z
	goto	S_P_L		; S curve Petrol / LPG

; linear so A or B
	movlw	H'FF'
	movwf	DISP2		; clear other displays
	movwf	DISP3
	movlw	B'00001010'	; 7-segment code A, 
	btfsc	AB_VALUE,0	; clear is A set is B
	movlw	B'00101001'	; 7-segment code b,
	movwf	DISP1
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

S_P_L
; S curve
; display 1
	movlw 	B'00010001'	; 7-segment code S. 
	movwf	DISP1
; display 2
	movlw 	B'01101101'	; 7-segment code L
	btfsc	S_LIN,0		; if set then petrol A/F S curve
;			  fgae.cdb	
	movlw	B'01101000'	; 7-segment code U
	movwf	DISP2
; display 3
	movlw 	B'01101101'	; 7-segment code L
	btfss	S_LIN,0		; if set then petrol A/F S curve
;			  fgae.cdb	
	movlw	B'00001110'	; 7-segment code P
	movwf	DISP3

	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

DP_SHOW
; 4. Decimal point X,XX or XX.X bar segments C and F, portb 2 and 7
	bcf		TRAN_FLG,0 		; prevent transfer in interrupt 
	movlw	B'01111011'		; 
	movwf	DISP4
	movlw	B'00001010'		; 7-segment code A, 
	btfsc	AB_VALUE,0		; clear is A set is B
	movlw	B'00101001'		; 7-segment code b,
	movwf	DISP1
	movwf	DISP2			; 
	movwf	DISP3
	btfsc	AB_VALUE,0
	goto	B_DP1
	btfsc	DECIMALA,0		; if set show in DISP1
	bcf		DISP1,3
	btfsc	DECIMALA,1		; if set show in DISP2
	bcf		DISP2,3
	bsf		TRAN_FLG,0 		; allow interrupt after transfer
	goto	RUN

B_DP1
	btfsc	DECIMALB,0		; if set show in DISP1
	bcf		DISP1,3
	btfsc	DECIMALB,1		; if set show in DISP2
	bcf		DISP2,3
	bsf		TRAN_FLG,0 		; allow interrupt after transfer
	goto	RUN

LOW_LIMIT_SHOW
; 5. Lower limit value bar segments C and G, portb 2 and 6
	bcf		TRAN_FLG,0 	; prevent transfer in interrupt
	movlw	B'10111011'	; 
	movwf	DISP4
	movf	LOWER_LIMA0,w	; ms A
	btfsc	AB_VALUE,0
	movf	LOWER_LIMB0,w	; ms B
	movwf	BIN_0

	movf	LOWER_LIMA1,w	; ls A
	btfsc	AB_VALUE,0
	movf	LOWER_LIMB1,w	; ls B	;
	movwf	BIN_1

	movf	DECIMALA,w	; decimal point A
	btfsc	AB_VALUE,0
	movf	DECIMALB,w	; decimal point B
	movwf	ZRO_FLG		; set flag for 0 suppression and dP
	call	BCD_AND_LOAD; convert and load
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

UPPER_LIMIT_SHOW
; 6. upper limit value segments C and B, portb 2 and 0
	bcf		TRAN_FLG,0 		; prevent transfer in interrupt
	movlw	B'11111010'		; 
	movwf	DISP4
	movf	UPPER_LIMA0,w	; ms A
	btfsc	AB_VALUE,0
	movf	UPPER_LIMB0,w	; ms B
	movwf	BIN_0

	movf	UPPER_LIMA1,w	; ls A
	btfsc	AB_VALUE,0
	movf	UPPER_LIMB1,w	; ls B	;
	movwf	BIN_1

	movf	DECIMALA,w	; decimal point A
	btfsc	AB_VALUE,0
	movf	DECIMALB,w	; decimal point B
	movwf	ZRO_FLG		; set flag for 0 suppression and dP
	call	BCD_AND_LOAD; convert and load
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

DOT_BAR_SHOW
; 7. dot or bar (doT or bAr or bCn) all segments; 0 = dot, 1 = BAR, 2 = BAR centred
	bcf		TRAN_FLG,0 		; prevent transfer in interrupt	; 
	movlw	B'00000000'		; 
	movwf	DISP4

	btfsc	DOT_BAR,0
	goto	BAR_SHOW
	btfsc	DOT_BAR,1
	goto	BAR_CENTRED_SHOW
; dot when zero
	movlw	B'10101000'	; 7-segment code d
	movwf	DISP1
	movlw	B'10001001'	; 7-segment code o, (note; for use with T for oT)
	movwf	DISP2		; clear other displays
	movlw	B'01001111'	; 7-segment code T, (note: use with o for oT)
	movwf	DISP3
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN
BAR_SHOW
; if in S display mode only show bCn and doT
	movf	S_LIN,w
	btfss	STATUS,Z
	goto	BAR_CENTRED_SHOW
	
	movlw	B'00101001'	; 7-segment code b,
	movwf	DISP1
	movlw	B'00001010'	; 7-segment code A,
	movwf	DISP2		; clear other displays
	movlw	B'01001110'	; 7-segment code r, 
	movwf	DISP3
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

BAR_CENTRED_SHOW
	movlw	B'00101001'	; 7-segment code b,
	movwf	DISP1
	movlw	B'01001101'	; 7-segment code C,
	movwf	DISP2		; clear other displays
	movlw	B'10101011'	; 7-segment code n, 
	movwf	DISP3
	bsf		TRAN_FLG,0 	; allow transfer in interrupt 
	goto	RUN

; extra 7-segment codes
;   fgae.cdb
; B'00001010'	; 7-segment code A, 
; B'00101001'	; 7-segment code b, 
; B'01001101'	; 7-segment code C, 
; B'10101000'	; 7-segment code d, 
; B'00001101'	; 7-segment code E, 
; B'00001111'	; 7-segment code F, 
; B'00101010'	; 7-segment code H, 
; B'01101111'	; 7-segment code I, 
; B'11101000'	; 7-segment code J, 
; B'01101101'	; 7-segment code L,	
; B'10101011'	; 7-segment code n, 
; B'10001001'	; 7-segment code o, (note; for use with T in oT)
; B'00001110'	; 7-segment code P, 
; B'01001110'	; 7-segment code r, 
; B'01001111'	; 7-segment code T, (note: use with o in oT)
; B'01101000'	; 7-segment code U, 
; B'00111000'	; 7-segment code Y, 

; *****************************************************************************************
; Subroutines

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return

; Subroutine to convert from 16-bit binary to 3-digit BCD (packed)
; Binary value is in BIN0 & BIN1. BIN0 is high byte BIN1 is low byte
; Result in BCD is in BCD0, BCD1 & BCD2.  
; BCD0 is MSB, BCD2 is LSB

BCD_AND_LOAD	
	bcf		STATUS,C	; clear carry bit
	movlw	D'16'
	movwf	CNT_16		; 16 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
	clrf	BCD_2

LOOPBCD
	rlf		BIN_1,f
	rlf		BIN_0,f		; LSB shift left binary registers
	rlf		BCD_2,f		; LSB shift left BCD registers
	rlf		BCD_1,f
	rlf		BCD_0,f

	decfsz	CNT_16,f	; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
; LOAD into display
	movf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	bcf		TRAN_FLG,0 	; stop transfer in interrupt
	movwf	DISP3

; check ZRO_FLG bits
	btfsc	ZRO_FLG,0	; if set then X.XX
	goto	X_XX		; no leading 0 suppression decimal point at DP1
	btfsc	ZRO_FLG,1	; if set then -X.X where - is a 0 blanking or XX.X
	goto	XX_X
; both flags clear so no dp but DISP1 and DISP2 '0' suppression
XXX; or _XX or __X
	movf	BCD_1,w		; check ms byte
	andlw	H'0F'		; get ls bit
	btfss	STATUS,Z
	goto	DISP1_LDY
	movlw	H'FF'
	movwf	DISP1		; DISP1 unlit 
; check DISP2
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	btfss	STATUS,Z
	goto	DISP2_LDX
	movlw	H'FF'
	movwf	DISP2		; DISP2 unlit 
	return
DISP2_LDX
	call	SEVEN_SEG	; seven segment value
	movwf	DISP2
	return

DISP1_LDY
	call	SEVEN_SEG	; seven segment value
	movwf	DISP1		; DISP1 lit with all but a 0
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; seven segment value
	movwf	DISP2		; DISP2 unlit 
	return

X_XX; no leading '0' suppression decimal at first digit X.XX
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	movwf	DISP2
	movf	BCD_1,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	movwf	DISP1
	bcf		DISP1,3		; decimal point on
	return

XX_X; leading zero suppression on ms digit, dp at second digit XX.X or _X.X
	movf	BCD_1,w		; check ms byte
	andlw	H'0F'		; get ls bit
	btfss	STATUS,Z
	goto	DISP1_LD
	movlw	H'FF'		; unlit DISP1
	goto	DISP1_LD1
DISP1_LD
	call	SEVEN_SEG	; seven segment value
DISP1_LD1
	movwf	DISP1		; DISP1 lit with all but a 0
; display 2
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	movwf	DISP2
	bcf		DISP2,3		; decimal point on
	return

; subroutine decimal adjust
DECADJ
	movlw	BCD_2		; BCD LSB address
	movwf	FSR			; pointer for BCD2
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_1
	movwf	FSR
	call 	ADJBCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR			; pointer for BCD_0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD
	movlw	H'03'		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-2)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	H'30'		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return				; end subroutine

; subroutine to read EEPROM memory 

EEREAD
	bcf		STATUS,RP1	; select bank 0
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE
EEWRITE	
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
	bsf 	INTCON,GIE	; enable interrupts
WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	return				; value written 


EEWRITE_X ; no interrupt enable	
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR4	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR4			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITEX
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITEX		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	return				; value written 

; read flash data
FLASHREAD
	bcf		STATUS,RP1	; select bank 0
	bsf 	STATUS,RP1	; select memory bank 2
	movf	FLASH0,w
	movwf	EEADRH
	movf	FLASH1,w
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bsf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read flash
	nop
	nop
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	movwf	DATA1
	movf	EEDATH,w
	movwf	DATA0
	bcf		STATUS,RP1	; select bank 0
	return

; Math routines

;

; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U    	CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1,F
                MOVF            BARGB1,W
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                RLF             AARGB0,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LA 		RLF             AARGB0,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46L8         RLF             AARGB1,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LB         RLF             AARGB1,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46L16        RLF             AARGB2,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LC 		RLF             AARGB2,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        	ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
UOK46L			RETURN

; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0,1
;               16 bit unsigned fixed point multiplier in BARGB0,1

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0,1,2,3



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   0x08
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0x00
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  0x08
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
		        RETURN

;****************************************************************************

; S-Curve lookup data table 

; airfuel ratio lookup table 5mV per count
	org	H'800'
; AIRFUEL
	DE	D'0'		; 
	DE	D'0'		; 
	DE	D'0'
	DE	D'206'
	DE	D'191'
	DE	D'184'
	DE	D'176'		; count 6
	DE	D'175'
	DE	D'171'		; 8
	DE	D'168'
	DE	D'163'
	DE	D'162'
	DE	D'160'
	DE	D'159'
	DE	D'157'
	DE	D'156'		; 15
	DE	D'156'
	DE	D'155'
	DE	D'154'
	DE	D'154'
	DE	D'153'
	DE	D'152'
	DE	D'152'		;  22
	DE	D'151'
	DE	D'151'
	DE	D'151'
	DE	D'150'
	DE	D'150'
	DE	D'150'	
	DE	D'150'		; 29
	DE	D'150'
	DE	D'149'
	DE	D'149'
	DE	D'149'
	DE	D'149'
	DE	D'149'
	DE	D'149'		; 36
	DE	D'149'
	DE	D'149'
	DE	D'149'
	DE	D'149'
	DE	D'149'
	DE	D'149'
	DE	D'149'		; 43
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'		; 50
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'	
	DE	D'148'
	DE	D'148'		; 58
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'		; 65
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'		; 72
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'	
	DE	D'148'		; 80
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'
	DE	D'148'		; 87
	DE	D'148'
	DE	D'148'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'		;94
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'		; 101
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'	
	DE	D'147'		; 108
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'		; 115
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'		; 122
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'	
	DE	D'147'		; 130
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'147'
	DE	D'146'
	DE	D'146'		; 137
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'		; 144
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'
	DE	D'146'		; 151
	DE	D'145'
	DE	D'145'
	DE	D'145'
	DE	D'145'
	DE	D'145'
	DE	D'144'	
	DE	D'144'		; 158
	DE	D'144'
	DE	D'143'
	DE	D'143'
	DE	D'142'
	DE	D'142'
	DE	D'142'
	DE	D'141'		; 165
	DE	D'140'
	DE	D'140'
	DE	D'139'
	DE	D'139'
	DE	D'138'
	DE	D'138'
	DE	D'137'		; 172
	DE	D'136'
	DE	D'135'
	DE	D'133'
	DE	D'132'
	DE	D'130'
	DE	D'128'	
	DE	D'125'		; 179
	DE	D'119'
	DE	D'118'
	DE	D'118'
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'		; 186
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'		; 193
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'
	DE	D'0'	
	DE	D'0'		; 200
	DE	D'0'		; 201
		
; ******************************************************************************************

; led bargraph lookup table (DOT display)each count represents ~4.88mV from 0V to 1V in 204 steps
; ie 1023 for 10-bit A-D for 5V or 1023/5 for 1V. So 204 steps at 4.88mV
	org	H'900'
; LEDDOT 
	DE	B'11111100'	; 0 bit 1 LED7 on
	DE	B'11111100'	; 1 bit 1 LED7 on
	DE	B'11111100'	; 2 bit 1 LED7 on
	DE	B'11111100'	; 3 bit 1 LED7 on
	DE	B'11111100'	; 4 bit 1 LED7 on
	DE	B'11111100'	; 5 bit 1 LED7 on
	DE	B'11111100'	; 6 bit 1 LED7 on
	DE	B'11111100'	; 7 bit 1 LED7 on
	DE	B'11111000'	; 8 bit 1,2 LED6&7 
	DE	B'11111000'	; 9 
	DE	B'11111000'	; 10 
 	DE	B'11111000'	; 11 
	DE	B'11111000'	; 12 
 	DE	B'11111010'	; 13 bit 2 LED6 on
	DE	B'11111010'	; 14 bit 2 LED6 on
 	DE	B'11111010'	; 15 bit 2 LED6 on
	DE	B'11111010'	; 16 bit 2 LED6 on
 	DE	B'11111010'	; 17 bit 2 LED6 on
	DE	B'11111010'	; 18 bit 2 LED6 on
 	DE	B'11111010'	; 19 bit 2 LED6 on
	DE	B'11111010'	; 20 bit 2 LED6 on
 	DE	B'11111010'	; 21 bit 2 LED6 on
	DE	B'11111010'	; 22 bit 2 LED6 on
 	DE	B'11111010'	; 23 
	DE	B'11111010'	; 24 
	DE	B'11111010'	; 25 
	DE	B'11111010'	; 26 
	DE	B'11111010'	; 27 
	DE	B'11111010'	; 28 
	DE	B'11110010'	; 29 bit 3,2 LED5&6 on
	DE	B'11110010'	; 30 bit 3,2 LED5&6 on
	DE	B'11110010'	; 31 bit 3,2 LED5&6 on
	DE	B'11110010'	; 32 
	DE	B'11110010'	; 33
	DE	B'11110010'	; 34 
	DE	B'11110010'	; 35 
	DE	B'11110010'	; 36 
	DE	B'11110010'	; 37 
	DE	B'11110010'	; 38 
	DE	B'11110010'	; 39 
	DE	B'11110010'	; 40 
	DE	B'11110110'	; 41 bit 3 LED5 on
	DE	B'11110110'	; 42 bit 3 LED5 on
	DE	B'11110110'	; 43 bit 3 LED5 on
	DE	B'11110110'	; 44 bit 3 LED5 on
	DE	B'11110110'	; 45 bit 3 LED5 on
	DE	B'11110110'	; 46 bit 3 LED5 on
	DE	B'11110110'	; 47 bit 3 LED5 on
	DE	B'11110110'	; 48 bit 3 LED5 on
	DE	B'11110110'	; 49 bit 3 LED5 on
	DE	B'11110110'	; 50 bit 3 LED5 on
	DE	B'11110110'	; 51 bit 3 LED5 on
	DE	B'11110110'	; 52 bit 3 LED5 on
	DE	B'11110110'	; 53 bit 3 LED5 on
	DE	B'11110110'	; 54 bit 3 LED5 on
	DE	B'11110110'	; 55 bit 3 LED5 on
	DE	B'11110110'	; 56 
	DE	B'11110110'	; 57 
	DE	B'11110110'	; 58 
	DE	B'11110110'	; 59 
	DE	B'11110110'	; 60 
	DE	B'11110110'	; 61 
	DE	B'11110110'	; 62 
	DE	B'11110110'	; 63 
	DE	B'11110110'	; 64 
	DE	B'11110110'	; 65 
	DE	B'11110110'	; 66 
	DE	B'11110110'	; 67 
	DE	B'11110110'	; 68 
	DE	B'01110110'	; 69 bit 7,3 LED4&5 on
	DE	B'01110110'	; 70 bit 7,3 LED4&5 on
	DE	B'01110110'	; 71 bit 7,3 LED4&5 on
	DE	B'01110110'	; 72 bit 7,3 LED4&5 on
	DE	B'01110110'	; 73 bit 7,3 LED4&5 on
	DE	B'01110110'	; 74 bit 7,3 LED4&5 on
	DE	B'01110110'	; 75 bit 7,3 LED4&5 on
	DE	B'01110110'	; 76 bit 7,3 LED4&5 on
	DE	B'01110110'	; 77 bit 7,3 LED4&5 on
	DE	B'01110110'	; 78 bit 7,3 LED4&5 on
	DE	B'01110110'	; 79 bit 7,3 LED4&5 on
	DE	B'01110110'	; 80 bit 7,3 LED4&5 on
	DE	B'01110110'	; 81 bit 7,3 LED4&5 on
	DE	B'01110110'	; 82 bit 7,3 LED4&5 on
	DE	B'01110110'	; 83 bit 7,3 LED4&5 on
	DE	B'01110110'	; 84 bit 7,3 LED4&5 on
	DE	B'01110110'	; 85 bit 7,3 LED4&5 on
	DE	B'01110110'	; 86 bit 7,3 LED4&5 on
	DE	B'01110110'	; 87 bit 7,3 LED4&5 on
	DE	B'01110110'	; 88 bit 7,3 LED4&5 on
	DE	B'01110110'	; 89 
	DE	B'01110110'	; 90 
	DE	B'01110110'	; 91 
	DE	B'01110110'	; 92 
	DE	B'01110110'	; 93 
	DE	B'01110110'	; 94
	DE	B'01110110'	; 95 
	DE	B'01110110'	; 96 
	DE	B'01110110'	; 97 
	DE	B'01110110'	; 98 
	DE	B'01110110'	; 99 
	DE	B'01110110'	; 100 
	DE	B'01110110'	; 101 
	DE	B'01110110'	; 102 
	DE	B'01110110'	; 103 
	DE	B'01110110'	; 104 
	DE	B'01110110'	; 105 
	DE	B'01110110'	; 106 
	DE	B'01110110'	; 107 
	DE	B'01111110'	; 108 bit 7 LED4 on
	DE	B'01111110'	; 109 bit 7 LED4 on
	DE	B'01111110'	; 110 bit 7 LED4 on
	DE	B'01111110'	; 111 bit 7 LED4 on
	DE	B'01111110'	; 112 bit 7 LED4 on
	DE	B'01111110'	; 113 bit 7 LED4 on
	DE	B'01111110'	; 114 bit 7 LED4 on
	DE	B'01111110'	; 115 bit 7 LED4 on
	DE	B'01111110'	; 116 bit 7 LED4 on
	DE	B'01111110'	; 117 bit 7 LED4 on
	DE	B'01111110'	; 118 bit 7 LED4 on
	DE	B'01111110'	; 119 bit 7 LED4 on
	DE	B'01111110'	; 120 bit 7 LED4 on
	DE	B'01111110'	; 121 bit 7 LED4 on
	DE	B'01111110'	; 122 bit 7 LED4 on
	DE	B'01111110'	; 123 bit 7 LED4 on
	DE	B'01111110'	; 124 bit 7 LED4 on
	DE	B'01111110'	; 125 bit 7 LED4 on
	DE	B'01111110'	; 126 bit 7 LED4 on
	DE	B'01111110'	; 127 bit 7 LED4 on
	DE	B'01111110'	; 128 bit 7 LED4 on
	DE	B'01111110'	; 129 bit 7 LED4 on
	DE	B'01111110'	; 130 bit 7 LED4 on
	DE	B'01101110'	; 131 bit 4,7 LED3,4 on
	DE	B'01101110'	; 132 bit 4,7 LED3,4 on
	DE	B'01101110'	; 133 bit 4,7 LED3,4 on
	DE	B'01101110'	; 134 bit 4,7 LED3,4 on
	DE	B'01101110'	; 135 bit 4,7 LED3,4 on
	DE	B'01101110'	; 136 bit 4,7 LED3,4 on
	DE	B'01101110'	; 137 bit 4,7 LED3,4 on
	DE	B'01101110'	; 138 bit 4,7 LED3,4 on
	DE	B'01101110'	; 139 bit 4,7 LED3,4 on
	DE	B'01101110'	; 140 bit 4,7 LED3,4 on
	DE	B'11101110'	; 141 bit 4 LED3 on
	DE	B'11101110'	; 142 bit 4 LED3 on
	DE	B'11101110'	; 143 bit 4 LED3 on
	DE	B'11101110'	; 144 bit 4 LED3 on
	DE	B'11101110'	; 145 bit 4 LED3 on
	DE	B'11101110'	; 146 bit 4 LED3 on
	DE	B'11101110'	; 147 bit 4 LED3 on
	DE	B'11101110'	; 148 bit 4 LED3 on
	DE	B'11101110'	; 149 bit 4 LED3 on
	DE	B'11101110'	; 150 bit 4 LED3 on
	DE	B'11101110'	; 151 bit 4 LED3 on
	DE	B'11101110'	; 152 bit 4 LED3 on
	DE	B'11001110'	; 153 bit 5,4 LED2&3 on
	DE	B'11001110'	; 154 bit 5,4 LED2&3 on
	DE	B'11001110'	; 155 bit 5,4 LED2&3 on
	DE	B'11001110'	; 156 bit 5,4 LED2&3 on
	DE	B'11001110'	; 157 bit 5,4 LED2&3 on
	DE	B'11001110'	; 158 bit 5,4 LED2&3 on
	DE	B'11001110'	; 159 bit 5,4 LED2&3 on
	DE	B'11001110'	; 160 bit 5,4 LED2&3 on
	DE	B'11001110'	; 161 bit 5,4 LED2&3 on
	DE	B'11001110'	; 162 bit 5,4 LED2&3 on
	DE	B'11001110'	; 163 bit 5,4 LED2&3 on
	DE	B'11001110'	; 164 bit 5,4 LED2&3 on
	DE	B'11011110'	; 165 bit 5 LED2 on
	DE	B'11011110'	; 166 bit 5 LED2 on
	DE	B'11011110'	; 167 bit 5 LED2 on
	DE	B'11011110'	; 168 bit 5 LED2 on
	DE	B'11011110'	; 169 bit 5 LED2 on
	DE	B'11011110'	; 170 bit 5 LED2 on
	DE	B'11011110'	; 171 bit 5 LED2 on
	DE	B'11011110'	; 172 bit 5 LED2 on
	DE	B'11011110'	; 173 bit 5 LED2 on
	DE	B'11011110'	; 174 bit 5 LED2 on
	DE	B'10011110'	; 175 bit 6,5 LED1&2 on
	DE	B'10011110'	; 176 bit 6,5 LED1&2 on
	DE	B'10011110'	; 177 bit 6,5 LED1&2 on
	DE	B'10011110'	; 178 bit 6,5 LED1&2 on
	DE	B'10111110'	; 179 bit 6 LED1 on
	DE	B'10111110'	; 180 bit 6 LED1 on
	DE	B'10111110'	; 181 bit 6 LED1 on
	DE	B'10111110'	; 182 bit 6 LED1 on
	DE	B'10111110'	; 183 bit 6 LED1 on
	DE	B'10111110'	; 184 bit 6 LED1 on
	DE	B'10111110'	; 185 bit 6 LED1 on
	DE	B'10111110'	; 186 bit 6 LED1 on
	DE	B'10111110'	; 187 bit 6 LED1 on
	DE	B'10111110'	; 188 bit 6 LED1 on
	DE	B'10111110'	; 189 bit 6 LED1 on
	DE	B'10111110'	; 190 bit 6 LED1 on
	DE	B'10111110'	; 191 bit 6 LED1 on
	DE	B'10111110'	; 192 bit 6 LED1 on
	DE	B'10111110'	; 193 bit 6 LED1 on
	DE	B'10111110'	; 194 bit 6 LED1 on
	DE	B'10111110'	; 195 bit 6 LED1 on
	DE	B'10111110'	; 196 bit 6 LED1 on
	DE	B'10111110'	; 197 bit 6 LED1 on
	DE	B'10111110'	; 198 bit 6 LED1 on
	DE	B'10111110'	; 199 bit 6 LED1 on
	DE	B'10111110'	; 200 bit 6 LED1 on
	DE	B'10111110'	; 201 bit 6 LED1 on
	
; ************************************************************
; LED bargraph display bar mode
	
	org	H'A00'
; LEDBAR
	DE	B'01110000'	; 0 LED4-7 on
	DE	B'01110000'	; 1 
	DE	B'01110000'	; 2 
	DE	B'01110000'	; 3 
	DE	B'01110000'	; 4 
	DE	B'01110000'	; 5 
	DE	B'01110000'	; 6 
	DE	B'01110000'	; 7 
	DE	B'01110000'	; 8 
	DE	B'01110010'	; 9 LED4-6
 	DE	B'01110010'	; 10 
	DE	B'01110010'	; 11 
 	DE	B'01110010'	; 12 
	DE	B'01110010'	; 13 
 	DE	B'01110010'	; 14 
	DE	B'01110010'	; 15 
 	DE	B'01110010'	; 16 
	DE	B'01110010'	; 17 
 	DE	B'01110010'	; 18 
	DE	B'01110010'	; 19 
 	DE	B'01110010'	; 20 
	DE	B'01110010'	; 21 
 	DE	B'01110010'	; 22 
	DE	B'01110010'	; 23 
 	DE	B'01110010'	; 24 
	DE	B'01110010'	; 25 
 	DE	B'01110010'	; 26 
	DE	B'01110010'	; 27 
	DE	B'01110110'	; 28 LED 4 & 5
	DE	B'01110110'	; 29 
 	DE	B'01110110'	; 30 
	DE	B'01110110'	; 31 
 	DE	B'01110110'	; 32 
	DE	B'01110110'	; 33
	DE	B'01110110'	; 34 
 	DE	B'01110110'	; 35 
	DE	B'01110110'	; 36 
 	DE	B'01110110'	; 37 
	DE	B'01110110'	; 38 
	DE	B'01110110'	; 39 
	DE	B'01110110'	; 40 
	DE	B'01110110'	; 41 
	DE	B'01110110'	; 42 
	DE	B'01110110'	; 43 
	DE	B'01110110'	; 44 
	DE	B'01110110'	; 45 
	DE	B'01110110'	; 46 
	DE	B'01110110'	; 47 
	DE	B'01110110'	; 48 
	DE	B'01110110'	; 49 
	DE	B'01110110'	; 50 
	DE	B'01110110'	; 51 
	DE	B'01110110'	; 52 
	DE	B'01110110'	; 53 
	DE	B'01110110'	; 54 
	DE	B'01110110'	; 55 
	DE	B'01110110'	; 56 
	DE	B'01110110'	; 57 
	DE	B'01110110'	; 58
	DE	B'01110110'	; 59 
	DE	B'01110110'	; 60 
	DE	B'01110110'	; 61 
	DE	B'01110110'	; 62 
	DE	B'01110110'	; 63
	DE	B'01110110'	; 64 
	DE	B'01110110'	; 65 
	DE	B'01110110'	; 66 
	DE	B'01110110'	; 67 
	DE	B'01111110'	; 68 LED4
	DE	B'01111110'	; 69 
	DE	B'01111110'	; 70
	DE	B'01111110'	; 71 LED4
	DE	B'01111110'	; 72 
	DE	B'01111110'	; 73 
	DE	B'01111110'	; 74 
	DE	B'01111110'	; 75 
	DE	B'01111110'	; 76 
	DE	B'01111110'	; 77
	DE	B'01111110'	; 78
	DE	B'01111110'	; 79 LED4
	DE	B'01111110'	; 80 
	DE	B'01111110'	; 81 
	DE	B'01111110'	; 82 
	DE	B'01111110'	; 83 
	DE	B'01111110'	; 84 
	DE	B'01111110'	; 85
	DE	B'01111110'	; 87 LED4
	DE	B'01111110'	; 88 
	DE	B'01111110'	; 89 
	DE	B'01111110'	; 90 
	DE	B'01111110'	; 91 
	DE	B'01111110'	; 92 
	DE	B'01111110'	; 93
	DE	B'01111110'	; 94 LED4
	DE	B'01111110'	; 95 
	DE	B'01111110'	; 96 
	DE	B'01111110'	; 97 
	DE	B'01111110'	; 98 
	DE	B'01111110'	; 99 
	DE	B'01111110'	; 100
	DE	B'01111110'	; 101 LED4
	DE	B'01111110'	; 102 
	DE	B'01111110'	; 103 
	DE	B'01111110'	; 104 
	DE	B'01111110'	; 105 
	DE	B'01111110'	; 106 
	DE	B'01111110'	; 107
	DE	B'01111110'	; 108 LED4
	DE	B'01111110'	; 109 
	DE	B'01111110'	; 110 
	DE	B'01111110'	; 111 
	DE	B'01111110'	; 112 
	DE	B'01111110'	; 113 
	DE	B'01111110'	; 114 
	DE	B'01111110'	; 115 
	DE	B'01111110'	; 116 
	DE	B'01111110'	; 117 
	DE	B'01111110'	; 118 
	DE	B'01111110'	; 119 
	DE	B'01111110'	; 120 
	DE	B'01111110'	; 121 
	DE	B'01111110'	; 122 
	DE	B'01111110'	; 123 
	DE	B'01111110'	; 124 
	DE	B'01111110'	; 125 
	DE	B'01111110'	; 126 
	DE	B'01111110'	; 127 
	DE	B'01111110'	; 128 
	DE	B'01111110'	; 129 
	DE	B'01111110'	; 130 
	DE	B'01111110'	; 131 
	DE	B'01111110'	; 132 
	DE	B'01111110'	; 133 
	DE	B'01111110'	; 134 
	DE	B'01111110'	; 135 
	DE	B'01111110'	; 136
	DE	B'01111110'	; 137 
	DE	B'01111110'	; 138 
	DE	B'01111110'	; 139 
	DE	B'01101110'	; 140 bit 4,7 LED3,4 on
	DE	B'01101110'	; 141 bit 4,7 LED3,4 on
	DE	B'01101110'	; 142 bit 4,7 LED3,4 on
	DE	B'01101110'	; 143 bit 4,7 LED3,4 on
	DE	B'01101110'	; 144 bit 4,7 LED3,4 on
	DE	B'01101110'	; 145 bit 4,7 LED3,4 on
	DE	B'01101110'	; 146 bit 4,7 LED3,4 on
	DE	B'01101110'	; 147 bit 4,7 LED3,4 on
	DE	B'01101110'	; 148 bit 4,7 LED3,4 on
	DE	B'01101110'	; 149 bit 4,7 LED3,4 on
	DE	B'01101110'	; 150 bit 4,7 LED3,4 on
	DE	B'01101110'	; 151 bit 4,7 LED3,4 on
	DE	B'01101110'	; 152 bit 4,7 LED3,4 on
	DE	B'01101110'	; 153 bit 4,7 LED3,4 on
	DE	B'01101110'	; 154 bit 4,7 LED3,4 on
	DE	B'01101110'	; 155 bit 4,7 LED3,4 on
	DE	B'01101110'	; 156 bit 4,7 LED3,4 on
	DE	B'01101110'	; 157 bit 4,7 LED3,4 on
	DE	B'01101110'	; 168 bit 4,7 LED3,4 on
	DE	B'01101110'	; 159 bit 4,7 LED3,4 on
	DE	B'01001110'	; 160 LED2,3,4 on
	DE	B'01001110'	; 161 
	DE	B'01001110'	; 162
	DE	B'01001110'	; 163 
	DE	B'01001110'	; 164 
	DE	B'01001110'	; 165 
	DE	B'01001110'	; 166 
	DE	B'01001110'	; 167 
	DE	B'01001110'	; 168 
	DE	B'01001110'	; 169 
	DE	B'01001110'	; 170 
	DE	B'01001110'	; 171 
	DE	B'01001110'	; 172 
	DE	B'01001110'	; 173 
	DE	B'01001110'	; 174
	DE	B'01001110'	; 165 
	DE	B'01001110'	; 166 
	DE	B'01001110'	; 167  
	DE	B'00001110'	; 168 LED 1-4 on
	DE	B'00001110'	; 169 
	DE	B'00001110'	; 170 
	DE	B'00001110'	; 171 
	DE	B'00001110'	; 172 
	DE	B'00001110'	; 173 
	DE	B'00001110'	; 174 
	DE	B'00001110'	; 175 
	DE	B'00001110'	; 176 
	DE	B'00001110'	; 177 
	DE	B'00001110'	; 178 
	DE	B'00001110'	; 179 
	DE	B'00001110'	; 180
	DE	B'00001110'	; 181 
	DE	B'00001110'	; 182 
	DE	B'00001110'	; 183 
	DE	B'00001110'	; 184 
	DE	B'00001110'	; 185 
	DE	B'00001110'	; 186
	DE	B'00001110'	; 187 
	DE	B'00001110'	; 188 
	DE	B'00001110'	; 189 
	DE	B'00001110'	; 190 
	DE	B'00001110'	; 191 
	DE	B'00001110'	; 192
	DE	B'00001110'	; 193 
	DE	B'00001110'	; 194 
	DE	B'00001110'	; 195 
	DE	B'00001110'	; 196 
	DE	B'00001110'	; 197 
	DE	B'00001110'	; 198
	DE	B'00001110'	; 199 
	DE	B'00001110'	; 200 
	DE	B'00001110'	; 201 


	end

	
	
