/*
  Camera Timer V1.0.3
  Copyright (C) 2008 Andrew J. Armstrong

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
  02111-1307  USA

  Author:
  Andrew J. Armstrong <andrew_armstrong@unwired.com.au>
*/

/*
-------------------------------------------------------------------------------

NAME     - CameraTimer

FUNCTION - This is a one-button time-lapse camera timer initially developed for
           a Canon 400D Digital SLR camera. It can be used to take photographs
           at intervals from 1 second to 65535 seconds (approximately 18 hours).

FEATURES - - Single button user interface
           - Shooting interval from 1 to 65535 seconds
           - Audible feedback via a piezo speaker
           - Visual feedback via a bi-color LED
           - Uses only 385/1023 bytes of ROM and 30/64 bytes of RAM.
           
USAGE    - - Connect the shutter control output to the camera's 2.5 mm remote
             control socket.
           - Turn on the camera.
           - Frame and focus on the subject.
           - Slide the camera lens auto focus (AF) switch to Manual.
           - Turn off the camera's LCD display to save power.
           - Turn on the camera timer. The LED will glow RED to indicate "set"
             mode.
           - Use the button to enter the desired shooting interval in seconds
             (see Examples below).
           - Press and hold the button for 1 second to enter "shoot" mode. The
             RED LED will extinguish to indicate "shoot" mode. The first photo
             will be taken immediately upon entering "shoot" mode and at each
             selected interval thereafter. The LED will briefly glow GREEN when
             each photograph is taken.

EXAMPLES - 1. To select a shooting interval of 120 seconds:
              - Turn on the camera timer -> high/low beep (means READY)
                RED light on = "set" mode
              - To enter the first digit (1):
                Short press -> low beep (means 0)
                Short press -> high beep (means +1)
                Wait for more than 1 second -> high/low beep (means OK)
                Interval now = 1 second
              - To enter the next digit (2):
                Short press -> low beep (means 0)
                Short press -> high beep (means +1)
                Short press -> high beep (means +1)
                Wait for more than 1 second -> high/low beep (means OK)
                Interval now = 12 seconds
              - To enter the next digit (0):
                Short press -> low beep (means 0)
                Wait for more than 1 second -> high/low beep (means OK)
                Interval now = 120 seconds
              - To exit "set" mode and enter "shoot" mode:
                Long press -> high beep, RED light off = "shoot" mode.

           2. To exit "shoot" mode and set a different shooting interval:
              - Long press -> high beep, RED light on = "set" mode entered.
              - Enter a new shooting interval as shown in (1) above

           3. To suspend shooting and resume with the same interval later:
              - Short press -> no beep, RED light on = "set" mode entered.
              - Wait until you want to resume shooting (finish beer etc)
              - Long press -> high beep for every second that the button is
                held down. You can use this audible timing signal to
                synchronise with your watch in order to resume shooting at a
                particular time.
              - Release button -> no beep, RED light off = "shoot" mode.

           4. To use the shooting interval that you last set:
              - Turn on the camera timer -> RED light on = "set" mode
              - Long press -> high beep, RED light off = "shoot" mode entered
                using the last set interval. The last interval you set is
                remembered in the onboard EEPROM across power ups.

NOTES    - 1. If when setting an interval you overshoot the digit you wanted
              just keep doing short button presses until you hear a low beep
              (which means 0) and begin counting each subsequent press until
              you reach the desired digit.
              
           2. If you forget where you are up to when setting an interval,
              simply turn the camera timer off and on and begin setting the
              interval again. Alternatively, enter "shoot" mode (long press)
              and then enter "set" mode again (long press).

           3. If you attempt to enter "shoot" mode with a shooting interval
              of 0, you will hear three low beeps to indicate that this is
              invalid and you will remain in "set" mode (RED light on).
              
           4. No attempt is made to validate the interval you enter. If you
              enter 99999 (which is greater than the maximum of 65535) then
              interval set will be some number between 0 and 65535 - all of
              which will be accepted (but not what you expected) except for 0
              which will cause the "fail" beep sequence to be heard.
              
           5. The last shooting interval that you use is remembered in the
              onboard EEPROM across power ups - so if you have a favourite
              shooting interval then you need only set it once.

PINS     - For the 12F675:
           -PIN-  ---PIC FUNCTION(S)-----------------   ---APP FUNCTION---
           Pin 1  Vdd
           Pin 2  GP5         T1CKI   OSC1    CLKIN     Shutter control
           Pin 3  GP4   AN3   ~T1G    OSC2    CLKOUT    Beeper control
           Pin 4  GP3         ~MCLR   Vpp               Control button
           Pin 5  GP2   AN2   T0CKI   COUT              Tone control
           Pin 6  GP1   AN1   CIN-    Vref    ICSPCLK   Mode indicator LED
           Pin 7  GP0   AN0   CIN+            ICSPDAT   n/c
           Pin 8  Vss

AUTHORS  - Init Name                 Email
           ---- -------------------- ------------------------------------------
           AJA  Andrew J. Armstrong  andrew_armstrong@unwired.com.au

HISTORY  - Date     Ver   By  Reason (most recent at the top please)
           -------- ----- --- -------------------------------------------------
           20081101 1.0.3 AJA Disallow appending digits on leaving shoot mode.
           20081020 1.0.2 AJA Revert to using GPIO.RED_LED for shoot mode, but
                              use a 2.2k button pull-up resistor of instead of
                              33k.
           20081018 1.0.1 AJA Use flag for shoot mode.
           20080929 1.0.0 AJA Initial version.
           
-------------------------------------------------------------------------------
*/
#define SHUTTER       GP5
#define BEEPER        GP4
#define BUTTON        GP3
#define TONE          GP2
#define RED_LED       GP1

#define BUTTON_PRESSED !GPIO.BUTTON
#define SHOOT_MODE    !GPIO.RED_LED

#define OUTPUT        0
#define INPUT         1

#define ON            1
#define OFF           0

#define PRESCALER                8
#define NOMINAL_XTAL_FREQ        4000000
#define FREQ_ADJUSTMENT          NOMINAL_XTAL_FREQ * 4 / 100
#define XTAL_FREQ                (NOMINAL_XTAL_FREQ + FREQ_ADJUSTMENT)
#define CLOCK_FREQ               XTAL_FREQ/4
#define TICKS_PER_SECOND         CLOCK_FREQ/PRESCALER
#define INTERRUPTS_IN_ONE_SECOND 8
#define LONG_PRESS               8
#define LONG_RELEASE             8
#define TICKS_PER_INTERRUPT      TICKS_PER_SECOND/INTERRUPTS_IN_ONE_SECOND
#define AVERAGE_ISR_INSTRUCTIONS 32
#define OVERHEAD                 4+AVERAGE_ISR_INSTRUCTIONS
#define TMR1_INITIAL_VALUE       (0xFFFF - TICKS_PER_INTERRUPT + OVERHEAD)
#define TMR1_INITIAL_LO          TMR1_INITIAL_VALUE & 0xff
#define TMR1_INITIAL_HI          TMR1_INITIAL_VALUE >> 8

void shoot(void);
unsigned char isButtonPressed(void);
void beep(void);
void beepLo(void);
void beepMed(void);
void beepHi(void);
void beepOK(void);
void beepFail(void);
void delay(void);

// The following are marked volatile because they are referenced in the ISR
volatile unsigned int   nInterruptsRemaining;
volatile unsigned short nDuration;


// The following are not volatile
union
{
  unsigned int n;
  unsigned short byte[2];
}                       nSecondsBetweenShots;
unsigned int            nSecondsRemaining;
unsigned short          nDigit;

unsigned char           bFlags;
#define bDigitPending   bFlags.F1
#define bLongPress      bFlags.F2

void main(void)
{
//----------------------------------------------------------------------------
// Set up bank 0 registers
//----------------------------------------------------------------------------

  nInterruptsRemaining = 0; // Timer interrupts remaining in this second
  nDuration = 0;            // Interrupts counted for button presses/releases
  nSecondsBetweenShots.n = 0; // Seconds to wait between each shot
  nSecondsRemaining = 0;    // Seconds remaining until next shot is taken
  nDigit = 9;               // Cause the first button press to cycle to zero
  bFlags = 0;               // Reset all flags to false

  GPIO = 0b00000000;        // Turn off all pins

  T1CON.T1CKPS1 = 1;        // Timer1 clock prescaler = 8 (PS1:PS0 = 11)
  T1CON.T1CKPS0 = 1;

//----------------------------------------------------------------------------
// Set up bank 1 registers
//----------------------------------------------------------------------------

  CMCON  = 0b00000111;      // Turn off the comparator <-- DO THIS OR SUFFER!
  ANSEL  = 0b00000000;      // All pins are digital (not analog) <-- AND THIS!
  TRISIO = 0b00001000;      // All pins are output (except GP3)
  PIE1.T1IE = 1;            // Enable interrupt on TMR1 overflow

//----------------------------------------------------------------------------
// Set up registers common to both banks
//----------------------------------------------------------------------------

  INTCON.PEIE = 1;		      // Enable peripheral interrupts (needed for TMR1)
  INTCON.GIE = 1;	          // Enable interrupts globally
  TMR1L = TMR1_INITIAL_LO;  // Set the low byte of the timer interval
  TMR1H = TMR1_INITIAL_HI;  // Set the high byte of the timer interval
  T1CON.TMR1ON = 1;         // Turn on Timer1
  beepOK();                 // Indicate that we are powered up and ready
  GPIO.RED_LED = ON;        // Indicate we are in "set" mode

  for(;;)                   // This loops hundreds of times each second...
  {

//----------------------------------------------------------------------------
// Shooting
//----------------------------------------------------------------------------

    if (SHOOT_MODE)           // If we are shooting pictures
    {
      nDuration = 0;          // Duration of button press (interrupt count)
      if (isButtonPressed())  // Pressing the button cancels shoot mode
      {
        GPIO.SHUTTER = OFF;   // Release the shutter immediately
        GPIO.RED_LED = ON;    // Indicate we are now in "set" mode
        nSecondsBetweenShots.n = 0; // Reset the shooting interval
        beepOK();             // Indicate shooting interval is reset to zero
      }
      else
      {
        if (nInterruptsRemaining == 0)  // If a whole second has elapsed
        {
          nInterruptsRemaining = INTERRUPTS_IN_ONE_SECOND;
          nSecondsRemaining--;
          if (nSecondsRemaining == 0)   // If the shooting interval has elapsed
          {
            nSecondsRemaining = nSecondsBetweenShots.n;
            shoot();    // Take the next shot
          }
        }
      }
    }

//----------------------------------------------------------------------------
// Setting interval
//----------------------------------------------------------------------------

    else                // Else we are setting the interval between shots
    {
      bLongPress = 0;
      if (isButtonPressed())
      {
        nDuration = 0;  // Duration of button press (interrupt count)
        while (BUTTON_PRESSED) // Wait for button to be released
        {
          if (nDuration >= LONG_PRESS)
          {
            bLongPress = 1;
            beepHi();   // Indicate about to enter shoot mode
            nDuration = 0;
          }
        }
        nDuration = 0;  // Duration of button release (interrupt count)
        if (!bLongPress)
        {               // A short press cycles up to the next digit
          nDigit++;
          bDigitPending = 1;
          if (nDigit > 9)
          {
            nDigit = 0; // Wrap from 9 to 0
            beepLo();   // Emit a low beep for 0 (so the user starts counting)
          }
          else
            beepHi();   // Emit a high beep for 1 to 9
        }
      }

      if (bDigitPending)
      {
        if (nDuration >= LONG_RELEASE)
        {               // A long release means the user wants this digit
          beepOK();     // Indicate the next digit is accepted
          // The following appends the next digit...
          nSecondsBetweenShots.n = nSecondsBetweenShots.n * 10 + nDigit;
          nDigit = 9;   // Ready for next digit entry
          bDigitPending = 0;  // Wait for user to select the next digit
        }
      }
      
      if (bLongPress)
      {
        if (nSecondsBetweenShots.n == 0)
        {
          nSecondsBetweenShots.byte[0] = Eeprom_read(0);
          nSecondsBetweenShots.byte[1] = Eeprom_read(1);
        }
        if (nSecondsBetweenShots.n == 0)
          beepFail();   // Indicate that the shooting interval is invalid
        else
        {
          Eeprom_write(0, nSecondsBetweenShots.byte[0]);
          Eeprom_write(1, nSecondsBetweenShots.byte[1]);
          nInterruptsRemaining = INTERRUPTS_IN_ONE_SECOND;
          nSecondsRemaining = nSecondsBetweenShots.n;
          GPIO.RED_LED = OFF;     // Indicate we are not in "set" mode
          shoot();
        }
      }
    }
  }
}

void shoot(void)
{
  GPIO.SHUTTER = ON;    // Press the shutter button
  delay();              // Hold the shutter button for a bit
  GPIO.SHUTTER = OFF;   // Release the shutter button
}

void delay(void)
{
  Delay_ms(50);
}

unsigned char isButtonPressed(void)
{
  if (BUTTON_PRESSED)   // Low means button is pressed
  {
    Delay_us(500);      // Debounce interval...
    Delay_us(500);      // ...of 1 ms
    if (BUTTON_PRESSED) // If still low after 1 ms
      return 1;         // Then indicate button is pressed
  }
  return 0;             // Else indicate button is not pressed
}


void beepOK(void)
{
  beepHi();
  delay();
  beepLo();
}

void beepFail()
{
  delay();
  beepLo();
  delay();
  beepLo();
  delay();
  beepLo();
}

void beepLo(void)
{
  TRISIO.TONE = INPUT;  // Deactivate tone control pin (floating)
  beep();
}
void beepMed(void)
{
  GPIO.TONE = OFF;      // Set medium frequency
  TRISIO.TONE = OUTPUT; // Activate tone control pin
  beep();
  TRISIO.TONE = INPUT;  // Float the tone control pin
}
void beepHi(void)
{
  GPIO.TONE = ON;       // Set high frequency
  TRISIO.TONE = OUTPUT; // Activate tone control pin
  beep();
  TRISIO.TONE = INPUT;  // Float the tone control pin
}

void beep(void)
{
  GPIO.BEEPER = ON;     // Turn beeper on
  delay();              // Wait a bit
  GPIO.BEEPER = OFF;    // Turn beeper off
}

void interrupt(void)    // In this app it is always a Timer1 interrupt...
{
  if (PIR1.TMR1IF)      // If it's a Timer1 interrupt
  {
    PIR1.TMR1IF = 0;	  // Clear the timer interrupt flag
    nInterruptsRemaining--;    // Decrement interrupts remaining in this second
    nDuration++;        // Number of interrupts since last reset
    T1CON.TMR1ON = 0;   // Turn off Timer1 so we can reset its value
    TMR1L = TMR1_INITIAL_LO;
    TMR1H = TMR1_INITIAL_HI;
    T1CON.TMR1ON = 1;   // Turn on Timer1
  }
}
