; Wideband Controller
; Uses assembly code for PIC16F88 microcontroller


;	ERRORLEVEL -302
;	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB3 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_ON & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; Define variables at memory locations

; Bank 0 RAM
IMPEDANCE	equ	H'20'	; impedance on/off flag
HEAT		equ	H'21'	; heater value
HEAT_COUNT	equ	H'22'	; heater counter for PWM
DRIVE_COUNT	equ	H'23'	; PWM drive counter
WIDEBAND0	equ	H'24'	; wideband value ms
WIDEBAND1	equ	H'25'	; wideband value ls
S_CURVE		equ	H'26'	; S curve output
CURRENT_IP0	equ	H'27'	; IP current (PWM output value) ms	
CURRENT_IP1	equ	H'28'	; IP current (PWM output value) ls
STOREH		equ	H'29'	; ms byte store
STOREL		equ	H'2A'	; ls byte store
BATT0		equ	H'2B'	; battery volts ms byte
BATT1		equ	H'2C'	; battery volts ls byte
RAMP		equ	H'2D'	; heater ramp				
COUNT1		equ	H'2E'	; counter for interrupt counts
HALFSEC		equ	H'2F'	; half second counter
FLAG		equ	H'30'	; flag indicator
VH_EFF0		equ	H'31'	; heater effective voltage ms
VH_EFF1		equ	H'32'	; heater effective voltage ls
MIN_COUNT	equ	H'33'	; minute counter
ERROR_FLG	equ	H'34'	; error flag
HEAT_I		equ	H'35'	; transfer new heat value at end of cycle
SWINGH0		equ	H'36'	; voltage swing high across VS impedance ms byte
SWINGH1		equ	H'37'	; voltage swing high across VS impedance ls byte
SWINGL0		equ	H'38'	; voltage swing low across VS impedance ms byte
SWINGL1		equ	H'39'	; voltage swing low across VS impedance ls byte	
STORE3		equ	H'3A'	; delay counter
SPAN0		equ	H'3B'	; impedance measuremant voltage span ms byte
SPAN1		equ	H'3C'	; impedance measuremant voltage span ls byte	
HEATX		equ	H'3D'	; heater value temporary store
FLASH0		equ	H'3E'	; flash address ms
FLASH1		equ	H'3F'	; flash address	ls
DATA0		equ	H'40'	; flash data ms
DATA1		equ	H'41'	; flash data ls	
FOUR_SEC	equ	H'42'	; four seconds counter
CALC0		equ	H'43'	; calculation temporary value ms byte
RATE		equ	H'44'	; ramp rate counter
DUTY_FLASH	equ	H'45'	; duty cycle flash for heater control
MILLISEC	equ	H'46'	; 10ms counter for S-curve rate
DATA_FLSH	equ	H'47'	; data flash
SENSE_IP0	equ	H'48'	; IP sense ms	
SENSE_IP1	equ	H'49'	; IP sense ls
VS_AVG0		equ	H'4A'	; Vs average ms
VS_AVG1		equ	H'4B'	; Vs average ls
TEMP_AV0	equ	H'4C'	; temporary ms
TEMP_AV1	equ	H'4D'	; temporary ls
TEMP_PWM0	equ H'4E'	; temp PWM ms
TEMP_PWM1	equ H'4F'	; temp PWM ls
S_RESP		equ	H'50'	; s curve response
VS_IP0		equ	H'51'	; VS/IP	ms
VS_IP1		equ	H'52'	; VS/IP ls
TWO_COUNT	equ	H'53'	; A/D AN2 counter
SWING_FLG	equ	H'54'	; swing flag
MAX0		equ	H'55'	; calculation for wideband PWM max value ms byte
MAX1		equ	H'56'	; calculation for wideband PWM max value ls byte
SUB0		equ	H'57'	; subtract value
SUB1		equ	H'58'	; subtract value
DIV1		equ	H'59'	; division value
MULT1		equ	H'5A'	; multiplier value
TEMPZ		equ	H'5B'	; temporary

; math routines
TEMP		equ	H'5E'	; temp file
REMB0		equ	H'5F'	; remainder ms
REMB1		equ	H'60'
REMB2		equ	H'61'	; remainder
REMB3		equ	H'62'	; remainder ls
TEMP1		equ H'63'	; temporary
TEMPD		equ	H'64'
AARGB3		equ	H'65'	; ls of argument A
AARGB2      equ H'66'
AARGB1      equ H'67'
AARGB0      equ H'68'	; most significant byte of argument A
TEMPB1      equ H'69'	; temporary
TEMPB0      equ H'6A'
BARGB3      equ H'6B'	; least significant byte of argument B
BARGB2      equ H'6C'	; 
BARGB1      equ H'6D'	; 
BARGB0      equ H'6E'	; most significant byte of argument B
LOOPCOUNT   equ H'6F'  	; loop counter

; All Banks RAM
STATUS_TMP 	equ H'70'	; temp storage for status during interrupt
W_TMP		equ	H'71'	; temporary storage for w during interrupt
PCLATH_STO	equ	H'72'	; PCLATH storage
TEMPX		equ	H'73'	; temporary

START		equ	H'74'	; startup flag (when running from car supply. make sure engine is started
;						; by waiting till supply goes above 13V. Only when J1 installed) 
LOCK		equ	H'75'	; locked control for pump cell
THIRTEEN	equ	H'76'	; 13V set flag	
DIV_S		equ	H'77'	; averaging value for S curve
ADDN_S0		equ	H'78'	; averaging addition ms byts
ADDN_S1		equ	H'79'	; averging addition ls byte
COUNT		equ	H'7A'	; counter during averaging count up
STORE0		equ	H'7B'	; temp store
STORE1		equ	H'7C'	; temp store
ALL_COUNT	equ	H'7D'	; all averaged counter for S-curve

	
; Bank 1 RAM
; indirect addressing so no label is required
; Using H'A0' to H'DF' for S-Curve averaging 
; E3 to E8 Test of Current_IP0,1
 

; define reset and interrupt vector start addresses

	org		0  			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	nop
	org     4			; interrupt vector 0004h, start interrupt routine here
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; bank 0
	bcf		STATUS,RP1
	movf	PCLATH,w	; keep PCLATH
	movwf	PCLATH_STO	; store PC lath
	bcf		PCLATH,3	; page 0
	bcf		PCLATH,4
	goto	INTERRUPT	; start of interrupt routine

; ******************************
; Lookup table for PWM max for duty cycle driven heater
; max varies with battery supply voltage Duty cycle = (VHeff/Vbatt)**2
; returns duty cycle/255
LOOK_PWM
	addwf	PCL,f		; add value of display to program counter
	retlw	D'163'		; when 15V supply for A-D value/4 = 255 and for 12V limit
	retlw	D'164'		; when 14.94V for A-D value/4 = 254 and for 12V limit
	retlw	D'165'		; when 14.88V for A-D value/4 = 253 and for 12V limit
	retlw	D'167'		; when 14.82V for A-D value/4 = 252 and for 12V limit
	retlw	D'168'		; when 14.76V for A-D value/4 = 251 and for 12V limit
	retlw	D'169'		; when 14.71V for A-D value/4 = 250 and for 12V limit
	retlw	D'171'		; when 14.65V for A-D value/4 = 249 and for 12V limit
	retlw	D'172'		; when 14.59V for A-D value/4 = 248 and for 12V limit
	retlw	D'174'		; when 14.53V for A-D value/4 = 247 and for 12V limit
	retlw	D'175'		; when 14.47V for A-D value/4 = 246 and for 12V limit
	retlw	D'176'		; when 14.41V for A-D value/4 = 245 and for 12V limit
	retlw	D'178'		; when 14.35V for A-D value/4 = 244 and for 12V limit
	retlw	D'179'		; when 14.29V for A-D value/4 = 243 and for 12V limit
	retlw	D'181'		; when 14.24V for A-D value/4 = 242 and for 12V limit
	retlw	D'182'		; when 14.18V for A-D value/4 = 241 and for 12V limit
	retlw	D'184'		; when 14.12V for A-D value/4 = 240 and for 12V limit
	retlw	D'185'		; when 14.06V for A-D value/4 = 239 and for 12V limit
	retlw	D'187'		; when 14.00V for A-D value/4 = 238 and for 12V limit
	retlw	D'188'		; when 13.94V for A-D value/4 = 237 and for 12V limit
	retlw	D'190'		; when 13.88V for A-D value/4 = 236 and for 12V limit
	retlw	D'192'		; when 13.82V for A-D value/4 = 235 and for 12V limit
	retlw	D'193'		; when 13.76V for A-D value/4 = 234 and for 12V limit
	retlw	D'195'		; when 13.71V for A-D value/4 = 233 and for 12V limit
	retlw	D'197'		; when 13.65V for A-D value/4 = 232 and for 12V limit
	retlw	D'198'		; when 13.59V for A-D value/4 = 231 and for 12V limit
	retlw	D'200'		; when 13.53V for A-D value/4 = 230 and for 12V limit
	retlw	D'202'		; when 13.47V for A-D value/4 = 229 and for 12V limit
	retlw	D'204'		; when 13.41V for A-D value/4 = 228 and for 12V limit
	retlw	D'205'		; when 13.35V for A-D value/4 = 227 and for 12V limit
	retlw	D'207'		; when 13.29V for A-D value/4 = 226 and for 12V limit
	retlw	D'209'		; when 13.24V for A-D value/4 = 225 and for 12V limit
	retlw	D'211'		; when 13.18V for A-D value/4 = 224 and for 12V limit
	retlw	D'213'		; when 13.12V for A-D value/4 = 223 and for 12V limit
	retlw	D'215'		; when 13.06V for A-D value/4 = 222 and for 12V limit
	retlw	D'217'		; when 13.00V for A-D value/4 = 221 and for 12V limit
	retlw	D'219'		; when 12.94V for A-D value/4 = 220 and for 12V limit
	retlw	D'221'		; when 12.88V for A-D value/4 = 219 and for 12V limit
	retlw	D'223'		; when 12.82V for A-D value/4 = 218 and for 12V limit
	retlw	D'225'		; when 12.76V for A-D value/4 = 217 and for 12V limit
	retlw	D'227'		; when 12.71V for A-D value/4 = 216 and for 12V limit
	retlw	D'229'		; when 12.65V for A-D value/4 = 215 and for 12V limit
	retlw	D'231'		; when 12.59V for A-D value/4 = 214 and for 12V limit
	retlw	D'233'		; when 12.53V for A-D value/4 = 213 and for 12V limit
	retlw	D'236'		; when 12.47V for A-D value/4 = 212 and for 12V limit
	retlw	D'238'		; when 12.41V for A-D value/4 = 211 and for 12V limit
	retlw	D'240'		; when 12.35V for A-D value/4 = 210 and for 12V limit
	retlw	D'242'		; when 12.29V for A-D value/4 = 209 and for 12V limit
	retlw	D'245'		; when 12.24V for A-D value/4 = 208 and for 12V limit
	retlw	D'247'		; when 12.18V for A-D value/4 = 207 and for 12V limit
	retlw	D'250'		; when 12.12V for A-D value/4 = 206 and for 12V limit
	retlw	D'252'		; when 12.06V for A-D value/4 = 205 and for 12V limit
	retlw	D'255'		; when 12.00V for A-D value/4 = 204 and for 12V limit
	
; ******************************************************************************

; start interrupt by saving w and status registers before altered by interrupt routine
INTERRUPT	
	
	bcf		INTCON,TMR0IF

; impedance drive output set at start of interrupt to ensure a good square wave
; not affected by routine length
; check if impedance is on	
	btfss	IMPEDANCE,0	;
	goto	NEG_DRV		; 
	btfss	PORTA,7
	goto	SET7
	bcf		PORTA,7		; impedance drive 5V square wave

	goto	NEG_DRV
SET7
	bsf		PORTA,7		; impedance drive 5V square wave

NEG_DRV ; negative supply driver
	btfss	PORTA,6
	goto	SET6
	bcf		PORTA,6		; negative supply generation driver
	goto	FLG_CK
SET6
	bsf		PORTA,6		; negative supply generation driver	

; flag check
FLG_CK
	btfsc	FLAG,1		; if flag set no A/D allowed as it may occur in main routine
	goto	COUNT_RUN
			
; check if impedance is on	
	btfss	IMPEDANCE,0
	goto	COUNT_RUN

; check impedance every ~10ms
	movlw	D'4'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_1
	movlw	D'44'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_1
	movlw	D'84'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_1
	movlw	D'124'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_1
	movlw	D'164'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_1
	movlw	D'204'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfss	STATUS,Z
	goto	CK2

; for when RA7 is low
; measure voltage across impedance of Nernst cell
; for VS detection and impedance measurement
; A/D conversion
; channel 0
CH0_1
	bcf		ADCON0,3
	bcf		ADCON0,4
	bcf		ADCON0,5
	clrf	SWING_FLG	; data not ready flag

; wait >20us to charge input capacitance. wait 100us for square wave to shape flat
	movlw	D'25'
	movwf	STORE3
WAIT2C1
	decfsz	STORE3,f
	goto	WAIT2C1	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	SWINGL1		; ls voltage when RA7 is low (or possible high instead)
	movf	ADRESH,w
	movwf	SWINGL0		; ms byte
	goto	COUNT_RUN

CK2
	
; check impedance every ~10ms
	movlw	D'3'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_2
	movlw	D'43'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_2
	movlw	D'83'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_2
	movlw	D'123'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_2
	movlw	D'163'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH0_2
	movlw	D'203'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfss	STATUS,Z
	goto	CH6

; for when RA7 is high
; measure voltage across impedance of Nernst cell
; A/D conversion
; channel 0
CH0_2
	bcf		ADCON0,3
	bcf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'25'
	movwf	STORE3
WAIT2C2
	decfsz	STORE3,f
	goto	WAIT2C2	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	SWINGH1		; ls byte when RA7 is high (possibly low instead)
	movf	ADRESH,w
	movwf	SWINGH0		; ms byte
	bsf		SWING_FLG,0	; data ready
	goto	COUNT_RUN

CH6
; check Ip sense every ~10ms

	movlw	D'2'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH6_2
	movlw	D'42'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH6_2
	movlw	D'82'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH6_2
	movlw	D'122'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH6_2
	movlw	D'162'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH6_2
	movlw	D'202'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfss	STATUS,Z
	goto	COUNT_RUN

; A/D conversion
; channel 6
CH6_2
	bcf		ADCON0,3
	bsf		ADCON0,4
	bsf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT2C6
	decfsz	STORE3,f
	goto	WAIT2C6	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	SENSE_IP1
	movf	ADRESH,w
	movwf	SENSE_IP0	; ms byte
	bsf		FLAG,1		; set when end of acquisition set 

COUNT_RUN
	decfsz	COUNT1,f	; counter
	goto	POWER
	movlw	D'244'		
	movwf	COUNT1		; set at 244 counts for 62.5ms period
	decfsz	RATE,f		; ramp rate counter
	goto	HALF_DEC
	bsf		FLAG,0		; flag for each rate increment
	movlw	D'3'
	movwf	RATE		; reset rate counter
HALF_DEC
	movf	DATA_FLSH,f ; data flash
	btfss	STATUS,Z
	decf	DATA_FLSH,f	; reduce to 0	 

	decfsz	HALFSEC,f
	goto	POWER
	bsf		FLAG,2		; uncleared flag indicates when both voltage swings 
						; across Nernst cell have been made
	movf	FOUR_SEC,w	; four seconds counter
	btfss	STATUS,Z	; when zero do not decrease
	decf	FOUR_SEC,f
	movlw	D'8'
	movwf	HALFSEC
	movf	MIN_COUNT,w	; minute counter (120 half seconds)
	btfss	STATUS,Z	; do not decrease if zero
	decf	MIN_COUNT,f	; decrease to zero minute counter

POWER
; power/ heat LED

; flash when heater ready 
	btfss	FLAG,7			; when set, heater ready
	goto	SET_1

; flash LED1 with duty related to heater control
FLASH_LED1
	btfsc	DUTY_FLASH,1	; if set light fully
	goto	SET_1
	btfss	DUTY_FLASH,0	; flash flag
	goto	CLR_1

HALF_DUTY
	btfsc	HALFSEC,2
	goto	SET_1
CLR_1
	bcf		PORTB,1
	goto	ERR_1
SET_1
	bsf		PORTB,1

ERR_1
; error/data LED flash when error
	btfss	ERROR_FLG,7
	goto	ERR_OFF

; flash LED2
FLASH2
	btfsc	HALFSEC,2	
	goto	CLR2
	bsf		PORTB,2	
	goto	HEATER
CLR2
	bcf		PORTB,2
	goto	HEATER

ERR_OFF
; check new data flash indication required
	btfss	ERROR_FLG,6
	goto	ERR_LED_OFF
; if timer finished off
	movf	DATA_FLSH,w
	btfsc	STATUS,Z		; when zero clear flag
	bcf		ERROR_FLG,6		; data flag off
	btfsc	ERROR_FLG,6		; if set then flash LED	
	bsf		PORTB,2			; LED on
	goto	HEATER
ERR_LED_OFF	
	bcf		PORTB,2		; error LED off

; heater PWM driver
HEATER
	incfsz	HEAT_COUNT,f	; heater counter
	goto	DO_PWM
	movf	HEAT,w			; transfer new pwm value at end of cycle
	movwf	HEAT_I			; prevents glitches in PWM
DO_PWM
	movf	HEAT_I,w		; heat value
	btfsc	STATUS,Z		; if zero clear
	goto	CLR_4
	subwf	HEAT_COUNT,w
	btfss	STATUS,C
	goto	SET_4
	btfsc	STATUS,Z
	goto	SET_4
CLR_4
	bcf		PORTA,4
	goto	DRIVE
SET_4 
	bsf		PORTA,4

DRIVE ;(multiplexer and PWM)

; clear multiplexer, reset new PWM for 0-5V, S curve or IP, then set multiplexer for required output
	incf	DRIVE_COUNT,f	; increase each time
; wide test
; clear multiplexer and set PWM, then set multiplexer and drive with PWM
	movlw	D'1'			; 
	subwf	DRIVE_COUNT,w	; PWM drive counter
	btfss	STATUS,C		; 
	goto	CLEAR_MULTIPLEXER		; multiplexer off
	movlw	D'4'	
	subwf	DRIVE_COUNT,w	; PWM drive counter	 
	btfss	STATUS,C		; when less than value set PWM for wideband out
	goto	WIDE_OUT1		; set PWM for wideband
	movlw	D'9'	
	subwf	DRIVE_COUNT,w	; PWM drive counter 
	btfss	STATUS,C		; when <= value set multiplexer output
	goto	WIDE_OUT2
; S-curve test
; clear multiplexer and set PWM, then set multiplexer and drive with PWM
	movlw	D'11'	
	subwf	DRIVE_COUNT,w	; PWM drive counter
	btfss	STATUS,C		; 
	goto	CLEAR_MULTIPLEXER		; multiplexer off
	movlw	D'14'		
	subwf	DRIVE_COUNT,w	; PWM drive counter
	btfss	STATUS,C		; 
	goto	S_CURVE6		; when less than value set pwm for S curve
	movlw	D'19'	
	subwf	DRIVE_COUNT,w	; PWM drive counter
	btfss	STATUS,C		; when <= value set multiplexer output
	goto	S_CURVE7
; IP test
; clear multiplexer and set PWM, then set multiplexer and drive with PWM
	movlw	D'21'	
	subwf	DRIVE_COUNT,w	; PWM drive counter			
	btfss	STATUS,C		; 
	goto	CLEAR_MULTIPLEXER		; multiplexer off
	movlw	D'24'	
	subwf	DRIVE_COUNT,w	; PWM drive counter		
	btfss	STATUS,C		; when less than value set IP
	goto	IP_OUT12
	movlw	D'29'	
	subwf	DRIVE_COUNT,w	; PWM drive counter
	btfss	STATUS,C		; when <= value set multiplexer output
	goto	IP_OUT13
; clear counter
	clrf	DRIVE_COUNT		; ready to start again
	decf	DRIVE_COUNT,f	; set at FF
	goto	RECLAIM

CLEAR_MULTIPLEXER
	movlw	B'11001111'		; bits 4 and 5 clear
	andwf	PORTB,f			; set multiplexer to 0
	goto	RECLAIM

; load pwm for 0-5V out
WIDE_OUT1
	movf	WIDEBAND0,w		; ms byte first
	movwf	STOREH			; store
	movf	WIDEBAND1,w		; ls byte
	movwf	STOREL			; store

LOAD_PWM
	bcf		CCP1CON,4		; ls bits PWM operation
	btfsc	STOREL,0		; wideband ls		
	bsf		CCP1CON,4		; ls bits PWM operation
	bcf		CCP1CON,5		; ls bits PWM operation
	btfsc	STOREL,1		; wideband ls		
	bsf		CCP1CON,5		; ls bits PWM operation	
; move right to find ms byte
	rrf		STOREH,f
	rrf		STOREL,f
	rrf		STOREH,f
	rrf		STOREL,w
	movwf	CCPR1L			; ms byte PWM
	goto	RECLAIM
WIDE_OUT2
; set multiplexer for wideband 0-5V out
	movf	PORTB,w
	iorlw	B'00010000'	; set bit 4
	andlw	B'11011111'	; bit 4 set and 5 clear for the '1' output
	movwf	PORTB		; set multiplexer to 2
	goto	RECLAIM

S_CURVE6

; ls bits
	movf	S_CURVE,w
	movwf	STOREL
	bcf		CCP1CON,4	; ls bits PWM operation
	btfsc	STOREL,0	; S-curve value	
	bsf		CCP1CON,4	; ls bits PWM operation
	bcf		CCP1CON,5	; ls bits PWM operation
	btfsc	STOREL,1	; s curve value		
	bsf		CCP1CON,5	; ls bits PWM operation	
	bcf		STATUS,C	; carry clear
	rrf		STOREL,f
	bcf		STATUS,C
	rrf		STOREL,w	
	movwf	CCPR1L		; ms byte PWM
	goto	RECLAIM
S_CURVE7
; set multiplexer for S-curve
	movf	PORTB,w
	iorlw	B'00100000'	; set bit 5
	andlw	B'11101111'	; bit 5 set and 4 clear for the '2' output
	movwf	PORTB		; set multiplexer to 1
	goto	RECLAIM

IP_OUT12
; load pwm for 0-5V out
	movf	CURRENT_IP0,w	; ms byte first
	movwf	STOREH		; store
	movf	CURRENT_IP1,w	; ls byte
	movwf	STOREL			; store
	goto	LOAD_PWM

IP_OUT13
; set multiplexer for Ip
	movf	PORTB,w
	iorlw	B'00110000'	; set bit 4 and 5
	andlw	B'11111111'	; bit 4 and 5 set for the '3' output
	movwf	PORTB		; set multiplexer to 3

; end of interrupt reclaim w and status 
RECLAIM
	movf	PCLATH_STO,w; restore PCLATH	
	movwf	PCLATH
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

;***************************************************************************************

; RESET		
; Set ports A & B

MAIN
; initial values
	
; set inputs/outputs
	movlw	H'00'		; set all outputs low
	movwf	PORTA
	movwf	PORTB
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'11000001'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'00000000'	; 
	movwf	OPTION_REG	; TMRO prescaler is 2, (3.906kHz)
	movlw   B'00101111'	; I/O 
	movwf   TRISA		; port A data direction register

; analog inputs, A/D
	movlw	B'01101111'	;  AN0,1,2,3,5,6 are analog inputs 
	movwf	ANSEL
	movlw	B'11000000'	; right justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0

	movlw	B'01000000'	; Fosc, channel etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on

	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01111000'	; for 8MHz
	movwf	OSCCON		; osc
; pwm set
	movlw	H'FE'
	movwf	PR2			; PWM period register
	bcf		STATUS,RP0	; memory bank 0
	movlw	H'00'		; 0% duty 
	movwf	CCPR1L		; ms byte of PWM
	movlw	B'00000000'
	movwf	T2CON
	bsf		T2CON,2		; enable timer 2
	movlw	B'00001100'	; set PWM mode
	movwf	CCP1CON		; enable PWM operation

; Initialise
	clrf	IMPEDANCE	; impedance measurement off
	movlw	H'1'		; initial VH effective for heater D504 or H1F8 for 7.4V effective
	movwf	VH_EFF0		; heater effective voltage ms
	movlw	H'F8'		; initial VH effective for heater D504 or H1F8
	movwf	VH_EFF1		; heater effective voltage ls

; Temperature of sensor degrees C -40, -10, 20,  50
; VH effective			standard  7.4, 7.8, 8.2, 8.6
; fast (with all requirements)	  9.0, 9.5, 10.0, 10.5

; for 7.8V Veff is D532 or H214		
; for 8.2V Veff is D559 or H22F

; other values calculated by multiplying required effective voltage by 68.2 
; and converting to a hexadecimal value (2-byte)
;(actual calulation: /3 for battery voltage divider then /5 for value compared to A/D reference,
; multiply by 1023 for 10-bit range). Result is the same as x 68.2


FAST_LIGHT ;(fast heatup allowed if installed correctly in exhaust) 

; allow instruction for fast light off (fast heatup)
; For fast light off remove semicolon at the beginning of the following line

;	btfsc	PORTB,0		; if J1 in then fast light off starts at 9V instead

	goto 	NO_FAST_LIGHT
	movlw	H'2'		; initial VH effective for heater D613 or H265 for 9.0VHeffective
	movwf	VH_EFF0		; heater effective voltage ms
	movlw	H'65'		; initial VH effective for heater D613 or H265
	movwf	VH_EFF1		; heater effective voltage ls

NO_FAST_LIGHT
	clrf	LOCK		; out of lock 
	clrf	ERROR_FLG	; error flag
	clrf	HEAT		; heater PWM calculated value
	clrf	HEAT_I		; interrupt heat value
	clrf	SWING_FLG	; amplitude swing flag
	clrf	START		; startup flag
	bcf		THIRTEEN,0	; 13V flag

; set PWM 
	clrf	WIDEBAND0	; wideband value ms
	clrf	WIDEBAND1	; wideband value ls
	clrf	S_CURVE		; S curve output

; timers
	clrf	DRIVE_COUNT	; PWM drive counter
	movlw	D'3'
	movwf	RATE		; ramp rate 187.5ms	
	clrf	DATA_FLSH	; data flash
	movlw	D'244'		
	movwf	COUNT1		; 62.5ms counter (x 8 for 500ms)
	movlw	D'60'
	movwf	MIN_COUNT	; minute counter to 60 for 1/2 minute (30 seconds) countdown
	movlw	D'8'
	movwf	FOUR_SEC	; four seconds (8-half seconds)
	movwf	HALFSEC		; half second
	clrf	DUTY_FLASH	; duty cycle flash for heater control
	
; clear the 64 S-Curve averaging registers A0 to DF

	movlw	H'A0'		; start of bank 1 registers
	movwf	FSR
CLEAR_ALL
	clrf	INDF		; clear 
	incf	FSR,f
	movf	FSR,w
	sublw	H'DF'		; if over DF then finished clearing 
	btfsc	STATUS,C
	goto	CLEAR_ALL

	clrf	ADDN_S0		; averaging addition ms byts
	clrf	ADDN_S1		; averging addition ls byte

; flags
	clrf	FLAG		; indicator flags
	bsf		FLAG,0		; 187.5ms flag


; measure VS/IP offset and so set current to pump cell to 0mA	
; A/D conversion
; channel 1
	bsf		ADCON0,3
	bcf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT2VS1
	decfsz	STORE3,f
	goto	WAIT2VS1	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	CURRENT_IP1	; IP current ls 
	movwf	VS_IP1		; VS/IP ms byte
	movf	ADRESH,w
	movwf	CURRENT_IP0	; IP current ms	
	movwf	VS_IP0		; VS/IP ls byte

; measure S curve response rate setting
; A/D conversion
; channel 2
	bcf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT_S1
	decfsz	STORE3,f
	goto	WAIT_S1	
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	S_RESP		; temporary ls
	rrf		ADRESH,f
	rrf		S_RESP,f	; remove ls bits for 8-bit A/D
	rrf		ADRESH,f
	rrf		S_RESP,f	; S curve response
	movf	S_RESP,w
	movwf	ALL_COUNT	; averaging counter. Only change S-CURVE after first averaging set has occurred 
	
; measure battery	
; A/D conversion
; channel 3
MEAS_BATT

	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'11'
	movwf	STORE3
WAITB20
	decfsz	STORE3,f
	goto	WAITB20	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	BATT1
	movf	ADRESH,w
	movwf	BATT0

; interrupt enable 
	bsf		INTCON,TMR0IE; set interrupt enable for TMR0 
	bsf		INTCON,GIE	; set global interrupt enable 
	bsf		IMPEDANCE,0	; impedance on

; check for car installation (J1 in)
	btfsc	PORTB,0		; when set not an installation
	goto	HEATER_RUN	; heater startup
	btfsc	THIRTEEN,0	; when set 13V has been reached
	goto	HEATER_RUN

; installation so check battery voltage. Only start at 13V or greater
	movlw	H'03'			; 13V max V 
	subwf	BATT0,w			; ms battery  
	movwf	CALC0
	movlw	H'76'			; 13V max 
	subwf	BATT1,w			; ls 
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfss	CALC0,7			; if set then battery V < 13V
	goto	HEATER_RUN
	movlw	D'8'			; keep timer set
	movwf	FOUR_SEC
	goto	MEAS_BATT		; measure battery

; program returns here
HEATER_RUN
	bsf		THIRTEEN,0		; 13V set flag

; HEATER Control
; initial power up phase starts at VH effective of 7.4V. Ramps up at 0.4V per second 
; to 12V where the impedance of the Nernst cell is checked
; Current must not exceed 4A
; Duty cycle = (VH effective/ Vbattery ) squared
; battery voltage : H3FF from A/D is 15V

; check for ~10ms
	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_Z
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_Z
	
MEASURE_BATT
; measure battery	
; A/D conversion
; channel 3
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT20
	decfsz	STORE3,f
	goto	WAIT20	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	BATT1
	movf	ADRESH,w
	movwf	BATT0

	decfsz	TWO_COUNT,f	; decrease counter and only measure CH2 occasionally
	goto	OFFSET_MEAS

; measure S curve response rate setting
; A/D conversion
; channel 2
	bcf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT_S2
	decfsz	STORE3,f
	goto	WAIT_S2	
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	S_RESP		; temporary ls
	rrf		ADRESH,f
	rrf		S_RESP,f	; remove ls bits for 8-bit A/D
	rrf		ADRESH,f
	rrf		S_RESP,f	; S curve response

OFFSET_MEAS
; measure VS/IP offset
	movf	TWO_COUNT,w	; measure when count is 127
	xorlw	D'127'
	btfss	STATUS,Z
	goto	HEAT_I_AD	

; A/D conversion
; channel 1
	bsf		ADCON0,3
	bcf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT2VS11
	decfsz	STORE3,f
	goto	WAIT2VS11	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	VS_IP1		; VS/IP ms byte
	movf	ADRESH,w
	movwf	VS_IP0		; VS/IP ls byte

HEAT_I_AD

; measure heater current	
; A/D conversion
; channel 5
	bsf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT20H
	decfsz	STORE3,f
	goto	WAIT20H	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	STORE1
	movf	ADRESH,w
	movwf	STORE0

; check for over 4A (0.4V across 0.1 ohm) equivalent to D82 in A/D
; compare with (D82). If over set error flag (ERROR_FLG,7)and stop heater.

	movf	STORE0,w	; high byte if zero check ls byte
	btfss	STATUS,Z
	goto	SET_ERR		; not zero so set error
; check for over D82
	movf	STORE1,w
	sublw	D'82'
	btfss	STATUS,C	; if > error
	goto	SET_ERR

; check if <8 for open circuit
; check after first 4-seconds
	movf	FOUR_SEC,w 	; 
	btfss	STATUS,Z	; 
	goto	RAMP_UP
	btfsc	FLAG,7		; ramp up phase finished if flag is set
	goto	RAMP_UP		; bypass o/c test after ramp up
	movf	STORE1,w
	sublw	D'8'
	btfsc	STATUS,C
	goto	SET_ERR
	bcf		ERROR_FLG,7	; no error so clear flag
	goto	RAMP_UP

; under or over-current so off
SET_ERR
	clrf	VH_EFF0
	clrf	VH_EFF1		; effective Voltage at 0
	bsf		ERROR_FLG,7	; set error flag
	clrf	IMPEDANCE	; impedance measurement off
	clrf	HEAT		; heater off
	bcf		PORTB,1		; heater LED off
; set PWM 
; wideband to lambda 1 (1.35V)
	movlw	H'1'
	movwf	WIDEBAND0	; wideband value ms
	movlw	H'15'
	movwf	WIDEBAND1	; wideband value ls
; S curve to 0.45V (D92)
	movlw	D'92'
	movwf	S_CURVE		; S curve output

; set current pump off 
; measure VS/IP offset	
; A/D conversion
; channel 1
	bsf		ADCON0,3
	bcf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'15'
	movwf	STORE3
WAIT2VS
	decfsz	STORE3,f
	goto	WAIT2VS	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	CURRENT_IP1	; IP current ls
	movf	ADRESH,w
	movwf	CURRENT_IP0	; IP current ms	
	bcf		FLAG,1		; clear ready for next ~10ms
SELF
	goto	SELF		; cycle without control

; check for ramp up phase
RAMP_UP
	bcf		FLAG,1		; clear ready for next ~10ms
	btfsc	FLAG,7		; ramp up phase finished if flag is set
	goto	HEAT_CON	; control loop of temperature

; check 187.5ms flag
	btfss	FLAG,0		; if set 187.5ms flag
	goto	DUTY_CALC	
	bcf		FLAG,0

; ramp up at 73.3mV/187.5ms (0.4V per second maximum ramp rate spec.) 
; equivalent to digital value of 5
; starts at 7.4V
; this is a Veff ramp up that needs the PWM to be calculated

	movlw	D'5'		; 73.3mV
	addwf	VH_EFF1,f	; heater effective voltage
	btfsc	STATUS,C
	incf	VH_EFF0,f	; increase ms byte when carry
	call	CK_LIMIT	; 
DUTY_CALC
	call	DUTY		; calculate duty cycle for Veff depending on battery Voltage
	movwf	HEAT
	goto	HEAT_CON 

CK_LIMIT
; stop increment at Veff = 13V or Veff = 12V. Use Veff = 13V at start and Veff = 12V after 1 minute.
; if MIN_COUNT is clear 30 seconds have elapsed and use Veff = 12V as the maximum
; 30 seconds allows 2 x 60 x 200h (max allowable hours at 13V is 200h) or 24000 starts
; equivalent to more than 240,000kms if average trip is 10kms or more.
; A Veff of 12V is allowable continuously

; check minute counter
	movf	MIN_COUNT,w	; minute counter
	btfsc	STATUS,Z
	goto	COMP12		; 12V maximum	

; set 13V as maximum
; compare with 13V [D886, H376]
	movlw	H'03'			; 13V max V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movlw	H'76'			; 13V max 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfsc	CALC0,7			; if set then V eff < 13V
	return
; Veff > 13V so fix at 13V
	movlw	H'03'
	movwf	VH_EFF0			; set at 13V max
	movlw	H'76'			;
	movwf	VH_EFF1			; fix at 13V max 
	return

COMP12
; set 12V as maximum after 60-seconds
; compare with 12V [D818, H332]
	movlw	H'03'			; 12V max V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movlw	H'32'			; 12V max 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfsc	CALC0,7			; if set then V eff < 12V
	return
; Veff > 12V so fix at 12V
	movlw	H'03'
	movwf	VH_EFF0			; set at 12V max
	movlw	H'32'
	movwf	VH_EFF1			; fix at 12V max 
	return

DUTY
; calculation of PWM duty cycle during ramp up
; calculate duty cycle based on Vbatt and Veff
; Vbatt is 15V with H3FF measured 
; duty cycle is (VH effective/VBatt)squared. Result is a fraction between 1 and 0 for a heat value of 
; between 255 and 0
; Method: multiply VH effective by 255 (to gain resolution in result) and divide by VBatt. 
; Then square the value and divide by 255 to get a range of 0-255.

; if Vbatt <= Veff then set HEAT at maximum 

	movf	BATT0,w			; ms battery V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movf	BATT1,w			; ls batt 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	
	btfsc	CALC0,7			; if set then battery V > V eff
	goto	DO_CALC
	movlw	H'FF'			; set heat at max
	return

DO_CALC

; calculate duty cycle based on Vbatt and Veff
	movf	VH_EFF0,w
	movwf	AARGB0
	movf	VH_EFF1,w
	movwf	AARGB1
	clrf	BARGB0
	movlw	H'FF'
	movwf	BARGB1
	bsf		PCLATH,3	; page 1
	call	FXM1616U	; multiply Veff x 255
	
	movf	BATT0,w		; battery volts divisor
	movwf	BARGB2
	movf	BATT1,w		; min of 4 for 16 bit result
	movwf	BARGB3	
	clrf	BARGB0
	clrf	BARGB1
	call	FXD3232U	; divide by battery Voltage

; shift
	movf	AARGB2,w	; numerator
	movwf	AARGB0
	movwf	BARGB0
	movf	AARGB3,w
	movwf	AARGB1
	movwf	BARGB1
	call	FXM1616U	; square the value

; divide by 255
	clrf	BARGB0
	clrf	BARGB1
	clrf	BARGB2	
	movlw	H'FF'	
	movwf	BARGB3
	call	FXD3232U
;	result in AARGB3
	bcf		PCLATH,3	; page 0
	movf	AARGB3,w
	return
	
HEAT_CON
; two loops check FLAG,7
; 1. during ramp up do not increase duty, but let ramp up process occur
; 2. after ramp up can alter heater drive at will. Uses lookup table to check if 
; the 12V Veff has been reached and alters the PWM value for control rather than Veff 
; set FLAG,7 when Nernst cell is 80 ohms indicating end of heatup phase
; when driving Nernst cell from RA7 with +/-2.5V ac via a 10.5k ohm impedance, 
; there is a 177mV swing from x 4.7 amplifier both high and low for 80 ohm on Nernst cell. 
; Equivalent to a change of 36 digital value from A/D

 
; check if first swing values have been measured
	btfss	FLAG,2		; cycle ready flag 
	goto	CURRENT_CONTROL

; take SWINGH0,1 from SWINGL0,1 (or vice versa if negative) to get difference in swing across Nernst cell
; (voltages measured in interrupt)

WAIT_SWING1 ; wait for data
	btfss	SWING_FLG,0		; if swing flag set can use data
	goto	WAIT_SWING1
	bcf		STATUS,GIE		; stop interrupt
	
; subtract Lower swing from Upper swing
	movf	SWINGL0,w		; ms lower 
	subwf	SWINGH0,w		; ms upper  
	movwf	SPAN0
	movf	SWINGL1,w		; ls lower 
	subwf	SWINGH1,w		; ls upper  
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry
	btfss	SPAN0,7			; if set then Lower swing > upper swing so rearrange calculation
	goto	COMPARE80

; subtract Upper swing from Lower swing
	movf	SWINGH0,w		; ms upper  
	subwf	SWINGL0,w		; ms lower 
	movwf	SPAN0
	movf	SWINGH1,w		; ls upper 
	subwf	SWINGL1,w		; ls lower
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry	
	
COMPARE80
	bsf		STATUS,GIE		; re allow interrupt
	movf	SPAN0,w
	btfss	STATUS,Z		; when ms byte of span is not 0, cell is >>80 ohms
	goto	SET_FF			; maximum heating allowed (either ramp up voltage if FLAG,7 not set
							; or at maximum (12V Veff) for FLAG,7 set.

CK_SPAN 
; D'36' is span for >80 ohms using 10.5k drive and 4.7 amplifier gain
	movf	SPAN1,w
	sublw	D'37'			; slightly >80 ohms
	btfss	STATUS,C		; if minus (c=0) >80 ohms
	goto	SET_FF			; maximum heating allowed (either ramp up voltage if FLAG,7 not set
							; or at maximum (12V Veff) for FLAG,7 set.

; when < = 80 ohms set FLAG,7 (ramp up complete)

	bsf		FLAG,7			; end of ramp up
; check span values
	movf	SPAN1,w
	xorlw	D'37'			; >80 ohms
	btfsc	STATUS,Z		; 
	goto	HEAT_SLOW
	movf	SPAN1,w
	xorlw	D'36'			; 80 ohms
	btfsc	STATUS,Z		; 
	goto	COOL_SLOW
; COOL
SET_00 ;(cool)
	clrf	DUTY_FLASH		; clear duty flash
	clrf	HEATX
	clrf	DUTY_FLASH		; stop duty flash
	goto	CHECK_MAX
; HEAT
SET_FF ;(heat)
	movf	HEAT,w			; get heat value
	movwf	HEATX			; place in working register
	btfss	FLAG,7			; if heatup ramp bypass
	goto	CHECK_MAX
	movlw	H'FF'			; set heat at max	
	movwf	HEATX
	bcf		DUTY_FLASH,0	; stop duty flash
	bsf		DUTY_FLASH,1	; lit fully for heat
	goto	CHECK_MAX

HEAT_SLOW
	movlw	D'255'
	movwf	HEATX
	bsf		DUTY_FLASH,0	; allow duty flash
	bcf		DUTY_FLASH,1	; clear lit fully 
	goto	CHECK_MAX
COOL_SLOW
	movlw	D'20'
	movwf	HEATX
	bsf		DUTY_FLASH,0	; allow duty flash
	bcf		DUTY_FLASH,1	; clear lit fully 
	goto	CHECK_MAX

; check 12V Veff limit
; check if maximum duty for Veff of 12V has been reached
CHECK_MAX
; if in ramp up heat mode bypass max check
	btfss	FLAG,7		
	goto	CURRENT_CONTROL	
	rrf		BATT0,f		; divide battery volts by 4
	rrf		BATT1,f
	rrf		BATT0,w
	rrf		BATT1,f

	movf	BATT1,w
	sublw	D'204'
	movlw	D'204'		; load w ready
	btfsc	STATUS,C	; if plus then set at 204 (12V)
	movwf	BATT1
	comf	BATT1,w		; reverse the order
	call	LOOK_PWM	; get PWM value for 12V Veff max for the given battery voltage

	movwf	STORE0		; temp value
	subwf	HEATX,w
	movf	STORE0,w	; load limit to w ready 
	btfsc	STATUS,C	; if negative value was ok	
	movwf	HEATX		; limit PWM

	movf	HEATX,w
	movwf	HEAT
	
CURRENT_CONTROL 
; keep current at zero until heater is in closed loop maintaining 750 degrees when FLAG,7 is set

	btfsc	FLAG,7	; 
	goto	CURRENT_LOOP

; measure VS/IP offset and so set current to pump cell to 0mA	
; A/D conversion

	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_1
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_1
; channel 1
	bsf		ADCON0,3
	bcf		ADCON0,4
	bcf		ADCON0,5
	
; wait >20us to charge input capacitance
	movlw	D'20'
	movwf	STORE3
WAIT2VSIP
	decfsz	STORE3,f
	goto	WAIT2VSIP	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	CURRENT_IP1	; IP current ls
	movf	ADRESH,w
	movwf	CURRENT_IP0	; IP current ms	

; set S-CURVE ouput to 0
	clrf	S_CURVE
	goto	HEATER_RUN

CURRENT_LOOP

; VS detect assumes 2.5V (average) at AN0 for 450mV Nernst Cell Voltage. Above 2.5 is rich. 
; Below 2.5V is lean

; VS is (SWINGL0,1 + SWINGH0,1)/2
; add

WAIT_SWING2 ; wait for data
	btfss	SWING_FLG,0		; if swing flag set can use data
	goto	WAIT_SWING2
	bcf		STATUS,GIE		; stop changes in interrupt
	bcf		SWING_FLG,0
	bcf		FLAG,1			; IP sense flag
	movf	SWINGL0,w
	addwf	SWINGH0,w
	movwf	VS_AVG0			; Ip sense average ms
	movf	SWINGL1,w
	addwf	SWINGH1,w
	bsf		STATUS,GIE		; allow changes in interrupt
	movwf	VS_AVG1			; Ip sense average ls
	btfsc	STATUS,C		; check carry	
	incf	VS_AVG0,f		; increase when ls byte overflows
; divide by 2
	bcf		STATUS,C		; carry clear
	rrf		VS_AVG0,f
	rrf		VS_AVG1,f

; if above 2.5V then mixture is rich so reduce CURRENT_IP0,IP1 in value
; if below 2.5V then mixture is lean so increase CURRENT_IP0,IP1 in value
; allow a hysteresis range above and below 2.5V because of sharp narrow band sensor output

; compare with 2.5V (H1FF)
; subtract VS_AVG0,1 from H1FF

	movf	VS_AVG0,w		; ms byte 
	sublw	H'1'			; ms 
	movwf	TEMP_AV0
	movf	VS_AVG1,w		; ls 
	sublw	H'FF'			; ls 
	movwf	TEMP_AV1
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry
	iorwf	TEMP_AV0,w		; check if both TEMP_AV0 and TEMP_AV1 are at 0
	btfsc	STATUS,Z		; if zero bypass PWM change
	goto	SET_OUTPUT
	btfsc	TEMP_AV0,7		; if set then VS_AVG0,1 > H1FF so rearrange calculation
	goto	REDUCE
	
; decrease hysteresis
; subtract D'1' (4.8mV) from VS_AVG0,1 equivalent to 1,2mV on VS
	movlw	D'1'			; ls 
	subwf	TEMP_AV1,w		; to 'w' for comparison with 0 (ls byte)
	movwf	TEMP_AV1		; equivalent to subwf TEMP_AV1,f but with value in w as well as f 
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry
;
	iorwf	TEMP_AV0,w		; check if both TEMP_AV0 and TEMP_AV1 are at 0
	btfsc	STATUS,Z		; if zero bypass PWM change
	goto	SET_OUTPUT
;
	btfss	TEMP_AV0,7		; if set then negative so set at 0
	goto	PROP_INC
	clrf	TEMP_AV0
	clrf	TEMP_AV1
	goto	SET_OUTPUT

; setup a proportionate increase rate
; divide offset from 2.5V by value x
PROP_INC 
	movf	TEMP_AV0,w		; ms
	movwf	AARGB0
	movf	TEMP_AV1,w
	movwf	AARGB1
	movlw	D'4'			;  divider (3 minimum)

	movwf	BARGB0
	bsf		PCLATH,3		; page 1
	call	DIV16_8			; divide
	bcf		PCLATH,3		; page 0
	movf	AARGB1,w		; result
	btfsc	STATUS,Z		; if zero use 1
	goto	INC1
	decf	AARGB1,w		; if a 1 check TEMP_AV1
	btfsc	STATUS,Z		; 
	goto	INC1

INCREASE
; add value to PWM
	movf	AARGB1,w		; result
INCREASE1
	addwf	CURRENT_IP1,w
	btfss	STATUS,C		; if carry increase ms byte if 2 or less
	goto	INC_LS
	bcf		STATUS,GIE
	movwf	CURRENT_IP1
	movf	CURRENT_IP0,w
	sublw	D'2'			; if 3 do not increase
	btfss	STATUS,C
	goto	SET_MAX			; no increase
	incf	CURRENT_IP0,f
	goto	SET_OUTPUT

SET_MAX
; set at 3FF
	movlw	H'FF'
	movwf	CURRENT_IP1
	movlw	H'03'
	movwf	WIDEBAND0		; ms PWM byte for wideband output
	movlw	H'FF'
	movwf	WIDEBAND1		; ls PWM byte for wideband output	
	bsf		ERROR_FLG,7		; error LED on at limit 
	goto	SET_OUTPUT1
INC_LS
	bcf		STATUS,GIE
	movwf	CURRENT_IP1
	goto	SET_OUTPUT

INC1
	movlw	D'01'
	goto	INCREASE1

REDUCE
; subtract H1FF (2.5V) from VS_AVG0,1
	movlw	H'1'
	subwf	VS_AVG0,w		; ms byte 
	movwf	TEMP_AV0
	movlw	H'FF'			; ls 
	subwf	VS_AVG1,w		; ls 
	movwf	TEMP_AV1
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry

; decrease hysteresis
; subtract D'1' (4.8mV) from VS_AVG0,1 equivalent to 1.2mV on Vs
	movlw	D'1'			; ls 
	subwf	TEMP_AV1,w		; w for comparison (ls byte)
	movwf	TEMP_AV1		;  
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry
;
	iorwf	TEMP_AV0,w		; check if both TEMP_AV0 and TEMP_AV1 are at 0
	btfsc	STATUS,Z		; if zero bypass PWM change
	goto	SET_OUTPUT
;	
	btfss	TEMP_AV0,7		; if set then negative so set at 0
	goto	PROP_DEC
	clrf	TEMP_AV0
	clrf	TEMP_AV1
	goto	SET_OUTPUT

; setup a proportionate increase rate
; divide offset from 2.5V by value x
PROP_DEC
	movf	TEMP_AV0,w		; ms
	movwf	AARGB0
	movf	TEMP_AV1,w
	movwf	AARGB1
	movlw	D'4' 			; value x, 3 minimum
	movwf	BARGB0
	bsf		PCLATH,3		; page 1
	call	DIV16_8			; divide
	bcf		PCLATH,3		; page 0
; reduce value from PWM
	movf	CURRENT_IP0,w	; ms byte 
	movwf	TEMP_PWM0
	movf	AARGB1,w		; result
	btfsc	STATUS,Z		; if zero use 1
	goto	DEC1
	decf	AARGB1,w		; result -1
	btfsc	STATUS,Z		; if zero use 1
	goto	DEC1			; 

RED_VALUE
	movf	AARGB1,w		; result
RED_VALUE1
	subwf	CURRENT_IP1,w	; ls 
	movwf	TEMP_PWM1
	btfss	STATUS,C
	decf	TEMP_PWM0,f		; decrease if carry
	btfsc	TEMP_PWM0,7		; if set then negative set at 0
	goto	ZERO_SET
; transfer to PWM
	bcf		STATUS,GIE	
	movf	TEMP_PWM0,w		; ms byte 
	movwf	CURRENT_IP0
	movf	TEMP_PWM1,w		; ls byte 
	movwf	CURRENT_IP1
	goto	SET_OUTPUT
DEC1
	movlw	D'1'
	goto	RED_VALUE1	
ZERO_SET
	bcf		STATUS,GIE
	clrf	CURRENT_IP0
	clrf	CURRENT_IP1	
	clrf	WIDEBAND0		; ms PWM byte for wideband output
	clrf	WIDEBAND1		; ls PWM byte for wideband output
	bsf		ERROR_FLG,7		; error LED on at limit 
	goto	SET_OUTPUT1


; allow 10ms for pump action
; Nernst cell at 450mV is set to 2.5V

SET_OUTPUT 
	
; When AN0 (Vs - 0.45V x 4.7V) is close to 2.5V (0.45V across Nernst cell)
; VS_AVG0,1 is the average AN0 value
; TEMP_AV1 is the offset from 2.5V 
; so if TEMP_AV1 is zero load values

	bcf		ERROR_FLG,7		; ERROR LED off
SET_OUTPUT1 

; *******************
; testing the impedance span (start) places impedance on wideband output 
;	movf	SPAN0,w
;	movwf	WIDEBAND0
;	movf	SPAN1,w
;	movwf	WIDEBAND1
;	bsf		STATUS,GIE
;	goto	HEATER_RUN
; testing the impedance span (end)
; ********************

	bsf		STATUS,GIE		; resume interrupt
	movf	TEMP_AV0,w
	btfss	STATUS,Z
	goto	NEW_RUN 		; if ms byte is zero check ls byte
	
	movf	TEMP_AV1,w		; if zero, load curve outputs
	btfss	STATUS,Z		; if 0 value 
	goto	HEATER_RUN		
	btfsc	LOCK,0			; if in lock last time (if remarked then no 2 x lock testing)
	goto	CK_CURRENT
	bsf		LOCK,0
	goto	HEATER_RUN
NEW_RUN
	clrf	LOCK
	goto	HEATER_RUN

CK_CURRENT ;

;	goto	DATA_READY			; if this line is remarked out, adds extra test for valid data
; check if Ip current is stable

; data stored in E3,E4; E5,E6 and E7,E8	
; move ms byte data along in time E5 to E7 and E3 to E5, CURRENT_IP0 to E3	
; move ls byte data along in time E6 to E8 and E4 to E6. CURRENT_IP1 to E4
; ms bytes
; E5 to E7
	movlw	H'E5'
	movwf	FSR
	movf	INDF,w
	incf	FSR,f
	incf	FSR,f
	movwf	INDF
; E3 to E5
	movlw	H'E3'
	movwf	FSR
	movf	INDF,w
	incf	FSR,f
	incf	FSR,f
	movwf	INDF
; CURRENT_IP0 to E3
	movlw	H'E3'
	movwf	FSR
	movf	CURRENT_IP0,w
	movwf	INDF
; ls bytes
; E6 to E8
	movlw	H'E6'
	movwf	FSR
	movf	INDF,w
	incf	FSR,f
	incf	FSR,f
	movwf	INDF
; E4 to E6
	movlw	H'E4'
	movwf	FSR
	movf	INDF,w
	incf	FSR,f
	incf	FSR,f
	movwf	INDF
; CURRENT_IP1 to E4
	movlw	H'E4'
	movwf	FSR
	movf	CURRENT_IP0,w
	movwf	INDF
; compare current values if all equal data is stable
; E3,E5 and E7
	movlw	H'E3'
	movwf	FSR
	movf	INDF,w
	incf	FSR,f
	incf	FSR,f
	xorwf	INDF,w		; E3 and E5
	btfss	STATUS,Z
	goto	HEATER_RUN	; data not equal
	movf	INDF,w		; 
	incf	FSR,f
	incf	FSR,f
	xorwf	INDF,w		; E5 and E7
	btfss	STATUS,Z
	goto	HEATER_RUN	; data not equal
; E4,E6 and E8
	movlw	H'E4'
	movwf	FSR
	movf	INDF,w
	incf	FSR,f
	incf	FSR,f
	xorwf	INDF,w		; E4 and E6
	btfss	STATUS,Z
	goto	HEATER_RUN	; data not equal
	movf	INDF,w		; 
	incf	FSR,f
	incf	FSR,f
	xorwf	INDF,w		; E6 and E8
	btfss	STATUS,Z
	goto	HEATER_RUN	; data not equal
DATA_READY
; when data is correct, flash DATA LED 
	bsf		LOCK,0
	movlw	D'2'			; 
	movwf	DATA_FLSH		; flash timer
	bsf		ERROR_FLG,6		; data flag

; Check for Air Calibration mode (VS/IP = 2V)or normal mode.

	movlw	D'02'			; VS/IP ms
	subwf	VS_IP0,w	
	btfsc	STATUS,C		; if positive normal mode
	goto	NORM_READINGS

AIR_READINGS; (LEAN only)

	clrf	S_CURVE			; S-curve output is 0

; For air readings Calibration VS/IP is 2.0
; TP4 is 2.343V
; Oxygen content of air is at 20.9% 
; set output at voltage to represent O2%.(20.9% = 2.09V)

; get (amplified) IP sense value and subtract from VS_IP
; SENSE_IP0,1 is IP sense at AN6
; subtract VS_IP from IP Sense
	btfss	FLAG,1			; if set can take reading
	goto	AIR_READINGS
	bcf		FLAG,1
	bcf		STATUS,GIE		; stop changes in interrupt
	movf	VS_IP0,w		; ms VS_IP 
	subwf	SENSE_IP0,w		; ms IP sense  
	movwf	TEMPB0
	movf	VS_IP1,w		; ls  
	subwf	SENSE_IP1,w		; ls  
	movwf	TEMPB1
	bsf		STATUS,GIE		; allow changes in interrupt
	btfss	STATUS,C
	decf	TEMPB0,f		; decrease if carry
	btfsc	TEMPB0,7		; if set then VS_IP > IP sense so clear
	goto	CLEAR_CURRENT_READING

; <= D55?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'55'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 55 
	goto	AIR1

; <= D109?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'109'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 109 
	goto	AIR2

; <= D153?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'153'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 153 
	goto	AIR3

; <= D226?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'226'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 226 
	goto	AIR4

; <= D411?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'9B'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > D411 
	goto	AIR5

; <= D512?
	movf	TEMPB0,w 		; ms 
	sublw	H'2'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'00'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 512 
	goto	AIR6

; set at 5V for over
	bsf		ERROR_FLG,7		; error
	bsf		WIDEBAND0,1
	bsf		WIDEBAND0,0
	movlw	H'FF'
	movwf	WIDEBAND1
	goto	HEATER_RUN
	
CLEAR_CURRENT_READING
	bsf		ERROR_FLG,7		; error
	clrf	WIDEBAND0
	clrf	WIDEBAND1
	goto	HEATER_RUN

AIR1
; 0 to 3%
; TEMPB0,1 between D0 and D55, output for wideband is from D0 to D61 
; 0V to 0.3V.
; Calculation for wideband PWM output is D61 - (((D55-TEMPB0,1)/D55) x D61).

	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'0'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'61'
	movwf	MAX1			; max value D285
	movlw	D'55'			; subtract value from TEMPB0,1
	movwf	SUB1			; subtract value ls	
	clrf	SUB0			; subtract value ms
	movlw	D'55'			; divide value
	movwf	DIV1
	movlw	D'61'
	movwf	MULT1
	goto	CALC_PWM_AIR
AIR2
;3-6% 
; TEMPB0,1 between D55 and D109, output for wideband is from D123 to 61 
; Calculation for wideband PWM output is D123 - (((D109-TEMPB0,1)/D54) x D62).

	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	D'0'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'123'
	movwf	MAX1			; max value D317
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'109'
	movwf	SUB1			; subtract value ls
	movlw	D'54'			; divide value
	movwf	DIV1
	movlw	D'62'
	movwf	MULT1
	goto	CALC_PWM_AIR
AIR3
; 6-8.29% 
; TEMPB0,1 between D109 and D153, output for wideband is from D169 to D123 
; Calculation for wideband PWM output is D169 - (((D153-TEMPB0,1)/D44) x D46).

	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	D'0'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'169'
	movwf	MAX1			; max value D317
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'153'
	movwf	SUB1			; subtract value ls
	movlw	D'44'			; divide value
	movwf	DIV1
	movlw	D'46'
	movwf	MULT1
	goto	CALC_PWM_AIR
AIR4
; 8.29-12% 
; TEMPB0,1 between D153 and D226, output for wideband is from D246 to D169 
; Calculation for wideband PWM output is D246 - (((D226-TEMPB0,1)/D73) x D77).

	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	D'0'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'246'
	movwf	MAX1			; max value D317
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'226'
	movwf	SUB1			; subtract value ls
	movlw	D'73'			; divide value
	movwf	DIV1
	movlw	D'77'
	movwf	MULT1
	goto	CALC_PWM_AIR
AIR5
; 12-20.9% 
; TEMPB0,1 between D226 and D411, output for wideband is from D426 to D246  
; Calculation for wideband PWM output is D428 - (((D411-TEMPB0,1)/D185) x D182).

	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'AC'
	movwf	MAX1			; max value D317
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'9B'
	movwf	SUB1			; subtract value ls
	movlw	D'185'			; divide value
	movwf	DIV1
	movlw	D'182'
	movwf	MULT1
	goto	CALC_PWM_AIR
AIR6
; 20.9-26.0% 
; TEMPB0,1 between D512 and D411, output for wideband is from D532 to D428 
; Calculation for wideband PWM output is D523 - (((D512-TEMPB0,1)/D101) x D104).

	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'14'
	movwf	MAX1			; max value D317
	movlw	H'2'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'00'
	movwf	SUB1			; subtract value ls
	movlw	D'101'			; divide value
	movwf	DIV1
	movlw	D'104'
	movwf	MULT1
	goto	CALC_PWM_AIR

NORM_READINGS

; get (amplified) IP sense value and subtract from VS_IP
; SENSE_IP0,1 is IP sense at AN6
; subtract VS_IP from IP Sense
	btfss	FLAG,1			; if set can take reading
	goto	NORM_READINGS
	bcf		FLAG,1
	movf	VS_IP0,w		; ms VS_IP 
	subwf	SENSE_IP0,w		; ms IP sense  
	movwf	TEMPB0
	movf	VS_IP1,w		; ls  
	subwf	SENSE_IP1,w		; ls  
	movwf	TEMPB1
	btfss	STATUS,C
	decf	TEMPB0,f		; decrease if carry
	btfss	TEMPB0,7		; if set then VS_IP > IP sense so rearrange calculation
	goto	SET_WIDE_LEAN

; subtract 
	movf	SENSE_IP0,w		; ms IP sense 
	subwf	VS_IP0,w		; ms VS_IP 
	movwf	TEMPB0
	movf	SENSE_IP1,w		; ls  
	subwf	VS_IP1,w		; ls 
	movwf	TEMPB1
	btfss	STATUS,C
	decf	TEMPB0,f		; decrease if carry	

SET_WIDE_RICH ;(Lambda <1)
; values obtained from calculated table. See wideband table at end of file
; <= D151?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'151'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 > 151 
	goto	RICH1
; <= 245?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'245'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 245
	goto	RICH2
; <= D348?
	movf	TEMPB0,w 		; ms 
	sublw	D'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'5C'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > D348
	goto	RICH3
; <= D596?
	movf	TEMPB0,w 		; ms 
	sublw	D'2'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'54'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 596
	goto	RICH4
; above D596 is out of range so error
	bcf		STATUS,GIE		; stop changes in interrupt
	clrf	WIDEBAND0		; clear ms PWM byte for wideband output	
	clrf	WIDEBAND1		; ls PWM byte for wideband output
	bsf		STATUS,GIE		; allow changes in interrupt
	bsf		ERROR_FLG,7		; ERROR LED on
	goto	NARROW_BAND

RICH1
; 1-0.9
; TEMPB0,1 between D0 and D151, Lambda is between 1 and 0.9, output for wideband is from D277 to D180 
; 1.35V to 0.88V.
; Calculation for wideband PWM output is D277 - (((TEMPB0,1- D'0')/D151) x D97). Note D97 = D277-D180
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'01'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'15'
	movwf	MAX1			; max value D269 = H010D
	movlw	D'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	D'151'			; divide value
	movwf	DIV1
	movlw	D'97'
	movwf	MULT1
	goto	CALC_PWM_WIDE_R

RICH2
; 0.9-0.85
; TEMPB0,1 between D151 and D245, Lambda is between 0.9 and 0.85, output for wideband is from D180 to D135 
; 0.88V to 0.66V.
; Calculation for wideband PWM output is D180 - (((TEMPB0,1-D151)/D94) x D45).
; Note D45 = D180-D135; D94 is D245-D151
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	D'00'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'180'
	movwf	MAX1			; max value 
	movlw	D'151'			; subtract value from TEMPB0,1
	clrf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	D'94'			; divide value
	movwf	DIV1
	movlw	D'45'
	movwf	MULT1
	goto	CALC_PWM_WIDE_R

RICH3
; 0.85-0.8
; TEMPB0,1 between D245 and D348, Lambda is between 0.85 and 0.8, output for wideband is from D135 to D90 
; 0.66V to 0.44V.
; Calculation for wideband PWM output is D135 - (((TEMPB0,1-D245)/D103) x D45).
; Note D45 = D135-D90; D103 is D348-D245
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	D'00'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'135'
	movwf	MAX1			; max value 
	movlw	D'245'			; subtract value from TEMPB0,1
	clrf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	D'103'			; divide value
	movwf	DIV1
	movlw	D'45'
	movwf	MULT1
	goto	CALC_PWM_WIDE_R

RICH4
; 0.8-0.7
; TEMPB0,1 between D348 and D596, Lambda is between 0.8 and 0.7, output for wideband is from D90 to D0 
; 0.44V to 0.0V.
; Calculation for wideband PWM output is D90 - (((TEMPB0,1-D348)/D248) x D90).
; Note D90 = D90-D0; D248 is D596-D348
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'0'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'90'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'5C'
	movwf	SUB1			; subtract value ls
	movlw	D'248'			; divide value
	movwf	DIV1
	movlw	D'90'
	movwf	MULT1
	goto	CALC_PWM_WIDE_R

SET_WIDE_LEAN
; <= D18?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'18'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 18 
	goto	LEAN1
; <= D37?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'37'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN2
; <= D55?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'55'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN3
; <= D73?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'73'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN4
; <= D91?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'91'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN5
; <= D109?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'109'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN6
; <= D128?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'128'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN7
; <= D146?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'146'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN8
; <= D164?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'164'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN9
; <= D183?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'183'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN10
; <= D201?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'201'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN11
; <= D219?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'219'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN12
; <= D238?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'238'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN13
; <= D257?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'1'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN14
; <= D276?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'14'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN15
; <= D295?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'27'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN16
; <= D306?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'32'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN17
; <= D314?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'3A'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN18
; <= D334?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'4E'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN19
; <= D345?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'59'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN20
; above D345 is out of range so error
	bcf		STATUS,GIE		; stop changes in interrupt
	movlw	H'03'
	movwf	WIDEBAND0		; set ms PWM byte for max wideband output
	movlw	H'FF'	
	movwf	WIDEBAND1		; set ls PWM byte for max wideband output
	bsf		STATUS,GIE		; allow changes in interrupt
	bsf		ERROR_FLG,7		; ERROR LED on
	goto	NARROW_BAND

LEAN1
; 1 to 1.026
; TEMPB0,1 between D0 and D18, Lambda is between 1 and 1.026, output for wideband is from D277 to D292 
; 1.43V to 1.39V.
; Calculation for wideband PWM output is D292 - (((D18-TEMPB0,1)/D18) x D15).
; Note D15 = D292-D277; D18 is D18-D0
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'24'
	movwf	MAX1			; max value D285
	movlw	D'18'			; subtract value from TEMPB0,1
	movwf	SUB1			; subtract value ls	
	clrf	SUB0			; subtract value ms
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'15'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN2
; 1.026 to 1.0534
; TEMPB0,1 between D18 and D37, output for wideband is from D292 to D317 
; Calculation for wideband PWM output is D317 - (((D37-TEMPB0,1)/D19) x D25).
; Note D25 = D317-D292; D19 is D37-D18
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	D'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'3D'
	movwf	MAX1			; max value D317
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'37'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'25'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN3
; 1.0534 to 1.0823
; TEMPB0,1 between D37 and D55, output for wideband is from  D317 to D343
; Calculation for wideband PWM output is D343 - (((D55-TEMPB0,1)/D18) x D26).
; Note D26 = D343-D317; D18 is D55-D37
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'57'
	movwf	MAX1			; max value D343
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'55'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'26'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN4
; 1.0823 to 1.112
; TEMPB0,1 between D55 and D73, output for wideband is from D343 to D370
; Calculation for wideband PWM output is D370 - (((D73-TEMPB0,1)/D18) x D27).
; Note MULT D27 = D370-D343; DIV D18 is D73-D55
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'72'
	movwf	MAX1			; max value D370
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'73'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'27'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN5
; 1.112 to 1.1445
; TEMPB0,1 between D73 and D91, output for wideband is from D399 to D370
; Calculation for wideband PWM output is D399 - (((D91-TEMPB0,1)/D18) x D29).
; Note MULT D29 = D399-D370; DIV D18 is D91-D73
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'8F'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'91'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'29'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN6 
; 1.1445 to 1.18
; TEMPB0,1 between D91 and D109, output for wideband is from D399 to D431
; Calculation for wideband PWM output is D431 - (((D109-TEMPB0,1)/D18) x D32).
; Note MULT D32 = D431-D399; DIV D18 is D109-D91
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'AF'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'109'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'32'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN7 
; 1.18 to 1.214
; TEMPB0,1 between D109 and D128, output for wideband is from D431 to D462
; Calculation for wideband PWM output is D462 - (((D128-TEMPB0,1)/D19) x D31).
; Note MULT D31 = D462-D431; DIV D19 is D128-D109
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'CE'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'128'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'31'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN8 
; 1.214 to 1.2516
; TEMPB0,1 between D128 and D146, output for wideband is from D462 to D495
; Calculation for wideband PWM output is D495 - (((D146-TEMPB0,1)/D18) x D33).
; Note MULT D33 = D495-D462; DIV D18 is D146-D128
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'EF'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'146'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'33'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN9 
; 1.2516 to 1.2916
; TEMPB0,1 between D146 and D164, output for wideband is from D531 to D495
; Calculation for wideband PWM output is D531 - (((D164-TEMPB0,1)/D18) x D36).
; Note MULT D36 = D531-D495; DIV D18 is D164-D146
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'13'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'164'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'36'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN10 
; 1.2916 to 1.33
; TEMPB0,1 between D164 and D183, output for wideband is from D531 to D566
; Calculation for wideband PWM output is D566 - (((D183-TEMPB0,1)/D19) x D35).
; Note MULT D35 = D566-D531; DIV D19 is D183-D164
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'36'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'183'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'35'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN11 
; 1.33 to 1.38
; TEMPB0,1 between D183 and D201, output for wideband is from D611 to D566
; Calculation for wideband PWM output is D611 - (((D201-TEMPB0,1)/D18) x D45).
; Note MULT D45 = D611-D566; DIV D18 is D201-D183
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'63'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'201'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'45'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN12 
; 1.38 to 1.43
; TEMPB0,1 between D201 and D219, output for wideband is from D611 to D656
; Calculation for wideband PWM output is D656 - (((D219-TEMPB0,1)/D18) x D45).
; Note MULT D45 = D656-D611; DIV D18 is D219-D201
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'90'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'219'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'45'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN13 
; 1.43 to 1.48
; TEMPB0,1 between D219 and D238, output for wideband is from D701 to D656
; Calculation for wideband PWM output is D701 - (((D238-TEMPB0,1)/D19) x D45).
; Note MULT D45 = D701-D656; DIV D19 is D238-D219
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'BD'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'238'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'45'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN14 
; 1.48 to 1.5346
; TEMPB0,1 between D238 and D257, output for wideband is from D701 to D750
; Calculation for wideband PWM output is D750 - (((D257-TEMPB0,1)/D19) x D49).
; Note MULT D49 = D750-D701; DIV D19 is D257-D238
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'EE'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'1'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'49'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN15 
; 1.5346 to 1.594
; TEMPB0,1 between D257 and D276, output for wideband is from D803 to D750
; Calculation for wideband PWM output is D803 - (((D276-TEMPB0,1)/D19) x D53).
; Note MULT D53 = D803-D750; DIV D19 is D276-D257
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'23'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'14'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'53'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN16 
; 1.594 to 1.658
; TEMPB0,1 between D276 and D295, output for wideband is from D803 to D861
; Calculation for wideband PWM output is D861 - (((D295-TEMPB0,1)/D19) x D53).
; Note MULT D58 = D861-D803; DIV D19 is D295-D276
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'5D'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'27'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'58'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN17 
; 1.658 to 1.7
; TEMPB0,1 between D295 and D306, output for wideband is from D898 to D861
; Calculation for wideband PWM output is D898 - (((D306-TEMPB0,1)/D11) x D31).
; Note MULT D37 = D898-D861; DIV D11 is D306-D295
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'82'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'32'
	movwf	SUB1			; subtract value ls
	movlw	D'11'			; divide value
	movwf	DIV1
	movlw	D'37'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN18 
; 1.7 to 1.73
; TEMPB0,1 between D306 and D314, output for wideband is from D898 to D925
; Calculation for wideband PWM output is D925 - (((D314-TEMPB0,1)/D8) x D27).
; Note MULT D27 = D925-D898; DIV D8 is D314-D306
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'9D'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'3A'
	movwf	SUB1			; subtract value ls
	movlw	D'8'			; divide value
	movwf	DIV1
	movlw	D'27'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN19 
; 1.73 to 1.80
; TEMPB0,1 between D314 and D334, output for wideband is from D925 to D988
; Calculation for wideband PWM output is D988 - (((D334-TEMPB0,1)/D20) x D63).
; Note MULT D63 = D988-D925; DIV D20 is D334-D314
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'DC'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'4E'
	movwf	SUB1			; subtract value ls
	movlw	D'20'			; divide value
	movwf	DIV1
	movlw	D'63'
	movwf	MULT1
	goto	CALC_PWM_WIDE
LEAN20 
; 1.80 to 1.84
; TEMPB0,1 between D334 and D345, output for wideband is from D1023 to D988
; Calculation for wideband PWM output is D1023 - (((D345-TEMPB0,1)/D11) x D35).
; Note MULT D35 = D1023-D988; DIV D11 is D345-D334
	bcf		ERROR_FLG,7		; ERROR LED off	
	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'FF'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'59'
	movwf	SUB1			; subtract value ls
	movlw	D'11'			; divide value
	movwf	DIV1
	movlw	D'35'
	movwf	MULT1
	goto	CALC_PWM_WIDE

CALC_PWM_WIDE_R
; RICH Calculation
; take subtract value from TEMPB0,1
	movf	SUB0,w 			; ms 
	subwf	TEMPB0,w		; subtract value
	movwf	AARGB0
	movf	SUB1,w			; ls  
	subwf	TEMPB1,w		; subtract value ls  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry	
	goto	MULT_ALL

CALC_PWM_WIDE
CALC_PWM_AIR
; lean calculation
; take  TEMPB0,1 from subtract value 
	movf	TEMPB0,w 		; ms 
	subwf	SUB0,w			; subtract value
	movwf	AARGB0
	movf	TEMPB1,w		; ls  
	subwf	SUB1,w			; subtract value ls  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry	

MULT_ALL
; multiply
	clrf	BARGB0
	movf	MULT1,w			; multiply value
	movwf	BARGB1
	bsf		PCLATH,3	; page 1
	call	FXM1616U		; multiply

; divide
	movf	DIV1,w
	movwf	BARGB3
	clrf	BARGB0
	clrf	BARGB1
	clrf	BARGB2
	call	FXD3232U		; divide
	bcf		PCLATH,3		; page 0
; subtract from maximum 
	
	movf	AARGB2,w 		; ms 
	subwf	MAX0,f			; subtract value
	movf	AARGB3,w		; ls  
	subwf	MAX1,f			; subtract value ls  
	btfss	STATUS,C
	decf	MAX0,f	

; place in PWM

	bcf		STATUS,GIE		; stop changes in interrupt
	movf	MAX0,w
	movwf	WIDEBAND0		; ms PWM byte for wideband output	
	movf	MAX1,w
	movwf	WIDEBAND1		; ls PWM byte for wideband output
	bsf		STATUS,GIE		; allow changes in interrupt

; Check for Air Calibration mode (VS/IP = 2V)

	movlw	D'02'			; VS/IP ms
	subwf	VS_IP0,w	
	btfss	STATUS,C		; if positive do S-Curve output
	goto	HEATER_RUN		; bypass S-curve


NARROW_BAND
;	incf	MILLISEC,f		; 10ms counter
;	btfss	MILLISEC,0
;	goto	HEATER_RUN		; bypass if not 20ms

; S_CURVE output
; Calculation for narrowband S-CURVE

; response rate varies with S_RESP setting 
; run through an averaging delay so values run to follow wideband values but slower response 	

	movf	S_RESP,w		; check response rate at VR2. (1.5 to 2-seconds for Bosch LSM11)
	movwf	DIV_S			; temporary value
; divide by 4 for 1-64 averages/ divisions 
	bcf		STATUS,C		; clear carry
	rrf		DIV_S,f
	bcf		STATUS,C		; clear carry
	rrf		DIV_S,f
	movlw	D'1'			; add 1 for a 1 to 64 range
	addwf	DIV_S,f		


; average over DIV_S count 20ms rate (64 ~ 1.2 seconds)

; averaging in A0-DF RAM. Shuffle data left to right

	movlw	H'DE'			; move value from DE to DF
	movwf	FSR
SHUFFLE
	movf	INDF,w			; get value
	incf	FSR,f
	movwf	INDF			; move to next ram address
	decf	FSR,f			; DE
	decf	FSR,f			; DD 
	movf	FSR,w	
; check if 9F (A1 -2)
	sublw	H'9F'
	btfss	STATUS,C
	goto	SHUFFLE			; shift data along
; all shuffled

; divide WIDEBAND0,1 by 4 
WIDE_4
	bcf		STATUS,C
	rrf		WIDEBAND0,w		; divide by 2
	movwf	ADDN_S0			; store
	rrf		WIDEBAND1,w
	movwf	ADDN_S1			; store
	bcf		STATUS,C
	rrf		ADDN_S0,f		; divide by 2 again		
	rrf		ADDN_S1,f		; 

; check if range is < 22 or > if 105 
; less than 22
	movlw	D'22'
	subwf	ADDN_S1,f
	btfss	STATUS,C		; if minus then less than 22
	clrf	ADDN_S1			; set at 0 if less
; compare to 83 (105 -22) (max range)
	movlw	D'83'
	subwf	ADDN_S1,w
	movlw	D'83'			; maximum
	btfsc	STATUS,C
	movwf	ADDN_S1			; set at 83 if >83
	
	movlw	H'A0'			; start of bank 1 registers
	movwf	FSR

; set S_CURVE value accordingly	
	bsf		PCLATH,3		; page 1
	movf	ADDN_S1,w		; get value
	call	LOOKUP_S_CURVE	; lookup table to set up S-curve output
	bcf		PCLATH,3		; page 0

; and place in H'A0'
	
	movwf	INDF

; Add values to average based on DIV_S	
	
	clrf	ADDN_S1
	clrf	ADDN_S0			; clear addition values
	clrf	COUNT
	incf	COUNT,f
ADD_ALL
	movf	INDF,w		; value
	addwf	ADDN_S1,f
	btfsc	STATUS,C	; if carry then increase ms byte
	incf	ADDN_S0,f
	incf	FSR,f
	incf	COUNT,f
	movf	COUNT,w
	subwf	DIV_S,w		; if over then finished adding 
	btfsc	STATUS,C
	goto	ADD_ALL

	movf	ADDN_S0,w
	movwf	AARGB0
	movf	ADDN_S1,w
	movwf	AARGB1
	movf	DIV_S,w		; divisor
	movwf	BARGB0
; divide for average
	bsf		PCLATH,3		; page 1
	call	DIV16_8			; divide
	bcf		PCLATH,3		; page 0

; check if first averaging has been complete
	movf	ALL_COUNT,w
	btfsc	STATUS,Z		; when zero use averaged values	
	goto	SET_S
	decf	ALL_COUNT,f
	goto	HEATER_RUN		; keep S-CURVE value set till full averaging has occurred.
SET_S
	movf	AARGB1,w		; divided value
	movwf	S_CURVE
	goto	HEATER_RUN
	

; *****************************************************************************************
; Subroutines

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return


; Page 1
	org	H'800'; page 1

LOOKUP_S_CURVE
; wideband value/4 if <D22 then 890mV out
; if >D105 then 30mV output
; start at D22
; S-curve output is effectively a divide by 4 value as it uses the 10-bit PWM and ls 8-bits
; so a D255 value in S_CURVE is 5V/4 or 1.25V max
	addwf	PCL,f	; add value to program counter
	retlw	D'182'	; 890mV (0.89/5V) x 1024 ; wideband /4 = 22 	
	retlw	D'182'	;  wideband /4 = 23 
	retlw	D'182'	;  wideband /4 = 24 
	retlw	D'182'	;  wideband /4 = 25 
	retlw	D'181'	;  wideband /4 = 26 
	retlw	D'181'	;  wideband /4 = 27 
	retlw	D'181'	;  wideband /4 = 28 
	retlw	D'180'	;  wideband /4 = 29 
	retlw	D'180'	;  wideband /4 = 30 
	retlw	D'180'	;  wideband /4 = 31 
	retlw	D'179'	;  wideband /4 = 32 
	retlw	D'179'	; 870mV  ; wideband /4 = 33
	retlw	D'179'	;  wideband /4 = 34 
	retlw	D'179'	;  wideband /4 = 35 
	retlw	D'179'	;  wideband /4 = 36 
	retlw	D'178'	;  wideband /4 = 37 
	retlw	D'178'	;  wideband /4 = 38 
	retlw	D'178'	;  wideband /4 = 39 
	retlw	D'177'	;  wideband /4 = 40 
	retlw	D'177'	;  wideband /4 = 41 
	retlw	D'176'	;  wideband /4 = 42 
	retlw	D'174'	;  wideband /4 = 43 
	retlw	D'174'	;  wideband /4 = 44 
	retlw	D'172'	; 860mV  ; wideband /4 = 45  
	retlw	D'172'	;  wideband /4 = 46 
	retlw	D'171'	;  wideband /4 = 47 
	retlw	D'170'	;  wideband /4 = 48 
	retlw	D'169'	;  wideband /4 = 49 
	retlw	D'168'	;  wideband /4 = 50 
	retlw	D'167'	;  wideband /4 = 51 
	retlw	D'166'	;  wideband /4 = 52 
	retlw	D'165'	;  wideband /4 = 53 
	retlw	D'163'	;  wideband /4 = 54 
	retlw	D'161'	;  wideband /4 = 55 
	retlw	D'159'	;  wideband /4 = 56 
	retlw	D'158'	;  wideband /4 = 57 
	retlw	D'156'	;  wideband /4 = 58 
	retlw	D'154'	;  wideband /4 = 59 
	retlw	D'153'	;  wideband /4 = 60 
	retlw	D'151'	;  wideband /4 = 61  
	retlw	D'148'	;  wideband /4 = 62 
	retlw	D'146'	;  wideband /4 = 63 
	retlw	D'141'	;  700mV  ;wideband /4 = 64 
	retlw	D'136'	;  wideband /4 = 65  
	retlw	D'124'	;  wideband /4 = 66 
	retlw	D'114'	;  wideband /4 = 67 
	retlw	D'100'	;  wideband /4 = 68 
	retlw	D'92'	;  450mV; wideband /4 = 69 
	retlw	D'63'	;  wideband /4 = 70 
	retlw	D'35'	;  wideband /4 = 71
 	retlw	D'26'	;  130mV ; wideband /4 = 72 
	retlw	D'20'	;  wideband /4 = 73 
	retlw	D'18'	;  wideband /4 = 74 
	retlw	D'16'	;  wideband /4 = 75 
	retlw	D'14'	;  wideband /4 = 76 
	retlw	D'13'	;  wideband /4 = 77 
	retlw	D'12'	;  wideband /4 = 78 
	retlw	D'12'	;  wideband /4 = 79 
	retlw	D'11'	;  wideband /4 = 80 
	retlw	D'11'	;  wideband /4 = 81 
	retlw	D'10'	;  wideband /4 = 82 
	retlw	D'10'	;  wideband /4 = 83 
	retlw	D'10'	;  wideband /4 = 84 
	retlw	D'9'	;  wideband /4 = 85 
	retlw	D'9'	;  wideband /4 = 86 
	retlw	D'9'	;  wideband /4 = 87 
	retlw	D'8'	;  wideband /4 = 88 
	retlw	D'8'	;  wideband /4 = 89 
	retlw	D'8'	;  wideband /4 = 90 
	retlw	D'8'	;  wideband /4 = 91 
	retlw	D'8'	;  wideband /4 = 92 
	retlw	D'8'	;  wideband /4 = 93 
	retlw	D'8'	;  wideband /4 = 94 
	retlw	D'8'	;  wideband /4 = 95 
	retlw	D'8'	;  wideband /4 = 96 
	retlw	D'7'	;  wideband /4 = 97 
	retlw	D'7'	;  wideband /4 = 98 
	retlw	D'7'	;  wideband /4 = 99 
	retlw	D'7'	;  wideband /4 = 100 
	retlw	D'7'	;  wideband /4 = 101 
	retlw	D'6'	;  wideband /4 = 102 
	retlw	D'6'	;  wideband /4 = 103 
	retlw	D'6'	;  wideband /4 = 104 
	retlw	D'6'	;  wideband /4 = 105 




; Math routines

;

; 32/32 Bit Unsigned Fixed Point Divide 32/32 -> 32.32
; Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point divisor in BARGB0, BARGB1, BARGB2, BARGB3
; Use: CALL FXD3232U
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point remainder in REMB0, REMB1, REMB2, REMB3
; Result: AARG, REM <-- AARG / BARG
; Max Timing: 4+1025+2 = 1031 clks
; Max Timing: 4+981+2 = 987 clks
; PM: 4+359+1 = 364 DM: 13
FXD3232U
	CLRF 	REMB0
	CLRF	REMB1
	CLRF 	REMB2
	CLRF 	REMB3
	call	UDIV3232L
	return

UDIV3232L 
; Max Timing: 24+6*32+31+31+6*32+31+31+6*32+31+31+6*32+31+16 = 1025 clks
; Min Timing: 24+6*31+30+30+6*31+30+30+6*31+30+30+6*31+30+3 = 981 clks
; PM: 359 DM: 13
	CLRF 	TEMP
	RLF 	AARGB0,W
	RLF 	REMB3,F
	MOVF 	BARGB3,W
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	RLF 	AARGB0,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232A 
	RLF 	AARGB0,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22LA
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LA

UADD22LA 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LA 
	RLF		AARGB0,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232A
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22L8
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L8

UADD22L8 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L8 
	RLF 	AARGB1,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232B 
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB1,0
	GOTO 	UADD22LB
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LB

UADD22LB 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LB 
	RLF 	AARGB1,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232B
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS	AARGB1,0
	GOTO 	UADD22L16
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L16

UADD22L16 	
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF	TEMP,F

UOK22L16
	RLF		AARGB2,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232C 
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22LC
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LC

UADD22LC
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LC 
	RLF 	AARGB2,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232C
	RLF 	AARGB3,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22L24
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L24

UADD22L24 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L24 
	RLF 	AARGB3,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232D 
    RLF  	AARGB3,W
    RLF    	REMB3,F
    RLF    	REMB2,F
    RLF    	REMB1,F
    RLF    	REMB0,F
    RLF    	TEMP,F
    MOVF   	BARGB3,W
 	BTFSS  	AARGB3,0
   	GOTO   	UADD22LD

    SUBWF  	REMB3,F
  	MOVF   	BARGB2,W
    BTFSS   STATUS,C
    INCFSZ  BARGB2,W
    SUBWF   REMB2,F
    MOVF    BARGB1,W
    BTFSS   STATUS,C
    INCFSZ  BARGB1,W
    SUBWF   REMB1,F
    MOVF    BARGB0,W
    BTFSS   STATUS,C
    INCFSZ  BARGB0,W
    SUBWF   REMB0,F
    CLRW
    BTFSS   STATUS,C
    MOVLW   H'1'
    SUBWF   TEMP,F
    GOTO    UOK22LD

UADD22LD      
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F
    CLRW
    BTFSC   STATUS,C
    MOVLW   H'1'
    ADDWF   TEMP,F
        
UOK22LD
	RLF     AARGB3,F

   	DECFSZ  LOOPCOUNT, F
    GOTO    LOOPU3232D

    BTFSC   AARGB3,0
    GOTO    UOK22L
    MOVF    BARGB3,W
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F

UOK22L
	RETURN


;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8		 
	  			CLRF            REMB0
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0,F

                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU1608A
                CLRF            TEMP1
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0,F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1,F

                BCF             STATUS,C
UOK68B          RLF             AARGB1,F

                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU1608B
                return

; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0,1
;               16 bit unsigned fixed point multiplier in BARGB0,1

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0,1,2,3

FXM1616U 	
			    CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
				RETLW   H'00'
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
				RETURN


	end

;Wideband data	
	
;Lambda	O2 concentration %	Ip mA	A/D value of Span relative to Vs/Ip @ AN1 and Ip sense @ AN6	0-5V curve output value(PWM decimal value)
				
;CALIBRATE	For Air 20.9%		Vs/Ip = 2V, TP4=2.343 IC5b has 560k resistors paralleled for 560k/2
; and gain of 12.73 for current measurement at AN6
; Vs/Ip = 2V(gain is 12.73 with 560k in parallel)	5V span is 50%
; output in volts is actual O2%
; lambda O2%	Ip		AN1-AN6		Wideband out	
;
; -	   26.0%	3.173 mA 2.5V (D512)	 2.6V (D532)
;*207	20.9%	2.55 mA	 2.01V (D411) 2.09V (D428)
;*2.42	12.0%	1.40 mA	 1.1V (D226)	 1.2V (D246)
;*1.7	8.29%	0.95 mA  0.748 (D153) 0.829 (D169)
;*1.43	6.0%	0.68 mA	  0.536 (D109)	 0.6V (D123)
;*1.18	3%  	0.34 mA	  0.268V (D55)	 0.3 (D61)
;*1.009	0%	   0 mA	      0V (D0)	     0V (D0)

;				
;				
;Standard range 0.7 to 1.84				
; lambda O2%	Ip		AN1-AN6			Wideband out				
;1.84	9.24%	1.07 mA	1.685V (D345)	5V (D1023)
;1.80	9%	1.036 mA	1.633V (D334)	4.83V (D988)
;1.73	8.5%	0.975 mA	1.536V (D314)	4.52V (D925)
;*1.7	8.29%	0.95mA mA	1.50V (D306)	4.39V (D898)
;1.658	8%	0.9158 mA	1.443V (D295)	4.21V (D861)
;1.594	7.5%	0.8568 mA	1.35V (D276)	3.92V (D803)
;1.5346	7%	0.7979 mA	1.257V (D257)	3.67V (D750)
;1.48	6.5%	0.73895 mA	1.164V (D238)	3.43V (D701)
;*1.43	6.0%	0.68 mA	1.07V (D219)	3.21V (D656)
;1.38	5.5%	0.6233 mA	0.982V (D201)	2.99V (D611)
;1.33	5%	0.5666 mA	0.893V (D183)	2.77V (D566)
;1.2916	4.5%	0.510 mA	0.804V (D164)	2.60V (D531)
;1.216	4%	0.45338 mA	0.714V (D146)	2.42V (D495)
;1.214	3.5%	0.39666 mA	0.625V (D128)	2.26V (D462)
;*1.18	3%	0.34 mA	0.54V (D109)	2.11V (D431)
;1.1445	2.5%	0.28333 mA	0.446V (D91)	1.95V (D399)
;1.112	2%	0.22666 mA	0.3571V (D73)	1.81V (D370)
;1.08227	1.5%	0.170 mA	0.2678V (D55)	1.68V (D343)
;1.0534	1%	0.1133 mA	0.1786V (D37)	1.55V (D317)
;1.026	0.5%	0.0566 mA	0.0893V (D18)	1.43V (D292)
;*1.009	0%	0 mA	0V (D0)	1.35V (D277)
;*0.9		-0.47 mA	0.74V (D151)	0.88V (D180)
;*0.85		-0.76 mA	1.2V (D245)	0.66V (D135)
;*0.8 		-1.08 mA	1.7V (D348)	0.44V (D90)
;*0.7		-1.85 mA	2.91V (D596)	0V (D0)
				
				
				


;*denotes tabled values Intermediate values use interpolation using the O2 concentration 
;against Ip graph and the lambda calculation ((XO2/3) +1)/1-4.76XO2.

;Span is Ip x 61.9 ohm x gain (560k/22k)
