;		GPSTimeDispV4.asm 	-- Control program for PIC16F877A
;		based 6-digit GPS Time Display add-on for the Silicon Chip
;		GPS-derived 10MHz Frequency Reference design published in
;		the March-April-May 2007 issues of Silicon Chip magazine.
;		Written by Jim Rowe.

;		Error in config word fixed, also bank switching errors
;		in initialisation sequence.
;		Autodim function routines also added (20/05/09)
;		Next change made to get UTC time from NEMA 0183 GPGGA data
;		sentences (instead of the GPRMC sentences used originally),
; 		in order to remove 300ms delay in displaying time every fifth
;		second -- caused by 1-3 lines of inserted GPGSV sentences
;		ahead of GPRMC sentences.
;		Next change made to remove bug causing random (?) 'blank display'
;		glitch when in daylight saving time mode.
;		Final change made to set BOREN config bit, to prevent corruption
;		of EEPROM data during power down. (Bug found by Ben Rampling VK6IC
;		-- thank you and Merv Thomas VK6BMT for advising us.)
;		(Last revision 25/01/2011 at 9:55am)
;		Time data is received from GPS receiver via a NEMA 0183 data
;		sentence (GPGGA), using the PIC's USART module -- which 
;		interrupts the processor when each character is received. 
;		The ISR then processes the received data for display.

;		The display should also work with other GPS receiver
;		modules, providing they output a standard GPGGA sentence.

;		(Note that PIC clock is 4MHz, so 1mc = 4 x 250ns = 1us)

; ***
	ERRORLEVEL -302
	ERRORLEVEL -306
; ***

	list	P=16F877A
	#include "p16f877A.inc"
	__config h'3F72'	; HS osc selected, WDT disabled, PWRT enabled,
						; BOR enabled, RB3 is digital I/O, code prot off,
						; Flash prog mem write protection off, RB7 and RB6 are
						; gen purpose I/O pins (not used for debugging)

;**********************************************************
;
;	First we define some handy bank selection macros
;
;	Select Register Bank 0
bank0	macro
	BCF	STATUS, RP0			; select Bank 0
	BCF STATUS, RP1
	BCF STATUS, IRP			; and also indir addr banks 0,1
	endm

;	Select Register Bank 1
bank1	macro
	BSF	STATUS, RP0			; Select Bank 1
	BCF STATUS, RP1
	BCF STATUS, IRP			; also indir addr banks 0,1
	endm

;	select Register Bank 2
bank2	macro
	BCF	STATUS, RP0			; Select Bank 2
	BSF STATUS, RP1
	BSF STATUS, IRP			; also indir addr banks 2,3
	endm

;	select Register Bank 3
bank3	macro
	BSF	STATUS, RP0			; Select Bank 3
	BSF STATUS, RP1
	BSF STATUS, IRP			; also indir addr banks 2,3
	endm
;
;	**********************************************************
;
;	Now we define data storage locations in RAM
	
; storage buffer for data chars received from GPS Rx in NMEA0183 sentences
NMEAstt	equ		h'20'	; first char of sentence buffer
NMEAend	equ		h'6F'	; last char of sentence buffer (79d chars max)

; regs for saving context during interrupt servicing
WSave	equ		h'70'	; w reg stored here
SSave	equ		h'71'	; status reg stored here
PCHiSave	equ h'72'	; PCLATH saved here
FSRSave		equ h'73'	; FSR saved here 

; variables at data memory locations
Counter1 equ	h'74'	; general purpose counter variable 1
Counter2 equ 	h'75'	; general purpose counter variable 2
Counter3 equ	h'76'	; general purpose counter variable 3
Keycode	 equ	h'77'	; switch scanning code stored (0/"1"/"2"/"3"/"4"/"5")
Temp1	equ		h'78'	; working storage 1
Temp2	equ		h'79'	; working storage 2
Temp3	equ		h'7A'	; working storage 3
BufPtr	equ		h'7B'	; Storage for current GPSRx buffer ptr (used in ISR)
TempChr	equ		h'7C'	; Temp storage for received GPS char

; Storage regs for flags, etc (stored in EEPROM also)
Flags	equ		h'7D'	; bit0: 1 = display in unchanged UTC
						; bit1: 1 = display in local standard time
						; bit2: 1 = display in local daylite saving time
ULOffseth	equ h'7E'	; offset between UTC & local time (hrs)
ULOffsetm	equ h'7F'	; offset between UTC & local time (mins)

; storage registers for current UTC time, all in binary/hex (Bank1)
UTCh10	equ		h'A0'	; UTC hours - tens
UTCh01	equ		h'A1'	; UTC hours - units
UTCm10	equ		h'A2'	; UTC minutes - tens
UTCm01	equ		h'A3'	; UTC minutes - units
UTCs10	equ		h'A4'	; UTC seconds - tens
UTCs01	equ		h'A5'	; UTC seconds - units
UTChhex	equ		h'A6'	; UTC hours in hex (00-17)
UTCmhex equ		h'A7'	; UTC minutes in hex (00-3B)

; storage registers for current local time (Bank1)
LSTh10		equ h'A8'	; Local time hours - tens
LSTh01		equ h'A9'	; Local time hours - units
LSTm10		equ h'AA'	; Local time minutes - tens
LSTm01		equ h'AB'	; Local time minutes - units
LSThhex		equ	h'AC'	; local time hours in hex (00-17)
LSTmhex		equ h'AD'	; local time minutes in hex (00-3B)

; storage registers for current digits/chars on DISP1-6 (Bank1)
Dchar1		equ h'B0'	; character code for DISP1 (hours x10)
Dchar2		equ h'B1'	; character code for DISP2 (hours x1)
Dchar3		equ h'B2'	; character code for DISP3 (minutes x10)
Dchar4		equ h'B3'	; character code for DISP4 (minutes x1)
Dchar5		equ h'B4'	; character code for DISP5 (seconds x10)
Dchar6		equ h'B5'	; character code for DISP6 (seconds x1)

; two more working data registers for byte splitting (Bank1)
Byte10	equ		h'B6'	; tens of units digit
Byte01	equ		h'B7'	; units digit

; data regs added for the display dimming function (bank1)
DonTime		equ	h'B8'	; digit on time value (x100us)
DoffTime	equ	h'B9'	; digit off time value (x100us)
Vbrite		equ h'BA'	; ambient light level (from ADC)
;
; *************************************************************
;
; now program begins

	ORG		h'00'		; normal start vector
	GOTO Initialise
	ORG		h'04'		; interrupt servicing vector
	GOTO IntService
	
Initialise:
	bank0				; first make sure we are in bank0
	CLRF PORTB			; clear PortB
	CLRF PORTA			; also PortA (turns off LEDs)
	CLRF INTCON			; disable all interrupts also
	CLRF ADCON0			; and the ADC module as well for the present
	CLRF RCSTA			; also clear RCSTA register
	bank1				; then set for Bank1, so we can...
	MOVLW h'07'			; disable the comparators
	MOVWF CMCON
	MOVLW h'49'			; set ADCON1 for Fosc/16, left justify, Vdd and Vss
	MOVWF ADCON1		; as Vmax & Vmin references for the ADC
	MOVLW h'1E'			; then set DonTime to 1Eh (30d -> 30x100us or 3ms)
	MOVWF DonTime
	MOVLW h'01'			; and DoffTime to 01h (-> 100us)
	MOVWF DoffTime		; for maximum brightness by default 
	CLRF PIE1			; clear all interrupt mask bits for clean start
	CLRF CVRCON			; also make sure VREF is off
	CLRF TRISA			; now clear TRISA so all PORTA pins are outputs
	MOVLW h'01'			; also set RB0 as an input, RB1-7 as outputs
	MOVWF TRISB			; by loading config register TRISB
	MOVLW h'C0'			; also make RC7(Rx) & RC6 (Tx) inputs, but RC0-RC5
	MOVWF TRISC			; outputs, by loading config register TRISC
	MOVLW h'1F'			; also make RD0-RD4 inputs, RD5-7 outputs
	MOVWF TRISD			; by loading its config register
	MOVLW h'01'			; now disable PSP, enable RE0 as an input
	MOVWF TRISE			; by clearing TRISE(4), setting TRISE(0)
	MOVLW h'80'			; then make sure PORTB pullups are disabled,
	MOVWF OPTION_REG	; and ext INT disabled
	MOVLW h'33'			; now set SPBRG reg for 4800 baud async serial comms
	MOVWF SPBRG			; (X = 51d/33h for 4800baud with 4MHz clock)
	BSF TXSTA, BRGH		; also set BRGH bit for high bit rate divisor
	BCF TXSTA, SYNC		; clear SYNC bit in TXSTA reg for async mode
	bank3				; then up to bank3 so we can
	BCF EECON1,7		; clear EEPGD bit to ensure we address data EEPROM
	BCF EECON1,2		; also clear WREN bit so no writes until we want to
	bank0				; then back to Bank0 so we can...
	BSF RCSTA, SPEN		; enable USART serial port (RC7->Rx, RC6->Tx)
	bank1				; then back to Bank1
	BCF TXSTA, TXEN		; transmit disabled for the present
	BSF PIE1, RCIE		; enable Rx interrupts (only) by setting RCIE bit
	bank0				; finally return to Bank0
	MOVLW h'40'			; now disable ext & RB interrupts, enable PEIE
	MOVWF INTCON		; interrupts only (for USART)	
	BSF RCSTA, CREN		; also enable reception by setting CREN
	CALL CleanSlate		; now reset GPS info for clean start
	MOVLW h'0A'			; also set UTC-locTh offset to +10 (hours)
	MOVWF ULOffseth		; (default -- for Australian EAST)
	MOVLW h'00'			; and set UTC-locTm offset to 0 (minutes)
	MOVWF ULOffsetm
	MOVLW h'01'			; and set Flags for UTC display as default
	MOVWF Flags
	MOVLW h'69'			; then turn on ADC, set for AN5 and conversion
	MOVWF ADCON0		; clock to Fosc/8 or Fosc/16
	CALL CheckMem		; now check EEPROM in case settings were saved
	CALL ShowTime		; then display time on DISP1-6
	BSF INTCON,GIE		; and finally enable global interrupts

LoopStart:
	CALL CheckLiteLvl	; first check ambient light level, set disp brightness
	bank0
	CALL DoAScan		; go look for a keypress
	MOVF Keycode,1		; check if a key was pressed
	BTFSC STATUS,Z 		; skip if Z=0 because Keycode wasn't zero
	GOTO DispUpdate		; Z=1, so no key pressed - just go & update display
	MOVLW "1"			; key was pressed, so check if it was S1 ("1")
	XORWF Keycode,0		; w will now have 0 if it was a match...
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a 1, and w -> 0)
	GOTO $+5			; Z=0 (because it wasn't a 1), so keep looking
	MOVLW h'01'			; it was a 1, so set Flags(0) for UTC display
	MOVWF Flags
	CALL SaveSettings	; then save settings in data EEPROM
	GOTO DispUpdate		; and move on
	MOVLW "2"			; not S1, so check if it was S2 ("2")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a "2")
	GOTO $+8			; Z=0, so try again
	BTFSS Flags,2		; was a 2, but see if Flags(2) was set (DLST)
	GOTO $+2			; if not, just go set Flags(1) for LST display
	CALL RestorOffseth	; it was, so returning from DLST mode - decrement offset
	MOVLW h'02'			; now set Flags(1) for LST display
	MOVWF Flags
	CALL SaveSettings	; then save settings in data EEPROM
	GOTO DispUpdate		; and move on 
	MOVLW "3"			; not S2, so check if it was S3 ("3")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a "3")
	GOTO $+8			; Z=0 (i.e., wasn't a "3"), so keep looking
	BTFSS Flags,1		; it was a 3, but see if Flags(1) was set (LST)
	GOTO $+2			; if not, just go set Flags(2) for DLST display
	CALL IncrOffseth	; it was, so increment ULoffseth
	MOVLW h'04'			; now set Flags(2) for DLST display
	MOVWF Flags
	CALL SaveSettings	; then save settings in data EEPROM
	GOTO DispUpdate		; and move on
	MOVLW "4"			; not S3, so check if it was S4 ("4")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a "4")
	GOTO $+4			; Z=0, so not "4" -- keep looking
	CALL IncrOffseth	; it was a 4, so go increment hours offset
	CALL SaveSettings	; then save settings in data EEPROM
	GOTO DispUpdate		; and move on
	CALL IncrOffsetm	; must have pressed S5, so change mins offset
	CALL SaveSettings	; and save settings in data EEPROM
DispUpdate:
	CALL ShowTime		; now update time displays and LEDs1-3	
	GOTO LoopStart		; and go back to start of loop
;	
;   Program's main loop ends here -- subroutines follow
;
;   ***********************************************************
;
LookupDig:
;	routine to lookup display code for a number in w (0-9),
;	then return with this code in w register. If numbers >9
;	are input due to corruption, a dash will be displayed.
;	NOTE: Should not be moved. Will 'misbehave' if it crosses
;	an 8-bit address boundary in prog memory space;
;
	ADDWF PCL,1			; add number to PC, to generate
						; a computed GOTO to one of the
						; lines below...
	RETLW h'FC'			; code to display '0'
	RETLW h'60'			; code to display '1'
	RETLW h'DA'			; code to display '2'
	RETLW h'F2'			; code to display '3'
	RETLW h'66'			; code to display '4'
	RETLW h'B6'			; code to display '5'
	RETLW h'BE'			; code to display '6'
	RETLW h'E0'			; code to display '7'
	RETLW h'FE'			; code to display '8'
	RETLW h'F6'			; code to display '9'
	RETLW h'02'			; code to display '-'
	RETLW h'02'			; code to display '-'
	RETLW h'02'			; code to display '-'
	RETLW h'02'			; code to display '-'
	RETLW h'02'			; code to display '-'
	RETLW h'02'			; code to display '-'
	RETURN				; just in case it falls through...
;
;	********************************************************
;
BounceWait:
	; routine to pause long enough for switch bounce to end
	MOVLW h'FA'			; set Counter2 for 250 loops (250ms approx)
	MOVWF Counter2
	CALL Delay1ms		; go away for 1ms
	DECFSZ Counter2, 1	; then decrement Counter2 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; reached zero, so return
;
;	*********************************************************
;
ByteSplit:
	; routine to split a hex byte in w (value 0-63) into two
	; decimal digits which are placed into Byte10 and Byte01
	; Enters, works & leaves in bank1, but relies on Temp3
	; address being = 7A/FA/17A/1FA
	ANDLW h'3F'			; first make sure byte value is less than 64
	MOVWF Temp3			; then save it in Temp3
	CLRF Byte10			; now initialise output digit values
	CLRF Byte01
Beg10:
	MOVLW h'0A'			; subtract 10d from input byte in Temp3
	SUBWF Temp3,0		; w = Temp3 - 0Ah
	BTFSS STATUS,C		;skip if we got a carry (Temp3 >= 10d)
	GOTO Beg1			;no carry, so Temp3 < 10d. Skip to Beg1
	BTFSS STATUS,Z		;carry was set, but check for zero
	GOTO PosRes			; no zero, so continue
	INCF Byte10,1		;0 and C, so must have been 10d exactly
	RETURN				;so just increment 10s & leave (in bank1)
PosRes:
	INCF Byte10,1		;positive result, so increment 10s
	MOVWF Temp3			;save remainder in w back to Temp3
	GOTO Beg10			;and continue looping
Beg1:
	INCF Byte01,1		;increment units -- must have at least one
	DECF Temp3,1		;now decrement Temp3
	BTFSS STATUS,Z		;skip if zero bit is set now
	GOTO Beg1			;not zero yet, so continue
	RETURN				;zero, so we're done -- leave (in bank1)
;
;	***********************************************************
;
CheckLiteLvl:
;	routine to check ambient light level via LDR board and ADC,
;	set DonTime and DoffTime for appropriate display brightness
;	(NOTE: enters in bank0 but leaves in bank1)
	BSF ADCON0,2		; to begin, set ADCON0(2) to start ADC
	BTFSC ADCON0,2		; then loop until conversion is done
	GOTO $-1			; (ADC clears ADCON0(2) when finished)
	MOVF ADRESH,0		; when done, fetch ADC high byte into w
	bank1
	MOVWF Vbrite		; then save in Vbrite

	MOVLW h'A0'			; now compare with A0h
	SUBWF Vbrite,0		; w now = Vbrite - A0
	BTFSC STATUS,C		; C=0 for neg result (i.e., Vbrite<A0h)
	GOTO Fullbright		; but C=1 means Vbrite >=A0, so set for 100%
	MOVLW h'60'			; <A0h, but now compare it with 60h
	SUBWF Vbrite,0		; w = Vbrite - 60h
	BTFSC STATUS,C		; C=0 for neg result (Vbrite <60h)
	GOTO Bright83pc		; but C=1 means Vbrite >=60, so set for 83%
	MOVLW h'30'			; <60h, but now compare it with 30h
	SUBWF Vbrite,0		; w = Vbrite - 30h
	BTFSC STATUS,C		; C=0 for neg result (Vbrite <30h)
	GOTO Bright66pc		; but C=1 means Vbrite >=30, so set for 66%
	MOVLW h'18'			; <30h, but now compare it with 18h
	SUBWF Vbrite,0		; w = Vbrite - 18h
	BTFSC STATUS,C		; C=0 for neg result (Vbrite < 18h)
	GOTO Bright50pc		; but C=1 means Vbrite >=18, so set for 50%
	MOVLW h'0C'			; <18h, but now compare it with 0Ch
	SUBWF Vbrite,0		; w = Vbrite - 0Ch
	BTFSC STATUS,C		; C=0 for neg result (Vbrite < 0Ch)
	GOTO Bright33pc		; but C=1 means Vbrite >=0C, so set for 33%

	MOVLW h'05'			; Vbrite <0Ch, so set DonTime to 05
	MOVWF DonTime
	MOVLW h'19'			; and DoffTime to 25d (19h)
	MOVWF DoffTime		; for 5/30 or 16% brightness
	RETURN				; then leave

Bright33pc:
	MOVLW h'0A'			; set DonTime to 10d (0Ah)
	MOVWF DonTime
	MOVLW h'14'			; and DoffTime to 20d (14h)
	MOVWF DoffTime		; for 10/30 or 33% brightness
	RETURN				; then leave

Bright50pc:
	MOVLW h'0F'			; set DonTime to 15d (0Fh)
	MOVWF DonTime
	MOVLW h'0F'			; and DoffTime to 15d (0Fh)
	MOVWF DoffTime		; for 15/30 or 50% brightness
	RETURN				; then leave

Bright66pc:
	MOVLW h'14'			; set DonTime to 20d (14h)
	MOVWF DonTime
	MOVLW h'0A'			; and DoffTime to 10d (0Ah)
	MOVWF DoffTime		; for 20/30 or 66% brightness
	RETURN				; then leave

Bright83pc:
	MOVLW h'19'			; set DonTime to 25d (19h)
	MOVWF DonTime
	MOVLW h'05'			; and DoffTime to 05d (05h)
	MOVWF DoffTime		; for 25/30 or 83% brightness
	RETURN				; then leave

Fullbright:
	MOVLW h'1E'			; set DonTime to 30d (1Eh)
	MOVWF DonTime
	MOVLW h'01'			; and DoffTime to 01d (01h)
	MOVWF DoffTime		; for 29/30 or 97% brightness
	RETURN				; then leave
;
;	***********************************************************
;
CheckMem:
	;routine to check if settings saved in data EEPROM, if so restore 'em
	; Enters in bank0, leaves in bank0 also
	CLRW					; set w for first address in EE memory (00h)
	MOVWF Temp1				; and save in Temp1 
	CALL ReadEE				; then read back data (returns with it in w)
	XORLW h'0F'				; check if it's first flag (0Fh)
	BTFSS STATUS,Z			; skip if Z=1 (i.e., it was a match)
	RETURN					; no first match, so leave now with settings unchanged
	INCF Temp1,1			; now set for second flag address (01h)
	MOVF Temp1,0			; copy into w
	CALL ReadEE				; and read data
	XORLW h'F0'				; check if it's second flag (F0h)
	BTFSS STATUS,Z			; skip if Z=1 (i.e., it was a match)
	RETURN					; no second match, so leave now with settings unchanged
	INCF Temp1,1			; did match, so set for first data address (02h)
	MOVF Temp1,0			; copy into w
	CALL ReadEE				; read it
	MOVWF Flags				; and save as Flags
	INCF Temp1,1			; now repeat for next address (03h)
	MOVF Temp1,0
	CALL ReadEE				; read it
	MOVWF ULOffseth			; and save as UTC-Local std time offset hours
	INCF Temp1,1			; now repeat for next address (04H)
	MOVF Temp1,0
	CALL ReadEE				; read it
	MOVWF ULOffsetm			; and save as UTC-Local std time offset minutes
	RETURN					; before returning with settings restored
;
;	****************************************************************
;
CleanSlate:
	; routine to clear GPS-derived info registers for clean start
	; enters & leaves in bank0, but indir addresses bank1 regs (IRP=0)
	MOVLW h'0F'			; now set counter for number of regs (A0-AE)
	MOVWF Counter3
	MOVLW h'A0'			; also set up address pointer start
	MOVWF FSR
WipeLoopStart:
	MOVLW h'00'			; now place a zero in w
	MOVWF INDF			; and store 0 in current ptr address
	DECFSZ Counter3,1	; now decrement counter & skip when zero
	GOTO $+2
	GOTO ClearDigs		; did reach zero, so all done -- move on
	INCF FSR,1			; Counter3 not zero yet, so increment FSR
	GOTO WipeLoopStart	; and loop back to continue
ClearDigs:
	MOVLW h'06'			; now set counter for number of regs (B0-B5)
	MOVWF Counter3
	MOVLW h'B0'			; set up address pointer start
	MOVWF FSR
Dig0loopSt:
	MOVLW h'02'			; now place code to display '-' in w
	MOVWF INDF			; and store char in current ptr address
	DECFSZ Counter3,1	; decrement counter & skip when zero
	GOTO $+2			; otherwise still going
	RETURN				; counter hit zero, so done -- leave
	INCF FSR,1			; increment FSR
	GOTO Dig0loopSt		; and loop back to continue
;
;	***********************************************************

Delay1ms:
	; routine to delay approx 1ms (1006mc) before returning
	MOVLW h'FA'			; set Counter1 for 250 loops
	MOVWF Counter1
	NOP
	DECFSZ Counter1, 1	; decrement Counter1 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; reached zero (250 x 4mc loops), so return

Delay100us:
	; routine to delay approx 100us (101mc) before returning
	MOVLW h'18'			; set Counter1 for 24d loops
	MOVWF Counter1
	NOP
	DECFSZ Counter1, 1	; decrement Counter1 & skip when zero
	GOTO $-2			; not zero yet, so loop back
	RETURN				; reached zero (23x4mc loops +9mc), so return

DlyDigOn:
	; routine to provide digit on-time delay (DonTime x 100us)
	; leaves in bank0...
	bank1
	MOVF DonTime,0		; first fetch DonTime into w
	bank0
	MOVWF Counter2		; then load it into Counter2
	CALL Delay100us		; pause now for 100us
	DECFSZ Counter2,1	; then decrement Counter2, skip when -> 0
	GOTO $-2			; if not zero yet, loop back
	RETURN				; reached zero, so return

DlyDigOff:
	; routine to provide digit off-time delay (DoffTime x 100us)
	; leaves in bank0...
	bank1
	MOVF DoffTime,0		; first fetch DoffTime into w
	bank0
	MOVWF Counter2		; then load it into Counter2
	CALL Delay100us		; pause now for 100us
	DECFSZ Counter2,1	; then decrement Counter2, skip when -> 0
	GOTO $-2			; if not zero yet, loop back
	RETURN				; reached zero, so return
;
;	***********************************************************
;
DoAScan:
	; routine to scan keys S1-S5, verify value for valid keypress.
	; Returns with code (or zero if no key pressed) saved in Keycode
	CALL ScanKeys		; go scan keyboard
	MOVWF Keycode		; then save return value in w to Keycode
	MOVF Keycode,1		; now check if Keycode = 0
	BTFSC STATUS, Z		; skip if Z=0 (ie Keycode not 0)
	RETURN				; otherwise return, as no key was pressed
	CALL BounceWait		; key was pressed, so wait for 250ms
	CALL ScanKeys		; & scan again to verify
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS,Z		; skip if Z=0 (ie w not 0, has a keycode)
	RETURN				; was zero, so false alarm - exit with w=0
	XORWF Keycode, 0	; not zero, so compare with first code
	BTFSC STATUS,Z		; skip if Z=0 (because codes didn't match)
	RETURN				; did match, so return with code in Keycode
	CALL BounceWait		; no match, so wait another 250ms
	GOTO DoAScan		; and back to scan again
;
;	***********************************************************
;
IncrOffseth:
;	routine to increment UTC-local time hours offset (ULOffseth)
;	(called when S4 pressed, or S3 AND we are coming from LST mode)
	MOVF ULOffseth,0	; first fetch hours offset in w
	ADDLW h'01'			; then add one
	MOVWF ULOffseth		; and save again (leaves in w as well)
	SUBLW h'17'			; now subtract it from 17h (= 23d)
	BTFSC STATUS,C		; w = 17h - ULOffseth (C=1 if pos or zero)
	GOTO $+2			; C=1, so ULOffseth <= 17h, so leave
	CLRF ULOffseth		; C=0 means (ULOffseth > 17h), so reset ULOffseth
	RETURN				; and return
;
;	***********************************************************
;
IncrOffsetm:
;	routine to increment UTC-local time minutes offset (ULOffsetm)
;	(called when S5 pressed)
	MOVF ULOffsetm,0	; first fetch minutes offset in w
	BTFSS STATUS,Z		; if it's zero (Z=1), skip
	GOTO $+4			; but if not (Z=0), go make it zero
	MOVLW h'1E'			; was zero, so make it 1Eh (=30d)
	MOVWF ULOffsetm		; and save again
	GOTO $+2			; then go
	CLRF ULOffsetm		; wasn't zero, so make it so
	RETURN				; and return

;	***********************************************************
;
PutInEE:
	; routine to write a data byte (in w) into current EEADR in
	; EE memory, also increment memory address in EEADR
	; (leaves in bank0)
	; Interrupts disabled before calling, enabled afterwards.
	bank2				; first swing up to bank2
	MOVWF EEDATA		; so we can move data into EEDATA
	bank3				; then move up to bank3 so we
	BCF EECON1, 7		; can clear EEPGD bit for data EEPROM access
	BSF EECON1, WREN	; and also enable EE writing
	MOVLW h'55'			; and carry out 'magic write sequence'
	MOVWF EECON2
	MOVLW h'AA'
	MOVWF EECON2
	BSF EECON1, WR		; so it does write the byte
	BCF EECON1, WREN	; then disable EE writing again
	BTFSC EECON1, WR	; now check to see if write has been done
	GOTO $-1			; loop back until it is (WR->0)
	bank0				; now down to bank0 so we can
	BTFSS PIR2, EEIF	; test EEIF bit (in PIR2)
	GOTO $-1			; loop back until it's set (write complete)
	BCF PIR2, EEIF		; once it is set, clear it ready for next write
	bank2				; then up to bank2 so we can
	INCF EEADR,1		; increment the data EEPROM address
	bank0				; and finally swing back down to bank0
	RETURN				; before returning
;
;	***********************************************************
;
RestorOffseth:
;	routine to decrement UTC-local time hours offset (ULOffseth)
;	(called when S2 pressed AND we are returning from DLST mode).
	MOVLW h'01'			; first load w with 01h
	SUBWF ULOffseth,1	; then subtract from ULOffseth (-> ULOffseth)
	BTFSC STATUS,C		; did it go negative? (C->0 if so)
	RETURN				; if not just return
	MOVLW h'17'			; but if it did, change it to 17h (23d)
	MOVWF ULOffseth		; and resave before we
	RETURN				; return
;
;	*********************************************************
;
ReadEE:
	; routine to read data byte from EE memory address (in w)
	; (enters and leaves in bank0)
	bank2				; first switch to bank2
	MOVWF EEADR			; so we can set up address register
	bank3				; then switch to bank3 so we
	BCF EECON1,7		; can  clear EEPGD bit for data EEPROM access
	BSF EECON1,RD		; and also set the read control bit
	bank2				; then switch back to bank2
	MOVF EEDATA,0		; and read the data into w reg
	bank0				; then swing down to bank0
	RETURN				; before we leave
;
;	***********************************************************
;
SaveSettings:
	; routine to save current settings in data EE memory
	; Enters & leaves in bank0, also disables interrupts during
	BCF INTCON, GIE		; first disable interrupts while we do it
	CLRW				; first set data EEPROM starting address
    bank2				; then up to bank2 so we can
	MOVWF EEADR			; save it
	MOVLW h'0F'			; then write first mem flag byte
	CALL PutInEE
	MOVLW h'F0'			; then write second mem flag byte
	CALL PutInEE		; (returns in bank0)
	MOVF Flags,0		; fetch Flags setting into w
	CALL PutInEE		; and then write it to EEPROM
	MOVF ULOffseth,0	; now do the UTC-local std time hours offset
	CALL PutInEE
	MOVF ULOffsetm,0	; and the UTC-local std time mins offset
	CALL PutInEE
	BSF INTCON, GIE		; finally re-enable interrupts
	RETURN				; before we return
;
;	********************************************************
;		
ScanKeys:
	; routine to look for a pressed key, return with code in w
	; (or with a null in w if no key is pressed)
	BTFSC PORTD, 0		; skip if RD0 is clear (S1 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "1"			; S1 pressed, so return with "1" in w
	BTFSC PORTD, 1		; next skip if RD1 is clear (S2 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "2"			; S2 pressed, so return with "2" in w
	BTFSC PORTD, 4		; next skip if RD4 is clear (S3 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "3"			; S3 pressed, so return with "3" in w
	BTFSC PORTD, 3		; next skip if RD3 is clear (S4 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "4"			; S4 pressed, so return with "4" in w
	BTFSC PORTD, 2		; next skip if RD2 is clear (S5 pressed)
	RETLW h'00'			; no keys pressed, so return with null in w reg
	RETLW "5"			; S5 pressed, so return with "5" in w
;
;	***********************************************************
;
ShowTime:
	; routine to turn on LED1 (UTC), LED2 (Loc std time) or LED3 (loc DLS time)
	; as desired (from Flags), then display selected time (HHMMss) on DISP1-6
	; Enters and leaves in bank0
	CLRF PORTA			; first clear PORTA to ensure LEDs are all off
	BTFSS Flags,0		; now check Flags(0)
	GOTO $+3			; if not set, keep looking
	BSF PORTA,2			; was set, so turn on LED1 to indicate UTC
	GOTO DispUTC		; then skip to display UTC
	BTFSS Flags,1		; how about Flags(1)?
	GOTO $+3			; if not set, keep looking
	BSF PORTA,1			; was set, so turn on LED2 to indicate LST time
	GOTO DispLST		; mode, then skip to display local time
	BSF PORTA,0			; Flags(2) must be set, so turn on LED3 for DLS time
	GOTO DispLST		; mode, then jump to display local time

DispUTC:
	bank1				; first up to Bank1, to access time values
	MOVF UTCh10,0		; now start by getting hoursx10 value in w
	CALL LookupDig		; fetch equiv display code
	MOVWF Dchar1		; and save it in Dchar1 ready for display

	MOVF UTCh01,0		; now for hours x1
	CALL LookupDig
	MOVWF Dchar2

	MOVF UTCm10,0		; then for minutes x10
	CALL LookupDig
	MOVWF Dchar3

	MOVF UTCm01,0		; and minutes x1
	CALL LookupDig
	MOVWF Dchar4

	GOTO ComSecs		; then jump to display secs (same for all)

DispLST:
;	this section shared by LST and DLST time display modes
	bank1				; first up to Bank1, to access time values
	MOVF LSTh10,0		; now first get hoursx10 value
	CALL LookupDig		; fetch equiv display code
	MOVWF Dchar1		; and save in Dchar1 ready for display

	MOVF LSTh01,0		; now repeat for the other values & digits
	CALL LookupDig
	MOVWF Dchar2

	MOVF LSTm10,0
	CALL LookupDig
	MOVWF Dchar3

	MOVF LSTm01,0
	CALL LookupDig
	MOVWF Dchar4

ComSecs:
	; this section shared by all three time functions
	MOVF UTCs10,0
	CALL LookupDig
	MOVWF Dchar5

	MOVF UTCs01,0
	CALL LookupDig
	MOVWF Dchar6

DisplayEm:
	; routine to display codes in Dchar1-6, in DISP1-6
	; (enters in mem bank1, leaves in bank0)
	MOVF Dchar1,0		; first get Dchar1 into w
	bank0				; swing down to bank0
	MOVWF PORTB			; and place in RB1-RB7
	BSF PORTC,2			; then set RC2 to turn on DISP1
	CALL DlyDigOn		; pause for dig on time
	BCF PORTC,2			; then turn off DISP1 again
	CALL DlyDigOff		; and pause again for off time

	bank1				; up to bank1
	MOVF Dchar2,0		; next get Dchar2 into w
	bank0				; down to bank0
	MOVWF PORTB			; and place in RB1-RB7
	BSF PORTC,3			; then set RC3 to turn on DISP2
	CALL DlyDigOn		; pause for dig on time
	BCF PORTC,3			; then turn off DISP2 again
	CALL DlyDigOff		; and pause again for off time

	bank1
	MOVF Dchar3,0		; next get Dchar3 into w
	bank0
	MOVWF PORTB			; and place it in RB1-RB7
	BSF PORTC,0			; then set RC0 to turn on DISP3
	CALL DlyDigOn		; pause for dig on time
	BCF PORTC,0			; then turn off DISP3 again
	CALL DlyDigOff		; and pause again for off time

	bank1
	MOVF Dchar4,0		; next get Dchar4 into w
	bank0
	MOVWF PORTB			; and place it in RB1-RB7
	BSF PORTC,1			; then set RC1 to turn on DISP4
	CALL DlyDigOn		; pause for dig on time
	BCF PORTC,1			; then turn off DISP4 again
	CALL DlyDigOff		; and pause again for off time

	bank1
	MOVF Dchar5,0		; next get Dchar5 into w
	bank0
	MOVWF PORTB			; and place it in RB1-RB7
	BSF PORTC,4			; then set RC4 to turn on DISP5
	CALL DlyDigOn		; pause for dig on time
	BCF PORTC,4			; then turn off DISP5 again
	CALL DlyDigOff		; and pause again for off time

	bank1
	MOVF Dchar6,0		; next get Dchar6 into w
	bank0
	MOVWF PORTB			; and place it in RB1-RB7
	BSF PORTC,5			; then set RC1 to turn on DISP6
	CALL DlyDigOn		; pause for dig on time
	BCF PORTC,5			; and turn off DISP6 again
	CALL DlyDigOff		; and pause again for off time

	CLRF PORTB			; finally clear PORTB and PORTC
	CLRF PORTC			; so all displays off again

	RETURN				; then return (in bank0)

;
;	***********************************************************
;
;   end of main routines -- interrupt servicing routines follow
	
IntService:
	; routine to service interrupts from USART module
	; when a character has been received from the GPS Rx
	MOVWF WSave			; first save context (w and status regs)
	SWAPF STATUS,0		; using SWAPF here to avoid STATUS change
	MOVWF SSave
	MOVF PCLATH,0		; also save PCLATH (PC hi bits) so we can
	MOVWF PCHiSave		; restore them before we leave
	MOVF FSR,0			; also save FSR reg for restoring on exit
	MOVWF FSRSave		; context saving now done

	bank0				; make sure we're in data memory bank0
	MOVF RCSTA,0		; now start by reading RCSTA register into w
	ANDLW h'06'			; mask off all except error flag bits (1,2)
	BTFSS STATUS,Z		; if Z=1, skip because no error
	GOTO Back4More		; but if Z=0, an error must have occurred so leave
	BTFSS PIR1,RCIF		; no error, so see if RCIF is set
	GOTO Back4More		; if it wasn't just depart again
	MOVF RCREG,0		; it was, so read char from RCREG (clears RCIF)
	MOVWF TempChr		; and save in TempChr
	BSF RCSTA,CREN		; set CREN again to re-enable USART receiving
	MOVLW h'2A'			; now check if char was an asterisk
	XORWF TempChr,0
	BTFSC STATUS,Z		; skip if it wasn't
	GOTO ParseNSave		; Z=1, so must be an * - go parse & save sentence	
	MOVLW h'0D'			; Z=0, so wasn't an *. Now check if it's a CR
	XORWF TempChr,0
	BTFSC STATUS,Z		; skip if Z=0, because it wasn't a CR	
	GOTO ResetBufPtr	; it was a CR, so just reset buf ptr & leave
	MOVLW h'0A'			; wasn't a CR, so check if it's a LF instead
	XORWF TempChr,0
	BTFSC STATUS,Z		; skip if Z=0, because it wasn't a LF
	GOTO ResetBufPtr	; must have been an LF, so reset ptr & leave
	MOVLW h'24'			; not LF either, so see if it's a "$" (line start)
	XORWF TempChr,0
	BTFSC STATUS,Z		; skip if Z=0, because it wasn't a $
	GOTO ResetBufPtr	; must have been a $, so reset buf ptr & leave
	MOVF BufPtr,0		; wasn't any of the above, so fetch buf pointer in w
	MOVWF FSR			; and load into FSR (indir addr reg)
 	MOVF TempChr,0		; then fetch char itself back into w
	MOVWF INDF			; and save it in current buffer location (via FSR)
	INCF FSR, 0			; then incr buf pointer & load into w
	GOTO SavePtrNGo		; ready to save in BufPtr and return

ParseNSave:
	; routine to read NMEA sentence buffer (020 - 06Fh, ie bank0)
	; and update saved GPS time info
	MOVLW h'20'			; first load FSR with buffer start addr
	MOVWF FSR
	MOVF INDF,0			; now read buffer's first char into w reg
	XORLW "G"			; and check if it's a G
	BTFSS STATUS,Z		; if Z=1, must be a G so skip
	GOTO ResetBufPtr	; not a G, so skip sentence altogether
	INCF FSR,1			; if it was a G, now check third char
	INCF FSR,1
	MOVF INDF,0			; by reading it into w reg
	XORLW "G"			; and checking if it's another G
	BTFSS STATUS,Z		; if Z=1, must be a G so skip
	GOTO ResetBufPtr	; Z=0, so it wasn't a G - just depart
	INCF FSR,1			; if it was a G, now check fourth char
	MOVF INDF,0
	XORLW "G"			; to check if it's a G also
	BTFSS STATUS,Z		; if Z=1, must be a third G so continue
	GOTO ResetBufPtr	; otherwise reset buf ptr & depart 
	MOVLW h'26'			; it was a G, so set FSR to read first GPS time char
	MOVWF FSR			; (because it must be a GPGGA sentence)

	MOVF INDF,0			; now read first time char
	ANDLW h'0F'			; strip off high nibble (ASCII->hex)
	bank1				; then set for bank1
	MOVWF UTCh10		; and save it in UTCh10 (UTC hours x10)
	bank0				; then back to bank0

	INCF FSR,1			; then fetch and save other UTC time chars
	MOVF INDF,0
	ANDLW h'0F'			; strip off high nibble (ASCII->hex)
	bank1
	MOVWF UTCh01		; the units of hours
	bank0

	INCF FSR,1
	MOVF INDF,0
	ANDLW h'0F'			; strip off high nibble (ASCII->hex)
	bank1
	MOVWF UTCm10		; the tens of minutes
	bank0

	INCF FSR,1
	MOVF INDF,0
	ANDLW h'0F'			; strip off high nibble (ASCII->hex)
	bank1
	MOVWF UTCm01		; the units of minutes
	bank0

	INCF FSR,1
	MOVF INDF,0
	ANDLW h'0F'			; strip off high nibble (ASCII->hex)
	bank1
	MOVWF UTCs10		; the tens of seconds
	bank0

	INCF FSR,1
	MOVF INDF,0
	ANDLW h'0F'			; strip off high nibble (ASCII->hex)
	bank1
	MOVWF UTCs01		; the units of seconds

UpdateLocal:
;	UTC values have been updated, so now we need to update
;	local time (LST or DLST) as well.
;	routine entered in bank1
	MOVF UTCh10,0		; first fetch UTC hours x10 into w
	MOVWF Byte10		; save in Byte10
	MOVF UTCh01,0		; then fetch UTC hours x1 into w
	MOVWF Byte01		; save in Byte01
	CALL FoldIntoHex	; then combine them into a single hex byte
	MOVWF UTChhex		; which we save in UTChhex (00 - 17h)

	MOVF UTCm10,0		; now fetch UTC mins x10 into w
	MOVWF Byte10		; save in Byte10
	MOVF UTCm01,0		; then fetch UTC mins x1 into w
	MOVWF Byte01		; save in Byte01
	CALL FoldIntoHex	; then combine them into a hex byte
	MOVWF UTCmhex		; which we save in UTCmhex (00 - 3Bh)
						; (also remains in w)
	bank0				; then down to Bank0
	ADDWF ULOffsetm,0	; now add UTC-LST mins offset (00 or 1Eh) into w
	bank1
	MOVWF LSTmhex		; and save the result in hex (also left in w)
	SUBLW h'3B'			; now subtract it from 3Bh (=59d), so w = 3B - LSTmhex
	BTFSC STATUS,C		; so if C=0 (neg result), LSTmhex > 3Bh
	GOTO $+6			; but if C=1, pos or zero result so LSTmhex <=3Bh
	MOVLW h'3C'			; C=0 means neg result (>3B), so subtract 3Ch from it
	SUBWF LSTmhex,1		; and save it again, but also
	MOVLW h'01'			; set LSThhex to 01, to save carry of 1 hour
	MOVWF LSThhex		;
	GOTO $+2			; then proceed to hours addition
	CLRF LSThhex		; since LSTmhex was <=3B, there is no carry to hours
						; so set LSThhex to 00 before proceeding to hours
	MOVF UTChhex,0		; now fetch UTC hours in hex, into w
	ADDWF LSThhex,0		; and add it to prepared LSThhex (00 or 01)
	bank0				; then down to bank0
	ADDWF ULOffseth,0	; so we can add in UTC-LST hours offset (00-17h)
	bank1				; then back up to bank1
	MOVWF LSThhex		; so we can save as LST hours (also left in w)
	SUBLW h'17'			; now subtract it from 17h (=23d) so w = 17 - LSThhex
	BTFSC STATUS,C		; skip if C=0 (negative result, so LSThhex >17h)
	GOTO TurnintoDigits	; C=1 means pos or zero result (LSThhex was <=17h) so move on
	MOVLW h'18'			; LSThhex was >17h,  so subtract 18h from it
	SUBWF LSThhex,1		; and save again, before dropping thru to...

TurnintoDigits:
;	routine to convert the LST hex values into digits, for display
;	(remains in bank1)
	MOVF LSThhex,0		; first get LSThhex back into w
	CALL ByteSplit		; then go split into digits
	MOVF Byte10,0		; then fetch tens digit and
	MOVWF LSTh10		; save it as LST hoursx10
	MOVF Byte01,0		; and do likewise for units digit
	MOVWF LSTh01
	MOVF LSTmhex,0		; now follow the same procedure for minutes
	CALL ByteSplit
	MOVF Byte10,0
	MOVWF LSTm10
	MOVF Byte01,0
	MOVWF LSTm01		; now we have done both hours and minutes
	
DropBack:
	bank0					; swing back to bank0

ResetBufPtr:
	MOVLW h'20'				; reset sentence buffer pointer for a new sentence
SavePtrNGo:
	MOVWF BufPtr			; save w in BufPtr for saving next char in buffer
	GOTO Back2Reality		; and leave to wait for another char
Back4More:
	; exit after finding USART comms error
	MOVF RCREG,0			; read char from RCREG so RCIF is cleared
	BCF RCSTA,CREN			; clear CREN bit in RCSTA (after receive error)
	BSF RCSTA,CREN			; then set it again to re-enable reception
Back2Reality:
	MOVF FSRSave, 0			; fetch saved FSR contents -> w
	MOVWF FSR				; and restore them to FSR
	MOVF PCHiSave,0			; fetch saved hi bits from PCLATH -> w
	MOVWF PCLATH			; and restore them to PCLATH
	SWAPF SSave,0			; now restore context (status & w regs)
	MOVWF STATUS			;
	SWAPF WSave,1			; using SWAPFs to avoid changing STATUS
	SWAPF WSave,0			;
	RETFIE					; and return, re-enabling ints (TOS->PC, 1->GIE)
;
;	************************************************************
;
FoldIntoHex:
;	routine to combine two BCD numbers (Byte10 and Byte01) into
;	a single 8-bit hex number so we can do a bit of maths.
;	Maximum value for Byte10 is 05, max for Byte01 is 09. Hex
;	resultant is returned in w register. Enters & leaves in bank1
	MOVF Byte10,0		; first fetch 10's byte into w
	BTFSC STATUS,Z		; skip if Z=0 (means Byte10 = 01,02,03,04 or 05)
	GOTO Addem			; Z=1, so Byte10 = 0. Just add them
	XORLW h'01'			; now test if it's a 1
	BTFSC STATUS,Z		; skip if Z=0, because it isn't a 1
	GOTO Itza1			; but if Z=1, Byte10 = 01 so go Itza1
	MOVF Byte10,0		; not 1, so fetch Byte10 back into w
	XORLW h'02'			; so we can test if it's a 2
	BTFSC STATUS,Z		; skip if Z=0, because it isn't a 2 either
	GOTO Itza2			; Z=1, so must be a 2. Go handle it
	MOVF Byte10,0		; not 2, so fetch Byte10 back into w
	XORLW h'03'			; so we can test if it's a 3
	BTFSC STATUS,Z		; skip if Z=0, because it isn't a 3 either
	GOTO Itza3			; Z=1, so must be a 3. Go handle it
	MOVF Byte10,0		; not 3, so fetch Byte10 back into w
	XORLW h'04'			; so we can test if it's a 4
	BTFSC STATUS,Z		; skip if Z=0, because it isn't a 4 either
	GOTO Itza4			; Z=1, so must be a 4. Go handle it
	MOVLW h'32'			; must be a 5, so prepare to add 32h to Byte01
	GOTO Addem			; and jump to do it
Itza4:
	MOVLW h'28'			; it's a 4, so prepare to add 28h to Byte01
	GOTO Addem			; and jump to do it
Itza3:
	MOVLW h'1E'			; it's a 3, so prepare to add 1Eh to Byte01
	GOTO Addem			; and jump to do it	
Itza2: 
	MOVLW h'14'			; must be 02, so prepare to add 14h to Byte01
	GOTO Addem			; and jump to do it
Itza1:
	MOVLW h'0A'			; is 01, so prepare to add 0Ah to hoursx1
Addem:
	ADDWF Byte01,0		; now add w to Byte01, -> w
	RETURN				; and return, with result in w
	
	END
	
