/*

Ethernet Time Module: add on for the Ethernet Web Server
September 2009, Mauro Grassi (c)

Compiling Instructions:

Enable Integer Promotions ON, Small Code Model, Large Data Model, Multi Bank, Enable All Optimizations, No Procedural Abstraction

Version history:

1.00: basic structure.
1.10: improved version with compacted TIME_T structure (made of BYTES rather than ints) to save memory and enlarge stack.
1.20: detects when no synchronising packet received (timeout feature).
1.30: added support for automatic baud rate detection. If the baudrate is changed in the ethernet web server, simply disable the SNTP module, wait
	  for the time module to go into timeout and then reenable the time module. The new baud rate will be detected. Works with 4.90 web server and over.
1.32: small improvements goes with version 4.92 of the Ethernet Web Server. Added MODE_FIRMWARE to display the firmware version.
1.34: changed so that it begins timed out.
1.36: changed key system substantially. moved all interrupt handlers to top priority.
1.40: improved version, tested with Ethernet Web Server Version 4.94.

*/

#include "p18f1320.h"
#include <math.h>

#pragma config OSC      = INTIO2
#pragma config DEBUG	= OFF
#pragma config IESO     = ON
#pragma config PWRT     = ON
#pragma config FSCM		= OFF
#pragma config BOR      = OFF
#pragma config BORV		= 27
#pragma config WDT      = OFF
#pragma config WDTPS    = 32768
#pragma config MCLRE    = OFF
#pragma config LVP      = OFF			// low voltage programming
#pragma	config STVR		= ON			// reset on stack underflow/overflow
#pragma config CP0      = OFF
#pragma config CP1      = OFF
#pragma config CPB      = OFF
#pragma config CPD      = OFF
#pragma config WRT0     = OFF
#pragma config WRT1     = OFF
#pragma config WRTB     = OFF      		// Boot Block Write Protection
#pragma config WRTC     = OFF
#pragma config WRTD     = OFF
#pragma config EBTR0    = OFF
#pragma config EBTR1    = OFF
#pragma config EBTRB    = OFF

/*
Seven Segment is as follows:

 	 --7--
 5 |       | 6
  	 --4--
 0 |       | 2
 	 --1--      o 3

*/

#define VERSION 				1.40
#define MAGIC_BYTE				0x76
#define DEBUG_PRESS				0
#define SCREEN_REFRESH_PERIOD	0xF0			// 0xF0 is around 60Hz
// these delays are measured in sixteenth of a second units
#define BUTTON_LONG_PRESS_DELAY		20
#define BUTTON_IDLE_DELAY			1
#define BUTTON_PRESS_DELAY			1

#define NO_KEY		0
#define LONG_KEY	1
#define SHORT_KEY	2

#define BUTTON0		(PORTAbits.RA5)
#define CATHODE0	(PORTAbits.RA3)
#define CATHODE1	(PORTAbits.RA2)
#define CATHODE2	(PORTAbits.RA1)
#define CATHODE3	(PORTAbits.RA0)
#define SEG0		(PORTBbits.RB5) 
#define SEG1		(PORTBbits.RB0) 
#define SEG2		(PORTAbits.RA6) 
#define SEG3		(PORTAbits.RA7)	
#define SEG4		(PORTBbits.RB7)	
#define SEG5		(PORTBbits.RB6) 
#define SEG7		(PORTBbits.RB3) 
#define SEG6		(PORTBbits.RB2) 

#define VAL0			1
#define VAL1			2
#define VAL2			4
#define VAL3			8
#define VAL4			16
#define VAL5			32
#define VAL6			64
#define VAL7			128

#define SECONDS			0
#define MINUTES			1
#define HOURS			2
#define DAY				3
#define MONTH			4
#define YEAR			5
#define WDAY			6
#define MAGIC_HEADER	0xF4
#define CHARBASE		55
#define MODE_DISPLAY_TIME_HH_MM            0    //Display Time HH:MM mode
#define MODE_DISPLAY_TIME_MM_SS            1    //Display Time MM:SS mode
#define MODE_DISPLAY_DATE_DD_MM            2    //Display Date DD:MM mode
#define MODE_DISPLAY_YEAR_YYYY             3    //Display Year YYYY mode
#define MODE_DISPLAY_ALL                   4	//Display Extended Scrolling Time Mode
#define MODE_DISPLAY_TIME_SCROLLING        5	//Display Scrolling Time Mode
#define MODE_FIRMWARE					   6	//Display Firmware version
#define MODE_OFF						   7	//Display Off
#define MODE_MODULUS					   8	//Total Number
#define EPOCH_YEAR 		1970
#define TIMEOUT_PERIOD	700
#define SYNC_TIMEOUT	300

#define DISPLAY_MODE_ADDR			16
#define TWENTY_FOUR_HOUR_MODE_ADDR	17
#define DAY_MONTH_ORDER_ADDR		18

typedef unsigned char byte;

typedef struct
{
  unsigned char secs;			// Seconds.	[0-60] 
  unsigned char mins;			// Minutes.	[0-59] 
  unsigned char hours;		    // Hours.	[0-23] 
  unsigned char day;			// Day.		[1-31] 
  unsigned char month;		    // Month.	[0-11] 
  unsigned char year;			// Year	since EPOCH
  unsigned char wday;           // weekday (0=Monday, 6=Sunday)
  unsigned char week;			// the first, second, third or fourth such weekday of the month
  unsigned char first;			// the first of the month is this weekday (0=Monday, 6=Sunday)
  unsigned char daylightSaving;	// 1 if daylightsaving time 0 otherwise
} TIME_T;

unsigned char buttonReady;
unsigned int buttonLatch;
unsigned char buttonKey;
unsigned int timeOut;
TIME_T currentTimeData;
TIME_T timeData;
unsigned char display[5];
unsigned char decp;
unsigned int sixteenthSecs;		
unsigned char digit;
unsigned char dispTime;
unsigned char twentyfourhourmode;
unsigned char daymonthorder;
unsigned char displayCount;
unsigned char *timePtr;
unsigned char lastReceivedUART;
unsigned char indexUART;
unsigned char displayMode;

void displayClockMode(unsigned char);
void myISRHigh(void);
void myISR(void);

#pragma code _HIGH_INTERRUPT_VECTOR = 0x000008
void _high_ISR (void)
{
    _asm goto myISRHigh _endasm
}

#pragma code _LOW_INTERRUPT_VECTOR = 0x000018
void _low_ISR (void)
{
    _asm goto myISR _endasm
}

void DelayMs(int delay)
{
	int n;
	while(delay--)
	{
	for(n = 0; n <80; n++) 
	{
	_asm
		nop
	_endasm
	}	
   }
}

void StartWrite(void)
{
    /*
     * A write command can be prematurely terminated by MCLR or WDT reset
     */
    //byte x;

	DelayMs(1);						// protection against too many writes and wearing out the memory!
	/*
	x=TMR1L;
	while(x!=0)
	{
     x=TMR1L;    
    }   	
    */
	INTCONbits.GIE=0;
	EECON2 = 0x55;
    EECON2 = 0xAA;
    EECON1bits.WR = 1;
    INTCONbits.GIE=1;
}
//end StartWrite

int ReadEEPROM(byte address)
{
    EECON1 = 0x00;
    EEADR =  address;
    EECON1bits.RD = 1;
    return (int)EEDATA;
}

byte WriteEEPROM(byte address, byte data)
{
		byte i;
		i=ReadEEPROM(address);
		if(i!=data)
		{
		EEADR=(byte)address;
        EEDATA = data;
        EECON1 = 0b00000100;    		// Setup writes: EEPGD=0,WREN=1
        StartWrite();
        while(EECON1bits.WR!=0);       	// Block till WR bit is clear
		}
		return ReadEEPROM(address);
}

unsigned rom char
sevenSegment[64]=
{
(VAL0 | VAL1 | VAL2 | VAL5 | VAL7 | VAL6),				// 0
(VAL6 | VAL2),											// 1
(VAL7 | VAL6 | VAL4 | VAL0 | VAL1),						// 2
(VAL7 | VAL6 | VAL4 | VAL2 | VAL1),						// 3
(VAL5 | VAL4 | VAL6 | VAL2),							// 4
(VAL7 | VAL5 | VAL4 | VAL2 | VAL1),						// 5
(VAL7 | VAL5 | VAL4 | VAL0 | VAL1 | VAL2),				// 6
(VAL7 | VAL6 | VAL2),									// 7
(VAL0 | VAL1 | VAL2 | VAL4 | VAL5 | VAL6 | VAL7),		// 8
(VAL2 | VAL4 | VAL5 | VAL6 | VAL7),						// 9
(VAL5 | VAL7 | VAL6 | VAL4 | VAL0 | VAL2),				// A
(VAL5 | VAL4 | VAL2 | VAL1 | VAL0),						// B
(VAL7 | VAL5 | VAL0 | VAL1),							// C
(VAL6 | VAL4 | VAL0 | VAL1 | VAL2),						// D
(VAL5 | VAL4 | VAL7 | VAL0 | VAL1),						// E
(VAL5 | VAL4 | VAL7 | VAL0),							// F
(VAL5 | VAL4 | VAL6 | VAL7 | VAL2 | VAL1),				// G
(VAL5 | VAL6 | VAL4 | VAL0 | VAL2),						// H
(VAL5 | VAL0),											// I
(VAL6 | VAL2 | VAL1),									// J
(VAL5 | VAL4 | VAL0 | VAL2 | VAL6),						// K
(VAL5 | VAL0 | VAL1),									// L
0,
(VAL4 | VAL0 | VAL2),									// N
(VAL0 | VAL4 | VAL2 | VAL1),							// O
(VAL5 | VAL6 | VAL4 | VAL7 | VAL0),						// P
(VAL6 | VAL7 | VAL4 | VAL5 | VAL2),						// Q
(VAL0 | VAL4),											// R
(VAL7 | VAL5 | VAL4 | VAL1 | VAL2),						// S
(VAL5 | VAL0 | VAL4 | VAL1),							// T
(VAL0 | VAL1 | VAL4),									// U
0,
0,
0,
(VAL5 | VAL4 | VAL6 | VAL2 | VAL1),						// Y
(VAL7 | VAL6 | VAL4 | VAL0 | VAL1),						// Z
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0,
0
};

void copyTimeData(TIME_T* dest, TIME_T* source)
{
	unsigned char* ip;
	unsigned char* op;
	unsigned char i;

	ip=(unsigned char*)source;
	op=(unsigned char*)dest;

	i=0;
	while(i<sizeof(TIME_T))
	{
	*op++=*ip++;
	i++;
	}

	if(daymonthorder)
	{
		// swap day and month
		i=dest->day;
		dest->day=dest->month;
		dest->month=i;
	}

	if(twentyfourhourmode==0)
	{
		// adjust
		if(dest->hours>12)dest->hours-=12;
	}
}

/*
void putDigitMask(unsigned char value, unsigned char decp)
{
	if(value & VAL0)SEG0=0; else SEG0=1;
	if(value & VAL1)SEG1=0; else SEG1=1;
	if(value & VAL2)SEG2=0; else SEG2=1;
	if(decp)SEG3=0; else SEG3=1;
	if(value & VAL4)SEG4=0; else SEG4=1;
	if(value & VAL5)SEG5=0; else SEG5=1;
	if(value & VAL6)SEG6=0; else SEG6=1;
	if(value & VAL7)SEG7=0; else SEG7=1;
}
*/

void initDisplay(void)
{
	ADCON1=0x7F;
	TRISB=0x00;
	TRISA=0x30;
	digit=0;
	display[0]=0;
	display[1]=0;
	display[2]=0;
	display[3]=0;
	decp=0;
	CATHODE0=0;
	CATHODE1=0;
	CATHODE2=0;
	CATHODE3=0;
	SEG0=1;
	SEG1=1;
	SEG2=1;
	SEG3=1;
	SEG4=1;
	SEG5=1;
	SEG6=1;
	SEG7=1;
	T1CON=0x11;
	PR2=0xFF;
//	T2CON=0X7F;
	T3CON=0x11;
	PIR1bits.TMR1IF=0;
	PIE1bits.TMR1IE=1;
	IPR1bits.TMR1IP=1;
	PIR2bits.TMR3IF=0;
	IPR2bits.TMR3IP=1;
	PIE2bits.TMR3IE=1;
//	PIR1bits.TMR2IF=0;
//	IPR1bits.TMR2IP=1;
//	PIE1bits.TMR2IE=1;
}

void initUART(void)
{
	lastReceivedUART=0;
	indexUART=0;
	timePtr=(unsigned char*)&currentTimeData;
	BAUDCTL=0x08;
	SPBRGH=0;
	SPBRG=16;
	TRISBbits.TRISB4=1;
	TXSTA=0x24;
	RCSTA=0x90;
	PIR1bits.RCIF=0;
	IPR1bits.RCIP=1;
	PIE1bits.RCIE=1;
}

void initSequence(void)
{
	OSCCON=0x73;
	PIR1=0;
	PIR2=0;
	PIE1=0;
	PIE2=0;
	INTCON=0;
	sixteenthSecs=0;
	timeOut=0;
	dispTime=0;
	displayMode=0;
	displayCount=0;
	twentyfourhourmode=1;
	daymonthorder=0;
	buttonReady=0;
	buttonLatch=0;
	buttonKey=NO_KEY;
	initDisplay();
	initUART();
	RCONbits.IPEN=1;	// enable prioritized interrupts!
	INTCONbits.GIEL=0;
	INTCONbits.GIE=1;
}

void disAdec(int value, int cursor)
{
	display[cursor & 3]=sevenSegment[((value/10)%10)];
	display[((cursor+1) & 3)]=sevenSegment[value % 10];
}

void disInt(int value, int cursor)
{
	display[cursor & 3]=sevenSegment[(value/1000)%10];
	display[((cursor+1) & 3)]=sevenSegment[(value/100)%10];
	display[((cursor+2) & 3)]=sevenSegment[(value/10)%10];
	display[((cursor+3) & 3)]=sevenSegment[value % 10];
}

void disIntHex(int cursor, unsigned int value)
{
	display[cursor & 3]=sevenSegment[(value>>12) & 0x0F];
	display[((cursor+1) & 3)]=sevenSegment[(value>>8) & 0x0F];
	display[((cursor+2) & 3)]=sevenSegment[(value>>4) & 0x0F];
	display[((cursor+3) & 3)]=sevenSegment[value & 0x0F];
}

/*
void refreshDisplay(void)
{
	if(digit<3)digit++; else digit=0;
	CATHODE0=0;
	CATHODE1=0;
	CATHODE2=0;
	CATHODE3=0;
	switch(digit)
	{
		case 0:
			putDigitMask(display[0], decp & 8);
			CATHODE0=1;
			break;
		case 1:
			putDigitMask(display[1], decp & 4);
			CATHODE1=1;
			break;
		case 2:
			putDigitMask(display[2], decp & 2);
			CATHODE2=1;
			break;
		case 3:
			putDigitMask(display[3], decp & 1);
			CATHODE3=1;
			break;
		default:
			break;
	}
}
*/

#pragma interrupt myISRHigh
void myISRHigh(void)
{
	byte value, idecp;
	if(PIR1bits.RCIF)
	{
//			RCSTAbits.CREN=0;
			lastReceivedUART=RCREG;
			if(lastReceivedUART==MAGIC_HEADER)
			{
				indexUART=0;
				timePtr=(unsigned char*)&currentTimeData;
				timeOut=TIMEOUT_PERIOD;
			}
			else
			if(dispTime==0)
			{
				if(indexUART<(sizeof(TIME_T)))
				{
				*timePtr++=lastReceivedUART;
				indexUART++;	
				if(indexUART>=(sizeof(TIME_T)))dispTime=1;
				}
			}
		PIR1bits.RCIF=0;
	} 

	if(PIR1bits.TMR1IF)
	{
		T1CONbits.TMR1ON=0;
		if(timeOut)timeOut--; 
		if(digit<3)digit++; else digit=0;
		switch(digit)
		{
		case 0:
		default:
			CATHODE3=0;
			idecp=decp & 8;
			break;
		case 1:
			CATHODE0=0;
			idecp=decp & 4;
			break;
		case 2:
			CATHODE1=0;
			idecp=decp & 2;
			break;
		case 3:
			CATHODE2=0;
			idecp=decp & 1;
			break;
		}
		value=display[digit];
		if(value & VAL0)SEG0=0; else SEG0=1;
		if(value & VAL1)SEG1=0; else SEG1=1;
		if(value & VAL2)SEG2=0; else SEG2=1;
		if(idecp)SEG3=0; 		else SEG3=1;
		if(value & VAL4)SEG4=0; else SEG4=1;
		if(value & VAL5)SEG5=0; else SEG5=1;
		if(value & VAL6)SEG6=0; else SEG6=1;
		if(value & VAL7)SEG7=0; else SEG7=1;
		switch(digit)
		{
		case 0:
		default:
			CATHODE0=1;
			break;
		case 1:
			CATHODE1=1;
			break;
		case 2:
			CATHODE2=1;
			break;
		case 3:
			CATHODE3=1;
			break;
		}
		TMR1H=SCREEN_REFRESH_PERIOD;
		TMR1L=0x00;
		T1CONbits.TMR1ON=1;
		PIR1bits.TMR1IF=0;
	}

	if(PIR2bits.TMR3IF)
	{
		if(sixteenthSecs<960)sixteenthSecs++; else sixteenthSecs=0;
		if((!timeOut)&&((sixteenthSecs % 32)==0))BAUDCTLbits.ABDEN=1;
		switch(buttonReady)
		{
			case 0:
				buttonLatch=0;
				buttonReady++;
				break;
			case 1:
				if(BUTTON0==0)buttonReady=0; else buttonLatch++;
				if(buttonLatch>BUTTON_IDLE_DELAY)buttonReady++;
				break;
			case 2:
				buttonLatch=0;
				buttonReady++;
				break;
			case 3:
				if(BUTTON0==1)buttonReady=2; else buttonLatch++;
				if(buttonLatch>BUTTON_PRESS_DELAY)buttonReady++;
				break;
			case 4:
				buttonLatch++;
				if(buttonLatch>BUTTON_LONG_PRESS_DELAY){ buttonKey=LONG_KEY; buttonReady++; }
				if(BUTTON0==1){ buttonKey=SHORT_KEY; buttonReady++; }
				break;
			default:
			case 5:
				if(buttonKey==NO_KEY)buttonReady=0;
				break;
		}
		PIR2bits.TMR3IF=0;
	}

/*
	if(PIR1bits.TMR2IF)
	{
	PIR1bits.TMR2IF=0;
	}	
*/
}

#pragma interrupt myISR
void myISR(void)
{

}

char getKey(void)
{
	char c;
	PIR1bits.RCIF=0;
	while(PIR1bits.RCIF==0);
	PIR1bits.RCIF=0;
	c=RCREG;
	return c;
}

void displayClockMode(unsigned char iclockmode)
{
		int  i, j;

		switch(iclockmode)
		{
			case MODE_DISPLAY_TIME_HH_MM:
					copyTimeData(&timeData, &currentTimeData);				
					disAdec(timeData.hours,0);
					disAdec(timeData.mins,2);
					if((timeData.secs & 0x02)==0)decp=0; else decp=4;	
					//if(twentyfourhourmode)decp|=8;
					break;

			case MODE_DISPLAY_TIME_MM_SS:
					copyTimeData(&timeData, &currentTimeData);				
					disAdec(timeData.mins,0);
					disAdec(timeData.secs,2);
					if((timeData.secs & 0x01)!=0)decp=0; else decp=4;
					break;

			case MODE_DISPLAY_DATE_DD_MM:
					// display date mode
					displayCount=displayCount % 4;
					i=displayCount;
					if(i==0)
					{
						display[0]=sevenSegment['D'-CHARBASE];		// d
						display[1]=sevenSegment['A'-CHARBASE];		// A	
						display[2]=sevenSegment['Y'-CHARBASE];		// Y			
						display[3]=0;
						decp=0;
					} else
					{		
						copyTimeData(&timeData, &currentTimeData);
						disAdec(timeData.day,0);
						disAdec(timeData.month,2);
						decp=4;
					}
					displayCount++;
					break;

			case MODE_DISPLAY_YEAR_YYYY:
				// display the year
				displayCount=displayCount % 4;
				i=displayCount;
				if(i==0)
				{
				display[0]=sevenSegment['Y'-CHARBASE];		// Y
				display[1]=sevenSegment['E'-CHARBASE];		// E
				display[2]=sevenSegment['A'-CHARBASE];		// A
				display[3]=sevenSegment['R'-CHARBASE];		// R
				decp=0;
				} else
				{
				copyTimeData(&timeData, &currentTimeData);
				disInt(timeData.year+EPOCH_YEAR, 0);
				decp=0;
				}
				displayCount++;
				break;

			case MODE_DISPLAY_ALL:
				displayCount=displayCount % 28;
				i=displayCount;
				if(i==0)
				{
					copyTimeData(&timeData, &currentTimeData);
					display[0]=0;
					display[1]=0;
					display[2]=0;
					display[3]=0;
					display[4]=0;
					decp=0;
				} else
				if(i==1)
				{
					display[4]=sevenSegment[ (timeData.hours/10) & 0x0F];
					decp=0;
				} else
				if(i==2)
				{
					display[4]=sevenSegment[ (timeData.hours % 10)& 0x0F];
					decp=0;
				} 
				else
				if(i==3)
				{
					display[4]=sevenSegment[ (timeData.mins /10) & 0x0F];
					decp=2;
				}
				else
				if(i==4)
				{
					display[4]=sevenSegment[ (timeData.mins % 10) & 0x0F];
					decp=4;
				}
				else
				if(i==5)
				{
					display[4]=0;
					decp=8;
				} else
				if(i==6)
				{
					display[4]=sevenSegment['D'-CHARBASE];		// d
					decp=0;
				} else
				if(i==7)
				{
					display[4]=sevenSegment['A'-CHARBASE];		// A
					decp=0;
				} else
				if(i==8)
				{
					display[4]=sevenSegment['Y'-CHARBASE];		// Y			
					decp=0;
				} else
				if(i==9)
				{
					display[4]=0;
					decp=0;
				} else
				if(i==10)
				{
					display[4]=sevenSegment[ (timeData.day/10) & 0x0F];
					decp=0;
				} else
				if(i==11)
				{
					display[4]=sevenSegment[ (timeData.day % 10) & 0x0F];
					decp=0;
				} else
				if(i==12)
				{
					display[4]=sevenSegment[ ((timeData.month)/10) & 0x0F];
					decp=2;
				}
				else
				if(i==13)
				{
					display[4]=sevenSegment[ ((timeData.month)%10) & 0x0F];
					decp=4;
				}
				else
				if(i==14)
				{
					display[4]=0;
					decp=8;
				} else
				if(i==15)
				{
					display[4]=sevenSegment['Y'-CHARBASE];		// Y
					decp=0;
				} else
				if(i==16)
				{
					display[4]=sevenSegment['E'-CHARBASE];		// E
					decp=0;
				} else
				if(i==17)
				{
					display[4]=sevenSegment['A'-CHARBASE];		// A
					decp=0;	
				} else
				if(i==18)
				{
					display[4]=sevenSegment['R'-CHARBASE];		// r
					decp=0;
				} else
				if(i==19)
				{		
					display[4]=0;
					decp=0;
				} else
				if(i==20)
				{
					j=(timeData.year+EPOCH_YEAR);
					display[4]=sevenSegment[ (j/1000) & 0x0F];
					decp=0;
				}
				else
				if(i==21)
				{
					j=(timeData.year+EPOCH_YEAR);
					j=j%1000;
					display[4]=sevenSegment[ (j/100) & 0x0F];
					decp=0;
				}
				else
				if(i==22)
				{
					j=(timeData.year+EPOCH_YEAR);
					j=j % 100;
					display[4]=sevenSegment[ (j/10) & 0x0F];
					decp=0;
				} else
				if(i==23)
				{
					j=(timeData.year+EPOCH_YEAR);
					j=j % 10;
					display[4]=sevenSegment[ (j & 0x0F)];
					decp=0;
				} else
				if(i==24)
				{
					display[4]=0;
					decp=0;
				} else
				if(i==25)
				{
					display[4]=0;
					decp=0;
				} else
				if(i==26)
				{
					display[4]=0;
					decp=0;
				} else	
				if(i==27)
				{
					display[4]=0;
					decp=0;
				} 
				display[0]=display[1];
				display[1]=display[2];
				display[2]=display[3];
				display[3]=display[4];			// shift one place
				displayCount++;
				break;

			case MODE_DISPLAY_TIME_SCROLLING:
				displayCount=displayCount % 10;
				i=displayCount;
				if(i==0)
				{
					copyTimeData(&timeData, &currentTimeData);
					display[0]=0;
					display[1]=0;
					display[2]=0;
					display[3]=0;
					display[4]=0;
					decp=0;
				} else
				if(i==1)
				{
					display[4]=sevenSegment[ (timeData.hours/10) & 0x0F];
					decp=0;
				} else
				if(i==2)
				{
					display[4]=sevenSegment[ (timeData.hours % 10)& 0x0F];
					decp=0;
				} 
				else
				if(i==3)
				{
					display[4]=sevenSegment[ (timeData.mins /10) & 0x0F];
					decp=2;
				}
				else
				if(i==4)
				{
					display[4]=sevenSegment[ (timeData.mins % 10) & 0x0F];
					decp=5;
				}
				else
				if(i==5)
				{
					display[4]=sevenSegment[ (timeData.secs /10) & 0x0F];
					decp=10;
				} 
				else
				if(i==6)
				{
					display[4]=sevenSegment[ (timeData.secs % 10) & 0x0F];
					decp=20;
				}
				else
				if(i==7)
				{
					display[4]=0;
					decp=8;
				} else
				if(i==8)
				{
					display[4]=0;
					decp=0;
				} else
				if(i==9)
				{
					display[4]=0;
					decp=0;
				}
				display[0]=display[1];
				display[1]=display[2];
				display[2]=display[3];
				display[3]=display[4];			// shift one place
				displayCount++;
				break;


		case MODE_FIRMWARE:
				i=(int)VERSION;
				j=(int)((((float)VERSION-(float)i)*100)+0.5);
				display[0]=sevenSegment['F'-CHARBASE];				// F
				display[1]=sevenSegment[((int)VERSION) & 0x0F];			
				disAdec(j, 2);
				decp=4;
				break;

		case MODE_OFF:
		default:
				displayCount=displayCount % 10;
				if(displayCount==0)
				{
				display[0]=sevenSegment['O'-CHARBASE];		// O
				display[1]=sevenSegment['F'-CHARBASE];		// F	
				display[2]=sevenSegment['F'-CHARBASE];		// F
				display[3]=0;
				decp=0;
				}
				else
				{
				display[0]=0;
				display[1]=0;
				display[2]=0;
				display[3]=0;
				decp=0;
				}
				if(displayCount<9)displayCount++; else displayCount=1;
				break;
		}
}
	
void getSettings(void)
{
	if(ReadEEPROM(0)==MAGIC_BYTE)
	{
		displayMode=ReadEEPROM(DISPLAY_MODE_ADDR);
		twentyfourhourmode=ReadEEPROM(TWENTY_FOUR_HOUR_MODE_ADDR);
		daymonthorder=ReadEEPROM(DAY_MONTH_ORDER_ADDR);
	}
}

void saveSettings(void)
{
	WriteEEPROM(DISPLAY_MODE_ADDR, displayMode);
	WriteEEPROM(TWENTY_FOUR_HOUR_MODE_ADDR, twentyfourhourmode);
	WriteEEPROM(DAY_MONTH_ORDER_ADDR, daymonthorder);
	WriteEEPROM(0, MAGIC_BYTE);
}

void advanceMode(void)
{
	displayCount=0;
	if(displayMode<(MODE_MODULUS-1))displayMode++; else displayMode=0;
	saveSettings();
}

void decreaseMode(void)
{
	displayCount=0;
	if(displayMode>0)displayMode--; else displayMode=MODE_MODULUS-1;
	saveSettings();
}

void main(void)
{
#if (DEBUG_PRESS)
	int longPress;
	int shortPress;
	longPress=0;
	shortPress=0;
#endif
	initSequence();
	getSettings();
	saveSettings();
	buttonKey=NO_KEY;
	while(1)
	{
		if(timeOut)
		{
		if(dispTime)
		{
#if (DEBUG_PRESS)
		disAdec(longPress, 0);
		disAdec(shortPress, 2);
#else
		displayClockMode(displayMode);
#endif
		dispTime=0;
		}
		}
		else
		{
			displayCount=0;
			if(displayMode!=MODE_OFF)
			{
			display[0]=VAL4;
			display[1]=VAL4;
			display[2]=VAL4;
			display[3]=VAL4;
			if(sixteenthSecs & 16)decp=0; else decp=4;
			} else
			{
			display[0]=0;
			display[1]=0;
			display[2]=0;
			display[3]=0;
			decp=0;
			}
		}

		if(buttonKey!=NO_KEY)
		{
			if(timeOut)
			{
				if(buttonKey==LONG_KEY)
				{
					// long press is context sensitive...
#if (DEBUG_PRESS)
					longPress++;
#endif
					switch(displayMode)
					{
						case MODE_DISPLAY_TIME_HH_MM: 
								twentyfourhourmode^=1;
								saveSettings();
								break;
						case MODE_DISPLAY_ALL:                   
						case MODE_DISPLAY_TIME_SCROLLING:
						case MODE_DISPLAY_YEAR_YYYY:            
						case MODE_OFF:
						case MODE_FIRMWARE:
								displayCount=0;
								break;

						case MODE_DISPLAY_DATE_DD_MM:
								daymonthorder^=1;
								saveSettings();
								break;
						case MODE_DISPLAY_TIME_MM_SS:
								decreaseMode();
								break;
					}
				} else
				{
				// short press
#if (DEBUG_PRESS)
				shortPress++;
#endif
				advanceMode();
				}
			}
			buttonKey=NO_KEY;
		}
	}
}
