
#include "p33Fxxxx.h"
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "SPDIF.h"
#include "Wave_Config.h"
#include "main.h"
#include "db.h"
#include "crc8.h"

unsigned char sweep_is_paused;

static const unsigned char pro_channel_data[24] = { 1|4, 8, 4|8|32, 0, 0, 0, 'S', 'C', 'S', 'G', 'T', 'e', 's', 't', 0, 0, 0, 0, 0, 0, 0, 0, 64|128, 0 };
void compute_channel_status(unsigned char pro, unsigned char emph) {
    unsigned char data[24], i;

    if( pro ) {
        memcpy(data, pro_channel_data, 24);
        data[0] |= (emph?8:0)|(sample_rate==44?64:0)|(sample_rate==48?128:0);
        data[23] = crc8(data, 23);
    } else {
        memset(data, 0, 24);
        data[0] = 4|(emph?8:0);
        if( sample_rate != 44 ) {
            data[3] = 2;
        }
    }
    for( i = 0; i < 192; ++i ) {
        chan_status[i] = (data[i>>3] & (1<<(i&7))) ? 0xc0 : 0x00;
    }
}

void setup_int_ptr() {
    DACBufferHead = DAC_BUFFER_SIZE/2;
    DACBufferTail = 0;

    if( wavetype == Sine ) {
        switch(mode) {
        case Null:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_Null;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_Null;
                dac_int_ptr = &mode_analog_Null;
            }
            break;
        case Mixed:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_Sine_Mixed;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_Sine_Mixed;
                dac_int_ptr = &mode_analog_Sine_Mixed;
            }
            break;
        case Pulsed:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_Sine_Pulsed;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_Sine_Pulsed;
                dac_int_ptr = &mode_analog_Sine_NonMixed;
            }
            break;
        case Sweep:
            if( sweep_is_paused ) {
                if( sample_rate == 48 ) {
                    int_ptr = &mode_dual_Sine_Plain;
                    dac_int_ptr = &mode_analog_DoNothing;
                } else {
                    int_ptr = &mode_Sine_Plain;
                    dac_int_ptr = &mode_analog_Sine_Plain;
                }
            } else {
                if( sample_rate == 48 ) {
                    if( sweepdir == sweep_down ) {
                        int_ptr = &mode_dual_Sine_Sweep_Geometric_rev;
                    } else {
                        int_ptr = &mode_dual_Sine_Sweep_Geometric;
                    }
                    dac_int_ptr = &mode_analog_DoNothing;
                } else {
                    if( sweepdir == sweep_down ) {
                        int_ptr = &mode_Sine_Sweep_Geometric_rev;
                    } else {
                        int_ptr = &mode_Sine_Sweep_Geometric;
                    }
                    dac_int_ptr = &mode_analog_Sine_NonMixed;
                }
            }
            break;
        default:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_Sine_Plain;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_Sine_Plain;
                dac_int_ptr = &mode_analog_Sine_Plain;
            }
            break;
        }
    } else {
        switch(mode) {
        case Null:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_Null;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_Null;
                dac_int_ptr = &mode_analog_Null;
            }
            break;
        case Mixed:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_NonSine_Mixed;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_NonSine_Mixed;
                dac_int_ptr = &mode_analog_NonSine_Mixed;
            }
            break;
        case Pulsed:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_NonSine_Pulsed;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_NonSine_Pulsed;
                dac_int_ptr = &mode_analog_NonSine_Pulsed;
            }
            break;
        case Sweep:
            if( sweep_is_paused ) {
                if( sample_rate == 48 ) {
                    int_ptr = &mode_dual_NonSine_Plain;
                    dac_int_ptr = &mode_analog_DoNothing;
                } else {
                    int_ptr = &mode_NonSine_Plain;
                    dac_int_ptr = &mode_analog_NonSine_Plain;
                }
            } else {
                if( sample_rate == 48 ) {
                    if( sweepdir == sweep_down ) {
                        int_ptr = &mode_dual_NonSine_Sweep_Geometric_rev;
                    } else {
                        int_ptr = &mode_dual_NonSine_Sweep_Geometric;
                    }
                    dac_int_ptr = &mode_analog_DoNothing;
                } else {
                    if( sweepdir == sweep_down ) {
                        int_ptr = &mode_NonSine_Sweep_Geometric_rev;
                        dac_int_ptr = &mode_analog_NonSine_Sweep_Geometric_rev;
                    } else {
                        int_ptr = &mode_NonSine_Sweep_Geometric;
                        dac_int_ptr = &mode_analog_NonSine_Sweep_Geometric;
                    }
                }
            }
            break;
        default:
            if( sample_rate == 48 ) {
                int_ptr = &mode_dual_NonSine_Plain;
                dac_int_ptr = &mode_analog_DoNothing;
            } else {
                int_ptr = &mode_NonSine_Plain;
                dac_int_ptr = &mode_analog_NonSine_Plain;
            }
            break;
        }
    }
}

static unsigned short fix_freq_(unsigned short freq, unsigned char rate) {
    switch(rate) {
    case 44:
        return ((unsigned long)freq*1920+1)/882;
        break;
    case 48:
        return freq << 1;
        break;
    default:
        return freq;
    }
}
static unsigned long time_to_samples_(unsigned short time, unsigned char rate) {
    switch(rate) {
    case 44:
        return 4410L * (unsigned long)time;
    case 48:
        return 4800L * (unsigned long)time;
    default:
        return 9600L * (unsigned long)time;
    }
}
static unsigned long ms_to_samples_(unsigned short ms, unsigned char rate) {
    switch(rate) {
    case 44:
        return 441L * (unsigned long)ms / 10L;
    case 48:
        return 48L * (unsigned long)ms;
    default:
        return 96L * (unsigned long)ms;
    }
}

static unsigned short fix_freq(unsigned short freq) {
    return fix_freq_(freq, sample_rate);
}
static unsigned long time_to_samples(unsigned short time) {
    return time_to_samples_(time, sample_rate);
}
static unsigned long ms_to_samples(unsigned short ms) {
    return ms_to_samples_(ms, sample_rate);
}


void set_sample_rate(unsigned char rate, unsigned char analog) {
    unsigned char pllfbd;

    if( rate == sample_rate && is_analog == analog ) {
        return;
    }

    if( analog ) {
        DCICON1bits.DCIEN = 0;
    } else {
        DAC1CONbits.DACEN = 0;
    }

#ifdef REVISIONB
    RPINR24bits.CSCKR = rate == 44 ?  3 :  4;
#else
    RPINR24bits.CSCKR = rate == 44 ?  4 :  3;
#endif
    DCICON3bits.BCG   = rate == 48 ?  2 :  0;
    DCICON1bits.CSCKD = rate == 48 ?  0 :  1;
    pllfbd            = rate == 96 ? 24 : 22;
    DAC1STATbits.LITYPE = analog ? 1 : 0;  // interrupt if FIFO is empty / not full
    DAC1STATbits.RITYPE = analog ? 1 : 0;  // interrupt if FIFO is empty / not full

    if( pllfbd != PLLFBD ) {
        PLLFBD = pllfbd;

        while( !OSCCONbits.LOCK )
            ;
    }

    ACLKCONbits.APSTSCLR = rate == 48 ? 6 : 7; // divide DAC clock by 1 or 2
    DAC1CONbits.DACEN = rate == 48 || analog;
    DCICON1bits.DCIEN = !analog;
    RPOR1bits.RP2R    = analog ? 20 : 13; // set RB2 = OC3 if analog, DCI Data Output if digital

    sample_rate = rate;
    is_analog = analog;

    sample_counter |= 0x40000000;

    setup_int_ptr();
}

static void set_new_dbs(unsigned char left, unsigned char right, unsigned char shift) {
    register unsigned short scale = dbscale[left] >> shift;
    if( scale != sine_scale_l ) {
        ampl_change_scale_l = scale;
        ampl_change_counter_l = 201;
    }

    scale = dbscale[right] >> shift;
    if( scale != sine_scale_r ) {
        ampl_change_scale_r = scale;
        ampl_change_counter_r = 201;
    }
}

static void set_new_scales(unsigned short left, unsigned short right) {
    if( sine_scale_l != left || sine_scale_r != right ) {
        ampl_change_scale_l = left;
        ampl_change_scale_r = right;
        ampl_change_counter_l = 201;
    }
}

static void set_new_scales_loop(unsigned short lt_left, unsigned short lt_right, unsigned short gt_left, unsigned short gt_right, unsigned long sample_limit) {
    while(1) {
        if( sample_counter < sample_limit ) {
            set_new_scales(lt_left, lt_right);
            if( sample_counter < sample_limit ) {
                return;
            }
        } else {
            set_new_scales(gt_left, gt_right);
            if( sample_counter >= sample_limit ) {
                return;
            }
        }
    }
}

static unsigned long calc_delta(unsigned short phase) {
    return (unsigned long)(360 - phase) * 800L / 3L;
}

static void lock_sine_offsets(unsigned short delta) {
    register unsigned long old_offset, new_offset;
    do {
        old_offset = sine_offset_l;
        new_offset = old_offset + delta;
        if( new_offset >= 96000 ) {
            new_offset -= 96000;
        }
        sine_offset_r = new_offset;
    } while( sine_offset_l != old_offset );
}

void set_locked(wavetype_t wt, unsigned short freq, unsigned short phase, unsigned char l_db, unsigned char r_db) {
    freq = fix_freq(freq);
    sine_freq_l = sine_freq_r = freq;
    if( mode == Locked ) {
        register unsigned long delta;

        delta = calc_delta(phase);
        if( delta == 96000 ) {
            delta = 0;
        }
        lock_sine_offsets(delta);

        set_new_dbs(l_db, r_db, 0);

        if( wavetype != wt ) {
            wavetype = wt;
            setup_int_ptr();
        }
    } else {
        int_ptr = &mode_Null;
        dac_int_ptr = &mode_analog_Null;
        wavetype = wt;
        mode = Locked;
        sine_offset_l = 24000;
        sine_offset_r = sine_offset_l + calc_delta(phase);
        if( sine_offset_r >= 96000 ) {
            sine_offset_r -= 96000;
        }
        sine_scale_l = dbscale[l_db];
        sine_scale_r = dbscale[r_db];
        setup_int_ptr();
    }
}

void set_mixed_independent(mode_t _mode, wavetype_t wt, unsigned short l_freq, unsigned short r_freq, unsigned char l_db, unsigned char r_db, unsigned char l_mute, unsigned char r_mute) {
    l_freq = fix_freq(l_freq);
    r_freq = fix_freq(r_freq);
    sine_freq_l = l_freq;
    sine_freq_r = r_freq;

    if( mode == _mode ) {
        set_new_dbs(l_db, r_db, _mode == Mixed ? 1 : 0);

        lock_sine_offsets(0);

        if( wavetype != wt ) {
            wavetype = wt;
            setup_int_ptr();
        }
    } else {
        int_ptr = &mode_Null;
        dac_int_ptr = &mode_analog_Null;
        wavetype = wt;
        mode = _mode;
        sine_offset_l = sine_offset_r = 24000;
        sine_scale_l = _mode == Mixed ? (dbscale[l_db]>>1) : dbscale[l_db];
        sine_scale_r = _mode == Mixed ? (dbscale[r_db]>>1) : dbscale[r_db];
        mute_l = l_mute;
        mute_r = r_mute;
        setup_int_ptr();

        sample_counter |= 0x40000000;
    }
}

void set_independent(wavetype_t wt, unsigned short l_freq, unsigned short r_freq, unsigned char l_db, unsigned char r_db) {
    set_mixed_independent(Independent, wt, l_freq, r_freq, l_db, r_db, 0, 0);
}

void set_mixed(wavetype_t wt, unsigned short a_freq, unsigned short b_freq, unsigned char a_db, unsigned char b_db, unsigned char l_mute, unsigned char r_mute) {
    set_mixed_independent(Mixed, wt, a_freq, b_freq, a_db, b_db, l_mute, r_mute);
}

static void setup_pulsed_vars(unsigned char on_db, unsigned char off_db, unsigned short on_time, unsigned short off_time) {
    pulse_on_time = on_time;
    pulse_off_time = off_time;
    pulse_on_scale = dbscale[on_db];
    pulse_off_scale = dbscale[off_db];
    pulse_on_samples = ms_to_samples(on_time);
    pulse_total_samples = ms_to_samples(on_time + off_time);
}

void set_pulsed(wavetype_t wt, unsigned short freq, unsigned short on_time, unsigned short off_time, unsigned char on_db, unsigned char off_db, unsigned char l_mute, unsigned char r_mute) {
    freq = fix_freq(freq);
    sine_freq_l = sine_freq_r = freq;

    if( mode == Pulsed ) {
        setup_pulsed_vars(on_db, off_db, on_time, off_time);

        set_new_scales_loop(mute_l ? 0 : pulse_on_scale, mute_r ? 0 : pulse_on_scale, mute_l ? 0 : pulse_off_scale, mute_r ? 0 : pulse_off_scale, pulse_on_samples);

        if( wavetype != wt ) {
            wavetype = wt;
            setup_int_ptr();
        }
    } else {
        int_ptr = &mode_Null;
        dac_int_ptr = &mode_analog_Null;
        wavetype = wt;
        mode = Pulsed;
        sine_offset_l = sine_offset_r = 24000;
        sine_scale_l = dbscale[l_mute ? 99 : on_db];
        sine_scale_r = dbscale[r_mute ? 99 : on_db];
        mute_l = l_mute;
        mute_r = r_mute;

        setup_pulsed_vars(on_db, off_db, on_time, off_time);

        ampl_change_counter_l = 0;
        setup_int_ptr();

        sample_counter |= 0x40000000;
    }
}

static unsigned long calc_sweep_rate_arith(unsigned long samples, unsigned long from, unsigned long to) {
  return ((unsigned long)((to-from)&255)<<24) / samples + ((((unsigned long)((to-from)&~255)<<16) / samples)<<8);
}
static unsigned long calc_sweep_rate_geom(unsigned long samples, unsigned long from, unsigned long to) {
  return (unsigned long)(11631178.0f/(samples / (log((float)to/(float)from) / log(2))) + 1.5/*0.5*/);
}

void set_sweep(wavetype_t wt, sweeptype_t st, unsigned short start_freq, unsigned short finish_freq, unsigned short sweep_time, unsigned short off_time, unsigned char db, unsigned char l_mute, unsigned char r_mute) {
    unsigned short fixed_start_freq = fix_freq(start_freq);
    unsigned short fixed_finish_freq = fix_freq(finish_freq);

    if( mode == Sweep && sweeptype == st && fixed_start_freq == sweep_start_freq && fixed_finish_freq == sweep_finish_freq && sweep_time == sweep_on_time && off_time == sweep_off_time ) {
        register unsigned short scale;

        scale = dbscale[db];
        sweep_scale = scale;
        set_new_scales_loop(mute_l ? 0 : scale, mute_r ? 0 : scale, 0, 0, sweep_on_samples);

        lock_sine_offsets(0);

        if( wavetype != wt ) {
            wavetype = wt;
            setup_int_ptr();
        }
    } else {
        unsigned long on_samples;

        dac_int_ptr = &mode_analog_Null;
        wavetype = wt;
        sweeptype = st;
        mode = Sweep;
        sine_freq_l = sine_freq_r = fixed_start_freq;
        sine_offset_l = sine_offset_r = 0;
        sine_scale_l = dbscale[l_mute ? 99 : db];
        sine_scale_r = dbscale[r_mute ? 99 : db];

        sweep_start_freq = fixed_start_freq;
        sweep_finish_freq = fixed_finish_freq;
        sweep_on_time = sweep_time;
        sweep_off_time = off_time;
        sweep_scale = dbscale[db];
        sweep_on_samples = on_samples = time_to_samples(sweep_time);
        sweep_total_samples = time_to_samples(sweep_time + off_time * 10);

        switch(st) {
        case arithmetic:
            if( fixed_finish_freq >= fixed_start_freq ) {
                sweep_linear_rate = calc_sweep_rate_arith(on_samples, fixed_start_freq, fixed_finish_freq);
                sweepdir = sweep_up;
            } else {
                sweep_linear_rate = calc_sweep_rate_arith(on_samples, fixed_finish_freq, fixed_start_freq);
                sweepdir = sweep_down;
            }
            sweep_rate = 0;
            break;
//      case geometric:
        default:
            {
                unsigned short rate;
                if( fixed_finish_freq >= fixed_start_freq ) {
                    rate = calc_sweep_rate_geom(on_samples, fixed_start_freq, fixed_finish_freq);
                    sweepdir = sweep_up;
                } else {
                    rate = calc_sweep_rate_geom(on_samples, fixed_finish_freq, fixed_start_freq);
                    sweepdir = sweep_down;
                }
                if( rate < 1 ) {
                    sweep_rate = 1;
                } else {
                    sweep_rate = rate;
                }
                sweep_linear_rate = 0;
            }
            break;
        }

        sweep_counter = 0;

        sweep_is_paused = 0;

        setup_int_ptr();

        sample_counter |= 0x40000000;
    }
}

void trigger_manual_sweep() {
    // Atomically force the sample counter to overflow,
    // forcing the interrupt code to start a new sweep.
    // Hopefully this code compiles to only set the upper
    // word of the varaible.
    sample_counter |= 0x40000000;
    if( sweep_is_paused ) {
        pause_resume_manual_sweep();
    }
}

void pause_resume_manual_sweep() {
    sweep_is_paused ^= 1;
    setup_int_ptr();
    if( sweep_is_paused && sample_counter >= sweep_on_samples ) {
        // There are only three valid states for manual sweep:
        //    1) sweep running
        //    2) sweep paused (when it would otherwise be running)
        //    3) sweep waiting for trigger (during the off time)
        // Since paused only makes sense while it would otherwise be running,
        // if we try to pause while it's waiting for trigger, we just go
        // back to waiting.
        sweep_is_paused = 0;
        setup_int_ptr();
    }
}
