
#include "p33Fxxxx.h"
#include "LCD.h"
#include "Interface.h"

#define LCD_DELAY 5

#define LCD_RS  5
#define LCD_RW  6
#define LCD_E   7
#define LCD_DB4 8
#define LCD_DB5 9
#define LCD_DB6 10
#define LCD_DB7 11

#define LCD_INS_CLEARDISPLAY                   1
#define LCD_INS_RETURNHOME                     2
#define LCD_INS_SETENRTYMODE(dir,shift)       (4|((dir)>0?2:0)|((shift)?1:0))
#define LCD_INS_DISPLAYONOFF(disp,curs,blink) (8|((disp)?4:0)|((curs)?2:0)|((blink)?1:0))
#define LCD_INS_DISPLAYSHIFT(right)           (16|8|((right)?4:0))
#define LCD_INS_MOVECURSOR(right)             (16|  ((right)?4:0))
#define LCD_INS_SETFUNCTION(dl,nl,font)       (32|((dl)?16:0)|((nl)?8:0)|((font)?4:0))
#define LCD_INS_SETCGADDRESS(addr)            (64|(addr))
#define LCD_INS_SETDDADDRESS(addr)            (128|(addr))

// If this, this flag causes the code to check that anything written to the LCD reads
// back correctly, in case of flakey whatever.
#define LCD_VERIFY


inline static void Set_LCD_RS(unsigned char val) {
    LATAbits.LATA4  = val;
}
inline static void Set_LCD_RW(unsigned char val) {
    LATBbits.LATB6  = val;
}
inline static void Set_LCD_E(unsigned char val) {
    LATBbits.LATB7  = val;
}
inline static void Set_LCD_Data(unsigned char val) {
    LATB = (LATB&~0x0F00)|( ((unsigned short)val) << 8 );
}
#ifdef LCD_VERIFY
inline static unsigned char Get_LCD_Data_High() {
    return (PORTB >> 4) & 240;
}
inline static unsigned char Get_LCD_Data_Low() {
    return (PORTB >> 8) & 15;
}
#endif
inline static unsigned char Get_LCD_Busy() {
    return PORTBbits.RB11;
}


inline static void Set_LCD_Inputs() {
    TRISB |= 0x0F00;
}

inline static void Set_LCD_Outputs() {
    TRISB &= ~0x0F00;
}

static void LCD_Setup() {
    TRISAbits.TRISA4  = 0;
    TRISBbits.TRISB6  = 0;
    TRISBbits.TRISB7  = 0;
    Set_LCD_Inputs();
    Set_LCD_RS(0);
    Set_LCD_RW(0);
    Set_LCD_E(0);
}

#define Nop() {__asm__ volatile ("nop");}
static void delay(unsigned char nops) {
    while( nops ) {
        Nop();
        --nops;
    }
}
static void lcd_delay() {
    delay(LCD_DELAY);
}

static void LCD_WaitBusy() {
    unsigned char busy;
    Set_LCD_RW(1);
    do {
        lcd_delay();
        Set_LCD_E(1);
        lcd_delay();
        busy = Get_LCD_Busy();
        Set_LCD_E(0);
        lcd_delay();
        Set_LCD_E(1);
        lcd_delay();
        Set_LCD_E(0);
    } while( busy );
    Set_LCD_RW(0);    
}

#ifdef LCD_VERIFY
static unsigned char LCD_Read() {
    unsigned char ret;

    LCD_WaitBusy();

    Set_LCD_RS(1);
    Set_LCD_RW(1);

    lcd_delay();
    Set_LCD_E(1);
    lcd_delay();
    ret = Get_LCD_Data_High();
    Set_LCD_E(0);
    lcd_delay();
    Set_LCD_E(1);
    lcd_delay();
    ret |= Get_LCD_Data_Low();
    Set_LCD_E(0);

    Set_LCD_RW(0);    
    Set_LCD_RS(0);

    return ret;
}
#endif

static void LCD_InstructionX(unsigned char instruction, unsigned char mode) {
    LCD_WaitBusy();
    if( mode == 0 ) {
        Set_LCD_RS(1);
    }
    Set_LCD_Outputs();

    Set_LCD_Data(instruction>>4);
    lcd_delay();
    Set_LCD_E(1);
    lcd_delay();
    Set_LCD_E(0);
    lcd_delay();
    
    Set_LCD_Data(instruction&15);
    Set_LCD_E(1);
    lcd_delay();
    Set_LCD_E(0);
    lcd_delay();

    Set_LCD_Inputs();
    if( mode == 0 ) {
        Set_LCD_RS(0);
    }
}
#define LCD_Instruction(a) LCD_InstructionX((a),1)
#define LCD_Data(a) LCD_InstructionX((a),0)

void LCD_Init() {
    unsigned char i = 0;

    LCD_Setup();

#ifdef REVISIONB
    // set up timer two for PWM, and OC1 and OC2 for LCD contrast and brightness respectively
    T2CONbits.TCS = 0;
    T2CONbits.TCKPS = 0;
    TMR2 = 0;
    PR2 = 399;
    T2CONbits.TON = 1;
    OC1CONbits.OCTSEL = 2-2;
    OC1CONbits.OCM = 6;
    OC1RS = lcd_brightness*2;
    RPOR2bits.RP5R = 18; // set RB5 = Output Compare 1
    OC2CONbits.OCTSEL = 2-2;
    OC2CONbits.OCM = 6;
    OC2RS = lcd_contrast*4;
    AD1PCFGLbits.PCFG2 = 1; // set RB0 to be a digital output
    RPOR0bits.RP0R = 19; // set RB0 = Output Compare 2
#endif

    // do this four times since if it doesn't work the LCD won't go on at all
    for( i = 0; i < 4; ++i ) {
        LCD_Instruction(LCD_INS_SETFUNCTION(0,0,0));
        LCD_Instruction(LCD_INS_SETFUNCTION(0,1,0));
        LCD_Instruction(LCD_INS_DISPLAYONOFF(1,0,0));
        LCD_Instruction(LCD_INS_CLEARDISPLAY);
    }
}

static void LCD_WriteString(const char* str) {
    while( *str ) {
        LCD_Data(*str);
        ++str;
    }
}

#ifdef LCD_VERIFY
static unsigned char LCD_VerifyString(const char* str) {
    unsigned char ret = 0;
    while( *str ) {
        if( LCD_Read() != ((unsigned char*)str)[0] ) {
            break;
        }
        ++str;
        ++ret;
    }
    return ret;
}
#endif

void update_lcd_line(unsigned char line, const char* data) {
#ifdef LCD_VERIFY
    unsigned char pos = 0;
    unsigned char ins;
    unsigned char iter = 0;
    do {
        ins = LCD_INS_SETDDADDRESS(pos | (line ? 0x40 : 0x00));
        LCD_Instruction(ins);
        LCD_WriteString(data + pos);
        LCD_Instruction(ins);
        pos += LCD_VerifyString(data + pos);
    } while( data[pos] != '\0' && ++iter < 16 );
#else
    unsigned char ins = LCD_INS_SETDDADDRESS(line ? 0x40 : 0x00);
    LCD_Instruction(ins);
    LCD_WriteString(data);
#endif
}

static unsigned char cursor_is_on;
void set_lcd_cursor(unsigned char line, unsigned char col, unsigned char on) {
    if( on ) {
        if( !cursor_is_on ) {
            lcd_delay();
            LCD_Instruction(LCD_INS_DISPLAYONOFF(1,1,0));
            cursor_is_on = 1;
        }
        lcd_delay();
        LCD_Instruction(LCD_INS_SETDDADDRESS(col|(line?0x40:0)));
    } else {
        if( cursor_is_on ) {
            lcd_delay();
            LCD_Instruction(LCD_INS_DISPLAYONOFF(1,0,0));
            cursor_is_on = 0;
        }
    }
}

unsigned char buttons_pressed, buttons_canceled;
static unsigned short key_repeat;

void LCD_Interface() {
    unsigned char i;
    unsigned short last_timer = 0, second_timer = 0;
    unsigned short btns[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

    // set up timer 1 as our button timer
    T1CONbits.TCS = 0;
    T1CONbits.TCKPS = 0;
    TMR1 = 0;
    PR1 = 0xFFFF;
    T1CONbits.TON = 1;

#ifndef REVISIONB
    // set up timer two for PWM, and OC1 and OC2 for LCD contrast and brightness respectively
    T2CONbits.TCS = 0;
    T2CONbits.TCKPS = 1;
    TMR2 = 0;
    PR2 = 99;
    T2CONbits.TON = 1;
    OC1CONbits.OCTSEL = 2-2;
    OC1CONbits.OCM = 6;
    OC1RS = 25;
    RPOR2bits.RP5R = 18; // set RB5 = Output Compare 1
    OC2CONbits.OCTSEL = 2-2;
    OC2CONbits.OCM = 6;
    OC2RS = 10;
    AD1PCFGLbits.PCFG2 = 1; // set RB0 to be a digital output
    RPOR0bits.RP0R = 19; // set RB0 = Output Compare 2
#endif

    reinit_mode();

    update_lcd();

    LATAbits.LATA0 = 0;
    LATAbits.LATA1 = 0;
#ifdef REVISIONC
    TRISAbits.TRISA0 = 0;
    TRISAbits.TRISA1 = 0;
#endif

    while(1) {
        CNPU1bits.CN15PUE = 1;
        CNPU2bits.CN16PUE = 1;
        CNPU2bits.CN21PUE = 1;
        CNPU2bits.CN22PUE = 1;

#ifdef REVISIONC
	    LATAbits.LATA1 = 1;
		delay(LCD_DELAY*3);
#else
        TRISAbits.TRISA0 = 0;
        lcd_delay();
        lcd_delay();
#endif

        btns[0] = (btns[0]<<1)|(PORTBbits.RB8  ? 0 : 1);
        btns[1] = (btns[1]<<1)|(PORTBbits.RB9  ? 0 : 1);
        btns[2] = (btns[2]<<1)|(PORTBbits.RB10 ? 0 : 1);
        btns[3] = (btns[3]<<1)|(PORTBbits.RB11 ? 0 : 1);

#ifdef REVISIONC
	    LATAbits.LATA1 = 0;
	    LATAbits.LATA0 = 1;
		delay(LCD_DELAY*10);
#else
        TRISAbits.TRISA0 = 1;
        TRISAbits.TRISA1 = 0;
#endif
        lcd_delay();
        lcd_delay();

        btns[4] = (btns[4]<<1)|(PORTBbits.RB8  ? 0 : 1);
        btns[5] = (btns[5]<<1)|(PORTBbits.RB9  ? 0 : 1);
        btns[6] = (btns[6]<<1)|(PORTBbits.RB10 ? 0 : 1);
        btns[7] = (btns[7]<<1)|(PORTBbits.RB11 ? 0 : 1);

#ifdef REVISIONC
	    LATAbits.LATA0 = 0;
#else
        TRISAbits.TRISA1 = 1;
#endif

        CNPU1bits.CN15PUE = 0;
        CNPU2bits.CN16PUE = 0;
        CNPU2bits.CN21PUE = 0;
        CNPU2bits.CN22PUE = 0;

        for( i = 0; i < 8; ++i ) {
            unsigned char pressed = btns[i] == 0xFFFF;
            if( pressed != ((buttons_pressed>>i)&1) ) {
                key_repeat = 0;
                if( pressed ) {
                    if( i == KEY_LMUTE && (buttons_pressed&(1<<KEY_RMUTE)) ) {
                        if( !(buttons_canceled & (1<<KEY_RMUTE)) ) {
                            buttons_canceled |=  (1<<KEY_RMUTE);
                            handle_key(KEY_RMUTE);
                        }
                        handle_key(KEY_CHANSWAP);
                    } else if( i == KEY_RMUTE && (buttons_pressed&(1<<KEY_LMUTE)) ) {
                        if( !(buttons_canceled & (1<<KEY_LMUTE)) ) {
                            buttons_canceled |=  (1<<KEY_LMUTE);
                            handle_key(KEY_LMUTE);
                        }
                        handle_key(KEY_CHANSWAP);
                    } else {
                        handle_key(i);
                    }
                    update_lcd();
                } else {
                    buttons_canceled &= ~(1<<i);
                }
                buttons_pressed ^= 1<<i;
                break;
            }
        }
        if( buttons_pressed == (1<<KEY_LEFT) || buttons_pressed == (1<<KEY_RIGHT) || buttons_pressed == (1<<KEY_UP) || buttons_pressed == (1<<KEY_DOWN) ) {
            ++key_repeat;
            if( !(key_repeat&127) ) {
                unsigned char temp = key_repeat>>7;
                if( temp == 4 || temp == 8 || temp == 11 || temp == 14 || temp == 16 || temp == 18 || temp == 19 ) {
                    unsigned char key = 0;
                    unsigned char temp2 = buttons_pressed;
                    while( temp2 != 1 ) {
                        ++key;
                        temp2 >>= 1;
                    }
                    handle_key(key);
                    update_lcd();
                    if( temp == 19 ) {
                        key_repeat -= 256;
                    }
                }
            }
        }

        if( PR1 == 0xFFFF ) {
            last_timer &= 0x8000;

            while( last_timer == (TMR1&0x8000) )
                ;
            last_timer ^= 0x8000;
        } else {
            // don't delay plus make this loop go twice as fast since in sleep mode the chip is too slow to run it at the normal rate
            second_timer |= 1;
        }

        ++second_timer;
        if( second_timer == 1220 ) {
            once_per_second();
            four_times_per_second();
            second_timer = 0;
        } else if( second_timer == 1220/4 || second_timer == 1220*2/4 || second_timer == 1220*3/4 ) {
            four_times_per_second();
        }
    }
}
