
#include <string.h>
#include "p33Fxxxx.h"
#include "DEE Emulation 16-bit.h"
#include "Interface.h"
#include "LCD.h"
#include "Wave_Config.h"
#include "main.h"


typedef struct {
    unsigned char line, startx, length, size;
    unsigned short max;
    void* ptr;
} edit_field;


Mode intmode;
unsigned short freq = 1000, freq_l = 1000, freq_r = 1000, freq_start = 20, freq_finish = 20000;
unsigned char  db, db_l, db_r, db_on, db_off = 30;
unsigned short phase_diff = 180;
unsigned short pulse_on = 100, pulse_off = 900;
unsigned short sweep_on = 100, sweep_off = 1;
unsigned char l_mute, r_mute;
unsigned char edit_pos;
unsigned char in_settings;
unsigned char pro_mode, emph_enabled;
unsigned short regulator_cal = 300;
unsigned short low_battery = 4000;
unsigned char sleeping;

SettingsMode settings_mode;
#ifdef DEFAULT_RATE_48
unsigned char samplerate_index = 1;
#else
unsigned char samplerate_index = 2;
#endif
unsigned char wavetype_index;// = 0;
unsigned char lcd_brightness = 25;
unsigned char lcd_contrast = 50;
unsigned char settings_bank;// = 0;
unsigned char sweep_index;// = 0;
unsigned char settings_load, settings_save;
unsigned char settings_loaded, settings_saved;

static unsigned short battery_readings[16];
static unsigned char cur_battery_reading, num_battery_readings;
static unsigned char low_bat_warn, flat_bat_warn;

#ifdef DEFAULT_RATE_48
static unsigned char last_samplerate_index = 1;
#else
static unsigned char last_samplerate_index = 2;
#endif



static const char* mode_names[] = {
    "Locked",
    "Indep.",
    "Mixed ",
    "Pulsed",
    "Sweep "
};

static const char const * samplerate_choices[] = {
    " 44100Hz", " 48000Hz", " 96000Hz", "Analogue"
};
static const unsigned char samplerates[] = { 44, 48, 96, 96 };
static const unsigned short max_freqs[4] = { 22050, 24000, 48000, 48000 };
static const char const * wavetype_choices[] = {
    "      Sine", "    Square", "  Triangle", "SawtoothUp", "SawtoothDn"
};
static const char const * sweep_choices[] = {
    "Exponential", "     Linear"
};
static const char const * conpro_choices[] = {
    "Consumer", " AES/EBU"
};
static const char const * emph_choices[] = {
    " Normal", "PreEmph"
};

#ifdef DEFAULT_RATE_48
#define DEFAULT_MAX_FREQ 24000
#else
#define DEFAULT_MAX_FREQ 48000
#endif

static edit_field locked_fields[] = {
    { 0, 0, 5, 2, DEFAULT_MAX_FREQ, &freq },
    { 0, 12, 2, 1, 99, &db_l },
    { 1, 0, 3, 2, 360, &phase_diff },
    { 1, 12, 2, 1, 99, &db_r },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field independent_fields[] = {
    { 0, 3, 5, 2, DEFAULT_MAX_FREQ, &freq_l },
    { 0, 12, 2, 1, 99, &db_l },
    { 1, 3, 5, 2, DEFAULT_MAX_FREQ, &freq_r },
    { 1, 12, 2, 1, 99, &db_r },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field pulse_fields[] = {
    { 0, 3, 5, 2, DEFAULT_MAX_FREQ, &freq },
    { 0, 12, 2, 1, 99, &db_on },
    { 1, 0, 4, 2, 9999, &pulse_on },
    { 1, 5, 3, 2, 999, &pulse_off },
    { 1, 12, 2, 1, 99, &db_off },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field sweep_fields[] = {
    { 0, 0, 5, 2, DEFAULT_MAX_FREQ, &freq_start },
    { 0, 12, 2, 1, 99, &db },
    { 1, 0, 5, 2, DEFAULT_MAX_FREQ, &freq_finish },
    { 1, 8, 3, 2, 999, &sweep_on },
    { 1, 13, 2, 2, 99, &sweep_off },
    { 2, 0, 0, 0, 0, 0 }
};


static edit_field config_rate[] = {
    { 0, 15, 1, 1, 3, &samplerate_index },
    { 1, 15, 1, 1, 4, &wavetype_index },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field config_lcd[] = {
    { 0, 12, 3, 1, 100, &lcd_brightness },
    { 1, 12, 3, 1, 100, &lcd_contrast },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field config_sweep[] = {
    { 1, 15, 1, 1, 1, &sweep_index },
};

static edit_field config_bits[] = {
    { 0, 7, 1, 1, 1, &pro_mode },
    { 0, 15, 1, 1, 1, &emph_enabled },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field config_batt[] = {
    { 0, 12, 3, 2, 999, &regulator_cal },
    { 1, 10, 4, 2, 9999, &low_battery },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field config_bank[] = {
    { 0, 15, 1, 1, 9, &settings_bank },
    { 1,  5, 1, 1, 0, &settings_load },
    { 1, 13, 1, 1, 0, &settings_save },
    { 2, 0, 0, 0, 0, 0 }
};

static edit_field config_status[] = {
    { 2, 0, 0, 0, 0, 0 }
};


static edit_field* edit_fields[2][7] = {
    { locked_fields, independent_fields, independent_fields, pulse_fields, sweep_fields, 0, 0 },
    { config_rate, config_status, config_sweep, config_bits, config_batt, config_lcd, config_bank }
};



#define SETTING_SIZE 36
// data size: 12*2 + 11 + 1 = 36 bytes
static void save_settings(unsigned short* tohere) {
    unsigned char* dest;
    tohere[0]  = freq;             ++tohere;
    tohere[0]  = freq_l;           ++tohere;
    tohere[0]  = freq_r;           ++tohere;
    tohere[0]  = freq_start;       ++tohere;
    tohere[0]  = freq_finish;      ++tohere;
    tohere[0]  = phase_diff;       ++tohere;
    tohere[0]  = pulse_on;         ++tohere;
    tohere[0]  = pulse_off;        ++tohere;
    tohere[0]  = sweep_on;         ++tohere;
    tohere[0]  = sweep_off;        ++tohere;
    tohere[0]  = regulator_cal;    ++tohere;
    tohere[0]  = low_battery;      ++tohere;
    dest = (unsigned char*)tohere;
    dest[0]   = intmode;          ++dest;
    dest[0]   = db;               ++dest;
    dest[0]   = db_l|(l_mute<<7); ++dest;
    dest[0]   = db_r|(r_mute<<7); ++dest;
    dest[0]   = db_on;            ++dest;
    dest[0]   = db_off;           ++dest;
    dest[0]   = samplerate_index; ++dest;
    dest[0]   = wavetype_index;   ++dest;
    dest[0]   = lcd_brightness;   ++dest;
    dest[0]   = lcd_contrast;     ++dest;
    dest[0]   = sweep_index|(pro_mode?64:0)|(emph_enabled?128:0); ++dest;
    dest[0]   = 0x5e;//           ++dest;
}

static void load_settings(const unsigned short* fromhere) {
    const unsigned char* source;
    freq = fromhere[0];             ++fromhere;
    freq_l = fromhere[0];           ++fromhere;
    freq_r = fromhere[0];           ++fromhere;
    freq_start = fromhere[0];       ++fromhere;
    freq_finish = fromhere[0];      ++fromhere;
    phase_diff = fromhere[0];       ++fromhere;
    pulse_on = fromhere[0];         ++fromhere;
    pulse_off = fromhere[0];        ++fromhere;
    sweep_on = fromhere[0];         ++fromhere;
    sweep_off = fromhere[0];        ++fromhere;
    regulator_cal = fromhere[0];    ++fromhere;
    low_battery = fromhere[0];      ++fromhere;
    source = (const unsigned char*)fromhere;
    intmode = source[0];            ++source;
    db = source[0];                 ++source;
    db_l = source[0];               ++source;
    db_r = source[0];               ++source;
    db_on = source[0];              ++source;
    db_off = source[0];             ++source;
    samplerate_index = source[0];   ++source;
    wavetype_index = source[0];     ++source;
    lcd_brightness = source[0];     ++source;
    lcd_contrast = source[0];       ++source;
    pro_mode = (source[0]&64) != 0;
    emph_enabled = (source[0]&128) != 0;
    sweep_index = source[0]&63;//   ++source;

    l_mute = db_l >> 7;
    db_l &= 127;
    r_mute = db_r >> 7;
    db_r &= 127;
}


static void get_eeprom_data(unsigned char which, unsigned short* tohere) {
    unsigned char i, end;
    i = which*(SETTING_SIZE/2);
    end = i + (SETTING_SIZE/2);
    while( i < end ) {
        tohere[0] = DataEERead(i);
        ++tohere;
        ++i;
    }
}
static void set_eeprom_data(unsigned char which, const unsigned short* fromhere) {
    unsigned char i, end;
    i = which*(SETTING_SIZE/2);
    end = i + (SETTING_SIZE/2);
    while( i < end ) {
        DataEEWrite(fromhere[0], i);
        ++fromhere;
        ++i;
    }
}
void load_settings_bank(unsigned char bank) {
    unsigned short data[(SETTING_SIZE/2)];
    get_eeprom_data(bank, data);
    // this check lets us avoid loading settings that are not valid (hopefully)
    if( ((unsigned char*)data)[SETTING_SIZE-1] == 0x5E ) {
        load_settings(data);
    }
}
void save_settings_bank(unsigned char bank) {
    unsigned short data[(SETTING_SIZE/2)];
    save_settings(data);
    set_eeprom_data(bank, data);
}




static void set_cursor(const edit_field* pfield, unsigned char pos) {
    if( pfield->ptr == &sweep_on && pos == 2 ) {
        pos = 3;
    } else if( pfield->ptr == &low_battery && pos > 0 ) {
        ++pos;
    }
    set_lcd_cursor(pfield->line, pfield->startx+pos, pfield->line < 2);
}

static void print_num_int(char* tohere, unsigned short num, unsigned char digits, char prefill) {
    char* dest = tohere + digits - 1;
    dest[0] = '0' + (num%10);
    while( --dest >= tohere ) {
        num /= 10;
        if( num > 0 ) {
            dest[0] = '0' + (num%10);
        } else {
            dest[0] = prefill;
        }
    }
}
static void print_num(char* tohere, unsigned short num, unsigned char digits) {
    print_num_int(tohere, num, digits, ' ');
}
static void print_num_leading0s(char* tohere, unsigned short num, unsigned char digits) {
    print_num_int(tohere, num, digits, '0');
}

static void print_db(char* tohere, unsigned short num, unsigned char digits) {
    if( num == 99 ) {
        //'-99dB' => ' off '
        memcpy(tohere-1, " off ", 5);
    } else {
        print_num(tohere, num, digits);
    }
}

static void set_new_clock_source(unsigned char source) {
    __builtin_write_OSCCONH(source);
    __builtin_write_OSCCONL(0x01);
}

static void enter_sleep_mode() {
    DAC1CONbits.DACEN = 0;   // disable DAC if it's active
    DCICON1bits.DCIEN = 0;   // disable DCI if it's active

    set_new_clock_source(6); // switch clock source to FRCDIV16

    PR1 = 0x1FF;
}
static void leave_sleep_mode() {
    set_new_clock_source(3); // switch clock source to PRIPLL

    PR1 = 0xFFFF;

    if( samplerate_index != 3 ) {
      DCICON1bits.DCIEN = 1; // re-enable DCI
    }
    if( samplerate_index == 3 || samplerates[samplerate_index] == 48 ) {
      DAC1CONbits.DACEN = 1; // re-enable DAC
    }
}


static unsigned short get_battery_voltage() {
    unsigned long total = 0;
    unsigned short ret;
    unsigned char i;
    if( !buttons_pressed ) {
        unsigned short result;
        AD1CON1bits.SAMP = 1;
        while( !AD1CON1bits.DONE )
            ;
        result = ADC1BUF0;
        // 0 .. 4095 -> 0 .. 13.2V (nom.)
        // result returned is in mV
        battery_readings[cur_battery_reading] = ((unsigned long)result * (unsigned long)(3000+regulator_cal))>>10;
        cur_battery_reading = (cur_battery_reading+1)&15;
        if( num_battery_readings < 16 ) {
            ++num_battery_readings;
        }
    }
    for( i = 0; i < num_battery_readings; ++i ) {
        total += battery_readings[i];
    }
    if( num_battery_readings ) {
        unsigned short flat_battery = low_battery - 200;

        ret = total / num_battery_readings;
        if( ret < low_battery ) {
            if( low_bat_warn < 32 ) {
                ++low_bat_warn;
            }
        } else {
            if( low_bat_warn > 0 ) {
                --low_bat_warn;
            }
        }

        if( flat_battery > 3500 ) {
            flat_battery = 3500;
        }
        if( ret < flat_battery ) {
            if( flat_bat_warn < 76 ) {
                ++flat_bat_warn;
            }
        } else if( ret > flat_battery + 200 ) {
            if( flat_bat_warn > 0 ) {
                --flat_bat_warn;
            }
        }
    } else {
        ret = 0;
    }
    return ret;
}

static const char* init_lcd[2][7][2] = {
    {
        { "xxxxxHz    -xxdB", "xxx\xdf       -xxdB" },
        { "L: xxxxxHz -xxdB", "R: xxxxxHz -xxdB" },
        { "A: xxxxxHz -xxdB", "B: xxxxxHz -xxdB" },
        { "   xxxxxHz -xxdB", "xxxx/xxxms -xxdB" },
        { "xxxxxHz    -xxdB", "xxxxxHz xx.x/xxs" }
    }, {
        { "Rate:    xxxxxHz", "Wave: " },
        { "Status:   ",       "        LR      " },
        { "Sweep:          ", "     " },
        { "",                 "Battery: " },
        { "3.3V Cal: 3.xxxV", "Low Bat.: x.xxxV" },
        { "Brightness: xxx%", "Contrast:   xxx%" },
        { "Bank:          x", "  Load    Save  " }
    }
};

static void display_battery_voltage(char* here) {
    unsigned short battery_mV = get_battery_voltage();
    print_num(here, battery_mV/1000, 2);
    here[2] = '.';
    print_num_leading0s(here+3, battery_mV%1000, 3);
    here[6] = 'V';
}

void update_lcd() {
    unsigned char pos;
    char buf0[17], buf1[17];
    const edit_field* pfield;

    if( sleeping ) {
        strcpy(buf0, "  Battery Flat! ");
        memset(buf1, ' ', 16);
        display_battery_voltage(buf1+4);
        pos = 0;
        pfield = config_status;
    } else {
        strcpy(buf0, init_lcd[in_settings][in_settings ? settings_mode : intmode][0]);
        strcpy(buf1, init_lcd[in_settings][in_settings ? settings_mode : intmode][1]);

        pos = edit_pos;
        pfield = edit_fields[in_settings][in_settings ? settings_mode : intmode];
        while( pos >= pfield->length && pfield->ptr ) {
            pos -= pfield->length;
            ++pfield;
        }

        if( in_settings ) {
            switch(settings_mode) {
            case Settings_Rate:
                strcpy(buf0+8, samplerate_choices[samplerate_index]);
                strcpy(buf1+6, wavetype_choices[wavetype_index]);
                break;
            case Settings_Status:
                strcpy(buf0+10, mode_names[intmode]);
                if( intmode == Mode_Sweep && sweep_is_paused ) {
                    strcpy(buf0, "Paused");
                }
                if( intmode == Mode_Pulse || intmode == Mode_Sweep ) {
                    unsigned short freq = sine_freq_l;
                    switch(sample_rate) {
                    default:
//		    case 96:
                        break;
                    case 48:
                        freq >>= 1;
                        break;
                    case 44:
                        freq = ((unsigned long)freq*882+1)/1920;
                        break;
                    }
                    strcpy(buf1+5, "Hz LR -xxdB");
                    print_num(buf1, freq, 5);
                    print_num(buf1+12, intmode == Mode_Sweep ? (sample_counter < sweep_on_samples ? db : 99) : (sine_scale_l == pulse_on_scale ? db_on : db_off), 2);
                    break;
                }
                if( l_mute ) {
                    buf1[8] = 'l';
                }
                if( r_mute ) {
                    buf1[9] = 'r';
                }
                break;
            case Settings_Sweep:
                strcpy(buf1+5, sweep_choices[sweep_index]);
                break;
            case Settings_Bits:
                {
                    strcpy(buf0+0, conpro_choices[pro_mode]);
                    buf0[8] = ' ';
                    strcpy(buf0+9, emph_choices[emph_enabled]);
                    display_battery_voltage(buf1+9);
                    break;
                }
            case Settings_Battery:
                print_num_leading0s(buf0+12, regulator_cal, 3);
                buf1[10] = (low_battery/1000) + '0';
                print_num_leading0s(buf1+12, low_battery%1000, 3);
                break;
            case Settings_LCD:
                print_num(buf0+12, lcd_brightness, 3);
                print_num(buf1+12, lcd_contrast, 3);
                break;
            case Settings_Bank:
                buf0[15] = '0' + settings_bank;
                if( settings_loaded ) {
                    buf1[6] = 'e';
                    buf1[7] = 'd';
                }
                if( settings_saved ) {
                    buf1[14] = 'd';
                }
                break;
            }
        } else {
            if( intmode == Mode_Sweep ) {
                print_num(buf0, freq_start, 5);
                print_db (buf0+12, db, 2);
                if( l_mute ) {
                    buf0[9] = 'l';
                }
                if( r_mute ) {
                    buf0[10] = 'r';
                }
                print_num(buf1, freq_finish, 5);
                print_num(buf1+8, sweep_on, 3);
                buf1[11] = buf1[10];
                buf1[10] = '.';
                if( sweep_off == 99 ) {
                    buf1[13] = 'm';
                    buf1[14] = 'a';
                    buf1[15] = 'n';
                } else {
                    print_num(buf1+13, sweep_off, 2);
                }
            } else {
                switch(intmode) {
                case Mode_Locked:
                    print_num(buf0, freq, 5);
                    print_db (buf0+12, db_l, 2);
                    print_num(buf1, phase_diff, 3);
                    print_db (buf1+12, db_r, 2);
                    break;
//              case Mode_Independent:
//              case Mode_Mixed:
                default:
                    print_num(buf0+3, freq_l, 5);
                    print_db (buf0+12, db_l, 2);
                    print_num(buf1+3, freq_r, 5);
                    print_db (buf1+12, db_r, 2);
                    break;
                case Mode_Pulse:
                    print_num(buf0+3, freq, 5);
                    print_db (buf0+12, db_on, 2);
                    print_num(buf1, pulse_on, 4);
                    print_num(buf1+5, pulse_off, 3);
                    print_db (buf1+12, db_off, 2);
                    break;
                }
                if( l_mute ) {
                    buf0[11] = '_';
                }
                if( r_mute ) {
                    buf1[11] = '_';
                }
            }
        }
    }
    update_lcd_line(0, buf0);
    update_lcd_line(1, buf1);
    set_cursor(pfield, pos); 
}

static inline void swapb(unsigned char* a, unsigned char* b) {
    *a ^= *b;
    *b ^= *a;
    *a ^= *b;
}

static inline void swapw(unsigned short* a, unsigned short* b) {
    *a ^= *b;
    *b ^= *a;
    *a ^= *b;
}

static void menu_freq_replace(unsigned short old, unsigned short new) {
    unsigned char i;
    for( i = 0; i < 5; ++i ) {
        edit_field* field = edit_fields[0][i];
        while( field->line != 2 ) {
            if( field->max == old ) {
                field->max = new;
            }
            if( new < old ) {
                if( field->size == 2 ) {
                    if( ((unsigned short*)field->ptr)[0] > new ) {
                        ((unsigned short*)field->ptr)[0] = new;
                    }
                } else {
                    if( ((unsigned char*)field->ptr)[0] > new ) {
                        ((unsigned char*)field->ptr)[0] = new;
                    }
                }
            }
            ++field;
        }
    }
}


void reinit_mode() {
    unsigned char db_l_mute, db_r_mute;

    set_sample_rate(samplerates[samplerate_index], samplerate_index == 3);

    if( samplerate_index != last_samplerate_index ) {
        menu_freq_replace(max_freqs[last_samplerate_index], max_freqs[samplerate_index]);
        last_samplerate_index = samplerate_index;
    }

    mode = Null; // force recomputation
    db_l_mute = l_mute ? 99 : db_l;
    db_r_mute = r_mute ? 99 : db_r;

    switch(intmode) {
    case Mode_Locked:
        set_locked((wavetype_t)wavetype_index, freq, phase_diff, db_l_mute, db_r_mute);
        break;
    case Mode_Independent:
        set_independent((wavetype_t)wavetype_index, freq_l, freq_r, db_l_mute, db_r_mute);
        break;
    case Mode_Mixed:
        set_mixed((wavetype_t)wavetype_index, freq_l, freq_r, db_l, db_r, l_mute, r_mute);
        break;
    case Mode_Pulse:
        set_pulsed((wavetype_t)wavetype_index, freq, pulse_on, pulse_off, db_on, db_off, l_mute, r_mute);
        break;
    case Mode_Sweep:
        set_sweep((wavetype_t)wavetype_index, (sweeptype_t)(sweep_index^1), freq_start, freq_finish, sweep_on, sweep_off, db, l_mute, r_mute);
        break;
    }
}

static const edit_field* get_this_edit_field() {
    return edit_fields[in_settings][in_settings ? settings_mode : intmode];
}
static unsigned char get_edit_field_count() {
    const edit_field* pfield = get_this_edit_field();
    unsigned char total = 0;
    while( pfield->length ) {
        total += pfield->length;
        ++pfield;
    }
    return total;
}

void handle_key(int key) {
    if( key == KEY_LEFT || key == KEY_RIGHT ) {
        if( in_settings && settings_mode == Settings_Status ) {
            if( intmode == Mode_Sweep ) {//&& sweep_off == 99 ) {
                // manual sweep mode
                if( key == KEY_RIGHT ) {
                    trigger_manual_sweep();
                } else {
                    pause_resume_manual_sweep();
                }
            }
        } else {
            unsigned char total = get_edit_field_count();
            if( key == KEY_LEFT ) {
                if( edit_pos == 0 )
                    edit_pos = total;
                else
                    --edit_pos;
            } else {
                if( edit_pos < total )
                    ++edit_pos;
                else
                    edit_pos = 0;
            }
        }
    } else if( key == KEY_UP || key == KEY_DOWN ) {
        const edit_field* pfield = get_this_edit_field();
        unsigned char pos = edit_pos;
        while( pos >= pfield->length && pfield->length ) {
            pos -= pfield->length;
            ++pfield;
        }
        if( pfield->length ) {
            pos = pfield->length-pos-1;
            if( pfield->size == 2 ) {
                unsigned short ldelta = 1;
                while( pos ) {
                    ldelta *= 10;
                    --pos;
                }
                if( key == KEY_DOWN ) {
                    if( ((unsigned short*)pfield->ptr)[0] < ldelta ) {
                        if( ((unsigned short*)pfield->ptr)[0] == 0 ) {
                            // avoid doing any reinitialisation etc.
                            return;
                        }
                        ((unsigned short*)pfield->ptr)[0] = 0;
                    } else {
                        ((unsigned short*)pfield->ptr)[0] -= ldelta;
                    }
                } else {
                    if( ((unsigned short*)pfield->ptr)[0] == pfield->max ) {
                        // avoid doing any reinitialisation etc.
                        return;
                    }
                    ((unsigned short*)pfield->ptr)[0] += ldelta;
                    if( ((unsigned short*)pfield->ptr)[0] > pfield->max ) {
                        ((unsigned short*)pfield->ptr)[0] = pfield->max;
                    }
                }
            } else {
                unsigned char ldelta = 1;
                while( pos ) {
                    ldelta *= 10;
                    --pos;
                }
                if( key == KEY_DOWN ) {
                    if( ((unsigned char*)pfield->ptr)[0] < ldelta ) {
                        if( ((unsigned char*)pfield->ptr)[0] == 0 ) {
                            // avoid doing any reinitialisation etc.
                            return;
                        }
                        ((unsigned char*)pfield->ptr)[0] = 0;
                    } else {
                        ((unsigned char*)pfield->ptr)[0] -= ldelta;
                    }
                } else {
                    if( ((unsigned char*)pfield->ptr)[0] == pfield->max && pfield->max != 0 ) {
                        // avoid doing any reinitialisation etc.
                        return;
                    }
                    ((unsigned char*)pfield->ptr)[0] += ldelta;
                    if( ((unsigned char*)pfield->ptr)[0] > pfield->max ) {
                        ((unsigned char*)pfield->ptr)[0] = pfield->max;
                    }
                }
                if( pfield->ptr == &settings_load ) {
                    unsigned char old_pro = pro_mode, old_emph = emph_enabled;
                    load_settings_bank(settings_bank);
                    reinit_mode();
                    if( pro_mode != old_pro || emph_enabled != old_emph ) {
                        compute_channel_status(pro_mode, emph_enabled);
                    }
                    settings_loaded = 1;
                } else if( pfield->ptr == &settings_save ) {
                    save_settings_bank(settings_bank);
                    settings_saved = 1;
                } else if( pfield->ptr == &settings_bank ) {
                    settings_loaded = settings_saved = 0;
                }
            }
        } else {
            if( in_settings ) {
                settings_loaded = settings_saved = 0;
                if( key == KEY_DOWN && settings_mode == Settings_Bank ) {
                    settings_mode = Settings_Rate;
                } else if( key == KEY_UP && settings_mode == Settings_Rate ) {
                    settings_mode = Settings_Bank;
                } else {
                    settings_mode += (key == KEY_DOWN ? 1 : -1);
                }
            } else {
                if( key == KEY_DOWN && intmode == Mode_Sweep ) {
                    intmode = Mode_Locked;
                } else if( key == KEY_UP && intmode == Mode_Locked ) {
                    intmode = Mode_Sweep;
                } else {
                    intmode += (key == KEY_DOWN ? 1 : -1);
                }
            }
            hide_cursor();
        }

        if( in_settings ) {
            if( (settings_mode == Settings_Rate || settings_mode == Settings_Sweep) && pfield->length ) {
                reinit_mode();
            } else if( settings_mode == Settings_LCD && pfield->length ) {
#ifdef REVISIONB
                OC1RS = lcd_brightness*2;
                OC2RS = lcd_contrast*4;
#else
                OC1RS = lcd_brightness;
                OC2RS = lcd_contrast;
#endif
            } else if( settings_mode == Settings_Bits && pfield->length ) {
                compute_channel_status(pro_mode, emph_enabled);
            }
        } else {
            reinit_mode();
        }
    } else if( key == KEY_MIDDLE ) {
        unsigned char total = get_edit_field_count();
        if( edit_pos != total ) {
            edit_pos = total;
        } else {
            in_settings ^= 1;
            settings_loaded = settings_saved = 0;
            hide_cursor();
        }
    } else if( key == KEY_LMUTE ) {
        l_mute ^= 1;
        reinit_mode();
    } else if( key == KEY_RMUTE ) {
        r_mute ^= 1;
        reinit_mode();
    } else if( key == KEY_CHANSWAP ) {
        switch(intmode) {
        case Mode_Locked:
            swapb(&db_l, &db_r);
            swapb(&l_mute, &r_mute);
			phase_diff = 360-phase_diff;
            break;
        case Mode_Independent:
        case Mode_Mixed:
            swapb(&db_l, &db_r);
            swapb(&l_mute, &r_mute);
            swapw(&freq_l, &freq_r);
            break;
        case Mode_Pulse:
        case Mode_Sweep:
            swapb(&l_mute, &r_mute);
            break;
        }
        reinit_mode();
    }
}

void hide_cursor() {
    edit_pos = get_edit_field_count();
}

void once_per_second() {
    if( in_settings && settings_mode == Settings_Bits ) {
        update_lcd(); // update battery voltage reading once per second
    } else {
        get_battery_voltage(); // continue reading battery voltage for long term averaging
    }

    if( low_bat_warn > 16 ) {
#ifdef REVISIONB
        unsigned char level = flat_bat_warn > 60 ? 10 : 20;
        OC1RS = OC1RS == level ? 0 : level;
#else
        OC1RS = OC1RS == 10 ? 0 : 10;
#endif
    } else {
#ifdef REVISIONB
        OC1RS = lcd_brightness*2;
#else
        OC1RS = lcd_brightness;
#endif
    }

    if( flat_bat_warn > 60 && !sleeping ) {
        enter_sleep_mode();
        sleeping = 1;
        update_lcd();
    } else if( flat_bat_warn == 0 && sleeping ) {
        leave_sleep_mode();
        sleeping = 0;
        update_lcd();
    }

    if( intmode == Sweep && sweep_off == 99 ) {
        // manual sweep mode
        if( (sample_counter>>16) > (sweep_on_samples>>16)+1 ) {
            sample_counter -= 0x10000L;
        }
    }
}
void four_times_per_second() {
    if( (in_settings && settings_mode == Settings_Status) || sleeping ) {
        update_lcd(); // keep status up to date
    }
}
