;	16bDigitalPot.asm -- Control program for Silicon Chip 16-bit Digital
;	Potentiometer, based on PIC16F877A micro running at 4.00MHz
;	(so Fosc/4 = 1MHz, 1 machine cycle = 1us)
;	Written by Jim Rowe. Last revised 4/06/2010

; ***
	ERRORLEVEL -302
	ERRORLEVEL -306
; ***

	list P=16F877A
	#include "p16f877A.inc"
	__config h'3F3A'	; HS osc selected, WDT disabled, PWRT enabled,
						; BOR disabled, RB3 is digital I/O, code prot off,
						; Flash prog mem write protection off, RB6 & RB7 are
						; general purpose I/O pins (not used for debugging)

;**********************************************************
;
;	First define some handy macros
;
;	Select Register Bank 0
bank0	macro
	errorlevel	+302		; Re-enable bank warning
	BCF		STATUS,RP0		; Select Bank 0
	BCF STATUS, RP1
	BCF STATUS, IRP			; and also indir addr banks 0,1
	endm

;	Select Register Bank 1
bank1	macro
	BSF		STATUS,RP0		; Select Bank 1
	errorlevel	-302		; disable warning
	endm

;	select Register Bank 2
bank2	macro
	BCF	STATUS, RP0			; Select Bank 2
	BSF STATUS, RP1
	BSF STATUS, IRP			; also indir addr banks 2,3
	endm

;	select Register Bank 3
bank3	macro
	BSF	STATUS, RP0			; Select Bank 3
	BSF STATUS, RP1
	BSF STATUS, IRP			; also indir addr banks 2,3
	endm
;
;	**********************************************************

;	Swap bytes in register file via W
swap	macro	this,that
	MOVF	this,w		; get this
	XORWF	that,f		; Swap using Microchip
	XORWF	that,w		; Tips'n Tricks
	XORWF	that,f		; #18
	MOVWF	this
	endm

;	Copy bytes in register file via W
copy	macro	from,to
	MOVF	from,W
	MOVWF	to
	endm

;	Prepare to call or goto page1 (800-FFFh)
Ppage1	macro
	BCF PCLATH,4	; clear bit4 of PCLATH
	BSF PCLATH,3	; but set bit3
	endm

;	Prepare to call or goto page0 (000-7FFh)
Ppage0	macro
	BCF PCLATH,4	; clear bit4 of PCLATH
	BCF PCLATH,3	; and also clear bit3
	endm

;	**********************************************************
;     STATUS bit definitions (used in FPRF24.TXT)

#define	_C	STATUS,0
#define	_Z	STATUS,2
;
;	**********************************************************

;  storage for counters & other variables used by program, in data memory
Counter1	equ	h'20'	; gp counter variable 1
Counter2	equ h'21'	; gp counter variable 2
Counter3	equ h'22'	; gp counter variable 3
Counter4	equ h'23'	; gp counter variable 4

Temp1	equ		h'24'	; working storage 1
Temp2	equ		h'25'	; working storage 2
Temp3	equ		h'26'	; working storage 3

Keycode		equ	h'27'	; storage for keypressed code

OutFlag		equ h'28'	; flag for current output status
						; b0 = 0 (00h) for output off,
						; b0 = 1 (01h) for output on

CurrInH		equ h'29'	; current Vin level (16-bit Fixed), high byte
CurrInL		equ h'2A'	; current Vin level, low byte

CurrOutH	equ h'2B'	; current Vout level, high byte
CurrOutL	equ h'2C'	; current Vout level, low byte

InLoopCtr	equ h'2D'	; keypad numeric entry loop counter
HbyteWkg	equ h'2E'	; working register for high byte of 16b number
LbyteWkg	equ h'2F'	; working register for low byte of 16b number

;	registers for saving Vin and Vout display digits, numeric entry digits
InDig1		equ h'30'
InDig2		equ h'31'
InDig3		equ h'32'
InDig4		equ h'33'
InDig5		equ h'34'
InDig6		equ h'35'

OutDig1		equ h'36'
OutDig2		equ h'37'
OutDig3		equ h'38'
OutDig4		equ h'39'
OutDig5		equ h'3A'
OutDig6		equ h'3B'

EntryDig1	equ h'40'
EntryDig2	equ h'41'
EntryDig3	equ h'42'
EntryDig4	equ h'43'
EntryDig5	equ h'44'
EntryDig6	equ h'45'

;	storage for current Vout value in 24-bit form, after calculation
VoutEXP		equ h'46'	; exponent
VoutB0		equ h'47'	; MSB & sign bit
VoutB1		equ h'48'	; LSB

;	storage for current Vin value in 24-bit form, after calculation
VinEXP		equ h'49'	; exponent
VinB0		equ h'4A'	; MSB & sign bit
VinB1		equ h'4B'	; and LSB

;	***********************************************************
;   Floating Point Stack & other locations used by FPRF24.TXT
;	routines
;
AARGB7	equ h'50'	; AARGB7 byte for FP argument A
AARGB6	equ h'51'	; AARGB6 byte
AARGB5	equ h'52'	; AARGB5 byte
AARGB4	equ	h'53'	; AARGB4 byte
AARGB3	equ h'54'	; AARGB3
AARGB2	equ h'55'	; AARGB2
AARGB1	equ h'56'	; AARGB1
AARGB0	equ h'57'	; AARGB0
AEXP	equ h'58'	; 8 bit biased exponent for argument A

BARGB3	equ h'59'	; BARGB3 byte for argument B
BARGB2	equ h'5A'	; BARGB2
BARGB1	equ h'5B'	; BARGB1
BARGB0	equ h'5C'	; BARGB0
BEXP	equ h'5D'	; 8 bit biased exponent for argument B

TEMPB3	equ h'5E'	; TEMPB3 byte
TEMPB2	equ h'5F'	; TEMPB2 byte
TEMPB1	equ h'60'	; TEMPB1 byte
TEMPB0	equ h'61'	; TEMPB0 byte

CARGB1	equ h'62'	; CARGB1 byte for argument C
CARGB0	equ h'63'	; CARGB0 byte
CEXP	equ h'64'	; 8 bit biased exponent for argument C

DARGB3	equ h'65'	; DARGB3 byte for argument D
DARGB2	equ h'66'	; DARGB2 byte
DARGB1	equ h'67'	; DARGB1 byte
DARGB0	equ	h'68'	; DARGB0 byte
DEXP	equ	h'69'	; 8-bit biased exponent for argument D

EARGB3	equ h'6A'	; needed by EXP1024, it seems

SIGN	equ	h'6B'	; location for saving sign in MSB
FPFLAGS	equ h'6C'	; floating point library exception flags
FPError	equ h'6D'	; floating point routine error code (FFh = error)

;	Fixed point storage locations

REMB3	equ h'70'	; remainder LSB
REMB2	equ h'71'	; remainder middle byte
REMB1	equ h'72'	; remainder upper middle byte
REMB0	equ h'73'	; remainder MSB
LOOPCOUNT	equ h'74'	; loop counter

;	Locations used by float_ascii to store result digits

	cblock h'75'
	ones			; where units digit is stored (75h)
	tenths			; where tenths digit is stored (76h)
	hundredths		; where hundredths digit is stored (77h)
	thousandths		; where thousandths digit is stored (78h)
	tenthous		; where ten thousandths digit is stored (79h)
	endc

digit_count	equ	h'7A'	; digit counter used by float_ascii 

;       floating point library exception flag bits
;
IOV     equ     0	; bit0 = integer overflow flag
FOV     equ     1   ; bit1 = FP overflow flag
FUN     equ     2   ; bit2 = FP underflow flag
FDZ     equ     3   ; bit3 = FP divide by zero flag
NAN		equ		4	; bit4 = not-a-number exception flag
DOM		equ		5	; bit5 = domain error exception flag
RND     equ     6   ; bit6 = FP rounding flag
					; 0 = truncation
                    ; 1 = unbiased rounding to nearest LSB
SAT     equ     7   ; bit7 = FP saturate flag
					; 0 = term on exception w/out saturation
					; 1 = term on exception with saturation
					; to appropriate value

EXP		equ	AEXP 
TEMP	equ	TEMPB0

;	define assembler constants
B0		equ	0
B1		equ	1
B2		equ	2
B3		equ	3
B4		equ	4
B5		equ	5
B6		equ	6
B7		equ	7

MSB		equ	7
LSB		equ	0

;       Floating point literal constants
;
EXPBIAS   equ h'7F'		; = 127d, bias for exponents

; 	*************************************************************

; now program begins

	ORG		h'00'		; normal start vector
	GOTO Initialise
	ORG		h'04'		; interrupt service vector
	GOTO Initialise		; (interrupts not used here)

Initialise:
; first we initialise the PIC itself
	bank0
	CLRF PORTA			; clear PortA
	CLRF PORTB			; and PortB
	CLRF PORTC			; and PortC
	CLRF PORTD			; and PortD
	CLRF PORTE			; and PortE
	CLRF INTCON			; disable all interrupts also
	CLRF RCSTA			; also clear RCSTA register to disable UART
	CLRF T1CON			; and clear T1 timer control register
	CLRF TMR1H			; also its H & L registers
	CLRF TMR1L 
	BCF ADCON0,0		; and shut down A/D module
	bank1
	MOVLW h'07'			; disable the comparators
	MOVWF CMCON
	MOVLW h'06'			; also the analog inputs, so RA0-3 can
	MOVWF ADCON1		; work as digital I/O pins
	CLRF CVRCON			; and also the comp voltage reference
	CLRF PIE1			; clear all interrupt mask bits
	MOVLW h'0F'			; now set RA0-3 as inputs, RA4-5 as outputs
	MOVWF TRISA
	CLRF TRISB			; then set RB0-7 as outputs
	CLRF TRISC			; do the same for RC0-7
	CLRF TRISD			; and for RD0 - RD7 as well
	CLRF TRISE			; also disable PSP mode & set RE0-2 as outputs
	bank0
	CLRF PIR1			; where we clear all periph int flags
	CALL DispInit		; now initialise LCD module
	CALL SetDefaults	; set default initial conditions
	CALL Display1		; then give initial greeting screen
	CALL Wait3Sec		; pause 3 seconds for user to read
	CALL Display2		; then show operating status screen

LoopStart:
;	we begin main loop by scanning keyswitches, to see if one
;	has been pressed
	CALL DoAScan		; scan keys, then
	MOVLW "A"			; check if it was "A"
	XORWF Keycode,0		; W will now have 0 if it was a match...
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was an A, and W -> 0)
	GOTO $+3			; Z=0 (because it wasn't an A), so try again
	CALL SetNuOutput	; it was an A, so go enter new output level
	GOTO MoveOn			; then move on (display & loop back)
	MOVLW "C"			; not S4, so check if it was S12 ("C")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a C)
	GOTO $+3			; Z=0, so keep looking
	CALL OutputTogl		; it was a C, so go toggle output
	GOTO MoveOn			; and move on
	MOVLW "D"			; not S12, so check if it was S16 ("D")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a D)
	GOTO MoveOn	
	CALL SetNuInput		; it was a D; so go enter new input volts
MoveOn:
	CALL Display2		; something has changed, so display status
	CALL Pause200ms		; pause for about 200ms
	GOTO LoopStart		; then back to start of loop
	
;	end of main loop -- subroutines follow

;**************************************************************
;
BounceWait:
;	routine to delay approx 50ms before returning
	MOVLW h'C8'			; set Counter2 for 200d outer loops
	MOVWF Counter2
	CALL Delay250us		; then wait 250us
	DECFSZ Counter2, F	; then decrement Counter2 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return
;
;	***********************************************************
;
Check4FPE:
	; routine to check if floating point functions had errors
	IORLW h'00'			; check if w came back with 00h or FFh
	BTFSC STATUS,Z		; if Z=0, must have been FFh (= an error)
	RETURN				; if Z=1, must have been 00h (= no error)
	MOVF FPFLAGS,0		; was an error, so fetch FP error flags in w
	ADDLW h'30'			; add 30h to turn into ASCII char
	MOVWF Keycode		; then save in Keycode
	CALL ErrorDisp		; so it will appear in error message
	RETURN				; and then return to resume anyway

;	***********************************************************
;
ClearEntryDigs:
;	routine to clear the entry digits, prior to entering new input
;	or output voltage settings
	MOVLW h'20'
	MOVWF EntryDig1
	MOVWF EntryDig2
	MOVWF EntryDig3
	MOVWF EntryDig4
	MOVWF EntryDig5
	MOVWF EntryDig6
	RETURN				; return when all six are cleared
;
;	***********************************************************
;
ClearLCD:
	;routine to clear LCD and reset address counter
	MOVLW h'01'			; clear display & reset addr ptr
	CALL DispAddress
	CALL Delay160ms		; pause 160ms to give it time to clear
	CALL Delay160ms		; and again, just for sloooow LCDs
	RETURN				; then return
;
;	***********************************************************	

Delay250us:
;   routine to delay approx 250us before return
;	delay is 4mc for entry & setup, 4mc for exit & return
;	plus 4mc for each loop. We use 60 loops, so total delay
;	is (4 x 60)mc + 8mc = 248mc = 248us
	MOVLW h'3C'			; set Counter1 for 60 loops
	MOVWF Counter1
	NOP					; 1mc delay for timing
	DECFSZ Counter1, F	; decrement Ctr1 & skip when zero
	GOTO $-2			; didn't hit zero, so loop back
	RETURN				; reached zero, so return

Delay10ms:
;	routine to delay approx 10ms before returning
	MOVLW h'28'			; set Counter2 for 40d outer loops
	MOVWF Counter2
	CALL Delay250us		; then wait 250us
	DECFSZ Counter2, F	; then decrement Counter2 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return

Delay160ms:
;	routine to delay approx 160ms before returning
	MOVLW h'10'			; set counter3 for 16 loops
	MOVWF Counter3
	CALL Delay10ms		; then wait 10ms
	DECFSZ Counter3,F	; decrement Counter3 & skip when zero
	GOTO $-2			; otherwise keep looping
	RETURN				; reached zero, so return

;	***********************************************************

DispAddress:
	;routine to translate & load display address (in w reg) into LCD
	BCF PORTC,6			; first set RS pin of LCD low, for instr/addr
	CALL Nibbles2LCD	; then send addr/cmd nibbles to LCD
	BCF PORTC,6			; make sure RS is is left low
	GOTO BusyWait		; then jump to delay 250us before return
	
DisplayData:
	;routine to display a data byte in w reg at the current LCD address
	BSF PORTC,6			; RS pin of LCD high, for sending data
	CALL Nibbles2LCD	; then send data nibbles to LCD

BusyWait:
	; routine to wait until LCD module is not busy, after writing
	CALL Delay250us		; go pause for 250us
	RETURN				; then return

DispInit:
	; routine to initialise LCD display module
	CALL Delay160ms		; first wait about 160ms before proceeding
	BCF PORTC,6			; first set LCD's RS line low (RC6) for instr/addr
	BCF PORTC,7			; also set EN line low (RC7)
	MOVLW h'03'			; load init code into RC0-4
	MOVWF PORTC
	CALL ToggleEN		; then toggle EN to write to LCD
	CALL Delay10ms		; then wait about 10ms
	BCF PORTC,6			; make sure RS is still low
	BCF PORTC,0			; now change code to 20h
	CALL ToggleEN		; toggle EN to write
	CALL Delay10ms		; pause 10ms again	
	MOVLW h'28'			; now set LCD functions (4 bit i/f, 2 lines, 5x7 chars)
	CALL DispAddress	; (also delays for 160us)
	MOVLW h'0C'			; also set display mode (disp on, no blink or cursor)
	CALL DispAddress	; (also delays for 160us)
	CALL ClearLCD		; then clear LCD screen (& delay 320ms)
	MOVLW h'06'			; and finally set entry mode (increm addr, no shift)
	CALL DispAddress	; (also delays for 160us)
	RETURN				; should now be set up & ready to go, so leave

Display1:
	; routine to display initial greeting info on LCD
	MOVLW h'80'			; first set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "S"			; then send "SC 16bit Digital"
	CALL DisplayData
	MOVLW "C"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "1"
	CALL DisplayData
	MOVLW "6"
	CALL DisplayData
	MOVLW "b"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW " " 
	CALL DisplayData
	MOVLW "D"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "g"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "a"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW " "			; and display " Potentiometer "
	CALL DisplayData
	MOVLW "P"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "m"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	RETURN				; before leaving

Display2:
	; routine to display operating status on LCD
	MOVLW h'80'			; first set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW " "			; then send " Output="
	CALL DisplayData
	MOVLW "O"
	CALL DisplayData
	MOVLW "u"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "p"
	CALL DisplayData
	MOVLW "u"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "="
	CALL DisplayData

	BTFSS OutFlag,0		; now check output flag bit
	GOTO OutputOFF		; if not set, skip to show OFF
	MOVF OutDig1,0		; otherwise show output volts
	CALL DisplayData
	MOVF OutDig2,0
	CALL DisplayData
	MOVF OutDig3,0
	CALL DisplayData
	MOVF OutDig4,0
	CALL DisplayData
	MOVF OutDig5,0
	CALL DisplayData
	MOVF OutDig6,0
	CALL DisplayData
	MOVLW "V"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	GOTO Line2

OutputOFF:
	MOVLW " "			; output is off, so show " OFF    "
	CALL DisplayData
	MOVLW "O"
	CALL DisplayData
	MOVLW "F"
	CALL DisplayData
	MOVLW "F"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	
Line2:
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW " "			; and display " (Input=NN.NNNV)"
	CALL DisplayData
	MOVLW "("
	CALL DisplayData
	MOVLW "I"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW "p"
	CALL DisplayData
	MOVLW "u"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "="
	CALL DisplayData
	MOVF InDig1,0
	CALL DisplayData
	MOVF InDig2,0
	CALL DisplayData
	MOVF InDig3,0
	CALL DisplayData
	MOVF InDig4,0
	CALL DisplayData
	MOVF InDig5,0
	CALL DisplayData
	MOVF InDig6,0
	CALL DisplayData
	MOVLW "V"
	CALL DisplayData
	MOVLW ")"
	CALL DisplayData
	RETURN				; before leaving

Display3:
	; display routine for setting input voltage
	MOVLW h'80'			; set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "S"			; then send "Set Input Volts  "
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "I"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW "p" 
	CALL DisplayData
	MOVLW "u"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "V"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData

ShowLine2:
	MOVLW h'C0'			; now move down to line 2 of LCD
	CALL DispAddress
	MOVLW " "			; then show " Level: "
	CALL DisplayData
	MOVLW "L"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "v"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW ":"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVF EntryDig1,0	; followed by blanks or entered digits
	CALL DisplayData
	MOVF EntryDig2,0
	CALL DisplayData
	MOVF EntryDig3,0
	CALL DisplayData
	MOVF EntryDig4,0
	CALL DisplayData
	MOVF EntryDig5,0
	CALL DisplayData
	MOVF EntryDig6,0
	CALL DisplayData
	MOVLW "V"			; and end up line with "V "
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	RETURN				; before leaving

;	**************************************************************
Display4:
	; display routine for setting output voltage
	MOVLW h'80'			; set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "S"			; then send "Set Output Volts"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "O"
	CALL DisplayData
	MOVLW "u"
	CALL DisplayData
	MOVLW "t" 
	CALL DisplayData
	MOVLW "p"
	CALL DisplayData
	MOVLW "u"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "V"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	GOTO ShowLine2		; now go & display line 2 (shared with Display3)

;	***********************************************************
;
DoAScan:
	; routine to scan keypad keys via Scankeys, check after bounce delay
	; and save in Keycode if valid key press detected, before returning
	; (keeps looping if no valid keypress detected)
	CALL ScanKeys		; go scan keyboard
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not 0)
	GOTO DoAScan		; otherwise loop back & keep scanning
	MOVWF Keycode		; Z=0, so a key was pressed - save code
	CALL Pause100ms		; then wait for 100ms to allow for bounce
	CALL ScanKeys		; & scan again to verify
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS,Z		; skip if Z=0 (ie w not 0, because it has a keycode)
	GOTO DoAScan		; Z=1, so just a false alarm - loop back to start 
	XORWF Keycode, 0	; Z=0, so compare with first code (w -> 0 if match)
	BTFSS STATUS,Z		; skip if Z=1 (because codes did match)
	GOTO DoAScan		; no match - loop back to start
	RETURN				; codes match, so exit (with valid char in Keycode)
;
;	***********************************************************
;
EnteraNumber:
;	routine to enter a six-digit (including DP) number from keypad
;	into entry buffer (EntryDig1 - EntryDig6, set to blanks before
;	this routine is called)
	MOVLW h'40'			; first set digit entry buffer pointer
	MOVWF FSR
	MOVLW h'06'			; also InLoopCtr
	MOVWF InLoopCtr		; for entering up to 6 digits
	
NumEntryLoop:
 	CALL DoAScan		; now scan keys & save in Keycode when one presd
	MOVLW "A"			; check if it was "A"
	XORWF Keycode,0		; w will now have 0 if it was a match...
	BTFSC STATUS,Z		; skip if Z=0 (it wasn't an A)
	GOTO NumEntryLoop	; it was an A, so just ignore & loop back
	MOVLW "C"			; not an A, so try if it was a C
	XORWF Keycode,0		; w will now have 0 if it was a match
	BTFSC STATUS,Z		; skip if Z=0 (it wasn't a C)
	GOTO NumEntryLoop	; it was a C,  so just ignore & loop back
	MOVLW "D"			; not a C either, so try for a D
	XORWF Keycode,0		; w will now have 0 if it was a match
	BTFSC STATUS,Z		; skip if Z=0 (it wasn't a D)
	GOTO NumEntryLoop	; it was a D, so just ignore & loop back
	MOVLW "B"			; not a D, so try for a B
	XORWF Keycode,0		; w will now have 0 if it was a match
	BTFSC STATUS,Z		; skip if Z=0 (it wasn't a B)
	GOTO Backspace		; it was a B, so go do a backspace
	MOVLW "#"			; not a B, so try for a hash
	XORWF Keycode,0		; w will now have 0 if it was a match
	BTFSC STATUS,Z		; skip  if Z=0 (it wasn't a hash)
	GOTO EnterKey		; it was a hash (= Enter), so go process number
	MOVLW "*"			; not a hash, so try for a star
	XORWF Keycode,0		; w will now have 0 if it was a match
	BTFSC STATUS,Z		; skip if Z=0 (it wasn't a star)
	GOTO DecPoint		; it was a star, so go place DP as next digit
	MOVF Keycode,0		; must be a numeral, so fetch code in w
	GOTO SaveNum		; and go save in buffer	
DecPoint:
	MOVLW "."			; load w with DP
SaveNum:
	MOVWF INDF			; and save as next numeral in buffer
	CALL ShowLine2		; also show on entry screen Line2
	INCF FSR,1			; then increment buffer pointer
	DECFSZ InLoopCtr,1	; also decrement loop counter
	GOTO NumEntryLoop	; loop back if not entered 6
	RETURN				; but if we have done 6, leave

Backspace:
	DECF FSR,1			; first decrement buffer pointer
	INCF InLoopCtr,1	; then increment loop counter again
	MOVLW h'20'			; restore a space in orig position
	MOVWF INDF
	CALL ShowLine2		; and reshow entry screen Line2
	GOTO NumEntryLoop	; then loop back to get another

EnterKey:
	MOVLW h'30'			; enter key pressed, so fill up any
	MOVWF INDF			; remaining buffer positions with zeroes
	INCF FSR,1			; incrementing buffer pointer as we go
	DECFSZ InLoopCtr,1	; also decrementing loop counter
	GOTO $-3			; and looping back until done
	copy EntryDig5, EntryDig6	; now slide them all down by one
	copy EntryDig4, EntryDig5
	copy EntryDig3, EntryDig4
	copy EntryDig2, EntryDig3
	copy EntryDig1, EntryDig2	; so we can... 
	MOVLW h'30'					; add a leading zero
	MOVWF EntryDig1			
	RETURN				; now we're really done, so return
;
;	**********************************************************
ErrorDisp:
	; routine to display "FP Error" on LCD, then pause & return
	; (called only if FP functions flag an error)
	CALL ClearLCD
	MOVLW h'80'			; next set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "F"			; then send "FP Error "
	CALL DisplayData
	MOVLW "P"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "E"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "o" 
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVF Keycode,0		; then show error code (0 - 7)
	CALL DisplayData
	CALL Wait3Sec		; now pause for 3 seconds for user to see
	CALL ClearLCD		; then wipe away again
	RETURN				; and return

;	***********************************************************

MakeInDigs:
;	routine to copy entered-from-keypad digits into Input digits
	copy EntryDig1, InDig1
	copy EntryDig2, InDig2
	copy EntryDig3, InDig3
	copy EntryDig4, InDig4
	copy EntryDig5, InDig5
	copy EntryDig6, InDig6
	RETURN

;	***********************************************************

MakeInV:
;	Routine to convert keyed-in input digits into 16-bit binary
;	value, also convert into 24-bit FP form ready for calculation
	MOVLW h'30'			; first set indir addr ptr to InDig1
	CALL Turn2Binary	; then go convert InDigs to binary
	copy HbyteWkg, CurrInH	; and save as Input binary values
	copy LbyteWkg, CurrInL  ;
	copy CurrInH, AARGB0	; also place into AARGB0, AARGB1
	copy CurrInL, AARGB1
	CALL FLO24			; then go convert it into 24-bit FP form
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy AEXP,VinEXP	; next save 24b result in Vin regs
	copy AARGB0,VinB0
	copy AARGB1,VinB1
	RETURN					; then return

;	***********************************************************

MakeOutDigs:
;	routine to copy entered-from-keypad digits into Output digits
	copy EntryDig1, OutDig1
	copy EntryDig2, OutDig2
	copy EntryDig3, OutDig3
	copy EntryDig4, OutDig4
	copy EntryDig5, OutDig5
	copy EntryDig6, OutDig6
	RETURN

;	***********************************************************

MakeOutV:
;	Routine to convert keyed-in output digits into 16-bit binary 
;	value, then convert into 24-bit form and calculate corresponding
;	output value by multiplying by 8E7FFF (= FFFF in 24b form) and
;	then dividing by input voltage Vin (in 24b form). So Output =
;	(Vout x 65535)/Vin, which is then converted back to a 16-bit
;	right justified number, saved in CurrOutH&L and also ports B7D
 
	MOVLW h'36'			; first set indir addr ptr to OutDig1
	CALL Turn2Binary	; then go convert OutDigs to binary
	copy HbyteWkg, AARGB0	; and save in AARG regs
	copy LbyteWkg, AARGB1 ;
	CALL FLO24			; then go convert it into 24-bit FP form
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy AEXP,VoutEXP	; next save 24b result in Vout regs
	copy AARGB0,VoutB0
	copy AARGB1,VoutB1	; (AARG regs also still have it)

	MOVLW h'8E'			; now place 24b version of FFFFh into BARG regs
	MOVWF BEXP			; (found using Fprep: 65535d = FFFFh = 8E7FFF)
	MOVLW h'7F'
	MOVWF BARGB0
	MOVLW h'FF'
	MOVWF BARGB1
	CALL FPM24			; multiply the two (AARG x BARG -> AARG regs)	
	CALL Check4FPE		; (duck away to check for any FP errors)

	copy VinEXP, BEXP	; now place 24b version of Vin into BARG regs
	copy VinB0, BARGB0
	copy VinB1, BARGB1
	CALL FPD24			; do the division (AARG/BARG -> AARG regs)
	CALL Check4FPE		; (duck away to check for any FP errors)

	CALL INT2416		; now convert result into 16-bit fixed form
;	CALL Check4FPE		; (duck away to check for any FP errors)
	copy AARGB0, CurrOutH	; now copy high and low	bytes into
	copy AARGB1, CurrOutL	; current 16-bit Output volts regs
	copy CurrOutL,PORTD		; then copy L output byte to port D
	copy CurrOutH,PORTB		; and copy H output byte to port B
	RETURN					; then return
;
;	***********************************************************
;
MultWkgby10:
;	routine to multiply current 16-bit contents of wkg register
;	by 10. We do this by converting HbyteWkg + LbyteWkg to 24bit FP
;	format, then multiplying by 10d (= 822000 in PIC 24b format),
;	then convert the result back into 16-bit fixed pt form and
;	saving it again in the wkg regs before we leave
	copy HbyteWkg, AARGB0	; first copy high byte into AARGB0
	copy LbyteWkg, AARGB1	; and low byte into AARGB1
	CALL FLO24				; then go convert into 24-bit FP form
	CALL Check4FPE			; (duck away to check for FP errors)
;							; AARG regs now have it in 24-bit form
	MOVLW h'82'				; now place 24-bit form of 10d into
	MOVWF BEXP				; BARG regs, ready for multiply
	MOVLW h'20'
	MOVWF BARGB0
	MOVLW h'00'
	MOVWF BARGB1
	CALL FPM24				; now multiply them (AARG x BARG -> AARG)
	CALL Check4FPE			; (duck away to check for any FP errors)
	CALL INT2416			; result in AARG, so convert back into 16b
;	CALL Check4FPE			; (duck away to check for any FP errors)
	copy AARGB0, HbyteWkg	; so save 16b result back into wkg regs
	copy AARGB1, LbyteWkg
	RETURN					; WkgRegs now have x10 result, so return
;
;	***************************************************************	
;
Nibbles2LCD:
	; routine to test bits of data byte (passed in w) then send
	; to LCD module as two nibbles (high nibble first)
	MOVWF Temp2			; first save byte to be sent in Temp2
	BTFSC Temp2,7		; now test bit 7 (upper nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	BTFSC Temp2,6		; now test bit 6
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,5		; now test bit 5
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,4		; now test bit 4
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	CALL ToggleEN		; now toggle EN to write hi nibble to LCD

	BTFSC Temp2,3		; next test bit 3 (lower nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	BTFSC Temp2,2		; now test bit 2
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,1		; now test bit 1
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,0		; now test bit 0
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	CALL ToggleEN		; toggle EN again to write low nibble to LCD
	RETURN				; and return, since both nibbles now sent
;
;	***********************************************************
;
OutputTogl:
;	routine to toggle potentiometer output status (on/off)
	BTFSS OutFlag,0		; check status of output flag (bit 0)
	GOTO TurnitOn		; if not currently set, go set it & turn on
	BCF OutFlag,0		; was set, so clear it
	CLRF PORTB			; turn off output H byte (0 -> port B)
	CLRF PORTD			; and also output L byte (0 -> port D)
	CALL Pause200ms		; then pause a while
	RETURN				; before returning

TurnitOn:	
	BSF OutFlag,0		; wasn't set, so set it now
	copy CurrOutL,PORTD	; then copy L output byte to port D
	copy CurrOutH,PORTB	; and copy H output byte to port B
	CALL Pause200ms		; then pause a while
	RETURN				; before returning

;	***********************************************************

Pause100ms:
;	routine to delay about 100ms before returning
	MOVLW h'0A'			;set Counter 3 for 10 outer loops
	MOVWF Counter3
	GOTO $+3			; then skip 3 lines
Pause200ms:
;	routine to delay approx 200ms before returning
	MOVLW h'14'			; set Counter3 for 20 outer loops
	MOVWF Counter3
	CALL Delay10ms		; then wait 10ms
	DECFSZ Counter3, F	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return

;	***********************************************************
;		
ScanKeys:
	; routine to look for a pressed key, return with ASCII code
	; in w (or with a null in w if no key is pressed)
	CLRF PORTA			; first clear both port A and port E
	CLRF PORTE			; to start with a known-clear slate
	BSF PORTA, 4		; then to begin set RA4 high for top row
	CALL BounceWait		; wait about 50ms for RA4 to stabilise
	BTFSS PORTA, 0		; now check if RA0 is set ("1" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "1"			; S1 pressed, so return with "1" in w
	BTFSS PORTA, 1		; next try RA1 ("2" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "2"			; S2 pressed, so return with "2" in w
	BTFSS PORTA, 2		; next try RA2 ("3" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "3"			; S3 pressed, so return with "3" in w
	BTFSS PORTA, 3		; next try RA3 ("A" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "A"			; S4 pressed, so return with "A" in w
	BCF PORTA,4			; now drop RA4

	BSF PORTA,5			; and set RA5 high for next row
	BTFSS PORTA, 0		; now check if RA0 is set ("4" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "4"			; S5 pressed, so return with "4" in w
	BTFSS PORTA, 1		; next try RA1 ("5" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "5"			; S6 pressed, so return with "5" in w
	BTFSS PORTA, 2		; next try RA2 ("6" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "6"			; S7 pressed, so return with "6" in w
	BTFSS PORTA, 3		; next try RA3 ("B" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "B"			; S8 pressed, so return with "B" in w
	BCF PORTA,5			; now drop RA5

	BSF PORTE,0			; and set RE0 high for next row
	BTFSS PORTA, 0		; now check if RA0 is set ("7" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "7"			; S9 pressed, so return with "7" in w
	BTFSS PORTA, 1		; next try RA1 ("8" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "8"			; S10 pressed, so return with "8" in w
	BTFSS PORTA, 2		; next try RA2 ("9" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "9"			; S11 pressed, so return with "9" in w
	BTFSS PORTA, 3		; next try RA3 ("C" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "C"			; S12 pressed, so return with "C" in w
	BCF PORTE,0			; now drop RE0

	BSF PORTE,1			; and set RE1 high for last row
	BTFSS PORTA, 0		; now check if RA0 is set ("*" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "*"			; S13 pressed, so return with "*" in w
	BTFSS PORTA, 1		; next try RA1 ("0" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "0"			; S14 pressed, so return with "0" in w
	BTFSS PORTA, 2		; next try RA2 ("#" pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "#"			; S15 pressed, so return with "#" in w
	BTFSS PORTA, 3		; next try RA3 ("D" pressed)
	GOTO $+2			; otherwise prepare to return
	RETLW "D"			; S16 pressed, so return with "D" in w
	BCF PORTE,1			; no key pressed, so drop RE1
	RETLW h'00'			; and return with null in w reg
;
;	***********************************************************
;
SetDefaults:
;	Subroutine to set potentiometer's default initial conditions
	CLRF OutFlag		; first set output to off
	MOVLW h'27'			; now set CurrIn level to 10,000mV
	MOVWF CurrInH		; (10000 = 2710h)
	MOVLW h'10'
	MOVWF CurrInL
	MOVLW h'8C'			; also set 24b Vin values to 10,000mV
	MOVWF VinEXP		; (10000d = 8C1C40, found from Fprep)
	MOVLW h'1C'			; (needed for correct calculation of
	MOVWF VinB0			;  new Vout values if Vin is unchanged)
	MOVLW h'40'
	MOVWF VinB1
	MOVLW h'7F'			; also set CurrOut level to 5,000mV
	MOVWF CurrOutH		; (5000d = FFFFh/2 = 7FFFh)
	MOVLW h'FF'
	MOVWF CurrOutL
	MOVLW h'31'			; then set Input Volts display digits
	MOVWF InDig1
	MOVLW h'30'
	MOVWF InDig2
	MOVLW h'2E'			
	MOVWF InDig3
	MOVLW h'30'
	MOVWF InDig4
	MOVLW h'30'
	MOVWF InDig5
	MOVLW h'30'
	MOVWF InDig6
	MOVLW h'20'			; finally set Output Volts display digits
	MOVWF OutDig1
	MOVLW h'35'
	MOVWF OutDig2
	MOVLW h'2E'			
	MOVWF OutDig3
	MOVLW h'30'
	MOVWF OutDig4
	MOVLW h'30'
	MOVWF OutDig5
	MOVLW h'30'
	MOVWF OutDig6
	
	RETURN				; then leave

;	************************************************************
;
SetNuInput:
;	routine to enter a new input volts level from keypad switches
;	(entered after user presses "D" key)
	CALL Pause100ms		; first pause to skip key bounce
	CALL ClearEntryDigs	; then clear entry digits
	CALL Display3		; then display Involts entry screen
	CALL EnteraNumber	; call enter a number routine
	CALL MakeInDigs		; copy entered digits into Input digits
	CALL MakeInV		; also convert to binary input value

	MOVF InDig1,W		; now check if we have a leading zero
	XORLW h'30'			; by doing an Ex-OR
	BTFSS STATUS,Z		; skip if Z=1, because we got a match
	RETURN				; but if Z=0, no match so just return
	MOVLW h'20'			; did get a match, so make it a space
	MOVWF InDig1		; (for LZ blanking in display) 
	RETURN
;
;	***********************************************************
;
SetNuOutput:
;	routine to enter new output volts level from keypad switches
;	(entered after user presses "A" key)
	CALL Pause100ms		; first pause to skip key bounce
	CALL ClearEntryDigs	; then clear entry digits
	CALL Display4		; then display Outvolts entry screen
	CALL EnteraNumber	; enter a number
	CALL MakeOutDigs	; copy entered digits into Output digits
	CALL MakeOutV		; also convert to binary and calculate
;						  correct output values
	MOVF OutDig1,W		; now check if we have a leading zero
	XORLW h'30'			; by doing an Ex-OR
	BTFSS STATUS,Z		; skip if Z=1, because we got a match
	RETURN				; but if Z=0, no match so just return
	MOVLW h'20'			; did get a match, so make it a space
	MOVWF OutDig1		; (for LZ blanking in display) 
	RETURN				; and then return

;	***********************************************************
;
ToggleEN:
	; routine to toggle EN line of LCD, to write an instr or data nibble
	BSF PORTC,7			; take LCD's EN line high (RC7)
	NOP					; pause 2us (4mc) to let it stabilise
	NOP
	NOP
	NOP
	BCF PORTC,7			; then low again, to write into LCD controller
	RETURN				; then return

;	***********************************************************
;
Turn2Binary:
	; routine to convert 5-digit-plus-DP number into a 2-byte
	; binary number. Location of MSD digit is placed in w
	; before calling. Result is placed in HbyteWkg and LbyteWkg.
	MOVWF FSR			; first save initial digit address in FSR
	MOVLW h'06'			; also set loop ctr for 6 loops
	MOVWF InLoopCtr
	CLRF HbyteWkg		; then clear high & low bytes of working reg
	CLRF LbyteWkg
FetchChar:
	MOVF INDF,0			; now fetch a character into w
	XORLW h'2E'			; check if it's a decimal point
	BTFSC STATUS,Z		; skip if it isn't (Z=0)
	GOTO NoCarry		; but if it is, just move on
	MOVLW h'30'			; now place 30h in w
	SUBWF INDF,0		; subtract it from char, result -> w
	MOVWF Temp1			; then save this value in Temp1 (0 - 9h)
	CALL MultWkgby10	; while we multiply Wkg reg by 10
	MOVF Temp1,0		; now fetch value from Temp1
	ADDWF LbyteWkg,1	; add it to low byte of Wkg register
	BTFSS STATUS,C		; check if there was a carry
	GOTO NoCarry		; if not, skip carry section
	INCF HbyteWkg,1		; if there was, add 1 to high byte
NoCarry:
	INCF FSR,1			; now increment FSR for next char
	DECFSZ InLoopCtr,1	; also decrement loop ctr, skip if -> 0
	GOTO FetchChar		; if not, loop back to continue
	RETURN				; otherwise leave, since we're done
;
;	***********************************************************
;	
Wait3Sec:
;	routine to pause for about 3 seconds, to allow reading display
	MOVLW h'12'			; first set Counter4 for 18d loops
	MOVWF Counter4		; (because 18 x 160ms = 2.88sec)
	CALL Delay160ms		; then wait for 160ms
	DECFSZ Counter4,1	; now decrement Counter4, skip when zero
	GOTO $-2			; otherwise keep looping
	RETURN				; return when done
;
;	**********************************************************
;	include floating point routines
;	(in FPRF24.TXT)
;
	#include 	<FPRF24.TXT>

;	**********************************************************	
	END
