/*

            1 Reset    VCC 20
7Seg  <-----2 PD0   T  PB7 19-------> 7Seg
Buzzer<-----3 PD1   i  PB6 18-------> 7Seg
Xtal  <-----4 PA0   n  PB5 17-------> 7Seg
Xtal  <-----5 PA1   y  PB4-16-------> 7Seg
IR    <-----6 INT0  2  PB3-15-------> 7Seg
Trig  <-----7 PD3   3  PB2-14-------> 7Seg
Reset <-----8 PD4   1  PB1-13-------> 7Seg
Relay <-----9 PD5   3  PB0-12-------> 7Seg
            10 GND     PD6-11-------> 7Seg
                               
fuses: CKDIV8=0 CKOUT=1 SUT=01 BOD=100 CKSEL=1101 hfuse=11001001 (0xc9) lfuse=01011101 (0x5d)
*/

#define F_CPU 8000000UL
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <string.h>

unsigned char display[6], target[6];
unsigned char autocount[6], lastcount[6];
unsigned char display_on, dot_on, running, paused, beeping, beeping_min, dim, ack, ack_timer;
unsigned char settings[6]; // 0 = buzzer on, 1 = relay off/running/beeping, 2 = trigger off/low/high, 3 = reset off/low/high, 4 = auto count down, 5 = # minutes to beep

static void slow_clock() {
  CLKPR = _BV(CLKPCE);
  CLKPR = _BV(CLKPS1)|_BV(CLKPS0);
  TCCR1B = _BV(CS10);
}

static void fast_clock() {
  CLKPR = _BV(CLKPCE);
  CLKPR = 0;
  TCCR1B = _BV(CS11);
}

static void enable_display() {
  fast_clock();
  display_on = 1;
}

static void disable_display() {
  slow_clock();
  display_on = 0;
  display[0] = display[1] = display[2] = display[3] = display[4] = display[5] = 16;
}

static void save_display_to_eeprom(unsigned char addr) {
  unsigned char i;
  for( i = 0; i < 6; ++i ) {
    eeprom_write_byte((uint8_t*)(unsigned short)addr, display[i]);
    asm("wdr");
    ++addr;
  }
  disable_display();
}

static void load_from_eeprom(unsigned char pos, unsigned char* to_here, unsigned char max, unsigned char def) {
  unsigned char i, temp;
  i = 6;
  while( i-- ) {
    temp = eeprom_read_byte((uint8_t*)(unsigned short)pos);
    if( temp > max )
      temp = def;
    *to_here = temp;
    ++pos;
    ++to_here;
  }
}

static void load_settings() {
  load_from_eeprom(0, settings, 2, 1);
  load_from_eeprom(6, autocount, 9, 0);
  dim = ~eeprom_read_byte((uint8_t*)12);
}

///////////////////////////////////////////////////////////////////////
// IR

#define IR_IN_VECT              INT0_vect
#define IR_CFG_INTERRUPT()      sei(); MCUCR = _BV(ISC00);
#define IR_ENA_INTERRUPT()      GIMSK = _BV(INT0)
#define IR_DIS_INTERRUPT()      GIMSK = 0
#define IR_READ()              (PIND&_BV(PD2))

unsigned short rc5_code, rc5_timing;
unsigned char rc5_bit, rc5_reset_timer;
volatile unsigned short rc5_final_code;

inline static void cancel_rc5_reception() {
  TIMSK = 0;
  rc5_bit = 0;
}

ISR(TIMER1_OVF_vect) {
  if( rc5_bit == 1 || rc5_reset_timer ) {
    cancel_rc5_reception();
  } else {
    unsigned char level = IR_READ();
    if( ++rc5_bit == 15 ) {
      rc5_final_code = rc5_code|0x8000;
      cancel_rc5_reception();
    } else {
      rc5_code <<= 1;
      rc5_code |= level != 0;
      TCNT1 = 0 - rc5_timing;
      rc5_reset_timer = 1;
    }
  }
}

#define RC5_PERIOD (2*0.00088889)
#define RC5_CYCLES (F_CPU/8*RC5_PERIOD)
#define RC5_MIN    (RC5_CYCLES / 1.2)
#define RC5_MAX    (RC5_CYCLES * 1.2)

ISR(IR_IN_VECT) {
  if( rc5_bit == 0 ) {
    if( !IR_READ() ) {
      // start reception
      TCNT1 = 0 - (unsigned short)RC5_MAX;
      TIFR = _BV(TOV1);
      TIMSK = _BV(TOIE1);
      rc5_reset_timer = 0;
      rc5_bit = 1;
      rc5_code = 0;
    }
  } else if( rc5_bit == 1 ) {
    if( !IR_READ() ) {
      unsigned short temp = TCNT1 + (unsigned short)RC5_MAX;
      rc5_timing = temp>>1;
      cancel_rc5_reception();
      if( temp >= (unsigned short)RC5_MIN ) {
        TCNT1 = 0 - rc5_timing - (rc5_timing>>1);
        TIMSK = _BV(TOIE1);
        rc5_bit = 2;
      }
    }
  } else if( rc5_reset_timer ) {
    TCNT1 = 0 - rc5_timing - (rc5_timing>>1);
    rc5_reset_timer = 0;
  }
}

///////////////////////////////////////////////////////////////////////

static const unsigned char digits[10] = { 1|2|4|8|16|32, 2|4, 1|2|64|16|8, 1|2|64|4|8, 2|4|64|32, 1|32|64|4|8, 1|32|16|8|4|64, 1|2|4, 1|2|4|8|16|32|64, 1|2|4|8|32|64 };
static const unsigned char anodes[6] = { 3, 2, 6+16, 0, 6, 7 };
static const unsigned char cathodes[3][8] = {
  { 7, 6, 5, 4, 6+16, 0, 1, 0+16 },
  { 5, 6, 7, 0+16, 3, 2, 1, 4 },
  { 4, 5, 6+16, 2, 1, 0+16, 3, 0 }
};

static unsigned char get_bit(unsigned char which) {
  return 1<<(which&15);
}

inline static void set_cathode_on(unsigned char which) {
  unsigned char bit = get_bit(which);
  if( which & 16 )
    DDRD |= bit;
  else
    DDRB |= bit;
}
inline static void set_cathode_off(unsigned char which) {
  unsigned char bit = ~get_bit(which);
  if( which & 16 )
    DDRD &= bit;
  else
    DDRB &= bit;
}
inline static void set_anode_on(unsigned char which) {
  unsigned char bit = get_bit(which);
  if( which & 16 ) {
    DDRD |= bit;
    PORTD |= bit;
  } else {
    DDRB |= bit;
    PORTB |= bit;
  }
}
inline static void set_anode_off(unsigned char which) {
  unsigned char bit = ~get_bit(which);
  if( which & 16 ) {
    PORTD &= bit;
    DDRD &= bit;
  } else {
    PORTB &= bit;
    DDRB &= bit;
  }
}

static void mydelay() {
#ifdef JAYCAR_MODS
  _delay_ms(0.10);
#else
  _delay_ms(0.01);
#endif
}
static void mydelay2() {
#ifdef JAYCAR_MODS
  _delay_ms(0.05);
#else
  mydelay();
#endif
}
static void do_display() {
  unsigned char i, j, cath, temp;
  for( i = 0; i < 6; ++i ) {
    set_anode_on(anodes[i]);
    temp = display[i]&16 ? 0 : digits[display[i]];
    if( i == 1 && (running|beeping) )
      temp |= dot_on;
    else if( i == 5 )
      temp |= ack;
    for( j = 0; j < 8; ++j ) {
      cath = cathodes[i>>1][j];
      if( temp&1 )
        set_cathode_on(cath);
      mydelay();
      if( i < 4 )
        mydelay2();
      if( temp&1 )
        set_cathode_off(cath);
      temp >>= 1;
    }
    set_anode_off(anodes[i]);
    mydelay();
  }
}

static void increment_time() {
  unsigned char i;
  for( i = 5; i != (unsigned char)-1; --i ) {
    if( ++display[i] == (i == 2 || i == 4 ? 6 : 10 ) ) {
      display[i] = 0;
    } else {
      break;
    }
  }
}
static void decrement_time() {
  unsigned char i;
  for( i = 5; i != (unsigned char)-1; --i ) {
    if( --display[i] == (unsigned char)-1 ) {
      display[i] = (i == 2 || i == 4 ? 5 : 9);
    } else {
      break;
    }
  }
}

inline static void relay_on() {
  PORTD |= _BV(PD5);
}

inline static void relay_off() {
  PORTD &= ~_BV(PD5);
}

static void fix_overflow(unsigned char* target) {
  if( target[2] >= 6 ) {
    target[2] -= 6;
    if( target[1] < 9 && target[0] < 9 ) {
      if( ++target[1] == 10 ) {
        target[1] = 0;
        ++target[0];
      }
    }
  }
}

int main(void) {
  unsigned char timer = 0; // not necessary but removes warning
  unsigned char dm = 0;
  unsigned short last_ir = 0;
  unsigned char can_reset, should_trigger, should_reset, temp;

  // INITIALIZE
  // turn off analogue comparator, we don't need it
  ACSR |= _BV(ACD);

  disable_display();
  load_settings();
  DDRD = _BV(PD1)|_BV(PD3)|_BV(PD4)|_BV(PD5); // enable buzzer and relay pins and charge input filter caps
  PORTD = _BV(PD3)|_BV(PD4); // enable input pull-ups

  IR_CFG_INTERRUPT();
  IR_ENA_INTERRUPT();

  // set up timer for clock
  TCCR0A = _BV(WGM01);
  TCCR0B = _BV(CS12);
  OCR0A = 250-1;

  while(1) {
    asm volatile ("wdr");
    dot_on = timer >= 125/2 ? 128 : 0;
    if( ack_timer ) {
      if( --ack_timer == 0 )
        ack = 0;
    }
    if( display_on && (!beeping || dot_on) ) {
      if( beeping && settings[0] )
        PORTD |= _BV(PD1);
      else
        PORTD &= ~_BV(PD1);

      if( dm < dim ) {
        ++dm;
#ifdef JAYCAR_MODS
        mydelay();
        mydelay();
#else
        _delay_ms(0.20);
#endif
      } else {
        do_display();
        dm = 0;
      }
    } else {
      PORTD &= ~_BV(PD1);
    }
    if( TIFR & _BV(OCIE0A) ) {
      if( ++timer == 125 ) {
        timer = 0;
        if( running && !paused ) {
          if( running == 2 )
            decrement_time();
          else
            increment_time();
          if( !memcmp((unsigned char*)display, target, 6) ) {
            beeping_min = settings[5];
            running = 0;
            if( beeping_min ) {
              beeping = 60;
              if( settings[1] == 2 )
                relay_on();
              else
                relay_off();
            } else {
              goto StopBeeping;
            }
          }
        } else if( beeping ) {
          if( --beeping == 0 ) {
            if( --beeping_min == 0 ) {
            StopBeeping:
              relay_off();
              disable_display();
            } else {
              beeping = 60;
            }
          }
        }
      }
      TIFR |= _BV(OCIE0A);
    }
    if( rc5_final_code ) {
      unsigned char diff;
      unsigned short code;
      code = rc5_final_code;
      rc5_final_code = 0;
      diff = last_ir != code;
      last_ir = code;
      temp = code;
      ack = 0x80;
#ifdef JAYCAR_MODS
      ack_timer = 10;
#else
      ack_timer = 100;
#endif
      if( temp == 0x10 ) {
        // volume (brightness) down
        if( dim > 0 )
          --dim;
      } else if( temp == 0x11 ) {
        // volume (brightness) up
        if( dim < 32 )
          ++dim;
      } else if( temp == 0x0c ) {
        // standby (reset/clear)
      ResetCounter:
        running = 0;
        beeping = 0;
        goto StopBeeping;
      } else if( temp == 0x36 ) {
        // stop (pause)
        if( running )
          paused = 1;
      } else if( temp == 0x35 ) {
        // play (resume)
        paused = 0;
      } else if( temp == 0x3C ) {
        // start preprogrammed count
        goto StartProgrammedCount;
      } else if( !(running|beeping) ) {
        if( temp <= 9 && diff ) {
          // numbers 0-9
          memcpy(display, display+1, 5);
          display[5] = (unsigned char)temp;
          enable_display();
        } else if( display_on && (temp&~1) == 0x20 && !(display[5]&16) ) {
          unsigned char i;
          unsigned char* dest, * src;

        CountDownOrUp:
          if( beeping ) {
            dest = display;
            src = lastcount;
          } else {
            dest = lastcount;
            src = display;
          }
          memcpy(dest, src, 6);

          if( temp&1 ) {
            // channel (count) down
            for( i = 0; i < 6; ++i ) {
              display[i] &= 15;
              target[i] = 0;
            }
            running = 2;
          } else {
            // channel (count) up
            for( i = 0; i < 6; ++i ) {
              target[i] = display[i]&15;
              display[i] = 0;
            }
            fix_overflow(target+2);
            fix_overflow(target);
            running = 1;
          }
          paused = 0;
          timer = 0;
          beeping = 0;
          if( settings[1] == 1 )
            relay_on();
        } else if( temp == 0x0d || temp == 0x0a ) {
          // mute (configure)
          if( display_on )
            save_display_to_eeprom((0x0d-temp)<<1);
          if( temp == 0x0d )
            eeprom_write_byte((uint8_t*)12, ~dim);
          load_settings();
        }
      }
    }

    DDRD &= ~_BV(PD3); // stop charging input filter caps so we can sample inputs
    DDRD &= ~_BV(PD4);
    should_trigger = settings[2] + !(PIND&_BV(PD3)) == 2;
    should_reset   = settings[3] + !(PIND&_BV(PD4)) == 2;
    can_reset = running|beeping;
    if( should_trigger && !should_reset && !can_reset ) {
    StartProgrammedCount:
      memcpy(display, autocount, 6);
      enable_display();
      temp = settings[4];
      goto CountDownOrUp;
    } else if( should_reset && !should_trigger && can_reset ) {
      goto ResetCounter;
    }
  }

  return 0;
}
