;	************************************************************
;	ElectroRefmr.asm - firmware for PIC-based
;	Electrolytic Reformer & Cap Leakage Meter
;
;	Written by Jim Rowe for Silicon Chip, using a PIC16F88
;	processor running from its internal clock oscillator
;	at 8MHz, so one machine cycle (mc) = 0.5us.

;	Program last revised 16/07/2010.
;
;	This version of the program gives a choice of seven different
;	test/reform voltage apply times (10s/30s/1m/3m/10m/30m/60m),
;	with the HV generator now controlled by the PIC itself using
;	a second small relay (RLY2). Also takes the average of 10-
;	measurement sequences instead of single readings, to reduce the
;	effect of any ripple present on the HV converter's output.
;
;	It now also takes a measurement of the actual voltage across the
;	capacitor undergoing testing/reforming, and displays this voltage
;	along with the leakage current and remaining application time.
;
;	Note: Program makes use of 24-bit and 32-bit floating point &
;	fixed point maths routines for Microchip Technology Inc's
; 	8-bit PIC processors, written by Frank J. Testa and
;	described in MTI's App Notes AN575, AN617, AN660 and AN670,
; 	downloadable from the MTI website at www.microchip.com
;	(Routines used here are all in FPRF24.TXT)
;
;	**********************************************************
; 
;	CPU configuration
;
	list p=PIC16f88
	#include		"p16f88.inc"
	__CONFIG        _CONFIG1, h'3F39'
	__CONFIG		_CONFIG2, _IESO_OFF & _FCMEN_OFF
;
;**********************************************************
;
;	First define some handy macros
;
;	Select Register Bank 0
bank0	macro
	errorlevel	+302		; Re-enable bank warning
	BCF	STATUS,RP0		; Select Bank 0
	BCF STATUS, RP1
	BCF STATUS, IRP			; and also indir addr banks 0,1
	endm

;	Select Register Bank 1
bank1	macro
	BSF		STATUS,RP0		; Select Bank 1
	errorlevel	-302		; disable warning
	endm

;	Copy bytes in register file via W
copy	macro	from,to
	MOVF	from,W
	MOVWF	to
	endm

; 	convert a numeral byte in w into ASCII code (in w)
GetASCII	macro
	ANDLW h'0F'			; first mask off upper nibble, if any
	IORLW h'30'			; now make upper nibble h'30'
	endm

;	turn off power to HV converter via RA4, Q4 & RLY2
HVoff	macro
	BCF Flags,1		; clear HV flag
	BCF PORTA,4		; clear RA4 to turn off RLY2
	endm

;	turn on power to the HV converter, to begin testing
HVon	macro
	BSF Flags,1		; first set HV flag
	BSF PORTA,4		; then set RA4 to turn on RLY2 & converter
	endm

;	Prepare to call or goto page1 (800-FFFh)
Ppage1	macro
	BCF PCLATH,4	; clear bit4 of PCLATH
	BSF PCLATH,3	; but set bit3
	endm

;	Prepare to call or goto page0 (000-7FFh)
Ppage0	macro
	BCF PCLATH,4	; clear bit 4 of PCLATH
	BCF PCLATH,3	; also clear bit 3
	endm

; 	set BARG regs to 10d, before an FP operation
; 	(10d = 82 20 00 00, precalc using Fprep)
SetBARG10	macro
	MOVLW h'82'			; exponent first
	MOVWF BEXP
	MOVLW h'20'			; then MSB/sign byte
	MOVWF BARGB0
	CLRF BARGB1			; then remaining two bytes
	CLRF BARGB2
	endm

;	set the PIC for measurement on the 0-10mA range
SetHiRange	macro
	BSF Flags,0		; set range flag for 10mA range
	BCF PORTA,1		; also drop RA1 to turn on Q5 & RLY1
	CALL InitDig	; then re-initialise digits & multiplier
	endm

;	Swap bytes in register file via W
swap	macro	this,that
	MOVF	this,w		; get this
	XORWF	that,f		; Swap using Microchip
	XORWF	that,w		; Tips'n Tricks
	XORWF	that,f		; #18
	MOVWF	this
	endm

;	toggle EN pin of LCD module to write an instr or data nibble
ToggleEN	macro
	BSF PORTB,4			; take LCD's EN line high (RB4)
	GOTO $+1			; pause 2us (4mc) to let it stabilise
	GOTO $+1
	BCF PORTB,4			; then low again, to write into LCD controller
	endm
;
;	**********************************************************
;     STATUS bit definitions (used in FPRF24.TXT)

#define	_C	STATUS,0
#define	_Z	STATUS,2
;
;**************************************************************
;
;	storage register declarations:

Counter1	equ	h'20'	; gp counter variable 1
Counter2	equ h'21'	; gp counter variable 2
Counter3	equ h'22'	; gp counter variable 3
Counter4	equ h'23'	; gp counter variable 4
MLoopCtr	equ h'24'	; measurement loop counter
Temp1		equ	h'25'	; working storage location 1
Temp2		equ h'26'	; working storage location 2
Temp3		equ h'27'	; working storage location 3
Flags		equ h'28'	; current range, HV conv & test mode on/off flags
						; (bit 0: 1 = 10mA, 0 = 100uA,
; 						   bit 1: 1 = HV on, 0 = HV off
;						   bit 2: 1 = in test mode, 0 = not yet

IDig1		equ h'29'	; first digit of leakage current reading
IDig2		equ h'2A'	; second digit of leakage current reading
IDig3		equ h'2B'	; third digit of leakage current reading
IDig4		equ h'2C'	; fourth digit of leakage current reading
IDig5		equ h'2D'	; fifth digit (multiplier)
;
;	storage for Cap voltage reading display digits
VDig1		equ h'2E'	; first digit (hundreds)
VDig2		equ h'2F'	; second digit (tens)
VDig3		equ h'30'	; third digit (units)

;	storage for ADC reading in 24-bit form, after calculation
ADCEXP		equ h'31'	; exponent
ADCB0		equ h'32'	; MSB & sign bit
ADCB1		equ h'33'	; LSB

;	storage of timer values and display digits
Tonsec		equ h'34'	; HV conv on time setting in seconds (0/10/30)
Tonmin		equ h'35'	; HV conv on time setting in mins (0/1/3/10/30/60)
TCmin		equ h'36'	; current TMR0 time period, mins (0-60)
TCsec		equ h'37'	; current TMR0 time period, seconds (0-60)
TC25ms		equ h'38'	; current TMR0 time period, 25ms R/T steps (40/second)

TonSunits	equ h'39'	; HV conv on time display digit - units of seconds
TonStens	equ h'3A'	; HV conv on time display digit - 10s of seconds
TonMunits	equ h'3B'	; HV conv on time display digit - units of minutes
TonMtens	equ h'3C'	; HV conv on time display digit - 10s of minutes

TmrSunits	equ h'3D'	; current Tmr0 count display digit, units of seconds
TmrStens	equ h'3E'	; current Tmr0 count display digit, tens of seconds
TmrMunits	equ h'3F'	; current Tmr0 count display digit, units of minutes
TmrMtens	equ h'40'	; current Tmr0 count display digit, tens of minutes

;	regs for storage of Byte10 and Byte1 (output bytes of ByteSplit routine)
Byte10		equ h'41'	; decimal tens output
Byte1		equ h'42'	; decimal units output

;	regs for storage of variables used by Bin16toBCD conversion routine
Count		equ h'43'	; counter
Temp		equ h'44'	; temp storage
H_byte		equ h'45'	; higher input byte
L_byte		equ h'46'	; lower input byte
R0			equ h'47'	; storage for output MSD (in rightmost nibble)
R1			equ h'48'	; storage for next two output digit nibbles
R2			equ h'49'	; storage for last two output digit nibbles

Ones		equ h'4A'	; storage for least signif BCD digit in ASCII
Tens		equ h'4B'	; storage for next signif BCD digit in ASCII
Hundreds	equ h'4C'	; and the next digit
Thousands	equ h'4D'	; and the next digit

; 	regs for saving context during interrupt servicing
;	(at top of RAM so they can be accessed from all banks)
WSave 		equ h'7B'	; w reg saved here
SSave		equ h'7C'	; status reg saved here
PCHiSave 	equ h'7D'	; PCLATH saved here

;	***********************************************************
;   Floating Point Stack & other locations used by FPRF24.TXT
;	routines
;
AARGB7	equ h'50'	; AARGB7 byte for FP argument A
AARGB6	equ h'51'	; AARGB6 byte
AARGB5	equ h'52'	; AARGB5 byte
AARGB4	equ	h'53'	; AARGB4 byte
AARGB3	equ h'54'	; AARGB3
AARGB2	equ h'55'	; AARGB2
AARGB1	equ h'56'	; AARGB1
AARGB0	equ h'57'	; AARGB0
AEXP	equ h'58'	; 8 bit biased exponent for argument A

BARGB3	equ h'59'	; BARGB3 byte for argument B
BARGB2	equ h'5A'	; BARGB2
BARGB1	equ h'5B'	; BARGB1
BARGB0	equ h'5C'	; BARGB0
BEXP	equ h'5D'	; 8 bit biased exponent for argument B

TEMPB3	equ h'5E'	; TEMPB3 byte
TEMPB2	equ h'5F'	; TEMPB2 byte
TEMPB1	equ h'60'	; TEMPB1 byte
TEMPB0	equ h'61'	; TEMPB0 byte

CARGB1	equ h'62'	; CARGB1 byte for argument C
CARGB0	equ h'63'	; CARGB0 byte
CEXP	equ h'64'	; 8 bit biased exponent for argument C

DARGB3	equ h'65'	; DARGB3 byte for argument D
DARGB2	equ h'66'	; DARGB2 byte
DARGB1	equ h'67'	; DARGB1 byte
DARGB0	equ	h'68'	; DARGB0 byte
DEXP	equ	h'69'	; 8-bit biased exponent for argument D

EARGB3	equ h'6A'	; needed by EXP1024, it seems

SIGN	equ	h'6B'	; location for saving sign in MSB
FPFLAGS	equ h'6C'	; floating point library exception flags
FPError	equ h'6D'	; floating point routine error code (FFh = error)

;	Fixed point storage locations

REMB3	equ h'70'	; remainder LSB
REMB2	equ h'71'	; remainder middle byte
REMB1	equ h'72'	; remainder upper middle byte
REMB0	equ h'73'	; remainder MSB
LOOPCOUNT	equ h'74'	; loop counter

;	retained only to keep assembler happy (float_ascii no longer used)
digit_count	equ h'7A'	; digit counter used by float_ascii
tenthous	equ h'7E'	; also needed by float_ascii

;       floating point library exception flag bits
;
IOV     equ     0	; bit0 = integer overflow flag
FOV     equ     1   ; bit1 = FP overflow flag
FUN     equ     2   ; bit2 = FP underflow flag
FDZ     equ     3   ; bit3 = FP divide by zero flag
NAN		equ		4	; bit4 = not-a-number exception flag
DOM		equ		5	; bit5 = domain error exception flag
RND     equ     6   ; bit6 = FP rounding flag
					; 0 = truncation
                    ; 1 = unbiased rounding to nearest LSB
SAT     equ     7   ; bit7 = FP saturate flag
					; 0 = term on exception w/out saturation
					; 1 = term on exception with saturation
					; to appropriate value

EXP		equ	AEXP 
TEMP	equ	TEMPB0

;	define assembler constants
B0		equ	0
B1		equ	1
B2		equ	2
B3		equ	3
B4		equ	4
B5		equ	5
B6		equ	6
B7		equ	7

MSB		equ	7
LSB		equ	0

;       Floating point literal constants
;
EXPBIAS   equ h'7F'		; = 127d, bias for FP exponents

;*************************************************************

;	Program itself now begins
;
	org	h'00'		; normal start vector
	GOTO Initialise
	org h'04'		; interrupt service vector
	GOTO IntService

Initialise:
	; first we set up CPU and INTOSC, I/O ports and ADC module
	bank0			; make sure we're set for bank 0
	CLRF INTCON		; disable interrupts
	Ppage0			; make sure PCLATH is set for page0
	CLRF CCP1CON	; disable the CCP module
	CLRF RCSTA		; and also the serial port
	CLRF PORTA		; clear PORTA
	CLRF PORTB		; also PORTB
	bank1			; then switch to bank1
	MOVLW h'70'		; set INTOSC for 8MHz
	MOVWF OSCCON
	CLRF OSCTUNE	; and also set to centre frequency
	CLRF PIE1		; turn off peripheral interrupts
	CLRF CVRCON		; now disable comparator Vref module
	MOVLW h'07'		; and disable the comparators
	MOVWF CMCON		; (by setting them for mode 111)
	MOVLW h'C0'		; then set RB0-RB5 as outputs, RB6-7 as inputs
	MOVWF TRISB
	MOVLW h'AD'		; also set RA0, RA2, RA3, RA7 as inputs,
	MOVWF TRISA		; RA1, RA4, RA6 as outputs
	MOVLW h'2C'		; then set RA2 to be AN2 input, RB6 to be AN5 input 
	MOVWF ANSEL		; and RA3 to be Vref+ input
	MOVLW h'E0'		; now set ADC for right justify, range 0V to Vref+
	MOVWF ADCON1	; and divide system clock by 2
	MOVLW h'87'		; then set Option reg for TMR0 + internal clock
	MOVWF OPTION_REG	; and prescaler division ratio of 256
	bank0			; then down to bank 0 again, so we can
	MOVLW h'11'		; now turn on ADC, make Tad = Toscx4, reset ADC
	MOVWF ADCON0	; and set AN2 as current analog input channel
	CALL SetDefs	; set defaults (HVC off, 10mA range, Ton = 10sec)
	CALL DispInit	; now initialise the LCD module
	CALL Display1	; and show greeting display
	CALL Wait2pt4sec	; then pause for 2.4 seconds
	CALL Display2	; now display 'Begin Test' information

Mainloop:
;	Main operating loop begins here. We first check if any of the
;	3 buttons is pressed, and if so perform requested task.
;	If no button is pressed, we check if test mode flag is set;
;	if not we call Display2 and loop back. But if flag is set,
;	we take a reading and display it before looping back.

	BTFSC PORTA,0		; first check if S4 is pressed (RA0 = 0)
	GOTO $+3			; if not, (RA0 = 1), move on
	CALL IncrTime		; but if S4 pressed go increase time
	GOTO BacktoDisp2	; then move on
	BTFSC PORTA,7		; now check if S5 is pressed (RA7 = 0)
	GOTO $+3			; if not (RA7 = 1), move on
	CALL DecrTime		; but if S5 pressed, go decrease time
	GOTO BacktoDisp2	; then move on
	BTFSC PORTB,7		; now check if S3 is pressed (RB7 = 0)
	GOTO $+2			; if not (RB7 = 1), move on
	CALL S3Pressed		; but if S3 pressed, go start or stop HVconv
	BTFSC Flags,2		; now see if test mode flag has been set
	GOTO OKtogo			; if it has, go take a reading and display	
BacktoDisp2:
	CALL Display2		; test mode flag not set, so just show
	GOTO Mainloop		; Display2 and loop back

OKtogo:
	CALL TakeaReading	; test mode flag set, so take a reading etc
	CALL ReadVolts		; also measure cap voltage 
	BTFSS Flags,1		; now check if HVconv is off or on
	GOTO $+3			; if it's off, go display via Display5 
	CALL Display3		; but if it's on, display via Display3
	GOTO $+2			; (Display3 shows Vtest ON + reading)
	CALL Display5		; (Display5 shows Vtest OFF + reading)
	CALL Delay160ms		; pause for 160ms to allow for button release
	GOTO Mainloop		; then loop back to begin again
 
;	main program loop ends -- subroutines follow
;
;	***********************************************************
;
Bin16toBCD:
;	routine to convert a 16-bit fixed point binary number in H_byte
;	and L_byte into a 5-digit BCD number, left in R0, R1 and R2
;	(Adapted from Microchip Technology B16TOBCD routine -- AN526)

	BCF STATUS,0			; first clear the carry bit
	MOVLW h'10'				; set count for 16d/10h
	MOVWF Count
	CLRF R0					; and clear the output registers
	CLRF R1
	CLRF R2
Loop16:
	RLF L_byte,F			; now rotate L_byte left through carry
	RLF H_byte,F			; and do the same with H_byte, from carry
	RLF R2,F				; then continue through R2
	RLF R1,F				; and R1
	RLF R0,F				; and finally R0

	DECFSZ Count,F			; decrement count, skip if it -> 0
	GOTO adjDEC				; otherwise keep going by going to adjDEC
	RETLW h'0'				; done all 16 bits, so leave

adjDEC:
	MOVLW R2				; 
	MOVWF FSR				;
	CALL adjBCD
; 
	MOVLW R1				;
	MOVWF FSR				;
	CALL adjBCD				;
; 
	MOVLW R0				;
	MOVWF FSR				;
	CALL adjBCD				;
;
	GOTO Loop16
;
adjBCD:
	MOVLW h'03'				; 
	ADDWF 0,W				;
	MOVWF Temp 
	BTFSC Temp,3			; test if result > 7
	MOVWF 0					; 
	MOVLW h'30'				; 
	ADDWF 0,W				;
	MOVWF Temp 
	BTFSC Temp,7			; test if result > 7
	MOVWF 0					; save as MSD
	RETLW h'0'
;	
;**************************************************************
;
ByteSplit:
	; routine to split a byte in w (value 0-99) into two decimal
	; digits which are placed into Byte10 and Byte1
	ANDLW h'7F'			; first make sure byte value is less than 128
	MOVWF Temp3			; then save it in Temp3
	CLRF Byte10			; initialise output digit values
	CLRF Byte1
Beg10:
	MOVLW h'0A'			; subtract 10d from input byte in Temp3
	SUBWF Temp3,0		; w = Temp3 - 0Ah
	BTFSS STATUS,C		;skip if we got a carry (Temp3 >= 10d)
	GOTO Beg1			;no carry, so Temp3 < 10d. Skip to Beg1
	BTFSS STATUS,Z		;carry was set, but check for zero
	GOTO $+3			;no zero, so continue
	INCF Byte10,1		;0 and C, so must have been 10 exactly
	RETURN				;so just increment 10s & leave
	INCF Byte10,1		;positive result, so increment 10s
	MOVWF Temp3			;save remainder in w back to Temp3
	GOTO Beg10			;and continue looping
Beg1:
	INCF Byte1,1		;increment units -- must have at least one
	DECF Temp3,1		;now decrement Temp3
	BTFSS STATUS,Z		;skip if zero bit is set now
	GOTO Beg1			;not zero yet, so continue
	RETURN				;zero, so we're done -- leave
;
;	**************************************************************
Check4FPE:
	; routine to check if floating point functions had errors
	IORLW h'00'			; check if w came back with 00h or FFh
	BTFSC STATUS,Z		; if Z=0, must have been FFh (= an error)
	RETURN				; if Z=1, must have been 00h (= no error)
	CALL ErrorDisp		; was an error, so display message
	RETURN				; and then return to resume anyway
;
;	***********************************************************
;
ClearLCD:
	;routine to clear LCD and reset address counter
	MOVLW h'01'			; clear display & reset addr ptr
	CALL DispAddress
	CALL Delay160ms		; pause 160ms to give it time to clear
	CALL Delay160ms		; and again, just for sloooow LCDs
	RETURN				; then return
;
;	************************************************************
;
DecrTime:
;	routine to decrement test voltage application time, if not already
;	at minimum value of 10 seconds. (Called when S5 pressed)

	MOVLW h'0A'			; first check if time isn't already 10sec
	XORWF Tonsec,0
	BTFSC STATUS,Z		; skip if Z=0 (because it wasn't 10d)
	GOTO Showandgo		; but if Z=1, it is 10d so don't increment
	MOVLW h'3C'			; it wasn't 10d, so see if it's 60min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (ie, Z=1)
	GOTO Isit30min		; otherwise go see if it's 30min
	MOVLW h'1E'			; it was 60min, so make it 30min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO Showandgo		; and display before returning
Isit30min:
	MOVLW h'1E'			; it wasn't 60min, so see if it's 30min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (ie, Z=1)
	GOTO Isit10min		; otherwise go see if it's 10min
	MOVLW h'0A'			; it was 30min, so make it 10min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO Showandgo		; and display before returning
Isit10min:
	MOVLW h'0A'			; it wasn't 30min, so see if it's 10min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO Isit3min		; otherwise go see if it's 3min
	MOVLW h'03'			; it was 10min, so make it 3min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO Showandgo		; and display before returning
Isit3min:
	MOVLW h'03'			; it wasn't 10min, so see if it's 3min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO Isit1min		; otherwise go see if it's 1min
	MOVLW h'01'			; it was 3min, so make it 1min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO Showandgo		; and display before returning
Isit1min:
	MOVLW h'01'			; it wasn't 3min, so see if it's 1min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO Mustbe30sec	; otherwise it must be 30sec
	MOVLW h'1E'   		; it was 1min, so make it 30sec
	MOVWF Tonsec
	CLRF Tonmin			; and zero the minutes
	GOTO Showandgo		; and display before returning
Mustbe30sec:
	MOVLW h'0A'			; must be 30sec (10sec already checked)
	MOVWF Tonsec		; so make it 10sec
	CLRF Tonmin			; and zero the minutes

Showandgo:
	CALL SetTimeDigs	; first work out the time display digits
	CALL Display4		; then show the resultant setting
	CALL Wait1sec		; then pause for 1 sec for user to see
	RETURN				; before returning 

;	***********************************************************	
	
Delay1ms:
	;routine to delay approx 1ms (2058 x 0.5us = 1029us) before return
	MOVLW h'0A'			; set Counter1 for 10 outer loops
	MOVWF Counter1
OuterLoop:
	MOVLW h'42'			; and Counter2 for 66 inner loops
	MOVWF Counter2		; (66 x 3mc = 198mc, + 7mc)
	DECFSZ Counter2, 1	; decrement Counter2 & skip when zero
	GOTO $-1			; not zero yet, so loop back
	DECFSZ Counter1, 1	; did reach zero, so decrement Counter1
	GOTO OuterLoop		; didn't hit zero, so loop back
	RETURN				; reached zero (10 x 66 loops) so return

Delay10ms:
	;routine to delay approx 10ms before returning
	MOVLW h'0A'			; set Counter3 for 10 outer loops
	MOVWF Counter3
	CALL Delay1ms		; then wait 1ms
	DECFSZ Counter3, 1	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return	
	
Delay160ms:
	;routine to delay approx 160ms before returning
	MOVLW h'A0'			; set Counter3 for 160 outer loops
	MOVWF Counter3
	CALL Delay1ms		; then wait 1ms
	DECFSZ Counter3, 1	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return	
	
DispAddress:
	;routine to translate & load display address (in w reg) into LCD
	BCF PORTB,5			; first set RS pin of LCD low, for instr/addr
	CALL Nibbles2LCD	; then send addr/cmd nibbles to LCD
	BCF PORTB,5			; make sure RS is is left low
	GOTO BusyWait		; then jump to delay 250us before return
	
DisplayData:
	;routine to display a data byte in w reg at the current LCD address
	BSF PORTB,5			; RS pin of LCD high, for sending data
	CALL Nibbles2LCD	; then send data nibbles to LCD

BusyWait:
	; routine to wait until LCD module is not busy, after writing
	MOVLW h'7D'			; set delay counter for 125 loops
	MOVWF Counter1		; (should give about 125 x 4 x 0.5 = 250us)
	NOP
	DECFSZ Counter1,1	; decrement counter & skip when zero
	GOTO $-2			; loop back until we reach zero
	RETURN				; then return

DispInit:
	; routine to initialise LCD display module
	BCF PORTB,4			; first make sure EN and RS lines are low
	BCF PORTB,5
	CALL Delay160ms		; then wait about 160ms before proceeding
	BSF PORTB,0			; now load init code 03h into RB0-3 
	BSF PORTB,1			; (= DB4 to DB7, so 03 -> 30h)
	ToggleEN			; then toggle EN to write to LCD
	CALL Delay10ms		; then wait about 10ms
	ToggleEN			; then toggle EN to write to LCD again
	CALL Delay10ms		; then wait about 10ms again
	ToggleEN			; then toggle EN to write to LCD again
	CALL Delay10ms		; then wait about 10ms a third time
	BCF PORTB,5			; make sure RS is still low
	BCF PORTB,0			; now change code in RB to 02 (-> 20h)
	ToggleEN			; then toggle EN to write to LCD
	CALL Delay10ms		; then wait about 10ms one last time

	MOVLW h'28'			; now set LCD functions (4 bit i/f, 2 lines, 5x10 chars)
	CALL DispAddress	; (also delays for 200us)
	MOVLW h'0C'			; also set display mode (disp on, no blink or cursor)
	CALL DispAddress	; (also delays for 200us)
	CALL ClearLCD		; then clear LCD screen (& delay 320ms)
	MOVLW h'06'			; set entry mode (increm addr, no shift)
	CALL DispAddress	; (also delays for 200us)
	RETURN				; should now be set up & ready to go, so leave

Display1:
	; routine to display initial greeting info on LCD
	MOVLW h'80'			; first set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "S"			; then send "Silicon Chip Cap"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "c"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "C" 
	CALL DisplayData
	MOVLW "h"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "p"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "C"
	CALL DisplayData
	MOVLW "a"
	CALL DisplayData
	MOVLW "p"
	CALL DisplayData
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW "R"			; and display "Reformer/Lkg Mtr"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "f"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "m"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "/"
	CALL DisplayData
	MOVLW "L"
	CALL DisplayData
	MOVLW "k"
	CALL DisplayData
	MOVLW "g"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "M"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	RETURN				; before leaving

Display2:
	; routine to display Test begin info on LCD
	MOVLW h'80'			; first set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "S"			; then send "Set Volts & Test"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "V"
	CALL DisplayData
	MOVLW "o"
	CALL DisplayData
	MOVLW "l"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "s" 
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "&"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "T"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW "T"			; and display "Time,Press Start"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "m"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW ","
	CALL DisplayData
	MOVLW "P"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "S"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "a"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	RETURN				; before leaving

Display3:
; 	routine to display measurement info on LCD while
;	testing voltage is ON
	MOVLW h'80'			; set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "V"			; then send "Vtest=ON  "
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "="
	CALL DisplayData
	MOVLW "O" 
	CALL DisplayData
	MOVLW "N"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVF TmrMtens,0		; followed by test time digits
	CALL DisplayData
	MOVF TmrMunits,0
	CALL DisplayData
	MOVLW "m"			; with units
	CALL DisplayData
	MOVF TmrStens,0
	CALL DisplayData
	MOVF TmrSunits,0
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
D3Line2:
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW "V"			; then show "Vc="
	CALL DisplayData
	MOVLW "c"
	CALL DisplayData
	MOVLW "="
	CALL DisplayData
	MOVF VDig1,0		; then give 3 volts digits
	CALL DisplayData
	MOVF VDig2,0
	CALL DisplayData
	MOVF VDig3,0
	CALL DisplayData
	MOVLW " "			; followed by a space
	CALL DisplayData
	MOVLW "I"			; then "Ic="
	CALL DisplayData
	MOVLW "c"
	CALL DisplayData
	MOVLW "="
	CALL DisplayData	
	MOVF IDig1,0		; followed by Ix digits
	CALL DisplayData
	MOVF IDig2,0
	CALL DisplayData
	MOVF IDig3,0
	CALL DisplayData
	MOVF IDig4,0
	CALL DisplayData
	MOVF IDig5,0		; IDig5 will be either "m" (default) or micro
	CALL DisplayData
	MOVLW "A"			; followed by "A"
	CALL DisplayData
	RETURN				; before leaving
;
Display4:
;	routine to display currently set testing period -- used while
;	incrementing or decrementing the period by pressing S4 or S5
	CALL ClearLCD		; first wipe the screen
	MOVLW h'80'			; next set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "V"			; then send "Vtest ON Time = "
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "O" 
	CALL DisplayData
	MOVLW "N"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData	
	MOVLW "T"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "m"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "=" 
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW " "
	CALL DisplayData
	MOVF TonMtens,0		; then display minutes digits
	CALL DisplayData
	MOVF TonMunits,0
	CALL DisplayData
	MOVLW " "			; followed by " min, "
	CALL DisplayData
	MOVLW "m"
	CALL DisplayData
	MOVLW "i"
	CALL DisplayData
	MOVLW "n"
	CALL DisplayData
	MOVLW ","
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVF TonStens,0		; followed by seconds digits
	CALL DisplayData
	MOVF TonSunits,0
	CALL DisplayData
	MOVLW " "			; and ending with " sec "
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "c"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	RETURN				; before returning

Display5:
;	routine to display measurement info after testing voltage
;	has been turned OFF (but still in testing mode)
	MOVLW h'80'			; set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "V"			; then send "Vtest=OFF "
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "e"
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	MOVLW "t"
	CALL DisplayData
	MOVLW "="
	CALL DisplayData
	MOVLW "O" 
	CALL DisplayData
	MOVLW "F"
	CALL DisplayData
	MOVLW "F"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVF TmrMtens,0		; followed by test timer digits
	CALL DisplayData
	MOVF TmrMunits,0
	CALL DisplayData
	MOVLW "m"			; with units
	CALL DisplayData
	MOVF TmrStens,0
	CALL DisplayData
	MOVF TmrSunits,0
	CALL DisplayData
	MOVLW "s"
	CALL DisplayData
	GOTO D3Line2		; then go to show line2 via Display3
	
ErrorDisp:
	; routine to display "FP Error" on LCD, then pause
	; and return (called only if FP functions flag an error)
	CALL ClearLCD
	MOVLW h'80'			; next set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "F"			; then send "FP Error!"
	CALL DisplayData
	MOVLW "P"
	CALL DisplayData
	MOVLW " "
	CALL DisplayData
	MOVLW "E"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "o" 
	CALL DisplayData
	MOVLW "r"
	CALL DisplayData
	MOVLW "!"
	CALL DisplayData
	CALL Wait2pt4sec	; now pause for 2.4 seconds for user to see
	CALL ClearLCD		; then wipe away again
	CALL Display2		; restore normal display
	RETURN				; and return

;	*************************************************************
;
IncrTime:
;	routine to increment time for test voltage application, if not
;	already at maximum value of 60 mins. (Called when S4 pressed)
	MOVLW h'3C'			; first check if time isn't already 60min
	XORWF Tonmin,0
	BTFSC STATUS,Z		; skip if Z=0 (because it wasn't 60d)
	GOTO ShowB4wego		; but if Z=1, it is 60d so don't increment
	MOVLW h'1E'			; it wasn't 60min, so see if is is 30min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (ie, Z=1)
	GOTO Try10min		; otherwise go see if it's 10min
	MOVLW h'3C'			; it was 30min, so make it 60min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO ShowB4wego		; and display before returning
Try10min:
	MOVLW h'0A'			; wasn't 30min, so see if it's 10min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO Try3min		; otherwise go see if it's 3min
	MOVLW h'1E'			; it was 10min, so make it 30min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO ShowB4wego		; and display before returning
Try3min:
	MOVLW h'03'			; wasn't 10min, so see if it's 3min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO Try1min		; otherwise go see if it's 1min
	MOVLW h'0A'			; it was 3min, so make it 10min
	MOVWF Tonmin
	CLRF Tonsec
	CALL SetTimeDigs	; then make the display digits
	GOTO ShowB4wego		; and display before returning
Try1min:
	MOVLW h'01'			; wasn't 3min, so see if it's 1min
	XORWF Tonmin,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO Try30sec		; otherwise go see if it's 30sec
	MOVLW h'03'			; it was 1min, so make it 3min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO ShowB4wego		; and display before returning
Try30sec:
	MOVLW h'1E'			; wasn't 1min, so see if it's 30sec
	XORWF Tonsec,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO Try10sec		; otherwise go see if it's 10sec
	MOVLW h'01'			; it was 30sec, so make it 1min
	MOVWF Tonmin
	CLRF Tonsec
	GOTO ShowB4wego		; and display before returning
Try10sec:
	MOVLW h'0A'			; wasn't 30sec, so see if it's 10sec
	XORWF Tonsec,0
	BTFSS STATUS,Z		; skip if it was (Z=1)
	GOTO ShowB4wego		; otherwise just ignore, show & go
	MOVLW h'1E'			; it was 10 sec, so make it 30sec
	MOVWF Tonsec
	CLRF Tonmin
ShowB4wego:
	CALL SetTimeDigs	; first work out the time display digits
	CALL Display4		; then show the resultant setting
	CALL Wait1sec		; then pause for 1 sec for user to see
	RETURN				; before returning 
;	
;	*************************************************************	
;
InitDig:
;	routine to initialise display digits for leakage current and
;	cap voltage (setting multiplier IDig5 according to the range)
	MOVLW h'30'			; first load VDig1-3 & IDig1-4 with zero code
	MOVWF VDig1
	MOVWF VDig2
	MOVWF VDig3
	MOVWF IDig1
	MOVWF IDig2
	MOVWF IDig3
	MOVWF IDig4
	BTFSS Flags,0		; then test range flag (bit 0), skip if 1
	GOTO LowRange		; if not set, go to low range setup
	MOVLW "m"			; on high range, so make IDig5 = "m"
	MOVWF IDig5
	RETURN				; then return
LowRange:
	MOVLW h'E4'			; on low range, so make IDig5 = micro symbol
	MOVWF IDig5
	RETURN				; before returning
;
;	***********************************************************
;
Nibbles2LCD:
	; routine to test bits of data byte (passed in w) then send
	; to LCD module as two nibbles (high nibble first)
	MOVWF Temp2			; first save byte to be sent in Temp2
	BTFSC Temp2,7		; now test bit 7 (upper nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,3			; if it's a 0, clear RB3
	GOTO $+2			; and proceed to next bit
	BSF PORTB,3			; was a 1, so set RB3 instead
	BTFSC Temp2,6		; now test bit 6
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,2			; if it's a 0, clear RB2
	GOTO $+2			; and proceed to next bit
	BSF PORTB,2			; was a 1, so set RB2 instead
	BTFSC Temp2,5		; now test bit 5
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,1			; if it's a 0, clear RB1
	GOTO $+2			; and proceed to next bit
	BSF PORTB,1			; was a 1, so set RB1 instead
	BTFSC Temp2,4		; now test bit 4
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,0			; if it's a 0, clear RB0
	GOTO $+2			; and proceed to next bit
	BSF PORTB,0			; was a 1, so set RB0 instead
	ToggleEN			; now toggle EN to write hi nibble to LCD

	BTFSC Temp2,3		; next test bit 3 (lower nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,3			; if it's a 0, clear RB3
	GOTO $+2			; and proceed to next bit
	BSF PORTB,3			; was a 1, so set RB3 instead
	BTFSC Temp2,2		; now test bit 2
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,2			; if it's a 0, clear RB2
	GOTO $+2			; and proceed to next bit
	BSF PORTB,2			; was a 1, so set RB2 instead
	BTFSC Temp2,1		; now test bit 1
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,1			; if it's a 0, clear RB1
	GOTO $+2			; and proceed to next bit
	BSF PORTB,1			; was a 1, so set RB1 instead
	BTFSC Temp2,0		; now test bit 0
	GOTO $+3			; if it's a 1, skip down
	BCF PORTB,0			; if it's a 0, clear RB0
	GOTO $+2			; and proceed to next bit
	BSF PORTB,0			; was a 1, so set RB0 instead
	ToggleEN			; toggle EN again to write low nibble to LCD
	RETURN				; and return, since both nibbles sent
;
;	************************************************************
;
ReadVolts:
;	routine to take a reading of capacitor voltage via analog input
;	AN5, to be displayed as VDig1-3, then change analog input back
;	to AN2 for further current measurements.
	BCF PIR1,6			; first clear ADIF bit in PIR1, just in case
	MOVLW h'69'			; next turn on ADC, make Tad = Tosc/16, reset ADC
	MOVWF ADCON0		; and set AN5 as current analog input channel
	CALL Delay1ms		; allow 1ms delay to cover acquisition	
	BSF ADCON0,2		; now start ADC conversion by setting GO bit
	BTFSC ADCON0,2		; then wait until conversion complete by
	GOTO $-1			; looping back until GO bit cleared again
	MOVF ADRESH,0		; done, so fetch high byte of ADC result into w
	MOVWF AARGB0		; and place into AARGB0
	bank1
	MOVF ADRESL,0		; then fetch low byte of result as well
	bank0
	MOVWF AARGB1		; and place it into AARGB1
	CALL FLO24			; then go convert it into 24-bit FP form
	CALL Check4FPE		; (duck away to check for any FP errors)
	MOVLW h'7E'			; AARG regs have ADC reading in 24b FP form,
	MOVWF BEXP			; so now load BARG regs with scaling factor
	MOVLW h'22'			; 7E22A9 = 0.6359125d
	MOVWF BARGB0		; (cf theoretical factor of 0.635386)
	MOVLW h'A9'
	MOVWF BARGB1
	CALL FPM24			; now do the scaling (AARG*BARG -> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARG regs have Vc reading (000 - 650) in 24b FP format
	CALL INT2416		; next convert into a 16-bit fixed point number
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARGB0 and AARGB1 now have scaled Vc as a 16-bit fixed-pt number
;
	copy AARGB0, H_byte		; next transfer these bytes to H_byte
	copy AARGB1, L_byte		; and L_byte, as input for the 
	CALL Bin16toBCD			; binary to BCD conversion routine
;
	MOVF R2,W			; now fetch LSD BCD digit into W
	GetASCII			; convert to ASCII
	MOVWF Ones			; and save in Ones
;
	SWAPF R2,W			; then repeat for the other digits
	GetASCII
	MOVWF Tens
;
	MOVF R1,W
	GetASCII
	MOVWF Hundreds
;
	SWAPF R1,W
	GetASCII
	MOVWF Thousands
;
	MOVLW h'30'			; now see if Hundreds is zero
	XORWF Hundreds,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process VDig1 normally
	MOVLW h'20'			; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Hundreds,W		; not a leading zero, so fetch into W
	MOVWF VDig1			; and save Hundreds (or blank) as VDig1

	MOVLW h'30'			; now check Tens to see if it's a zero
	XORWF Tens,W		; Z will again = 1 if it is
	BTFSS STATUS,Z		; skip if that's what you find
	GOTO $+7			; otherwise go to process VDig2 normally
	MOVLW h'20'			; it is zero, but see if VDig1 is blank
	XORWF VDig1,W		; Z will again = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process VDig2 normally
	MOVLW h'20'			; VDig1 is blank, so blank VDig2 also
	GOTO $+2
	MOVF Tens,W			; not another zero, so fetch Tens into W
	MOVWF VDig2			; and save Tens (or blank) as VDig2

	copy Ones, VDig3	; then just copy Ones into VDig3
;	all Volts digits have been refreshed, so restore ADC input to AN2
;	before we depart
	BCF PIR1,6			; clear ADIF bit in PIR1, just in case
	MOVLW h'11'			; next turn on ADC, make Tad = Tosc/4, reset ADC
	MOVWF ADCON0		; and reset AN2 as current analog input channel	

	RETURN				; then just return
;
;	***********************************************************
S3Pressed:
;	routine to respond when S3 button (Test Start/Stop) is pressed
;	If HV gen is off, it is turned on to begin testing. But if HV
;	gen is already on, will turn it off (readings continue)

	BTFSC Flags,1		; first check HV on/off flag (1 = on)
	GOTO Turnoff		; if it's already on, go turn off
TurnOn:
	HVon				; but if it's off, turn it and Tmr0 on
	copy Tonmin, TCmin	; by first setting timer's initial
	copy Tonsec, TCsec	; minutes & seconds from set values
	copy TonSunits, TmrSunits	; updating timer digits as well
	copy TonStens, TmrStens		; (from TonS digits)
	copy TonMunits, TmrMunits
	copy TonMtens, TmrMtens 
	MOVLW h'28'			; then initialise the 25ms count (28h = 40d)
	MOVWF TC25ms		; (chosen because 40 x 25.088ms = 1.0035s)
	MOVLW h'3C'			; then preload Tmr0 with 60d, for correct time
	MOVWF TMR0			; (also clears prescaler, ready to go)
	BSF Flags,2			; also set test mode flag bit
	CALL Waithalfasec	; and pause briefly to allow key return
	CLRF INTCON			; now clear all interrupt flags and
	BSF INTCON,5		; re-enable only TMR0IE interrupts
	BSF INTCON,7		; finally set GIE bit to turn on interrupts
	RETURN				; then return when finished

Turnoff:
	HVoff				; converter was on, so turn it off
	CLRF INTCON			; also turn off interrupts (including TMR0)
	CALL Waithalfasec	; and pause briefly to allow key return
	RETURN				; return when finished
;
;	***********************************************************
;
SetDefs:
;	routine to set defaults during startup
	SetHiRange			; first set for 10mA current range
	HVoff				; then make sure HV conv is off
	BCF Flags,2			; also make sure test mode is off as yet
	MOVLW h'0A'			; now set Tonsec to 10d (default)
	MOVWF Tonsec
	CLRF Tonmin			; and clear Tonmin so it's zero minutes
	CALL SetTimeDigs	; then go and set time display digits
	copy Tonsec,TCsec	; also copy time settings to TCsec
	copy Tonmin,TCmin	; and TCmin, also copying display digits
	copy TonSunits, TmrSunits
	copy TonStens, TmrStens
	copy TonMunits, TmrMunits
	copy TonMtens, TmrMtens 
	RETURN				; and finally return

;	***********************************************************
;
SetTimeDigs:
;	routine to work out time display digits from Tonmin and Tonsec
	MOVF Tonmin,0	; first check Tonmin in case it's zero
	BTFSC STATUS,Z	; if it's not zero (Z=1), skip
	GOTO ZeroMin	; but if it is zero, go set min digits to " 0"
	CALL ByteSplit	; not zero, so go convert into Byte10 & Byte1
	MOVF Byte10,0	; now fetch Byte10 (tens) into w
	GetASCII		; convert it to ASCII
	MOVWF TonMtens	; and save it in TonMtens
	MOVF Byte1,0	; now fetch Byte1 (units) into w
	GetASCII		; convert it to ASCII
	MOVWF TonMunits	; and save it in TonMunits
	GOTO CheckSecs
ZeroMin:
	MOVLW h'20'		; we have zero minutes, so set digits to " 0"
	MOVWF TonMtens
	MOVLW h'30'
	MOVWF TonMunits

CheckSecs:
	MOVF Tonsec,0	; now check the seconds, in case it's zero
	BTFSC STATUS,Z	; if it's not zero, skip
	GOTO ZeroSec	; but if it is zero, go set seconds digits to " 0"
	CALL ByteSplit	; not zero, so go convert into Byte10 & Byte1
	MOVF Byte10,0	; now fetch Byte10 (tens) into w
	GetASCII		; convert it to ASCII
	MOVWF TonStens	; and save it in TonStens
	MOVF Byte1,0	; now fetch Byte1 (units) into w
	GetASCII		; convert it to ASCII
	MOVWF TonSunits	; and save it in TonMunits
	RETURN			; all done, so return

ZeroSec:
	MOVLW h'20'		; we do have zero seconds, so set digits to " 0"
	MOVWF TonStens
	MOVLW h'30'
	MOVWF TonSunits
	RETURN			; all done, so return
;
;	***********************************************************
;
TakeaReading:
;	routine to use the PIC's ADC module to take 10 readings, which
;	are added together and then divided by 10 to get an average. Then 
;	we check to see if an uprange is needed (ADC average = 1023) or
;	a downrange is needed (ADC average <10). Then it upranges or
;	downranges if necessary, before taking another set of readings and
;	then (or otherwise) calculating the corresponding leakage current
;	(Ix) value ready for display.

	MOVLW h'0A'			; first set MLoopCtr for a loop/sequence
	MOVWF MLoopCtr 		; of 10 readings
	MOVLW h'76'			; then set ADCEXP, ADCB0 and ADCB1 for a near
	MOVWF ADCEXP		; zero initial number (76 03 13 = 0.0020000338,
	MOVLW h'03'			; which is close enough to zero for our purposes)
	MOVWF ADCB0
	MOVLW h'13'
	MOVWF ADCB1

MeasLoop:
	BCF PIR1,6			; first clear ADIF bit in PIR1, just in case
	BSF ADCON0,2		; then start ADC conversion by setting GO bit
	BTFSC ADCON0,2		; then wait until conversion complete by
	GOTO $-1			; looping back until GO bit cleared again
	MOVF ADRESH,0		; done, so fetch high byte of ADC result into w
	MOVWF AARGB0		; and place into AARGB0
	bank1
	MOVF ADRESL,0		; then fetch low byte of result as well
	bank0
	MOVWF AARGB1		; and place it into AARGB1
	CALL FLO24			; then go convert it into 24-bit FP form
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy ADCEXP,BEXP	; now AARG reg have ADC reading, so bring
	copy ADCB0,BARGB0	; accumulated readings into BARG regs
	copy ADCB1,BARGB1
	CALL FPA24			; now add the two together using FPA24
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy AEXP,ADCEXP	; then save sum back into ADC reading regs
	copy AARGB0,ADCB0	; (so sum of readings will be accumulating
	copy AARGB1,ADCB1	;  in these regs)
	DECFSZ MLoopCtr,1	; decrement loop counter, skip if it -> 0
	GOTO MeasLoop		; otherwise loop back for another reading

	SetBARG10			; done 10, so load dec10 (24bFP) into BARG regs
	copy ADCEXP, AEXP	; and bring sum of the 10 readings into AARG regs
	copy ADCB0,AARGB0
	copy ADCB1,AARGB1
	CALL FPD24			; then call FPD24 to divide the sum by 10
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy AEXP,ADCEXP	; then save average back into ADC reading regs
	copy AARGB0,ADCB0	; (so these regs will now have the average
	copy AARGB1,ADCB1	; of the 10 readings, in 24b FP form)

;	AARG regs still have 24b version of average reading, so now check
;	for full scale reading, in case upranging may be needed 
	MOVLW h'88'			; now load 24b FP version of 1023
	MOVWF BEXP			; (pre calc using Fprep) into BARG regs
	MOVLW h'7F'			; so it can be used as subtractant
	MOVWF BARGB0		; (to see if we should uprange)
	MOVLW h'C0'
	MOVWF BARGB1
	CALL FPS24			; now subtract BARG from AARG (-> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;						  AARG regs now have (ADC result - 1023)
	BTFSS AARGB0,7		; so check sign bit of AARGB0, skip if set (neg result)
	GOTO ChekHirange	; positive result, so we do have a FS reading
	copy ADCEXP,AEXP	; neg result, so bring back average ADC reading
	copy ADCB0,AARGB0	; into AARG regs
	copy ADCB1,AARGB1	
	SetBARG10			; and load BARG regs with 10d in 24b FP form
	CALL FPS24			; now subtract BARG from AARG (-> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;						  AARG regs now have (ADC result - 10)
	BTFSS AARGB0,7		; so check sign bit of AARGB0, skip if set (neg result)
	GOTO KeepGoing		; otherwise continue to process (reading > 10)

	BTFSS Flags,0		; reading is <10, so check if we are in low range
	GOTO KeepGoing		; flag clear, so OK to continue (in low range already)
	BCF Flags,0			; flag set, so clear it for 100uA range
	BSF PORTA,1			; also raise RA1 to turn off Q5 & RLY1
	CALL InitDig		; then re-initialise digits & multiplier
	GOTO TakeaReading	; and go back for another set of readings

ChekHirange:
	BTFSC Flags,0		; we have a FS reading, so see if we are in hi range
	GOTO KeepGoing		; flag is set, so we are already - OK to continue
	SetHiRange			; flag is clear, so swing up to hi (10mA) range
	GOTO TakeaReading	; and go back for another set of readings

KeepGoing:
;	reading average in normal value range or at least we're on the right
;	range, so proceed to process reading and calculate leakage current
	copy ADCEXP,AEXP	; first bring back ADC reading
	copy ADCB0,AARGB0	; into AARG regs
	copy ADCB1,AARGB1

	MOVLW h'7C'			; first set FP multiplier for scaling to
	MOVWF BEXP			; 0-199 (7C4ED7 = 0.20199204)
	MOVLW h'4E'			; (cf theoretical multiplier = 0.2019931)
	MOVWF BARGB0
	MOVLW h'D7'
	MOVWF BARGB1
	CALL FPM24			; now do the scaling (AARG*BARG -> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARG regs have scaled Ix reading (000 - 199) in 24b FP format

MakeIxDigits:
	CALL INT2416		; next convert into a 16-bit fixed point number
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARGB0 and AARGB1 now have scaled Ix as a 16-bit fixed-pt number
;
	copy AARGB0, H_byte		; next transfer these bytes to H_byte
	copy AARGB1, L_byte		; and L_byte, as input for the 
	CALL Bin16toBCD			; binary to BCD conversion routine
;
	MOVF R2,W			; now fetch LSD BCD digit into W
	GetASCII			; convert to ASCII
	MOVWF Ones			; and save in Ones
;
	SWAPF R2,W			; then repeat for the other signif digits
	GetASCII
	MOVWF Tens
;
	MOVF R1,W
	GetASCII
	MOVWF Hundreds
;
ChekRange:
	BTFSS Flags,0		; now are we on high range or low range?
	GOTO LowRng			; flag clear, so we're on low range. Go thataway
HiRng:
	MOVLW h'30'			; high range, so first see if Hundreds is zero
	XORWF Hundreds,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process IDig1 normally
	MOVLW h'20'			; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Hundreds,W		; not a leading zero, so fetch Hundreds into W
	MOVWF IDig1			; and save Hundreds (or blank) as IDig1
	copy Tens, IDig2	; then copy Tens into IDig2 (even if zero)
	MOVLW h'2E'			; IDig3 becomes the decimal point
	MOVWF IDig3
	copy Ones, IDig4	; and just copy Ones into IDig4
	MOVLW h'6D'			; making IDig5 an 'm'
	MOVWF IDig5
	RETURN				; then return to display, etc

LowRng:
	MOVLW h'20'			; on low range, so first make IDig1 a blank
	MOVWF IDig1
	MOVWF h'30'			; now check if Hundreds is zero
	XORWF Hundreds,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process IDig2 normally
	MOVLW h'20'			; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Hundreds,W		; not a leading zero, so fetch Hundreds into W
	MOVWF IDig2			; and save Hundreds (or blank) as IDig2

	MOVWF h'30'			; now check if Tens is zero
	XORWF Tens,W		; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+7			; otherwise go to process IDig3 normally
	MOVLW h'20'			; Tens = 0, but check if IDig2 is blank
	XORWF IDig2,W		; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process IDig3 normally
	MOVLW h'20'			; IDig2 is blank, so make IDig3 blank also
	GOTO $+2
	MOVF Tens,W			; not a leading zero, so fetch Tens into W
	MOVWF IDig3			; then save Tens (or blank) as IDig3

	copy Ones, IDig4	; and just copy Ones into IDig4 (even if zero)
	MOVLW h'E4'			; making IDig5 a 'mu'
	MOVWF IDig5
	RETURN				; then return to display, etc
;
;	***********************************************************
;		
Wait2pt4sec:
;	routine to pause for about 2.4 seconds (15 x 160ms = 2.4s)
	MOVLW h'0F'			; set Counter4 for 15d loops
	GOTO $+4
Wait1sec:
;	routine to pause for about 1 second (7 x 160ms = 1.12s)
	MOVLW h'07'			; set Counter4 for 7d loops
	GOTO $+2
Waithalfasec:
;	routine to pause for about 0.5 second (3 x 160ms = 0.48s)
	MOVLW h'03'			; set Counter4 for 3 loops
	MOVWF Counter4
	CALL Delay160ms		; then wait for 160ms
	DECFSZ Counter4,1	; now decrement Counter4, skip when zero
	GOTO $-2			; otherwise keep looping
	RETURN				; return when done

;	**********************************************************
;	
; 	end of main routines -- interrupt routine follows
;	
IntService:
	; routine to service interrupts from TMR0
	; (every 25.088ms when timer is going)
	; NOTE - has been modified to avoid making any subr calls

	MOVWF WSave			; first save context (w & status regs)
	SWAPF STATUS,W		; using SWAPF to avoid STATUS change
	CLRF STATUS			; (ensures bank0 inside int routine)
	MOVWF SSave			; so SSave has nibble-swapped STATUS saved
	MOVF PCLATH,W		; also save PC hi bits so we can restore
	MOVWF PCHiSave		; them when we return from interrupt
	CLRF INTCON			; now turn off all interrupts
	Ppage0				; also make sure we are in page 0
	bank0				; and in register bank 0

	DECFSZ TC25ms,F		; to begin decrement TC25ms, skip if it ->0
	GOTO ResetTmr		; not =0, so just reset Tmr0 & return

	MOVF TCsec,F		; TC25ms did -> 0, so check TCsec
	BTFSC STATUS,Z		; skip if Z=0, because TCsec not = 0
	GOTO $+3			; TCsec already = 0, so go check TCmin
	DECF TCsec,F		; decrement TCsec
	GOTO Reset25ms		; then go, just resetting TC25ms

	MOVF TCmin,F		; TCsec = 0, so check TCmin
	BTFSC STATUS,Z		; skip if Z=0, because TCmin not 0
	GOTO Timezup		; Tcmin already 0, so time is up
	DECF TCmin,F		; Tcmin not 0 yet, so decrement it
	GOTO ResetTCsec		; and go, resetting TCsec, TC25ms

Timezup:
	HVoff				; time's up, so turn off HV & clear flag
	GOTO LeaveAtEnd		; then just return from interrupt

ResetTCsec:
	MOVLW h'3B'			; reset TCsec to 59d
	MOVWF TCsec

Reset25ms:
	MOVLW h'28'			; reset TC25ms to 40d (=28h)
	MOVWF TC25ms		; (40 x 25ms = 1sec)
;
UpdateTime:
;	routine to update timer display digits from current timer
;	values TCmin and TCsec, before we return from interrupt
	MOVF TCmin,W	; first check TCmin in case it's zero
	BTFSC STATUS,Z	; if it's not zero (Z=1), skip
	GOTO MinZero	; but if it is zero, go set min digits to " 0"

;	this next section avoids a call to Bytesplit subroutine
	ANDLW h'7F'		; first make sure byte value is less than 128
	MOVWF Temp3		; then save it in Temp3
	CLRF Byte10		; initialise output digit values
	CLRF Byte1
StTc10m:
	MOVLW h'0A'			; subtract 10d from input byte in Temp3
	SUBWF Temp3,W		; w = Temp3 - 0Ah
	BTFSS STATUS,C		; skip if we got a carry (Temp3 >= 10d)
	GOTO StTc1m			; no carry, so Temp3 < 10d. Skip to StTc1m
	BTFSS STATUS,Z		; carry was set, but check for zero
	GOTO $+3			; no zero, so continue
	INCF Byte10,F		; 0 and C, so must have been 10 exactly
	GOTO DoneSplitM		; so just increment 10s & leave
	INCF Byte10,F		; positive result, so increment 10s
	MOVWF Temp3			; save remainder in w back to Temp3
	GOTO StTc10m		; and continue looping
StTc1m:
	INCF Byte1,F		;increment units -- must have at least one
	DECF Temp3,F		;now decrement Temp3
	BTFSS STATUS,Z		;skip if zero bit is set now
	GOTO StTc1m			;not zero yet, so continue
;	(end of section avoiding a call to Bytesplit)

DoneSplitM:
	MOVF Byte10,W	; now fetch Byte10 (tens) into w
	GetASCII		; convert it to ASCII
	MOVWF TmrMtens	; and save it in TmrMtens
	MOVF Byte1,W	; then fetch Byte1 (units) into w
	GetASCII		; convert it to ASCII
	MOVWF TmrMunits	; and save it in TmrMunits
	GOTO SecsChek
MinZero:
	MOVLW h'20'		; we have zero minutes, so set digits to " 0"
	MOVWF TmrMtens
	MOVLW h'30'
	MOVWF TmrMunits

SecsChek:
	MOVF TCsec,W	; now check the seconds, in case it's zero
	BTFSC STATUS,Z	; if it's not zero, skip
	GOTO SecZero	; but if it is zero, go set seconds digits to " 0"

;	this next section again avoids a call to Bytesplit subroutine
	ANDLW h'7F'		; first make sure byte value is less than 128
	MOVWF Temp3		; then save it in Temp3
	CLRF Byte10		; initialise output digit values
	CLRF Byte1
StTc10s:
	MOVLW h'0A'			; subtract 10d from input byte in Temp3
	SUBWF Temp3,W		; w = Temp3 - 0Ah
	BTFSS STATUS,C		; skip if we got a carry (Temp3 >= 10d)
	GOTO StTc1s			; no carry, so Temp3 < 10d. Skip to StTc1s
	BTFSS STATUS,Z		; carry was set, but check for zero
	GOTO $+3			; no zero, so continue
	INCF Byte10,F		; 0 and C, so must have been 10 exactly
	GOTO DoneSplits		; so just increment 10s & go to DoneSplits
	INCF Byte10,F		; positive result, so increment 10s
	MOVWF Temp3			; save remainder in w back to Temp3
	GOTO StTc10s		; and continue looping
StTc1s:
	INCF Byte1,F		;increment units -- must have at least one
	DECF Temp3,F		;now decrement Temp3
	BTFSS STATUS,Z		;skip if zero bit is set now
	GOTO StTc1s			;not zero yet, so continue
;	(end of second section avoiding a call to Bytesplit)
;
DoneSplits:
	MOVF Byte10,W	; now fetch Byte10 (tens) into w
	GetASCII		; convert it to ASCII
	MOVWF TmrStens	; and save it in TmrStens
	MOVF Byte1,W	; then fetch Byte1 (units) into w
	GetASCII		; convert it to ASCII
	MOVWF TmrSunits	; and save it in TmrMunits
	GOTO ResetTmr	; all done, so reset timer and return

SecZero:
	MOVLW h'20'		; we do have zero seconds, so set digits to " 0"
	MOVWF TmrStens
	MOVLW h'30'
	MOVWF TmrSunits

ResetTmr:
	; routine to reset Tmr0 for interrupts each 25.088ms
	MOVLW h'3C'			; preload Tmr0 with 60d, for correct time
	MOVWF TMR0			; (also clears prescaler, ready to go)
	BSF INTCON,5		; then re-enable only TMR0IE interrupts
;
LeaveAtEnd:
;	now we can restore pre-interrupt context and return
	MOVF PCHiSave,W		; restore saved PC hi bits
	MOVWF PCLATH		; to PCLATH
	SWAPF SSave,W		; then restore status reg (using SWAPF so
	MOVWF STATUS		; STATUS now has re-swapped saved nibbles)
	SWAPF WSave,F		; and also restore w reg (using mirror pair
	SWAPF WSave,W		; of SWAPFs to avoid changing STATUS reg)
	RETFIE				; & return with TOS->PC, 1->GIE (= INTCON 7)
;
;	*******************************************************
;	include floating point routines
;	(in FPRF24.TXT)
;
	#include 	<FPRF24.TXT>


 	END

