/*******************************************************************************************************************************
Main.c

Version 1.0   May 2010
Version 1.1   Nov 2010 - Enabled power up timer and brownup tiner to combat startup issues.
Version 1.2   Dec 2010 - Rearranged startup sequence in InitEverything() to minimise startup issues

Go to http://geoffg.net/boatcomputer.html for updates, errata and helpfull notes
    
    Copyright 2009, 2010 Geoff Graham - http://geoffg.net
    This program is free software: you can redistribute it and/or modify it under the terms of the GNU General
    Public License as published by the Free Software Foundation, either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.  You should have received a copy of the GNU General Public License along with this program.
    If not, see <http://www.gnu.org/licenses/>.


This is the main source file for the GPS Boat Computer project.  The other files are:
    CCS-18F4550.h           Defines specific to the configuration of the 18F4550
    SG122232A_Driver_2_0.h  Driver for the SG122232A graphics LCD
    bootloader.h            The bootloader code in compiled form
    glcd_library_1_0a.h     Graphics library (fonts and drawing routines)
    
    Two other files are required.  These configure the USB protocol stack for this project.  Because of licensing
    restrictions these are not included in this source distribution.  Instructions for creating them are listed
    in the file "Modifying CCS USB driver to suit the GPS Car Computer.pdf"

Development Environment
    To compile this you need:
     - CCS C Compiler Version 4.0.057 or higher (www.ccsinfo.com)
     - Microchip MPLAB IDE Version 8.14 or higher (www.microchip.com)

Program summary:
    The program consists of five main sections:
    
    Timer 0 interrupt:      This occurs every 85.3uS and is used count down the timeout used to detect a faulty gps 
    module, sensing/debounce for the three push button switches, control of the backlight brightness and control of 
    the beeper used to signal overspeed.
    
    UART Receive Interrupt: Occurs on every character received from the GPS.  This routine echoes the received character 
    to the USB virtual serial port then processes it.  This involves stripping the leading $ and stripping and checking
    the checksum.  The received string is stored in GpsBuf[2][] and GpsDataReady is updated (0=no data, 1=buffer1, 2=buffer2).
    While the main program process this data the receive interrupt will be filling the second buffer.  As a message from 
    the GPS takes about 160mS the main program must finish in less than that time and reset GpsDataReady to zero otherwise 
    the next message will be lost.
    
    USB Interrupt and code:     The interrupt occurs when processing of a send or receive is required.  USB processing 
    occurs entirly in the background and does not affect the rest of the program.
    
    Main Program Loop:      This runs continuously.  It checks for received data from the GPS and processes it.  It
    also runs a state machine that switches execution to various functions depending on the value of the current
    state.  Each state represents a different display that the user sees and the function called by the state machine
    is responsible for displaying the data and processing any pertinant button presses.  The state will be switched to a 
    different state if the user presses a button that calls up a new display.
    
    Graphics and LCD driver:    Drawing on the LCD is controlled by glcd_library_1_0a.h which includes lines, circles
    and three fonts.  These routines call SG122232A_Driver_2_0.h which maintains an image of the LCD in a buffer in RAM.
    When instructed the driver will dump the buffer into the LCD resulting in a very fast update.
    
    
********************************************************************************************************************************/


// define the version number shown on the startup screen
#define VERSION         "1.2"

// comment out the following if you do NOT want to use the Flow Sensor and use instead the fuel injector method
#define USE_FUEL_SENSOR

// the following controls the build and its use with the bootloader
//    USE_BOOTLOADER = 0    Will not include the bootloader and will run in low memory as normal.  Use a programmer to load.
//    USE_BOOTLOADER = 1    Will include the bootloader.  The result is a single HEX file for use with a programmer
//    USE_BOOTLOADER = 2    Will compile an update. This can only be loaded into flash by the bootloader.
#define USE_BOOTLOADER  2

// comment out the following line if you do not want the USB serial emulator
#define USE_USB         1                                   


/*******************************************************************************************************************************/

#include <18F4550.h>
#device ICD=TRUE
#device adc=8
#use delay(clock=48000000)

#include "CCS-18F4550.h"

#FUSES HSPLL                    //High speed Osc (> 4mhz for PCM/PCH) (>10mhz for PCD)
#FUSES NOWDT                    //No Watch Dog Timer
#FUSES NOPROTECT                //Code not protected from reading
#FUSES BROWNOUT               	//Brownout reset enabled
#FUSES BORV43					//Brownout at 4.3V
#FUSES PUT                    	//Power Up Timer enabled
#FUSES NOCPD                    //No EE protection
#FUSES STVREN                   //Stack full/underflow will cause reset
#FUSES NODEBUG                  //Debug mode for ICD
#FUSES NOLVP                    //No low voltage prgming, B3(PIC16) or B5(PIC18) used for I/O
#FUSES NOWRT                    //Program memory not write protected
#FUSES NOWRTD                   //Data EEPROM not write protected
#FUSES NOIESO                   //Internal External Switch Over mode disabled
#FUSES NOFCMEN                  //Fail-safe clock monitor disabled
#FUSES NOPBADEN                 //PORTB pins are configured as analog input channels on RESET
#FUSES NOWRTC                   //configuration not registers write protected
#FUSES NOWRTB                   //Boot block not write protected
#FUSES NOEBTR                   //Memory not protected from table reads
#FUSES NOEBTRB                  //Boot block not protected from table reads
#FUSES NOCPB                    //No Boot Block code protection
#FUSES MCLR                     //Master Clear pin enabled
#FUSES NOXINST                  //Extended set extension and Indexed Addressing mode disabled (Legacy mode)
#FUSES PLL5                     //Divide By 5(20MHz oscillator input)
#FUSES CPUDIV1                  //System Clock by 1
#FUSES USBDIV                   //USB clock source comes from PLL divide by 2
#FUSES VREGEN                   //USB voltage regulator disabled
#FUSES NOICPRT                  //ICPRT disabled


/************************************************************************************************
Define the i/o pins
************************************************************************************************/
#define     TRISA_INIT          0b00001101  // this MUST agree with the #defines below
#define     CMP_IN_NEG          RA0     // input analog - Comparator negative
#define     LCD_A0              RA1     // output - LCD A0
#define     REF_OUT             RA2     // output analog - reference voltage
#define     CMP_IN_POS          RA3     // input analog - Comparator positive
#define     CMP_OUT             RA4     // output - Comparator output
#define     BEEPER              RA5     // output - the beeper used to signal overspeed

#define     TST {BEEPER = 1; BEEPER = 0;}
#define     TRISB_INIT          0b11111000  // this MUST agree with the #defines below
                                        // pullups available (all or nothing)
#define     LCD_RES             RB0     // output - LCD reset
#define     LCD_E               RB1     // output - LCD enable (data strobe)
#define     LCD_RW              RB2     // output - LCD Read/Write
#define     SET_BTN             RB3     // input - the set button
#define     UP_BTN              RB4     // input - the up button
#define     DWN_BTN             RB5     // input - the down button
#ifdef USE_FUEL_SENSOR
	#define     FLOWSENSE           RB6     // input - fuel flow sensor -also- ICD PGD
#else
	#define     FUEL_INJECTOR       RB6     // input - engine's fuel injector solenoid -also- ICD PGD
#endif
#define     CAR_LIGHTS          RB7     // input - wired to the car's headlamps -also- ICD PGC

#define     TRISC_INIT          0b10001000  // this MUST agree with the #defines below
#define     LCD_CS1             RC0     // output - LCD Chip Select 1
#define     LCD_CLK             RC1     // output - LCD clock used to drive the glass
#define     LCD_BKLIT           RC2     // output - LCD backlight
#define     UNAVAILABLE         RC3     // Unavailable on 18F2550 and 18F4550
#define     USB_DN              RC4     // output - USB D-
#define     USB_DP              RC5     // output - USB D+
#define     GPS_TX              RC6     // output - Data to GPS
#define     GPS_RX              RC7     // input - GPS data

#define     TRISD_INIT          0b00000000  // this MUST agree with the #defines below
#define     D0                  RD0     // input & output - Data bus to LCD
#define     D1                  RD1     // input & output - Data bus to LCD
#define     D2                  RD2     // input & output - Data bus to LCD
#define     D3                  RD3     // input & output - Data bus to LCD
#define     D4                  RD4     // input & output - Data bus to LCD
#define     D5                  RD5     // input & output - Data bus to LCD
#define     D6                  RD6     // input & output - Data bus to LCD
#define     D7                  RD7     // input & output - Data bus to LCD

#define     TRISE_INIT          0b00000110  // this MUST agree with the #defines below
#define     LCD_CS2             RE0     // output - LCD Chip Select 2
#define     UNASSIGNED1         RE1     // input - 
#define     IGNITION	        RE2     // input - from the 12V to the engine, used for engine runtime

#define     LCD_DATA            PORTD           // Data port for the LCD
#define     LCD_DATA_TRIS       TRISD           // and its associated direction register    


// define debugging signals - not used in production code
#define Mark1               { UNASSIGNED1 = 1; UNASSIGNED1 = 0; }
#define Mark1double         { UNASSIGNED1 = 1; UNASSIGNED1 = 0; UNASSIGNED1 = 1; UNASSIGNED1 = 0; }
#define Mark2               { UNASSIGNED2 = 1; UNASSIGNED2 = 0; }
#define Mark2double         { UNASSIGNED2 = 1; UNASSIGNED2 = 0; UNASSIGNED2 = 1; UNASSIGNED2 = 0; }
#define Mark3               { UNASSIGNED3 = 1; UNASSIGNED3 = 0; }
#define Mark3double         { UNASSIGNED2 = 1; UNASSIGNED2 = 0; UNASSIGNED2 = 1; UNASSIGNED2 = 0; }


/************************************************************************************************
Define the various states that the display can be in.  This is part of the main State Machine

To add a new display screen you must add a state:
    1.  Insert the new state label in enum StateValues
    2.  Insert the actions for this state in the corresponding position in the array StateAction
    3.  If it is a show/hideable state:
            a.  Insert the descriptive string for the new state in M_set_showlist
            b.  add an entry to the show/hide table in the eeprom configuration data
            c.  Add one to the value of HIDE_SIZE
    4.  Write a subrouting to process the new state
    5.  Insert the subroutine call in the switch statement in main()
    
************************************************************************************************/

// Values used by the state machine
// This lists all the states that the display can be in
enum StateValues {
    S_Time,
    S_Speed,
    S_Fuel,
    S_Eng,
    S_Poi1,
    S_Poi2,
    S_Poi3,
    S_Poi4,
    S_Poi5,
    S_Poi6,
    S_Poi7,
    S_Poi8,
    S_Posit,
    S_Sat,                              // must be the last normal screen (reachable by UP/DOWN buttons)
    
    S_TimeCfg,
    S_RevVidCfg,
    S_FuelCfg1,
    S_FuelCfg2,
    S_FuelCal,
    S_EngCfg,
    S_PoiCfg1,
    S_BrightDayCfg,
    S_BrightNightCfg,
    S_LAST_ENTRY
} State;

// State Change Table
// this lists the reaction of the state to various button pushes
// this is in EXACTLY the same sequence as the enum table above
// each entry specifies the state to change to when the button is pressed
// entry 1 is the Set button, 2 is the Up button and 3 is the Down button
// using the current state means that nothing will happen
const uint8 StateAction[S_LAST_ENTRY][3] = {
    S_TimeCfg, S_Sat, S_Speed,                                  // S_Time state
    S_RevVidCfg, S_Time, S_Fuel,                                // S_Speed state
    S_FuelCfg1, S_Speed, S_Eng,                                 // S_Fuel state
    S_EngCfg, S_Fuel, S_Poi1,                                   // S_Eng state
    S_PoiCfg1, S_Eng, S_Poi2,                                   // S_Poi1 state
    S_PoiCfg1, S_Poi1, S_Poi3,                                  // S_Poi2 state
    S_PoiCfg1, S_Poi2, S_Poi4,                                  // S_Poi3 state
    S_PoiCfg1, S_Poi3, S_Poi5,                                  // S_Poi4 state
    S_PoiCfg1, S_Poi4, S_Poi6,                                  // S_Poi5 state
    S_PoiCfg1, S_Poi5, S_Poi7,                                  // S_Poi6 state
    S_PoiCfg1, S_Poi6, S_Poi8,                                  // S_Poi7 state
    S_PoiCfg1, S_Poi7, S_Posit,                                 // S_Poi8 state
    S_BrightDayCfg, S_Poi8, S_Sat,                              // S_Posit state
    S_BrightDayCfg, S_Posit, S_Time,                            // S_Sat state - this MUST be the last normal state
    ///////////////////    place all normal screens (reachable by UP/DOWN buttons) before here
    S_Time, S_TimeCfg, S_TimeCfg,                               // S_TimeCfg state
    S_RevVidCfg, S_RevVidCfg, S_RevVidCfg,                      // S_RevVidCfg called from S_Speed
    S_FuelCfg1, S_FuelCfg1, S_FuelCfg1,                         // S_FuelCfg1 state
    S_FuelCfg2, S_FuelCfg2, S_FuelCfg2,                         // S_FuelCfg1 state
    S_Fuel, S_FuelCal, S_FuelCal,               	            // S_FuelCfg1 state
    S_EngCfg, S_EngCfg, S_EngCfg,                               // S_EngCfg state
    S_PoiCfg1, S_PoiCfg1, S_PoiCfg1,                            // S_PoiCfg1 state
    S_BrightNightCfg, S_BrightDayCfg, S_BrightDayCfg,           // S_BrightDayCfg state
    S_Sat, S_BrightNightCfg, S_BrightNightCfg                   // S_BrightNightCfg state
};


/************************************************************************************************
Configuration defines
************************************************************************************************/
#define GPS_TIMEOUT         255                                 // each unit is 26.3mS
#define GPS_BUFF_SIZE       84                                  // two buffers of this size are created                                         
#define LOW_SIGNAL_HOLD     10                                  // wait this many messages before removing low sig msg
#define BEEP_ON             150                                 // length of the two overspeed beeps in mS
#define BEEP_GAP            200                                 // gap between the two overspeed beeps in mS

// "factory default" values used on initial programming and when reset all is selected
#define D_state             S_Speed                             // state
#define D_tz                80                                  // timezone in hours * 10
#define D_bright_day        100                                 // day brightness
#define D_bright_night      20                                  // night brightness
#define D_fuel              25                                  // default multiplier for the fuel display
#define D_pointer           1                                   // compass pointer, 0=north, 1=heading
#define D_AutoScan          0                                   // Auto scan, 0=off, 2=running
#define D_RunMinutes		0									// Start with zero engine run hours
#define D_RunHoursLB		0									// Start with zero engine run hours
#define D_RunHoursHB		0									// Start with zero engine run hours
#define D_RunD				0									// engine runtime reset day
#define D_RunM				0									// engine runtime reset month
#define D_RunY				0									// engine runtime reset year
#define D_FlowCount			0									// current flow count (starts at zero)
#ifdef USE_FUEL_SENSOR
	#define D_FlowCalLB			0x74								// Flow calibrate LSB
	#define D_FlowCalHB			0x04								//   "      "     MSB
#else
	#define D_FlowCalLB			0xE8								// Flow calibrate LSB
	#define D_FlowCalHB			0x03								//   "      "     MSB
#endif
#define D_RevVid			0									// Reverse video flag

#define D_hide_time         1                                   // show/hide for the clock
#define D_hide_speed        0                                   // show/hide for the speed
#define D_hide_fuel      	0                                   // show/hide for fuel
#define D_hide_engine  		0                                   // show/hide for engine
#define D_hide_poi1      	0                                   // show/hide for poi1
#define D_hide_poi2      	0                                   // show/hide for poi2
#define D_hide_poi3      	0                                   // show/hide for poi3
#define D_hide_poi4      	2                                   // show/hide for poi4
#define D_hide_poi5      	2                                   // show/hide for poi5
#define D_hide_poi6      	2                                   // show/hide for poi6
#define D_hide_poi7      	2                                   // show/hide for poi7
#define D_hide_poi8      	2                                   // show/hide for poi8
#define D_hide_latlong      1                                   // show/hide for latlong
#define D_hide_signal       1                                   // show/hide for the signal



/************************************************************************************************
Implement the build options (see #define USE_BOOTLOADER  in the beginning of this file)
************************************************************************************************/
#if USE_BOOTLOADER == 1
    #build(reset=0x800, interrupt=0x808)
    #org 0, 0x7ff 
    void bootloader_code(void) {
        #asm
            goto 0x6FC              // this is a "magic" address found by disassembly of the bootloader
            nop
            nop
            goto 0x808
        #endasm
    }
    #include "bootloader.h"         // this is the bootloader (in object form)
#endif

#if USE_BOOTLOADER == 2
    #build(reset=0x800, interrupt=0x808)
    #org 0, 0x7ff {}
#endif

#include "SG12232A_Driver_2_0.h"
#include "glcd_library_1_0a.h"

#ifdef USE_USB
    #include "usb_cdc.h"
#endif


/************************************************************************************************
Global memory locations
************************************************************************************************/
bit NewData;                                                // true when new data is available for display
uint8 GpsTimeout;                                           // used to count down the timeout for gps data

bit SetBtn, SetBtnEvent;                                    // true if Set button pressed
uint8 UpBtn, UpBtnEvent;                                    // count of the number of Up button presses
uint8 DwnBtn, DwnBtnEvent;                                  // count of the number of Down button presses
uint8 AutoScan;                                             // 0=off, 1=waiting for both buttons to release, 2=auto scan running
uint8 SecCnt;                                               // counts seconds in the main loop for timing Auto Scan

uint8 StrBuf[28];

bit NoSignal, Startup;
bit BkLightTime;                                            // true if backlight is controlled by the time
uint8 BkLightValue;                                         // used to control the duty cycle of the backlight
bit RevVid;													// used to reverse the display (white on black)

uint8 UpdateSecCnt;
uint32 FlowCount;
#ifdef USE_FUEL_SENSOR
	uint16 FlowPeriod;
#else
	uint16 FuelCnt;
	uint16 FlowRate1, FlowRate2;
#endif

sint8 CurrentPOI = 0;
uint8 day, month, year;										// date from the gps
uint8 hour, min, sec;                                       // time from the gps (utc)
sint8 tz;                                                   // timezone in hours*10
sint16 minutes;                                             // local time
uint8 h, m;                                                 // local time
bit pm;                                                     // local time
    
uint8 LatBuf[16], LongBuf[16];
uint8 Speed;
sint16 Heading;

uint8 Sv[12];
uint8 SvCnt, SatUsed, FoundSatCnt;

uint8 GpsBuf[2][GPS_BUFF_SIZE];
uint8 GpsDataReady;


/************************************************************************************************
Declare functions
************************************************************************************************/
void InitEverything(void);
void GetGPSData(void);
void GPS_SetupUART(void);
void DoSpeed(void);
void DoRevVid(void);
void DoMsg(void);
void DoKmRemaining(void);
void CalcTime(void);
void DoTime(void);
void DoTimeCfg(void);
void DoFuel(void);
void DoFuelCfg1(void);
void DoFuelCfg2(void);
void DoFuelCal(void);
void DoPosit(void);
void DoEng(void);
void DoEngCfg(void);
void DoPoi(uint8);
void DoPoiCfg1(void);
void DrawSat(void);
void DoSat(void);
void DoBrightDayCfg(void);
void DoBrightNightCfg(void);
void ExtractGpsData(void);
uint16 GetGpsInt(uint8 *p, uint8 i);
uint16 GetGpsDecimal(uint8 *p, uint8 i);
void GetTime(uint8 *p);
void GetLatLong(uint8 *p, uint8 *b);
void Center812(uint8 *p, uint8 x, uint8 y);
void Center1116(uint8 *p, uint8 x, uint8 y);
void SetShowHide(void);
void ResetAll(void);
void GetDate(uint8 *p);
uint32 ConvLatLong(uint8 *p);


/************************************************************************************************
Initialise the eeprom configuration data.
************************************************************************************************/
#define E_state             0                               // index values of the stored configuration data
#define E_tz                1
#define E_bright_day        2
#define E_bright_night      3
#define E_fuel              4
#define E_AutoScan          5
#define E_RunMinutes		6
#define E_RunHoursHB		7
#define E_RunHoursLB		8
#define E_FlowCount0		9
#define E_FlowCount1		10
#define E_FlowCount2		11
#define E_FlowCount3		12
#define E_FlowCalHB			13
#define E_FlowCalLB			14
#define E_RevVid			15

#define E_POIStartX			16
#define E_POIStartY			E_POIStartX + 3

#define HIDE_START          0x70                            // start of the show/hide array in EEPROM
#define HIDE_SIZE           14                               // size of the show/hide array
#define E_hide_time         HIDE_START
#define E_hide_speed        HIDE_START + 1
#define E_hide_fuel         HIDE_START + 2
#define E_hide_engine       HIDE_START + 3
#define E_hide_poi1         HIDE_START + 4
#define E_hide_poi2         HIDE_START + 5
#define E_hide_poi3         HIDE_START + 6
#define E_hide_poi4         HIDE_START + 7
#define E_hide_poi5         HIDE_START + 8
#define E_hide_poi6         HIDE_START + 9
#define E_hide_poi7         HIDE_START + 10
#define E_hide_poi8         HIDE_START + 11
#define E_hide_latlong      HIDE_START + 12
#define E_hide_signal       HIDE_START + 13

#define BACKUP_START        0x80

#if HIDE_START + HIDE_SIZE > BACKUP_START
	#ERROR	Hide array too large - Not enough eeprom space.
#endif


// initialise the eeprom default values for the configurable items
#ROM int8 0xF00000 = {  
                        D_state,
                        D_tz,
                        D_bright_day,
                        D_bright_night,
                        D_fuel,
                        D_AutoScan,
                        D_RunMinutes,
						D_RunHoursHB,
						D_RunHoursLB,
						D_FlowCount,
						D_FlowCount,
						D_FlowCount,
						D_FlowCount,
						D_FlowCalHB,
						D_FlowCalLB,
						D_RevVid,
						0, 0, 0, 0, 0, 0, 				// POI 1
						0, 0, 0, 0, 0, 0, 				// POI 2
						0, 0, 0, 0, 0, 0, 				// POI 3
						0, 0, 0, 0, 0, 0, 				// POI 4
						0, 0, 0, 0, 0, 0, 				// POI 5
						0, 0, 0, 0, 0, 0, 				// POI 6
						0, 0, 0, 0, 0, 0, 				// POI 7
						0, 0, 0, 0, 0, 0 				// POI 8
                    }   
                                        
// initialise the eeprom default values for the show/hide table
#ROM int8 0xF00000 + HIDE_START = {
                        D_hide_time,
                        D_hide_speed,
                        D_hide_fuel,
                        D_hide_engine,
                        D_hide_poi1,
                        D_hide_poi2,
                        D_hide_poi3,
                        D_hide_poi4,
                        D_hide_poi5,
                        D_hide_poi6,
                        D_hide_poi7,
                        D_hide_poi8,
                        D_hide_latlong,
                        D_hide_signal
                        }

// initialise the eeprom backup values for the configurable items
#ROM int8 0xF00000 + BACKUP_START = {
                        D_state,
                        D_tz,
                        D_bright_day,
                        D_bright_night,
                        D_fuel,
                        D_AutoScan,
                        D_RunMinutes,
						D_RunHoursHB,
						D_RunHoursLB,
						D_FlowCount,
						D_FlowCount,
						D_FlowCount,
						D_FlowCount,
						D_FlowCalHB,
						D_FlowCalLB,
						D_RevVid,
						0, 0, 0, 0, 0, 0, 				// POI 1
						0, 0, 0, 0, 0, 0, 				// POI 2
						0, 0, 0, 0, 0, 0, 				// POI 3
						0, 0, 0, 0, 0, 0, 				// POI 4
						0, 0, 0, 0, 0, 0, 				// POI 5
						0, 0, 0, 0, 0, 0, 				// POI 6
						0, 0, 0, 0, 0, 0, 				// POI 7
						0, 0, 0, 0, 0, 0 				// POI 8
                    }   
                                        
// initialise the eeprom backup values for the show/hide table
#ROM int8 0xF00000 + BACKUP_START + HIDE_START = {
                        D_hide_time,
                        D_hide_speed,
                        D_hide_fuel,
                        D_hide_engine,
                        D_hide_poi1,
                        D_hide_poi2,
                        D_hide_poi3,
                        D_hide_poi4,
                        D_hide_poi5,
                        D_hide_poi6,
                        D_hide_poi7,
                        D_hide_poi8,
                        D_hide_latlong,
                        D_hide_signal
                        }
               

/************************************************************************************************
To save space we store strings in the program flash
************************************************************************************************/
const uint8     M_SiliconChip[] = {"BOAT COMPUTER"};

#ifdef USE_FUEL_SENSOR
	const uint8     M_Version[]     = {"VERSION "VERSION"A"};
#else
	const uint8     M_Version[]     = {"VERSION "VERSION"B"};
#endif

const uint8     M_srch[]        = {"SEARCHING"};
const uint8     M_NoSignal[]    = {"LOW SIGNAL"};
const uint8		M_Satfound[]	= {" SATELLITES"};
const uint8     M_err[]         = {"ERROR"};
const uint8		M_yes[]			= {"YES"};
const uint8		M_no[]			= {"NO"};
const uint8     M_gpsfault[]    = {"GPS FAULT"};
const uint8     M_kn[]          = {"KN"};
const uint8		M_reverse[]		= {"REVERSE\nDISPLAY? "};
const uint8     M_settime[]     = {"SET LOCAL TIME"};
const uint8		M_engine[]  	= {"ENGINE HOURS"};
const uint8		M_enginechoice[]= {"RESET HOURS\nTO ZERO? "};
const uint8		M_fueltitle[]	= {"FUEL/H   TOTAL"};
const uint8		M_fuelreset[]   = {"RESET TOTAL\nTO ZERO? "};
const uint8		M_fuelchoice[]	= {"CALIBRATE\nFUEL USED? "};
const uint8		M_fuelused[]	= {"SET FUEL USED"};
const uint8		M_poinotset[]	= {"NOT SET"};
const uint8		M_poinew[]		= {"SET POI TO THIS\nLOCATION? "};
const uint8		M_dashes[]		= {">100"};
const uint8     M_bright_day[]  = {"DAY BACKLIGHT"};
const uint8     M_bright_night[]= {"NIGHT BACKLIGHT"};
const uint8     M_set_showlist[]= {"CLOCK\0SPEED\0FUEL\0ENGINE HRS\0POI 1\0POI 2\0POI 3\0POI 4\0POI 5\0POI 6\0POI 7\0POI 8\0LAT/LONG\0SIGNAL"};
const uint8     M_set_show[]    = {"SHOW"};
const uint8     M_set_show_auto[]={"AUTO SCAN HIDE"};
const uint8     M_set_hide[]    = {"ALWAYS HIDE"};
const uint8     M_gps_reset[]   = {"$PSRF104,00,00,00,00,00,00,12,08*29\r\n"};



/************************************************************************************************
timer0 interrupt - this occurs every 85.3uS
************************************************************************************************/
#int_timer0
void timer0_interrupt() {
    static uint8 scale = 0;
    static uint8 SetBtnCnt = 3;
    static uint8 UpBtnCnt = 0;
    static uint8 DwnBtnCnt = 0;
    static uint8 HalfSecCnt = 0;
    uint8 i;
    
    #ifndef USE_FUEL_SENSOR
    	if(!FUEL_INJECTOR) FuelCnt++;                               // count if the fuel injectors are open
    #endif
    
    RBPU = 0;                                                       // turn pullups on while checking the buttons
    if(SET_BTN)  SetBtnCnt = 3;                                     // check the set button
    if(UP_BTN) UpBtnCnt = 0;                                        // and the up button
    if(DWN_BTN) DwnBtnCnt = 0;                                      // and down
    RBPU = 1;                                                       // turn off pullups, they interfere with injector detect
    
    if(++scale == 0) {                                              // this code is excuted once every 21.8 mS
        if(SetBtnCnt == 1) SetBtnEvent = true;
        if(SetBtnCnt) SetBtnCnt--;
        
        if(UpBtnCnt++ == 2) UpBtnEvent++;                           // first step
        if(UpBtnCnt == 55) { UpBtnEvent++; UpBtnCnt = 52; }         // after 2 sec do 10 steps per sec
        if(DwnBtnCnt++ == 2) DwnBtnEvent++;                         // first step
        if(DwnBtnCnt == 55) { DwnBtnEvent++; DwnBtnCnt = 52; }      // after 2 sec do 10 steps per sec

        // this section checks if auto scan has been triggered by pressing both up and down buttons simultaneously
        // auto scan will automatically step to the next display every 3 seconds and is cancelled by pressing up or down
        // The AutoScan variable has three states.  0 = not running
        //                                          1 = waiting for both buttons to be released
        //                                          2 or greater = auto scan running
        if(State <= S_Sat) {        
            if(!UP_BTN && !DWN_BTN && (UpBtnEvent || DwnBtnEvent))
                AutoScan = 1;
                
            if(AutoScan == 1)
                if(!UP_BTN || !DWN_BTN)
                    UpBtnEvent = DwnBtnEvent  = 0;
                else {
                    SecCnt = 0;
                    AutoScan = 2;
                }   
            
            if(AutoScan >= 2 && (UpBtnEvent || DwnBtnEvent || SetBtnEvent))
                SecCnt = AutoScan = UpBtnEvent = DwnBtnEvent  = 0;
        }
        else
            AutoScan = 0;   
            
        if(GpsTimeout) 
            GpsTimeout--;
                
        // set up the brightness of the backlight depending on the control mode and day/night
        // this routine makes a slow adjustment (1 percentage point every half second)
        // the result is saved in BkLightValue which is used above for PWM control
        HalfSecCnt++;
        if(HalfSecCnt > 22) {                               // this executes every 500mS
            HalfSecCnt = 0;
            if((BkLightTime && ((h < 6 && pm) || (h >= 6 && !pm))) || (!BkLightTime && !CAR_LIGHTS))
                i = E_bright_day;
            else
                i = E_bright_night;
            if(eeprom_read(i) > BkLightValue) BkLightValue++;
            if(eeprom_read(i) < BkLightValue) BkLightValue--;

        }
    set_pwm1_duty(BkLightValue);                    // 0 for full off, 100 for full on
    }   
}   




/************************************************************************************************
timer1 interrupt - this occurs every 10mS
************************************************************************************************/
#ifdef USE_FUEL_SENSOR
	#int_timer1
	void timer1_interrupt() {
		static uint16 period = 1400;
		static bit last = 0;
		
		// reset the timer with -1500 which, with a x8 prescale, will interrupt every 10mS
		TMR1H = 0xc5;
		TMR1L = 0x68 - TMR1L;
	
		if(period > 1500) 
			FlowPeriod = 0;					// after 15 seconds signal that flow is stopped
		else
			period++;
		
		if(last == 1 && FLOWSENSE == 0) {
			//FlowPeriod = period;																										// No averaging
			//FlowPeriod = (FlowPeriod + period) >> 1;																					// Averages the last 2 pulses
			FlowPeriod = (FlowPeriod + FlowPeriod + FlowPeriod + period) >> 2;															// Averages the last 4 pulses
			//FlowPeriod = (FlowPeriod + FlowPeriod + FlowPeriod + FlowPeriod + FlowPeriod + FlowPeriod + FlowPeriod + period) >> 3;	// Averages the last 8 pulses
			FlowCount++;
			period = 0;
		}
		last = FLOWSENSE;
	}		
#endif
		





/************************************************************************************************
UART receive interrupt
************************************************************************************************/
#int_rda
void rds_interrupt() {
    #define GET_1ST_CKSUM       253
    #define GET_2ND_CKSUM       254
    #define GET_WAITING         255
    
    static uint8 GpsBufSelect = 0;
    static uint8 CharCnt = GET_WAITING;
    static uint8 GpsChecksum;
    uint8 ch;
    
    // look for a framing or overrun error and clear
    if(OERR) { CREN = OFF; CREN = ON; ch = RCREG; ch = RCREG; CharCnt = GET_WAITING; }
    if(FERR) { ch = RCREG; ch = RCREG; CharCnt = GET_WAITING; }
    
    ch = (RCREG & 0x7f);                                    // get the character
    
    #ifdef USE_USB
        if(usb_enumerated()) usb_cdc_putc_fast(ch);         // send to the USB if we are connected
    #endif
    
    if(ch == 0x0d) CharCnt = GET_WAITING;
    
    if(ch == '$') 
        CharCnt = GpsChecksum = 0;
    else if(CharCnt != GET_WAITING) {
        if(ch == '*') {                                     // signals start of checksum
            GpsBuf[GpsBufSelect][CharCnt] = 0;
            CharCnt = GET_1ST_CKSUM;
        } else if(CharCnt == GET_1ST_CKSUM) {               // first checksum byte
            CharCnt = GET_2ND_CKSUM;
            ch = ch - '0';
            if(ch > 9) ch -= 'A' - 58;                      // convert from hex
            if(ch != (GpsChecksum >> 4)) CharCnt = GET_WAITING;     // and test
        } else if(CharCnt == GET_2ND_CKSUM) {               // second checksum byte
            ch = ch - '0';
            if(ch > 9) ch -= 'A' - 58;                      // convert from hex
            if(ch == (GpsChecksum & 0x0f)) {                // and test
                if(GpsDataReady == 0) {
                    GpsBufSelect++;
                    GpsDataReady = GpsBufSelect;            // success
                    if(GpsBufSelect > 1) GpsBufSelect = 0;
                }   
                GpsTimeout = GPS_TIMEOUT;
            }   
            CharCnt = GET_WAITING; 
        } else {
            GpsChecksum = GpsChecksum ^ ch;
            if(ch >= 'a') ch -= 'a' - 'A';                  // convert to uppercase
            GpsBuf[GpsBufSelect][CharCnt] = ch; CharCnt++;
            if(CharCnt >= GPS_BUFF_SIZE) CharCnt = GET_WAITING;
        }   
    }   
}   
    


/**********************************************************************************************
Main program
This first initialises everything then enters an endlass loop which consists of:
  -  Checking if a char has arrived over the USB and if so, sends it to the GPS
  -  If there is data from the GPS it decodes that data and loads it into global variables
  -  Calculates the Km and time remaining
  -  Calls the display handler for the current page on the LCD.  This and the handling of the 
     button presses is implemented in a state machine.
This loop executes at high speed (approx 5uS when there is nothing to be done)
**********************************************************************************************/
void main() {
    uint8 NewState;

    InitEverything();
    
    while(forever) {
        #ifdef USE_USB
            usb_task();
            if (usb_enumerated() && usb_cdc_kbhit() && TRMT) TXREG = usb_cdc_getc();
        #endif
        if(GpsDataReady) GetGpsData();

        if(NewData) if(SecCnt++ == 3) SecCnt = 1;                       // used to determine when to save the state in eeprom

        // check if we have an error and if so, display an appropiate message
        // there are three possible errors
        //  - GPS module failure (GpsTimeout == 0)
        //  - Startup (Startup = true).  Not strictly an error but displayed while finding satellites
        //  - Low signal (NoSignal = true)
        if(GpsTimeout == 0 || Startup || NoSignal) {
			RBPU = 0;                                                   // turn pullups on while checking the buttons
            if((Startup || NoSignal) && !SET_BTN)
                DoSat();                                                // show the satellite signal screen if Set pressed
            else {
                glcd_fillScreen(RevVid);
                
                // 
                if(GpsTimeout == 0)
                    strcpy(StrBuf, M_err);                              // timeout error (ie, gps not working)
                else if(Startup)
                    strcpy(StrBuf, M_srch);                             // starting up so we are searching
                else
                    strcpy(StrBuf, M_NoSignal);                         // low signal error
                Center812(StrBuf, 60, 0);                               // print the first line
                
                if(GpsTimeout == 0)
                    strcpy(StrBuf, M_gpsfault);
                else {
                    sprintf(StrBuf, "%2u", FoundSatCnt);
	                strcpy(StrBuf + 2, M_Satfound);
                }    
                Center812(StrBuf, 60, 17);                              // print the second line
                
                glcd_update();
                SecCnt = 0;
            }   
            RBPU = 1;													// turn off pullups
            SecCnt = SetBtnEvent = UpBtnEvent = DwnBtnEvent = 0;
        } 
        // if not an error we can display the data
        else {
            // First section of the state machine
            // Refer to the StateValue and StateAction tables defined in the front of this file
            // Here we process any button presses that will cause a change of state.  To do this we look up the 
            // current state and button that has been pressed to determine what the new state is.  If the new 
            // state is hidden we repeat the process for the hidden state until we reach a non hidden state.
            do {
                if(SetBtnEvent) {                                       // Set button
                    NewState = StateAction[State][0];
                    SetBtn = true;
                    SecCnt = 0;
                }   
                else if(UpBtnEvent) {                                   // Up button
                    NewState = StateAction[State][1];
                    UpBtn = UpBtnEvent;
                    SecCnt = 0;
                }   
                else if(DwnBtnEvent || (NewData && SecCnt == 2 && AutoScan == 2)) { // Down button or auto scan
                    NewState = StateAction[State][2];
                    if(DwnBtnEvent) SecCnt = 0;
                    DwnBtn = DwnBtnEvent;
                }
                else
                    break;
                
                // we have the new state.  Switch to it if it is NOT hidden, otherwise do the loop again for the new state
                State = NewState;
                if(State > S_Sat || (AutoScan == 0 && eeprom_read(HIDE_START + State) < 2) || (AutoScan >= 2 && eeprom_read(HIDE_START + State) < 1)) {
                    NewData = true;
                    SetBtnEvent = UpBtnEvent = DwnBtnEvent = 0;
                    break;
                }   
            } while (forever);
            
            // Second section of the state machine
            // Depending on the current state we call the appropiate function to display the data or process button presses.
            // Because the main loop executes very rapidly the function is called thousands of times a second.
            // It is up to the function to determine if it has to do anything by checking for various button presses, etc.
            switch(State) {
                case S_Time:            DoTime(); break;
                case S_TimeCfg:         DoTimeCfg(); break;
                case S_Speed:           DoSpeed(); break;
                case S_RevVidCfg:       DoRevVid(); break;
                case S_Fuel:            DoFuel(); break;
                case S_FuelCfg1:        DoFuelCfg1(); break;
                case S_FuelCfg2:        DoFuelCfg2(); break;
                case S_FuelCal:         DoFuelCal(); break;
                case S_Eng:             DoEng(); break;
                case S_EngCfg:          DoEngCfg(); break;
                case S_Poi1:            DoPoi(0); break;
                case S_Poi2:            DoPoi(1); break;
                case S_Poi3:            DoPoi(2); break;
                case S_Poi4:            DoPoi(3); break;
                case S_Poi5:            DoPoi(4); break;
                case S_Poi6:            DoPoi(5); break;
                case S_Poi7:            DoPoi(6); break;
                case S_Poi8:            DoPoi(7); break;
                case S_PoiCfg1:         DoPoiCfg1(); break;
                case S_Posit:           DoPosit(); break;
                case S_Sat:             DoSat(); break;
                case S_BrightDayCfg:    DoBrightDayCfg(); break;
                case S_BrightNightCfg:  DoBrightNightCfg(); break;
                }
            
            // Write the current state and autoscan status to eeprom.  This means that the state can be restored.
            // on power up. This code waits for 3 seconds before writing in case the change of state was caused by power off.
            if(NewData && SecCnt == 2) {
                if((State != eeprom_read(E_state)) && AutoScan == 0) eeprom_write(E_state, State);
                if(AutoScan != 1 && eeprom_read(E_AutoScan) != AutoScan) eeprom_write(E_AutoScan, AutoScan);
            }   
            NewData = SetBtn = UpBtn = DwnBtn = 0;                  // clear all flags
        }
    }                                                               // and continue looping forever
}
    
void InitEverything(void) {
    
    setup_adc_ports(NO_ANALOGS|VSS_VDD);
    setup_adc(ADC_OFF);
    setup_psp(PSP_DISABLED);
    setup_spi(SPI_SS_DISABLED);
    setup_wdt(WDT_OFF);
   
	// set up the outputs before we turn them on
    PORTE = PORTD = PORTC = PORTB = PORTA = 0;				
    LCD_CS2 = LCD_CS1 = LCD_A0 = LCD_RW = 1;
    LCD_E = 0;
    LCD_RES = 0;
    LCD_DATA = 0xff;
    
    // set pin directions
    TRISA = TRISA_INIT;
    TRISB = TRISB_INIT;
    TRISC = TRISC_INIT;
    TRISD = TRISD_INIT;
    TRISE = TRISE_INIT;

	mSec(20);												// let everything settle
    CVRCON = 0b11100110;                                    // voltage reference ON, output on RA2, 1.35V with 5.3V Vdd
    CMCON =  0b00000001;                                    // comparator 1 ON as independent with output
    
    UpdateSecCnt = 0;
    FlowCount = make32(eeprom_read(E_FlowCount3), eeprom_read(E_FlowCount2), eeprom_read(E_FlowCount1), eeprom_read(E_FlowCount0));
 
    GpsDataReady = 0;
    Heading = 0;
    NewData = false;
    GpsTimeout = GPS_TIMEOUT;
    FoundSatCnt = SatUsed = SvCnt = 0;
    BkLightValue = 50;
    RevVid = eeprom_read(E_RevVid);
    NoSignal = false;
    Startup = true;
    GPS_SetupUART();
    GpsDataReady = 0;
    SetBtnEvent = false;
    UpBtn = UpBtnEvent = 0;
    DwnBtn = DwnBtnEvent = 0;

    // get the current state and timezone from the eeprom
    AutoScan = eeprom_read(E_AutoScan);
    State = eeprom_read(E_state);
    tz = eeprom_read(E_tz);

    setup_timer_2(T2_DIV_BY_16,99,2);                       // setup timer 2 for PWM
    setup_ccp1(CCP_PWM);                                    // enable PWM
    setup_ccp2(CCP_PWM);                                    // enable PWM
    set_pwm1_duty(100);
    set_pwm2_duty(50);

    LCD_RES = 1;                                            // release the LCD from reset
	mSec(100);												// let everything settle
    GLCD_Init();                                            // LCD initialise
    glcd_fillScreen(RevVid);                                // clear our image buffer
    
    setup_timer_0(RTCC_INTERNAL | RTCC_8_BIT | RTCC_DIV_4);
    setup_timer_3(T3_DISABLED|T3_DIV_BY_1);
  
    #ifdef USE_FUEL_SENSOR
	    setup_timer_1(T1_INTERNAL|T1_DIV_BY_8);
	    enable_interrupts(INT_TIMER1);
    	disable_interrupts(GLOBAL);
	#else
    	setup_timer_1(T1_DISABLED|T1_DIV_BY_1);
	#endif

    enable_interrupts(INT_TIMER0);
    disable_interrupts(GLOBAL);
    enable_interrupts(INT_RDA);
    disable_interrupts(GLOBAL);

    RBPU = 0;                                               // turn on pullups
    if(!UP_BTN) SetShowHide();                              // Up button pressed, goto Show-Hide mode
    if(!DWN_BTN) {                                          // Down button pressed
        glcd_update();                                      // send a blank screen to the LCD
        ResetAll();                                         // reset everything
        while(!DWN_BTN);                                    // and wait for the button to be released
    }   
    RBPU = 1;                                               // turn off pullups
  
    // show the initial startup screen and wait 2 seconds
    strcpy(StrBuf, M_SiliconChip); Center812(StrBuf, 60, 0);
    strcpy(StrBuf, M_Version); Center812(StrBuf, 60, 17);
    glcd_update();
    mSec(2000);
   
    // clear any random button presses
    SetBtnEvent = false;
    UpBtn = UpBtnEvent = 0;
    DwnBtn = DwnBtnEvent = 0;
   
	#ifdef USE_USB
	    usb_init_cs();
	#endif
  
    enable_interrupts(GLOBAL);
}



/************************************************************************************************
Get data from the GPS module
The message from the gps module has been loaded into a buffer.  This first determines what sort
of message that it is, then extracts the various data that we want.
                    
Format of the NEMA messages that we are interested in:
    01234567890123456789012345678901234567890123456789012345678901234567890123456789
    $GPGGA,043400.000,3158.7598,S,11552.8693,E,1,05,3.4,25.0,M,-29.3,M,,0000*58
    ====== ======     =========== ============   ==     ====                 ==
    fixed   time       latitude   longtitude     sat  altitude            checksum
    header hhmmss     ddmm.mmmm   dddmm.mmmm     cnt   meters

    $GPRMC,043356.000,A,3158.7599,S,11552.8689,E,0.24,54.42,101008,,*20
    ====== ======     = =========== ============ ==== ===== ======   ==
    fixed   time   valid  latitude   longtitude  speed course date    checksum
    header hhmmss   data  ddmm.mmmm dddmm.mmmm   knots deg   ddmmyy
    
************************************************************************************************/
void GetGpsData(void) {
    uint8 *p, i;
    
    p = &GpsBuf[GpsDataReady - 1][0];
    
	// decode the GGA meassage (this gives us most of the data that we are interested in)
    if(p[2] == 'G' && p[3] == 'G' && p[4] == 'A') {
        if(GetGpsInt(p, 6) == 0)
            NoSignal = true;                                        // we don't have a good lock on enough satellites
        else {  
            GetTime(p);
            CalcTime();
            LatBuf[0] = ' ';
            GetLatLong(&p[17], &LatBuf[1]);
            LongBuf[0] = p[29];
            GetLatLong(&p[30], &LongBuf[1]);
            SatUsed = GetGpsInt(p, 7);
			#ifndef USE_FUEL_SENSOR
		    	FlowRate2 = FlowRate1;
				disable_interrupts(INT_TIMER0);
        	    FlowRate1 = FuelCnt;
				FuelCnt = 0;
				enable_interrupts(INT_TIMER0);
				FlowCount += FlowRate1;
			#endif
        }
        
        UpdateSecCnt++;
        if(UpdateSecCnt >= 180) {
	        UpdateSecCnt = 0;
	        eeprom_write(E_FlowCount0, make8(FlowCount, 0));
	        eeprom_write(E_FlowCount1, make8(FlowCount, 1));
	        eeprom_write(E_FlowCount2, make8(FlowCount, 2));
	        eeprom_write(E_FlowCount3, make8(FlowCount, 3));
	        if(IGNITION) {
		        if(eeprom_read(E_RunMinutes) == 19) {
			        eeprom_write(E_RunMinutes, 0);
			        eeprom_write(E_RunHoursLB, eeprom_read(E_RunHoursLB) + 1);
			        if(eeprom_read(E_RunHoursLB) == 0) eeprom_write(E_RunHoursHB, eeprom_read(E_RunHoursHB) + 1);
			    } else
			     	eeprom_write(E_RunMinutes, eeprom_read(E_RunMinutes) + 1);
			}
		}
		
        NewData = true;                                             // signal that we have new data for display

	// decode the GSV messages (this gives us the signal levels and we get three of them every four seconds)
    } else if(p[2] == 'G' && p[3] == 'S' && p[4] == 'V') {
        if(p[8] == '1') {                                           // first GSV message
            for(i = 0; i < 12; i++) Sv[i] = 0;                      // zero our array of signal levels
            SvCnt = GetGpsInt(p, 3);                                // nbr of satellites that should be in the sky
            for(i = 0; i < 4; i++) Sv[i] = GetGpsInt(p, 7 + (i * 4));   // signal levels of the first four satellites
        }   
        if(p[8] == '2') for(i = 0; i < 4; i++) Sv[i + 4] = GetGpsInt(p, 7 + (i * 4));   // second GSV message
        if(p[8] == '3') for(i =0; i < 4; i++) Sv[i + 8] = GetGpsInt(p, 7 + (i * 4));    // third
        if(p[6] == p[8]) for(FoundSatCnt = i = 0; i < 12; i++) if(Sv[i] > 3) FoundSatCnt++;

	// decode the RMC message (this gives us just speed and heading)
    } else if(p[2] == 'R' && p[3] == 'M' && p[4] == 'C') {
        if(p[17] != 'A') {
            NoSignal = true;                                        // we don't have a good lock on enough satellites
        } else {
            Startup = NoSignal = false;                             // good signal so reset the signal related flags

            Speed = GetGpsInt(p, 7);
            if(Speed > 0 && GetGpsDecimal(p, 7) > 50) Speed++;
            //Speed = 5;

            Heading = GetGpsInt(p, 8);
            if(GetGpsDecimal(p, 8) > 50) Heading++;
            //Heading++;
            //if(Heading > 359) Heading = 0;
            //Heading = 340; 
        }
    }
    
    GpsDataReady = 0;                                               // signal that we have processed the data
}



// Get the integer value of a field in a gps message
// p = pointer to the text buffer
// i = field number.  the first field after the $GPXXX is 1, the next 2, etc.
// this will stop at the first non numeric character and return a 16 bit unsigned integer
uint16 GetGpsInt(uint8 *p, uint8 i) {
    uint16 t;
    
    while(i) if(*p++ == ',') i--;
    t = 0;
    while(*p >= '0' && *p <= '9') t = (t * 10) + (*p++ - '0');  
    return t;
}   



// Get the date from the gps message
// p = pointer to the text buffer
// will fill the global variables day, month, year
void GetDate(uint8 *p) {
    uint8 i = 9;
    
    while(i) if(*p++ == ',') i--;
    day = (p[0] - '0') * 10 + (p[1] - '0');
    month = (p[2] - '0') * 10 + (p[3] - '0');
    year = (p[4] - '0') * 10 + (p[5] - '0');
}   



// Get the value after the decimal point in a field in a gps message
// p = pointer to the text buffer
// i = field number.  the first field after the $GPXXX is 1, the next 2, etc.
// this will stop at the first non numeric character and return a 16 bit unsigned integer
// representing the decimal value * 100 (ie, 2 digits)
// will return zero if no decimal point found
uint16 GetGpsDecimal(uint8 *p, uint8 i) {
    uint16 t;
    
    while(i) if(*p++ == ',') i--;
    while(*p != '.') {
        if(*p == ',') return 0;                                 // find the decimal point
        p++;
    }   
    t = 0;
    p++;
    if(*p >= '0' && *p <= '9') {
        t = (*p++ - '0') * 10;
        if(*p >= '0' && *p <= '9') t += (*p++ - '0');
    }   
    return t;
}   


// get the time from a gps message
void GetTime(uint8 *p) {
    //if(p[12] != '.' || p[16] != ',') return;
    hour = ((p[6] - '0') * 10) + (p[7] - '0');
    min = ((p[8] - '0') * 10) + (p[9] - '0');
    sec = ((p[10] - '0') * 10) + (p[11] - '0');
}   


// get latitude or longtitude from a gps message
void GetLatLong(uint8 *p, uint8 *b) {
    uint8 i;
    
    //if(p[4] != '.' || p[9] != ',' || p[11] != ',') return;
    *b++ = *p++; *b++ = *p++;
    *b++ = '['; *b++ = ' ';                         // note that [ is the degree symbol
    for(i = 0; i < 7; i++) *b++ = *p++;
    *b++ = '\''; *b++ = ' ', *b++ = *++p;
    *b = 0;
}   

sint32 GetThreeDigits(uint8 *p) {
	uint16 i;
	
	i = 0;
	if(*p >= '0') i = *p - '0';
	p++;
	i = (i * 10) + (*p - '0');
	p++;
	i = (i * 10) + (*p - '0');
	return i;
}	


#define CurrentLat		0
#define CurrentLong		1
uint32 ConvLatLong(bit b) {
	uint8 *p;
	sint32 t;
	
	if(b)
		p = LongBuf;
	else
		p = LatBuf;
	t = GetThreeDigits(p);
	t = t * 60 + GetThreeDigits(p + 4);
	t = t * 600 + ((GetThreeDigits(p + 8) * 6) / 10);
	p += 14;
	if(*p == 'S' || *p == 'W') t = -t;
	if(b)
		t += 180 * 60 * 60 * 10;
	else
		t += 90 * 60 * 60 * 10;
	return t;
}	
	
	
	
	
/************************************************************************************************
State Machine functions
The following functions are responsible for handling a specific screen on the LCD
They are called by the main state machine depending on the current state
Each function is responsible for checking the data available and button pressed flags and doing
what ever is necessary.
If there is nothing to do they should simply return.
The various flags should not be reset inside these functions.  It is the responsibility of the state
machine to reset them.
Each function called by the state machine has a name in the format of DoXxx where Xxx is the screen.
************************************************************************************************/


// convert the time from UTC into local time using the timezone
void CalcTime(void) {
    minutes = (sint16)hour * 60 + (sint16)min + (sint16)tz * 6;
    if(minutes < 0) minutes += 24 * 60;
    if(minutes > 24 * 60) minutes -= 24 * 60;
    h = minutes/60;
    if(h >= 12) { 
        pm = true; h -= 12;
    } else
        pm = false;
    if(h == 0) h = 12;
    m = minutes % 60;
}


void DoTime(void) {
    if(NewData) {
        glcd_fillScreen(RevVid);
        glcd_rect(43,7,46,13,true,!RevVid); glcd_rect(43,19,46,25,true,!RevVid);                  // draw the colon
        glcd_setxy(-5, 2); printf(glcd_putc2232, "%2u", h);
        glcd_setxy(49, 2);   printf(glcd_putc2232, "%02u", m);
        glcd_setxy(103, 1); 
        if(pm)
            glcd_putc812('P');
        else
            glcd_putc812('A');
        glcd_putc812('M');
        glcd_setxy(99, 16); printf(glcd_putc1116, "%02u", sec);
        glcd_update();
    }   
}   



// Implements the main code required to handle a yes/no selection
// Before being called you must load StrBuf with the prompt. 
// This will return zero if nothing has happened or 1 for NO or 2 for YES when the set button pressed
uint8 GetYesNo(void) {
	static bit InMenu = false;
	static bit b;
	uint8 t;

	if(SetBtn)
		if(InMenu) {
			// the user has pressed Set, return the currently selected choice
			InMenu = false;
			return b + 1;
		} else {
			// this is the first time so reset the choice to NO (the default)
			InMenu = true;
			b = 0;
		}
		
    if(UpBtn || DwnBtn) b = !b;
    
    glcd_setxy(2, 2);
    glcd_text812(StrBuf);            								// print the prompt

    if(b) {
    	strcpy(StrBuf, M_yes);
	    t = 22;
    } else {
    	strcpy(StrBuf, M_no);
	    t = 14;
	} 
    glcd_line(glcd_x, glcd_y + 12, glcd_x + t, glcd_y + 12, !RevVid);		// draw the underline
    glcd_text812(StrBuf);											// print YES or NO
    
    return 0;
}	




void DoTimeCfg(void) {
    if(UpBtn) {
        while(UpBtn) { UpBtn--; tz += 5; if(tz > 120) tz = -120; }
         eeprom_write(E_tz, tz); 
    }
    else if(DwnBtn) {
        while(DwnBtn) { DwnBtn--; tz -= 5; if(tz < -120) tz = 120; }
         eeprom_write(E_tz, tz);
    }
    
    if(NewData) {
        glcd_fillScreen(RevVid);
        CalcTime();
        strcpy(StrBuf, M_settime); Center812(StrBuf, 60, 0);            // print set time heading       
        sprintf(StrBuf, "%u:%02u %cM", h, m, pm?'P':'A');
        Center1116(StrBuf, 60, 17);
        glcd_update();
    }   
}   


// sine function lookup table.  Each step is one degrees starting from zero and going up to 89 degrees.
const uint8 lu[90] = {0, 4, 8, 13, 17, 22, 26, 31, 35, 40, 44, 48, 53, 57, 61, 66, 70, 74, 79, 83, 87, 91, 95, 100, 104, 108, 112, 116,
					  120, 124, 128, 131, 135, 139, 143, 146, 150, 154, 157, 161, 164, 167, 171, 174, 177, 181, 184, 187, 190, 193, 196,
                      198, 201, 204, 207, 209, 212, 214, 217, 219, 221, 223, 226, 228, 230, 232, 233, 235, 237, 238, 240, 242, 243, 244,
                      246, 247, 248, 249, 250, 251, 252, 252, 253, 254, 254, 255, 255, 255, 255, 255};


// get the x coordinate (horizontal displacement) given a compass reading
// d is the desired direction of the pointer in degrees
// c is the horizontal center of the arrow in pixels
uint8 GetX(uint16 d, uint8 c) {
    if(d < 90) return c + (lu[d] >> 4);
    if(d < 180) return c + (lu[89 - ((d-90))] >> 4);
    if(d < 270) return c - (lu[d-180] >> 4);
    return c - (lu[89 - (d-270)] >> 4); 
}       
    
    
// get the y coordinate (vertical displacement) given a compass reading
// d is the desired direction of the pointer in degrees
// c is the vertical center of the arrow in pixels
uint8 GetY(uint16 d, uint8 c) {
    if(d < 90) return c - (lu[89 - d] >> 4);
    if(d < 180) return c + (lu[d-90] >> 4);
    if(d < 270) return c + (lu[89 - (d-180)] >> 4);
    return c - (lu[d-270] >> 4); 
}       



void DrawPointer(uint16 CompassPoint, uint16 NbrLabel) {
    #define HEADING_CENTER  90
    uint8 x1, y1, x2, y2;
    sint16 i, j, k, cp;
    
    if(speed) {
		for(k = -2; k <= 2; k++) {
	        cp = CompassPoint + k;
	        if(cp >= 360) cp -= 360;
	        if(cp < 0) cp += 360;
	        for(i = - 16; i <= 16; i += 1) {
	            j = cp + i + 180;
	            if(j < 0) j += 360;
	            if(j >= 360) j -= 360;
	            x1 = GetX(cp, HEADING_CENTER);
	            y1 = GetY(cp, 16);
	            x2 = GetX(j, HEADING_CENTER);
	            y2 = GetY(j, 16);
	            glcd_line(x1, y1, x2, y2, !RevVid);
	            //glcd_line(x2, y2, x1, y1, !RevVid);                // compensate for undiagnosed bug in glcd_bar()
	        }
	 	}  
	
	    if(CompassPoint < 90 || (CompassPoint > 180 && CompassPoint <= 270))
	        i = 21;
	    else
	        i = 0;
	    glcd_setxy(95, i); printf(glcd_putc812, "%3Ld[", NbrLabel);     // display the heading
	} 
    glcd_circle(HEADING_CENTER, 16, 4, ON, !RevVid);
    glcd_circle(HEADING_CENTER, 16, 2, ON, RevVid);
    glcd_pixel(HEADING_CENTER, 16, !RevVid);
}




void DoSpeed(void) {
    
    if(NewData) {
        glcd_fillScreen(RevVid);
        glcd_setxy(0, 2); printf(glcd_putc2232, "%2u", Speed);                 // display the speed
        strcpy(StrBuf, M_kn); glcd_setxy(49, 1); glcd_text812(StrBuf);        // display knots
		DrawPointer(360 - Heading, Heading);
        glcd_update();
     }   
}   




void DoRevVid(void) {
	uint8 i;
	
    glcd_fillScreen(RevVid);
    strcpy(StrBuf, M_reverse); 
	i = GetYesNo();
	if(i) {
		i--;
		eeprom_write(E_RevVid, i);
		RevVid = i;
		State = S_Speed;
	}		
    glcd_update();
}   




void DoFuel(void){
	uint16 FuelRate;
	uint16 FuelUsed;
    if(NewData) {
		glcd_fillScreen(RevVid);
		
		#ifdef USE_FUEL_SENSOR
		    disable_interrupts(INT_TIMER1);
			FuelRate = (uint32)3600000 / (make32(0, 0, eeprom_read(E_FlowCalHB), eeprom_read(E_FlowCalLB)) * (uint32)FlowPeriod);
			if(FlowPeriod == 0) FuelRate = 0;
			FuelUsed = (FlowCount * 10) / make32(0, 0, eeprom_read(E_FlowCalHB), eeprom_read(E_FlowCalLB));
		    enable_interrupts(INT_TIMER1);
		#else
			FuelRate = ((uint32)(FlowRate1 + FlowRate2) * (uint32)9) / make32(0, 0, eeprom_read(E_FlowCalHB), eeprom_read(E_FlowCalLB));
			FuelUsed = (uint32)FlowCount / (make32(0, 0, eeprom_read(E_FlowCalHB), eeprom_read(E_FlowCalLB)) * (uint32)200);
		#endif

		strcpy(StrBuf, M_fueltitle); Center812(StrBuf, 59, 2);
		
		sprintf(StrBuf, "%u.%u", (uint8)(FuelRate/10), (uint8)(FuelRate % 10));	// display rate
		Center1116(StrBuf, 27, 16);

		sprintf(StrBuf, "%Lu.%Lu", FuelUsed/10, FuelUsed % 10);					// display used
		Center1116(StrBuf, 93, 16);

		glcd_update();
	}
}




void DoFuelCfg1(void){
    glcd_fillScreen(RevVid);
    strcpy(StrBuf, M_fuelreset);
	switch(GetYesNo()) {
		case 1:	State = S_FuelCfg2;
				GetYesNo();
				break;
				
		case 2: FlowCount = 0;
				eeprom_write(E_FlowCount0, 0);
				eeprom_write(E_FlowCount1, 0);
				eeprom_write(E_FlowCount2, 0);
				eeprom_write(E_FlowCount3, 0);
				State = S_Fuel;
				break;
	}			
    glcd_update();
}




void DoFuelCfg2(void){
    glcd_fillScreen(RevVid);
    strcpy(StrBuf, M_fuelchoice);
	switch(GetYesNo()) {
		case 1:	State = S_Fuel;
				break;
				
		case 2: State = S_FuelCal;
				break;
	}			
    glcd_update();
}




void DoFuelCal(void) {
	uint16 FuelUsed;
	uint16 cal;
	uint8 caladj;
    if(UpBtn || DwnBtn) {
	    cal = make16(eeprom_read(E_FlowCalHB), eeprom_read(E_FlowCalLB));
	    caladj = cal/200 + 1;
        while(UpBtn) { UpBtn--; cal -= caladj; }
        while(DwnBtn) { DwnBtn--; cal += caladj; }
        eeprom_write(E_FlowCalHB, WordHiByte(cal)); 
        eeprom_write(E_FlowCalLB, WordLoByte(cal)); 
    }
    
 //   if(NewData) {
        glcd_fillScreen(RevVid);
        strcpy(StrBuf, M_fuelused); Center812(StrBuf, 60, 2);            // print the title       
		#ifdef USE_FUEL_SENSOR
			FuelUsed = (FlowCount * 10) / make32(0, 0, eeprom_read(E_FlowCalHB), eeprom_read(E_FlowCalLB));
		#else
			FuelUsed = FlowCount / (make32(0, 0, eeprom_read(E_FlowCalHB), eeprom_read(E_FlowCalLB)) * (uint32)10);
		#endif
        sprintf(StrBuf, "%Lu.%Lu", FuelUsed/10, FuelUsed % 10);
        Center1116(StrBuf, 60, 16);
        glcd_update();
//    }   
}   





void DoEng(void){
	glcd_fillScreen(RevVid);
	strcpy(StrBuf, M_engine); Center812(StrBuf, 60, 2);        // display the title
	sprintf(StrBuf, "%Lu", make16(eeprom_read(E_RunHoursHB), eeprom_read(E_RunHoursLB)));
	Center1116(StrBuf, 60, 16);                 // display the run hours
	glcd_update();
}



void DoEngCfg(void){
    glcd_fillScreen(RevVid);
    strcpy(StrBuf, M_enginechoice); 
	switch(GetYesNo()) {
		case 1:	State = S_Eng;
				break;
				
		case 2: eeprom_write(E_RunHoursHB, 0);
				eeprom_write(E_RunHoursLB, 0);
				eeprom_write(E_RunMinutes, 0);
	}			
    glcd_update();
}
    

void DoPoi(uint8 poi){
	sint32 x32, y32;
	uint16 x, y, z, h;
	sint16 a;
	uint8 i;
	uint8 precision, quadrant;
	
	CurrentPOI = poi;
	
    if(NewData) {
	    glcd_fillScreen(RevVid);
	    glcd_setxy(2,2);
		printf(glcd_putc812, "POI %u", poi + 1);
	    if(eeprom_read(E_POIStartY + (poi * 6)) == 0) {
			glcd_setxy(2, 16);
			strcpy(StrBuf, M_poinotset);
			glcd_text812(StrBuf);
			glcd_update();
			return;
		}
		
	    // get the current position in tens of seconds with zero at the north pole greenwich meridian
		x32 = ConvLatLong(CurrentLong);
		y32 = ConvLatLong(CurrentLat);
		
		// find the difference between us and the poi
		i = E_POIStartX + (poi * 6);
		x32 = make32(0, eeprom_read(i++), eeprom_read(i++), eeprom_read(i++))  - x32;
		y32 = make32(0, eeprom_read(i++), eeprom_read(i++), eeprom_read(i))  - y32;

		// compensate for possible wrap around at the greenwich meridian
		if(x32 < -12960000/2) x32 = x32 + 12960000;
		if(x32 >= 12960000/2) x32 = x32 - 12960000;
		
		// now figure out what quadrant the POI is in (1 = North to West, 2 = west to south, etc)
		// and strip the sign off so all numbers are positive
		if(x32 >= 0 && y32 > 0)
			quadrant = 1;
		else if(x32 > 0 && y32 <= 0)
			quadrant = 2;
		else if(x32 <= 0 && y32 < 0)
			quadrant = 3;
		else
			quadrant = 4;
			
		x32  = abs(x32);	
		y32  = abs(y32);
			
		// Calculate the distance to the POI
		// The earth has a mean circumfrence of 40,041km or 21,260nm (1nm = 1 sec of the earth's circumfrence).  In this routine
		// we convert the difference in latitude and longtitude (in tenths of a second) from the POI to nautical miles or, if
		// we are very close, to metres.  We then use the Pythagorean theorem (hypotenuse = squareroot((a * a) + (b * b))) to
		// calculate the distance.  The variable "precision" is used to record the output precision:
		//		precision = 0	Distance is over 100nm so display dashes ("----")
		//		precision = 1   Distance is 3 digits expressed in metres (eg.  210m)
		//		precision = 2   Distance is x.xx digits expressed in nautical miles (eg.  1.04nm)
		//		precision = 3   Distance is xx.x digits expressed in nautical miles (eg.  10.4nm)
		//		precision = 4   Distance is 3 digits expressed in nautical miles (eg.  104nm)
		precision = 0;
		if(x32 < 70000 && y32 < 70000){
			precision = 2;
			x = x32 / 7;											// convert longitude to nautical miles
			y = y32 / 7;											// convert latitude to nautical miles
			while(x > 100 || y > 100) {								// scale x and y so that both are less than 100
				precision++;
				x = x / 10;  y = y /10;
			}
			if(precision == 2 && x < 11 && y < 11) {				// if we are close to the POI
				x = x32 * 3;  y = y32 * 3;							// convert to metres
				precision = 1;
			}	
				
			z = (x * x) + (y * y);
			for(h = 0; h * h < z; h++);								// get sqroot in h
		}
			
		switch(precision) {
			case 0:	strcpy(StrBuf, M_dashes);
					break;
			case 1: sprintf(StrBuf, "%3u", (uint8)h);
					break;
			case 2: sprintf(StrBuf, "%3.2w", (uint8)h);
					break;
			case 3: sprintf(StrBuf, "%3.1w", (uint8)h);
					break;
			case 4: sprintf(StrBuf, "%3u", (uint8)h);
					break;
		}
		glcd_setxy(0, 16);
		glcd_text1116(StrBuf);
		glcd_y = 20; glcd_x += 2;
		if(precision != 1) {
			glcd_putc812('N');
		}
		glcd_putc812('M');
		
		
		z = 0xffff/h;
		switch(quadrant) {
			case 1:	i = (x * z) >> 8;
					break;
			case 2:	i = (y * z) >> 8;
					break;
			case 3:	i = (x * z) >> 8;
					break;
			case 4:	i = (y * z) >> 8;
					break;
		}
		for(a = 0; i > lu[a]; a++);
		a = a - Heading + (90 * ((sint16)quadrant - 1));
		if(a < 0) a = a + 360;
		if(a > 359) a = a - 360;
		DrawPointer(a, a);
		
	    glcd_update();
	} 
}



void DoPoiCfg1(void) {
	uint32 n;
	uint8 i;
	
    glcd_fillScreen(RevVid);
    strcpy(StrBuf, M_poinew); 
	switch(GetYesNo()) {		
		case 2: i = E_POIStartX + (CurrentPOI * 6);
				n = ConvLatLong(CurrentLong);
				eeprom_write(i++, make8(n, 2));
				eeprom_write(i++, make8(n, 1));
				eeprom_write(i++, make8(n, 0));
				n = ConvLatLong(CurrentLat);
				eeprom_write(i++, make8(n, 2));
				eeprom_write(i++, make8(n, 1));
				eeprom_write(i, make8(n, 0));

		case 1:	State = S_Poi1 + CurrentPOI;
				break;
	}			
    glcd_update();
}
	




void DoPosit(void) {
    if(NewData) {
        glcd_fillScreen(RevVid);
        glcd_setxy(0, 2); glcd_text812(LatBuf);
        glcd_setxy(0, 16); glcd_text812(LongBuf);
        glcd_update();
    }   
}   



void DoSat(void) {
    uint8 i;
    
    #define BAR_START   50
    #define BAR_WIDTH   5
    
    if(NewData) {
        glcd_fillScreen(RevVid);
        for(i = 0; i < 12; i++) {
            if(Sv[i] > 62) Sv[i] = 62;                                  // make sure the value will plot within the LCD size
            glcd_rect(BAR_START + (i * (BAR_WIDTH + 1)), 31 - Sv[i]/2, BAR_START + (i * (BAR_WIDTH + 1)) + BAR_WIDTH - 1, 31, true, !RevVid);
        }
        glcd_setxy(0, 4); printf(glcd_putc812, "%2u SAT", SvCnt);
        if(SatUsed > SvCnt) SatUsed = SvCnt;
        if(SatUsed < 10)
            glcd_setxy(4, 17); 
        else
            glcd_setxy(0, 17); 
        printf(glcd_putc812, "USE %u", SatUsed);
        glcd_update();  
    }   
}   




void DoBrightCfg(uint8 ENbr) {
    uint8 i;
    i = eeprom_read(ENbr);
    
    if(UpBtn) {
        while(UpBtn--) if(i != 100) i++; 
        eeprom_write(ENbr, i);
    }   
    else if(DwnBtn) {
        while(DwnBtn--) if(i != 0) i--; 
        eeprom_write(ENbr, i);
    }   
    
    BkLightValue = i;
    
    if(NewData) {
        glcd_fillScreen(RevVid); 
        Center812(StrBuf, 60, 0);   
        sprintf(StrBuf, "%u%%", i);
        Center1116(StrBuf, 60, 16);
        glcd_update();
    }   
}   




void DoBrightDayCfg(void) {
    strcpy(StrBuf, M_bright_day);
    DoBrightCfg(E_bright_day);  
}   



void DoBrightNightCfg(void) {
    strcpy(StrBuf, M_bright_night);
    DoBrightCfg(E_bright_night);    
}   



/*********************************************************************************************
General purpose or utility functions
*********************************************************************************************/


// Set up the GPS serial interface for 4800 baud async receive and transmit
void GPS_SetupUART(void) {
    SPBRGH = 0;
    SPBRG = 155;                                                    // baud rate selector, 4800 baud
    //BAUDCON = 0b00100000;
    RCIE = OFF;                                                     // no interrupts
    TXEN = ON;                                                      // enable transmit
    BRGH = OFF;                                                     // low speed
    SPEN = ON;                                                      // enable serial port
    CREN = ON;                                                      // enable receive
}

    
        
void Center812(uint8 *p, uint8 x, uint8 y) {
    uint8 j;
    
    for(j = 0; p[j]; j++) x -= 4;
    glcd_setxy(x, y);
    glcd_text812(p);
}




void Center1116(uint8 *p, uint8 x, uint8 y) {
    uint8 j;
    
    for(j = 0; p[j]; j++) x -= 6;
    glcd_setxy(x, y);
    glcd_text1116(p);
}


// implements the Show-Hide mode
void SetShowHide(void) {
    uint8   st = 1;
    uint8 i, j;
    
    enable_interrupts(GLOBAL);
    glcd_update();

    while(!UP_BTN);
    UpBtnEvent = 0;
    
    while(forever) {
        glcd_fillScreen(RevVid); 
        j = i = 0;
        while(i != st) if(M_set_showlist[j++] == 0) i++;
        for(i = 0; M_set_showlist[j] != 0; i++, j++) StrBuf[i] = M_set_showlist[j];
        StrBuf[i] = 0;
        Center812(StrBuf, 60, 0);                           // print the heading
        
        i = eeprom_read(HIDE_START + st);
        switch(i) {                                         // copy the show/hide string
            case 0: strcpy(StrBuf, M_set_show); break;
            case 1: strcpy(StrBuf, M_set_show_auto); break;
            case 2: strcpy(StrBuf, M_set_hide); break;
        }
        Center812(StrBuf, 60, 17);                          // print the show/hide string
        glcd_update();
        
        if(SetBtnEvent) {
            if(i++ == 2) i = 0;
            eeprom_write(HIDE_START + st, i);
        }   
        if(DwnBtnEvent) if(++st == HIDE_SIZE) st = 0;
        if(UpBtnEvent) if(st-- == 0) st = HIDE_SIZE - 1;
        SetBtnEvent = UpBtnEvent = DwnBtnEvent = 0;
    }
}


// reset everything to "factory defaults"
void ResetAll(void) {
    uint8 i;
    
    mSec(500);
    
    // send the reset to factory defaults command to the GPS module
    for(i = 0; M_gps_reset[i]; i++) {
        while(!TXIF);
        TXREG = M_gps_reset[i];
    }
    // copy the backup default settings into the working area of the eeprom
    for(i = 0; i < HIDE_START + HIDE_SIZE; i++)
        eeprom_write(i, eeprom_read(BACKUP_START + i));
}   
