//***************************************************************************
//                     glcd_library.c
//
//  This file contains a library of routines for handling a graphics LCD
//             Modified to suit the GPS Car Computer project
//
//  This is specific to the CCS C Compiler and contains some code which is
//  derived from CCS libaries.  It is therefor subject to CCS copyright
//  restrictions and, in particular, can only be used by licencees of the
//  CCS C Compiler (which you need anyway to compile this code).
//
//
//****************************************************************************
//
//   GRAPHICS ROUTINES
//
//   glcd_line(x1,y1,x2,y2,colour)
//      * Draws a line from the first point to the second point
//        with the given colour.
//        - colour can be ON or OFF
//
//   glcd_rect(x1,y1,x2,y2,fill,colour)
//      * Draws a rectangle with upper left point (x1,y1) and lower
//        right point (x2,y2).
//        - fill can be YES or NO
//        - colour can be ON or OFF
//
//   glcd_bar(x1,y1,x2,y2,width,colour)
//      * Draws a bar (wide line) from the first point to the
//        second point.
//        - width is the number of pixels wide
//        - colour is ON or OFF
//
//   glcd_circle(x,y,radius,fill,colour)
//      * Draws a circle with center at (x,y)
//        - fill can be YES or NO
//        - colour can be ON or OFF
//
//******************************************************************************
//
//  TEXT WRITING ROUTINES
//
//  There are 3 fonts:
//
//   - 8x12 font uppercase only
// 
//   - 11x16 font numeric, uppercase and the lowercase letters m and h
//
//   - 22x32 font with numeric digits only
//
//   Any or all together will be linked by calling the appropriate functions.
// 
//
//   Before calling ANY text function the starting position and optionally other
//   characteristics must be set by calling glcd_setxy()
//
//   void glcd_setxy(x, y, size, colour)
//       x and y: The upper left coordinate of the first character
//       size:    An integer that scales the size of the text (default is 1)
//       colour:  ON or OFF (default is ON)
//
//   If you are happy with the defaults you only need to specify x & y, for example:
//       glcd_setxy(0, 0);          // top left hand corner
//
//   All text functions update x and y automatically so that the invisible cursor is
//   ready for the next character.  The character \r will return the cursor to the
//   left margin and \n will cause the y position to step down a line.
//
//   Note that character wrapping is not supported, strings are truncated at the
//   edge of the display.
//
//   The data is stored in global variables and, as a shortcut, these can be accessed
//   directly.  The variables are:
//      int8 glcd_x, glcd_y         // Coordinates
//      int8 glcd_size              // Size
//      int1 glcd_colour            // Colour
//
//   The writing functions are:
//
//   glcd_putc812(char)
//      * Write char in 8x12 font
//      * Note that the tild character is replaced with a degrees C symbol
//
//   glcd_text812(textptr)
//      * Write the null terminated text pointed to by textptr in 8x12 font
//
//   glcd_putc1116(char)
//      * Write char in 11x16 font
//      * Note that the tild character is replaced with a degrees C symbol
//
//   glcd_text1116(textptr)
//      * Write the null terminated text pointed to by textptr in 11x16 font
//
//   glcd_putc2232(char)
//      * Write char in 22x32 font
//      * Note that the tild character is replaced with a degrees C symbol
//
//   glcd_text2232(textptr)
//      * Write the null terminated text pointed to by textptr in 22x32 font
//
//   The glcd_putcXX(char) functions can be used in printf().  For example:
//      printf(glcd_char812, "Temperature is %d~", temperature);
//      * Note that the tild (~) character is replaced with a degrees C symbol
//        in the 8x12 font
// 
//
//**********************************************************************************
//    LCD DRIVER
//
//   The above routines are dependent on a graphics driver that exposes the following
//   function:
//
//   glcd_pixel(x,y,colour)
//      * Sets the pixel to the given colour.
//        - colour can be ON or OFF
//
//
//   Other functions that the driver should supply (but are not used by this library) are:
//
//   glcd_init()
//      * Initilise the display
//      * Must be called before any other function.
//
//   glcd_fillScreen(colour)
//      * Fills the entire LCD with the given colour.
//        - colour can be ON or OFF
//
//
//**********************************************************************************

signed int8 glcd_x, glcd_y;                                         // Coordinates used to specify where the next text will go
byte glcd_size;                                                     // Same for size
bit glcd_colour;                                                    // Same for colour

extern bit RevVid;

void glcd_setxy(x, y, sz = 1, cl = 1) {
     glcd_x = x; glcd_y = y;                                        // set the coordinates
     glcd_size = sz; glcd_colour = cl;                              // set the other factors
}

byte glcd_readByte(byte chip);
void glcd_writeByte(byte chip, byte data);
void glcd_fillScreen(int1 colour);

    
const BYTE TEXT_8x12_1[21][12]={ 0,0,0,0,0,0,0,0,0,0,0,0                // ' ' 32
                        0,24,60,60,60,24,24,0,24,24,0,0,                // '!' 33
                        54,54,54,20,0,0,0,0,0,0,0,0,                    // '"' 34
                        0,108,108,108,254,108,108,254,108,108,0,0,      // '#' 35
                        24,24,124,198,192,120,60,6,198,124,24,24,       // '$' 36
                        0,0,0,98,102,12,24,48,102,198,0,0,              // '%' 37
                        0,56,108,56,56,118,246,206,204,118,0,0,         // '&' 38
                        12,12,12,24,0,0,0,0,0,0,0,0,                    // ''' 39
                        0,12,24,48,48,48,48,48,24,12,0,0,               // '(' 40
                        0,48,24,12,12,12,12,12,24,48,0,0,               // ')' 41
                        0,0,0,108,56,254,56,108,0,0,0,0,                // '*' 42
                        0,0,0,24,24,126,24,24,0,0,0,0,                  // '+' 43
                        0,0,0,0,0,0,0,12,12,12,24,0,                    // ',' 44
                        0,0,0,0,0,254,0,0,0,0,0,0,                      // '-' 45
                        0,0,0,0,0,0,0,0,24,24,0,0,                      // '.' 46
                        0,0,2,6,12,24,48,96,192,128,0,0,                // '/' 47
//                      0,124,198,206,222,246,230,198,198,124,0,0,      // '0' 48 with a slash
                        0,124,198,198,198,198,198,198,198,124,0,0,      // '0',48 without a slash
                        0,24,120,24,24,24,24,24,24,126,0,0,             // '1' 49
                        0,124,198,198,12,24,48,96,198,254,0,0,          // '2' 50
                        0,124,198,6,6,60,6,6,198,124,0,0,               // '3' 51
                        0,12,28,60,108,204,254,12,12,12,0,0             // '4' 52
};
                        
const BYTE TEXT_8x12_2[21][12]={ 0,254,192,192,192,252,6,6,198,124,0,0, // '5' 53
                        0,124,198,192,192,252,198,198,198,124,0,0,      // '6' 54
                        0,254,198,12,24,48,48,48,48,48,0,0,             // '7' 55
                        0,124,198,198,198,124,198,198,198,124,0,0,      // '8' 56
                        0,124,198,198,198,126,6,6,198,124,0,0,          // '9' 57
                        0,0,0,12,12,0,0,12,12,0,0,0,                    // ':' 58
                        0,0,0,12,12,0,0,12,12,12,24,0,                  // ';' 59
                        0,12,24,48,96,192,96,48,24,12,0,0,              // '<' 60
                        0,0,0,0,254,0,254,0,0,0,0,0,                    // '=' 61
                        0,96,48,24,12,6,12,24,48,96,0,0,                // '>' 62
                        0,124,198,198,12,24,24,0,24,24,0,0,             // '?' 63
                        0,124,198,198,222,222,222,220,192,126,0,0,      // '@' 64
                        0,56,108,198,198,198,254,198,198,198,0,0,       // 'A' 65
                        0,252,102,102,102,124,102,102,102,252,0,0,      // 'B' 66
                        0,60,102,192,192,192,192,192,102,60,0,0,        // 'C' 67
                        0,248,108,102,102,102,102,102,108,248,0,0,      // 'D' 68
                        0,254,102,96,96,124,96,96,102,254,0,0,          // 'E' 69
                        0,254,102,96,96,124,96,96,96,240,0,0,           // 'F' 70
                        0,124,198,198,192,192,206,198,198,124,0,0,      // 'G' 71
                        0,198,198,198,198,254,198,198,198,198,0,0,      // 'H' 72
                        0,60,24,24,24,24,24,24,24,60,0,0                // 'I' 73
};
                        
const BYTE TEXT_8x12_3[18][12]={ 0,60,24,24,24,24,24,216,216,112,0,0,   // 'J' 74
                        0,198,204,216,240,240,216,204,198,198,0,0,      // 'K' 75
                        0,240,96,96,96,96,96,98,102,254,0,0,            // 'L' 76
                        0,198,198,238,254,214,214,214,198,198,0,0,      // 'M' 77
                        0,198,198,230,230,246,222,206,206,198,0,0,      // 'N' 78
                        0,124,198,198,198,198,198,198,198,124,0,0,      // 'O' 79
                        0,252,102,102,102,124,96,96,96,240,0,0,         // 'P' 80
                        0,124,198,198,198,198,198,198,214,124,6,0,      // 'Q' 81
                        0,252,102,102,102,124,120,108,102,230,0,0,      // 'R' 82
                        0,124,198,192,96,56,12,6,198,124,0,0,           // 'S' 83
                        0,126,90,24,24,24,24,24,24,60,0,0,              // 'T' 84
                        0,198,198,198,198,198,198,198,198,124,0,0,      // 'U' 85
                        0,198,198,198,198,198,198,108,56,16,0,0,        // 'V' 86
                        0,198,198,214,214,214,254,238,198,198,0,0,      // 'W' 87
                        0,198,198,108,56,56,56,108,198,198,0,0,         // 'X' 88
                        0,102,102,102,102,60,24,24,24,60,0,0,           // 'Y' 89
                        0,254,198,140,24,48,96,194,198,254,0,0,         // 'Z' 90
                        0,64,160,160,64,0,0,0,0,0,0,0                   // Degree symbol
};


const unsigned int16 TEXT_11x16_1[11][11]={
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                                                //   
        0, 0, 0, 124, 13311, 13311, 124, 0, 0, 0, 0,                                    //  !
        0, 0, 60, 60, 0, 0, 60, 60, 0, 0, 0,                                            //  "
        512, 7696, 8080, 1008, 638, 7710, 8080, 1008, 638, 30, 16,                      //  #
        0, 1144, 3324, 3276, 16383, 16383, 3276, 4044, 1928, 0, 0,                      //  $
        12288, 14392, 7224, 3640, 1792, 896, 448, 14560, 14448, 14392, 28,              //  %
        0, 7936, 16312, 12796, 8646, 14306, 7742, 7196, 13824, 8704, 0,                 //  &
        0, 0, 0, 39, 63, 31, 0, 0, 0, 0, 0,                                             //  '
        0, 0, 1008, 4092, 8190, 14343, 8193, 8193, 0, 0, 0,                             //  (
        0, 0, 8193, 8193, 14343, 8190, 4092, 1008, 0, 0, 0,                             //  )
        0, 3224, 3768, 992, 4088, 4088, 992, 3768, 3224, 0, 0,                          //  *
};

const unsigned int16 TEXT_11x16_2[11][11]={
        0, 384, 384, 384, 4080, 4080, 384, 384, 384, 0, 0                               //  +
        0, 0, 0, 47104, 63488, 30720, 0, 0, 0, 0, 0,                                    //  ,
        0, 384, 384, 384, 384, 384, 384, 384, 384, 0, 0,                                //  -
        0, 0, 0, 14336, 14336, 14336, 0, 0, 0, 0, 0,                                    //  .
        6144, 7168, 3584, 1792, 896, 448, 224, 112, 56, 28, 14,                         //  /
     // 2040, 8190, 7686, 13059, 12675, 12483, 12387, 12339, 6174, 8190, 2040,          //  0
     // 2040, 8190, 6150, 12291, 12291, 12291, 12291, 12291, 6150, 8190, 2040,          //  0
        2040, 4092, 7182, 14343, 12291, 12291, 12291, 14343, 7182, 4092, 2040,          //  0
        0, 0, 12300, 12300, 12302, 16383, 16383, 12288, 12288, 12288, 0,                //  1
        12316, 14366, 15367, 15875, 14083, 13187, 12739, 12515, 12407, 12350, 12316,    //  2
        3084, 7182, 14343, 12483, 12483, 12483, 12483, 12483, 14823, 8062, 3644,        //  3
        960, 992, 880, 824, 796, 782, 775, 16383, 16383, 768, 768,                      //  4
        3135, 7295, 14435, 12387, 12387, 12387, 12387, 12387, 14563, 8131, 3971         //  5
};

const unsigned int16 TEXT_11x16_3[11][11]={
        4032, 8176, 14840, 12508, 12494, 12487, 12483, 12483, 14787, 8064, 3840,        //  6
        3, 3, 3, 12291, 15363, 3843, 963, 243, 63, 15, 3,                               //  7
        3840, 8124, 14846, 12519, 12483, 12483, 12483, 12519, 14846, 8124, 3840,        //  8
        60, 126, 12519, 12483, 12483, 14531, 7363, 3779, 2023, 1022, 252,               //  9
        0, 0, 0, 7280, 7280, 7280, 0, 0, 0, 0, 0,                                       //  :
        0, 0, 0, 40048, 64624, 31856, 0, 0, 0, 0, 0,                                    //  ;
        0, 192, 480, 1008, 1848, 3612, 7182, 14343, 12291, 0, 0,                        //  <
        0, 1632, 1632, 1632, 1632, 1632, 1632, 1632, 1632, 1632, 0,                     //  =
        0, 12291, 14343, 7182, 3612, 1848, 1008, 480, 192, 0, 0,                        //  >
        28, 30, 7, 3, 14211, 14275, 227, 119, 62, 28, 0,                                //  ?
        4088, 8190, 6151, 13299, 14331, 13851, 14331, 14331, 13831, 1022, 504           //  @
};

const unsigned int16 TEXT_11x16_4[11][11]={
        14336, 16128, 2016, 1788, 1567, 1567, 1788, 2016, 16128, 14336, 0,              //  A
        16383, 16383, 12483, 12483, 12483, 12483, 12519, 14846, 8124, 3840, 0,          //  B
        1008, 4092, 7182, 14343, 12291, 12291, 12291, 14343, 7182, 3084, 0,             //  C
        16383, 16383, 12291, 12291, 12291, 12291, 14343, 7182, 4092, 1008, 0,           //  D
        16383, 16383, 12483, 12483, 12483, 12483, 12483, 12483, 12291, 12291, 0,        //  E
        16383, 16383, 195, 195, 195, 195, 195, 195, 3, 3, 0,                            //  F
        1008, 4092, 7182, 14343, 12291, 12483, 12483, 12483, 16327, 16326, 0,           //  G
        16383, 16383, 192, 192, 192, 192, 192, 192, 16383, 16383, 0,                    //  H
        0, 0, 12291, 12291, 16383, 16383, 12291, 12291, 0, 0, 0,                        //  I
        3584, 7680, 14336, 12288, 12288, 12288, 12288, 14336, 8191, 2047, 0,            //  J
        16383, 16383, 192, 480, 1008, 1848, 3612, 7182, 14343, 12291, 0                 //  K
};

const unsigned int16 TEXT_11x16_5[11][11]={
        16383, 16383, 12288, 12288, 12288, 12288, 12288, 12288, 12288, 12288, 0,        //  L
        16383, 16383, 30, 120, 480, 480, 120, 30, 16383, 16383, 0,                      //  M
        16383, 16383, 14, 56, 240, 960, 1792, 7168, 16383, 16383, 0,                    //  N
        1008, 4092, 7182, 14343, 12291, 12291, 14343, 7182, 4092, 1008, 0,              //  O
        16383, 16383, 387, 387, 387, 387, 387, 455, 254, 124, 0,                        //  P
        1008, 4092, 7182, 14343, 12291, 13827, 15879, 7182, 16380, 13296, 0,            //  Q
        16383, 16383, 387, 387, 899, 1923, 3971, 7623, 14590, 12412, 0,                 //  R
        3132, 7294, 14567, 12483, 12483, 12483, 12483, 14791, 8078, 3852, 0,            //  S
        0, 3, 3, 3, 16383, 16383, 3, 3, 3, 0, 0,                                        //  T
        2047, 8191, 14336, 12288, 12288, 12288, 12288, 14336, 8191, 2047, 0,            //  U
        7, 63, 504, 4032, 15872, 15872, 4032, 504, 63, 7, 0                             //  V
};

const unsigned int16 TEXT_11x16_6[4][11]={
        16383, 16383, 7168, 1536, 896, 896, 1536, 7168, 16383, 16383, 0,                //  W
        12291, 15375, 3612, 816, 480, 480, 816, 3612, 15375, 12291, 0,                  //  X
        3, 15, 60, 240, 16320, 16320, 240, 60, 15, 3, 0,                                //  Y
        12291, 15363, 15875, 13059, 12739, 12515, 12339, 12319, 12303, 12291, 0        //  Z
};        

const int16 TEXT_11x16_k[11]={0, 16383, 16383, 768, 1920, 4032, 7392, 14432, 12288, 0, 0};                     //  k
const int16 TEXT_11x16_h[11]={16383, 16383, 192, 96, 96, 96, 224, 16320, 16256, 0, 0};              //  h
const int16 TEXT_11x16_m[11]={16352, 16320, 224, 224, 16320, 16320, 224, 224, 16320, 16256, 0};     //  m


const unsigned int32 NBR0_22x32[22]={2097024, 8388576, 33554424, 67108860, 66060540, 130023486, 125829150, 251658271, 251658255, 251658255, 251658255, 251658255, 251658255, 251658255, 251658271, 125829150, 130023486, 66060540, 67108860, 33554424, 8388576, 2097024};
const unsigned int32 NBR1_22x32[22]={0, 0, 0, 0, 251658480, 251658480, 251658480, 251658488, 251658492, 251658494, 268435455, 268435455, 268435455, 268435455, 251658240, 251658240, 251658240, 251658240, 251658240, 251658240, 0, 0};
const unsigned int32 NBR2_22x32[22]={251659248, 251659256, 260047868, 264241406, 266338367, 267386911, 267911183, 268173327, 259915791, 255787023, 253722639, 252690447, 252174351, 251916303, 251787279, 251722767, 251690527, 251674431, 251666430, 251662332, 251660280, 251659232};
const unsigned int32 NBR3_22x32[22]={15728880, 32506104, 66060540, 132120702, 130023486, 260046879, 251719695, 251719695, 251719695, 251719695, 251719695, 251719695, 251719695, 251719695, 251719695, 260175887, 130284575, 132644414, 67092478, 33497084, 16650232, 8128496};
const unsigned int32 NBR4_22x32[22]={1044480, 1046528, 1047552, 1048064, 999168, 991104, 987072, 985056, 984048, 983544, 983292, 983166, 983103, 983103, 268435455, 268435455, 268435455, 268435455, 983040, 983040, 983040, 983040};
const unsigned int32 NBR5_22x32[22]={15736831, 32522239, 66076671, 132136959, 130038799, 260062223, 251673615, 251673615, 251673615, 251673615, 251673615, 251673615, 251673615, 251673615, 251673615, 251673615, 260078607, 130087951, 134215695, 67104783, 33546255, 8355855};
const unsigned int32 NBR6_22x32[22]={16775168, 33553920, 67108608, 134217600, 264372160, 260112352, 251720688, 251720184, 251719932, 251719806, 251719742, 251719711, 251719695, 251719695, 251719695, 251719695, 260173839, 264499215, 134209536, 67092480, 33521664, 8257536};
const unsigned int32 NBR7_22x32[22]={15, 15, 15, 15, 15, 15, 251658255, 264241167, 267386895, 268173327, 33488911, 8372239, 2093071, 523279, 130831, 32719, 8191, 2047, 511, 127, 15, 15};
const unsigned int32 NBR8_22x32[22]={8257536, 33521664, 67094496, 134213624, 264503292, 260177918, 251722815, 251721759, 251719695, 251719695, 251719695, 251719695, 251719695, 251719695, 251721759, 251722815, 260177918, 264503292, 134213624, 67094496, 33521664, 8257536};
const unsigned int32 NBR9_22x32[22]={4080, 8184, 16380, 32766, 251722815, 251721759, 251719695, 251719695, 251719695, 251719695, 260108303, 264302607, 132182031, 66121743, 33091599, 16576527, 8386591, 4193343, 2097150, 1048572, 65528, 8160};


// Purpose:       Draw a line on a graphic LCD using Bresenham's
//                line drawing algorithm
// Inputs:        (x1, y1) - the start coordinate
//                (x2, y2) - the end coordinate
//                colour - ON or OFF
// Dependencies:  glcd_pixel()
void glcd_line(int x1, int y1, int x2, int y2, int1 colour)
{
   signed int  x, y, addx, addy, dx, dy;
   signed long P;
   int i;
   dx = abs((signed int)(x2 - x1));
   dy = abs((signed int)(y2 - y1));
   x = x1;
   y = y1;

   if(x1 > x2)
      addx = -1;
   else
      addx = 1;
   if(y1 > y2)
      addy = -1;
   else
      addy = 1;

   if(dx >= dy)
   {
      P = 2*dy - dx;

      for(i=0; i<=dx; ++i)
      {
         glcd_pixel(x, y, colour);

         if(P < 0)
         {
            P += 2*dy;
            x += addx;
         }
         else
         {
            P += 2*dy - 2*dx;
            x += addx;
            y += addy;
         }
      }
   }
   else
   {
      P = 2*dx - dy;

      for(i=0; i<=dy; ++i)
      {
         glcd_pixel(x, y, colour);

         if(P < 0)
         {
            P += 2*dx;
            y += addy;
         }
         else
         {
            P += 2*dx - 2*dy;
            x += addx;
            y += addy;
         }
      }
   }
}


// Purpose:       Draw a rectangle on a graphic LCD
// Inputs:        (x1, y1) - the start coordinate
//                (x2, y2) - the end coordinate
//                fill  - YES or NO
//                colour - ON or OFF
// Dependencies:  glcd_pixel(), glcd_line()
void glcd_rect(int x1, int y1, int x2, int y2, int fill, int1 colour)
{
   if(fill)
   {
      int y, ymax;                          // Find the y min and max
      if(y1 < y2)
      {
         y = y1;
         ymax = y2;
      }
      else
      {
         y = y2;
         ymax = y1;
      }

      for(; y<=ymax; ++y)                    // Draw lines to fill the rectangle
         glcd_line(x1, y, x2, y, colour);
   }
   else
   {
      glcd_line(x1, y1, x2, y1, colour);      // Draw the 4 sides
      glcd_line(x1, y2, x2, y2, colour);
      glcd_line(x1, y1, x1, y2, colour);
      glcd_line(x2, y1, x2, y2, colour);
   }
}


// Purpose:       Draw a bar (wide line) on a graphic LCD
// Inputs:        (x1, y1) - the start coordinate
//                (x2, y2) - the end coordinate
//                width  - The number of pixels wide
//                colour - ON or OFF
void glcd_bar(int x1, int y1, int x2, int y2, int width, int1 colour)
{
   signed int  x, y, addx, addy, j;
   signed long P, dx, dy, c1, c2;
   int i;
   dx = abs((signed int)(x2 - x1));
   dy = abs((signed int)(y2 - y1));
   x = x1;
   y = y1;
   c1 = -dx*x1 - dy*y1;
   c2 = -dx*x2 - dy*y2;

   if(x1 > x2)
   {
      addx = -1;
      c1 = -dx*x2 - dy*y2;
      c2 = -dx*x1 - dy*y1;
   }
   else
      addx = 1;
   if(y1 > y2)
   {
      addy = -1;
      c1 = -dx*x2 - dy*y2;
      c2 = -dx*x1 - dy*y1;
   }
   else
      addy = 1;

   if(dx >= dy)
   {
      P = 2*dy - dx;

      for(i=0; i<=dx; ++i)
      {
         for(j=-(width/2); j<width/2+width%2; ++j)
         {
            if(dx*x+dy*(y+j)+c1 >= 0 && dx*x+dy*(y+j)+c2 <=0)
               glcd_pixel(x, y+j, colour);
         }
         if(P < 0)
         {
            P += 2*dy;
            x += addx;
         }
         else
         {
            P += 2*dy - 2*dx;
            x += addx;
            y += addy;
         }
      }
   }
   else
   {
      P = 2*dx - dy;

      for(i=0; i<=dy; ++i)
      {
         if(P < 0)
         {
            P += 2*dx;
            y += addy;
         }
         else
         {
            P += 2*dx - 2*dy;
            x += addx;
            y += addy;
         }
         for(j=-(width/2); j<width/2+width%2; ++j)
         {
            if(dx*x+dy*(y+j)+c1 >= 0 && dx*x+dy*(y+j)+c2 <=0)
               glcd_pixel(x+j, y, colour);
         }
      }
   }
}


// Purpose:       Draw a circle on a graphic LCD
// Inputs:        (x,y) - the center of the circle
//                radius - the radius of the circle
//                fill - YES or NO
//                colour - ON or OFF
void glcd_circle(int x, int y, int radius, int1 fill, int1 colour)
{
   signed int a, b, P;
   a = 0;
   b = radius;
   P = 1 - radius;

   do
   {
      if(fill)
      {
         glcd_line(x-a, y+b, x+a, y+b, colour);
         glcd_line(x-a, y-b, x+a, y-b, colour);
         glcd_line(x-b, y+a, x+b, y+a, colour);
         glcd_line(x-b, y-a, x+b, y-a, colour);
      }
      else
      {
         glcd_pixel(a+x, b+y, colour);
         glcd_pixel(b+x, a+y, colour);
         glcd_pixel(x-a, b+y, colour);
         glcd_pixel(x-b, a+y, colour);
         glcd_pixel(b+x, y-a, colour);
         glcd_pixel(a+x, y-b, colour);
         glcd_pixel(x-a, y-b, colour);
         glcd_pixel(x-b, y-a, colour);
      }

      if(P < 0)
         P+= 3 + 2*a++;
      else
         P+= 5 + 2*(a++ - b--);
    } while(a <= b);
}





// Purpose:       Write a char in 8x12 font on a graphic LCD
// Note:          This does not contain charactera after '['
//                The '[' char will be drawn as a degree symbol
void glcd_putc812(char c)
{
    int j, k, l, m;                     // Loop counters
    BYTE pixelData[12];                 // Stores character data
    
    if(c == '\r') { glcd_x = 0; return; }
    if(c == '\n') { glcd_x = 0; glcd_y += 14 * glcd_size; return; }

    if(c < '5') // Checks if the letter is in the first text array
       memcpy(pixelData, TEXT_8x12_1[c-' '], 12);
    else if(c < 'J') // Check if the letter is in the second array
       memcpy(pixelData, TEXT_8x12_2[c-'5'], 12);
    else if(c <= '[') // Check if the letter is in the third array
       memcpy(pixelData, TEXT_8x12_3[c-'J'], 12);
    else
       memcpy(pixelData, TEXT_8x12_1[0], 12);   // Default to space

    for(j=0; j<12; ++j) {        // Loop through character byte data
       for(k=0; k<8*glcd_size; ++k) {          // Loop through the horizontal pixels
           if(bit_test(pixelData[j], k)) { // Check if the pixel should be set
             for(l=0; l<glcd_size; ++l) {       // The next two loops change the character's size
                for(m=0; m<glcd_size; ++m) {
                   glcd_pixel(glcd_x+((7-k)*glcd_size)+l, glcd_y+m+j*glcd_size, !RevVid); // Draws the pixel
                }
             }
          }
       }
    }
    glcd_x += 8 * glcd_size;
}



// Purpose:       Write text string in 8x12 font on a graphic LCD
void glcd_text812(char* textptr)
{
    for(; *textptr != '\0'; ++textptr)                  // Loop through the passed string
        glcd_putc812(*textptr);
}




// Purpose:       Write a char in 11x16 font on a graphic LCD
void glcd_putc1116(char c)
{
    byte j, k, l, m;                     // Loop counters
    unsigned int16 pixelData[12];                 // Stores character data
    
    if(c == '\r') { glcd_x = 0; return; }
    if(c == '\n') { glcd_y += 16 * glcd_size; return; }

    if(c < ' ' || c > 'Z') {
        if(c == 'k')
            memcpy(pixelData, TEXT_11x16_k, 22);    // lowercase k
        else if(c == 'h')
            memcpy(pixelData, TEXT_11x16_h, 22);    // lowercase h
        else if(c == 'm')
            memcpy(pixelData, TEXT_11x16_m, 22);    // lowercase m
        else
            memcpy(pixelData, TEXT_11x16_1[0], 22); // Non printable defaults to a space
    }           
    else if(c < ' ' + 11)                           // Checks if the letter is in the first text array
       memcpy(pixelData, TEXT_11x16_1[c-' '], 22);
    else if(c < '+' + 11)                           // Checks if the letter is in the next text array
       memcpy(pixelData, TEXT_11x16_2[c-'+'], 22);
    else if(c < '6' + 11)                           // Checks if the letter is in the next text array
       memcpy(pixelData, TEXT_11x16_3[c-'6'], 22);
    else if(c < 'A' + 11)                           // Checks if the letter is in the next text array
       memcpy(pixelData, TEXT_11x16_4[c-'A'], 22);
    else if(c < 'L' + 11)                           // Checks if the letter is in the next text array
       memcpy(pixelData, TEXT_11x16_5[c-'L'], 22);
    else                                            // Must be in the last array
       memcpy(pixelData, TEXT_11x16_6[c-'W'], 22);
       
    for(j=0; j<11; ++j, glcd_x+=glcd_size) {        // Loop through character byte data
       for(k=0; k<16*glcd_size; ++k) {              // Loop through the vertical pixels
          if(bit_test(pixelData[j], k)) {           // Check if the pixel should be set
             for(l=0; l<glcd_size; ++l) {           // The next two loops change the character's size
                for(m=0; m<glcd_size; ++m) {
                   glcd_pixel(glcd_x+m, glcd_y+k*glcd_size+l, !RevVid); // Draw the pixel
                }
             }
          }
       }
    }
   glcd_x += glcd_size;
}



// Purpose:       Write text string in 11x16 font on a graphic LCD
void glcd_text1116(char* textptr)
{
    for(; *textptr != '\0'; ++textptr)                  // Loop through the passed string
        glcd_putc1116(*textptr);
}






// Purpose:       Write a char in 22x32 font on a graphic LCD
// Note:          This contains the numbers 0 to 9 only
void glcd_putc2232(char c)
{
    byte j, k;                                                      // Loop counters
    uint32 pixelData;                                               // Stores character data
    
       
    for(j=0; j<22; ++j, glcd_x++) {                                 // Loop through character data
        switch(c- '0') {
            case 0: pixelData = NBR0_22x32[j]; break;
            case 1: pixelData = NBR1_22x32[j]; break;
            case 2: pixelData = NBR2_22x32[j]; break;
            case 3: pixelData = NBR3_22x32[j]; break;
            case 4: pixelData = NBR4_22x32[j]; break;
            case 5: pixelData = NBR5_22x32[j]; break;
            case 6: pixelData = NBR6_22x32[j]; break;
            case 7: pixelData = NBR7_22x32[j]; break;
            case 8: pixelData = NBR8_22x32[j]; break;
            case 9: pixelData = NBR9_22x32[j]; break;
            default:    pixelData = 0;
        }   
       for(k=0; k<32; ++k) {                                        // Loop through the vertical pixels
          if(bit_test(pixelData, k)) {                              // Check if the pixel should be set
               glcd_pixel(glcd_x, glcd_y+k, !RevVid);           // Draw the pixel
          }
       }
    }
    glcd_x += 2;                                                    // 2 pix space after the char
}



// Purpose:       Write text string in 11x16 font on a graphic LCD
void glcd_text2232(char* textptr)
{
    for(; *textptr != '\0'; ++textptr)                              // Loop through the passed string
        glcd_putc2232(*textptr);
}

