;	************************************************************
;	HiResCounter.asm - firmware for PIC-based
;	High Resolution Digital Counter
;
;	Written by Jim Rowe for Silicon Chip, using a PIC16F877A
;	processor running from an external crystal at 8MHz,
;	so one machine cycle (mc) = 0.5us.
;
;	----------------------------------------------------------------------------
;	Version	Date 		Author		Modification
;	----------------------------------------------------------------------------
;	1.1     3/12/12		Jim Rowe	Initial Version
;	----------------------------------------------------------------------------
;	1.2     12/09/13	Peter Urban	Ensure IC14 is correctly reset
;                                   		before counting - to prevent bobble
;                                   		on the least significant digit,
;                                   		when measuring low frequencies.
;	----------------------------------------------------------------------------
;	1.21	7/12/14		Ben Cook	Improved button press detection
;                                   		algorithm in subroutine 'DoAScan'
;	----------------------------------------------------------------------------
;	
;	This program performs counting of the most significant eight
;	decades via Timer1, etc., together with reading of the four 
;	least significant decades (which use external chips), reading
;	the eight mode select switches, control of overall counter
;	operation, plus driving the multiplexed 12-digit LED display
;	and the eleven mode display LEDs. 
;
;	**********************************************************
; 
;	CPU configuration
;

	list P=16F877A
	#include "p16f877A.inc"
	__config h'3F3A'	; HS osc selected, WDT disabled, PWRT disabled,
						; BOR disabled, RB3 is digital I/O, code prot off,
						; Flash prog mem write protection off, RB6 & RB7 are
						; general purpose I/O pins (not used for debugging)
;
;**********************************************************
;
;	First define some handy macros
;
;	Select Register Bank 0
bank0	macro
	errorlevel	+302	; Re-enable bank warning
	BCF	STATUS,RP0		; select Bank 0
	BCF STATUS, RP1
	BCF STATUS, IRP		; & also indir addr banks 0,1
	endm

;	Select Register Bank 1
bank1	macro
	BSF	STATUS,RP0		; select Bank 1
	BCF	STATUS,RP1
	BCF STATUS,IRP
	errorlevel	-302	; disable warning
	endm

;	Select Register Bank 2
bank2	macro
	BCF	STATUS,RP0		; select Bank 2
	BSF	STATUS,RP1
	BSF STATUS,IRP
	errorlevel	-302	; disable warning
	endm

;	Select Register Bank 3
bank3	macro
	BSF	STATUS,RP0		; select Bank 3
	BSF	STATUS,RP1
	BSF STATUS,IRP
	errorlevel	-302	; disable warning
	endm

;	Copy bytes in register file via W
copy	macro	from,to
	MOVF	from,W
	MOVWF	to
	endm

;	Swap bytes in register file via W
swap	macro	this,that
	MOVF	this,w		; get this
	XORWF	that,f		; Swap using Microchip
	XORWF	that,w		; Tips'n Tricks
	XORWF	that,f		; #18
	MOVWF	this
	endm
;
;**************************************************************
;
;	storage register declarations:

D1val		equ	h'20'	; display digit 1 value (lsd)
D2val		equ h'21'	; display digit 2
D3val		equ h'22'	; display digit 3
D4val		equ h'23'	; display digit 4
D5val		equ h'24'	; display digit 5
D6val		equ h'25'	; display digit 6
D7val		equ	h'26'	; display digit 7
D8val		equ h'27'	; display digit 8
D9val		equ h'28'	; display digit 9
D10val		equ h'29'	; display digit 10
D11val		equ h'2A'	; display digit 11
D12val		equ h'2B'	; display digit 12 value (msd)

ModeByte	equ h'2C'	; mode data bits (-> PORTB, then IC23)
;						bit0: 0 = Frequency, 1 = Period
; 						bit1: 1 = Select Input Channel A 
;						bit2: 1 = Select TB gating of 1000s
;						bit3: 1 = Select Input Channel B
;						bit4: 1 = Select TB gating of 10s
;						bit5: 1 = Select TB gating of 1s
;						bit6: 1 = Select TB gating of 100s
;						bit7: 0 = Int TB, 1 = Ext TB
;
;
;	Segment display codes for digits 1-12 and indicator LEDs
D1code		equ h'30'	; display code (7 seg) for digit 1 (lsd)
D2code		equ h'31'	; display code (7 seg) for digit 2
D3code		equ h'32'	; display code (7 seg) for digit 3
D4code		equ h'33'	; display code (7 seg) for digit 4
D5code		equ h'34'	; display code (7 seg) for digit 5
D6code		equ h'35'	; display code (7 seg) for digit 6
D7code		equ h'36'	; display code (7 seg) for digit 7
D8code		equ h'37'	; display code (7 seg) for digit 8
D9code		equ h'38'	; display code (7 seg) for digit 9
D10code		equ h'39'	; display code (7 seg) for digit 10
D11code		equ h'3A'	; display code (7 seg) for digit 11
D12code		equ h'3B'	; display code (7 seg) for digit 12 (msd)
LED427code	equ h'3C'	; display code for LEDs 4-7
;						(all at RA0-3 address 0Ch)
;						bit0: = 1 for LED6 on (PERIOD)
;						bit1: = 1 for LED4 on (CH A INPUT)
;						bit2: not used
;						bit3: not used
;						bit4: = 1 for LED7 on (EXT TB)
;						bit5: = 1 for LED5 on (CH B INPUT)
;						bit6: not used
;						bit7: not used
LED123code	equ h'3D'	; display code for LEDs 1-3
;						(all at RA0-3 address 0Dh)
;						bit0: not used
;						bit1: not used
;						bit2: = 1 for LED2 on (FREQ, CH A)
;						bit3: = 1 for LED3 on (PERIOD)
;						bit4: not used
;						bit5: not used
;						bit6: = 1 for LED1 on (FREQ, CH B)
;						bit7: not used
LED811code	equ h'3E'	; display code for LEDs 8-11
;						(all at RA0-3 address 0Eh)
;						bit0: = 1 for LED10 on (100s GATING)
;						bit1: = 1 for LED8 on  (1s GATING)
;						bit2: not used
;						bit3: not used
;						bit4: = 1 for LED11 on (1000s GATING)
;						bit5: = 1 for LED9 on  (10s GATING)
;						bit6: not used
;						bit7: not used

DPdigPos	equ h'3F'	; bits to indicate digit position of DP
;						bit0: 1 = digit 1 has DP (ChA*1s gating or Period)
;						bit1: 1 = digit 2 has DP (ChA*10s gating)
;						bit2: 1 = digit 3 has DP (ChA*100s gating)
;						bit3: 1 = digit 4 has DP (ChA*1000s gating or ChB*1s)
;						bit4: 1 = digit 5 has DP (ChB*10s gating)
;						bit5: 1 = digit 6 has DP (ChB*100s gating)
;						bit6: 1 = digit 7 has DP (ChB*1000s gating)
;						bit7: not used
;
;	storage for other variables 
Temp1		equ h'40'	; temp storage location 1
Temp2		equ h'41'	; temp storage location 2
Temp3		equ h'42'	; temp storage location 3
CountVH		equ h'43'	; used to store TMR1 overflows
CountFlag	equ h'44'	; bit 0 = 1 for for counter running

Keycode		equ h'45'	; valid button press code stored here ("2"- "9")
KPFlag		equ h'46'	; bit 0 = 1 if there was a valid keypress

DispLCtr	equ h'47'	; counter used in main display mpx loop
Bsplit		equ h'48'	; temporary storage for input byte of ByteInto2
CurrChar	equ h'49'	; temporary storage for display code translation
Counter1	equ h'4A'	; counter used in delay loops
Counter2	equ h'4B'	; counter used in delay loops
Counter3	equ h'4C'	; counter used in delay loops

;	storage for regs during interrupts
WSave		equ h'4E'	; W reg saved here during interrupts
SSave		equ h'4F'	; status reg saved here during interrupts


;	regs for storage of Byte10 and Byte1 (output bytes of ByteSplit routine)
Byte10		equ h'50'	; decimal tens output
Byte1		equ h'51'	; decimal units output

;	regs for storage of variables used by Bin24toBCD conversion routine
Count		equ h'52'	; counter
Temp		equ h'53'	; temp storage
H_byte		equ h'54'	; highest input byte
M_byte		equ h'55'	; middle input byte
L_byte		equ h'56'	; lowest input byte
R0			equ h'57'	; storage for output nibbles MSD & MSD-1
R1			equ h'58'	; storage for next two output digit nibbles
R2			equ h'59'	; storage for next two output digit nibbles
R3			equ h'5A'	; storage for output nibbles LSD+1 & LSD

;BJC Storage for new debounce algo
ButtonPress         equ h'5B'   ; storage for key debouncer state register
KeyTriggered        equ h'5C'   ; bit 0 = 1 for a raw button press

;	*************************************************************

;	Program now begins
;
	org	h'00'		; normal start vector
	GOTO Initialise
	org h'04'		; interrupt service vector
	GOTO IntServ	; interrupt service routine (TMR1 overflow)

	org h'0008'		; program proper begins here
Initialise:
	; first we set up CPU and INTOSC, I/O ports and TMR1 module
	bank0			; first make sure we're set for bank 0
	MOVLW h'30'		; then send 30h to PORTA to make sure
	MOVWF PORTA		; displays & LEDs are all off to begin
	CLRF PORTB		; also clear PORTB, C and D
	CLRF PORTC
	CLRF PORTD
	MOVLW h'06'		; but set bits 1 and 2 of PORTE so power
	MOVWF PORTE		; to both input channels will be off
	CLRF INTCON		; disable all interrupts for the present
	CLRF RCSTA		; and also the serial port
	MOVLW h'07'		; now make sure TMR1 ia enabled & set up correctly
	MOVWF T1CON		; for ext clock via RC0, no synch & 1:1 prescaling
	CLRF TMR1H		; clear its H & L registers
	CLRF TMR1L
	CLRF T2CON		; disable timer 2
	CLRF CCP1CON	; disable the CCP module
	CLRF SSPCON		; and the SSP module
	CLRF ADCON0		; and the A/D conv module

	bank1			; then switch to bank1
	MOVLW h'80'		; set Option reg to disable PORTB pullups
	MOVWF OPTION_REG
	CLRF CVRCON		; now disable comparator Vref module
	MOVLW h'07'		; and disable the comparators
	MOVWF CMCON		; (by setting them for mode 111)
	CLRF TRISA		; now set RA0-RA5 as outputs
	CLRF TRISB		; and set RB0-RB7 as outputs
	MOVLW h'49'		; also set RC0, 3 & 6 as inputs
	MOVWF TRISC		; (but RC1, 2, 4, 5 & 7 as outputs)
	MOVLW h'FF'		; then set RD0-7 as inputs
	MOVWF TRISD
	CLRF TRISE		; and set RE0-2 as outputs
	MOVLW h'06'		; now make sure RE0-2 are not analog
	MOVWF ADCON1	; but are set for digital I/O
	CLRF PIE2		; turn off peripheral interrupts
	CLRF PIE1		; then enable TMR1 o/flow interrupts only
	BSF PIE1,0

	bank3			; then up to bank3 so we can
	BCF EECON1,7	; clear EEPGD bit to ensure we address data EEPROM
	BCF EECON1,2	; also clear WREN bit so no writes until we want to

	bank0			; then back to Bank0 so we can...
	CLRF LED123code	; clear LED status regs
	CLRF LED427code
	CLRF LED811code
	CLRF CountFlag	; and CountFlag
	MOVLW h'22'		; now load ModeByte with default settings
	MOVWF ModeByte	; (Freq, ChA, 1s gating & internal TB)
	CLRF PIR1		; then clear all periph int flags
	CALL CheckMem	; check EEPROM in case settings were saved last time
	CALL SetMode	; now use ModeByte to set up counter
	CALL DefDisp	; also provide default startup display
	CALL DispUpdate	; and show startup display (Silicon Chip)


;BJC Addition
    BCF KeyTriggered,0 ; Clear KeyTriggered Flag


Mainloop:
;	main operating loop begins here
	CALL DoAScan		; first check for a keypress
	BTFSS KPFlag,0		; now -- was there a valid keypress?
	GOTO MovingOn		; if not, skip the next section

	MOVLW "9"			; valid keypress, so check if it was S9 ("9")
	XORWF Keycode,0		; w will now have 0 if it was a match...
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a 9, and w -> 0)
	GOTO $+6			; Z=0 (because it wasn't a 9), so keep looking
	BSF ModeByte,2		; was a 9, so set ModeByte(2) for 1000s gating
	BCF ModeByte,4		; and clear the other three gating time bits
	BCF ModeByte,5
	BCF ModeByte,6
	GOTO SetnSave		; then setup counter & save before moving on

	MOVLW "8"			; not S9, so check if it was S8 ("8")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was an "8")
	GOTO $+6			; Z=0, so go try again
	BSF ModeByte,6		; was an 8, so set ModeByte(6) for 100s gating
	BCF ModeByte,5		; and clear the other three gating time bits
	BCF ModeByte,4
	BCF ModeByte,2 
	GOTO SetnSave		; then setup counter & save settings

	MOVLW "7"			; not S8, so check if it was S7 ("7")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a "7")
	GOTO $+6			; Z=0, so keep looking
	BSF ModeByte,4		; was a 7, so set ModeByte(4) for 10s gating
	BCF ModeByte,2		; and clear the other three gating time bits
	BCF ModeByte,5
	BCF ModeByte,6
	GOTO SetnSave		; then set up counter & save settings

	MOVLW "6"			; not S7, so check if it was S6 ("6")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a "6")
	GOTO $+6			; Z=0, so keep looking
	BSF ModeByte,5		; was a 6, so set ModeByte(5) for 1s gating
	BCF ModeByte,2		; and clear the other three gating time bits
	BCF ModeByte,4
	BCF ModeByte,6
	GOTO SetnSave		; then set up counter & save settings

	MOVLW "5"			; not S6, so check if it was S5 ("5")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a "5")
	GOTO $+4			; Z=0, so keep looking
	MOVLW h'80'			; was a 5, so toggle ModeByte(7) to change TB
	XORWF ModeByte,1 
	GOTO SetnSave		; then set up counter & save settings

	MOVLW "4"			; not S5, so check if it was S4 ("4")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a "4")
	GOTO NotS4			; Z=0, so keep looking
	MOVLW h'01'			; was a 4, so toggle ModeByte(0) for Freq/Period
	XORWF ModeByte,1
	BTFSS ModeByte,0	; if we are now in Period mode, skip
	GOTO FreqBack		; otherwise reset for ChA freq measurement
	BCF ModeByte, 1		; Period mode, so ensure that both ChA & ChB
	BCF ModeByte, 3		; frequency input gates are off
	GOTO SetnSave		; then set up counter & save settings

FreqBack:
	BSF ModeByte,1		; returning from period mode, so reset for
	BCF ModeByte,3		; channel A measurement
	GOTO SetnSave		; then set up counter & save settings

NotS4:
	MOVLW "3"			; not S4, so check if it was S3 ("3")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a "3")
	GOTO $+4			; Z=0, so keep looking
	BSF ModeByte,3		; was a 3, so set ModeByte(3) for ChB
	BCF ModeByte,1		; and clear ModeByte(1) to disable ChA
	GOTO SetnSave		; then set up counter & save settings

	MOVLW "2"			; not S3, so check if it was S2 ("2")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a "2")
	GOTO $+3			; Z=0, so just skip set & save
	BSF ModeByte,1		; was a 2, so set ModeByte(1) for ChA
	BCF ModeByte,3		; and clear ModeByte(3) to disable ChB

SetnSave:
	; a setting was changed, so reconfig counter & save settings
	CALL SetMode		; go set up counter from ModeByte
	CALL SaveSettings	; also save new settings for next time
	CLRF CountFlag		; and clear CountFlag for clean start

MovingOn:
	BTFSC CountFlag,0	; now see if counting in progress
	GOTO CheckIfClosed	; if it is, go see if gate has closed

CounterInit:
	; not counting, so setup and start counter
	BSF PORTC, 5		; first reset decades 1 and 2, gate control FFs
	BTFSC ModeByte,0	; START OF ADDED CODE. Skip next if in Freq mode
	GOTO CounterInit2	; (extra code not needed if in period mode)
	MOVF ModeByte, W	; copy ModeByte into W
	ANDLW h'F5'			; disable both ChA and ChB inputs temporarily
	IORLW h'01'			; and set bit 0 to 1, so we swing into Period mode
	MOVWF PORTB			; then copy to Port B
	BSF PORTC, 7		; and pulse PortC(7) to latch into IC23
	NOP					; pausing briefly to make sure it latches
	BCF PORTC, 7		; before removing latch signal again
	MOVF ModeByte, W	; now copy ModeByte into W again
	MOVWF PORTB			; then copy to Port B again
	NOP					; pausing briefly to ensure IC14 has been reset
	BSF PORTC, 7		; now pulse PortC(7) again to latch into IC23
	NOP					; pausing briefly to make sure it latches
	BCF PORTC, 7		; before removing latch signal once again again		
	; END OF ADDED CODE, as IC14 should now have been reset reliably
CounterInit2:
	NOP					; slight pause to make sure they do reset
	BCF PORTC,5			; then remove reset pulse again
	BSF PORTE,0			; then reset decades 3 and 4
	NOP					; slight pause to make sure
	BCF PORTE,0			; then remove reset pulse
	CLRF TMR1L			; now reset the TMR1 registers
	CLRF TMR1H
	CLRF CountVH		; also clear the very high count register (TMR1 o/f)
	MOVLW h'07'			; then make sure TMR1 ia enabled & set up correctly
	MOVWF T1CON			; for ext clock via RC0, no synch & 1:1 prescaling
	BSF PORTC,4			; then set main gate control FF to unlock gate
	NOP					; ready to begin counting on first TB clock edge
	BCF PORTC,4			; clearing it again so gate can open
	BSF CountFlag,0		; & setting CountFlag to indicate count under way
	BCF PIR1,0			; now clear TMR1 o/flow flag just to make sure
	BSF INTCON,PEIE		; then enable interrupts for TMR1
	BSF INTCON,GIE		; by setting PEIE & GIE bits

CheckIfClosed:
	BTFSC PORTC, 3		; now see if main gate has closed yet
	GOTO RefreshD		; if not, just skip to refresh display
	BCF CountFlag,0		; but if it has, clear CountFlag
	BCF T1CON, 0		; also stop counter before we read
	CALL Delay1ms		; then wait for counters to settle, before
	CALL CountUpdate	; going to process new count
	CALL LZblank		; then blank leading zeroes
	CALL FindCodes		; translate digit values into display codes
RefreshD:
	CALL DispUpdate		; refresh all displays & LEDs
	GOTO Mainloop		; then loop back to begin again
 
;	main program loop ends -- subroutines follow
;
;	***********************************************************
;
Bin24toBCD:
;	routine to convert a 24-bit fixed point binary number in H_byte,
;	M_byte and L_byte into an 8-digit BCD number, left in R0 (MSD), R1,
;	R2 & R3 (LSD). (Adapted from Microchip B16TOBCD routine - in AN526)

	BCF STATUS,0			; first clear the carry bit
	MOVLW h'18'				; set count for 24d/18h
	MOVWF Count
	CLRF R0					; and clear the output registers
	CLRF R1
	CLRF R2
	CLRF R3
Loop24:
	RLF L_byte,F			; now rotate L_byte left through carry
	RLF M_byte,F			; and do the same with M_byte, from carry
	RLF H_byte,F			; and also with H_byte, from carry
	RLF R3,F				; then continue through R3
	RLF R2,F				; and R2
	RLF R1,F				; and R1
	RLF R0,F				; and finally R0

	DECFSZ Count,F			; decrement count, skip if it -> 0
	GOTO adjDEC				; otherwise keep going by going to adjDEC
	RETLW h'0'				; done all 24 bits, so leave

adjDEC:
	MOVLW R3				;
	MOVWF FSR				;
	CALL adjBCD

	MOVLW R2				; 
	MOVWF FSR				;
	CALL adjBCD
; 
	MOVLW R1				;
	MOVWF FSR				;
	CALL adjBCD				;
; 
	MOVLW R0				;
	MOVWF FSR				;
	CALL adjBCD				;
;
	GOTO Loop24				; continue by looping back 
;
adjBCD:
	MOVLW h'03'				; 
	ADDWF 0,W				;
	MOVWF Temp 
	BTFSC Temp,3			; test if result > 7
	MOVWF 0					; 
	MOVLW h'30'				; 
	ADDWF 0,W				;
	MOVWF Temp 
	BTFSC Temp,7			; test if result > 7
	MOVWF 0					; save as MSD
	RETLW h'0'
;	
;**************************************************************
;
ByteInto2:
	; routine to split a byte (in w on entry) into two nibbles
	; which are placed into Byte10 and Byte1 before return

	MOVWF Bsplit		; first save byte in Bsplit
	CLRF Byte10			; and initialise output digit values
	CLRF Byte1
	copy Bsplit, Byte1	; now copy Bsplit into Byte1
	SWAPF Bsplit,1		; then swap nibbles in Bsplit
	copy Bsplit, Byte10	; before copying into Byte 10
	MOVLW h'0F'			; now load mask into w
	ANDWF Byte1,1		; so we can mask off upper nibble in Byte1
	ANDWF Byte10,1		; and also in Byte10

	RETURN				; we're done -- so leave
;
;	**************************************************************
;
CheckMem:
	; routine to check if settings saved in data EEPROM, if so restore 'em
	; Enters in bank0, leaves in bank0 also
	CLRW					; set w for first address in EE memory (00h)
	MOVWF Temp1				; and save in Temp1 
	CALL ReadEE				; then read back data (returns with it in w)
	XORLW h'0F'				; check if it's first flag (0Fh)
	BTFSS STATUS,Z			; skip if Z=1 (i.e., it was a match)
	RETURN					; no first match, so leave now with settings unchanged
	INCF Temp1,1			; now set for second flag address (01h)
	MOVF Temp1,W			; copy into w
	CALL ReadEE				; and read data
	XORLW h'F0'				; check if it's second flag (F0h)
	BTFSS STATUS,Z			; skip if Z=1 (i.e., it was a match)
	RETURN					; no second match, so leave now with settings unchanged
	INCF Temp1,1			; did match, so set for first data address (02h)
	MOVF Temp1,W			; copy into w
	CALL ReadEE				; read it
	MOVWF ModeByte			; and save as ModeByte
	RETURN					; before returning with settings restored
;
;	****************************************************************
;
CountUpdate:
	; routine to process a count and update display digits, etc
	; (Once gate has closed & counters have stopped)
	BSF PORTC,1			; set up to read decades 1&2 via IC19
	BCF PORTC,2			; by taking RC1 high & RC2 low
	NOP					; small pause to allow IC19 outputs to settle
	MOVF PORTD, W		; then read into w reg
	CALL ByteInto2		; split into two digit nibbles
	copy Byte1, D1val	; then copy low nibble into D1val
	copy Byte10, D2val	; and high nibble into D2val
	BSF PORTC,2			; now set up to read decades 3&4 via IC21
	BCF PORTC,1			; by taking RC2 high & RC1 low
	NOP					; with another pause for IC21 outputs to settle
	MOVF PORTD, W		; then read into w reg
	CALL ByteInto2		; split into two digits
	copy Byte1, D3val	; then copy into D3val
	copy Byte10, D4val	; and D4val
	BSF PORTC,1			; then disable IC21 again before moving inside PIC

	copy CountVH, H_byte	; now copy CountVH into H_byte
	copy TMR1H, M_byte		; and copy TMR1H into M_byte
	copy TMR1L, L_byte		; and copy TMR1L into L_byte
	CALL Bin24toBCD			; so we can convert them into BCD digits
	MOVF R3, W			; then copy R3 (LSB) into w
	CALL ByteInto2		; so we can split into two digits
	copy Byte1, D5val	; which will become D5val
	copy Byte10, D6val	; and D6val

	MOVF R2, W			; now do the same with R2
	CALL ByteInto2
	copy Byte1, D7val	; to produce D7val
	copy Byte10, D8val	; and D8val

	MOVF R1, W			; and do the same with R1
	CALL ByteInto2
	copy Byte1, D9val	; to produce D9val
	copy Byte10, D10val ; and D10val

	MOVF R0, W			; then do the same with R0
	CALL ByteInto2
	copy Byte1, D11val	; to produce D11val
	copy Byte10,D12val	; and D12val

	RETURN				; all digits have been updated, so return
;
;	************************************************************
;
DefDisp:
;	routine to provide default digit display codes upon counter startup

	MOVLW h'67'			; first make D1code a "P"
	MOVWF D1code
	MOVLW h'48'			; then make D2code a "I"
	MOVWF D2code
	MOVLW h'6B'			; and so on
	MOVWF D3code
	MOVLW h'A6'	
	MOVWF D4code
	MOVLW h'00'	
	MOVWF D5code
	MOVLW h'6E'
	MOVWF D6code
	MOVLW h'EE'
	MOVWF D7code
	MOVLW h'A6'
	MOVWF D8code
	MOVLW h'48'
	MOVWF D9code
	MOVLW h'A2'
	MOVWF D10code
	MOVLW h'48'
	MOVWF D11code
	MOVLW h'AD'
	MOVWF D12code

	RETURN					; then leave as all 12 digits done
;	
;	****************************************************************
;	
Delay1ms:
	; routine to delay approx 1ms (2058 x 0.5us = 1029us) before return
	MOVLW h'0A'			; set Counter1 for 10 outer loops
	MOVWF Counter1
OuterLoop:
	MOVLW h'42'			; and Counter2 for 66 inner loops
	MOVWF Counter2		; (66 x 3mc = 198mc, + 7mc = 205)
	DECFSZ Counter2, 1	; decrement Counter2 & skip when zero
	GOTO $-1			; not zero yet, so loop back
	DECFSZ Counter1, 1	; did reach zero, so decrement Counter1
	GOTO OuterLoop		; didn't hit zero, so loop back
	RETURN				; reached zero (10 x 66 loops) so return

Delay200ms:
	; routine to delay approx 200ms before returning
	MOVLW h'C8'			; load w for 200 outer loops
	GOTO $+2			; then go set Counter3
Delay20ms:
	; routine to delay approx 20ms before returning
	MOVLW h'14'			; load w for 20 outer loops
	MOVWF Counter3		; now set Counter3
	CALL Delay1ms		; then wait 1ms
	DECFSZ Counter3, 1	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return	
;
;	***********************************************************
;
DispUpdate:
	; routine to refresh multiplexed displays & indicator LEDs
	; (one display sequence = 15 x 1ms = 15ms approx)
	CLRF DispLCtr		; first clear DispLCtr (00 = Dig1 address)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig1
	MOVF D1code,w		; then fetch D1code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D2 (01h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig2
	MOVF D2code,w		; then fetch D2code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D3 (02h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig3
	MOVF D3code,w		; then fetch D3code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D4 (03h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig4
	MOVF D4code,w		; then fetch D4code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D5 (04h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig5
	MOVF D5code,w		; then fetch D5code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D6 (05h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig6
	MOVF D6code,w		; then fetch D6code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D7 (06h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig7
	MOVF D7code,w		; then fetch D7code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D8 (07h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig8
	MOVF D8code,w		; then fetch D8code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D9 (08h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig9
	MOVF D9code,w		; then fetch D9code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D10 (09h)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig10
	MOVF D10code,w		; then fetch D10code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D11 (0Ah)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig11
	MOVF D11code,w		; then fetch D11code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for digit D12 (0Bh)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable Dig12
	MOVF D12code,w		; then fetch D12code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for LEDs 4-7 (0Ch)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable LEDs4-7
	MOVF LED427code,w	; then fetch LED427code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for LEDs 1-3 (0Dh)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable LEDs1-3
	MOVF LED123code,w	; then fetch LED123code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	INCF DispLCtr,1		; now increment counter for LEDs 8-11 (0Eh)
	MOVF DispLCtr, w	; now fetch digit or LED address
	MOVWF PORTA			; and send to Port A to enable LEDs8-11
	MOVF LED811code,w	; then fetch LED811code (segment code)
	MOVWF PORTB			; and send to Port B for display
	CALL Delay1ms		; then pause for visibility

	MOVLW h'30'			; done them all, so load 30h into
	MOVWF PORTA			; Port A to blank display (RA4=RA5=1)
	RETURN				; finally leave when done
;
;	***********************************************************
;	
DoAScan:
	; This subroutine scans the keys on the Front Panel and performs switch
	; debouncing for all keys.Saves in 'Keycode' if valid key press
	; Shift register 'ButtonPress' detects 8 consecutive 5ms cycles of a valid key
	; press returned by subroutine 'ScanKeys'. If the register fills, then
	; KPFlag and KeyTriggered are set. This prevents the routine running a second
	; time for the same key-press

    	CLRF KPFlag             ; KPFlag = 0
    	MOVLW h'0'
    	MOVWF ButtonPress       ; ButtonPress = 0
ScanLoop:
    	CALL ScanKeys           ; go scan FP buttons
    	ADDLW h'00'             ; w=w+0
    	BTFSC STATUS, Z         ; skip if Z=0 (ie w not 0)
    	goto NoKeys				; otherwise just leave
    	MOVWF Keycode           ; Z=0, so a button was pressed - save code
    	BTFSC KeyTriggered, 0   ; Is KeyTriggered = 1?
    	RETURN                  ; Yes, so return
    	RLF ButtonPress,1       ; Shift ButtonPress bits left by 1
    	MOVLW h'1'
    	IORWF ButtonPress,1     ; ButtonPress = ButtonPress OR '1'
    	CALL Delay1ms
    	CALL Delay1ms
    	CALL Delay1ms
    	CALL Delay1ms
    	CALL Delay1ms           ; Wait 5ms
    	BTFSS ButtonPress,7     ; Is ButtonPress Bit 7 = '1'?
    	GOTO ScanLoop           ; No, so loop round again
    	BSF KPFlag,0            ; Persistance detected, set KPFlag
    	BSF KeyTriggered,0      ; Set the KeyTriggered Flag
    	RETURN                  ; Return
  NoKeys:
    	BCF KeyTriggered, 0     ; KeyTriggered = 0
    	RETURN                  ; Return

;
;	*************************************************************
;
FindCodes:
;	routine to find actual segment display codes for numeric displays,
;	adding decimal point, ready for display multiplexing
	MOVF D1val, w		; fetch D1val into w
	CALL LookItUp		; then go look up its code
	MOVWF D1code		; before saving it in D1code

	MOVF D2val, w		; fetch D2val into w
	CALL LookItUp		; then go look up its code
	MOVWF D2code		; and save it in D2code

	MOVF D3val, w		; fetch D3val into w
	CALL LookItUp		; go look up its code
	MOVWF D3code		; and save it in D3code

	MOVF D4val, w		; fetch D4val into w
	CALL LookItUp		; go look up its code
	MOVWF D4code		; and save it in D4code

	MOVF D5val, w		; fetch D5val into w
	CALL LookItUp		; go look up its code
	MOVWF D5code		; and save it in D5code

	MOVF D6val, w		; fetch D6val into w
	CALL LookItUp		; go look up its code
	MOVWF D6code		; and save it in D6code

	MOVF D7val, w		; fetch D7val into w
	CALL LookItUp		; go look up its code
	MOVWF D7code		; and save it in D7code

	MOVF D8val, w		; fetch D8val into w
	CALL LookItUp		; then go look up its code
	MOVWF D8code		; before saving it in D8code

	MOVF D9val, w		; fetch D9val into w
	CALL LookItUp		; then go look up its code
	MOVWF D9code		; and save it in D9code

	MOVF D10val, w		; fetch D10val into w
	CALL LookItUp		; go look up its code
	MOVWF D10code		; and save it in D10code

	MOVF D11val, w		; fetch D11val into w
	CALL LookItUp		; go look up its code
	MOVWF D11code		; and save it in D11code

	MOVF D12val, w		; fetch D12val into w
	CALL LookItUp		; go look up its code
	MOVWF D12code		; and save it in D12code

DPplace:
	BTFSS DPdigPos,0	; now check if DP should be with Dig1
	GOTO $+3			; if not, keep looking
	BSF D1code, 4		; but if so, set DP bit in D1code
	RETURN				; and leave

	BTFSS DPdigPos,1	; next check if DP should be with Dig2
	GOTO $+3			; if not, keep looking
	BSF D2code, 4		; but if so, set DP bit in D2code
	RETURN				; and leave

	BTFSS DPdigPos,2	; next check if DP should be with Dig3
	GOTO $+3			; if not, keep looking
	BSF D3code, 4		; but if so, set DP bit in D3code
	RETURN				; and leave

	BTFSS DPdigPos,3	; next check if DP should be with Dig4
	GOTO $+3			; if not, keep looking
	BSF D4code, 4		; but if so, set DP bit in D4code & skip
	RETURN				; and leave

	BTFSS DPdigPos,4	; next check if DP should be with Dig5
	GOTO $+3			; if not, keep looking
	BSF D5code, 4		; but if so, set DP bit in D5code
	RETURN				; and leave

	BTFSS DPdigPos,5	; next check if DP should be with Dig6
	GOTO $+3			; if not, keep looking
	BSF D6code, 4		; but if so, set DP bit in D6code
	RETURN

	BTFSS DPdigPos,6	; finally check if DP should be with Dig7
	RETURN				; if not, just return
	BSF D7code, 4		; but if so, set DP bit in D7code
 	RETURN				; before we return
;
;	*************************************************************
;
LookItUp:
;	routine to find the segment display code for a value in the w
;	reg on entry. The matching code in the w register when it leaves...
;
	MOVWF CurrChar		; first save char temporarily
	MOVLW h'00'			; now test if it's a zero
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if didn't
	RETLW h'EE'			; but if we did, return with code for "0"

	MOVLW h'01'			; next test if it's a 1
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if didn't
	RETLW h'48'			; but if we did, return with code for "1"

	MOVLW h'02'			; next test if it's a 2
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'C7'			; but if we did, return with code for "2"

	MOVLW h'03'			; next test if it's a 3
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'CD'			; but if we did, return with code for "3"

	MOVLW h'04'			; next test if it's a 4
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'69'			; but if we did, return with code for "4"

	MOVLW h'05'			; next test if it's a 5
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'AD'			; but if we did, return with code for "5"

	MOVLW h'06'			; next test if it's a 6
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'AF'			; but if we did, return with code for "6"

	MOVLW h'07'			; next test if it's a 7
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'4C'			; but if we did, return with code for "7"

	MOVLW h'08'			; next test if it's an 8
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'EF'			; but if we did, return with code for "8"

	MOVLW h'09'			; next test if it's a 9
	XORWF CurrChar, w	; Z will = 1 if we get a match
	BTFSC STATUS, Z		; so skip if we didn't
	RETLW h'ED'			; but if we did, return with code for "9"

	RETLW h'00'			; none of the above, so return with a blank
;
;	*************************************************************
;
LZblank:
;	routine to blank leading zeroes from D12val-D1val display,
;	before conversion into display codes
	MOVF D12val, w		; fetch D12val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D12val		; a 'blank' value instead & continue
 
	MOVF D11val, w		; fetch D11val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D11val		; a 'blank' value instead & continue

 	MOVF D10val, w		; fetch D10val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D10val		; a 'blank' value instead & continue

 	MOVF D9val, w		; fetch D9val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D9val			; a 'blank' value instead & continue

 	MOVF D8val, w		; fetch D8val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D8val			; a 'blank' value instead & continue

 	MOVF D7val, w		; fetch D7val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D7val			; a 'blank' value instead & continue

  	MOVF D6val, w		; fetch D6val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D6val			; a 'blank' value instead & continue

 	MOVF D5val, w		; fetch D5val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D5val			; a 'blank' value instead & continue

 	MOVF D4val, w		; fetch D4val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D4val			; a 'blank' value instead & continue

 	MOVF D3val, w		; fetch D3val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D3val			; a 'blank' value instead & continue

 	MOVF D2val, w		; fetch D2val into w
	BTFSS STATUS,Z		; (Z=1 if it's zero)
	RETURN				; not zero, so we can leave now
	MOVLW h'0A'			; but if it was a zero, replace with
	MOVWF D2val			; a 'blank' value instead

	RETURN				; then return, since D1val can be 0
;
;	*************************************************************
;
PutInEE:
	; routine to write a data byte (in w) into current EEADR in
	; EE memory, also increment memory address in EEADR
	; (leaves in bank0)
	; Interrupts disabled before calling, enabled afterwards.
	bank2				; first swing up to bank2
	MOVWF EEDATA		; so we can move data into EEDATA
	bank3				; then move up to bank3 so we
	BCF EECON1, 7		; can clear EEPGD bit for data EEPROM access
	BSF EECON1, WREN	; and also enable EE writing
	MOVLW h'55'			; and carry out 'magic write sequence'
	MOVWF EECON2
	MOVLW h'AA'
	MOVWF EECON2
	BSF EECON1, WR		; so it does write the byte
	BCF EECON1, WREN	; then disable EE writing again
	BTFSC EECON1, WR	; now check to see if write has been done
	GOTO $-1			; loop back until it is (WR->0)
	bank0				; now down to bank0 so we can
	BTFSS PIR2, EEIF	; test EEIF bit (in PIR2)
	GOTO $-1			; loop back until it's set (write complete)
	BCF PIR2, EEIF		; once it is set, clear it ready for next write
	bank2				; then up to bank2 so we can
	INCF EEADR,1		; increment the data EEPROM address
	bank0				; and finally swing back down to bank0
	RETURN				; before returning
;
;	*********************************************************
;
ReadEE:
	; routine to read data byte from EE memory address (in w)
	; (enters and leaves in bank0)
	bank2				; first switch to bank2
	MOVWF EEADR			; so we can set up address register
	bank3				; then switch to bank3 so we
	BCF EECON1,7		; can  clear EEPGD bit for data EEPROM access
	BSF EECON1,RD		; and also set the read control bit
	bank2				; then switch back to bank2
	MOVF EEDATA,0		; and read the data into w reg
	bank0				; then swing down to bank0
	RETURN				; before we leave
;
;	***********************************************************
;
SaveSettings:
	; routine to save current settings in data EE memory
	; Enters & leaves in bank0, also disables interrupts during
	BCF INTCON, GIE		; first disable interrupts while we do it
	CLRW				; first set data EEPROM starting address
    bank2				; then up to bank2 so we can
	MOVWF EEADR			; save it
	MOVLW h'0F'			; then write first mem flag byte
	CALL PutInEE
	MOVLW h'F0'			; then write second mem flag byte
	CALL PutInEE		; (returns in bank0)
	MOVF ModeByte, W	; fetch ModeByte into w
	CALL PutInEE		; and then write it to EEPROM

	BSF INTCON, GIE		; finally re-enable interrupts
	RETURN				; before we return
;
;	***********************************************************
;		
ScanKeys:
	; routine to look for a pressed key (switches S2-S9), return with
	; ASCII code for switch number in w (or with a null in w
	; if no key is pressed)
	BCF PORTC,6			; first clear bit 6 of port C
	MOVLW h'FF'			; now make all bits of port B = 1
	MOVWF PORTB
	BTFSS PORTC, 6		; now test RC(6), to see if set
	RETLW h'00'			; if not, just return with null in w
	CLRF PORTB			; but if set, clear portB
	BCF PORTC,6			; and also clear RC(6) again
FindSwitch:
	BSF PORTB, 0		; first check if it's S9
	BTFSS PORTC, 6		;
	GOTO $+2			; otherwise keep looking
	RETLW "9"			; S9 pressed, so return with "9" in w
	BCF PORTB, 0		; not S9, so
	BSF PORTB, 1		; try for S8
	BTFSS PORTC, 6
	GOTO $+2			; otherwise keep looking
	RETLW "8"			; S8 pressed, so return with "8" in w
	BCF PORTB, 1		; not S8 either, so
	BSF PORTB, 2		; try for S7
	BTFSS PORTC, 6
	GOTO $+2			; otherwise keep looking
	RETLW "7"			; S7 pressed, so return with "7" in w
	BCF PORTB, 2		; not S7 either, so
	BSF PORTB, 3		; try for S6
	BTFSS PORTC, 6
	GOTO $+2			; otherwise keep looking
	RETLW "6"			; S6 pressed, so return with "6" in w
	BCF PORTB, 3		; not S6 either, so
	BSF PORTB, 4		; try for S5
	BTFSS PORTC, 6
	GOTO $+2			; otherwise keep looking
	RETLW "5"			; S5 pressed, so return with "5" in w
	BCF PORTB, 4		; not S5 either, so
	BSF PORTB, 5		; try for S4
	BTFSS PORTC, 6
	GOTO $+2			; otherwise keep looking
	RETLW "4"			; S4 pressed, so return with "4" in w
	BCF PORTB, 5		; not S4 either, so
	BSF PORTB, 6		; try for S3
	BTFSS PORTC, 6
	GOTO $+2			; otherwise keep looking
	RETLW "3"			; S3 pressed, so return with "3" in w
	BCF PORTB, 6		; not S3 either, so
	BSF PORTB, 7		; try for S2
	BTFSS PORTC, 6
	RETLW h'00'			; no key pressed now, so return with null
	RETLW "2"			; S2 pressed, so return with "2" in w
;
;	***********************************************************
;
SetMode:
;	routine to set up counter using ModeByte as the reference.
;	Saves ModeByte in IC23, also sets correct indicator LEDs
; 	and digit position for the DP (decimal point) in the display

	BCF PORTC, 7		; first make sure RC(7) = 0
	MOVF ModeByte, 0	; then copy ModeByte into w
	MOVWF PORTB			; and load into Port B
	BSF PORTC, 7		; now pulse LE input of IC23 to latch it
	NOP					; pausing briefly to make sure
	BCF PORTC, 7		; before restoring RC(7) = 0

	BTFSS ModeByte, 1	; now check if ModeByte(1) = 1 (sel ChA)
	GOTO $+5			; if not, move on
	BCF PORTE, 2		; but if set, turn on power to ChA (RE2->0)
	BSF PORTE, 1		; making sure ChB is off (RE1->1)
	BSF LED427code, 1	; also set bit for LED4 (ChA)
	BCF LED427code, 5	; while clearing bit for LED5 (ChB)

	BTFSS ModeByte, 3	; next check if ModeByte(3) = 1 (sel ChB)
	GOTO $+5			; if not, move on
	BCF PORTE, 1		; but if set, turn on power to ChB (RE1->0)
	BSF PORTE, 2		; making sure ChA is off (RE2->1)
	BSF LED427code, 5	; also set bit for LED5 (ChB)
	BCF LED427code, 1	; while clearing bit for LED4 (ChA)

	BTFSS ModeByte, 7	; next check if ModeByte(7) = 1 (Ext TB)
	GOTO $+3			; if not, skip to turn off LED7
	BSF LED427code, 4	; but if it's set, set bit for LED7 (ExtTB)
	GOTO $+2			; & then skip next
	BCF LED427code, 4	; must be Int TB, so make sure LED7 is off

	BTFSS ModeByte, 0	; next check if ModeByte(0) = 1 (Period)
	GOTO FreqMode		; if not, move on (must be freq meas mode)
	BSF LED427code, 0	; but if set, set bit for LED6 (Period)
	BCF LED427code, 4	; but clear bit for LED7 (ext TB)
	BSF LED123code, 3	; also set bit for LED3 (Period)
	BCF LED123code, 2	; but clear bit for LED2 (Freq*A)
	BCF LED123code, 6	; and also for LED1 (Freq*B)
	CLRF LED811code		; plus turn off LEDs8-11
	CLRF DPdigPos 		; then clear DPdigPos
	BSF DPdigPos, 0		; and just set DPdigPos(0)
	BCF PORTE, 2		; then turn on power for ChA input
	BSF PORTE, 1		; making sure ChB power is off
	RETURN				; then leave

FreqMode:
	BCF LED427code, 0	; must be Freq mode, so turn off LED6
	BCF LED123code, 3	; and also LED3
	BTFSS ModeByte, 1	; then check if we're using ChA
	GOTO $+4			; if not, must be ChB so skip next 3
	BSF LED123code, 2	; ChA, so turn on LED2
	BCF LED123code, 6	; and turn off LED1
	GOTO $+3			; then move on
	BSF LED123code, 6	; ChB, so turn on LED1
	BCF LED123code, 2	; and turn off LED2

GatingCheck:
	BTFSS ModeByte, 5	; now check if ModeByte(5) = 1 (1s gating)
	GOTO $+9			; if not, move on
	CLRF LED811code		; but if set, clear all bits for LED811code 	
	BSF LED811code, 1	; and just set bit for LED8 (1s gating)
	CLRF DPdigPos		; then clear DPdigPos
	BTFSS ModeByte, 1	; now check if we are using ChA
	GOTO $+3			; if not, skip next
	BSF DPdigPos, 0		; if yes, set DPdigPos(0)
	GOTO $+2			; and then move on
	BSF DPdigPos, 3		; must be ChB, so set DPdigPos(3) instead

	BTFSS ModeByte, 4	; now check if ModeByte(4) = 1 (10s gating)
	GOTO $+9			; if not, move on
	CLRF LED811code		; but if set, clear all bits for LED811code 	
	BSF LED811code, 5	; and just set bit for LED9 (10s gating)
	CLRF DPdigPos		; then clear DPdigPos
	BTFSS ModeByte, 1	; now check if we are using ChA
	GOTO $+3			; if not, skip next
	BSF DPdigPos, 1		; if yes, set DPdigPos(1)
	GOTO $+2			; and then move on
	BSF DPdigPos, 4		; must be ChB, so set DPdigPos(4) instead

	BTFSS ModeByte, 6	; now check if ModeByte(6) = 1 (100s gating)
	GOTO $+9			; if not, move on
	CLRF LED811code		; but if set, clear all bits for LED811code 	
	BSF LED811code, 0	; and just set bit for LED10 (100s gating)
	CLRF DPdigPos		; then clear DPdigPos
	BTFSS ModeByte, 1	; now check if we are using ChA
	GOTO $+3			; if not, skip next
	BSF DPdigPos, 2		; if yes, set DPdigPos(2)
	GOTO $+2			; and then move on
	BSF DPdigPos, 5		; must be ChB, so set DPdigPos(5) instead

	BTFSS ModeByte, 2	; now check if ModeByte(2) = 1 (1000s gating)
	GOTO $+9			; if not, move on
	CLRF LED811code		; but if set, clear all bits for LED811code 	
	BSF LED811code, 4	; and just set bit for LED11 (1000s gating)
	CLRF DPdigPos		; then clear DPdigPos
	BTFSS ModeByte, 1	; now check if we are using ChA
	GOTO $+3			; if not, skip next
	BSF DPdigPos, 3		; if yes, set DPdigPos(3)
	GOTO $+2			; and then move on
	BSF DPdigPos, 6		; must be ChB, so set DPdigPos(6) instead

	RETURN				; all done, so we can return
;
;	***********************************************************
;
;   end of main routines -- interrupt servicing routine follows
	
IntServ:
	; routine to service interrupts (from TMR1 overflow)

	MOVWF WSave			; first save context (w and status regs)
	SWAPF STATUS,0		; using SWAPF here to avoid STATUS change
	MOVWF SSave

	bank0				; make sure we're in data memory bank0
	INCF CountVH,1		; increment CountVH to register a TMR1 o/f
	BCF PIR1,0			; then clear TMR1IF flag for next time
	
	SWAPF SSave,0			; restore context (status & w regs)
	MOVWF STATUS			;
	SWAPF WSave,1			; using SWAPFs to avoid changing STATUS
	SWAPF WSave,0			;
	RETFIE					; & return, re-enabling ints (TOS->PC, 1->GIE)
;
;	************************************************************
;
 	END


