/*
		Interface for 2 4-digit, common cathode 7 segment LED displays in a multiplexed configuration.
*/



#include "7SEG.h"

unsigned char Display[2][5];  // A buffer for our 2x4 character 7segment display system
unsigned char DisplayDP[3]; //a flag for each display to indicate whether the decimal point on the last digit should be lit

// Used for the LED pulse width modulation; these will be the counter values at which to turn on/off the leds
unsigned int OCR1A_ON;
unsigned int OCR1A_OFF;


volatile unsigned char DisplayStatus; // On or off
volatile unsigned char EnabledDigit=1;	// Which digit (of the four in each 7segment display) should be illuminated?
volatile unsigned char EnabledDisplay=1; // Which of the two 7 segment displays should be illuminated?


unsigned char 	num2char(unsigned char num);


ISR (TIMER1_COMPA_vect) // Timing for display multiplex
{
	Process_7Seg_Multiplex();
}

void Initialise_7SEG()
{
	// Configure the IO pins associated with the 7segment display: Set as output, then take them low.
		// LED SEGMENTS
		sbi(DDR(SEG_A_PORT),SEG_A_PIN);
		sbi(DDR(SEG_B_PORT),SEG_B_PIN);
		sbi(DDR(SEG_C_PORT),SEG_C_PIN);
		sbi(DDR(SEG_D_PORT),SEG_D_PIN);
		sbi(DDR(SEG_E_PORT),SEG_E_PIN);
		sbi(DDR(SEG_F_PORT),SEG_F_PIN);
		sbi(DDR(SEG_G_PORT),SEG_G_PIN);
		sbi(DDR(SEG_DP_PORT),SEG_DP_PIN);

		cbi(PORT(SEG_A_PORT),SEG_A_PIN);
		cbi(PORT(SEG_B_PORT),SEG_B_PIN);
		cbi(PORT(SEG_C_PORT),SEG_C_PIN);
		cbi(PORT(SEG_D_PORT),SEG_D_PIN);
		cbi(PORT(SEG_E_PORT),SEG_E_PIN);
		cbi(PORT(SEG_F_PORT),SEG_F_PIN);
		cbi(PORT(SEG_G_PORT),SEG_G_PIN);
		cbi(PORT(SEG_DP_PORT),SEG_DP_PIN);

		// DIGIT CATHODES
		sbi(DDR(D1D1_ENABLE_PORT), D1D1_ENABLE_PIN);
		sbi(DDR(D1D2_ENABLE_PORT), D1D2_ENABLE_PIN);
		sbi(DDR(D1D3_ENABLE_PORT), D1D3_ENABLE_PIN);
		sbi(DDR(D1D4_ENABLE_PORT), D1D4_ENABLE_PIN);
		cbi(PORT(D1D1_ENABLE_PORT), D1D1_ENABLE_PIN);
		cbi(PORT(D1D2_ENABLE_PORT), D1D2_ENABLE_PIN);
		cbi(PORT(D1D3_ENABLE_PORT), D1D3_ENABLE_PIN);
		cbi(PORT(D1D4_ENABLE_PORT), D1D4_ENABLE_PIN);

		sbi(DDR(D2D1_ENABLE_PORT), D2D1_ENABLE_PIN);
		sbi(DDR(D2D2_ENABLE_PORT), D2D2_ENABLE_PIN);
		sbi(DDR(D2D3_ENABLE_PORT), D2D3_ENABLE_PIN);
		sbi(DDR(D2D4_ENABLE_PORT), D2D4_ENABLE_PIN);
		cbi(PORT(D2D1_ENABLE_PORT), D2D1_ENABLE_PIN);
		cbi(PORT(D2D2_ENABLE_PORT), D2D2_ENABLE_PIN);
		cbi(PORT(D2D3_ENABLE_PORT), D2D3_ENABLE_PIN);
		cbi(PORT(D2D4_ENABLE_PORT), D2D4_ENABLE_PIN);

	// Configure timer 1 (16bit) which does the pulse-width modulation for the 7segment displays
	TCNT1=0x00;
	TCCR1B|=((1<<WGM12)|(1<<CS12)); //256 prescaler, CTC mode

	// We calculated all these settings in 7SEG.h
	OCR1A_ON=(DIGIT_ADDRESS_TIME/10)*DUTY_CYCLE;
	OCR1A_OFF=DIGIT_ADDRESS_TIME-OCR1A_ON;
	OCR1A=OCR1A_ON;
	TIMSK1|=(1<<OCIE1A);


}

//////////////////////////////////////////////////////////////////////////////////
//  void Process_7Seg_Multiplex()												//
//																				//
//	We have enough output pins on the AVR to power one 7-segment character at a //
//  time. In order to use all 8 characters (2x 4 digit) we need to multiplex.   //
//  We will scan through each character and illuminate it briefly. Done fast    //
//  enough, our human eyes will think they're all lit at once.                  //
//																				//
//////////////////////////////////////////////////////////////////////////////////

void Process_7Seg_Multiplex()
{
	if(DisplayStatus==ON)
	{
		EnableDigit(EnabledDisplay,OFF);
		DisplayStatus=OFF;
		EnabledDigit++;
		if((EnabledDigit>4) & (EnabledDisplay==TOP))
		{
			EnabledDigit=1;
			EnabledDisplay=BOTTOM;
		}
		if((EnabledDigit>4) & (EnabledDisplay==BOTTOM))
		{
			EnabledDigit=1;
			EnabledDisplay=TOP;
		}
		if(EnabledDigit==3)
			DisplayChar(Display[EnabledDisplay][EnabledDigit], DisplayDP[EnabledDisplay]);
		else
			DisplayChar(Display[EnabledDisplay][EnabledDigit], 0);


		OCR1A=OCR1A_OFF;
	}
	else
	{
		EnableDigit(EnabledDisplay,EnabledDigit);
		DisplayStatus=ON;
		OCR1A=OCR1A_ON;
	}

}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//  void DisplayUnsignedInt(unsigned char WhichDisplay, unsigned int Number, unsigned char show_DP)  	//
//																										//
//	This function will display the specified 4-digit number on the specified display.					//
//  (i.e. it will take care of figuring out which segments need to be lit to spell out a given number	//
//  																									//
//	If show_DP=1, the decimal point on the third character will be lit (allowing 1dp precision)			//
//																										//
//	In a couple of places I use the character 'X' - this just means don't light up anything				//
//																										//
//////////////////////////////////////////////////////////////////////////////////////////////////////////

void DisplayUnsignedInt(unsigned char WhichDisplay, unsigned int Number, unsigned char show_DP)
{
	unsigned int temp;
	unsigned char j=0; //j here is a flag that keeps track of whether the most significant digit has ocurred yet. It determines whether we should print a zero or a nothing

	DisplayDP[WhichDisplay]=FALSE;
	if(show_DP==TRUE)
		DisplayDP[WhichDisplay]=TRUE;


	if(Number>9999)
		Number=9999;
	temp=Number/1000;
	if(temp>0)
	{
		Display[WhichDisplay][1]=num2char(temp);
		j=1;
	}
	else
		Display[WhichDisplay][1]='X';
	Number-=temp*1000;
	temp=Number/100;

	if(temp>0)
	{
		Display[WhichDisplay][2]=num2char(temp);
		j=1;
	}
	else if(j==0)
			Display[WhichDisplay][2]='X';
	else if(j==1)
			Display[WhichDisplay][2]='0';

	Number-=temp*100;
	temp=Number/10;
	if(temp>0)
	{
		Display[WhichDisplay][3]=num2char(temp);
		j=1;
	}
	else
			Display[WhichDisplay][3]='0';
	Number-=temp*10;
	temp=Number;
	Display[WhichDisplay][4]=num2char(temp);
}



//////////////////////////////////////////////////////////////////////////////////
//  unsigned char num2char(unsigned char num)									//
//																				//
//	Single digit number to ascii conversion										//
//																				//
//////////////////////////////////////////////////////////////////////////////////

unsigned char num2char(unsigned char num)  // single digit only!
{
	return num+48;
}



//////////////////////////////////////////////////////////////////////////////////
//  void DisplayChar(unsigned char Character, unsigned char decimalpoint)		//
//																				//
//	This will light up the segments required to show the specified Character	//
// (which in this case can be a number from 0-9 or a hyphen)					//
//																				//
//////////////////////////////////////////////////////////////////////////////////

void DisplayChar(unsigned char Character, unsigned char decimalpoint)
{
	// First turn all segments off (so we start from a known state)
	cbi(PORT(SEG_A_PORT),SEG_A_PIN);
	cbi(PORT(SEG_B_PORT),SEG_B_PIN);
	cbi(PORT(SEG_C_PORT),SEG_C_PIN);
	cbi(PORT(SEG_D_PORT),SEG_D_PIN);
	cbi(PORT(SEG_E_PORT),SEG_E_PIN);
	cbi(PORT(SEG_F_PORT),SEG_F_PIN);
	cbi(PORT(SEG_G_PORT),SEG_G_PIN);
	cbi(PORT(SEG_DP_PORT),SEG_DP_PIN);

	// Now go through and turn on all the required LED segments
	if(decimalpoint==TRUE)
		sbi(PORT(SEG_DP_PORT),SEG_DP_PIN);

	switch(Character)
	{
		case '0':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					sbi(PORT(SEG_D_PORT),SEG_D_PIN);
					sbi(PORT(SEG_E_PORT),SEG_E_PIN);
					sbi(PORT(SEG_F_PORT),SEG_F_PIN);
					break;

		case '1': 	sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					break;

		case '2':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_D_PORT),SEG_D_PIN);
					sbi(PORT(SEG_E_PORT),SEG_E_PIN);
					sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;

		case '3':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					sbi(PORT(SEG_D_PORT),SEG_D_PIN);
					sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;

		case '4':	sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					sbi(PORT(SEG_F_PORT),SEG_F_PIN);
					sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;

		case '5':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					sbi(PORT(SEG_D_PORT),SEG_D_PIN);
					sbi(PORT(SEG_F_PORT),SEG_F_PIN);
					sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;

		case '6':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					sbi(PORT(SEG_D_PORT),SEG_D_PIN);
					sbi(PORT(SEG_E_PORT),SEG_E_PIN);
					sbi(PORT(SEG_F_PORT),SEG_F_PIN);
					sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;

		case '7':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					break;

		case '8':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					sbi(PORT(SEG_D_PORT),SEG_D_PIN);
					sbi(PORT(SEG_E_PORT),SEG_E_PIN);
					sbi(PORT(SEG_F_PORT),SEG_F_PIN);
					sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;

		case '9':	sbi(PORT(SEG_A_PORT),SEG_A_PIN);
					sbi(PORT(SEG_B_PORT),SEG_B_PIN);
					sbi(PORT(SEG_C_PORT),SEG_C_PIN);
					sbi(PORT(SEG_D_PORT),SEG_D_PIN);
					sbi(PORT(SEG_F_PORT),SEG_F_PIN);
					sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;

		case '-':	sbi(PORT(SEG_G_PORT),SEG_G_PIN);
					break;
	}
}


//////////////////////////////////////////////////////////////////////////////////
//  void EnableDigit(unsigned char Display, unsigned char Digit)				//
//																				//
//	This function 'switches on' a character within the 2x4 7segment displays.	//
//  It is called by the multiplexing routine									//
//																				//
//////////////////////////////////////////////////////////////////////////////////

void EnableDigit(unsigned char Display, unsigned char Digit)
{

	// First turn all character enable lines off (so we start from a known state)
	cbi(PORT(D1D1_ENABLE_PORT), D1D1_ENABLE_PIN);
	cbi(PORT(D1D2_ENABLE_PORT), D1D2_ENABLE_PIN);
	cbi(PORT(D1D3_ENABLE_PORT), D1D3_ENABLE_PIN);
	cbi(PORT(D1D4_ENABLE_PORT), D1D4_ENABLE_PIN);
	cbi(PORT(D2D1_ENABLE_PORT), D2D1_ENABLE_PIN);
	cbi(PORT(D2D2_ENABLE_PORT), D2D2_ENABLE_PIN);
	cbi(PORT(D2D3_ENABLE_PORT), D2D3_ENABLE_PIN);
	cbi(PORT(D2D4_ENABLE_PORT), D2D4_ENABLE_PIN);

	// Now enable the correct digit:
	if(Display==BOTTOM)
	{
		switch(Digit)
		{
			case 1: sbi(PORT(D1D1_ENABLE_PORT), D1D1_ENABLE_PIN); break;
			case 2: sbi(PORT(D1D2_ENABLE_PORT), D1D2_ENABLE_PIN); break;
			case 3: sbi(PORT(D1D3_ENABLE_PORT), D1D3_ENABLE_PIN); break;
			case 4: sbi(PORT(D1D4_ENABLE_PORT), D1D4_ENABLE_PIN); break;
		}
	}
	if(Display==TOP)
	{
		switch(Digit)
		{
			case 1: sbi(PORT(D2D1_ENABLE_PORT), D2D1_ENABLE_PIN); break;
			case 2: sbi(PORT(D2D2_ENABLE_PORT), D2D2_ENABLE_PIN); break;
			case 3: sbi(PORT(D2D3_ENABLE_PORT), D2D3_ENABLE_PIN); break;
			case 4: sbi(PORT(D2D4_ENABLE_PORT), D2D4_ENABLE_PIN); break;
		}
	}


}
