/*
		Interface for 2 Hall effect flow sensors (Jaycar PN ZD1202)
		The sensors produce a pulse every time a certain volume of liquid has passed. Nominally 500uL; the value can be adjusted in FLOW.h

		For the intended application the flow rates can be too small to accurately count pulses, so instead they are timed.
		This requires 1 external interrupt and 1 8bit timer per sensor (the 8bit timer is resolution extended to 16bits)
*/

#include "FLOW.h"


unsigned int 	Pulse_Period[2][BUFFER_SIZE]; 	// Pulse_Period[0][9] holds the 9th pulse period of channel 0
unsigned char 	Pulse_Period_Pointer[2];  		// Points to the last COMPLETED sample in the pulse period array. So Pointer=7 means the most recent sample is stored in Pulse_Period[7]
unsigned char 	Timer0_Resolution_Extender;		// Used to extend an 8bit timer to 16bits
unsigned char 	Timer2_Resolution_Extender;
unsigned long 	Num_Pulses[2];


///// INPUT 1 ////
ISR (INT0_vect)
{
	Process_INT0();
}

ISR (TIMER0_OVF_vect) // Timing for INT0
{
	Process_TIMER0_OVF();
}
//////////////////


///// INPUT 2 ////
ISR (INT1_vect)
{
	Process_INT1();
}

ISR (TIMER2_OVF_vect) // Timing for INT2
{
	Process_TIMER2_OVF();
}
//////////////////


//////////////////////////////////////////////////////////////////////////////////
// void Initialise_Flow_Sensing()												//
//																				//
//	Ensures the interrupt pins and 8bit timers are correctly configured			//
//																				//
//////////////////////////////////////////////////////////////////////////////////

void Initialise_Flow_Sensing()
{
	EICRA|=((1<<ISC11)|(1<<ISC10)|(1<<ISC01)|(1<<ISC00));//Both interrupts trigger on a rising edge
	sbi(EIMSK,INT0);	// Enable interrupts from INT0
	sbi(EIMSK,INT1);	// Enable interrupts from INT1

	TCCR0B|=(1<<CS12);			   //256 prescaler - @8MHz, one tick is 32us. With an 8bit resolution extender, 2.1s capacity
	TCCR2B|=((1<<CS22)|(1<<CS21)); //256 prescaler

	// Configure the interrupt pins (PORTD.2 & PORTD.3) as inputs
	cbi(DDRD,2);
	cbi(DDRD,3);
	sbi(PORTD,2);
	sbi(PORTD,3);

	Timer0_Resolution_Extender=0;
	Timer2_Resolution_Extender=0;

	TCNT0=0x00;
	TCNT2=0x00;

	TIMSK2|=(1<<TOIE2);
	TIMSK0|=(1<<TOIE0);
}


//////////////////////////////////////////////////////////////////////////////////
// void Process_INT0() / Process_INT1()											//
//																				//
//	When an edge comes in on the INT pin we need to:							//
//		1) Stop the timer and record how long ago the previous pulse was		//
//		2) Start the timer to record until the next pulse						//
//																				//
//	There is also some basic debouncing in here, otherwise we would record a	//
//	bunch of short (fake) periods every transition. I've done this by insisting	//
//	that it's not a real pulse unless it's taken more than 5.4ms				//
//////////////////////////////////////////////////////////////////////////////////


void Process_INT0()
{
	if(Timer0_Resolution_Extender>0) // Debouncing - 1 tick on the resolution extender is 5.4ms. This is long enough to avoid bounce but quick enough for typical pulse rates.
	{

		//This is a real edge
		Num_Pulses[FLOW_INPUT_1]++;
		SaveDataPoint(FLOW_INPUT_1);
		TCNT0=0x00;
		Timer0_Resolution_Extender=0;
	}
}

void Process_INT1()
{
	if(Timer2_Resolution_Extender>0) // Debouncing - 1 tick on the resolution extender is 5.4ms. This is long enough to avoid bounce but quick enough for typical pulse rates.
	{
		//This is a real edge
		Num_Pulses[FLOW_INPUT_2]++;
		SaveDataPoint(FLOW_INPUT_2);
		TCNT2=0x00;
		Timer2_Resolution_Extender=0;
	}
}


//////////////////////////////////////////////////////////////////////////////////
// void Process_TIMER0_OVF() / Process_TIMER2_OVF()								//
//																				//
//	I'm using 8 bit timers to measure the pulse period just because I don't 	//
//	have two 16bit timers available. To cover the full range of possible periods//
//	we need to be able to time for quite a while (~1second) but also with  		//
//	high precision. So it really does need to be 16bit. 						//
//																				//
//	To get around the hardware limitations I'm extending the resolution with 	//
//	an 8bit char - every time the 8bit overflows you increment the char by one. //
//	This is in effect a 16 bit timer.											//
//																				//
//////////////////////////////////////////////////////////////////////////////////

void Process_TIMER0_OVF()
{
	if(Timer0_Resolution_Extender<255)
		Timer0_Resolution_Extender++;
	else	// This has taken too long to be a normal pulse... if you keep waiting you'll probably wait forever. So just save what you've got:
		SaveDataPoint(FLOW_INPUT_1);
}

void Process_TIMER2_OVF()
{
	if(Timer2_Resolution_Extender<255)
		Timer2_Resolution_Extender++;
	else
		SaveDataPoint(FLOW_INPUT_2);
}


//////////////////////////////////////////////////////////////////////////////////
// void SaveDataPoint(unsigned char ChannelNumber)								//
//																				//
//	Every time an edge comes in you need to look at the timer and record    	//
//	what the pulse period was. This function does that, saving to a buffer (int	//
//	array). Once the buffer is full we start overwriting it; there is a pointer	//
//	system in place to keep track of where any given sample should be stored	//
//	in the buffer																//
//																				//
//////////////////////////////////////////////////////////////////////////////////


void SaveDataPoint(unsigned char ChannelNumber)
{
	if(Pulse_Period_Pointer[ChannelNumber]<(BUFFER_SIZE-1))
		Pulse_Period_Pointer[ChannelNumber]++;
	else
		Pulse_Period_Pointer[ChannelNumber]=0;

	if(ChannelNumber==FLOW_INPUT_1)
		Pulse_Period[ChannelNumber][Pulse_Period_Pointer[ChannelNumber]]=TCNT0+(Timer0_Resolution_Extender<<8);
	else
		Pulse_Period[ChannelNumber][Pulse_Period_Pointer[ChannelNumber]]=TCNT2+(Timer2_Resolution_Extender<<8);

}

//////////////////////////////////////////////////////////////////////////////////
// unsigned int AveragePulsePeriod(unsigned char ChannelNumber)					//
//																				//
//	Calculates the average pulse period for the specified input channel. The    //
//  number of samples averaged is specified by the INTEGRATION_COUNT define.    //
//  Note that the period returned is in terms of clock ticks, not milliseconds  //
//////////////////////////////////////////////////////////////////////////////////

unsigned int AveragePulsePeriod(unsigned char ChannelNumber)
{
	unsigned char i;
	unsigned char j=Pulse_Period_Pointer[ChannelNumber];
	unsigned long Total=0;

	for(i=0;i<INTEGRATION_COUNT;i++)
	{
		Total+=Pulse_Period[ChannelNumber][j];
		if(j>0)
			j--;
		else
			j=BUFFER_SIZE-1;   // This business with j is to deal with having to wrap around the (circular) buffer
	}

	// You've now calculated the total in clock ticks
	return Total/INTEGRATION_COUNT;
}

//////////////////////////////////////////////////////////////////////////////////
// 	unsigned int PulsePeriod2FlowRate(unsigned int period)
//
//	Given a pulse period this will do the maths and tell you what flowrate it corresponds to
//  (in units of 10x LPH, so 32.1 LPH would be reported as 321)
//
//  The calculation goes:
//  LPH	= (microlitres per pulse) * (litres per microlitre) * (pulses per clock tick) * (clock ticks per second) * (seconds per hour)
//		= (microlitres per pulse) * 	1/1000000			*	  1/period		   *       31250			  *    3600
//		= ((microlitres per pulse) * 112.5 ) / period
//
//	(We've assumed an 8MHz clock with a 256 prescaler to get the 31250 figure)
//
//	But because this is an 8bit microcontroller and we have to keep everything as integers, we'll put in another
//  factor of 100, do the maths then remember to divide by 100 at the end. So the final calculation looks like:
//  LPH = ((microlitres per pulse) * 11250 ) / period
//
//////////////////////////////////////////////////////////////////////////////////

unsigned int PulsePeriod2FlowRate(unsigned int period)
{
	unsigned long temp=MICROLITRES_PER_PULSE;
	unsigned int temp2;
	temp=(temp*11250)/period;

	// Round accurately to 1dp
	temp2=(temp/10)*10;
	temp2=temp-temp2;
	if(temp2>=5)
		temp+=10;
	if(temp>100)
		return (unsigned int) (temp/10);
	else // It's an infinite pulse ... the flow has stopped! Shall we call that 9999 or 0? I chose 0.
		return 0;
}

//////////////////////////////////////////////////////////////////////////////////
// 	unsigned int PulsePeriod2Milliseconds(unsigned int period)					//
//																				//
//	Converts a period from units of timer 'ticks' to milliseconds				//
//																				//
//  At 8MHz with a 256 prescaler, 1 timer 'tick' is equal to 32microseconds,	//
//  the maths follows from there.												//
//////////////////////////////////////////////////////////////////////////////////

unsigned int PulsePeriod2Milliseconds(unsigned int period) //returns the period in ms
{
	unsigned long temp=period;
	unsigned int temp2;
	temp=(temp*32)/100;
	temp2=(temp/10)*10;
	temp2=temp-temp2;
	if(temp2>=5)
		temp+=10;
	if(temp<2000)
		return (temp/10);
	else
		return 0;
}

//////////////////////////////////////////////////////////////////////////////////
// 	unsigned int Differential_FlowRate()										//
//																				//
//	Returns the difference between the two measured flows 						//
//  (in units of 10x LPH, so 321 is 32.1)										//
//																				//
//  Always returns the absolute value! (i.e. always a positive number)			//
//																				//
//////////////////////////////////////////////////////////////////////////////////
unsigned int Differential_FlowRate()
{
	unsigned int temp0=PulsePeriod2FlowRate(AveragePulsePeriod(0));
	unsigned int temp1=PulsePeriod2FlowRate(AveragePulsePeriod(1));
	if(temp1>temp0)
		return temp1-temp0;
	else
		return temp0-temp1;
}

//////////////////////////////////////////////////////////////////////////////////
// 	unsigned int TotalFuelUsed()												//
//																				//
//	Returns the integrated differential flow rate (i.e. what volume of fluid   	//
//  has gone in but not out) since the unit was powered up.						//
//	Units are 10x LPH, so 302 would mean 30.2L									//
//																				//
//////////////////////////////////////////////////////////////////////////////////
unsigned int TotalFuelUsed()
{
	unsigned long temp;
	unsigned int temp2;

	if(Num_Pulses[FLOW_INPUT_2]>Num_Pulses[FLOW_INPUT_1])
		temp=(((Num_Pulses[FLOW_INPUT_2]-Num_Pulses[FLOW_INPUT_1])*MICROLITRES_PER_PULSE)/10000);
	else
		temp=(((Num_Pulses[FLOW_INPUT_1]-Num_Pulses[FLOW_INPUT_2])*MICROLITRES_PER_PULSE)/10000);

	// Round accurately to 1dp
	temp2=(unsigned int) ((temp/10)*10);
	if((temp-temp2)>=5)
		temp+=10;

	return (unsigned int) temp/10;
}





