

; Latching Relay 
; includes watchdog timer to allow a wake from sleep to save quiescent power 

	list P=16F88
	#include p16f88.inc
	ERRORLEVEL -302
	ERRORLEVEL -306

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB3 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

EEPROM1			equ	H'00'	; non-volatile storage for MOM_SET

; Bank 0 RAM
DELCNT			equ	H'20'	; delay counter 
BATTERY			equ	H'21'	; battery voltage
TIMER			equ	H'22'	; timer for momentary operation
PULSE			equ	H'23'	; pulse period setting
SECONDS			equ	H'24'	; counter for seconds
EDGE			equ	H'25'	; edge select (H, L or H+L)
OPERATION		equ	H'26'	; operation (toggle, follow or momentary)		
MULTIPLIER		equ	H'27'	; timer multiplier
WATCHDOG_COUNT	equ	H'28'	; watchdog timeout counter
VALUE_1			equ	H'29'	; delay counter 1
VALUE_2			equ	H'2A'	; delay counter 2
CYCLE_LOOP		equ	H'2B'	; cycle loop counter for battery voltage measurement
BATT_LOW		equ	H'2C'	; battery low flag
TEMP			equ	H'2D'	; working register
TEMP2			equ	H'2E'	; working register
TIMER1			equ	H'2F'	; counter timer
TIMER2			equ	H'30'	; counter timer
INPUT_LEV		equ	H'31'	; input level
RELAY_ON_OFF	equ	H'32'	; relay on or off 
TIMER1_WORKING	equ	H'33'	; working counter timer
TIMER2_WORKING	equ	H'34'	; working counter timer
TIMER_RUN		equ	H'35'	; timer run flag
MOM_SET			equ	H'36'	; momentary set. bit 0 determines powerup and timing state
TIMERH			equ	H'37'	; timer1 store ms byte
TIMERL			equ	H'38'	; timer1 store ls byte
COUNTER			equ	H'39'	; watchdog timer counter

; Math routine divide
AARGB0			equ H'69'	; argument Ms
AARGB1			equ H'6A'	; argument ls
AARGB3			equ	H'6B'
BARGB0			equ H'6C'	; argument Ms
REMB0			equ	H'6D'	; remainder
TEMP1			equ	H'6E'	; temp register
LOOPCOUNT		equ	H'6F'	; loop counter

; All Banks RAM
; Interrupt store registers 
W_TMP			equ	H'70'	; storage of w before interrupt
STATUS_TMP		equ	H'71'	; status storage before interrupt

	ORG     2100

	DE	D'00'	; default timing state and startup state

; start at memory 0
	org	0
	goto	SETUP
	org	4
	goto	INTERRUPT


;**********************************************************************************************

SETUP

	clrf	PORTB		; outputs low
	clrf	PORTA
; set inputs/outputs
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'11001101'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'00111100'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'10000111'	; settings (pullups disabled, TMR0/256)
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'01100100'	; AN2, AN5 and AN6 are analog inputs
	movwf	ANSEL
	movlw	B'00000000'	; left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'11000100'	; Fosc, channel 2 etc
	movwf	ADCON0
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000000'	; 31.25kHz operation. Period = (1/31.25kHz)= 32us
	movwf	OSCCON		; 
	bcf		STATUS,RP0	; select memory bank 0
; timer 1
	movlw	B'00000001'	; timer 1 prescaler /1, fosc/4. count every  32us x 4 = 128us
	movwf	T1CON		; overflow at 8.3886080s (128us x 65536)
	bsf		T1CON,0		; timer 1 on

	bsf		STATUS,RP1	; select memory bank 2
	movlw	B'00001010'	; watchdog prescaler
	movwf	WDTCON		; divide by 512 for 32.768ms timeout
	bcf		STATUS,RP1	; select memory bank 0

; initial conditions
INITIAL
	bsf		PORTB,1			; set high	 
	clrf	TIMER1_WORKING	; timer
	clrf	TIMER2_WORKING
	clrf	BATT_LOW		; battery low flag
	clrf	TIMER_RUN		; timer run
	movwf	CYCLE_LOOP		; cycle loop counter
	movlw	D'6'
	movwf	COUNTER

; start up delay
	movlw	D'255'			; set delay period value 2 
	movwf	VALUE_2			; VALUE_2 has 255
	call	LP_3			; delay
	
; measure and find selected values and operation

	call	OPTIONS			; find selected options H/H+L/L edge
							; toggle/momentary/follow input
							; timer multiplier
	call	TIMER_VAL		; timer and pulse values
	
	movlw	EEPROM1
	call	EEREAD			; get current MOM_SET value
	movwf	MOM_SET

; check S2 option at startup
	btfss	PORTB,3			; if set then momentary option setting
	goto	INPUT_READ
	incf	MOM_SET,w
	call	EEWRITE			; write to EEPROM	
	movlw	EEPROM1
	call	EEREAD			; get current MOM_SET value
	movwf	MOM_SET
LOOP_TILL_OPEN
	call	DELAY3
	btfsc	PORTB,3			; if set then option setting
	goto	LOOP_TILL_OPEN	; wait for switch
INPUT_READ
	movf	PORTB,w			; read RB2
	andlw	B'00000100'		; single out RB2
	movwf	INPUT_LEV		; input level

ALLOW_INTERRUPTS
; allow interrupts
	bsf		STATUS,RP0		; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 overflow interrupt
	bcf		STATUS,RP0		; select memory bank 0
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag
	bsf		INTCON,PEIE		; enable periperal interrupts 	 
	bsf		INTCON,GIE

; initialise relay
; check if follow option
; check edge operation
	movf	OPERATION,w		; operation (toggle, follow or momentary)
	xorlw	B'00000011'		; follow		
	btfss	STATUS,Z		; check for equal (follow)
	goto	OTHER_OP

; input monitor. monitor RB2. Look for change 

	movf	PORTB,w			; 
	andlw	B'00000100'		; 
	movwf	TEMP			; RB2 value stored
	comf	TEMP,w
	andlw	B'00000100'		; make sure input level store differs from actual level
	movwf	INPUT_LEV		; place in input level store
	goto	CYCLE

OTHER_OP
	movf	OPERATION,w		; operation (toggle, follow or momentary)
	btfss	STATUS,Z		; check for equal (toggle)
	goto	MOM_OP

; Alternate state
; initial state set by Multiplier
ALTER_STATE
	movf	MULTIPLIER,w
	movwf	RELAY_ON_OFF	; relay state at power up
	btfsc	RELAY_ON_OFF,0	; check requirement status 
	goto	RELAY_START

; reset (RA7, RB5 high)
	bcf		PORTB,4
	bcf		PORTA,6
	bsf		PORTA,7
	bsf		PORTB,5
	goto	PULSE_DELAY_X
	
RELAY_START
	
; set (RB4, RA6 high)
	bcf		PORTA,7
	bcf		PORTB,5
	bsf		PORTB,4
	bsf		PORTA,6

PULSE_DELAY_X
	movf	PULSE,w
	call	PULSE_DELAY		; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7
	
	goto	CYCLE

MOM_OP ; momentary operation
	btfsc	MOM_SET,0
	goto	ALT_PWER
	call	RELAY_OFF		; relay off at power up
	goto	CYCLE
ALT_PWER
	call	RELAY_ON
	
CYCLE ; beginining of normal running loop
; Loop using sleep function and watchdog timer to waken after a 32.768ms sleep

	clrwdt					; watchdog timer cleared
	bsf		STATUS,RP1		; select memory bank 2
	bsf		WDTCON,0		; watchdog on /512 for 32.768ms timeout
	sleep					; stop operations

; awakes with watchdog timeout
	bcf		WDTCON,0		; watchdog off
	bcf		STATUS,RP1		; select memory bank 0


	movf	TIMER_RUN,w		; check if timer 1 time period is running (only during timer for 0-5m and 0-5h)
	btfsc	STATUS,Z		; when TIMER_RUN is clear, bypass timer1 correction
	goto	DRIVE_INPUTS

	movf	COUNTER,w
	btfsc	STATUS,Z		; when zero compensate timer
	goto	COMP_TIMER1	
	decfsz	COUNTER,f
	goto	DRIVE_INPUTS
	
; add to timer1 to compensate for the loss of count during sleep. Add compensation for timer loss
; during addition code
COMP_TIMER1	
	movlw	D'6'
	movwf	COUNTER
; stop timer 1	
	bcf		T1CON,0			; timer 1 off
	bcf		STATUS,GIE		; stop interrupts

; read timer 1 and store
	movf	TMR1L,w			; timer 1 low byte
	movwf	TIMERL			; working timer value
	movf	TMR1H,w			; timer 1 high byte
	movwf	TIMERH			; working timer value

; add offsets to working timer value

	movlw	D'6'			; timer	(255 x COUNTER value)the 255 to timer compensates for 32.768ms of watchdog timer
	addwf	TIMERH,f		; add to high byte required due to the watchdog timer period
 	btfsc	STATUS,C		; if carry then set flag
	bsf		PIR1,TMR1IF		; set flag
	movlw	D'21'			; compensate loss due to shutting timer off to read/write
	addwf	TIMERL,f
	movlw	D'01'			; be ready with a 1
	btfsc	STATUS,C		; if carry
	addwf	TIMERH,f		; increase ms byte
	btfsc	STATUS,C		; if carry then set flag
	bsf		PIR1,TMR1IF		; set flag
; transfer working timer values to to timer1
	movf	TIMERH,w
	movwf	TMR1H
	movf	TIMERL,w
	movwf	TMR1L			; 
	bsf		T1CON,0			; timer 1 on
	bsf		STATUS,GIE		; re-enable interrupt

DRIVE_INPUTS
; drive RA3, RA4 and RB0 low to keep output from floating
	
	movf	PORTA,w
	andlw	B'11100111'
	movwf	PORTA			; RA3, RA4 low (when set as outputs)
	bcf		PORTB,0			; RB0 low (when set as an output)

; RA3, RA4 set as a low output
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00100110'		; RA3, RA4 output
	movwf	TRISA			; port A data direction register

; RB0 set as a low output
	movlw	B'11001100'		; RB0 output
	movwf	TRISB			; port B data direction register

; RA3, RA4 set as an input
	movlw	B'00111110'		; RA3 input
	movwf	TRISA			; port A data direction register
; RB0 set as an input
	movlw	B'11001101'		; RB0 input
	movwf	TRISB			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0

; Loop

	decfsz	CYCLE_LOOP,f
	goto	CYCLE2
;	movlw	D'xx'			; set cycle count. These two instructions only required if CYCLE_LOOP is not 255
;	movwf	CYCLE_LOOP
	
; check battery if low <3.83V (=11.5V when VR1 = 11k). High batt over 12V (or 4V at input)
	call	BATTERY_V		; measure battery voltage
	movf	BATTERY,w
	sublw	D'195'			; take battery voltage from  11.5V (3.83V after division)(D195)
	btfss	STATUS,C		; when positive then under voltage
	goto	CK_BATT_HI		; check if battery is above 12V 

; battery low
	btfsc 	BATT_LOW,0		; if battery low flag set already relay off
	goto	CK_BATT_HI
	bsf		BATT_LOW,0		; set battery low flag
	bcf		PORTB,1			; low on low battery saving power.
; switch off relay when battery is low. 
; find Set/Reset selection
	bcf		PORTA,0			; drive off to optocoupler LED and Set/Reset input low
	btfsc	PORTA,1			; set (high) reset (low)
	goto	SET_HI1
SET_LO1
; reset (RA7, RB5 high)
	bcf		PORTB,4
	bcf		PORTA,6
	bsf		PORTA,7
	bsf		PORTB,5
	goto	PULSE_DELAY1
SET_HI1
; set (RB4, RA6 high)
	bcf		PORTA,7
	bcf		PORTB,5
	bsf		PORTA,6
	bsf		PORTB,4

PULSE_DELAY1
	movf	PULSE,w
	call	PULSE_DELAY		; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7

CK_BATT_HI	; check if battery is above 12V 
	btfss	BATT_LOW,0		; check if battery low flag is set
	goto	CYCLE2			; if not low battery bypass battery check
	movf	BATTERY,w
	sublw	D'204'			; take battery voltage from 12V (4V after division)(D'204')
	btfsc	STATUS,C		; when negative then over 12V
	goto	CYCLE2								 

	clrf	BATT_LOW		; battery low flag cleared

	bsf		PORTB,1			; set pullup high
	movlw	D'250'			; delay to charge input capacitor
	call	PULSE_DELAY		; delay for charging (w=255V=500ms)

; if follow mode change relay state flag so relay switches to correct state

	movf	OPERATION,w
	xorlw	H'03'			; follow
	btfss	STATUS,Z
	goto	RETURN_STATE
; input monitor. monitor RB2. Look for change 

	movf	PORTB,w			; 
	andlw	B'00000100'		; 
	movwf	TEMP			; RB2 value stored
	goto	HI_BATT_RETURN
	
RETURN_STATE
; if toggle mode (Alternate) set to power up state

	movf	OPERATION,w
	btfsc	STATUS,Z
	goto	ALTER_STATE		; alternate power up state
; 
; if momentary set relay off
	movf	OPERATION,w
	xorlw	H'01'			; momentary
	btfss	STATUS,Z
	goto	CYCLE2
	btfsc	MOM_SET,0
	goto	ALT_PWER1
	call	RELAY_OFF		; relay off at power up
	goto	CYCLE2
ALT_PWER1
	call	RELAY_ON

CYCLE2
	
; look at RB3 for a high via SET switch

	btfss	PORTB,3			; if high SET switch is pressed
	goto	INPUT

; switch relay off
	bcf		RELAY_ON_OFF,0	; relay requirement flag
	

; drive the relay off (see set/reset)
; set PULSE_RUN flag
	bcf		PORTA,0			; drive off to optocoupler LED and Set/Reset input
	btfss	PORTA,1			; set (high) reset (low)
	goto	SET_HI2X
RESET_LO2X
	btfsc	MOM_SET,0		; timing state for relay
	goto	S_HI2X
; reset (RA6, RB4 high)
R_LO2X
	bcf		PORTA,7
	bcf		PORTB,5
	bsf		PORTA,6
	bsf		PORTB,4
	goto	PULSE_DELY2X
SET_HI2X
	btfsc	MOM_SET,0		; timing state for relay
	goto	R_LO2X
; set (RB5, RA7 high)
S_HI2X
	bcf		PORTB,4
	bcf		PORTA,6
	bsf		PORTA,7
	bsf		PORTB,5

PULSE_DELY2X
	clrf 	TIMER_RUN	; end of Timer
	movf	PULSE,w
	call	PULSE_DELAY	; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7

	call	OPTIONS			; find selected options H/H+L/L edge
							; toggle/momentary/follow input
							; timer multiplier
	call	TIMER_VAL		; timer and pulse values
	clrf	TIMER_RUN		; timer off

; check if RB3 still high
LOOP_S2
	btfss	PORTB,3
	goto	INPUT
	bsf		PORTA,0			; keep set till S2 is open	
	call	DELAY2
	goto	LOOP_S2

INPUT
; input monitor. monitor RB2. Look for change 
; avoid input check when there is a low battery
	btfsc	BATT_LOW,0		; low batt. flag
	goto	CYCLE
	movf	PORTB,w			; 
	andlw	B'00000100'		; 
	movwf	TEMP			; RB2 value stored
	xorwf	INPUT_LEV,w		; compare with previous input level for RB2
	btfsc	STATUS,Z		; when equal no change
	goto	CYCLE

HI_BATT_RETURN
	movf	TEMP,w
	movwf	INPUT_LEV		; store new input

	movlw	D'50'			; delay to prevent relay change at power off
	call	PULSE_DELAY		; delay for charging (w=255V=500ms)

; changed input

; test for follow operation
; check edge operation
	movf	OPERATION,w		; operation (toggle, follow or momentary)
	xorlw	B'00000011'		; follow		
	btfsc	STATUS,Z		; check for equal (follow)
	goto	FOLLOW_RUN

; toggle or momentary options remaining
; check edge select
	movf	EDGE,w			; edge select (H, L or H+L)
	btfss	STATUS,Z
	goto 	CK_L_LH
; change on low edge
	btfsc	TEMP,2			; RB2 input level	
	goto	CYCLE			; not a high to low so bypass
; yes low edge
; check required operation
	movf	OPERATION,w		; operation (toggle, follow or momentary)		
	btfss	STATUS,Z		; check for zero (toggle)
	goto	NEXT_OP

; toggle
ALTERNATE
	btfsc	BATT_LOW,0		; if battery low bypass relay change
	goto	CYCLE
	btfss	RELAY_ON_OFF,0	; check requirement status and toggle requirement
	goto	SET_RELAY_ON
SET_RELAY_OFF
	bcf		RELAY_ON_OFF,0	; relay requirement flag
	call	RELAY_OFF
	goto	CYCLE
SET_RELAY_ON
	bsf		RELAY_ON_OFF,0	; relay requirement flag
	call	RELAY_ON
	goto	CYCLE

CK_L_LH
; low or L+H edges
	movf	EDGE,w			; edge select (H, L or H+L)
	xorlw	D'1'
	btfss	STATUS,Z		
	goto 	CK_LH
; change on high edge
	btfss	TEMP,2			; RB2 input level	
	goto	CYCLE			; not a high to low so bypass
; yes high edge
; check required operation
	movf	OPERATION,w		; operation (toggle, follow or momentary)		
	btfss	STATUS,Z		; check for zero (toggle)
	goto	NEXT_OP
; toggle
	goto	ALTERNATE

CK_LH
; low and high
	movf	EDGE,w			; edge select (H, L or H+L)
	xorlw	D'3'
	btfss	STATUS,Z		
	goto 	CYCLE
; check required operation
	movf	OPERATION,w		; operation (toggle, follow or momentary)		
	btfsc	STATUS,Z		; check for zero (toggle)
	goto	ALTERNATE

NEXT_OP
	
; momentary
; set timer if action is for momentary selection

	movf	OPERATION,w	; check if momentary
	xorlw	D'01'
	btfss	STATUS,Z	; if momentary check if Timer_run flag set
	goto	CYCLE

; check for H, H+L and L edge
	movf	EDGE,w			; edge select (H, L or H+L)
	btfss	STATUS,Z
	goto 	CK_L_LH1
; change on low edge
	btfsc	TEMP,2			; RB2 input level	
	goto	CYCLE			; not a low to high so bypass
	goto	RUN_MOMENTARY
; low or L+H edges
CK_L_LH1
	movf	EDGE,w			; edge select (H, L or H+L)
	xorlw	D'1'
	btfss	STATUS,Z		
	goto 	CK_LH1
; change on high edge
	btfss	TEMP,2			; RB2 input level	
	goto	CYCLE			; not a high to low so bypass
	goto	RUN_MOMENTARY

CK_LH1
	movf	EDGE,w			; edge select (H, L or H+L)
	xorlw	D'3'
	btfss	STATUS,Z		; change on H+L		
	goto 	CYCLE

; if timer is 0-50s then use a delay loop rather than timer1

RUN_MOMENTARY
; check if 0-50s
	movf	MULTIPLIER,w
	btfss	STATUS,Z		; when zero 0-50s
	goto	MIN_HOUR	
; relay on
	clrf	TIMER_RUN		; flag off
;	bsf		RELAY_ON_OFF,0	; relay requirement flag
	btfsc	BATT_LOW,0		; if battery low bypass relay on
	goto	CYCLE

	btfsc	MOM_SET,0		; timing state for relay
	goto	ALT_MOM1
	call	RELAY_ON
	goto 	CONT_MOM1
ALT_MOM1
	call	RELAY_OFF

CONT_MOM1
; VR3, 0-255 = 0-50s. ~196ms/step with 6 x 32.768ms watchdog timeouts/ 196ms step
; TIMER value sets delay

	movf	TIMER,w		; get timer value
   	movwf	DELCNT		; working counter
	btfsc	STATUS,Z	; if DELCNT=0 bypass timeout
	goto	RESET_RELAY	; reset relay	

; Loop using sleep function and watchdog timer to waken after a 32.768ms sleep

	clrwdt					; watchdog timer cleared
	bsf		STATUS,RP1		; select memory bank 2
	bsf		WDTCON,0		; watchdog on /512 for 32.768ms timeout
	bcf		STATUS,RP1		; select memory bank 0
SET_COUNT	
	movlw	D'06'			; initialise to 6	
	movwf	WATCHDOG_COUNT	; watchdog timeout counter
WAIT
	btfsc	PORTB,3
	goto	END_OF_TIMER
	sleep					; stop operations; awakes with watchdog timeout

; drive RA3, RA4 and RB0 low to keep output from floating
	movf	PORTA,w
	andlw	B'11100111'
	movwf	PORTA
	bcf		PORTB,0	
; RA3, RA4 set as a low output
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00100110'		; RA3, RA4 output
	movwf	TRISA			; port A data direction register
; RB0 set as a low output
	movlw	B'11001100'		; RB0 output
	movwf	TRISB			; port B data direction register
; RA3, RA4 set as an input
	movlw	B'00111110'		; RA3 input
	movwf	TRISA			; port A data direction register
; RB0 set as an input
	movlw	B'11001101'		; RB0 input
	movwf	TRISB			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0

; watchdog timer count
	decfsz	WATCHDOG_COUNT,f; count of 6-timeouts for 196ms
	goto	WAIT
	decfsz	DELCNT,f		; timeout delay (0-256 steps)
	goto	SET_COUNT

; end of counter period
END_OF_TIMER
	bsf		STATUS,RP1		; select memory bank 2
	bcf		WDTCON,0		; watchdog off
	bcf		STATUS,RP1		; select memory bank 0
	clrwdt

; reset 
RESET_RELAY
	movf	PORTB,w			; ensure any RB2 level changes that occurred during timing are ignored
	andlw	B'00000100'		; 
	movwf	INPUT_LEV		; place in input level store

; check state
	btfsc	MOM_SET,0		; timing state for relay
	goto	SET_RELAY_ON
	goto	SET_RELAY_OFF

MIN_HOUR
	btfsc	TIMER_RUN,0		; when set bypass because already timing
	goto	CYCLE

; clear timer1
	bcf		T1CON,0		; timer 1 off
	bcf		STATUS,GIE
	clrf	TMR1L
	clrf	TMR1H
	bcf		PIR1,TMR1IF		; clear flag
	bsf		STATUS,GIE
	bsf		T1CON,0		; timer 1 on
	
	movf	MULTIPLIER,w
	xorlw	D'3'
	btfss	STATUS,Z		; when 3 0-5m
	goto	HOURS

; timer 1 is 8.3886080s. (128us x 65536)
; for 0-5m (0-300s) use a multiply counter of up to 36 for 302s
	movf	TIMER,w
	movwf	AARGB1		; then divide max 255 by 7 for max of 36
	clrf	AARGB0
	movlw	D'7'
	movwf	BARGB0
	call	DIV16_8		; divide 255 (max) by 7
	movf	AARGB1,w	; result
	btfsc	STATUS,Z	; if zero set at 1
	movlw	D'1'
	movwf	TIMER1
	movwf	TIMER1_WORKING
	movlw	D'01'		; minimum of 1 
	movwf	TIMER2
	movwf	TIMER2_WORKING
	bsf		TIMER_RUN,0	; timer run flag set (this then has the interrupt switch off the relay at timeout)
; relay on
;	bsf		RELAY_ON_OFF,0	; relay requirement flag
	btfsc	BATT_LOW,0		; if battery low bypass relay on
	goto	CYCLE

	btfsc	MOM_SET,0		; timing state for relay
	goto	ALT_MOM2
	call	RELAY_ON
	goto 	CYCLE
ALT_MOM2
	call	RELAY_OFF
	goto	CYCLE

HOURS
; 0-5H (TIMER1 = up to 128, TIMER2 = 17)

; for 0-5h use a multiply counter of up to 128 x 17 for 2176. At 8.3886s each gives 18253s=5.07h
	movf	TIMER,w
	movwf	AARGB1		; then divide max 255 by 2 for max of 128
	clrf	AARGB0
	movlw	D'2'
	movwf	BARGB0
	call	DIV16_8		; divide 255 (max) by 2
	movf	AARGB1,w	; result
	btfsc	STATUS,Z	; if zero set at 1
	movlw	D'1'
	movwf	TIMER1
	movwf	TIMER1_WORKING
	movlw	D'17'		; 128 x 17 = 2176. 8.388s x 2176=18252s or 5hours
	movwf	TIMER2
	movwf	TIMER2_WORKING
	bsf		TIMER_RUN,0	; timer run flag set (this then has the interrupt switch off the relay at tiemout)	
; relay on
;	bsf		RELAY_ON_OFF,0	; relay requirement flag
	btfsc	BATT_LOW,0		; if battery low bypass relay on
	goto	CYCLE
	btfsc	MOM_SET,0		; timing state for relay
	goto	ALT_MOM3
	call	RELAY_ON
	goto 	CYCLE
ALT_MOM3
	call	RELAY_OFF
	goto	CYCLE
	goto	CYCLE

FOLLOW_RUN
; follow
; relay follows input level
; on or off depends on High or low sense EDGE
; bit 2 in TEMP has current input level
	btfsc	TEMP,2			; if input is high turn off/on relay dependent on edge input
	goto	FOLLOW_1
	btfss	EDGE,0	
	goto	SET_RELAY_ON_M
	goto	SET_RELAY_OFF_M
FOLLOW_1
	btfss	EDGE,0
	goto	SET_RELAY_OFF_M	
	goto	SET_RELAY_ON_M

SET_RELAY_OFF_M
;	bcf		RELAY_ON_OFF,0	; relay requirement flag
	btfsc	BATT_LOW,0		; if battery low bypass operation
	goto	CYCLE
	call	RELAY_OFF_M
	goto	CYCLE
SET_RELAY_ON_M
;	bsf		RELAY_ON_OFF,0	; relay requirement flag
	btfsc	BATT_LOW,0		; if battery low bypass operation
	goto	CYCLE
	call	RELAY_ON_M
	goto	CYCLE
	
;***************************************************

; INTERRUPT
; start interrupt by saving w and status registers 
	
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP		; status in status_tmp 
	bcf		STATUS,RP0		; bank select
	bcf 	STATUS,RP1		; select memory bank 0 

	bcf		PIR1,TMR1IF		; clear flag

; check for timer end
	btfss	TIMER_RUN,0		; check if Timer_run flag set
	goto	RECLAIM
	movf	OPERATION,w		; check if momentary
	xorlw	D'01'
	btfss	STATUS,Z		; momentary 
	goto	RECLAIM

; check if timers are zero
	movf	TIMER1_WORKING,w
	btfsc 	STATUS,Z			; if zero bypass	
 	goto	T2
	decfsz	TIMER1_WORKING,f	; decrease
	goto	RECLAIM
	movf	TIMER1,w
	movwf	TIMER1_WORKING
T2	movf	TIMER2_WORKING,w
	btfsc 	STATUS,Z			; if zero bypass	
 	goto	RECLAIM
	decfsz	TIMER2_WORKING,f
	goto	RECLAIM
DRV_OFF	
; drive the relay off (see set/reset)
; set PULSE_RUN flag
	bcf		PORTA,0			; drive off to optocoupler LED and Set/Reset input
	btfss	PORTA,1			; set (high) reset (low)
	goto	SET_HI2
RESET_LO2
	btfsc	MOM_SET,0		; timing state for relay
	goto	S_HI2
; reset (RA6, RB4 high)
R_LO2
	bcf		PORTA,7
	bcf		PORTB,5
	bsf		PORTA,6
	bsf		PORTB,4
	goto	PULSE_DELY2
SET_HI2
	btfsc	MOM_SET,0		; timing state for relay
	goto	R_LO2
; set (RB5, RA7 high)
S_HI2
	bcf		PORTB,4
	bcf		PORTA,6
	bsf		PORTA,7
	bsf		PORTB,5

PULSE_DELY2
	clrf 	TIMER_RUN	; end of Timer
	movf	PULSE,w
	call	PULSE_DELAY	; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7
	bcf		T1CON,0			; timer 1 off
		
RECLAIM
; end of interrupt reclaim w and status 
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

;**************************************************************
; Subroutines

RELAY_OFF	
	btfsc	BATT_LOW,0		; if battery low flag cleared can switch relay
	return

; find Set/Reset selection
	bcf		PORTA,0			; drive off to optocoupler LED and Set/Reset input low
	btfsc	PORTA,1			; set (high) reset (low)
	goto	SET_HI3
SET_LO3
; reset (RA7, RB5 high)
	bcf		PORTB,4
	bcf		PORTA,6
	bsf		PORTA,7
	bsf		PORTB,5
	goto	PULSE_DELAY4
SET_HI3
; set (RB4, RA6 high)
	bcf		PORTA,7
	bcf		PORTB,5
	bsf		PORTA,6
	bsf		PORTB,4

PULSE_DELAY4
	movf	PULSE,w
	call	PULSE_DELAY		; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7
	return

; *****
RELAY_ON
	btfsc	BATT_LOW,0		; if battery low flag cleared can switch relay
	return

; find Set/Reset selection
	bcf		PORTA,0			; drive off to optocoupler LED and Set/Reset input low
	btfss	PORTA,1			; set (high) reset (low)
	goto	SET_HI4
SET_LO4
; reset (RA7, RB5 high)
	bcf		PORTB,4
	bcf		PORTA,6
	bsf		PORTA,7
	bsf		PORTB,5
	goto	PULSE_DELAY3
SET_HI4
; set (RB4, RA6 high)
	bcf		PORTA,7
	bcf		PORTB,5
	bsf		PORTA,6	
	bsf		PORTB,4
	
PULSE_DELAY3
	movf	PULSE,w
	call	PULSE_DELAY		; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7
	return

; *****
RELAY_OFF_M	
	btfsc	BATT_LOW,0		; if battery low flag cleared can switch relay
	return
; reset (RA7, RB5 high)
	bcf		PORTB,4
	bcf		PORTA,6
	bsf		PORTA,7
	bsf		PORTB,5
	movf	PULSE,w
	call	PULSE_DELAY		; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7
	return

; *****
RELAY_ON_M
	btfsc	BATT_LOW,0		; if battery low flag cleared can switch relay
	return
; reset (RA7, RB5 high)
	
	bcf		PORTA,7
	bcf		PORTB,5
	bsf		PORTA,6
	bsf		PORTB,4
	movf	PULSE,w
	call	PULSE_DELAY		; subroutine for pulse length (255=5V=500ms)

; clear drive after pulse
	bcf		PORTB,5
	bcf		PORTB,4
	call	DELAY3			; relay off at Q3 or Q4 first so diodes can clamp the coil 	
	bcf		PORTA,6
	bcf		PORTA,7
	return

; *****
; Battery Voltage reading
BATTERY_V

; RA0 must be high  to switch on the optocoupler allowing connection of the resistive voltage divider.

	bsf		PORTA,0			; drive optocoupler LED
	call	DELAY2			; short delay ~2ms
CH_2AD

; set analog input address
	bcf		ADCON0,5
	bsf		ADCON0,4
	bcf		ADCON0,3		; 
	call	DEL_AD			; convert to digital
	movf	ADRESH,w		; A/D value
	movwf	BATTERY			; battery voltage
	bcf		PORTA,0			; drive off to optocoupler LED and Set/Reset input
	return

; options.
OPTIONS

; edge H, H+L or L edge trigger
	clrf	EDGE			; set at 0 at start
	bsf		EDGE,0			; initial setting
	bsf		PORTA,0			; drive optocoupler LED and options link high

; RA3 set as a high output
	bsf		PORTA,3	
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00110110'		; RA3 output
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; RA3 set as an input
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00111110'		; RA3 input
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; check RA3. If low then the link ties input low, so L edge selection
	btfss	PORTA,3
	goto 	OPERATE_X		; keep EDGE value at 0

; H or H+L edge trigger
; RA3 set as a low output
	bcf		PORTA,3	
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00110110'		; RA3 output
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; RA3 set as an input
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00111110'		; RA3 input
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; check RA3. If high then the link ties input high, so H edge selection
	movlw	B'00000000'		; high
	btfss	PORTA,3	
	movlw	B'00000011'		; L+H
	movwf	EDGE			; EDGE=1 if input high, =11 if low meaning input is open
	
OPERATE_X 

; toggle, momentary or follow input
	clrf	OPERATION		; set at 0 at start
	bsf		PORTA,0			; drive optocoupler LED and options link high

; RA4 set as a high output
	bsf		PORTA,4	
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00101110'		; RA4 output
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; RA4 set as an input
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00111110'		; RA4 input
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; check RA4. If low then the link ties input low, so toggle edge selection
	btfss	PORTA,4
	goto 	MULTIPLY		; keep OPERATE value at 0

; check for a high input selection
	
; momentary or follow
; RA4 set as a low output
	bcf		PORTA,4	
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00101110'		; RA4 output
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; RA4 set as an input
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'00111110'		; RA4 input
	movwf	TRISA			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; check RA4. If high then the link ties input high, so momentary selection
	movlw	B'00000001'		; momentary
	btfss	PORTA,4	
	movlw	B'00000011'		; follow
	movwf	OPERATION		; OPERATION =1 if input high, =11 if low meaning input is open

MULTIPLY
	
; 0-50s, 0-5m, 0-5h
	clrf	MULTIPLIER		; set at 0 at start
	bsf		PORTA,0			; drive optocoupler LED and options link high

; RB0 set as a high output
	bsf		PORTB,0	
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'11001100'		; RB0 output
	movwf	TRISB			; port B data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; RB0 set as an input
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'11001101'		; RB0 input
	movwf	TRISB			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms

	btfsc	PORTB,0			; check RB0. If low then the link ties input low, so 0-50s
	goto	MULT_HI
	bcf		PORTA,0			; options link low
	return					; keep MULTIPLIER value at 0

MULT_HI
; check for a high input selection
; 0-5m, 0-5h
; RB0 set as a low output
	bsf		PORTA,0			; options link high
	bcf		PORTB,0	
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'11001100'		; RB0 output
	movwf	TRISB			; port B data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; RB0 set as an input
	bsf		STATUS,RP0		; select memory bank 1	
	movlw	B'11001101'		; RB0 input
	movwf	TRISB			; port A data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; check RB0. If high then the link ties input high, so 0-5h selection
	movlw	B'00000001'
	btfss	PORTB,0	
	movlw	B'00000011'		; 0-5m
	movwf	MULTIPLIER		; MULTIPLIER=1 if input high, =11 if low meaning input is open
	bcf		PORTA,0			; options link low
	return


; *********************

TIMER_VAL
; Timer and Pulse values

; RB3 is an input normally or a high value output 
	bsf		PORTB,3			; set output high
	bsf		STATUS,RP0		; select memory bank 1
	movlw	B'11000101'		; set portB,3 an output 
	movwf	TRISB			; port B data direction register
	bcf		STATUS,RP0		; select memory bank 0
	call	DELAY2			; short delay ~2ms
; Channel 6 A/D value (Pulse)
CH_6AD
; set analog input address
	bsf		ADCON0,5
	bsf		ADCON0,4
	bcf		ADCON0,3		; A/D No.6
	call	DEL_AD			; convert to digital
	movf	ADRESH,w		; A/D value
	movwf	PULSE

; Channel 5 A/D value (TIMER)
CH_5AD
; set analog input address
	bsf		ADCON0,5
	bcf		ADCON0,4
	bsf		ADCON0,3		; A/D No.6
	call	DEL_AD			; convert to digital
	movf	ADRESH,w		; A/D value
	movwf	TIMER
; RB3 an input
	bsf		STATUS,RP0		; select memory bank 1
	movlw	B'11001101'		; set portB,3 an input 
	movwf	TRISB			; port B data direction register
	bcf		STATUS,RP0		; select memory bank 0
	return

; ****************************
; delays

; DELAY for A/D acquisition
DEL_AD
	bsf		ADCON0,ADON		; A/D on
	movlw	D'1'			; 
	movwf	DELCNT
DEL1
	decfsz	DELCNT,f
	goto	DEL1

	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV1
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV1
	bcf		ADCON0,ADON		; A/D off
	return

; delay for pulse drive period
PULSE_DELAY				; subroutine for pulse length (255=5V=500ms DELAY)
   	movwf	DELCNT
	movf	DELCNT,w
	btfsc	STATUS,Z	; if DELCNT=0 return
	return
DELAY_P
	movlw	D'2'		; 
	movwf	VALUE_2
DELAY_M
	nop
	nop
	decfsz	VALUE_2,f
	goto 	DELAY_M
	decfsz	DELCNT,f
	goto	DELAY_P
	return	

DELAY2
	movlw	D'5'		; set delay period value 2 
DELAY_2
	movwf	VALUE_2		; VALUE_2 = w
LP_3
	decfsz	VALUE_2,f	; decrease VALUE_2 skip if zero
	goto 	LP_3
	return	

DELAY3
	movlw	D'15'		; set delay period value 2 
	movwf	VALUE_2		; VALUE_2 = w
LP_4
	decfsz	VALUE_2,f	; decrease VALUE_2 skip if zero
	goto 	LP_4
	return	

; subroutine to read EEPROM memory 

EEREAD
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM

; EEPROM write with interrupts disabled

EEWRITE

	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register

	bsf		STATUS,RP0	; bank 3
WR4	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR4			; not written yet
	bsf		EECON1,WREN	; enable write
	bsf		STATUS,RP0
	bsf		STATUS,RP1	; bank 3		
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITE1	

	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE1		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0
	bcf		STATUS,RP1	; bank 0 
	return				; value written

;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8		   	CLRF            REMB0
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B          RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
                return

 
 end
