; *****************************************************************
; Project:		 solartrak1_1.mcp
; Source file:	 solartrak1_1.asm
; Include files: P16f88.inc
;				 registers.inc
;				 constants.inc
;				 macros.inc
; last revision: 11 June, 2011
; Author:		 Herman Nacinovich
; *****************************************************************

	; **********************************************************************************
	; * NOTES: FOR WIG-WAG OUTPUT CONNECT RA4 DIRECTLY TO VCC OR USE PULLUP RESISTOR.
	; * 	   FOR SINGLE-ENDED OUTPUT CONNECT RA4 DIRECTLY TO VSS OR USE SHORTING LINK.
	; *        SPEED OUTPUT ON RB6
	; *		   REVERSE OUTPUT ON RB7
	; **********************************************************************************

	; -----------------------------------------------------------------------
	; define processor and configuration constants
	; -----------------------------------------------------------------------
	list		p=pic16f88
	__config	_CONFIG1, _BODEN_ON & _CP_OFF & _PWRTE_ON & _WDT_ON & _INTRC_IO & _CCP1_RB3 & _MCLR_OFF
	; -----------------------------------------------------------------------
	; list include files
	; -----------------------------------------------------------------------
	#include 	<p16f88.inc>
	#include	<registers.inc>
	#include	<constants.inc>
	#include 	<macros.inc>	
; ------------------ START OF PROGRAMME -------------------------
	org		0x00
initialise
	; -----------------------------------------------------------------
	banksel	OPTION_REG
	; -----------------------------------------------------------------
	; disable PORTB pullups, set TMR0 prescale to 1:4
	; -----------------------------------------------------------------
	movlw	_RBPU_DIS | _PRESC_4
	movwf	OPTION_REG		; disable PORTB pullups, prescale (TMR0) = 1:4
	; -----------------------------------------------------------------
	; set RA0, RA1 to analogue i/o mode
	; -----------------------------------------------------------------
	movlw	0x03
	movwf	ANSEL			; set RA1,RA0 analogue i/o
	; -----------------------------------------------------------------
	; set internal oscillator to 4 MHz
	; -----------------------------------------------------------------
	movlw	b'01100000'		; Fosc = 4 MHz	
	movwf	OSCCON
	; -----------------------------------------------------------------
	; set RB0, RB1, RB6 to input mode, rest to outputs
	; -----------------------------------------------------------------
	movlw	0x43			; RB0, RB1, RB6, RB7 inputs
	movwf	TRISB			; rest output
	; -----------------------------------------------------------------
	banksel	0
	; -----------------------------------------------------------------
	bcf		reverse			; ensure reverse output = 0
	movlw	_FOSC_8			
	; -----------------------------------------------------------------
	; set A/D clock to Fosc / 8
	; -----------------------------------------------------------------
	movwf	ADCON0			; A/D clock = Fosc / 8
	; -----------------------------------------------------------------
	; initialise T1CON
	; -----------------------------------------------------------------
	movlw	T1_PRESC_8
	movwf	T1CON			; set TMR1 prescale to 1:8
	; -----------------------------------------------------------------
	; initialise gp registers
	; -----------------------------------------------------------------
	clrf	flags
	clrf	easthi
	clrf	eastlo
	clrf	westhi
	clrf	westlo
	clrf	timerhi
	clrf	timerlo
; -----------------------------------------------------------------
main
	clrwdt
	call	get_east		; get input from east photocell
	call	get_west		; get input from west photocell
	call	test			; test inputs and set flags accordingly
	btfsc	FLAG_REV		; reverse mode?
	goto	motor_rev		; yes
	btfsc	FLAG_FWD		; forward mode?
	goto	motor_fwd		; yes
	; -----------------------------------------------------------------
motor_stop
	banksel	TRISB
	bsf		output			; output = hi-z
	banksel	0
	bcf		reverse			; saves power if a reversing relay is used
	bcf		FLAG_RUN
	goto	main
	; -----------------------------------------------------------------
motor_rev
	; -----------------------------------------------------------------
	; check if east limit switch was recently open
	; -----------------------------------------------------------------
	btfsc	FLAG_EASTSW		; switch recently open?
	goto	motor_stop		; yes
	; -----------------------------------------------------------------
	; check if east limit switch is currently open
	; -----------------------------------------------------------------
	btfss	eastsw			; switch currently open?
	goto	$+3				; no
	; -----------------------------------------------------------------
	bsf		FLAG_EASTSW		
	goto	motor_stop		; yes
	; -----------------------------------------------------------------
	bsf		reverse
	bcf		FLAG_WESTSW		; clear flag once in reverse mode
	; -----------------------------------------------------------------
	; if pgm_link = 1 (open) then output is bidirectional (output = 0 corresponds to motor reverse)
	; -----------------------------------------------------------------
	btfsc	pgm_link		; link open?
	bcf		output			; yes
	; -----------------------------------------------------------------
	; if pgm_link = 0 (closed) then output is unidirectional(output = 1 corresponds to either motor forward or motor reverse)
	; -----------------------------------------------------------------
	btfss	pgm_link		; link closed
	bsf		output			; yes
	; -----------------------------------------------------------------
	btfss	FLAG_RUN		; currently in run mode?
	goto	motor_start		; no
	; -----------------------------------------------------------------
	btfss	PIR1,TMR1IF		; TMR1 timeout?
	goto	main			; no
	; -----------------------------------------------------------------
	banksel	TRISB
	bcf		output			; yes: set to output mode
	; -----------------------------------------------------------------
	banksel	0
	bcf		T1CON,TMR1ON	; turn TMR1 off
	goto	main
	; -----------------------------------------------------------------
motor_fwd
	; -----------------------------------------------------------------
	; check if west limit switch was recently open
	; -----------------------------------------------------------------
	btfsc	FLAG_WESTSW		; switch recently open?
	goto	motor_stop		; yes
	; -----------------------------------------------------------------
	; check if west limit switch currently open
	; -----------------------------------------------------------------
	btfss	westsw			; west limit switch currrently open?
	goto	$+3				; no
	; -----------------------------------------------------------------
	bsf		FLAG_WESTSW		
	goto	motor_stop		; yes
	; -----------------------------------------------------------------
	bcf		FLAG_EASTSW		; clear for when reversing
	bcf		reverse			
	bsf		output
	; -----------------------------------------------------------------
	btfss	FLAG_RUN		; currently in run mode?
	goto	motor_start		; no
	; -----------------------------------------------------------------
	btfss	PIR1,TMR1IF		; TMR1 timeout?
	goto	main			; no
	; -----------------------------------------------------------------
	banksel	TRISB
	btfsc	FLAG_RUN		; currently in run mode?
	bcf		output			; yes: set port pin for output mode
	; -----------------------------------------------------------------
	banksel	0
	bcf		T1CON,TMR1ON	; turn TMR1 off
	goto	main
	; -----------------------------------------------------------------
motor_start
	; ***************************************************************
	; In some situations a relay driven by reverse output may be used.
	; For these situations a small delay is introduced  to ensure relay
	; contacts are fully open or closed before motor starts.
	; **********************************************************
	bcf		T1CON,TMR1ON	; stop TMR1 timer
	movlw	high (T1_PRESET); preset TMR1
	movwf	TMR1H
	movlw	low (T1_PRESET)
	movwf	TMR1L
	bcf		PIR1,TMR1IF		; clear TMR1 interrupt flag
	bsf		T1CON,TMR1ON	; start timer
	bsf		FLAG_RUN		; flag run mode
	goto	main
; -----------------------------------------------------------------
; function:		get_east
; description:	acquire current vaue of east sensor and
;				compare with previous (average) value
;				to obtain new (average) value
; comments:		Averaging algorithm based on the following
;				formula:
;
;				average = [(n-1)* old average + current reading)
;				(where n = d'256')
;
;				Although the A/D converter in the PIC16F88 is 
;				capable	of 10-bit resolution only the upper 8
;				bits are used (to use the full 10 bits would
;				add slight complexity). This loss in resolution 
;				is more than compensated by the averaging process
;				and in the present application it doesn't seem
; 				worth bothering to use the full 10-bit capability.
; ------------------------------------------------------
get_east
	banksel	0
	sel_east		
	call	convert			; initiate adc and wait for completion
	; ------------------------------------------
	; subtract high byte (current average) from low byte (current average)
	; ------------------------------------------
	movfw	easthi
	subwf	eastlo,F
	skpc
	decf	easthi,F
	; ------------------------------------------
	; add new measurement to current average (low byte)
	; ------------------------------------------
	movfw 	ADRESH
	addwf	eastlo,F
	; -----------------------------------------------
	; adjust current average (high byte) if necessary
	; -----------------------------------------------
	skpnc
	incf	easthi,F	; <-average value
	return
; ------------------------------------------------------
; function:		get_west
; description:	acquire current vaue of west sensor and
;				compared with previous (average) value
;				to obtain new (average) value
; comments:		see function get_east for details
; ------------------------------------------------------
get_west
	banksel	0
	sel_west
	movwf	ADCON0			; sel. RA1, adc = ON
	call	convert			; initiate adc and wait for completion
	; -------------------------------------------------------------------
	; subtract high byte (current average) from low byte (current average)
	; -------------------------------------------------------------------
	movfw	westhi
	subwf	westlo,F
	skpc
	decf	westhi,F
	; -------------+-----------------------------------
	; add new measurement to current average (low byte)
	; -------------------------------------------------
	movfw	ADRESH
	addwf	westlo,F
	; -----------------------------------------------
	; adjust current average (high byte) if necessary
	; -----------------------------------------------
	skpnc
	incf	westhi,F	; <-average value
	return
; -------------------------------------------------------------------	
; test inputs from east and west photocells and set/clear flags accordingly
; -------------------------------------------------------------------	
test
	bcf		FLAG_STOP		; default	
	btfsc	FLAG_FWD		; forward mode?	
	goto	test_east		; yes
	btfsc	FLAG_REV		; reverse mode?
	goto	test_east		; yes
	bsf		FLAG_STOP
	goto	check_timer		; else...
	; -------------------------------------------------------------------
	; test if east input is less than or greater than west input
	; -------------------------------------------------------------------	
test_east
	; -----------------------------------------------------
	; if west = east then do nothing
	; -----------------------------------------------------
	movfw	west
	subwf	east,W
	skpnz				; west = east?
	goto	test_cloud	; Yes
	; -------------------------------------------------------------------	
	; if west > east then FLAG_FWD = 1 else FLAG_FWD = 0
	; -----------------------------------------------------
	bcf		FLAG_FWD	; default
	skpc			 	; west > east?
	bsf		FLAG_FWD	; yes
	; -------------------------------------------------------------------	
	; test if inputs exceed cloud threshold
	; -------------------------------------------------------------------	
test_cloud
	; -------------------------------------------------------------------	
	movlw	THRESH_CLOUD ; default
	btfss	FLAG_FWD	; forward mode?
	addlw	d'3'		; no
	subwf	west,W
	skpc				; west < cloud threshold?
	bcf		FLAG_FWD	; yes
	; -------------------------------------------------------------------	
	; test if inputs exceed dark threshold
	; -------------------------------------------------------------------	
test_dark
	movlw	THRESH_DARK	
	btfsc	FLAG_REV	; currently in reverse mode?
	addlw	d'7'		; yes: then increase threshold for noise immunity
	; -------------------------------------------------------------------	
	bsf		FLAG_REV	; default
	subwf	west,W
	skpnc				; west > threshold?
	bcf		FLAG_REV	; yes
	; -------------------------------------------------------------------	
	sublw	0
	addwf	west,W		; restore previous value W
	subwf	east,W
	skpnc				; east > threshold?
	bcf		FLAG_REV	; yes
	btfsc	FLAG_FWD	; forward mode?
	return				; yes
	btfsc	FLAG_REV	; reverse mode
	return				; yes
	btfss	FLAG_STOP	; previously stop mode?
	goto	init_timer	; no
	return				; else...	
; -------------------------------------------------------------------	
init_timer
	movlw	high (TIMEOUT)
	movwf	timerhi
	movlw	low (TIMEOUT)
	movwf	timerlo
	return
; ---------------------------------------------------------------------------------	
; check if timer is active (timer is activated on change from run mode to stop mode)
; ---------------------------------------------------------------------------------	
check_timer
	movf	timerhi,F
	skpz					; high byte, timer = 0?
	goto	update_timer	; no
	movf	timerlo,F		
	skpnz					; low byte, timer = 0?
	goto	test_east		; yes
; -------------------------------------------------------------------	
; function:		update_timer
; description:	update timer based on TMR0
;				timer is decremented so long as timer is active
;	 		    timeout when timerhi:timerlo = 0x0000
;				note: timer is active if timerhi:timerlo > 0x0000
; -------------------------------------------------------------------	
update_timer
	btfss	INTCON,TMR0IF	; TMR0 rollover?
	return					; no
; -------------------------------------------------------------------	
	bcf		INTCON,TMR0IF
	decfsz	timerlo,F
	return
	movf	timerhi,F
	skpz					; timeout?
	decf	timerhi,F		; no
	return					
; -------------------------------------------------------------------	
; initialise A/D conversion and wait for completion
; -------------------------------------------------------------------	
convert		
	delay	d'12'
	bsf		ADCON0,GO
loop=$
	btfsc	ADCON0,GO	; conversion complete?
	goto	loop		; no
	return
; -------------------------------------------------------------------	
; generate delay from 8 us to approx. 1 ms (see <delay> macro)
; -------------------------------------------------------------------	
delay_x		
	nop
	nop
	nop
loop=$
	addlw	-1
	skpz
	goto	loop
	return
end_delay_x
; ================================================================
	end

