 

; Ultrasonic ranger for water level metering
; LED1 minimum, LED10 maximum.

	list P=16F88
	#include p16f88.inc
	ERRORLEVEL -302
	ERRORLEVEL -306

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB3 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO 

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; EEPROM
; initial settings with 331m/s sound rate at 0 degrees C (compensated over temperature)
; min calibration is for 331mm and so is 1ms send and 1ms return. 2ms is a count of D500 for 4us/count
; max calibration setting is for 1m at 3.02ms send and 3.02ms return. 6.04ms is a count of D1510 for 4us/count
; dot or bar; dot
; initially unprogrammed at HFF
; programmed at the very first startup with default values when the test location is not H55

EEPROMTST		equ	H'0'	; test location checks if EEPROM programmed or default loaded
EEPROM0			equ	H'1'	; non-volatile storage of MIN_COUNT_H, minimum calibration counter value ms byte
EEPROM1			equ	H'2'	; non-volatile storage of MIN_COUNT_L, minimum calibration counter value ls byte
EEPROM2			equ	H'3'	; non-volatile storage of MAX_COUNT_H, maximum calibration counter value ms byte
EEPROM3			equ	H'4'	; non-volatile storage of MAX_COUNT_L, maximum calibration counter value ls byte
EEPROM4			equ	H'5'	; non-volatile storage of dot or bar
EEPROM5			equ	H'6'	; non-volatile storage of COUNT_H, minimum counter value ms byte
EEPROM6			equ	H'7'	; non-volatile storage of COUNT_L, minimum counter value ls byte	

; Bank 0 RAM
DELCNT			equ	H'20'	; delay counter 
VALUE_1			equ	H'21'	; delay loop counter1
VALUE_2		 	equ	H'22'	; delay loop counter2
CYCLE			equ	H'23'	; cycles burst
THERMISTOR		equ	H'24'	; Thermistor reading
MIN_COUNT_H		equ	H'25'	; minimum calibration counter value ms byte
MIN_COUNT_L		equ	H'26'	; minimum calibration counter value ls byte
MAX_COUNT_H		equ	H'27'	; maximum calibration counter value ms byte
MAX_COUNT_L		equ	H'28'	; maximum calibration counter value ls byte
LED_DISPLAY		equ	H'29'	; LED display value
CALIBRATION		equ	H'2A'	; calibration flag max or min
RANGE_M			equ	H'2B'	; current range reading ms byte
RANGE_L			equ	H'2C'	; current range reading ls byte
COUNTER			equ	H'2D'	; 0.26s overflow counter TMR1
RATE			equ	H'2E'	; update rate counter
DOT_BAR			equ	H'2F'	; dot or bar display
LAST_VALUE		equ	H'30'	; last value for display for comparison with new display value
COUNT_H			equ	H'31'	; minimum counter value ms byte
COUNT_L			equ	H'32'	; minimum counter value ls byte

; math routines
TEMP1			equ	H'5C'
TEMPB0			equ	H'5D'
TEMPB1			equ	H'5E'
TEMPB2			equ	H'5F'
TEMP			equ H'60'
REMB3			equ H'61'
REMB2			equ	H'62'
REMB1      		equ H'63'
REMB0			equ	H'64'
AARGB5			equ	H'65'
AARGB4      	equ H'66'
AARGB3			equ	H'67'
AARGB2      	equ H'68'
AARGB1      	equ H'69'
AARGB0      	equ H'6A'	; most significant byte of argument A
BARGB3      	equ H'6B'
BARGB2      	equ H'6C'
BARGB1      	equ H'6D'
BARGB0      	equ H'6E'	; most significant byte of argument B
LOOPCOUNT   	equ H'6F'  	; division counter
	

; **********************************

; start at memory 0
	org	0
	goto	SETUP

; lookup table for speed of sound in air with respect to temperature
SOUND_SPEED
; covers from 70 degrees C to 0 degrees C. A/D NTC reading from D37 to D197
; lookup table assumes 37 has been subtracted so starts at line 0 for D37
; 'w' return value is the lsb speed of sound of the Hex value. Eg D331m/s is represented as
; (D331-D256) or D75. D255 (H100) must be added to this for the complete speed value
	
	addwf	PCL,f	; add value to program counter
	retlw	D'115'	; 0, 70 degrees C NTC value after D37 is subtracted (max temperature reading)
	retlw	D'114'
	retlw	D'113'
	retlw	D'113'
	retlw	D'112'
	retlw	D'112'	; 5, 65 degrees C. (NTC value is 42, minus 37=5)
	retlw	D'112'
	retlw	D'111'
	retlw	D'111'
	retlw	D'111'
	retlw	D'111'
	retlw	D'110'
	retlw	D'110'
	retlw	D'110'
	retlw	D'110'	; 14, 60 degrees C. (NTC value is 51, minus 37 is 14)
	retlw	D'109'
	retlw	D'109'
	retlw	D'109'
	retlw	D'108'
	retlw	D'108'
	retlw	D'108'
	retlw	D'107'	; 21. 55 degrees C. (NTC value is 58, minus 37 is 21)
	retlw	D'107'
	retlw	D'107'
	retlw	D'106'
	retlw	D'106'
	retlw	D'106'
	retlw	D'105'
	retlw	D'105'
	retlw	D'105'
	retlw	D'104'	; 30. 50 degrees C. (NTC value is 67)
	retlw	D'104'
	retlw	D'104'
	retlw	D'103'
	retlw	D'103'
	retlw	D'103'
	retlw	D'102'
	retlw	D'102'
	retlw	D'102'
	retlw	D'101'	; 39. 45 degrees C (NTC value is 76)
	retlw	D'101'
	retlw	D'101'
	retlw	D'101'
	retlw	D'100'
	retlw	D'100'
	retlw	D'100'
	retlw	D'100'
	retlw	D'99'
	retlw	D'99'
	retlw	D'99'
	retlw	D'99'
	retlw	D'98'	; 51. 40 degrees C (NTC value 88)
	retlw	D'98'
	retlw	D'98'
	retlw	D'98'
	retlw	D'97'
	retlw	D'97'
	retlw	D'97'
	retlw	D'97'
	retlw	D'96'
	retlw	D'96'
	retlw	D'96'
	retlw	D'96'
	retlw	D'95'	; 63. 35 degrees C (NTC value 100)
	retlw	D'95'
	retlw	D'95'
	retlw	D'95'
	retlw	D'95'
	retlw	D'95'
	retlw	D'95'
	retlw	D'94'
	retlw	D'94'
	retlw	D'94'
	retlw	D'94'
	retlw	D'94'
	retlw	D'94'
	retlw	D'93'	; 76. 30 degrees C (NTC value 113)
	retlw	D'93'
	retlw	D'93'
	retlw	D'93'
	retlw	D'93'
	retlw	D'92'
	retlw	D'92'
	retlw	D'92'
	retlw	D'92'
	retlw	D'92'
	retlw	D'91'
	retlw	D'91'
	retlw	D'91'
	retlw	D'91'
	retlw	D'90'	; 90. 25 degrees C (NTC value 127)
	retlw	D'90'
	retlw	D'90'
	retlw	D'90'
	retlw	D'90'
	retlw	D'89'
	retlw	D'89'
	retlw	D'89'
	retlw	D'89'
	retlw	D'89'
	retlw	D'88'
	retlw	D'88'
	retlw	D'88'
	retlw	D'88'
	retlw	D'87'	; 104. 20 degrees C (NTC value 141)
	retlw	D'87'
	retlw	D'87'
	retlw	D'87'
	retlw	D'87'
	retlw	D'87'
	retlw	D'86'
	retlw	D'86'
	retlw	D'86'
	retlw	D'86'
	retlw	D'86'
	retlw	D'85'
	retlw	D'85'
	retlw	D'85'
	retlw	D'85'
	retlw	D'85'
	retlw	D'84'	; 120. 15 degrees C (NTC value 157)
	retlw	D'84'
	retlw	D'84'
	retlw	D'83'
	retlw	D'83'
	retlw	D'83'
	retlw	D'83'
	retlw	D'82'
	retlw	D'82'
	retlw	D'82'
	retlw	D'81'
	retlw	D'81'
	retlw	D'81'
	retlw	D'81'	; 133. 10 degrees C (NTC value 170)
	retlw	D'81'
	retlw	D'81'
	retlw	D'81'
	retlw	D'81'
	retlw	D'80'
	retlw	D'80'
	retlw	D'80'
	retlw	D'80'
	retlw	D'79'
	retlw	D'79'
	retlw	D'79'
	retlw	D'79'
	retlw	D'79'
	retlw	D'78'	; 147. 5 degrees C (NTC value 184)
	retlw	D'78'
	retlw	D'78'
	retlw	D'78'
	retlw	D'78'
	retlw	D'77'	
	retlw	D'77'
	retlw	D'77'
	retlw	D'77'
	retlw	D'76'
	retlw	D'76'
	retlw	D'76'
	retlw	D'75'
	retlw	D'75'	; 160. 0 degrees C (NTC value 197)
	retlw	D'75'
	retlw	D'75'
	retlw	D'75'
	retlw	D'75'	; 164
	retlw	D'75'; 

;**********************************************************************************************

SETUP

	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off

; set inputs/outputs
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'01001010'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'00100000'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000111'	; settings (pullups enabled, TMR0/256)
	movwf	OPTION_REG

	movlw	B'01111000'	; 8MHz operation 8MHz
	movwf	OSCCON	
	
; analog inputs, A/D

	movlw	B'00100000'	; AN5 is an analog input
	movwf	ANSEL

	movlw	B'01000000'	; 8MHz operation left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'01101000'	; Fosc, channel 5
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on

	movlw	B'00110000'	; 8MHz operation timer 1 prescaler /8, fosc/4
	movwf	T1CON
	bsf		T1CON,0		; timer 1 on with 4us per count, 1.44mm resolution
	movlw	B'00001010'	; compare mode software flag set
	movwf	CCP1CON

; initialise
	clrf	CALIBRATION	; calibration flag max or min
	movlw	H'FF'
	movwf	RATE		; update rate, fast. Then slow when RATE=0
	clrf	RANGE_M		; current range reading ms byte
	clrf	RANGE_L		; current range reading ls byte
	movlw	D'4'
	clrf	LAST_VALUE	; last value for display for comparison with new display value

; test if EEPROM programmed

	movlw	EEPROMTST	; test EEPROM for H55
	call	EEREAD
	xorlw	H'55'		; compare with 55	
	btfss	STATUS,Z	; if 55 then EEPROM locations are programmed 
	call	DEFAULT		; load default values. EEPROM not programmed	
	
; get calibration values stored in EEPROM
; min range
	movlw	EEPROM0
	call	EEREAD		; sets EEADR
	movwf	MIN_COUNT_H	; minimum calibration counter value ms byte
	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movwf	MIN_COUNT_L	; minimum calibration counter value ls byte

; max range 
	movlw	EEPROM2
	call	EEREAD		; sets EEADR
	movwf	MAX_COUNT_H	; maximum calibration counter value ms byte
	movlw	EEPROM3
	call	EEREAD		; sets EEADR
	movwf	MAX_COUNT_L	; maximum calibration counter value ls byte

; dot/bar 
	movlw	EEPROM4
	call	EEREAD		; sets EEADR
	movwf	DOT_BAR

; min counter
	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movwf	COUNT_H		; minimum counter value ms byte
	movlw	EEPROM6
	call	EEREAD		; sets EEADR
	movwf	COUNT_L		; minimum counter value ls byte

; check S1 closed to alter DOT_BAR

	btfsc	PORTB,1		; when closed change DOT_BAR
	goto	PRELIMINARY
	movlw	H'FF'		; 100ms	
	call	DELAY3		; debounce
	btfsc	PORTB,1		; when closed change DOT_BAR
	goto	RUN_40kHz
	incf	DOT_BAR,F	; change display
	movlw	EEPROM4
	call	EEREAD		; sets EEADR
	movf	DOT_BAR,w
	call	EEWRITE		; write value
; show display
	btfss	DOT_BAR,0	; clear = dot
	goto 	DOT_SHOW
; LEDs 6-10
BAR_SHOW ; show bargraph	
	movlw	B'11000000'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB 
	goto	LOOP_SW

DOT_SHOW; show dot display	
	bcf		PORTA,7		; LED6 on	

; wait for switch to open
LOOP_SW
	btfss	PORTB,1		; when open 
	goto	LOOP_SW
	movlw	H'FF'		; 100ms	
	call	DELAY3		; debounce
	btfss	PORTB,1		; when open 
	goto	LOOP_SW
; LEDs off
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off
	goto 	RUN_40kHz
PRELIMINARY ; startup delay to have receive circuit settle
	call	DELAYX	; 500ms

; ***************************************************
; 40kHz drive burst signal for ultrasonic sender
RUN_40kHz
; clear and start timer1 counting
	bcf		T1CON,0		; timer 1 off
	clrf	TMR1L
	clrf	TMR1H		; clear timer 1
	bcf		PIR1,TMR1IF	; clear overflow flag
	bsf		T1CON,0		; timer 1 on
; 40kHz start
	call	BURST_40_kHz
	goto 	END_BURST

; subroutine to drive ultrasonic transducer
BURST_40_kHz	
	movlw	D'15'
	movwf	CYCLE		; cycle burst counter

CONT_40kHz
	
	bsf		PORTB,0		; RB0 high, Q2, Q3 on
	nop
	nop
	call	DELAY2
	bcf		PORTB,0		; output low, transistors off
	nop	
	nop	
	nop
	bsf		PORTB,2		; RB2 high Q1, Q4 on
	nop
	nop
	call	DELAY2
	bcf		PORTB,2		; transistors off
	nop	
	decfsz	CYCLE,F
	goto	CONT_40kHz
	return

; **********************************
END_BURST

; compare timer counter with COUNT_H, COUNT_L (minimum counter values)
	
	movf	COUNT_H,w		; ms min count
	movwf	CCPR1H
	movf	COUNT_L,w		; ls min count
	movwf	CCPR1L			; compare value
	bcf		PIR1,CCP1IF		; compare flag
BLANKING
	btfsc	PIR1,TMR1IF		; if set then timed out
	goto	ERROR_READ
	btfss	PIR1,CCP1IF		; when set continue
	goto	BLANKING		; wait till timer1 count is equal or > min count

; * test
;	bsf	PORTB,2			; oscilloscope tracer to set/check minimum count
;	bcf	PORTB,2
; * test

; monitor RB3 input for a low (detection of return signal)

INPUT_MON; monitor input
	
	btfsc	PIR1,TMR1IF	; if set then timed out
	goto	ERROR_READ
	btfsc	PORTB,3		; when low, indicates signal detection
	goto 	INPUT_MON

; signal detected
	call	DELAY2
	btfsc	PORTB,3
	goto	INPUT_MON	; small delay removes glitch error

TEMPERATURE
; read NTC for temperature compensation
	call	DEL_AD			; convert NTC Thermistor reading

	movlw	D'37'
	subwf	THERMISTOR,f	; NTC Thermistor value. Take away 37
	btfss	STATUS,C		; when C=0 negative so use 0
	clrf	THERMISTOR		; zero minimum
; compare with 160
	movlw	D'160'
	subwf	THERMISTOR,w
	movlw	D'160'			; clamp value at 160 if over
	btfsc	STATUS,C		; if C=0 value is 160 or less
	movwf	THERMISTOR		; store 160 in THERMISTOR register if over 160

	movf	THERMISTOR,w	; get Thermistor value (160) or original value if <160
	call	SOUND_SPEED		; look up speed of sound value (ls byte)with respect to temperature
							; (requires the addition of 255).

; ********************************
; Calculations (temperature compensation)

; multiply timer1 value by this temperature compensated speed value and divide by 331 (zero deg C value)
	movwf	AARGB2			; 'w' has lookup value from SOUND_SPEED (ls byte)
	clrf	AARGB1
	bsf		AARGB1,0		; add the 255 value (see SOUND_SPEED lookup) 
	clrf	AARGB0			; ms byte
  	movf	TMR1H,w			; timer ms byte
	movwf	BARGB1
	movf	TMR1L,w			; timer ls byte
	movwf	BARGB2			
	clrf	BARGB0			; multiplier ms byte cleared
	call	FXM2424U		; multiply. Result in AARGB0 to AARGB5
; shift values right
	movf	AARGB2,w
	movwf	AARGB0
	movf	AARGB3,w
	movwf	AARGB1
	movf	AARGB4,w
	movwf	AARGB2
	movf	AARGB5,w
	movwf	AARGB3

; divisor at D331 (H14B)
	clrf	BARGB0
	clrf	BARGB1
	movlw	H'01'
	movwf	BARGB2
	movlw	H'4B'
	movwf	BARGB3

	call	FXD3232U		; divide

; result in AARGB0,1,2,3 but since it is 16 bit only in AARGB2,3. 

; If AARGB1 is not zero: error.
	movf	AARGB1,w		; check for over
	btfss	STATUS,Z
	goto	ERROR_READ		; overrange counter value
; If AARGB0 is not zero: error
	movf	AARGB0,w		; check for over
	btfss	STATUS,Z
	goto	ERROR_READ

	movf	AARGB2,w		; transfer values to RANGE
	movwf	RANGE_M			;
	movf	AARGB3,w		;
	movwf	RANGE_L			;

; subtract 10 to ensure last LED lights 

	movlw	D'10'
	subwf	RANGE_L,f
	btfss	STATUS,C	; if carry,decrease ms byte
	decf	RANGE_M,f

; **************************************************

CALC
; convert range value to a value between minimum and maximum counter values 	
; (MIN_COUNT_H, MIN_COUNT_L, MAX_COUNT_H, MAX_COUNT_L)
; For Dot
; Calculation is counter value for 0 deg C and normalised over the min and max range x 18
; 18 x (RANGE_M and RANGE_L - MIN_COUNT_H, MIN_COUNT_L)/(MAX_COUNT_H, MAX_COUNT_L - MIN_COUNT_H, MIN_COUNT_L) 
; gives a value 0-18 for the 10 LED display. A 0-1 is for full (lowest count so full) and 18 for empty 	
; For Bar value is divided by 2 for 0-9 range

; Subtract first. RANGE_M, RANGE_L - MIN_COUNT_H, MIN_COUNT_L
; if zero or negative set LED_DISPLAY at 0
	movf	MIN_COUNT_H,w	; ms result
	subwf	RANGE_M,w
	movwf	BARGB1			; store
	movf	MIN_COUNT_L,w	; ls result
	subwf	RANGE_L,w
	movwf	BARGB2			; store
	btfss	STATUS,C
	decf	BARGB1,F		; decrease if carry
	btfsc	BARGB1,7		; if bit 7 set then RANGE_M, RANGE_L < MIN_COUNT_H, MIN_COUNT_L  
	goto	LED1_DRIVE		; drive top LED 
; check for zero from subtraction
	movf	BARGB1,w
	btfss	STATUS,Z		; if zero check ls byte
	goto	MULT18
	movf	BARGB2,w
	btfsc	STATUS,Z
	goto	LED1_DRIVE		; drive LED1 if RANGE_M, RANGE_L - MIN_COUNT_H, MIN_COUNT_L = 0

MULT18

; Multiply by 18
; multiply BARGB1, BARGB2 subtraction value x 18

	movlw	D'18'
	movwf	AARGB2			; ls byte
	clrf	AARGB1
	clrf	AARGB0			; ms byte
  	clrf	BARGB0			; ms byte cleared
	call	FXM2424U		; multiply. Result in AARGB0 to AARGB5
; shift values right
	movf	AARGB2,w
	movwf	AARGB0
	movf	AARGB3,w
	movwf	AARGB1
	movf	AARGB4,w
	movwf	AARGB2
	movf	AARGB5,w
	movwf	AARGB3

; subtract MAX_COUNT_H, MAX_COUNT_L - MIN_COUNT_H, MIN_COUNT_L
; if zero or negative, error. (CAL_ERR)

	movf	MIN_COUNT_H,w	; ms result
	subwf	MAX_COUNT_H,w
	movwf	BARGB2			; store
	movf	MIN_COUNT_L,w	; ls result
	subwf	MAX_COUNT_L,w
	movwf	BARGB3			; store
	btfss	STATUS,C
	decf	BARGB2,F		; decrease if carry
	btfsc	BARGB2,7		; if bit 7 set then MAX_COUNT_H, MAX_COUNT_L < MIN_COUNT_H, MIN_COUNT_L 
	goto	CAL_ERR			; calibration error
 
; check for zero from subtraction
	movf	BARGB2,w
	btfss	STATUS,Z		; if zero check ls byte
	goto	DIV
	movf	BARGB3,w
	btfsc	STATUS,Z
	goto	CAL_ERR			; calibration error if subtraction = 0

DIV
; divisor 
	clrf	BARGB0			; clear ms bytes
	clrf	BARGB1
	call	FXD3232U		; divide

; result in AARGB0,1,2,3 but since result should be 0-18 result is only in AARGB3.
	
	movf	AARGB3,w
	movwf	LED_DISPLAY		; LED display value

; if result is >18 set LED_DISPLAY at 18 

	movf	LED_DISPLAY,w
	sublw	D'18'			; if LED_DISPLAY > 18 set at 18
	btfsc	STATUS,C
	goto	DISPLAY_COMPARE	; 18 or less so no change to value required
	movlw	D'18'			; over 18 so set at 18 max
	movwf	LED_DISPLAY

DISPLAY_COMPARE ; two sequential values must be the same otherwise slow rate is reset to fast rate
	movf	LED_DISPLAY,w
	xorwf	LAST_VALUE,w	; compare with last value
	btfsc	STATUS,Z		
	goto	DISPLAY			; if last value is the same as new value (LED_DISPLAY) then display 

; values are not the same so transfer new value (LED_DISPLAY) to last value

	movf	LED_DISPLAY,w
	movwf	LAST_VALUE
; reset to fast rate
	movlw	H'FF'
	movwf	RATE			; reset to fast display rate
	goto	DISPLAY

; *****************************

LED1_DRIVE
	clrf	LED_DISPLAY		; LED_DISPLAY=0. LED1 lit (Dot). All LEDs lit (Bar) 
	
DISPLAY; select display mode
	btfsc	DOT_BAR,0
	goto	BAR_DISPLAY

DOT_DISPLAY
; display
; all LEDs off
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off

; light LED(s)according to 0-18 in LED_DISPLAY. LED10 top for 0, LED1 bottom for 18
; compare LED_DISPLAY with 0-18

; 1
	movlw	D'18'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED1
; 2
	movlw	D'17'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED1_2
; 3
	movlw	D'16'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED2
; 4
	movlw	D'15'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED2_3
; 5
	movlw	D'14'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED3
; 6
	movlw	D'13'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED3_4
; 7
	movlw	D'12'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED4
; 8
	movlw	D'11'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED4_5
; 9
	movlw	D'10'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED5
; 10
	movlw	D'9'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED5_6
; 11
	movlw	D'8'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED6
; 12
	movlw	D'7'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED6_7
; 13
	movlw	D'6'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED7
; 14
	movlw	D'5'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED7_8
; 15
	movlw	D'4'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED8
; 16
	movlw	D'3'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED8_9
; 17
	movlw	D'2'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED9
; 18
	movlw	D'1'
	xorwf	LED_DISPLAY,w
	btfsc	STATUS,Z
	goto	LED9_10
; 19
	goto	LED10

; Light LEDs required for DOT display
LED1	
	movlw	B'11101111'
	movwf	PORTA		; LED 1
	goto	CHECK_S1
LED1_2
	movlw	B'11100111'
	movwf	PORTA		; LED 1 & 2 on
	goto	CHECK_S1
	
LED2
	movlw	B'11110111'
	movwf	PORTA		; LED2 on
	goto	CHECK_S1
LED2_3
	movlw	B'11110011'
	movwf	PORTA		; LED2 & 3 on
	goto	CHECK_S1	; 

LED3
	movlw	B'11111011'	; LED3 on
	movwf	PORTA
	goto	CHECK_S1
LED3_4
	movlw	B'11111001'	; LED3 & 4 on
	movwf	PORTA
	goto	CHECK_S1

LED4
	movlw	B'11111101'	; LED4 on
	movwf	PORTA		; 
	goto	CHECK_S1
LED4_5
	movlw	B'11111100'	; LED4 & 5 on
	movwf	PORTA		; 
	goto	CHECK_S1	; 

LED5
	movlw	B'11111110'	; LED5 on
	movwf	PORTA		; 
	goto	CHECK_S1
LED5_6
	movlw	B'01111110'	; LED5 & 6 on
	movwf	PORTA		; 
	goto	CHECK_S1

LED6
	movlw	B'01111111'	; LED6 on
	movwf	PORTA		; 
	goto	CHECK_S1
LED6_7
	movlw	B'00111111'	; LED 6 & 7 on
	movwf	PORTA		; 
	goto	CHECK_S1

LED7
	movlw	B'10111111'	; LED7 on
	movwf	PORTA		; 
	goto	CHECK_S1
LED7_8
	movlw	B'10111111'	; LED7 on
	movwf	PORTA
LED8
	movlw	B'01111010'	; RB0, RB2 low Q1-Q6 off, LED 8 on
	movwf	PORTB
	goto	CHECK_S1

LED8_9
	movlw	B'01011010'	; RB0, RB2 low Q1-Q6 off, LED 8 & 9 on
	movwf	PORTB
	goto	CHECK_S1

LED9
	movlw	B'11011010'	; RB0, RB2 low Q1-Q6 off, LED 9 on
	movwf	PORTB
	goto	CHECK_S1
LED9_10
	movlw	B'11001010'	; RB0, RB2 low Q1-Q6 off, LED 9 & 10 on
	movwf	PORTB
	goto	CHECK_S1

LED10
	movlw	B'11101010'	; RB0, RB2 low Q1-Q6 off, LED 10 on
	movwf	PORTB
	goto	CHECK_S1

; *************************************************
BAR_DISPLAY
; display
; all LEDs off
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off
	bcf		STATUS,C	; carry cleared
	rrf		LED_DISPLAY,F	; divide by 2 for bargraph

; light LED(s)according to 0-9 in LED_DISPLAY. LED1 bottom for 0, LED10 top for 9
; compare LED_DISPLAY with 0-9

; 1
CHECK_1
	movlw	D'09'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_2
; LED1
	movlw	B'11001111'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB 
	goto	CHECK_S1	; switch closure check
; 2
CHECK_2
	movlw	D'08'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_3
; LEDs 1-2	
	movlw	B'11000111'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB 
	goto	CHECK_S1
; 3
CHECK_3
	movlw	D'07'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_4
; LEDs 1-3	
	movlw	B'11000011'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB
	goto	CHECK_S1
; 4
CHECK_4
	movlw	D'06'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_5
; LEDs 1-4	
	movlw	B'11000001'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB
	goto	CHECK_S1
; 5
CHECK_5
	movlw	D'05'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_6
; LEDs 1-5	
	movlw	B'11000000'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB
	goto	CHECK_S1
; 6
CHECK_6
	movlw	D'04'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_7
; LEDs 1-6	
	movlw	B'01000000'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB
	goto	CHECK_S1
; 7
CHECK_7
	movlw	D'03'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_8
; LEDs 1-7	
	movlw	B'00000000'	; LEDs on
	movwf 	PORTA
	movlw 	B'10110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB
	goto	CHECK_S1
; 8
CHECK_8
	movlw	D'2'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	CHECK_9
; LEDs 1-8	
	movlw	B'00000000'	; LEDs on
	movwf 	PORTA
	movlw 	B'00110000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB
	goto	CHECK_S1
; 9
CHECK_9
	movlw	D'1'
	xorwf	LED_DISPLAY,w
	btfss	STATUS,Z
	goto	ALL_LED		; 
; LEDs 1-9	
	movlw	B'00000000'	; LEDs on
	movwf 	PORTA
	movlw 	B'00010000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB
	goto	CHECK_S1

ALL_LED
	movlw	B'00000000'	; LEDs on
	movwf 	PORTA
	movlw 	B'00000000'	; RB0, RB2 low Q1-Q6 off, LEDs on
	movwf	PORTB 

; ****************************************

; check for S1 cal pressed
; press to change from Min to Max selection with flashing min/max LED
; press and hold to store last range value after correction to 0 degrees C value 
CHECK_S1
	btfsc	PORTB,1		; if low switch is pressed
	goto	BURST_DELAY

; check if calibration or default restore
; read NTC and if shorted restore defaults
	call	DEL_AD		; convert NTC Thermistor reading

	movlw	D'20'
	subwf	THERMISTOR,f	; NTC Thermistor value. Take away 20
	btfsc	STATUS,C		; when C=0 negative restore defaults
	goto	CALIBRATE_SET

; restore default values

; indicate default action. LEDs 1 and 10 lit
	movlw	B'11101010'	; RB0, RB2 low Q1-Q6 off, LEDs off except LED10
	movwf	PORTB
	movlw	B'11101111'
	movwf	PORTA		; LEDs off except LED1
	
	call	DEFAULT		; load default values
	call	DELAY		; 500ms
	goto	BURST_DELAY


; *****************************************
; Calibration
CALIBRATE_SET
	clrf	TMR1H		; clear timer
	clrf	TMR1L
	bcf		PIR1,TMR1IF	; clear overflow flag
	bsf		T1CON,0		; timer 1 on to provide LED flashing rate at 0.26s with overflow flag

	movlw	H'FF'
	movwf	RATE		; reset update rate to fast.

; all LEDs off
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off
	movlw	D'16'		; 4-seconds
	movwf	COUNTER		; counter decreased with TMR1 overflow each 0.26s
CAL_CYC
	btfsc	CALIBRATION,0		; calibration flag either 0 = minimum value, 1 = maximum value
	goto	MAX_CAL
; flash LED min or max
MIN_CAL
	btfss	PORTA,4		; minimum LED (LED1) flash
	goto	SET_MIN
	bcf		PORTA,4		; Min LED on
	goto	CHK_OVERFLOW
SET_MIN
	bsf		PORTA,4		; LED off
	goto	CHK_OVERFLOW
MAX_CAL
	btfss	PORTB,4		; maximum LED (LED10) flash
	goto	SET_MAX
	bcf		PORTB,4		; Max LED on
	goto	CHK_OVERFLOW
SET_MAX
	bsf		PORTB,4		; LED off

CHK_OVERFLOW	
	btfss	PIR1,TMR1IF	; when set change LED to on or off
	goto	CHK_OVERFLOW; wait for overflow at 0.26s 

; after 0.26s
	bcf		PIR1,TMR1IF	; cleared flag when set
	btfsc	PORTB,1		; if high switch is released
	goto	RELEASED	; switch released before timeout for calibration
	decfsz	COUNTER,F	; decrease counter
	goto	CAL_CYC		; continue cycle

	btfsc	CALIBRATION,0
	goto	SET_LED	
	bcf		PORTA,4		; MIN LED1 on
	goto	CAL_CLEAR
SET_LED
	bcf		PORTB,4		; Max LED on
CAL_CLEAR	
; counter zero
; clear and start timer1 counting
	bcf		T1CON,0		; timer 1 off
	clrf	TMR1L
	clrf	TMR1H		; clear timer 1
	bcf		PIR1,TMR1IF	; clear overflow flag
	bsf		T1CON,0		; timer 1 on
	
; send burst
	call	BURST_40_kHz
	
; monitor RB3 input for a low (detection of return signal)
	btfss	CALIBRATION,0	; check calibration flag 
	goto	DISTANCE		; distance measure

; close distance measure
	call	DELAY4		; wait short period 

INPUT_MON3; monitor input
	
	btfsc	PIR1,TMR1IF	; if set then timed out
	goto	ERROR_READ
	btfsc	PORTB,3		; when low, indicates signal detection
	goto 	INPUT_MON3
	goto	RECEIVED2
DISTANCE
	movlw	H'01'		; ms min count
	movwf	CCPR1H
	movlw	H'F4'		; ls min count
	movwf	CCPR1L			; compare value
	bcf		PIR1,CCP1IF		; compare flag
INPUT_MON2
	btfsc	PIR1,TMR1IF		; if set then timed out
	goto	ERROR_READ
	btfss	PIR1,CCP1IF		; when set continue
	goto	INPUT_MON2		; wait till timer1 count is equal or > min count
	btfsc	PORTB,3			; when low, indicates signal detection
	goto 	INPUT_MON2
	
RECEIVED2 ; signal detected
	
; get counter 1 value
	bcf		T1CON,0		; stop timer 1

; read NTC for temperature compensation
	call	DEL_AD		; convert NTC Thermistor reading

	movlw	D'37'
	subwf	THERMISTOR,f	; NTC Thermistor value. Take away 37
	btfss	STATUS,C		; when C=0 negative so use 0
	clrf	THERMISTOR		; zero minimum
; compare with 160
	movlw	D'160'
	subwf	THERMISTOR,w
	movlw	D'160'			; clamp value at 160 if over
	btfsc	STATUS,C		; if C=0 value is 160 or less
	movwf	THERMISTOR		; store 160 in THERMISTOR register if over 160

	movf	THERMISTOR,w	; get Thermistor value (160) or original value if <160
	call	SOUND_SPEED		; look up speed of sound value (ls byte)with respect to temperature
							; (requires the addition of 255).
; Calculations

; multiply timer1 value by this temperature compensated speed value and divide by 331 (zero deg C value)
	movwf	AARGB2			; 'w' has lookup value from SOUND_SPEED (ls byte)
	clrf	AARGB1
	bsf		AARGB1,0		; add the 255 value (see SOUND_SPEED lookup) 
	clrf	AARGB0			; ms byte
  	movf	TMR1H,w			; timer ms byte
	movwf	BARGB1
	movf	TMR1L,w			; timer ls byte
	movwf	BARGB2			
	clrf	BARGB0			; multiplier ms byte cleared
	call	FXM2424U		; multiply. Result in AARGB0 to AARGB5
; shift values right
	movf	AARGB2,w
	movwf	AARGB0
	movf	AARGB3,w
	movwf	AARGB1
	movf	AARGB4,w
	movwf	AARGB2
	movf	AARGB5,w
	movwf	AARGB3

; divisor at D331 H14B
	clrf	BARGB0
	clrf	BARGB1
	movlw	H'01'
	movwf	BARGB2
	movlw	H'4B'
	movwf	BARGB3

	call	FXD3232U		; divide
; result in AARGB0,1,2,3 but since it is 16 bit only in AARGB2,3. 

; If AARGB1 is not zero: error.
	movf	AARGB1,w		; check for over
	btfss	STATUS,Z
	goto	ERROR_READ		; overrange counter value
; If AARGB1 is not zero: error
	movf	AARGB0,w		; check for over
	btfss	STATUS,Z
	goto	ERROR_READ

; store values
	movf	AARGB2,w		; transfer values to RANGE
	movwf	RANGE_M			
	movf	AARGB3,w		
	movwf	RANGE_L			

	btfss	CALIBRATION,0	; check calibration flag 
	goto	STORE

; store min counter value when min values calibration is set 

; subtract 10 from value so reading can work down to that minimum and show final LED
	movf	RANGE_M,w
	movwf	TEMPB0
	movlw	D'10'
	subwf	RANGE_L,w
	movwf	TEMPB1
	btfss	STATUS,C	; if carry,decrease ms byte
	decf	TEMPB0,f

	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movf	TEMPB0,w	; range ms byte
	movwf	COUNT_H
	call	EEWRITE

	movlw	EEPROM6
	call	EEREAD		; sets EEADR
	movf	TEMPB1,w	; range ls byte
	movwf	COUNT_L
	call	EEWRITE	

STORE
; store new min or max calibration values
	btfss	CALIBRATION,0	; check calibration flag 
	goto	MAX_COUNT
MIN_COUNT
	movlw	EEPROM0
	call	EEREAD		; sets EEADR
	movf	RANGE_M,w	; current range reading ms byte
	movwf	MIN_COUNT_H	; minimum calibration counter value ms byte
	call	EEWRITE

	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movf	RANGE_L,w	; current range reading ls byte
	movwf	MIN_COUNT_L	; minimum calibration counter value ls byte
	call	EEWRITE

	goto	CAL_END	

MAX_COUNT

	movlw	EEPROM2
	call	EEREAD		; sets EEADR
	movf	RANGE_M,w	; current range reading ms byte
	movwf	MAX_COUNT_H	; maximum calibration counter value ms byte
	call	EEWRITE

	movlw	EEPROM3
	call	EEREAD		; sets EEADR
	movf	RANGE_L,w	; current range reading ls byte
	movwf	MAX_COUNT_L	; maximum calibration counter value ls byte
	call	EEWRITE

CAL_END
RELEASED	
; if S1 released before counter ends toggle CALIBRATION bit 0 so it toggles from min to max or max to min
	incf	CALIBRATION,F	; next calibration (min or max)
	movlw	D'20'			; delay to indicate a break
	call	DELAYX1
	bsf		PORTA,4
	bsf		PORTB,4			; LEDs off
	movlw	D'20'			; delay to indicate a break
	call	DELAYX1
	goto	RUN_40kHz
; ********************************************
BURST_DELAY
; add delay between bursts
	movf	RATE,w		; update rate
	btfsc	STATUS,Z	; when 0 stop decrementing
	goto	SLOW_RATE
	decf	RATE,f		; decreased from FF (reset condition) (remark out for no slow rate)
	goto	FAST_RATE	
SLOW_RATE
	movlw	H'FF'		; 
	call	DELAY1		; run delay twice
FAST_RATE
	call	DELAYX		; 500ms 
	goto	RUN_40kHz

; ********************************************
; error indications
ERROR_READ
; flash LED1 through to LED10 for no signal from Ultrasonic ranging
	clrf	TMR1H		; clear timer
	clrf	TMR1L
	bcf		PIR1,TMR1IF	; clear overflow flag
	bsf		T1CON,0		; timer 1 on to provide LED flashing rate at 0.26s with overflow flag

	movlw	H'FF'
	movwf	RATE		; reset to fast update rate

; all LEDs off
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off
	movlw	D'10'		; one cycle of LEDs
	movwf	COUNTER		; counter overflow at 0.26s

; flash LED min to max
FLASH_ERR
; LEDs off to start each flash
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off
; cycle LEDs with counter
	movf	COUNTER,w
	xorlw	D'10'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTA,4		; LED1
	
	movf	COUNTER,w
	xorlw	D'9'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTA,3		; LED2

	movf	COUNTER,w
	xorlw	D'8'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTA,2		; LED3

	movf	COUNTER,w
	xorlw	D'7'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTA,1		; LED4

	movf	COUNTER,w
	xorlw	D'6'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTA,0		; LED5

	movf	COUNTER,w
	xorlw	D'5'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTA,7		; LED6

	movf	COUNTER,w
	xorlw	D'4'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTA,6		; LED7

	movf	COUNTER,w
	xorlw	D'3'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTB,7		; LED8

	movf	COUNTER,w
	xorlw	D'2'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTB,5		; LED9

	movf	COUNTER,w
	xorlw	D'1'		; if = LED driven
	btfsc	STATUS,Z
	bcf		PORTB,4		; LED10

CHK_ERR_OVER	
	btfss	PIR1,TMR1IF	; when set change LED to on or off
	goto	CHK_ERR_OVER; wait for overflow at 0.26s 

; after 0.26s
	bcf		PIR1,TMR1IF	; cleared flag when set
	decfsz	COUNTER,F	; decrease counter
	goto	FLASH_ERR	; flash LEDs continue cycle
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off
	goto	CHECK_S1	; check switch S1 or burst delay

; ***********************************

CAL_ERR
; flash LED1 and LED10 for incorrect calibration (min > max)
	clrf	TMR1H		; clear timer
	clrf	TMR1L
	bcf		PIR1,TMR1IF	; clear overflow flag
	bsf		T1CON,0		; timer 1 on to provide LED flashing rate at 0.26s with overflow flag

	movlw	H'FF'
	movwf	RATE		; reset to fast update rate

; all LEDs off
	movlw	B'11111010'	; RB0, RB2 low Q1-Q6 off, LEDs off
	movwf	PORTB
	movlw	B'11111111'
	movwf	PORTA		; LEDs off
	movlw	D'16'		; 4-seconds
	movwf	COUNTER		; counter overflow at 0.26s

; flash LED min and max
FLASH_CAL_ERR
	btfss	PORTB,4		; minimum LED (LED10) flash
	goto	FLASH_CAL_ERR1
	bcf		PORTB,4		; LED10 on
	bsf		PORTA,4		; LED1 off
	goto	CHK_CAL_ERR_OVER
FLASH_CAL_ERR1
	bsf		PORTB,4		; LED10 off
	bcf		PORTA,4		; LED1 on

CHK_CAL_ERR_OVER	
	btfss	PIR1,TMR1IF	; when set change LED to on or off
	goto	CHK_CAL_ERR_OVER; wait for overflow at 0.26s 

; after 0.26s
	bcf		PIR1,TMR1IF	; cleared flag when set
	decfsz	COUNTER,F	; decrease counter
	goto	FLASH_CAL_ERR	; flash LEDs, continue cycle
	bsf		PORTB,4		; LED10 off
	bsf		PORTA,4		; LED1 off
	goto	CHECK_S1	; check switch S1 or BURST_DELAY

;***************************************
; Subroutines

DEFAULT ; set defaults

	movlw	EEPROMTST	; test EEPROM
	call	EEREAD
	movlw 	H'55'
	call	EEWRITE
	
; min range default
	movlw	EEPROM0
	call	EEREAD		; sets EEADR
	movlw	H'01'		; default min range reading ms byte
	movwf	MIN_COUNT_H	; minimum calibration counter value ms byte
	call	EEWRITE

	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movlw	H'F4'		; default min range reading ls byte
	movwf	MIN_COUNT_L	; minimum calibration counter value ls byte
	call	EEWRITE

; max range default
	movlw	EEPROM2
	call	EEREAD		; sets EEADR
	movlw	H'05'		; default max range reading ms byte
	movwf	MAX_COUNT_H	; maximum calibration counter value ms byte
	call	EEWRITE

	movlw	EEPROM3
	call	EEREAD		; sets EEADR
	movlw	H'E6'		; default min range reading ls byte
	movwf	MAX_COUNT_L	; maximum calibration counter value ls byte
	call	EEWRITE

; dot/bar default
	movlw	EEPROM4
	call	EEREAD		; sets EEADR
	clrw
	movwf	DOT_BAR
	call	EEWRITE		; write value

; min
	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movlw	H'01'		; default min range reading ms byte
	movwf	COUNT_H		; minimum counter value ms byte
	call	EEWRITE

	movlw	EEPROM6
	call	EEREAD		; sets EEADR
	movlw	H'20'		; default min range reading ls byte
	movwf	COUNT_L		; minimum counter value ls byte
	call	EEWRITE
	return

; delays

; delay 500ms
DELAY
    movlw	D'5'		; 
DELAY_1	
DELAY1
	movwf	DELCNT

DELAY_M
	btfss	PORTB,1		; if low, Cal. switch is pressed
	return
	movlw	D'255'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
LP_1
	movlw	D'255'		; set delay period value 2 
	movwf	VALUE_2		; VALUE_2 = w
LP_2
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1
	decfsz	DELCNT,f
	goto	DELAY_M
	return	

DELAY2
	movlw	D'4'		; 	
	movwf	VALUE_1		; VALUE_1 = w
LP_3
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_3
	return	

DELAY3
 	movwf	VALUE_1		; VALUE_1 = w
LP_5
	movlw	D'255'		; set delay period value 2 
	movwf	VALUE_2		; VALUE_2 = w
LP_4
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_4
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_5
	return

DELAY4
	movlw	D'45'		; 	
	movwf	VALUE_1		; VALUE_1 = w
LP_6
	nop
	nop
	nop
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_6
	return	

; DELAY for A/D acquisition
DEL_AD
	movlw	D'100'			; for 8MHz operation
	movwf	DELCNT
DEL1
	decfsz	DELCNT,f
	goto	DEL1

	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV1
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV1
	movf	ADRESH,w		; ms bits
	movwf	THERMISTOR
	return

; delay 500ms
DELAYX
    movlw	D'5'		; 
DELAYX_1	
DELAYX1
	movwf	DELCNT
DELAY_XM
	movlw	D'255'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
LPX_1
	movlw	D'255'		; set delay period value 2 
	movwf	VALUE_2		; VALUE_2 = w
LPX_2
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LPX_2
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LPX_1
	decfsz	DELCNT,f
	goto	DELAY_XM
	return	

; subroutine to read EEPROM memory 

EEREAD
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM

; EEPROM write 

EEWRITE
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf		STATUS,RP0	; bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	bsf		STATUS,RP0
	bsf		STATUS,RP1	; bank 3		
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITE1	

	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE1		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0
	bcf		STATUS,RP1	; bank 0 
	return				; value written
	
;*********************************
; 24x24 Bit Unsigned Fixed Point Multiply 24x24 -> 48
; Input: 24 bit unsigned fixed point multiplicand in AARGB0,1,2
; 24 bit unsigned fixed point multiplier in BARGB0,1,2
; Use: CALL FXM2424U
; Output: 48 bit unsigned fixed point product in AARGB0
; Result: AARG <-- AARG x BARG
; Max Timing: 9+501+2 = 512 clks
; Min Timing: 9+150 = 159 clks

FXM2424U
	CLRF 	AARGB3 ; clear partial product
	CLRF 	AARGB4
	CLRF 	AARGB5
	MOVF 	AARGB0,W
	MOVWF 	TEMPB0
	MOVF 	AARGB1,W
	MOVWF 	TEMPB1
	MOVF 	AARGB2,W
	MOVWF 	TEMPB2

	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
LOOPUM2424A
	RRF 	BARGB2,F
	BTFSC 	STATUS,C
	GOTO 	ALUM2424NAP
	DECFSZ 	LOOPCOUNT,F
	GOTO	LOOPUM2424A
	MOVWF 	LOOPCOUNT
LOOPUM2424B
	RRF 	BARGB1,F
	BTFSC 	STATUS,C
	GOTO	BLUM2424NAP
	DECFSZ	LOOPCOUNT,F
	GOTO	LOOPUM2424B
	MOVWF	LOOPCOUNT
LOOPUM2424C
	RRF		BARGB0,F
	BTFSC 	STATUS,C
	GOTO 	CLUM2424NAP
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPUM2424C
	CLRF 	AARGB0
	CLRF 	AARGB1
	CLRF 	AARGB2
	RETLW 	0x00
CLUM2424NAP
	BCF 	STATUS,C
	GOTO 	CLUM2424NA
BLUM2424NAP
	BCF 	STATUS,C
	GOTO 	BLUM2424NA
ALUM2424NAP
	BCF 	STATUS,C
	GOTO 	ALUM2424NA
ALOOPUM2424
	RRF 	BARGB2,F
	BTFSS 	STATUS,C
	GOTO 	ALUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF	AARGB0,F
ALUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	ALOOPUM2424
	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
BLOOPUM2424
	RRF 	BARGB1,F
	BTFSS 	STATUS,C
	GOTO 	BLUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF 	AARGB0,F
BLUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	RRF 	AARGB4,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	BLOOPUM2424
	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
CLOOPUM2424
	RRF 	BARGB0,F
	BTFSS 	STATUS,C
	GOTO 	CLUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF 	AARGB0,F
CLUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	RRF 	AARGB4,F
	RRF 	AARGB5,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	CLOOPUM2424
	return

;********************************
; 32/32 Bit Unsigned Fixed Point Divide 32/32 -> 32.32
; Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point divisor in BARGB0, BARGB1, BARGB2, BARGB3
; Use: CALL FXD3232U
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point remainder in REMB0, REMB1, REMB2, REMB3
; Result: AARG, REM <-- AARG / BARG
; Max Timing: 4+1025+2 = 1031 clks
; Max Timing: 4+981+2 = 987 clks
; PM: 4+359+1 = 364 DM: 13
FXD3232U
	CLRF 	REMB0
	CLRF	REMB1
	CLRF 	REMB2
	CLRF 	REMB3
	call	UDIV3232L
	return

UDIV3232L 
; Max Timing: 24+6*32+31+31+6*32+31+31+6*32+31+31+6*32+31+16 = 1025 clks
; Min Timing: 24+6*31+30+30+6*31+30+30+6*31+30+30+6*31+30+3 = 981 clks
; PM: 359 DM: 13
	CLRF 	TEMP
	RLF 	AARGB0,W
	RLF 	REMB3,F
	MOVF 	BARGB3,W
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	RLF 	AARGB0,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232A 
	RLF 	AARGB0,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22LA
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LA

UADD22LA 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LA 
	RLF		AARGB0,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232A
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22L8
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L8

UADD22L8 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L8 
	RLF 	AARGB1,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232B 
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB1,0
	GOTO 	UADD22LB
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LB

UADD22LB 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LB 
	RLF 	AARGB1,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232B
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS	AARGB1,0
	GOTO 	UADD22L16
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L16

UADD22L16 	
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF	TEMP,F

UOK22L16
	RLF		AARGB2,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232C 
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22LC
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LC

UADD22LC
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LC 
	RLF 	AARGB2,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232C
	RLF 	AARGB3,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22L24
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L24

UADD22L24 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L24 
	RLF 	AARGB3,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232D 
    RLF  	AARGB3,W
    RLF    	REMB3,F
    RLF    	REMB2,F
    RLF    	REMB1,F
    RLF    	REMB0,F
    RLF    	TEMP,F
    MOVF   	BARGB3,W
 	BTFSS  	AARGB3,0
   	GOTO   	UADD22LD

    SUBWF  	REMB3,F
  	MOVF   	BARGB2,W
    BTFSS   STATUS,C
    INCFSZ  BARGB2,W
    SUBWF   REMB2,F
    MOVF    BARGB1,W
    BTFSS   STATUS,C
    INCFSZ  BARGB1,W
    SUBWF   REMB1,F
    MOVF    BARGB0,W
    BTFSS   STATUS,C
    INCFSZ  BARGB0,W
    SUBWF   REMB0,F
    CLRW
    BTFSS   STATUS,C
    MOVLW   H'1'
    SUBWF   TEMP,F
    GOTO    UOK22LD

UADD22LD      
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F
    CLRW
    BTFSC   STATUS,C
    MOVLW   H'1'
    ADDWF   TEMP,F
        
UOK22LD
	RLF     AARGB3,F

   	DECFSZ  LOOPCOUNT, F
    GOTO    LOOPU3232D

    BTFSC   AARGB3,0
    GOTO    UOK22L
    MOVF    BARGB3,W
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F

UOK22L

	RETURN

 end
