'---------------------------------------------------------------------------------------------------
' Parallel Loading ATmega328 controlled AD9850 DDS Signal Generator
' Author: Andrew Woodfield
' Date: October 2016 / April 2018 / February 2019 / February 2020
' File Name: Parallel_Load_SigGen_AD9850_328_Vxx.bas
'---------------------------------------------------------------------------------------------------
' Program Description:

' An ATmega328 is used to control a Chinese-made AD9850 DDS module and a standard 2 line x 16
' character alphanumeric LCD display in a simple AM/FM/CW signal generator. The ATmega328 runs from
' the internal 8MHz RC clock.
'
' The generator covers from 100kHz to about 50MHz with a low cost AD9850 DDS module. Additional
' software and hardware allows use of the AD9850/1 DDS aliased outputs to generate beyond this
' range, usually limited by the DDS clock to less the Nyquist rate i.e. half the sampling/clock
' frequency used in the DDS module e.g. The usual 125 MHz clock in an AD9850 module usually limits
' DDS outputs to less than 62.5MHz, and typical use is constrained by aliasing to a maximum of
' 50MHz. However, by using an additional on-board highpass LC filter, this generator can cover an
' additional range (typically) from about 70 - 120 MHz.with the AD9850 module with reduced output
' and with useful output from 130 - 150MHz where software limits the upper edge of tuning. Further
' modifications to the HP filter would allow selection of other alias output ranges.
'
' The signal generator starts at 10.000 MHz. A low cost rotary encoder is used to adjust the signal
' generator output frequency and set the start and end frequencies for the scan mode.
'
' AM is generated using the current controlled amplitude features of the AD9850/1. PWM is used to
' control the AM modulation frequency and the AD9850 output level. Generating reasonable quality FM
' modulation is more demanding and requires the DDS frequency data be parallel loaded (5 bytes
' per frequency write) into the AD9850 using a short assembly code subroutine.
'
' A scanning mode is also provided in the generator to permit alignment of bandpass filters and
' similar tests. The implicit SINx/x amplitude rolloff of a DDS oscillator will impact any tests
' done across a wide frequency range. A future software upgrade may provide some basic output level
' compensation.
'
' Three push button switches are fitted to control the signal generator. The STEP switch selects
' the frequency tuning step size and the number of steps per scan in the scanning mode. This switch
' is the push-in switch normally integrated on the rotary encoder. The MODE switch selects the
' modulation mode - CW, AM, or FM. Three FM modulation settings are available: 2.5kHz deviation
' (i.e. narrowband, and shown on the LCD as FM-NB), 5kHz (traditional land mobile modulation, shown
' as FM-WB on the LCD) or 50kHz deviation (Broadcast FM modulation, shown as FM-BC on the display).
' The MODE switch allows selection of the 'Scanning' mode on the signal generator in which the
' generator is unmodulated.
'
' Once in the scanning mode, the third SCAN pushbutton, along with the rotary encoder and its STEP
' switch, allows setting of the start and end frequencies for the scan, and the number of steps per
' scan. It also starts and stops the generator scanning. Scanning can also be stopped by pressing
' the MODE button. The generator stops scanning and displays the last frequency it was generating
' during the scanning mode and returns to normal signal generator unmodulated operation.
'
' This version is for LEVEL-type rotary encoders
'---------------------------------------------------------------------------------------------------
' ATmega328 Fuse settings:
'
' LOCK Byte: 0ffh (No locks)
' EXTd Byte: 0ffh (BOD disabled)
' HIGH Byte: 0d9h
'   RSTDISBL 1 Not disabled
'       DWEN 1 DW not enabled
'      SPIEN 0 SPI programming enabled
'      WDTON 1 Watchdog timer off
'     EESAVE 1 EEPROM not preserved in erase
'    BOOTSZ1 0 Boot ROM size
'    BOOTSZ0 0
'    BOOTRST 1 No boot vector at startup
'
' LOW Byte:  0e2h
'     CKDIV8 1 NOT divided by 8
'      CKOUT 1 Not enabled
'       SUT1 1 Slow rising power
'       SUT0 0
'     CKSEL3 0 8 MHz internal RC clock
'     CKSEL2 0
'     CKSEL1 1
'     CKSEL0 0
'
'---------------------------------------------------------------------------------------------------
' Compiler Directives (some fundamental hardware issues)

$regfile = "m328def.dat"                                    'tells bascom which device is being used
$crystal = 8000000                                          'bascom needs to know how fast it is going
                                                             'Note: Internal RC clock is used!!!
'---------------------------------------------------------------------------------------------------
' Hardware Setups
' set up direction of all ports

Config Portb = Output
Config Portc = Output
Config Portd = Input

Config Pinc.6 = Input                                       'reset
Config PortD.4 = output                                     'led
config PORTD.5 = output                                     'ad9850 modulation
config PORTD.6 = output                                     'lcd
config PORTD.7 = output                                     'lcd

Set Portd.0                                                 'mode switch - writing 1 to input turns on pullup resistor
Set PortD.1                                                 'step switch
Set Portd.2                                                 'rotary encoder phase 0
Set Portd.3                                                 'rotary encoder phase 1

' Hardware Aliases:

Stepkey Alias Pind.1                                        'hardware alias for step button
Modekey Alias PinD.0                                        'CW/AM/FM/SCAN button
Qpin0 Alias Pind.2                                          'quadradure encoder pin 0
Qpin1 Alias Pind.3                                          'quadradure encoder pin 1

Ddsdata Alias PortB                                         'dds module
Fq_ud Alias Portc.4
W_clk Alias Portc.5
ad_rst Alias Portd.4

' Initialise ports so hardware starts correctly
' ...Done in code below...

'------------------------------------------------------------------
' Declare Variables

Dim Frq As dword                                            'DDS data for stored frequency
Dim dds1 As Byte At Frq Overlay                             'mirrored variables for dds write (LSB)
Dim dds2 As Byte At Frq + 1 Overlay
Dim dds3 As Byte At Frq + 2 Overlay
Dim dds4 As Byte At Frq + 3 Overlay                         'MSB

Dim Frqz As dword                                           'used to calculate current DDS data value

Dim Stepsize As Byte                                        'tuning step used for sig gen frequency tuning

Dim Frequency As Dword                                      'four byte UNsigned variable
Dim F1 As Byte At Frequency Overlay                         'mirrored variables for dds calc
Dim F2 As Byte At Frequency + 1 Overlay
Dim F3 As Byte At Frequency + 2 Overlay
Dim F4 As Byte At Frequency + 3 Overlay

Dim Cvdds As Byte                                           'counter variable for dds routine

Dim Countspin As Byte
Dim Delta As Dword
Dim modeval As Byte                                         'CW=0, AM=1, NBFM=2, FM=3, WBFM=4, SCAN=5

Dim Encval As Byte                                          'used in encoder interrupt
Dim Enctemp As Byte
Dim Letemp As Byte
Dim Lastencoder As Byte

Dim Dialpointer As Byte                                     'for LCD display routine
Dim Dialchar As Byte
Dim Dialcount As Byte
Dim S As String * 10                                        'for conversion of frequency dword value to chars
Dim Q As String * 3
Dim R As Byte                                               'temp stores for display routine

Dim Update As Bit
Dim Showfreq As Bit

dim fmcycle as byte                                         'state counter for fm interrupt routine
dim fmdev as dword                                          'deviation step
dim fmhdev as dword                                         'half deviation step

dim amcycle as byte                                         'counter for am modulation state machine
dim amlevel as byte                                         'sets am modulation via PWM register compare0b

dim startfreq as dword                                      'start frequency for scan function
dim endfreq as dword                                        'end frequency for scan function
dim scandelta as dword                                      'frequency step increment while scanning
dim scansteps as byte                                       'sets number of steps in scan (0=10, 1=20, 2=50, 3=100, 4= 200, 5=500)
dim scanmode as byte                                        '0=enter start frequency, 1=enter end frequency, 2=enter number of steps per scan, 3=start scanning
dim scanentryval as byte                                    'saves scanmode for use in variable update and scanning routines
dim startfreqstepsize as byte                               'tuning step for setting startfreq
dim endfreqstepsize as byte                                 'tuning step for setting endfreq
dim tempval as byte                                         'for scan display routine
dim deltadivider as word                                    'calculated freq step for scanning
dim scancount as word                                       'loop counter for scanning routine
dim adjustok as bit                                         'flags ability to change start/end freq for scan

' Initialise Variables
' ...Done in code below...

' Declare Constants

Const Timedelay = 25                                        'delay for main routine (ms)

' Declare subroutines
Declare Sub Displaylcd                                      'display stored DDS frequency
Declare Sub Calculate                                       'convert freq to DDS data
Declare Sub Sendfreq                                        'send DDS freq data to DDS
declare sub scandisplay                                     'display current scan settings
'
'------------------------------------------------------------------
' Program starts here
'
osccal = 240                                                'low cost encoder enhancement
'
reset ad_rst                                                'ensure ad9850 reset line starts low
waitms 100                                                  'to calm things down

'startup message on LCD first

Reset Update                                                'used in interrupt routine with encoder

Config Lcd = 16 * 2
Config Lcdpin = Pin , Db4 = Portc.0 , Db5 = Portc.1 , Db6 = Portc.2 , Db7 = Portc.3 , E = Portd.7 , Rs = Portd.6

Deflcdchar 1 , 16 , 16 , 16 , 16 , 16 , 16 , 31 , 32        'marker with pointer at 0 or 5
Deflcdchar 2 , 8 , 8 , 8 , 8 , 8 , 24 , 31 , 32             'marker with pointer at 1 or 6
Deflcdchar 3 , 4 , 4 , 4 , 4 , 4 , 20 , 31 , 32             'marker with pointer at 2 or 7
Deflcdchar 4 , 2 , 2 , 2 , 2 , 2 , 18 , 31 , 32             'marker with pointer at 3 or 8
Deflcdchar 5 , 1 , 1 , 1 , 1 , 1 , 17 , 31 , 32             'marker with pointer at 4 or 9
Deflcdchar 6 , 32 , 32 , 32 , 32 , 32 , 16 , 31 , 32        'marker w/o pointer
Deflcdchar 7 , 8 , 12 , 30 , 31 , 30 , 12 , 8 , 32          'scan mode - update pointer (points from left to right)
Cls


Lcd "ZL2PD AD9850 DDS"
Lowerline
Lcd "Signal Generator"
Wait 1

'ad9850 clock is running by now so reset chip

set ad_rst                                                  'reset the ad9850
waitms 2                                                    'duration here is not important
reset ad_rst
waitms 2                                                    'so we can see it happen

'enable the device output

set portd.5                                                 'turn ad9850 RF output fully on

'parallel load power up sequence

reset w_clk                                                 'initialise the various control pins
reset fq_ud
portB = 0                                                   'ddsdata=0
waitms 2                                                    'allow time to take effect

set w_clk                                                   'now do the parallel load powerup signalling
waitms 2
reset w_clk                                                 'by toggling w_clk pin

set fq_ud
waitms 2
reset fq_ud                                                 'and then the update pin

waitms 10                                                   'allow time to take effect

'configure the startup DDS frequency and basic configuration

Frequency = 10000000                                        'initialise frequency to 10MHz
Stepsize = 3                                                'and tuning step size to 1kHz
modeval = 0                                                 'CW
Countspin = 128                                             'initialise interrupt variable
Delta = 0
scandelta = 0

Startfreq = 1000000                                         'initialise scanning variables at power up
endfreq = 30000000                                          '1-30MHz, 100 steps
scandelta = 0                                               'freq increment for scan starts at zero
scansteps = 4                                               '0 = 10 , 1 = 20 , 2 = 50 , 3 = 100 , 4 = 200 , 5 = 500 steps per scan
scanmode = 0                                                '0=update startfreq, 1=update endfreq, 2=update scansteps, 3=run scan
scanentryval = 0                                            'mirrors scanmode
startfreqstepsize = 3                                       '100kHz steps
endfreqstepsize = 3                                         '100kHz steps

Cls

Calculate                                                   'get startup dds value
Sendfreq                                                    'and send it to dds
Displaylcd                                                  'and display frequency for first time

Config Int0 = change
Config Int1 = change
Const Debouncetime = 2

On Int0 Getencoder
On Int1 Getencoder

Enable Interrupts
Enable Int0
Enable Int1


'timer0 is used to generate a 31kHz clocked PWM output (OC0B i.e. PD5). This passes via an RC low pass
'filter and the resulting 0-5V DC level drives a voltage-controlled current sink. This current
'controls the AD9850 RF output level

'timer0 runs continuously so if compare0b=255 then oc0b output on portd.5 is always high and
'AD9850 RF output is set to full amplitude

'Config Timer0 = Pwm , Prescale = 1 , pwm = on , Compare B Pwm = Clear Up , compare0b = 255       'normal pwm (16kHz)  and turns 100% on

TCCR0a = &B00100011                                         'configure fast pwm (32kHz)
TCCR0B = &B00000001
compare0b = 255                                             'and turn 100% on

'timer 1 is clocked from the processor RC clock (8MHz). An interrupt is generated every 62.5uS
' (TCNT1 loaded in mainline code) which then updates the value loaded in timer0 to change the
' AD9850 RF output amplitude 16 times per 1000Hz cycle (1mS)

config timer1 = timer , prescale = 1
stop timer1
on OVF1 ammod                                               'AM modulation 1kHz tone generating interrupt routine
enable OVF1                                                 'enable overflow interrupt

'timer 2 is also clocked from the processor RC clock (8MHz) divided by 8. An interrupt is
'generated approx every 50uS (TCNT2 loaded in mainline code) which then updates the AD9850 output
'frequency about 20 times per 1000Hz cycle (1mS)

config timer2 = timer , prescale = 8                        'timer2 starts with this command so...
stop timer2
on OVF2 fmmod                                               'FM modulation interrupt routine
enable OVF2                                                 'enable overflow interrupt



'*************************************************************************
'         read the input switches
Do

If Stepkey = 0 and modekey = 1 Then                         'STEP keypress

   if modeval <> 5 then                                     'for normal sig gen operation
    Incr Stepsize
    If Stepsize = 7 Then Stepsize = 1
    R = Lookup(stepsize , Curpos)                           'get location of cursor
    Locate 1 , R                                            'and move it there
   end if

   if modeval = 5 then                                      'scan set/run mode
      select case scanentryval                              '0=update startfreq, 1=update endfreq, 2=update scansteps, 3=run scan
        Case 0 :
            incr startfreqstepsize                          'increment the pointer
            if startfreqstepsize = 6 then startfreqstepsize = 1
            R = lookup(startfreqstepsize , scancurpos)
            locate 1 , R                                    'and place underline to mark the right spot
        Case 1 :
            incr endfreqstepsize                            'increment the pointer
            if endfreqstepsize = 6 then endfreqstepsize = 1
            R = lookup(endfreqstepsize , scancurpos)
            locate 2 , R                                    'and place underline to mark the right spot
        Case 2 :
            incr scansteps                                  '0 = 10 , 1 = 20 , 2 = 50 , 3 = 100 , 4 = 200 , 5 = 500 steps per scan
            if scansteps = 6 then scansteps = 0
            Locate 2 , 12                                   'update the LCD directly
            Select Case scansteps                           'scansteps : (0 = 10 , 1 = 20 , 2 = 50 , 3 = 100 , 4 = 200 , 5 = 500)
              Case 0 : Lcd "  10 "
              Case 1 : Lcd "  20 "
              case 2 : Lcd "  50 "
              Case 3 : Lcd " 100 "
              Case 4 : lcd " 200 "
              Case 5 : lcd " 500 "
            End Select
      End Select                                            'note that case for scanmode=3 (run scan) is ignored / does nothing here
   end if

End If

while Stepkey = 0 and modekey = 1                           'wait for STEP key to be released by user
wend
waitms 10                                                   'mini-debounce


If Modekey = 0 and stepkey = 1 Then                         'MODE keypress
   Incr modeval

   if modeval = 1 then
     compare0b = 127                                        'set AM modulation PWM for 50% duty cycle
     amcycle = 255                                          'initialise am modulation to -1 to avoid offset
     TCNT1 = &HFE0C                                         'load timer 1 for 62.5uS interrupt (16 steps per 1kHz sinewave cycle of 1000uS)
     start timer1                                           'and start AM
   else
      stop timer1                                           'turn AM modulation PWM off
      compare0b = 255                                       'and turn ad9850 output fully on
   end if

   if modeval = 2 then                                      'narrow-band FM with 2.5kHz peak deviation
      tCNT2 = &HCE                                          'preload timer2 for 50uS interrupt rate
      fmdev = 12885
      fmhdev = 6442                                         'half-step
      fmcycle = 255                                         'initialise state counter to -1 to avoid carrier offset
      start timer2                                          'start FM modulator routine
   else
      stop timer2                                           'turn off FM modulation
   end if

   if modeval = 3 then                                      'standard FM with 5kHz peak deviation
      tCNT2 = &HCE                                          'preload timer2 for 50uS interrupt rate
      fmdev = 25770
      fmhdev = 12885
      fmcycle = 255                                         'initialise state counter to -1 to avoid carrier offset
      start timer2                                          'start FM modulator routine
   else
      if modeval <> 2 then stop timer2                      'turn off FM modulation (unless we just turned it on!)
   end if

   if modeval = 4 then                                      'broadcast FM with 50kHz peak deviation
      tCNT2 = &HCE                                          'preload timer2 for 50uS interrupt rate
      fmdev = 429497
      fmhdev = 214748
      fmcycle = 255                                         'initialise state counter to -1 to avoid carrier offset
      start timer2                                          'start FM modulator routine
   else
      if modeval <> 2 and modeval <> 3 then stop timer2     'turn off FM modulation (unless we just turned it on!)
   end if

   if modeval = 5 then                                      'select the SCAN mode
      scanmode = 0                                          'initialise scanmode where 0=update startfreq, 1=update endfreq, 2=update scansteps, 3=run scan
      scandisplay                                           'now show display for scan mode with current values of start, stop freq and number of steps
      compare0b = 0                                         'turn ad9850 output off for now - should be in CW mode at this point with timer 1 and timer 2 stopped
   end if

   if modeval = 6 then
      modeval = 0                                           'restore  modeval for CW (0) if necessary
      if scanentryval = 3 then frequency = startfreq        'if we have just left the scanning routine then restore the output frequency to the start frequency to tidy up
   end if                                                   'update of dds will be done via the update flag

   set update                                               'to flag display update is required to show new mode

end if                                                      'for MODE key

while modekey = 0 and stepkey = 1                           'wait for MODE key to be released by user
wend
waitms 10                                                   'mini-debounce


if modekey = 0 and stepkey = 0 then                         'if SCAN key is pressed

   if modeval = 5 then                                      'if we are in the SCAN mode (display has been changed in MODE key routine)...
      select case scanmode                                  '0=update startfreq, 1=update endfreq, 2=update scansteps, 3=run scan
         case 0:
            compare0b = 0                                   'turn ad9850 output off (in case just looping around in SCAN key routine)
            locate 1 , 2                                    'point lcd cursor at the right spot and...
            Lcd Chr(7)                                      'write the arrow char there to show user the currently selected scan parameter
            scanentryval = 0                                'and inform the interrupt routine which variable to update
            Locate 2 , 12                                   'and rewrite the scansteps value if previously said 'SCAN' from scanning mode
            Select Case scansteps                           'scansteps : (0 = 10 , 1 = 20 , 2 = 50 , 3 = 100 , 4 = 200 , 5 = 500)
              Case 0 : Lcd "  10 "
              Case 1 : Lcd "  20 "
              case 2 : Lcd "  50 "
              Case 3 : Lcd " 100 "
              Case 4 : lcd " 200 "
              Case 5 : lcd " 500 "
            End Select
            R = lookup(startfreqstepsize , scancurpos)      'find the digit in the selected parameter to be updated
            locate 1 , R                                    'and place an underline there
            reset update                                    'reset the update flag to ensure screen is not overwritten
            set adjustok                                    'and flag ok to interrupt routine to update variables
         case 1:
            locate 1 , 2                                    'point lcd cursor at the right spot and...
            Lcd ":"                                         'update previously selected scan parameter's pointer
            locate 2 , 2                                    'point lcd cursor at the right spot and...
            Lcd Chr(7)                                      'write the arrow char there to show user the currently selected scan parameter
            R = lookup(endfreqstepsize , scancurpos)        'find the digit in the selected parameter to be updated
            locate 2 , R                                    'and place underline there
            scanentryval = 1                                'and inform the interrupt routine which variable to update
         case 2:
            locate 2 , 2                                    'point lcd cursor at the right spot and...
            Lcd ":"                                         'update previously selected scan parameter's pointer
            locate 1 , 11                                   'point lcd cursor at the right spot and...
            Lcd Chr(7)                                      'write the arrow char there to show user the currently selected scan parameter
            scanentryval = 2                                'and inform the interrupt routine which variable to update
         case 3:
            locate 1 , 11                                   'point lcd cursor at the right spot and...
            Lcd " "                                         'erase the previous arrow char
            scanentryval = 3                                'and inform the interrupt routine NOT to update any variable
            reset adjustok
            frequency = startfreq                           'get ready to write new scan value to DDS
            scandelta = endfreq                             'calculate required scan freq step
            scandelta = scandelta - startfreq               'endfreq-startfreq
            Select Case scansteps                           'scansteps : (0 = 10 , 1 = 20 , 2 = 50 , 3 = 100 , 4 = 200 , 5 = 500)
              Case 0 : deltadivider = 10
              Case 1 : deltadivider = 20
              case 2 : deltadivider = 50
              Case 3 : deltadivider = 100
              Case 4 : deltadivider = 200
              Case 5 : deltadivider = 500
            End Select
            scandelta = scandelta \ deltadivider            'so calculate the required scan freq step
            scancount = 0                                   'reset the scan loop counter
            calculate                                       'determine new DDS parameters
            sendfreq                                        'and send them to DDS
            compare0b = 255                                 'turn ad9850 output 100% on
            locate 2 , 12                                   'point lcd cursor at the right spot and...
            Lcd " SCAN"                                     'show scan is running
            set update                                      'to start scanning
       end select
       incr scanmode                                        'all done so incr scanmode for next time
       if scanmode = 4 then scanmode = 0
   end if
end if

while modekey = 0 and stepkey = 0                           'wait for SCAN key to be released by user
wend
waitms 10                                                   'mini-debounce


if update = 1 then                                          'encoder has turned or a display update has been requested!

   if modeval <> 5 then                                     'if we are in normal sig gen modes...

      Disable interrupts

      If Stepsize = 1 Then Delta = 10
      If Stepsize = 2 Then Delta = 100
      If Stepsize = 3 Then Delta = 1000
      If Stepsize = 4 Then Delta = 10000
      If Stepsize = 5 Then Delta = 100000
      If Stepsize = 6 Then Delta = 1000000


      If Countspin < 128 Then Frequency = Frequency - Delta
'      if frequency < 100000 then frequency = 100000         'limit generator to 100kHz
      If Countspin > 128 Then Frequency = Frequency + Delta
'      if frequency > 150000000 then frequency = 150000000   'limit generator to 150MHz


      Countspin = 128                                       'reset for next time
      Reset Update                                          'for next time

      Enable Interrupts

      if modeval = 2 or modeval = 3 or modeval = 4 then stop timer2       'must pause FM modulation while DDS frequency is updated

      Calculate                                             'get new dds value
      Sendfreq                                              'and send it to dds
      Displaylcd                                            'and flag display update

      if modeval = 2 or modeval = 3 or modeval = 4 then start timer2       'and now restart FM if mode is selected


   else                                                     'we ARE in scan mode (modeval=5)

      if scanentryval = 0 or scanentryval = 1 and adjustok = 1 then       'set/change start or end scan freq

         if scanentryval = 0 then tempval = startfreqstepsize       'get correct stepsize value for selected parameter (1-6, but 6 ignored)
         if scanentryval = 1 then tempval = endfreqstepsize

         Disable interrupts

         If tempval = 1 Then delta = 1000                   'these values may require adjustment depending on the
         If tempval = 2 Then delta = 10000                  'specific mechanical operation of the rotary encoder used
         If tempval = 3 Then delta = 100000
         If tempval = 4 Then delta = 1000000
         If tempval = 5 Then delta = 10000000

         If Countspin < 128 Then
            if scanentryval = 0 then startfreq = startfreq - Delta
            if scanentryval = 1 then endfreq = endfreq - delta
         end if

         If Countspin > 128 Then
            if scanentryval = 0 then startfreq = startfreq + Delta
            if scanentryval = 1 then endfreq = endfreq + delta
         end if

         Countspin = 128                                    'reset for next time
         Reset Update

         Enable Interrupts

         scandisplay                                        'and update the scandisplay to show user the change

      end if

                                                            'note: scanmode=2 case where scanstep is updated is handled in STEP key routine

      if scanentryval = 3 then                              'the scanning mode is selected  Note: No reset of UPDATE flag here

         incr scancount
         if scancount < deltadivider then
            frequency = frequency + scandelta               'add increment to current scan frequency
         else
            scancount = 0                                   'reset the values to scan again
            frequency = startfreq
         end if

         Calculate                                          'get new dds value
         Sendfreq                                           'and send it to dds

      end if

   end if

end if


Loop


'**********************************************************************


'**********************************************************************
' change in levels on int0 or int1 will call this routine
' routine is adapted from millier routine in Circuit Cellar #152
' and takes about 30uS at 8 MHz (excluding debouncing delays)
'**********************************************************************

Getencoder:

   encval = pind                                            'read pins
   Encval = Encval And 12                                   'mask off required pins (seems to need 2 steps)
   enctemp = Encval                                         'save this new encoder value
   Shift Encval , Right , 2
   Letemp = Lastencoder
   Encval = Encval + Letemp                                 'now have both old and new encoder levels
   Letemp = Lookup(encval , Relookup)                       'saves/equalises time c/w nested XOR/IF
   If Letemp = 1 Then incr Countspin
   If Letemp = -1 Then decr Countspin
   Set Update
   Lastencoder = Enctemp                                    'save the new encoder value for next time
   waitms debouncetime

   eifr = 3                                                 'and clear any outstanding interrupt before exiting

Return                                                      'from interrupt

Relookup:
Data 0 , -1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , -1 , 0

'*************************************************************************


'**********************************************************************
'    Interrupt Routine for Frequency Modulation
'**********************************************************************

fmmod:
   tCNT2 = &HCE                                             'reload timer2 for next 50uS interrupt
   incr fmcycle
   if fmcycle = 20 then fmcycle = 0                         'count through simple 20 state machine
   select case fmcycle
      case 0 to 2 : frq = frq + fmdev                       'fmdev sets peak deviation
      case 3 to 4 : frq = frq + fmhdev                      'to better emulate sinewave
      case 5 to 6 : frq = frq - fmhdev
      case 7 to 12 : frq = frq - fmdev
      case 13 to 14 : frq = frq - fmhdev
      case 15 to 16 : frq = frq + fmhdev
      case 17 to 19 : frq = frq + fmdev
   end select
   sendfreq
return

'**********************************************************************
'    Interrupt Routine for Amplitude Modulation
'**********************************************************************

ammod:
   TCNT1 = &HFE0C                                           'reload timer1 for next 62.5uS interrupt
   incr amcycle
   if amcycle = 16 then amcycle = 0                         '16 state machine
   select case amcycle
      case 0 : amlevel = 100                                'sets mid-modulation "dc" level
      case 1 to 4 : amlevel = amlevel + 14                  'add level step to compare1b
      case 5 to 12 : amlevel = amlevel - 14                 'subtract level step from compare1b
      case 13 to 15 : amlevel = amlevel + 14                'add level step to compare1b
   end select
   compare0b = amlevel                                      'write new value to PWM register in timer0
return

'**********************************************************************
'   Subroutines
'**********************************************************************
'
'
'**********************************************************************
'    Addition and multiplication for DDS - Assumes 125MHz crystal
'**********************************************************************
Sub Calculate

   Frq = 0
   Frqz = 0
   Frqz = F1 * 34
   Frq = Frq + Frqz
   Frqz = F2 * 8796
   Frq = Frq + Frqz
   Frqz = F3 * 2251799
   Frq = Frq + Frqz
   Frqz = F4 * 576460752
   Frq = Frq + Frqz

End Sub


'**********************************************************************
' Send frequency data to DDS using parallel load method
' Assembly code is used to achieve adequate update speed with the
' five bytes required for each frequency update being sent in parallel
' to the AD9850 along with framing (FQ_UD) and clocking (WCLK) signals
'**********************************************************************

Sub Sendfreq

   $asm

      CLR R1                                                'Start by sending control byte (00) to AD9850
      out portb , R1                                        'write data to port
      sbi Portc,5                                           'and toggle write clock bit...
      Cbi Portc,5

      LOADADR dds4 , X                                      'load address of dds4 (MSB) into register pair X
      Ld R1, X                                              'then load dds4 value into R1
      out portb , R1                                        'write data to port
      sbi Portc,5                                           'and toggle write clock bit...
      Cbi Portc,5

      LOADADR dds3 , X
      Ld R1, X
      out portb , R1
      sbi Portc,5
      Cbi Portc,5

      LOADADR dds2 , X
      Ld R1, X
      out portb , R1
      sbi Portc,5
      Cbi Portc,5

      LOADADR dds1 , X                                      'load address of dds1 (LSB) into register pair X
      Ld R1, X
      out portb , R1
      sbi Portc,5
      Cbi Portc,5

      sbi Portc,4                                           'flag end of dds write (loads data and reset DDS addr pointer)
      Cbi Portc,4

   $end asm

End Sub

'**********************************************************************
' Displays frequency on the 16*2 LCD to 1 Hz resolution
' with slide-rule type scale on top line
'**********************************************************************

Sub Displaylcd

Lowerline                                                   'point cursor at lower line of lcd to write the graphic display

S = Str(frequency)                                          'convert current frequency (dword) into string

' get value of 100 kHz digit Note: strings are always left justified
select case frequency

   case is > 99999999 : Q = Mid(s , 4 , 1)                  '100MHz or more
   case is > 9999999 : Q = Mid(s , 3 , 1)                   '10MHz or more
   case is > 999999 : Q = Mid(s , 2 , 1)                    '1MHz or more
   case else : Q = Mid(s , 1 , 1)                           'must be less than 1MHz

end select

Dialpointer = Val(q)                                        'and convert it back to a value
Incr Dialpointer                                            ' ...so it points to lcd location(1-10) where pointer char will be written

'now get value of 10 kHz digit

select case frequency

   case is > 99999999 : Q = Mid(s , 5 , 1)                  '100MHz or more
   case is > 9999999 : Q = Mid(s , 4 , 1)                   '10MHz or more
   case is > 999999 : Q = Mid(s , 3 , 1)                    '1MHz or more
   case else : Q = Mid(s , 2 , 1)                           'must be less than 1MHz

end select

R = Val(q)                                                  'convert to numeric value
Shift R , right                                             'value divided by 2
Dialchar = R + 1                                            'get 10's kHz figure to point to reqd pointer char (Range 1-5)

For Dialcount = 1 To 10                                     'now write the characters for the 'slide rule' type graphics display
   If Dialcount = Dialpointer Then
      Lcd Chr(dialchar)
     Else
      Lcd Chr(6)
   End If
Next Dialcount
lcd "      "                                                'and erase to end of the top line


Upperline                                                   'point cursor at top line to write the frequency

select case frequency                                       'parse string to extract and display MHz, kHz and Hz as MMM,KKK,HHH or MM,KKK,HHH or M,KKK,HHH or KKK,HHH .

   case is > 99999999 :                                     '100MHz or more
      Q = Mid(s , 1 , 3)
      Lcd Q ; ","
      Q = Mid(s , 4 , 3)
      Lcd Q ; ","
      Q = Mid(s , 7 , 3)
      Lcd Q

   case is > 9999999 :                                      '10MHz or more
      Lcd " "
      Q = Mid(s , 1 , 2)
      Lcd Q ; ","
      Q = Mid(s , 3 , 3)
      Lcd Q ; ","
      Q = Mid(s , 6 , 3)
      Lcd Q

   case is > 999999 :                                       '1MHz or more
      Lcd "  "
      Q = Mid(s , 1 , 1)
      Lcd Q ; ","
      Q = Mid(s , 2 , 3)
      Lcd Q ; ","
      Q = Mid(s , 5 , 3)
      Lcd Q

   case else :                                              'must be less than 1MHz
      Lcd "    "
      Q = Mid(s , 1 , 3)
      Lcd Q ; ","
      Q = Mid(s , 4 , 3)
      Lcd Q

end select

Locate 1 , 12
lcd " MHz "                                                 'and add units (also cleans up stray chars from scan display)

'Now write Mode details to the LCD on LOWER line

Locate 2 , 12                                               'shift LCD pointer
Select Case modeval
  Case 0 : Lcd "CW   "
  Case 1 : Lcd "AM   "
  Case 2 : Lcd "FM-NB"
  Case 3 : lcd "FM-WB"
  Case 4 : lcd "FM-BC"
End Select

'last step is to leave cursor underline at correct location (Must be last thng done to ensure cursor is shown correctly)
R = Lookup(stepsize , Curpos)                               'get location of frequency (underline) cursor
Locate 1 , R                                                'and move it there

End Sub

Curpos:
Data 10 , 10 , 9 , 7 , 6 , 5 , 3                            'first table data value is ignored

'*************************************************************************
' Routine to display scan settings on the 16*2 LCD including start and
' end frequencies, and number of steps per scan. Also writes cursor to
' data field to be updated
'*************************************************************************

Sub scandisplay

Upperline                                                   'point cursor at top line of lcd

lcd "S:"

S = Str(startfreq)                                          'convert current start frequency (dword) into string

If startfreq < 1000000 Then                                 'write MHz and kHz digits with dot separator
   Lcd "  0."
   Q = Mid(s , 1 , 3)
   lcd Q
else
   If startfreq < 10000000 then
      Lcd "  "
      Q = Mid(s , 1 , 1)
      lcd Q ; "."
      Q = Mid(s , 2 , 3)
      Lcd Q
   else
      if startfreq < 100000000 then
         Lcd " "
         Q = Mid(s , 1 , 2)
         lcd q ; "."
         Q = Mid(s , 3 , 3)
         Lcd Q
      else
         Q = Mid(s , 1 , 3)
         lcd q ; "."
         Q = Mid(s , 4 , 3)
         Lcd Q
      end if
   end if
End If
lcd "  "                                                    'add two extra spaces to clean up


Lowerline

lcd "E:"

S = Str(endfreq)                                            'convert current end frequency (dword) into string

If endfreq < 1000000 Then                                   'write MHz and kHz digits with dot separator
   Lcd "  0."
   Q = Mid(s , 1 , 3)
   lcd Q
else
   If endfreq < 10000000 then
      Lcd "  "
      Q = Mid(s , 1 , 1)
      lcd Q ; "."
      Q = Mid(s , 2 , 3)
      Lcd Q
   else
      if endfreq < 100000000 then
         Lcd " "
         Q = Mid(s , 1 , 2)
         lcd q ; "."
         Q = Mid(s , 3 , 3)
         Lcd Q
      else
         Q = Mid(s , 1 , 3)
         lcd q ; "."
         Q = Mid(s , 4 , 3)
         Lcd Q
      end if
   end if
End If
lcd " "                                                     'add an extra space to clean up here too


Locate 1 , 12                                               'and write number of scansteps
lcd "Steps"

Locate 2 , 12
Select Case scansteps                                       'scansteps : (0 = 10 , 1 = 20 , 2 = 50 , 3 = 100 , 4 = 200 , 5 = 500)
  Case 0 : Lcd "  10 "
  Case 1 : Lcd "  20 "
  case 2 : Lcd "  50 "
  Case 3 : Lcd " 100 "
  Case 4 : lcd " 200 "
  Case 5 : lcd " 500 "
End Select

End Sub

scancurpos:                                                 'used in step routine when in scan modes
Data 10 , 9 , 8 , 7 , 5 , 4 , 3 , 2                         'first table data value is ignored

'*************************************************************************