; File TURBOTMR.ASM
; Uses assembly code for PIC16F84 microcontroller

; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _HS_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

; RAM

ROLLING1	equ H'0C'	; threshold store MS byte
ROLLING2	equ	H'0D'	; threshold store 
ROLLING3	equ	H'0E'	; threshold store 
ROLLING4	equ	H'0F'	; threshold store 
ROLLING5	equ	H'10'	; threshold store 
ROLLING6	equ	H'11'	; threshold store 
ROLLING7	equ	H'12'	; threshold store 
ROLLING8	equ	H'13'	; threshold store 
ROLLING9	equ	H'14'	; threshold store 
ROLLING10	equ	H'15'	; threshold store 
ROLLING11	equ	H'16'	; threshold store 
ROLLING12	equ	H'17'	; threshold store 
ROLLING13 	equ	H'18'	; threshold store 
ROLLING14	equ	H'19'	; threshold store 
ROLLING15	equ	H'1A'	; threshold store 
ROLLING16	equ	H'1B'	; threshold store 
ROLLING17	equ	H'1C'	; threshold store 
ROLLING19	equ	H'1D'	; threshold store 
ROLLING20	equ	H'1E'	; threshold store 
ROLLING21	equ	H'1F'	; threshold store 
ROLLING22	equ	H'20'	; threshold store 
ROLLING23	equ	H'21'	; threshold store 
ROLLING24	equ	H'22'	; threshold store 
ROLLING25	equ	H'23'	; threshold store 
ROLLING26	equ	H'24'	; threshold store 
ROLLING27	equ	H'25'	; threshold store 
ROLLING28	equ	H'26'	; threshold store 
ROLLING29	equ	H'27'	; threshold store 
ROLLING30	equ	H'28'	; threshold store 
ROLLING31	equ	H'29'	; threshold store 
ROLLING32	equ	H'2A'	; threshold store LS byte

STATUS_TMP 	equ H'2C'	; temp storage for status during interrupt
W_TMP		equ	H'2D'	; temporary storage for w during interrupt
FSR_STO		equ	H'2E'	; storage of FSR in interrupt
ROLL_T		equ	H'2F'	; maximum idle time setting
ROLL_I		equ	H'30'	; rolling idle time ms
ROLL_J		equ	H'31'	; rolling idle time ls
ROLL_S		equ	H'32'	; rolling time storage
COUNT1		equ	H'33'	; timer counter
COUNT2		equ	H'34'	; timer counter 
COUNT3		equ	H'35'	; idle timing

CPWM		equ	H'37'	; pwm out from ROLL_I
CPWMM		equ	H'38'	; pwm counter
EXTN0		equ	H'39'	; delay extension and working register
EXTN		equ	H'3A'	; delay extension and working register
EXTN1		equ	H'3B'	; delay extension 
TEMP_3		equ	H'3C'	; delay register
ROLLT_MULT	equ	H'3D'	; multiplier for ROLL_T 
LOOP		equ	H'3E'	; loop counter

; math routines
CARGB0      equ 0x46	; subtractor
CARGB1      equ 0x47	; subtractor
TEMP1	    equ 0x48	; temporary register
AARGB3		equ	0x49	; working register
AARGB1      equ 0x4A	; ls byte numerator
AARGB0      equ 0x4B
BARGB0      equ 0x4C	; denominator
BARGB1      equ 0x4D	; denominator
REMB0       equ 0x4E    ; most significant byte of remainder
LOOPCOUNT   equ 0x4F    ; loop counter

	
; define reset and interrupt vector start addresses

	org	0	  			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org 4				; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine

;
;******************************************************************************************* 
; idle period lookup table converts S1 value (0 to 16) to a time value multiplier of 3.75s

IDLE_P
	addwf	PCL,f		; add value to program counter
	retlw	D'04'		; 4 x 3.75s = 15s
	retlw	D'08'		; 8 x 3.75s = 30s
	retlw	D'16'		; 16 x 3.75s = 1 minute
	retlw	D'24'		; 24 x 3.75s = 1.5 minutes
	retlw	D'32'		; 32 x 3.75 = 2 minutes
	retlw	D'40'		; 40 x 3.75s = 2.5 minutes
	retlw	D'48'		; 48 x 3.75s = 3 minutes
	retlw	D'56'		; 56 x 3.75 = 3.5 minutes 
	retlw	D'64'		; 64 x 3.75s = 4 minutes
	retlw	D'80'		; 80 x 3.75s = 5 minutes
	retlw	D'96'		; 96 x 3.75 = 6 minutes 
	retlw	D'112'		; 112 x 3.75s = 7 minutes
	retlw	D'144'		; 144 x 3.75 = 9 minutes 
	retlw	D'176'		; 176 x 3.75s = 11 minutes
	retlw	D'208'		; 208 x 3.75s = 13 minutes
	retlw	D'240'		; 240 x 3.75 = 15 minutes  

; INTERRUPT

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT

	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp 
	movf	FSR,W		; store FSR
	movwf	FSR_STO 
	bcf		STATUS,RP0	; select bank 0
	bcf		INTCON,T0IF	; clear TMRO interrupt flag

; check counters

	incfsz	COUNT1,f	; increase counter 1 skip if overflowed
	goto 	CHK_UPD		; goto check update time
	incf	COUNT2,f	; increase counter 2 when counter 1 overflows

; timers relate to 10MHz/4/256/8 interrupt rate
CHK_UPD
	movf	COUNT2,w	; MS Byte of update counter to w
	xorlw	0x07		; compare MS Byte counter with value (07)
	btfss	STATUS,Z	; skip if equal
	goto	CK_OUT		; not, so continue 
	movf	COUNT1,w	; LS Byte of update counter
	xorlw	0xD2		; D2
	btfss	STATUS,Z	; skip if equal
	goto	CK_OUT		; not 1.64s second yet (1.64s x 256 = 7 minutes) change for different time
	clrf	COUNT1		; clear timer counter 1
	clrf	COUNT2		; and timer 2
	goto	TMNG
CK_OUT
	btfss	PORTA,1		; if set do timing (relays on)	
	goto	PWM_CT		; drive PWM out

; timer	@ 3.75 seconds/256 for idle period (multiplied to get idle time see lookup table).

TMNG
	btfss	PORTA,1		; if timing (relays on) decrease timer
	goto	DEC_HLD
	incf	COUNT3,f	; increase counter 3 skip if overflowed
	movf	COUNT3,w	; LS Byte of idle counter to w
	xorlw	D'18'		; compare LS Byte counter with value
	btfss	STATUS,Z	; skip if equal
	goto	PWM_CT		; not, so continue 
	clrf	COUNT3		; clear timer counter 1
	movf	ROLL_I,w	; check value
	btfss	STATUS,Z	; if zero check next byte
	goto	CALL_SUB	; not zero 
	movf	ROLL_J,w	; check value
	btfsc	STATUS,Z	; if zero do not decrease
	goto	PWM_CT
CALL_SUB
	call	SUBTRACT	; decrease by 1
; if zero timed out
	movf	ROLL_I,w	; check value
	btfss	STATUS,Z	; if zero check ls byte
	goto	PWM_CT		; zero 
	movf	ROLL_J,w	; check value
	btfss	STATUS,Z	; if zero stop
	goto	PWM_CT
	goto	STOP_D

; subtract ROLL_I (ms) ROLL_J (ls) - (1)CARGBO CARGB1 
SUBTRACT
	movlw	0x01
	movwf	CARGB1		; 1
	clrf	CARGB0
	call	NEG_A		; complement of A
	movf	CARGB1,w 
	addwf	ROLL_J,f 	; add lsb
	btfsc	STATUS,C	; add carry
	incf	ROLL_I,f 
	movf	CARGB0,w 
	addwf	ROLL_I,f 
	return

NEG_A
	comf	CARGB1,f 
	incf	CARGB1,f 
	btfsc	STATUS,Z
	decf	CARGB0,f 
	comf	CARGB0,f 
	return	               

; check comparator signal load level to rolling counter

DEC_HLD
	movlw	ROLLING1	; first address of rolling counter
	movwf	FSR			; Indirect pointer
	movlw	D'32'		; number of counters
	movwf	LOOP
	btfss	PORTB,0		; check sense input
	goto	REVRSE		; reverse sense
	btfss	PORTA,2		; if RA2 high then overthreshold
	goto	NOT_OVR
OVER
	bsf		PORTB,2		; LED on
	bsf		STATUS,C	; set carry bit
	goto	ROLL_CNT	; move rolling counter
NOT_OVR
	bcf		PORTB,2		; LED off
	bcf		STATUS,C	; clear carry
	goto	ROLL_CNT
REVRSE
	btfsc	PORTA,2		; if low then overthreshold
	goto	NOT_OVR		
	goto	OVER
ROLL_CNT
	rrf		INDF,f		; move ms byte right (ROLLING counters store the successive RA2 sampled value in a 255 long chain)
	decfsz	LOOP,f		; if zero finished
	goto	MORE_LOOP
	goto	PWM_CT
MORE_LOOP
	incf	FSR,f		; next byte
	goto	ROLL_CNT

; check for a low on RA0, set RA1 high if not timed out
IGN_OFF
	btfsc	PORTA,0		; is RA0 low
	goto	END_INT
	btfsc	PORTA,1		; is RA1 high (relays on)
	goto	END_INT
	
; debounce RA0
	
; Delay
	movlw	D'25'		; 25 = 10ms delay extension
	movwf	EXTN1
WAIT2
	call	DELMOR		; delay
	decfsz	EXTN1,f		; out from delay when zero
	goto	WAIT2

	btfsc	PORTA,0		; is RA0 low
	goto	END_INT
	btfsc	PORTA,1		; is RA1 high (relays on)
	goto	END_INT

; if counter zero it has timed out
	
	movf	ROLL_I,w	; check value
	btfss	STATUS,Z	; if zero check ls byte
	goto	SET_A		; not zero 
	movf	ROLL_J,w	; check value
	btfss	STATUS,Z	; if zero stop
	goto	SET_A		; not zero
	goto	STOP_D		; timer stopped
SET_A
	bsf		PORTA,1		; set high as not timed out
 	goto	END_INT

; pwm output of rolling measurement (% over threshold in the rolling period)

PWM_CT
	movf	CPWMM,w		; 255 count check if 0
	btfsc	STATUS,Z
	goto	ZRO_CT		; zero so reload values
	movf	CPWM,w		; counter
	btfsc	STATUS,Z	; if zero clear RA3 (PWM out)	
	goto	ZRO_PWM
	decfsz	CPWM,f		; decrease value until 0
	goto	NO_LO
ZRO_PWM
	bcf		PORTA,3		; pwm out low
NO_LO
	decfsz	CPWMM,f		; decrease 255 counter til 0 then reload to 255 
	goto	RECLAIM
ZRO_CT
	movlw	0xFF		; start next pwm cycle
	movwf	CPWMM		; restore counter to 255
	movf	ROLLT_MULT,w; rolling multiplier
	movwf	CPWM		; counter
	btfss	STATUS,Z	; if zero do not set RA3
	bsf		PORTA,3		; pwm high for CPWM count

; end of interrupt reclaim w and status 

RECLAIM	
	btfss	PORTA,0		; is RA0 low
	goto	IGN_OFF		; if low check for valid ignition off signal
END_INT
	movf	FSR_STO,w	; FSR storage
	movwf	FSR			; restore FSR
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register

	retfie				; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'11110011'	; (RB0,1-RB4-7 input)
	movwf	TRISB		; port B data direction register
	movlw	B'00000010'	; 
	movwf	OPTION_REG	; TMRO prescaler /8, PORTB pullups enabled
	movlw   B'00010101'	; (RA1,RA3 outputs, RA0,RA2, RA4 input)
	movwf   TRISA		; A port data direction register
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'00000000'	; RA outputs low
	movwf	PORTA		; portA outputs low
	bcf		PORTB,2		; threshold led off
		
; initial conditions
	movlw	ROLLING1	; first address of rolling counter
	movwf	FSR			; Indirect pointer
	movlw	D'32'		; number of counters
	movwf	LOOP
CLR_ROLL
	clrf	INDF		; clear
	decfsz	LOOP,f		; if zero finished
	goto	MORE_LP
	goto	CONT_CLR

MORE_LP
	incf	FSR,f		; next byte
	goto	CLR_ROLL
CONT_CLR	
	clrf	COUNT2		; reload counters
	clrf	COUNT1
	clrf	COUNT3
	clrf	CPWMM		; pwm output counter
	clrf	CPWM		; pwm out
	clrf	ROLL_I		; timing 
	clrf	ROLL_J
	clrf	ROLLT_MULT	; multiplier

	movlw	0x40		; delay ~3s
	movwf	EXTN1
DLY1
	movlw	0xFF		; delay extension
	movwf	EXTN
DLY2
	call	DELMOR		; delay
	decfsz	EXTN,f		; out from delay when zero
	goto	DLY2
	decfsz	EXTN1,f
	goto	DLY1
	
; convert idle time switch setting to a time period value

	movlw	0x00
	movwf	EXTN		; switch read value set to 0
	btfss	PORTB,7		; ms bit of selection switch
	bsf		EXTN,3
	btfss	PORTB,5		; next significant bit
	bsf		EXTN,2
	btfss	PORTB,4		; next significant bit 
	bsf		EXTN,1
	btfss	PORTB,6		; next significant bit
	bsf		EXTN,0		; value now between 00 and 0F
	movf	EXTN,w		; get value
	call	IDLE_P		; get multiplier value
	movwf	ROLL_T		; this is the maximum idle time setting 15s to 15 minutes

; interrupt enable and program now runs

	bsf	INTCON,T0IE	; set interrupt enable for TMR0 
	bsf	INTCON,GIE	; set global interrupt enable for above

; check if timing on or off

T_ONOFF	
	btfss	PORTA,1		; check output drive to relays ie timing on or off
	goto	CALCUL		; turbo timing off so update timing 

; Check for reset
	
CK_RST
	btfsc	PORTB,1		; check if reset switch closed
	goto	T_ONOFF
	movlw	0xFF		; delay extension
	movwf	EXTN
WAIT_T
	call	DELMOR		; delay
	decfsz	EXTN,f		; out from delay when zero
	goto	WAIT_T
	btfsc	PORTB,1		; check if reset switch closed
	goto	T_ONOFF
STOP_D
	bcf		INTCON,GIE	; global interrupt enable 
	bcf		PORTA,1		; relays off
	bcf		PORTA,3		; pwm output low
DLYX
	movlw	0x10		; delay ~1s
	movwf	EXTN1
STPD
	movlw	0xFF		; delay extension
	movwf	EXTN
WAIT0
	call	DELMOR		; delay
	decfsz	EXTN,f		; out from delay when zero
	goto	WAIT0
	decfsz	EXTN1,f
	goto	STPD		; loop till power switched off via relay
;	goto	DLYX
	goto	MAIN		; start again
	
; calculate timer value

; ROLL_T (max idle period) x calculated rolling value
; calculate rolling value
; ROLLING1, ROLLING2.....ROLLING32 have 255 bits. Each bit represents the comparator level in time (ms bit is the most recent
; ls bit the status of the comparator some 7 minutes earlier, The number of 1's are added to get the value.) Weighting is applied 
; so the latest comparator readings have more effect than those registered earlier. There are 4 blocks. Block 1 is the first 8 bytes
; Rolling 1 to Rolling 8. Block 2 is Rolling 9 to Rolling 16. Block 3 is Rolling 17 to Rolling 24. Block 4 is Rolling 25 to Rolling 32.
; Block 1 has a 0.35 loading, Block 2 a 0.26 loading, Block 3 a 0.21 loading and block 4 a 0.18 loading. (without loading each would
; have the same 0.25 loading) 
; multiply block 1 x 1.4, block 2 by 1.04, block 3 x 0.84 and block 4 x 0.72 (total is 4)
; without loading multiply each by 1 (total 4) 

CALCUL
	movlw	ROLLING1
	movwf	FSR
	call	BLOCK_ADD	; block1 value in EXTN
; add weighting
; multiply and divide
; Block 1 weighting x140/100 for x1.4
	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'140'		; x 140
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'100'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB1,w
	 
; move to ROLL_S 
	movwf	ROLL_S 

	call	BLOCK_ADD	; block2 value in EXTN
; add weighting
; multiply and divide
; Block 2 weighting x208/200 for x1.04
	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'208'		; x 208
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'200'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB1,w
	 
; Add each successive block to get final value
	addwf	ROLL_S,f 

	call	BLOCK_ADD	; block3 value in EXTN
; add weighting
; multiply and divide
; Block 3 weighting x210/250 for x 0.84
	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'210'		; x 210
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'250'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB1,w
	 
; Add each successive block to get final value
	addwf	ROLL_S,f 

	call	BLOCK_ADD	; block4 value in EXTN
; add weighting
; multiply and divide
; Block 4 weighting x180/250 for x 0.72

	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'180'		; x 180
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'250'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB1,w
	 
; Add each successive block to get final value
	addwf	ROLL_S,f 
	movlw	0xFF
	btfsc	STATUS,C	; if carry set overflow so make ROLL_S FF
	movwf	ROLL_S

; check ROLL_S for under count of 3
	
	movlw	0x04
	subwf	ROLL_S,w
	btfss	STATUS,C	; if carry clr then ROLL_S less than 3
	clrf	ROLL_S		; clear if less than 3

; multiply by factor to compensate for the fact that it is rare to maintain 100% overthreshold fo 7 minutes

	movlw	0x04		; factor
	movwf	AARGB0
	movf	ROLL_S,w
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply
	movf	AARGB1,w	; ls byte
	movwf	ROLL_S
	movf	AARGB0,w	; ms byte
	btfsc	STATUS,Z	; if zero continue
	goto	MULT_TS
	movlw	0xFF
	movwf	ROLL_S		; set at maximum 		
	 	
; multiply ROLL_T (maximum value) by ROLL_S for ROLLING idle time

MULT_TS
	movf	ROLL_T,w	; maximum time
	movwf	AARGB0
	movf	ROLL_S,w
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply

; place in ROLL_I, ROLL_J (MS,LS bytes) This is actual timeout period

	movf	AARGB1,w	; ls byte	
	movwf	ROLL_J
	movf	AARGB0,w	; ms byte
	movwf	ROLL_I

	movf	ROLL_S,w
	movwf	ROLLT_MULT	; factor that ROLL_T (max time) is altered by (factor from rolling ROLL_S)
	goto	T_ONOFF	

; ******************************************************************************
; Subroutines

; subroutine to add the number of 1's in each block
BLOCK_ADD
	clrf	EXTN		; start at 0
	movlw	0x08		; eight bytes
	movwf	LOOPCOUNT
NXT	call	ADD_BIT		; add the number of 1's in this byte
	addwf	EXTN,f		; storage
	incf	FSR,f
	decfsz	LOOPCOUNT,f
	goto	NXT
	return
ADD_BIT
	clrf	EXTN0		; start at zero
	btfsc	INDF,7		; is bit 7 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,6		; is bit 6 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,5		; is bit 5 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,4		; is bit 4 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,3		; is bit 3 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,2		; is bit 2 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,1		; is bit 1 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,0		; is bit 0 set
	incf	EXTN0,f		; add 1 if set
	movf	EXTN0,w
	return		

; delay 

DELMOR	
	movlw 	0xFF		; delay period
	movwf	TEMP_3
SMALER
	decfsz	TEMP_3,f	; reduce temp_3
	goto	SMALER		; temp_3 smaller by 1
	return				; end delay


; 8 x 8 multiply

EIGHTEIGHT      CLRF    AARGB1          ; clear partial product
UMUL0808L       MOVLW   0x08
                MOVWF   LOOPCOUNT
                MOVF    AARGB0,W

LOOPUM0808A
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    LUM0808NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808A

                CLRF    AARGB0
                RETLW   0x00

LUM0808NAP
                BCF     STATUS,C
                GOTO    LUM0808NA

LOOPUM0808
                RRF     BARGB0, F
                BTFSC   STATUS,C
                ADDWF   AARGB0, F
LUM0808NA       RRF    	AARGB0, F
                RRF    	AARGB1, F
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808

                return             


;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8      	CLRF            REMB0
                MOVLW           0x08
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           0x08
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B          RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
                return

	end	
